! ***********************************************************************
!
!   Copyright (C) 2011  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************


      module mix_info

      use const_def
      use num_lib
      use utils_lib
      use star_private_def
      use mlt_def

      implicit none


      contains


      subroutine set_mixing_info(s, ierr)
         ! set convection variables cdc and conv_vel starting from local MLT results.
         ! overshooting can also be added.
         use rates_def, only: i_rate
         use chem_def, only: ipp, icno, i3alf
         use overshoot, only: add_overshooting
         use turbulent_diffusion, only: &
            add_turbulent_diffusion, add_radiation_turbulence
         use mix_smoothing, only: set_newly_non_conv
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         
         integer :: nz, i, k, max_conv_bdy, max_mix_bdy
         real(dp) :: c, rho_face
         real(dp), pointer, dimension(:) :: eps_h, eps_he, eps_z, cdc_factor
         
         logical :: dbg

         include 'formats'
         
         ierr = 0
         dbg = .false.

         nullify(eps_h, eps_he, eps_z, cdc_factor)

         if (dbg) write(*, *) 'set_mixing_info'

         ierr = 0
         max_conv_bdy = 10 ! will automatically be increased if necessary
         max_mix_bdy = 10 ! will automatically be increased if necessary
         nz = s% nz
         
         s% num_conv_boundaries = 0
         if (.not. associated(s% conv_bdy_loc)) allocate(s% conv_bdy_loc(max_conv_bdy))
         if (.not. associated(s% conv_bdy_q)) allocate(s% conv_bdy_q(max_conv_bdy))
         if (.not. associated(s% top_conv_bdy)) allocate(s% top_conv_bdy(max_conv_bdy))
         if (.not. associated(s% burn_h_conv_region)) allocate(s% burn_h_conv_region(max_conv_bdy))
         if (.not. associated(s% burn_he_conv_region)) allocate(s% burn_he_conv_region(max_conv_bdy))
         if (.not. associated(s% burn_z_conv_region)) allocate(s% burn_z_conv_region(max_conv_bdy))
         
         s% num_mix_boundaries = 0
         if (.not. associated(s% mix_bdy_loc)) allocate(s% mix_bdy_loc(max_mix_bdy))
         if (.not. associated(s% mix_bdy_q)) allocate(s% mix_bdy_q(max_mix_bdy))
         if (.not. associated(s% top_mix_bdy)) allocate(s% top_mix_bdy(max_mix_bdy))
         if (.not. associated(s% burn_h_mix_region)) allocate(s% burn_h_mix_region(max_mix_bdy))
         if (.not. associated(s% burn_he_mix_region)) allocate(s% burn_he_mix_region(max_mix_bdy))
         if (.not. associated(s% burn_z_mix_region)) allocate(s% burn_z_mix_region(max_mix_bdy))

         s% cdc(1:nz) = s% mlt_cdc(1:nz)
         s% D_mix(1:nz) = s% mlt_D(1:nz)
         s% conv_vel(1:nz) = s% mlt_vc(1:nz)
         s% mixing_type(1:nz) = s% mlt_mixing_type(1:nz)
         
         if (s% remove_mixing_glitches) then
            
            if (dbg) k = 128
            
            if (dbg) write(*,*) 'call remove_tiny_mixing'
            call remove_tiny_mixing(s, ierr)
            if (failed('remove_tiny_mixing')) return
            
            if (dbg) write(*,*) 'call remove_mixing_singletons'
            call remove_mixing_singletons(s, ierr)
            if (failed('remove_mixing_singletons')) return
            
            if (dbg) write(*,*) 'call close_convection_gaps'
            call close_convection_gaps(s, ierr)
            if (failed('close_convection_gaps')) return
            
            if (dbg) write(*,*) 'call close_thermo_haline_gaps'
            call close_thermo_haline_gaps(s, ierr)
            if (failed('close_thermo_haline_gaps')) return
            
            if (dbg) write(*,*) 'call remove_thermohaline_dropouts'
            call remove_thermohaline_dropouts(s, ierr)
            if (failed('remove_thermohaline_dropouts')) return
            
            if (dbg) write(*,*) 'call close_semiconvection_gaps'
            call close_semiconvection_gaps(s, ierr)
            if (failed('close_semiconvection_gaps')) return
            
            if (dbg) write(*,*) 'call remove_embedded_semiconvection'
            call remove_embedded_semiconvection(s, ierr)
              if (failed('remove_embedded_semiconvection')) return
            
         end if

         if (dbg) write(*,*) 'call do_mix_envelope'
         call do_mix_envelope(s)

         call do_alloc(ierr)
         if (ierr /= 0) return
         
         do k=1,nz
            eps_h(k) = s% eps_nuc_categories(i_rate,ipp,k) + &
                       s% eps_nuc_categories(i_rate,icno,k)
            eps_he(k) = s% eps_nuc_categories(i_rate,i3alf,k)
            eps_z(k) = s% eps_nuc(k) - (eps_h(k) + eps_he(k))
         end do
         
         if (dbg) write(*,*) 'call set_mlt_boundary_info'
         call set_mlt_boundary_info(s, ierr)
         if (failed('set_mlt_boundary_info')) return
         
         if (dbg) write(*,*) 'call locate_convection_boundaries'
         call locate_convection_boundaries( &
            s, nz, eps_h, eps_he, eps_z, s% mstar, s% q, s% cdc, s% grada_at_face, s% gradr, ierr)
         if (failed('locate_convection_boundaries')) return
        
         if (dbg) write(*,*) 'call add_overshooting'
         call add_overshooting(s, ierr)
         if (failed('add_overshooting')) return
        
         if (dbg) write(*,*) 'call add_turbulent_diffusion'
         call add_turbulent_diffusion(s, ierr)
         if (failed('add_turbulent_diffusion')) return
        
         if (dbg) write(*,*) 'call add_radiation_turbulence'
         call add_radiation_turbulence(s, ierr)
         if (failed('add_radiation_turbulence')) return
         
         if (dbg) write(*,*) 'call set_cz_bdy_mass'
         call set_cz_bdy_mass(s, ierr)
         if (failed('set_cz_bdy_mass')) return
         
         cdc_factor(1) = 1
         s% D_mix(1) = 0
         do k = 2, nz
            rho_face = (s% dq(k-1)*s% rho(k) + s% dq(k)*s% rho(k-1))/(s% dq(k-1) + s% dq(k))
            cdc_factor(k) = (4*pi*s% r(k)**2*rho_face)**2
            s% D_mix(k) = s% cdc(k)/cdc_factor(k)
            if (is_bad_num(s% D_mix(k))) then
               write(*,2) 's% D_mix(k)', k, s% D_mix(k)
               stop 'set_mixing_info'
            end if
         end do
         if (s% set_min_D_mix .and. s% ye(nz) >= s% min_center_Ye_for_min_D_mix) then
            do k=1,nz
               if (s% D_mix(k) >= s% min_D_mix) cycle
               s% D_mix(k) = s% min_D_mix
               s% mixing_type(k) = minimum_mixing
            end do
         end if

         if (s% use_other_mixing) then
            if (dbg) write(*,*) 'call other_mixing'
            call s% other_mixing(s% id, ierr)
            if (failed('other_mixing')) return
         end if
         
         if (dbg) write(*,*) 'call set_newly_non_conv'
         call set_newly_non_conv(s, ierr)
         if (failed('set_newly_non_conv')) return
         
         s% D_mix_non_rotation(1:nz) = s% D_mix(1:nz)

         if (s% rotation_flag) then
         
            call update_rotation_mixing_info(s,ierr)
            if (failed('update_rotation_mixing_info')) return
                     
            do k = 2, nz
               s% cdc(k) = s% D_mix(k)*cdc_factor(k)               
               if (s% D_mix(k) /= 0 .and. s% mixing_type(k) == no_mixing) then
                  !write(*,2) 'mixing_type = rotation_mixing', k
                  s% mixing_type(k) = rotation_mixing
               end if
               if (is_bad_num(s% cdc(k))) then
                  write(*,2) 's% cdc(k)', k, s% cdc(k)
                  write(*,2) 's% D_mix(k)', k, s% D_mix(k)
                  write(*,2) 'cdc_factor(k)', k, cdc_factor(k)
                  write(*,2) 's% am_nu(k)', k, s% am_nu(k)
                  write(*,2) 's% D_DSI(k)', k, s% D_DSI(k)
                  write(*,2) 's% D_SH(k)', k, s% D_SH(k)
                  write(*,2) 's% D_SSI(k)', k, s% D_SSI(k)
                  write(*,2) 's% D_ES(k)', k, s% D_ES(k)
                  write(*,2) 's% D_GSF(k)', k, s% D_GSF(k)
                  write(*,2) 's% D_ST(k)', k, s% D_ST(k)
                  stop 'set_mixing_info'
               end if
            end do
            s% cdc(1) = s% cdc(2)

         end if

         if (dbg) write(*,*) 'call locate_mixing_boundaries'
         call locate_mixing_boundaries(s, eps_h, eps_he, eps_z, ierr)
         if (failed('locate_mixing_boundaries')) return

         call check(2)
                  
         call dealloc
         
         if (dbg) stop 'mixing'

         contains
         
         logical function failed(str)
            character (len=*), intent(in) :: str
            if (ierr == 0) then
               failed = .false.
               return
            end if
            if (s% report_ierr .or. dbg) &
               write(*,*) 'set_mixing_info failed in call to ' // trim(str)
            failed = .true.
            call dealloc
         end function failed
         
         subroutine do_alloc(ierr)
            use alloc
            use utils_lib
            integer, intent(out) :: ierr
            call non_crit_get_work_array(s, eps_h, nz, nz_alloc_extra, 'mix_info', ierr)
            if (ierr /= 0) return            
            call non_crit_get_work_array(s, eps_he, nz, nz_alloc_extra, 'mix_info', ierr)
            if (ierr /= 0) return            
            call non_crit_get_work_array(s, eps_z, nz, nz_alloc_extra, 'mix_info', ierr)
            if (ierr /= 0) return            
            call non_crit_get_work_array(s, cdc_factor, nz, nz_alloc_extra, 'mix_info', ierr)
            if (ierr /= 0) return            
         end subroutine do_alloc
         
         subroutine dealloc
            use alloc
            use utils_lib
            call non_crit_return_work_array(s, eps_h, 'mix_info')            
            call non_crit_return_work_array(s, eps_he, 'mix_info')            
            call non_crit_return_work_array(s, eps_z, 'mix_info')            
            call non_crit_return_work_array(s, cdc_factor, 'mix_info')            
         end subroutine dealloc
         
         subroutine check(i)
            integer, intent(in) :: i
            integer :: k
            include 'formats'
            do k = 1, s% nz
               if (is_bad_num(s% D_mix(k))) then
                  write(*,3) 's% D_mix(k)', i, k, s% D_mix(k)
                  stop 'mix info'
               end if
            end do
         end subroutine check
      
      end subroutine set_mixing_info
      
      
      subroutine set_mlt_boundary_info(s, ierr)
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         
         integer :: k, mt, mt1, mt2, nz
         logical :: ledoux_flag
         real(dp) :: dg0, dg1
         
         include 'formats'

         ! NOTE: this routine must be called BEFORE overshooting is done.
         
         ierr = 0
         nz = s% nz
         s% mixing_type_change_dq(1:nz) = 0
         
         do k = 2, nz
            mt1 = s% mixing_type(k-1)
            mt2 = s% mixing_type(k)
            if (mt1 == mt2) cycle
            if (mt1 > mt2) then
               mt = mt1; mt1 = mt2; mt2 = mt
            end if
            ! mt1 < mt2
            if (mt1 == no_mixing) then
               if (mt2 == convective_mixing .or. mt2 == semiconvective_mixing) then
                  ledoux_flag = .false.
               else if (mt2 == thermo_haline_mixing) then
                  ledoux_flag = .true.
               else
                  write(*,3) 'mt1', k, mt1
                  write(*,3) 'mt2', k, mt2
                  stop 'error in set_mlt_boundary_info 1'
               end if
            else if (mt1 == convective_mixing) then
               if (mt2 == semiconvective_mixing) then
                  ledoux_flag = .true.
               else if (mt2 == thermo_haline_mixing) then
                  ledoux_flag = .false.
               else
                  stop 'error in set_mlt_boundary_info 2'
               end if
            else if (mt1 == semiconvective_mixing) then
               if (mt2 == thermo_haline_mixing) then
                  ledoux_flag = .true.
               else
                  stop 'error in set_mlt_boundary_info 3'
               end if
            else
               stop 'set_mlt_boundary_info'
            end if
            if (ledoux_flag) then ! interp for gradL == gradr
               dg0 = s% gradL(k-1) - s% gradr(k-1)
               dg1 = s% gradL(k) - s% gradr(k)
            else ! interp for grada_at_face == gradr
               dg0 = s% grada_at_face(k-1) - s% gradr(k-1)
               dg1 = s% grada_at_face(k) - s% gradr(k)
            end if
            if (dg0*dg1 >= 0) return
            s% mixing_type_change_dq(k-1) = find0(0d0,dg0,s% dq(k-1),dg1)
            if (.false. .and. s% q(k) < 0.4) write(*,2) 'mixing_type_change_dq', k-1, &
               s% q(k-1) - s% mixing_type_change_dq(k-1)
         end do

      end subroutine set_mlt_boundary_info
      
      
      subroutine set_cz_bdy_mass(s, ierr)
         type (star_info), pointer :: s
         integer, intent(out) :: ierr

         logical :: in_convective_region
         integer :: k, j, nz
         logical, parameter :: dbg = .false.

         include 'formats'
         ierr = 0
         nz = s% nz

         s% n_conv_regions = 0
         in_convective_region = (s% mixing_type(nz) == convective_mixing)
         if (in_convective_region) then
            s% n_conv_regions = 1
            s% cz_bot_mass(1) = s% M_center
         end if
         
         if (dbg) write(*,*) 'initial in_convective_region', in_convective_region
         
         do k=nz-1, 2, -1
            if (in_convective_region) then
               if (s% mixing_type(k) /= convective_mixing) then ! top of convective region
                  s% cz_top_mass(s% n_conv_regions) = &
                     s% M_center + (s% q(k) - s% mixing_type_change_dq(k))*s% xmstar
                  in_convective_region = .false.
               end if
            else
               if (s% mixing_type(k) == convective_mixing) then ! bottom of convective region
                  if (s% n_conv_regions < max_num_mixing_regions) then
                     s% n_conv_regions = s% n_conv_regions + 1
                     s% cz_bot_mass(s% n_conv_regions) = &
                        s% M_center + (s% q(k) - s% mixing_type_change_dq(k))*s% xmstar
                  end if
                  in_convective_region = .true.
               end if
            end if
         end do
         if (in_convective_region) then
            s% cz_top_mass(s% n_conv_regions) = s% mstar
         end if

         if (dbg) then
            write(*,*)
            write(*,2) 'set_mixing_info s% n_conv_regions', s% n_conv_regions
            do j = 1, s% n_conv_regions
               write(*,2) 'conv region', j, s% cz_bot_mass(j)/Msun, s% cz_top_mass(j)/Msun
            end do
            write(*,*)
         end if
         
      end subroutine set_cz_bdy_mass

      
      subroutine locate_convection_boundaries( &
            s, nz, eps_h, eps_he, eps_z, mstar, q, cdc, grada, gradr, ierr)
         !use star_utils, only: scale_height
         type (star_info), pointer :: s
         integer, intent(in) :: nz
         real(dp), dimension(:), intent(in) :: eps_h, eps_he, eps_z
         real(dp), intent(in) :: mstar
         real(dp), pointer, dimension(:) :: q, cdc, grada, gradr
         integer, intent(out) :: ierr
         
         logical :: in_convective_region
         integer :: k, k_bot, i, j, iounit, max_conv_bdy
         real(dp) :: dgrad00, dgradp1, turnover_time, &
            bot_Hp, bot_r, top_Hp, top_r, dr
         
         logical, parameter :: dbg = .false.
         logical, parameter :: write_debug = .false.
         
         include 'formats'

         ierr = 0
         
         if (write_debug) then
            iounit = alloc_iounit(ierr)
            if (ierr /= 0) then
               write(*, *) 'alloc_iounit failed'
               return
            end if
            open(iounit, file=trim('debug.data'), action='write', iostat=ierr)
            if (ierr /= 0) then
               write(*, *) 'open debug.data failed'
               call free_iounit(iounit)
               return
            end if
            write(*,*) 'write debug.data'
            write(iounit,*) 'nz', nz
            write(iounit,1) 'mstar', mstar
            do k=1,nz
               write(iounit,2) 'q', k, q(k)
               write(iounit,2) 'cdc', k, cdc(k)
               write(iounit,2) 'eps_h', k, eps_h(k)
               write(iounit,2) 'eps_he', k, eps_he(k)
               write(iounit,2) 'eps_z', k, eps_z(k)
            end do
         end if

         max_conv_bdy = size(s% conv_bdy_q, dim=1)
         s% conv_bdy_q(:) = 0
         s% conv_bdy_loc(:) = 0
         s% top_conv_bdy(:) = .false.
         s% burn_h_conv_region(:) = .false.
         s% burn_he_conv_region(:) = .false.
         s% burn_z_conv_region(:) = .false.
         bot_Hp = 0; bot_r = 0; top_Hp = 0; top_r = 0; dr = 0
         
         s% num_conv_boundaries = 0
         in_convective_region = (s% mixing_type(nz) == convective_mixing)
         k_bot = nz
         turnover_time = 0
         do k=nz-1, 2, -1
            if (in_convective_region) then
               if (s% mixing_type(k) /= convective_mixing) then
                  call end_of_convective_region
                  if (.false. .and. s% m(k) < Msun) &
                     write(*,'(a80,99f16.5)') 'log(turnover_time/yr), dt/turnover_time, top m', &
                        log10(turnover_time/secyer), s% dt/turnover_time, s% m(k)/Msun
               else
                  turnover_time = turnover_time + (s% rmid(k-1) - s% rmid(k))/s% conv_vel(k)
               end if
            else ! in non-convective region
               if (s% mixing_type(k) == convective_mixing) then ! start of a convective region
                  if (s% num_conv_boundaries == max_conv_bdy) then
                     call realloc(ierr)
                     if (ierr /= 0) then
                        if (write_debug) call free_iounit(iounit)
                        return
                     end if
                  end if
                  s% num_conv_boundaries = s% num_conv_boundaries+1
                  i = s% num_conv_boundaries
                  k_bot = k+1
                  if (k == 1) then
                     s% conv_bdy_q(i) = 1
                  else ! bottom of region is between k+1 and k
                     s% conv_bdy_q(i) = s% q(k) - s% mixing_type_change_dq(k)
                  end if
                  s% top_conv_bdy(i) = .false.
                  s% conv_bdy_loc(i) = k_bot 
                     ! bottom of region is between k_bot and k_bot-1
                  in_convective_region = .true.
                  bot_r = s% r(k_bot)
                  !bot_Hp = scale_height(s,k_bot,.true.)
                  !write(*,2) 'bot_Hp', k_bot, bot_Hp, s% scale_height(k_bot)
                  bot_Hp = s% scale_height(k_bot)
                  !stop 'mix_info'
                  turnover_time = 0
               end if
            end if
         end do
         
         if (in_convective_region) then
            k = 1 ! end at top
            call end_of_convective_region
         end if

         if (write_debug) then
            write(iounit,*) 's% num_conv_boundaries', s% num_conv_boundaries
            do j=1,s% num_conv_boundaries
               write(iounit,2) 's% conv_bdy_q', j, s% conv_bdy_q(j)
               write(iounit,*) 's% top_conv_bdy', j, s% top_conv_bdy(j)
               write(iounit,*) 's% burn_h_conv_region', j, s% burn_h_conv_region(j)
               write(iounit,*) 's% burn_he_conv_region', j, s% burn_he_conv_region(j)
               write(iounit,*) 's% burn_z_conv_region', j, s% burn_z_conv_region(j)
               write(iounit,*) 's% conv_bdy_loc', j, s% conv_bdy_loc(j)
            end do
            close(iounit)
            call free_iounit(iounit)
         end if
         
         if (dbg) then
            write(*,*) 's% num_conv_boundaries', s% num_conv_boundaries
            do j=1,s% num_conv_boundaries
               write(*,*) 's% conv_bdy_q', j, s% conv_bdy_q(j)
               write(*,*) 's% top_conv_bdy', j, s% top_conv_bdy(j)
               write(*,*) 's% burn_h_conv_region', j, s% burn_h_conv_region(j)
               write(*,*) 's% burn_he_conv_region', j, s% burn_he_conv_region(j)
               write(*,*) 's% burn_z_conv_region', j, s% burn_z_conv_region(j)
               write(*,*) 's% conv_bdy_loc', j, s% conv_bdy_loc(j)
               write(*,*) 'mixing type', s% mixing_type(s% conv_bdy_loc(j)-3:s% conv_bdy_loc(j)+3)
            end do
            write(*,*)
            !stop 'locate_convection_boundaries'
         end if

         
         contains         
         
         
         subroutine end_of_convective_region()
            integer :: max_eps_loc, kk, op_err
            real(dp) :: max_eps, eps, Hp
            logical :: end_dbg
            
            9 format(a40, 3i7, 99(1pe26.16))
            include 'formats'
            
            in_convective_region = .false.
            
            end_dbg = .false.
            
            top_r = s% r(k)
            top_Hp = s% scale_height(k)
            dr = top_r - bot_r
            Hp = (bot_Hp + top_Hp)/2
            if (end_dbg) then
               write(*,2) 'end_of_convective_region: k', k
               write(*,2) 'end_of_convective_region: k_bot', k_bot
               write(*,2) 'nz', nz
               write(*,*) 'dr/Hp < s% prune_bad_cz_min_Hp_height', &
                  dr/Hp < s% prune_bad_cz_min_Hp_height
            end if
            
            if (dr/Hp < s% prune_bad_cz_min_Hp_height .and. s% prune_bad_cz_min_Hp_height > 0) then
               if (end_dbg) write(*,2) 'dr/Hp < s% prune_bad_cz_min_Hp_height', &
                  k, dr/Hp, s% prune_bad_cz_min_Hp_height
               max_eps = maxval(eps_h(k:k_bot) + eps_he(k:k_bot) + eps_z(k:k_bot))
               if (end_dbg) write(*,3) 'max_eps', k, k_bot, max_eps, &
                  10**s% prune_bad_cz_min_log_eps_nuc
               if (max_eps < 10**s% prune_bad_cz_min_log_eps_nuc &
                     .and. all(s% mixing_type(k+1:k_bot-1) /= thermo_haline_mixing)) then
                  if (end_dbg) then
                     write(*,'(a)') 'end_of_convective_region: remove convection zone because too small'
                     write(*,2) 'dr/Hp', k, dr/Hp
                     write(*,2) 'prune_bad_cz_min_Hp_height', k, s% prune_bad_cz_min_Hp_height
                     write(*,2) 'dr/Rsun', k, dr/Rsun
                     write(*,2) 'top_r/Rsun', k, top_r/Rsun
                     write(*,2) 'bot_r/Rsun', k, bot_r/Rsun
                     write(*,2) 'Hp/Rsun', k, Hp/Rsun
                     write(*,2) 'top_Hp/Rsun', k, top_Hp/Rsun
                     write(*,2) 'bot_Hp/Rsun', k, bot_Hp/Rsun
                     write(*,*)
                     !write(*,3) 'use_gradr_for_gradT: dr/Hp', k, k_bot, dr/Hp
                     write(*,*)
                     stop 'mix'
                  end if
                  do kk = k, k_bot ! this includes the radiative points at boundaries
                     call set_use_gradr(s,kk)
                     s% cdc(kk) = 0
                     s% D_mix(kk) = 0
                     s% conv_vel(kk) = 0
                     s% mixing_type(kk) = no_mixing
                  end do
                  if (s% num_conv_boundaries > 0) &
                     s% num_conv_boundaries = s% num_conv_boundaries-1
                  !stop
                  
                  
                  return
               end if
            end if
            
            !write(*,3) 'redo mlt', k, k_bot, dr, s% mixing_length_alpha*min(top_Hp, bot_Hp)
            if (dr < s% mixing_length_alpha*min(top_Hp, bot_Hp)) then
               !write(*,*) 'dr is small -- redo mlt'
!$OMP PARALLEL DO PRIVATE(kk,op_err)
               do kk = k, k_bot
                  op_err = 0
                  call redo1_mlt(s,kk,dr,op_err)
                  if (op_err /= 0) ierr = op_err
               end do
!$OMP END PARALLEL DO
            else
               !write(*,*) 'dr is okay'
            end if
            
            if (s% num_conv_boundaries == max_conv_bdy) then
               call realloc(ierr)
               if (ierr /= 0) return
            end if            
            
            s% num_conv_boundaries = s% num_conv_boundaries+1
            i = s% num_conv_boundaries
            
            end_dbg = .false. !dbg ! .or. (i == 1) .or. (i == 12)

            ! check for burning in region
            max_eps = -1d99
            max_eps_loc = 0
            if (end_dbg) then
               write(*,*) 'search for max eps', k, k_bot, nz
            end if
            do kk=k,min(nz,k_bot+1)
               eps = eps_h(kk) + eps_he(kk) + eps_z(kk)
               !if (end_dbg) write(*,2) 'eps', kk, eps
               if (eps > max_eps) then
                  max_eps = eps
                  max_eps_loc = kk
               end if
            end do
            if (end_dbg) write(*,2) 'max eps', max_eps_loc, max_eps
            if (max_eps > 1d0) then
               if (eps_h(max_eps_loc) > 0.5d0*max_eps) then
                  s% burn_h_conv_region(i) = .true.
                  if (i > 1) s% burn_h_conv_region(i-1) = .true.
               else if (eps_he(max_eps_loc) > 0.5d0*max_eps) then
                  s% burn_he_conv_region(i) = .true.
                  if (i > 1) s% burn_he_conv_region(i-1) = .true.
               else
                  s% burn_z_conv_region(i) = .true.
                  if (i > 1) s% burn_z_conv_region(i-1) = .true.
               end if
            end if
            
            if (k == 1) then
               s% conv_bdy_q(i) = 1
            else 
               ! top of region is between k+1 and k
               s% conv_bdy_q(i) = s% q(k) - s% mixing_type_change_dq(k)
               !if (s% conv_bdy_q(i) < 0.4) then
               !   write(*,2) 's% conv_bdy_q(i)', i, s% conv_bdy_q(i)
               !end if
            end if
            s% top_conv_bdy(i) = .true.
            s% conv_bdy_loc(i) = k
            if (end_dbg) write(*,3) 'end_of_convective_region', i, k, s% cdc(k), s% cdc(k+1)
            
            !if (end_dbg) stop 'debug: end_of_convective_region'
            
         end subroutine end_of_convective_region
         
         
         subroutine redo1_mlt(s,k,dr,ierr)
            use mlt_info, only: do1_mlt
            type (star_info), pointer :: s         
            integer, intent(in) :: k
            real(dp), intent(in) :: dr
            integer, intent(out) :: ierr
            real(dp) :: Hp, reduced_alpha
            include 'formats'
            ierr = 0
            Hp = s% scale_height(k)
            if (dr >= s% mixing_length_alpha*Hp) return
            ! convection zone is smaller than mixing length
            if (s% remove_mixing_glitches .and. dr < 0.1d0*s% mixing_length_alpha*Hp) then ! just erase it
               s% cdc(k) = 0
               s% D_mix(k) = 0
               s% conv_vel(k) = 0
               s% mixing_type(k) = no_mixing
               return
            end if
            ! redo it with reduced alpha to keep mixing length small enough
            reduced_alpha = min(1d0,dr/Hp)*0.5*dr/Hp ! mixing_length <= 0.5*dr
            !write(*,2) 'reduced_alpha', k, reduced_alpha, s% mixing_length_alpha
            !write(*,2) 'cdc before redo1_mlt', k, s% cdc(k), reduced_alpha
            call do1_mlt(s,k,reduced_alpha,ierr)
            s% cdc(k) = s% mlt_cdc(k)
            !write(*,2) 'cdc after', k, s% cdc(k)
            s% D_mix(k) = s% mlt_D(k)
            s% conv_vel(k) = s% mlt_vc(k)
            s% mixing_type(k) = s% mlt_mixing_type(k)
         end subroutine redo1_mlt
         
      
         subroutine realloc(ierr)
            use utils_lib
            integer, intent(out) :: ierr
            
            integer :: sz
            
            sz = size(s% conv_bdy_q, dim=1)
            
            ierr = 0
            max_conv_bdy = 2*(10+max_conv_bdy)
            
            call realloc_double(s% conv_bdy_q,max_conv_bdy,ierr)
            if (ierr /= 0) return
            
            call realloc_integer(s% conv_bdy_loc,max_conv_bdy,ierr)
            if (ierr /= 0) return
            
            call realloc_logical(s% top_conv_bdy,max_conv_bdy,ierr)
            if (ierr /= 0) return
            
            call realloc_logical(s% burn_h_conv_region,max_conv_bdy,ierr)
            if (ierr /= 0) return
            
            call realloc_logical(s% burn_he_conv_region,max_conv_bdy,ierr)
            if (ierr /= 0) return
            
            call realloc_logical(s% burn_z_conv_region,max_conv_bdy,ierr)
            if (ierr /= 0) return

            s% conv_bdy_q(sz+1:max_conv_bdy) = 0
            s% conv_bdy_loc(sz+1:max_conv_bdy) = 0
            s% top_conv_bdy(sz+1:max_conv_bdy) = .false.
            s% burn_h_conv_region(sz+1:max_conv_bdy) = .false.
            s% burn_he_conv_region(sz+1:max_conv_bdy) = .false.
            s% burn_z_conv_region(sz+1:max_conv_bdy) = .false.
         
         end subroutine realloc
                
      end subroutine locate_convection_boundaries
      

      subroutine set_use_gradr(s,k)
         use mlt_info
         type (star_info), pointer :: s
         integer, intent(in) :: k
         call switch_to_no_mixing(s,k)
         call switch_to_radiative(s,k)
         !s% use_gradr_for_gradT(k) = .true.
      end subroutine set_use_gradr      
      
     
      subroutine locate_mixing_boundaries(s, eps_h, eps_he, eps_z, ierr)
         type (star_info), pointer :: s
         real(dp), dimension(:), intent(in) :: eps_h, eps_he, eps_z
         integer, intent(out) :: ierr
         
         logical :: in_mixing_region
         integer :: k, k_bot, i, j, iounit, max_mix_bdy, nz
         
         logical, parameter :: dbg = .false.
         
         include 'formats'

         ierr = 0
         nz = s% nz
         
         max_mix_bdy = size(s% mix_bdy_q, dim=1)
         s% mix_bdy_q(:) = 0
         s% mix_bdy_loc(:) = 0
         s% top_mix_bdy(:) = .false.
         s% burn_h_mix_region(:) = .false.
         s% burn_he_mix_region(:) = .false.
         s% burn_z_mix_region(:) = .false.
         
         s% num_mix_boundaries = 0
         s% num_mix_regions = 0
         in_mixing_region = (s% mixing_type(nz) /= no_mixing)
         k_bot = nz
         do k=nz-1, 2, -1
            if (in_mixing_region) then
               if (s% mixing_type(k) == no_mixing) call end_of_mixing_region
            else ! in non-mixing region
               if (s% mixing_type(k) /= no_mixing) then ! start of a mixing region
                  if (s% num_mix_boundaries == max_mix_bdy) then
                     call realloc(ierr)
                     if (ierr /= 0) return
                  end if
                  s% num_mix_boundaries = s% num_mix_boundaries+1
                  i = s% num_mix_boundaries
                  k_bot = k+1
                  if (k == 1) then
                     s% mix_bdy_q(i) = 1
                  else ! bottom of region is between k+1 and k
                     s% mix_bdy_q(i) = s% q(k) - s% mixing_type_change_dq(k)
                  end if
                  s% top_mix_bdy(i) = .false.
                  s% mix_bdy_loc(i) = k_bot
                  in_mixing_region = .true.
               end if
            end if
         end do
         
         if (in_mixing_region) then
            k = 1 ! end at top
            call end_of_mixing_region
         end if

         
         contains         
         
         
         subroutine end_of_mixing_region()
            integer :: max_eps_loc, kk
            real(dp) :: max_eps, eps
            
            9 format(a40, 3i7, 99(1pe26.16))
            include 'formats'
            
            in_mixing_region = .false.
            
            if (s% num_mix_boundaries == max_mix_bdy) then
               call realloc(ierr)
               if (ierr /= 0) return
            end if            
            
            s% num_mix_regions = s% num_mix_regions+1
            s% num_mix_boundaries = s% num_mix_boundaries+1
            i = s% num_mix_boundaries

            ! check for burning in region
            max_eps = -1d99
            max_eps_loc = 0
            do kk=k,min(nz,k_bot+1)
               eps = eps_h(kk) + eps_he(kk) + eps_z(kk)
               if (eps > max_eps) then
                  max_eps = eps
                  max_eps_loc = kk
               end if
            end do
            if (max_eps > 1d0) then
               if (eps_h(max_eps_loc) > 0.5d0*max_eps) then
                  s% burn_h_mix_region(i) = .true.
                  if (i > 1) s% burn_h_mix_region(i-1) = .true.
               else if (eps_he(max_eps_loc) > 0.5d0*max_eps) then
                  s% burn_he_mix_region(i) = .true.
                  if (i > 1) s% burn_he_mix_region(i-1) = .true.
               else
                  s% burn_z_mix_region(i) = .true.
                  if (i > 1) s% burn_z_mix_region(i-1) = .true.
               end if
            end if
            
            if (k == 1) then
               s% mix_bdy_q(i) = 1
            else 
               ! top of region is between k+1 and k
               s% mix_bdy_q(i) = s% q(k) - s% mixing_type_change_dq(k)
            end if
            s% top_mix_bdy(i) = .true.
            s% mix_bdy_loc(i) = k
            
         end subroutine end_of_mixing_region
         
      
         subroutine realloc(ierr)
            use utils_lib
            integer, intent(out) :: ierr
            
            integer :: sz
            
            sz = size(s% mix_bdy_q, dim=1)
            
            ierr = 0
            max_mix_bdy = 2*(10+max_mix_bdy)
            
            call realloc_double(s% mix_bdy_q,max_mix_bdy,ierr)
            if (ierr /= 0) return
            
            call realloc_integer(s% mix_bdy_loc,max_mix_bdy,ierr)
            if (ierr /= 0) return
            
            call realloc_logical(s% top_mix_bdy,max_mix_bdy,ierr)
            if (ierr /= 0) return
            
            call realloc_logical(s% burn_h_mix_region,max_mix_bdy,ierr)
            if (ierr /= 0) return
            
            call realloc_logical(s% burn_he_mix_region,max_mix_bdy,ierr)
            if (ierr /= 0) return
            
            call realloc_logical(s% burn_z_mix_region,max_mix_bdy,ierr)
            if (ierr /= 0) return

            s% mix_bdy_q(sz+1:max_mix_bdy) = 0
            s% mix_bdy_loc(sz+1:max_mix_bdy) = 0
            s% top_mix_bdy(sz+1:max_mix_bdy) = .false.
            s% burn_h_mix_region(sz+1:max_mix_bdy) = .false.
            s% burn_he_mix_region(sz+1:max_mix_bdy) = .false.
            s% burn_z_mix_region(sz+1:max_mix_bdy) = .false.
         
         end subroutine realloc
         
                
      end subroutine locate_mixing_boundaries

      
      
      subroutine remove_tiny_mixing(s, ierr)
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         
         integer :: k, nz
         logical, parameter :: dbg = .false.
         real(dp) :: tiny
         
         include 'formats'
         
         if (dbg) write(*,*) 'remove_tiny_mixing'
         
         ierr = 0
         nz = s% nz
         
         tiny = s% clip_D_limit
         do k=1,nz
            if (s% D_mix(k) < tiny) then
               s% cdc(k) = 0
               s% D_mix(k) = 0
               s% conv_vel(k) = 0
               s% mixing_type(k) = no_mixing
            end if
         end do
      
      end subroutine remove_tiny_mixing
      
      
      ! remove single point mixing or non-mixing regions
      ! NOTE: does not remove thermohaline singletons
      subroutine remove_mixing_singletons(s, ierr)
         !use star_utils, only: scale_height
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         
         integer :: k, nz
         logical, parameter :: dbg = .false.
         real(dp) :: lambda
         
         include 'formats'
         
         if (dbg) write(*,*) 'remove_mixing_singletons'
         
         ierr = 0
         nz = s% nz
         
         do k=2,nz-1
            if (s% cdc(k) == 0) then
               if (s% cdc(k-1) /= 0 .and. s% cdc(k+1) /= 0) then
                  s% cdc(k) = (s% cdc(k-1) + s% cdc(k+1))/2
                  s% D_mix(k) = s% cdc(k)/(4*pi*s% r(k)**2*s% rho(k))**2
                  !lambda = s% mixing_length_alpha*(scale_height(s,k-1) + scale_height(s,k+1))/2
                  lambda = s% mlt_mixing_length(k) 
                  s% conv_vel(k) = 3*s% D_mix(k)/lambda
                  s% mixing_type(k) = max(s% mixing_type(k-1), s% mixing_type(k+1))
               if (dbg) write(*,3) 'remove radiative singleton', k, nz
               end if
            else if (s% okay_to_remove_mixing_singleton) then 
               if (s% cdc(k-1) == 0 .and. s% cdc(k+1) == 0) then
                  !if (s% mixing_type(k) /= thermo_haline_mixing) then
                     call set_use_gradr(s,k)
                     s% cdc(k) = 0
                     s% D_mix(k) = 0
                     s% conv_vel(k) = 0
                     s% mixing_type(k) = no_mixing
                     if (dbg) write(*,3) 'remove mixing singleton', k, nz
                  !end if
               end if
            end if
         end do
         
         if (s% cdc(1) == 0) then
            if (s% cdc(2) /= 0) then
               s% cdc(1) = s% cdc(2)
               s% D_mix(1) = s% D_mix(2)
               s% conv_vel(1) = s% conv_vel(2)
               s% mixing_type(1) = s% mixing_type(2)
               if (dbg) write(*,3) 'remove radiative singleton', 1, nz
            end if
         else
            if (s% cdc(2) == 0) then
               !if (s% mixing_type(1) /= thermo_haline_mixing) then
                  call set_use_gradr(s,1)
                  s% cdc(1) = 0
                  s% D_mix(1) = 0
                  s% conv_vel(1) = 0
                  s% mixing_type(1) = no_mixing
                  if (dbg) write(*,2) 'remove mixing singleton', 1
               !end if
            end if
         end if
         
         if (s% cdc(nz) == 0) then
            if (s% cdc(nz-1) /= 0) then
               s% cdc(nz) = s% cdc(nz-1)
               s% D_mix(nz) = s% D_mix(nz-1)
               s% conv_vel(nz) = s% conv_vel(nz-1)
               s% mixing_type(nz) = s% mixing_type(nz-1)
               if (dbg) write(*,2) 'remove radiative singleton: s% cdc(nz-1)', nz, s% cdc(nz-1)
            end if
         else
            if (s% cdc(nz-1) == 0) then
               !if (s% mixing_type(nz) /= thermo_haline_mixing) then
                  call set_use_gradr(s,nz)
                  s% cdc(nz) = 0
                  s% D_mix(nz) = 0
                  s% conv_vel(nz) = 0
                  s% mixing_type(nz) = no_mixing
                  if (dbg) write(*,2) 'remove mixing singleton: s% cdc(nz)', nz, s% cdc(nz)
               !end if
            end if
         end if
      
      end subroutine remove_mixing_singletons
      
      
      subroutine close_convection_gaps(s, ierr)
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         call close_gaps(s, convective_mixing, s% min_convective_gap, ierr)
      end subroutine close_convection_gaps
      
      
      subroutine close_thermo_haline_gaps(s, ierr)
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         call close_gaps(s, thermo_haline_mixing, s% min_thermo_haline_gap, ierr)
      end subroutine close_thermo_haline_gaps
      
      
      subroutine close_semiconvection_gaps(s, ierr)
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         call close_gaps(s, semiconvective_mixing, s% min_semiconvection_gap, ierr)
      end subroutine close_semiconvection_gaps
      
      
      subroutine close_gaps(s, mix_type, min_gap, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: mix_type
         real(dp), intent(in) :: min_gap
         integer, intent(out) :: ierr
         
         integer :: k, nz
         logical :: in_region
         real(dp) :: rtop, rbot, Hp
         integer :: ktop, kbot ! k's for gap         
         logical, parameter :: dbg = .false.        
         include 'formats'         
         ierr = 0         
         if (min_gap < 0) return         
         nz = s% nz        
         in_region = (s% mixing_type(nz) == mix_type)
         rbot = 0
         kbot = nz
         do k=nz-1, 2, -1
            if (in_region) then
               if (s% mixing_type(k) /= mix_type) then ! end of region
                  kbot = k+1
                  rbot = s% r(kbot)
                  in_region = .false.
               end if
            else
               if (s% mixing_type(k) == mix_type) then ! start of region
                  ktop = k
                  rtop = s% r(ktop)
                  Hp = s% P(ktop)/(s% rho(ktop)*s% grav(ktop))
                  if (rtop - rbot < Hp*min_gap) then
                     if (kbot < nz) then
                        s% cdc(ktop+1:kbot-1) = (s% cdc(ktop) + s% cdc(kbot))/2
                        s% D_mix(ktop+1:kbot-1) = &
                           (s% D_mix(ktop) + s% D_mix(kbot))/2
                        s% conv_vel(ktop+1:kbot-1) = (s% conv_vel(ktop) + s% conv_vel(kbot))/2
                        s% mixing_type(ktop+1:kbot-1) = mix_type
                        if (dbg) write(*,3) 'close mixing gap', &
                              ktop+1, kbot-1, (rtop - rbot)/Hp, rtop - rbot, Hp
                     else
                        s% cdc(ktop+1:kbot) = s% cdc(ktop)
                        s% D_mix(ktop+1:kbot) = s% D_mix(ktop)
                        s% conv_vel(ktop+1:kbot) = s% conv_vel(ktop)
                        s% mixing_type(ktop+1:kbot) = mix_type
                        if (dbg) write(*,3) 'close mixing gap', &
                           ktop+1, kbot, (rtop - rbot)/Hp, rtop - rbot, Hp
                     end if
                  end if
                  in_region = .true.
               end if
            end if
         end do
         
      end subroutine close_gaps
      
      
      ! if find radiative region embedded in thermohaline,
      ! and max(gradL - grada) in region is < 1d-3
      ! and region height is < min_thermo_haline_dropout
      ! then convert the region to thermohaline
      subroutine remove_thermohaline_dropouts(s, ierr)
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         
         integer :: k, nz, j
         logical :: in_region
         real(dp) :: rtop, rbot, Hp, q_upper, q_lower, alfa, beta
         integer :: ktop, kbot ! k's for gap
         logical :: all_small 
         logical, parameter :: dbg = .false.        
         include 'formats'         
         ierr = 0         
         nz = s% nz        
         rbot = s% r(nz)
         kbot = nz-1
         in_region = (s% mixing_type(kbot) == no_mixing)
         all_small = .false.
         do k=nz-2, 2, -1
            if (in_region) then
               if (s% mixing_type(k) == no_mixing) then ! check if okay
                  if (s% gradL(k) - s% grada_at_face(k) > s% max_dropout_gradL_sub_grada) &
                     all_small = .false.
               else ! end of radiative region
                  ktop = k+1
                  rtop = s% r(ktop)
                  Hp = s% P(ktop)/(s% rho(ktop)*s% grav(ktop))
                  q_upper = s% q(ktop-1)
                  q_lower = s% q(kbot+1)
                  !write(*,3) 'ktop kbot', ktop, kbot
                  if (.false. .and. ktop > 1100 .and. kbot < 1400) &
                     write(*,3) '(rtop - rbot)/Hp', ktop, kbot, (rtop - rbot)/Hp
                  if (rtop - rbot < Hp*s% min_thermo_haline_dropout .and. &
                      s% mixing_type(ktop-1) == thermo_haline_mixing .and. &
                      s% mixing_type(kbot+1) == thermo_haline_mixing .and. &
                      q_upper - q_lower > 1d-20 .and. all_small) then
                     do j = ktop, kbot ! interpolate in q
                        alfa = (s% q(j) - q_lower)/(q_upper - q_lower)
                        beta = 1 - alfa
                        s% cdc(j) = alfa*s% cdc(ktop-1) + beta*s% cdc(kbot+1)
                        s% D_mix(j) = alfa*s% D_mix(ktop-1) + beta*s% D_mix(kbot+1)
                        s% conv_vel(j) = alfa*s% conv_vel(ktop-1) + beta*s% conv_vel(kbot+1)
                        s% mixing_type(j) = thermo_haline_mixing
                     end do
                  end if
                  in_region = .false.
               end if
            else
               if (s% mixing_type(k) == no_mixing) then ! start of region
                  kbot = k
                  rbot = s% r(kbot)
                  in_region = .true.
                  all_small = (s% gradL(k) - s% grada_at_face(k) <= s% max_dropout_gradL_sub_grada)
               end if
            end if
         end do
         
      end subroutine remove_thermohaline_dropouts
      
      
      subroutine remove_embedded_semiconvection(s, ierr)
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         
         integer :: k, nz
         logical :: in_region
         integer :: kbot, ktop
         
         logical, parameter :: dbg = .false.
         
         include 'formats'
         
         ierr = 0
         if (.not. s% remove_embedded_semiconvection) return
         
         nz = s% nz
         
         in_region = check(nz)
         kbot = nz
         do k=nz-1, 2, -1
            if (in_region) then
               if (.not. check(k)) then ! end of region
                  ktop = k+1
                  in_region = .false.
                  call clean_region
               end if
            else ! not in region
               if (check(k)) then ! start of region
                  kbot = k
                  in_region = .true.
               end if
            end if
         end do
         
         if (in_region) then
            ktop = 1
            call clean_region
         end if
         
         
         contains
         
         
         subroutine clean_region
            integer :: kbot1, ktop1
            include 'formats'
            if (dbg) write(*,3) 'clean_region semiconvective', kbot, ktop
            ! move top to below top convective region
            do while (s% mixing_type(ktop) == convective_mixing)
               ktop = ktop + 1
               if (ktop >= kbot) return
            end do
            if (dbg) write(*,2) 'new ktop 1', ktop
            ! move top to below top semiconvective region
            do while (s% mixing_type(ktop) == semiconvective_mixing)
               ktop = ktop + 1
               if (ktop >= kbot) return
            end do
            if (dbg) write(*,2) 'new ktop 2', ktop
            ! move bot to above bottom convective region
            do while (s% mixing_type(kbot) == convective_mixing)
               kbot = kbot - 1
               if (ktop >= kbot) return
            end do
            if (dbg) write(*,2) 'new kbot 1', kbot
            ! move bot to above bottom semiconvective region
            do while (s% mixing_type(kbot) == semiconvective_mixing)
               kbot = kbot - 1
               if (ktop >= kbot) return
            end do
            if (dbg) write(*,2) 'new kbot 2', kbot
            ! convert any semiconvective region between kbot and ktop
            kbot1 = kbot
            do while (kbot1 > ktop)
               ! move kbot1 to bottom of next semiconvective region
               do while (s% mixing_type(kbot1) == convective_mixing)
                  kbot1 = kbot1 - 1
                  if (kbot1 <= ktop) return
               end do
               ktop1 = kbot1
               ! move ktop1 to top of semiconvective region
               do while (s% mixing_type(ktop1) == semiconvective_mixing)
                  ktop1 = ktop1 - 1
                  if (ktop1 <= ktop) return
               end do
               s% D_mix(ktop1+1:kbot1) = s% D_mix(ktop1)
               s% conv_vel(ktop1+1:kbot1) = s% conv_vel(ktop1)
               s% mixing_type(ktop1+1:kbot1) = convective_mixing
               if (dbg) write(*,3) 'merge semiconvective island', kbot1, ktop1+1
               kbot1 = ktop1
            end do
         end subroutine clean_region
         
         
         logical function check(k)
            integer, intent(in) :: k
            check = &
               (s% mixing_type(k) == semiconvective_mixing) .or. &
               (s% mixing_type(k) == convective_mixing)
         end function check
         
      end subroutine remove_embedded_semiconvection

      
      subroutine do_mix_envelope(s)
         type (star_info), pointer :: s
         real(dp) :: T_mix_limit         
         integer :: j, k, i, nz
         
         include 'formats'
         
         T_mix_limit = s% T_mix_limit
         !write(*,1) 'T_mix_limit', T_mix_limit
         if (T_mix_limit <= 0) return
         nz = s% nz
         j = 0
         do k = 1, nz ! search inward until find T >= T_mix_limit
            if (s% T(k) >= T_mix_limit) then
               j = k; exit
            end if
         end do
         if (j==0) j=nz ! all T < T_mix_limit
         ! find base of innermost convection that has T < T_mix_limit
         i = 0
         do k = j, 1, -1
            if (s% mixing_type(k) == convective_mixing) then
               i = k; exit
            end if
         end do
         if (i == 0) then
            !write(*,*) 'no convection in region with T < T_mix_limit at point ', j
            return ! no convection in region with T < T_mix_limit
         end if
         ! extend convection region to surface
         j = maxloc(s% cdc(1:i), dim=1)
         s% conv_vel(1:i) = s% conv_vel(j)
         s% cdc(1:i) = s% cdc(j)
         s% D_mix(1:i) = s% D_mix(j)
         s% mixing_type(1:i) = convective_mixing
         
         !write(*,*) 'extend convection to surface from point ', i
         !return

         
      end subroutine do_mix_envelope


      subroutine get_convection_sigmas(s, dt, ierr)
         use chem_def, only: chem_isos
         type (star_info), pointer :: s         
         real(dp), intent(in) :: dt
         integer, intent(out) :: ierr
         
         integer :: nz, k, j, species
         real(dp) :: sig_term_limit ! sig_term_limit is used to help convergence
                     
         real(dp) :: siglim, xmstar, dq00, dqm1, cdcterm, dmavg, rho_face, &
            cdc, max_sig, D, xm1, x00, xp1, dm, dX, X, cushion, limit, &
            Tc, full_off, full_on, alfa
         logical :: in_convective_region
         real(dp), dimension(:), pointer :: sig

         include 'formats'
         
         sig_term_limit = s% sig_term_limit

         ierr = 0
         xmstar = s% xmstar
         sig => s% sig
         nz = s% nz
         
         sig(1) = 0d0
         do k = 2, nz
            D = s% D_mix(k)
            if (D == 0) then
               sig(k) = 0d0
               cycle
            end if
            rho_face = (s% dq(k-1)*s% rho(k) + s% dq(k)*s% rho(k-1))/(s% dq(k-1) + s% dq(k))
            cdc = D*(4*pi*s% r(k)**2*rho_face)**2
            cdcterm = s% mix_factor*cdc
            dq00 = s% dq(k)
            dqm1 = s% dq(k-1)
            dmavg = xmstar*(dq00+dqm1)/2
            sig(k) = cdcterm/dmavg
            if (is_bad_num(sig(k))) sig(k) = 1d99
            if (.false. .and. is_bad_num(sig(k))) then
               ierr = -1
               return
               write(*,2) 'sig(k)', k, sig(k)
               write(*,2) 'rho_face', k, rho_face
               write(*,2) 'cdc', k, cdc
               write(*,2) 'cdcterm', k, cdcterm
               write(*,2) 'dq00', k, dq00
               write(*,2) 'dqm1', k, dqm1
               write(*,2) 's% D_mix(k)', k, s% D_mix(k)
               write(*,2) 's% cdc(k)', k, s% cdc(k)
               write(*,2) 's% mlt_cdc(k)', k, s% mlt_cdc(k)
               stop 'debug: get_convection_sigmas'
            end if
         end do

         ! can get numerical problems unless limit sig
         max_sig = maxval(sig)
         if (max_sig < 1) return
         do k = 2, nz
            if (k > 1) then
               siglim = sig_term_limit*xmstar*min(s% dq(k),s% dq(k-1))/dt
            else
               siglim = sig_term_limit*xmstar*s% dq(k)/dt
            end if
            s% sig_div_siglim(k) = sig(k)/max(1d-99,siglim)
!            if (k == -1) then
!               write(*,2) 'sig(k)', k, sig(k)
!               write(*,2) 'siglim', k, siglim
!            end if
            if (sig(k) > siglim) then
               sig(k) = siglim
            end if
         end do
         
         species = s% species
         ! limit sigma to avoid negative mass fractions
         cushion = 10d0
         Tc = s% T(s% nz)
         full_off = s% Tcenter_max_for_sig_min_factor_full_off
         if (Tc <= full_off) return
         full_on = s% Tcenter_min_for_sig_min_factor_full_on
         limit = s% sig_min_factor_for_high_Tcenter
         if (Tc < full_on) then
            alfa = (Tc - full_off)/(full_on - full_off)
            ! limit is full on for alfa = 1 and limit is 1 for alfa = 0
            limit = limit*alfa + (1d0 - alfa)
         end if
         if (limit >= 1d0) return
         do k=2,nz
            siglim = sig(k)
            do j=1,species
               xm1 = s% xa(j,k-1)
               x00 = s% xa(j,k)
               if (xm1 > x00) then
                  X = xm1
                  dX = xm1 - x00
                  dm = s% dm(k-1)
               else
                  X = x00
                  dX = x00 - xm1
                  dm = s% dm(k)
                  dX = -dX
               end if
               if (k == -1639 .and.j == 23) then
                  write(*,1) 'cushion*dt*dX*siglim', cushion*dt*dX*siglim
                  write(*,1) 'cushion', cushion
                  write(*,1) 'dt', dt
                  write(*,1) 'dX', dX
                  write(*,1) 'siglim', siglim
                  write(*,1) 'cushion*dt*dX*siglim', cushion*dt*dX*siglim
                  write(*,1) 'dm*X', dm*X
                  write(*,1) 'X', X
                  write(*,1) 'dm', dm
                  write(*,*) 'cushion*dt*dX*siglim > dm*X', cushion*dt*dX*siglim > dm*X
               end if
               if (cushion*dt*dX*siglim > dm*X) then
                  if (k == -1639 .and.j == 23) write(*,2) 'limit sig for x < 0 ' // &
                     trim(chem_isos% name(s% chem_id(j))), k, dm*X/(dt*dX*cushion)/siglim
                     
                  siglim = dm*X/(dt*dX*cushion)
               end if
            end do
            sig(k) = max(limit*sig(k), siglim)
         end do

      end subroutine get_convection_sigmas
      
      
      subroutine update_rotation_mixing_info(s, ierr)
         use rotation_mix_info, only: set_rotation_mixing_info         
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         
         integer :: k, nz, k0
         
         include 'formats'
         
         ierr = 0
         nz = s% nz
         
         call set_rotation_mixing_info(s, ierr)
         if (ierr /= 0) return
      
         ! include rotation part for mixing abundances
         do k = 2, nz
            s% D_mix(k) = &
               max(s% D_mix_non_rotation(k), &
                   s% am_D_mix_factor*max(s% D_visc(k), &
                     s% D_DSI(k) + &
                     s% D_SH(k) + &
                     s% D_SSI(k) + &
                     s% D_ES(k) + &
                     s% D_GSF(k) + &
                     s% D_ST(k) &
                     ))
         end do

         if (s% set_uniform_am_nu) then ! set large am_nu to get uniform omega
            s% am_nu(1:nz) = s% uniform_am_nu
         else
   			do k=1,nz
               s% am_nu(k) = s% am_nu_factor*max(s% D_visc(k), &
                  s% am_nu_non_rotation_factor*s% D_mix_non_rotation(k) + &
                  s% am_nu_DSI_factor*s% D_DSI(k) + &
                  s% am_nu_SH_factor*s% D_SH(k) + &
                  s% am_nu_SSI_factor*s% D_SSI(k) + &
                  s% am_nu_ES_factor*s% D_ES(k) + &
                  s% am_nu_GSF_factor*s% D_GSF(k) + &
                  s% am_nu_ST_factor*s% nu_ST(k))
   				if (s% am_nu(k) < 0) then
   				   ierr = -1
   				   return
   				end if
   			end do
         end if
      
         if (s% use_other_am_mixing) then
            call s% other_am_mixing(s% id, ierr)
            if (ierr /= 0) return
         end if
      
         if (s% set_min_am_nu .and. s% ye(nz) >= s% min_center_Ye_for_min_am_nu) &
            s% am_nu(1:nz) = max(s% min_am_nu, s% am_nu(1:nz))

      end subroutine update_rotation_mixing_info
   
         
      
      end module mix_info
