! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************


      module mesh_functions

      use star_private_def
      use const_def
      use num_lib
      use utils_lib
      use chem_def

      implicit none

      contains

      
      integer function get_net_iso(s, species)
         use chem_lib, only: chem_get_iso_id
         type (star_info), pointer :: s      
         character (len=*) :: species
         integer :: j
         get_net_iso = 0
         if (len_trim(species) == 0) return
         j = chem_get_iso_id(species)
         if (j <= 0) then
            write(*,*) 'unknown species name for mesh function: ' // trim(species)
            write(*,*) 'len_trim(species)', len_trim(species)
            return
         end if
         get_net_iso = s% net_iso(j) ! 0 if species not in current net
      end function get_net_iso

      
      logical function do_mass_function(s,species,weight,j)
         type (star_info), pointer :: s      
         character (len=iso_name_length) :: species
         real(dp), intent(in) :: weight
         integer, intent(out) :: j
         j = 0
         if (weight > 0) j = get_net_iso(s, species)
         do_mass_function = (j > 0)
      end function do_mass_function


      integer function num_mesh_functions(s)
         type (star_info), pointer :: s      
         integer :: i, j, k
         i = 0
         if (s% use_other_mesh_functions) &
            call s% how_many_other_mesh_fcns(s% id, i)
         if (s% convective_bdy_weight > 0) i=i+1
         if (s% P_function_weight > 0) i=i+1
         if (s% T_function1_weight > 0) i=i+1
         if (s% T_function2_weight > 0) i=i+1
         if (s% R_function_weight > 0) i=i+1
         if (s% R_function2_weight > 0) i=i+1
         if (s% M_function_weight > 0) i=i+1
         if (s% gradT_function_weight > 0) i=i+1
         if (s% omega_function_weight > 0 .and. s% rotation_flag) i=i+1
         do k=1,num_xa_function
            if (do_mass_function( &
                  s, s% xa_function_species(k), s% xa_function_weight(k), j)) &
               i=i+1
         end do
         num_mesh_functions = i
      end function num_mesh_functions
      
      

      subroutine set_mesh_function_data(s, nfcns, names, vals1, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: nfcns
         character (len=*) :: names(:)
         real(dp), pointer :: vals1(:) ! =(nz, nfcns)
         integer, intent(out) :: ierr
         
         integer :: i, nz, j, k, i_other
         logical, parameter :: dbg = .false.
         real(dp), dimension(:,:), pointer :: vals
         
         ierr = 0
         nz = s% nz
         
         vals(1:nz,1:nfcns) => vals1(1:nz*nfcns)
         
         i_other = 0
         if (s% use_other_mesh_functions) then
            call s% how_many_other_mesh_fcns(s% id, i_other)
            if (i_other > 0) then
               if (i_other > nfcns) then
                  ierr = -1
                  return
               end if
               call s% other_mesh_fcn_data(s% id, i_other, names, vals1, ierr)
               if (ierr /= 0) return
            end if
         end if

         i = i_other            
         if (s% P_function_weight > 0) then
            i = i+1; names(i) = 'P_function'
         end if      
         if (s% T_function1_weight > 0) then
            i = i+1; names(i) = 'T_function1'
         end if      
         if (s% T_function2_weight > 0) then
            i = i+1; names(i) = 'T_function2'
         end if         
         if (s% R_function_weight > 0) then
            i = i+1; names(i) = 'R_function'
         end if         
         if (s% R_function2_weight > 0) then
            i = i+1; names(i) = 'R_function2'
         end if         
         if (s% M_function_weight > 0) then
            i = i+1; names(i) = 'M_function'
         end if         
         if (s% gradT_function_weight > 0) then
            i = i+1; names(i) = 'gradT_function'
         end if         
         if (s% omega_function_weight > 0 .and. s% rotation_flag) then
            i = i+1; names(i) = 'omega_function'
         end if         
         if (s% convective_bdy_weight > 0) then
            i = i+1; names(i) = 'newly_nonconv'
         end if      
         do k=1,num_xa_function
            if (do_mass_function(s, s% xa_function_species(k), s% xa_function_weight(k), j)) then
               i = i+1; names(i) = trim(s% xa_function_species(k))
               !write(names(i),'(a,i1)') 'xa_function_', k
            end if         
         end do
         if (i /= nfcns) then
            write(*,*) 'error in set_mesh_function_names: incorrect nfcns'
            ierr = -1
         end if
      
!x$OMP PARALLEL DO PRIVATE(i,k)
         do i=i_other+1,nfcns
         
            if (ierr /= 0) cycle
            if (dbg) write(*,*) trim(names(i))

            if (names(i) == 'P_function') then
               do k=1,nz 
                  vals(k,i) = s% P_function_weight * s% lnP(k)/ln10
               end do
                  
            else if (names(i) == 'T_function1') then
               do k=1,nz 
                  vals(k,i) = s% T_function1_weight*s% lnT(k)/ln10
               end do
                  
            else if (names(i) == 'T_function2') then
               do k=1,nz 
                  vals(k,i) = &
                     s% T_function2_weight*log10(s% T(k) / (s% T(k) + s% T_function2_param))
               end do
               
            else if (names(i) == 'R_function') then
               do k=1,nz 
                  vals(k,i) = &
                     s% R_function_weight*log10(1 + (s% r(k)/Rsun)/s% R_function_param)
               end do
               
            else if (names(i) == 'R_function2') then
               do k=1,nz 
                  vals(k,i) = &
                     s% R_function2_weight * &
                     min(s% R_function2_param1, max(s% R_function2_param2,s% r(k)/s% r(1)))
               end do
               
            else if (names(i) == 'M_function') then
               do k=1,nz 
                  vals(k,i) = &
                  s% M_function_weight*log10(1 + (s% xmstar*s% q(k)/Msun)/s% M_function_param)
               end do
               
            else if (names(i) == 'gradT_function') then
               do k=1,nz 
                  vals(k,i) = s% gradT_function_weight*s% gradT(k)
               end do
               
            else if (names(i) == 'omega_function') then
               do k=1,nz 
                  vals(k,i) = s% omega_function_weight*log10(max(1d-99,s% omega(k)))
               end do

            else if (names(i) == 'newly_nonconv') then
               call do_newly_nonconvective(i)
               !call do_mix_type_change(i)
               
            else 
               do k=1,num_xa_function
                  call do1_xa_function(k,i)
               end do
               
            end if

         end do
!x$OMP END PARALLEL DO

         contains
         
         
         subroutine do_mix_type_change(i)
            integer, intent(in) :: i
            integer :: k, j
            if (s% dt < s% convective_bdy_min_dt_yrs*secyer) return
            do k=1,nz
               if (s% dq(k) < s% convective_bdy_dq_limit) cycle
               if (s% mixing_type_change_dq(k) /= 0) then
                  vals(k,i) = s% convective_bdy_weight
                  cycle
               end if
               cycle
               
               
               
               if (k > 1) then
                  if (s% mixing_type_change_dq(k-1) /= 0) then
                     vals(k,i) = s% convective_bdy_weight
                     cycle
                  end if
               end if
               if (k < nz) then
                  if (s% mixing_type_change_dq(k+1) /= 0) then
                     vals(k,i) = s% convective_bdy_weight
                     cycle
                  end if
               end if
            end do
            do j=1,5
               do k=2,nz
                  vals(k,i) = vals(k,i) + vals(k-1,i)
               end do
            end do
            
         end subroutine do_mix_type_change
         
         
         subroutine do_newly_nonconvective(i)
            use mlt_def, only: convective_mixing
            integer, intent(in) :: i
            integer :: k, ktop, j
            logical :: in_region, convective_below, convective_above
            
            logical, parameter :: dbg = .false.
            
            include 'formats'
            
            vals(1:nz,i) = 0
            
            if (s% dt < s% convective_bdy_min_dt_yrs*secyer) return
            in_region = s% newly_nonconvective(1)
            ktop = 1
            convective_above = .false.


            
            if (dbg) then ! debug
               write(*,2) 'initial check', i
               do k=1,nz
                  if (vals(k,i) /= 0) write(*,3) 'vals(k,i)', k, i, vals(k,i)
               end do
               do k=1,nz
                  if (s% newly_nonconvective(k)) write(*,2) 's% newly_nonconvective(k)', k
               end do
               write(*,*)
               !stop 'do_newly_nonconvective'
            end if

            do k=2,nz
               if (in_region) then
                  if (.not. s% newly_nonconvective(k)) then ! end of region
                     convective_below = (s% mixing_type(k) == convective_mixing)
                     if (dbg) write(*,3) 'call check_newly_nonconv_region', ktop, k-1
                     call check_newly_nonconv_region( &
                        i, ktop, k-1, convective_below, convective_above)
                     in_region = .false.
                  end if
               else if (s% newly_nonconvective(k)) then ! start of region
                  if (dbg) write(*,2) 'start of newly_nonconvective region', k
                  convective_above = (s% mixing_type(k-1) == convective_mixing)
                  in_region = .true.
                  ktop = max(1,k-1)
               end if
            end do
            
            if (dbg) then ! debug
               write(*,2) 'middle check', i
               do k=1,nz
                  if (vals(k,i) /= 0) write(*,3) 'vals(k,i)', k, i, vals(k,i)
               end do
               write(*,*)
               stop 'do_newly_nonconvective'
            end if
            
            
            if (in_region) then
               convective_below = .false.
               if (dbg) write(*,3) 'call check_newly_nonconv_region', ktop, nz
               call check_newly_nonconv_region( &
                  i, ktop, nz, convective_below, convective_above)
            end if
            
            !if (.true.) then ! also include convective boundaries
            if (dbg) write(*,2) 's% num_conv_boundaries', s% num_conv_boundaries
            do j = 1, s% num_conv_boundaries
               k = s% conv_bdy_loc(j)
               if (dbg) write(*,3) 'convective boundaries', j, k
               vals(k,i) = s% convective_bdy_weight
               if (k < nz .and. s% top_conv_bdy(i)) then
                  vals(k+1,i) = s% convective_bdy_weight
               else if (k > 1 .and. .not. s% top_conv_bdy(i)) then
                  vals(k-1,i) = s% convective_bdy_weight
               end if
            end do
            !end if
            
            if (dbg) then ! debug
               write(*,2) 'final check', i
               do k=1,nz
                  if (vals(k,i) /= 0) write(*,3) 'vals(k,i)', k, i, vals(k,i)
               end do
               write(*,*)
               stop 'do_newly_nonconvective'
            end if
            
            do k=2,nz
               vals(k,i) = vals(k,i) + vals(k-1,i)
            end do
            
         end subroutine do_newly_nonconvective
         
            
         subroutine check_newly_nonconv_region( &
               i, ktop, kbot, convective_below, convective_above)
            integer, intent(in) :: i, ktop, kbot
            logical, intent(in) :: convective_below, convective_above
            integer :: k
            real(dp) :: qtop, qbot, dq, q_start, q_end
            
            include 'formats'
            
            qtop = s% q(ktop)
            qbot = s% q(kbot)
            dq = sum(s% dq(ktop:kbot-1))
            q_start = qtop
            q_end = qbot
            q_end = max(0d0, q_end - dq)
            q_start = min(1d0, q_start + dq)
            do k=1,nz
               if (s% q(k) <= q_start .and. s% q(k) >= q_end) then
                  if (s% dq(k) >= s% convective_bdy_dq_limit) then
                     vals(k,i) = s% convective_bdy_weight
                  else if (s% convective_bdy_dq_limit > 0) then
                     vals(k,i) = s% convective_bdy_weight * &
                        s% dq(k)/s% convective_bdy_dq_limit
                  end if
               end if
            end do
            !stop 'check_newly_nonconv_region'
            
         end subroutine check_newly_nonconv_region
          
          
         subroutine do1_xa_function(k,i)
            integer, intent(in) :: k,i
            real(dp) :: weight, param
            integer :: j, m
            if (len_trim(s% xa_function_species(k))==0) return
            if (trim(names(i)) /= trim(s% xa_function_species(k))) return
            j = get_net_iso(s, s% xa_function_species(k))
            if (j <= 0) then
               ierr = -1
            else
               weight = s% xa_function_weight(k)
               param = s% xa_function_param(k)
               do m=1,s% nz
                  vals(m,i) = weight*log10(s% xa(j,m) + param)
               end do
            end if
         end subroutine do1_xa_function
         

      end subroutine set_mesh_function_data


      end module mesh_functions
