! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************


      module mesh_adjust

      use const_def
      use star_def
      use utils_lib, only:is_bad_num
      use chem_def

      implicit none

      
      real(dp), parameter :: eta_limit = -1d-6
            
      
      logical, parameter :: dbg = .false.

      
      contains
      

      subroutine do_mesh_adjust( &
            s, nz, nz_old, xh_old, xa_old, &
            energy_old, eta_old, lnd_old, lnPgas_old, lnE_old, &
            j_rot_old, omega_old, conv_vel_old, Del_T_old, &
            dlnd_dt_old, dlnPgas_dt_old, dlnT_dt_old, &
            nu_ST_old, D_ST_old, D_DSI_old, D_SH_old, &
            D_SSI_old, D_ES_old, D_GSF_old, &
            cell_type, comes_from, &
            dq_old, xq_old, xh, xa, dq, xq, ierr)
         use interp_1d_def
         use interp_1d_lib
         use star_utils, only: use_xh_to_update_i_rot_and_j_rot
         type (star_info), pointer :: s
         integer, intent(in) :: nz, nz_old
         integer, dimension(:), pointer :: cell_type, comes_from
         real(dp), dimension(:), pointer :: &
            dq_old, xq_old, dq, xq, energy_old, eta_old, &
            lnd_old, lnPgas_old, lnE_old, conv_vel_old, Del_T_old, &
            dlnd_dt_old, dlnPgas_dt_old, dlnT_dt_old, &
            brunt_B_old, j_rot_old, omega_old, &
            nu_ST_old, D_ST_old, D_DSI_old, D_SH_old, &
            D_SSI_old, D_ES_old, D_GSF_old
         real(dp), dimension(:,:), pointer :: xh_old, xa_old
         real(dp), dimension(:,:), pointer :: xh, xa
         real(dp), dimension(:), pointer :: &
            lnT_c0, lnT_c1, lnT_c2, interp_work, &
            energy_c0, energy_c1, energy_c2, density_new, new_vals, interp_y_old, &
            interp_x_old, interp_x_new, interp_r_old, interp_r_new
         integer, intent(out) :: ierr
         
         real(dp) :: dxa, xmstar, mstar, sumx
         real(dp), pointer, dimension(:) :: xq_old_plus1
         real(dp), pointer :: lnT_old(:)
         integer, pointer, dimension(:) :: old_loc
         character (len=256) :: message
         integer :: k, j, op_err, nzlo, nzhi, nzlo_old, nzhi_old, species
         logical :: interpolate_xa, found_bad_one
         real(dp), pointer :: p2(:,:)

         real(dp), dimension(:), pointer :: xa1_c0, xa1_c1, xa1_c2
         real(dp), dimension(:,:), pointer :: xa_c0, xa_c1, xa_c2
         
         include 'formats'

         ierr = 0
         species = s% species
         xmstar = s% xmstar
         mstar = xmstar + s% M_center
         interpolate_xa = s% mesh_interpolate_xa

         ! check xq's
         do k=1,nz
            if (xq(k) < 0 .or. xq(k) > 1) then
               ierr = -1
               return
               
               write(*,*) 'k', k
               write(*,*) 'xq(k)', xq(k)
               stop 'debug: do_mesh_adjust'
            end if
         end do
         
         if (dbg) write(*,*) 'enter do_mesh_adjust'
         
         nzlo = 0
         do k = 1, nz
            if (cell_type(k) /= unchanged_type) then
               if (dbg) write(*,2) 'nzlo changed', k
               nzlo = k; exit
            end if
         end do
         if (nzlo == 0) then
            if (dbg) write(*,2) 'no cells changed'
            nzlo = nz
         end if
         
         nzhi = nzlo
         do k = nz, nzlo, -1
            if (cell_type(k) /= unchanged_type) then
               if (dbg) write(*,2) 'nzhi changed', k
               nzhi = k; exit
            end if
         end do
         
         ! extend range for purposes of interpolation
         if (nzhi < nz) nzhi = nzhi+1
         if (nzlo > 1) nzlo = nzlo-1
         
         nzlo_old = comes_from(nzlo)
         if (nzhi == nz) then
            nzhi_old = nz_old
         else
            nzhi_old = comes_from(nzhi+1)
         end if
         
         call do_alloc1(ierr)
         if (ierr /= 0) return
         
         do k=1,nz_old 
            lnT_old(k) = xh_old(s% i_lnT,k)
            xq_old_plus1(k) = xq_old(k)
         end do
         ! add point at true center so can interpolate xq_new > xq_old(nz_old)
         xq_old_plus1(nz_old+1) = 1
         
         ! cubic interpolate by mass to get new L and Volume
         ! set lnR and lnd from new Volume and new mass
         
         if (dbg) write(*,*) 'call do_FL'
         call do_FL( &
            s, nz, nz_old, nzlo, nzhi, comes_from, &
            xh, xh_old, xq, xq_old_plus1, ierr)
         if (failed('do_FL')) return

         if (dbg) write(*,*) 'call do_lnR_and_lnd'
         call do_lnR_and_lnd( &
            s, nz, nz_old, nzlo, nzhi, cell_type, comes_from, &
            xh, xh_old, xmstar, lnd_old, lnPgas_old, &
            dq, dq_old, xq, xq_old_plus1, density_new, ierr)
         if (failed('do_lnR_and_lnd')) return

         ! note: density_new is only set for nzlo to nzhi

         if (dbg) write(*,*) 'call set_old_locs'
         call set_old_locs
         
         if (s% v_flag) then
            call do_alloc2(ierr)
            if (ierr /= 0) return
            do k=1, nz_old
               interp_r_old(k) = exp(xh_old(s% i_lnR,k))
            end do
            do k=1, nz
               interp_r_new(k) = exp(xh(s% i_lnR,k))
            end do
            if (s% v_flag) then ! interp v by 1/r to smooth dlnd/dt
               do k = 1, nz_old
                  interp_x_old(k) = 1/interp_r_old(k)
                  interp_y_old(k) = xh_old(s% i_vel,k)
               end do
               do k=1,nz
                  interp_x_new(k) = 1/interp_r_new(k)
               end do
               call interpolate_vector( &
                     nz_old, interp_x_old, nz, interp_x_new, &
                     interp_y_old, new_vals, &
                     interp_pm, pm_work_size, interp_work, ierr)
               if (failed('interpolate_vector for velocity')) return
               do k=1,nz
                  xh(s% i_vel,k) = new_vals(k)
               end do
            end if
            call dealloc2
         end if

         if (s% rotation_flag) then
            call adjust_omega(s, nz, nz_old, comes_from, &
               xq_old, xq, dq_old, dq, xh, j_rot_old, ierr)
            if (failed('adjust_omega')) return            
            call do_interp_pt_val( &
               s, nz, nz_old, nzlo, nzhi, s% nu_ST, nu_ST_old, &
               0d0, xq, xq_old_plus1, .true., ierr)
            if (failed('nu_ST')) return
            call do_interp_pt_val( &
               s, nz, nz_old, nzlo, nzhi, s% D_ST, D_ST_old, &
               0d0, xq, xq_old_plus1, .true., ierr)
            if (failed('D_ST')) return
            call do_interp_pt_val( &
               s, nz, nz_old, nzlo, nzhi, s% D_DSI, D_DSI_old, &
               0d0, xq, xq_old_plus1, .true., ierr)
            if (failed('D_DSI')) return
            call do_interp_pt_val( &
               s, nz, nz_old, nzlo, nzhi, s% D_SH, D_SH_old, &
               0d0, xq, xq_old_plus1, .true., ierr)
            if (failed('D_SH')) return
            call do_interp_pt_val( &
               s, nz, nz_old, nzlo, nzhi, s% D_SSI, D_SSI_old, &
               0d0, xq, xq_old_plus1, .true., ierr)
            if (failed('D_SSI')) return
            call do_interp_pt_val( &
               s, nz, nz_old, nzlo, nzhi, s% D_ES, D_ES_old, &
               0d0, xq, xq_old_plus1, .true., ierr)
            if (failed('D_ES')) return
            call do_interp_pt_val( &
               s, nz, nz_old, nzlo, nzhi, s% D_GSF, D_GSF_old, &
               0d0, xq, xq_old_plus1, .true., ierr)
            if (failed('D_GSF')) return
         end if
         
         if (s% min_T_for_time_averaged_conv_velocity < 1d12) then
            call do_interp_pt_val( &
               s, nz, nz_old, nzlo, nzhi, s% conv_vel, conv_vel_old, &
               0d0, xq, xq_old_plus1, .true., ierr)
            if (failed('conv_vel')) return
            call do_interp_pt_val( &
               s, nz, nz_old, nzlo, nzhi, s% Del_T, Del_T_old, &
               0d0, xq, xq_old_plus1, .true., ierr)
            if (failed('Del_T')) return
         end if
         
         call do_interp_cell_var( &
            s, nz, nz_old, nzlo, nzhi, xq, xq_old_plus1, &
            dq, dq_old, s% dlnd_dt, dlnd_dt_old, ierr)
         if (failed('eps_nuc')) return
         
         call do_interp_cell_var( &
            s, nz, nz_old, nzlo, nzhi, xq, xq_old_plus1, &
            dq, dq_old, s% dlnPgas_dt, dlnPgas_dt_old, ierr)
         if (failed('eps_nuc')) return

         call do_interp_cell_var( &
            s, nz, nz_old, nzlo, nzhi, xq, xq_old_plus1, &
            dq, dq_old, s% dlnT_dt, dlnT_dt_old, ierr)
         if (failed('eps_grav')) return
         
         ! since the mass fractions must add to 1 everywhere in the cell, the slopes must add to 0.
         do k=nzlo_old,nzhi_old  ! 1,nz_old  !
         
            if (s% mesh_adjust_get_T_from_E) then
               p2(1:1,1:nz_old) => energy_old(1:nz_old)
               call get1_lpp(k, 1, nz_old, 1, dq_old, p2, &
                  s% mesh_adjust_use_quadratic, energy_c0, energy_c1, energy_c2)
            end if

            p2(1:1,1:nz_old) => lnT_old(1:nz_old)
            call get1_lpp(k, 1, nz_old, 1, dq_old, p2, &
                  s% mesh_adjust_use_quadratic, lnT_c0, lnT_c1, lnT_c2)
         
            ! since we must adjust things to make the sum of xa's = 1, only do linear reconstruction.
            do j=1,species
               call get1_lpp(k, species, nz_old, j, dq_old, xa_old, &
                              .false., xa_c0(:,j), xa_c1(:,j), xa_c2(:,j))
               if (.false.) write(*,2) 'xa_c0,c1,c2', j, xa_c0(k,j), xa_c1(k,j), xa_c2(k,j)
            end do
            
            if (.not. interpolate_xa) then
               sumx = sum(xa_old(1:species,k))
               do j=1,species
                  xa_c0(k,j) = xa_old(j,k)/sumx ! make sure that adds to 1
                  xa_c2(k,j) = 0 ! no curvature terms
               end do
               
               ! only reduce magnitude of slopes so don't risk producing values out of [0..1] range
               if (sum(xa_c1(k,:)) > 0) then
                  j = maxloc(xa_c1(k,:), dim=1)
               else
                  j = minloc(xa_c1(k,:), dim=1)
               end if
               xa_c1(k,j) = 0
               xa_c1(k,j) = -sum(xa_c1(k,:))
               ! check for valid fractions at boundaries; set slopes to 0 if find a bad one.
               do j=1,species
                  dxa = abs(xa_c1(k,j))*dq_old(k)/2
                  if (xa_c0(k,j) + dxa > 1 .or. xa_c0(k,j) - dxa < 0) then
                     xa_c1(k,:) = 0
                     exit
                  end if
                  if (.false.) write(*,2) '2nd xa_c0,c1,c2', j, xa_c0(k,j), xa_c1(k,j), xa_c2(k,j)
               end do
            end if
            
         end do
         
         if (failed('adjust_mesh nz_old parallel loop')) return
         
         if (interpolate_xa) then
            call do_xa_interp( &
               s, nz, nz_old, nzlo, nzhi, species, &
               xa, xa_old, xq, xq_old_plus1, ierr)
            if (failed('do_xa_interp')) return
         end if
         
         if (dbg) write(*,*) 'do xa and lnT'
         do k = 1, nz

            op_err = 0
         
            if (.not. interpolate_xa) then
               ! calculate new abundances to conserve species
               call do_xa( &
                  s, nz, nz_old, k, species, cell_type, comes_from, xa, xa_old, &
                  xa_c0, xa_c1, xa_c2, xq, dq, xq_old, dq_old, old_loc, &
                  s% mesh_adjust_use_quadratic, op_err)
               if (op_err /= 0) then
                  write(message,*) 'do_xa for k', k
                  ierr = op_err
               end if
            end if

            ! calculate new temperatures to conserve energy
            call do_lnT( &
               s, nz_old, k, species, cell_type, comes_from, &
               xa, xh, xh_old, &
               xq, dq, xq_old, eta_old, &
               lnT_c0, lnT_c1, lnT_c2, &
               energy_c0, energy_c1, energy_c2, old_loc, density_new, op_err)
            if (op_err /= 0) then
               write(message,*) 'do_lnT for k', k
               ierr = op_err
            end if
         
         end do

         if (failed(message)) return
         
         if (dbg) write(*,*) 'call check_species_conservation'
         call check_species_conservation(species,ierr)
         
         call dealloc1
         

         contains
         

         subroutine do_alloc1(ierr)
            use alloc
            integer, intent(out) :: ierr            
            call get_integer_work_array(s, old_loc, nz, nz_alloc_extra, ierr)
            if (ierr /= 0) return            
            call non_crit_get_work_array(s, xq_old_plus1, nz_old+1, nz_alloc_extra, 'mesh_adjust', ierr)
            if (ierr /= 0) return          
            call non_crit_get_work_array(s, lnT_old, nz_old, nz_alloc_extra, 'mesh_adjust', ierr)
            if (ierr /= 0) return           
            call non_crit_get_work_array(s, energy_c0, nz_old, nz_alloc_extra, 'mesh_adjust', ierr)
            if (ierr /= 0) return           
            call non_crit_get_work_array(s, energy_c1, nz_old, nz_alloc_extra, 'mesh_adjust', ierr)
            if (ierr /= 0) return            
            call non_crit_get_work_array(s, energy_c2, nz_old, nz_alloc_extra, 'mesh_adjust', ierr)
            if (ierr /= 0) return            
            call non_crit_get_work_array(s, lnT_c0, nz_old, nz_alloc_extra, 'mesh_adjust', ierr)
            if (ierr /= 0) return            
            call non_crit_get_work_array(s, lnT_c1, nz_old, nz_alloc_extra, 'mesh_adjust', ierr)
            if (ierr /= 0) return            
            call non_crit_get_work_array(s, lnT_c2, nz_old, nz_alloc_extra, 'mesh_adjust', ierr)
            if (ierr /= 0) return            
            call non_crit_get_work_array(s, density_new, nz, nz_alloc_extra, 'mesh_adjust', ierr)
            if (ierr /= 0) return            
            call non_crit_get_work_array(s, xa1_c0, nz_old*species, nz_alloc_extra*species, 'mesh_adjust', ierr)
            if (ierr /= 0) return
            xa_c0(1:nz_old,1:species) => xa1_c0(1:nz_old*species)
            call non_crit_get_work_array(s, xa1_c1, nz_old*species, nz_alloc_extra*species, 'mesh_adjust', ierr)
            if (ierr /= 0) return
            xa_c1(1:nz_old,1:species) => xa1_c1(1:nz_old*species)
            call non_crit_get_work_array(s, xa1_c2, nz_old*species, nz_alloc_extra*species, 'mesh_adjust', ierr)
            if (ierr /= 0) return
            xa_c2(1:nz_old,1:species) => xa1_c2(1:nz_old*species)
         end subroutine do_alloc1
         
         
         subroutine dealloc1
            use alloc
            call return_integer_work_array(s, old_loc)            
            call non_crit_return_work_array(s, xq_old_plus1, 'mesh_adjust')            
            call non_crit_return_work_array(s, lnT_old, 'mesh_adjust')            
            call non_crit_return_work_array(s, energy_c0, 'mesh_adjust')            
            call non_crit_return_work_array(s, energy_c1, 'mesh_adjust')            
            call non_crit_return_work_array(s, energy_c2, 'mesh_adjust')            
            call non_crit_return_work_array(s, lnT_c0, 'mesh_adjust')           
            call non_crit_return_work_array(s, lnT_c1, 'mesh_adjust')            
            call non_crit_return_work_array(s, lnT_c2, 'mesh_adjust')            
            call non_crit_return_work_array(s, density_new, 'mesh_adjust')            
            call non_crit_return_work_array(s, xa1_c0, 'mesh_adjust')            
            call non_crit_return_work_array(s, xa1_c1, 'mesh_adjust')            
            call non_crit_return_work_array(s, xa1_c2, 'mesh_adjust')
         end subroutine dealloc1
         

         subroutine do_alloc2(ierr)
            use alloc
            use utils_lib
            integer, intent(out) :: ierr
            call non_crit_get_work_array(s, interp_x_old, nz_old, nz_alloc_extra, 'mesh_adjust', ierr)
            if (ierr /= 0) return            
            call non_crit_get_work_array(s, interp_x_new, nz, nz_alloc_extra, 'mesh_adjust', ierr)
            if (ierr /= 0) return            
            call non_crit_get_work_array(s, new_vals, nz, nz_alloc_extra, 'mesh_adjust', ierr)
            if (ierr /= 0) return            
            call non_crit_get_work_array(s, interp_y_old, nz_old, nz_alloc_extra, 'mesh_adjust', ierr)
            if (ierr /= 0) return 
            call non_crit_get_work_array(s, interp_work, nz_old*pm_work_size, nz_alloc_extra, 'mesh_adjust', ierr)
            if (ierr /= 0) return
            call non_crit_get_work_array(s, interp_r_old, nz_old, nz_alloc_extra, 'mesh_adjust', ierr)
            if (ierr /= 0) return            
            call non_crit_get_work_array(s, interp_r_new, nz, nz_alloc_extra, 'mesh_adjust', ierr)
            if (ierr /= 0) return            
         end subroutine do_alloc2
         
         
         subroutine dealloc2
            use alloc
            use utils_lib
            call non_crit_return_work_array(s, interp_x_old, 'mesh_adjust')            
            call non_crit_return_work_array(s, interp_x_new, 'mesh_adjust')            
            call non_crit_return_work_array(s, new_vals, 'mesh_adjust')            
            call non_crit_return_work_array(s, interp_y_old, 'mesh_adjust')            
            call non_crit_return_work_array(s, interp_work, 'mesh_adjust')
            call non_crit_return_work_array(s, interp_r_old, 'mesh_adjust')            
            call non_crit_return_work_array(s, interp_r_new, 'mesh_adjust')            
         end subroutine dealloc2


         subroutine set_old_locs
            real(dp) :: dq_sum_new, dq_sum_old
            logical :: okay
            integer :: k_old, k, j
            include 'formats'
            ! setup map from new k's to old ones
            ! for k=1:nz, set old_loc(k) s.t. 
               ! if old_loc(k) == nz_old then
                  ! xq(k) >= xq_old(old_loc(k))
               ! else
                  ! xq_old(old_loc(k)+1) > xq(k) >= xq_old(old_loc(k))
            k_old = 1
            old_loc(1) = 1
            do k = 2,nz
               do while (k_old <= nz_old)
                  if (xq_old(k_old) > xq(k)) exit
                  k_old = k_old + 1
               end do
               old_loc(k) = k_old-1
            end do            
         end subroutine set_old_locs
         
         
         logical function failed(msg)
            character (len=*) :: msg
            if (ierr == 0) then
               failed = .false.
               return
            end if
            failed = .true.
            if (dbg) write(*, *) 'mesh_revisions failed in ' // trim(msg)
            call dealloc1
            return
         end function failed

         
         subroutine check_species_conservation(species,ierr)
            integer, intent(in) :: species
            integer, intent(out) :: ierr
            integer :: j, k, jbad
            real(dp) :: old_total, new_total
            logical :: okay
            include 'formats'
            ierr = 0
            okay = .true.
            jbad = -1
!x$OMP PARALLEL DO PRIVATE(j, old_total, new_total)  >>> OMP makes it worse
            do j=1,species
               old_total = dot_product(xa_old(j,1:nz_old),dq_old(1:nz_old))
               if (old_total < 1d-9) cycle
               new_total = dot_product(xa(j,1:nz),dq(1:nz))
               if (abs(new_total - old_total) > 1d-4) then ! check for major problems
                  ierr = -1
                  jbad = j
                  okay = .false.
                  if (dbg) then
                     write(*,*) 'problem with conservation of species ' // chem_isos% name(s% chem_id(j))
                     write(*,1) 'new mass fraction', new_total
                     write(*,1) 'old mass fraction', old_total
                     write(*,1) 'new - old', new_total - old_total
                     write(*,1) '(new - old)/old', (new_total - old_total) / old_total
                     write(*,*)
                  end if
               end if
            end do
!x$OMP END PARALLEL DO
            if (okay) return
            ierr = -1
            write(*,*)
            do j=1,species
               old_total = dot_product(xa_old(j,1:nz_old),dq_old(1:nz_old))
               if (old_total < 1d-9) cycle
               new_total = dot_product(xa(j,1:nz),dq(1:nz))
               write(*,2) 'new - old mass fraction ' // chem_isos% name(s% chem_id(j)), &
                     j, new_total-old_total
            end do
            write(*,*)
            j = jbad
            do k=2, nz
               if (comes_from(k) == comes_from(k-1)) cycle
               old_total = dot_product(xa_old(j,1:comes_from(k)-1),dq_old(1:comes_from(k)-1))
               if (old_total < 1d-9) cycle
               new_total = dot_product(xa(j,1:k-1),dq(1:k-1))
               write(*,2) 'partial new - old ' // chem_isos% name(s% chem_id(j)), k, &
                  new_total-old_total, new_total, old_total
            end do
            write(*,*)
            do k=415, nz
               write(*,'(a30,99i6)') 'cell_type(k)', k, cell_type(k), comes_from(k)
            end do
            write(*,*)
            write(*,2) 'xq', 439, xq(439)
            write(*,2) 'xq_old', 429, xq_old(429)
            write(*,2) 'dq_old', 429, dq_old(429)
            write(*,2) 'dq', 439, dq(439)
            write(*,*)
            write(*,2) 'xq', 424, xq(424)
            write(*,2) 'xq_old', 428, xq_old(428)
            write(*,2) 'dq_old', 428, dq_old(428)
            write(*,2) 'sum dq', 424, sum(dq(424:438))
            write(*,*)
            write(*,2) 'xq_old + dq_old', 428, xq_old(428) + dq_old(428)
            write(*,2) 'xq_old', 429, xq_old(429)
            write(*,*)
            write(*,2) 'xq + sum dq', 424, xq(424) + sum(dq(424:438))
            write(*,2) 'xq', 439, xq(439)
            write(*,*)
            write(*,1) 'sum dq_old', sum(dq_old(1:nz_old))            
            
            write(*,2) 'dq_old', 427, dq_old(427)
            write(*,2) 'sum new', 416, sum(dq(416:423))
            write(*,2) 'dq_old - sum new', 427, dq_old(427) - sum(dq(416:423))
            write(*,2) 'dq_old', 428, dq_old(428)
            write(*,2) 'sum new', 424, sum(dq(424:438))
            write(*,2) 'dq_old - sum new', 428, dq_old(428) - sum(dq(424:438))
         end subroutine check_species_conservation
         

      end subroutine do_mesh_adjust

      
      
      subroutine do_xa_interp( &
            s, nz, nz_old, nzlo, nzhi, species, xa, xa_old, xq, xq_old_plus1, ierr)
         use interp_1d_def
         use interp_1d_lib
         type (star_info), pointer :: s
         integer, intent(in) :: nz, nz_old, nzlo, nzhi, species
         real(dp), dimension(:,:), pointer :: xa, xa_old
         real(dp), dimension(:), pointer :: xq, xq_old_plus1
         integer, intent(out) :: ierr
         integer, parameter :: nwork = pm_work_size
         real(dp), pointer, dimension(:) :: xnew, xold, qmid_new, qmid_old
         real(dp), pointer, dimension(:) :: work
         integer :: n, i, j, k
         ierr = 0
         
         n = nzhi - nzlo + 1
         call do_alloc(ierr)
         if (ierr /= 0) return
         
         do i = 1, n
            k = i + nzlo - 1
            if (k < nz) then
               qmid_new(i) = (xq(k) + xq(k+1))/2
            else
               qmid_new(i) = (xq(k) + 1)/2
            end if
         end do

         do k=1,nz_old
            qmid_old(k) = (xq_old_plus1(k) + xq_old_plus1(k+1))/2
         enddo
         
         do j=1,species
            do k=1,nz_old
               xold(k) = xa_old(j,k)
            end do
            call interpolate_vector( &
                  nz_old, qmid_old, n, qmid_new, xold, xnew, interp_pm, nwork, work, ierr)
            if (ierr /= 0) then
               call dealloc
               return
            end if
            do k=1,n
               xa(j,nzlo+k-1) = xnew(k)
            end do
         end do
         
         n = nzlo - 1
         if (n > 0) then
            do k=1,n
               do j=1,species
                  xa(j,k) = xa_old(j,k)
               end do
            end do
         end if
         
         if (nzhi < nz) then
            n = nz - nzhi - 1 ! nz-n = nzhi+1
            do j=0,n 
               xa(:,nz-j) = xa_old(:,nz_old-j)
            enddo
         end if
         
         call dealloc
         
         contains
         
         subroutine do_alloc(ierr)
            use alloc
            use utils_lib
            integer, intent(out) :: ierr
            real(dp), pointer :: p(:)
            call non_crit_get_work_array(s, qmid_new, n, nz_alloc_extra, 'mesh_adjust', ierr)
            if (ierr /= 0) return
            call non_crit_get_work_array(s, xnew, n, nz_alloc_extra, 'mesh_adjust', ierr)
            if (ierr /= 0) return
            call non_crit_get_work_array(s, xold, n, nz_alloc_extra, 'mesh_adjust', ierr)
            if (ierr /= 0) return
            call non_crit_get_work_array(s, qmid_old, nz_old, nz_alloc_extra, 'mesh_adjust', ierr)
            if (ierr /= 0) return
            call non_crit_get_work_array(s, work, (nz_old+1)*nwork, nz_alloc_extra, 'mesh_adjust', ierr)
         end subroutine do_alloc
         
         subroutine dealloc
            use alloc
            use utils_lib
            real(dp), pointer :: p(:)
            call non_crit_return_work_array(s, qmid_new, 'mesh_adjust')            
            call non_crit_return_work_array(s, xnew, 'mesh_adjust')            
            call non_crit_return_work_array(s, xold, 'mesh_adjust')            
            call non_crit_return_work_array(s, qmid_old, 'mesh_adjust')            
            call non_crit_return_work_array(s, work, 'mesh_adjust')
         end subroutine dealloc
         
      end subroutine do_xa_interp
      
      
      subroutine do_FL( &
            s, nz, nz_old, nzlo, nzhi, comes_from, &
            xh, xh_old, xq, xq_old_plus1, ierr)
         use interp_1d_def
         use interp_1d_lib
         type (star_info), pointer :: s
         integer, intent(in) :: nz, nz_old, nzlo, nzhi, comes_from(:)
         real(dp), dimension(:,:), pointer :: xh, xh_old
         real(dp), dimension(:), pointer :: xq, xq_old_plus1
         integer, intent(out) :: ierr
         integer, parameter :: nwork = pm_work_size
         real(dp), pointer, dimension(:) :: L_old_plus1, L_new, xq_new, work
         integer :: n, i_lum, k

         include 'formats'
         
         ierr = 0
         i_lum = s% i_lum
         n = nzhi - nzlo + 1
         call do_alloc(ierr)
         if (ierr /= 0) return
         
         do k=1,nz_old
            L_old_plus1(k) = xh_old(i_lum,k)
         end do
         L_old_plus1(nz_old+1) = s% L_center
         
         do k=1,n
            xq_new(k) = xq(nzlo+k-1)
         end do
         
         call interpolate_vector( &
               nz_old+1, xq_old_plus1, n, xq_new, &
               L_old_plus1, L_new, interp_pm, nwork, work, ierr)
         if (ierr /= 0) then
            call dealloc
            return
            
            write(*,*) 'interpolate_vector failed in do_FL for remesh'
            stop 'debug: mesh adjust: do_FL'
         end if
         
         do k=nzlo,nzhi
            xh(i_lum,k) = L_new(k+1-nzlo)
         end do
         
         n = nzlo - 1
         if (n > 0) then
            do k=1,n
               xh(i_lum,k) = xh_old(i_lum,k)
            end do
         end if
         
         if (nzhi < nz) then
            n = nz - nzhi - 1 ! nz-n = nzhi+1
            do k=0,n
               xh(i_lum,nz-k) = xh_old(i_lum,nz_old-k)
            end do
         end if
                  
         call dealloc
         
         contains
         
         subroutine do_alloc(ierr)
            use alloc
            use utils_lib
            integer, intent(out) :: ierr
            real(dp), pointer :: p(:)
            call non_crit_get_work_array(s, L_old_plus1, nz_old+1, nz_alloc_extra, 'mesh_adjust', ierr)
            if (ierr /= 0) return
            call non_crit_get_work_array(s, L_new, n, nz_alloc_extra, 'mesh_adjust', ierr)
            if (ierr /= 0) return
            call non_crit_get_work_array(s, xq_new, n, nz_alloc_extra, 'mesh_adjust', ierr)
            if (ierr /= 0) return
            call non_crit_get_work_array(s, work, (nz_old+1)*nwork, nz_alloc_extra, 'mesh_adjust', ierr)
            if (ierr /= 0) return
         end subroutine do_alloc
         
         subroutine dealloc
            use alloc
            use utils_lib
            real(dp), pointer :: p(:)
            call non_crit_return_work_array(s, L_old_plus1, 'mesh_adjust')            
            call non_crit_return_work_array(s, L_new, 'mesh_adjust')            
            call non_crit_return_work_array(s, xq_new, 'mesh_adjust')            
            call non_crit_return_work_array(s, work, 'mesh_adjust')
         end subroutine dealloc

      end subroutine do_FL
      

      subroutine do_interp_pt_val( &
            s, nz, nz_old, nzlo, nzhi, val, val_old, center_val, &
            xq, xq_old_plus1, force_non_negative, ierr)
         use interp_1d_def
         use interp_1d_lib
         type (star_info), pointer :: s
         integer, intent(in) :: nz, nz_old, nzlo, nzhi
         real(dp), dimension(:), pointer :: val, val_old
         real(dp), intent(in) :: center_val
         real(dp), dimension(:), pointer :: xq, xq_old_plus1
         logical, intent(in) :: force_non_negative
         integer, intent(out) :: ierr
         integer, parameter :: nwork = pm_work_size
         real(dp), pointer, dimension(:) :: val_old_plus1, val_new, xq_new, work
         integer :: n, k

         include 'formats'
         
         ierr = 0
         n = nzhi - nzlo + 1
         call do_alloc(ierr)
         if (ierr /= 0) return
         
         do k=1,nz_old
            val_old_plus1(k) = val_old(k)
         end do
         val_old_plus1(nz_old+1) = center_val
         
         do k=1,n
            xq_new(k) = xq(nzlo+k-1)
         end do
         
         call interpolate_vector( &
               nz_old+1, xq_old_plus1, n, xq_new, &
               val_old_plus1, val_new, interp_pm, nwork, work, ierr)
         if (ierr /= 0) then
            call dealloc
            return
         end if
         
         do k=nzlo,nzhi
            val(k) = val_new(k+1-nzlo)
         end do
         
         n = nzlo - 1
         if (n > 0) then
            do k=1,n
               val(k) = val_old(k)
            end do
         end if
         
         if (nzhi < nz) then
            n = nz - nzhi - 1 ! nz-n = nzhi+1
            do k=0,n 
               val(nz-k) = val_old(nz_old-k)
            end do
         end if
         
         if (force_non_negative) then
            do k=nzlo,nzhi
               if (val(k) < 0) val(k) = 0
            end do
         end if
         
         call dealloc
         
         contains
         
         subroutine do_alloc(ierr)
            use alloc
            use utils_lib
            integer, intent(out) :: ierr
            real(dp), pointer :: p(:)
            call non_crit_get_work_array(s, val_old_plus1, nz_old+1, nz_alloc_extra, 'mesh_adjust', ierr)
            if (ierr /= 0) return
            call non_crit_get_work_array(s, val_new, n, nz_alloc_extra, 'mesh_adjust', ierr)
            if (ierr /= 0) return
            call non_crit_get_work_array(s, xq_new, n, nz_alloc_extra, 'mesh_adjust', ierr)
            if (ierr /= 0) return
            call non_crit_get_work_array(s, work, (nz_old+1)*nwork, nz_alloc_extra, 'mesh_adjust', ierr)
            if (ierr /= 0) return
         end subroutine do_alloc
         
         subroutine dealloc
            use alloc
            use utils_lib
            real(dp), pointer :: p(:)
            call non_crit_return_work_array(s, val_old_plus1, 'mesh_adjust')            
            call non_crit_return_work_array(s, val_new, 'mesh_adjust')            
            call non_crit_return_work_array(s, xq_new, 'mesh_adjust')            
            call non_crit_return_work_array(s, work, 'mesh_adjust')
         end subroutine dealloc

      end subroutine do_interp_pt_val
      
      
      subroutine do_interp_cell_var( &
            s, nz, nz_old, nzlo, nzhi, xq, xq_old_plus1, &
            dq, dq_old, val, val_old, ierr)
         use interp_1d_def
         use interp_1d_lib
         type (star_info), pointer :: s
         integer, intent(in) :: nz, nz_old, nzlo, nzhi
         real(dp), dimension(:), pointer :: &
            xq, xq_old_plus1, val, val_old, dq, dq_old
         integer, intent(out) :: ierr
         integer, parameter :: nwork = pm_work_size
         
         real(dp), pointer, dimension(:) :: val_old_plus1, val_new, &
            mid_xq_new, mid_xq_old_plus1, work
         integer :: n, i, j, k
         
         ierr = 0
         n = nzhi - nzlo + 1
         call do_alloc(ierr)
         if (ierr /= 0) return
         
         do k=1,nz_old
            val_old_plus1(k) = val_old(k)
            mid_xq_old_plus1(k) = xq_old_plus1(k) + 0.5d0*dq_old(k)
         end do
         val_old_plus1(nz_old+1) = val_old_plus1(nz_old)
         mid_xq_old_plus1(nz_old+1) = 1
         do i=1,n 
            mid_xq_new(i) = xq(nzlo+i-1) + 0.5d0*dq(nzlo+i-1)
         end do
         
         call interpolate_vector( &
               nz_old+1, mid_xq_old_plus1, n, mid_xq_new, &
               val_old_plus1, val_new, interp_pm, nwork, work, ierr)
         if (ierr /= 0) then
            call dealloc
            return
            
            write(*,*) 'interpolate_vector failed in do_interp_cell_var for remesh'
            stop 'debug: mesh adjust: do_val'
         end if
         
         do i=1,n
            val(nzlo+i-1) = val_new(i)
         end do
         
         n = nzlo - 1
         if (n > 0) then
            do i=1,n
               val(i) = val_old(i)
            end do
         end if
         
         if (nzhi < nz) then
            n = nz - nzhi - 1 ! nz-n = nzhi+1
            do i=0,n
               val(nz-i) = val_old(nz_old-i)
            end do
         end if
         
         call dealloc
         
         contains
         
         subroutine do_alloc(ierr)
            use alloc
            use utils_lib
            integer, intent(out) :: ierr
            real(dp), pointer :: p(:)
            call non_crit_get_work_array(s, val_old_plus1, nz_old+1, nz_alloc_extra, 'mesh_adjust', ierr)
            if (ierr /= 0) return
            call non_crit_get_work_array(s, mid_xq_old_plus1, nz_old+1, nz_alloc_extra, 'mesh_adjust', ierr)
            if (ierr /= 0) return
            call non_crit_get_work_array(s, val_new, n, nz_alloc_extra, 'mesh_adjust', ierr)
            if (ierr /= 0) return
            call non_crit_get_work_array(s, mid_xq_new, n, nz_alloc_extra, 'mesh_adjust', ierr)
            if (ierr /= 0) return
            call non_crit_get_work_array(s, work, (nz_old+1)*nwork, nz_alloc_extra, 'mesh_adjust', ierr)
            if (ierr /= 0) return
         end subroutine do_alloc
         
         subroutine dealloc
            use alloc
            use utils_lib
            real(dp), pointer :: p(:)
            call non_crit_return_work_array(s, val_old_plus1, 'mesh_adjust')            
            call non_crit_return_work_array(s, mid_xq_old_plus1, 'mesh_adjust')            
            call non_crit_return_work_array(s, val_new, 'mesh_adjust')            
            call non_crit_return_work_array(s, mid_xq_new, 'mesh_adjust')            
            call non_crit_return_work_array(s, work, 'mesh_adjust')
         end subroutine dealloc

      end subroutine do_interp_cell_var
      
      
      subroutine do_interp_cell_val( &
            s, nz, nz_old, nzlo, nzhi, val_new_out, val_old, &
            xq, xq_old_plus1, dq, dq_old, ierr)
         use interp_1d_def
         use interp_1d_lib
         type (star_info), pointer :: s
         integer, intent(in) :: nz, nz_old, nzlo, nzhi
         real(dp), dimension(:), pointer :: val_new_out, val_old
         real(dp), dimension(:), pointer :: xq, xq_old_plus1, dq, dq_old
         integer, intent(out) :: ierr
         integer, parameter :: nwork = pm_work_size
         
         real(dp), pointer, dimension(:) :: val_old_plus1, &
            val_new, mid_xq_new, mid_xq_old_plus1, work
         integer :: n, i, j, k
         
         ierr = 0
         n = nzhi - nzlo + 1
         call do_alloc(ierr)
         if (ierr /= 0) return
         
         do k=1,nz_old
            val_old_plus1(k) = val_old(k)
            mid_xq_old_plus1(k) = xq_old_plus1(k) + 0.5d0*dq_old(k)
         end do
         val_old_plus1(nz_old+1) = val_old_plus1(nz_old)
         mid_xq_old_plus1(nz_old+1) = 1
         do i=1,n
            mid_xq_new(i) = xq(nzlo+i-1) + 0.5d0*dq(nzlo+i-1)
         end do
         
         call interpolate_vector( &
               nz_old+1, mid_xq_old_plus1, n, mid_xq_new, &
               val_old_plus1, val_new, interp_pm, nwork, work, ierr)
         if (ierr /= 0) then
            call dealloc
            return
         end if
         
         do i=1,n
            val_new_out(nzlo+i-1) = val_new(i)
         end do
         
         n = nzlo - 1
         if (n > 0) then
            do i=1,n
               val_new_out(i) = val_old(i)
            end do
         end if
         
         if (nzhi < nz) then
            n = nz - nzhi - 1 ! nz-n = nzhi+1
            do i=0,n
               val_new_out(nz-i) = val_old(nz_old-i)
            end do
         end if
         
         call dealloc
         
         contains
         
         subroutine do_alloc(ierr)
            use alloc
            use utils_lib
            integer, intent(out) :: ierr
            real(dp), pointer :: p(:)
            call non_crit_get_work_array(s, val_old_plus1, nz_old+1, nz_alloc_extra, 'mesh_adjust', ierr)
            if (ierr /= 0) return
            call non_crit_get_work_array(s, mid_xq_old_plus1, nz_old+1, nz_alloc_extra, 'mesh_adjust', ierr)
            if (ierr /= 0) return
            call non_crit_get_work_array(s, mid_xq_new, n, nz_alloc_extra, 'mesh_adjust', ierr)
            if (ierr /= 0) return
            call non_crit_get_work_array(s, val_new, n, nz_alloc_extra, 'mesh_adjust', ierr)
            if (ierr /= 0) return
            call non_crit_get_work_array(s, work, (nz_old+1)*nwork, nz_alloc_extra, 'mesh_adjust', ierr)
            if (ierr /= 0) return
         end subroutine do_alloc
         
         subroutine dealloc
            use alloc
            use utils_lib
            real(dp), pointer :: p(:)
            call non_crit_return_work_array(s, val_old_plus1, 'mesh_adjust')            
            call non_crit_return_work_array(s, mid_xq_old_plus1, 'mesh_adjust')            
            call non_crit_return_work_array(s, mid_xq_new, 'mesh_adjust')            
            call non_crit_return_work_array(s, val_new, 'mesh_adjust')            
            call non_crit_return_work_array(s, work, 'mesh_adjust')
         end subroutine dealloc

      end subroutine do_interp_cell_val
      
      
      subroutine do_lnR_and_lnd( &
            s, nz, nz_old, nzlo, nzhi, cell_type, comes_from, &
            xh, xh_old, xmstar, lnd_old, lnPgas_old, &
            dq, dq_old, xq, xq_old_plus1, density_new, ierr)
         use interp_1d_def
         use interp_1d_lib
         use num_lib
         type (star_info), pointer :: s
         integer, intent(in) :: nz, nz_old, nzlo, nzhi, cell_type(:), comes_from(:)
         real(dp), dimension(:,:), pointer :: xh, xh_old
         real(dp), intent(in) :: xmstar
         real(dp), dimension(:), pointer :: lnd_old, lnPgas_old
         real(dp), dimension(:), pointer :: xq, dq, dq_old, xq_old_plus1, density_new
         ! note: density_new is only set for nzlo to nzhi
         integer, intent(out) :: ierr

         integer, parameter :: nwork = pm_work_size
         integer :: k, kk, n, interp_lo, interp_hi, interp_n, &
            hint, i_lnR, i_xlnd, i_lnPgas
         real(dp), pointer :: work(:), Vol_old_plus1(:), Vol_new(:), &
            interp_Vol_new(:), interp_xq(:)
         real(dp) :: Vol_min, Vol_max, cell_Vol, Vol_center, Vm1, V00, Vp1
         
         logical, parameter :: dbg = .false.
         
         include 'formats'
         
         ! NOTE: for interpolating volume, need to add point at true center
            
         ierr = 0
         i_lnR = s% i_lnR
         i_xlnd = s% i_xlnd
         i_lnPgas = s% i_lnPgas
         
         interp_lo = max(1, nzlo-1)
         interp_hi = min(nz, nzhi+1)
         interp_n = interp_hi - interp_lo + 1
         
         if (dbg) then
            write(*,2) 'interp_lo', interp_lo
            write(*,2) 'interp_hi', interp_hi
            write(*,2) 'interp_n', interp_n
            write(*,2) 'enter do_lnR_and_lnd lnd_old(nz_old)/ln10', nz_old, lnd_old(nz_old)/ln10
         end if
         
         call do_alloc(ierr)
         if (ierr /= 0) return
         
         do k=1,nz_old
            Vol_old_plus1(k) = (pi4/3)*exp(3*xh_old(i_lnR,k))
         end do
         Vol_center = (pi4/3)*s% R_center**3
         Vol_old_plus1(nz_old+1) = Vol_center
         
         ! testing -- check for Vol_old_plus1 strictly decreasing         
         do k = 2, nz_old+1
            if (Vol_old_plus1(k) >= Vol_old_plus1(k-1)) then
               ierr = -1
               if (.not. dbg) then
                  call dealloc
                  return
               end if
               
               write(*,3) 'bad old vol', k, nz_old
               write(*,1) 'Vol_old_plus1(k)', Vol_old_plus1(k)
               write(*,1) 'Vol_old_plus1(k-1)', Vol_old_plus1(k-1)
               write(*,2) 'lnr_old', k, xh_old(i_lnR,k)
               write(*,2) 'lnr_old', k+1, xh_old(i_lnR,k+1)
               write(*,2) 'dlnr_old', k, xh_old(i_lnR,k) - xh_old(i_lnR,k+1)
               write(*,2) 'r_old', k, exp(xh_old(i_lnR,k))
               write(*,2) 'r_old', k+1, exp(xh_old(i_lnR,k+1))
               write(*,1) 's% R_center', s% R_center
               write(*,2) 'r_old - s% R_center', k, exp(xh_old(i_lnR,k)) - s% R_center
               write(*,1) 'r(nz_old)', exp(xh_old(i_lnR,nz_old))
               write(*,2) 'diff r_old', k, exp(xh_old(i_lnR,k)) - exp(xh_old(i_lnR,k+1))
               write(*,*)
               stop 'debug: mesh adjust: do_lnR_and_lnd'
            end if
         end do
         
         ! testing -- check for q strictly decreasing
         do k = 2, nz
            if (xq(k) <= xq(k-1)) then
               ierr = -1
               if (.not. dbg) return
               
               write(*,3) 'bad xq', k, nz, xq(k), xq(k-1)
               stop 'debug: mesh adjust: do_lnR_and_lnd'
            end if
         end do
         
         do k=1,interp_n
            interp_xq(k) = xq(interp_lo+k-1)
         end do

         call interpolate_vector( &
               nz_old+1, xq_old_plus1, interp_n, interp_xq, Vol_old_plus1, &
               interp_Vol_new, interp_pm, nwork, work, ierr)
         if (ierr /= 0) then
            call dealloc
            if (.not. dbg) return
            write(*,*) 'failed in interpolate_vector'
            stop 'debug: mesh_adjust'
         end if
         
         do k=1,interp_n
            Vol_new(interp_lo+k-1) = interp_Vol_new(k)
         end do
         
         if (dbg) then
            write(*,1) 'xq_old', xq_old_plus1(971:976)
            write(*,1) 'dq_old', dq_old(971:976)
            write(*,1) 'xq_new', xq(1012:1017)
            write(*,1) 'dq_new', dq(1012:1017)
            write(*,1) 'Vol_old', Vol_old_plus1(971:976)
            write(*,1) 'Vol_new', Vol_new(1012:1017)
            write(*,*)
            write(*,3) 'new comes from', 1012, comes_from(1012)
            write(*,3) 'new comes from', 1013, comes_from(1013)
            write(*,3) 'new comes from', 1014, comes_from(1014)
            write(*,3) 'new comes from', 1015, comes_from(1015)
            write(*,3) 'new comes from', 1016, comes_from(1016)
            write(*,3) 'new comes from', 1017, comes_from(1017)
            write(*,*)
            stop
         end if
         
         if (Vol_new(interp_lo+1) >= Vol_new(interp_lo)) then
            Vol_new(interp_lo+1) = (Vol_new(interp_lo) + Vol_new(interp_lo+2))/2
            if (dbg) write(*,2) 'fix Vol_new at lo+1', interp_lo+1, Vol_new(interp_lo+1)
            if (Vol_new(interp_lo+1) >= Vol_new(interp_lo)) then
               ierr = -1; call dealloc
               if (.not. dbg) return
               write(*,*) '(Vol_new(interp_lo+1) >= Vol_new(interp_lo))'
               stop 'debug: mesh_adjust'
            end if
         end if
         
         hint = 0
         do k = interp_lo+1, interp_hi-1
            if (Vol_new(k+1) >= Vol_new(k) .or. Vol_new(k) >= Vol_new(k-1)) then
               if (dbg) write(*,2) 'fix interpolated Vol_new', &
                  k, Vol_new(k+1), Vol_new(k), Vol_new(k-1)
!               k_old = binary_search(nz_old+1, Vol_old_plus1, hint, Vol_new(k))
!               if (k_old < 1 .or. k_old > nz_old) then
!                  ierr = -1
!                  if (.not. dbg) return
!                  write(*,1) 'Vol_min', Vol_min
!                  write(*,1) 'Vol_max', Vol_max
!                  stop 'debug: mesh_adjust'
!               end if
               Vol_min = minval(Vol_new(k-1:k+1))
               Vol_max = maxval(Vol_new(k-1:k+1))
               if (Vol_min == Vol_max .or. is_bad_num(Vol_min) .or. is_bad_num(Vol_max)) then
                  ierr = -1
                  if (.not. dbg) return
                  write(*,1) 'Vol_min', Vol_min
                  write(*,1) 'Vol_max', Vol_max
                  stop 'debug: mesh_adjust'
               end if
               Vm1 = Vol_new(k-1)
               V00 = Vol_new(k)
               Vp1 = Vol_new(k+1)
               Vol_new(k-1) = Vol_max
               Vol_new(k) = (Vol_max + Vol_min)/2
               Vol_new(k+1) = Vol_min
               if (dbg) write(*,2) 'new Vol_new',  &
                  k, Vol_new(k+1), Vol_new(k), Vol_new(k-1)
               if (Vol_new(k+1) >= Vol_new(k) .or. Vol_new(k) >= Vol_new(k-1)) then
                  ierr = -1
                  if (.not. dbg) return
                  
                  write(*,*)
                  write(*,*) 'new'
                  write(*,1) 'Vol_new(k-1)', Vol_new(k-1)
                  write(*,1) 'Vol_new(k)', Vol_new(k)
                  write(*,1) 'Vol_new(k+1)', Vol_new(k+1)
                  write(*,*)
                  write(*,*) 'Vol_new(k+1) >= Vol_new(k)', Vol_new(k+1) >= Vol_new(k)
                  write(*,*) 'Vol_new(k) >= Vol_new(k-1)', Vol_new(k) >= Vol_new(k-1)
                  write(*,*)
                  write(*,3) 'cell_type(k+1)', k+1, cell_type(k+1)
                  write(*,3) 'cell_type(k)', k, cell_type(k)
                  write(*,3) 'cell_type(k-1)', k-1, cell_type(k-1)
                  write(*,*)
                  write(*,3) 'comes_from(k+1)', k+1, comes_from(k+1)
                  write(*,3) 'comes_from(k)', k, comes_from(k)
                  write(*,3) 'comes_from(k-1)', k-1, comes_from(k-1)
                  write(*,*)
                  write(*,2) 'Vol_old_plus1(comes_from(k)+1)', &
                     comes_from(k+1), Vol_old_plus1(comes_from(k)+1)
                  write(*,2) 'Vol_old_plus1(comes_from(k))', &
                     comes_from(k), Vol_old_plus1(comes_from(k))
                  write(*,2) 'Vol_old_plus1(comes_from(k)-1)', &
                     comes_from(k-1), Vol_old_plus1(comes_from(k)-1)
                  write(*,*)
                  write(*,2) 'lnr_old', comes_from(k)+1, xh_old(i_lnR,comes_from(k)+1)
                  write(*,2) 'lnr_old', comes_from(k), xh_old(i_lnR,comes_from(k))
                  write(*,2) 'lnr_old', comes_from(k)-1, xh_old(i_lnR,comes_from(k)-1)
                  write(*,*)
                  write(*,2) 'dlnr_old', comes_from(k), &
                     xh_old(i_lnR,comes_from(k)) - xh_old(i_lnR,comes_from(k)+1)
                  write(*,2) 'dlnr_old', comes_from(k)-1, &
                     xh_old(i_lnR,comes_from(k)-1) - xh_old(i_lnR,comes_from(k))
                  write(*,*)
                  write(*,2) 'Vp1', k+1, Vp1
                  write(*,2) 'V00', k, V00
                  write(*,2) 'Vm1', k-1, Vm1
                  write(*,*)
                  write(*,2) 'dV', k, Vm1 - V00
                  write(*,2) 'dV', k+1, V00 - Vp1
                  write(*,*)
                  write(*,2) 'r_old', comes_from(k)+1, exp(xh_old(i_lnR,comes_from(k)+1))
                  write(*,2) 'r_old', comes_from(k), exp(xh_old(i_lnR,comes_from(k)))
                  write(*,2) 'r_old', comes_from(k)-1, exp(xh_old(i_lnR,comes_from(k)-1))
                  write(*,*)
                  write(*,2) 'dr_old', comes_from(k)-1, &
                     exp(xh_old(i_lnR,comes_from(k)-1)) - exp(xh_old(i_lnR,comes_from(k)))
                  write(*,2) 'dr_old', comes_from(k), &
                     exp(xh_old(i_lnR,comes_from(k))) - exp(xh_old(i_lnR,comes_from(k)+1))
                  write(*,*)
                  do kk=1, nz
                     if (comes_from(kk) == 626) &
                        write(*,3) 'comes_from(kk)', kk, comes_from(kk), &
                        Vol_new(kk) - Vol_new(kk+1), Vol_new(kk)
                  end do
                  write(*,*)
                  stop 'debug: do_lnR_and_lnd in mesh adjust: interpolation gave non-pos volume'
               end if
            end if
         end do
         
         xh(i_lnR,nzlo:min(nz,nzhi+1)) = -1d99
!x$OMP PARALLEL DO PRIVATE(k, cell_Vol)  >>> OMP makes it worse
         do k = nzlo, min(nzhi,nz-1)
            if (ierr /= 0) cycle
            
            if (xh(i_lnR,k) < -1d90) call set1(k)
            if (xh(i_lnR,k+1) < -1d90) call set1(k+1)
            
            if (cell_type(k) == unchanged_type) then
               if (i_xlnd /= 0) then
                  xh(i_xlnd,k) = lnd_old(comes_from(k))
               else if (i_lnPgas /= 0) then
                  xh(i_lnPgas,k) = lnPgas_old(comes_from(k))
               else
                  stop 'debug: do_lnR_and_lnd in mesh adjust: need either lnd or lnPgas'
               end if
               density_new(k) = exp(lnd_old(comes_from(k)))
               cycle
            end if

            if (xh(i_lnR,k) <= xh(i_lnR,k+1)) then
               if (dbg) then
                  write(*,*)
                  write(*,2) 'k', k
                  write(*,1) 'R(k)/Rsun', exp(xh(i_lnR,k))/Rsun
                  write(*,1) 'Vol_new(k)', Vol_new(k)
                  write(*,1) 'Vol_new(k+1)', Vol_new(k+1)
                  write(*,1) 'cell_Vol', Vol_new(k)-Vol_new(k+1)
                  write(*,1) 'dq(k)', dq(k)
                  write(*,1) 'xq(k)', xq(k)
                  write(*,1) 'xq(k+1)', xq(k+1)
                  write(*,2) 'nz', nz
                  write(*,2) 'comes_from(k)', comes_from(k)
                  write(*,2) 'comes_from(k+1)', comes_from(k+1)
                  write(*,2) 'vol_old', comes_from(k), Vol_old_plus1(comes_from(k))
                  write(*,2) 'vol_old', comes_from(k)+1, Vol_old_plus1(comes_from(k)+1)
                  write(*,2) 'cell_vol_old', comes_from(k), &
                     Vol_old_plus1(comes_from(k)) - Vol_old_plus1(comes_from(k)+1)
                  write(*,*)
                  write(*,1) 'xh(i_lnR,k)', xh(i_lnR,k)
                  write(*,1) 'xh(i_lnR,k+1)', xh(i_lnR,k+1)
                  write(*,*)
                  write(*,*) 'cell_type(k) == unchanged_type', cell_type(k) == unchanged_type
                  write(*,*) 'cell_type(k+1) == unchanged_type', cell_type(k+1) == unchanged_type
                  write(*,*)
                  write(*,*) 'do_lnR_and_lnd: xh(i_lnR,k) <= xh(i_lnR,k+1)'
                  stop
               end if
               ierr = -1; cycle
            end if
            
            cell_Vol = Vol_new(k)-Vol_new(k+1)
            if (cell_Vol <= 0) then
               if (dbg) then
                  write(*,1) 'cell_Vol', cell_Vol
                  write(*,1) 'Vol_new(k)', Vol_new(k)
                  write(*,1) 'Vol_new(k+1)', Vol_new(k+1)
                  write(*,*) 'k', k
                  write(*,*) 'do_lnR_and_lnd: cell_Vol <= 0'
               end if
               ierr = -1; cycle
            end if
            if (dq(k) <= 0) then
               if (dbg) then
                  write(*,1) 'dq(k)', dq(k)
                  write(*,*) 'k', k
                  write(*,*) 'do_lnR_and_lnd: dq(k) <= 0'
               end if
               ierr = -1; cycle
            end if
            density_new(k) = xmstar*dq(k)/cell_Vol
            if (i_xlnd /= 0) then
               xh(i_xlnd,k) = log(density_new(k))
            else
               xh(i_lnPgas,k) = -1d99 ! fix later after set T
               !write(*,2) 'need to fix lnPgas', k
            end if
            
         end do
!x$OMP END PARALLEL DO

         if (ierr /= 0) then
            if (.not. dbg) return
            
            stop 'debug: failed in mesh adjust do_lnR_and_lnd'
         end if
         
         n = nzlo - 1
         if (n > 0) then
            if (i_xlnd /= 0) then
               do k=1,n
                  xh(i_xlnd,k) = lnd_old(k)
               end do
            else
               do k=1,n
                  xh(i_lnPgas,k) = lnPgas_old(k)
               end do
            end if
            do k=1,n
               xh(i_lnR,k) = xh_old(i_lnR,k)
               density_new(k) = exp(lnd_old(k))
            end do
         end if
         
         if (nzhi < nz) then
            n = nz - nzhi - 1 ! nz-n = nzhi+1
            if (i_xlnd /= 0) then
               do k=0,n
                  xh(i_xlnd,nz-k) = lnd_old(nz_old-k)
               end do
            else
               do k=0,n
                  xh(i_lnPgas,nz-k) = lnPgas_old(nz_old-k)
               end do
            end if
            do k=0,n
               xh(i_lnR,nz-k) = xh_old(i_lnR,nz_old-k)
               density_new(nz-k) = exp(lnd_old(nz_old-k))
            end do
         else ! nzhi == nz
            density_new(nz) = xmstar*dq(nz) / (Vol_new(nz) - Vol_center)
            if (i_xlnd /= 0) then
               xh(i_xlnd,nz) = log(density_new(nz))
            else
               xh(i_lnPgas,nz) = -1d99 ! fix later after set T
            end if
            xh(i_lnR,nz) = log(Vol_new(nz)/(pi4/3))/3
         end if

         call dealloc
         
         contains
         
         subroutine do_alloc(ierr)
            use alloc
            use utils_lib
            integer, intent(out) :: ierr
            call non_crit_get_work_array(s, Vol_old_plus1, nz_old+1, nz_alloc_extra, 'mesh_adjust', ierr)
            if (ierr /= 0) return
            call non_crit_get_work_array(s, Vol_new, nz, nz_alloc_extra, 'mesh_adjust', ierr)
            if (ierr /= 0) return
            call non_crit_get_work_array(s, interp_xq, interp_n, nz_alloc_extra, 'mesh_adjust', ierr)
            if (ierr /= 0) return
            call non_crit_get_work_array(s, interp_Vol_new, interp_n, nz_alloc_extra, 'mesh_adjust', ierr)
            if (ierr /= 0) return
            call non_crit_get_work_array(s, work, (nz_old+1)*nwork, nz_alloc_extra, 'mesh_adjust', ierr)
            if (ierr /= 0) return
         end subroutine do_alloc
         
         subroutine dealloc
            use alloc
            use utils_lib
            call non_crit_return_work_array(s, Vol_old_plus1, 'mesh_adjust')            
            call non_crit_return_work_array(s, Vol_new, 'mesh_adjust')            
            call non_crit_return_work_array(s, interp_xq, 'mesh_adjust')            
            call non_crit_return_work_array(s, interp_Vol_new, 'mesh_adjust')            
            call non_crit_return_work_array(s, work, 'mesh_adjust')
         end subroutine dealloc
         
         subroutine set1(k)
            integer, intent(in) :: k
            include 'formats'
            if (cell_type(k) == unchanged_type) then
               xh(i_lnR,k) = xh_old(i_lnR,comes_from(k))
            else
               xh(i_lnR,k) = log(Vol_new(k)/(pi4/3))/3
            end if
         end subroutine set1
         

      end subroutine do_lnR_and_lnd
      
      
      real(dp) function get_cell_dq(k, nz, dq)
         integer, intent(in) :: k, nz
         real(dp), intent(in) :: dq(:)
         if (k == nz) then
            get_cell_dq = dq(k) + dq(k-1)/2
         else if (k == 1) then
            get_cell_dq = dq(1)/2
         else
            get_cell_dq = (dq(k) + dq(k-1))/2
         end if
      end function get_cell_dq
      
      
      subroutine do_xa( &
            s, nz, nz_old, k, species, cell_type, comes_from, &
            xa, xa_old, xa_c0, xa_c1, xa_c2, &
            xq, dq, xq_old,  dq_old, old_loc, mesh_adjust_use_quadratic, ierr)
         use chem_def, only: chem_isos
         type (star_info), pointer :: s
         integer, intent(in) :: nz, nz_old, species, k, cell_type(:), comes_from(:)
         real(dp), dimension(:,:), pointer :: xa, xa_old, xa_c0, xa_c1, xa_c2
         real(dp), dimension(:), pointer :: xq, dq, xq_old,  dq_old
         integer, pointer :: old_loc(:)
         logical, intent(in) :: mesh_adjust_use_quadratic
         integer, intent(out) :: ierr
         
         integer :: j, jj, k_old, k_old_last, kdbg
         real(dp) :: xq_outer, cell_dq, xa_sum, total(species)
         logical :: dbg_get_integral
            
         include 'formats'
         
         ierr = 0
         
         kdbg = -1074
         
         if (cell_type(k) == unchanged_type) then
            do j=1,species
               xa(j,k) = xa_old(j,comes_from(k))
            end do
            return
         end if
         
         xq_outer = xq(k)
         if (k == nz) then
            cell_dq = 1 - xq_outer
         else
            cell_dq = dq(k)
         end if
         
         k_old = old_loc(k)
         
         ! sum the old abundances between xq_outer and xq_inner
         dbg_get_integral = .false.
         total(:) = 0
         do j=1,species
            dbg_get_integral = (k == kdbg) .and. (j == 1) ! h1
            if (dbg_get_integral) write(*,2) trim(chem_isos% name(s% chem_id(j)))
            call get_xq_integral( &
               k_old, nz_old, xq_old, xq_outer, cell_dq, &
               mesh_adjust_use_quadratic, xa_c0(:,j), xa_c1(:,j), xa_c2(:,j), &
               total(j), dbg_get_integral, k_old_last, ierr)
         end do
         
         xa(:,k) = total(:)/cell_dq
         
         if (k == kdbg) then
            do j=1,species
               write(*,2) 'new ' // trim(chem_isos% name(s% chem_id(j))), k, xa(j,k)
            end do
         end if
         
         do j=1,species
            if (xa(j,k) > 1 + 1d-8 .or. xa(j,k) < -1d-8) then
               ierr = -1
               return
               
               do jj=1,species
                  write(*,1) 'xa ' // trim(chem_isos% name(s% chem_id(jj))), xa(jj,k)
               end do
               write(*,*)
               write(*,2) 'sum xa', k, sum(xa(:,k))
               write(*,*)
               write(*,2) 'xa ' // trim(chem_isos% name(s% chem_id(j))), k, xa(j,k)
               write(*,*)
               write(*,2) 'xq_outer', k, xq_outer
               write(*,2) 'xq_inner', k, xq_outer + cell_dq
               write(*,2) 'cell_dq', k, cell_dq
               write(*,*)
               write(*,2) 'xq_old(k_old)', k_old, xq_old(k_old)
               write(*,2) 'xq_inner(k_old)', k_old, xq_old(k_old)+dq_old(k_old)
               write(*,2) 'dq_old(k_old)', k_old, dq_old(k_old)
               write(*,*)
               write(*,2) 'xa_c0(k_old,j)', k_old, xa_c0(k_old,j)
               write(*,2) 'xa_c1(k_old,j)', k_old, xa_c1(k_old,j)
               write(*,2) 'xa_c2(k_old,j)', k_old, xa_c2(k_old,j)
               write(*,*)
               write(*,2) 'old outer', k_old, xa_c0(k_old,j) + xa_c1(k_old,j)*dq_old(k_old)/2
               write(*,2) 'old inner', k_old, xa_c0(k_old,j) - xa_c1(k_old,j)*dq_old(k_old)/2
               write(*,*)
               stop 'debug: mesh adjust: do_xa'
            end if
         end do
         
         xa_sum = sum(xa(:,k))
         !write(*,1) 'xa_sum', xa_sum
         
         if (is_bad_num(xa_sum)) then
            ierr = -1
            return
            
            write(*,*) 'xa_sum', xa_sum
            write(*,*) 'bug in revise mesh, do_xa bad num: k', k
            stop 'debug: mesh adjust: do_xa'
         end if
         
         if (abs(1-xa_sum) > 1d-3) then
            ierr = -1
            return
            
            write(*,*) '(abs(1-xa_sum) > 1d-3)'
            write(*,1) '1-xa_sum', 1-xa_sum
            write(*,*) 'bug in revise mesh xa_sum, do_xa: k', k
            write(*,*) ' k_old', k_old
            write(*,*) ' k_old_last', k_old_last
            write(*,*) 'nz_old', nz_old
            
            write(*,*)
            do j=1,species
               write(*,1) 'xa ' // trim(chem_isos% name(s% chem_id(j))), xa(j,k)
            end do
            write(*,*)
            write(*,1) 'xq(k)', xq(k)
            if (k < nz) write(*,1) 'xq(k+1)', xq(k+1)
            write(*,1) '1-xq(k)', 1-xq(k)
            if (k < nz) write(*,1) '1-xq(k+1)', 1-xq(k+1)
            write(*,*)
            write(*,1) 'xq_old(k_old)', xq_old(k_old)
            write(*,1) 'xq_old(k_old)+cell_dq', xq_old(k_old)+cell_dq
            write(*,1) '1-xq_old(k_old)', 1-xq_old(k_old)
            write(*,1) '1-(xq_old(k_old)+cell_dq)', 1-(xq_old(k_old)+cell_dq)
            write(*,1) 'cell_dq', cell_dq
            if (k_old < nz_old) then
               write(*,1) 'xq_old(k_old+1)', xq_old(k_old+1)
               if (k_old+1 < nz_old) write(*,1) 'xq_old(k_old+2)', xq_old(k_old+2)
               write(*,1) '1-xq_old(k_old+1)', 1-xq_old(k_old+1)
               if (k_old+1 < nz_old) write(*,1) '1-xq_old(k_old+2)', 1-xq_old(k_old+2)
            end if
            write(*,*)
            write(*,1) 'cell_dq', cell_dq
            write(*,1) 'dq_old(k_old)', dq_old(k_old)
            if (k_old < nz_old) write(*,1) 'dq_old(k_old+1)', dq_old(k_old+1)
            write(*,*)
            
            write(*,2) 'sum xa_old', k_old, sum(xa_old(:,k_old))
            write(*,2) 'sum xa_old', k_old+1, sum(xa_old(:,k_old+1))
            
            write(*,*)
            stop 'debug: mesh adjust: do_xa, bad xa_sum'
         end if
         
         xa(:,k) = xa(:,k) / xa_sum

      end subroutine do_xa
      
      
      subroutine do_lnT( &
            s, nz_old, k, &
            species, cell_type, comes_from, &
            xa, xh, xh_old, &
            xq, dq, xq_old, eta_old, &
            lnT_c0, lnT_c1, lnT_c2, &
            energy_c0, energy_c1, energy_c2, old_loc, density_new, ierr)
         use eos_def
         type (star_info), pointer :: s
         integer, intent(in) :: nz_old, k, species, cell_type(:), comes_from(:)
         integer, pointer, dimension(:) :: old_loc
         real(dp), dimension(:,:), pointer :: xa, xh, xh_old
         real(dp), dimension(:), pointer :: &
            xq, dq, xq_old,  eta_old, density_new, &
            lnT_c0, lnT_c1, lnT_c2, &
            energy_c0, energy_c1, energy_c2
         integer, intent(out) :: ierr
         
         integer :: k_old, k_old_last, i_lnT
         real(dp) :: &
            Rho, logRho, xq_outer, cell_dq, avg_energy, &
            sum_lnT, avg_lnT, new_lnT, sum_energy, new_xa(species)
         logical :: dbg_get_integral
         
         include 'formats'
         
         ierr = 0
         i_lnT = s% i_lnT
         new_xa(:) = xa(:,k)
         
         if (cell_type(k) == unchanged_type) then
            k_old = comes_from(k)
            xh(i_lnT, k) = xh_old(i_lnT, k_old)
            call fix_lnPgas(ierr)
            return
         end if
         
         xq_outer = xq(k)
         cell_dq = dq(k)         
         k_old = old_loc(k)
         
         ! find average lnT between q_outer and q_inner
         dbg_get_integral = .false.
         call get_xq_integral( &
            k_old, nz_old, xq_old, xq_outer, cell_dq, &
            s% mesh_adjust_use_quadratic, lnT_c0, lnT_c1, lnT_c2, sum_lnT, dbg_get_integral, &
            k_old_last, ierr)
         if (ierr /= 0) then
            if (dbg) write(*,*) 'get_xq_integral failed for average lnT'
            return
         end if
         avg_lnT = sum_lnT/cell_dq

         if (is_bad_num(avg_lnT) .or. avg_lnT < 0 .or. avg_lnT > 100) then
            ierr = -1
            return
            
            write(*,2) 'avg_lnT', k_old, avg_lnT
            write(*,*) 'bug in revise mesh, do_lnT'
            stop 'debug: mesh adjust: do_lnT'
         end if

         if (.not. s% mesh_adjust_get_T_from_E) then
            xh(i_lnT, k) = avg_lnT    
            call fix_lnPgas(ierr)        
            return
         end if

         if (eta_old(k_old) >= eta_limit) then
            xh(i_lnT, k) = avg_lnT     
            call fix_lnPgas(ierr)       
            return
         end if
         
         if (dbg) write(*,2) 'eta_old(k_old)', k_old, eta_old(k_old)
         
         ! find average internal energy between q_outer and q_inner
         dbg_get_integral = .false.
         call get_xq_integral( &
            k_old, nz_old, xq_old, xq_outer, cell_dq, &
            s% mesh_adjust_use_quadratic, energy_c0, energy_c1, energy_c2, &
            sum_energy, dbg_get_integral, k_old_last, ierr)
         if (ierr /= 0) then
            if (dbg) write(*,*) 'get_xq_integral failed for average internal energy'
            if (.not. dbg) return
            stop 'debug: mesh adjust: do_lnT'
         end if
         avg_energy = sum_energy/cell_dq

         if (is_bad_num(avg_energy)) then
            ierr = -1
            if (dbg) write(*,*) 'is_bad_num(avg_energy)'
            if (.not. dbg) return
            
            write(*,*) 'avg_energy', avg_energy
            write(*,*) 'bug in revise mesh, do_lnT'
            stop 'debug: mesh adjust: do_lnT'
         end if
         
         ! call eos to calculate lnT from new internal energy
         
         Rho = density_new(k)
         logRho = log10(Rho)
         call set_lnT_for_energy( &
            s, k, &
            s% net_iso(ih1), s% net_iso(ihe3), s% net_iso(ihe4), species, new_xa, &
            Rho, logRho, avg_energy, avg_lnT, new_lnT, ierr)
         
         if (ierr /= 0) then
            if (dbg) write(*,*) 'set_lnT_for_energy failed', k
            xh(i_lnT, k) = avg_lnT    
            ierr = 0    
            call fix_lnPgas(ierr)    
            if (.not. dbg) return
            
            write(*,*) 'do_lnT ierr for k', k
            stop 'debug: failed in revise mesh do_lnT'
         end if
         
         xh(i_lnT,k) = new_lnT
         call fix_lnPgas(ierr)
         
         
         contains
         
         
         subroutine fix_lnPgas(ierr)
            use chem_lib, only: basic_composition_info
            use micro, only: eval_lnPgas
            use eos_def
            integer, intent(out) :: ierr
            
            real(dp) :: &
               X, Y, Z, rho, lnd, T, lnT, lnPgas, &
               abar, zbar, z2bar, mass_correction, ye, sumx
            
            include 'formats'
            
            ierr = 0
            if (i_lnPgas == 0) return
            if (xh(i_lnPgas,k) > -1d90) return
            ! use xh(i_lnT, k) and new_density(k) to calculate xh(i_lnPgas,k)
         
            call basic_composition_info(species, s% chem_id, new_xa(:), X, Y, &
               abar, zbar, z2bar, ye, mass_correction, sumx)  
            Z = 1 - (X + Y)
            
            rho = density_new(k)
            lnd = log(rho)
            
            lnT = xh(i_lnT,k)
            T = exp(lnT)
            
            if (dbg .and. k == 803) then
               write(*,2) 'call eval_lnPgas', k
               write(*,1) 'T', T
               write(*,1) 'lnT/ln10', lnT/ln10
               write(*,1) 'Rho', Rho
               write(*,1) 'lnd/ln10', lnd/ln10
               write(*,1) 'Z', Z
               write(*,1) 'X', X
               write(*,1) 'Y', Y
               write(*,1) 'abar', abar
               write(*,1) 'zbar', zbar
            end if
            
            call eval_lnPgas( &
               s, z, x, new_xa, abar, zbar, T, lnT, Rho, lnd, &
               .false., lnPgas, ierr)
            
            if (dbg .and. ierr /= 0) then
               write(*,2) 'ierr', ierr
               write(*,2) 'done fix_lnPgas', k, lnPgas/ln10
               write(*,*)
               stop 'fix_lnPgas'
            end if
               
            xh(i_lnPgas,k) = lnPgas
               
               
         end subroutine fix_lnPgas
         

      end subroutine do_lnT


      subroutine set_lnT_for_energy( &
            s, k, &
            h1, he3, he4, species, xa, &
            Rho, logRho, energy, lnT_guess, lnT, ierr)
         use eos_lib, only: eosDT_get_T
         use eos_def
         use chem_lib, only: basic_composition_info
         type (star_info), pointer :: s
         integer, intent(in) :: k, h1, he3, he4, species
         real(dp), intent(in)  :: xa(species), Rho, logRho, energy, lnT_guess
         real(dp), intent(out)  :: lnT
         integer, intent(out) :: ierr
         
         real(dp) :: &
               X, Y, Z, logT_tol, other_tol, other, &
               logT_guess, logT_bnd1, logT_bnd2, other_at_bnd1, other_at_bnd2, &
               logT_result, res(num_eos_basic_results), d_dlnd(num_eos_basic_results), &
               d_dlnT(num_eos_basic_results), helm_res(num_helm_results), &
               d_dabar(num_eos_basic_results), d_dzbar(num_eos_basic_results), &
               abar, zbar, z2bar, ye, mass_correction, sumx
         integer:: which_other, max_iter, eos_calls
         
         integer, parameter :: k_dbg = -1

         include 'formats'
         
         ierr = 0
         
         call basic_composition_info(species, s% chem_id, xa(:), X, Y, &
               abar, zbar, z2bar, ye, mass_correction, sumx)  
         
         which_other = i_lnE
         other = log(energy)
         other_tol = 1d-6*ln10
         
         max_iter = 100
         Z = max(0d0, min(1d0, 1d0 - (X + Y)))

         logT_tol = 1d-6
         logT_guess = lnT_guess/ln10

         logT_bnd1 = arg_not_provided! 3d0
         other_at_bnd1 = arg_not_provided! 5d10
         logT_bnd2 = arg_not_provided! 10d0
         other_at_bnd2 = arg_not_provided! 2d27
         
         ! use density and energy to get temperature
         if (s% use_other_eos) then
            call s% other_eosDT_get_T( &
               s% id, k, s% eos_handle, Z, X, abar, zbar, &
               species, s% chem_id, s% net_iso, xa, &
               logRho, which_other, other, &
               logT_tol, other_tol, max_iter, logT_guess, &
               logT_bnd1, logT_bnd2, other_at_bnd1, other_at_bnd2, &
               logT_result, res, d_dlnd, d_dlnT, d_dabar, d_dzbar, &
               eos_calls, ierr)
         else
            call eosDT_get_T( &
               s% eos_handle, Z, X, abar, zbar, &
               species, s% chem_id, s% net_iso, xa, &
               logRho, which_other, other, &
               logT_tol, other_tol, max_iter, logT_guess, &
               logT_bnd1, logT_bnd2, other_at_bnd1, other_at_bnd2, &
               logT_result, res, d_dlnd, d_dlnT, d_dabar, d_dzbar, &
               helm_res, eos_calls, ierr)
            if (ierr /= 0 .and. dbg) then
               write(*,1) 'eosDT_get_T ierr', ierr
               write(*,1) 'Z', Z
               write(*,1) 'X', X
               write(*,1) 'abar', abar
               write(*,1) 'zbar', zbar
               write(*,1) 'logRho', logRho
               write(*,1) 'logT_guess', logT_guess
               write(*,1) 's% eta(k)', s% eta(k)
               write(*,*)
            end if
         end if

         lnT = logT_result*ln10
         
         if (ierr /= 0 .or. is_bad_num(lnT)) then
            ierr = -1
            return
            
            call show
            stop 'debug: set_lnT_for_energy'
         end if
         
         contains
         
         subroutine show
            include 'formats'
            write(*,*)
            write(*,*) 'set_lnT_for_energy ierr', ierr
            write(*,*) 'k', k
            write(*,*)
            write(*,1) 'lnT =', lnT
            write(*,*)
            write(*,1) 'logRho =', logRho
            write(*,1) 'logT_guess =', logT_guess
            write(*,1) 'logT_bnd1 =', logT_bnd1
            write(*,1) 'logT_bnd2 =', logT_bnd2
            write(*,1) 'energy =', energy
            write(*,1) 'Z =', Z
            write(*,1) 'X =', X
            write(*,1) 'abar =', abar
            write(*,1) 'zbar =', zbar
            write(*,1) 'logT_tol =', logT_tol
            write(*,1) 'other_tol =', other_tol
            write(*,*)
            write(*,*)
         end subroutine show
      
      end subroutine set_lnT_for_energy
      
      
      ! Stiriba, Youssef, Appl, Numer. Math. 45, 499-511. 2003.
         
         ! LPP-HARMOD -- local piecewise parabolic reconstruction 
      
         ! interpolant is derived to conserve integral of v in cell k
         ! interpolant slope at cell midpoint is harmonic mean of slopes between adjacent cells
         ! where these slopes between cells are defined as the difference between cell averages
         ! divided by the distance between cell midpoints.
         ! interpolant curvature based on difference between the midpoint slope
         ! and the smaller in magnitude of the slopes between adjacent cells.

         ! interpolant f(dq) = a + b*dq + (c/2)*dq^2, with dq = q - q_midpoint
         ! c0 holds a's, c1 holds b's, and c2 holds c's.


      subroutine get1_lpp(k, ldv, nz, j, dq, v, quad, c0, c1, c2)
         integer, intent(in) :: k, ldv, nz, j
         real(dp), intent(in) :: dq(:) ! (nz)
         real(dp), intent(in) :: v(:,:) ! (ldv,nz)
         logical, intent(in) :: quad
         real(dp), dimension(:) :: c0, c1, c2
         
         real(dp) :: vbdy1, vbdy2, dqhalf, sm1, s00, sprod
         real(dp), parameter :: rel_curvature_limit = 0.1d0
         
         logical :: dbg
         
         include 'formats'
         
         if (k == 1 .or. k == nz) then
            call set_const
            return
         end if
         
         dbg = .false.
         !dbg = (k == 30 .and. j == 3) ! .false.
         
         sm1 = (v(j,k-1) - v(j,k)) / ((dq(k-1) + dq(k))/2)
         s00 = (v(j,k) - v(j,k+1)) / ((dq(k) + dq(k+1))/2)
         
         sprod = sm1*s00
         if (sprod <= 0) then 
            ! at local min or max, so set slope and curvature to 0.
            call set_const
            return
         end if

         if (.not. quad) then
            c0(k) = v(j,k)
            c1(k) = (sm1 + s00)/2 ! use average to smooth abundance transitions
            c2(k) = 0 ! Yan Wang fixed this -- it was left out initially.
         else 
            c1(k) = sprod*2/(s00 + sm1) ! harmonic mean slope
            if (abs(sm1) <= abs(s00)) then
               c2(k) = (sm1 - c1(k))/(2*dq(k))
            else
               c2(k) = (c1(k) - s00)/(2*dq(k))
            end if
            c0(k) = v(j,k) - c2(k)*dq(k)**2/24
         end if
         
         ! check values at edges for monotonicity
         dqhalf = dq(k)/2
         vbdy1 = c0(k) + c1(k)*dqhalf + c2(k)/2*dqhalf**2 ! value at face(k)
         vbdy2 = c0(k) - c1(k)*dqhalf + c2(k)/2*dqhalf**2 ! value at face(k+1)
         if ((v(j,k-1) - vbdy1)*(vbdy1 - v(j,k)) < 0 .or. &
             (v(j,k) - vbdy2)*(vbdy2 - v(j,k+1)) < 0) then
            if (dbg) then
               write(*,*) 'non-monotonic'
               write(*,2) 'v(j,k-1)', k-1, v(j,k-1)
               write(*,2) 'vbdy1', k, vbdy1
               write(*,2) 'v(j,k)', k, v(j,k)
               write(*,2) 'vbdy2', k, vbdy2
               write(*,2) 'v(j,k+1)', k+1, v(j,k+1)
               write(*,*)
               write(*,2) 'v(j,k-1) - vbdy1', k, v(j,k-1) - vbdy1
               write(*,2) 'vbdy1 - v(j,k+1)', k, vbdy1 - v(j,k+1)
               write(*,*)
               write(*,2) 'v(j,k-1) - vbdy2', k, v(j,k-1) - vbdy2
               write(*,2) 'vbdy2 - v(j,k+1)', k, vbdy2 - v(j,k+1)
               write(*,*)
               write(*,2) 'sm1', k, sm1
               write(*,2) 's00', k, s00
               write(*,*)
               stop 'debug: get1_lpp'
            end if
            c2(k) = 0
            if (abs(sm1) <= abs(s00)) then
               c1(k) = sm1
            else
               c1(k) = s00
            end if
         end if
         
         contains
         
         subroutine set_const
            c0(k) = v(j,k)
            c1(k) = 0
            if (quad) c2(k) = 0
         end subroutine set_const
      
      end subroutine get1_lpp

      
      subroutine get_xq_integral( &
            k_old_in, nz_old, xq_old, xq_outer, dq, &
            quad, c0, c1, c2, integral, dbg, k_old_last, ierr)
         ! integrate val(j,:) from xq_inner to xq_outer, with xq_inner = xq_outer + dq
         integer, intent(in) :: k_old_in, nz_old
         real(dp), intent(in) :: xq_old(:), xq_outer, dq
         logical, intent(in) :: quad
         real(dp), intent(in), dimension(:) :: c0, c1, c2 ! coefficients
         real(dp), intent(out) :: integral
         logical, intent(in) :: dbg
         integer, intent(out) :: k_old_last, ierr
         
         integer :: k, k_old
         real(dp) :: a, b, c, old_xq_inner, old_xq_outer, xq_inner, &
            xq_overlap_outer, xq_overlap_inner, dq1, sum_dqs, old_xq_mid, &
            v_overlap_outer, v_overlap_inner, dq_outer, dq_inner, avg_value
            
         include 'formats'
         
         ierr = 0
         k_old = k_old_in
         ! move starting k_old if necessary
         do
            if (k_old <= 1) exit
            if (xq_old(k_old) <= xq_outer) exit
            k_old = k_old - 1
         end do
         xq_inner = xq_outer + dq
         old_xq_inner = xq_old(k_old)
         sum_dqs = 0
         integral = 0d0
         
         do k = k_old, nz_old
            if (dq <= sum_dqs) exit
            old_xq_outer = old_xq_inner
            if (k == nz_old) then
               old_xq_inner = 1
            else
               old_xq_inner = xq_old(k+1)
            end if
            xq_overlap_outer = max(xq_outer, old_xq_outer)
            xq_overlap_inner = min(xq_inner, old_xq_inner)
            
            if (dbg) then
               write(*,2) 'xq_overlap_outer', k, xq_overlap_outer
               write(*,2) 'xq_outer', k, xq_outer
               write(*,2) 'old_xq_outer', k, old_xq_outer
               write(*,2) 'xq_overlap_inner', k, xq_overlap_inner
               write(*,2) 'xq_inner', k, xq_inner
               write(*,2) 'old_xq_inner', k, old_xq_inner
            end if

            if (sum_dqs == 0 .and. xq_overlap_outer == xq_outer .and. xq_overlap_inner == xq_inner) then 
               ! fully contained
               !if (dbg .and. k_old == 29 .and. j == 3) write(*,*) 'fully contained'
               xq_inner = xq_outer + dq
               xq_overlap_inner = xq_inner
               dq1 = dq
            else if (old_xq_inner >= xq_inner) then ! this is the last one
               dq1 = dq - sum_dqs
            else
               dq1 = max(0d0, xq_overlap_inner-xq_overlap_outer)
            end if
            sum_dqs = sum_dqs + dq1
            ! interpolant f(dq) = a + b*dq + (c/2)*dq^2, with dq = q - q_midpoint
            a = c0(k)
            b = c1(k)
            if (quad) then
               c = c2(k)
            else
               c = 0
            end if

            if (dq1 == 0 .or. (b==0 .and. c==0)) then
               avg_value = a
            else 
               old_xq_mid = (old_xq_outer + old_xq_inner)/2
               dq_outer = old_xq_mid - xq_overlap_outer
               dq_inner = old_xq_mid - xq_overlap_inner

               if (.false.) then ! debugging
                  avg_value = a
               else if (c == 0) then
                  ! use slope to estimate average value in the region being used
                  if (dbg) write(*,*) 'use slope to estimate average value'
                  v_overlap_outer = a + dq_outer*b
                  v_overlap_inner = a + dq_inner*b
                  avg_value = (v_overlap_outer + v_overlap_inner)/2
               else ! use quadratic reconstruction
                  if (dbg) write(*,*) 'use quadratic reconstruction'
                  avg_value = &
                     a + b*(dq_inner + dq_outer)/2 + &
                        c*(dq_inner**2 + dq_inner*dq_outer + dq_outer**2)/6
               end if
            end if
            integral = integral + dq1*avg_value
            if (dbg) then
               write(*,2) 'a', k, a
               write(*,2) 'b', k, b
               write(*,2) 'c', k, c
               write(*,2) 'dq1', k, dq1
               write(*,2) 'avg_value', k, avg_value
               write(*,2) 'integral', k, integral
               write(*,*)
            end if
            k_old_last = k
            if (old_xq_inner >= xq_inner) exit ! this is the last one

         end do
      
      end subroutine get_xq_integral

      
      subroutine adjust_omega(s, nz, nz_old, comes_from, &
            old_xq, new_xq, old_dq, new_dq, xh, old_j_rot, ierr)
         use alloc
         type (star_info), pointer :: s
         integer, intent(in) :: nz, nz_old
         integer, dimension(:), pointer :: comes_from
         real(dp), dimension(:), intent(in) :: &
            old_xq, new_xq, old_dq, new_dq, old_j_rot
         real(dp), intent(in) :: xh(:,:)
         integer, intent(out) :: ierr         
         integer :: k, op_err, old_k, new_k
         real(dp) :: old_j_tot, new_j_tot
			real(dp), pointer, dimension(:) :: &
			   old_xout, new_xout, old_dqbar, new_dqbar
         include 'formats.dek'       
         ierr = 0
         
         call non_crit_get_work_array(s, old_xout, nz_old, nz_alloc_extra, 'adjust_omega', ierr)
         if (ierr /= 0) return
         call non_crit_get_work_array(s, old_dqbar, nz_old, nz_alloc_extra, 'adjust_omega', ierr)
         if (ierr /= 0) return
         call non_crit_get_work_array(s, new_xout, nz, nz_alloc_extra, 'adjust_omega', ierr)
         if (ierr /= 0) return
         call non_crit_get_work_array(s, new_dqbar, nz, nz_alloc_extra, 'adjust_omega', ierr)
         if (ierr /= 0) return
			   
			old_xout(1) = old_xq(1)
			old_dqbar(1) = old_dq(1)/2
			do k=2,nz_old
				old_xout(k) = old_xout(k-1) + old_dqbar(k-1)
				old_dqbar(k) = (old_dq(k-1) + old_dq(k))/2
			end do
			old_dqbar(nz_old) = old_dq(nz_old-1)/2 + old_dq(nz_old)
			
			new_xout(1) = new_xq(1)
			new_dqbar(1) = new_dq(1)/2
			do k=2,nz
				new_xout(k) = new_xout(k-1) + new_dqbar(k-1)
				new_dqbar(k) = (new_dq(k-1) + new_dq(k))/2
			end do
			new_dqbar(nz) = new_dq(nz-1)/2 + new_dq(nz)
			
!$OMP PARALLEL DO PRIVATE(k, op_err)
         do k = 1, nz
            op_err = 0
            call adjust1_omega(s, k, nz, nz_old, comes_from, &
               old_xout, new_xout, old_dqbar, new_dqbar, old_j_rot, xh, op_err)
            if (op_err /= 0) ierr = op_err
         end do
!$OMP END PARALLEL DO

         call non_crit_return_work_array(s, old_xout, 'adjust_omega')
         call non_crit_return_work_array(s, old_dqbar, 'adjust_omega')
         call non_crit_return_work_array(s, new_xout, 'adjust_omega')
         call non_crit_return_work_array(s, new_dqbar, 'adjust_omega')
			
      end subroutine adjust_omega

      
      subroutine adjust1_omega(s, k, nz, nz_old, comes_from, &
            old_xout, new_xout, old_dqbar, new_dqbar, old_j_rot, xh, ierr)
         ! set new value for s% omega(k)
         use utils_lib, only: is_bad_num
         type (star_info), pointer :: s
         integer, intent(in) :: k, nz, nz_old
         integer, dimension(:), pointer :: comes_from
         real(dp), dimension(:), intent(in) :: &
            old_xout, new_xout, old_dqbar, new_dqbar, old_j_rot
         real(dp), intent(in) :: xh(:,:)
         integer, intent(out) :: ierr
         
         real(dp) :: xq_outer, xq_inner, j_tot, xq0, xq1, new_point_dqbar, dq_sum, dq
         integer :: kk, k_outer, j
         
         integer, parameter :: k_dbg = -1
         
         include 'formats.dek'
         
         ierr = 0                  
         xq_outer = new_xout(k)
         new_point_dqbar = new_dqbar(k)
         if (k < nz) then
            xq_inner = xq_outer + new_point_dqbar
         else
            xq_inner = 1d0
         end if
         
         if (k == k_dbg) then
            write(*,2) 'xq_outer', k, xq_outer
            write(*,2) 'xq_inner', k, xq_inner
            write(*,2) 'new_point_dqbar', k, new_point_dqbar
         end if
         
         !write(*,*)
         !write(*,2) 'xq_outer', k, xq_outer
         
         dq_sum = 0d0
         j_tot = 0
         if (xq_outer >= old_xout(nz_old)) then
            ! new contained entirely in old center zone
            k_outer = nz_old
            if (k == k_dbg) &
               write(*,2) 'new contained in old center', &
                  k_outer, old_xout(k_outer)
         else if (k == 1) then
            k_outer = 1
         else
            k_outer = comes_from(k-1)
         end if

         do kk = k_outer, nz_old ! loop until reach m_inner
            
            if (kk == nz_old) then
               xq1 = 1d0
            else
               xq1 = old_xout(kk+1)
            end if
            if (xq1 <= xq_outer) cycle

            xq0 = old_xout(kk)
            if (xq0 >= xq_inner) then
               if (dq_sum < new_point_dqbar .and. kk > 1) then 
                  ! need to add a bit more from the previous source
                  dq = new_point_dqbar - dq_sum
                  dq_sum = new_point_dqbar
                  j_tot = j_tot + old_j_rot(kk-1)*dq
               
                  if (.false. .or. k == k_dbg) &
                     write(*,3) 'new k contains some of old kk-1', &
                        k, kk, old_j_rot(kk-1)*dq, old_j_rot(kk-1), dq, j_tot/dq_sum, j_tot, dq_sum

                  end if
               exit
            end if
            
            if (xq1 < xq_outer) then
               ierr = -1
               !return
               write(*,*)
               write(*,*) 'k', k
               write(*,*) 'kk', kk
               write(*,1) 'xq1', xq1
               write(*,1) 'xq_outer', xq_outer
               write(*,*) 'xq1 < xq_outer'
               stop 'debugging: adjust1_omega'
            end if
            
            if (xq0 >= xq_outer .and. xq1 <= xq_inner) then ! entire old kk is in new k
               
               dq = old_dqbar(kk)
               dq_sum = dq_sum + dq
               
               if (dq_sum > new_point_dqbar) then 
                  ! dq too large -- numerical roundoff problems
                  dq = dq - (new_point_dqbar - dq_sum)
                  dq_sum = new_point_dqbar
               end if
               
               j_tot = j_tot + old_j_rot(kk)*dq
               
               if (.false. .or. k == k_dbg) &
                  write(*,3) 'new k contains all of old kk', &
                     k, kk, old_j_rot(kk)*dq, old_j_rot(kk), dq, j_tot/dq_sum, j_tot, dq_sum
               
            else if (xq0 <= xq_outer .and. xq1 >= xq_inner) then ! entire new k is in old kk
            
               dq = new_dqbar(k)
               dq_sum = dq_sum + dq
               j_tot = j_tot + old_j_rot(kk)*dq
               
               if (.false. .or. k == k_dbg) &
                  write(*,3) 'all new k is in old kk', &
                     k, kk, old_j_rot(kk)*dq, old_j_rot(kk), dq, j_tot/dq_sum, j_tot, dq_sum
               
            else ! only use the part of old kk that is in new k
            
               if (k == k_dbg) then
                  write(*,*) 'only use the part of old kk that is in new k', xq_inner <= xq1
                  write(*,1) 'xq_outer', xq_outer
                  write(*,1) 'xq_inner', xq_inner
                  write(*,1) 'xq0', xq0
                  write(*,1) 'xq1', xq1
                  write(*,1) 'dq_sum', dq_sum
                  write(*,1) 'new_point_dqbar', new_point_dqbar
                  write(*,1) 'new_point_dqbar - dq_sum', new_point_dqbar - dq_sum
               end if
            
               if (xq_inner <= xq1) then ! this is the last part of new k
               
                  if (k == k_dbg) write(*,3) 'this is the last part of new k', k, kk

                  dq = new_point_dqbar - dq_sum
                  dq_sum = new_point_dqbar

               else ! we avoid this case if possible because of numerical roundoff
               
                  if (k == k_dbg) write(*,3) 'we avoid this case if possible', k, kk
               
                  dq = max(0d0, xq1 - xq_outer)
                  if (dq_sum + dq > new_point_dqbar) dq = new_point_dqbar - dq_sum
                  dq_sum = dq_sum + dq

               end if
               
               j_tot = j_tot + old_j_rot(kk)*dq
               
               if (.false. .or. k == k_dbg) &
                  write(*,3) 'new k use only part of old kk', &
                     k, kk, old_j_rot(kk)*dq, old_j_rot(kk), dq, j_tot/dq_sum, j_tot, dq_sum
               
               if (dq <= 0) then
                  ierr = -1
                  !return
                  write(*,*) 'dq <= 0', dq
                  stop 'debugging: adjust1_omega'
               end if
               
            end if
            
            if (dq_sum >= new_point_dqbar) then
               if (k == k_dbg) then
                  write(*,2) 'exit for k', k
                  write(*,2) 'dq_sum', kk, dq_sum
                  write(*,2) 'new_point_dqbar', kk, new_point_dqbar
               end if
               exit
            end if
            
         end do
			
			s% j_rot(k) = j_tot/dq_sum
			s% i_rot(k) = (2d0/3d0)*exp(2*xh(s% i_lnR, k))
         s% omega(k) = s% j_rot(k)/s% i_rot(k)
         
         if (k_dbg == k) then
            write(*,2) 's% omega(k)', k, s% omega(k)
            write(*,2) 's% j_rot(k)', k, s% j_rot(k)
            write(*,2) 's% i_rot(k)', k, s% i_rot(k)
            if (s% model_number > 1925) stop 'debugging: adjust1_omega'
         end if
               
      end subroutine adjust1_omega


      end module mesh_adjust
      
