! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module init

      use star_private_def
      use const_def

      implicit none
      
      integer, parameter :: do_create_pre_ms_model = 0
      integer, parameter :: do_load_zams_model = 1
      integer, parameter :: do_load_saved_model = 2
      integer, parameter :: do_create_initial_model = 3

      contains
      
      
      subroutine do_star_init( &
            my_mesa_dir, chem_isotopes_filename, &
            kappa_file_prefix, kappa_CO_prefix, kappa_lowT_prefix, &
            kappa_blend_logT_upper_bdy, kappa_blend_logT_lower_bdy, &
            kappa_type2_logT_lower_bdy, other_kappa_file_prefix, &
            eos_file_prefix, other_eos_prefix, eosDT_Z1_suffix, eosPT_Z1_suffix, &
            net_reaction_filename, rate_tables_dir, rates_cache_suffix, &
            ionization_file_prefix, ionization_Z1_suffix, &
            eosDT_cache_dir, eosPT_cache_dir, ionization_cache_dir, &
            kap_cache_dir, rates_cache_dir, weaklib_cache_dir, &
            ierr)
         use mod_paquette_coeffs, only: initialise_collision_integrals
         use hydro_rotation, only: init_rotation
         use alloc, only: init_alloc
         character (len=*), intent(in) :: &
            my_mesa_dir, chem_isotopes_filename, net_reaction_filename, rate_tables_dir, &
            kappa_file_prefix, kappa_CO_prefix, kappa_lowT_prefix, &
            other_kappa_file_prefix, &
            eosDT_Z1_suffix, eosPT_Z1_suffix, &
            eos_file_prefix, other_eos_prefix, rates_cache_suffix, &
            ionization_file_prefix, ionization_Z1_suffix, &
            eosDT_cache_dir, eosPT_cache_dir, ionization_cache_dir, &
            kap_cache_dir, rates_cache_dir, weaklib_cache_dir
         real(dp), intent(in) :: kappa_blend_logT_upper_bdy, kappa_blend_logT_lower_bdy, &
            kappa_type2_logT_lower_bdy
         integer, intent(out) :: ierr
         ! ierr will be 0 for a normal return. 
         ! ierr nonzero means something went wrong.
         integer :: iam, nprocs, nprow, npcol
         include 'formats'
         ierr = 0
         call initialise_collision_integrals
         call init_alloc
         !call superlu_dist_before(iam, nprocs, nprow, npcol, ierr)
            ! for iam /= 0, this doesn't return until it is time to quit
         if (ierr /= 0) then
            write(*,*) 'superlu_dist_before returned ierr', ierr
            return
         end if
         !if (iam /= 0) then
         !   call superlu_dist_after(ierr)
         !   stop 'do_star_init'
         !end if       
         call stardata_init( &
            my_mesa_dir, chem_isotopes_filename, &
            kappa_file_prefix, kappa_CO_prefix, kappa_lowT_prefix, &
            kappa_blend_logT_upper_bdy, kappa_blend_logT_lower_bdy, &
            kappa_type2_logT_lower_bdy, other_kappa_file_prefix, &
            eos_file_prefix, other_eos_prefix, eosDT_Z1_suffix, eosPT_Z1_suffix, &
            net_reaction_filename, rate_tables_dir, rates_cache_suffix, &
            ionization_file_prefix, ionization_Z1_suffix, &
            eosDT_cache_dir, eosPT_cache_dir, ionization_cache_dir, &
            kap_cache_dir, rates_cache_dir, weaklib_cache_dir, &
            ierr)
      	if (ierr /= 0) then
            write(*,*) 'failed in stardata_init'
            return
      	end if
         call init_rotation(ierr)
      	if (ierr /= 0) then
            write(*,*) 'failed in init_rotation'
            return
      	end if
         !if (nprow*npcol > 1) then
         !   write(*,'(a,x,3i3)') 'MPI nprocs nprow npcol', nprocs, nprow, npcol
         !end if
      end subroutine do_star_init


      subroutine do_star_shutdown
         !use mtx_lib, only: superlu_dist_quit_work, superlu_dist_after
         use micro, only: shutdown_microphys
         integer :: ierr
         !call superlu_dist_quit_work(ierr)  
         !call superlu_dist_after(ierr)
         !call shutdown_microphys ! skip this for now
      end subroutine do_star_shutdown
            
      
      integer function alloc_star_data(ierr)
         use kap_lib
         use eos_lib
         use rates_def, only: rates_reaction_id_max, rates_NACRE_if_available
         use chem_def, only: num_categories
         use net, only: default_set_which_rates, default_set_rate_factors
         use mod_other_wind, only: null_other_wind
         use mod_other_adjust_mdot, only: null_other_adjust_mdot
         use mod_other_torque, only: default_other_torque
         use mod_other_torque_implicit, only: default_other_torque_implicit
         use mod_other_energy, only: default_other_energy
         use mod_other_energy_implicit, only: default_other_energy_implicit
         use mod_other_mixing, only: null_other_mixing
         use mod_other_am_mixing, only: null_other_am_mixing
         use mod_other_atm, only: null_other_atm
         use mod_other_diffusion, only: null_other_diffusion
         use mod_other_mlt, only: null_other_mlt
         use mod_other_neu, only: null_other_neu
         use mod_other_cgrav, only: default_other_cgrav
         use mod_other_opacity_factor, only: default_other_opacity_factor
         use mod_other_pgstar_plots, only: null_other_pgstar_plots_info
         use mod_other_mesh_functions
         use mod_other_eps_grav, only: null_other_eps_grav
         use mod_other_get_pulsation_info, only: null_other_get_pulsation_info
         use mod_other_write_pulsation_info, only: null_other_write_pulsation_info
         use mod_other_history_data_initialize, only: null_other_history_data_initialize
         use mod_other_history_data_add_model, only: null_other_history_data_add_model
         use mod_other_eos
         use mod_other_kap
         
         
         integer, intent(out) :: ierr
         
         type (star_info), pointer :: s
         integer, parameter :: init_alloc_nvar = 20
         character (len=32) :: extra_name
         integer :: i
                  
         ierr = 0
         
         alloc_star_data = alloc_star(ierr)
         if (ierr /= 0) return
         
         call get_star_ptr(alloc_star_data, s, ierr)
         if (ierr /= 0) return

         s% eos_handle = alloc_eos_handle(ierr)
         if (ierr /= 0) return
         
         s% kap_handle = alloc_kap_handle(ierr)
         if (ierr /= 0) return
         
         s% net_handle = 0        
         s% generations = 0         
         s% total_num_jacobians = 0
         s% num_burn_max_iters = 0
         
         s% nvar_hydro = 0                 
         s% nvar_chem = 0                 
         s% nvar = 0     
                     
         s% nz = 0     
         s% net_name = ''  
         s% species = 0                  
         s% num_reactions = 0  
               
         s% nz_old = 0         
         s% nz_older = 0         
         
         s% lnPgas_flag = .false.
         s% v_flag = .false.
         s% rotation_flag = .false.
         
         s% have_previous_rotation_info = .false.
         s% have_previous_conv_vel = .false.
         s% have_prev_step_info = .false.

         s% just_wrote_terminal_header = .false.
         s% just_did_backup = .false.
         
         s% mstar_dot = 0

         s% surf_lnT = 0
         s% surf_lnd = 0
         s% surf_lnPgas = 0
         s% surf_lnR = 0
         s% surf_v = 0
         s% surf_lnS = 0
        
         s% termination_code = -1
         
         s% prev_create_atm_R0_div_R = -1
         
         s% screening_mode_value = -1
         
         s% dt = -1
         s% dt_next = -1
         
         s% i_xlnd = 0
         s% i_lnT = 0
         s% i_lnR = 0
         s% i_lum = 0
         s% i_lnPgas = 0
         s% i_vel = 0
         s% i_chem1 = 0
         
         s% equP = 0
         s% equT = 0
         s% equR = 0 
         s% equL = 0
         s% equv = 0
         s% equchem1 = 0
         
         nullify(s% dq)
         nullify(s% dq_old)
         nullify(s% dq_older)

         nullify(s% conv_vel)
         nullify(s% conv_vel_old)
         nullify(s% conv_vel_older)

         nullify(s% Del_T)
         nullify(s% Del_T_old)
         nullify(s% Del_T_older)

         nullify(s% nu_ST)
         nullify(s% nu_ST_old)
         nullify(s% nu_ST_older)

         nullify(s% D_DSI)
         nullify(s% D_DSI_old)
         nullify(s% D_DSI_older)

         nullify(s% D_SH)
         nullify(s% D_SH_old)
         nullify(s% D_SH_older)

         nullify(s% D_SSI)
         nullify(s% D_SSI_old)
         nullify(s% D_SSI_older)

         nullify(s% D_ES)
         nullify(s% D_ES_old)
         nullify(s% D_ES_older)

         nullify(s% D_GSF)
         nullify(s% D_GSF_old)
         nullify(s% D_GSF_older)

         nullify(s% q)
         nullify(s% q_old)
         nullify(s% q_older)

         nullify(s% xa)
         nullify(s% xa_old)
         nullify(s% xa_older)

         nullify(s% xh)
         nullify(s% xh_old)
         nullify(s% xh_older)
         
         nullify( &
            s% op_mono_umesh1, s% op_mono_ff1, &
            s% op_mono_rs1, s% op_mono_s1)
         s% op_mono_nptot = 0
         s% op_mono_ipe = 0
         s% op_mono_nrad = 0 
         s% op_mono_n = 0

         nullify(s% atm_structure)
         s% atm_structure_num_pts = 0
         
         nullify(s% chem_id)

         nullify(s% which_rates)
         s% set_which_rates => default_set_which_rates
         
         nullify(s% rate_factors)
         s% set_rate_factors => default_set_rate_factors
         
         allocate(s% nameofvar(init_alloc_nvar),stat=ierr)
         if (ierr /= 0) return
         
         allocate(s% nameofequ(init_alloc_nvar),stat=ierr)
         if (ierr /= 0) return
         
         allocate(s% ode_var(init_alloc_nvar),stat=ierr)
         if (ierr /= 0) return

         allocate(s% category_factors(num_categories))
         
         do i = 1, max_num_profile_extras
            if (i < 10) then
               write(extra_name,'(a,i1)') 'profile_extra_', i
            else
               write(extra_name,'(a,i2)') 'profile_extra_', i
            end if
            s% profile_extra_name(i) = trim(extra_name)
         end do
         
         do i = 1, max_num_history_extras
            if (i < 10) then
               write(extra_name,'(a,i1)') 'history_extra_', i
            else
               write(extra_name,'(a,i2)') 'history_extra_', i
            end if
            s% history_extra_name(i) = trim(extra_name)
         end do

         nullify(s% history_values)
         nullify(s% history_value_is_integer)
         nullify(s% history_names)
         nullify(s% history_names_dict)
         s% number_of_history_columns = -1
         s% model_number_of_history_values = -1
         s% need_to_set_history_names_etc = .true.
         
         s% pgstar_prev_burn_max = 0
         s% pgstar_prev_burn_min = 0
 
         nullify(s% prev_mesh_xm)
         nullify(s% prev_mesh_f1_lnS)
         
         nullify(s% other_star_info)
         
         nullify(s% bcyclic_klu_storage)
         nullify(s% bcyclic_odd_storage)
         nullify(s% bcyclic_sprs_shared_ia)
         nullify(s% bcyclic_sprs_shared_ja)
         s% bcyclic_sprs_shared_net_name = ''
         s% bcyclic_shared_sprs_nonzeros = 0
         
         nullify(s% burn_klu_storage)
         nullify(s% burn_sprs_shared_ia)
         nullify(s% burn_sprs_shared_ja)
         s% burn_sprs_shared_net_name = ''
         s% burn_shared_sprs_nonzeros = 0
         s% bcyclic_nvar_hydro = 0

         nullify(s% ipar_decsol)
         nullify(s% hydro_iwork)
         nullify(s% rpar_decsol)
         nullify(s% hydro_work)
         
         s% other_wind => null_other_wind
         s% other_adjust_mdot => null_other_adjust_mdot
         s% other_mixing => null_other_mixing
         s% other_am_mixing => null_other_am_mixing
         s% other_torque => default_other_torque
         s% other_torque_implicit => default_other_torque_implicit
         s% other_energy => default_other_energy
         s% other_energy_implicit => default_other_energy_implicit
         s% other_cgrav => default_other_cgrav
         s% other_opacity_factor => default_other_opacity_factor
         s% other_atm => null_other_atm
         s% other_diffusion => null_other_diffusion
         s% other_mlt => null_other_mlt
         s% other_neu => null_other_neu
         s% other_eps_grav => null_other_eps_grav

         s% other_eosDT_get => null_other_eosDT_get
         s% other_eosDT_get_T => null_other_eosDT_get_T
         s% other_eosDT_get_Rho => null_other_eosDT_get_Rho
         
         s% other_eosPT_get => null_other_eosPT_get
         s% other_eosPT_get_T => null_other_eosPT_get_T
         s% other_eosPT_get_Pgas => null_other_eosPT_get_Pgas
         s% other_eosPT_get_Pgas_for_Rho => null_other_eosPT_get_Pgas_for_Rho

         s% other_kap_get_Type1 => null_other_kap_get_Type1
         s% other_kap_get_Type2 => null_other_kap_get_Type2

         s% other_pgstar_plots_info => null_other_pgstar_plots_info
         s% how_many_other_mesh_fcns => null_how_many_other_mesh_fcns
         s% other_mesh_fcn_data => null_other_mesh_fcn_data         

         s% other_write_pulsation_info => null_other_write_pulsation_info         
         s% other_get_pulsation_info => null_other_get_pulsation_info

         s% other_history_data_initialize => null_other_history_data_initialize   
         s% other_history_data_add_model => null_other_history_data_add_model
         
      end function alloc_star_data    
      

      subroutine create_pre_ms_model(id, ierr)
         use star_utils, only: std_write_internals_to_file
         integer, intent(in) :: id
         integer, intent(out) :: ierr
         character (len=0) :: model_dir
         call model_builder( &
            id, model_dir, do_create_pre_ms_model, &
            .false., 'restart_photo', ierr)
      end subroutine create_pre_ms_model
      

      subroutine create_initial_model(id, ierr)
         use star_utils, only: std_write_internals_to_file
         integer, intent(in) :: id
         integer, intent(out) :: ierr
         character (len=0) :: model_dir
         call model_builder( &
            id, model_dir, do_create_initial_model, &
            .false., 'restart_photo', ierr)
      end subroutine create_initial_model
      

      subroutine load_zams_model(id, ierr)
         integer, intent(in) :: id
         integer, intent(out) :: ierr
         call model_builder( &
            id, '', do_load_zams_model, &
            .false., 'restart_photo', ierr)
      end subroutine load_zams_model
      

      subroutine load_saved_model(id, model_fname, ierr)
         integer, intent(in) :: id
         character (len=*), intent(in) :: model_fname
         integer, intent(out) :: ierr
         call model_builder( &
            id, model_fname, do_load_saved_model, &
            .false., 'restart_photo', ierr)
      end subroutine load_saved_model
      

      subroutine load_restart_photo(id, restart_filename, ierr)
         integer, intent(in) :: id
         character (len=*), intent(in) :: restart_filename
         integer, intent(out) :: ierr
         call model_builder( &
            id, '', do_load_zams_model, .true., restart_filename, ierr)
      end subroutine load_restart_photo


      ! for both zams and pre-main-sequence
      subroutine model_builder( &
            id, model_info, do_which, restart, restart_filename, ierr)
         use net, only: set_net
         use alloc, only: set_var_info
         use model_in, only: read_restart
         use init_model, only: get_zams_model
         use report, only: do_report
         use star_utils, only: set_phase_of_evolution, yrs_for_init_timestep
         use adjust_xyz, only: set_z, set_y
         use pre_ms_model, only: build_pre_ms_model
         use create_initial_model, only: build_initial_model
         use read_model, only: do_read_saved_model, finish_load_model, set_zero_age_params
         use relax, only: do_relax_to_limit, do_relax_mass, &
            do_relax_mass_scale, do_relax_num_steps
         integer, intent(in) :: id, do_which
         character (len=*), intent(in) :: model_info, restart_filename
         logical, intent(in) :: restart
         integer, intent(out) :: ierr
         
         type (star_info), pointer :: s
         real(dp) :: initial_mass, initial_z, dlgm_per_step
         
         real(dp), parameter :: lg_max_abs_mdot = -1000 ! use default
         real(dp), parameter :: change_mass_years_for_dt = 1
         real(dp), parameter :: min_mass_for_create_pre_ms = 0.03d0
         logical :: restore_at_end
         integer :: k

         include 'formats'
         
         ierr = 0
         
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) return

         initial_mass = s% initial_mass
         initial_z = s% initial_z
         s% dt = 0
         s% termination_code = -1
         
         if (restart) then
            s% doing_first_model_of_run = .false.
            s% doing_first_model_after_restart = .true.
            call read_restart(s, restart_filename, ierr)            
            if (ierr /= 0) return         
            s% M_center = s% mstar - s% xmstar
            call check_initials
            call set_net(s, s% net_name, ierr)
            if (ierr /= 0) return
            call finish_load_model(s, ierr)
            if (s% max_years_for_timestep > 0) &
               s% dt_next = min(s% dt_next, secyer*s% max_years_for_timestep)
            return
         end if

         s% doing_first_model_of_run = .true.
         s% doing_first_model_after_restart = .false.
         call set_zero_age_params(s)
         
         if (do_which == do_load_saved_model) then
            s% dt_next = -1
            call do_read_saved_model(s, model_info, ierr)
            if (ierr /= 0) return
            call check_initials
            if (s% dt_next < 0) s% dt_next = yrs_for_init_timestep(s)*secyer
         else
            s% net_name = s% default_net_name
            s% species = 0
            s% lnPgas_flag = .false.
            s% v_flag = .false.
            s% rotation_flag = .false.
            s% star_mass = s% initial_mass
            s% mstar = s% initial_mass*Msun
            s% M_center = s% mstar - s% xmstar
            call set_var_info(s, ierr)
            if (ierr /= 0) then
               write(*,*) 'failed in set_var_info'
               return
            end if
            select case (do_which)
               case (do_create_initial_model)
                  call set_net(s, s% net_name, ierr)
                  if (ierr /= 0) then
                     write(*,*) 'failed in set_net'
                     return
                  end if
                  call build_initial_model(s, ierr)
                  if (ierr /= 0) then
                     write(*,*) 'failed in build_initial_model'
                     return
                  end if
                  s% generations = 1
                  s% dt_next = 1d-5*secyer
                  !write(*,'(a)') ' done create initial model'
                  !write(*,*)
               case (do_create_pre_ms_model)
                  if (s% initial_mass < min_mass_for_create_pre_ms) then
                     write(*,*)
                     write(*,*)
                     write(*,*)
                     write(*,'(a,1x,f5.2)') 'sorry: cannot create pre-ms smaller than', &
                        min_mass_for_create_pre_ms
                     write(*,'(a)') &
                        'please create pre-ms and then relax to lower mass as a separate operation'
                     write(*,*)
                     write(*,'(a)') 'here is an example:'
                     write(*,'(a)') 'in your inlist &controls section, set initial_mass = 0.03'
                     write(*,'(a)') 'in the &star_job section, add something like these lines'
                     write(*,'(a)') '  relax_mass_scale = .true.'
                     write(*,'(a)') '  dlgm_per_step = 1d-3 ! log10(delta M/Msun/step)'
                     write(*,'(a)') '  new_mass = 2.863362d-3 ! 3 Mjupiter in Msun units'
                     write(*,'(a)') '  change_mass_years_for_dt = 1'
                     write(*,*)
                     write(*,*)
                     write(*,*)
                     ierr = -1
                     return
                  end if
                  call set_net(s, s% net_name, ierr)
                  if (ierr /= 0) then
                     write(*,*) 'failed in set_net'
                     return
                  end if
                  call build_pre_ms_model(id, s, s% nvar_hydro, s% species, ierr)
                  if (ierr /= 0) then
                     write(*,*) 'failed in build_pre_ms_model'
                     return
                  end if
                  s% generations = 1
                  s% dt_next = 1d-5*secyer
                  !write(*,'(a)') ' done create pre main-sequence model'
                  !write(*,*)
               case (do_load_zams_model)
                  s% generations = 1
                  call get_zams_model(s, s% zams_filename, ierr)
                  if (ierr /= 0) then
                     write(*,*) 'failed in get_zams_model'
                     return
                  end if
                  if (s% dt_next < 0) s% dt_next = yrs_for_init_timestep(s)*secyer
               case default
                  write(*,*) 'bad value for do_which in build_model'
                  ierr = -1
                  return
            end select
         end if

         s% extra_heat(1:s% nz) = 0

         call finish_load_model(s, ierr)
         if (ierr /= 0) then
            write(*,*) 'failed in finish_load_model'
            return
         end if
         if (s% max_years_for_timestep > 0) &
            s% dt_next = min(s% dt_next, secyer*s% max_years_for_timestep)
         call set_phase_of_evolution(s)
         
         if (do_which == do_create_pre_ms_model) then
            if (s% mstar > s% initial_mass*Msun) then ! need to reduce mass
               write(*,1) 'reduce mass to', s% initial_mass
               !call do_relax_mass_scale( &
               !   s% id, s% initial_mass, 0.005d0, 1d0, ierr)
               call do_relax_mass(s% id, s% initial_mass, lg_max_abs_mdot, ierr)
               if (ierr /= 0) then
                  write(*,*) 'failed in do_relax_mass'
                  return
               end if
            else if (s% mstar < s% initial_mass*Msun) then ! need to increase mass
               write(*,1) 'increase mass to', s% initial_mass
               call do_relax_mass_scale( &
                  s% id, s% initial_mass, 0.005d0, 1d0, ierr)
               !call do_relax_mass(s% id, s% initial_mass, lg_max_abs_mdot, ierr)
               if (ierr /= 0) then
                  write(*,*) 'failed in do_relax_mass'
                  return
               end if
            end if
            call do_relax_num_steps( &
               s% id, s% pre_ms_relax_num_steps, s% dt_next, ierr) 
            if (ierr /= 0) then
               write(*,*) 'failed in do_relax_num_steps'
               return
            end if
         else if (do_which == do_create_initial_model) then
            call do_relax_num_steps( &
               s% id, s% initial_model_relax_num_steps, s% dt_next, ierr) 
            if (ierr /= 0) then
               write(*,*) 'failed in do_relax_num_steps'
               return
            end if
         end if

         s% doing_first_model_of_run = .true.
         
         
         contains
         
         subroutine check_initials
            include 'formats'
            if (abs(initial_mass - s% initial_mass) > 1d-3*initial_mass .and. initial_mass > 0) then
               write(*,1) "WARNING -- inlist initial_mass ignored", initial_mass
               write(*,1) "using saved initial_mass instead", s% initial_mass
               write(*,*)
            end if
            if (abs(initial_z - s% initial_z) > 1d-3*initial_z .and. initial_z > 0) then
               write(*,1) "WARNING -- inlist initial_z ignored", initial_z
               write(*,1) "using saved initial_z instead", s% initial_z
               write(*,*)
            end if
         end subroutine check_initials
         
      end subroutine model_builder
      
      
      logical function doing_restart()
         integer :: ios
         open(unit=99, file='restart_photo', action='read', status='old', iostat=ios)
         if (ios == 0) then
            doing_restart = .true.
            close(99)
         else
            doing_restart = .false.
         end if
      end function doing_restart

         
      end module init
