! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module hydro_vars
      
      use star_private_def
      use const_def
      use chem_def, only: chem_isos
      use utils_lib, only:is_bad_num
      
      implicit none

      logical, parameter :: dbg = .false.      

      contains
      
      
      subroutine set_vars(s, dt, ierr)
         type (star_info), pointer :: s
         real(dp), intent(in) :: dt
         integer, intent(out) :: ierr
         logical, parameter :: &
            skip_time_derivatives = .false., &
            skip_m_grav_and_grav = .false., &
            skip_do_burn_mix = .true., &
            skip_net = .false., &
            skip_neu = .false., &
            skip_kap = .false., &
            skip_grads = .false., &
            skip_rotation = .false., &
            skip_nse_fractions = .false., &
            skip_brunt = .false., &
            skip_mixing_info = .false.
         if (dbg) write(*, *) 'set_vars'
         call set_some_vars( &
            s, skip_time_derivatives, skip_m_grav_and_grav, skip_do_burn_mix, &
            skip_net, skip_neu, skip_kap, skip_grads, skip_rotation, &
            skip_brunt, skip_nse_fractions, skip_mixing_info, dt, ierr)
      end subroutine set_vars
      
      
      subroutine set_final_vars(s, dt, ierr)
         use alloc, only: non_crit_get_work_array, non_crit_return_work_array
         type (star_info), pointer :: s
         real(dp), intent(in) :: dt
         integer, intent(out) :: ierr
         
         logical, parameter :: &
            skip_time_derivatives = .true., &
            skip_m_grav_and_grav = .false., &
            skip_do_burn_mix = .true., &
            skip_net = .false., &
            skip_neu = .false., &
            skip_kap = .false., &
            skip_grads = .false., &
            skip_rotation = .true., &
            skip_brunt = .false., &
            skip_nse_fractions = .true., &
            skip_mixing_info = .true.
         integer :: nz, ierr1            
         real(dp), pointer, dimension(:) :: &
            cv, nu_st, d_st, d_dsi, d_sh, d_ssi, d_es, d_gsf
            
         ! save and restore mixing coeffs needed for time smoothing
         ierr = 0
         nz = s% nz
         
         nullify(cv, nu_st, d_st, d_dsi, d_sh, d_ssi, d_es, d_gsf)
         
         if (s% rotation_flag) then
            call get_cpy(s% nu_ST, nu_st, ierr1)
            if (ierr1 /= 0) ierr = -1
            call get_cpy(s% D_ST, d_st, ierr1)
            if (ierr1 /= 0) ierr = -1
            call get_cpy(s% D_DSI, d_dsi, ierr1)
            if (ierr1 /= 0) ierr = -1
            call get_cpy(s% D_SH, d_sh, ierr1)
            if (ierr1 /= 0) ierr = -1
            call get_cpy(s% D_SSI, d_ssi, ierr1)
            if (ierr1 /= 0) ierr = -1
            call get_cpy(s% D_ES, d_es, ierr1)
            if (ierr1 /= 0) ierr = -1
            call get_cpy(s% D_GSF, d_gsf, ierr1)
            if (ierr1 /= 0) ierr = -1
         end if
         
         if (s% min_T_for_time_averaged_conv_velocity < 1d12) then
            call get_cpy(s% conv_vel, cv, ierr1)
            if (ierr1 /= 0) ierr = -1
         end if
         
         if (ierr == 0) then
            if (dbg) write(*, *) 'set_vars'
            
            ! save values used during the step before they are replaced
            ! by values at end of step
            
            s% n_conv_regions_older = s% n_conv_regions_old
            s% n_conv_regions_old = s% n_conv_regions
         
            s% cz_bot_mass_older(:) = s% cz_bot_mass_old(:)
            s% cz_bot_mass_old(:) = s% cz_bot_mass(:)
         
            s% cz_top_mass_older(:) = s% cz_top_mass_old(:)
            s% cz_top_mass_old(:) = s% cz_top_mass(:)

            call set_some_vars( &
               s, skip_time_derivatives, skip_m_grav_and_grav, skip_do_burn_mix, &
               skip_net, skip_neu, skip_kap, skip_grads, skip_rotation, skip_brunt, &
               skip_nse_fractions, skip_mixing_info, dt, ierr)
            
         end if
         
         if (s% rotation_flag) then
            call restore(s% nu_ST, nu_st)
            call restore(s% D_ST, d_st)
            call restore(s% D_DSI, d_dsi)
            call restore(s% D_SH, d_sh)
            call restore(s% D_SSI, d_ssi)
            call restore(s% D_ES, d_es)
            call restore(s% D_GSF, d_gsf)         
            s% have_previous_rotation_info = .true.
         end if
         
         if (s% min_T_for_time_averaged_conv_velocity < 1d12) then
            call restore(s% conv_vel, cv)
            s% have_previous_conv_vel = .true.
         end if

            
         contains
         
         
         subroutine get_cpy(src,cpy,ierr)
            real(dp), pointer, dimension(:) :: src, cpy
            integer, intent(out) :: ierr
            integer :: k
            ierr = 0
            call non_crit_get_work_array( &
               s, cpy, nz, nz_alloc_extra, 'set_final_vars', ierr)
            if (ierr /= 0) return
            do k=1,nz
               cpy(k) = src(k)
            end do
         end subroutine get_cpy
         
         
         subroutine restore(src,cpy)
            real(dp), pointer, dimension(:) :: src, cpy
            integer :: k
            if (.not. associated(cpy)) return
            do k=1,nz
               src(k) = cpy(k)
            end do
            call non_crit_return_work_array(s, cpy, 'set_final_vars')
         end subroutine restore
         
         
      end subroutine set_final_vars
      
      
      subroutine reset_vars_after_diffusion(s, dt, ierr)
         type (star_info), pointer :: s
         real(dp), intent(in) :: dt
         integer, intent(out) :: ierr
         logical, parameter :: &
            skip_time_derivatives = .true., &
            skip_m_grav_and_grav = .true., &
            skip_do_burn_mix = .true., &
            skip_net = .false., &
            skip_neu = .false., &
            skip_kap = .false., &
            skip_grads = .false., &
            skip_rotation = .false., &
            skip_brunt = .true., &
            skip_nse_fractions = .true., &
            skip_mixing_info = .true.
         if (dbg) write(*, *) 'reset_vars_after_diffusion'
         call set_some_vars( &
            s, skip_time_derivatives, skip_m_grav_and_grav, skip_do_burn_mix, &
            skip_net, skip_neu, skip_kap, skip_grads, skip_rotation, &
            skip_brunt, skip_nse_fractions, skip_mixing_info, dt, ierr)
      end subroutine reset_vars_after_diffusion

      
      subroutine set_some_vars( &
            s, skip_time_derivatives, skip_m_grav_and_grav, skip_do_burn_mix, &
            skip_net, skip_neu, skip_kap, skip_grads, skip_rotation, &
            skip_brunt, skip_nse_fractions, skip_mixing_info, dt, ierr)
         type (star_info), pointer :: s
         logical, intent(in) :: &
            skip_time_derivatives, skip_m_grav_and_grav, skip_do_burn_mix, &
            skip_net, skip_neu, skip_kap, skip_grads, &
            skip_rotation, skip_brunt, skip_nse_fractions, skip_mixing_info
         real(dp), intent(in) :: dt
         integer, intent(out) :: ierr

         logical, parameter :: skip_other_cgrav = .false.
         logical, parameter :: skip_basic_vars = .false.
         logical, parameter :: skip_micro_vars = .false.
         
         include 'formats'

         call update_vars(s, &
            skip_time_derivatives, skip_basic_vars, skip_micro_vars, &
            skip_m_grav_and_grav, skip_do_burn_mix, skip_net, skip_neu, skip_kap, &
            skip_grads, skip_rotation, skip_brunt, skip_other_cgrav, &
            skip_nse_fractions, skip_mixing_info, dt, ierr)
         if (ierr /= 0) then
            if (s% report_ierr .or. dbg) &
               write(*, *) 'set_some_vars: update_vars returned ierr', ierr
            return
         end if

      end subroutine set_some_vars
      
      
      subroutine update_vars(s, &
            skip_time_derivatives, skip_basic_vars, skip_micro_vars, &
            skip_m_grav_and_grav, skip_do_burn_mix, skip_net, skip_neu, skip_kap, &
            skip_grads, skip_rotation, skip_brunt, skip_other_cgrav, &
            skip_nse_fractions, skip_mixing_info, dt, ierr)
         use utils_lib, only: has_bad_num, is_bad_num
         use mlt_def, only: no_mixing
         use star_utils, only: &
            eval_irradiation_heat, set_dm_bar, set_m_and_dm, get_tau
         use mix_info, only: set_mixing_info
         type (star_info), pointer :: s 
         logical, intent(in) :: &
            skip_time_derivatives, skip_basic_vars, skip_micro_vars, &
            skip_m_grav_and_grav, skip_do_burn_mix, skip_net, skip_neu, skip_kap, skip_grads, &
            skip_rotation, skip_brunt, skip_other_cgrav, &
            skip_nse_fractions, skip_mixing_info
         real(dp), intent(in) :: dt
         integer, intent(out) :: ierr
         
         integer :: i_xlnd, i_lnPgas, i_lnT, i_lnR, i_lum, i_vel, &
            j, k, species, nvar_chem, nzlo, nz, k_below_just_added
         real(dp) :: dt_inv, del_t
         
         include 'formats'
         
         ierr = 0
         nz = s% nz
         nzlo = 1
         k_below_just_added = s% k_below_just_added
         species = s% species
         nvar_chem = s% nvar_chem
         i_xlnd = s% i_xlnd
         i_lnPgas = s% i_lnPgas
         i_lnT = s% i_lnT
         i_lnR = s% i_lnR
         i_lum = s% i_lum
         i_vel = s% i_vel         
         
         do j=1,s% nvar_hydro
            if (j == i_xlnd) then
               do k=nzlo,nz
                  s% lnd(k) = s% xh(i_xlnd,k)
               end do
            else if (j == i_lnPgas) then
               do k=nzlo,nz
                  s% lnPgas(k) = s% xh(i_lnPgas,k)
               end do
            else if (j == i_lnT) then
               do k=nzlo,nz
                  s% lnT(k) = s% xh(i_lnT,k)
               end do
            else if (j == i_lnR) then
               do k=nzlo,nz
                  s% lnR(k) = s% xh(i_lnR,k)
               end do
            else if (j == i_lum) then
               do k=nzlo,nz
                  s% L(k) = s% xh(i_lum, k)
               end do
            else if (j == i_vel) then
               do k=nzlo,nz
                  s% v(k) = s% xh(i_vel,k)
               end do
            end if
         end do
         
         if (i_vel == 0) then
            do k=nzlo,nz
               s% v(k) = 0d0
            end do
         end if
         
         call set_m_and_dm(s)
         call set_dm_bar(s% nz, s% dm, s% dm_bar)

         if (.not. skip_mixing_info) then
            s% mixing_type(nzlo:nz) = no_mixing
            s% adjust_mlt_gradT_fraction(nzlo:nz) = -1 ! can be set by other_mixing
         end if
         
         if (.not. skip_time_derivatives) then
            ! lagrangian time derivatives
            if (s% generations < 2 .or. dt <= 0 .or. s% nz /= s% nz_old) then
         
               if (i_xlnd /= 0) s% dlnd_dt(nzlo:nz) = 0
               if (i_lnPgas /= 0) s% dlnPgas_dt(nzlo:nz) = 0
               s% dlnT_dt(nzlo:nz) = 0
               s% dlnR_dt(nzlo:nz) = 0
               if (s% v_flag) s% dv_dt(nzlo:nz) = 0
               s% dVARdot_dVAR = 0
            
            else
         
               dt_inv = 1/dt
               s% dVARdot_dVAR = dt_inv
            
               do k=1,k_below_just_added-1         
                  del_t = s% del_t_for_just_added(k)
                  if (k == 1 .or. del_t <= 0d0) then
                     s% dlnR_dt(k) = 0
                     if (i_vel /= 0) s% dv_dt(k) = 0       
                     if (i_xlnd /= 0) s% dlnd_dt(k) = 0        
                     if (i_lnPgas /= 0) s% dlnPgas_dt(k) = 0        
                     s% dlnT_dt(k) = 0  
                  else
                     s% dlnR_dt(k) = (s% lnR(k) - s% lnR(1))/del_t
                     if (i_vel /= 0) s% dv_dt(k) = (s% v(k) - s% v(1))/del_t            
                     if (i_xlnd /= 0) s% dlnd_dt(k) = (s% lnd(k) - s% lnd(1))/del_t            
                     if (i_lnPgas /= 0) s% dlnPgas_dt(k) = (s% lnPgas(k) - s% lnPgas(1))/del_t            
                     s% dlnT_dt(k) = (s% lnT(k) - s% lnT(1))/del_t            
                  end if           
               end do
               
               if (i_xlnd /= 0) then
                  do k=k_below_just_added,nz 
                     s% dlnd_dt(k) = &
                        (s% xh(i_xlnd,k) - s% lnd_for_d_dt(k))*dt_inv
                     if (is_bad_num(s% dlnd_dt(k))) then
                        ierr = -1
                        if (s% report_ierr) &
                           write(*,2) 'update_vars: bad dlnd_dt', k, &
                              s% dlnd_dt(k), s% lnd_for_d_dt(k), s% xh(i_xlnd,k)
                        stop 'update_vars'
                     end if
                  end do
               end if
               if (i_lnPgas /= 0) then
                  do k=k_below_just_added,nz 
                     s% dlnPgas_dt(k) = (s% xh(i_lnPgas,k) - s% lnPgas_for_d_dt(k))*dt_inv
                     if (is_bad_num(s% dlnPgas_dt(k))) then
                        ierr = -1
                        if (s% report_ierr) &
                           write(*,2) 'update_vars: bad dlnPgas_dt', k, &
                              s% dlnPgas_dt(k), s% lnPgas_for_d_dt(k), s% xh(i_lnPgas,k)
                     end if
                  end do
               end if
               do k=k_below_just_added,nz 
                  s% dlnT_dt(k) = (s% xh(i_lnT,k) - s% lnT_for_d_dt(k))*dt_inv
                  if (is_bad_num(s% dlnT_dt(k))) then
                     ierr = -1
                     if (s% report_ierr) &
                        write(*,2) 'update_vars: bad dlnT_dt', k, s% dlnT_dt(k)
                  end if
               end do
               do k=k_below_just_added,nz 
                  s% dlnR_dt(k) = (s% xh(i_lnR,k) - s% lnR_for_d_dt(k))*dt_inv
                  if (is_bad_num(s% dlnR_dt(k))) then
                     ierr = -1
                     if (s% report_ierr) &
                        write(*,2) 'update_vars: bad dlnR_dt', k, s% dlnR_dt(k)
                  end if
               end do
               if (s% v_flag) then
                  do k=k_below_just_added,nz 
                     s% dv_dt(k) = (s% xh(i_vel,k) - s% v_for_d_dt(k))*dt_inv
                     if (is_bad_num(s% dv_dt(k))) then
                        ierr = -1
                        if (s% report_ierr) &
                           write(*,2) 'update_vars: bad dv_dt', k, s% dv_dt(k)
                     end if
                  end do
               end if
            
            end if
            
         end if
                  
         call set_hydro_vars( &
            s, nzlo, nz, skip_basic_vars, &
            skip_micro_vars, skip_m_grav_and_grav, skip_do_burn_mix, skip_net, skip_neu, &
            skip_kap, skip_grads, skip_rotation, skip_brunt, skip_other_cgrav, &
            skip_nse_fractions, skip_mixing_info, dt, ierr)
         if (ierr /= 0) return         
         
         if (.not. skip_mixing_info) then
            call set_mixing_info(s, ierr)
            if (ierr /= 0) return
            if (s% op_split_gradT_fraction_grada) then
               call set_adjust_mlt_gradT_fraction(s, ierr)
               if (ierr /= 0) return
            end if
         end if

         if (s% Teff <= 0) then
            call set_Teff(s, ierr) 
            if (ierr /= 0) return
         end if
         
         call get_tau(s, s% tau)
         
         s% extra_heat(:) = s% extra_power_source
         s% d_extra_heat_dv00(nzlo:nz) = 0
         s% d_extra_heat_dvp1(nzlo:nz) = 0
         s% d_extra_heat_dlnR00(nzlo:nz) = 0
         s% d_extra_heat_dlnRp1(nzlo:nz) = 0
         if (.not. s% use_other_energy_implicit) then
            s% extra_heat(nzlo:nz) = 0
            s% d_extra_heat_dlnd(nzlo:nz) = 0
            s% d_extra_heat_dlnT(nzlo:nz) = 0         
            if (s% use_other_energy) then
               call s% other_energy(s% id, ierr)
               if (ierr /= 0) then
                  if (s% report_ierr .or. dbg) &
                     write(*, *) 'set_vars: other_energy returned ierr', ierr
                  return
               end if
            else if (s% inject_uniform_extra_heat /= 0) then 
               do k=1,s% nz 
                  s% extra_heat(k) = s% inject_uniform_extra_heat
               end do
            end if
         end if
         
         if (s% irradiation_flux /= 0) then
            do k=nzlo,nz
               s% irradiation_heat(k) = eval_irradiation_heat(s,k)
            end do
         else
            s% irradiation_heat(nzlo:nz) = 0
         end if
         
      end subroutine update_vars

         
      subroutine set_adjust_mlt_gradT_fraction(s,ierr)
         use eos_def
         type (star_info), pointer :: s 
         integer, intent(out) :: ierr   
         integer :: k     
         include 'formats'
         ierr = 0
         do k=2,s% nz
            if (abs(s% grada_at_face(k) - s% gradr(k)) < 1d-50) then
               s% adjust_mlt_gradT_fraction(k) = 0d0
            else
               s% adjust_mlt_gradT_fraction(k) = &
                  (s% gradT(k) - s% gradr(k))/ &
                     (s% grada_at_face(k) - s% gradr(k))
            end if
         end do
         s% adjust_mlt_gradT_fraction(1) = 0d0 ! not used
      end subroutine set_adjust_mlt_gradT_fraction
      
      
      subroutine set_Teff(s, ierr)
         use atm_lib, only: atm_option
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         real(dp) :: r_phot, L_surf
         logical, parameter :: skip_partials = .true.
         real(dp) :: Teff, &
            lnT_surf, dlnT_dL, dlnT_dlnR, dlnT_dlnM, dlnT_dlnkap, &
            lnP_surf, dlnP_dL, dlnP_dlnR, dlnP_dlnM, dlnP_dlnkap
         ierr = 0         
         call set_Teff_info_for_eqns(s, skip_partials, r_phot, L_surf, Teff, &
            lnT_surf, dlnT_dL, dlnT_dlnR, dlnT_dlnM, dlnT_dlnkap, &
            lnP_surf, dlnP_dL, dlnP_dlnR, dlnP_dlnM, dlnP_dlnkap, &
            ierr)
      end subroutine set_Teff
      
      
      subroutine set_Teff_info_for_eqns(s, skip_partials, r_phot, L_surf, Teff, &
            lnT_surf, dlnT_dL, dlnT_dlnR, dlnT_dlnM, dlnT_dlnkap, &
            lnP_surf, dlnP_dL, dlnP_dlnR, dlnP_dlnM, dlnP_dlnkap, &
            ierr)
         use atm_lib, only: atm_option
         use star_utils, only: get_r_phot
         type (star_info), pointer :: s
         logical, intent(in) :: skip_partials
         real(dp), intent(out) :: r_phot, L_surf, Teff, &
            lnT_surf, dlnT_dL, dlnT_dlnR, dlnT_dlnM, dlnT_dlnkap, &
            lnP_surf, dlnP_dL, dlnP_dlnR, dlnP_dlnM, dlnP_dlnkap
         integer, intent(out) :: ierr

         integer :: which_atm_option, off_table_option
         
         r_phot = get_r_phot(s)
         L_surf = s% L(1)
         
         which_atm_option = atm_option(s% which_atm_option, ierr)
         if (ierr /= 0) then
            write(*,*) 'unknown value for which_atm_option ' // trim(s% which_atm_option)
            return
         end if
      
         off_table_option = atm_option(s% which_atm_off_table_option, ierr)  
         if (ierr /= 0) then
            write(*,*) 'unknown value for which_atm_off_table_option ' // trim(s% which_atm_off_table_option)
            return
         end if
         
         call get_surf_PT( &
            s, which_atm_option, off_table_option, r_phot, L_surf, skip_partials, &
            Teff, lnT_surf, dlnT_dL, dlnT_dlnR, dlnT_dlnM, dlnT_dlnkap, &
            lnP_surf, dlnP_dL, dlnP_dlnR, dlnP_dlnM, dlnP_dlnkap, &
            ierr)
         if (ierr /= 0) then
            return
         end if

         s% Teff = Teff
         s% L_phot = L_surf/Lsun
         s% photosphere_L = s% L_phot
         s% photosphere_r = r_phot/Rsun

      end subroutine set_Teff_info_for_eqns
      
      
      subroutine set_hydro_vars( &
            s, nzlo, nzhi, skip_basic_vars, skip_micro_vars, skip_m_grav_and_grav, &
            skip_do_burn_mix, skip_net, skip_neu, skip_kap, skip_grads, skip_rotation, &
            skip_brunt, skip_other_cgrav, skip_nse_fractions, skip_mixing_info, dt, ierr)
         use micro, only: set_micro_vars
         use mlt_info, only: set_mlt_vars, set_grads, adjust_gradT
         use star_utils, only: update_time, total_times, &
            set_k_thermaltime_eq_accretiontime, &
            set_m_grav_and_grav, set_scale_height, get_fraction_NSE_burn
         use hydro_rotation, only: set_rotation_info
         use brunt, only: do_brunt_N2

         type (star_info), pointer :: s         
         integer, intent(in) :: nzlo, nzhi
         logical, intent(in) :: &
            skip_basic_vars, skip_micro_vars, skip_m_grav_and_grav, &
            skip_do_burn_mix, skip_net, skip_neu, skip_kap, skip_brunt, &
            skip_grads, skip_rotation, skip_other_cgrav, &
            skip_nse_fractions, skip_mixing_info
         real(dp), intent(in) :: dt
         integer, intent(out) :: ierr
         
         integer :: nz, num_nse, k
         integer :: time0, clock_rate
         real(dp) :: total_all_before
         logical :: do_Pvisc
         
         include 'formats'

         if (dbg) write(*, *) 'set_hydro_vars', nzlo, nzhi

         ierr = 0
         nz = s% nz
         
         if (.not. skip_basic_vars) then
            if (dbg) write(*, *) 'call set_basic_vars'
            if (s% doing_timing) then
               total_all_before = total_times(s)
               call system_clock(time0,clock_rate)
            end if
            call set_basic_vars(s, nzlo, nzhi, skip_other_cgrav, ierr)
            if (s% doing_timing) &
               call update_time(s, time0, total_all_before, s% time_set_basic_vars)
            if (failed('set_basic_vars')) return
         end if

         if (.not. skip_nse_fractions) then
            num_nse = 0
            do k=nzlo,nzhi
               s% nse_fraction(k) = get_fraction_NSE_burn(s,k)
               if (s% nse_fraction(k) > 0) num_nse = num_nse + 1
            end do
            if (num_nse > 0) write(*,3) 'num_nse', num_nse, s% model_number
         end if

         if (.not. skip_micro_vars) then
            if (dbg) write(*, *) 'call set_micro_vars'
            call set_micro_vars( &
               s, nzlo, nzhi, skip_do_burn_mix, skip_net, skip_neu, skip_kap, dt, ierr)
            if (failed('set_micro_vars')) return
         end if
         
         if (.not. skip_m_grav_and_grav) then 
            ! don't change m_grav or grav during newton iteratons
            if (dbg) write(*, *) 'call set_m_grav_and_grav'
            call set_m_grav_and_grav(s)
         end if
         
         if (dbg) write(*, *) 'call set_scale_height'
         call set_scale_height(s)
         
         if (dbg) write(*, *) 'call set_k_thermaltime_eq_accretiontime'
         call set_k_thermaltime_eq_accretiontime(s)

         if (s% rotation_flag .and. .not. skip_rotation) then
            if (dbg) write(*, *) 'call set_rotation_info'
            if (s% doing_timing) then
               total_all_before = total_times(s)
               call system_clock(time0,clock_rate)
            end if
            call set_rotation_info(s, ierr)
            if (s% doing_timing) &
               call update_time(s, time0, total_all_before, s% time_set_rotation_vars)
            if (failed('set_rotation_info')) return
         end if
         
         do_Pvisc = (s% v_flag .and. s% use_artificial_viscosity) .or. &
            s% use_other_viscosity_pressure
         if (do_Pvisc) then
            if (dbg) write(*, *) 'call set_Pvisc'
            call set_Pvisc(s, nzlo, nzhi, ierr)
            if (failed('set_Pvisc')) return
         else
            s% Pvisc(nzlo:nzhi) = 0
         end if
         
         if (.not. skip_brunt) then
            if (dbg) write(*,*) 'call do_brunt_N2'
            call do_brunt_N2(s, nzlo, nzhi, ierr)
            if (failed('do_brunt_N2')) return
         end if

         if (.not. skip_grads) then            
            if (dbg) write(*, *) 'call set_grads'
            call set_grads(s, ierr)
            if (failed('set_grads')) return
         end if
         
         if (dbg) write(*, *) 'call set_mlt_vars'
         if (s% doing_timing) then
            total_all_before = total_times(s)
            call system_clock(time0,clock_rate)
         end if
         call set_mlt_vars(s, nzlo, nzhi, ierr)
         if (s% doing_timing) &
            call update_time(s, time0, total_all_before, s% time_set_mlt_vars)
         if (failed('set_mlt_vars')) return
         
         
         contains
         
         logical function failed(str)
            character (len=*), intent(in) :: str
            if (ierr == 0) then
               failed = .false.
               return
            end if
            if (s% report_ierr .or. dbg) write(*,*) 'set_hydro_vars failed in call to ' // trim(str)
            failed = .true.
         end function failed

      end subroutine set_hydro_vars
      
      
      subroutine check_rs(s, ierr)
         type (star_info), pointer :: s         
         integer, intent(out) :: ierr
         integer :: k
         logical :: okay
         include 'formats'
         ierr = 0
         okay = .true.
         do k=2, s% nz
            if (s% r(k) > s% r(k-1)) then
               if (s% report_ierr) then
                  write(*,2) 's% r(k) > s% r(k-1)', k, &
                     s% r(k)/Rsun, s% r(k-1)/Rsun, s% r(k)/Rsun-s% r(k-1)/Rsun
               end if
               okay = .false.
            end if
         end do
         if (okay) return
         ierr = -1
         if (s% report_ierr) write(*,*)
      end subroutine check_rs
      
      
      subroutine set_dlnP_dm(s, nzlo, nzhi, dt, ierr)
         type (star_info), pointer :: s         
         integer, intent(in) :: nzlo, nzhi
         real(dp), intent(in) :: dt
         integer, intent(out) :: ierr
         integer :: k
         
         real(dp) :: &
            d_dlnPdm_dlnRp1, d_dlnPdm_dlnR00, d_dlnPdm_dlnRm1, &
            d_dlnPdm_dlnq, d_dlnPdm_dlndqm1, d_dlnPdm_dlndq00, &
            d_dlnPdm_dvelp1, d_dlnPdm_dvel00, d_dlnPdm_dvelm1, &
            d_dlnPdm_dlndp1, d_dlnPdm_dlnd00, d_dlnPdm_dlndm1, &
            d_dlnPdm_dlnTp1, d_dlnPdm_dlnT00, d_dlnPdm_dlnTm1, &         
            d_dlnPdm_dlnPgas00_const_T, d_dlnPdm_dlnT00_const_Pgas, & 
            d_dlnPdm_dlnPgasm1_const_T, d_dlnPdm_dlnTm1_const_Pgas, & 
            d_dlnPdm_dL00, Ppoint, &
            dPpoint_dlnd00, dPpoint_dlndm1, dPpoint_dlnT00, dPpoint_dlnTm1, &
            dPpoint_dlnPgas00_const_T, dPpoint_dlnPgasm1_const_T, &
            dPpoint_dlnT00_const_Pgas, dPpoint_dlnTm1_const_Pgas
         
         do k=nzlo, nzhi
            call do_dlnP_dm(s, k, dt, &
               d_dlnPdm_dlnRp1, d_dlnPdm_dlnR00, d_dlnPdm_dlnRm1, &
               d_dlnPdm_dlnq, d_dlnPdm_dlndqm1, d_dlnPdm_dlndq00, &
               d_dlnPdm_dvelp1, d_dlnPdm_dvel00, d_dlnPdm_dvelm1, &
               d_dlnPdm_dlndp1, d_dlnPdm_dlnd00, d_dlnPdm_dlndm1, &
               d_dlnPdm_dlnTp1, d_dlnPdm_dlnT00, d_dlnPdm_dlnTm1, &         
               d_dlnPdm_dlnPgas00_const_T, d_dlnPdm_dlnT00_const_Pgas, & 
               d_dlnPdm_dlnPgasm1_const_T, d_dlnPdm_dlnTm1_const_Pgas, & 
               d_dlnPdm_dL00, &
               Ppoint, &
               dPpoint_dlnd00, dPpoint_dlndm1, dPpoint_dlnT00, dPpoint_dlnTm1, &
               dPpoint_dlnPgas00_const_T, dPpoint_dlnPgasm1_const_T, &
               dPpoint_dlnT00_const_Pgas, dPpoint_dlnTm1_const_Pgas, &
               ierr)
            if (ierr /= 0) return
         end do
         
      end subroutine set_dlnP_dm
      
      
      subroutine set_basic_vars( &
            s, nzlo, nzhi, skip_other_cgrav, ierr)
         use star_utils, only: total_angular_momentum, set_i_rot, set_omega
         type (star_info), pointer :: s         
         integer, intent(in) :: nzlo, nzhi
         logical, intent(in) :: skip_other_cgrav
         integer, intent(out) :: ierr
         integer :: j, k, species
         real(dp) :: twoGmrc2, r2, alfa, beta, sum_xa
         
         include 'formats'
         
         if (dbg) write(*,4) 'enter set_basic_vars: nzlo, nzhi, nz', nzlo, nzhi, s% nz
         ierr = 0
         species = s% species
         s% L_phot = s% L(1)/Lsun
         
!$OMP PARALLEL DO PRIVATE(j,k,twoGmrc2,r2,sum_xa)
         do k=nzlo, nzhi
            if (s% lnPgas_flag) then
               s% Pgas(k) = exp(s% lnPgas(k))
            else
               s% rho(k) = exp(s% lnd(k))
            end if
            s% T(k) = exp(s% lnT(k))
            s% r(k) = exp(s% lnR(k))
            r2 = s% r(k)**2
            s% area(k) = pi4*r2
            do j=1,species 
               s% xa(j,k) = max(0d0, min(1d0, s% xa(j,k)))
            end do
            sum_xa = sum(s% xa(1:species,k))
            if (abs(sum_xa - 1d0) > 1d-12) then
               do j=1,species
                  s% xa(j,k) = s% xa(j,k)/sum_xa
               end do
            end if
         end do    
!$OMP END PARALLEL DO

         call set_rmid_and_Amid(s, nzlo, nzhi, ierr)
         
         if (s% rotation_flag) then
            call set_i_rot(s)
            call set_omega(s, 'hydro_vars')
         end if
         
         if (.not. skip_other_cgrav) then
            if (s% use_other_cgrav) then
               call s% other_cgrav(s% id, ierr)
               if (ierr /= 0) then
                  if (s% report_ierr .or. dbg) write(*, *) 'other_cgrav returned ierr', ierr
                  return
               end if
            else
               s% cgrav(nzlo:nzhi) = standard_cgrav
            end if
         end if
         
      end subroutine set_basic_vars

      
      subroutine set_rmid_and_Amid(s, nzlo, nzhi, ierr)
         type (star_info), pointer :: s         
         integer, intent(in) :: nzlo, nzhi
         integer, intent(out) :: ierr
         
         integer :: k, nz
         
         include 'formats'
         
         ierr = 0
         nz = s% nz
!$OMP PARALLEL DO PRIVATE(k)
         do k=nzlo, nzhi
            call do_Amid(k)
         end do
!$OMP END PARALLEL DO
         
         
         contains
         
         
         subroutine do_Amid(k) ! set Amid, dAmid_dlnR00, and dAmid_dlnRp1
            integer, intent(in) :: k
            real(dp) :: r003, rp13, rmid, rmid2, drmid2_dlnR00, drmid2_dlnRp1
            r003 = s% r(k)**3
            if (k < nz) then
               rp13 = s% r(k+1)**3
            else
               rp13 = s% R_center**3
            end if
            rmid = ((r003 + rp13)/2)**(1d0/3d0)
            s% rmid(k) = rmid
            rmid2 = rmid**2
            drmid2_dlnR00 = r003/rmid
            drmid2_dlnRp1 = rp13/rmid
            s% Amid(k) = 4*pi*rmid2
            s% dAmid_dlnR00(k) = 4*pi*drmid2_dlnR00
            s% dAmid_dlnRp1(k) = 4*pi*drmid2_dlnRp1
         end subroutine do_Amid
         

      end subroutine set_rmid_and_Amid
      
      
      subroutine set_Pvisc(s, nzlo, nzhi, ierr)
         type (star_info), pointer :: s         
         integer, intent(in) :: nzlo, nzhi
         integer, intent(out) :: ierr
         
         integer :: k, nz
         real(dp) :: l1, l22
         
         include 'formats'
         
         ierr = 0
         if (s% use_other_viscosity_pressure) then
            call s% other_viscosity_pressure(s% id, ierr)
            return
         end if
         
         l1 = s% l1_coef
         l22 = s% l2_coef**2
         nz = s% nz
!x$OMP PARALLEL DO PRIVATE(k)
         do k=nzlo, nzhi
            call do_Pvisc(k)
         end do
!x$OMP END PARALLEL DO
         
         
         contains
         
         
         subroutine do_Pvisc(k) ! set Pvisc and its partials
            use eos_def, only: i_gamma1
            integer, intent(in) :: k
            real(dp) :: dA_dlnR, &
               cs, d_cs_dlnd, d_cs_dlnT, &
               dv, d_dv_dv00, d_dv_dvp1, &
               etav, d_etav_dlnd, d_etav_dlnT, &
               d_etav_dv00, d_etav_dvp1, &
               c00, cp1, Amid2, d_c00_dlnR00, d_c00_dlnRp1, d_cp1_dlnR00, d_cp1_dlnRp1, &
               del, d_del_dlnR00, d_del_dlnRp1, d_del_dv00, d_del_dvp1

            if (l1 == 0) then
               cs = 0; d_cs_dlnd = 0; d_cs_dlnT = 0
            else
               cs = sqrt(s% gamma1(k)*s% P(k)/s% rho(k))
               d_cs_dlnd = (s% d_eos_dlnd(i_gamma1,k)/s% gamma1(k) + s% chiRho(k) - 1)*cs/2
               d_cs_dlnT = (s% d_eos_dlnT(i_gamma1,k)/s% gamma1(k) + s% chiT(k))*cs/2
            end if
            
            if (k == nz) then
               dv = 0; d_dv_dv00 = 0; d_dv_dvp1 = 0
            else if (s% v(k+1) <= s% v(k)) then
               dv = 0; d_dv_dv00 = 0; d_dv_dvp1 = 0
            else
               dv = s% v(k+1) - s% v(k)
               d_dv_dv00 = -1
               d_dv_dvp1 = 1
            end if
            
            etav = 0.75d0*s% rho(k)*(l1*cs + l22*dv)
            if (etav == 0) then
               s% Pvisc(k) = 0
               s% dPvisc_dlnd(k) = 0
               s% dPvisc_dlnT(k) = 0
               s% dPvisc_dlnR00(k) = 0
               s% dPvisc_dvel00(k) = 0
               s% dPvisc_dlnRp1(k) = 0
               s% dPvisc_dvelp1(k) = 0
               return
            end if
            d_etav_dlnd = 0.75d0*s% rho(k)*(l1*(cs + d_cs_dlnd) + l22*dv)
            d_etav_dlnT = 0.75d0*s% rho(k)*l1*d_cs_dlnT
            d_etav_dv00 = 0.75d0*s% rho(k)*l22*d_dv_dv00
            d_etav_dvp1 = 0.75d0*s% rho(k)*l22*d_dv_dvp1
            
            c00 = (1 - s% area(k)/(3*s% Amid(k)))
            Amid2 = s% Amid(k)**2
            dA_dlnR = 2*s% area(k)
            d_c00_dlnR00 = (s% area(k)*s% dAmid_dlnR00(k) - s% Amid(k)*dA_dlnR)/(3*Amid2)
            d_c00_dlnRp1 = s% area(k)*s% dAmid_dlnRp1(k)/(3*Amid2)
            if (k < nz) then
               cp1 = (1 - s% area(k+1)/(3*s% Amid(k)))
               d_cp1_dlnR00 = s% area(k+1)*s% dAmid_dlnR00(k)/(3*Amid2)
               dA_dlnR = 2*s% area(k+1)
               d_cp1_dlnRp1 = (s% area(k+1)*s% dAmid_dlnRp1(k) - s% Amid(k)*dA_dlnR)/(3*Amid2)
               del = s% v(k)*c00 - s% v(k+1)*cp1
               d_del_dlnR00 = s% v(k)*d_c00_dlnR00 - s% v(k+1)*d_cp1_dlnR00
               d_del_dlnRp1 = s% v(k)*d_c00_dlnRp1 - s% v(k+1)*d_cp1_dlnRp1
               d_del_dv00 = c00
               d_del_dvp1 = -cp1
            else
               del = s% v(k)*c00
               d_del_dlnR00 = s% v(k)*d_c00_dlnR00
               d_del_dv00 = c00
               d_del_dvp1 = 0
               d_del_dlnRp1 = 0
            end if
            
            s% Pvisc(k) = -etav*del
            s% dPvisc_dlnd(k) = -d_etav_dlnd*del
            s% dPvisc_dlnT(k) = -d_etav_dlnT*del
            s% dPvisc_dlnR00(k) = -etav*d_del_dlnR00
            s% dPvisc_dvel00(k) = -(d_etav_dv00*del + etav*d_del_dv00)
            if (k < nz) then
               s% dPvisc_dlnRp1(k) = -etav*d_del_dlnRp1
               s% dPvisc_dvelp1(k) = -(d_etav_dvp1*del + etav*d_del_dvp1)
            end if
         
         end subroutine do_Pvisc
         
         
      end subroutine set_Pvisc
      
      
      real(dp) function get_Teff(s, ierr) result(Teff)
         use atm_lib, only: atm_option
         use star_utils, only: get_r_phot
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         
         integer :: which_atm_option, off_table_option
         real(dp) :: r_phot, L_surf
         logical :: skip_partials
         real(dp) :: &
            lnT_surf, dlnT_dL, dlnT_dlnR, dlnT_dlnM, dlnT_dlnkap, &
            lnP_surf, dlnP_dL, dlnP_dlnR, dlnP_dlnM, dlnP_dlnkap
            
         ierr = 0
         
         r_phot = get_r_phot(s)
         L_surf = s% L(1)
         skip_partials = .true.
         
         which_atm_option = atm_option(s% which_atm_option, ierr)
         if (ierr /= 0) then
            write(*,*) 'unknown value for which_atm_option ' // trim(s% which_atm_option)
            return
         end if
      
         off_table_option = atm_option(s% which_atm_off_table_option, ierr)  
         if (ierr /= 0) then
            write(*,*) 'unknown value for which_atm_off_table_option ' // &
               trim(s% which_atm_off_table_option)
            return
         end if
         
         call get_surf_PT( &
            s, which_atm_option, off_table_option, r_phot, L_surf, skip_partials, &
            Teff, lnT_surf, dlnT_dL, dlnT_dlnR, dlnT_dlnM, dlnT_dlnkap, &
            lnP_surf, dlnP_dL, dlnP_dlnR, dlnP_dlnM, dlnP_dlnkap, &
            ierr)
         if (ierr /= 0) then
            return
         end if

      end function get_Teff

      
      subroutine get_surf_PT( &
            s, which_atm_option, off_table_option, r_phot, L_surf, skip_partials, &
            Teff, lnT_surf, dlnT_dL, dlnT_dlnR, dlnT_dlnM, dlnT_dlnkap, &
            lnP_surf, dlnP_dL, dlnP_dlnR, dlnP_dlnM, dlnP_dlnkap, &
            ierr)
         use atm_lib
         use atm_def
         use chem_def
         use create_atm, only: get_Paczynski_atm_surf_PT

         type (star_info), pointer :: s
         integer, intent(in) :: which_atm_option, off_table_option
         real(dp), intent(in) :: r_phot, L_surf
         logical, intent(in) :: skip_partials
         real(dp), intent(out) :: Teff, &
            lnT_surf, dlnT_dL, dlnT_dlnR, dlnT_dlnM, dlnT_dlnkap, &
            lnP_surf, dlnP_dL, dlnP_dlnR, dlnP_dlnM, dlnP_dlnkap
         integer, intent(out) :: ierr
         
         real(dp) :: opacity, X, Y, Z, M, tau_surface, tau_center_cell1, &
            kap_surf, err, P_surf_atm, P_surf
         integer :: iters, num_atm_structure_points
         real(dp), pointer :: atm_structure_results(:,:)
         logical :: save_atm_structure_info
         
         include 'formats'
      
         opacity = s% opacity(1)
         X = s% X(1); Y = s% Y(1); Z = 1 - (X+Y)
         tau_surface = s% tau_factor*s% tau_base ! tau at outer edge of cell 1
         M = s% m_grav(1)
         
         !write(*,1) 'get_surf_PT tau_surface', tau_surface

         !write(*,*) 's% dtau1_start > tau_surface', s% dtau1_start > tau_surface
         if (s% use_other_atm) then
            call s% other_atm( &
               s% id, M, r_phot, L_surf, X, Z, opacity, Teff, &
               lnT_surf, dlnT_dL, dlnT_dlnR, dlnT_dlnM, dlnT_dlnkap, &
               lnP_surf, dlnP_dL, dlnP_dlnR, dlnP_dlnM, dlnP_dlnkap, &
               which_atm_option, s% atm_switch_to_grey_as_backup, ierr)
         else if (which_atm_option == atm_grey_and_kap) then
            call atm_get_grey_and_kap( &
               tau_surface, opacity, &
               s% cgrav(1), M, r_phot, L_surf, X, Z, s% abar(1), s% zbar(1), & 
               s% species, s% chem_id, s% net_iso, s% xa(:,1), &
               s% atm_grey_and_kap_max_tries, s% atm_grey_and_kap_atol, s% atm_grey_and_kap_rtol, &
               s% eos_handle, s% kap_handle, &
               lnT_surf, dlnT_dL, dlnT_dlnR, dlnT_dlnM, & 
               lnP_surf, dlnP_dL, dlnP_dlnR, dlnP_dlnM, & 
               kap_surf, Teff, iters, err, ierr)
            dlnT_dlnkap = 0
            dlnP_dlnkap = 0
            if (s% trace_atm_grey_and_kap .and. .not. skip_partials) then
               write(*,2) 'grey&kap: itrs, err, kap_s, kap1', &
                  iters, err, kap_surf, opacity
            end if    
         else if (which_atm_option == atm_Paczynski_grey) then
            call get_Paczynski_atm_surf_PT( &
               s, r_phot, L_surf, skip_partials, Teff, &
               lnT_surf, dlnT_dL, dlnT_dlnR, dlnT_dlnM, dlnT_dlnkap, &
               lnP_surf, dlnP_dL, dlnP_dlnR, dlnP_dlnM, dlnP_dlnkap, &
               ierr)
            if (s% trace_atm_Paczynski_grey .and. .not. skip_partials) then
               write(*,*) 
               write(*,'(99a20)') &
                  'Paczynski_grey', 'Teff', 'lnT', 'dlnT_dL', 'dlnT_dlnR', 'lnP', 'dlnP_dL', 'dlnP_dlnR'
               write(*,'(i20,99e20.10)') &
                  s% model_number, Teff, lnT_surf, dlnT_dL, dlnT_dlnR, lnP_surf, dlnP_dL, dlnP_dlnR
            end if    
         else if (which_atm_option == atm_grey_irradiated) then
            if (s% atm_grey_irradiated_simple_kap_th .and. &
                  s% atm_grey_irr_kap_v_div_kap_th > 0) &
               opacity = s% atm_grey_irradiated_kap_v/s% atm_grey_irr_kap_v_div_kap_th
            call atm_grey_irradiated_get( &
               s% atm_grey_irradiated_T_eq, s% atm_grey_irradiated_kap_v, &
               opacity, s% atm_grey_irradiated_simple_kap_th, &
               s% atm_grey_irr_kap_v_div_kap_th, s% atm_grey_irradiated_P_surf, &
               s% cgrav(1), M, r_phot, L_surf, X, Z, s% abar(1), s% zbar(1), & 
               s% species, s% chem_id, s% net_iso, s% xa(:,1), &
               s% atm_grey_irradiated_max_tries, s% atm_grey_irradiated_atol, &
               s% atm_grey_irradiated_rtol, &
               s% eos_handle, s% kap_handle, &
               lnT_surf, dlnT_dL, dlnT_dlnR, dlnT_dlnM, & 
               kap_surf, tau_surface, Teff, iters, err, ierr)
            s% tau_factor = tau_surface/s% tau_base
            lnP_surf = log(s% atm_grey_irradiated_P_surf)
            dlnP_dL = 0
            dlnP_dlnR = 0
            dlnP_dlnM = 0
            dlnT_dlnkap = 0
            dlnP_dlnkap = 0
            if (s% trace_atm_grey_irradiated .and. .not. skip_partials) then
               write(*,2) 'grey_irradiated: itrs, err, kap_s, kap1, tau_surface, Teff, T', &
                  iters, err, kap_surf, opacity, tau_surface, Teff, exp(lnT_surf)
            end if    
         else if (atm_by_integration(which_atm_option)) then
            save_atm_structure_info = .false.
            nullify(atm_structure_results)
            call atm_get_int_T_tau( &
               s% atm_int_errtol, s% cgrav(1), M, r_phot, L_surf, X, Z, s% abar(1), s% zbar(1), & 
               s% species, s% chem_id, s% net_iso, s% xa(:,1), &
               which_atm_option, s% eos_handle, s% kap_handle, save_atm_structure_info, &
               tau_surface, skip_partials, Teff, &
               lnT_surf, dlnT_dL, dlnT_dlnR, dlnT_dlnM, dlnT_dlnkap, &
               lnP_surf, dlnP_dL, dlnP_dlnR, dlnP_dlnM, dlnP_dlnkap, &
               num_atm_structure_points, atm_structure_results, ierr)
         else if (atm_by_table(which_atm_option)) then
            if (s% tau_factor /= 1) then
               write(*,*) 'cannot simultaneously have atm by table and tau_factor /= 1'
               ierr = -1
               return
            end if
            s% tau_base = atm_tau_base(which_atm_option, ierr)
            if (ierr == 0) call atm_get_table( &
               which_atm_option, off_table_option, &
               s% cgrav(1), M, r_phot, L_surf, &
               lnT_surf, dlnT_dL, dlnT_dlnR, dlnT_dlnM, & 
               lnP_surf, dlnP_dL, dlnP_dlnR, dlnP_dlnM, & 
               opacity, X, Z, s% abar(1), s% zbar(1), & 
               s% species, s% chem_id, s% net_iso, s% xa(:,1), &
               s% atm_grey_and_kap_max_tries, &
               s% atm_grey_and_kap_atol, s% atm_grey_and_kap_rtol, &
               s% eos_handle, s% kap_handle, Teff, ierr)
            dlnT_dlnkap = 0
            dlnP_dlnkap = 0
         else
            call atm_get_grey( &
               tau_surface, s% cgrav(1), M, r_phot, L_surf, opacity, Teff, &
               lnT_surf, dlnT_dL, dlnT_dlnR, dlnT_dlnM, dlnT_dlnkap, &
               lnP_surf, dlnP_dL, dlnP_dlnR, dlnP_dlnM, dlnP_dlnkap, &
               ierr)
         end if
         
         if (s% surface_extra_Pgas /= 0d0) then
            P_surf_atm = exp(lnP_surf)
            P_surf = P_surf_atm + s% surface_extra_Pgas
            if (P_surf < 1d-50) then
               lnP_surf = -50*ln10
               dlnP_dL = 0
               dlnP_dlnR = 0
               dlnP_dlnM = 0
               dlnP_dlnkap = 0
            else
               lnP_surf = log(P_surf)
               dlnP_dL = dlnP_dL*P_surf_atm/P_surf
               dlnP_dlnR = dlnP_dlnR*P_surf_atm/P_surf
               dlnP_dlnM = dlnP_dlnM*P_surf_atm/P_surf
               dlnP_dlnkap = dlnP_dlnkap*P_surf_atm/P_surf
            end if
         end if

         !call show
         !stop 'get PT surf'
         
         if (is_bad_num(lnT_surf) .or. is_bad_num(lnP_surf)) then
            call show
            ierr = -1
            return
            !stop 'get_surf_PT'
         end if

         if (ierr /= 0) then
            if (s% report_ierr) then
               write(*, *) 'atm_get returned ierr', ierr
               write(*, *)
            end if
            return
         end if
         
         
         contains
         
         subroutine show
            include 'formats'
            write(*,1) 'M =', M
            write(*,1) 'R =', r_phot
            write(*,1) 'L =', L_surf
            write(*,1) 'X =', X
            write(*,1) 'Z =', Z
            write(*,1) 'opacity =', opacity
            write(*,2) 'which_atm_option =', which_atm_option
            write(*,*) trim(s% which_atm_option)
            write(*,*) 's% atm_switch_to_grey_as_backup', s% atm_switch_to_grey_as_backup
            write(*,*)
            write(*,1) 's% tau_factor', s% tau_factor
            write(*,*)
            write(*,1) 'Teff', Teff
            write(*,1) 'logT_surf', lnT_surf/ln10
            write(*,1) 'logP_surf', lnP_surf/ln10
            write(*,*)
            write(*,1) 'lnT_surf', lnT_surf
            write(*,1) 'lnP_surf', lnP_surf
            write(*,*)
            write(*,1) 'dlnT_dL', dlnT_dL
            write(*,1) 'dlnT_dlnR', dlnT_dlnR
            write(*,1) 'dlnT_dlnM', dlnT_dlnM
            write(*,1) 'dlnT_dlnkap', dlnT_dlnkap
            write(*,*)
            write(*,1) 'dlnP_dL', dlnP_dL
            write(*,1) 'dlnP_dlnR', dlnP_dlnR
            write(*,1) 'dlnP_dlnM', dlnP_dlnM
            write(*,1) 'dlnP_dlnkap', dlnP_dlnkap
            write(*,*)
         end subroutine show
         
      end subroutine get_surf_PT
      
      
      subroutine get_dVARDOT_dVAR_pt(s, k, dt, d_dvarm1, d_dvar00, d_dvarp1)
         type (star_info), pointer :: s 
         integer, intent(in) :: k
         real(dp), intent(in) :: dt
         real(dp), intent(out) :: d_dvarm1, d_dvar00, d_dvarp1
         call get_dVARDOT_dVAR_cell(s, k, dt, d_dvarm1, d_dvar00, d_dvarp1)
      end subroutine get_dVARDOT_dVAR_pt
      
      
      subroutine get_dVARDOT_dVAR_cell(s, k, dt, d_dvarm1, d_dvar00, d_dvarp1)
         type (star_info), pointer :: s 
         integer, intent(in) :: k
         real(dp), intent(in) :: dt
         real(dp), intent(out) :: d_dvarm1, d_dvar00, d_dvarp1
         
         if (k >= s% k_below_just_added .or. dt <= 0) then
            d_dvarm1 = 0
            d_dvar00 = s% dVARdot_dVAR
            d_dvarp1 = 0
            return
         end if
         
         d_dvarm1 = 0
         d_dvar00 = 1d0/max(1d-99,s% del_t_for_just_added(k))
         d_dvarp1 = 0

      end subroutine get_dVARDOT_dVAR_cell
            

      real(dp) function eval_cell_collapse_timescale(s) result(dt)
         type (star_info), pointer :: s
         
         integer :: nz, k
         real(dp) :: dr, dv, cell_dt
         include 'formats'
         
         dt = 1d99
         if (.not. s% v_flag) return
         
         nz = s% nz
         do k=1,nz-1
            dv = s% v(k+1) - s% v(k)
            if (dv <= 0) cycle
            dr = s% r(k) - s% r(k+1)
            cell_dt = dr/dv
            if (cell_dt < dt) dt = cell_dt
         end do
         
      end function eval_cell_collapse_timescale
      

      real(dp) function eval_chem_timescale(s) result(dt)
         type (star_info), pointer :: s

         real(dp), pointer, dimension(:) :: sig
         integer :: nz, k, j, species, min_k, min_j
         real(dp) :: dt0, dm, sig00, sigp1, dxdt_nuc, dxdt_chem, &
            x00, xm1, xp1, dx00, dxp1, flux00, fluxp1, dxdt_mix
         
         include 'formats'
         
         dt = 1d99
         min_k = 0
         min_j = 0
         nz = s% nz
         species = s% species
         sig => s% sig
         
         do k = 1, s% nz
            dm = s% dm(k)
            sig00 = sig(k)   
            if (k < nz) then
               sigp1 = sig(k+1)
            else
               sigp1 = 0
            end if
            do j=1,species
               if (.not. s% do_burn) then
                  dxdt_nuc = 0
               else
                  dxdt_nuc = s% dxdt_nuc(j,k)
               end if           
               x00 = s% xa(j,k)
               if (x00 < 1d-10) cycle
               if (.not. s% do_mix) then
                  dxdt_mix = 0
               else
                  if (k > 1) then
                     xm1 = s% xa(j,k-1)
                     dx00 = xm1 - x00
                     flux00 = -sig00*dx00
                  else
                     flux00 = 0
                  end if         
                  if (k < nz) then
                     xp1 = s% xa(j,k+1)
                     dxp1 = x00 - xp1
                     fluxp1 = -sigp1*dxp1
                  else
                     fluxp1 = 0
                  end if
                  dxdt_mix = (fluxp1 - flux00)/dm
               end if
               dxdt_chem = dxdt_nuc + dxdt_mix
               if (dxdt_chem >= 0) cycle   
               dt0 = max(1d-20, x00)/max(1d-50, abs(dxdt_chem))
               if (dt0 < dt) then
                  dt = dt0
                  min_k = k
                  min_j = j
               end if
            end do
         end do
         
         return
         
         j = min_j
         k = min_k
         write(*,3) 'chem timescale', j, k, dt, s% xa(j,k), s% dxdt_nuc(j,k) 
         
      end function eval_chem_timescale


      subroutine do_dlnP_dm(s, k, dt, &
            d_dlnPdm_dlnRp1, d_dlnPdm_dlnR00, d_dlnPdm_dlnRm1, &
            d_dlnPdm_dlnq, d_dlnPdm_dlndqm1, d_dlnPdm_dlndq00, &
            d_dlnPdm_dvelp1, d_dlnPdm_dvel00, d_dlnPdm_dvelm1, &
            d_dlnPdm_dlndp1, d_dlnPdm_dlnd00, d_dlnPdm_dlndm1, &
            d_dlnPdm_dlnTp1, d_dlnPdm_dlnT00, d_dlnPdm_dlnTm1, &         
            d_dlnPdm_dlnPgas00_const_T, d_dlnPdm_dlnT00_const_Pgas, & 
            d_dlnPdm_dlnPgasm1_const_T, d_dlnPdm_dlnTm1_const_Pgas, & 
            d_dlnPdm_dL00, &
            Ppoint, &
            dPpoint_dlnd00, dPpoint_dlndm1, dPpoint_dlnT00, dPpoint_dlnTm1, &
            dPpoint_dlnPgas00_const_T, dPpoint_dlnPgasm1_const_T, &
            dPpoint_dlnT00_const_Pgas, dPpoint_dlnTm1_const_Pgas, &
            ierr)
         type (star_info), pointer :: s 
         integer, intent(in) :: k
         real(dp), intent(in) :: dt
         real(dp), intent(out) :: &
            d_dlnPdm_dlnRp1, d_dlnPdm_dlnR00, d_dlnPdm_dlnRm1, &
            d_dlnPdm_dlnq, d_dlnPdm_dlndqm1, d_dlnPdm_dlndq00, &
            d_dlnPdm_dvelp1, d_dlnPdm_dvel00, d_dlnPdm_dvelm1, &
            d_dlnPdm_dlndp1, d_dlnPdm_dlnd00, d_dlnPdm_dlndm1, &
            d_dlnPdm_dlnTp1, d_dlnPdm_dlnT00, d_dlnPdm_dlnTm1, &         
            d_dlnPdm_dlnPgas00_const_T, d_dlnPdm_dlnT00_const_Pgas, & 
            d_dlnPdm_dlnPgasm1_const_T, d_dlnPdm_dlnTm1_const_Pgas, & 
            d_dlnPdm_dL00, &
            Ppoint, &
            dPpoint_dlnd00, dPpoint_dlndm1, dPpoint_dlnT00, dPpoint_dlnTm1, &
            dPpoint_dlnPgas00_const_T, dPpoint_dlnPgasm1_const_T, &
            dPpoint_dlnT00_const_Pgas, dPpoint_dlnTm1_const_Pgas
         integer, intent(out) :: ierr
         
         real(dp) :: &
            m, dv_dt, dlnP_dm, fac, vfac, &
            rPterm, accel, d_rPterm_dlnR, dlnP_dm_accel_term, d_accel_dlnR, &
            d_rPterm_dlnd00, d_rPterm_dlnT00, d_rPterm_dlndm1, d_rPterm_dlnTm1, &
            d_rPterm_dlnPgas00_const_T, d_rPterm_dlnT00_const_Pgas, &
            d_rPterm_dlnPgasm1_const_T, d_rPterm_dlnTm1_const_Pgas, &
            twoGmrc2, invfac, gr_factor, dtwoGmrc2_dlnR, d_invfac_dlnR, d_gr_factor_dlnR00, &
            d_conv_dP_term_dlnPgas00_const_T, d_conv_dP_term_dlnPgasm1_const_T, &
            d_conv_dP_term_dlnT00_const_Pgas, d_conv_dP_term_dlnTm1_const_Pgas, &
            d_accel_dlnR00, d_accel_dlnRp1, d_accel_dlnRm1, &
            d_dvardt_dvarm1, d_dvardt_dvar00, d_dvardt_dvarp1, &
            dfac_dlnd00, dfac_dlnT00, dfac_dlndm1, dfac_dlnTm1, dfac_dlnR, dfac_dL, &
            dfac_dlnPgas00_const_T, dfac_dlnPgasm1_const_T, &
            dfac_dlnT00_const_Pgas, dfac_dlnTm1_const_Pgas
            
         logical :: local_v_flag, dbg

         include 'formats'
         
         ierr = 0
         
         dbg = .false.

         s% dlnP_dm(k) = 0
         dlnP_dm = 0
         d_dlnPdm_dlnRp1 = 0
         d_dlnPdm_dlnR00 = 0
         d_dlnPdm_dlnRm1 = 0
         d_dlnPdm_dlnq = 0
         d_dlnPdm_dlndqm1 = 0
         d_dlnPdm_dlndq00 = 0
         d_dlnPdm_dvelp1 = 0
         d_dlnPdm_dvel00 = 0
         d_dlnPdm_dvelm1 = 0
         d_dlnPdm_dlndp1 = 0
         d_dlnPdm_dlnd00 = 0
         d_dlnPdm_dlndm1 = 0
         d_dlnPdm_dlnTp1 = 0
         d_dlnPdm_dlnT00 = 0
         d_dlnPdm_dlnTm1 = 0
         d_dlnPdm_dlnPgas00_const_T = 0
         d_dlnPdm_dlnT00_const_Pgas = 0
         d_dlnPdm_dlnPgasm1_const_T = 0
         d_dlnPdm_dlnTm1_const_Pgas = 0
         d_dlnPdm_dL00 = 0

         call eval_dlnPdm_qhse(s, k, m, &
            dlnP_dm, &
            d_dlnPdm_dlnR00, d_dlnPdm_dlnq, d_dlnPdm_dL00, &
            d_dlnPdm_dlnd00, d_dlnPdm_dlnT00, &
            d_dlnPdm_dlndm1, d_dlnPdm_dlnTm1, &
            d_dlnPdm_dlnPgas00_const_T, d_dlnPdm_dlnT00_const_Pgas, &
            d_dlnPdm_dlnPgasm1_const_T, d_dlnPdm_dlnTm1_const_Pgas, &
            Ppoint, &
            dPpoint_dlnd00, dPpoint_dlndm1, dPpoint_dlnT00, dPpoint_dlnTm1, &
            dPpoint_dlnPgas00_const_T, dPpoint_dlnPgasm1_const_T, &
            dPpoint_dlnT00_const_Pgas, dPpoint_dlnTm1_const_Pgas, &
            ierr)
         if (ierr /= 0) return
         s% dlnP_dm(k) = dlnP_dm

         d_dlnPdm_dlnRm1 = 0

         if (is_bad_num(s% dlnP_dm(k))) then
            write(*,*) 'k', k
            write(*,1) 'dlnP_dm(k)', s% dlnP_dm(k)
            write(*,*)
            ierr = -1
            return
         end if
         
         if (s% conv_dP_term(k) /= 0 .and.  s% conv_dP_term_factor > 0) then 
            ! include effect of convective turbulence on pressure gradient
            
            fac = 1d0/(1d0 + s% conv_dP_term_factor*s% conv_dP_term(k))

            s% dlnP_dm(k) = s% dlnP_dm(k)*fac
            
            dfac_dlnd00 = -fac**2*s% conv_dP_term_factor*s% d_conv_dP_term_dlnd00(k)
            dfac_dlnT00 = -fac**2*s% conv_dP_term_factor*s% d_conv_dP_term_dlnT00(k)
            dfac_dlndm1 = -fac**2*s% conv_dP_term_factor*s% d_conv_dP_term_dlndm1(k)
            dfac_dlnTm1 = -fac**2*s% conv_dP_term_factor*s% d_conv_dP_term_dlnTm1(k)
            dfac_dlnR = -fac**2*s% conv_dP_term_factor*s% d_conv_dP_term_dlnR(k)
            dfac_dL = -fac**2*s% conv_dP_term_factor*s% d_conv_dP_term_dL(k)
                                
            if (s% lnPgas_flag) then
               d_conv_dP_term_dlnPgas00_const_T = &
                  dfac_dlnd00*s% dlnRho_dlnPgas_const_T(k)
               d_conv_dP_term_dlnPgasm1_const_T = &
                  dfac_dlndm1*s% dlnRho_dlnPgas_const_T(k-1)
               d_conv_dP_term_dlnT00_const_Pgas = &
                  dfac_dlnT00 + dfac_dlnd00*s% dlnRho_dlnT_const_Pgas(k)
               d_conv_dP_term_dlnTm1_const_Pgas = &
                  dfac_dlnTm1 + dfac_dlndm1*s% dlnRho_dlnT_const_Pgas(k-1)
                  
               dfac_dlnPgas00_const_T = &
                  -fac**2*s% conv_dP_term_factor*d_conv_dP_term_dlnPgas00_const_T
               dfac_dlnPgasm1_const_T = &
                  -fac**2*s% conv_dP_term_factor*d_conv_dP_term_dlnPgasm1_const_T
               dfac_dlnT00_const_Pgas = &
                  -fac**2*s% conv_dP_term_factor*d_conv_dP_term_dlnT00_const_Pgas
               dfac_dlnTm1_const_Pgas = &
                  -fac**2*s% conv_dP_term_factor*d_conv_dP_term_dlnTm1_const_Pgas
                  
               d_dlnPdm_dlnPgas00_const_T = d_dlnPdm_dlnPgas00_const_T*fac + &
                  s% dlnP_dm(k)*dfac_dlnPgas00_const_T
               d_dlnPdm_dlnT00_const_Pgas = d_dlnPdm_dlnT00_const_Pgas*fac + &
                  s% dlnP_dm(k)*dfac_dlnT00_const_Pgas
               d_dlnPdm_dlnPgasm1_const_T = d_dlnPdm_dlnPgasm1_const_T*fac + &
                  s% dlnP_dm(k)*dfac_dlnPgasm1_const_T
               d_dlnPdm_dlnTm1_const_Pgas = d_dlnPdm_dlnTm1_const_Pgas*fac + &
                  s% dlnP_dm(k)*dfac_dlnTm1_const_Pgas
            else
               d_dlnPdm_dlnd00 = d_dlnPdm_dlnd00*fac + s% dlnP_dm(k)*dfac_dlnd00
               d_dlnPdm_dlnT00 = d_dlnPdm_dlnT00*fac + s% dlnP_dm(k)*dfac_dlnT00
               d_dlnPdm_dlndm1 = d_dlnPdm_dlndm1*fac + s% dlnP_dm(k)*dfac_dlndm1
               d_dlnPdm_dlnTm1 = d_dlnPdm_dlnTm1*fac + s% dlnP_dm(k)*dfac_dlnTm1
            end if
            
            d_dlnPdm_dlnR00 = d_dlnPdm_dlnR00*fac + s% dlnP_dm(k)*dfac_dlnR
            d_dlnPdm_dL00 = d_dlnPdm_dL00*fac + s% dlnP_dm(k)*dfac_dL

         end if
         
         if (s% use_gr_factors) then ! GR gravity factor = 1/sqrt(1-2Gm/(rc^2))
            ! note: this uses m_grav
            
            twoGmrc2 = 2*s% cgrav(k)*s% m_grav(k)/(s% r(k)*clight**2)
            invfac = sqrt(1d0 - twoGmrc2)
            gr_factor = 1d0/invfac
            
            dtwoGmrc2_dlnR = -twoGmrc2
            d_invfac_dlnR = -dtwoGmrc2_dlnR/(2*invfac)
            d_gr_factor_dlnR00 = -d_invfac_dlnR/invfac**2
            
            if (dbg) then
               write(*,2) 'std s% dlnP_dm(k)', k, s% dlnP_dm(k)
               write(*,2) 'gr_factor', k, gr_factor
               write(*,2) 'new s% dlnP_dm(k)', k, s% dlnP_dm(k)*gr_factor
               !write(*,2) '', k, 
               !write(*,2) '', k, 
               !write(*,2) '', k, 
            end if

            s% dlnP_dm(k) = s% dlnP_dm(k)*gr_factor                        
            d_dlnPdm_dlnR00 = &
               d_dlnPdm_dlnR00*gr_factor + s% dlnP_dm(k)*d_gr_factor_dlnR00
            
         end if
         
         
         if (s% v_flag) then ! include velocity term
               
            local_v_flag = &
               (s% xh_old(s% i_lnT,k)/ln10 >= s% velocity_logT_lower_bound)
            
            fac = s% accel_factor
            if (s% use_mass_corrections) fac = fac*s% mass_correction_start(k)
            
            if (local_v_flag) then
               dv_dt = s% dv_dt(k) ! = (v(k) - vstart(k))/dt
            else ! assume vstart(k) = 0 and constant acceleration dv_dt so vfinal(k) = dv_dt*dt
               ! v(k) = dr/dt = average velocity = (vstart + vfinal)/2 = dv_dt*dt/2 when vstart = 0
               ! so (1/2)*dv_dt*dt = v(k) 
               dv_dt = 2d0*s% v(k)/dt
            end if
            accel = dv_dt
            call get_dVARDOT_dVAR_pt(s, k, dt, d_dvardt_dvarm1, d_dvardt_dvar00, d_dvardt_dvarp1)
            
            rPterm = 1/(4 * pi * s% r(k)**2 * Ppoint)            
            dlnP_dm_accel_term = -fac*accel*rPterm
            
            s% dlnP_dm(k) = s% dlnP_dm(k) + dlnP_dm_accel_term
            
            if (s% lnPgas_flag) then
               d_rPterm_dlnPgas00_const_T = -rPterm*dPpoint_dlnPgas00_const_T/Ppoint
               d_rPterm_dlnT00_const_Pgas = -rPterm*dPpoint_dlnT00_const_Pgas/Ppoint
               d_rPterm_dlnPgasm1_const_T = -rPterm*dPpoint_dlnPgasm1_const_T/Ppoint
               d_rPterm_dlnTm1_const_Pgas = -rPterm*dPpoint_dlnTm1_const_Pgas/Ppoint
            else
               d_rPterm_dlnd00 = -rPterm*dPpoint_dlnd00/Ppoint
               d_rPterm_dlnT00 = -rPterm*dPpoint_dlnT00/Ppoint
               d_rPterm_dlndm1 = -rPterm*dPpoint_dlndm1/Ppoint
               d_rPterm_dlnTm1 = -rPterm*dPpoint_dlnTm1/Ppoint
            end if
            d_rPterm_dlnR = -2*rPterm

            if (s% lnPgas_flag) then
               d_dlnPdm_dlnPgas00_const_T = &
                  d_dlnPdm_dlnPgas00_const_T - fac*accel*d_rPterm_dlnPgas00_const_T
               d_dlnPdm_dlnT00_const_Pgas = &
                  d_dlnPdm_dlnT00_const_Pgas - fac*accel*d_rPterm_dlnT00_const_Pgas
               d_dlnPdm_dlnPgasm1_const_T = &
                  d_dlnPdm_dlnPgasm1_const_T - fac*accel*d_rPterm_dlnPgasm1_const_T
               d_dlnPdm_dlnTm1_const_Pgas = &
                  d_dlnPdm_dlnTm1_const_Pgas - fac*accel*d_rPterm_dlnTm1_const_Pgas
            else
               d_dlnPdm_dlnd00 = d_dlnPdm_dlnd00 - fac*accel*d_rPterm_dlnd00
               d_dlnPdm_dlnT00 = d_dlnPdm_dlnT00 - fac*accel*d_rPterm_dlnT00
               d_dlnPdm_dlndm1 = d_dlnPdm_dlndm1 - fac*accel*d_rPterm_dlndm1
               d_dlnPdm_dlnTm1 = d_dlnPdm_dlnTm1 - fac*accel*d_rPterm_dlnTm1
            end if

            d_dlnPdm_dlnR00 = d_dlnPdm_dlnR00 - fac*accel*d_rPterm_dlnR
            
            if (local_v_flag) then
               d_dlnPdm_dvel00 = -fac*d_dvardt_dvar00*rPterm
               d_dlnPdm_dvelp1 = -fac*d_dvardt_dvarp1*rPterm
               d_dlnPdm_dvelm1 = -fac*d_dvardt_dvarm1*rPterm
            else
               d_dlnPdm_dvel00 = -fac*rPterm*2d0/dt
            end if
            
         end if
            

      end subroutine do_dlnP_dm
      
      
      subroutine eval_dlnPdm_qhse(s, k, &
            m, dlnP_dm_qhse, d_dlnPdm_dlnR, d_dlnPdm_dlnq, d_dlnPdm_dL, &
            d_dlnPdm_dlnd00, d_dlnPdm_dlnT00, &
            d_dlnPdm_dlndm1, d_dlnPdm_dlnTm1, &
            d_dlnPdm_dlnPgas00_const_T, d_dlnPdm_dlnT00_const_Pgas, &
            d_dlnPdm_dlnPgasm1_const_T, d_dlnPdm_dlnTm1_const_Pgas, &
            Ppoint, &
            dPpoint_dlnd00, dPpoint_dlndm1, dPpoint_dlnT00, dPpoint_dlnTm1, &
            dPpoint_dlnPgas00_const_T, dPpoint_dlnPgasm1_const_T, &
            dPpoint_dlnT00_const_Pgas, dPpoint_dlnTm1_const_Pgas, &
            ierr)
         type (star_info), pointer :: s 
         integer, intent(in) :: k
         real(dp), intent(out) :: &
            m, dlnP_dm_qhse, d_dlnPdm_dlnR, d_dlnPdm_dlnq, d_dlnPdm_dL, &
            d_dlnPdm_dlnd00, d_dlnPdm_dlnT00, &
            d_dlnPdm_dlndm1, d_dlnPdm_dlnTm1, &
            d_dlnPdm_dlnPgas00_const_T, d_dlnPdm_dlnT00_const_Pgas, &
            d_dlnPdm_dlnPgasm1_const_T, d_dlnPdm_dlnTm1_const_Pgas, &
            Ppoint, &
            dPpoint_dlnd00, dPpoint_dlndm1, dPpoint_dlnT00, dPpoint_dlnTm1, &
            dPpoint_dlnPgas00_const_T, dPpoint_dlnPgasm1_const_T, &
            dPpoint_dlnT00_const_Pgas, dPpoint_dlnTm1_const_Pgas
         integer, intent(out) :: ierr
         
         real(dp) :: alfa, r, r3, r4, d_r4_dlnR, dlnq_dq, &
            dlnP_dm_qhse0, fs, dfs_dlnT, dfs_dlnR, dfs_dlnd, dfs_dL, &
            dfs_dlnPgas00_const_T, dfs_dlnT00_const_Pgas, correction_factor
         logical :: lnPgas_flag

         include 'formats'
         
         ierr = 0
         lnPgas_flag = s% lnPgas_flag
         
         ! basic eqn is dP = -G m / (4 pi r^4)
         ! divide by <P> to make it unitless
         ! simple average is adequate for <P> since is only for normalizing the equation.
         ! however, be careful to use same <P> for both sides of equation..... 
         
         ! for rotation, multiply by factor fp
         
         ! for tau < 2/3, multiply by Paczynski factor for dilution of radiation
            ! B. Paczynski, 1969, Acta Astr., vol. 19.  eqn 13
                  
         ! dlnP_dm_qhse = -G m / (4 pi r^4 <P>)
         r = s% r(k)
         r3 = r**3
         r4 = r*r3
         d_r4_dlnR = 4*r4
         
         if (k == 1) then
            alfa = 1
            Ppoint = alfa*s% P(k)
            dPpoint_dlndm1 = 0
            dPpoint_dlnTm1 = 0
            dPpoint_dlnPgasm1_const_T = 0
            dPpoint_dlnTm1_const_Pgas = 0
         else
            alfa = s% dq(k-1)/(s% dq(k-1) + s% dq(k))
            Ppoint = alfa*s% P(k) + (1-alfa)*s% P(k-1)
            if (lnPgas_flag) then
               dPpoint_dlnPgasm1_const_T = (1-alfa)*s% Pgas(k-1)
               dPpoint_dlnTm1_const_Pgas = (1-alfa)*4*s% Prad(k-1)
            else
               dPpoint_dlndm1 = (1-alfa)*s% P(k-1)*s% chiRho(k-1)
               dPpoint_dlnTm1 = (1-alfa)*s% P(k-1)*s% chiT(k-1)
            end if
         end if
         
         if (lnPgas_flag) then
            dPpoint_dlnPgas00_const_T = alfa*s% Pgas(k)
            dPpoint_dlnT00_const_Pgas = alfa*4*s% Prad(k)
         else
            dPpoint_dlnd00 = alfa*s% P(k)*s% chiRho(k)
            dPpoint_dlnT00 = alfa*s% P(k)*s% chiT(k)
         end if
         

         if (s% use_mass_corrections) then
            correction_factor = &
               s% mass_correction_start(k) + &
               s% P_div_rho_start(k)/clight**2
               
            if (correction_factor > 1.05 .or. correction_factor < 0.95) then
               write(*,2) 'bad correction factor?', k, correction_factor
               write(*,2) 's% mass_correction_start(k)', k, s% mass_correction_start(k)
               write(*,2) 's% P_div_rho_start(k)/clight**2', k, s% P_div_rho_start(k)/clight**2
               stop 1
            end if
            
            m = s% m_grav(k)
         else
            correction_factor = 1
            m = s% m(k)
         end if
         dlnP_dm_qhse = -s% cgrav(k)*m*correction_factor/(pi4*r4*Ppoint)

         if (s% rotation_flag .and. s% use_dlnP_dm_rotation_correction) &
            dlnP_dm_qhse = dlnP_dm_qhse*s% fp_rot(k)
         dlnP_dm_qhse0 = dlnP_dm_qhse
         
         if (s% tau(k) >= 2d0/3d0) then
            fs = 0
            dfs_dlnT = 0
            dfs_dlnR = 0
            dfs_dlnd = 0
            dfs_dL = 0
         else ! (tau < 2d0/3d0) then ! B. Paczynski, 1969, Acta Astr., vol. 19, 1.
            fs = (1d0 - 1.5d0*s% tau(k))*(2*crad*s% T(k)**3d0*sqrt(s% r(k)))/ &
                  (3d0*s% cgrav(k)*m*s% rho(k))*(s% L(k)/(8d0*pi*boltz_sigma))**0.25d0  ! eqn 15
            dfs_dlnT = 3d0*fs
            dfs_dlnR = 0.5d0*fs
            dfs_dlnd = -fs
            dfs_dL = 0.25d0*fs/s% L(k)
            dlnP_dm_qhse = dlnP_dm_qhse0*(1 + fs) ! eqn 13
         end if

         d_dlnPdm_dlnR = -(d_r4_dlnR/r4)*dlnP_dm_qhse + dlnP_dm_qhse0*dfs_dlnR
         dlnq_dq = 1/s% q(k)
         d_dlnPdm_dlnq = dlnP_dm_qhse/s% q(k)/dlnq_dq
         
         if (lnPgas_flag) then      
            dfs_dlnPgas00_const_T = dfs_dlnd*s% dlnRho_dlnPgas_const_T(k)
            dfs_dlnT00_const_Pgas = dfs_dlnT + dfs_dlnd*s% dlnRho_dlnT_const_Pgas(k)
            d_dlnPdm_dlnPgas00_const_T = -(dPpoint_dlnPgas00_const_T/Ppoint)*dlnP_dm_qhse + &
               dlnP_dm_qhse0*dfs_dlnPgas00_const_T
            d_dlnPdm_dlnT00_const_Pgas = -(dPpoint_dlnT00_const_Pgas/Ppoint)*dlnP_dm_qhse + &
               dlnP_dm_qhse0*dfs_dlnT00_const_Pgas
            d_dlnPdm_dlnPgasm1_const_T = -(dPpoint_dlnPgasm1_const_T/Ppoint)*dlnP_dm_qhse
            d_dlnPdm_dlnTm1_const_Pgas = -(dPpoint_dlnTm1_const_Pgas/Ppoint)*dlnP_dm_qhse
         else
            d_dlnPdm_dlnd00 = -(dPpoint_dlnd00/Ppoint)*dlnP_dm_qhse + dlnP_dm_qhse0*dfs_dlnd
            d_dlnPdm_dlnT00 = -(dPpoint_dlnT00/Ppoint)*dlnP_dm_qhse + dlnP_dm_qhse0*dfs_dlnT
            d_dlnPdm_dlndm1 = -(dPpoint_dlndm1/Ppoint)*dlnP_dm_qhse
            d_dlnPdm_dlnTm1 = -(dPpoint_dlnTm1/Ppoint)*dlnP_dm_qhse
         end if
         
         d_dlnPdm_dL = dlnP_dm_qhse0*dfs_dL

         if (is_bad_num(dlnP_dm_qhse)) then
            ierr = -1
            if (s% report_ierr) then
               write(*,*) 'eval_dlnPdm_qhse: is_bad_num(dlnP_dm_qhse)'
               write(*,2) 'dlnP_dm_qhse', k, dlnP_dm_qhse
               write(*,2) 's% tau(k)', k, s% tau(k)
               write(*,2) 's% fp_rot(k)', k, s% fp_rot(k)
               write(*,2) 'r4', k, r4
               write(*,2) 'Ppoint', k, Ppoint
               write(*,2) 'correction_factor', k, correction_factor
               write(*,2) 'm', k, m
               write(*,2) 's% cgrav(k)', k, s% cgrav(k)
               stop
            end if
            return
         end if

      end subroutine eval_dlnPdm_qhse

         
      end module hydro_vars

