! ***********************************************************************
!
!   Copyright (C) 2012  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module hydro_mtx
      
      use star_private_def
      use utils_lib, only: is_bad_num, has_bad_num
      use num_lib, only: safe_log10
      use const_def
      
      use num_def
      
      implicit none
      
      logical, parameter :: dbg = .false. 
      
      integer, parameter :: ipar_id = 1
      integer, parameter :: ipar_first_call = 2
      integer, parameter :: hydro_lipar = ipar_first_call
      
      integer, parameter :: rpar_dt = 1
      integer, parameter :: hydro_lrpar = 1

      
      ! for inspectB debugging
      real(dp), pointer :: debug_previous_data(:,:)


      contains
      

      subroutine set_newton_vars(s, iter, dx, xscale, dt, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: iter
         real(dp), dimension(:,:) :: dx, xscale
         real(dp), intent(in) :: dt
         integer, intent(out) :: ierr
         call set_vars_for_solver(s, 1, s% nz, iter, dx, xscale, dt, ierr)
      end subroutine set_newton_vars


      subroutine set_vars_for_solver(s, nzlo, nzhi, iter, dx, xscale, dt, ierr)
         use utils_lib, only: has_bad_num, is_bad_num
         use const_def, only: secyer, Msun, Lsun, Rsun
         use hydro_vars, only: set_hydro_vars, set_rmid_and_Amid
         use chem_def
         use star_utils
         type (star_info), pointer :: s
         integer, intent(in) :: nzlo, nzhi, iter
         real(dp), dimension(:,:) :: dx, xscale
         real(dp), intent(in) :: dt
         integer, intent(out) :: ierr

         logical, parameter :: &
            skip_basic_vars = .true., &
            skip_micro_vars = .false., &
            skip_kap = .false., &
            skip_neu = .false., &
            skip_grads = .true., &
            skip_rotation = .true., &
            skip_nse_fractions = .true., &
            skip_m_grav_and_grav = .true., &
            skip_brunt = .true., &
            skip_mixing_info = .true., &
            skip_other_cgrav = .true.
         logical :: do_chem, do_struct, try_again, skip_net, skip_do_burn_mix
         integer :: i, j, k, kk, klo, khi, &
            i_xlnd, i_lnPgas, i_lnT, i_lnR, i_lum, i_vel, &
            fe56, nvar, nvar_chem, species, i_chem1, nz, nvar_hydro
         real(dp), dimension(:, :), pointer :: xh_pre, xa_pre
         integer :: time0, clock_rate, op_err, kbad, &
            cnt, max_fixes, loc(2), k_lo, k_hi, k_const_mass
         real(dp) :: total_all_before, twoGmrc2, r2, xavg, &
            dq_sum, max_xa_err_ratio, xa_err_norm, dt_inv, min_xa_hard_limit
         
         include 'formats'

         if (dbg) write(*, *) 'set_vars_for_solver'
         ierr = 0
                  
         do_chem = (s% do_burn .or. s% do_mix)
         do_struct = (s% do_struct_hydro .or. s% do_struct_thermo)
         
         min_xa_hard_limit = current_min_xa_hard_limit(s)
         
         !restore old op-split scheme as option here   (see 4979)
         if (s% operator_coupling_choice == 0) then
            skip_net = .false.
         else
            skip_net = .true.
         end if
         skip_do_burn_mix = .true.
         
         xh_pre => s% xh_pre
         xa_pre => s% xa_pre
         
         i_xlnd = s% i_xlnd
         i_lnPgas = s% i_lnPgas
         i_lnT = s% i_lnT
         i_lnR = s% i_lnR
         i_lum = s% i_lum
         i_vel = s% i_vel
         
         nz = s% nz
         nvar_chem = s% nvar_chem
         species = s% species
         nvar_hydro = s% nvar_hydro
         i_chem1 = s% i_chem1

         fe56 = s% net_iso(ife56)
         if (fe56 /= 0) fe56 = i_chem1+fe56-1
         
         if (do_chem) then ! include abundances
            nvar = s% nvar
         else ! only structure
            nvar = nvar_hydro
         end if

         if (dt == 0) then
            dt_inv = 0
         else 
            dt_inv = 1/dt
         end if
         s% dVARdot_dVAR = dt_inv
      
         if (s% doing_timing) then
            total_all_before = total_times(s)
            call system_clock(time0,clock_rate)
         end if
      
         if (do_chem) then
            do k=1,nz
               do j=1,species
                  s% xa(j,k) = xa_pre(j,k) + dx(j+nvar_hydro,k)
               end do
            end do
            max_fixes = 0 !5
            do cnt=1,max_fixes
               loc = minloc(s% xa(1:species,1:nz))
               j = loc(1)
               k = loc(2)
               if (s% xa(j,k) >= 1d-3*min_xa_hard_limit) exit ! too good to fix
               if (s% xa(j,k) < min_xa_hard_limit) then
                  if (s% report_ierr) then
                     khi = nz
                     do kk=k+1,nz
                        if (s% xa(j,kk) < min_xa_hard_limit) cycle
                        khi = kk-1; exit
                     end do
                     klo = 1
                     do kk=k-1,1,-1
                        if (s% xa(j,kk) < min_xa_hard_limit) cycle
                        klo = kk+1; exit
                     end do
                     do k=klo,khi
                        write(*,2) &
                           'negative ' // trim(chem_isos% name(s% chem_id(j))), &
                           k, s% xa(j,k), xa_pre(j,k), dx(nvar_hydro+j,k), s% m(k)/Msun
                     end do
                  end if
                  ierr = -1
                  return
                  exit ! too bad to fix
               end if
               if (k == 1) then  
                  k_lo = 1; k_hi = 2
               else if (k == nz) then
                  k_lo = nz-1; k_hi = nz
               else if (s% sig(k) > s% sig(k+1)) then
                  k_lo = k-1; k_hi = k
               else if (s% sig(k+1) > 0) then
                  k_lo = k; k_hi = k+1
               else
                  exit
               end if
               try_again = .true.
               do while (try_again .and. sum(s% xa(j,k_lo:k_hi)*s% dq(k_lo:k_hi)) < 0)
                  try_again = .false.
                  if (k_lo > 1) then
                     if (s% sig(k_lo) > 0) then
                        k_lo = k_lo-1
                        try_again = .true.
                     end if
                  end if
                  if (k_hi < nz) then
                     if (s% sig(k_hi+1) > 0) then
                        k_hi = k_hi+1
                        try_again = .true.
                     end if
                  end if
               end do
               !write(*,3) 'no extend', k_lo, k_hi
               if (.not. try_again) exit
               dq_sum = sum(s% dq(k_lo:k_hi))
               if (s% report_ierr) then
                  write(*,5) 'fix xa(j,k_lo:k_hi)', j, k_lo, k, k_hi, s% xa(j,k), &
                     sum(s% xa(j,k_lo:k_hi)*s% dq(k_lo:k_hi))/dq_sum, dq_sum
                  !stop
                end if
               do j=1,species
                  xavg = sum(s% xa(j,k_lo:k_hi)*s% dq(k_lo:k_hi))/dq_sum
                  do kk=k_lo,k_hi
                     s% xa(j,kk) = xavg
                  end do
               end do
            end do
         end if

         kbad = 0
!x$OMP PARALLEL DO PRIVATE(k,op_err)
         do k=1,nz
            op_err = 0
            call set1(k,.false.,op_err)
            if (op_err /= 0) then
               kbad = k; ierr = op_err
            end if
         end do
!x$OMP END PARALLEL DO
         if (ierr /= 0) then
            if (s% report_ierr) then
               do k=1,nz ! report the errors sequentially
                  call set1(k,.true.,op_err)
               end do
               write(*,4) 'set_vars_for_solver failed: model, k, nz', s% model_number, kbad, nz
               stop 'set_vars_for_solver'
            end if            
            return
         end if

         do k=1,nz
            if (is_bad_num(s% L(k))) then
               if (s% report_ierr) write(*,2) 'set_vars_for_solver L', k, s% L(k), &
                  xh_pre(i_lum,k) + dx(i_lum,k), &
                  xh_pre(i_lum,k), dx(i_lum,k)
               ierr = -1
               return               
               stop
            end if
         end do
         
         if (ierr /= 0) then
            if (s% report_ierr) then
            
               do k=1,nz
                  if (abs(1d0 - sum(s% xa(:,k))) > 1d-3) then
                     write(*,2) 'set_vars_for_solver: bad xa sum', k, &
                        sum(s% xa(:,k)), sum(xa_pre(:,k)), sum(dx(i_chem1:nvar,k))
                     write(*,'(51x,a)') 'xa, xa_pre+dx, xa_pre, dx'
                     do j=1,species
                        write(*,2) trim(chem_isos% name(s% chem_id(j))), k, &
                           s% xa(j,k), xa_pre(j,k) + dx(i_chem1-1+j,k), &
                           xa_pre(j,k), dx(i_chem1-1+j,k)
                     end do
                     
                     exit
                     
                  end if
               end do
               write(*,*)
               
            end if
            return
         end if
         
         if (do_struct) then
            if ( & !(.not. s% rotation_flag) .and. &
                (.not. s% doing_numerical_jacobian)) then
               if (s% lnPgas_flag) then
                  call edit_lnR_for_lnPgas(s, xh_pre, dx, dt)
               else
                  call edit_lnR(s, xh_pre, dx, dt)
               end if
            end if
!x$OMP PARALLEL DO PRIVATE(k,twoGmrc2,r2)
            do k=1,nz
               s% r(k) = exp(s% lnR(k))
               r2 = s% r(k)**2
               s% area(k) = pi4*r2
               ! note: m_grav is held constant during newton iterations
               s% grav(k) = s% cgrav(k)*s% m_grav(k)/r2
               if (s% use_gr_factors) then ! GR gravity factor = 1/sqrt(1-2Gm/(rc^2))
                  twoGmrc2 = 2*s% cgrav(k)*s% m_grav(k)/(s% r(k)*clight**2)
                  s% grav(k) = s% grav(k)/sqrt(1d0 - twoGmrc2)
               end if
            end do    
!x$OMP END PARALLEL DO
         end if

         call set_rmid_and_Amid(s, 1, nz, ierr)
         if (ierr /= 0) then
            if (s% report_ierr) write(*, *) 'set_rmid_and_Amid returned ierr', ierr
            return
         end if

         if (s% rotation_flag) then ! change i_rot to match new r, omega to match new i_rot
            call set_i_rot(s)
            call set_omega(s, 'hydro_mtx')
         end if

         if (dbg) write(*, *) 'call set_hydro_vars'
         call set_hydro_vars( &
            s, 1, nz, skip_basic_vars, skip_micro_vars, &
            skip_m_grav_and_grav, skip_do_burn_mix, skip_net, skip_neu, skip_kap, &
            skip_grads, skip_rotation, skip_brunt, skip_other_cgrav, &
            skip_nse_fractions, skip_mixing_info, dt, ierr)
         if (ierr /= 0) then
            if (s% report_ierr) write(*,2) 'set_hydro_vars returned ierr', ierr
            return
         end if
      
         if (s% doing_timing) &
            call update_time(s, time0, total_all_before, s% time_set_newton_vars)
         
         !write(*,*) 'call dump_struct from hydro_mtx'
         !call dump_struct(s)
         !write(*,*) 'done call dump_struct from hydro_mtx'

         
         contains
         
         
         subroutine set1(k,report,ierr)
            use chem_def, only: chem_isos
            integer, intent(in) :: k
            logical, intent(in) :: report
            integer, intent(out) :: ierr
            
            real(dp) :: x(nvar)
            ! setting x = xh_pre + dx is necessary because of numerical issues.
            ! we want to ensure that we've calculated the variables using exactly the
            ! same values for x as will be returned as the final result.
            real(dp) :: twoGmrc2, r2, sum_xa
            integer :: j, i
            real(dp) :: del_t, starting_value, alfa, beta
            
            logical, parameter :: check_for_bad_nums = .false.
            
            include 'formats'
            ierr = 0
            
            if (do_struct) then
            
               x(1:nvar_hydro) = xh_pre(1:nvar_hydro,k) + dx(1:nvar_hydro,k)
               if (i_xlnd /= 0) then
                  s% lnd(k) = x(i_xlnd)
                  s% rho(k) = exp(s% lnd(k))
                  if (check_for_bad_nums .and. is_bad_num(s% rho(k))) then
                     if (report) write(*,2) 'bad num rho', k, s% rho(k)
                     ierr = -1
                  end if
               end if
               
               if (i_lnPgas /= 0) then
                  s% lnPgas(k) = x(i_lnPgas)
                  s% Pgas(k) = exp(s% lnPgas(k))
                  if (check_for_bad_nums .and. is_bad_num(s% Pgas(k))) then
                     if (report) write(*,2) 'bad num Pgas', k, s% Pgas(k)
                     ierr = -1
                  end if
               end if
               
               s% lnT(k) = x(i_lnT)
               s% T(k) = exp(s% lnT(k))
               if (check_for_bad_nums .and. is_bad_num(s% T(k))) then
                  if (report) write(*,2) 'bad num T', k, s% T(k)
                  ierr = -1
               end if
               
               if (k == -1) then
                  write(*,2) 'set_vars_for_solver dlogT new old', k, &
                     dx(i_lnT,k)/ln10, s% lnT(k)/ln10, &
                     xh_pre(i_lnT,k)/ln10
               end if

               s% lnR(k) = x(i_lnR)
               if (check_for_bad_nums .and. is_bad_num(s% lnR(k))) then
                  if (report) write(*,2) 'bad num lnR', k, s% lnR(k)
                  ierr = -1
               end if
               
               s% L(k) = x(i_lum)
               if (check_for_bad_nums .and. is_bad_num(s% L(k))) then
                  if (report) write(*,2) 'bad num L', k, s% L(k)
                  ierr = -1
               end if
               
               if (i_vel /= 0) then
                  s% v(k) = x(i_vel)
                  if (check_for_bad_nums .and. is_bad_num(s% v(k))) then
                     if (report) write(*,2) 'bad num v', k, s% v(k)
                     ierr = -1
                  end if
               end if

               ! set time derivatives
               if (dt == 0) then
                  s% dlnd_dT(k) = 0
                  s% dlnPgas_dT(k) = 0
                  s% dlnT_dt(k) = 0
                  s% dlnR_dt(k) = 0
                  if (i_vel /= 0) s% dv_dt(k) = 0
                  
               else if (k >= s% k_const_mass) then
                  ! use dx to get better accuracy
                  
                  if (i_xlnd /= 0) s% dlnd_dt(k) = dx(i_xlnd,k)*dt_inv                  
                  if (i_lnPgas /= 0) s% dlnPgas_dt(k) = dx(i_lnPgas,k)*dt_inv                  
                  s% dlnT_dt(k) = dx(i_lnT,k)*dt_inv                 
                  s% dlnR_dt(k) = dx(i_lnR,k)*dt_inv                 
                  if (i_vel /= 0) s% dv_dt(k) = dx(i_vel,k)*dt_inv
                  
               else if (k >= s% k_below_just_added) then
                  
                  s% dlnT_dt(k) = (x(i_lnT) - s% lnT_for_d_dt(k))*dt_inv                  
                  s% dlnR_dt(k) = (x(i_lnR) - s% lnR_for_d_dt(k))*dt_inv                  
                  if (i_xlnd /= 0) &
                     s% dlnd_dt(k) = (x(i_xlnd) - s% lnd_for_d_dt(k))*dt_inv                  
                  if (i_lnPgas /= 0) &
                     s% dlnPgas_dt(k) = (x(i_lnPgas) - s% lnPgas_for_d_dt(k))*dt_inv                  
                  if (i_vel /= 0) s% dv_dt(k) = (x(i_vel) - s% v_for_d_dt(k))*dt_inv 
                  
                  if (.false. .and. k == s% k_below_just_added) then
                     write(*,2) 's% dlnT_dt(k)', k, s% dlnT_dt(k)
                     write(*,2) 'x(i_lnT)', k, x(i_lnT)
                     write(*,2) 's% lnT_for_d_dt(k)', k, s% lnT_for_d_dt(k)
                     write(*,2) 'xh_pre(i_lnT,1)', 1, xh_pre(i_lnT,1)
                     write(*,2) 'dt', k, dt
                     write(*,*)
                  end if
                  
               ! k < s% k_below_just_added
               else if (k > 1 .and. s% del_t_for_just_added(k) > 0d0) then 
               
                  ! del_t is time since this cell entered at the surface
                  del_t = s% del_t_for_just_added(k)
                  ! interpolate starting values
                  alfa = del_t/dt
                  beta = 1d0 - alfa
                  starting_value = alfa*xh_pre(i_lnT,1) + beta*s% surf_lnT
                  s% dlnT_dt(k) = (x(i_lnT) - starting_value)/del_t          
                  starting_value = alfa*xh_pre(i_lnR,1) + beta*s% surf_lnR
                  s% dlnR_dt(k) = (x(i_lnR) - starting_value)/del_t
                  if (i_xlnd /= 0) then
                     starting_value = alfa*xh_pre(i_xlnd,1) + beta*s% surf_lnd
                     s% dlnd_dt(k) = (x(i_xlnd) - starting_value)/del_t    
                  end if    
                  if (i_lnPgas /= 0) then
                     starting_value = alfa*xh_pre(i_lnPgas,1) + beta*s% surf_lnPgas
                     s% dlnPgas_dt(k) = (x(i_lnPgas) - starting_value)/del_t   
                  end if          
                  if (i_vel /= 0) then
                     starting_value = alfa*xh_pre(i_vel,1) + beta*s% surf_v
                     s% dv_dt(k) = (x(i_vel) - starting_value)/del_t
                  end if    
                  
                  if (.false. .and. k == s% k_below_just_added-1) then
                     write(*,2) 's% dlnT_dt(k)', k, s% dlnT_dt(k)
                     write(*,2) 'x(i_lnT)', k, x(i_lnT)
                     write(*,2) 'starting_value', k, starting_value
                     write(*,2) 'xh_pre(i_lnT,1)', 1, xh_pre(i_lnT,1)
                     write(*,2) 's% surf_lnT', 1, s% surf_lnT
                     write(*,2) 'del_t', k, del_t
                     write(*,2) 'alfa', k, alfa
                     write(*,2) 'beta', k, beta
                     write(*,*)
                     
                  end if
                  
               else ! k == 1 and k < s% k_below_just_added, so new surface cell
               
                  s% dlnT_dt(k) = 0  
                  s% dlnR_dt(k) = 0
                  s% dlnd_dt(k) = 0    
                  s% dlnPgas_dt(k) = 0    
                  s% dv_dt(k) = 0
                  
               end if
               
            end if
            
            if (do_chem) call check1_chem(s, k, min_xa_hard_limit, report, ierr)
         
         end subroutine set1
         
         
      end subroutine set_vars_for_solver

               
      subroutine check1_chem(s, k, min_xa_hard_limit, report, ierr)
         use chem_def, only: chem_isos
         type (star_info), pointer :: s
         integer, intent(in) :: k
         real(dp), intent(in) :: min_xa_hard_limit
         logical, intent(in) :: report
         integer, intent(out) :: ierr

         integer :: j, species
         real(dp) :: sum_xa
         logical :: okay

         include 'formats'
         
         ierr = 0
         species = s% species
         okay = .true.
         if (min_xa_hard_limit > -1d50) then
            do j=1,species
               if (s% xa(j,k) < min_xa_hard_limit) then
                  if (report) then
                     write(*,3) &
                        'bad negative xa, sig, logT ' // &
                        trim(chem_isos% name(s% chem_id(j))), j, k, &
                        s% xa(j,k), s% sig(k), s% lnT(k)/ln10
                     if (.false.) then
                        write(*,3) &
                           'bad negative xa xa_pre xa-xa_pre ' // &
                           trim(chem_isos% name(s% chem_id(j))), j, k, &
                           s% xa(j,k), s% xa_pre(j,k), &
                           s% xa(j,k) - s% xa_pre(j,k)
                        write(*,3) &
                           'bad negative logT T logRho rho ' // &
                           trim(chem_isos% name(s% chem_id(j))), j, k, &
                           s% lnT(k)/ln10, s% T(k), s% lnd(k)/ln10, s% rho(k)
                        write(*,3) &
                           'bad negative m/Msun q r/Rsun ' // &
                           trim(chem_isos% name(s% chem_id(j))), j, k, &
                           s% m(k)/Msun, s% q(k), s% r(k)/Rsun
                        write(*,3) &
                           'bad negative sig ' // &
                           trim(chem_isos% name(s% chem_id(j))), j, k, &
                           s% sig(max(1,k-1)), s% sig(k), &
                           s% sig(min(s% nz, k+1)), s% sig(min(s% nz, k+2))
                        write(*,3) &
                           'bad negative: xa neighbors ' // &
                           trim(chem_isos% name(s% chem_id(j))), j, k, &
                           s% xa(j,max(1,k-2):min(s% nz, k+2))
                        write(*,3) &
                           'bad negative: xa_pre neighbors ' // &
                           trim(chem_isos% name(s% chem_id(j))), j, k, &
                           s% xa_pre(j,max(1,k-2):min(s% nz, k+2))
                        write(*,*)
                     end if
                     okay = .false.
                  end if
                  ierr = -1
                  s% why_Tlim = Tlim_neg_X
                  if (.not. report) return
               else if (.false. .and. report .and. s% xa(j,k) < -1d-6) then
                  write(*,2) &
                     'negative ' // &
                     trim(chem_isos% name(s% chem_id(j))), &
                     k, s% xa(j,k), s% xa_pre(j,k), &
                     s% xa(j,k) - s% xa_pre(j,k), &
                     s% m(k)/Msun, s% nse_fraction(k)
               end if
            end do
         else if (.false. .and. s% report_ierr) then
            do j=1,species
               if (s% xa(j,k) < -1d-3) then
                  write(*,2) &
                     'negative ' // &
                     trim(chem_isos% name(s% chem_id(j))), &
                     k, s% xa(j,k), s% xa_pre(j,k), &
                     s% xa(j,k) - s% xa_pre(j,k), &
                     s% m(k)/Msun, s% nse_fraction(k)
               end if
            end do
         end if
         if (.false. .and. report .and. .not. okay) then
            do j=1,species
               write(*,2) &
                  trim(chem_isos% name(s% chem_id(j))), &
                  k, s% xa(j,k), s% xa_pre(j,k), &
                  s% xa(j,k) - s% xa_pre(j,k)
            end do
            write(*,*)
            !stop 'check1_chem'
         end if

         do j=1,species
            s% xa(j,k) = max(0d0, min(1d0, s% xa(j,k)))
         end do

         sum_xa = sum(s% xa(1:species,k))
         if (is_bad_num(sum_xa)) then
            if (report) then
               write(*,'(a60,i8,99f20.10)') 'bad num sum X', k, s% m(k)/Msun, sum_xa
            end if
            ierr = -1
            s% why_Tlim = Tlim_bad_Xsum
            if (.not. report) return
         end if
         if (abs(sum_xa - 1d0) > s% sum_xa_tolerance) then
            if (report) then
               write(*,2) &
                  'bad sumX', k, &
                  sum_xa, sum(s% xa_pre(1:species,k)), &
                  sum_xa - sum(s% xa_pre(1:species,k))
               if (.false.) then
                  write(*,2) &
                     'bad sumX sig', k, &
                     s% sig(max(1,k-1)), s% sig(k), &
                     s% sig(min(s% nz, k+1)), s% sig(min(s% nz, k+2))
                  write(*,2) &
                     'bad sumX logT T logRho rho', k, &
                     s% lnT(k)/ln10, s% T(k), s% lnd(k)/ln10, s% rho(k)
                  write(*,2) &
                     'bad sumX m/Msun q r/Rsun', k, &
                     s% m(k)/Msun, s% q(k), s% r(k)/Rsun
                  write(*,2) &
                     'bad sumX sum_xa sum(dxdt_nuc)', k, &
                     sum_xa, sum(s% dxdt_nuc(:,k))
                  write(*,*)
               end if
            end if
           ierr = -1
            s% why_Tlim = Tlim_bad_Xsum
            okay = .false.
            if (.not. report) return
         end if
         if (.false. .and. report .and. .not. okay) then
            do j=1,species
               write(*,2) &
                  trim(chem_isos% name(s% chem_id(j))), &
                  k, s% xa(j,k), s% xa_pre(j,k), &
                  s% xa(j,k) - s% xa_pre(j,k)
            end do
            write(*,*)
            write(*,2) 'sum_xa', k, sum_xa
            write(*,*)
            !stop 'check1_chem bad sum X'
         end if

         if (abs(sum_xa - 1d0) > 1d-12) then
            do j=1,species
               s% xa(j,k) = s% xa(j,k)/sum_xa
            end do
         end if
         
         if (s% xa_clip_limit > 0) then
            do j=1,species
               if (s% xa(j,k) < s% xa_clip_limit) s% xa(j,k) = 0d0
            end do
         end if
         
      end subroutine check1_chem
         
         
      subroutine dump_struct(s)
         type (star_info), pointer :: s
         integer :: k, j, i
         
         include 'formats.dek'
         
         do k=1,s% nz
            write(*,2) 'dq', k, s% dq(k)
            write(*,2) 'm', k, s% m(k)
            write(*,2) 'T', k, s% T(k)
            write(*,2) 'rho', k, s% rho(k)
            write(*,2) 'Pgas', k, s% Pgas(k)
            write(*,2) 'L', k, s% L(k)
            write(*,2) 'r', k, s% r(k)
            write(*,2) 'grada', k, s% grada(k)
            write(*,2) 'opacity', k, s% opacity(k)
            write(*,2) 'd_opacity_dlnd', k, s% d_opacity_dlnd(k)
            write(*,2) 'd_opacity_dlnT', k, s% d_opacity_dlnT(k)
            write(*,2) 'eps_nuc', k, s% eps_nuc(k)
            write(*,2) 'd_epsnuc_dlnd', k, s% d_epsnuc_dlnd(k)
            write(*,2) 'd_epsnuc_dlnT', k, s% d_epsnuc_dlnT(k)
            write(*,2) 'non_nuc_neu', k, s% non_nuc_neu(k)
            write(*,2) 'd_nonnucneu_dlnd', k, s% d_nonnucneu_dlnd(k)
            write(*,2) 'd_nonnucneu_dlnT', k, s% d_nonnucneu_dlnT(k)
            write(*,2) 'eps_grav', k, s% eps_grav(k)
            write(*,2) 'd_eps_grav_dlndm1', k, s% d_eps_grav_dlndm1(k)
            write(*,2) 'd_eps_grav_dlnd00', k, s% d_eps_grav_dlnd00(k)
            write(*,2) 'd_eps_grav_dlndp1', k, s% d_eps_grav_dlndp1(k)
            write(*,2) 'd_eps_grav_dlnTm1', k, s% d_eps_grav_dlnTm1(k)
            write(*,2) 'd_eps_grav_dlnT00', k, s% d_eps_grav_dlnT00(k)
            write(*,2) 'd_eps_grav_dlnTp1', k, s% d_eps_grav_dlnTp1(k)
            write(*,2) 'gradT', k, s% gradT(k)
            write(*,2) 'd_gradT_dlnd00', k, s% d_gradT_dlnd00(k)
            write(*,2) 'd_gradT_dlndm1', k, s% d_gradT_dlndm1(k)
            write(*,2) 'dlnP_dm', k, s% dlnP_dm(k)
            write(*,2) 'dlnT_dm', k, s% dlnT_dm(k)
            write(*,2) 'dL_dm', k, s% dL_dm(k)
            !write(*,2) '', k, s% (k)
            do j=1,s% species
               write(*,3) 'xa(j,k)', j, k, s% xa(j,k)
            end do
         end do
         
      
      end subroutine dump_struct
      
         
      subroutine edit_lnR(s, xh_pre, dx, dt) 
         ! uses mass and density to set radius
         type (star_info), pointer :: s
         real(dp), dimension(:, :) :: xh_pre
         real(dp), dimension(:, :) :: dx
         real(dp), intent(in) :: dt
         real(dp) :: vol00, volp1, cell_vol
         integer :: k, nz
         vol00 = (4*pi/3)*s% R_center**3
         nz = s% nz         
         do k=nz, 1, -1
            volp1 = vol00
            cell_vol = s% dm(k)/s% rho(k)
            vol00 = volp1 + cell_vol
            s% lnR(k) = log(vol00/(4*pi/3))/3
            dx(s% i_lnR,k) = s% lnR(k) - xh_pre(s% i_lnR,k)
            if (dt > 0 .and. k >= s% k_below_just_added) &
               s% dlnR_dt(k) = (s% lnR(k) - s% lnR_for_d_dt(k))/dt
         end do
         call edit_dlnR_dt_above_k_below_just_added(s, xh_pre)
      end subroutine edit_lnR

      
      subroutine edit_lnR_for_lnPgas(s, xh_pre, dx, dt) 
         type (star_info), pointer :: s
         ! cannot use density since it hasn't been calculated yet
         real(dp), dimension(:, :) :: xh_pre
         real(dp), dimension(:, :) :: dx
         real(dp), intent(in) :: dt
         integer :: k, kk, k_outer, k_inner, cnt, nz, k_const_mass
         real(dp) :: dq_total, dq_sum, &
            r3_outer, r3_inner, r3, alfa, beta
         logical, parameter :: dbg = .false.
         include 'formats'
         cnt = 0
         nz = s% nz
         k_const_mass = s% k_const_mass
         do k=1, nz-1
            if (s% lnR(k) > s% lnR(k+1)) cycle
            ! find k_outer s.t. lnR(k+1) < lnR(k_outer)
            if (dbg) write(*,*)
            if (dbg) write(*,2) 'bad s% lnR(k)', k, s% lnR(k)
            if (dbg) write(*,2) 'bad s% lnR(k+1)', k+1, s% lnR(k+1)
            cnt = cnt+1
            k_outer = 1
            do kk = k,1,-1
               if (s% lnR(k+1) < s% lnR(kk)) then
                  k_outer = kk; exit
               end if
            end do
            ! find k_inner s.t. lnR(k_inner) < lnR(k)
            k_inner = nz
            do kk = k+1, nz-1
               if (s% lnR(kk) < s% lnR(k)) then
                  k_inner = kk; exit
               end if
            end do
            dq_total = sum(s% dq(k_outer:k_inner-1))
            dq_sum = 0
            r3_outer = exp(3*s% lnR(k_outer))
            r3_inner = exp(3*s% lnR(k_inner))
            if (dbg) write(*,2) 's% lnR(k_outer)', k_outer, s% lnR(k_outer)
            do kk=k_outer+1, k_inner-1
               dq_sum = dq_sum + s% dq(kk-1)
               alfa = dq_sum/dq_total
               beta = 1 - alfa
               r3 = alfa*r3_inner + beta*r3_outer
               s% lnR(kk) = log(r3)/3
               if (dbg) write(*,2) 'new s% lnR(kk)', kk, s% lnR(kk)
               dx(s% i_lnR,kk) = s% lnR(kk) - xh_pre(s% i_lnR,kk)
               if (dt > 0 .and. kk >= s% k_below_just_added) &
                  s% dlnR_dt(kk) = (s% lnR(kk) - s% lnR_for_d_dt(kk))/dt
            end do
            if (dbg) write(*,2) 's% lnR(k_inner)', k_inner, s% lnR(k_inner)
         end do
         call edit_dlnR_dt_above_k_below_just_added(s, xh_pre)
      end subroutine edit_lnR_for_lnPgas     
      
      
      subroutine edit_dlnR_dt_above_k_below_just_added(s, xh_pre)
         type (star_info), pointer :: s
         real(dp), dimension(:, :) :: xh_pre
         integer :: k, k_below_just_added
         real(dp) :: lnR_pre
         k_below_just_added = s% k_below_just_added
         if (k_below_just_added == 1) return
         lnR_pre = xh_pre(s% i_lnR,1)
         s% dlnR_dt(1) = 0
         do k = 2, k_below_just_added - 1
            if (s% del_t_for_just_added(k) <= 0d0) then
               s% dlnR_dt(k) = 0d0
            else
               s% dlnR_dt(k) = (s% lnR(k) - lnR_pre)/s% del_t_for_just_added(k)
            end if
         end do
      end subroutine edit_dlnR_dt_above_k_below_just_added
            

      subroutine enter_setmatrix( &
            iter, nvar_in, nz, neqns, dx, dx_init, xscale, xder, need_solver_to_eval_jacobian, &
            ldA, A1, idiag, lrpar, rpar, lipar, ipar, ierr)
         use mtx_lib, only: write_3point_jacobian_info
         use mtx_def, only: lapack
         use star_utils, only: update_time, total_times
         integer, intent(in) :: iter, nvar_in, nz, neqns ! (neqns = nvar*nz)
         real(dp), pointer, dimension(:,:) :: dx, dx_init, xscale, xder ! (nvar, nz)
         logical, intent(out) :: need_solver_to_eval_jacobian
         integer, intent(in) :: ldA ! leading dimension of A
         real(dp), pointer, dimension(:) :: A1
         integer, intent(inout) :: idiag 
         integer, intent(in) :: lrpar, lipar
         real(dp), intent(inout) :: rpar(:) ! (lrpar)
         integer, intent(inout) :: ipar(:) ! (lipar)
         integer, intent(out) :: ierr

         type (star_info), pointer :: s                  
         real(dp), pointer, dimension(:,:) :: A ! (ldA, neqns)
         integer :: i, j, k, cnt, nnz, nzlo, nzhi
         logical, parameter :: save_jac_plot_data = .false.  
         real(dp), parameter :: epsder_struct = 1d-7
         real(dp), parameter :: epsder_chem = 1d-7
         real(dp) :: dt
         integer :: id, nvar, nvar_hydro
         logical :: dbg_enter_setmatrix, do_chem
         real(dp), pointer :: blk3(:, :, :, :)
         integer :: time0, clock_rate
         real(dp) :: total_all_before
         
         include 'formats'
         
         id = ipar(ipar_id)
         dbg_enter_setmatrix = dbg

         ierr = 0   
         
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) then
            ierr = -1
            return
         end if

         if (s% doing_timing) then
            total_all_before = total_times(s)
            call system_clock(time0,clock_rate)
         end if

         if (dbg_enter_setmatrix) write(*, '(/,/,/,/,/,/,a)') 'enter_setmatrix'
         
         if (s% model_number == 1) then
            s% num_jacobians = s% num_jacobians + 1
            if (s% num_jacobians > 60 .and. mod(s% num_jacobians,10) == 0) &
               write(*,*) 'first model is slow to converge: num tries', s% num_jacobians
         end if
         
         dt = rpar(rpar_dt)
         
         do_chem = (s% do_burn .or. s% do_mix)
         if (do_chem) then ! include abundances
            nvar = s% nvar
         else ! only structure
            nvar = s% nvar_hydro
         end if
         
         if (nvar /= nvar_in) then
            write(*,*) 'enter_setmatrix: bad value for nvar_in'
            ierr = -1
            return
         end if
         
         if (s% hydro_call_number == s% hydro_dump_call_number &
            .and. iter == s% hydro_dump_iter_number) s% numerical_jacobian = .true.

         if (s% numerical_jacobian) then
            write(*,*) 'enter_setmatrix: s% numerical_jacobian', s% numerical_jacobian
            write(*,*) 'enter_setmatrix: s% hydro_matrix_type', s% hydro_matrix_type
            
            if (.false.) then
               write(*,*) 'call eval_equ_for_numerical_jacobian'
               call eval_equ_for_numerical_jacobian(s, dt, xscale, ierr)
               if (ierr /= 0) stop 1
               stop 'after eval_equ_for_numerical_jacobian'
            end if
            
            need_solver_to_eval_jacobian = .true.
            s% doing_numerical_jacobian = .true.
            nvar_hydro = s% nvar_hydro
            do j=1, nvar_hydro
               do k=1,nz
                  xder(j,k) = epsder_struct* &
                     max(1d0, abs(s% xh_pre(j,k)+dx(j,k)), abs(s% xh_pre(j,k)+dx_init(j,k)))
               end do
            end do
            if (nvar > nvar_hydro) then
               do j=nvar_hydro+1, nvar
                  do k=1,nz
                     xder(j,k) = epsder_chem* &
                        max(1d0, abs(s% xa_pre(j-nvar_hydro,k)+dx(j,k)), &
                                          abs(s% xa_pre(j-nvar_hydro,k)+dx_init(j,k)))
                  end do
               end do
            end if
            call check_xder_lnR
            write(*,*) 'done enter_setmatrix'
            return
         end if

         s% idiag = idiag
         A(1:ldA,1:neqns) => A1(1:ldA*neqns)
         s% jacobian(1:ldA,1:neqns) => A1(1:ldA*neqns)
                  
         if (s% hydro_matrix_type == block_tridiag_dble_matrix_type .or. &
             s% hydro_matrix_type == block_tridiag_quad_matrix_type) then
!$OMP PARALLEL DO PRIVATE(i,j)
            do i=1,neqns
               do j=1,lda
                  A(j,i) = 0
               end do
            end do
!$OMP END PARALLEL DO
            i = nvar*nvar*nz
            if (size(A1,dim=1) < 3*i) then
               write(*,*) 'enter_setmatrix: size(A1,dim=1) < 3*i', size(A1,dim=1), 3*i
               ierr = -1
               return
            end if
            s% ublk(1:nvar,1:nvar,1:nz) => A1(1:i)
            s% dblk(1:nvar,1:nvar,1:nz) => A1(i+1:2*i)
            s% lblk(1:nvar,1:nvar,1:nz) => A1(2*i+1:3*i)
         end if

         !s% doing_numerical_jacobian = .false.
         
         if (dbg_enter_setmatrix) &
            write(*, *) 'call eval_partials with doing_numerical_jacobian = .false.'
         call eval_partials(s, dt, xscale, ierr)
         if (ierr /= 0) return

         if (dbg_enter_setmatrix) write(*, *) 'finished enter_setmatrix'
         need_solver_to_eval_jacobian = .false.

         if (s% doing_timing) &
            call update_time(s, time0, total_all_before, s% time_newton_enter_setmatrix)
         
         if (.false. .and. s% hydro_matrix_type == block_tridiag_dble_matrix_type) then
            write(*,*) 'at end of enter_setmatrix'
            do k = 536,538
               do j=1,nvar
                  write(*,2) 'equ ' // trim(s% nameofequ(j)), k, s% equ(j,k)
               end do
               write(*,*)
               do j=1,nvar
                  do i=1,nvar
                     if (s% dblk(i,j,k) /= 0) &
                        write(*,2) 'dblk ' // trim(s% nameofequ(i)) // &
                           '_' // trim(s% nameofvar(j)), k, s% dblk(i,j,k)
                  end do
               end do
               write(*,*)
               do j=1,nvar
                  do i=1,nvar
                     if (s% lblk(i,j,k) /= 0) &
                        write(*,2) 'lblk ' // trim(s% nameofequ(i)) // &
                           '_' // trim(s% nameofvar(j)), k, s% lblk(i,j,k)
                  end do
               end do
               write(*,*)
               do j=1,nvar
                  do i=1,nvar
                     if (s% ublk(i,j,k) /= 0) &
                        write(*,2) 'ublk ' // trim(s% nameofequ(i)) // &
                           '_' // trim(s% nameofvar(j)), k, s% ublk(i,j,k)
                  end do
               end do
               write(*,*)
               write(*,*)
            end do
            !stop 'dump_partials'
         end if

         !write(*,*) 'done enter_setmatrix'
         
         if (.not. save_jac_plot_data) return

         write(*, *) 'save jacobian plot data'
         nzlo = s% hydro_jacobian_nzlo
         nzhi = s% hydro_jacobian_nzhi
         if (nzhi < nzlo) nzhi = nz
         call write_3point_jacobian_info( &
               ldA, nvar*nz, A, idiag, nvar, nz, nzlo, nzhi, 1, nvar, 1, nvar, &
               xscale, s% nameofvar, s% nameofequ, 'plot_data/jacobian_data')
         write(*,*) 'enter_setmatrix: hit CR to continue'
         read(*,*)
         
         contains
         
         
         subroutine clip_mtx(clip_limit)
            real(dp), intent(in) :: clip_limit
            include 'formats'
            cnt = 0
            nnz = 0
            if (s% hydro_matrix_type == block_tridiag_dble_matrix_type .or. &
                s% hydro_matrix_type == block_tridiag_quad_matrix_type) then
               do k=1,nz
                  do j=1,nvar
                     do i=1,nvar
                        if (k < nz .and. s% ublk(i,j,k) /= 0) then
                           nnz = nnz+1
                           if (abs(s% ublk(i,j,k)) < clip_limit) then
                              cnt = cnt + 1
                              s% ublk(i,j,k) = 0
                           end if
                        end if
                        if (s% dblk(i,j,k) /= 0) then
                           nnz = nnz+1
                           if (abs(s% dblk(i,j,k)) < clip_limit) then
                              cnt = cnt + 1
                              s% dblk(i,j,k) = 0
                           end if
                        end if
                        if (k > 1 .and. s% lblk(i,j,k) /= 0) then
                           nnz = nnz+1
                           if (abs(s% lblk(i,j,k)) < clip_limit) then
                              cnt = cnt + 1
                              s% lblk(i,j,k) = 0
                           end if
                        end if
                     end do
                  end do
               end do
            else
               do i=1,neqns
                  do j=idiag-(2*nvar-1),idiag+(2*nvar-1)
                     if (A(j,i) /= 0) then
                        nnz = nnz+1
                        if (abs(A(j,i)) < clip_limit) then
                           cnt = cnt + 1
                           A(j,i) = 0
                        end if
                     end if
                  end do
               end do
            end if

         end subroutine clip_mtx
         
         
         subroutine write_block_data(ierr)
            use mtx_lib, only: mtx_write_block_tridiagonal
            integer, intent(out) :: ierr
            integer :: iounit, k, i, j, nvar, nz
            character (len=64) :: fname_out
            include 'formats'
            if (s% nvar > 99) then
               write(fname_out,'(a,i3,a)') 'block_tri_', s% nvar, '.data'
            else
               write(fname_out,'(a,i2,a)') 'block_tri_', s% nvar, '.data'
            end if
            nvar = s% nvar
            nz = s% nz
            write(*,'(a,2i6)') 'save test matrix info to ' // trim(fname_out), nvar, nz
            iounit = 33
            ierr = 0
            open(iounit,file=trim(fname_out),iostat=ierr)
            if (ierr /= 0) return
            call mtx_write_block_tridiagonal(iounit,nvar,nz,s% lblk,s% dblk,s% ublk,ierr)
            close(iounit)
            if (ierr /= 0) return
            write(*,'(a,2i6)') 'done saving test matrix info to ' // trim(fname_out), nvar, nz
         end subroutine write_block_data
         
         
         subroutine write_1block_data
            use const_def
            integer :: k, i, j, klo, khi, nvar_lo, nvar_hi
            character (len=64) :: fname_out
            include 'formats'
            klo=1265 ! !s% nz
            khi=1267 ! !s% nz
            nvar_lo = 1
            nvar_hi = s% nvar
            do i = nvar_lo, nvar_hi
               write(*,'(a14)',advance='no') trim(s% nameofvar(i))
               if (i <= s% nvar_hydro) then
                  do k = klo, khi
                     write(*,'(e14.4)',advance='no') s% xh(i,k)
                  end do
               else
                  do k = klo, khi
                     write(*,'(e14.4)',advance='no') s% xa(i-s% nvar_hydro,k)
                  end do
               end if
               write(*,*)
            end do
            do k=klo,khi
               write(*,2) 'k', k, s% sig(k)
               write(*,'(a14)',advance='no') ''
               do i=nvar_lo, nvar_hi
                  write(*,'(a14)',advance='no') trim(s% nameofvar(i))
               end do
               write(*,*)
               do i=nvar_lo, nvar_hi
                  write(*,'(a14)',advance='no') trim(s% nameofequ(i))
                  do j=nvar_lo, nvar_hi
                     write(*,'(e14.4)',advance='no') s% dblk(i,j,k)
                  end do
                  write(*,*)
               end do
            end do
            stop 'write_1block_data'
         end subroutine write_1block_data
         
         
         subroutine check_xder_lnR
            integer :: i_lnR, k
            real(dp) :: lnR00, lnRm1, lnRp1, dlnR_prev, ddx, ddx_limit
            include 'formats'
            i_lnR = s% i_lnR
            do k=1,nz
               ddx = xder(i_lnR,k)
               lnR00 = s% xh_pre(i_lnR,k) + dx(i_lnR,k)
               if (ddx > 0) then ! compare lnR00 to lnRm1
                  if (k == 1) cycle
                  lnRm1 = s% xh_pre(i_lnR,k-1) + dx(i_lnR,k-1)
                  ddx_limit = 0.5d0*(lnRm1 - lnR00)
                  if (ddx > ddx_limit) then
                     xder(i_lnR,k) = ddx_limit
                  end if
               else ! compare lnR00 to lnRp1
                  if (k == nz) exit
                  lnRp1 = s% xh_pre(i_lnR,k+1) + dx(i_lnR,k+1)
                  ddx_limit = 0.5d0*(lnRp1 - lnR00) ! ddx and ddx_limit both < 0
                  if (ddx < ddx_limit) then
                     xder(i_lnR,k) = ddx_limit
                  end if
               end if
            end do            
         end subroutine check_xder_lnR
         
            
      end subroutine enter_setmatrix
      
      
      subroutine eval_equ_for_numerical_jacobian(s, dt, xscale, ierr)
         use hydro_eqns, only:eval_equ
         type (star_info), pointer :: s
         real(dp), intent(in) :: dt
         real(dp), pointer, dimension(:,:) :: xscale ! (nvar, nz)
         integer, intent(out) :: ierr
         
         logical :: do_chem
         integer :: nvar
         logical, parameter :: skip_partials = .true.
         ierr = 0
         do_chem = (s% do_burn .or. s% do_mix)
         if (do_chem) then ! include abundances
            nvar = s% nvar
         else ! only structure
            nvar = s% nvar_hydro
         end if
         call eval_equ(s, nvar, dt, skip_partials, xscale, ierr)
         if (ierr /= 0) then
            if (s% report_ierr) write(*, *) &
               'eval_equ_for_numerical_jacobian: eval_equ returned ierr', ierr
            return
         end if
         
      end subroutine eval_equ_for_numerical_jacobian
      
      
      subroutine eval_partials(s, dt, xscale, ierr)
         use hydro_eqns, only: eval_equ
         type (star_info), pointer :: s
         real(dp), intent(in) :: dt
         real(dp), pointer, dimension(:,:) :: xscale ! (nvar, nz)
         integer, intent(out) :: ierr
         
         logical :: do_chem
         integer :: nvar
         logical, parameter :: skip_partials = .false.
         ierr = 0
         do_chem = (s% do_burn .or. s% do_mix)
         if (do_chem) then ! include abundances
            nvar = s% nvar
         else ! only structure
            nvar = s% nvar_hydro
         end if
         call eval_equ(s, nvar, dt, skip_partials, xscale, ierr)
         if (ierr /= 0) then
            if (s% report_ierr) write(*, *) 'eval_partials: eval_equ returned ierr', ierr
            return
         end if
         
      end subroutine eval_partials
      

      subroutine exit_setmatrix( &
            iter, nvar_in, nz, neqns, dx, ldA, A1, idiag, xscale, lrpar, rpar, lipar, ipar, ierr)
         use mtx_lib, only:write_3point_jacobian_info
         use const_def, only: Rsun
         integer, intent(in) :: iter, nvar_in, nz, neqns
         real(dp), pointer :: dx(:,:) ! (nvar, nz)
         integer, intent(in) :: ldA ! leading dimension of A
         real(dp), pointer, dimension(:) :: A1 ! jacobian data
         integer, intent(inout) :: idiag ! row of A with the matrix diagonal entries
         real(dp), pointer, dimension(:,:) :: xscale ! (nvar, nz)
         integer, intent(in) :: lrpar, lipar
         real(dp), intent(inout) :: rpar(:) ! (lrpar)
         integer, intent(inout) :: ipar(:) ! (lipar)
         integer, intent(out) :: ierr

         type (star_info), pointer :: s
         real(dp), pointer, dimension(:,:) :: A ! (ldA, neqs)
         integer :: nvar, id, nvar_hydro, equP, equT, equR, equL, equv, j, i, k, equ_k, var_k
         real(dp), pointer :: save_A1(:,:), save_A2(:,:)
         logical :: save_numjac_plot_data, test_analytical_jacobian, dbg_exit, do_chem 
         real(dp) :: dt
         
         include 'formats'
         
         dbg_exit = dbg         
         id = ipar(ipar_id)
         
         ierr = 0
         
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) then
            ierr = -1
            return
         end if
         
         if (dbg_exit) then
            write(*, *) 'exit_setmatrix'
            write(*, 2) 'nvar', nvar
            write(*, 2) 'species', s% species
            write(*, 2) 'nvar_hydro', s% nvar_hydro
            write(*, 2) 'nz', nz
            write(*, 2) 'neqns', neqns
            write(*, 2) 'idiag', idiag
            write(*, 2) 'ldA', ldA
            write(*, *) 
         end if
         
         dt = rpar(rpar_dt)
         
         do_chem = (s% do_burn .or. s% do_mix)
         if (do_chem) then ! include abundances
            nvar = s% nvar
         else ! only structure
            nvar = s% nvar_hydro
         end if
         
         if (nvar /= nvar_in .or. nz /= s% nz .or. neqns /= nvar*nz) then
            write(*,*) 'nvar', nvar
            write(*,*) 'nvar_in', nvar_in
            write(*,*) 'nz', nz
            write(*,*) 's% nz', s% nz
            ierr = -1
            return
         end if

         s% doing_numerical_jacobian = .false.

         if (.not. s% numerical_jacobian) return

         ! numerical jacobian is only used for debugging the analytical one
         
         if ((s% hydro_call_number /= s% hydro_dump_call_number) &
            .and. (s% hydro_dump_call_number > 0)) return
            
         if (s% hydro_dump_iter_number > 0 &
            .and. iter /= s% hydro_dump_iter_number) return
         
         save_numjac_plot_data = s% hydro_save_numjac_plot_data        
         
         test_analytical_jacobian = save_numjac_plot_data &
                  .or. (s% hydro_dump_call_number > 0)
         
         nvar_hydro = s% nvar_hydro
         
         equP = s% equP
         equT = s% equT
         equR = s% equR
         equL = s% equL
         equv = s% equv

         if (.not. test_analytical_jacobian) then
            if (save_numjac_plot_data) call write_numeric(nvar)
            return
         end if
         
         s% idiag = idiag
         A(1:ldA,1:neqns) => A1(1:ldA*neqns)
         s% jacobian(1:ldA,1:neqns) => A1(1:ldA*neqns)

         
         allocate(save_A1(ldA,neqns),save_A2(ldA,neqns),stat=ierr)
         if (ierr /= 0) then
            write(*,*) 'allocate failed in exit_setmatrix'
            return
         end if
         
         ! save_A1 gets the numerical jacobian
         do j=1,neqns
            do i=1,ldA
               save_A1(i,j) = A(i,j)
               A(i,j) = 0
            end do
         end do
         
         write(*,*) 'exit_setmatrix: evaluate analytic partials' 
         
         !k = 28
         !write(*,2) 'r(k)/Rsun', k, s% r(k)/Rsun
         !write(*,*)
         
         ! evaluate analytic partials
         !write(*,2) 'lnr(k)', 304, s% lnr(304)
         write(*,*) 'exit_setmatrix: call set_newton_vars'
         call set_newton_vars(s, iter, dx, xscale, dt, ierr)
         !write(*,2) 'lnr(k)', 304, s% lnr(304)
         if (ierr /= 0) then
            if (s% report_ierr) write(*, *) 'exit_setmatrix: set_newton_vars returned ierr', ierr
            return
         end if
         
         !write(*,*)
         !write(*,2) 'r(k)/Rsun', k, s% r(k)/Rsun
         !write(*,*)
         !write(*,*)
         !write(*,*)

         write(*,*) 'exit_setmatrix: call eval_partials'
         call eval_partials(s, dt, xscale, ierr)
         if (ierr /= 0) return
         write(*,*) 'exit_setmatrix: done eval_partials'

         if (save_numjac_plot_data) then
            call write_jacobians(s,nvar,save_A1,save_A2)
         end if
         
         ! restore numeric jacobian
         do j=1,neqns
            do i=1,ldA
               A(i,j) = save_A1(i,j)
            end do
         end do

         deallocate(save_A1,save_A2)

         
         contains
         
         
         subroutine get_names(nvar, nmvar, nmequ)
            use chem_def
            integer, intent(in) :: nvar
            character*32, dimension(nvar) :: nmvar, nmequ
            integer :: i
            integer, pointer :: chem_id(:)
            nmvar(1:nvar_hydro) = s% nameofvar(1:nvar_hydro)
            nmequ(1:nvar_hydro) = s% nameofequ(1:nvar_hydro)
            if (nvar == nvar_hydro) return
            chem_id => s% chem_id
            do i=nvar_hydro+1,nvar
               write(nmvar(i),'(a)') 'var_' // trim(chem_isos% name(chem_id(i-nvar_hydro)))
               write(nmequ(i),'(a)') 'equ_' // trim(chem_isos% name(chem_id(i-nvar_hydro)))
            end do
         end subroutine get_names
            
            
         subroutine write_numeric(nvar)
            integer, intent(in) :: nvar
            character*32, dimension(nvar) :: nmvar, nmequ
            real(dp) :: rcond
            call get_names(nvar, nmvar, nmequ)
            write(*, *) 'save numerical jacobian plot data'
            call write_3point_jacobian_info( &
                  ldA, neqns, A, idiag, nvar, nz, 1, nz, 1, nvar, 1, nvar, &
                  xscale, nmvar, nmequ, 'plot_data/numerical_jacobian')
            stop 'debug: write_numeric'
         end subroutine write_numeric
                     
            
         subroutine write_jacobians(s,nvar,save_A1,save_A2)
            use star_utils, only: std_write_internals_to_file
            type (star_info), pointer :: s
            integer, intent(in) :: nvar
            real(dp), pointer :: save_A1(:,:), save_A2(:,:)
            real(dp) :: rcond
            character*32, dimension(nvar) :: nmvar, nmequ
            
            integer :: i, j, nzlo, nzhi
            
            call get_names(nvar, nmvar, nmequ)

            nzlo = s% hydro_jacobian_nzlo
            nzhi = s% hydro_jacobian_nzhi
            if (nzhi < nzlo) nzhi = nz
            
            write(*, *) 'save analytic jacobian plot data'
            call write_3point_jacobian_info( &
                  ldA, neqns, A, idiag, nvar, nz, nzlo, nzhi, 1, nvar, 1, nvar, &
                  xscale, nmvar, nmequ, 'plot_data/jacobian_data')
            !write(*, *) 'done write_3point_jacobian_info'
               
            ! restore numeric jacobian
            do j=1,neqns
               do i=1,ldA
                  A(i,j) = save_A1(i,j)
               end do
            end do

            write(*, *) 'save numerical jacobian plot data'
            call write_3point_jacobian_info( &
                  ldA, neqns, A, idiag, nvar, nz, nzlo, nzhi, 1, nvar, 1, nvar, &
                  xscale, nmvar, nmequ, 'plot_data/numerical_jacobian')
         
            deallocate(save_A1, save_A2)
            
            call std_write_internals_to_file(s% id, 0)
         
            stop 'done write_jacobians: exit_setmatrix'
            
         end subroutine write_jacobians
         
         
      end subroutine exit_setmatrix


      subroutine copy_one_to_3point_jacobian( &
            ldA, neqns, nvar, nz, nzlo, nzhi, A, idiag, eqn, em1, e00, ep1, xscale)
         integer, intent(in) :: ldA ! leading dimension of A
         real(dp), intent(out) :: A(ldA, neqns) ! the jacobian matrix
         ! A(idiag+q-v, v) = partial of equation(q) wrt variable(v)
         integer, intent(in) :: idiag, neqns, nvar, nz, nzlo, nzhi, eqn
         real(dp), dimension(nvar, nvar, nz), intent(in) :: em1, e00, ep1         
         real(dp), intent(in) :: xscale(nvar, nz)         
         integer :: var, k, dk, ii, jj         
         do dk = -1, 1 ! 3 point stencil
            do var = 1, nvar
               !write(*,*) 'dk, var', dk, var
               ii = eqn - var - nvar*dk + idiag
               jj = var + nvar*(dk-1)
               select case(dk)
                  case(-1) 
                     do k=max(2,nzlo),nzhi
                        A(ii,jj+nvar*k) = em1(eqn, var, k)*xscale(var, k-1)
                     end do
                  case(0) 
                     do k=nzlo,nzhi
                        A(ii,jj+nvar*k) = e00(eqn, var, k)*xscale(var, k)
                     end do
                  case(1) 
                     do k=nzlo,min(nz-1,nzhi)
                        A(ii,jj+nvar*k) = ep1(eqn, var, k)*xscale(var, k+1)
                     end do
               end select
            end do
         end do
      end subroutine copy_one_to_3point_jacobian

      
      subroutine copy_all_from_3point_jacobian( &
               ldA, neqns, nvar, nz, nzlo, nzhi, A, idiag, em1, e00, ep1, xscale)
         integer, intent(in) :: ldA ! leading dimension of A
         integer, intent(in) :: neqns, nvar, nz, nzlo, nzhi
         real(dp), intent(in) :: A(ldA, nvar*nz) ! the jacobian matrix
         ! A(idiag+q-v, v) = partial of equation(q) wrt variable(v)
         integer, intent(in) :: idiag
         real(dp), dimension(nvar, nvar, nz), intent(out) :: em1, e00, ep1         
         real(dp), intent(in) :: xscale(nvar, nz)
         integer :: eqn
         em1 = 0; e00 = 0; ep1 = 0
         do eqn=1, nvar
            call copy_one_from_3point_jacobian( &
               ldA, neqns, nvar, nz, 1, nz, A, idiag, eqn, em1, e00, ep1, xscale)
         end do
      end subroutine copy_all_from_3point_jacobian

      
      subroutine copy_one_from_3point_jacobian( &
               ldA, neqns, nvar, nz, nzlo, nzhi, A, idiag, eqn, em1, e00, ep1, xscale)
         integer, intent(in) :: ldA ! leading dimension of A
         integer, intent(in) :: neqns, nvar, nz, nzlo, nzhi
         integer, intent(in) :: idiag, eqn
         real(dp), intent(in) :: A(ldA, nvar*nz) ! the jacobian matrix
         ! A(idiag+q-v, v) = partial of equation(q) wrt variable(v)
         real(dp), dimension(nvar, nvar, nz), intent(out) :: em1, e00, ep1         
         ! em1(i, j, k) is partial of equation i of cell k wrt variable j of cell k-1
         ! e00(i, j, k) is partial of equation i of cell k wrt variable j of cell k
         ! ep1(i, j, k) is partial of equation i of cell k wrt variable j of cell k+1
         real(dp), intent(in) :: xscale(nvar, nz)
         integer :: var, k, dk, ii, jj
         ! move partials from A to em1, e00, and ep1
         do dk = -1, 1 ! 3 point stencil
            do var = 1, nvar
               ii = eqn - var - nvar*dk + idiag
               jj = var + nvar*(dk-1)
               select case(dk)
                  case(-1) 
                     do k=max(2,nzlo),nzhi
                        em1(eqn, var, k) = A(ii, jj+nvar*k)/xscale(var, k-1)
                     end do
                  case(0) 
                     do k=nzlo,nzhi
                        e00(eqn, var, k) = A(ii, jj+nvar*k)/xscale(var, k)
                     end do
                  case(1) 
                     do k=nzlo,min(nz-1,nzhi)
                        ep1(eqn, var, k) = A(ii, jj+nvar*k)/xscale(var, k+1)
                     end do
               end select
            end do
         end do
      end subroutine copy_one_from_3point_jacobian
      
      
      subroutine store1_in_3point_jacobian( &
            ldA, neqns, nvar, nz, A, idiag, eqn, var, k, dk, value, xscale)
         integer, intent(in) :: ldA ! leading dimension of A
         integer, intent(in) :: neqns, nvar, nz, idiag, var, eqn, k, dk
         real(dp), intent(out) :: A(ldA, nvar*nz) ! the jacobian matrix
         ! A(idiag+q-v, v) = partial of equation(q) wrt variable(v)
         real(dp), intent(in) :: value, xscale(nvar, nz)
         integer :: ii, jj
         ii = eqn - var - nvar*dk + idiag
         jj = var + nvar*(dk-1)
         select case(dk)
            case(-1) 
               A(ii, jj+nvar*k) = value*xscale(var, k-1)
            case(0) 
               A(ii, jj+nvar*k) = value*xscale(var, k)
            case(1) 
               A(ii, jj+nvar*k) = value*xscale(var, k+1)
         end select
      end subroutine store1_in_3point_jacobian
         

      subroutine show_jacobian_block(s, equ_k, var_k, ldA, neqns, nvar, nz, A, idiag, xscale)
         ! show J(i,j) for i in equations for equ_k and j in variables for var_k
         type (star_info), pointer :: s
         integer, intent(in) :: equ_k ! k for equations
         integer, intent(in) :: var_k ! k for vars
         integer, intent(in) :: ldA ! leading dimension of A
         integer, intent(in) :: neqns, nvar, nz, idiag
         real(dp), intent(in) :: A(ldA, neqns) ! the jacobian matrix in banded diagonal form
         real(dp), intent(in) :: xscale(nvar, nz)
         
         integer :: i, j, nvar_hydro, equ, var
         real(dp) :: mtx(nvar,nvar)
         logical :: all_zeros, row_all_zeros(nvar), col_all_zeros(nvar)
         
         call get_one_from_3point_jacobian( &
            equ_k, var_k, ldA, neqns, nvar, nz, A, idiag, mtx, xscale)
            
         nvar_hydro = s% nvar_hydro
         
         write(*,*)
         write(*,*)
         write(*,'(12x)', advance='no')
         do var = 1, nvar_hydro
            write(*, '(a24)', advance='no') trim(s% nameofvar(var))
         end do
         write(*,*)
         do equ = 1, nvar_hydro
            write(*, '(a12)', advance='no') trim(s% nameofequ(equ))
            do var = 1, nvar_hydro
               write(*, '(e24.10)', advance='no') mtx(equ,var)
            end do
            write(*,*)
         end do
         write(*,*)
         write(*,*)
         write(*,*) 'equ_k', equ_k
         write(*,*) 'var_k', var_k
         write(*,*)
         write(*,*)
         
         !stop 'debug: show_jacobian_block'
         
         
         
 1       format(i5, 1pe26.16)
         write(*, *)
         do i=1, nvar
            write(*, '(a15, i5)') 'column', i
            all_zeros = .true.
            col_all_zeros(i) = .true.
            do j=1, nvar
               if (mtx(j, i) == 0) then
                  write(*, '(i5, i15)') j, 0
               else
                  write(*, 1) j, mtx(j, i)
                  all_zeros = .false.
                  col_all_zeros(i) = .false.
               end if
            end do
            if (all_zeros) write(*, *) &
                        '****************************** ALL ZEROS ******************************'
            write(*, *)
         end do
         
         write(*, *)
         do i=1, nvar
            write(*, '(a15, i5)') 'row', i
            all_zeros = .true.
            row_all_zeros(i) = .true.
            do j=1, nvar
               if (mtx(i, j) == 0) then
                  write(*, '(i5, i15)') j, 0
               else
                  write(*, 1) j, mtx(i, j)
                  all_zeros = .false.
                  row_all_zeros(i) = .false.
               end if
            end do
            if (all_zeros) write(*, *)  &
                        '****************************** ALL ZEROS ******************************'
            write(*, *)
         end do
         
         write(*, *)
         do i=1, nvar
            if (col_all_zeros(i)) write(*, *) 'all zeros in column', i
            if (row_all_zeros(i)) write(*, *) 'all zeros in row', i
         end do
         write(*, *)


      end subroutine show_jacobian_block

      
      subroutine get_one_from_3point_jacobian( &
               equ_k, var_k, ldA, neqns, nvar, nz, A, idiag, mtx, xscale)
         integer, intent(in) :: equ_k ! k for equations
         integer, intent(in) :: var_k ! k for vars
         integer, intent(in) :: ldA ! leading dimension of A
         integer, intent(in) :: neqns, nvar, nz
         integer, intent(in) :: idiag
         real(dp), intent(in) :: A(ldA, nvar*nz) ! the jacobian matrix
         ! A(idiag+q-v, v) = partial of equation(q) wrt variable(v)
         real(dp), dimension(nvar, nvar), intent(out) :: mtx
         real(dp), intent(in) :: xscale(nvar, nz)         
         integer :: dk, ii, jj, eqn, var       
         ! move partials from A to mtx         
         dk = var_k - equ_k
         if (abs(dk) > 1) then ! BUG
            write(*,*) 'get_one_from_3point_jacobian: bad dk', dk
            mtx = 0
            return
         end if
         do eqn = 1, nvar
            do var = 1, nvar
               ii = eqn - var - nvar*dk + idiag
               jj = var + nvar*(dk-1)
               mtx(eqn, var) = A(ii, jj+nvar*equ_k)!/xscale(var, var_k)
            end do
         end do
      end subroutine get_one_from_3point_jacobian
      

      end module hydro_mtx

