! ***********************************************************************
!
!   Copyright (C) 2012  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************


      module hydro_eqns

      use star_private_def
      use const_def
      use utils_lib, only: is_bad_num
      use star_utils, only: em1, e00, ep1
      use hydro_vars, only: eval_dlnPdm_qhse, do_dlnP_dm
      
      

      implicit none

      real(dp), parameter :: one = 1, zero = 0
      
      logical, parameter :: dbg = .false.   
      
      integer, parameter :: dbg_cell = -1
      

      contains


      subroutine eval_equ(s, nvar, dt, skip_partials, xscale, ierr)
         type (star_info), pointer :: s         
         integer, intent(in) :: nvar
         real(dp), intent(in) :: dt ! time since start of current timestep
         logical, intent(in) :: skip_partials
         real(dp), pointer, dimension(:,:) :: xscale ! (nvar, nz)
         integer, intent(out) :: ierr
         call eval_equ_for_solver(s, nvar, 1, s% nz, dt, skip_partials, xscale, ierr)
      end subroutine eval_equ
      

      subroutine eval_equ_for_solver( &
            s, nvar, nzlo, nzhi, dt, skip_partials, xscale, ierr)
         use chem_def
         use mesh_functions
         use hydro_chem_eqns, only: do_chem_eqns, do1_chem_eqns
         use star_utils, only: update_time, total_times

         type (star_info), pointer :: s         
         integer, intent(in) :: nvar, nzlo, nzhi
         real(dp), intent(in) :: dt ! time since start of current timestep
         logical, intent(in) :: skip_partials
         real(dp), pointer, dimension(:,:) :: xscale ! (nvar, nz)
         integer, intent(out) :: ierr

         integer :: &
            equP, equT, equR, equL, equv, equchem1, &
            k, j, nvar_hydro, nvar_chem, nz, op_err, matrix_type
         integer :: &
            i_xlnd, i_lnPgas, i_lnR, i_lnT, i_lum, i_vel, &
            i_chem1, i_xh1, i_xhe4, kmax_equ(nvar), species
         real(dp) :: max_equ(nvar), total_all_before
         real(dp) :: dVARdot_dVAR, L_phot_old, dot_factor
         real(dp), dimension(:), pointer :: &
            L, lnR, lnP, lnT, dL_dm, dlnP_dm, dlnT_dm, lnE
         real(dp), dimension(:, :), pointer :: equ
         logical :: v_flag, dump_for_debug, do_Pvisc, do_chem, do_mix
         integer :: time0, clock_rate
         
         include 'formats'
         
         ierr = 0
         
         if (s% doing_timing) then
            total_all_before = total_times(s)
            call system_clock(time0,clock_rate)
         end if

         dump_for_debug = .false. !(s% hydro_call_number == 3)
         !dump_for_debug = .true.         
         
         do_Pvisc = (s% v_flag .and. s% use_artificial_viscosity) .or. &
            s% use_other_viscosity_pressure
         
         do_mix = s% do_mix
         do_chem = (do_mix .or. s% do_burn)
         
         s% max_x_expected = 1d0
         s% min_x_expected = 0d0
         
         call unpack
         
         dot_factor = max(1d0, dt)
         
         if (dbg) write(*, *) 'eqns', dt
         
         if (.not. (s% do_struct_hydro .or. s% do_struct_thermo)) then 
         
            if (.not. do_chem) then
               write(*,*) 'bug: all false: do_chem, do_struct_hydro, and do_struct_thermo'
               stop 'eval_equ_for_solver'
            end if
            ! hold structure constant while solve burn and/or mix
            do j=1,nvar_hydro
               call null_eqn(j)
            end do
            if (ierr == 0) &
               call do_chem_eqns(s, xscale, nvar, equchem1, nvar_chem, skip_partials, dt, equ, ierr)
               
         else ! solving structure equations
            
            if (s% use_other_energy_implicit) then
               call s% other_energy_implicit(s% id, ierr)
               if (ierr /= 0) then
                  if (s% report_ierr) write(*,*) 'eval_equ_for_solver: ierr from other_energy_implicit'
                  return
               end if
            end if

!$OMP PARALLEL DO PRIVATE(op_err,k)
            do k = nzlo, nzhi
               if (.not. skip_partials .and. &
                     matrix_type == block_tridiag_dble_matrix_type) then
                  s% dblk(:,:,k) = 0
                  s% ublk(:,:,k) = 0
                  s% lblk(:,:,k) = 0
               end if
               op_err = 0
               if (s% do_struct_hydro) then
                  call do1_density_eqn(k,op_err)
                  if (op_err /= 0) ierr = op_err            
                  if (k > 1) call do1_pressure_eqn(k,op_err)
                  if (op_err /= 0) ierr = op_err 
                  if (equv /= 0) call do1_velocity_eqn(k,op_err)
                  if (op_err /= 0) ierr = op_err 
               end if
               if (s% do_struct_thermo) then
                  call do1_luminosity_eqn(k,op_err)
                  if (op_err /= 0) ierr = op_err            
                  if (k > 1) call do1_temperature_eqn(k,op_err)
                  if (op_err /= 0) ierr = op_err
                  if (op_err /= 0) ierr = op_err 
               end if                                
               if (do_chem) then
                  call do1_chem_eqns( &
                     s, xscale, k, nvar, equchem1, nvar_chem, &
                     skip_partials, dt, equ, op_err)
                  if (op_err /= 0) ierr = op_err
               end if
            end do      
!$OMP END PARALLEL DO

            if (ierr == 0 .and. nzlo == 1 .and. &
                  (s% do_struct_hydro .or. s% do_struct_thermo)) then
               if (dbg) write(*,*) 'call PT_eqns_surf'
               call PT_eqns_surf(s, xscale, equ, skip_partials, nvar, ierr)
               if (dbg) write(*,*) 'done PT_eqns_surf'
            end if

            if (.not. s% do_struct_hydro) then
               call dummy_eqn(equR,i_lnR,nzlo,nzhi) 
               call dummy_eqn(equP,i_xlnd,max(2,nzlo),nzhi)
               if (equv /= 0) call dummy_eqn(equv,i_vel,nzlo,nzhi)
            end if
            
            if (.not. s% do_struct_thermo) then
               call dummy_eqn(equL,i_lum,nzlo,nzhi)
               call dummy_eqn(equT,i_lnT,max(2,nzlo),nzhi)
            end if
            
         end if

         if (ierr /= 0) then
            if (s% report_ierr) write(*,*) 'eval_equ_for_solver: ierr after eval equations'
            return
         end if

         if (s% hydro_check_everything .and. &
               s% hydro_call_number == s% hydro_dump_call_number) then ! check everything
            call check_everything
         end if
         
         if (s% doing_timing) call update_time(s, time0, total_all_before, s% time_eval_eqns)
         
         if (dump_for_debug .and. .not. skip_partials) call dump_equ_and_partials
         
         return
         if (s% model_number /= 121) return
         
         !call dump_equL_info ! TESTING
         
         !call dump_equ ! TESTING
         
         
         contains
         
         
         subroutine unpack
         
            include 'formats'
            
            nz = s% nz
            species = s% species
            nvar_hydro = s% nvar_hydro
            matrix_type = s% hydro_matrix_type
            
            lnT => s% lnT
            lnR => s% lnR
            L => s% L
            lnP => s% lnP
            lnE => s% lnE
            equ(1:nvar,1:nz) => s% equ1(1:nvar*nz)
            dVARdot_dVAR = s% dVARdot_dVAR
            
            equP = s% equP
            equT = s% equT
            equR = s% equR
            equL = s% equL
            equv = s% equv
            
            equchem1 = s% equchem1
            
            i_xlnd = s% i_xlnd
            i_lnPgas = s% i_lnPgas
            i_lnT = s% i_lnT
            i_lnR = s% i_lnR
            i_lum = s% i_lum
            i_vel = s% i_vel

            i_chem1 = s% i_chem1
            i_xh1 = i_chem1-1+s% net_iso(ih1)
            i_xhe4 = i_chem1-1+s% net_iso(ihe4)
            
            L_phot_old = s% L_phot_old
            v_flag = s% v_flag
            
            nvar_chem = s% nvar_chem
            
            dL_dm => s% dL_dm
            dlnP_dm => s% dlnP_dm
            dlnT_dm => s% dlnT_dm
            
         end subroutine unpack
         
         
         subroutine null_eqn(j)
            integer, intent(in) :: j
            integer :: k
            do k=nzlo,nzhi
               equ(j,k) = 0 ! s% xs(j,k) - s% xs_pre_pass(j,k)
               if (.not. skip_partials) call e00(s, xscale,j,j,k,nvar,one)
            end do
         end subroutine null_eqn
         
               
         subroutine do1_density_eqn(k,ierr) ! mass-volume-density relation
            integer, intent(in) :: k
            integer, intent(out) :: ierr
            real(dp) :: rp13, dm, rho, dr3, dequ_ddr3, ddr3_dlnd, &
               ddr3_dlnPgas_const_T, ddr3_dlnT_const_Pgas
               
            include 'formats'

            ierr = 0
      
            dm = s% dm(k)
            rho = s% rho(k)
            ! dm/rho is cell volume
            dr3 = (dm/rho)/(pi4/3)
         
            if (k < nz) then
               rp13 = s% r(k+1)**3
            else
               rp13 = s% R_center**3
            end if
            ! dm = (pi4/3)*(r(k)**3 - rp13)*rho
            ! r(k)**3 = rp13 + (dm/rho)/(pi4/3) = rp13 + dr3            
            equ(equR, k) = lnR(k) - log(rp13 + dr3)/3
         
            if (skip_partials) return

            call e00(s, xscale, equR, i_lnR, k, nvar, one)
            if (k < nz) call ep1(s, xscale, equR, i_lnR, k, nvar, -rp13/(rp13 + dr3))

            dequ_ddr3 = -one_third/(rp13 + dr3)
            ddr3_dlnd = -dr3
            
            if (s% lnPgas_flag) then
               ddr3_dlnPgas_const_T = ddr3_dlnd*s% dlnRho_dlnPgas_const_T(k)
               call e00(s, xscale, equR, i_lnPgas, k, nvar, dequ_ddr3*ddr3_dlnPgas_const_T)
               ddr3_dlnT_const_Pgas = ddr3_dlnd*s% dlnRho_dlnT_const_Pgas(k)
               call e00(s, xscale, equR, i_lnT, k, nvar, dequ_ddr3*ddr3_dlnT_const_Pgas)
            else
               call e00(s, xscale, equR, i_xlnd, k, nvar, dequ_ddr3*ddr3_dlnd)
            end if
            
         end subroutine do1_density_eqn


         subroutine do1_luminosity_eqn(k,ierr) ! luminosity gradient (energy conservation)
            use rates_def
            use mod_eps_grav, only: eval_eps_grav_and_partials
            use chem_def, only: iprot

            integer, intent(in) :: k
            integer, intent(out) :: ierr
         
            integer, pointer :: reaction_id(:) ! maps net reaction number to reaction id

            real(dp) :: L00, Lp1, Lx, dA_dlnR, diff, diff_old, &
               dm_max, dm, L_scale, dLdm, partial, &
               d_dLdM_dlndm1, d_dLdM_dlnTm1, &
               d_dLdM_dlnd00, d_dLdM_dlnT00, &
               d_dLdM_dlndp1, d_dLdM_dlnTp1, &
               d_dLdM_dlnR00, d_dLdM_dlnRp1, &
               d_dLdM_dL00, d_dLdM_dLp1, &
               d_faclnd_dlnd, d_faclnT_dlnT, &
               eps_burn, d_eps_burn_dlnd, d_eps_burn_dlnT, d_epsnuc_dlnd, d_epsnuc_dlnT, &
               non_nuc_neu, d_nonnucneu_dlnd, d_nonnucneu_dlnT, faclnd, faclnT, &
               Rho_new, Rho_start, T_new, T_start, dln_dlnd, dln_dlnT, &
               dL_expected, dL_actual, eps_nuc, Flx, dLx, d_dLx_epsnuc, &
               d_del_dlnR00, d_del_dlnRp1, d_del_dvel00, d_del_dvelp1, &
               d_dLAV_dlnd, d_dLAV_dlnT, d_dLAV_dlnR00, d_dLAV_dlnRp1, &
               d_dLAV_dvel00, d_dLAV_dvelp1, dLAV, del, qmid, eps_extra, &
               d_eps_burn_dlnPgas_const_T, d_eps_burn_dlnT_const_Pgas, &
               d_dLdM_dlnPgasm1_const_T, d_dLdM_dlnPgas00_const_T, d_dLdM_dlnPgasp1_const_T, &
               d_dLdM_dlnTm1_const_Pgas, d_dLdM_dlnT00_const_Pgas, d_dLdM_dlnTp1_const_Pgas, &
               d_dLAV_dlnT_const_Pgas, d_dLAV_dlnPgas_const_T, d_dLdM_dv00, d_dLdM_dvp1, &
               dLx_dLp1, dLx_dlnT00_const_Pgas, dLx_dlnTm1_const_Pgas, dLx_dlnTp1_const_Pgas, &
               dLx_dlnT00, dLx_dlnTm1, dLx_dlnTp1, dLx_dlnd00, dLx_dx, &
               dLx_dlnPgas00_const_T, dLx_dlnR00, dLx_dv00, dLx_dL00, &
               dLx_dlnRp1, dLx_dvp1, dLx_dlnPgasp1_const_T, &
               dLx_dlndp1, dLx_dlnPgasm1_const_T, dLx_dlndm1, &
               other_equ, other_f
               
            real(dp), parameter :: x_lim_for_d_epsnuc_dx = 1d-6

            integer :: j
            logical :: dbg
               
            include 'formats'
            ierr = 0
            
            call eval_eps_grav_and_partials(s, k, dt, ierr) ! get eps_grav info
            if (ierr /= 0) return

            dm = s% dm(k)
            
            eps_nuc = s% eps_nuc(k)
            d_epsnuc_dlnd = s% d_epsnuc_dlnd(k)
            d_epsnuc_dlnT = s% d_epsnuc_dlnT(k)

            ! dLdm is an average rate over the timestep.
            ! since non_nuc_neu is an instantaneous value,
            ! for consistency we use average.
            non_nuc_neu = 0.5d0*(s% non_nuc_neu_start(k) + s% non_nuc_neu(k))
            d_nonnucneu_dlnd = 0.5d0*s% d_nonnucneu_dlnd(k)
            d_nonnucneu_dlnT = 0.5d0*s% d_nonnucneu_dlnT(k)
            
            eps_burn = eps_nuc - non_nuc_neu + s% extra_heat(k) + s% irradiation_heat(k)
                     
            dLdm = s% eps_grav(k) + eps_burn ! optional artificial viscosity term treated below
            dL_expected = dm*dLdm
         
            if (do_Pvisc) then
               if (k < nz) then
                  del = s% v(k)*(3*s% Amid(k) - s% area(k)) - s% v(k+1)*(3*s% Amid(k) - s% area(k+1))
               else
                  del = s% v(k)*(3*s% Amid(k) - s% area(k)) ! V_center = 0
               end if
               dLAV = 0.5d0*s% Pvisc(k)*del
               dL_expected = dL_expected + dLAV
               dLdm = dL_expected/dm
            else
               del = 0
            end if
         
            L00 = L(k)
            if (k < nz) then
               Lp1 = L(k+1)
            else
               Lp1 = s% L_center
            end if
            dL_actual = L00 - Lp1
            s% dL_dm(k) = dL_actual/dm
            
            dLx = dLdm*dm ! dL expected
            Lx = Lp1 + dLx ! Lx = L00 expected = Lp1 + dLdm*dm
                     
            if (.not. s% doing_first_model_of_run) then
               L_scale = max(1d-3*Lsun, abs(s% xh_pre(i_lum,k)))
            else
               L_scale = Lsun
            endif
            
            ! energy conservation (luminosity equation)
            equ(equL, k) = Lx/L_scale - L00/L_scale
            
            dbg = .false. ! (k == 1763) !.and. (abs(equ(equL,k)) > 9d6)
            if (dbg) then
               write(*,2) 'L', k, s% L(k)
               write(*,2) 'Lp1', k+1, Lp1
               write(*,2) 'dm', k, dm
               write(*,2) '(L-Lp1)/dm', k, (s% L(k)-Lp1)/dm
               write(*,2) '(L-Lx)/dm', k, (s% L(k)-Lx)/dm
               write(*,2) 'dLdm', k, dLdm
               write(*,2) 'eps_nuc', k, eps_nuc
               write(*,2) 'non_nuc_neu', k, non_nuc_neu
               write(*,2) 's% extra_heat(k)', k, s% extra_heat(k)
               write(*,2) 's% irradiation_heat(k)', k, s% irradiation_heat(k)
               write(*,2) 'eps_burn', k, eps_burn
               write(*,2) 's% eps_grav(k)', k, s% eps_grav(k)
               write(*,2) 'equ(equL, k)', k, equ(equL, k)
               write(*,*)
            end if
            if (dbg .and. .false.) then
               write(*,2) 's% eps_grav(k)', k, s% eps_grav(k)
               write(*,*)
               write(*,2) 'eps_burn', k, eps_burn
               write(*,*)
               write(*,2) 'eps_nuc', k, eps_nuc
               write(*,2) 'd_epsnuc_dlnd', k, d_epsnuc_dlnd
               write(*,2) 'd_epsnuc_dlnT', k, d_epsnuc_dlnT
               write(*,*)
               write(*,2) 'logT', k, s% lnT(k)/ln10
               write(*,2) 'logT_start(k)', k, s% lnT_start(k)/ln10
               write(*,2) 'dlogT', k, (s% lnT(k) - s% lnT_start(k))/ln10
               write(*,*)
               write(*,2) 'logRho', k, s% lnd(k)/ln10
               write(*,2) 'logRho_start(k)', k, s% lnd_start(k)/ln10
               write(*,2) 'dlogRho', k, (s% lnd(k) - s% lnd_start(k))/ln10
               write(*,*)
               write(*,2) 'L', k, s% L(k)
               write(*,2) 'L_start', k, s% L_start(k)
               write(*,2) 'L - L_start', k, s% L(k) - s% L_start(k)
               write(*,*)
               write(*,2) 'Lp1', k+1, Lp1
               write(*,2) 'dLx', k, dLx
               write(*,*)
               write(*,2) 'dLdm', k, dLdm
               write(*,2) 'dLdm_start', k, (s% L_start(k) - s% L_start(k+1))/dm
               write(*,2) 'dLdm - dLdm_start', k, dLdm - (s% L_start(k) - s% L_start(k+1))/dm
               write(*,*)
               write(*,2) 'dm', k, dm
               write(*,2) 'L_scale', k, L_scale
               write(*,2) 'Lx', k, Lx
               write(*,2) 'L00', k, L00
               write(*,2) 'Lx - L00', k, Lx - L00
               write(*,2) 'equ(equL, k)', k, equ(equL, k)
               write(*,*)
               !if (abs(equ(equL,k)) > 9d6) stop 'equL'
            end if
      
            if (is_bad_num(equ(equL, k))) then
               ierr = -1
               if (s% report_ierr) write(*,2) 'equ(equL, k)', k, equ(equL, k)
               return
            end if

            if (skip_partials) return
         
            d_dLdM_dlndm1 = 0
            d_dLdM_dlnd00 = 0
            d_dLdM_dlndp1 = 0
            d_dLdM_dlnTm1 = 0
            d_dLdM_dlnT00 = 0
            d_dLdM_dlnTp1 = 0
            
            d_eps_burn_dlnd = d_epsnuc_dlnd - d_nonnucneu_dlnd + s% d_extra_heat_dlnd(k)
            d_eps_burn_dlnT = d_epsnuc_dlnT - d_nonnucneu_dlnT + s% d_extra_heat_dlnT(k)
      
            d_dLdM_dlnR00 = s% d_eps_grav_dlnR00(k) + s% d_extra_heat_dlnR00(k)
            d_dLdM_dlnRp1 = s% d_eps_grav_dlnRp1(k) + s% d_extra_heat_dlnRp1(k)
            d_dLdM_dL00 = s% d_eps_grav_dL00(k)
            d_dLdM_dLp1 = 0

            d_dLdM_dv00 = s% d_eps_grav_dv00(k) + s% d_extra_heat_dv00(k)
            d_dLdM_dvp1 = s% d_eps_grav_dvp1(k) + s% d_extra_heat_dvp1(k)

            if (s% lnPgas_flag) then
               
               d_eps_burn_dlnPgas_const_T = d_eps_burn_dlnd*s% dlnRho_dlnPgas_const_T(k)
               d_eps_burn_dlnT_const_Pgas = &
                  d_eps_burn_dlnT + d_eps_burn_dlnd*s% dlnRho_dlnT_const_Pgas(k)
      
               d_dLdM_dlnPgasm1_const_T = s% d_eps_grav_dlnPgasm1_const_T(k)
               d_dLdM_dlnPgas00_const_T = &
                  s% d_eps_grav_dlnPgas00_const_T(k) + d_eps_burn_dlnPgas_const_T
               d_dLdM_dlnPgasp1_const_T = s% d_eps_grav_dlnPgasp1_const_T(k)
            
               d_dLdM_dlnTm1_const_Pgas = s% d_eps_grav_dlnTm1_const_Pgas(k)
               d_dLdM_dlnT00_const_Pgas = &
                  s% d_eps_grav_dlnT00_const_Pgas(k) + d_eps_burn_dlnT_const_Pgas
               d_dLdM_dlnTp1_const_Pgas = s% d_eps_grav_dlnTp1_const_Pgas(k)

            else
      
               d_dLdM_dlndm1 = s% d_eps_grav_dlndm1(k)
               d_dLdM_dlnd00 = s% d_eps_grav_dlnd00(k) + d_eps_burn_dlnd
               d_dLdM_dlndp1 = s% d_eps_grav_dlndp1(k)
            
               d_dLdM_dlnTm1 = s% d_eps_grav_dlnTm1(k)
               d_dLdM_dlnT00 = s% d_eps_grav_dlnT00(k) + d_eps_burn_dlnT
               d_dLdM_dlnTp1 = s% d_eps_grav_dlnTp1(k)
               
               d_dLdM_dlnPgasm1_const_T = 0
               d_dLdM_dlnPgas00_const_T = 0
               d_dLdM_dlnPgasp1_const_T = 0
            
               d_dLdM_dlnTm1_const_Pgas = 0
               d_dLdM_dlnT00_const_Pgas = 0
               d_dLdM_dlnTp1_const_Pgas = 0

            end if
            
            dLx_dlnT00 = d_dLdm_dlnT00*dm
            dLx_dlnTm1 = d_dLdM_dlnTm1*dm
            dLx_dlnTp1 = d_dLdM_dlnTp1*dm
            dLx_dlnd00 = d_dLdm_dlnd00*dm
            dLx_dlnR00 = d_dLdM_dlnR00*dm
            dLx_dv00 = d_dLdM_dv00*dm
            dLx_dL00 = d_dLdM_dL00*dm
            dLx_dLp1 = d_dLdM_dLp1*dm
            dLx_dlnRp1 = d_dLdM_dlnRp1*dm
            dLx_dvp1 = d_dLdM_dvp1*dm
            dLx_dlndp1 = d_dLdM_dlndp1*dm
            dLx_dlndm1 = d_dLdM_dlndm1*dm
            
            if (s% lnPgas_flag) then
               dLx_dlnT00_const_Pgas = d_dLdm_dlnT00_const_Pgas*dm
               dLx_dlnTm1_const_Pgas = d_dLdM_dlnTm1_const_Pgas*dm
               dLx_dlnTp1_const_Pgas = d_dLdM_dlnTp1_const_Pgas*dm
               dLx_dlnPgasp1_const_T = d_dLdM_dlnPgasp1_const_T*dm
               dLx_dlnPgas00_const_T = d_dLdm_dlnPgas00_const_T*dm
               dLx_dlnPgasm1_const_T = d_dLdM_dlnPgasm1_const_T*dm
            else
               dLx_dlnT00_const_Pgas = 0
               dLx_dlnTm1_const_Pgas = 0
               dLx_dlnTp1_const_Pgas = 0
               dLx_dlnPgasp1_const_T = 0
               dLx_dlnPgas00_const_T = 0
               dLx_dlnPgasm1_const_T = 0
            end if
            
            call e00(s, xscale, equL, i_lum, k, nvar, dLx_dL00/L_scale - 1/L_scale)
            if (k < nz) call ep1( &
               s, xscale, equL, i_lum, k, nvar, dLx_dLp1/L_scale + 1/L_scale)
            
            if (s% lnPgas_flag) then
               call e00(s, xscale, equL, i_lnT, k, nvar, dLx_dlnT00_const_Pgas/L_scale)
               if (k > 1) then
                  call em1(s, xscale, equL, i_lnT, k, nvar, dLx_dlnTm1_const_Pgas/L_scale)
               end if
               if (k < nz) then
                  call ep1(s, xscale, equL, i_lnT, k, nvar, dLx_dlnTp1_const_Pgas/L_scale)
               end if
            else
               call e00(s, xscale, equL, i_lnT, k, nvar, dLx_dlnT00/L_scale)
               if (k > 1) call em1(s, xscale, equL, i_lnT, k, nvar, dLx_dlnTm1/L_scale)
               if (k < nz) call ep1(s, xscale, equL, i_lnT, k, nvar, dLx_dlnTp1/L_scale)
            end if

            if (s% do_struct_hydro) then
               if (s% lnPgas_flag) then
                  call e00(s, xscale, equL, i_lnPgas, k, nvar, dLx_dlnPgas00_const_T/L_scale)
               else
                  call e00(s, xscale, equL, i_xlnd, k, nvar, dLx_dlnd00/L_scale)
               end if
               
               call e00(s, xscale, equL, i_lnR, k, nvar, dLx_dlnR00/L_scale)
               if (i_vel /= 0) call e00(s, xscale, equL, i_vel, k, nvar, dLx_dv00/L_scale)
               if (k < nz) then
                  call ep1(s, xscale, equL, i_lnR, k, nvar, dLx_dlnRp1/L_scale)
                  if (i_vel /= 0) call ep1(s, xscale, equL, i_vel, k, nvar, dLx_dvp1/L_scale)
                  if (s% lnPgas_flag) then
                     call ep1(s, xscale, equL, i_lnPgas, k, nvar, dLx_dlnPgasp1_const_T/L_scale)
                  else
                     call ep1(s, xscale, equL, i_xlnd, k, nvar, dLx_dlndp1/L_scale)
                  end if
               end if
               if (k > 1) then
                  if (s% lnPgas_flag) then
                     call em1(s, xscale, equL, i_lnPgas, k, nvar, dLx_dlnPgasm1_const_T/L_scale)
                  else
                     call em1(s, xscale, equL, i_xlnd, k, nvar, dLx_dlndm1/L_scale)
                  end if
               end if
            end if
            
            if (do_chem .and. & 
                  (s% dxdt_nuc_factor > 0d0 .or. s% mix_factor > 0d0)) then
               do j=1,nvar_chem
                  dLx_dx = s% d_epsnuc_dx(j,k)*dm
                  call e00(s, xscale, equL, i_chem1+j-1, k, nvar, dLx_dx/L_scale)
               end do
            end if

            if (.not. do_Pvisc) return
         
            if (k < nz) then
               !del = s% v(k)*(3*s% Amid(k) - s% area(k)) - s% v(k+1)*(3*s% Amid(k) - s% area(k+1))
               dA_dlnR = 2*s% area(k)
               d_del_dlnR00 = &
                  s% v(k)*(3*s% dAmid_dlnR00(k) - dA_dlnR) - s% v(k+1)*3*s% dAmid_dlnR00(k)
               dA_dlnR = 2*s% area(k+1)
               d_del_dlnRp1 = &
                  s% v(k)*3*s% dAmid_dlnRp1(k) - s% v(k+1)*(3*s% dAmid_dlnRp1(k) - dA_dlnR)
               d_del_dvel00 = (3*s% Amid(k) - s% area(k))
               d_del_dvelp1 = -(3*s% Amid(k) - s% area(k+1))
            else
               !del = s% v(k)*(3*s% Amid(k) - s% area(k))
               dA_dlnR = 2*s% area(k)
               d_del_dlnR00 = s% v(k)*(3*s% dAmid_dlnR00(k) - dA_dlnR)
               d_del_dlnRp1 = 0
               d_del_dvel00 = (3*s% Amid(k) - s% area(k))
               d_del_dvelp1 = 0
            end if
         
            !dLAV = 0.5d0*s% Pvisc(k)*del
            d_dLAV_dlnd = 0.5d0*s% dPvisc_dlnd(k)*del
            d_dLAV_dlnT = 0.5d0*s% dPvisc_dlnT(k)*del
            d_dLAV_dlnR00 = 0.5d0*s% Pvisc(k)*d_del_dlnR00
            d_dLAV_dvel00 = 0.5d0*s% Pvisc(k)*d_del_dvel00
            if (k < nz) then
               d_dLAV_dlnRp1 = 0.5d0*s% Pvisc(k)*d_del_dlnRp1
               d_dLAV_dvelp1 = 0.5d0*s% Pvisc(k)*d_del_dvelp1
            else
               d_dLAV_dlnRp1 = 0
               d_dLAV_dvelp1 = 0
            end if
         
            !dL_expected = dL_expected + dLAV
            !equ(equL, k) = (dL_expected - dL_actual)/L_scale
            
            if (s% lnPgas_flag) then
               d_dLAV_dlnT_const_Pgas = &
                  d_dLAV_dlnT + d_dLAV_dlnd*s% dlnRho_dlnT_const_Pgas(k)
               call e00(s, xscale, equL, i_lnT, k, nvar, &
                           d_dLAV_dlnT_const_Pgas/L_scale)
            else
               call e00(s, xscale, equL, i_lnT, k, nvar, d_dLAV_dlnT/L_scale)
            end if

            if (s% do_struct_hydro) then
               if (s% lnPgas_flag) then   
                  d_dLAV_dlnPgas_const_T = d_dLAV_dlnd*s% dlnRho_dlnPgas_const_T(k)
                  call e00(s, xscale, equL, i_lnPgas, k, nvar, &
                           d_dLAV_dlnPgas_const_T/L_scale)
               else
                  call e00(s, xscale, equL, i_xlnd, k, nvar, d_dLAV_dlnd/L_scale)
               end if
               call e00(s, xscale, equL, i_lnR, k, nvar, d_dLAV_dlnR00/L_scale)
               call e00(s, xscale, equL, i_vel, k, nvar, d_dLAV_dvel00/L_scale)
               if (k < nz) then
                  call ep1(s, xscale, equL, i_lnR, k, nvar, d_dLAV_dlnRp1/L_scale)
                  call ep1(s, xscale, equL, i_vel, k, nvar, d_dLAV_dvelp1/L_scale)
               end if
            end if

         end subroutine do1_luminosity_eqn
         
               
         subroutine do1_pressure_eqn(k,ierr)
            ! compare lnP(k-1) to lnP(k)
            use chem_def, only: chem_isos
            use eos_def, only: i_lnPgas

            integer, intent(in) :: k
            integer, intent(out) :: ierr
         
            real(dp) :: &
               lnP_surf, delm, d_delm_dqm1, d_delm_dq00, P00, Pm1, &
               dPr, Ppoint, Ppoint_inv, &
               dPpoint_dlnd00, dPpoint_dlndm1, dPpoint_dlnT00, dPpoint_dlnTm1, &
               dPpoint_dlnPgas00_const_T, dPpoint_dlnPgasm1_const_T, &
               dPpoint_dlnT00_const_Pgas, dPpoint_dlnTm1_const_Pgas, &
               d_Ppoint_inv_dlndm1, d_Ppoint_inv_dlnd00, d_Ppoint_inv_dlnTm1, d_Ppoint_inv_dlnT00, &
               d_Ppoint_inv_dlnPgas00_const_T, d_Ppoint_inv_dlnPgasm1_const_T, &
               d_Ppoint_inv_dlnT00_const_Pgas, d_Ppoint_inv_dlnTm1_const_Pgas, &
               lnPdiff, d_lnPdiff_dlnP00, d_lnPdiff_dlnPm1, var_test, equ_test, &
               d_dlnPdm_dlnRp1, d_dlnPdm_dlnR00, d_dlnPdm_dlnRm1, &
               d_dlnPdm_dlnq, d_dlnPdm_dlndqm1, d_dlnPdm_dlndq00, &
               d_dlnPdm_dvelp1, d_dlnPdm_dvel00, d_dlnPdm_dvelm1, &
               d_dlnPdm_dlndp1, d_dlnPdm_dlnd00, d_dlnPdm_dlndm1, &
               d_dlnPdm_dlnTp1, d_dlnPdm_dlnT00, d_dlnPdm_dlnTm1, &
               d_dlnPdm_dlnPgas00_const_T, d_dlnPdm_dlnT00_const_Pgas, & 
               d_dlnPdm_dlnPgasm1_const_T, d_dlnPdm_dlnTm1_const_Pgas, & 
               d_delm_dlndq00, d_delm_dlndqm1, diff, diff_old, &
               dA_dlnR, delQ, dlnPQ, Ainv, &
               d_delQ_dlnd00, d_delQ_dlnT00, d_delQ_dlnR00, d_delQ_dvel00, &
               d_delQ_dlndm1, d_delQ_dlnTm1, d_delQ_dlnRm1, d_delQ_dvelm1, &
               d_delQ_dlnRp1, d_delQ_dvelp1, d_Ainv_dlnR00, &
               d_delQ_dlnPgas00_const_T, d_delQ_dlnPgasm1_const_T, &
               d_delQ_dlnT00_const_Pgas, d_delQ_dlnTm1_const_Pgas, &
               d_dlnPQ_dlnd00, d_dlnPQ_dlnT00, d_dlnPQ_dlnR00, d_dlnPQ_dvel00, &
               d_dlnPQ_dlndm1, d_dlnPQ_dlnTm1, d_dlnPQ_dlnRm1, d_dlnPQ_dvelm1, &
               d_dlnPQ_dlnRp1, d_dlnPQ_dvelp1, &
               r003, rp13, T1, lnT1, lnP1, &
               dlnT1_dL, dlnT1_dlnR, &
               dlnP1_dL, dlnP1_dlnR, &
               dlnP1_dlnm, dlnP1_dlnkap, &
               dlnkap_dlnd, dlnkap_dlnT, &
               dlnP1_dlnd, dlnP1_dlnT, &
               dlnP1_dlnR00, dlnP1_dlnRp1, d_dlnPdm_dL00, &
               dlnP00_dlnPgas_const_T, dlnP00_dlnT_const_Pgas, &
               dlnPm1_dlnPgas_const_T, dlnPm1_dlnT_const_Pgas, &
               d_lnPdiff_dlnTm1_const_Pgas, d_lnPdiff_dlnT00_const_Pgas, &
               d_lnPdiff_dlnPpgas00_const_T, d_lnPdiff_dlnPgasm1_const_T, &
               d_dlnPQ_dlnPgas00_const_T, d_dlnPQ_dlnPgasm1_const_T, &
               d_dlnPQ_dlnT00_const_Pgas, d_dlnPQ_dlnTm1_const_Pgas
            integer :: j   
            logical :: dbg 

            real(dp), parameter :: lnd0 = -1.7048902326588365D+01
            real(dp), parameter :: equ0 = -9.9870027208667683D-05
            real(dp), parameter :: dlnP_dm0 = -2.3395769706524674D-38

            include 'formats'
            ierr = 0

            dbg = .false.
         
            ! compare lnP(k-1) to lnP(k)
         
            call do_dlnP_dm(s, k, dt, &
               d_dlnPdm_dlnRp1, d_dlnPdm_dlnR00, d_dlnPdm_dlnRm1, &
               d_dlnPdm_dlnq, d_dlnPdm_dlndqm1, d_dlnPdm_dlndq00, &
               d_dlnPdm_dvelp1, d_dlnPdm_dvel00, d_dlnPdm_dvelm1, &
               d_dlnPdm_dlndp1, d_dlnPdm_dlnd00, d_dlnPdm_dlndm1, &
               d_dlnPdm_dlnTp1, d_dlnPdm_dlnT00, d_dlnPdm_dlnTm1, &         
               d_dlnPdm_dlnPgas00_const_T, d_dlnPdm_dlnT00_const_Pgas, & 
               d_dlnPdm_dlnPgasm1_const_T, d_dlnPdm_dlnTm1_const_Pgas, & 
               d_dlnPdm_dL00, &
               Ppoint, &
               dPpoint_dlnd00, dPpoint_dlndm1, dPpoint_dlnT00, dPpoint_dlnTm1, &
               dPpoint_dlnPgas00_const_T, dPpoint_dlnPgasm1_const_T, &
               dPpoint_dlnT00_const_Pgas, dPpoint_dlnTm1_const_Pgas, &
               ierr)
            if (ierr /= 0) return            
         
            if (k == 1) then
               delm = s% dm(k)/2
               Pm1 = 0
            else
               delm = (s% dm(k) + s% dm(k-1))/2
               Pm1 = s% P(k-1)
            end if
         
            P00 = s% P(k)
            dPr = Pm1 - P00
            Ppoint_inv = 1/Ppoint
      
            ! basic eqn is dP = -G m / (4 pi r^4)
            ! divide by <P> to make it unitless
            ! simple average is adequate for <P> since is only for normalizing the equation.
            ! however, be careful to use same <P> for both sides of equation..... 
         
            lnPdiff = dPr/Ppoint ! use this in place of lnP(k-1)-lnP(k)
         
            equ(equP, k) = delm*dlnP_dm(k) - lnPdiff
            
            if (k == -1 .or. dbg) then
               write(*,2) 'equ(equP, k)', k, equ(equP, k)
               write(*,2) 'delm', k, delm
               write(*,2) 'dlnP_dm(k)', k, dlnP_dm(k)
               write(*,2) 'delm*dlnP_dm(k)', k, delm*dlnP_dm(k)
               write(*,2) 'P00', k, P00
               write(*,2) 'Pm1', k-1, Pm1
               write(*,2) 'Ppoint', k, Ppoint
               write(*,2) 'lnPdiff', k, lnPdiff
               write(*,2) 's% T(k)', k, s% T(k)
               write(*,2) 's% rho(k)', k, s% rho(k)
               !write(*,2) '', k, 
               write(*,*)
            end if
         
            if (k == 1 .or. .not. do_Pvisc) then ! to prevent bogus warnings
               Ainv = 0
               delQ = 0
            else
               delQ = s% Pvisc(k-1)*(3*s% Amid(k-1) - s% area(k)) - &
                      s% Pvisc(k)*(3*s% Amid(k) - s% area(k))
               Ainv = 1/s% area(k)
               dlnPQ = -0.5d0*delQ*Ppoint_inv*Ainv
               equ(equP, k) = equ(equP, k) + dlnPQ
            end if
                     
            if (is_bad_num(equ(equP, k))) then
               ierr = -1
               if (s% report_ierr) write(*,*) 'P_eqn: is_bad_num(equ(equP, k))'
               return
            end if

            if (.false. .and. s% hydro_call_number == 17417 .and. k == 304 .and. &
                  skip_partials .and. abs(lnd0 - s% lnd(k)) > 1d-12*abs(lnd0)) then
               write(*,*)
               write(*,*) 'numeric'
               write(*,*) 'do_Pvisc', do_Pvisc
               write(*,2) 's% lnd(k)', k, s% lnd(k)
               write(*,2) 'vs', k, lnd0
               write(*,*)
               write(*,2) 'equ(equP, k)', k, equ(equP, k)
               write(*,2) 'vs', k, equ0
               write(*,*)
               write(*,2) 'dequP/dlnd', k, (equ(equP,k) - equ0)/(s% lnd(k) - lnd0)
               write(*,2) 'vs', k, 6.5608434652647929D-02
               write(*,*)
               write(*,2) 'delm*dlnP_dm(k)', k, delm*dlnP_dm(k)
               write(*,2) 'vs', k, delm*(-1.2431625967331853D-31)
               write(*,*)
               write(*,2) 'lnPdiff', k, lnPdiff
               write(*,2) 'vs', k, (-5.1427058225696771D-02)
               write(*,*)
               write(*,2) 'd_dlnPdm_dlnd00', k, (dlnP_dm(k) - (-1.2431625967331853D-31))/(s% lnd(k) - lnd0)
               write(*,2) 'vs', k, 4.0778941386363592D-33
               write(*,*)
               write(*,2) 'd_lnPdiff_dlnP00', k, (lnPdiff - (-5.1427058225696771D-02))/(s% lnd(k) - lnd0)
               write(*,2) 'vs', k, -9.9933881442088235D-01
               write(*,*)
               !stop
            end if
            
            if (.false. .and. s% hydro_call_number == 17417 .and. k == 304) then
               write(*,2) 'dlnP_dm,cgrav,m,r,Ppoint', k, dlnP_dm(k), s% cgrav(k), s% m(k), s% r(k), Ppoint
            end if
            
            if (skip_partials) return

            d_lnPdiff_dlnPm1 = P00*Pm1/Ppoint**2
            d_lnPdiff_dlnP00 = -d_lnPdiff_dlnPm1
         
            call e00(s, xscale, equP, i_lnR, k, nvar, delm*d_dlnPdm_dlnR00)
            call e00(s, xscale, equP, i_lum, k, nvar, delm*d_dlnPdm_dL00)
            
            if (s% lnPgas_flag) then
               dlnP00_dlnPgas_const_T = s% Pgas(k)/s% P(k)
               dlnP00_dlnT_const_Pgas = 4*s% Prad(k)/s% P(k)
               d_lnPdiff_dlnPpgas00_const_T = d_lnPdiff_dlnP00*dlnP00_dlnPgas_const_T
               d_lnPdiff_dlnT00_const_Pgas = d_lnPdiff_dlnP00*dlnP00_dlnT_const_Pgas
               call e00(s, xscale, equP, i_lnPgas, k, nvar, &
                     delm*d_dlnPdm_dlnPgas00_const_T - d_lnPdiff_dlnPpgas00_const_T)
               if (s% do_struct_thermo) &
                  call e00(s, xscale, equP, i_lnT, k, nvar, &
                        delm*d_dlnPdm_dlnT00_const_Pgas - d_lnPdiff_dlnT00_const_Pgas)
            else
               call e00(s, xscale, equP, i_xlnd, k, nvar, &
                     delm*d_dlnPdm_dlnd00 - d_lnPdiff_dlnP00*s% chiRho(k))
               if (s% do_struct_thermo) &
                  call e00(s, xscale, equP, i_lnT, k, nvar, &
                        delm*d_dlnPdm_dlnT00 - d_lnPdiff_dlnP00*s% chiT(k))
            end if
            
            if (k > 1) then
               call em1(s, xscale, equP, i_lnR, k, nvar, delm*d_dlnPdm_dlnRm1)
               if (s% lnPgas_flag) then
                  dlnPm1_dlnPgas_const_T = s% Pgas(k-1)/s% P(k-1)
                  dlnPm1_dlnT_const_Pgas = 4*s% Prad(k-1)/s% P(k-1)
                  d_lnPdiff_dlnPgasm1_const_T = d_lnPdiff_dlnPm1*dlnPm1_dlnPgas_const_T
                  d_lnPdiff_dlnTm1_const_Pgas = d_lnPdiff_dlnPm1*dlnPm1_dlnT_const_Pgas
                  call em1(s, xscale, equP, i_lnPgas, k, nvar, &
                     delm*d_dlnPdm_dlnPgasm1_const_T - d_lnPdiff_dlnPgasm1_const_T)
                  if (s% do_struct_thermo) &
                     call em1(s, xscale, equP, i_lnT, k, nvar, &
                        delm*d_dlnPdm_dlnTm1_const_Pgas - d_lnPdiff_dlnTm1_const_Pgas)
               else
                  call em1(s, xscale, equP, i_xlnd, k, nvar, &
                     delm*d_dlnPdm_dlndm1 - d_lnPdiff_dlnPm1*s% chiRho(k-1))
                  if (s% do_struct_thermo) &
                     call em1(s, xscale, equP, i_lnT, k, nvar, &
                        delm*d_dlnPdm_dlnTm1 - d_lnPdiff_dlnPm1*s% chiT(k-1))
               end if
            end if

            if (s% v_flag) then
               call e00(s, xscale, equP, i_vel, k, nvar, delm*d_dlnPdm_dvel00)
               if (k > 1) call em1(s, xscale, equP, i_vel, k, nvar, delm*d_dlnPdm_dvelm1)
            end if
         
            if (k == 1 .or. .not. do_Pvisc) return
            
            !delQ = s% Pvisc(k-1)*(3*s% Amid(k-1) - s% area(k))
            !     - s% Pvisc(k)*(3*s% Amid(k) - s% area(k))
         
            d_delQ_dlnd00 = -s% dPvisc_dlnd(k)*(3*s% Amid(k) - s% area(k))
            d_delQ_dlnT00 = -s% dPvisc_dlnT(k)*(3*s% Amid(k) - s% area(k))
            
            dA_dlnR = 2*s% area(k)
            d_delQ_dlnR00 = &
               s% dPvisc_dlnRp1(k-1)*(3*s% Amid(k-1) - s% area(k)) &
               + s% Pvisc(k-1)*(3*s% dAmid_dlnRp1(k-1) - dA_dlnR) &
               - s% dPvisc_dlnR00(k)*(3*s% Amid(k) - s% area(k)) &
               - s% Pvisc(k)*(3*s% dAmid_dlnR00(k) - dA_dlnR)
            d_delQ_dvel00 = &
               s% dPvisc_dvelp1(k-1)*(3*s% Amid(k-1) - s% area(k)) &
               - s% dPvisc_dvel00(k)*(3*s% Amid(k) - s% area(k))
         
            d_delQ_dlndm1 = s% dPvisc_dlnd(k-1)*(3*s% Amid(k-1) - s% area(k))
            d_delQ_dlnTm1 = s% dPvisc_dlnT(k-1)*(3*s% Amid(k-1) - s% area(k))
            d_delQ_dlnRm1 = &
               s% dPvisc_dlnR00(k-1)*(3*s% Amid(k-1) - s% area(k)) &
               + s% Pvisc(k-1)*3*s% dAmid_dlnR00(k-1)
            d_delQ_dvelm1 = s% dPvisc_dvel00(k-1)*(3*s% Amid(k-1) - s% area(k))
         
            if (k < nz) then
               d_delQ_dlnRp1 = -s% dPvisc_dlnRp1(k)*(3*s% Amid(k) - s% area(k))
               d_delQ_dvelp1 = -s% dPvisc_dvelp1(k)*(3*s% Amid(k) - s% area(k))
            else
               d_delq_dlnRp1 = 0
               d_delQ_dvelp1 = 0
            end if               
         
            !Ainv = 1/s% area(k)
            dA_dlnR = 2*s% area(k)
            d_Ainv_dlnR00 = -dA_dlnR*Ainv**2
         
            !dlnPQ = -0.5d0*delQ*Ppoint_inv*Ainv
         
            d_dlnPQ_dlnR00 = &
               -0.5d0*(d_delQ_dlnR00*Ainv + delQ*d_Ainv_dlnR00)*Ppoint_inv
            d_dlnPQ_dvel00 = -0.5d0*d_delQ_dvel00*Ppoint_inv*Ainv
         
            d_dlnPQ_dlnRm1 = -0.5d0*d_delQ_dlnRm1*Ppoint_inv*Ainv
            d_dlnPQ_dvelm1 = -0.5d0*d_delQ_dvelm1*Ppoint_inv*Ainv
         
            !equ(equP, k) = equ(equP, k) + dlnPQ

            call e00(s, xscale, equP, i_lnR, k, nvar, d_dlnPQ_dlnR00)
            call em1(s, xscale, equP, i_lnR, k, nvar, d_dlnPQ_dlnRm1)

            call e00(s, xscale, equP, i_vel, k, nvar, d_dlnPQ_dvel00)         
            call em1(s, xscale, equP, i_vel, k, nvar, d_dlnPQ_dvelm1)
         
            if (s% lnPgas_flag) then               
               d_delQ_dlnPgas00_const_T = d_delQ_dlnd00*s% dlnRho_dlnPgas_const_T(k)
               d_delQ_dlnT00_const_Pgas = &
                  d_delQ_dlnT00 + d_delQ_dlnd00*s% dlnRho_dlnT_const_Pgas(k)
               d_delQ_dlnPgasm1_const_T = d_delQ_dlndm1*s% dlnRho_dlnPgas_const_T(k-1)             
               d_delQ_dlnTm1_const_Pgas = &
                  d_delQ_dlnTm1 + d_delQ_dlndm1*s% dlnRho_dlnT_const_Pgas(k-1)
               d_Ppoint_inv_dlnPgas00_const_T = -dPpoint_dlnPgas00_const_T*Ppoint_inv**2
               d_Ppoint_inv_dlnPgasm1_const_T = -dPpoint_dlnPgasm1_const_T*Ppoint_inv**2
               d_Ppoint_inv_dlnT00_const_Pgas = -dPpoint_dlnT00_const_Pgas*Ppoint_inv**2
               d_Ppoint_inv_dlnTm1_const_Pgas = -dPpoint_dlnTm1_const_Pgas*Ppoint_inv**2
               d_dlnPQ_dlnPgas00_const_T = &
                  -0.5d0*(d_delQ_dlnPgas00_const_T*Ppoint_inv + delQ*d_Ppoint_inv_dlnPgas00_const_T)*Ainv
               call e00(s, xscale, equP, i_lnPgas, k, nvar, d_dlnPQ_dlnPgas00_const_T)
               d_dlnPQ_dlnPgasm1_const_T = &
                  -0.5d0*(d_delQ_dlnPgasm1_const_T*Ppoint_inv + delQ*d_Ppoint_inv_dlnPgasm1_const_T)*Ainv
               call em1(s, xscale, equP, i_lnPgas, k, nvar, d_dlnPQ_dlnPgasm1_const_T)
               if (s% do_struct_thermo) then
                  d_dlnPQ_dlnT00_const_Pgas = &
                     -0.5d0*(d_delQ_dlnT00_const_Pgas*Ppoint_inv + delQ*d_Ppoint_inv_dlnT00_const_Pgas)*Ainv
                  call e00(s, xscale, equP, i_lnT, k, nvar, d_dlnPQ_dlnT00_const_Pgas)
                  d_dlnPQ_dlnTm1_const_Pgas = &
                     -0.5d0*(d_delQ_dlnTm1_const_Pgas*Ppoint_inv + delQ*d_Ppoint_inv_dlnTm1_const_Pgas)*Ainv
                  call em1(s, xscale, equP, i_lnT, k, nvar, d_dlnPQ_dlnTm1_const_Pgas)
               end if
            else
               d_Ppoint_inv_dlnd00 = -dPpoint_dlnd00*Ppoint_inv**2
               d_Ppoint_inv_dlndm1 = -dPpoint_dlndm1*Ppoint_inv**2
               d_Ppoint_inv_dlnT00 = -dPpoint_dlnT00*Ppoint_inv**2
               d_Ppoint_inv_dlnTm1 = -dPpoint_dlnTm1*Ppoint_inv**2
               d_dlnPQ_dlnd00 = &
                  -0.5d0*(d_delQ_dlnd00*Ppoint_inv + delQ*d_Ppoint_inv_dlnd00)*Ainv
               call e00(s, xscale, equP, i_xlnd, k, nvar, d_dlnPQ_dlnd00)
               d_dlnPQ_dlndm1 = &
                  -0.5d0*(d_delQ_dlndm1*Ppoint_inv + delQ*d_Ppoint_inv_dlndm1)*Ainv
               call em1(s, xscale, equP, i_xlnd, k, nvar, d_dlnPQ_dlndm1)
               if (s% do_struct_thermo) then
                  d_dlnPQ_dlnT00 = &
                     -0.5d0*(d_delQ_dlnT00*Ppoint_inv + delQ*d_Ppoint_inv_dlnT00)*Ainv
                  call e00(s, xscale, equP, i_lnT, k, nvar, d_dlnPQ_dlnT00)
                  d_dlnPQ_dlnTm1 = &
                     -0.5d0*(d_delQ_dlnTm1*Ppoint_inv + delQ*d_Ppoint_inv_dlnTm1)*Ainv
                  call em1(s, xscale, equP, i_lnT, k, nvar, d_dlnPQ_dlnTm1)
               end if
            end if
         
            if (k < nz) then
               d_dlnPQ_dlnRp1 = -0.5d0*d_delQ_dlnRp1*Ppoint_inv*Ainv
               d_dlnPQ_dvelp1 = -0.5d0*d_delQ_dvelp1*Ppoint_inv*Ainv
               call ep1(s, xscale, equP, i_lnR, k, nvar, d_dlnPQ_dlnRp1)
               call ep1(s, xscale, equP, i_vel, k, nvar, d_dlnPQ_dvelp1)
            end if

         end subroutine do1_pressure_eqn
         
               
         subroutine do1_temperature_eqn(k,ierr)
            integer, intent(in) :: k
            integer, intent(out) :: ierr
            
            ! if not in quasi-hydrostatic-equilibrium, this is not correct.
            call do1_quasi_HSE_T_eqn(k,ierr)
         
         end subroutine do1_temperature_eqn
         
               
         subroutine do1_quasi_HSE_T_eqn(k,ierr)

            integer, intent(in) :: k
            integer, intent(out) :: ierr
            
            real(dp) :: alfa, beta, r, dr_dL00, dr_dlnR00, dr_dlnd00, dr_dlnT00, &
               dr_dlndm1, dr_dlnTm1, dr_lnq00, dr_lndq00, dr_lndqm1, m, dlnPdm, &
               d_dlnPdm_dlnR, d_dlnPdm_dlnq, d_dlnPdm_dL, d_dlnPdm_dlnd00, d_dlnPdm_dlnT00, &
               d_dlnPdm_dlndm1, d_dlnPdm_dlnTm1, &
               d_dlnPdm_dlnPgas00_const_T, d_dlnPdm_dlnT00_const_Pgas, &
               d_dlnPdm_dlnPgasm1_const_T, d_dlnPdm_dlnTm1_const_Pgas, &
               dP_dlnPgas00_const_T, dP_dlnPgasm1_const_T, &
               dP_dlnT00_const_Pgas, dP_dlnTm1_const_Pgas, &
               grad_star, d_grad_star_dL, d_grad_star_dlnR, &       
               d_grad_star_dlnd00, d_grad_star_dlndm1, &    
               d_grad_star_dlnT00, d_grad_star_dlnTm1, &
               Ppoint, &
               dPpoint_dlnd00, dPpoint_dlndm1, dPpoint_dlnT00, dPpoint_dlnTm1, &
               dPpoint_dlnPgas00_const_T, dPpoint_dlnPgasm1_const_T, &
               dPpoint_dlnT00_const_Pgas, dPpoint_dlnTm1_const_Pgas, &
               d_dlnTdm_dLum, d_dlnTdm_dlnR, d_dlnTdm_dlnd00, &
               d_dlnTdm_dlnT00, d_dlnTdm_dlndm1, d_dlnTdm_dlnTm1, &
               delm, d_delm_dqm1, d_delm_dq00, T00, Tm1, dT, Tpoint, &
               lnTdiff, d_lnTdiff_dlnT00, d_lnTdiff_dlnTm1, d_dlnTdm_dlnq, &
               d_delm_dlndq00, d_delm_dlndqm1, diff, diff_old, &
               d_grad_star_dlnT00_const_Pgas, d_grad_star_dlnTm1_const_Pgas, &
               d_grad_star_dlnPgas00_const_T, d_grad_star_dlnPgasm1_const_T, &
               d_dlnTdm_dlnPgas00_const_T, d_dlnTdm_dlnPgasm1_const_T, &
               d_dlnTdm_dlnT00_const_Pgas, d_dlnTdm_dlnTm1_const_Pgas, &
               dr_dlnPgas00_const_T, dr_dlnPgasm1_const_T, &
               dr_dlnT00_const_Pgas, dr_dlnTm1_const_Pgas, &
               other, other_equ, FL, other_FL

            
            include 'formats'
            ierr = 0
            ! compare lnT(k-1) to lnT(k)

            call eval_dlnPdm_qhse(s, k, m, &
               dlnPdm, d_dlnPdm_dlnR, d_dlnPdm_dlnq, d_dlnPdm_dL, &
               d_dlnPdm_dlnd00, d_dlnPdm_dlnT00, &
               d_dlnPdm_dlndm1, d_dlnPdm_dlnTm1, &
               d_dlnPdm_dlnPgas00_const_T, d_dlnPdm_dlnT00_const_Pgas, &
               d_dlnPdm_dlnPgasm1_const_T, d_dlnPdm_dlnTm1_const_Pgas, &
               Ppoint, &
               dPpoint_dlnd00, dPpoint_dlndm1, dPpoint_dlnT00, dPpoint_dlnTm1, &
               dPpoint_dlnPgas00_const_T, dPpoint_dlnPgasm1_const_T, &
               dPpoint_dlnT00_const_Pgas, dPpoint_dlnTm1_const_Pgas, &
               ierr)
            if (ierr /= 0) return
         
            call eval_grad_star_info( &
               s, k, grad_star, d_grad_star_dL, d_grad_star_dlnR, &
               d_grad_star_dlnd00, d_grad_star_dlndm1, &
               d_grad_star_dlnT00, d_grad_star_dlnTm1, &
               d_grad_star_dlnT00_const_Pgas, d_grad_star_dlnTm1_const_Pgas, &
               d_grad_star_dlnPgas00_const_T, d_grad_star_dlnPgasm1_const_T, &
               ierr)
            if (ierr /= 0) return
                     
            s% dlnT_dm(k) = dlnPdm*grad_star
         
            delm = (s% dm(k) + s% dm(k-1))/2
            Tm1 = s% T(k-1)
            alfa = s% dm(k-1)/(s% dm(k-1) + s% dm(k))
            beta = 1 - alfa

            T00 = s% T(k)
            dT = Tm1 - T00
            Tpoint = alfa*T00 + beta*Tm1
            lnTdiff = dT/Tpoint ! use this in place of lnT(k-1)-lnT(k)
         
            r = delm*dlnT_dm(k) - lnTdiff
            equ(equT, k) = r
            
            if (is_bad_num(equ(equT, k))) then
               ierr = -1
               if (s% report_ierr) write(*,2) 'equ(equT, k)', k, equ(equT, k)
               return
               write(*,2) 'equ(equT, k)', k, equ(equT, k)
               write(*,2) 'lnTdiff', k, lnTdiff
               write(*,2) 'delm', k, delm
               write(*,2) 'dlnT_dm(k)', k, dlnT_dm(k)
               write(*,2) 'dlnPdm', k, dlnPdm
               write(*,2) 'grad_star', k, grad_star
               stop 'equT'
            end if
            
            other = 8.3276707878306060D+40
            other_equ = 1.4606961473706548D-12
            other_FL = 9.4222987519533120D+01
         
            if (skip_partials) return
         
            d_lnTdiff_dlnTm1 = T00*Tm1/Tpoint**2
            d_lnTdiff_dlnT00 = -d_lnTdiff_dlnTm1

            d_dlnTdm_dLum = dlnPdm*d_grad_star_dL + d_dlnPdm_dL*grad_star
            dr_dL00 = delm*d_dlnTdm_dLum

            d_dlnTdm_dlnR = dlnPdm*d_grad_star_dlnR + d_dlnPdm_dlnR*grad_star
            dr_dlnR00 = delm*d_dlnTdm_dlnR
      
            call e00(s, xscale, equT, i_lum, k, nvar, dr_dL00)
            
            if (s% lnPgas_flag) then
               d_dlnTdm_dlnT00_const_Pgas = &
                  dlnPdm*d_grad_star_dlnT00_const_Pgas + d_dlnPdm_dlnT00_const_Pgas*grad_star
               dr_dlnT00_const_Pgas = delm*d_dlnTdm_dlnT00_const_Pgas - d_lnTdiff_dlnT00
               call e00(s, xscale, equT, i_lnT, k, nvar, dr_dlnT00_const_Pgas)
               d_dlnTdm_dlnTm1_const_Pgas =  &
                  dlnPdm*d_grad_star_dlnTm1_const_Pgas + d_dlnPdm_dlnTm1_const_Pgas*grad_star
               dr_dlnTm1_const_Pgas = delm*d_dlnTdm_dlnTm1_const_Pgas - d_lnTdiff_dlnTm1
               call em1(s, xscale, equT, i_lnT, k, nvar, dr_dlnTm1_const_Pgas)
            else
               d_dlnTdm_dlnT00 = dlnPdm*d_grad_star_dlnT00 + d_dlnPdm_dlnT00*grad_star
               dr_dlnT00 = delm*d_dlnTdm_dlnT00 - d_lnTdiff_dlnT00
               call e00(s, xscale, equT, i_lnT, k, nvar, dr_dlnT00)
               d_dlnTdm_dlnTm1 = dlnPdm*d_grad_star_dlnTm1 + d_dlnPdm_dlnTm1*grad_star
               dr_dlnTm1 = delm*d_dlnTdm_dlnTm1 - d_lnTdiff_dlnTm1
               call em1(s, xscale, equT, i_lnT, k, nvar, dr_dlnTm1)
            end if

            if (s% do_struct_hydro) then
               call e00(s, xscale, equT, i_lnR, k, nvar, dr_dlnR00)
               if (s% lnPgas_flag) then
                  d_dlnTdm_dlnPgas00_const_T = &
                     dlnPdm*d_grad_star_dlnPgas00_const_T + d_dlnPdm_dlnPgas00_const_T*grad_star
                  dr_dlnPgas00_const_T = delm*d_dlnTdm_dlnPgas00_const_T
                  call e00(s, xscale, equT, i_lnPgas, k, nvar, dr_dlnPgas00_const_T)
                  d_dlnTdm_dlnPgasm1_const_T = &
                     dlnPdm*d_grad_star_dlnPgasm1_const_T + d_dlnPdm_dlnPgasm1_const_T*grad_star
                  dr_dlnPgasm1_const_T = delm*d_dlnTdm_dlnPgasm1_const_T
                  call em1(s, xscale, equT, i_lnPgas, k, nvar, dr_dlnPgasm1_const_T)
               else
                  d_dlnTdm_dlnd00 = dlnPdm*d_grad_star_dlnd00 + d_dlnPdm_dlnd00*grad_star
                  dr_dlnd00 = delm*d_dlnTdm_dlnd00
                  call e00(s, xscale, equT, i_xlnd, k, nvar, dr_dlnd00)
                  d_dlnTdm_dlndm1 = dlnPdm*d_grad_star_dlndm1 + d_dlnPdm_dlndm1*grad_star
                  dr_dlndm1 = delm*d_dlnTdm_dlndm1
                  call em1(s, xscale, equT, i_xlnd, k, nvar, dr_dlndm1)
               end if
            end if
            
            if (.false. .and. k==1762) then
               write(*,*)
               write(*,*)
               write(*,2) 'equ(equT,k)', k, equ(equT,k)
               write(*,2) 'delm', k, delm
               write(*,2) 'dlnT_dm(k)', k, dlnT_dm(k)
               write(*,2) 'lnTdiff', k, lnTdiff
               write(*,*)
               write(*,2) 'd_dlnTdm_dlnT00', k, d_dlnTdm_dlnT00
               write(*,2) 'dlnPdm', k, dlnPdm
               write(*,2) 'd_grad_star_dlnT00', k, d_grad_star_dlnT00
               write(*,2) 'd_dlnPdm_dlnT00', k, d_dlnPdm_dlnT00
               write(*,2) 'grad_star', k, grad_star
               write(*,*) 's% lnPgas_flag', s% lnPgas_flag
               
               write(*,2) 'dr_dlnT00', k, dr_dlnT00
               write(*,2) 'delm*d_dlnTdm_dlnT00', k, delm*d_dlnTdm_dlnT00
               write(*,2) 'd_lnTdiff_dlnT00', k, d_lnTdiff_dlnT00
               write(*,2) 'd_dlnTdm_dlnT00', k, d_dlnTdm_dlnT00
               write(*,*)
               
            end if

         end subroutine do1_quasi_HSE_T_eqn

         
         subroutine do1_velocity_eqn(k,ierr)
            use hydro_vars, only: get_dVARDOT_dVAR_pt
            integer, intent(in) :: k
            integer, intent(out) :: ierr
            real(dp) :: d_dlnRdt_dlnRm1, d_dlnRdt_dlnR00, d_dlnRdt_dlnRp1
            real(dp) :: r, dlnR_dt, cs
            ierr = 0
            cs = s% csound_init(k)
            if (s% xh_old(s% i_lnT,k)/ln10 < s% velocity_logT_lower_bound .and. &
                s% dt/secyer < s% max_dt_yrs_for_velocity_logT_lower_bound) then ! no velocities
               equ(equv, k) = s% v(k)/cs
               if (skip_partials) return
               call e00(s, xscale, equv, i_vel, k, nvar, one/cs)     
               return
            end if
            dlnR_dt = s% dlnR_dt(k)
            call get_dVARDOT_dVAR_pt(s, k, dt, d_dlnRdt_dlnRm1, d_dlnRdt_dlnR00, d_dlnRdt_dlnRp1)
            r = s% r(k)
            equ(equv, k) = (dlnR_dt*r - s% v(k))/cs
            if (skip_partials) return
            call e00(s, xscale, equv, i_vel, k, nvar, -one/cs)     
            call e00(s, xscale, equv, i_lnR, k, nvar, r*(dlnR_dt + d_dlnRdt_dlnR00)/cs)
            if (k > 1 .and. d_dlnRdt_dlnRm1 /= 0) &
               call em1(s, xscale, equv, i_lnR, k, nvar, r*d_dlnRdt_dlnRm1/cs)
            if (k < s% nz .and. d_dlnRdt_dlnRp1 /= 0) &
               call ep1(s, xscale, equv, i_lnR, k, nvar, r*d_dlnRdt_dlnRp1/cs)
         end subroutine do1_velocity_eqn
         
         
         subroutine dummy_eqn(j,i,nzlo,nzhi)
            integer, intent(in) :: j, i, nzlo, nzhi
            integer :: k
            do k=nzlo,nzhi
               equ(j,k) = 0 !s% xs(i,k) - s% xs_pre_pass(i,k)
               if (.not. skip_partials) call e00(s, xscale,j,i,k,nvar,one)
            end do
         end subroutine dummy_eqn


         subroutine check(j_equ, k)
            integer, intent(in) :: j_equ, k
            integer :: j
            logical, parameter :: dump_it = .false.
            if (abs(equ(j_equ, k)) > max_equ(j_equ)) then
               max_equ(j_equ) = abs(equ(j_equ, k))
               kmax_equ(j_equ) = k
            end if
            
            8 format(a30, 2i6, 1pe26.16, 3x, a)
            9 format(a30, 3i6, 1pe26.16, 3x, a)
                        
            if (dump_it .or. is_bad_num(equ(j_equ, k))) then
               write(*,8) 'equ(j_equ, k)', j_equ, k, equ(j_equ, k), trim(s% nameofequ(j_equ))
               write(*,*) 'skip_partials', skip_partials
            end if
         end subroutine check
         
         
         subroutine dump_equL_info
            integer :: k, j, k0, k1
            include 'formats'
            write(*,*) 'dump_equL_info'
            do k=1,s% nz
               do j=1,nvar
                  write(*,2) 's% lnE(k)', k, s% lnE(k)
                  write(*,2) 's% P(k)', k, s% P(k)
                  write(*,2) 's% rho(k)', k, s% rho(k)
                  write(*,2) 's% T(k)', k, s% T(k)
                  write(*,2) 's% Cv(k)', k, s% Cv(k)
                  write(*,2) 's% dE_dRho(k)', k, s% dE_dRho(k)
                  write(*,2) 's% chiT(k)', k, s% chiT(k)
                  write(*,2) 's% chiRho(k)', k, s% chiRho(k)
                  write(*,2) 's% d_eps_grav_dlnT00(k)', k, s% d_eps_grav_dlnT00(k)
                  write(*,2) 's% d_eps_grav_dlnd00(k)', k, s% d_eps_grav_dlnd00(k)
                  write(*,2) 's% eps_grav(k)', k, s% eps_grav(k)
               end do
               write(*,*)
            end do
            stop 'dump_equL_info'      
         end subroutine dump_equL_info


         subroutine dump_equ
            integer :: k, j, k0, k1
            include 'formats'
            do k=1,s% nz
               do j=1,nvar
                  write(*,2) 'equ ' // trim(s% nameofequ(j)), k, equ(j, k)
               end do
               write(*,*)
            end do
         end subroutine dump_equ


         subroutine dump_equ_and_partials
            integer :: k, j, i
            include 'formats'
            do k=1,s% nz
               do j=1,nvar
                  write(*,2) 'equ ' // trim(s% nameofequ(j)), k, equ(j, k)
               end do
               write(*,*)
            end do
            !stop 'dump_equ'      
         end subroutine dump_equ_and_partials


         subroutine dump_some_equ
            integer :: k, j, i
            include 'formats'
            do k=536, 538
               do j=1,nvar
                  write(*,2) 'equ ' // trim(s% nameofequ(j)), k, equ(j, k)
               end do
               write(*,*)
            end do
            !stop 'dump_some_equ'      
         end subroutine dump_some_equ


         subroutine check_everything
            use star_utils, only: std_write_internals_to_file
            integer :: k, j, k0, k1
            include 'formats'
            write(*,*) 'check everything', nvar, nz
            max_equ(:) = 0; kmax_equ(:) = 0
            do k=nzlo,nzhi
               do j=1, nvar
                  call check(j, k)
               end do
            end do
            write(*,*)
            do j=1,nvar
               write(*,3) trim(s% nameofequ(j)), j, kmax_equ(j), max_equ(j)
            end do
            write(*,*)
            ! set k0 and k1 to locations where want more info
            k0 = 1; k1 = 0
            if (k0 <= k1) then
               do k=k0,k1
                  do j=1,nvar
                     write(*,2) 'equ ' // trim(s% nameofequ(j)), k, equ(j, k)
                  end do
                  write(*,*)
               end do
               write(*,*)
               write(*,*)
               do k=k0,k1
                  write(*,2) '1-q(k)', k, 1-s% q(k)
               end do
               write(*,*)
               write(*,*)
               do k=k0,k1
                  write(*,2) 'lnd(k)/ln10', k, s% lnd(k)/ln10
               end do
               write(*,*)
               write(*,*)
               do k=k0,k1
                  write(*,2) 'lnT(k)/ln10', k, s% lnT(k)/ln10
               end do
               write(*,*)
               write(*,*)
               do k=k0,k1
                  do j=1,nvar_chem
                     write(*,2) 'xa_old(j,k) ' // trim(chem_isos% name(s% chem_id(j))), &
                        j, s% xa_old(j,k)
                  end do
               end do
               write(*,*)
               write(*,*)
               do k=k0,k1
                  do j=1,nvar_chem
                     write(*,2) 'xa(j,k) ' // trim(chem_isos% name(s% chem_id(j))), j, s% xa(j,k)
                  end do
               end do
            end if
            write(*,*) 'nz', nz
            write(*,*)
            write(*,*) 'finished check everything'
            !stop 'check_everything'
            write(*,*)
            write(*,*)
            write(*,*)
            call std_write_internals_to_file(s% id, 0)
            
            if (skip_partials) return
            
         end subroutine check_everything
         

      end subroutine eval_equ_for_solver


      subroutine PT_eqns_surf(s, xscale, equ, skip_partials, nvar, ierr)
         use hydro_vars, only: set_Teff_info_for_eqns
         use chem_def
         use atm_def
         use atm_lib, only: atm_option
         
         type (star_info), pointer :: s         
         real(dp), pointer :: xscale(:,:)
         real(dp), pointer :: equ(:,:)
         logical, intent(in) :: skip_partials
         integer, intent(in) :: nvar
         integer, intent(out) :: ierr
         
         integer :: i_xlnd, i_lnPgas, i_lnT, i_lnR, i_lum, equP, equT
         real(dp) :: r, L, Teff, &
            lnT_surf, dlnTsurf_dL, dlnTsurf_dlnR, dlnTsurf_dlnM, dlnTsurf_dlnkap, &
            lnP_surf, dlnPsurf_dL, dlnPsurf_dlnR, dlnPsurf_dlnM, dlnPsurf_dlnkap
         real(dp) :: &
            dlnT_bc_dlnd, dlnT_bc_dlnT, dlnT_bc_dlnR, &
            dlnT_bc_dL, dlnP_bc_dlnd, dlnP_bc_dlnT, dlnP_bc_dlnR, dlnP_bc_dL, &
            dlnkap_dlnd, dlnkap_dlnT, dPinv_dlnd, dPinv_dlnT, dP0, dT0, &
            P_surf, T_surf, dlnP_bc_dlnPsurf, dlnT_bc_dlnTsurf, P_bc, T_bc, lnT_bc, lnP_bc, &
            dP0_dlnR, dT0_dlnR, dT0_dlnT, dT0_dlnd, dT0_dL, dlnP_bc_dP0, dlnT_bc_dT0, &
            dlnP_bc_dlnPgas_const_T, dlnP_bc_dlnT_const_Pgas, dlnP_dlnPgas_const_T, &
            dlnP_dlnT_const_Pgas, dlnT_bc_dlnPgas_const_T, dlnT_bc_dlnT_const_Pgas
         
         include 'formats'
         
         ierr = 0
         
         if (s% L(1) <= 0) then
            ierr = -1
            if (s% report_ierr) write(*, *) 'P_eqn_surf L <= 0', ierr
            return
         end if
         
         call set_Teff_info_for_eqns(s, skip_partials, r, L, Teff, &
            lnT_surf, dlnTsurf_dL, dlnTsurf_dlnR, dlnTsurf_dlnM, dlnTsurf_dlnkap, &
            lnP_surf, dlnPsurf_dL, dlnPsurf_dlnR, dlnPsurf_dlnM, dlnPsurf_dlnkap, &
            ierr)
         if (ierr /= 0) then
            if (s% report_ierr) then
               write(*,*) 'P_eqn_surf: ierr from set_Teff_info_for_eqns'
            end if
            return
         end if

         equP = s% equP
         equT = s% equT

         i_lnPgas = s% i_lnPgas
         i_xlnd = s% i_xlnd
         i_lnT = s% i_lnT
         i_lnR = s% i_lnR
         i_lum = s% i_lum
         
         ! P_surf and T_surf are at outer boundary of cell 1
         P_surf = exp(lnP_surf)
         T_surf = exp(lnT_surf)
         s% P_surf = P_surf
         s% T_surf = T_surf

         ! offset P and T from outer edge of cell 1 to center of cell 1
         dP0 = s% surf_bc_offset_factor*s% cgrav(1)*s% mstar*s% dm(1)/(8*pi*r**4)
         dT0 = dP0*s% gradT(1)*s% T(1)/s% P(1)
         
         P_bc = P_surf + dP0
         T_bc = T_surf + dT0

         lnP_bc = log(P_bc)
         lnT_bc = log(T_bc)
         
         if (s% do_struct_hydro) then
            equ(equP, 1) = lnP_bc - s% lnP(1)
            if (is_bad_num(equ(equP, 1))) then
               write(*,1) 'equ(equP, 1)', equ(equP, 1)
               write(*,1) 's% lnP(1)', s% lnP(1)
               write(*,1) 'lnP_bc', lnP_bc
               write(*,1) 'lnP_surf', lnP_surf
               write(*,1) 'P_surf', P_surf
               write(*,1) 'dP0', dP0
               write(*,1) 'r', r
               write(*,1) 's% dm(1)', s% dm(1)
               write(*,1) 's% mstar', s% mstar
               write(*,1) 's% surf_bc_offset_factor', s% surf_bc_offset_factor
               stop 'PT_eqns_surf'
            end if
         else
            equ(equP,1) = 0
         end if
         
         if (s% do_struct_thermo) then
            equ(equT, 1) = lnT_bc - s% lnT(1)        
         else
            equ(equT,1) = 0
         end if

         if (skip_partials) return

         !dT0 = dP0*s% gradT(1)*s% T(1)/s% P(1)
         
         dP0_dlnR = -4*dP0         
         dT0_dlnR = -4*dT0 + dP0*s% d_gradT_dlnR(1)*s% T(1)/s% P(1)
         
         dPinv_dlnT = -s% chiT(1)/s% P(1)
         dT0_dlnT = &
            dT0 + &
            dP0*s% d_gradT_dlnT00(1)*s% T(1)/s% P(1) + &
            dP0*s% gradT(1)*s% T(1)*dPinv_dlnT
         dPinv_dlnd = -s% chiRho(1)/s% P(1)
         dT0_dlnd = &
            dP0*s% d_gradT_dlnd00(1)*s% T(1)/s% P(1) + &
            dP0*s% gradT(1)*s% T(1)*dPinv_dlnd
            
         dT0_dL = dP0*s% d_gradT_dL(1)*s% T(1)/s% P(1)
         
         dlnP_bc_dP0 = 1/P_bc
         dlnT_bc_dT0 = 1/T_bc
         
         dlnP_bc_dlnPsurf = P_surf/P_bc
         dlnT_bc_dlnTsurf = T_surf/T_bc

         dlnkap_dlnd = s% d_opacity_dlnd(1)/s% opacity(1)
         dlnkap_dlnT = s% d_opacity_dlnT(1)/s% opacity(1)

         dlnP_bc_dlnd = dlnP_bc_dlnPsurf*dlnPsurf_dlnkap*dlnkap_dlnd
         dlnP_bc_dlnT = dlnP_bc_dlnPsurf*dlnPsurf_dlnkap*dlnkap_dlnT

         dlnT_bc_dlnT = dlnT_bc_dlnTsurf*dlnTsurf_dlnkap*dlnkap_dlnT + dlnT_bc_dT0*dT0_dlnT
         dlnT_bc_dlnd = dlnT_bc_dlnTsurf*dlnTsurf_dlnkap*dlnkap_dlnd + dlnT_bc_dT0*dT0_dlnd

         if (s% do_struct_thermo) then ! temperature eqn
         
            if (s% lnPgas_flag) then
               dlnT_bc_dlnT_const_Pgas = &
                  dlnT_bc_dlnT + dlnT_bc_dlnd*s% dlnRho_dlnT_const_Pgas(1)
               call e00(s, xscale, equT, i_lnT, 1, nvar, dlnT_bc_dlnT_const_Pgas - 1)
            else
               call e00(s, xscale, equT, i_lnT, 1, nvar, dlnT_bc_dlnT - 1)
            end if
            
            dlnT_bc_dL = dlnT_bc_dlnTsurf*dlnTsurf_dL + dlnT_bc_dT0*dT0_dL
            call e00(s, xscale, equT, i_lum, 1, nvar, dlnT_bc_dL)
            
            if (s% do_struct_hydro) then ! partial of temperature eqn wrt lnR and (lnd or lnPgas)
            
               dlnT_bc_dlnR = dlnT_bc_dlnTsurf*dlnTsurf_dlnR + dlnT_bc_dT0*dT0_dlnR
               call e00(s, xscale, equT, i_lnR, 1, nvar, dlnT_bc_dlnR)

               if (s% lnPgas_flag) then
                  dlnT_bc_dlnPgas_const_T = dlnT_bc_dlnd*s% dlnRho_dlnPgas_const_T(1)
                  call e00(s, xscale, equT, i_lnPgas, 1, nvar, dlnT_bc_dlnPgas_const_T)
               else
                  call e00(s, xscale, equT, i_xlnd, 1, nvar, dlnT_bc_dlnd)
               end if
               
            end if
            
         else ! dummy eqn
         
            call e00(s,xscale,equT,i_lnT,1,nvar,one)
            
         end if

         if (s% do_struct_hydro) then ! pressure eqn
            
            if (s% lnPgas_flag) then
               dlnP_bc_dlnPgas_const_T = dlnP_bc_dlnd*s% dlnRho_dlnPgas_const_T(1)
               dlnP_dlnPgas_const_T = s% Pgas(1)/s% P(1)
               call e00(s, xscale, equP, i_lnPgas, 1, nvar, &
                  dlnP_bc_dlnPgas_const_T - dlnP_dlnPgas_const_T)
            else
               call e00(s, xscale, equP, i_xlnd, 1, &
                  nvar, dlnP_bc_dlnd - s% chiRho(1))
            end if
            
            dlnP_bc_dlnR = dlnP_bc_dlnPsurf*dlnPsurf_dlnR + dlnP_bc_dP0*dP0_dlnR
            call e00(s, xscale, equP, i_lnR, 1, nvar, dlnP_bc_dlnR)
            
            if (s% do_struct_thermo) then ! partial of pressure eqn wrt lnT
               
               if (s% lnPgas_flag) then
                  dlnP_bc_dlnT_const_Pgas = &
                     dlnP_bc_dlnT + dlnP_bc_dlnd*s% dlnRho_dlnT_const_Pgas(1)
                  dlnP_dlnT_const_Pgas = 4*s% Prad(1)/s% P(1)
                  call e00(s, xscale, equP, i_lnT, 1, &
                  nvar, dlnP_bc_dlnT_const_Pgas - dlnP_dlnT_const_Pgas)
               else
                  call e00(s, xscale, equP, i_lnT, 1, &
                  nvar, dlnP_bc_dlnT - s% chiT(1))
               end if
               
               dlnP_bc_dL = dlnP_bc_dlnPsurf*dlnPsurf_dL
               call e00(s, xscale, equP, i_lum, 1, nvar, dlnP_bc_dL)
               
            end if
            
         else ! dummy eqn
            
            if (s% lnPgas_flag) then
               call e00(s,xscale,equP,i_lnPgas,1,nvar,one)
            else
               call e00(s,xscale,equP,i_xlnd,1,nvar,one)
            end if
            
         end if

      end subroutine PT_eqns_surf


      subroutine eval_grad_star_info( &
            s, k, grad_star, d_grad_star_dL, d_grad_star_dlnR, &
            d_grad_star_dlnd00, d_grad_star_dlndm1, &
            d_grad_star_dlnT00, d_grad_star_dlnTm1, &
            d_grad_star_dlnT00_const_Pgas, d_grad_star_dlnTm1_const_Pgas, &
            d_grad_star_dlnPgas00_const_T, d_grad_star_dlnPgasm1_const_T, &
            ierr)
         type (star_info), pointer :: s 
         integer, intent(in) :: k        
         real(dp), intent(out) :: grad_star, d_grad_star_dL, d_grad_star_dlnR, &
            d_grad_star_dlnd00, d_grad_star_dlndm1, d_grad_star_dlnT00, d_grad_star_dlnTm1, &
            d_grad_star_dlnT00_const_Pgas, d_grad_star_dlnTm1_const_Pgas, &
            d_grad_star_dlnPgas00_const_T, d_grad_star_dlnPgasm1_const_T
         integer, intent(out) :: ierr         
         include 'formats'
              
         ierr = 0      
         
         grad_star = s% gradT(k)  
          
         d_grad_star_dL = s% d_gradT_dL(k)   
         d_grad_star_dlnR = s% d_gradT_dlnR(k)

         d_grad_star_dlnd00 = s% d_gradT_dlnd00(k)
         d_grad_star_dlndm1 = s% d_gradT_dlndm1(k)   
         d_grad_star_dlnT00 = s% d_gradT_dlnT00(k)
         d_grad_star_dlnTm1 = s% d_gradT_dlnTm1(k)         
         
         if (s% lnPgas_flag) then
            d_grad_star_dlnPgas00_const_T = &
               d_grad_star_dlnd00*s% dlnRho_dlnPgas_const_T(k)
            d_grad_star_dlnT00_const_Pgas = &
               d_grad_star_dlnT00 + d_grad_star_dlnd00*s% dlnRho_dlnT_const_Pgas(k)
            if (k > 1) then
               d_grad_star_dlnPgasm1_const_T = &
                  d_grad_star_dlndm1*s% dlnRho_dlnPgas_const_T(k-1)
               d_grad_star_dlnTm1_const_Pgas = &
                  d_grad_star_dlnTm1 + d_grad_star_dlndm1*s% dlnRho_dlnT_const_Pgas(k-1)
            else
               d_grad_star_dlnPgasm1_const_T = 0
               d_grad_star_dlnTm1_const_Pgas = 0
            end if
         end if

      end subroutine eval_grad_star_info
      


      end module hydro_eqns

