! ***********************************************************************
!
!   Copyright (C) 2012  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************


      module hydro_chem_eqns

      use star_private_def
      use const_def
      use utils_lib, only: is_bad_num

      implicit none
      

      contains

      subroutine do_chem_eqns( &
            s, xscale, nvar, equchem1, species, skip_partials, dt, equ, ierr)
         type (star_info), pointer :: s
         real(dp), pointer :: xscale(:,:) ! (nvar, nz)
         integer, intent(in) :: nvar, equchem1, species
         logical, intent(in) :: skip_partials
         real(dp), intent(in) :: dt
         real(dp), intent(out) :: equ(:,:)
         integer, intent(out) :: ierr         
         integer :: k, op_err
         include 'formats'         
         ierr = 0
!$OMP PARALLEL DO PRIVATE(k,op_err)
         do k = 1, s% nz
            if (ierr /= 0) cycle
            call do1_chem_eqns( &
               s, xscale, k, nvar, equchem1, species, skip_partials, dt, equ, op_err)
            if (op_err /= 0) ierr = op_err      
         end do
!$OMP END PARALLEL DO
      end subroutine do_chem_eqns


      subroutine do1_chem_eqns( &
            s, xscale, k, nvar, equchem1, species, skip_partials, dt_in, equ, ierr)
            
         use chem_def
         use num_lib, only: safe_log10
         use net_lib, only: show_net_contents, show_net_reactions, &
            show_net_params
         use rates_def, only: reaction_Name, i_rate
         use star_utils, only: em1, e00, ep1
         use solve_mix, only: revise_avg_mix_dxdt
         use solve_burn, only: revise_avg_burn_dxdt
         
         type (star_info), pointer :: s
         real(dp), pointer :: xscale(:,:) ! (nvar, nz)
         integer, intent(in) :: k, nvar, equchem1, species
         logical, intent(in) :: skip_partials
         real(dp), intent(in) :: dt_in
         real(dp), intent(out) :: equ(:,:)
         integer, intent(out) :: ierr

         integer, pointer :: reaction_id(:) ! maps net reaction number to reaction id
         integer :: nz, i_xlnd, i_lnPgas, i_lnT, i_lnR, op_err, j, i, jj, ii
         real(dp), pointer, dimension(:) :: sig
         real(dp), dimension(species) :: save_dx_burning, save_dx_mixing
         real(dp) :: &
            dt, dx_expected_dxa, dx_expected, dx_actual, x_actual, x_expected, &
            dq, dm, dx_burning, dx_mixing, dequ, dxdt_nuc, dxdt_mix, &
            xprev, sum_xprev, sum_dxdt_nuc, dx_expected_dlnd, dx_expected_dlnT, &
            d_dxdtmix_dx00, d_dxdtmix_dxm1, d_dxdtmix_dxp1, &
            d_dxmix_dx00, d_dxmix_dxm1, d_dxmix_dxp1, sum_dx_burning, sum_dx_mixing, &
            x00, xp1, xm1, dx00, dxp1, sigavg, d, w_out00, w_outp1, w_out, w_in, &
            sig00, sigp1, flux00, dflux00_dxm1, dflux00_dx00, &
            fluxp1, dfluxp1_dx00, dfluxp1_dxp1, eqn_scale, d_dxdt_dx, &
            dequ_dlnd, dequ_dlnT, dequ_dlnPgas_const_T, dequ_dlnT_const_Pgas
         logical :: revised_dx_mixing

         include 'formats'
         
         ierr = 0
         
         dt = dt_in
         
         nz = s% nz
         i_xlnd = s% i_xlnd
         i_lnPgas = s% i_lnPgas
         i_lnT = s% i_lnT
         i_lnR = s% i_lnR
         sig => s% sig
         
         dq = s% dq(k)
         dm = s% dm(k)
         w_out00 = 0
         w_outp1 = 1
                  
         sum_xprev = 0
         sum_dxdt_nuc = 0
         d_dxdtmix_dxp1 = 0
         d_dxdtmix_dxm1 = 0
         d_dxdtmix_dx00 = 0
         d_dxmix_dxp1 = 0
         d_dxmix_dxm1 = 0
         d_dxmix_dx00 = 0

         sig00 = sig(k)
   
         if (k < s% nz) then
            sigp1 = sig(k+1)
         else
            sigp1 = 0
         end if
         
         if (s% do_mix) then
      
            if (k > 1) then
               dflux00_dxm1 = -sig00
               dflux00_dx00 = sig00
            else
               dflux00_dxm1 = 0
               dflux00_dx00 = 0
            end if
      
            if (k < s% nz) then
               dfluxp1_dx00 = -sigp1
               dfluxp1_dxp1 = sigp1
            else
               dfluxp1_dx00 = 0
               dfluxp1_dxp1 = 0
            end if
         
            d_dxdtmix_dx00 = (dfluxp1_dx00 - dflux00_dx00)/dm
            d_dxdtmix_dxm1 = -dflux00_dxm1/dm
            d_dxdtmix_dxp1 = dfluxp1_dxp1/dm

            d_dxmix_dx00 = d_dxdtmix_dx00*dt
            d_dxmix_dxm1 = d_dxdtmix_dxm1*dt
            d_dxmix_dxp1 = d_dxdtmix_dxp1*dt

         else
         
            dx_mixing = 0
            d_dxmix_dx00 = 0
            d_dxmix_dxm1 = 0
            d_dxmix_dxp1 = 0
            
         end if
         
         sum_dx_burning = 0
         sum_dx_mixing = 0

         do j=1,species ! composition equation for species j in cell k
      
            i = equchem1+j-1
            
            x00 = s% xa(j,k)
            xprev = s% xa_pre(j,k)
            if (xprev < 0) then
               write(*,3) 's% xa_pre(j,k)', j, k, s% xa_pre(j,k)
               ierr = -1
               return
            end if
            x_actual = x00
            dx_actual = x_actual - xprev
         
            if (s% do_burn) then
            
               dxdt_nuc = s% dxdt_nuc(j,k)
               
            else if (s% do_mix) then ! use average burn rate
            
               dxdt_nuc = s% avg_burn_dxdt(j,k)
               
            else
            
               dxdt_nuc = 0
               
            end if
            
            if (s% do_mix) then
            
               if (k > 1) then
                  xm1 = s% xa(j,k-1)
                  dx00 = xm1 - x00
                  flux00 = -sig00*dx00
               else
                  flux00 = 0
               end if
         
               if (k < s% nz) then
                  xp1 = s% xa(j,k+1)
                  dxp1 = x00 - xp1
                  fluxp1 = -sigp1*dxp1
               else
                  fluxp1 = 0
               end if
               dxdt_mix = (fluxp1 - flux00)/dm
               
            else if (s% do_burn) then ! use average mix rate
            
               dxdt_mix = s% avg_mix_dxdt(j,k)
               
            else
            
               dxdt_mix = 0
               
            end if
                  
            dx_burning = dxdt_nuc*dt
            dx_mixing = dxdt_mix*dt
            
            eqn_scale = xscale(i,k)
         
            dx_expected = dx_mixing + dx_burning
            x_expected = xprev + dx_expected
            
            if (s% lnT(k)/ln10 >= s% min_logT_for_checking_x_expected) then
               if (x_expected > s% max_x_expected) s% max_x_expected = x_expected
               if (x_expected < s% min_x_expected) s% min_x_expected = x_expected
               if (x_expected > s% max_allowed_x_expected .or. &
                     x_expected < s% min_allowed_x_expected) then
                  if (s% report_ierr .or. s% trace_bad_x_expected) &
                     write(*,'(a55,3x,2i6,99f12.5)') trim(chem_isos% name(s% chem_id(j))) // &
                        ' bad x_expected at m/Msun, r/Rsun, logT', s% model_number, &
                        k, x_expected, s% m(k)/Msun, s% r(k)/Rsun, s% lnT(k)/ln10
                  ierr = -1
                  return
               end if
            end if
            
            revised_dx_mixing = .false.
            
            sum_dx_burning = sum_dx_burning + dx_burning
            sum_dx_mixing = sum_dx_mixing + dx_mixing
            save_dx_burning(j) = dx_burning
            save_dx_mixing(j) = dx_mixing
         
            equ(i,k) = (x_expected - x_actual)/eqn_scale            

            if (is_bad_num(equ(i,k))) then
               write(*,2) 'equ(i,k) ' // trim(chem_isos% name(s% chem_id(j))), k, equ(i,k)
               write(*,2) 'dx_expected', k, dx_expected
               write(*,2) 'dx_actual', k, dx_actual
               write(*,2) 'eqn_scale', k, eqn_scale
               stop 'chem eqn'
            end if
            
            !if (j == 18 .and. k==610 .and. .not. skip_partials) then
            !   write(*,2) 'dx_expected_dlnT', k, dt*s% dxdt_dT(j,k)*s% T(k) 
            !end if
            
            !if (.not. skip_partials) then
            !   if (x_expected < -10 .or. x_expected > 10) then
            !      write(*,3) 'x_expected ' // trim(chem_isos% name(s% chem_id(j))) // ' logT x_actual', &
            !         s% model_number, k, x_expected, s% lnT(k)/ln10, x_actual
            !   end if
            !end if

            if (.false.) then
            !if (j == 10 .and. k==638 .and. .not. skip_partials) then
            !if (.true. .and. equ(i,k) > 1d9 .and. s% hydro_call_number == 12266) then
               write(*,*)
               write(*,2) 'model_number, dt', s% model_number, dt
               write(*,2) 'equ(i,k) ' // trim(chem_isos% name(s% chem_id(j))), k, equ(i,k)
               write(*,2) 'dx_expected', k, dx_expected
               write(*,2) 'dx_actual', k, dx_actual
               write(*,2) 'dx_burning', k, dx_burning
               write(*,2) 'dx_mixing', k, dx_mixing
               write(*,2) 'xprev', k, xprev
               write(*,2) 'x_actual', k, x_actual
               write(*,2) 'x_expected', k, x_expected
               write(*,2) 'eqn_scale', k, eqn_scale
               write(*,2) 'logT', k, s% lnT(k)/ln10
               write(*,2) 'logRho', k, s% lnd(k)/ln10
               write(*,2) 'prev logT', k, s% lnT_start(k)/ln10
               write(*,2) 'prev logRho', k, s% lnd_start(k)/ln10
               write(*,*)
               write(*,2) 'dx_expected_dlnT', k, dt*s% dxdt_dT(j,k)*s% T(k) 
               write(*,*)
               write(*,2) 'dt*dxdt_nuc/dxdt_dT', k, dt*s% dxdt_nuc(j,k)/s% dxdt_dT(j,k)
               write(*,2) 'dt*dxdt_nuc/dxdt_dT/T', k, dt*s% dxdt_nuc(j,k)/s% dxdt_dT(j,k)/s% T(k)
               write(*,*)
               write(*,2) 's% dxdt_nuc(j,k)', k, s% dxdt_nuc(j,k)
               write(*,2) 's% dxdt_drho(j,k)', k, s% dxdt_drho(j,k)
               write(*,2) 's% dxdt_dT(j,k)', k, s% dxdt_dT(j,k)
               write(*,*)
               !do jj=1,species
               !   write(*,2) 'd_dxdt_dx ' // trim(chem_isos% name(s% chem_id(jj))), k, s% d_dxdt_dx(j,jj,k)
               !end do
               write(*,*)
               if (x_expected < 0 .or. s% hydro_call_number == 541) then
                  if ((dx_mixing < 0 .and. -dx_mixing > x_actual).or. s% hydro_call_number == 541) then
                     write(*,2) 'xm1', k-1, xm1
                     write(*,2) 'x00', k, x00
                     write(*,2) 'xp1', k+1, xp1
                     write(*,2) 'dx00', k, dx00
                     write(*,2) 'dxp1', k, dxp1
                     write(*,2) 'sig00', k, sig00
                     write(*,2) 'sigp1', k, sigp1
                     write(*,2) 'flux00', k, flux00
                     write(*,2) 'fluxp1', k, fluxp1
                     write(*,2) 'flux00/dm', k, flux00/dm
                     write(*,2) 'fluxp1/dm', k, fluxp1/dm
                     write(*,2) 'dt*flux00/dm', k, dt*flux00/dm
                     write(*,2) 'dt*fluxp1/dm', k, dt*fluxp1/dm
                     write(*,2) 'dxdt_mix', k, dxdt_mix
                     write(*,2) 'dt*dxdt_mix', k, dt*dxdt_mix
                     write(*,*)
                  end if
               end if
               
               
               write(*,*)
               !if (equ(i,k) > 1d9) stop 'chem eqn'
               !stop 'chem eqn'
            end if
         
            if (is_bad_num(equ(i,k))) then
               ierr = -1
               if (s% report_ierr) write(*,3) 'chem eqn equ(i,k)', i, k, equ(i,k)
               return
            end if
      
            if (skip_partials) cycle
         
            call e00(s, xscale, i, i, k, nvar, -1d0/eqn_scale)
         
            ! jacobian terms for x_expected
            if (s% do_burn) then
         
               do jj=1,species
                  ii = equchem1+jj-1
                  dx_expected_dxa = dt*s% d_dxdt_dx(j,jj,k)
                  dequ = dx_expected_dxa/eqn_scale
                  call e00(s, xscale, i, ii, k, nvar, dequ)
               end do
      
               dx_expected_dlnd = dt*s% dxdt_drho(j,k)*s% rho(k)
               dequ_dlnd = dx_expected_dlnd/eqn_scale
            
               if (s% do_struct_hydro) then
                  if (s% lnPgas_flag) then
                     dequ_dlnPgas_const_T = dequ_dlnd*s% dlnRho_dlnPgas_const_T(k)
                     call e00(s, xscale, i, i_lnPgas, k, nvar, dequ_dlnPgas_const_T)
                  else
                     call e00(s, xscale, i, i_xlnd, k, nvar, dequ_dlnd)
                  end if
               end if
            
               if (s% do_struct_thermo) then
                  dx_expected_dlnT = dt*s% dxdt_dT(j,k)*s% T(k) 
                  dequ_dlnT = dx_expected_dlnT/eqn_scale
                  if (s% lnPgas_flag) then
                     dequ_dlnT_const_Pgas = &
                        dequ_dlnT + dequ_dlnd*s% dlnRho_dlnT_const_Pgas(k)
                     call e00(s, xscale, i, i_lnT, k, nvar, dequ_dlnT_const_Pgas)
                  else
                     call e00(s, xscale, i, i_lnT, k, nvar, dequ_dlnT)
                  end if
               end if
               
               if (.not. s% do_mix) then ! set avg_burn_dxdt for use by mixing
                  s% avg_burn_dxdt(j,k) = s% dxdt_nuc(j,k)
               end if

            end if
         
            if (s% do_mix .and. .not. revised_dx_mixing) then 
                       
               dx_expected_dxa = d_dxmix_dx00
               dequ = dx_expected_dxa/eqn_scale
               call e00(s, xscale, i, i, k, nvar, dequ)               
               if (k > 1) then
                  dx_expected_dxa = d_dxmix_dxm1
                  dequ = dx_expected_dxa/eqn_scale
                  call em1(s, xscale, i, i, k, nvar, dequ)
               end if               
               if (k < nz) then
                  dx_expected_dxa = d_dxmix_dxp1
                  dequ = dx_expected_dxa/eqn_scale
                  call ep1(s, xscale, i, i, k, nvar, dequ)
               end if    
               
               if (.not. s% do_burn) & ! set avg_mix_dxdt for use by burn
                  s% avg_mix_dxdt(j,k) = dx_actual/dt - s% avg_burn_dxdt(j,k)
               
            end if
         
         end do
         
         if (s% do_burn .and. .not. s% do_mix) then ! ensure that avg_burn_dxdt sums to 0
            call revise_avg_burn_dxdt(s, k, species, s% avg_burn_dxdt, dt, ierr)
            if (ierr /= 0) then
               write(*,2) 'do1_chem_eqns failed in revise_avg_burn_dxdt', k
               return
            end if
         end if
         
         if (s% do_mix .and. .not. s% do_burn) then ! ensure that avg_mix_dxdt sums to 0
            call revise_avg_mix_dxdt(s, k, species, s% avg_mix_dxdt, dt, ierr)
            if (ierr /= 0) then
               write(*,2) 'do1_chem_eqns failed in revise_avg_mix_dxdt', k
               return
            end if
         end if
         

      end subroutine do1_chem_eqns


      end module hydro_chem_eqns

