! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module history
      
      use star_private_def
      use const_def
      use chem_def
      use history_specs
      
      implicit none
      



         
      logical, parameter :: open_close_log = .true.



      contains
      
      
      subroutine do_get_data_for_history_columns( &
            s, id_extra, &
            how_many_extra_history_columns, &
            data_for_extra_history_columns, &
            ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: id_extra
         interface
            include 'extra_history_cols.inc'
         end interface
         integer, intent(out) :: ierr
         logical, parameter :: write_flag = .false.
         call do_history_info( &
            s, id_extra, &
            how_many_extra_history_columns, &
            data_for_extra_history_columns, &
            write_flag, ierr)
      end subroutine do_get_data_for_history_columns


      subroutine write_history_info( &
            s, id_extra, &
            how_many_extra_history_columns, &
            data_for_extra_history_columns, &
            ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: id_extra
         interface
            include 'extra_history_cols.inc'
         end interface
         integer, intent(out) :: ierr
         logical, parameter :: write_flag = .true.
         call do_history_info( &
            s, id_extra, &
            how_many_extra_history_columns, &
            data_for_extra_history_columns, &
            write_flag, ierr)
      end subroutine write_history_info


      subroutine do_history_info( &
            s, id_extra, &
            how_many_extra_history_columns, &
            data_for_extra_history_columns, &
            write_flag, ierr)
         use utils_lib, only: alloc_iounit, free_iounit, &
            integer_dict_create_hash, integer_dict_free
         use star_utils, only: eval_csound
         use chem_def, only: category_name
         use rates_def, only: rates_reaction_id_max, i_rate
         type (star_info), pointer :: s
         integer, intent(in) :: id_extra
         interface
            include 'extra_history_cols.inc'
         end interface
         logical, intent(in) :: write_flag
         integer, intent(out) :: ierr
         
         character (len=strlen) :: fname, dbl_fmt, int_fmt, txt_fmt
         integer :: numcols, io, i, nz, col, j, i0, num_extra_cols, n
         integer, parameter :: num_epsnuc_out = 12
         real(dp) :: &
            epsnuc_out(num_epsnuc_out), csound_surf, v_surf, envelope_fraction_left
         integer :: mixing_regions, mix_relr_regions, burning_regions
         integer, pointer :: mixing_type(:), mix_relr_type(:), burning_type(:)
         character (len=maxlen_history_column_name), pointer :: extra_col_names(:)
         real(dp), pointer :: extra_col_vals(:)
         
         character (len=maxlen_profile_column_name), pointer :: &
            names(:) ! (num_history_columns)
         real(dp), pointer :: vals(:) ! (num_history_columns)
         logical, pointer :: is_int(:) ! (num_history_columns)
         
         include 'formats'
         
         dbl_fmt = s% star_history_dbl_format
         int_fmt = s% star_history_int_format
         txt_fmt = s% star_history_txt_format
         
         ierr = 0
         
         if (.not. associated(s% history_column_spec)) then
            numcols = 0
         else
            numcols = size(s% history_column_spec, dim=1)
         end if
         num_extra_cols = how_many_extra_history_columns(s, s% id, id_extra)
         n = numcols + num_extra_cols
         if (n == 0) then
            write(*,*) 'WARNING: do not have any output specified for logs.'
            ierr = -1
            return
         end if

         if (s% number_of_history_columns < 0) then
            s% number_of_history_columns = n
         else if (s% number_of_history_columns /= n) then
            if (associated(s% history_values)) then
               deallocate(s% history_values)
               nullify(s% history_values)
            end if
            if (associated(s% history_names)) then
               deallocate(s% history_names)
               nullify(s% history_names)
            end if
            if (associated(s% history_value_is_integer)) then
               deallocate(s% history_value_is_integer)
               nullify(s% history_value_is_integer)
            end if
            if (associated(s% history_names_dict)) then
               call integer_dict_free(s% history_names_dict)
               nullify(s% history_names_dict)
            end if
            s% need_to_set_history_names_etc = .true.
            s% number_of_history_columns = n
         end if
         
         if (.not. associated(s% history_values)) then
            allocate(s% history_values(n))
         else if (size(s% history_values,dim=1) /= n) then
            ierr = -1
            write(*,2) 'bad size s% history_values', &
               size(s% history_values,dim=1), n
         end if
         vals => s% history_values
         
         if (.not. associated(s% history_names)) then
            allocate(s% history_names(n))
         else if (size(s% history_names,dim=1) /= n) then
            ierr = -1
            write(*,2) 'bad size s% history_names', &
               size(s% history_names,dim=1), n
         end if
         if (s% need_to_set_history_names_etc) then
            names => s% history_names
         else
            nullify(names)
         end if
         
         if (.not. associated(s% history_value_is_integer)) then
            allocate(s% history_value_is_integer(n))
         else if (size(s% history_value_is_integer,dim=1) /= n) then
            ierr = -1
            write(*,2) 'bad size s% history_value_is_integer', &
               size(s% history_value_is_integer,dim=1), n
         end if
         if (s% need_to_set_history_names_etc) then
            is_int => s% history_value_is_integer
         else
            nullify(is_int)
         end if
         
         nz = s% nz
         nullify(mixing_type)
         nullify(mix_relr_type)
         nullify(burning_type)
         nullify(extra_col_names)
         nullify(extra_col_vals)
         
         if (write_flag) then
            io = alloc_iounit(ierr)
            if (ierr /= 0) return
         else
            io = -1 ! set to invalid value to trigger complaint if use it by mistake
         end if
         if (num_extra_cols > 0) then
            allocate( &
               extra_col_names(num_extra_cols), extra_col_vals(num_extra_cols), stat=ierr)
            if (ierr /= 0) then
               if (io > 0) call free_iounit(io)
               return
            end if
            call data_for_extra_history_columns( &
               s, s% id, id_extra, num_extra_cols, extra_col_names, extra_col_vals, ierr)
            if (ierr /= 0) then
               deallocate(extra_col_names, extra_col_vals)
               if (io > 0) call free_iounit(io)
               return
            end if
         end if
         
         i0 = 1
         if (write_flag .and. (open_close_log .or. s% model_number == -100)) then
            fname = trim(s% log_directory) // '/' // trim(s% star_history_name)
            if ((.not. history_file_exists(fname,io)) .or. &
                  s% doing_first_model_of_run .or. s% need_to_set_history_names_etc) then
               ierr = 0
               if (len_trim(s% star_history_header_name) > 0) then
                  fname = trim(s% log_directory) // '/' // trim(s% star_history_header_name)
               end if
               open(unit=io, file=trim(fname), action='write', iostat=ierr)
            else
               i0 = 3            
               open(unit=io, file=trim(fname), action='write', position='append', iostat=ierr)
            end if
            if (ierr /= 0) then
               write(*,*) 'failed to open ' // trim(fname)
               if (io > 0) call free_iounit(io)
               return
            end if
         end if
         
         csound_surf = eval_csound(s,1,ierr)
         if (ierr /= 0) return
         
         if (.not. s% v_flag) then
            v_surf =  s% r(1)*s% dlnR_dt(1)
         else
            v_surf = s% v(1)
         end if
         
         if (s% initial_mass > s% he_core_mass) then
            envelope_fraction_left = &
               (s% star_mass - s% he_core_mass)/(s% initial_mass - s% he_core_mass)
         else
            envelope_fraction_left = 1
         end if
         
         if (write_flag .and. i0 == 1) then ! write parameters at start of log
            call s% other_history_data_initialize(s% id, ierr)
            if (ierr /= 0) return
            do i=1,3
               col = 0
               call write_integer(io, col, i, 'version_number', version_number)
               call write_val(io, col, i, 'burn_min1', s% burn_min1)
               call write_val(io, col, i, 'burn_min2', s% burn_min2)
               write(io,*)
            end do
            write(io,*)
         end if

         do i=i0,3 ! add a row to the log
         
            call s% other_history_data_add_model(s% id, ierr)
            if (ierr /= 0) return
            col = 0
            if (i==3) then
            
               if (write_flag .and. i0 == 1 .and. len_trim(s% star_history_header_name) > 0) then
                  close(io)
                  fname = trim(s% log_directory) // '/' // trim(s% star_history_name)
                  open(unit=io, file=trim(fname), action='write', status='replace', iostat=ierr)
                  if (ierr /= 0) then
                     call free_iounit(io); return
                  end if
               end if
               
               epsnuc_out(1:4) = s% burn_zone_mass(1:4,1)
               epsnuc_out(5:8) = s% burn_zone_mass(1:4,2)
               epsnuc_out(9:12) = s% burn_zone_mass(1:4,3)
               
               mixing_regions = count_output_mix_regions(mixing_offset)               
               if (mixing_regions > 0) then
                  allocate(mixing_type(nz), stat=ierr)
                  if (ierr /= 0) exit
                  call set_mix_types(mixing_type, mixing_regions)
                  call prune_weak_mixing_regions(mixing_type, mixing_regions)
               end if
               
               mix_relr_regions = count_output_mix_regions(mix_relr_offset)               
               if (mix_relr_regions > 0) then
                  allocate(mix_relr_type(nz), stat=ierr)
                  if (ierr /= 0) exit
                  call set_mix_types(mix_relr_type, mix_relr_regions)
                  call prune_weak_mixing_regions(mix_relr_type, mix_relr_regions)
               end if
               
               burning_regions = count_output_burn_regions()               
               if (burning_regions > 0) then
                  allocate(burning_type(nz), stat=ierr)
                  if (ierr /= 0) exit
                  call set_burn_types
               end if
               
            end if
            
            do j=1,numcols
               call do_col(i,j)
            end do
            do j=1,num_extra_cols
               call do_extra_col(i,j)
            end do
            if (write_flag) write(io,*)
            
         end do
         
         if (open_close_log) close(io)
         
         if (io > 0) call free_iounit(io)
         
         if (associated(mixing_type)) deallocate(mixing_type)
         if (associated(mix_relr_type)) deallocate(mix_relr_type)
         if (associated(burning_type)) deallocate(burning_type)
         if (associated(extra_col_names)) deallocate(extra_col_names)
         if (associated(extra_col_vals)) deallocate(extra_col_vals)
         
         s% model_number_of_history_values = s% model_number
         
         if (s% need_to_set_history_names_etc) then
            call integer_dict_create_hash(s% history_names_dict, ierr)
         end if
            
         s% need_to_set_history_names_etc = .false.


         contains


         subroutine do_extra_col(pass, j)
            integer, intent(in) :: pass, j
            if (pass == 1) then
               if (write_flag) write(io, fmt=int_fmt, advance='no') j + numcols
            else if (pass == 2) then
               call do_name(j + numcols, extra_col_names(j))
            else if (pass == 3) then
               call do_val(j + numcols, extra_col_vals(j))
            end if
         end subroutine do_extra_col


         subroutine do_name(j, col_name)
            use utils_lib, only: integer_dict_define
            integer, intent(in) :: j
            integer :: ierr
            character (len=*), intent(in) :: col_name
            if (write_flag) write(io, fmt=txt_fmt, advance='no') trim(col_name)
            if (associated(names)) names(j) = trim(col_name)
            if (s% need_to_set_history_names_etc) then
               call integer_dict_define(s% history_names_dict, col_name, j, ierr)
               if (ierr /= 0) write(*,*) 'failed in integer_dict_define ' // trim(col_name)
            end if
         end subroutine do_name
         
         
         integer function count_output_burn_regions()
            integer :: j, cnt, c, i, ii
            cnt = 0
            do j=1,numcols
               c = s% history_column_spec(j)
               if (c > burning_offset .and. c <= burning_offset + idel) then
                  i = c - burning_offset
                  ii = (i+1)/2
                  if (ii > cnt) cnt = ii
               end if
            end do
            count_output_burn_regions = cnt
         end function count_output_burn_regions
         
         
         subroutine set_burn_types
            integer :: cnt, min_ktop, min_kbot, imax, i, prev_cnt, k
            real(dp) :: val
            logical, parameter :: dbg = .false.
            do k=1,nz
               val = s% eps_nuc(k) - s% non_nuc_neu(k)
               burning_type(k) = int(sign(1d0,val)*max(0d0,1d0+log10(abs(val))))
            end do
            ! remove smallest regions until <= burning_regions remain
            imax = nz
            do i=1,imax
               call count_regions(burning_type, cnt, min_ktop, min_kbot)
               if (dbg) write(*,*) 'count_regions', cnt
               if (cnt <= burning_regions) return
               if (i > 1 .and. cnt >= prev_cnt) then
                  write(*,*) 'bug in set_burn_types: cnt, prev_cnt', cnt, prev_cnt
                  if (dbg) stop 'debug: set_burn_types'
                  return
               end if
               prev_cnt = cnt
               if (dbg) write(*,*) 'remove_region', min_ktop, min_kbot, cnt
               call remove_region(burning_type, min_ktop, min_kbot)
            end do
            if (dbg) stop 'debug: set_burn_types'
         end subroutine set_burn_types

         
         integer function count_output_mix_regions(mx_offset)
            integer, intent(in) :: mx_offset
            integer :: j, cnt, c, i, ii
            cnt = 0
            do j=1,numcols
               c = s% history_column_spec(j)
               if (c > mx_offset .and. c < mx_offset+idel) then
                  i = c - mx_offset
                  ii = (i+1)/2
                  if (ii > cnt) cnt = ii
               end if
            end do
            count_output_mix_regions = cnt
         end function count_output_mix_regions
         
         
         subroutine prune_weak_mixing_regions(mx_type, mx_regions)
            use mlt_def, only: no_mixing
            integer :: mx_type(:), mx_regions
            real(dp) :: D_max_in_region, D_cutoff
            integer :: min_ktop, min_kbot
            integer :: i
            logical, parameter :: dbg = .false.
            include 'formats'
            D_cutoff = s% mixing_D_limit_for_log
            do i = 1, 10000
               call find_weakest_mixing_region( &
                  mx_type, mx_regions, D_max_in_region, min_ktop, min_kbot)
               if (D_max_in_region > D_cutoff) then
                  if (dbg) write(*,3) 'done', min_ktop, min_kbot, D_max_in_region, D_cutoff
                  return
               end if
               if (dbg) write(*,3) 'remove_weak_mixing_region', min_ktop, min_kbot, D_max_in_region
               if (dbg) write(*,3) 'i model mass', &
                  i, s% model_number, s% m(min_kbot)/Msun, s% m(min_ktop)/Msun
               if (dbg) write(*,*)
               mx_type(min_ktop:min_kbot) = no_mixing
            end do
         end subroutine prune_weak_mixing_regions
         
         
         subroutine find_weakest_mixing_region( &
               mx_type, mx_regions, D_max_in_region, min_ktop, min_kbot)
            use mlt_def
            integer :: mx_type(:), mx_regions
            real(dp), intent(out) :: D_max_in_region
            integer, intent(out) :: min_ktop, min_kbot
            
            integer :: k, kbot, cur_type
            real(dp) :: D_max
            logical, parameter :: dbg = .false.
            include 'formats'
            min_ktop = 1
            min_kbot = nz
            D_max_in_region = 1d99
            kbot = nz
            cur_type = mx_type(nz)
            do k=nz-1,1,-1
               if (cur_type == mx_type(k)) cycle
               ! k is bottom of new region; k+1 is top of region cnt
               if (cur_type /= no_mixing &
                     .and. cur_type /= thermo_haline_mixing &
                     .and. cur_type /= semiconvective_mixing) then
                  D_max = maxval(s% D_mix_non_rotation(k+1:kbot))
                  if (D_max < D_max_in_region) then
                     D_max_in_region = D_max; min_ktop = k+1; min_kbot = kbot
                  end if
               end if
               kbot = k
               cur_type = mx_type(k)
            end do
            if (cur_type == no_mixing) then
               D_max = maxval(s% D_mix_non_rotation(1:kbot))
               if (D_max < D_max_in_region) then
                  D_max_in_region = D_max; min_ktop = 1; min_kbot = kbot
               end if
            end if

         end subroutine find_weakest_mixing_region
         
         
         subroutine set_mix_types(mx_type, mx_regions)
            integer :: mx_type(:), mx_regions
            
            integer :: cnt, min_ktop, min_kbot, imax, i, prev_cnt, k
            logical, parameter :: dbg = .false.
            
            do k=1,nz
               mx_type(k) = s% mixing_type(k)
            end do
            ! remove smallest regions until <= mixing_regions remain
            imax = nz
            do i=1,imax
               call count_regions(mx_type, cnt, min_ktop, min_kbot)
               if (dbg) write(*,*) 'count_regions', cnt
               if (cnt <= mx_regions) exit
               if (i > 1 .and. cnt >= prev_cnt) then
                  write(*,*) 'bug in set_mix_types: cnt, prev_cnt', cnt, prev_cnt
                  if (dbg) stop 'set_mix_types'
                  return
               end if
               prev_cnt = cnt
               if (dbg) write(*,*) 'remove_region', min_ktop, min_kbot, cnt
               call remove_region(mx_type, min_ktop, min_kbot)
            end do        
            if (dbg) call show_regions(mx_type)
         end subroutine set_mix_types
         
         
         subroutine count_regions(mx_type, cnt, min_ktop, min_kbot)
            integer, intent(in) :: mx_type(:)
            integer, intent(out) :: cnt, min_ktop, min_kbot
            integer :: k, kbot, cur_type
            real(dp) :: prev_qtop, qtop, dq, min_dq
            logical, parameter :: dbg = .false.
            include 'formats'
            cnt = 1
            min_ktop = 1
            min_kbot = nz
            prev_qtop = 0
            min_dq = 1
            kbot = nz
            cur_type = mx_type(nz)
            do k=nz-1,1,-1
               if (cur_type == mx_type(k)) cycle
               ! k is bottom of new region; k+1 is top of region cnt
               qtop = s% q(k+1)
               dq = qtop - prev_qtop
               !if (dbg) write(*,3) 'dq', k+1, cnt, dq, min_dq
               if (dq < min_dq) then
                  min_dq = dq; min_ktop = k+1; min_kbot = kbot
                  if (dbg) write(*,3) &
                     'loop min_ktop min_kbot min_dq', min_ktop, min_kbot, min_dq
               end if
               kbot = k
               cnt = cnt+1
               cur_type = mx_type(k)
               prev_qtop = qtop
            end do       
            qtop = 1
            dq = qtop - prev_qtop
            if (dq < min_dq) then
               min_dq = dq; min_ktop = 1; min_kbot = kbot
               if (dbg) write(*,3) &
                  'final min_ktop min_kbot min_dq', min_ktop, min_kbot, min_dq
            end if
            !if (dbg) write(*,3) 'dq', 1, cnt, dq, min_dq
         end subroutine count_regions
         
         
         subroutine show_regions(mx_type)
            integer, intent(in) :: mx_type(:)
            integer :: k, kbot, cur_type
            include 'formats'
            kbot = nz
            cur_type = mx_type(nz)
            do k=nz-1,1,-1
               if (cur_type == mx_type(k)) cycle
               ! k is bottom of new region; k+1 is top of region cnt
               write(*,4) 'mix region', cur_type, k+1, kbot, s% m(k+1)/Msun, s% m(kbot)/Msun
               kbot = k
               cur_type = mx_type(k)
            end do       
            write(*,4) 'mix region', cur_type, 1, kbot, s% m(1)/Msun, s% m(kbot)/Msun
         end subroutine show_regions
         
         
         subroutine remove_region(mx_type, min_ktop, min_kbot)
            integer, intent(inout) :: mx_type(:)
            integer, intent(in) :: min_ktop, min_kbot
            integer :: new_type
            logical, parameter :: dbg = .false.
            include 'formats'
            if (dbg) then
               write(*,2) 'q top', min_ktop, s% q(min_ktop)
               write(*,2) 'q bot', min_kbot, s% q(min_kbot)
               write(*,2) 'dq', min_kbot, s% q(min_ktop) - s% q(min_kbot)
            end if
            if (min_ktop > 1) then ! merge with above
               new_type = mx_type(min_ktop-1)
            else if (min_kbot < nz) then ! merge with below
               new_type = mx_type(min_kbot+1)
            else
               write(*,*) 'confusion in args for remove_region', min_ktop, min_kbot
               return
            end if
            mx_type(min_ktop:min_kbot) = new_type
            if (dbg) write(*,2) 'new type', min_kbot, new_type
         end subroutine remove_region
         
         
         integer function region_top(mx_type,j)
            integer, intent(in) :: mx_type(:), j
            integer :: k, cnt, cur_type
            cnt = 1
            cur_type = mx_type(nz)
            do k=nz-1,1,-1
               if (cur_type == mx_type(k)) cycle
               ! k is start of new region; k+1 is top of region cnt
               if (cnt == j) then
                  region_top = k+1
                  return
               end if
               cnt = cnt+1
               cur_type = mx_type(k)
            end do
            if (cnt == j) then
               region_top = 1
            else
               region_top = -1
            end if
         end function region_top
         
         
         real(dp) function interpolate_burn_bdy(k) result(val)
            use mlt_def
            use num_lib, only: find0
            integer, intent(in) :: k
            integer :: bv, bv0, bv1
            real(dp) :: eps, eps0, eps1, d0, d1, q0, q1
            include 'formats'
            if (k <= 0) then
               val = -1; return
            end if
            val = s% q(k)
            if (k == 1) return
            eps1 = s% eps_nuc(k) - s% eps_nuc_neu_total(k) - s% non_nuc_neu(k)
            bv1 = sign(1d0,eps1)*log10(max(1d0,abs(eps1)))
            eps0 = s% eps_nuc(k-1) - s% eps_nuc_neu_total(k-1) - s% non_nuc_neu(k-1)
            bv0 = sign(1d0,eps0)*log10(max(1d0,abs(eps0)))
            bv = max(bv0,bv1)
            eps = 10**bv
            d0 = eps0 - eps
            d1 = eps1 - eps
            if (d0*d1 >= 0) return
            q1 = s% q(k) - s% dq(k)*0.5d0
            q0 = s% q(k-1) - s% dq(k-1)*0.5d0
            val = find0(q1,d1,q0,d0)
         end function interpolate_burn_bdy
         

         subroutine do_col(pass, j)
            integer, intent(in) :: pass, j
            if (pass == 1) then
               call do_col_pass1
            else if (pass == 2) then
               call do_col_pass2(j)
            else if (pass == 3) then
               call do_col_pass3(s% history_column_spec(j))
            end if
         end subroutine do_col
         
         
         subroutine do_col_pass1 ! write the column number
            col = col+1
            if (write_flag) write(io, fmt=int_fmt, advance='no') col
         end subroutine do_col_pass1
         
         
         subroutine do_col_pass2(j) ! get the column name
            integer, intent(in) :: j
            character (len=100) :: col_name
            character (len=10) :: str
            integer :: c, i, ii
            c = s% history_column_spec(j)
            if (c > burning_offset) then
               i = c - burning_offset
               ii = (i+1)/2
               if (ii > burning_regions) burning_regions = ii
               if (ii < 10) then
                  write(str,'(i1)') ii
               else if (ii < 100) then
                  write(str,'(i2)') ii
               else
                  write(str,'(i3)') ii
               end if
               if (mod(i,2)==1) then ! burning type
                  col_name = 'burn_type_' // trim(str)
               else ! location of top
                  col_name = 'burn_qtop_' // trim(str)
               end if
            else if (c > mix_relr_offset) then
               i = c - mix_relr_offset
               ii = (i+1)/2
               if (ii > mix_relr_regions) mix_relr_regions = ii
               if (ii < 10) then
                  write(str,'(i1)') ii
               else if (ii < 100) then
                  write(str,'(i2)') ii
               else
                  write(str,'(i3)') ii
               end if
               if (mod(i,2)==1) then
                  col_name = 'mix_relr_type_' // trim(str)
               else ! location of top
                  col_name = 'mix_relr_top_' // trim(str)
               end if
            else if (c > mixing_offset) then
               i = c - mixing_offset
               ii = (i+1)/2
               if (ii > mixing_regions) mixing_regions = ii
               if (ii < 10) then
                  write(str,'(i1)') ii
               else if (ii < 100) then
                  write(str,'(i2)') ii
               else
                  write(str,'(i3)') ii
               end if
               if (mod(i,2)==1) then ! mixing type
                  col_name = 'mix_type_' // trim(str)
               else ! location of top
                  col_name = 'mix_qtop_' // trim(str)
               end if            
            else if (c > c_d_eps_dlnT_offset) then
               i = c - c_d_eps_dlnT_offset
               col_name = 'c_d_eps_dlnT_' // trim(category_name(i))
            else if (c > c_d_eps_dlnd_offset) then
               i = c - c_d_eps_dlnd_offset
               col_name = 'c_d_eps_dlnd_' // trim(category_name(i))
            else if (c > c_log_eps_burn_offset) then
               i = c - c_log_eps_burn_offset
               col_name = 'c_log_eps_burn_' // trim(category_name(i))
            else if (c > max_eps_nuc_offset) then
               i = c - max_eps_nuc_offset
               col_name = 'max_eps_nuc_log_' // trim(chem_isos% name(i))
            else if (c > cz_top_max_offset) then
               i = c - cz_top_max_offset
               col_name = 'cz_top_log_' // trim(chem_isos% name(i))
            else if (c > cz_max_offset) then
               i = c - cz_max_offset
               col_name = 'cz_log_' // trim(chem_isos% name(i))
            else if (c > log_surface_xa_offset) then
               i = c - log_surface_xa_offset
               col_name = 'log_surface_' // trim(chem_isos% name(i))
            else if (c > log_center_xa_offset) then
               i = c - log_center_xa_offset
               col_name = 'log_center_' // trim(chem_isos% name(i))
            else if (c > log_average_xa_offset) then
               i = c - log_average_xa_offset
               col_name = 'log_average_' // trim(chem_isos% name(i))
            else if (c > log_total_mass_offset) then
               i = c - log_total_mass_offset
               col_name = 'log_total_mass_' // trim(chem_isos% name(i))
            else if (c > total_mass_offset) then
               i = c - total_mass_offset
               col_name = 'total_mass_' // trim(chem_isos% name(i))
            else if (c > category_offset) then
               i = c - category_offset
               col_name = category_name(i)
            else if (c > average_xa_offset) then
               i = c - average_xa_offset
               col_name = 'average_' // trim(chem_isos% name(i))
            else if (c > surface_xa_offset) then
               i = c - surface_xa_offset
               col_name = 'surface_' // trim(chem_isos% name(i))
            else if (c > center_xa_offset) then
               i = c - center_xa_offset
               col_name = 'center_' // trim(chem_isos% name(i))
            else
               col_name = trim(history_column_name(c))
            end if
            call do_name(j, col_name)
         end subroutine do_col_pass2
         
         
         subroutine do_col_pass3(c) ! get the column value
            use rates_def
            use mlt_def, only: convective_mixing
            integer, intent(in) :: c
            integer :: i, ii, k, int_val
            logical :: is_int_val
            real(dp) :: val, val1, Ledd, power_photo, frac
            int_val = 0; val = 0; is_int_val = .false.
            if (c > burning_offset) then
               i = c - burning_offset
               ii = (i+1)/2
               k = region_top(burning_type, ii)
               if (mod(i,2)==1) then ! burning type
                  is_int_val = .true.
                  if (k > 0) then
                     int_val = burning_type(k)
                  else
                     int_val = -9999
                  end if
               else ! location of top
                  val = interpolate_burn_bdy(k)
               end if
            else if (c > mix_relr_offset) then
               i = c - mix_relr_offset
               ii = (i+1)/2
               k = region_top(mix_relr_type, ii)
               if (mod(i,2)==1) then ! mixing type
                  is_int_val = .true.
                  if (k > 0) then
                     int_val = mix_relr_type(k)
                  else
                     int_val = -1
                  end if
               else ! r/rstar location of boundary
                  if (k <= 1) then
                     val = 1d0
                  else
                     frac = s% mixing_type_change_dq(k-1)/s% dq(k-1)
                     val = (1d0 - frac)*s% r(k-1)**3 + frac*s% r(k)**3
                     val = (val**(1d0/3d0))/s% r(1)
                  end if
               end if
            else if (c > mixing_offset) then
               i = c - mixing_offset
               ii = (i+1)/2
               k = region_top(mixing_type, ii)
               if (mod(i,2)==1) then ! mixing type
                  is_int_val = .true.
                  if (k > 0) then
                     int_val = mixing_type(k)
                  else
                     int_val = -1
                  end if
               else ! q location of boundary
                  if (k <= 1) then
                     val = 1d0
                  else
                     val = s% q(k-1) - s% mixing_type_change_dq(k-1)
                  end if
               end if
            else
               call history_getval( &
                  s, c, val, int_val, is_int_val, &
                  nz, v_surf, csound_surf, envelope_fraction_left, epsnuc_out, ierr)
               if (ierr /= 0) then
                  write(*,*) 'missing log info for ' // trim(history_column_name(c)), j, k
                  return
               end if
            end if
            if (is_int_val) then
               call do_int_val(j,int_val)
            else
               call do_val(j,val)
            end if
         end subroutine do_col_pass3
         
         
         subroutine do_val(j, val)
            use utils_lib, only: is_bad_num
            integer, intent(in) :: j
            real(dp), intent(in) :: val
            if (write_flag) then
               if (is_bad_num(val)) then
                  write(io, fmt=dbl_fmt, advance='no') -1d99
               else
                  write(io, fmt=dbl_fmt, advance='no') val
               end if
            end if
            if (associated(vals)) vals(j) = val
            if (associated(is_int)) is_int(j) = .false.
         end subroutine do_val
         
         
         subroutine do_int_val(j, val)
            integer, intent(in) :: j
            integer, intent(in) :: val
            if (write_flag) write(io, fmt=int_fmt, advance='no') val
            if (associated(vals)) vals(j) = dble(val)
            if (associated(is_int)) is_int(j) = .true.
         end subroutine do_int_val
                  
         
      end subroutine do_history_info


      subroutine write_integer(io, col, pass, name, val)
         integer, intent(in) :: io, pass
         integer, intent(inout) :: col
         character (len=*), intent(in) :: name
         integer, intent(in) :: val
         if (pass == 1) then
            col = col+1
            write(io, fmt='(i28, 1x)', advance='no') col
         else if (pass == 2) then
            write(io, fmt='(a28, 1x)', advance='no') trim(name)
         else if (pass == 3) then
            write(io, fmt='(i28, 1x)', advance='no') val
         end if
      end subroutine write_integer
      
      
      subroutine write_val(io, col, pass, name, val)
         integer, intent(in) :: io, pass
         integer, intent(inout) :: col
         character (len=*), intent(in) :: name
         real(dp), intent(in) :: val
         if (pass == 1) then
            col = col+1
            write(io, fmt='(i28, 1x)', advance='no') col
         else if (pass == 2) then
            write(io, fmt='(a28, 1x)', advance='no') trim(name)
         else if (pass == 3) then
            write(io, fmt='(1pe28.16e3, 1x)', advance='no') val
         end if
      end subroutine write_val
      
      
      subroutine history_getval( &
            s, c, val, int_val, is_int_val, &
            nz, v_surf, csound_surf, envelope_fraction_left, epsnuc_out, ierr)
         use rates_def, only: i_rate
         use star_utils, only: &
            dt_Courant, eval_Ledd, center_value, &
            omega_crit, get_L_rad, get_phi_Joss, get_L_rad_div_Ledd
         type (star_info), pointer :: s
         integer, intent(in) :: c, nz
         real(dp), intent(in) :: &
            v_surf, csound_surf, envelope_fraction_left, epsnuc_out(:)
         real(dp), intent(out) :: val
         integer, intent(out) :: int_val
         logical, intent(out) :: is_int_val
         integer, intent(out) :: ierr
         integer :: k, i
         real(dp) :: Ledd, L_rad, phi_Joss, power_photo, tmp, r
         
         include 'formats'
         
         ierr = 0
         is_int_val = .false.
         int_val = 0
         val = 0



         if (c > c_d_eps_dlnT_offset) then
            i = c - c_d_eps_dlnT_offset
            val = s% center_eps_burn_dT(i)*(10d0**s% log_center_temperature)
         else if (c > c_d_eps_dlnd_offset) then
            i = c - c_d_eps_dlnd_offset
            val = s% center_eps_burn_dRho(i)*(10d0**s% log_center_density)
         else if (c > c_log_eps_burn_offset) then
            i = c - c_log_eps_burn_offset
            val = safe_log10(abs(s% center_eps_burn(i))) ! abs is for photo
         else if (c > max_eps_nuc_offset) then
            i = c - max_eps_nuc_offset
            val = safe_log10(max_eps_nuc_log_x(s% net_iso(i)))
         else if (c > cz_top_max_offset) then
            i = c - cz_top_max_offset
            val = safe_log10(cz_top_max_log_x(s% net_iso(i)))
         else if (c > cz_max_offset) then
            i = c - cz_max_offset
            val = safe_log10(cz_max_log_x(s% net_iso(i)))
         else if (c > log_surface_xa_offset) then
            i = c - log_surface_xa_offset
            val = safe_log10(surface_avg_x(s,s% net_iso(i)))
         else if (c > log_center_xa_offset) then
            i = c - log_center_xa_offset
            val = safe_log10(central_avg_x(s,s% net_iso(i)))
         else if (c > log_average_xa_offset) then
            i = c - log_average_xa_offset
            val = safe_log10(star_avg_x(s,s% net_iso(i)))
         else if (c > log_total_mass_offset) then
            i = c - log_total_mass_offset
            val = safe_log10(star_avg_x(s,s% net_iso(i))*s% xmstar/Msun)
         else if (c > total_mass_offset) then
            i = c - total_mass_offset
            val = star_avg_x(s,s% net_iso(i))*s% xmstar/Msun
         else if (c > category_offset) then
            i = c - category_offset
            val = category_L(i)
         else if (c > average_xa_offset) then
            i = c - average_xa_offset
            val = star_avg_x(s,s% net_iso(i))
         elseif (c > surface_xa_offset) then
            i = c - surface_xa_offset
            val = surface_avg_x(s,s% net_iso(i))
         else if (c > center_xa_offset) then
            i = c - center_xa_offset
            val = central_avg_x(s,s% net_iso(i))
         else 
            
            select case(c)
         
            case(h_model_number)
               is_int_val = .true.
               int_val = s% model_number
            case(h_log_star_age)
               val = safe_log10(s% star_age)
            case(h_star_age)
               val = s% star_age
            case(h_star_mass)
               val = s% star_mass
            case(h_log_xmstar)
               val = safe_log10(s% xmstar)
            case(h_delta_mass)
               val = s% star_mass - s% initial_mass
            case(h_star_mdot)
               val = s% star_mdot
            case(h_log_abs_mdot)
               val = safe_log10(abs(s% star_mdot))

            case(h_mdot_timescale)
               !write(*,1) 'mdot_timescale: s% star_mdot', s% star_mdot
               !write(*,1) 'mdot_timescale: s% star_mass', s% star_mass
               val = s% star_mass/max(1d-99,abs(s% star_mdot))
               !write(*,1) 'mdot_timescale', val
               
            case(h_kh_div_mdot_timescales)
               val = s% kh_timescale/ &
                  (s% star_mass/max(1d-99,abs(s% star_mdot)))
            case(h_dlnR_dlnM)
               if (abs(s% star_mdot) > 1d-99) &
                  val = (s% lnR(1) - s% lnR_start(1)) / &
                     ((s% mstar - s% mstar_old)/(0.5d0*(s% mstar + s% mstar_old)))
            case(h_star_gravitational_mass)
               val = s% m_grav(1)/Msun
            case(h_star_mass_grav_div_mass)
               val = s% m_grav(1)/s% m(1)

            case(h_time_step)
               val = s% time_step
            case(h_e_thermal)
               val = sum(s% dm(1:s% nz)*s% T(1:s% nz)*s% cp(1:s% nz))
            case(h_log_dt)
               val = safe_log10(s% time_step)
            case(h_total_angular_momentum)
               val = s% total_angular_momentum
            case(h_log_total_angular_momentum)
               val = safe_log10(s% total_angular_momentum)
            case(h_num_zones)
               int_val = s% nz
               is_int_val = .true.
            case(h_num_retries)
               int_val = s% num_retries
               is_int_val = .true.
            case(h_num_backups)
               int_val = s% num_backups
               is_int_val = .true.
            case(h_num_jacobians)
               int_val = s% num_jacobians
               is_int_val = .true.
            case(h_total_num_jacobians)
               int_val = s% total_num_jacobians
               is_int_val = .true.
            case(h_h1_czb_mass)
               val = s% h1_czb_mass
            case(h_surf_c12_minus_o16)
               val = s% surface_c12 - s% surface_o16
            case(h_surf_num_c12_div_num_o16)
               val = (16d0/12d0)*s% surface_c12/max(1d-99,s% surface_o16)
            case(h_conv_mx1_top)
               val = s% conv_mx1_top
            case(h_conv_mx1_bot)
               val = s% conv_mx1_bot
            case(h_conv_mx2_top)
               val = s% conv_mx2_top
            case(h_conv_mx2_bot)
               val = s% conv_mx2_bot
            case(h_mx1_top)
               val = s% mx1_top
            case(h_mx1_bot)
               val = s% mx1_bot
            case(h_mx2_top)
               val = s% mx2_top
            case(h_mx2_bot)
               val = s% mx2_bot
            case(h_conv_mx1_top_r)
               val = s% conv_mx1_top_r
            case(h_conv_mx1_bot_r)
               val = s% conv_mx1_bot_r
            case(h_conv_mx2_top_r)
               val = s% conv_mx2_top_r
            case(h_conv_mx2_bot_r)
               val = s% conv_mx2_bot_r
            case(h_mx1_top_r)
               val = s% mx1_top_r
            case(h_mx1_bot_r)
               val = s% mx1_bot_r
            case(h_mx2_top_r)
               val = s% mx2_top_r
            case(h_mx2_bot_r)
               val = s% mx2_bot_r
            case(h_epsnuc_M_1)
               val = epsnuc_out(1)
            case(h_epsnuc_M_2)
               val = epsnuc_out(2)
            case(h_epsnuc_M_3)
               val = epsnuc_out(3)
            case(h_epsnuc_M_4)
               val = epsnuc_out(4)
            case(h_epsnuc_M_5)
               val = epsnuc_out(5)
            case(h_epsnuc_M_6)
               val = epsnuc_out(6)
            case(h_epsnuc_M_7)
               val = epsnuc_out(7)
            case(h_epsnuc_M_8)
               val = epsnuc_out(8)
            case(h_power_h_burn)
               val = s% power_h_burn
            case(h_log_LH)
               val = safe_log10(s% power_h_burn)
            case(h_power_he_burn)
               val = s% power_he_burn
            case(h_log_LHe)
               val = safe_log10(s% power_he_burn)
            case(h_log_LZ)
               power_photo = dot_product(s% dm(1:nz), s% eps_nuc_categories(i_rate,iphoto,1:nz))/Lsun
               val = safe_log10(s% power_nuc_burn - (power_photo + s% power_h_burn + s% power_he_burn))
            case(h_luminosity)
               val = s% L(1)/Lsun
            case(h_log_L)
               val = safe_log10(s% L(1)/Lsun)
            case(h_log_Lneu)
               val = safe_log10(s% power_neutrinos)
         
            case(h_log_Teff)
               val = safe_log10(s% Teff)
            case(h_effective_T)
               val = s% Teff
            case(h_photosphere_L)
               val = s% photosphere_L
            case(h_photosphere_r)
               val = s% photosphere_r
         
            case(h_radius_cm)
               val = s% R(1)
            case(h_log_R_cm)
               val = safe_log10(s% R(1))
            case(h_radius)
               val = s% R(1)/Rsun
            case(h_log_R)
               val = safe_log10(s% R(1)/Rsun)
         
            case(h_gravity)
               val = s% grav(1)
            case(h_log_g)
               val = safe_log10(s% grav(1))
               
            case(h_log_max_T)
               val = s% log_max_temperature
            case(h_log_cntr_T)
               val = s% log_center_temperature
            case(h_log_cntr_Rho)
               val = s% log_center_density
            case(h_log_cntr_P)
               val = s% log_center_pressure
            case(h_log_center_T)
               val = s% log_center_temperature
            case(h_log_center_Rho)
               val = s% log_center_density
            case(h_log_center_P)
               val = s% log_center_pressure
               
            case(h_max_T)
               val = 10d0**s% log_max_temperature
            case(h_center_T)
               val = 10d0**s% log_center_temperature
            case(h_center_Rho)
               val = 10d0**s% log_center_density
            case(h_center_P)
               val = 10d0**s% log_center_pressure
               
            case(h_v_surf)
               val = v_surf
            case(h_v_surf_div_v_kh)
               val = v_surf/(s% photosphere_r/s% kh_timescale)
            case(h_v_div_csound_surf)
               val = v_surf/csound_surf
            case(h_surface_accel_div_grav)
               if (s% dt > 0 .and. s% generations > 1) then
                  val = (v_surf - s% v_surf_old)/(s% dt*s% grav(1))
               else
                  val = 0
               end if
            case(h_log_L_div_Ledd)
               Ledd = eval_Ledd(s)
               val = safe_log10(s% L(1)/Ledd)
               
            case(h_lum_div_Ledd)
               Ledd = eval_Ledd(s)
               val = s% L(1)/Ledd
               
            case(h_gradT_excess_alpha)
               val = s% gradT_excess_alpha
            case(h_gradT_excess_min_beta)
               val = s% gradT_excess_min_beta
            case(h_gradT_excess_max_lambda)
               val = s% gradT_excess_max_lambda
               
            case(h_max_L_rad_div_Ledd)
               do k=1,s% nz
                  tmp = get_L_rad_div_Ledd(s,k)
                  if (tmp > val) val = tmp
               end do
               
            case(h_max_L_rad_div_Ledd_div_phi_Joss)
               do k=1,s% nz
                  tmp = get_L_rad_div_Ledd(s,k)
                  phi_Joss = get_phi_Joss(s,k)
                  if (tmp/phi_Joss > val) val = tmp/phi_Joss
               end do
               
            case(h_max_conv_dP_term)
               if (s% conv_dP_term_factor > 0) val = maxval(s% conv_dP_term(1:s% nz))

            case(h_surf_avg_j_rot)
               if (s% rotation_flag) val = s% j_rot_avg_surf
            case(h_surf_avg_omega)
               if (s% rotation_flag) val = s% omega_avg_surf
            case(h_surf_avg_omega_crit)
               if (s% rotation_flag) val = s% omega_crit_avg_surf
            case(h_surf_avg_omega_div_omega_crit)
               if (s% rotation_flag) val = s% omega_div_omega_crit_avg_surf 
               
            case(h_surf_avg_v_rot)
               if (s% rotation_flag) val = s% v_rot_avg_surf*1d-5 ! km/sec
            case(h_surf_avg_v_crit)
               if (s% rotation_flag) val = s% v_crit_avg_surf*1d-5 ! km/sec
            case(h_surf_avg_v_div_v_crit)
               if (s% rotation_flag) val = s% v_div_v_crit_avg_surf 
               
            case(h_surf_avg_L_div_Ledd)
               val = s% L_div_Ledd_avg_surf 
            case(h_surf_avg_opacity)
               val = s% opacity_avg_surf 
            case(h_surf_avg_logT)
               val = s% logT_avg_surf 
            case(h_surf_avg_logRho)
               val = s% logRho_avg_surf 
               
               
            case (h_kh_mdot_limit)
               if (s% rotation_flag) val = &
                  s% rotational_mdot_kh_fac*s% star_mass/s% kh_timescale
            case (h_log_rotational_mdot_boost)
               if (s% rotation_flag) val = safe_log10(s% rotational_mdot_boost)
            case (h_rotational_mdot_boost)
               if (s% rotation_flag) val = s% rotational_mdot_boost

            case(h_min_Pgas_div_P)
               val = minval(s% Pgas(1:nz)/s% P(1:nz))
            case(h_center_degeneracy)
               val = s% center_degeneracy
               
            case(h_log_center_eps_nuc)
               val = safe_log10(s% center_eps_nuc)
            case(h_center_eps_nuc)
               val = s% center_eps_nuc
            case(h_d_center_eps_nuc_dlnT)
               val = s% d_center_eps_nuc_dlnT
            case(h_d_center_eps_nuc_dlnd)
               val = s% d_center_eps_nuc_dlnd
               
            case(h_center_dlogT)
               val = s% dt*center_value(s, s% dlnT_dt)/ln10
            case(h_center_dlogRho)
               val = s% dt*center_value(s, s% dlnd_dt)/ln10
               
            case(h_center_dlnT_dt)
               val = center_value(s, s% dlnT_dt)
            case(h_center_dlnT_dt_prev_step)
               if (s% have_prev_step_info) &
                  val = center_value(s, s% dlnT_dt_prev_step)
                  
            case(h_center_dlnd_dt)
               val = center_value(s, s% dlnd_dt)
            case(h_center_dlnd_dt_prev_step)
               if (s% have_prev_step_info) &
                  val = center_value(s, s% dlnd_dt_prev_step)

            case(h_center_dL_dm)
               val = s% center_dL_dm
            case(h_center_non_nuc_neu)
               val = s% center_non_nuc_neu
            case(h_center_eps_grav)
               val = s% center_eps_grav
            case(h_center_gamma)
               val = s% center_gamma
            case(h_center_zbar)
               val = s% center_zbar
            case(h_center_abar)
               val = s% center_abar
            case(h_center_mu)
               val = s% center_mu
            case(h_center_ye)
               val = s% center_ye
            case(h_center_entropy)
               val = s% center_entropy
            case(h_compactness_parameter)
               if (s% m(1) > 2.5*Msun) then
                  do k=s% nz-1, 1, -1
                     if (s% m(k) > 2.5*Msun) exit
                  end do
                  r = s% r(k+1) + (s% r(k)-s% r(k+1))*(2.5*Msun - s% m(k+1))/s% dm(k)
                  val = 2.5/(r/1d8)
               end if
            case(h_iron_core_infall)
               val = s% iron_core_infall*1d-5 ! convert to km/sec
            case(h_center_omega)
               val = s% center_omega
            case(h_center_omega_div_omega_crit)
               val = s% center_omega_div_omega_crit

            case(h_surf_r_equatorial_div_r)
               val = s% r_equatorial(1)/s% r(1)
            case(h_surf_r_polar_div_r)
               val = s% r_polar(1)/s% r(1)
         
            case(h_he_core_mass)
               val = s% he_core_mass
            case(h_he_core_radius)
               val = s% he_core_radius
            case(h_he_core_lgT)
               val = s% he_core_lgT
            case(h_he_core_lgRho)
               val = s% he_core_lgRho
            case(h_he_core_L)
               val = s% he_core_L
            case(h_he_core_v)
               val = s% he_core_v
            case(h_he_core_omega)
               val = s% he_core_omega
            case(h_he_core_omega_div_omega_crit)
               val = s% he_core_omega_div_omega_crit
         
            case(h_c_core_mass)
               val = s% c_core_mass
            case(h_c_core_radius)
               val = s% c_core_radius
            case(h_c_core_lgT)
               val = s% c_core_lgT
            case(h_c_core_lgRho)
               val = s% c_core_lgRho
            case(h_c_core_L)
               val = s% c_core_L
            case(h_c_core_v)
               val = s% c_core_v
            case(h_c_core_omega)
               val = s% c_core_omega
            case(h_c_core_omega_div_omega_crit)
               val = s% c_core_omega_div_omega_crit
         
            case(h_o_core_mass)
               val = s% o_core_mass
            case(h_o_core_radius)
               val = s% o_core_radius
            case(h_o_core_lgT)
               val = s% o_core_lgT
            case(h_o_core_lgRho)
               val = s% o_core_lgRho
            case(h_o_core_L)
               val = s% o_core_L
            case(h_o_core_v)
               val = s% o_core_v
            case(h_o_core_omega)
               val = s% o_core_omega
            case(h_o_core_omega_div_omega_crit)
               val = s% o_core_omega_div_omega_crit
         
            case(h_si_core_mass)
               val = s% si_core_mass
            case(h_si_core_radius)
               val = s% si_core_radius
            case(h_si_core_lgT)
               val = s% si_core_lgT
            case(h_si_core_lgRho)
               val = s% si_core_lgRho
            case(h_si_core_L)
               val = s% si_core_L
            case(h_si_core_v)
               val = s% si_core_v
            case(h_si_core_omega)
               val = s% si_core_omega
            case(h_si_core_omega_div_omega_crit)
               val = s% si_core_omega_div_omega_crit
               
            case(h_fe_core_mass)
               val = s% fe_core_mass
            case(h_fe_core_radius)
               val = s% fe_core_radius
            case(h_fe_core_lgT)
               val = s% fe_core_lgT
            case(h_fe_core_lgRho)
               val = s% fe_core_lgRho
            case(h_fe_core_L)
               val = s% fe_core_L
            case(h_fe_core_v)
               val = s% fe_core_v
            case(h_fe_core_omega)
               val = s% fe_core_omega
            case(h_fe_core_omega_div_omega_crit)
               val = s% fe_core_omega_div_omega_crit
         
            case(h_envelope_mass)
               val = s% star_mass - s% he_core_mass
            case(h_envelope_fraction_left)
               val = envelope_fraction_left
            case(h_tau10_mass)
               val = s% tau10_mass
            case(h_tau10_radius)
               val = s% tau10_radius
            case(h_tau10_lgP)
               val = s% tau10_lgP
            case(h_tau10_T)
               val = 10**s% tau10_lgT
            case(h_tau10_lgT)
               val = s% tau10_lgT
            case(h_tau10_lgRho)
               val = s% tau10_lgRho
            case(h_tau10_L)
               val = s% tau10_L
            case(h_tau100_mass)
               val = s% tau100_mass
            case(h_tau100_radius)
               val = s% tau100_radius
            case(h_tau100_lgP)
               val = s% tau100_lgP
            case(h_tau100_T)
               val = 10**s% tau100_lgT
            case(h_tau100_lgT)
               val = s% tau100_lgT
            case(h_tau100_lgRho)
               val = s% tau100_lgRho
            case(h_tau100_L)
               val = s% tau100_L
            case(h_dynamic_timescale)
               val = s% dynamic_timescale
            case(h_kh_timescale)
               val = s% kh_timescale
            case(h_nuc_timescale)
               val = s% nuc_timescale
            case(h_log_chem_timescale)
               val = safe_log10(s% chem_timescale)
            case(h_log_cell_collapse_timescale)
               val = safe_log10(s% cell_collapse_timescale)
            case(h_log_chem_timescale_div_time_step)
               val = safe_log10(s% chem_timescale/s% time_step)
            case(h_eps_grav_integral)
               val = dot_product(s% dm(1:nz), s% eps_grav(1:nz))/Lsun
            case(h_extra_L)
               val = dot_product(s% dm(1:nz), s% extra_heat(1:nz))/Lsun
            case(h_log_extra_L)
               val = safe_log10(dot_product(s% dm(1:nz), s% extra_heat(1:nz))/Lsun)
            case(h_log_abs_Lgrav)
               val = safe_log10(abs(dot_product(s% dm(1:nz), s% eps_grav(1:nz))/Lsun))
            case(h_log_Lnuc)
               power_photo = dot_product(s% dm(1:nz), s% eps_nuc_categories(i_rate,iphoto,1:nz))/Lsun
               val = safe_log10(s% power_nuc_burn - power_photo)
            case(h_log_Lnuc_sub_log_L)
               power_photo = dot_product(s% dm(1:nz), s% eps_nuc_categories(i_rate,iphoto,1:nz))/Lsun
               val = safe_log10(s% power_nuc_burn - power_photo)
               val = val - safe_log10(s% L(1)/Lsun)
            case(h_mass_loc_of_max_eps_nuc)
               k = maxloc(s% eps_nuc(1:nz), dim=1)
               val = (s% m(k) - s% dm(k)/2)/Msun
            case(h_mass_ext_to_max_eps_nuc)
               k = maxloc(s% eps_nuc(1:nz), dim=1)
               val = (1d0 - s% q(k) + 0.5d0*s% dq(k))*s% xmstar/Msun
         
            case(h_trace_mass_location)
               val = s% trace_mass_location
            case(h_trace_mass_radius)
               val = s% trace_mass_radius
            case(h_trace_mass_lgT)
               val = s% trace_mass_lgT
            case(h_trace_mass_lgRho)
               val = s% trace_mass_lgRho
            case(h_trace_mass_L)
               val = s% trace_mass_L
            case(h_trace_mass_v)
               val = s% trace_mass_v
            case(h_trace_mass_omega)
               val = s% trace_mass_omega
            case(h_trace_mass_omega_div_omega_crit)
               val = s% trace_mass_omega_div_omega_crit

            case(h_trace_mass_lgP)
               val = s% trace_mass_lgP
            case(h_trace_mass_g)
               val = s% trace_mass_g
            case(h_trace_mass_X)
               val = s% trace_mass_X
            case(h_trace_mass_Y)
               val = s% trace_mass_Y
            case(h_trace_mass_edv_H)
               val = s% trace_mass_edv_H
            case(h_trace_mass_edv_He)
               val = s% trace_mass_edv_He
            case(h_trace_mass_scale_height)
               val = s% trace_mass_scale_height
            case(h_trace_mass_dlnX_dr)
               val = s% trace_mass_dlnX_dr
            case(h_trace_mass_dlnY_dr)
               val = s% trace_mass_dlnY_dr
            case(h_trace_mass_dlnRho_dr)
               val = s% trace_mass_dlnRho_dr
         
            case(h_max_T_lgT)
               val = s% max_T_lgT
            case(h_max_T_mass)
               val = s% max_T_mass
            case(h_max_T_radius)
               val = s% max_T_radius
            case(h_max_T_lgRho)
               val = s% max_T_lgRho
            case(h_max_T_L)
               val = s% max_T_L
            case(h_max_T_eps_nuc)
               val = s% max_T_eps_nuc
         
            case(h_max_eps_h)
               val = s% max_eps_h
            case(h_max_eps_h_lgT)
               val = s% max_eps_h_lgT
            case(h_max_eps_h_lgRho)
               val = s% max_eps_h_lgRho
            case(h_max_eps_h_m)
               val = s% max_eps_h_m/Msun
            case(h_max_eps_h_xm)
               val = (s% mstar - s% max_eps_h_m)/Msun
            case(h_max_eps_h_lgR)
               val = s% max_eps_h_lgR
            case(h_max_eps_h_lgP)
               val = s% max_eps_h_lgP
            case(h_max_eps_h_opacity)
               val = s% max_eps_h_opacity
         
            case(h_max_eps_he)
               val = s% max_eps_he
            case(h_max_eps_he_lgT)
               val = s% max_eps_he_lgT
            case(h_max_eps_he_lgRho)
               val = s% max_eps_he_lgRho
            case(h_max_eps_he_m)
               val = s% max_eps_he_m/Msun
            case(h_max_eps_he_xm)
               val = (s% mstar - s% max_eps_he_m)/Msun
            case(h_max_eps_he_lgR)
               val = s% max_eps_he_lgR
            case(h_max_eps_he_lgP)
               val = s% max_eps_he_lgP
            case(h_max_eps_he_opacity)
               val = s% max_eps_he_opacity
         
            case(h_max_eps_z)
               val = s% max_eps_z
            case(h_max_eps_z_lgT)
               val = s% max_eps_z_lgT
            case(h_max_eps_z_lgRho)
               val = s% max_eps_z_lgRho
            case(h_max_eps_z_m)
               val = s% max_eps_z_m/Msun
            case(h_max_eps_z_xm)
               val = (s% mstar - s% max_eps_z_m)/Msun
            case(h_max_eps_z_lgR)
               val = s% max_eps_z_lgR
            case(h_max_eps_z_lgP)
               val = s% max_eps_z_lgP
            case(h_max_eps_z_opacity)
               val = s% max_eps_z_opacity
         
            case(h_max_eps_nuc)
               val = s% max_eps_nuc
            case(h_max_eps_nuc_lgT)
               val = s% max_eps_nuc_lgT
            case(h_max_eps_nuc_lgRho)
               val = s% max_eps_nuc_lgRho
            case(h_max_eps_nuc_m)
               val = s% max_eps_nuc_m/Msun
            case(h_max_eps_nuc_xm)
               val = (s% mstar - s% max_eps_nuc_m)/Msun
            case(h_max_eps_nuc_lgR)
               val = s% max_eps_nuc_lgR
            case(h_max_eps_nuc_lgP)
               val = s% max_eps_nuc_lgP
            case(h_max_eps_nuc_opacity)
               val = s% max_eps_nuc_opacity
            case(h_max_eps_nuc_cp)
               val = s% max_eps_nuc_cp
            case(h_max_eps_nuc_t_heat)
               if (s% max_eps_nuc >= 0) then
                  val = s% max_eps_nuc_cp*10**s% max_eps_nuc_lgT/s% max_eps_nuc
               else
                  val = 0
               end if

            case(h_max_eps_nuc_csound)
               val = s% csound(s% max_eps_nuc_k)
            case(h_max_eps_nuc_pi_r_div_cs)
               val = pi*s% r(s% max_eps_nuc_k)/s% csound(s% max_eps_nuc_k)
            case(h_max_eps_nuc_H)
               val = s% scale_height(s% max_eps_nuc_k)
            case(h_max_eps_nuc_H_div_cs)
               val = s% scale_height(s% max_eps_nuc_k)/s% csound(s% max_eps_nuc_k)

            case(h_log_surf_opacity)
               val = safe_log10(s% opacity(1))
            case(h_log_surf_density)
               val = s% lnd(1)/ln10
            case(h_surface_temperature)
               val = s% T_surf
            case(h_log_surf_temperature)
               val = safe_log10(s% T_surf)
            case(h_surface_optical_depth)
               val = s% tau_base*s% tau_factor
            case(h_log_surf_optical_depth)
               val = safe_log10(s% tau_base*s% tau_factor)
            case(h_log_surf_P)
               val = safe_log10(s% P_surf)
            case(h_log_surf_pressure)
               val = safe_log10(s% P_surf)
            case(h_log_surf_z)
               val = 0
               if (s% net_iso(ih1) /= 0) val = val + s% xa(s% net_iso(ih1),1)
               if (s% net_iso(ih2) /= 0) val = val + s% xa(s% net_iso(ih2),1)
               if (s% net_iso(ihe3) /= 0) val = val + s% xa(s% net_iso(ihe3),1)
               if (s% net_iso(ihe4) /= 0) val = val + s% xa(s% net_iso(ihe4),1)
               val = safe_log10(1d0 - val)
               
            case(h_log_Ledd)
               val = safe_log10(eval_Ledd(s)/Lsun)
         
            case(h_dt_Courant)
               val = dt_Courant(s)
            case(h_log_dt_Courant)
               val = safe_log10(dt_Courant(s))
            case(h_dt_Courant_yr)
               val = dt_Courant(s)/secyer
            case(h_log_dt_Courant_yr)
               val = safe_log10(dt_Courant(s)/secyer)
            case(h_dt_div_dt_Courant)
               val = s% dt/dt_Courant(s)
            case(h_log_dt_div_dt_Courant)
               val = safe_log10(s% dt/dt_Courant(s))

            case(h_cz_bot_mass)
               if (s% largest_conv_mixing_region == 0) then
                  val = 0
               else
                  k = s% mixing_region_bottom(s% largest_conv_mixing_region)
                  if (k == s% nz) then
                     val = s% M_center/Msun
                  else
                     val = s% m(k)/Msun
                  end if
               end if
            case(h_cz_mass)
               if (s% largest_conv_mixing_region == 0) then
                  val = 0
               else
                  k = s% mixing_region_bottom(s% largest_conv_mixing_region)
                  val = s% m(k)/Msun
               end if
            case(h_cz_log_xmass)
               if (s% largest_conv_mixing_region == 0) then
                  val = -99
               else
                  k = s% mixing_region_bottom(s% largest_conv_mixing_region)
                  val = safe_log10(s% xmstar*sum(s% dq(1:k-1)))
               end if
            case(h_cz_log_xmsun)
               if (s% largest_conv_mixing_region == 0) then
                  val = -99
               else
                  k = s% mixing_region_bottom(s% largest_conv_mixing_region)
                  val = safe_log10(s% xmstar*sum(s% dq(1:k-1))/Msun)
               end if
            case(h_cz_xm)
               if (s% largest_conv_mixing_region == 0) then
                  val = 0
               else
                  k = s% mixing_region_bottom(s% largest_conv_mixing_region)
                  val = s% xmstar*sum(s% dq(1:k-1))/Msun
               end if
            case(h_cz_logT)
               if (s% largest_conv_mixing_region == 0) then
                  val = 0
               else
                  k = s% mixing_region_bottom(s% largest_conv_mixing_region)
                  val = s% lnT(k)/ln10
               end if
            case(h_cz_logRho)
               if (s% largest_conv_mixing_region == 0) then
                  val = 0
               else
                  k = s% mixing_region_bottom(s% largest_conv_mixing_region)
                  val = s% lnd(k)/ln10
               end if
            case(h_cz_logP)
               if (s% largest_conv_mixing_region == 0) then
                  val = 0
               else
                  k = s% mixing_region_bottom(s% largest_conv_mixing_region)
                  val = s% lnP(k)/ln10
               end if
            case(h_cz_log_column_depth)
               if (s% largest_conv_mixing_region == 0) then
                  val = 0
               else
                  k = s% mixing_region_bottom(s% largest_conv_mixing_region)
                  val = safe_log10(s% xmstar*sum(s% dq(1:k-1))/(4*pi*s% r(k)**2))
               end if
            case(h_cz_log_radial_depth)
               if (s% largest_conv_mixing_region == 0) then
                  val = 0
               else
                  k = s% mixing_region_bottom(s% largest_conv_mixing_region)
                  val = safe_log10(s% r(1) - s% r(k))
               end if
            case(h_cz_luminosity)
               if (s% largest_conv_mixing_region == 0) then
                  val = 0
               else
                  k = s% mixing_region_bottom(s% largest_conv_mixing_region)
                  val = s% L(k)/Lsun
               end if
            case(h_cz_log_tau)
               if (s% largest_conv_mixing_region == 0) then
                  val = 0
               else
                  k = s% mixing_region_bottom(s% largest_conv_mixing_region)
                  val = safe_log10(s% tau(k))
               end if
            case(h_cz_opacity)
               if (s% largest_conv_mixing_region == 0) then
                  val = 0
               else
                  k = s% mixing_region_bottom(s% largest_conv_mixing_region)
                  val = s% opacity(k)
               end if
            case(h_cz_log_eps_nuc)
               if (s% largest_conv_mixing_region == 0) then
                  val = 0
               else
                  k = s% mixing_region_bottom(s% largest_conv_mixing_region)
                  val = safe_log10(s% eps_nuc(k))
               end if
            case(h_cz_t_heat)
               if (s% largest_conv_mixing_region == 0) then
                  val = 0
               else
                  k = s% mixing_region_bottom(s% largest_conv_mixing_region)
                  if (s% eps_nuc(k) <= 0) then
                     val = 1d99
                  else
                     val = s% Cp(k)*s% T(k)/s% eps_nuc(k)
                  end if
               end if
            case(h_cz_eta)
               if (s% largest_conv_mixing_region == 0) then
                  val = 0
               else
                  k = s% mixing_region_bottom(s% largest_conv_mixing_region)
                  val = s% eta(k)
               end if
            case(h_cz_csound)
               if (s% largest_conv_mixing_region == 0) then
                  val = 0
               else
                  k = s% mixing_region_bottom(s% largest_conv_mixing_region)
                  val = s% csound(k)
               end if
            case(h_cz_scale_height)
               if (s% largest_conv_mixing_region == 0) then
                  val = 0
               else
                  k = s% mixing_region_bottom(s% largest_conv_mixing_region)
                  val = s% scale_height(k)
               end if
            case(h_cz_grav)
               if (s% largest_conv_mixing_region == 0) then
                  val = 0
               else
                  k = s% mixing_region_bottom(s% largest_conv_mixing_region)
                  val = s% grav(k)
               end if
            case(h_cz_bot_radius)
               if (s% largest_conv_mixing_region == 0) then
                  val = 0
               else
                  k = s% mixing_region_bottom(s% largest_conv_mixing_region)
                  if (k == s% nz) then
                     val = s% R_center/Rsun
                  else
                     val = s% R(k)/Rsun
                  end if
               end if
            case(h_cz_zone)
               if (s% largest_conv_mixing_region == 0) then
                  k = 0
               else
                  k = s% mixing_region_bottom(s% largest_conv_mixing_region)
               end if
               int_val = k
               is_int_val = .true.
            case(h_cz_omega)
               if (s% largest_conv_mixing_region == 0 .or. .not. s% rotation_flag) then
                  val = 0
               else
                  k = s% mixing_region_bottom(s% largest_conv_mixing_region)
                  val = s% omega(k)
               end if
            case(h_cz_omega_div_omega_crit)
               if (s% largest_conv_mixing_region == 0 .or. .not. s% rotation_flag) then
                  val = 0
               else
                  k = s% mixing_region_bottom(s% largest_conv_mixing_region)
                  val = s% omega(k)/omega_crit(s,k)
               end if

            case(h_cz_top_mass)
               if (s% largest_conv_mixing_region == 0) then
                  val = 0
               else
                  k = s% mixing_region_top(s% largest_conv_mixing_region)
                  val = s% m(k)/Msun
               end if
            case(h_cz_top_log_xmass)
               if (s% largest_conv_mixing_region == 0) then
                  val = -99
               else
                  k = s% mixing_region_top(s% largest_conv_mixing_region)
                  val = safe_log10(s% xmstar*sum(s% dq(1:k-1)))
               end if
            case(h_cz_top_log_xmsun)
               if (s% largest_conv_mixing_region == 0) then
                  val = -99
               else
                  k = s% mixing_region_top(s% largest_conv_mixing_region)
                  val = safe_log10(s% xmstar*sum(s% dq(1:k-1))/Msun)
               end if
            case(h_cz_top_xm)
               if (s% largest_conv_mixing_region == 0) then
                  val = 0
               else
                  k = s% mixing_region_top(s% largest_conv_mixing_region)
                  val = s% xmstar*sum(s% dq(1:k-1))/Msun
               end if
            case(h_cz_top_logT)
               if (s% largest_conv_mixing_region == 0) then
                  val = 0
               else
                  k = s% mixing_region_top(s% largest_conv_mixing_region)
                  val = s% lnT(k)/ln10
               end if
            case(h_cz_top_logRho)
               if (s% largest_conv_mixing_region == 0) then
                  val = 0
               else
                  k = s% mixing_region_top(s% largest_conv_mixing_region)
                  val = s% lnd(k)/ln10
               end if
            case(h_cz_top_logP)
               if (s% largest_conv_mixing_region == 0) then
                  val = 0
               else
                  k = s% mixing_region_top(s% largest_conv_mixing_region)
                  val = s% lnP(k)/ln10
               end if
            case(h_cz_top_log_column_depth)
               if (s% largest_conv_mixing_region == 0) then
                  val = 0
               else
                  k = s% mixing_region_top(s% largest_conv_mixing_region)
                  val = safe_log10(s% xmstar*sum(s% dq(1:k-1))/(4*pi*s% r(k)**2))
               end if
            case(h_cz_top_log_radial_depth)
               if (s% largest_conv_mixing_region == 0) then
                  val = 0
               else
                  k = s% mixing_region_top(s% largest_conv_mixing_region)
                  val = safe_log10(s% r(1) - s% r(k))
               end if
            case(h_cz_top_luminosity)
               if (s% largest_conv_mixing_region == 0) then
                  val = 0
               else
                  k = s% mixing_region_top(s% largest_conv_mixing_region)
                  val = s% L(k)/Lsun
               end if
            case(h_cz_top_log_tau)
               if (s% largest_conv_mixing_region == 0) then
                  val = 0
               else
                  k = s% mixing_region_top(s% largest_conv_mixing_region)
                  val = safe_log10(s% tau(k))
               end if
            case(h_cz_top_opacity)
               if (s% largest_conv_mixing_region == 0) then
                  val = 0
               else
                  k = s% mixing_region_top(s% largest_conv_mixing_region)
                  val = s% opacity(k)
               end if
            case(h_cz_top_log_eps_nuc)
               if (s% largest_conv_mixing_region == 0) then
                  val = 0
               else
                  k = s% mixing_region_top(s% largest_conv_mixing_region)
                  val = safe_log10(s% eps_nuc(k))
               end if
            case(h_cz_top_t_heat)
               if (s% largest_conv_mixing_region == 0) then
                  val = 0
               else
                  k = s% mixing_region_top(s% largest_conv_mixing_region)
                  if (s% eps_nuc(k) <= 0) then
                     val = 1d99
                  else
                     val = s% Cp(k)*s% T(k)/s% eps_nuc(k)
                  end if
               end if
            case(h_cz_top_eta)
               if (s% largest_conv_mixing_region == 0) then
                  val = 0
               else
                  k = s% mixing_region_top(s% largest_conv_mixing_region)
                  val = s% eta(k)
               end if
            case(h_cz_top_csound)
               if (s% largest_conv_mixing_region == 0) then
                  val = 0
               else
                  k = s% mixing_region_top(s% largest_conv_mixing_region)
                  val = s% csound(k)
               end if
            case(h_cz_top_scale_height)
               if (s% largest_conv_mixing_region == 0) then
                  val = 0
               else
                  k = s% mixing_region_top(s% largest_conv_mixing_region)
                  val = s% scale_height(k)
               end if
            case(h_cz_top_grav)
               if (s% largest_conv_mixing_region == 0) then
                  val = 0
               else
                  k = s% mixing_region_top(s% largest_conv_mixing_region)
                  val = s% grav(k)
               end if
            case(h_cz_top_radius)
               if (s% largest_conv_mixing_region == 0) then
                  val = 0
               else
                  k = s% mixing_region_top(s% largest_conv_mixing_region)
                  val = s% R(k)/Rsun
               end if
            case(h_mass_conv_core)
               val = s% mass_conv_core
            case(h_cz_top_zone_logdq)
               if (s% largest_conv_mixing_region == 0) then
                  k = 0
                  val = 0
               else
                  k = s% mixing_region_top(s% largest_conv_mixing_region)
                  val = safe_log10(s% dq(k))
               end if
            case(h_cz_top_zone)
               if (s% largest_conv_mixing_region == 0) then
                  k = 0
               else
                  k = s% mixing_region_top(s% largest_conv_mixing_region)
               end if
               int_val = k
               is_int_val = .true.
            case(h_cz_top_omega)
               if (s% largest_conv_mixing_region == 0 .or. .not. s% rotation_flag) then
                  val = 0
               else
                  k = s% mixing_region_top(s% largest_conv_mixing_region)
                  val = s% omega(k)
               end if
            case(h_cz_top_omega_div_omega_crit)
               if (s% largest_conv_mixing_region == 0 .or. .not. s% rotation_flag) then
                  val = 0
               else
                  k = s% mixing_region_top(s% largest_conv_mixing_region)
                  val = s% omega(k)/omega_crit(s,k)
               end if
         
            case(h_max_gradT_div_grada)
               val = 0
               do k = 1, nz
                  if (s% gradT(k)/s% grada_at_face(k) > val) &
                     val = s% gradT(k)/s% grada_at_face(k)
               end do
            case(h_max_gradT_sub_grada)
               val = 0
               do k = 1, nz
                  if (s% gradT(k) - s% grada_at_face(k) > val) &
                     val = s% gradT(k) - s% grada_at_face(k)
               end do
            case(h_min_log_mlt_Gamma)
               val = 1d99
               do k = 1, nz
                  if (s% mlt_Gamma(k) > 0 .and. s% mlt_Gamma(k) < val) val = s% mlt_Gamma(k)
               end do
               val = safe_log10(val)
         
            case(h_max_conv_vel_div_csound)
               val = 0
               do k = 1, nz
                  if (s% q(k) > s% max_conv_vel_div_csound_maxq) cycle
                  if (s% conv_vel(k)/s% csound(k) > val) val = s% conv_vel(k)/s% csound(k)
               end do
         
            case(h_min_t_eddy)
               val = 1d99
               do k = 1, nz
                  if (s% conv_vel(k) <= 0) cycle
                  if (s% scale_height(k)/s% conv_vel(k) < val) &
                     val = s% scale_height(k)/s% conv_vel(k)
               end do
         
            case(h_max_x_expected)
               val = s% max_x_expected
            case(h_min_x_expected)
               val = s% min_x_expected
         
            case(h_elapsed_time)
               val = s% total_elapsed_time
         
            case(h_delta_nu)
               val = 1d6/(2*s% photosphere_acoustic_r)
            case(h_delta_Pg)
               val = s% delta_Pg
            case(h_log_delta_Pg)
               val = safe_log10(s% delta_Pg)
            case(h_nu_max)
               val = s% nu_max
            case(h_nu_max_3_4th_div_delta_nu)
               val = s% nu_max**0.75d0/(1d6/(2*s% photosphere_acoustic_r))
            case(h_acoustic_cutoff)
               val = s% acoustic_cutoff
            case(h_acoustic_radius)
               val = s% photosphere_acoustic_r
            case(h_gs_per_delta_nu)
               if (s% nu_max <= 0 .or. s% delta_Pg <= 0) then
                  val = 0
               else
                  val = 1d6/(2*s% photosphere_acoustic_r) ! delta_nu
                  val = 1d6*val/(s% nu_max**2*s% delta_Pg)
               end if
            case(h_ng_for_nu_max)
               if (s% nu_max <= 0 .or. s% delta_Pg <= 0) then
                  val = 0
               else
                  val = 1d6/(s% nu_max*s% delta_Pg)
               end if

            case(h_int_k_r_dr_nu_max_Sl1)
               val = get_int_k_r_dr(s,1,1.0d0)
            case(h_int_k_r_dr_2pt0_nu_max_Sl1)
               val = get_int_k_r_dr(s,1,2d0)
            case(h_int_k_r_dr_0pt5_nu_max_Sl1)
               val = get_int_k_r_dr(s,1,0.5d0)
            case(h_int_k_r_dr_nu_max_Sl2)
               val = get_int_k_r_dr(s,2,1.0d0)
            case(h_int_k_r_dr_2pt0_nu_max_Sl2)
               val = get_int_k_r_dr(s,2,2d0)
            case(h_int_k_r_dr_0pt5_nu_max_Sl2)
               val = get_int_k_r_dr(s,2,0.5d0)
            case(h_int_k_r_dr_nu_max_Sl3)
               val = get_int_k_r_dr(s,3,1.0d0)
            case(h_int_k_r_dr_2pt0_nu_max_Sl3)
               val = get_int_k_r_dr(s,3,2d0)
            case(h_int_k_r_dr_0pt5_nu_max_Sl3)
               val = get_int_k_r_dr(s,3,0.5d0)
                        
            case (h_op_split_diff)
               val = s% op_split_diff

            case (h_min_L)
               val = minval(s% L(1:nz))/Lsun
            case (h_min_dL_dm)
               val = minval(s% dL_dm(1:nz))
            case (h_min_dL_dm_m)
               val = s% m(minloc(s% dL_dm(1:nz),dim=1))/Msun

            case (h_k_const_mass)
               int_val = s% k_const_mass
               is_int_val = .true.
            case (h_q_const_mass)
               val = s% q(s% k_const_mass)
            case (h_logxq_const_mass)
               val = safe_log10(sum(s% dq(1:s% k_const_mass-1)))

            case (h_k_below_just_added)
               int_val = s% k_below_just_added
               is_int_val = .true.
            case (h_q_below_just_added)
               val = s% q(s% k_below_just_added)
            case (h_logxq_below_just_added)
               val = safe_log10(sum(s% dq(1:s% k_below_just_added-1)))
               
            case (h_k_below_recently_added)
               int_val = s% k_below_recently_added
               is_int_val = .true.
            case (h_q_below_recently_added)
               val = s% q(s% k_below_recently_added)
            case (h_logxq_below_recently_added)
               val = safe_log10(sum(s% dq(1:s% k_below_recently_added-1)))
               
            case (h_k_thermaltime_eq_accretiontime)
               int_val = s% k_thermaltime_eq_accretiontime
               is_int_val = .true.
            case (h_q_thermaltime_eq_accretiontime)
               val = s% q(s% k_thermaltime_eq_accretiontime)
            case (h_logxq_thermaltime_eq_accretiontime)
               val = safe_log10(sum(s% dq(1:s% k_thermaltime_eq_accretiontime-1)))
               
            case (h_k_below_mdot_eps_grav)
               int_val = s% k_below_mdot_eps_grav
               is_int_val = .true.
            case (h_q_below_mdot_eps_grav)
               val = s% q(s% k_below_mdot_eps_grav)
            case (h_logxq_below_mdot_eps_grav)
               val = safe_log10(sum(s% dq(1:s% k_below_mdot_eps_grav-1)))


            case (h_operator_coupling_choice)
               int_val = s% operator_coupling_choice
               is_int_val = .true.

            case (h_diffusion_solver_steps)
               int_val = s% num_diffusion_solver_steps
               is_int_val = .true.

            case (h_diffusion_solver_iters)
               int_val = s% num_diffusion_solver_iters
               is_int_val = .true.

            case (h_num_burn_max_iters)
               int_val = s% num_burn_max_iters
               is_int_val = .true.

            case (h_version_number)
               int_val = version_number
               is_int_val = .true.
         
            case default
               ierr = -1
         
            end select
         
         end if
         
         
         contains
         
         
         real(dp) function max_eps_nuc_log_x(j)
            integer, intent(in) :: j
            real(dp) :: sum_x, sum_dq
            integer :: k
            max_eps_nuc_log_x = 0
            if (j == 0) return
            k = maxloc(s% eps_nuc, dim=1)
            if (k < 1 .or. k > s% nz) return
            max_eps_nuc_log_x = s% xa(j,k)
         end function max_eps_nuc_log_x 


         real(dp) function cz_top_max_log_x(j)
            integer, intent(in) :: j
            real(dp) :: sum_x, sum_dq
            integer :: k
            cz_top_max_log_x = 0
            if (s% largest_conv_mixing_region == 0) return
            k = s% mixing_region_top(s% largest_conv_mixing_region)
            if (j == 0 .or. k <= 1) return
            cz_top_max_log_x = s% xa(j,k-1)
         end function cz_top_max_log_x         


         real(dp) function cz_max_log_x(j)
            integer, intent(in) :: j
            real(dp) :: sum_x, sum_dq
            integer :: k
            cz_max_log_x = 0
            if (s% largest_conv_mixing_region == 0) return
            k = s% mixing_region_bottom(s% largest_conv_mixing_region)
            if (j == 0 .or. k <= 1) return
            cz_max_log_x = s% xa(j,k-1)
         end function cz_max_log_x         


         real(dp) function category_L(i)
            integer, intent(in) :: i
            if (i == 0) then
               category_L = 0
               return
            end if
            category_L = &
               safe_log10(dot_product(s% eps_nuc_categories(i_rate,i,1:nz),s% dm(1:nz))/Lsun)
         end function category_L         


         real(dp) function reaction_eps_nuc(i)
            integer, intent(in) :: i
            if (i == 0) then
               reaction_eps_nuc = 0
               return
            end if
            reaction_eps_nuc = &
               safe_log10(dot_product(s% reaction_eps_nuc(i_rate,i,1:nz),s% dm(1:nz))/Lsun)
         end function reaction_eps_nuc         
      
      end subroutine history_getval
      
      
      real(dp) function get_int_k_r_dr(s, el, nu_factor)
         use utils_lib, only: is_bad_num
         type (star_info), pointer :: s
         integer, intent(in) :: el
         real(dp), intent(in) :: nu_factor
         
         real(dp) :: integral, cs2, r2, n2, sl2, omega2, &
            L2, kr2, dr, r0_outer, r0_inner, sl2_next
         integer :: k, k1, k_inner, k_outer  
         
         logical :: dbg
            
         include 'formats'
         
         dbg = .false. !(el == 1 .and. nu_factor == 1d0)
         
         
         get_int_k_r_dr = 0
         L2 = el*(el+1)
         omega2 = (1d-6*2*pi*s% nu_max*nu_factor)**2
         
         ! k_inner and k_outer are bounds of evanescent region
         
         ! k_outer is outermost k where Sl2 <= omega2 at k-1 and Sl2 > omega2 at k
         ! 1st find outermost where Sl2 <= omega2
         k1 = 0
         do k = 1, s% nz
            r2 = s% r(k)**2
            cs2 = s% csound_at_face(k)**2
            sl2 = L2*cs2/r2
            if (sl2 <= omega2) then
               k1 = k; exit
            end if
         end do
         if (k1 == 0) return
         ! then find next k where Sl2 >= omega2
         k_outer = 0
         do k = k1+1, s% nz
            r2 = s% r(k)**2
            cs2 = s% csound_at_face(k)**2
            sl2 = L2*cs2/r2
            if (sl2 > omega2) then
               k_outer = k; exit
            end if
         end do
         if (k_outer == 0) return
         
         ! k_inner is next k where N2 >= omega2 at k+1 and N2 < omega2 at k
         k_inner = 0
         do k = k_outer+1, s% nz
            if (s% brunt_N2(k) >= omega2) then
               k_inner= k; exit
            end if
         end do
         if (k_inner == 0) return
         
         integral = 0
         do k = k_inner-1, k_outer, -1
            r2 = s% r(k)**2
            cs2 = s% csound_at_face(k)**2
            n2 = s% brunt_N2(k)
            sl2 = L2*cs2/r2
            if (k == k_inner-1) then
               ! interpolate to estimate where in cell k N2 == omega2
               ! use that location in place of rmid(k)
               if (n2 <= omega2 .and. omega2 <= s% brunt_N2(k+1)) then
                  r0_inner = (find0(s% r(k)**3, n2-omega2, s% r(k+1)**3, s% brunt_N2(k+1)-omega2))**(1d0/3d0)
                  if (r0_inner < s% r(k+1) .or. r0_inner > s% r(k)) then
                     if (dbg) write(*,1) 'r0_inner confusion', s% r(k+1), r0_inner, s% r(k)
                  end if
               else
                  r0_inner = s% r(k+1)
                  if (dbg) write(*,2) 'N2 == omega2 failed for k_inner', k_inner
               end if
               dr = s% rmid(k-1) - r0_inner
            else if (k == k_outer) then ! find where in cell k-1 Sl2 == omega2
               ! use that location in place of rmid(k-1)
               sl2_next = L2*(s% csound(k-1)/s% r(k-1))**2
               if (sl2 >= omega2 .and. omega2 >= sl2_next) then
                  r0_outer = (find0(s% r(k)**3, sl2-omega2, s% r(k-1)**3, sl2_next-omega2))**(1d0/3d0)
                  if (r0_outer < s% r(k) .or. r0_outer > s% r(k-1)) then
                     if (dbg) write(*,1) 'r0_outer confusion', s% r(k), r0_outer, s% r(k-1)
                  end if
               else
                  r0_outer = s% r(k-1)
                  if (dbg) write(*,2) 'Sl2 == omega2 failed for k_outer', k_outer
               end if
               dr = r0_outer - s% rmid(k)
            else
               dr = s% rmid(k-1) - s% rmid(k)
            end if
            kr2 = (1 - n2/omega2)*(1 - Sl2/omega2)*omega2/cs2
            if (kr2 < 0 .and. n2 < omega2 .and. omega2 < Sl2) &
               integral = integral + sqrt(-kr2)*dr
         end do
         
         
         if (integral == 0) return
         get_int_k_r_dr = 2*integral/ln10
         
         if (dbg) write(*,3) 'r0 inner outer', &
            k_inner, k_outer, r0_inner/Rsun, r0_outer/Rsun, get_int_k_r_dr
            
         if (.not. is_bad_num(get_int_k_r_dr)) return
         
         write(*,2) 'el', el
         write(*,1) 'nu_factor', nu_factor
         write(*,1) 's% nu_max*nu_factor', s% nu_max*nu_factor
         write(*,1) 'log10 nu_max*nu_factor', log10(s% nu_max*nu_factor)
         
         write(*,1) 'get_int_k_r_dr', get_int_k_r_dr
         write(*,1) 'integral', integral
         write(*,2) 'k_inner', k_inner
         write(*,2) 'k_outer', k_outer
         stop 'get_int_k_r_dr'
         
      end function get_int_k_r_dr


      real(dp) function central_avg_x(s,j)
         type (star_info), pointer :: s
         integer, intent(in) :: j
         real(dp) :: sum_x, sum_dq
         integer :: k
         if (j == 0) then
            central_avg_x = 0
            return
         end if
         sum_x = 0
         sum_dq = 0
         do k = s% nz, 1, -1
            sum_x = sum_x + s% xa(j,k)*s% dq(k)
            sum_dq = sum_dq + s% dq(k)
            if (sum_dq >= s% center_avg_value_dq) exit
         end do
         central_avg_x = sum_x/sum_dq
      end function central_avg_x         


      real(dp) function surface_avg_x(s,j)
         type (star_info), pointer :: s
         integer, intent(in) :: j
         real(dp) :: sum_x, sum_dq
         integer :: k
         if (j == 0) then
            surface_avg_x = 0
            return
         end if
         sum_x = 0
         sum_dq = 0
         do k = 1, s% nz
            sum_x = sum_x + s% xa(j,k)*s% dq(k)
            sum_dq = sum_dq + s% dq(k)
            if (sum_dq >= s% surface_avg_abundance_dq) exit
         end do
         surface_avg_x = sum_x/sum_dq
      end function surface_avg_x         


      real(dp) function star_avg_x(s,j)
         type (star_info), pointer :: s
         integer, intent(in) :: j
         if (j == 0) then
            star_avg_x = 0
            return
         end if
         star_avg_x = dot_product(s% xa(j,1:s% nz),s% dq(1:s% nz))/sum(s% dq(1:s% nz))
      end function star_avg_x         


      subroutine get_history_specs(s, num, names, specs)

         use star_utils, only: eval_csound
         use utils_lib
         use utils_def
         
         type (star_info), pointer :: s
         integer, intent(in) :: num
         character (len=*), intent(in) :: names(:)
         integer, intent(out) :: specs(:)
         
         integer :: i, ierr, n, j, iounit, t
         logical :: special_case
         character (len=256) :: buffer, string
         
         include 'formats'
         ierr = 0
         if (num <= 0) return
         iounit = -1
         do i = 1, num
            buffer = names(i)
            n = len_trim(buffer) + 1
            buffer(n:n) = ' '
            j = 0
            t = token(iounit, n, j, buffer, string)
            if (t /= name_token) then
               write(*,*) 'bad value for name of history item ' // trim(names(i))
               ierr = -1
               exit
            end if
            special_case = .false.
            specs(i) = do1_history_spec( &
               iounit, t, n, j, string, buffer, special_case, ierr)
            if (ierr /= 0 .or. special_case) then
               write(*,*) 'get_history_specs failed for ' // trim(names(i))
               specs(i) = -1
               ierr = 0
            end if
         end do            
      
      end subroutine get_history_specs


      subroutine get_history_values(s, num, specs, &
            is_int_value, int_values, values, failed_to_find_value)

         use star_utils, only: eval_csound
         use utils_lib
         use utils_def
         
         type (star_info), pointer :: s
         integer, intent(in) :: num
         integer, intent(in) :: specs(:)
         logical, intent(out) :: is_int_value(:)
         integer, intent(out) :: int_values(:)
         real(dp), intent(out) :: values(:)
         logical, intent(out) :: failed_to_find_value(:)
         
         integer :: i, c, int_val, ierr, n, t, j, iounit
         real(dp) :: val, epsnuc_out(12), v_surf, csound_surf, envelope_fraction_left
         logical :: is_int_val, special_case
         character (len=256) :: buffer, string
         
         include 'formats'
         ierr = 0
         if (num <= 0) return
         
         epsnuc_out(1:4) = s% burn_zone_mass(1:4,1)
         epsnuc_out(5:8) = s% burn_zone_mass(1:4,2)
         epsnuc_out(9:12) = s% burn_zone_mass(1:4,3)
         csound_surf = eval_csound(s,1,ierr)
         
         if (.not. s% v_flag) then
            v_surf =  s% r(1)*s% dlnR_dt(1)
         else
            v_surf = s% v(1)
         end if
         if (s% initial_mass > s% he_core_mass) then
            envelope_fraction_left = &
               (s% star_mass - s% he_core_mass)/(s% initial_mass - s% he_core_mass)
         else
            envelope_fraction_left = 1
         end if
                  
         do i = 1, num
            failed_to_find_value(i) = .false.
            c = specs(i)
            if (c <= 0) then
               failed_to_find_value(i) = .true.
            else
               call history_getval( &
                  s, c, values(i), int_values(i), is_int_value(i), &
                  s% nz, v_surf, csound_surf, envelope_fraction_left, epsnuc_out, ierr)
               if (ierr /= 0) then
                  failed_to_find_value(i) = .true.
                  ierr = 0                  
               end if
            end if
         end do            
      
      end subroutine get_history_values

      
      subroutine get_iso_val(s,str,val,ierr)
         use chem_lib, only: get_nuclide_index
         type (star_info), pointer :: s
         character (len=*), intent(in) :: str
         real(dp), intent(out) :: val
         integer, intent(out) :: ierr
         integer :: n, split, id, i
         ierr = 0
         val = 0
         n = len_trim(str)
         split = 0
         do i=1,n
            if (str(i:i) == ' ') then
               split = i
               exit
            end if
         end do
         if (split <= 1 .or. split >= n) then ! no interior space to split str
            ierr = -1
            return
         end if
         id = get_nuclide_index(str(split+1:n))
         if (id <= 0) then ! not a valid iso name
            ierr = -1
            return
         end if
         i = s% net_iso(id)
         select case (str(1:split-1))             
            case('center')
               val = central_avg_x(s,i)               
            case('surface')
               val = surface_avg_x(s,i)              
            case('average')
               val = star_avg_x(s,i)               
            case('total_mass')
               val = star_avg_x(s,i)*s% xmstar/Msun              
            case('log_total_mass')
               val = safe_log10(star_avg_x(s,i)*s% xmstar/Msun)               
            case('log_average')
               val = safe_log10(star_avg_x(s,i))               
            case('log_center')
               val = safe_log10(central_avg_x(s,i))               
            case('log_surface')
               val = safe_log10(surface_avg_x(s,i))                  
            case default
               ierr = -1
         end select
      end subroutine get_iso_val




      end module history

