! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module mod_eps_grav
      
      use star_private_def
      use const_def
      use chem_def, only: chem_isos
      use utils_lib, only:is_bad_num
      use hydro_vars, only: get_dVARDOT_dVAR_cell
      
      implicit none


      contains
      
      
      subroutine eval_eps_grav_and_partials(s, k, dt, ierr)
         type (star_info), pointer :: s 
         integer, intent(in) :: k
         real(dp), intent(in) :: dt
         integer, intent(out) :: ierr
         real(dp) :: f
         include 'formats'
         ierr = 0
         
         s% eps_grav(k) = 0
         s% d_eps_grav_dlndm1(k) = 0
         s% d_eps_grav_dlnd00(k) = 0
         s% d_eps_grav_dlndp1(k) = 0
         s% d_eps_grav_dlnTm1(k) = 0
         s% d_eps_grav_dlnT00(k) = 0
         s% d_eps_grav_dlnTp1(k) = 0
         s% d_eps_grav_dlnR00(k) = 0
         s% d_eps_grav_dL00(k) = 0
         s% d_eps_grav_dlnPgasm1_const_T(k) = 0
         s% d_eps_grav_dlnPgas00_const_T(k) = 0
         s% d_eps_grav_dlnPgasp1_const_T(k) = 0
         s% d_eps_grav_dlnTm1_const_Pgas(k) = 0
         s% d_eps_grav_dlnT00_const_Pgas(k) = 0
         s% d_eps_grav_dlnTp1_const_Pgas(k) = 0
         s% d_eps_grav_dlnRp1(k) = 0
         s% d_eps_grav_dv00(k) = 0
         s% d_eps_grav_dvp1(k) = 0

         call eval1_eps_grav_and_partials(s, k, dt, ierr)
         if (ierr /= 0) return
         
         if (s% use_other_eps_grav) then
            ! note: call this after 1st doing the standard calculation
            call s% other_eps_grav(s% id, k, dt, ierr)
            if (ierr /= 0) return
         end if
         
         f = s% eps_grav_factor
         if (abs(f - 1d0) < 1d-12) return
         
         s% eps_grav(k) = f*s% eps_grav(k)
         s% d_eps_grav_dlndm1(k) = f*s% d_eps_grav_dlndm1(k)
         s% d_eps_grav_dlnd00(k) = f*s% d_eps_grav_dlnd00(k)
         s% d_eps_grav_dlndp1(k) = f*s% d_eps_grav_dlndp1(k)
         s% d_eps_grav_dlnTm1(k) = f*s% d_eps_grav_dlnTm1(k)
         s% d_eps_grav_dlnT00(k) = f*s% d_eps_grav_dlnT00(k)
         s% d_eps_grav_dlnTp1(k) = f*s% d_eps_grav_dlnTp1(k)
         s% d_eps_grav_dlnR00(k) = f*s% d_eps_grav_dlnR00(k)
         s% d_eps_grav_dL00(k) = f*s% d_eps_grav_dL00(k)
         s% d_eps_grav_dlnPgas00_const_T(k) = f*s% d_eps_grav_dlnPgas00_const_T(k)
         s% d_eps_grav_dlnPgasm1_const_T(k) = f*s% d_eps_grav_dlnPgasm1_const_T(k)
         s% d_eps_grav_dlnPgasp1_const_T(k) = f*s% d_eps_grav_dlnPgasp1_const_T(k)
         s% d_eps_grav_dlnTm1_const_Pgas(k) = f*s% d_eps_grav_dlnTm1_const_Pgas(k)
         s% d_eps_grav_dlnT00_const_Pgas(k) = f*s% d_eps_grav_dlnT00_const_Pgas(k)
         s% d_eps_grav_dlnTp1_const_Pgas(k) = f*s% d_eps_grav_dlnTp1_const_Pgas(k)
         s% d_eps_grav_dlnRp1(k) = f*s% d_eps_grav_dlnRp1(k)
         s% d_eps_grav_dv00(k) = f*s% d_eps_grav_dv00(k)
         s% d_eps_grav_dvp1(k) = f*s% d_eps_grav_dvp1(k)
         
         
      end subroutine eval_eps_grav_and_partials
      
   
      subroutine eval1_eps_grav_and_partials(s, k, dt, ierr)
         use eos_def
         use eos_lib
         use mlt_def
         type (star_info), pointer :: s 
         integer, intent(in) :: k
         real(dp), intent(in) :: dt
         integer, intent(out) :: ierr
         
         real(dp) :: &
            dlnd_dt, dlnPgas_dt, dlnT_dt, &
            d_dVARdt_dVARm1, d_dVARdt_dVAR00, d_dVARdt_dVARp1, &
            a1, da1_dlnd, da1_dlnT, &
            T1, dT1_dlnd, dT1_dlnTm1, dT1_dlnT00, dT1_dlnTp1, dT1_dlnd_dt, dT1_d_dlnTdt, &
            a2, da2_dlnd, da2_dlnT, &
            T2, dT2_dlnT, dT2_dlndm1, dT2_dlnd00, dT2_dlndp1, &
            T3, dT3_dlnd, dT3_dlnT, Gamma, &
            mu, mu_start, dmu_dt, dE_dmu, d_dEdmu_dlnT

         include 'formats'
         
         ierr = 0         
            
         if (k < s% k_below_just_added) then
             if (s% zero_eps_grav_in_just_added_material) return
             if (s% accretion_entropy_at_surface > 0) then
               call do_eps_grav_using_S(ierr)
               if (is_bad_num(s% eps_grav(k))) then
                  ierr = -1
                  if (s% report_ierr) &
                     write(*,*) 'do_eps_grav_using_S -- bad value for eps_grav'
                  return
               end if
               return
             end if
         end if
      
         if (s% use_lnS_for_eps_grav) then
            call do_eps_grav_using_S(ierr)
         else ! check Gamma to see if need to use lnS form
            Gamma = Plasma_Coupling_Parameter(s% T(k), s% rho(k), s% abar(k), s% zbar(k))
            if (Gamma >= s% Gamma_lnS_eps_grav_full_on) then
               call do_eps_grav_using_S(ierr)
            else if (Gamma <= s% Gamma_lnS_eps_grav_full_off) then
               if (s% lnPgas_flag) then
                  call do_eps_grav_with_lnPgas
               else
                  call do_eps_grav_with_lnd
               end if
            else
               call blend_both_values
            end if
            
            if (s% include_composition_in_eps_grav) then
               mu = s% abar(k)/(1 + s% zbar(k)) ! assume complete ionization
               mu_start = s% abar_start(k)/(1 + s% zbar_start(k))
               dmu_dt = (mu - mu_start)/dt
               dE_dmu = -1.5d0*cgas*s% T(k)/mu**2
               d_dEdmu_dlnT = dE_dmu
               s% eps_grav_composition_term(k) = -dE_dmu*dmu_dt
               s% eps_grav(k) = &
                  s% eps_grav(k) + s% eps_grav_composition_term(k)
               s% d_eps_grav_dlnT00(k) = s% d_eps_grav_dlnT00(k) &
                  - d_dEdmu_dlnT*dmu_dt
            end if
            
         end if

         if (is_bad_num(s% eps_grav(k))) then
            ierr = -1
            if (s% report_ierr) &
               write(*,*) 'eval_eps_grav_and_partials -- bad value for eps_grav'
            return
         end if

                  
         contains


         subroutine blend_both_values
            real(dp) :: alfa, beta, &
               eps_grav, d_eps_grav_dlndm1, d_eps_grav_dlnd00, d_eps_grav_dlndp1, &
               d_eps_grav_dlnTm1, d_eps_grav_dlnT00, d_eps_grav_dlnTp1, &
               d_eps_grav_dlnR00, d_eps_grav_dL00, d_eps_grav_dlnPgas00_const_T, &
               d_eps_grav_dlnPgasm1_const_T, d_eps_grav_dlnPgasp1_const_T, &
               d_eps_grav_dlnTm1_const_Pgas, d_eps_grav_dlnT00_const_Pgas, &
               d_eps_grav_dlnTp1_const_Pgas, d_eps_grav_dlnRp1, &
               d_eps_grav_dv00, d_eps_grav_dvp1
            
            alfa = (Gamma - s% Gamma_lnS_eps_grav_full_off) / &
               (s% Gamma_lnS_eps_grav_full_on - s% Gamma_lnS_eps_grav_full_off)
            ! alfa is fraction of lnS form in result
            beta = 1d0 - alfa
            ! beta is fraction of non-lnS form in result

            if (alfa >= 1 .or. alfa <= 0) then
               ierr = -1
               if (s% report_ierr) &
                  write(*,*) 'eval_eps_grav_and_partials -- error in blend_both_values'
               return
            end if
            
            call do_eps_grav_using_S(ierr)
            if (ierr /= 0) return
            
            ! copy results
            eps_grav = s% eps_grav(k)
            d_eps_grav_dlndm1 = s% d_eps_grav_dlndm1(k)
            d_eps_grav_dlnd00 = s% d_eps_grav_dlnd00(k)
            d_eps_grav_dlndp1 = s% d_eps_grav_dlndp1(k)
            d_eps_grav_dlnTm1 = s% d_eps_grav_dlnTm1(k)
            d_eps_grav_dlnT00 = s% d_eps_grav_dlnT00(k)
            d_eps_grav_dlnTp1 = s% d_eps_grav_dlnTp1(k)
            d_eps_grav_dlnR00 = s% d_eps_grav_dlnR00(k)
            d_eps_grav_dL00 = s% d_eps_grav_dL00(k)
            d_eps_grav_dlnPgas00_const_T = s% d_eps_grav_dlnPgas00_const_T(k)
            d_eps_grav_dlnPgasm1_const_T = s% d_eps_grav_dlnPgasm1_const_T(k)
            d_eps_grav_dlnPgasp1_const_T = s% d_eps_grav_dlnPgasp1_const_T(k)
            d_eps_grav_dlnTm1_const_Pgas = s% d_eps_grav_dlnTm1_const_Pgas(k)
            d_eps_grav_dlnT00_const_Pgas = s% d_eps_grav_dlnT00_const_Pgas(k)
            d_eps_grav_dlnTp1_const_Pgas = s% d_eps_grav_dlnTp1_const_Pgas(k)
            d_eps_grav_dlnRp1 = s% d_eps_grav_dlnRp1(k)
            d_eps_grav_dv00 = s% d_eps_grav_dv00(k)
            d_eps_grav_dvp1 = s% d_eps_grav_dvp1(k)
            
            if (s% lnPgas_flag) then
               call do_eps_grav_with_lnPgas
            else
               call do_eps_grav_with_lnd
            end if
            
            ! combine results
            s% eps_grav(k) = alfa*eps_grav + beta*s% eps_grav(k)
            s% d_eps_grav_dlndm1(k) = alfa*d_eps_grav_dlndm1 + beta*s% d_eps_grav_dlndm1(k)
            s% d_eps_grav_dlnd00(k) = alfa*d_eps_grav_dlnd00 + beta*s% d_eps_grav_dlnd00(k)
            s% d_eps_grav_dlndp1(k) = alfa*d_eps_grav_dlndp1 + beta*s% d_eps_grav_dlndp1(k)
            s% d_eps_grav_dlnTm1(k) = alfa*d_eps_grav_dlnTm1 + beta*s% d_eps_grav_dlnTm1(k)
            s% d_eps_grav_dlnT00(k) = alfa*d_eps_grav_dlnT00 + beta*s% d_eps_grav_dlnT00(k)
            s% d_eps_grav_dlnTp1(k) = alfa*d_eps_grav_dlnTp1 + beta*s% d_eps_grav_dlnTp1(k)
            s% d_eps_grav_dlnR00(k) = alfa*d_eps_grav_dlnR00 + beta*s% d_eps_grav_dlnR00(k)
            s% d_eps_grav_dL00(k) = alfa*d_eps_grav_dL00 + beta*s% d_eps_grav_dL00(k)
            s% d_eps_grav_dlnPgas00_const_T(k) = &
               alfa*d_eps_grav_dlnPgas00_const_T + beta*s% d_eps_grav_dlnPgas00_const_T(k)
            s% d_eps_grav_dlnPgasm1_const_T(k) = &
               alfa*d_eps_grav_dlnPgasm1_const_T + beta*s% d_eps_grav_dlnPgasm1_const_T(k)
            s% d_eps_grav_dlnPgasp1_const_T(k) = &
               alfa*d_eps_grav_dlnPgasp1_const_T + beta*s% d_eps_grav_dlnPgasp1_const_T(k)
            s% d_eps_grav_dlnTm1_const_Pgas(k) = &
               alfa*d_eps_grav_dlnTm1_const_Pgas + beta*s% d_eps_grav_dlnTm1_const_Pgas(k)
            s% d_eps_grav_dlnT00_const_Pgas(k) = &
               alfa*d_eps_grav_dlnT00_const_Pgas + beta*s% d_eps_grav_dlnT00_const_Pgas(k)
            s% d_eps_grav_dlnTp1_const_Pgas(k) = &
               alfa*d_eps_grav_dlnTp1_const_Pgas + beta*s% d_eps_grav_dlnTp1_const_Pgas(k)
            s% d_eps_grav_dlnRp1(k) = alfa*d_eps_grav_dlnRp1 + beta*s% d_eps_grav_dlnRp1(k)
            s% d_eps_grav_dv00(k) = alfa*d_eps_grav_dv00 + beta*s% d_eps_grav_dv00(k)
            s% d_eps_grav_dvp1(k) = alfa*d_eps_grav_dvp1 + beta*s% d_eps_grav_dvp1(k)
         
         end subroutine blend_both_values


         subroutine do_eps_grav_using_S(ierr)
            use interp_1d_lib
            use interp_1d_def
            use alloc
            integer, intent(out) :: ierr
            
            ! eps_grav = -T*ds/dt
            integer :: nwork, sz
            real(dp), pointer :: work(:)
            real(dp) :: del_t, xm, entropy, lnS_prev, S_prev, &
               T, dS_dlnT, dS_dlnd
            real(dp) :: thin_radiative_eps_grav         
            logical :: testing_thin_radiative
            
            include 'formats'
            ierr = 0
            
            if (dt == 0) then
               s% eps_grav(k) = 0
               return
            end if
            
            testing_thin_radiative = (.false. .and. s% mixing_type(k) /= convective_mixing)
            
            if (testing_thin_radiative) then ! test vs thin_radiative
               call do_thin_radiative(.true.)
               thin_radiative_eps_grav = s% eps_grav(k)
               s% eps_grav(k) = 0
            end if

            entropy = exp(s% lnS(k))
            T = s% T(k)
            
            if (k >= s% k_below_just_added) then
               del_t = dt
               if (k >= s% k_const_mass) then
                  S_prev = exp(s% lnS_pre(k))
               else
                  if (.not. s% have_prev_mesh_lnS_interpolant) then
!$OMP critical (create_lnS_interpolant)
                     if (.not. s% have_prev_mesh_lnS_interpolant) then
                        nwork = pm_work_size
                        sz = nwork*s% prev_mesh_nz
                        call non_crit_get_work_array( &
                           s, work, sz, 0, 'do_eps_grav_using_S', ierr)
                        if (ierr == 0) then
                           call interp_pm( &
                              s% prev_mesh_xm, s% prev_mesh_nz, &
                              s% prev_mesh_f1_lnS, &
                              nwork, work, ierr)
                           call non_crit_return_work_array( &
                              s, work, 'do_eps_grav_using_S')
                           if (ierr == 0) &
                              s% have_prev_mesh_lnS_interpolant = .true.
                        end if
                     end if
!$OMP end critical (create_lnS_interpolant)
                  end if
                  if (ierr /= 0) return
                  xm = sum(s% dm(1:k-1)) - (s% xmstar - s% xmstar_old) 
                     ! use xm wrt old mass for interpolation
                  call interp_value( &
                     s% prev_mesh_xm, s% prev_mesh_nz, s% prev_mesh_f1_lnS, &
                     xm, lnS_prev, ierr)
                  if (ierr /= 0) then
                  end if
                  S_prev = exp(lnS_prev)
               end if
            else ! k < k_below_just_added
               del_t = s% del_t_for_just_added(k)
               if (del_t <= 0d0) return ! leave eps_grav = 0
               if (s% accretion_entropy_at_surface > 0) then
                  S_prev = s% accretion_entropy_at_surface
               else
                  S_prev = exp(s% lnS_pre(1))
               end if
            end if
                        
            s% eps_grav(k) = -T*(entropy - S_prev)/del_t

            if (testing_thin_radiative) then
               write(*,2) 'eps_grav vs thin_radiative', k, &
                  abs(s% eps_grav(k) - thin_radiative_eps_grav)/s% eps_grav(k)
               write(*,2) 's% eps_grav(k)', k, s% eps_grav(k)
               write(*,2) 'thin_radiative_eps_grav', k, thin_radiative_eps_grav
               write(*,2) 'entropy', k, entropy
               write(*,2) 'S_prev', k, S_prev
               write(*,2) 'exp(s% lnS_pre(1))', k, exp(s% lnS_pre(1))
               write(*,2) 'del_t', k, del_t
               write(*,2) 'dt', k, dt
               write(*,*)
               !stop 'eps_grav'
            end if
            
            dS_dlnT = entropy*s% d_eos_dlnT(i_lnS,k)
            dS_dlnd = entropy*s% d_eos_dlnd(i_lnS,k)
            s% d_eps_grav_dlnT00(k) = -T*dS_dlnT/del_t + s% eps_grav(k)
            s% d_eps_grav_dlnd00(k) = -T*dS_dlnd/del_t

            if (s% lnPgas_flag) then
               s% d_eps_grav_dlnPgas00_const_T(k) = &
                  s% d_eps_grav_dlnd00(k)*s% dlnRho_dlnPgas_const_T(k)
               s% d_eps_grav_dlnT00_const_Pgas(k) = &
                  s% d_eps_grav_dlnT00(k) + &
                  s% d_eps_grav_dlnd00(k)*s% dlnRho_dlnT_const_Pgas(k)
            end if
               
         end subroutine do_eps_grav_using_S


         subroutine do_eps_grav_with_lnPgas
         
            !s% eps_grav(k) = -s% T(k)*s% cp(k)*(dlnT_dt - s% grada(k)*dlnP_dt)
            ! dlnP_dt = (dPgas_dt + dPrad_dt)/P
            !         = (Pgas*dlnPgas_dt + 4*Prad*dlnT_dt)/P
            !s% eps_grav(k) = -s% T(k)*s% cp(k)* &
            !     ((1 - s% grada(k)*4*Prad/P)*dlnT_dt - s% grada(k)*Pgas/P*dlnPgas_dt)
            
            real(dp) :: dlnd_dlnPgas00, dlnd_dlnPgasm1, dlnd_dlnPgasp1, dgrada_dlnPgas, &
               dT1_dlnPgas, dT2_dlnPgasm1, dT2_dlnPgas00, dT2_dlnPgasp1, dT3_dlnPgas, &
               d_eps_grav_dlnPgasm1, d_eps_grav_dlnPgas00, d_eps_grav_dlnPgasp1, &
               dP_dlnPgas, dPinv_dlnPgas, dPrad_dlnT, &
               da1_dlnPgas, da2_dlnPgas, dP_dlnT_const_Pgas, dPinv_dlnT_const_Pgas, &
               da1_dlnPgas_const_T, da1_dlnT_const_Pgas, da2_dlnPgas_const_T, &
               da2_dlnT_const_Pgas, dcp_dlnPgas_const_T, dcp_dlnT_const_Pgas, &
               dgrada_dlnPgas_const_T, dgrada_dlnT_const_Pgas, dPinv_dlnPgas_const_T, &
               dPgas_dlnPgas_const_T, dPrad_dlnT_const_Pgas, dP_dlnPgas_const_T, &
               dT1_dlnPgas_const_T, dT1_dlnT00_const_Pgas, dT1_dlnTm1_const_Pgas, &
               dT1_dlnTp1_const_Pgas, dT2_dlnPgas00_const_T, dT2_dlnPgasm1_const_T, &
               dT2_dlnPgasp1_const_T, dT2_dlnT_const_Pgas, dT3_dlnPgas_const_T, &
               dT3_dlnT_const_Pgas

            dlnPgas_dt = s% dlnPgas_dt(k)     
            dlnT_dt = s% dlnT_dt(k)         
            call get_dVARDOT_dVAR_cell(s, k, dt, d_dVARdt_dVARm1, d_dVARdt_dVAR00, d_dVARdt_dVARp1)
            
            dgrada_dlnPgas_const_T = s% d_eos_dlnd(i_grad_ad,k)*s% dlnRho_dlnPgas_const_T(k)
            dgrada_dlnT_const_Pgas = &
               s% d_eos_dlnT(i_grad_ad,k) + s% d_eos_dlnd(i_grad_ad,k)*s% dlnRho_dlnT_const_Pgas(k)
            
            dgrada_dlnPgas_const_T = s% d_eos_dlnd(i_grad_ad,k)*s% dlnRho_dlnPgas_const_T(k)
            dgrada_dlnT_const_Pgas = &
               s% d_eos_dlnT(i_grad_ad,k) + s% d_eos_dlnd(i_grad_ad,k)*s% dlnRho_dlnT_const_Pgas(k)

            dcp_dlnPgas_const_T = s% d_eos_dlnd(i_cp,k)*s% dlnRho_dlnPgas_const_T(k)
            dcp_dlnT_const_Pgas = &
               s% d_eos_dlnT(i_cp,k) + s% d_eos_dlnd(i_cp,k)*s% dlnRho_dlnT_const_Pgas(k)
            
            dPgas_dlnPgas_const_T = s% Pgas(k)
            dP_dlnPgas_const_T = dPgas_dlnPgas_const_T
            dPinv_dlnPgas_const_T = -dP_dlnPgas_const_T/s% P(k)**2
            
            dPrad_dlnT_const_Pgas = 4*s% Prad(k)
            dP_dlnT_const_Pgas = dPrad_dlnT_const_Pgas
            dPinv_dlnT_const_Pgas = -dP_dlnT_const_Pgas/s% P(k)**2
            
            a1 = 1 - s% grada(k)*4*s% Prad(k)/s% P(k)
            da1_dlnPgas_const_T = &
               - dgrada_dlnPgas_const_T*4*s% Prad(k)/s% P(k) &
               - s% grada(k)*4*s% Prad(k)*dPinv_dlnPgas_const_T
            da1_dlnT_const_Pgas = &
               - dgrada_dlnT_const_Pgas*4*s% Prad(k)/s% P(k) &
               - s% grada(k)*4*dPrad_dlnT_const_Pgas/s% P(k) &
               - s% grada(k)*4*s% Prad(k)*dPinv_dlnT_const_Pgas
            
            T1 = dlnT_dt*a1
            dT1_dlnPgas_const_T = dlnT_dt*da1_dlnPgas_const_T
            dT1_dlnT00_const_Pgas = d_dVARdt_dVAR00*a1 + dlnT_dt*da1_dlnT_const_Pgas
            dT1_dlnTm1_const_Pgas = d_dVARdt_dVARm1*a1
            dT1_dlnTp1_const_Pgas = d_dVARdt_dVARp1*a1
            
            a2 = s% grada(k)*s% Pgas(k)/s% P(k)
            da2_dlnPgas_const_T = &
               dgrada_dlnPgas_const_T*s% Pgas(k)/s% P(k) + &
               s% grada(k)*dPgas_dlnPgas_const_T/s% P(k) + &
               s% grada(k)*s% Pgas(k)*dPinv_dlnPgas_const_T
            da2_dlnT_const_Pgas = &
               dgrada_dlnT_const_Pgas*s% Pgas(k)/s% P(k) + &
               s% grada(k)*s% Pgas(k)*dPinv_dlnT_const_Pgas
            
            T2 = dlnPgas_dt*a2
            dT2_dlnT_const_Pgas = dlnPgas_dt*da2_dlnT_const_Pgas
            dT2_dlnPgas00_const_T = d_dVARdt_dVAR00*a2 + dlnPgas_dt*da2_dlnPgas_const_T
            dT2_dlnPgasm1_const_T = d_dVARdt_dVARm1*a2
            dT2_dlnPgasp1_const_T = d_dVARdt_dVARp1*a2
            
            T3 = -s% T(k)*s% cp(k)
            dT3_dlnPgas_const_T = -s% T(k)*dcp_dlnPgas_const_T
            dT3_dlnT_const_Pgas =  -(s% T(k)*s% cp(k) + s% T(k)*dcp_dlnT_const_Pgas)

            s% eps_grav(k) = T3*(T1-T2)
         
            if (is_bad_num(s% eps_grav(k))) then
               ierr = -1
               if (s% report_ierr) write(*,*) 'eval_eps_grav_and_partials -- bad value for eps_grav'
               return
            end if

            s% d_eps_grav_dlnPgasm1_const_T(k) = -T3*dT2_dlnPgasm1_const_T
            s% d_eps_grav_dlnPgasp1_const_T(k) = -T3*dT2_dlnPgasp1_const_T
            s% d_eps_grav_dlnPgas00_const_T(k) = &
               (T3*(dT1_dlnPgas_const_T - dT2_dlnPgas00_const_T) + dT3_dlnPgas_const_T*(T1-T2))
            
            s% d_eps_grav_dlnTm1_const_Pgas(k) = T3*dT1_dlnTm1_const_Pgas
            s% d_eps_grav_dlnTp1_const_Pgas(k) = T3*dT1_dlnTp1_const_Pgas
            s% d_eps_grav_dlnT00_const_Pgas(k) = &
               (T3*(dT1_dlnT00_const_Pgas - dT2_dlnT_const_Pgas) + dT3_dlnT_const_Pgas*(T1-T2))

         end subroutine do_eps_grav_with_lnPgas


         subroutine do_eps_grav_with_lnd
            integer :: j
            include 'formats'
         
            !s% eps_grav(k) = -s% T(k)*s% cp(k)* &
            !      ((1-s% grada(k)*s% chiT(k))*dlnT_dt &
            !        - s% grada(k)*s% chiRho(k)*dlnd_dt)
            
            real(dp) :: dT1_dlnTdot, dT2_dlnddot
            
            dlnd_dt = s% dlnd_dt(k)
            dlnT_dt = s% dlnT_dt(k)
            
            call get_dVARDOT_dVAR_cell(s, k, dt, d_dVARdt_dVARm1, d_dVARdt_dVAR00, d_dVARdt_dVARp1)

            a1 = 1 - s% grada(k)*s% chiT(k)
            da1_dlnd = -(s% d_eos_dlnd(i_grad_ad,k)*s% chiT(k) + s% grada(k)*s% d_eos_dlnd(i_chiT,k))
            da1_dlnT = -(s% d_eos_dlnT(i_grad_ad,k)*s% chiT(k) + s% grada(k)*s% d_eos_dlnT(i_chiT,k))
   
            T1 = dlnT_dt*a1
            dT1_dlnd = dlnT_dt*da1_dlnd
   
            dT1_d_dlnTdt = a1
            dT1_dlnT00 = d_dVARdt_dVAR00*a1 + dlnT_dt*da1_dlnT
            dT1_dlnTm1 = d_dVARdt_dVARm1*a1
            dT1_dlnTp1 = d_dVARdt_dVARp1*a1
            dT1_dlnTdot = 0
            
            a2 = s% grada(k)*s% chiRho(k)
            da2_dlnd = s% d_eos_dlnd(i_grad_ad,k)*s% chiRho(k) + s% grada(k)*s% d_eos_dlnd(i_chiRho,k)
            da2_dlnT = s% d_eos_dlnT(i_grad_ad,k)*s% chiRho(k) + s% grada(k)*s% d_eos_dlnT(i_chiRho,k)
   
            T2 = dlnd_dt*a2
            dT2_dlnT = dlnd_dt*da2_dlnT
            
            dT2_dlnd00 = d_dVARdt_dVAR00*a2 + dlnd_dt*da2_dlnd
            dT2_dlndm1 = d_dVARdt_dVARm1*a2
            dT2_dlndp1 = d_dVARdt_dVARp1*a2
            dT2_dlnddot = 0

            T3 = -s% T(k)*s% cp(k)
            dT3_dlnd = -s% T(k)*s% d_eos_dlnd(i_Cp,k)
            dT3_dlnT = -s% T(k)*(s% cp(k) + s% d_eos_dlnT(i_Cp,k))
   
            ! eps_grav = T3*(T1-T2)
            s% eps_grav(k) = T3*(T1-T2)
         
            s% d_eps_grav_dlndm1(k) = -T3*dT2_dlndm1
            s% d_eps_grav_dlndp1(k) = -T3*dT2_dlndp1
            s% d_eps_grav_dlnd00(k) = (T3*(dT1_dlnd - dT2_dlnd00) + dT3_dlnd*(T1-T2))
            
            s% d_eps_grav_dlnTm1(k) = T3*dT1_dlnTm1
            s% d_eps_grav_dlnTp1(k) = T3*dT1_dlnTp1
            s% d_eps_grav_dlnT00(k) = (T3*(dT1_dlnT00 - dT2_dlnT) + dT3_dlnT*(T1-T2))

            if (.false. .and. k == 12) then
               write(*,*) 'do_eps_grav_with_lnd'
               write(*,2) 's% lnS(k)/ln10', k, s% lnS(k)/ln10
               write(*,2) 's% lnS_pre(k)/ln10', k, s% lnS_pre(k)/ln10
               write(*,2) '(lnS - lnS_pre)/ln10', k, (s% lnS(k) - s% lnS_pre(k))/ln10
               write(*,2) 's% rho(k)', k, s% rho(k)
               write(*,2) 's% T(k)', k, s% T(k)               
               write(*,2) 's% grada(k)', k, s% grada(k)
               write(*,2) 's% chiRho(k)', k, s% chiRho(k)
               write(*,2) 's% chiT(k)', k, s% chiT(k)
               write(*,2) 'dt', k, s% dt
               write(*,2) 's% eps_grav(k)', k, s% eps_grav(k)
               write(*,2) 's% dlnT_dt(k)', k, s% dlnT_dt(k)
               write(*,2) 's% dlnd_dt(k)', k, s% dlnd_dt(k)
               write(*,2) 's% dlnR_dt(k)', k, s% dlnR_dt(k)
               write(*,2) 'T1-T2', k, T1-T2
               write(*,2) 'T1', k, T1
               write(*,2) 'T2', k, T2
               write(*,2) 'T3', k, T3
            end if
            
            
            
         end subroutine do_eps_grav_with_lnd
         
         
         subroutine do_thin_radiative(testing)
            logical, intent(in) :: testing
         
            ! use Lars' formula for eps_grav in thin radiative envelope during accretion
            ! Townsley & Bildsten, The Astrophysical Journal, 600:390–403, 2004 January 1
            
            real(dp) :: inv_4pi_r4, inv_4pi_r4_P, dg, G_m_mdot, CpT, &
               d_inv_4pi_r4_P_dlnR, d_inv_4pi_r4_P_dlnd, d_inv_4pi_r4_P_dlnT, &
               d_dg_dlnd00, d_dg_dlnT00, d_dg_dlndm1, d_dg_dlnTm1, d_dg_dlnR, d_dg_dL, &
               d_CpT_dlnd, d_CpT_dlnT, d_CpT_dlnPgas_const_T, d_CpT_dlnT_const_Pgas, &
               d_dg_dlnPgas00_const_T, d_dg_dlnPgasm1_const_T, d_dg_dlnT00_const_Pgas, &
               d_dg_dlnTm1_const_Pgas, d_inv_4pi_r4_P_dlnPgas_const_T, &
               d_inv_4pi_r4_P_dlnT_const_Pgas, dP_dlnPgas_const_T, dP_dlnT_const_Pgas, &
               dPinv_dlnPgas_const_T, dPinv_dlnT_const_Pgas, gradT_mid, mmid
            
            include 'formats'
            
            inv_4pi_r4 = 1/(4*pi*s% rmid(k)**4)
            inv_4pi_r4_P = inv_4pi_r4/s% P(k)
            gradT_mid = 05d0*(s% gradT(k) + s% gradT(k+1))
            dg = s% grada(k) - gradT_mid
            mmid = 0.5d0*(s% m(k) + s% m(k+1))
            G_m_mdot = s% cgrav(k)*mmid*s% mstar_dot
            CpT = s% Cp(k)*s% T(k)
            
            s% eps_grav(k) = dg*G_m_mdot*CpT*inv_4pi_r4_P
            
            if (testing) then
               write(*,2) 'dg', k, dg
               write(*,2) 's% grada(k)', k, s% grada(k)
               write(*,2) 'gradT_mid', k, gradT_mid
               write(*,2) 'G_m_mdot', k, G_m_mdot
               write(*,2) 'CpT', k, CpT
               write(*,2) 'inv_4pi_r4_P', k, inv_4pi_r4_P
               write(*,2) 'eps_grav', k, s% eps_grav(k)
               write(*,2) 'Cp_T_Mdot_div_L', k, CpT*s% mstar_dot/s% L(k)
               write(*,*)
               return
            end if
            
            stop 'do_thin_radiative'
            ! need to be revised to use cell centered values for gradT and r.
            
            d_inv_4pi_r4_P_dlnR = -4*inv_4pi_r4_P
            
            d_dg_dlnR = -s% d_gradT_dlnR(k)
            d_dg_dL = -s% d_gradT_dL(k)

            d_dg_dlnd00 = s% d_eos_dlnd(i_grad_ad,k) - s% d_gradT_dlnd00(k)
            d_dg_dlnT00 = s% d_eos_dlnT(i_grad_ad,k) - s% d_gradT_dlnT00(k)
         
            d_dg_dlndm1 = -s% d_gradT_dlndm1(k)
            d_dg_dlnTm1 = -s% d_gradT_dlnTm1(k)

            d_CpT_dlnd = s% d_eos_dlnd(i_Cp,k)*s% T(k)
            d_CpT_dlnT = s% d_eos_dlnT(i_Cp,k)*s% T(k) + CpT 

            s% d_eps_grav_dlnR00(k) = d_dg_dlnR*G_m_mdot*CpT*inv_4pi_r4_P
            s% d_eps_grav_dL00(k) = d_dg_dL*G_m_mdot*CpT*inv_4pi_r4_P
            
            if (s% lnPgas_flag) then
               
               dP_dlnPgas_const_T = s% Pgas(k)
               dP_dlnT_const_Pgas = 4*s% Prad(k)
               
               dPinv_dlnPgas_const_T = -dP_dlnPgas_const_T/s% P(k)**2
               dPinv_dlnT_const_Pgas = -dP_dlnT_const_Pgas/s% P(k)**2
               
               d_inv_4pi_r4_P_dlnPgas_const_T = inv_4pi_r4*dPinv_dlnPgas_const_T
               d_inv_4pi_r4_P_dlnT_const_Pgas = inv_4pi_r4*dPinv_dlnT_const_Pgas
               
               d_CpT_dlnPgas_const_T = d_CpT_dlnd*s% dlnRho_dlnPgas_const_T(k)
               d_CpT_dlnT_const_Pgas = &
                  d_CpT_dlnT + d_CpT_dlnd*s% dlnRho_dlnT_const_Pgas(k)
               
               d_dg_dlnPgas00_const_T = d_dg_dlnd00*s% dlnRho_dlnPgas_const_T(k)
               d_dg_dlnT00_const_Pgas = &
                  d_dg_dlnT00 + d_dg_dlnd00*s% dlnRho_dlnT_const_Pgas(k)
               
               if (k > 1) then
                  d_dg_dlnPgasm1_const_T = d_dg_dlndm1*s% dlnRho_dlnPgas_const_T(k-1)
                  d_dg_dlnTm1_const_Pgas = &
                     d_dg_dlnTm1 + d_dg_dlndm1*s% dlnRho_dlnT_const_Pgas(k-1)
               else
                  d_dg_dlnPgasm1_const_T = 0
                  d_dg_dlnTm1_const_Pgas = 0
               end if
               
               s% d_eps_grav_dlnPgasm1_const_T(k) = &
                  d_dg_dlnPgasm1_const_T*G_m_mdot*CpT*inv_4pi_r4_P
               s% d_eps_grav_dlnPgas00_const_T(k) = G_m_mdot*( &
                  d_dg_dlnPgas00_const_T*CpT*inv_4pi_r4_P + &
                  dg*d_CpT_dlnPgas_const_T*inv_4pi_r4_P + &
                  dg*CpT*d_inv_4pi_r4_P_dlnPgas_const_T)
               s% d_eps_grav_dlnPgasp1_const_T(k) = 0
               
               s% d_eps_grav_dlnTm1_const_Pgas(k) = &
                  d_dg_dlnTm1_const_Pgas*G_m_mdot*CpT*inv_4pi_r4_P
               s% d_eps_grav_dlnT00_const_Pgas(k) = G_m_mdot*( &
                  d_dg_dlnT00_const_Pgas*CpT*inv_4pi_r4_P + &
                  dg*d_CpT_dlnT_const_Pgas*inv_4pi_r4_P + &
                  dg*CpT*d_inv_4pi_r4_P_dlnT_const_Pgas)
               s% d_eps_grav_dlnTp1_const_Pgas(k) = 0
            
            else

               d_inv_4pi_r4_P_dlnd = -s% chiRho(k)*inv_4pi_r4_P
               d_inv_4pi_r4_P_dlnT = -s% chiT(k)*inv_4pi_r4_P

               s% d_eps_grav_dlndm1(k) = d_dg_dlndm1*G_m_mdot*CpT*inv_4pi_r4_P
               s% d_eps_grav_dlndp1(k) = 0
               s% d_eps_grav_dlnd00(k) = G_m_mdot*( &
                  d_dg_dlnd00*CpT*inv_4pi_r4_P + dg*d_CpT_dlnd*inv_4pi_r4_P + dg*CpT*d_inv_4pi_r4_P_dlnd)

               s% d_eps_grav_dlnTm1(k) = d_dg_dlnTm1*G_m_mdot*CpT*inv_4pi_r4_P
               s% d_eps_grav_dlnTp1(k) = 0
               s% d_eps_grav_dlnT00(k) = G_m_mdot*( &
                  d_dg_dlnT00*CpT*inv_4pi_r4_P + dg*d_CpT_dlnT*inv_4pi_r4_P + dg*CpT*d_inv_4pi_r4_P_dlnT)
            
            end if
            
      
         end subroutine do_thin_radiative
         

      end subroutine eval1_eps_grav_and_partials

         
      end module mod_eps_grav

