! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module alloc

      use star_private_def
      use utils_lib, only: fill_with_NaNs

      implicit none
      
      
      logical, parameter :: fill_arrays_with_NaNs = .false.


      integer, parameter :: do_deallocate = 0
      integer, parameter :: do_allocate = 1
      integer, parameter :: do_check_size = 2
      integer, parameter :: do_remove_from_center = 3
      integer, parameter :: do_copy_pointers_and_resize = 4
      
      logical, parameter :: zero_when_allocate = .false.
      
      !logical, parameter :: work_array_debug = .true.
      logical, parameter :: work_array_debug = .false.
      logical, parameter :: quad_array_debug = .false.
      
      logical, parameter :: work_array_trace = .false.
      logical, parameter :: quad_array_trace = .false.
         
         
      ! working storage
      
      type work_array_pointer
         real(dp), dimension(:), pointer :: p
      end type work_array_pointer
      integer, parameter :: num_work_arrays = 250
      type (work_array_pointer), target :: &
         work_pointers(num_work_arrays)
      
      type quad_array_pointer
         real(qp), dimension(:), pointer :: p
      end type quad_array_pointer
      integer, parameter :: num_quad_arrays = 250
      type (quad_array_pointer), target :: &
         quad_pointers(num_quad_arrays)
      
      type int_work_array_pointer
         integer, dimension(:), pointer :: p
      end type int_work_array_pointer
      integer, parameter :: num_int_work_arrays = 250
      type (int_work_array_pointer), target :: &
         int_work_pointers(num_int_work_arrays)
      
      type logical_work_array_pointer
         logical, dimension(:), pointer :: p
      end type logical_work_array_pointer
      integer, parameter :: num_logical_work_arrays = 250
      type (logical_work_array_pointer), target :: &
         logical_work_pointers(num_logical_work_arrays)

      integer :: num_calls, num_returns
      integer :: num_allocs, num_deallocs
      
      
 
      contains
      
      
      subroutine alloc_extras(id, liwork, lwork, ierr)
         integer, intent(in) :: liwork, lwork, id
         integer, intent(out) :: ierr
         type (star_info), pointer :: s
         
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) return
         
         if (associated(s% extra_iwork)) deallocate(s% extra_iwork)
         if (associated(s% extra_iwork_old)) deallocate(s% extra_iwork_old)
         if (associated(s% extra_iwork_older)) deallocate(s% extra_iwork_older)
         
         if (associated(s% extra_work)) deallocate(s% extra_work)
         if (associated(s% extra_work_old)) deallocate(s% extra_work_old)
         if (associated(s% extra_work_older)) deallocate(s% extra_work_older)
         
         allocate( &
            s% extra_iwork(liwork), s% extra_work(lwork), &
            s% extra_iwork_old(liwork), s% extra_work_old(lwork), &
            s% extra_iwork_older(liwork), s% extra_work_older(lwork), &
            stat=ierr)
         if (ierr == 0) then
            s% len_extra_iwork = liwork
            s% len_extra_work = lwork
            if (fill_arrays_with_NaNs) then
               call fill_with_NaNs(s% extra_work)
               call fill_with_NaNs(s% extra_work_old)
               call fill_with_NaNs(s% extra_work_older)
            end if
         end if
         
      end subroutine alloc_extras
      
      
      subroutine dealloc_extras(s)
         type (star_info), pointer :: s
         if (associated(s% extra_iwork)) then
            deallocate(s% extra_iwork)
            nullify(s% extra_iwork)
         end if
         if (associated(s% extra_iwork_old)) then
            deallocate(s% extra_iwork_old)
            nullify(s% extra_iwork_old)
         end if
         if (associated(s% extra_iwork_older)) then
            deallocate(s% extra_iwork_older)
            nullify(s% extra_iwork_older)
         end if
         s% len_extra_iwork = 0
         if (associated(s% extra_work)) then
            deallocate(s% extra_work)
            nullify(s% extra_work)
         end if
         if (associated(s% extra_work_old)) then
            deallocate(s% extra_work_old)
            nullify(s% extra_work_old)
         end if
         if (associated(s% extra_work_older)) then
            deallocate(s% extra_work_older)
            nullify(s% extra_work_older)
         end if
         s% len_extra_work = 0
      end subroutine dealloc_extras
      
      
      subroutine update_nreactions_allocs(s, ierr)
         use rates_def, only: num_rvs
         use utils_lib, only: realloc_double, realloc_double2, realloc_double3
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         integer :: nz, num_reactions
         ierr = 0
         nz = s% nz
         num_reactions = s% num_reactions
         call realloc_double3(s% rate_screened, num_rvs, num_reactions, nz, ierr)
         if (ierr /= 0) return
         call realloc_double3(s% rate_raw, num_rvs, num_reactions, nz, ierr)
         if (ierr /= 0) return
         call realloc_double3(s% reaction_eps_nuc, num_rvs, num_reactions, nz, ierr)
         if (ierr /= 0) return      
      end subroutine update_nreactions_allocs


      subroutine update_nvar_allocs(s, nvar_hydro_old, nvar_chem_old, ierr)
         use utils_lib, only: realloc_double, realloc_double2, realloc_double3, realloc_quad2
         type (star_info), pointer :: s
         integer, intent(in) :: nvar_hydro_old, nvar_chem_old
         integer, intent(out) :: ierr
         
         integer :: nvar, nz, species, nvar_chem, nvar_hydro
         
         include 'formats'
         
         ierr = 0
         nvar_hydro = s% nvar_hydro
         nvar_chem = s% nvar_chem
         species = s% species
         
         if ((nvar_chem_old == nvar_chem) .and. (nvar_hydro_old == nvar_hydro)) return
         
         nvar = nvar_chem + nvar_hydro         
         s% nvar = nvar
         nz = s% nz
         
         call realloc_double2(s% xh, nvar_hydro, nz, ierr)
         if (ierr /= 0) return
         
         call realloc_double2(s% xh_old, nvar_hydro, s% nz_old, ierr)
         if (ierr /= 0) return
         
         call realloc_double2(s% xh_older, nvar_hydro, s% nz_older, ierr)
         if (ierr /= 0) return
         
         call realloc_double(s% equ1, nvar*nz, ierr)
         if (ierr /= 0) return
         s% equ(1:nvar,1:nz) => s% equ1(1:nvar*nz)
         
         call realloc_double2(s% xh_pre, nvar_hydro, s% nz, ierr)
         if (ierr /= 0) return
         
         call update_nreactions_allocs(s, ierr)
                  
      end subroutine update_nvar_allocs
      
      
      subroutine free_star_data(id, ierr)
         integer, intent(in) :: id
         integer, intent(out) :: ierr
         type (star_info), pointer :: s
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) return
         call free_arrays(s)
      end subroutine free_star_data
      
      
      subroutine free_arrays(s)
         use kap_lib
         use eos_lib
         use net_lib, only: free_net_handle
         use star_private_def, only:free_star
         type (star_info), pointer :: s
         type (star_info), pointer :: c => null()
         integer :: ierr
         call free_eos_handle(s% eos_handle) 
         call free_kap_handle(s% kap_handle)        
         call free_net_handle(s% net_handle)
         call star_info_arrays(s, c, do_deallocate, ierr)
         call dealloc_extras(s)
         call free_hydro(s)
         call free_star(s)
      end subroutine free_arrays
      
      
      subroutine free_hydro(s)
         type (star_info), pointer :: s
         if (associated(s% chem_id)) then
            deallocate(s% chem_id)
            nullify(s% chem_id)
         end if
         if (associated(s% net_iso)) then
            deallocate(s% net_iso)
            nullify(s% net_iso)
         end if
         if (associated(s% nameofvar)) then
            deallocate(s% nameofvar)
            nullify(s% nameofvar)
         end if
         if (associated(s% nameofequ)) then
            deallocate(s% nameofequ)
            nullify(s% nameofequ)
         end if
         if (associated(s% ode_var)) then
            deallocate(s% ode_var)
            nullify(s% ode_var)
         end if
      end subroutine free_hydro
      
      
      subroutine check_sizes(s, ierr)
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         type (star_info), pointer :: c => null()
         call star_info_arrays(s, c, do_check_size, ierr)
         if (ierr /= 0) then
            write(*,*) 'check_sizes failed for s'
            return
         end if
         if (s% generations <= 1) return
         call star_info_old_arrays(s, do_check_size, ierr)
         if (ierr /= 0) then
            write(*,*) 'check_sizes failed for s old'
            return
         end if
         if (s% generations <= 2) return
         call star_info_older_arrays(s, do_check_size, ierr)
         if (ierr /= 0) then
            write(*,*) 'check_sizes failed for s older'
            return
         end if
      end subroutine check_sizes
      
      
      subroutine alloc_star_info_old_arrays(s, ierr)
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         call star_info_old_arrays(s, do_allocate, ierr)
      end subroutine alloc_star_info_old_arrays

      
      subroutine star_info_old_arrays(s, action, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: action
         integer, intent(out) :: ierr
         
         integer :: nz, species, nvar_hydro
         
         ierr = 0
         
         nz = s% nz_old
         nvar_hydro = s% nvar_hydro
         species = s% species

         do ! just so can exit in case of failure
         
            call do1D(s% conv_vel_old, nz, action, ierr)
            if (failed('conv_vel_old')) exit
            call do1D(s% Del_T_old, nz, action, ierr)
            if (failed('Del_T_old')) exit

            call do1D(s% nu_ST_old, nz, action, ierr)
            if (failed('nu_ST_old')) exit
            call do1D(s% D_ST_old, nz, action, ierr)
            if (failed('D_ST_old')) exit
            call do1D(s% D_DSI_old, nz, action, ierr)
            if (failed('D_DSI_old')) exit
            call do1D(s% D_SH_old, nz, action, ierr)
            if (failed('D_SH_old')) exit
            call do1D(s% D_SSI_old, nz, action, ierr)
            if (failed('D_SSI_old')) exit
            call do1D(s% D_ES_old, nz, action, ierr)
            if (failed('D_ES_old')) exit
            call do1D(s% D_GSF_old, nz, action, ierr)
            if (failed('D_GSF_old')) exit
            
            call do1D(s% dq_old, nz, action, ierr)
            if (failed('dq_old')) exit
            call do1D(s% q_old, nz, action, ierr)
            if (failed('q_old')) exit
            call do1D(s% omega_old, nz, action, ierr)
            if (failed('omega_old')) exit
            call do2D(s% xa_old, species, nz, action, ierr)
            if (failed('xa_old')) then
               write(*,*) 'size(xa_old,dim=1)', size(s% xa_old,dim=1)
               write(*,*) 'species', species
               write(*,*) 'size(xa_old,dim=2)', size(s% xa_old,dim=2)
               write(*,*) 's% nz_old', s% nz_old
               exit
            end if
            call do2D(s% xh_old, nvar_hydro, nz, action, ierr)
            if (failed('xh_old')) exit

            return
         end do
         
         ierr = -1
         
         contains
         
         logical function failed(str)
            character (len=*), intent(in) :: str
            failed = .false.
            if (ierr == 0) return
            write(*,*) 'star_info_old_arrays failed for ' // trim(str)
            failed = .true.
         end function failed
             
      end subroutine star_info_old_arrays
      
      
      subroutine alloc_star_info_older_arrays(s, ierr)
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         call star_info_older_arrays(s, do_allocate, ierr)
      end subroutine alloc_star_info_older_arrays
      
      
      subroutine star_info_older_arrays(s, action, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: action
         integer, intent(out) :: ierr
         
         integer :: nz, species, nvar_hydro
         ierr = 0
         
         nz = s% nz_older
         nvar_hydro = s% nvar_hydro
         species = s% species

         do ! just so can exit on failure
         
            call do1D(s% conv_vel_older, nz, action, ierr)
            if (failed('conv_vel_older')) exit
            call do1D(s% Del_T_older, nz, action, ierr)
            if (failed('Del_T_older')) exit
            
            call do1D(s% nu_ST_older, nz, action, ierr)
            if (failed('nu_ST_older')) exit
            call do1D(s% D_ST_older, nz, action, ierr)
            if (failed('D_ST_older')) exit
            call do1D(s% D_DSI_older, nz, action, ierr)
            if (failed('D_DSI_older')) exit
            call do1D(s% D_SH_older, nz, action, ierr)
            if (failed('D_SH_older')) exit
            call do1D(s% D_SSI_older, nz, action, ierr)
            if (failed('D_SSI_older')) exit
            call do1D(s% D_ES_older, nz, action, ierr)
            if (failed('D_ES_older')) exit
            call do1D(s% D_GSF_older, nz, action, ierr)
            if (failed('D_GSF_older')) exit
            
            call do1D(s% dq_older, nz, action, ierr)
            if (failed('dq_older')) exit
            call do1D(s% q_older, nz, action, ierr)
            if (failed('q_older')) exit
            call do1D(s% omega_older, nz, action, ierr)
            if (failed('omega_older')) exit
            call do2D(s% xa_older, species, nz, action, ierr)
            if (failed('xa_older')) exit
            call do2D(s% xh_older, nvar_hydro, nz, action, ierr)
            if (failed('xh_older')) exit
            
            return
         end do
         
         ierr = -1
         write(*,*) 'action', action
         
         contains
         
         logical function failed(str)
            character (len=*), intent(in) :: str
            failed = .false.
            if (ierr == 0) return
            write(*,*) 'star_info_older_arrays failed for ' // trim(str)
            failed = .true.
         end function failed
             
      end subroutine star_info_older_arrays
      
      
      subroutine alloc_star_info_prv_arrays(s, ierr)
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         call star_info_prv_arrays(s, do_allocate, ierr)
      end subroutine alloc_star_info_prv_arrays
      
      
      subroutine star_info_prv_arrays(s, action, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: action
         integer, intent(out) :: ierr
         
         integer :: nz, species, nvar
         ierr = 0
         return
         
         
         
         !nz = s% nz_prv
         nvar = s% nvar
         species = s% species

         do ! just so can exit on failure
         
            !call do1D(s% q_prv, nz, action, ierr)
            if (failed('q_prv')) exit
            
            return
         end do
         
         ierr = -1
         write(*,*) 'action', action
         
         contains
         
         logical function failed(str)
            character (len=*), intent(in) :: str
            failed = .false.
            if (ierr == 0) return
            write(*,*) 'star_info_prv_arrays failed for ' // trim(str)
            failed = .true.
         end function failed
             
      end subroutine star_info_prv_arrays


      subroutine free_star_info_arrays(s)
         type (star_info), pointer :: s
         integer :: ierr
         type (star_info), pointer :: c => null()
         call star_info_arrays(s, c, do_deallocate, ierr)
         if (ierr /= 0) write(*,*) 'free_star_info_arrays failed'
      end subroutine free_star_info_arrays
      
      
      subroutine allocate_star_info_arrays(s, ierr)
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         type (star_info), pointer :: c => null()
         call star_info_arrays(s, c, do_allocate, ierr)
         if (ierr /= 0) write(*,*) 'allocate_star_info_arrays failed'
      end subroutine allocate_star_info_arrays
      
      
      subroutine prune_star_info_arrays(s, ierr)
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         type (star_info), pointer :: c => null()
         call star_info_arrays(s, c, do_remove_from_center, ierr)
         if (ierr /= 0) write(*,*) 'prune_star_info_arrays failed'
      end subroutine prune_star_info_arrays
      
      
      subroutine resize_star_info_arrays(s, c, ierr)
         type (star_info), pointer :: s, c
         integer, intent(out) :: ierr
         call star_info_arrays(s, c, do_copy_pointers_and_resize, ierr)
         if (ierr /= 0) write(*,*) 'resize_star_info_arrays failed'
      end subroutine resize_star_info_arrays


      subroutine star_info_arrays(s, c_in, action_in, ierr)
         use eos_def, only:num_eos_basic_results
         use rates_def, only:num_rvs
         use chem_def, only: num_categories
         use const_def, only: standard_cgrav
         type (star_info), pointer :: s, c_in
         integer, intent(in) :: action_in
         integer, intent(out) :: ierr
         
         integer :: nz, species, num_reactions, &
            nvar, nvar_hydro, nvar_chem, nz_new, action
         type (star_info), pointer :: c
         
         include 'formats'
         
         ierr = 0
         
         nz = s% nz
         species = s% species
         num_reactions = s% num_reactions
         nvar = s% nvar
         nvar_hydro = s% nvar_hydro
         nvar_chem = s% nvar_chem         
         nz_new = nz + nz_alloc_extra
         
         c => s
         action = action_in
         if (action == do_copy_pointers_and_resize) then
            if (associated(c_in)) then
               c => c_in
            else ! nothing to copy, so switch to allocate
               action = do_allocate
            end if
         end if
         
         do ! just so can exit on failure

            if (failed('dq')) exit
            call do1(s% dq, c% dq)
            if (failed('dq')) exit
            call do1(s% q, c% q)
            if (failed('q')) exit           
            call do2(s% xa, c% xa, species)
            if (failed('xa')) exit            
            call do2(s% xh, c% xh, nvar_hydro)
            if (failed('xh')) exit
            call do2(s% xh_pre, c% xh_pre, nvar_hydro)
            if (failed('xh_pre')) exit

            call do1(s% lnd, c% lnd)
            if (failed('lnd')) exit
            call do1(s% lnT, c% lnT)
            if (failed('lnT')) exit
            call do1(s% lnR , c% lnR)
            if (failed('lnR')) exit
            call do1(s% L, c% L)
            if (failed('L')) exit
            call do1(s% v, c% v)
            if (failed('v')) exit
            
            call do1(s% j_rot, c% j_rot)
            if (failed('j_rot')) exit
         
            call do1(s% dlnd_dt, c% dlnd_dt)
            if (failed('dlnd_dt')) exit
            call do1(s% dlnPgas_dt, c% dlnPgas_dt)
            if (failed('dlnPgas_dt')) exit
            call do1(s% dlnT_dt, c% dlnT_dt)
            if (failed('dlnT_dt')) exit
            call do1(s% dlnR_dt, c% dlnR_dt)
            if (failed('dlnR_dt')) exit
            call do1(s% dv_dt, c% dv_dt)
            if (failed('dv_dt')) exit
            
            call do1(s% m, c% m)
            if (failed('m')) exit
            call do1(s% dm, c% dm)
            if (failed('dm')) exit
            call do1(s% dm_bar, c% dm_bar)
            if (failed('dm_bar')) exit
            
            call do1(s% T, c% T)
            if (failed('T')) exit
            call do1(s% rho, c% rho)
            if (failed('rho')) exit
            call do1(s% r, c% r)
            if (failed('r')) exit
            call do1(s% rmid, c% rmid)
            if (failed('rmid')) exit
            
            call do1(s% X, c% X)
            if (failed('X')) exit
            call do1(s% Y, c% Y)
            if (failed('Y')) exit
            call do1(s% abar, c% abar)
            if (failed('abar')) exit
            call do1(s% zbar, c% zbar)
            if (failed('zbar')) exit
            call do1(s% z2bar, c% z2bar)
            if (failed('z2bar')) exit
            call do1(s% ye, c% ye)
            if (failed('ye')) exit
            
            call do1(s% mass_correction, c% mass_correction)
            if (failed('mass_correction')) exit
            call do1(s% m_grav, c% m_grav)
            if (failed('m_grav')) exit
            
            call do1(s% P, c% P)
            if (failed('P')) exit
            call do1(s% lnP, c% lnP)
            if (failed('lnP')) exit
            call do1(s% Prad, c% Prad)
            if (failed('Prad')) exit
            call do1(s% lnPgas, c% lnPgas)
            if (failed('lnPgas')) exit
            call do1(s% Pgas, c% Pgas)
            if (failed('Pgas')) exit
            call do1(s% lnE, c% lnE)
            if (failed('lnE')) exit
            call do1(s% grada, c% grada)
            if (failed('grada')) exit
            call do1(s% dE_dRho, c% dE_dRho)
            if (failed('dE_dRho')) exit
            call do1(s% Cv, c% Cv)
            if (failed('Cv')) exit
            call do1(s% Cp, c% Cp)
            if (failed('Cp')) exit
            call do1(s% lnS, c% lnS)
            if (failed('lnS')) exit
            call do1(s% d_entropy_dlnd, c% d_entropy_dlnd)
            if (failed('d_entropy_dlnd')) exit
            call do1(s% d_entropy_dlnT, c% d_entropy_dlnT)
            if (failed('d_entropy_dlnT')) exit
            call do1(s% gamma1, c% gamma1)
            if (failed('gamma1')) exit
            call do1(s% gamma3, c% gamma3)
            if (failed('gamma3')) exit
            call do1(s% eta, c% eta)
            if (failed('eta')) exit
            call do1(s% theta_e, c% theta_e)
            if (failed('theta_e')) exit
            call do1(s% gam, c% gam)
            if (failed('gam')) exit
            call do1(s% mu, c% mu)
            if (failed('mu')) exit
            call do1(s% lnfree_e, c% lnfree_e)
            if (failed('lnfree_e')) exit
            call do1(s% chiRho, c% chiRho)
            if (failed('chiRho')) exit
            call do1(s% chiT, c% chiT)
            if (failed('chiT')) exit
            call do2(s% d_eos_dlnd, c% d_eos_dlnd, num_eos_basic_results)
            if (failed('d_eos_dlnd')) exit
            call do2(s% d_eos_dlnT, c% d_eos_dlnT, num_eos_basic_results)
            if (failed('d_eos_dlnT')) exit
            call do2(s% d_eos_dabar, c% d_eos_dabar, num_eos_basic_results)
            if (failed('d_eos_dabar')) exit
            call do2(s% d_eos_dzbar, c% d_eos_dzbar, num_eos_basic_results)
            if (failed('d_eos_dzbar')) exit
            call do1(s% dlnRho_dlnPgas_const_T, c% dlnRho_dlnPgas_const_T)
            if (failed('dlnRho_dlnPgas_const_T')) exit
            call do1(s% dlnRho_dlnT_const_Pgas, c% dlnRho_dlnT_const_Pgas)
            if (failed('dlnRho_dlnT_const_Pgas')) exit
            
            ! other model variables
            call do1(s% velocity, c% velocity)
            if (failed('velocity')) exit
            call do1(s% csound, c% csound)
            if (failed('csound')) exit
            call do1(s% csound_at_face, c% csound_at_face)
            if (failed('csound_at_face')) exit
            call do1(s% rho_face, c% rho_face)
            if (failed('rho_face')) exit
            call do1(s% scale_height, c% scale_height)
            if (failed('scale_height')) exit
            call do1(s% v_div_csound, c% v_div_csound)
            if (failed('v_div_csound')) exit
            call do1(s% entropy, c% entropy)
            if (failed('entropy')) exit
            call do1(s% grav, c% grav)
            if (failed('grav')) exit
            call do1(s% tau, c% tau)
            if (failed('tau')) exit
            call do1(s% dr_div_csound, c% dr_div_csound)
            if (failed('dr_div_csound')) exit
            
            call do1(s% area, c% area)
            if (failed('area')) exit
            call do1(s% Amid, c% Amid)
            if (failed('Amid')) exit
            call do1(s% dAmid_dlnR00, c% dAmid_dlnR00)
            if (failed('dAmid_dlnR00')) exit
            call do1(s% dAmid_dlnRp1, c% dAmid_dlnRp1)
            if (failed('dAmid_dlnRp1')) exit
            
            call do1(s% omega, c% omega)
            if (failed('omega')) exit
            call do1(s% i_rot, c% i_rot)
            if (failed('i_rot')) exit
            call do1(s% fp_rot, c% fp_rot)
            if (failed('fp_rot')) exit
            call do1(s% ft_rot, c% ft_rot)
            if (failed('ft_rot')) exit
            call do1(s% r_polar, c% r_polar)
            if (failed('r_polar')) exit
            call do1(s% r_equatorial, c% r_equatorial)
            if (failed('r_equatorial')) exit
            call do1(s% am_nu, c% am_nu)
            if (failed('am_nu')) exit
            call do1(s% am_sig, c% am_sig)
            if (failed('am_sig')) exit

            call do1(s% adjust_mlt_gradT_fraction, c% adjust_mlt_gradT_fraction)
            if (failed('adjust_mlt_gradT_fraction')) exit

            call do1(s% domega_dlnR, c% domega_dlnR)
            if (failed('domega_dlnR')) exit
            call do1(s% richardson_number, c% richardson_number)
            if (failed('richardson_number')) exit
            
            call do1(s% D_mix_non_rotation, c% D_mix_non_rotation)
            if (failed('D_mix_non_rotation')) exit
            call do1(s% D_visc, c% D_visc)
            if (failed('D_visc')) exit
            call do1(s% D_DSI, c% D_DSI)
            if (failed('D_DSI')) exit
            call do1(s% D_SH, c% D_SH)
            if (failed('D_SH')) exit
            call do1(s% D_SSI, c% D_SSI)
            if (failed('D_SSI')) exit
            call do1(s% D_ES, c% D_ES)
            if (failed('D_ES')) exit
            call do1(s% D_GSF, c% D_GSF)
            if (failed('D_GSF')) exit

            call do1(s% D_ST, c% D_ST)
            if (failed('D_ST')) exit
            call do1(s% nu_ST, c% nu_ST)
            if (failed('nu_ST')) exit
            call do1(s% omega_shear, c% omega_shear)
            if (failed('omega_shear')) exit

            call do1(s% dynamo_B_r, c% dynamo_B_r)
            if (failed('dynamo_B_r')) exit
            call do1(s% dynamo_B_phi, c% dynamo_B_phi)
            if (failed('dynamo_B_phi')) exit

            
            call do1(s% opacity, c% opacity)
            if (failed('opacity')) exit
            call do1(s% d_opacity_dlnd, c% d_opacity_dlnd)
            if (failed('d_opacity_dlnd')) exit
            call do1(s% d_opacity_dlnT, c% d_opacity_dlnT)
            if (failed('d_opacity_dlnT')) exit
            call do1(s% kap_frac_Type2, c% kap_frac_Type2)
            if (failed('kap_frac_Type2')) exit
            
            call do1(s% eps_nuc, c% eps_nuc)
            if (failed('eps_nuc')) exit
            call do1(s% d_epsnuc_dlnd, c% d_epsnuc_dlnd)
            if (failed('d_epsnuc_dlnd')) exit
            call do1(s% d_epsnuc_dlnT, c% d_epsnuc_dlnT)
            if (failed('d_epsnuc_dlnT')) exit
            call do2(s% d_epsnuc_dx, c% d_epsnuc_dx, species)
            if (failed('d_epsnuc_dx')) exit

            call do1(s% eps_nuc_neu_total, c% eps_nuc_neu_total)
            if (failed('eps_nuc_neu_total')) exit

            call do2(s% dxdt_nuc, c% dxdt_nuc, species)
            if (failed('dxdt_nuc')) exit
            call do2(s% dxdt_dRho, c% dxdt_dRho, species)
            if (failed('dxdt_dRho')) exit
            call do2(s% dxdt_dT, c% dxdt_dT, species)
            if (failed('dxdt_dT')) exit
            call do3(s% d_dxdt_dx, c% d_dxdt_dx, species, species)
            if (failed('d_dxdt_dx')) exit
            
            if (action /= do_check_size) then
               call do3(s% rate_screened, c% rate_screened, num_rvs, num_reactions)
               if (failed('rate_screened')) exit
               call do3(s% rate_raw, c% rate_raw, num_rvs, num_reactions)
               if (failed('rate_raw')) exit
               call do3(s% reaction_eps_nuc, c% reaction_eps_nuc, num_rvs, num_reactions)
               if (failed('reaction_eps_nuc')) exit
               call do3(s% eps_nuc_categories, c% eps_nuc_categories, num_rvs, num_categories)
               if (failed('eps_nuc_categories')) exit
               call do2(s% luminosity_by_category, c% luminosity_by_category, num_categories)
               if (failed('luminosity_by_category')) exit
            end if

            call do1(s% nse_fraction, c% nse_fraction)
            if (failed('nse_fraction')) exit
                    
            call do2(s% diffusion_D_self, c% diffusion_D_self, species)
            if (failed('diffusion_D_self')) exit
            call do2(s% edv, c% edv, species)
            if (failed('edv')) exit
            call do2(s% v_rad, c% v_rad, species)
            if (failed('v_rad')) exit
            call do2(s% g_rad, c% g_rad, species)
            if (failed('g_rad')) exit
            call do2(s% typical_charge, c% typical_charge, species)
            if (failed('typical_charge')) exit
            call do2(s% diffusion_dX, c% diffusion_dX, species)
            if (failed('diffusion_dX')) exit
            call do1(s% E_field, c% E_field)
            if (failed('E_field')) exit
                  
            call do1(s% non_nuc_neu, c% non_nuc_neu)
            if (failed('non_nuc_neu')) exit
            call do1(s% d_nonnucneu_dlnd, c% d_nonnucneu_dlnd)
            if (failed('d_nonnucneu_dlnd')) exit
            call do1(s% d_nonnucneu_dlnT, c% d_nonnucneu_dlnT)
            if (failed('d_nonnucneu_dlnT')) exit
            
            call do1(s% nonnucneu_plas, c% nonnucneu_plas)
            if (failed('nonnucneu_plas')) exit
            call do1(s% nonnucneu_brem, c% nonnucneu_brem)
            if (failed('nonnucneu_brem')) exit
            call do1(s% nonnucneu_phot, c% nonnucneu_phot)
            if (failed('nonnucneu_phot')) exit
            call do1(s% nonnucneu_pair, c% nonnucneu_pair)
            if (failed('nonnucneu_pair')) exit
            call do1(s% nonnucneu_reco, c% nonnucneu_reco)
            if (failed('nonnucneu_reco')) exit
            
            call do1(s% extra_opacity_factor, c% extra_opacity_factor)
            if (failed('extra_opacity_factor')) exit

            call do1(s% irradiation_heat, c% irradiation_heat)
            if (failed('irradiation_heat')) exit
            call do1(s% extra_heat, c% extra_heat)
            if (failed('extra_heat')) exit
            call do1(s% d_extra_heat_dlnd, c% d_extra_heat_dlnd)
            if (failed('d_extra_heat_dlnd')) exit
            call do1(s% d_extra_heat_dlnT, c% d_extra_heat_dlnT)
            if (failed('d_extra_heat_dlnT')) exit
            call do1(s% d_extra_heat_dv00, c% d_extra_heat_dv00)
            if (failed('d_extra_heat_dv00')) exit
            call do1(s% d_extra_heat_dvp1, c% d_extra_heat_dvp1)
            if (failed('d_extra_heat_dvp1')) exit
            call do1(s% d_extra_heat_dlnR00, c% d_extra_heat_dlnR00)
            if (failed('d_extra_heat_dlnR00')) exit
            call do1(s% d_extra_heat_dlnRp1, c% d_extra_heat_dlnRp1)
            if (failed('d_extra_heat_dlnRp1')) exit

            call do1(s% extra_jdot, c% extra_jdot)
            if (failed('extra_jdot')) exit
            call do1(s% extra_omegadot, c% extra_omegadot)
            if (failed('extra_omegadot')) exit

            call do1(s% d_extra_jdot_domega_m1, c% d_extra_jdot_domega_m1)
            if (failed('d_extra_jdot_domega_m1')) exit
            call do1(s% d_extra_omegadot_domega_m1, c% d_extra_omegadot_domega_m1)
            if (failed('d_extra_omegadot_domega_m1')) exit
            call do1(s% d_extra_jdot_domega_00, c% d_extra_jdot_domega_00)
            if (failed('d_extra_jdot_domega_00')) exit
            call do1(s% d_extra_omegadot_domega_00, c% d_extra_omegadot_domega_00)
            if (failed('d_extra_omegadot_domega_00')) exit
            call do1(s% d_extra_jdot_domega_p1, c% d_extra_jdot_domega_p1)
            if (failed('d_extra_jdot_domega_p1')) exit
            call do1(s% d_extra_omegadot_domega_p1, c% d_extra_omegadot_domega_p1)
            if (failed('d_extra_omegadot_domega_p1')) exit

            call do1(s% cgrav, c% cgrav)
            if (failed('cgrav')) exit
            if (action == do_allocate .or. action == do_copy_pointers_and_resize) &
               s% cgrav(1:nz) = standard_cgrav
            
            call do1(s% eps_grav, c% eps_grav)
            if (failed('eps_grav')) exit
            call do1(s% d_eps_grav_dlndm1, c% d_eps_grav_dlndm1)
            if (failed('d_eps_grav_dlndm1')) exit
            call do1(s% d_eps_grav_dlnd00, c% d_eps_grav_dlnd00)
            if (failed('d_eps_grav_dlnd00')) exit
            call do1(s% d_eps_grav_dlndp1, c% d_eps_grav_dlndp1)
            if (failed('d_eps_grav_dlndp1')) exit
            call do1(s% d_eps_grav_dlnTm1, c% d_eps_grav_dlnTm1)
            if (failed('d_eps_grav_dlnTm1')) exit
            call do1(s% d_eps_grav_dlnT00, c% d_eps_grav_dlnT00)
            if (failed('d_eps_grav_dlnT00')) exit
            call do1(s% d_eps_grav_dlnTp1, c% d_eps_grav_dlnTp1)
            if (failed('d_eps_grav_dlnTp1')) exit
            call do1(s% d_eps_grav_dlnR00, c% d_eps_grav_dlnR00)
            if (failed('d_eps_grav_dlnR00')) exit
            call do1(s% d_eps_grav_dL00, c% d_eps_grav_dL00)
            if (failed('d_eps_grav_dL00')) exit
            call do1(s% d_eps_grav_dlnPgasm1_const_T, c% d_eps_grav_dlnPgasm1_const_T)
            if (failed('d_eps_grav_dlnPgasm1_const_T')) exit
            call do1(s% d_eps_grav_dlnPgas00_const_T, c% d_eps_grav_dlnPgas00_const_T)
            if (failed('d_eps_grav_dlnPgas00_const_T')) exit
            call do1(s% d_eps_grav_dlnPgasp1_const_T, c% d_eps_grav_dlnPgasp1_const_T)
            if (failed('d_eps_grav_dlnPgasp1_const_T')) exit
            call do1(s% d_eps_grav_dlnTm1_const_Pgas, c% d_eps_grav_dlnTm1_const_Pgas)
            if (failed('d_eps_grav_dlnTm1_const_Pgas')) exit
            call do1(s% d_eps_grav_dlnT00_const_Pgas, c% d_eps_grav_dlnT00_const_Pgas)
            if (failed('d_eps_grav_dlnT00_const_Pgas')) exit
            call do1(s% d_eps_grav_dlnTp1_const_Pgas, c% d_eps_grav_dlnTp1_const_Pgas)
            if (failed('d_eps_grav_dlnTp1_const_Pgas')) exit
            call do1(s% d_eps_grav_dlnRp1, c% d_eps_grav_dlnRp1)
            if (failed('d_eps_grav_dlnRp1')) exit
            call do1(s% d_eps_grav_dv00, c% d_eps_grav_dv00)
            if (failed('d_eps_grav_dv00')) exit
            call do1(s% d_eps_grav_dvp1, c% d_eps_grav_dvp1)
            if (failed('d_eps_grav_dvp1')) exit
            call do1(s% eps_grav_composition_term, c% eps_grav_composition_term)
            if (failed('eps_grav_composition_term')) exit

            call do1_integer(s% num_steps, c% num_steps)
            if (failed('num_steps')) exit
            call do1_integer(s% mtx_solve, c% mtx_solve)
            if (failed('mtx_solve')) exit
            call do1_integer(s% mtx_factor, c% mtx_factor)
            if (failed('mtx_factor')) exit
            call do1(s% avg_order, c% avg_order)
            if (failed('avg_order')) exit

            call do1_integer(s% mlt_mixing_type, c% mlt_mixing_type)
            if (failed('mlt_mixing_type')) exit
            call do1(s% mlt_mixing_length, c% mlt_mixing_length)
            if (failed('mlt_mixing_length')) exit
            call do1(s% mlt_D, c% mlt_D)
            if (failed('mlt_D')) exit
            call do1(s% mlt_vc, c% mlt_vc)
            if (failed('mlt_vc')) exit
            call do1(s% mlt_Gamma, c% mlt_Gamma)
            if (failed('mlt_Gamma')) exit
            call do1(s% gradT_sub_grada, c% gradT_sub_grada)
            if (failed('gradT_sub_grada')) exit
            call do1(s% grada_at_face, c% grada_at_face)
            if (failed('grada_at_face')) exit

            call do1(s% mlt_cdc, c% mlt_cdc)
            if (failed('mlt_cdc')) exit
            call do1(s% Del_T, c% Del_T)
            if (failed('Del_T')) exit
            call do1(s% cdc, c% cdc)
            if (failed('cdc')) exit
            
            call do1(s% D_mix, c% D_mix)
            if (failed('D_mix')) exit
            call do1(s% conv_vel, c% conv_vel)
            if (failed('conv_vel')) exit
            call do1_integer(s% mixing_type, c% mixing_type)
            if (failed('mixing_type')) exit
            call do1(s% mixing_type_change_dq, c% mixing_type_change_dq)
            if (failed('mixing_type_change_dq')) exit
            call do1_logical(s% newly_nonconvective, c% newly_nonconvective)
            if (failed('newly_nonconvective')) exit
            
            call do1(s% actual_gradT, c% actual_gradT)
            if (failed('actual_gradT')) exit
            call do1(s% gradT, c% gradT)
            if (failed('gradT')) exit
            call do1(s% d_gradT_dlnd00, c% d_gradT_dlnd00)
            if (failed('d_gradT_dlnd00')) exit
            call do1(s% d_gradT_dlnT00, c% d_gradT_dlnT00)
            if (failed('d_gradT_dlnT00')) exit
            call do1(s% d_gradT_dlndm1, c% d_gradT_dlndm1)
            if (failed('d_gradT_dlndm1')) exit
            call do1(s% d_gradT_dlnTm1, c% d_gradT_dlnTm1)
            if (failed('d_gradT_dlnTm1')) exit
            call do1(s% d_gradT_dlnR, c% d_gradT_dlnR)
            if (failed('d_gradT_dlnR')) exit
            call do1(s% d_gradT_dL, c% d_gradT_dL)
            if (failed('d_gradT_dL')) exit
            
            call do1(s% gradr, c% gradr)
            if (failed('gradr')) exit
            call do1(s% d_gradr_dlnd00, c% d_gradr_dlnd00)
            if (failed('d_gradr_dlnd00')) exit
            call do1(s% d_gradr_dlnT00, c% d_gradr_dlnT00)
            if (failed('d_gradr_dlnT00')) exit
            call do1(s% d_gradr_dlndm1, c% d_gradr_dlndm1)
            if (failed('d_gradr_dlndm1')) exit
            call do1(s% d_gradr_dlnTm1, c% d_gradr_dlnTm1)
            if (failed('d_gradr_dlnTm1')) exit
            call do1(s% d_gradr_dlnR, c% d_gradr_dlnR)
            if (failed('d_gradr_dlnR')) exit
            call do1(s% d_gradr_dL, c% d_gradr_dL)
            if (failed('d_gradr_dL')) exit
            
            call do1(s% conv_dP_term, c% conv_dP_term)
            if (failed('conv_dP_term')) exit
            call do1(s% d_conv_dP_term_dlnd00, c% d_conv_dP_term_dlnd00)
            if (failed('d_conv_dP_term_dlnd00')) exit
            call do1(s% d_conv_dP_term_dlnT00, c% d_conv_dP_term_dlnT00)
            if (failed('d_conv_dP_term_dlnT00')) exit
            call do1(s% d_conv_dP_term_dlndm1, c% d_conv_dP_term_dlndm1)
            if (failed('d_conv_dP_term_dlndm1')) exit
            call do1(s% d_conv_dP_term_dlnTm1, c% d_conv_dP_term_dlnTm1)
            if (failed('d_conv_dP_term_dlnTm1')) exit
            call do1(s% d_conv_dP_term_dlnR, c% d_conv_dP_term_dlnR)
            if (failed('d_conv_dP_term_dlnR')) exit
            call do1(s% d_conv_dP_term_dL, c% d_conv_dP_term_dL)
            if (failed('d_conv_dP_term_dL')) exit
            
            call do1(s% grad_density, c% grad_density)
            if (failed('grad_density')) exit
            call do1(s% grad_temperature, c% grad_temperature)
            if (failed('grad_temperature')) exit
            call do1(s% gradL, c% gradL)
            if (failed('gradL')) exit
            call do1(s% gradL_composition_term, c% gradL_composition_term)
            if (failed('gradL_composition_term')) exit
            call do1(s% gradL_composition_term_start, c% gradL_composition_term_start)
            if (failed('gradL_composition_term_start')) exit
            
            call do1_integer( &
               s% dominant_iso_for_thermohaline, c% dominant_iso_for_thermohaline)
            if (failed('dominant_iso_for_thermohaline')) exit

            call do1(s% sig, c% sig)
            if (failed('sig')) exit
            call do1(s% sig_div_siglim, c% sig_div_siglim)
            if (failed('sig_div_siglim')) exit
            call do1(s% am_sig_div_am_siglim, c% am_sig_div_am_siglim)
            if (failed('am_sig_div_am_siglim')) exit
            
            call do1(s% cdc_prev, c% cdc_prev)
            if (failed('cdc_prev')) exit
            
            call do1(s% brunt_N2, c% brunt_N2)
            if (failed('brunt_N2')) exit
            call do1(s% brunt_N2_composition_term, c% brunt_N2_composition_term)
            if (failed('brunt_N2_composition_term')) exit
            call do1(s% brunt_B, c% brunt_B)
            if (failed('brunt_B')) exit
            
            call do1(s% dlnP_dm, c% dlnP_dm)
            if (failed('dlnP_dm')) exit
            call do1(s% dlnT_dm, c% dlnT_dm)
            if (failed('dlnT_dm')) exit
            call do1(s% dL_dm, c% dL_dm)
            if (failed('dL_dm')) exit

            call do1(s% Pvisc, c% Pvisc)
            if (failed('Pvisc')) exit
            call do1(s% dPvisc_dlnd, c% dPvisc_dlnd)
            if (failed('dPvisc_dlnd')) exit
            call do1(s% dPvisc_dlnT, c% dPvisc_dlnT)
            if (failed('dPvisc_dlnT')) exit
            call do1(s% dPvisc_dlnR00, c% dPvisc_dlnR00)
            if (failed('dPvisc_dlnR00')) exit
            call do1(s% dPvisc_dvel00, c% dPvisc_dvel00)
            if (failed('dPvisc_dvel00')) exit
            call do1(s% dPvisc_dlnRp1, c% dPvisc_dlnRp1)
            if (failed('dPvisc_dlnRp1')) exit
            call do1(s% dPvisc_dvelp1, c% dPvisc_dvelp1)
            if (failed('dPvisc_dvelp1')) exit
            
            call do1(s% L_nuc_burn, c% L_nuc_burn)
            if (failed('L_nuc_burn')) exit
            call do1(s% L_nuc_photo, c% L_nuc_photo)
            if (failed('L_nuc_photo')) exit
            call do2(s% L_nuc_by_category, c% L_nuc_by_category, num_categories)
            if (failed('L_nuc_by_category')) exit
            call do1(s% L_neutrinos, c% L_neutrinos)
            if (failed('L_neutrinos')) exit
            call do1(s% L_grav, c% L_grav)
            if (failed('L_grav')) exit

            call do1(s% d_abar_dlnd, c% d_abar_dlnd)
            if (failed('d_abar_dlnd')) exit
            call do1(s% d_abar_dlnT, c% d_abar_dlnT)
            if (failed('d_abar_dlnT')) exit
            call do1(s% d_zbar_dlnd, c% d_zbar_dlnd)
            if (failed('d_zbar_dlnd')) exit
            call do1(s% d_zbar_dlnT, c% d_zbar_dlnT)
            if (failed('d_zbar_dlnT')) exit

            call do1(s% del_t_for_just_added, c% del_t_for_just_added)
            if (failed('del_t_for_just_added')) exit      
                 
            call do1(s% lnd_for_d_dt, c% lnd_for_d_dt)
            if (failed('lnd_for_d_dt')) exit            
            call do1(s% lnPgas_for_d_dt, c% lnPgas_for_d_dt)
            if (failed('lnPgas_for_d_dt')) exit            
            call do1(s% lnT_for_d_dt, c% lnT_for_d_dt)
            if (failed('lnT_for_d_dt')) exit            
            call do1(s% lnR_for_d_dt, c% lnR_for_d_dt)
            if (failed('lnR_for_d_dt')) exit            
            call do1(s% v_for_d_dt, c% v_for_d_dt)
            if (failed('v_for_d_dt')) exit            
            
            call do2(s% xa_pre, c% xa_pre, species)
            if (failed('xa_pre')) exit

            call do1(s% lnS_pre, c% lnS_pre)
            if (failed('lnS_pre')) exit
            
            call do1(s% lnd_start, c% lnd_start)
            if (failed('lnd_start')) exit
            call do1(s% lnP_start, c% lnP_start)
            if (failed('lnP_start')) exit
            call do1(s% lnT_start, c% lnT_start)
            if (failed('lnT_start')) exit
            call do1(s% lnR_start, c% lnR_start)
            if (failed('lnR_start')) exit
            call do1(s% L_start, c% L_start)
            if (failed('L_start')) exit
            call do1(s% omega_start, c% omega_start)
            if (failed('omega_start')) exit
            call do1(s% Z_start, c% Z_start)
            if (failed('Z_start')) exit
            call do1(s% ye_start, c% ye_start)
            if (failed('ye_start')) exit
            call do1(s% csound_init, c% csound_init)
            if (failed('csound_init')) exit
            call do1(s% i_rot_start, c% i_rot_start)
            if (failed('i_rot_start')) exit
            call do1(s% P_div_rho_start, c% P_div_rho_start)
            if (failed('P_div_rho_start')) exit
            call do1(s% mass_correction_start, c% mass_correction_start)
            if (failed('mass_correction_start')) exit
            call do1(s% eps_nuc_start, c% eps_nuc_start)
            if (failed('eps_nuc_start')) exit
            call do1(s% non_nuc_neu_start, c% non_nuc_neu_start)
            if (failed('non_nuc_neu_start')) exit
            call do2(s% dxdt_nuc_start, c% dxdt_nuc_start, species)
            if (failed('dxdt_nuc_start')) exit
            call do2(s% luminosity_by_category_start, &
               c% luminosity_by_category_start, num_categories)
            if (failed('luminosity_by_category_start')) exit
            
            call do1(s% gradr_start, c% gradr_start)
            if (failed('gradr_start')) exit
            call do1(s% gradL_start, c% gradL_start)
            if (failed('gradL_start')) exit
            call do1(s% gradT_start, c% gradT_start)
            if (failed('gradT_start')) exit
            call do1(s% actual_gradT_start, c% actual_gradT_start)
            if (failed('actual_gradT_start')) exit
            call do1(s% grada_at_face_start, c% grada_at_face_start)
            if (failed('grada_at_face_start')) exit

            call do1(s% grada_start, c% grada_start)
            if (failed('grada_start')) exit
            call do1(s% chiT_start, c% chiT_start)
            if (failed('chiT_start')) exit
            call do1(s% chiRho_start, c% chiRho_start)
            if (failed('chiRho_start')) exit
            call do1(s% cp_start, c% cp_start)
            if (failed('cp_start')) exit
            call do1(s% cv_start, c% cv_start)
            if (failed('cv_start')) exit
            call do1(s% T_start, c% T_start)
            if (failed('T_start')) exit
            
            call do1(s% abar_start, c% abar_start)
            if (failed('abar_start')) exit
            call do1(s% zbar_start, c% zbar_start)
            if (failed('zbar_start')) exit
            call do1(s% mu_start, c% mu_start)
            if (failed('mu_start')) exit
            
            call do1(s% dlnd_dt_prev_step, c% dlnd_dt_prev_step)
            if (failed('dlnd_dt_prev_step')) exit
            call do1(s% dlnPgas_dt_prev_step, c% dlnPgas_dt_prev_step)
            if (failed('dlnPgas_dt_prev_step')) exit
            call do1(s% dlnT_dt_prev_step, c% dlnT_dt_prev_step)
            if (failed('dlnT_dt_prev_step')) exit

            call do1(s% max_burn_correction, c% max_burn_correction)
            if (failed('max_burn_correction')) exit
            call do1(s% avg_burn_correction, c% avg_burn_correction)
            if (failed('avg_burn_correction')) exit
            
            call do1_neq(s% equ1, c% equ1)
            if (failed('equ1')) exit
            if (action == do_remove_from_center) then
               s% equ(1:nvar,1:nz) => s% equ1(1:nvar*nz)
            else if (action /= do_check_size .and. action /= do_deallocate) then
               !write(*,2) 'action', action
               !write(*,2) 'size(s% equ1,dim=1)', size(s% equ1,dim=1)
               !write(*,2) 'nvar*nz_new', nvar*nz_new
               !write(*,2) 'nz_new', nz_new
               !write(*,2) 'nvar', nvar
               s% equ(1:nvar,1:nz_new) => s% equ1(1:nvar*nz_new)
            end if

            if (action /= do_copy_pointers_and_resize) then
               call do2D_dim1(s% profile_extra, nz, max_num_profile_extras, action, ierr)
               call do2D_dim1(s% history_extra, nz, max_num_history_extras, action, ierr)
            else
               deallocate(s% profile_extra, s% history_extra)
               allocate(s% profile_extra(nz, max_num_profile_extras), stat=ierr)
               allocate(s% history_extra(nz, max_num_history_extras), stat=ierr)
            end if
            if (failed('pstar extras')) exit
            
            return
         end do
         ierr = -1
         
         
         contains
         
         
         subroutine do1(ptr, other)
            real(dp), dimension(:), pointer :: ptr, other
            real(dp), dimension(:), pointer :: tmp
            if (action == do_copy_pointers_and_resize) then
               ptr => other
               if (nz_new <= size(ptr,dim=1)) return
               deallocate(ptr)
               allocate(ptr(nz_new), stat=ierr)
            else if (action == do_allocate) then
               call do1D(ptr, nz_new, action, ierr)
            else
               call do1D(ptr, nz, action, ierr)
            end if
         end subroutine do1
         
         
         subroutine do1_neq(ptr, other)
            real(dp), dimension(:), pointer :: ptr, other
            real(dp), dimension(:), pointer :: tmp
            if (action == do_copy_pointers_and_resize) then
               ptr => other
               if (nvar*nz_new <= size(ptr,dim=1)) return
               deallocate(ptr)
               allocate(ptr(nvar*nz_new), stat=ierr)
            else if (action == do_allocate) then
               call do1D(ptr, nvar*nz_new, action, ierr)
            else
               call do1D(ptr, nvar*nz, action, ierr)
               if (ierr == 0) return
               write(*,*) 'size(ptr,dim=1)', size(ptr,dim=1)
               write(*,*) 'nvar*nz', nvar*nz
               write(*,*) 'nvar', nvar
               write(*,*) 'nz', nz
               write(*,*) 'nz_new', nz_new
               stop 'do1_neq'
            end if
         end subroutine do1_neq
         
         
         subroutine do1_integer(ptr, other)
            integer, dimension(:), pointer :: ptr, other
            integer, dimension(:), pointer :: tmp
            if (action == do_copy_pointers_and_resize) then
               ptr => other
               if (nz_new <= size(ptr,dim=1)) return
               deallocate(ptr)
               allocate(ptr(nz_new), stat=ierr)
            else if (action == do_allocate) then
               call do1D_integer(ptr, nz_new, action, ierr)
            else
               call do1D_integer(ptr, nz, action, ierr)
            end if
         end subroutine do1_integer
         
         
         subroutine do2_integer(ptr, other, sz1)
            integer, dimension(:,:), pointer :: ptr, other
            integer, intent(in) :: sz1
            real(dp), dimension(:,:), pointer :: tmp
            if (action == do_copy_pointers_and_resize) then
               ptr => other
               if (sz1 == size(ptr, dim=1) .and. nz_new <= size(ptr, dim=2)) return
               deallocate(ptr)
               allocate(ptr(sz1, nz_new), stat=ierr)
            else if (action == do_allocate) then
               call do2D_integer(ptr, sz1, nz_new, action, ierr)
            else
               call do2D_integer(ptr, sz1, nz, action, ierr)
            end if
         end subroutine do2_integer
         
         
         subroutine do1_logical(ptr, other)
            logical, dimension(:), pointer :: ptr, other
            logical, dimension(:), pointer :: tmp
            if (action == do_copy_pointers_and_resize) then
               ptr => other
               if (nz_new <= size(ptr,dim=1)) return
               deallocate(ptr)
               allocate(ptr(nz_new), stat=ierr)
            else if (action == do_allocate) then
               call do1D_logical(ptr, nz_new, action, ierr)
            else
               call do1D_logical(ptr, nz, action, ierr)
            end if
         end subroutine do1_logical
         
         
         subroutine do2(ptr, other, sz1)
            real(dp), dimension(:,:), pointer :: ptr, other
            integer, intent(in) :: sz1
            real(dp), dimension(:,:), pointer :: tmp
            if (action == do_copy_pointers_and_resize) then
               ptr => other
               if (sz1 == size(ptr, dim=1) .and. nz_new <= size(ptr, dim=2)) return
               deallocate(ptr)
               allocate(ptr(sz1, nz_new), stat=ierr)
            else if (action == do_allocate) then
               call do2D(ptr, sz1, nz_new, action, ierr)
            else
               call do2D(ptr, sz1, nz, action, ierr)
            end if
         end subroutine do2
         
         
         subroutine do3(ptr, other, sz1, sz2)
            real(dp), dimension(:,:,:), pointer :: ptr, other
            integer, intent(in) :: sz1, sz2
            real(dp), dimension(:,:,:), pointer :: tmp
            if (action == do_copy_pointers_and_resize) then
               ptr => other
               if (sz1 == size(ptr, dim=1) .and. sz2 == size(ptr, dim=2) &
                     .and. nz_new <= size(ptr, dim=3)) return
               deallocate(ptr)
               allocate(ptr(sz1, sz2, nz_new), stat=ierr)
            else if (action == do_allocate) then
               call do3D(ptr, sz1, sz2, nz_new, action, ierr)
            else
               call do3D(ptr, sz1, sz2, nz, action, ierr)
            end if
         end subroutine do3
         
         
         subroutine do2_quad(ptr, other, sz1)
            real(qp), dimension(:,:), pointer :: ptr, other
            integer, intent(in) :: sz1
            real(qp), dimension(:,:), pointer :: tmp
            if (action == do_copy_pointers_and_resize) then
               ptr => other
               if (sz1 == size(ptr, dim=1) .and. nz_new <= size(ptr, dim=2)) return
               deallocate(ptr)
               allocate(ptr(sz1, nz_new), stat=ierr)
            else if (action == do_allocate) then
               call do2D_quad(ptr, sz1, nz_new, action, ierr)
            else
               call do2D_quad(ptr, sz1, nz, action, ierr)
            end if
         end subroutine do2_quad

         
         logical function failed(str)
            character (len=*), intent(in) :: str
            include 'formats'
            failed = .false.
            if (ierr == 0) return
            write(*,*) 'star_info_arrays failed for ' // trim(str)
            write(*,2) 'species', species
            write(*,2) 'nz', nz
            failed = .true.
         end function failed
         
         
      end subroutine star_info_arrays

            
      subroutine do1D(ptr, sz, action, ierr)
         real(dp), dimension(:), pointer :: ptr
         integer, intent(in) :: sz, action
         integer, intent(out) :: ierr
         real(dp), dimension(:), pointer :: ptr2
         integer :: old_sz, j
         ierr = 0
         select case(action)
            case (do_deallocate)
               if (associated(ptr)) then
                  deallocate(ptr)
                  nullify(ptr)
               end if
            case (do_allocate)
               allocate(ptr(sz), stat=ierr)
               if (zero_when_allocate) ptr = 0
               if (fill_arrays_with_NaNs) call fill_with_NaNs(ptr)
            case (do_check_size)
               if (size(ptr,dim=1) < sz) ierr = -1
            case (do_remove_from_center)
               allocate(ptr2(sz), stat=ierr)
               old_sz = size(ptr,dim=1)
               if (old_sz < sz) then
                  ierr = -1; return
               end if
               do j=1,sz
                  ptr2(j) = ptr(j)
               end do
               deallocate(ptr)
               if (ierr /= 0) return
               ptr => ptr2
         end select
      end subroutine do1D
      
      
      subroutine do2D(ptr, sz1, sz2, action, ierr)
         use utils_lib, only: fill_with_NaNs_2D
         real(dp), dimension(:,:), pointer:: ptr
         integer, intent(in) :: sz1, sz2, action
         integer, intent(out) :: ierr
         real(dp), dimension(:,:), pointer :: ptr2
         integer :: old_sz2, j, i
         ierr = 0
         select case(action)
            case (do_deallocate)
               if (associated(ptr)) then
                  deallocate(ptr)
                  nullify(ptr)
               end if
            case (do_allocate)
               allocate(ptr(sz1, sz2), stat=ierr)
               if (zero_when_allocate) ptr = 0
               if (fill_arrays_with_NaNs) call fill_with_NaNs_2D(ptr)
            case (do_check_size)
               if (size(ptr,dim=1) /= sz1) ierr = -1
               if (size(ptr,dim=2) < sz2) ierr = -1
            case (do_remove_from_center)
               allocate(ptr2(sz1, sz2), stat=ierr)
               old_sz2 = size(ptr,dim=2)
               if (old_sz2 < sz2) then
                  ierr = -1; return
               end if
               do i=1,sz1
                  do j=1,sz2
                     ptr2(i,j) = ptr(i,j)
                  end do
               end do
               deallocate(ptr)
               if (ierr /= 0) return
               ptr => ptr2
         end select
      end subroutine do2D
      
      
      subroutine do2D_quad(ptr, sz1, sz2, action, ierr)
         real(qp), dimension(:,:), pointer:: ptr
         integer, intent(in) :: sz1, sz2, action
         integer, intent(out) :: ierr
         real(qp), dimension(:,:), pointer :: ptr2
         integer :: old_sz2, j, i
         ierr = 0
         select case(action)
            case (do_deallocate)
               if (associated(ptr)) then
                  deallocate(ptr)
                  nullify(ptr)
               end if
            case (do_allocate)
               allocate(ptr(sz1, sz2), stat=ierr)
               if (zero_when_allocate) ptr = 0
            case (do_check_size)
               if (size(ptr,dim=1) /= sz1) ierr = -1
               if (size(ptr,dim=2) < sz2) ierr = -1
            case (do_remove_from_center)
               allocate(ptr2(sz1, sz2), stat=ierr)
               old_sz2 = size(ptr,dim=2)
               if (old_sz2 < sz2) then
                  ierr = -1; return
               end if
               do i=1,sz1
                  do j=1,sz2 
                     ptr2(i,j) = ptr(i,j)
                  end do
               end do
               deallocate(ptr)
               if (ierr /= 0) return
               ptr => ptr2
         end select
      end subroutine do2D_quad
      
      
      subroutine do2D_dim1(ptr, sz1, sz2, action, ierr)
         use utils_lib, only: fill_with_NaNs_2D
         real(dp), dimension(:,:), pointer:: ptr
         integer, intent(in) :: sz1, sz2, action
         integer, intent(out) :: ierr
         real(dp), dimension(:,:), pointer :: ptr2
         integer :: old_sz1, j,i
         ierr = 0
         select case(action)
            case (do_deallocate)
               if (associated(ptr)) then
                  deallocate(ptr)
                  nullify(ptr)
               end if
            case (do_allocate)
               allocate(ptr(sz1, sz2), stat=ierr)
               if (zero_when_allocate) ptr = 0
               if (fill_arrays_with_NaNs) call fill_with_NaNs_2D(ptr)
            case (do_check_size)
               if (size(ptr,dim=1) /= sz1) ierr = -1
               if (size(ptr,dim=2) < sz2) ierr = -1
            case (do_remove_from_center)
               allocate(ptr2(sz1, sz2), stat=ierr)
               old_sz1 = size(ptr,dim=1)
               if (old_sz1 < sz1) then
                  ierr = -1; return
               end if
               do j=1,sz2
                  do i=1,sz1
                     ptr2(i,j) = ptr(i,j)
                  end do
               end do
               deallocate(ptr)
               if (ierr /= 0) return
               ptr => ptr2
         end select
      end subroutine do2D_dim1
      
      
      subroutine do3D(ptr, sz1, sz2, sz3, action, ierr)
         use utils_lib, only: fill_with_NaNs_3D
         real(dp), dimension(:,:,:), pointer:: ptr
         integer, intent(in) :: sz1, sz2, sz3, action
         integer, intent(out) :: ierr
         real(dp), dimension(:,:,:), pointer :: ptr2
         integer :: old_sz3, j, i, k
         ierr = 0
         select case(action)
            case (do_deallocate)
               if (associated(ptr)) then
                  deallocate(ptr)
                  nullify(ptr)
               end if
            case (do_allocate)
               allocate(ptr(sz1, sz2, sz3), stat=ierr)
               if (zero_when_allocate) ptr = 0
               if (fill_arrays_with_NaNs) call fill_with_NaNs_3D(ptr)
            case (do_check_size)
               if (size(ptr,dim=1) /= sz1) ierr = -1
               if (size(ptr,dim=2) /= sz2) ierr = -1
               if (size(ptr,dim=3) < sz3) ierr = -1
            case (do_remove_from_center)
               allocate(ptr2(sz1, sz2, sz3), stat=ierr)
               old_sz3 = size(ptr,dim=3)
               if (old_sz3 < sz3) then
                  ierr = -1; return
               end if
               do k=1,sz3
                  do j=1,sz2
                     do i=1,sz1
                        ptr2(i,j,k) = ptr(i,j,k)
                     end do
                  end do
               end do
               deallocate(ptr)
               if (ierr /= 0) return
               ptr => ptr2
         end select
      end subroutine do3D
      
      
      subroutine do4D(ptr, sz1, sz2, sz3, sz4, action, ierr)
         use utils_lib, only: fill_with_NaNs_4D
         real(dp), dimension(:,:,:,:), pointer:: ptr
         integer, intent(in) :: sz1, sz2, sz3, sz4, action
         integer, intent(out) :: ierr
         real(dp), dimension(:,:,:,:), pointer :: ptr2
         integer :: old_sz4, i, j, k, m
         ierr = 0
         select case(action)
            case (do_deallocate)
               if (associated(ptr)) then
                  deallocate(ptr)
                  nullify(ptr)
               end if
            case (do_allocate)
               allocate(ptr(sz1, sz2, sz3, sz4), stat=ierr)
               if (zero_when_allocate) ptr = 0
               if (fill_arrays_with_NaNs) call fill_with_NaNs_4D(ptr)
            case (do_check_size)
               if (size(ptr,dim=1) /= sz1) ierr = -1
               if (size(ptr,dim=2) /= sz2) ierr = -1
               if (size(ptr,dim=3) /= sz3) ierr = -1
               if (size(ptr,dim=4) < sz4) ierr = -1
            case (do_remove_from_center)
               allocate(ptr2(sz1, sz2, sz3, sz4), stat=ierr)
               old_sz4 = size(ptr,dim=4)
               if (old_sz4 < sz4) then
                  ierr = -1; return
               end if
               do m=1,sz4
                  do k=1,sz3
                     do j=1,sz2
                        do i=1,sz1
                           ptr2(i,j,k,m) = ptr(i,j,k,m)
                        end do
                     end do
                  end do
               end do
               deallocate(ptr)
               if (ierr /= 0) return
               ptr => ptr2
         end select
      end subroutine do4D
      
      
      subroutine do1D_integer(ptr, sz, action, ierr)
         integer, dimension(:), pointer:: ptr
         integer, intent(in) :: sz, action
         integer, intent(out) :: ierr
         integer, dimension(:), pointer :: ptr2
         integer :: old_sz, j
         ierr = 0
         select case(action)
            case (do_deallocate)
               if (associated(ptr)) then
                  deallocate(ptr)
                  nullify(ptr)
               end if
            case (do_allocate)
               allocate(ptr(sz), stat=ierr)
               if (zero_when_allocate) ptr = 0
            case (do_check_size)
               if (size(ptr,dim=1) < sz) ierr = -1
            case (do_remove_from_center)
               allocate(ptr2(sz), stat=ierr)
               old_sz = size(ptr,dim=1)
               if (old_sz < sz) then
                  ierr = -1; return
               end if
               do j=1,sz
                  ptr2(j) = ptr(j)
               end do
               deallocate(ptr)
               if (ierr /= 0) return
               ptr => ptr2
         end select
      end subroutine do1D_integer
      
      
      subroutine do2D_integer(ptr, sz1, sz2, action, ierr)
         integer, dimension(:, :), pointer:: ptr
         integer, intent(in) :: sz1, sz2, action
         integer, intent(out) :: ierr
         integer, dimension(:,:), pointer :: ptr2
         integer :: old_sz2, j, i
         ierr = 0
         select case(action)
            case (do_deallocate)
               if (associated(ptr)) then
                  deallocate(ptr)
                  nullify(ptr)
               end if
            case (do_allocate)
               allocate(ptr(sz1, sz2), stat=ierr)
               if (zero_when_allocate) ptr = 0
            case (do_check_size)
               if (size(ptr,dim=1) /= sz1) ierr = -1
               if (size(ptr,dim=2) < sz2) ierr = -1
            case (do_remove_from_center)
               allocate(ptr2(sz1, sz2), stat=ierr)
               old_sz2 = size(ptr,dim=2)
               if (old_sz2 < sz2) then
                  ierr = -1; return
               end if
               do j=1,sz2
                  do i=1,sz1
                     ptr2(i,j) = ptr(i,j)
                  end do
               end do
               deallocate(ptr)
               if (ierr /= 0) return
               ptr => ptr2
         end select
      end subroutine do2D_integer
      
      
      subroutine do1D_logical(ptr, sz, action, ierr)
         logical, dimension(:), pointer:: ptr
         integer, intent(in) :: sz, action
         integer, intent(out) :: ierr
         logical, dimension(:), pointer :: ptr2
         integer :: old_sz, j
         ierr = 0
         select case(action)
            case (do_deallocate)
               if (associated(ptr)) then
                  deallocate(ptr)
                  nullify(ptr)
               end if
            case (do_allocate)
               allocate(ptr(sz), stat=ierr)
               if (zero_when_allocate) ptr = .false.
            case (do_check_size)
               if (size(ptr,dim=1) < sz) ierr = -1
            case (do_remove_from_center)
               allocate(ptr2(sz), stat=ierr)
               old_sz = size(ptr,dim=1)
               if (old_sz < sz) then
                  ierr = -1; return
               end if
               do j=1,sz
                  ptr2(j) = ptr(j)
               end do
               deallocate(ptr)
               if (ierr /= 0) return
               ptr => ptr2
         end select
      end subroutine do1D_logical
      
      
      subroutine do_remove_center(id, k, ierr)
         use star_utils, only: set_qs
         integer, intent(in) :: id, k
         integer, intent(out) :: ierr
         type (star_info), pointer :: s 
         real(dp) :: old_xmstar
         include 'formats'
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) then
            write(*,*) 'do_remove_center: get_star_ptr ierr', ierr
            return
         end if
         if (k <= 0 .or. k > s% nz) return
         old_xmstar = s% xmstar
         s% xmstar = (1d0 - s% q(k))*old_xmstar
         s% M_center = s% mstar - s% xmstar
         s% R_center = s% r(k)
         s% L_center = s% L(k)
         s% nz = k-1
         s% dq(1:k-1) = s% dq(1:k-1)*old_xmstar/s% xmstar
         call set_qs(s% nz, s% q, s% dq, ierr)
         if (ierr /= 0) return
         s% generations = 1 ! memory leak, but not worth worrying about
         write(*,1) 'old_xmstar', old_xmstar
         write(*,1) 's% xmstar', s% xmstar
         write(*,1) 'old_xmstar/s% xmstar', old_xmstar/s% xmstar
         call prune_star_info_arrays(s, ierr)
      end subroutine do_remove_center


      subroutine set_var_info(s, ierr)
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         
         integer :: i

         ierr = 0
         i = 0
         
         s% ode_var(:) = .false.
         
         ! put structure first, abundances last.
         
         i = i+1
         if (s% lnPgas_flag) then
            s% i_xlnd = 0
            s% i_lnPgas = i
            s% equP = s% i_lnPgas
         else
            s% i_lnPgas = 0
            s% i_xlnd = i
            s% equP = s% i_xlnd
         end if

         i = i+1
         s% i_lnT = i
         s% equT = s% i_lnT
         
         i = i+1
         s% i_lum = i
         s% equL = s% i_lum
         
         i = i+1
         s% i_lnR = i
         s% equR = s% i_lnR
                  
         if (s% v_flag) then
            i = i+1; s% i_vel = i
         else
            s% i_vel = 0
         end if
         s% equv = s% i_vel

         s% nvar_hydro = i
         
         s% i_chem1 = s% nvar_hydro + 1
         s% equchem1 = s% i_chem1

         s% nvar = s% nvar_hydro + s% nvar_chem
         
         ! Names of the variables
         if (s% i_xlnd /= 0) s% nameofvar(s% i_xlnd) = 'lnd'
         if (s% i_lnPgas /= 0) s% nameofvar(s% i_lnPgas) = 'lnPgas'
         if (s% i_lnT /= 0) s% nameofvar(s% i_lnT) = 'lnT'
         if (s% i_lnR /= 0) s% nameofvar(s% i_lnR) = 'lnR'
         if (s% i_lum /= 0) s% nameofvar(s% i_lum) = 'lum'
         if (s% i_vel /= 0) s% nameofvar(s% i_vel) = 'vel'

         ! Names of the equations
         if (s% equP /= 0) s% nameofequ(s% equP) = 'equP'
         if (s% equT /= 0) s% nameofequ(s% equT) = 'equT'
         if (s% equR /= 0) s% nameofequ(s% equR) = 'equR'
         if (s% equL /= 0) s% nameofequ(s% equL) = 'equL'
         if (s% equv /= 0) s% nameofequ(s% equv) = 'equv'
      
         ! chem names are done later by set_chem_names when have set up the net

      end subroutine set_var_info
      
      
      subroutine set_chem_names(s)
         use chem_def
         type (star_info), pointer :: s
         integer ::  old_size, i, j, cid
         
         include 'formats'
         
         if (s% nvar_hydro == 0) return ! not ready to set chem names yet
         
         old_size = size(s% nameofvar,dim=1)
         if (old_size < s% nvar) then
            call realloc(s% nameofvar)
            call realloc(s% nameofequ)
            call realloc_logical(s% ode_var)
         end if
         do i=1, s% nvar_chem
            cid = s% chem_id(i)
            j = s% nvar_hydro+i
            s% nameofvar(j) = trim(chem_isos% name(cid))
            s% nameofequ(j) = 'equ_' // trim(chem_isos% name(cid))
            s% ode_var(j) = .true.
         end do
         
         contains
         
         subroutine realloc(p)
            character (len=name_len), dimension(:), pointer :: p
            character (len=name_len), dimension(:), pointer :: old_p
            integer :: cpy_len, j
            old_p => p
            old_size = size(p,dim=1)
            allocate(p(s% nvar))
            cpy_len = min(old_size, s% nvar)
            do j=1,cpy_len
               p(j) = old_p(j)
            end do
            deallocate(old_p)         
         end subroutine realloc
         
         subroutine realloc_logical(p)
            logical, dimension(:), pointer :: p
            logical, dimension(:), pointer :: old_p
            integer :: cpy_len, j
            old_p => p
            old_size = size(p,dim=1)
            allocate(p(s% nvar))
            cpy_len = min(old_size, s% nvar)
            do j=1,cpy_len
               p(j) = old_p(j)
            end do
            deallocate(old_p)         
         end subroutine realloc_logical
         
      end subroutine set_chem_names

      
      subroutine set_lnPgas_flag(id, lnPgas_flag, ierr)
         integer, intent(in) :: id
         logical, intent(in) :: lnPgas_flag
         integer, intent(out) :: ierr         
         type (star_info), pointer :: s
         integer :: k, nz
         logical, parameter :: dbg = .false.
         
         include 'formats'
               
         ierr = 0
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) return
         if (s% lnPgas_flag .eqv. lnPgas_flag) return
         
         nz = s% nz
         s% lnPgas_flag = lnPgas_flag         

         call set_var_info(s, ierr)
         if (ierr /= 0) return    

         if (lnPgas_flag) then
            s% xh(s% i_lnPgas,1:nz) = s% lnPgas(1:nz)
         else
            s% xh(s% i_xlnd,1:nz) = s% lnd(1:nz)
         end if
         
         s% generations = 1 ! for now, don't bother with old and older models

      end subroutine set_lnPgas_flag

      
      subroutine set_v_flag(id, v_flag, ierr)
         integer, intent(in) :: id
         logical, intent(in) :: v_flag
         integer, intent(out) :: ierr         
         type (star_info), pointer :: s
         integer :: nvar_hydro_old, k, nz
         real(dp) :: cs
         logical, parameter :: dbg = .false.
         
         include 'formats'
               
         ierr = 0
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) return
         if (s% v_flag .eqv. v_flag) return
         
         nz = s% nz
         s% v_flag = v_flag         
         nvar_hydro_old = s% nvar_hydro
         
         if (.not. v_flag) then ! remove i_vel's
            call del(s% xh)
            call del(s% xh_pre)
            if (associated(s% xh_old) .and. s% generations > 1) call del(s% xh_old)
            if (associated(s% xh_older) .and. s% generations > 2) call del(s% xh_older)
         end if

         call set_var_info(s, ierr)
         if (ierr /= 0) return    
              
         call update_nvar_allocs(s, nvar_hydro_old, s% nvar_chem, ierr)
         if (ierr /= 0) return
         
         call check_sizes(s, ierr)
         if (ierr /= 0) return

         if (v_flag) then ! insert i_vel's
            call insert(s% xh)
            call insert(s% xh_pre)
            s% xh(s% i_vel,1:nz) = s% r(1:nz)*s% dlnR_dt(1:nz)
            if (dbg) then
               do k=1,20
                  cs = sqrt(s% gamma1(k)*s% P(k)/s% rho(k))
                  write(*,2) 'v/cs', k, s% xh(s% i_vel,k)/cs, s% xh(s% i_vel,k), cs
               end do
            end if
            if (associated(s% xh_old) .and. s% generations > 1) call insert(s% xh_old)
            if (associated(s% xh_older) .and. s% generations > 2) call insert(s% xh_older)
         end if
         
         call set_chem_names(s)
         
         contains
         
         subroutine del(xs)
            real(dp) :: xs(:,:)
            integer :: j, i_vel
            if (size(xs,dim=2) < nz) return
            i_vel = s% i_vel
            do j = i_vel+1, nvar_hydro_old
               xs(j-1,1:nz) = xs(j,1:nz)
            end do
         end subroutine del
         
         subroutine insert(xs)
            real(dp) :: xs(:,:)
            integer :: j, i_vel
            if (size(xs,dim=2) < nz) return
            i_vel = s% i_vel
            do j = s% nvar_hydro, i_vel+1, -1
               xs(j,1:nz) = xs(j-1,1:nz)
            end do
            xs(i_vel,1:nz) = 0
         end subroutine insert

      end subroutine set_v_flag

      
      subroutine set_rotation_flag(id, rotation_flag, ierr)
         use star_utils, only: set_i_rot
         integer, intent(in) :: id
         logical, intent(in) :: rotation_flag
         integer, intent(out) :: ierr         
         type (star_info), pointer :: s
         
         include 'formats'
         
         ierr = 0
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) return
         if (s% rotation_flag .eqv. rotation_flag) return

         s% rotation_flag = rotation_flag   
         s% omega(1:s% nz) = 0    
         
         if (.not. rotation_flag) return
         
         call set_i_rot(s)
         s% j_rot(1:s% nz) = 0    
         
         call zero_array(s% nu_ST)
         call zero_array(s% D_ST)
         call zero_array(s% D_DSI)
         call zero_array(s% D_SH)
         call zero_array(s% D_SSI)
         call zero_array(s% D_ES)
         call zero_array(s% D_GSF)
         
         call zero_array(s% nu_ST_old)
         call zero_array(s% D_ST_old)
         call zero_array(s% D_DSI_old)
         call zero_array(s% D_SH_old)
         call zero_array(s% D_SSI_old)
         call zero_array(s% D_ES_old)
         call zero_array(s% D_GSF_old)

         call zero_array(s% nu_ST_older)
         call zero_array(s% D_ST_older)
         call zero_array(s% D_DSI_older)
         call zero_array(s% D_SH_older)
         call zero_array(s% D_SSI_older)
         call zero_array(s% D_ES_older)
         call zero_array(s% D_GSF_older)
            
            
         contains
         
         subroutine zero_array(d)
            real(dp), pointer :: d(:)
            if (.not. associated(d)) return
            d(:) = 0
         end subroutine zero_array
            

      end subroutine set_rotation_flag     
      
      
      subroutine realloc_work_array(s, ptr, oldsz, newsz, extra, str, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: oldsz, newsz, extra
         real(dp), pointer :: ptr(:)
         character (len=*), intent(in) :: str
         integer, intent(out) :: ierr
         real(dp), pointer :: tmp(:)
         integer :: k
         tmp => ptr
         call get_work_array(s, ptr, newsz, extra, str, ierr)
         if (ierr /= 0) return
         do k=1,min(oldsz,newsz)
            ptr(k) = tmp(k)
         end do
         call return_work_array(s, tmp, str)
      end subroutine realloc_work_array
      
      
      subroutine get_work_array(s, ptr, sz, extra, str, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: sz, extra
         real(dp), pointer :: ptr(:)
         character (len=*), intent(in) :: str
         integer, intent(out) :: ierr
         call do_get_work_array(s, .true., ptr, sz, extra, str, ierr)
      end subroutine get_work_array
      
      
      ! okay to use this if sure don't need reentrant allocation
      subroutine non_crit_get_work_array(s, ptr, sz, extra, str, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: sz, extra
         real(dp), pointer :: ptr(:)
         character (len=*), intent(in) :: str
         integer, intent(out) :: ierr
         call do_get_work_array(s, .false., ptr, sz, extra, str, ierr)
      end subroutine non_crit_get_work_array
      
      
      subroutine do_get_work_array(s, crit, ptr, sz, extra, str, ierr)
         use utils_lib, only: fill_with_NaNs
         type (star_info), pointer :: s
         logical, intent(in) :: crit
         integer, intent(in) :: sz, extra
         real(dp), pointer :: ptr(:)
         character (len=*), intent(in) :: str
         integer, intent(out) :: ierr
         
         integer :: i
         logical :: okay

         ierr = 0
         
         if (work_array_debug) then
            allocate(ptr(sz + extra), stat=ierr)
            if (fill_arrays_with_NaNs) call fill_with_NaNs(ptr)
            return
         end if
         
         okay = .false.
         
         if (crit) then
!$omp critical (alloc_work_array)
            num_calls = num_calls + 1         
            do i = 1, num_work_arrays
               if (get1(i)) then
                  okay = .true.
                  exit
               end if
            end do
!$omp end critical (alloc_work_array)
         else
            num_calls = num_calls + 1         
            do i = 1, num_work_arrays
               if (get1(i)) then
                  okay = .true.
                  exit
               end if
            end do
         end if
         if (okay) return
         
         allocate(ptr(sz + extra), stat=ierr)
         num_allocs = num_allocs + 1
         if (fill_arrays_with_NaNs) call fill_with_NaNs(ptr)
         
         if (work_array_trace) write(*,*) 'allocate new work array'
         
         contains
         
         logical function get1(itry)
            integer, intent(in) :: itry
            real(dp), pointer :: p(:)
            include 'formats'
            if (.not. associated(work_pointers(itry)% p)) then
               get1 = .false.
               return
            end if
            p => work_pointers(itry)% p
            work_pointers(itry)% p => null()
            if (size(p,dim=1) < sz) then
               if (work_array_trace) &
                  write(*,4) 'enlarge work array ' // trim(str), &
                     itry, size(p,dim=1), sz + extra
               deallocate(p)
               allocate(p(sz + extra), stat=ierr)
            else
               if (work_array_trace) &
                  write(*,4) 'use work array ' // trim(str), itry, size(p,dim=1)
            end if
            ptr => p
            get1 = .true. 
            if (fill_arrays_with_NaNs) call fill_with_NaNs(ptr)
         end function get1
      
      end subroutine do_get_work_array
      
      
      subroutine return_work_array(s, ptr, str)
         type (star_info), pointer :: s
         real(dp), pointer :: ptr(:)
         character (len=*), intent(in) :: str
         call do_return_work_array(s, .true., ptr, str)
      end subroutine return_work_array


      ! okay to use this if sure don't need reentrant allocation
      subroutine non_crit_return_work_array(s, ptr, str)
         type (star_info), pointer :: s
         real(dp), pointer :: ptr(:)
         character (len=*), intent(in) :: str
         if (.not. associated(ptr)) return
         call do_return_work_array(s, .false., ptr, str)
      end subroutine non_crit_return_work_array
      
      
      subroutine do_return_work_array(s, crit, ptr, str)
         type (star_info), pointer :: s
         logical, intent(in) :: crit
         real(dp), pointer :: ptr(:)
         character (len=*), intent(in) :: str
         
         integer :: i
         logical :: okay

         if (.not. associated(ptr)) return
         
         if (work_array_debug) then
            deallocate(ptr)
            return
         end if
         
         okay = .false.
         if (crit) then
!$omp critical (alloc_work_array)
            num_returns = num_returns + 1
            do i=1,num_work_arrays
               if (return1(i)) then
                  okay = .true.
                  exit
               end if
            end do
!$omp end critical (alloc_work_array)
         else
            num_returns = num_returns + 1
            do i=1,num_work_arrays
               if (return1(i)) then
                  okay = .true.
                  exit
               end if
            end do
         end if
         if (okay) return
         
         deallocate(ptr)
         num_deallocs = num_deallocs + 1
         
         contains
         
         logical function return1(itry)
            integer, intent(in) :: itry
            include 'formats'
            if (associated(work_pointers(itry)% p)) then
               return1 = .false.
               return
            end if
            if (work_array_trace) &
               write(*,3) 'return work array ' // trim(str), itry, size(ptr,dim=1)
            work_pointers(itry)% p => ptr
            ptr => null()
            return1 = .true.
         end function return1
      
      end subroutine do_return_work_array
      
      
      subroutine get_quad_array(s, ptr, sz, extra, str, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: sz, extra
         real(qp), pointer :: ptr(:)
         character (len=*), intent(in) :: str
         integer, intent(out) :: ierr
         call do_get_quad_array(s, .true., ptr, sz, extra, str, ierr)
      end subroutine get_quad_array
      
      
      ! okay to use this if sure don't need reentrant allocation
      subroutine non_crit_get_quad_array(s, ptr, sz, extra, str, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: sz, extra
         real(qp), pointer :: ptr(:)
         character (len=*), intent(in) :: str
         integer, intent(out) :: ierr
         call do_get_quad_array(s, .false., ptr, sz, extra, str, ierr)
      end subroutine non_crit_get_quad_array
      
      
      subroutine do_get_quad_array(s, crit, ptr, sz, extra, str, ierr)
         type (star_info), pointer :: s
         logical, intent(in) :: crit
         integer, intent(in) :: sz, extra
         real(qp), pointer :: ptr(:)
         character (len=*), intent(in) :: str
         integer, intent(out) :: ierr
         
         integer :: i
         logical :: okay

         ierr = 0
         
         if (quad_array_debug) then
            allocate(ptr(sz + extra), stat=ierr)
            return
         end if
         
         okay = .false.
         if (crit) then
!$omp critical (alloc_quad_work_array)
            num_calls = num_calls + 1         
            do i = 1, num_quad_arrays
               if (get1(i)) then
                  okay = .true.
                  exit
               end if
            end do
!$omp end critical (alloc_quad_work_array)
         else
            num_calls = num_calls + 1         
            do i = 1, num_quad_arrays
               if (get1(i)) then
                  okay = .true.
                  exit
               end if
            end do
         end if
         if (okay) return
         
         allocate(ptr(sz + extra), stat=ierr)
         num_allocs = num_allocs + 1
         
         if (quad_array_trace) write(*,*) 'allocate new quad array'
         
         contains
         
         logical function get1(itry)
            integer, intent(in) :: itry
            real(qp), pointer :: p(:)
            include 'formats'
            if (.not. associated(quad_pointers(itry)% p)) then
               get1 = .false.
               return
            end if
            p => quad_pointers(itry)% p
            quad_pointers(itry)% p => null()
            if (size(p,dim=1) < sz) then
               if (quad_array_trace) &
                  write(*,4) 'enlarge quad array ' // trim(str), itry, size(p,dim=1), sz + extra
               deallocate(p)
               allocate(p(sz + extra), stat=ierr)
            else
               if (quad_array_trace) &
                  write(*,4) 'use quad array ' // trim(str), itry, size(p,dim=1)
            end if
            ptr => p
            get1 = .true. 
         end function get1
      
      end subroutine do_get_quad_array
      
      
      subroutine return_quad_array(s, ptr, str)
         type (star_info), pointer :: s
         real(qp), pointer :: ptr(:)
         character (len=*), intent(in) :: str
         call do_return_quad_array(s, .true., ptr, str)
      end subroutine return_quad_array


      ! okay to use this if sure don't need reentrant allocation
      subroutine non_crit_return_quad_array(s, ptr, str)
         type (star_info), pointer :: s
         real(qp), pointer :: ptr(:)
         character (len=*), intent(in) :: str
         if (.not. associated(ptr)) return
         call do_return_quad_array(s, .false., ptr, str)
      end subroutine non_crit_return_quad_array
      
      
      subroutine do_return_quad_array(s, crit, ptr, str)
         type (star_info), pointer :: s
         logical, intent(in) :: crit
         real(qp), pointer :: ptr(:)
         character (len=*), intent(in) :: str
         
         integer :: i
         logical :: okay

         if (.not. associated(ptr)) return
         
         if (quad_array_debug) then
            deallocate(ptr)
            return
         end if
         
         okay = .false.
         if (crit) then
!$omp critical (alloc_quad_work_array)
            num_returns = num_returns + 1
            do i=1,num_quad_arrays
               if (return1(i)) then
                  okay = .true.
                  exit
               end if
            end do
!$omp end critical (alloc_quad_work_array)
         else
            num_returns = num_returns + 1
            do i=1,num_quad_arrays
               if (return1(i)) then
                  okay = .true.
                  exit
               end if
            end do
         end if
         if (okay) return
         
         deallocate(ptr)
         num_deallocs = num_deallocs + 1
         
         contains
         
         logical function return1(itry)
            integer, intent(in) :: itry
            include 'formats'
            if (associated(quad_pointers(itry)% p)) then
               return1 = .false.
               return
            end if
            if (quad_array_trace) &
               write(*,3) 'return quad array ' // trim(str), itry, size(ptr,dim=1)
            quad_pointers(itry)% p => ptr
            ptr => null()
            return1 = .true.
         end function return1
      
      end subroutine do_return_quad_array
      
      
      subroutine realloc_integer_work_array(s, ptr, oldsz, newsz, extra, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: oldsz, newsz, extra
         integer, pointer :: ptr(:)
         integer, intent(out) :: ierr
         integer, pointer :: itmp(:)
         integer :: k
         itmp => ptr
         call get_integer_work_array(s, ptr, newsz, extra, ierr)
         if (ierr /= 0) return
         do k=1,min(oldsz,newsz)
            ptr(k) = itmp(k)
         end do
         call return_integer_work_array(s, itmp)
      end subroutine realloc_integer_work_array


      subroutine get_integer_work_array(s, ptr, sz, extra, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: sz, extra
         integer, pointer :: ptr(:)
         integer, intent(out) :: ierr
         
         integer :: i
         logical :: okay

         ierr = 0
         
         if (work_array_debug) then
            allocate(ptr(sz + extra), stat=ierr)
            return
         end if
         
         okay = .false.
!$omp critical (alloc_integer_work_array)
         num_calls = num_calls + 1
         do i=1,num_int_work_arrays
            if (get1(i)) then
               okay = .true.
               exit
            end if
         end do
!$omp end critical (alloc_integer_work_array)
         if (okay) return
         
         allocate(ptr(sz + extra), stat=ierr)
         num_allocs = num_allocs + 1
         
         if (work_array_trace) &
            write(*,*) 'allocate new integer work array'
         
         contains
         
         logical function get1(itry)
            integer, intent(in) :: itry
            integer, pointer :: p(:)
            include 'formats'
            if (.not. associated(int_work_pointers(i)% p)) then
               get1 = .false.
               return
            end if
            p => int_work_pointers(i)% p
            int_work_pointers(i)% p => null()
            if (size(p,dim=1) < sz) then
               if (work_array_trace) &
                  write(*,3) 'enlarge integer work array', size(p,dim=1), sz + extra
               deallocate(p)
               allocate(p(sz + extra), stat=ierr)
            end if
            ptr => p
            get1 = .true. 
         end function get1
      
      end subroutine get_integer_work_array

      
      subroutine return_integer_work_array(s, ptr)
         type (star_info), pointer :: s
         integer, pointer :: ptr(:)
         
         integer :: i
         logical :: okay

         if (.not. associated(ptr)) return
         
         if (work_array_debug) then
            deallocate(ptr)
            return
         end if
         
         okay = .false.
!$omp critical (alloc_integer_work_array)
         num_returns = num_returns + 1
         do i=1,num_int_work_arrays
            if (return1(i)) then
               okay = .true.
               exit
            end if
         end do
!$omp end critical (alloc_integer_work_array)
         if (okay) return
         
         deallocate(ptr)
         num_deallocs = num_deallocs + 1
         
         contains
         
         logical function return1(itry)
            integer, intent(in) :: itry
            integer, pointer :: p(:)
            if (associated(int_work_pointers(itry)% p)) then
               return1 = .false.
               return
            end if
            int_work_pointers(itry)% p => ptr
            ptr => null()
            return1 = .true.
         end function return1
      
      end subroutine return_integer_work_array
      
      
      subroutine get_logical_work_array(s, ptr, sz, extra, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: sz, extra
         logical, pointer :: ptr(:)
         integer, intent(out) :: ierr
         
         integer :: i
         logical :: okay

         ierr = 0
         
         if (work_array_debug) then
            allocate(ptr(sz + extra), stat=ierr)
            return
         end if
         
         okay = .false.
!$omp critical (alloc_logical_work_array)
         num_calls = num_calls + 1
         do i=1,num_logical_work_arrays
            if (get1(i)) then
               okay = .true.
               exit
            end if
         end do
!$omp end critical (alloc_logical_work_array)
         if (okay) return
         
         allocate(ptr(sz + extra), stat=ierr)
         num_allocs = num_allocs + 1
         
         if (work_array_trace) &
            write(*,*) 'allocate new logical work array'
         
         contains
         
         logical function get1(itry)
            integer, intent(in) :: itry
            logical, pointer :: p(:)
            include 'formats'
            if (.not. associated(logical_work_pointers(itry)% p)) then
               get1 = .false.
               return
            end if
            p => logical_work_pointers(itry)% p
            logical_work_pointers(itry)% p => null()
            if (size(p,dim=1) < sz) then
               if (work_array_trace) &
                  write(*,3) 'enlarge logical work array', size(p,dim=1), sz + extra
               deallocate(p)
               allocate(p(sz + extra), stat=ierr)
            end if
            ptr => p
            get1 = .true. 
         end function get1
      
      end subroutine get_logical_work_array

      
      subroutine return_logical_work_array(s, ptr)
         type (star_info), pointer :: s
         logical, pointer :: ptr(:)
         
         integer :: i
         logical :: okay

         if (.not. associated(ptr)) return
         
         if (work_array_debug) then
            deallocate(ptr)
            return
         end if
         
         okay = .false.
!$omp critical (alloc_logical_work_array)
         num_returns = num_returns + 1
         do i=1,num_logical_work_arrays
            if (return1(i)) then
               okay = .true.
               exit
            end if
         end do
!$omp end critical (alloc_logical_work_array)
         if (okay) return
         
         deallocate(ptr)
         num_deallocs = num_deallocs + 1
         
         contains
         
         logical function return1(itry)
            integer, intent(in) :: itry
            logical, pointer :: p(:)
            if (associated(logical_work_pointers(itry)% p)) then
               return1 = .false.
               return
            end if
            logical_work_pointers(itry)% p => ptr
            ptr => null()
            return1 = .true.
         end function return1
      
      end subroutine return_logical_work_array
      
      
      subroutine init_alloc
         integer :: i
         
         num_calls=0; num_returns=0
         num_allocs=0; num_deallocs=0
         do i=1,num_work_arrays
            work_pointers(i)%p => null()
         end do
         do i=1,num_int_work_arrays
            int_work_pointers(i)%p => null()
         end do
         do i=1,num_logical_work_arrays
            logical_work_pointers(i)%p => null()
         end do

      end subroutine init_alloc
      

      subroutine size_work_arrays
         integer :: sz, num, i
         
         include 'formats'
         sz = 0; num = 0
         do i=1,num_work_arrays
            sz = sz + get_size(i)
         end do
         do i=1,num_int_work_arrays
            sz = sz + get_size_i(i)
         end do
         do i=1,num_logical_work_arrays
            sz = sz + get_size_l(i)
         end do
         
         write(*,'(a,5x,99i8)') &
            'work_arrays: num sz calls returns diff', &
            num, sz, num_calls, num_returns, num_calls-num_returns, &
            num_allocs, num_deallocs, num_allocs-num_deallocs
         write(*,*)
         
         contains
         
         integer function get_size(i)
            integer, intent(in) :: i
            if (associated(work_pointers(i)% p)) then
               get_size = size(work_pointers(i)% p,dim=1)
               num = num + 1
            else
               get_size = 0
            end if
         end function get_size
         
         integer function get_size_i(i)
            integer, intent(in) :: i
            if (associated(int_work_pointers(i)% p)) then
               get_size_i = size(int_work_pointers(i)% p,dim=1)
               num = num + 1
            else
               get_size_i = 0
            end if
         end function get_size_i
         
         integer function get_size_l(i)
            integer, intent(in) :: i
            if (associated(logical_work_pointers(i)% p)) then
               get_size_l = size(logical_work_pointers(i)% p,dim=1)
               num = num + 1
            else
               get_size_l = 0
            end if
         end function get_size_l

      end subroutine size_work_arrays


      end module alloc


