! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module adjust_mesh_support

      use star_private_def
      use const_def
      use report, only:std_write_internals_to_file

      implicit none

      integer, parameter :: max_allowed_gvals = 50
      
      logical, parameter :: dbg = .false.

      contains
      
      
      subroutine get_gval_info( &
            s, delta_gval_max, gvals1, nz, &
            eps_h, eps_he, eps_z, num_gvals, gval_names, ierr)
         use chem_def
         use num_lib, only: safe_log10, find0
         use rates_def
         use star_utils, only: get_tau
         use alloc            
         use mesh_functions, only: set_mesh_function_data
         type (star_info), pointer :: s      
         real(dp), dimension(:), pointer :: eps_h, eps_he, eps_z, delta_gval_max
         real(dp), dimension(:), pointer :: gvals1
         integer, intent(in) :: nz, num_gvals
         integer, intent(out) :: ierr
         character (len=32), intent(out) :: gval_names(max_allowed_gvals)
         
         integer :: j, k
         logical, parameter :: dbg = .false.
         real(dp), pointer, dimension(:) :: src
         real(dp) :: eps_min_for_delta, &
               dlog_eps_dlogP_full_off, dlog_eps_dlogP_full_on, alfa_czb, alfa_os
         real(dp), dimension(:,:), pointer :: gvals
         
         gvals(1:nz,1:num_gvals) => gvals1(1:nz*num_gvals)
         
         include 'formats'
         
         ierr = 0      
         
         eps_min_for_delta = 10**s% mesh_dlog_eps_min_for_extra
         
         dlog_eps_dlogP_full_off = s% mesh_dlog_eps_dlogP_full_off
         dlog_eps_dlogP_full_on = s% mesh_dlog_eps_dlogP_full_on
         delta_gval_max(1:nz) = 1

         call set_mesh_function_data(s, num_gvals, gval_names, gvals1, ierr)
         if (ierr /= 0) return

         call set_alfas

         call get_work_array(s, src, nz, nz_alloc_extra, 'adjust_mesh_support', ierr)
         if (ierr /= 0) then
            write(*,*) 'allocate failed in ', nz
            return
         end if

         call set_delta_gval_max(src)
         
         call smooth_gvals(nz,src,num_gvals,gvals)
         
         call return_work_array(s, src, 'adjust_mesh_support')
         
         
         contains
                  
         
         subroutine set_delta_gval_max(src)
            real(dp), pointer :: src(:)
            real(dp) :: P_exp, beta
         
            !k = 2100
            !write(*,2) '1 delta_gval_max(k)', k, delta_gval_max(k)
            
            if (s% mesh_Pgas_div_P_exponent /= 0) then
               P_exp = s% mesh_Pgas_div_P_exponent
               do k=1,nz
                  beta = s% Pgas(k)/s% P(k)
                  delta_gval_max(k) = delta_gval_max(k)*(beta**P_exp)
               end do
            end if
         
            if (s% xtra_coef_above_xtrans /= 1 .or. &
                s% xtra_coef_below_xtrans /= 1) &
               call do_xtrans_coef
         
            if (s% xtra_coef_a_l_nb_czb /= 1 .or. &
               s% xtra_coef_a_l_hb_czb /= 1 .or. &
               s% xtra_coef_a_l_heb_czb /= 1 .or. &
               s% xtra_coef_a_l_zb_czb /= 1 .or. &
               s% xtra_coef_b_l_nb_czb /= 1 .or. &
               s% xtra_coef_b_l_hb_czb /= 1 .or. &
               s% xtra_coef_b_l_heb_czb /= 1 .or. &
               s% xtra_coef_b_l_zb_czb /= 1 .or. &
               s% xtra_coef_a_u_nb_czb /= 1 .or. &
               s% xtra_coef_a_u_hb_czb /= 1 .or. &
               s% xtra_coef_a_u_heb_czb /= 1 .or. &
               s% xtra_coef_a_u_zb_czb /= 1 .or. &
               s% xtra_coef_b_u_nb_czb /= 1 .or. &
               s% xtra_coef_b_u_hb_czb /= 1 .or. &
               s% xtra_coef_b_u_heb_czb /= 1 .or. &
               s% xtra_coef_b_u_zb_czb /= 1) &
               call do_conv_bdy_coef
            
            !write(*,2) '2 delta_gval_max(k)', k, delta_gval_max(k)
         
            if (s% xtra_coef_os_above_nonburn /= 1 .or. &
                s% xtra_coef_os_below_nonburn /= 1 .or. &
                s% xtra_coef_os_above_burn_h /= 1 .or. &
                s% xtra_coef_os_below_burn_h /= 1 .or. &
                s% xtra_coef_os_above_burn_he /= 1 .or. &
                s% xtra_coef_os_below_burn_he /= 1 .or. &
                s% xtra_coef_os_above_burn_z /= 1 .or. &
                s% xtra_coef_os_below_burn_z /= 1) &
               call do_overshoot_coef
            
            !write(*,2) '3 delta_gval_max(k)', k, delta_gval_max(k)
            
            do j=1,num_mesh_logX
               if (s% mesh_dlogX_dlogP_extra(j) > 0 .and. s% mesh_dlogX_dlogP_extra(j) < 1) &
                  call do_mesh_dlogX_coef(s,j)
            end do
         
            !write(*,2) '4 delta_gval_max(k)', k, delta_gval_max(k)
         
            call do1_dlog_eps_dlogP_coef(s% mesh_dlog_pp_dlogP_extra, ipp)
            call do1_dlog_eps_dlogP_coef(s% mesh_dlog_cno_dlogP_extra, icno)
            call do1_dlog_eps_dlogP_coef(s% mesh_dlog_3alf_dlogP_extra, i3alf)

            call do1_dlog_eps_dlogP_coef(s% mesh_dlog_burn_c_dlogP_extra, i_burn_c)
            call do1_dlog_eps_dlogP_coef(s% mesh_dlog_burn_n_dlogP_extra, i_burn_n)
            call do1_dlog_eps_dlogP_coef(s% mesh_dlog_burn_o_dlogP_extra, i_burn_o)
            call do1_dlog_eps_dlogP_coef(s% mesh_dlog_burn_ne_dlogP_extra, i_burn_ne)
            call do1_dlog_eps_dlogP_coef(s% mesh_dlog_burn_na_dlogP_extra, i_burn_na)
            call do1_dlog_eps_dlogP_coef(s% mesh_dlog_burn_mg_dlogP_extra, i_burn_mg)
            call do1_dlog_eps_dlogP_coef(s% mesh_dlog_burn_si_dlogP_extra, i_burn_si)
            call do1_dlog_eps_dlogP_coef(s% mesh_dlog_burn_s_dlogP_extra, i_burn_s)
            call do1_dlog_eps_dlogP_coef(s% mesh_dlog_burn_ar_dlogP_extra, i_burn_ar)
            call do1_dlog_eps_dlogP_coef(s% mesh_dlog_burn_ca_dlogP_extra, i_burn_ca)
            call do1_dlog_eps_dlogP_coef(s% mesh_dlog_burn_ti_dlogP_extra, i_burn_ti)
            call do1_dlog_eps_dlogP_coef(s% mesh_dlog_burn_cr_dlogP_extra, i_burn_cr)
            call do1_dlog_eps_dlogP_coef(s% mesh_dlog_burn_fe_dlogP_extra, i_burn_fe)

            call do1_dlog_eps_dlogP_coef(s% mesh_dlog_cc_dlogP_extra, icc)
            call do1_dlog_eps_dlogP_coef(s% mesh_dlog_co_dlogP_extra, ico)
            call do1_dlog_eps_dlogP_coef(s% mesh_dlog_oo_dlogP_extra, ioo)
   
            call do1_dlog_eps_dlogP_coef(s% mesh_dlog_pnhe4_dlogP_extra, ipnhe4)
            call do1_dlog_eps_dlogP_coef(s% mesh_dlog_photo_dlogP_extra, iphoto)
            call do1_dlog_eps_dlogP_coef(s% mesh_dlog_other_dlogP_extra, iother)
            
            !write(*,2) '5 delta_gval_max(k)', k, delta_gval_max(k)

            forall (k=1:nz) src(k) = delta_gval_max(k)
         
            do j=1,3
               delta_gval_max(1) = (2*src(1) + src(2))/3
               do k=2,nz-1
                  delta_gval_max(k) = (src(k-1) + src(k) + src(k+1))/3
               end do
               delta_gval_max(nz) = (2*src(nz) + src(nz-1))/3
               if (j == 3) exit
               src(1) = (2*delta_gval_max(1) + delta_gval_max(2))/3
               do k=2,nz-1
                  src(k) = &
                     (delta_gval_max(k-1) + delta_gval_max(k) + delta_gval_max(k+1))/3
               end do
               src(nz) = (2*delta_gval_max(nz) + delta_gval_max(nz-1))/3
            end do
            
            !k = 1000
            !write(*,2) '6 delta_gval_max(k)', k, delta_gval_max(k)
         end subroutine set_delta_gval_max


         subroutine set_alfas
            real(dp) :: he_cntr, full_off, full_on
            include 'formats'
            he_cntr = s% xa(s% net_iso(ihe4),nz)
            full_off = s% xtra_coef_czb_full_off
            full_on = s% xtra_coef_czb_full_on
            if (he_cntr >= full_off) then
               alfa_czb = 0
            else if (he_cntr <= full_on) then
               alfa_czb = 1
            else
               alfa_czb = (he_cntr - full_off)/(full_on - full_off)
            end if 
            full_off = s% xtra_coef_os_full_off
            full_on = s% xtra_coef_os_full_on
            if (he_cntr >= full_off) then
               alfa_os = 0
            else if (he_cntr <= full_on) then
               alfa_os = 1
            else
               alfa_os = (he_cntr - full_off)/(full_on - full_off)
            end if
         end subroutine set_alfas
         
                     
         subroutine do_mesh_dlogX_coef(s, which)
            use chem_lib, only: chem_get_iso_id
            type (star_info), pointer :: s      
            integer, intent(in) :: which
            real(dp) :: &
               logX_min_for_extra, dlogX_extra, dlogX_full_on, dlogX_full_off
            integer :: k, cid, j
            real(dp) :: X_min_for_extra, dlogX, xtra_coef, extra, coef
            logical :: dbg
            include 'formats'
            if (len_trim(s% mesh_logX_species(which)) == 0) return
            cid = chem_get_iso_id(s% mesh_logX_species(which))
            if (cid <= 0) return
            j = s% net_iso(cid)
            if (j == 0) return
            logX_min_for_extra = s% mesh_logX_min_for_extra(which)
            dlogX_extra = s% mesh_dlogX_dlogP_extra(which)
            dlogX_full_on = s% mesh_dlogX_dlogP_full_on(which)
            dlogX_full_off = s% mesh_dlogX_dlogP_full_off(which)
            X_min_for_extra = 10**max(-50d0, logX_min_for_extra)
            do k=2, nz
               dbg = .false. !(k == 2100)
               if (dbg) then
                  write(*,2) 'do_mesh_dlogX_coef X_min_for_extra', j, X_min_for_extra
                  write(*,2) 's% xa(j,k)', k, s% xa(j,k)
                  write(*,2) 's% xa(j,k-1)', k-1, s% xa(j,k-1)
                  write(*,2) 'dlogX_full_off', k, dlogX_full_off
                  write(*,2) 'abs(log10(s% xa(j,k)/s% xa(j,k-1)))', k, abs(log10(s% xa(j,k)/s% xa(j,k-1)))
                  write(*,2) 'dlogX_full_on', k, dlogX_full_on
               end if
               if (s% xa(j,k) < X_min_for_extra .or. s% xa(j,k-1) < X_min_for_extra) cycle
               dlogX = abs(log10(s% xa(j,k)/s% xa(j,k-1)))
               if (dlogX <= dlogX_full_off) cycle 
               if (dlogX >= dlogX_full_on) then
                  extra = dlogX_extra
               else
                  extra = 1 - (1 - dlogX_extra) * &
                     (dlogX - dlogX_full_off) / (dlogX_full_on - dlogX_full_off)
               end if
               coef = extra
               if (dbg) write(*,2) 'coef', 2, coef, delta_gval_max(k)
               if (coef < delta_gval_max(k)) delta_gval_max(k) = coef
               if (coef < delta_gval_max(k-1)) delta_gval_max(k-1) = coef
               if (k < nz) then
                  if (coef < delta_gval_max(k+1)) delta_gval_max(k+1) = coef
               end if
            end do
         end subroutine do_mesh_dlogX_coef
         

         subroutine do1_dlog_eps_dlogP_coef(dlog_eps_dlogP_extra, cat)
            use chem_def, only: icno
            real(dp), intent(in) :: dlog_eps_dlogP_extra
            integer, intent(in) :: cat
            integer :: k
            real(dp) :: eps, epsm1, dlog_eps, dlogP, dlog_eps_dlogP, &
               extra, new_max, maxv
            include 'formats'
            
            if (dlog_eps_dlogP_extra <= 0 .or. dlog_eps_dlogP_extra >=1) return
            maxv = maxval(s% eps_nuc_categories(i_rate,cat,1:nz))
            if (maxv < eps_min_for_delta) return
            
            do k=2, nz
            
               eps = s% eps_nuc_categories(i_rate,cat,k)
               if (eps < eps_min_for_delta) cycle
               
               epsm1 = s% eps_nuc_categories(i_rate,cat,k-1)
               if (epsm1 < eps_min_for_delta) cycle
               
               maxv = maxval(s% eps_nuc_categories(i_rate,:,k))
               if (maxv /= eps) cycle
               
               dlogP = (s% lnP(k) - s% lnP(k-1))/ln10
               if (dlogP < -50d0) cycle

               dlog_eps = abs(log10(eps/epsm1))
               
               dlog_eps_dlogP = dlog_eps/dlogP
               if (dlog_eps_dlogP <= dlog_eps_dlogP_full_off) cycle 
               
               if (dlog_eps_dlogP >= dlog_eps_dlogP_full_on) then
                  extra = dlog_eps_dlogP_extra
               else
                  extra = 1 - (1 - dlog_eps_dlogP_extra) * &
                     (dlog_eps_dlogP - dlog_eps_dlogP_full_off) / &
                        (dlog_eps_dlogP_full_on - dlog_eps_dlogP_full_off)
               end if
               new_max = extra
               if (new_max < delta_gval_max(k)) then
                  delta_gval_max(k) = new_max
               end if
               if (new_max < delta_gval_max(k-1)) then
                  delta_gval_max(k-1) = new_max
               end if
               if (k < nz) then
                  if (new_max < delta_gval_max(k+1)) delta_gval_max(k+1) = new_max
               end if
               
            end do
         end subroutine do1_dlog_eps_dlogP_coef
         
         
         subroutine do_xtrans_coef
            integer :: k, nz, prev_jmax, jmax
            real(dp) :: xtra_coef, xtra_dist, Hp, r_extra
            
            logical, parameter :: dbg = .false.
            
            include 'formats'
            
            if (dbg) write (*,*) 'do_xtrans_coef'
            
            nz = s% nz
                        
            xtra_coef = s% xtra_coef_below_xtrans
            xtra_dist = s% xtra_dist_below_xtrans
            if (xtra_coef /= 1) then ! first go from surface to center doing below transitions
               prev_jmax = maxloc(s% xa(:,1),dim=1)
               k = 2
               do while (k <= nz)
                  jmax = maxloc(s% xa(:,k),dim=1)
                  if (jmax /= prev_jmax) then ! at a change in max abundance species
                     prev_jmax = jmax
                     Hp = s% P(k)/(s% rho(k)*s% grav(k))
                     r_extra = max(0d0, s% r(k) - xtra_dist*Hp)
                     do ! toward center from a transition
                        if (s% r(k) < r_extra) exit
                        if (xtra_coef < delta_gval_max(k)) delta_gval_max(k) = xtra_coef
                        if (k == nz) exit
                        k = k+1
                     end do
                  end if
                  k = k+1
               end do
            end if
            
            xtra_coef = s% xtra_coef_above_xtrans
            xtra_dist = s% xtra_dist_above_xtrans
            if (xtra_coef /= 1) then ! now go from center to surface doing above transitions
               prev_jmax = maxloc(s% xa(:,nz),dim=1)
               k = nz-1
               do while (k >= 1)
                  jmax = maxloc(s% xa(:,k),dim=1)
                  if (jmax /= prev_jmax) then ! at a change in max abundance species
                     prev_jmax = jmax
                     Hp = s% P(k)/(s% rho(k)*s% grav(k))
                     r_extra = max(0d0, s% r(k) + xtra_dist*Hp)
                     do ! toward surface from a transition
                        if (s% r(k) > r_extra) exit
                        if (xtra_coef < delta_gval_max(k)) delta_gval_max(k) = xtra_coef
                        if (k == 1) exit
                        k = k-1
                     end do
                  end if
                  k = k-1
               end do
            end if
            
         end subroutine do_xtrans_coef
         

         subroutine do_conv_bdy_coef
            use mlt_def, only: convective_mixing
            integer :: k, kk, nz, max_eps_loc, k_bdy, k_nxt
            real(dp) :: xtra_coef_b, xtra_dist_b, xtra_coef_a, xtra_dist_a
            real(dp) :: coef, dist, Hp, r_extra, max_eps, eps
            logical :: in_convective_region
            
            logical, parameter :: dbg = .false.
            
            include 'formats'
            
            if (dbg) write (*,1) 'do_conv_bdy_coef', alfa_czb
            
            if (alfa_czb == 0) return
            
            nz = s% nz
                        
            ! first go from surface to center doing lower boundaries of convective zones
            in_convective_region = (s% mixing_type(1) == convective_mixing)
            k = 2
            max_eps = -1d99
            max_eps_loc = -1
            do while (k <= nz)
               eps = eps_h(k) + eps_he(k) + eps_z(k)
               if (in_convective_region) then
                  if (s% mixing_type(k) == convective_mixing) then
                     if (eps > max_eps) then
                        max_eps = eps
                        max_eps_loc = k
                     end if
                  else
                     in_convective_region = .false.
                     Hp = s% P(k)/(s% rho(k)*s% grav(k))
                     if (max_eps < 1d0) then
                        xtra_coef_a = s% xtra_coef_a_l_nb_czb
                        xtra_dist_a = s% xtra_dist_a_l_nb_czb
                        xtra_coef_b = s% xtra_coef_b_l_nb_czb
                        xtra_dist_b = s% xtra_dist_b_l_nb_czb
                     else if (eps_h(max_eps_loc) > 0.5d0*max_eps) then
                        xtra_coef_a = s% xtra_coef_a_l_hb_czb
                        xtra_dist_a = s% xtra_dist_a_l_hb_czb
                        xtra_coef_b = s% xtra_coef_b_l_hb_czb
                        xtra_dist_b = s% xtra_dist_b_l_hb_czb
                     else if (eps_he(max_eps_loc) > 0.5d0*max_eps) then
                        xtra_coef_a = s% xtra_coef_a_l_heb_czb
                        xtra_dist_a = s% xtra_dist_a_l_heb_czb
                        xtra_coef_b = s% xtra_coef_b_l_heb_czb
                        xtra_dist_b = s% xtra_dist_b_l_heb_czb
                     else
                        xtra_coef_a = s% xtra_coef_a_l_zb_czb
                        xtra_dist_a = s% xtra_dist_a_l_zb_czb
                        xtra_coef_b = s% xtra_coef_b_l_zb_czb
                        xtra_dist_b = s% xtra_dist_b_l_zb_czb
                     end if
                     coef = xtra_coef_b*alfa_czb + (1-alfa_czb)
                     k_bdy = k                
                     if (coef > 0 .and. coef /= 1) then
                        dist = xtra_dist_b*alfa_czb
                        r_extra = max(0d0, s% r(k) - dist*Hp)
                        do ! do below a lower czb
                           if (s% r(k) < r_extra) exit
                           if (coef < delta_gval_max(k)) delta_gval_max(k) = coef
                           if (k == nz) exit
                           k = k+1
                        end do
                     end if
                     k_nxt = k
                     k = k_bdy          
                     coef = xtra_coef_a*alfa_czb + (1-alfa_czb)
                     if (coef > 0 .and. coef /= 1) then
                        dist = xtra_dist_a*alfa_czb
                        r_extra = min(s% r(1), s% r(k) + dist*Hp)
                        do ! do above a lower czb
                           if (s% r(k) > r_extra) exit
                           if (coef < delta_gval_max(k)) delta_gval_max(k) = coef
                           if (k == 1) exit
                           k = k-1
                        end do
                     end if
                     if (dbg) write(*,*) 'extra applied in range', k, k_bdy, k_nxt
                     k = k_nxt
                  end if

               else if (s% mixing_type(k) == convective_mixing) then
                  in_convective_region = .true.
                  max_eps = eps
                  max_eps_loc = k
               end if
               k = k+1
            end do
            
            ! now go from center to surface doing upper boundaries of convective zones
            in_convective_region = (s% mixing_type(nz) == convective_mixing)
            k = nz-1
            max_eps = -1d99
            max_eps_loc = -1
            do while (k >= 1)
               eps = eps_h(k) + eps_he(k) + eps_z(k)
               if (in_convective_region) then
                  if (s% mixing_type(k) == convective_mixing) then
                     if (eps > max_eps) then
                        max_eps = eps
                        max_eps_loc = k
                     end if
                  else
                     in_convective_region = .false.
                     Hp = s% P(k)/(s% rho(k)*s% grav(k))
                     if (max_eps < 1d0) then
                        xtra_coef_a = s% xtra_coef_a_u_nb_czb
                        xtra_dist_a = s% xtra_dist_a_u_nb_czb
                        xtra_coef_b = s% xtra_coef_b_u_nb_czb
                        xtra_dist_b = s% xtra_dist_b_u_nb_czb
                     else if (eps_h(max_eps_loc) > 0.5d0*max_eps) then
                        xtra_coef_a = s% xtra_coef_a_u_hb_czb
                        xtra_dist_a = s% xtra_dist_a_u_hb_czb
                        xtra_coef_b = s% xtra_coef_b_u_hb_czb
                        xtra_dist_b = s% xtra_dist_b_u_hb_czb
                     else if (eps_he(max_eps_loc) > 0.5d0*max_eps) then
                        xtra_coef_a = s% xtra_coef_a_u_heb_czb
                        xtra_dist_a = s% xtra_dist_a_u_heb_czb
                        xtra_coef_b = s% xtra_coef_b_u_heb_czb
                        xtra_dist_b = s% xtra_dist_b_u_heb_czb
                     else
                        xtra_coef_a = s% xtra_coef_a_u_zb_czb
                        xtra_dist_a = s% xtra_dist_a_u_zb_czb
                        xtra_coef_b = s% xtra_coef_b_u_zb_czb
                        xtra_dist_b = s% xtra_dist_b_u_zb_czb
                     end if
                     coef = xtra_coef_a*alfa_czb + (1-alfa_czb)
                     k_bdy = k
                     if (coef > 0 .and. coef /= 1) then
                        dist = xtra_dist_a*alfa_czb
                        r_extra = min(s% r(1), s% r(k) + dist*Hp)
                        do ! above upper czb
                           if (s% r(k) > r_extra) exit
                           if (coef < delta_gval_max(k)) delta_gval_max(k) = coef
                           if (k == 1) exit
                           k = k-1
                        end do
                     end if
                     k_nxt = k
                     k = k_bdy
                     coef = xtra_coef_b*alfa_czb + (1-alfa_czb)
                     if (coef > 0 .and. coef /= 1) then
                        dist = xtra_dist_b*alfa_czb
                        r_extra = max(0d0, s% r(k) - dist*Hp)
                        do ! below upper czb
                           if (s% r(k) < r_extra) exit
                           if (coef < delta_gval_max(k)) delta_gval_max(k) = coef
                           if (k == nz) exit
                           k = k+1
                        end do
                        if (dbg) write(*,'(a30,3i6,e20.10)') 'extra applied at upper czb', k_nxt, k_bdy, k
                        k = k_nxt
                     end if
                  end if
               else if (s% mixing_type(k) == convective_mixing) then
                  in_convective_region = .true.
                  max_eps = eps
                  max_eps_loc = k
               end if
               k = k-1
            end do
            
         end subroutine do_conv_bdy_coef

         
         subroutine do_overshoot_coef
            use mlt_def, only: overshoot_mixing, convective_mixing
            integer :: k, kk, nz, max_eps_loc
            real(dp) :: xtra_coef, xtra_dist, coef, Hp, r_extra, max_eps, eps
            logical :: in_convective_region
            logical, parameter :: dbg = .false.
            
            include 'formats'
            
            if (alfa_os == 0) return

            nz = s% nz
            
            ! first go from surface to center doing below convective boundaries
            in_convective_region = (s% mixing_type(1) == convective_mixing)
            k = 2
            max_eps = -1d99
            max_eps_loc = -1
            do while (k <= nz)
               eps = eps_h(k) + eps_he(k) + eps_z(k)
               if (in_convective_region) then
                  if (s% mixing_type(k) == convective_mixing) then
                     if (eps > max_eps) then
                        max_eps = eps
                        max_eps_loc = k
                     end if
                  else
                     in_convective_region = .false.
                     if (max_eps < 1d0) then
                        xtra_coef = s% xtra_coef_os_below_nonburn
                        xtra_dist = s% xtra_dist_os_below_nonburn
                     else if (eps_h(max_eps_loc) > 0.5d0*max_eps) then
                        xtra_coef = s% xtra_coef_os_below_burn_h
                        xtra_dist = s% xtra_dist_os_below_burn_h
                     else if (eps_he(max_eps_loc) > 0.5d0*max_eps) then
                        xtra_coef = s% xtra_coef_os_below_burn_he
                        xtra_dist = s% xtra_dist_os_below_burn_he
                     else
                        xtra_coef = s% xtra_coef_os_below_burn_z
                        xtra_dist = s% xtra_dist_os_below_burn_z
                     end if
                     xtra_coef = xtra_coef*alfa_os + (1-alfa_os)
                     if (xtra_coef > 0 .and. xtra_coef /= 1) then
                        coef = xtra_coef
                        do
                           if (s% mixing_type(k) /= overshoot_mixing) exit
                           if (coef < delta_gval_max(k)) delta_gval_max(k) = coef
                           if (k == nz) exit
                           k = k+1
                        end do
                        if (xtra_dist > 0) then                        
                           Hp = s% P(k)/(s% rho(k)*s% grav(k))
                           r_extra = max(0d0, s% r(k) - xtra_dist*Hp)
                           if (dbg) write(*,2) 'extra below overshoot region', &
                              k, s% r(k)/Rsun, Hp/Rsun, r_extra/Rsun                        
                           do
                              if (s% r(k) < r_extra) exit
                              if (coef < delta_gval_max(k)) delta_gval_max(k) = coef
                              if (k == nz) exit
                              k = k+1
                           end do
                        end if
                     end if
                     if (dbg) write(*,2) 'done with extra below overshoot region', k
                     if (dbg) write(*,*)
                  end if
               else if (s% mixing_type(k) == convective_mixing) then
                  in_convective_region = .true.
                  max_eps = eps
                  max_eps_loc = k
               end if
               k = k+1
            end do
            
            ! now go from center to surface doing above convective boundaries
            in_convective_region = (s% mixing_type(nz) == convective_mixing)
            k = nz-1
            max_eps = -1d99
            max_eps_loc = -1
            do while (k >= 1)
               eps = eps_h(k) + eps_he(k) + eps_z(k)
               if (in_convective_region) then
                  if (s% mixing_type(k) == convective_mixing) then
                     if (eps > max_eps) then
                        max_eps = eps
                        max_eps_loc = k
                     end if
                  else
                     in_convective_region = .false.
                     if (max_eps < 1d0) then
                        xtra_coef = s% xtra_coef_os_above_nonburn
                        xtra_dist = s% xtra_dist_os_above_nonburn
                     else if (eps_h(max_eps_loc) > 0.5d0*max_eps) then
                        xtra_coef = s% xtra_coef_os_above_burn_h
                        xtra_dist = s% xtra_dist_os_above_burn_h
                     else if (eps_he(max_eps_loc) > 0.5d0*max_eps) then
                        xtra_coef = s% xtra_coef_os_above_burn_he
                        xtra_dist = s% xtra_dist_os_above_burn_he
                     else
                        xtra_coef = s% xtra_coef_os_above_burn_z
                        xtra_dist = s% xtra_dist_os_above_burn_z
                     end if
                     xtra_coef = xtra_coef*alfa_os + (1-alfa_os)
                     if (xtra_coef > 0 .and. xtra_coef /= 1) then
                        coef = xtra_coef
                        do
                           if (s% mixing_type(k) /= overshoot_mixing) exit
                           if (coef < delta_gval_max(k)) delta_gval_max(k) = coef
                           if (k == 1) exit
                           k = k-1
                        end do
                        if (xtra_dist > 0) then                        
                           Hp = s% P(k)/(s% rho(k)*s% grav(k))
                           r_extra = min(s% r(1), s% r(k) + xtra_dist*Hp)
                           if (dbg) write(*,2) 'extra above overshoot region', &
                              k, s% r(k)/Rsun, Hp/Rsun, r_extra/Rsun
                           do
                              if (s% r(k) > r_extra) exit
                              if (coef < delta_gval_max(k)) delta_gval_max(k) = coef
                              if (k == 1) exit
                              k = k-1
                           end do
                        end if
                     end if
                     if (dbg) write(*,2) 'done with extra above overshoot region', k
                     if (dbg) write(*,*)
                  end if
               else if (s% mixing_type(k) == convective_mixing) then
                  in_convective_region = .true.
                  max_eps = eps
                  max_eps_loc = k
               end if
               k = k-1
            end do
            
         end subroutine do_overshoot_coef
         
         
      end subroutine get_gval_info
      
      
      subroutine single_peak(nz,src)
         integer, intent(in) :: nz
         real(dp), pointer :: src(:)
         integer :: k, kmax
         real(dp) :: prev, val
         kmax = maxloc(src(1:nz), dim=1)
         val = src(kmax)
         do k=kmax+1,nz 
            prev = val
            val = src(k)
            if (val > prev) val = prev
            src(k) = val
         end do
         val = src(kmax)
         do k=kmax-1,1,-1 
            prev = val
            val = src(k)
            if (val > prev) val = prev
            src(k) = val
         end do
      end subroutine single_peak
      
      
      subroutine increasing_inward(nz,src)
         integer, intent(in) :: nz
         real(dp), pointer :: src(:)
         integer :: k
         real(dp) :: prev, val
         val = src(1)
         do k=2,nz 
            prev = val
            val = src(k)
            if (val < prev) val = prev
            src(k) = val
         end do
      end subroutine increasing_inward
      
      
      subroutine decreasing_outward(nz,src)
         integer, intent(in) :: nz
         real(dp), pointer :: src(:)
         integer :: k
         real(dp) :: prev, val
         val = src(nz)
         do k=nz-1,1,-1 
            prev = val
            val = src(k)
            if (val > prev) val = prev
            src(k) = val
         end do
      end subroutine decreasing_outward
      
      
      subroutine increasing_outward(nz,src)
         integer, intent(in) :: nz
         real(dp), pointer :: src(:)
         integer :: k
         real(dp) :: prev, val
         val = src(nz)
         do k=nz-1,1,-1 
            prev = val
            val = src(k)
            if (val < prev) val = prev
            src(k) = val
         end do
      end subroutine increasing_outward
      
      
      subroutine smooth_gvals(nz,src,num_gvals,gvals)
         integer, intent(in) :: nz, num_gvals
         real(dp), pointer :: src(:), gvals(:,:)
         integer :: k, i
         
         do i=1,num_gvals
         
            forall (k=1:nz) src(k) = gvals(k,i)
         
            gvals(1,i) = (2*src(1) + src(2))/3
            forall (k=2:nz-1) gvals(k,i) = (src(k-1) + src(k) + src(k+1))/3
            gvals(nz,i) = (2*src(nz) + src(nz-1))/3

            src(1) = (2*gvals(1,i) + gvals(2,i))/3
            forall (k=2:nz-1) src(k) = (gvals(k-1,i) + gvals(k,i) + gvals(k+1,i))/3
            src(nz) = (2*gvals(nz,i) + gvals(nz-1,i))/3
      
            gvals(1,i) = (2*src(1) + src(2))/3
            forall (k=2:nz-1) gvals(k,i) = (src(k-1) + src(k) + src(k+1))/3
            gvals(nz,i) = (2*src(nz) + src(nz-1))/3
            
         end do

      end subroutine smooth_gvals
      
      
      subroutine set_boundary_values(s, src, dest, j)
         type (star_info), pointer :: s
         real(dp), pointer :: src(:), dest(:, :)
         integer, intent(in) :: j
         integer :: k, nz
         nz = s% nz
         dest(1,j) = src(1)
         do k=2,nz
            dest(k,j) = (src(k-1)+src(k))/2
         end do
      end subroutine set_boundary_values


      subroutine check_validity(s, ierr)
         type (star_info), pointer :: s      
         integer, intent(out) :: ierr
         
         integer :: k, nz
         
         include 'formats'
         
         ierr = 0      
         nz = s% nz
         
         do k=1, nz-1
            if (s% xh(s% i_lnR, k) <= s% xh(s% i_lnR, k+1)) then
               ierr = -1
               if (s% report_ierr) then
                  write(*, *) 'at start of remesh: negative cell volume for cell', k
                  write(*, *) 
                  write(*,2) 's% xh(s% i_lnR, k)', k, s% xh(s% i_lnR, k)
                  write(*,2) 's% xh(s% i_lnR, k+1)', k+1, s% xh(s% i_lnR, k+1)
                  write(*, *) 
                  write(*, *) 's% model_number', s% model_number
                  write(*, *) 's% nz', s% nz
                  write(*, *) 's% num_retries', s% num_retries
                  write(*, *) 's% num_backups', s% num_backups
                  write(*, *) 
               end if
               return
            end if
            if (s% dq(k) <= 0) then
               ierr = -1
               if (s% report_ierr) then
                  write(*, *) 'at start of remesh: non-positive cell mass for cell', k
                  write(*, *) 's% model_number', s% model_number
                  write(*, *) 's% nz', s% nz
                  write(*, *) 's% num_retries', s% num_retries
                  write(*, *) 's% num_backups', s% num_backups
                  write(*, *) 
               end if
               return
            end if
         end do
         
      end subroutine check_validity

      

      end module adjust_mesh_support


