! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************


      module adjust_mass

      use star_private_def
      use const_def
      use chem_def, only: ih1, ihe4, ic12, in14, io16

      implicit none
      
      
      logical, parameter :: dbg_adjm = .false.
      
      
      contains
      
      
      subroutine do_adjust_mass(s, species, ierr)
         use adjust_xyz, only: get_xa_for_accretion
         use utils_lib, only:has_bad_num, is_bad_num
         use star_utils, only: report_xa_bad_nums
         use num_lib, only: safe_log10
         use chem_def
         
         type (star_info), pointer :: s
         integer, intent(in) :: species
         integer, intent(out) :: ierr
         
         real(dp) :: &
            dt, delta_m, old_mstar, new_mstar, old_J, new_J, factor, &
            frac, env_mass, mmax, alfa, new_xmstar, old_xmstar, &
				q_for_just_added, q_for_recently_added, xq_for_mdot_eps, sum_dq
			
			real(dp), target :: xaccrete_array(species)
         real(dp), dimension(:), pointer :: &
            xaccrete, rxm_old, rxm_new, old_cell_mass, new_cell_mass, &
            oldloc, newloc, oldval, newval, work, xm_old, xm_new
         real(dp), dimension(:), pointer :: xa_old1
         real(dp), dimension(:,:), pointer :: xa_old
         
         integer :: j, k, k_const_mass, nz
         
         logical, parameter :: dbg = .false.
         
         include 'formats'
         
         ierr = 0
         
         xaccrete => xaccrete_array
         s% angular_momentum_removed = 0
         
         if (dbg) write(*,*) 'do_adjust_mass'

         ! NOTE: don't assume that vars are all set at this point.
         ! use values from s% xh(:,:) and s% xa(:,:) only.
         ! e.g., don't use s% lnT(:) -- use s% xh(s% i_lnT,:) instead.
         
         ! except: omega, i_rot, j_rot have been set, and total_angular_momentum.
         
         s% k_const_mass = 1
         
         
         s% k_below_mdot_eps_grav = 1

         q_for_recently_added = 1d0 ! change later if are adding mass
         s% k_below_recently_added = 1
         q_for_just_added = 1d0
         s% k_below_just_added = 1
         
         nz = s% nz         
         dt = s% dt         
         delta_m = dt*s% mstar_dot

         if (s% super_eddington_wind_mdot /= 0 .and. &
               s% super_eddington_wind_mdot > -s% mstar_dot) then
            s% mstar_dot = -s% super_eddington_wind_mdot
            delta_m = dt*s% mstar_dot
            !write(*,1) 'adjust_mass switch to super_eddington', s% super_eddington_wind_mdot
         else if (delta_m == 0 &
            .or. (delta_m < 0 .and. s% star_mass <= s% min_star_mass_for_loss) &
            .or. (delta_m > 0 .and. s% max_star_mass_for_gain > 0 &
                  .and. s% star_mass >= s% max_star_mass_for_gain)) then
            if (dbg) write(*,*) 'do_adjust_mass return 2'
            return
         end if

         !write(*,1) 'lg wind_mdot in adjust_mass', log10(-s% mstar_dot/(Msun/secyer))
         
         old_mstar = s% mstar
         old_xmstar = s% xmstar
         
         new_mstar = old_mstar + delta_m
         new_xmstar = old_xmstar + delta_m
         
         if (delta_m > 0 .and. s% max_star_mass_for_gain > 0 &
               .and. new_mstar > Msun*s% max_star_mass_for_gain) then
            new_mstar = Msun*s% max_star_mass_for_gain
            delta_m = new_mstar - old_mstar
         else if (delta_m < 0 .and. new_mstar < Msun*s% min_star_mass_for_loss) then
            new_mstar = Msun*s% min_star_mass_for_loss
            delta_m = new_mstar - old_mstar
         end if
         
         frac = old_xmstar/new_xmstar
         new_xmstar = old_xmstar/frac
         if (new_xmstar <= 0) then
            ierr = -1
            return
         end if
         s% xmstar = new_xmstar
         s% mstar = s% xmstar + s% M_center
         
         if (dbg_adjm) then
            env_mass = old_mstar - s% he_core_mass*Msun
            write(*,'(a40,f26.16)') 'env_mass/old_mstar', env_mass/old_mstar
            write(*,*)
            write(*,1) 'delta_m/old_mstar', delta_m/old_mstar
            write(*,1) 's% he_core_mass*Msun', s% he_core_mass*Msun
            write(*,1) 'env_mass', env_mass
            write(*,1) 'delta_m/env_mass', delta_m/env_mass
            write(*,1) 'log10(abs(delta_m/env_mass))', safe_log10(abs(delta_m/env_mass))
            write(*,*)
         end if
         
         call do_alloc(ierr)
         if (ierr /= 0) return
         
         do k=1,nz
            old_cell_mass(k) = old_xmstar*s% dq(k)
         end do
         xm_old(1) = 0
         do k=2,nz
            xm_old(k) = xm_old(k-1) + old_cell_mass(k-1)
         end do            
         
         if (delta_m > 0) then
            s% angular_momentum_removed = 0
         else
            s% angular_momentum_removed = angular_momentum_removed(ierr)
            if (ierr /= 0) return
         end if
         
         call revise_q_and_dq( &
            s, nz, old_xmstar, new_xmstar, k_const_mass, ierr)
         if (ierr /= 0) then
            if (s% report_ierr) write(*, *) 'revise_q_and_dq failed in adjust mass'
            call dealloc
            return
         end if
         
         s% k_const_mass = k_const_mass

         do k=1,nz
            new_cell_mass(k) = new_xmstar*s% dq(k)
         end do
         
         xm_new(1) = 0
         do k=2,nz
            xm_new(k) = xm_new(k-1) + new_cell_mass(k-1)
         end do            

         xq_for_mdot_eps = &
            s% depth_factor_for_k_below_mdot_eps_grav*abs(delta_m)/new_xmstar
         s% k_below_mdot_eps_grav = nz
         sum_dq = 0
         do k = 1, nz
            if (sum_dq >= xq_for_mdot_eps) then
               s% k_below_mdot_eps_grav = k; exit
            end if
            sum_dq = sum_dq + s% dq(k)
         end do
         
         if (delta_m > 0) then
         
            q_for_just_added = old_xmstar/new_xmstar
            s% k_below_just_added = nz
            do k = 1, nz
               if (s% q(k) <= q_for_just_added) then
                  s% k_below_just_added = k; exit
               end if
            end do
            
            !write(*,2) 'set del_t_for_just_added using dt', s% model_number, dt
            do k = 1, s% k_below_just_added-1
               s% del_t_for_just_added(k) = min(dt,xm_new(k+1)/s% mstar_dot)
               if (s% del_t_for_just_added(k) < s% min_del_T_div_dt*dt) then
                  s% del_t_for_just_added(k) = s% min_del_T_div_dt*dt
               end if
               !write(*,2) 'del_t_for_just_added/dt', k, s% del_t_for_just_added(k)/dt
            end do
         
            q_for_recently_added = &
               old_xmstar/(old_xmstar + s% factor_for_recently_added*delta_m)
            s% k_below_recently_added = nz
            do k = 1, nz
               if (s% q(k) <= q_for_recently_added) then
                  s% k_below_recently_added = k; exit
               end if
            end do
            
         end if
         
         if (s% show_info_for_recently_added .and. s% k_below_recently_added > 1) &
            write(*,2) 'recently_added: k_below, log 1-q', &
               s% k_below_recently_added, safe_log10(1d0 - q_for_recently_added)
                  
         do k=1,nz
            do j=1,species
               xa_old(j,k) = s% xa(j,k)
            end do
         end do

         if (delta_m < 0) then
            xaccrete(1:species) = 0 ! xaccrete not used when removing mass
         else ! set xaccrete for composition of added material
            if (s% accrete_same_as_surface) then
               do j=1,species
                  xaccrete(j) = xa_old(j,1)
               end do
            else
               call get_xa_for_accretion(s, xaccrete, ierr)
               if (ierr /= 0) then
                  if (s% report_ierr) write(*, *) 'get_xa_for_accretion failed in adjust mass'
                  call dealloc
                  return
               end if               
            end if
         end if
         
         mmax = max(old_mstar, new_mstar)
         
         ! rxm_old and rxm_new are for interpolating by mass
         ! but instead of using mass coord, we want to use external mass
         ! in order to get better accuracy near the surface.
         ! to simplify this, the zero point is the same for both rxm_old and rxm_new.
         ! that makes rxm_new = rxm_old for k >= k_const_mass.
         if (delta_m < 0) then
            rxm_old(1) = 0
            rxm_new(1) = -delta_m ! note that rxm_new(1) > 0 since delta_m < 0
         else
            rxm_old(1) = delta_m
            rxm_new(1) = 0
         end if
         do k = 2, nz
            rxm_old(k) = rxm_old(k-1) + old_cell_mass(k-1)
            if (k >= k_const_mass) then
               rxm_new(k) = rxm_old(k)
               new_cell_mass(k) = old_cell_mass(k)
            else
               rxm_new(k) = rxm_new(k-1) + new_cell_mass(k-1)
            end if
         end do

         call set_lnd_lnT_lnPgas_for_d_dt( &
            s, nz, k_const_mass, s% k_below_just_added, &
            rxm_old, rxm_new, old_cell_mass, new_cell_mass, &
            delta_m, old_xmstar, new_xmstar, &
            oldloc, newloc, oldval, newval, work, ierr)
         if (ierr /= 0) then
            if (s% report_ierr) &
               write(*, *) 'set_lnd_lnT_lnPgas_for_d_dt failed in adjust mass' 
            call dealloc
            return
         end if
               
         call set_lnR_v_for_d_dt( &
            s, nz, k_const_mass, s% k_below_just_added, &
            rxm_old, rxm_new, delta_m, old_xmstar, new_xmstar, &
            oldloc, newloc, oldval, newval, work, ierr)         
         if (ierr /= 0) then
            if (s% report_ierr) &
               write(*, *) 'set_lnR_v_for_d_dt failed in adjust mass' 
            call dealloc
            return
         end if

         call set_xa(s, nz, k_const_mass, species, xa_old, xaccrete, &
            rxm_old, rxm_new, mmax, old_cell_mass, new_cell_mass, ierr)  
         if (ierr /= 0) then
            if (s% report_ierr) &
               write(*, *) 'set_xa failed in adjust mass' 
            call dealloc
            return
         end if
         
         if (s% rotation_flag) then     
            call set_omega( &
            	s, nz, k_const_mass, delta_m, &
            	rxm_old, rxm_new, mmax, old_cell_mass, new_cell_mass, &
            	! reuse these work arrays
            	oldloc, newloc, oldval, newval, xm_old, xm_new, &            	
            	ierr)
            if (ierr /= 0) then
               if (s% report_ierr) &
                  write(*, *) 'set_omega failed in adjust mass' 
               call dealloc
               return
            end if
         end if
                  
         call dealloc
         
         if (dbg_adjm) stop 'debugging: do_adjust_mass'
         if (dbg) write(*,*) 'do_adjust_mass return'

         
         contains
                        
         
         real(dp) function angular_momentum_removed(ierr) result(J)
				! when call this, s% j_rot is still for old mass
            integer, intent(out) :: ierr
            integer :: k
            real(dp) :: r2, dmm1, dm00, dm, dm_sum, dm_lost
            include 'formats'
            ierr = 0
            J = 0
            if (.not. s% rotation_flag) return
            dm00 = 0
            dm_sum = 0
            dm_lost = -delta_m
            do k = 1, nz
               dmm1 = dm00
               dm00 = old_cell_mass(k)
               dm = 0.5d0*(dmm1+dm00)
               if (dm_sum + dm > dm_lost) then
                  dm = dm_lost - dm_sum
                  dm_sum = dm_lost
					else
						dm_sum = dm_sum + dm
               end if
               J = J + dm*s% j_rot(k)
               if (dm_sum == dm_lost) exit
            end do
         end function angular_momentum_removed


         real(dp) function eval_total_angular_momentum(s,cell_mass,nz_last) result(J)
            type (star_info), pointer :: s
				real(dp) :: cell_mass(:)
            integer, intent(in) :: nz_last
            integer :: k
            real(dp) :: dmm1, dm00, dm
            include 'formats'
            J = 0
            if (.not. s% rotation_flag) return
            dm00 = 0
            do k = 1, nz_last
               dmm1 = dm00
               dm00 = cell_mass(k)
               if (k == s% nz) then
                  dm = 0.5d0*dmm1+dm00
               else if (k == nz_last) then
                  dm = 0.5d0*dmm1
               else
                  dm = 0.5d0*(dmm1+dm00)
               end if
               J = J + dm*s% j_rot(k)
            end do
         end function eval_total_angular_momentum
         

         subroutine do_alloc(ierr)
            use interp_1d_def
            use alloc
            integer, intent(out) :: ierr
            ierr = 0            
            call non_crit_get_work_array(s, rxm_old, nz, nz_alloc_extra, 'adjust_mass rxm_old', ierr)
            if (ierr /= 0) return            
            call non_crit_get_work_array(s, rxm_new, nz, nz_alloc_extra, 'adjust_mass rxm_new', ierr)
            if (ierr /= 0) return            
            call non_crit_get_work_array(s, old_cell_mass, nz, nz_alloc_extra, 'adjust_mass old_cell_mass', ierr)
            if (ierr /= 0) return            
            call non_crit_get_work_array(s, new_cell_mass, nz, nz_alloc_extra, 'adjust_mass new_cell_mass', ierr)
            if (ierr /= 0) return            
            call non_crit_get_work_array(s, xa_old1, species*nz, nz_alloc_extra, 'adjust_mass xa_old', ierr)
            if (ierr /= 0) return   
            xa_old(1:species,1:nz) => xa_old1(1:species*nz)         
            call non_crit_get_work_array(s, oldloc, nz, nz_alloc_extra, 'adjust_mass oldloc', ierr)
            if (ierr /= 0) return            
            call non_crit_get_work_array(s, newloc, nz, nz_alloc_extra, 'adjust_mass newloc', ierr)
            if (ierr /= 0) return            
            call non_crit_get_work_array(s, oldval, nz, nz_alloc_extra, 'adjust_mass oldval', ierr)
            if (ierr /= 0) return            
            call non_crit_get_work_array(s, newval, nz, nz_alloc_extra, 'adjust_mass newval', ierr)
            if (ierr /= 0) return            
            call non_crit_get_work_array(s, work, nz*pm_work_size, nz_alloc_extra, 'adjust_mass work', ierr)
            if (ierr /= 0) return            
            call non_crit_get_work_array(s, xm_old, nz, 0, 'adjust_mass', ierr)
            if (ierr /= 0) return            
            call non_crit_get_work_array(s, xm_new, nz, 0, 'adjust_mass', ierr)
            if (ierr /= 0) return
         end subroutine do_alloc
         
         
         subroutine dealloc
            use alloc            
            call non_crit_return_work_array(s, rxm_old, 'adjust_mass rxm_old')            
            call non_crit_return_work_array(s, rxm_new, 'adjust_mass rxm_new')            
            call non_crit_return_work_array(s, old_cell_mass, 'adjust_mass old_cell_mass')            
            call non_crit_return_work_array(s, new_cell_mass, 'adjust_mass new_cell_mass')            
            call non_crit_return_work_array(s, xa_old1, 'adjust_mass xa_old')            
            call non_crit_return_work_array(s, oldloc, 'adjust_mass oldloc')            
            call non_crit_return_work_array(s, newloc, 'adjust_mass newloc')            
            call non_crit_return_work_array(s, oldval, 'adjust_mass oldval')            
            call non_crit_return_work_array(s, newval, 'adjust_mass newval')            
            call non_crit_return_work_array(s, work, 'adjust_mass work')                  
            call non_crit_return_work_array(s, xm_old, 'adjust_mass xm_old')
            call non_crit_return_work_array(s, xm_new, 'adjust_mass xm_new')
         end subroutine dealloc
         
         
      end subroutine do_adjust_mass
            
      
      subroutine revise_q_and_dq( &
            s, nz, old_xmstar, new_xmstar, k_const_mass, ierr)
         use star_utils, only: normalize_dqs, set_qs
         type (star_info), pointer :: s
         integer, intent(in) :: nz
         real(dp), intent(in) :: old_xmstar, new_xmstar
         integer, intent(out) :: k_const_mass, ierr
         
         integer :: k, kA, kB, i_lnT, j00, jp1
         real(dp) :: lnTlim_A, lnTlim_B, qlim_A, qlim_B, sumdq, sumdq1, sumdq2, sumdq3
         real(dp) :: frac, lnTmax, lnT_A, lnT_B, qA, qB_old, qB_new, qfrac
         integer, parameter :: min_kA = 5
         logical :: dbg
         logical :: okay_to_move_kB_inward

         include 'formats'
         
         ierr = 0
         dbg = .false.
         
         okay_to_move_kB_inward = .false.
         
         lnTlim_A = ln10*s% adjust_mass_logT_limit_A
         lnTlim_B = ln10*s% adjust_mass_logT_limit_B
         
         qlim_A = s% adjust_mass_q_limit_A
         qlim_B = s% adjust_mass_q_limit_B
         
         frac = old_xmstar / new_xmstar
         i_lnT = s% i_lnT
         
         lnTmax = maxval(s% xh(i_lnT,1:nz))
         
         kA = min_kA
         lnT_A = min(lnTmax, lnTlim_A)
         do k = min_kA, nz
            if (s% xh(i_lnT,k) >= lnT_A .or. s% q(k) < qlim_A) then
               kA = k; exit
            end if
         end do
         qA = s% q(kA)
         
         !write(*,2) 'qA', kA, qA
         !write(*,2) 'qlim_A', kA, qlim_A
         
         kB = 0
         lnT_B = min(lnTmax, lnTlim_B)
         do k = kA+1, nz
            if (s% xh(i_lnT,k) >= lnT_B .or. s% q(k) < qlim_B) then
               kB = k; exit
            end if
         end do
         if (kB == 0) kB = nz
         qB_old = s% q(kB)
         
         if (dbg_adjm) then
            write(*,*) 'kA', kA
            write(*,1) 'qA', qA
            write(*,*) 'kB', kB
            write(*,1) 'qB_old', qB_old
            write(*,*)
            write(*,1) 'qA-qB_old', qA-qB_old
            write(*,*)
         end if
         
         !write(*,2) 'qlim_B', kB, qlim_B
         !write(*,2) 'qB_old', kB, qB_old
                  
         qB_new = qB_old * frac ! in order to keep m(kB) constant
         !write(*,2) 'frac', kB, frac
         
         
         
         do ! make sure qfrac is not too far from 1
            qfrac = (qA - qB_new) / max(1d-99,qA - qB_old)
            !write(*,2) 'qB_old', kB, qB_old
            !write(*,2) 'qB_new', kB, qB_new
            !write(*,2) 'qA - qB_old', kB, qA - qB_old
            !write(*,2) 'qA - qB_new', kB, qA - qB_new
            !write(*,2) '(qA-qB_new)/(qA-qB_old)', kB, (qA - qB_new)/(qA - qB_old)
            if (kB == nz) exit
            if (kB-kA > 10) then
               if (qfrac > 0.67d0 .and. qfrac < 1.5d0) exit
               if (qfrac > 0.50d0 .and. qfrac < 2.0d0) then
                  j00 = maxloc(s% xa(:,kB),dim=1) ! most abundant species at kB
                  jp1 = maxloc(s% xa(:,kB+1),dim=1) ! most abundant species at kB+1
                  if (j00 /= jp1) then ! change in composition.
                     if (dbg) write(*,*) 'change in composition.  back up kB.'
                     kB = max(1,kB-5)
                     exit
                  end if
               end if
            end if
            kB = kB+1
            qB_old = s% q(kB)
            qB_new = qB_old * frac
         end do

         !write(*,2) 'after adjust kA', kA, qA
         !write(*,2) 'after adjust kB', kB, qB_old, qB_new         
         !stop 'adjust_mass'
         
         k_const_mass = kB
         
         ! set new dq's
         ! s% dq(1:kA-1) unchanged
         s% dq(kA:kB-1) = s% dq(kA:kB-1)*qfrac
         s% dq(kB:nz) = s% dq(kB:nz)*frac
         
         if (dbg_adjm) then
            write(*,1) 'frac for kb:nz', frac
            write(*,1) 'qfrac for kA:kB-1', qfrac
            write(*,1) 'revise_q_and_dq sum dqs', sum(s% dq(1:nz))
            write(*,2) 'qfrac region', kB, qfrac, s% q(kB), s% lnT(kB)/ln10
            write(*,2) 'frac region', kA, frac, s% q(kA), s% lnT(kA)/ln10
            write(*,2) 'kA', kA
            write(*,2) 'kB', kB
            write(*,2) 'nz', nz
            write(*,*)
            stop 'adjust_mass'
         end if
         
         sumdq1 = 0 ! sumdq1 = sum dq(1:kA-1)
         sumdq2 = 0 ! sumdq2 = sum dq(kA:kB-1)
         sumdq3 = 0 ! sumdq3 = sum dq(kB-1:nz)
         sumdq = s% dq(1)
         do k = 2, nz
            if (k == kA) then
               sumdq1 = sumdq
               sumdq = 0
            end if
            if (k == kB) then
               sumdq2 = sumdq
               sumdq = 0
            end if
            sumdq = sumdq + s% dq(k)
         end do
         sumdq3 = sumdq
         
         frac = (1d0 - sumdq3)/(sumdq1 + sumdq2)
         s% dq(1:kB-1) = s% dq(1:kB-1)*frac
         
         ! set q's to match new dq's
         s% q(1) = 1d0
         sumdq = s% dq(1)
         do k = 2, nz-1
            s% q(k) = 1d0 - sumdq
            sumdq = sumdq + s% dq(k)
         end do
         s% q(nz) = 1d0 - sumdq
         
         if (s% q(nz) < 0.9d0*s% dq(nz) .or. &
               abs(s% q(nz)-s% dq(nz)) > 1d-3*s% dq(nz)) then
            ierr = -1
            if (s% report_ierr) &
               write(*,1) 'adjust mass: bad dqs at nz', &
                  s% q(nz)-s% dq(nz), s% q(nz), s% dq(nz)
         end if
      end subroutine revise_q_and_dq    

      
      subroutine set_xa( &
            s, nz, k_const_mass, species, xa_old, xaccrete, &
            old_cell_xbdy, new_cell_xbdy, mmax, old_cell_mass, new_cell_mass, ierr)
         ! set new values for s% xa(:,:)
         type (star_info), pointer :: s
         integer, intent(in) :: nz, k_const_mass, species
         real(dp), intent(in) :: mmax
         real(dp), intent(in), pointer :: xa_old(:, :), xaccrete(:)
         real(dp), dimension(:), intent(in), pointer :: &
            old_cell_xbdy, new_cell_xbdy, old_cell_mass, new_cell_mass ! (nz)
         integer, intent(out) :: ierr         
         integer :: k, j, op_err
         real(dp), parameter :: max_sum_abs = 10d0
         real(dp), parameter :: xsum_tol = 1d-2  
         include 'formats'       
         ierr = 0
         if (dbg_adjm) &
            write(*,2) 'set_xa: k_const_mass', k_const_mass
         if (k_const_mass < nz) then
            ! for k >= k_const_mass have m_new(k) = m_old(k),
            ! so no change in xa_new(:,k) for k > k_const_mass
            do k=k_const_mass+1,nz
               do j=1,species
                  s% xa(j,k) = xa_old(j,k)
               end do
            end do
         end if
!$OMP PARALLEL DO PRIVATE(k, op_err)
         do k = 1, k_const_mass
            op_err = 0
            call set1_xa(s, k, nz, species, xa_old, xaccrete, &
               old_cell_xbdy, new_cell_xbdy, mmax, old_cell_mass, new_cell_mass, op_err)
            if (op_err /= 0) ierr = op_err
         end do
!$OMP END PARALLEL DO

      end subroutine set_xa

      
      subroutine set1_xa(s, k, nz, species, xa_old, xaccrete, &
            old_cell_xbdy, new_cell_xbdy, mmax, old_cell_mass, new_cell_mass, ierr)
         ! set new values for s% xa(:,k)
         use num_lib, only: binary_search
         use utils_lib, only: is_bad_num
         use chem_def, only: chem_isos
         type (star_info), pointer :: s
         integer, intent(in) :: k, nz, species
         real(dp), intent(in) :: mmax
         real(dp), intent(in), pointer :: xa_old(:,:), xaccrete(:)
         real(dp), dimension(:), intent(in), pointer :: &
            old_cell_xbdy, new_cell_xbdy, old_cell_mass, new_cell_mass
         integer, intent(out) :: ierr
         
         real(dp) :: xm_outer, xm_inner, msum(species), sumx, &
            xm0, xm1, new_cell_dm, dm_sum, dm
         integer :: kk, k_outer, j
         
         integer, parameter :: k_dbg = -1
         logical, parameter :: xa_dbg = .false.
         
         logical, parameter :: do_not_mix_accretion = .false.
         
         include 'formats'
         
         ierr = 0
         msum(:) = -1 ! for testing
                  
         xm_outer = new_cell_xbdy(k)
         if (k == nz) then
            new_cell_dm = mmax - xm_outer - s% M_center
         else
            new_cell_dm = new_cell_mass(k)
         end if
         xm_inner = xm_outer + new_cell_dm
         
         dm_sum = 0d0
         
         if (xm_outer < old_cell_xbdy(1)) then ! there is some accreted material in new cell
            if (do_not_mix_accretion .or. xm_inner <= old_cell_xbdy(1)) then 
               ! new cell is entirely accreted material
               !write(*,2) 'new cell is entirely accreted material', k, new_cell_dm
               do j=1,species
                  s% xa(j,k) = xaccrete(j)
               end do
               return
            end if
            dm = min(new_cell_dm, old_cell_xbdy(1) - xm_outer)
            dm_sum = dm
            do j=1,species
               msum(j) = xaccrete(j)*dm
            end do
            xm_outer = old_cell_xbdy(1)
            k_outer = 1
         else ! new cell entirely composed of old material
            msum(:) = 0
            if (xm_outer >= old_cell_xbdy(nz)) then
               ! new cell contained entirely in old cell nz
               k_outer = nz
            else
               ! binary search for k_outer such that
               ! xm_outer >= old_cell_xbdy(k_outer)
               ! and old_cell_xbdy(k_outer+1) > xm_outer
               k_outer = binary_search(nz, old_cell_xbdy, 0, xm_outer)
               
               ! check
               if (k_outer <= 0 .or. k_outer > nz) then

                  ierr = -1
                  if (.not. xa_dbg) return

                  write(*,2) 'k', k
                  write(*,2) 'k_outer', k_outer
                  write(*,1) 'xm_outer', xm_outer
                  write(*,2) 'old_cell_xbdy(1)', 1, old_cell_xbdy(1)
                  write(*,2) 'old_cell_xbdy(nz)', nz, old_cell_xbdy(nz)
                  stop 'debugging: set1_xa'
               end if
               
               if (xm_outer < old_cell_xbdy(k_outer)) then

                  ierr = -1
                  if (.not. xa_dbg) return

                  write(*,*) 'k', k
                  write(*,*) 'k_outer', k_outer
                  write(*,1) 'xm_outer', xm_outer
                  write(*,1) 'old_cell_xbdy(k_outer)', old_cell_xbdy(k_outer)
                  write(*,*) '(xm_outer < old_cell_xbdy(k_outer))'
                  stop 'debugging: set1_xa'
               end if
               
               if (k_outer < nz) then
                  if (old_cell_xbdy(k_outer+1) <= xm_outer) then

                     ierr = -1
                     if (.not. xa_dbg) return

                     write(*,*) 'k', k
                     write(*,*) 'k_outer', k_outer
                     write(*,1) 'xm_outer', xm_outer
                     write(*,1) 'old_cell_xbdy(k_outer+1)', old_cell_xbdy(k_outer+1)
                     write(*,*) '(old_cell_xbdy(k_outer+1) <= xm_outer)'
                     stop 'debugging: set1_xa'
                  end if
               end if
               
            end if
         end if
         
         if (k == -1) then
            ierr = -1
            if (.not. xa_dbg) return
            
            write(*,2) 'nz', nz
            write(*,2) 'k_outer', k_outer
            write(*,1) 'xm_outer', xm_outer
            write(*,1) 'xm_inner', xm_inner
         end if

         do kk = k_outer, nz ! loop until reach m_inner
            xm0 = old_cell_xbdy(kk)
            
            if (xm0 >= xm_inner) then
               if (dm_sum < new_cell_dm .and. kk > 1) then 
                  ! need to add a bit more from the previous source cell
                  dm = new_cell_dm - dm_sum
                  dm_sum = new_cell_dm
                  do j=1,species
                     msum(j) = msum(j) + xa_old(j,kk-1)*dm
                  end do
               end if
               exit
            end if
            
            if (kk == nz) then
               xm1 = mmax - s% M_center
            else
               xm1 = old_cell_xbdy(kk+1)
            end if
            
            if (xm1 < xm_outer) then
               ierr = -1
               if (.not. xa_dbg) return
               write(*,*)
               write(*,*) 'k', k
               write(*,*) 'kk', kk
               write(*,1) 'xm1', xm1
               write(*,1) 'xm_outer', xm_outer
               write(*,*) 'xm1 < xm_outer'
               stop 'debugging: set1_xa'
            end if
            
            if (xm0 >= xm_outer .and. xm1 <= xm_inner) then ! entire old cell kk is in new cell k
               
               dm = old_cell_mass(kk)
               dm_sum = dm_sum + dm
               
               if (dm_sum > new_cell_dm) then 
                  ! dm too large -- numerical roundoff problems
                  dm = dm - (new_cell_dm - dm_sum)
                  dm_sum = new_cell_dm
               end if
               
               do j=1,species
                  msum(j) = msum(j) + xa_old(j,kk)*dm
               end do
               
            else if (xm0 <= xm_outer .and. xm1 >= xm_inner) then ! entire new cell k is in old cell kk
            
               dm = new_cell_mass(k)
               dm_sum = dm_sum + dm
               do j=1,species
                  msum(j) = msum(j) + xa_old(j,kk)*dm
               end do
               
            else ! only use the part of old cell kk that is in new cell k
            
               if (xm_inner <= xm1) then ! this is the last part of new cell k
               
                  dm = new_cell_dm - dm_sum
                  dm_sum = new_cell_dm

               else ! notice that we avoid this case if possible because of numerical roundoff
               
                  dm = max(0d0, xm1 - xm_outer)
                  if (dm_sum + dm > new_cell_dm) dm = new_cell_dm - dm_sum
                  dm_sum = dm_sum + dm

               end if
               
               do j=1,species
                  msum(j) = msum(j) + xa_old(j,kk)*dm
               end do
               
               if (dm <= 0) then
                  ierr = -1
                  if (.not. xa_dbg) return
                  write(*,*) 'dm <= 0', dm
                  stop 'debugging: set1_xa'
               end if
               
            end if
            
            if (dm_sum >= new_cell_dm) then
               exit
            end if
            
         end do

         ! revise and renormalize
         do j=1,species
            s% xa(j,k) = msum(j) / new_cell_mass(k)
         end do
         sumx = sum(s% xa(:,k))
         do j=1,species
            s% xa(j,k) = s% xa(j,k)/sumx
         end do
               
      end subroutine set1_xa

      
      subroutine set_omega( &
            s, nz, k_const_mass, delta_m, &
            old_cell_xbdy, new_cell_xbdy, mmax, old_cell_mass, new_cell_mass, &
         	! work arrays (nz)
         	old_xout, new_xout, old_dmbar, new_dmbar, old_j_rot, extra_work, &            	
            ierr)
         use star_utils, only: total_angular_momentum
         type (star_info), pointer :: s
         integer, intent(in) :: nz, k_const_mass
         real(dp), intent(in) :: mmax, delta_m
         real(dp), dimension(:), intent(in), pointer :: &
            old_cell_xbdy, new_cell_xbdy, old_cell_mass, new_cell_mass ! (nz)
			real(dp), pointer, dimension(:) :: &
			   old_xout, new_xout, old_dmbar, new_dmbar, old_j_rot, extra_work
         integer, intent(out) :: ierr       
           
         integer :: k, k0, op_err, old_k, new_k
         real(dp) :: old_j_tot, new_j_tot, goal_total_added, actual_total_added, &
            f, jtot_bdy, new_total, goal_total, err_total, &
            bdy_j, bdy_total, inner_total, outer_total
         include 'formats.dek'    
            
         ierr = 0
			   
			old_xout(1) = old_cell_xbdy(1)
			new_xout(1) = new_cell_xbdy(1)
			old_dmbar(1) = old_cell_mass(1)/2
			new_dmbar(1) = new_cell_mass(1)/2
			old_j_rot(1) = s% j_rot(1)
			do k=2,nz
				old_xout(k) = old_xout(k-1) + old_dmbar(k-1)
				new_xout(k) = new_xout(k-1) + new_dmbar(k-1)
				old_dmbar(k) = (old_cell_mass(k-1) + old_cell_mass(k))/2
				new_dmbar(k) = (new_cell_mass(k-1) + new_cell_mass(k))/2
			   old_j_rot(k) = s% j_rot(k)
			end do
			old_dmbar(nz) = old_cell_mass(nz-1)/2 + old_cell_mass(nz)
			new_dmbar(nz) = new_cell_mass(nz-1)/2 + new_cell_mass(nz)
         
         old_j_tot = dot_product(s% j_rot(1:nz),old_dmbar(1:nz))

			if (s% k_below_just_added == 1) then
			   k0 = 1
			else
			   k0 = s% k_below_just_added + 1
			end if
!$OMP PARALLEL DO PRIVATE(k, op_err)
         do k = 1, k_const_mass
            if (k < k0) then
               call set1_irot(s, k)
               cycle
            end if
            op_err = 0
            call set1_omega( &
               s, k, nz, old_xout, new_xout, mmax, old_dmbar, new_dmbar, old_j_rot, op_err)
            if (op_err /= 0) ierr = op_err
         end do
!$OMP END PARALLEL DO
         
         if (s% k_below_just_added > 1) then 
            ! set omega in cells with newly added material
            if (s% use_accreted_material_j) then
               actual_total_added = 0d0
               do k=1,s% k_below_just_added-2 ! remaining 2 done below
                  s% j_rot(k) = s% accreted_material_j
                  s% omega(k) = s% j_rot(k)/s% i_rot(k)
                  actual_total_added = actual_total_added + s% j_rot(k)*new_dmbar(k)
               end do
               k = s% k_below_just_added
               goal_total_added = delta_m*s% accreted_material_j
               goal_total = old_j_tot + goal_total_added
               inner_total = dot_product(s% j_rot(k+1:nz),new_dmbar(k+1:nz))
               outer_total = dot_product(s% j_rot(1:k-2),new_dmbar(1:k-2))
               bdy_total = goal_total - (inner_total + outer_total)
               bdy_j = bdy_total/sum(new_dmbar(k-1:k))
               if (bdy_j > 0) then
                  do k=s% k_below_just_added-1,s% k_below_just_added
                     s% j_rot(k) = bdy_j
                     s% omega(k) = s% j_rot(k)/s% i_rot(k)
                  end do
                  new_total = dot_product(s% j_rot(1:nz),new_dmbar(1:nz))
                  err_total = new_total - goal_total
               else
                  ierr = -1
               end if
            else ! use old surface omega in all the new material
               do k=1,s% k_below_just_added-1
                  s% omega(k) = s% omega(s% k_below_just_added)
                  s% j_rot(k) = s% omega(k)*s% i_rot(k)
               end do
            end if
         end if
			
      end subroutine set_omega


      subroutine set1_irot(s, k) ! using lnR_for_d_dt
         use star_utils, only: eval_i_rot
         type (star_info), pointer :: s
         integer, intent(in) :: k
         
         real(dp) :: r00, r003, ri, ro, rp13, rm13
      
         if (s% simple_i_rot_flag) then
			   s% i_rot(k) = (2d0/3d0)*exp(2*s% lnR_for_d_dt(k))
			else
			   r00 = exp(s% lnR_for_d_dt(k))
			   r003 = r00**3
			   if (k == s% nz) then
			      rp13 = s% R_center**3
			   else
			      rp13 = exp(3*s% lnR_for_d_dt(k+1))
			   end if
			   if (k == 1) then
			      rm13 = r003
			   else
			      rm13 = exp(3*s% lnR_for_d_dt(k-1))
			   end if
			   ri = ((r003 + rp13)/2)**(1d0/3d0)
			   ro = ((r003 + rm13)/2)**(1d0/3d0)
			   s% i_rot(k) = eval_i_rot(s, ri, r00, ro)
			end if
      
      end subroutine set1_irot

      
		! this works like set1_xa except shifted to cell edge instead of cell center
      subroutine set1_omega(s, k, nz, &
            old_xout, new_xout, mmax, old_dmbar, new_dmbar, old_j_rot, ierr)
         ! set new value for s% omega(k)
         use num_lib, only: binary_search
         use utils_lib, only: is_bad_num
         type (star_info), pointer :: s
         integer, intent(in) :: k, nz
         real(dp), intent(in) :: mmax
         real(dp), dimension(:), intent(in), pointer :: &
            old_xout, new_xout, old_dmbar, new_dmbar, old_j_rot
         integer, intent(out) :: ierr
         
         real(dp) :: xm_outer, xm_inner, j_tot, xm0, xm1, new_point_dmbar, &
            dm_sum, dm
         integer :: kk, k_outer, j
         
         integer, parameter :: k_dbg = -1
         
         include 'formats.dek'
         
         ierr = 0                  
         
         call set1_irot(s, k)

         xm_outer = new_xout(k)
         if (k == nz) then
            new_point_dmbar = mmax - xm_outer - s% M_center
         else
            new_point_dmbar = new_dmbar(k)
         end if
         xm_inner = xm_outer + new_point_dmbar
         
         if (k == k_dbg) then
            write(*,2) 'xm_outer', k, xm_outer
            write(*,2) 'xm_inner', k, xm_inner
            write(*,2) 'new_point_dmbar', k, new_point_dmbar
         end if
         
         !write(*,*)
         !write(*,2) 'xm_outer', k, xm_outer
         
         dm_sum = 0d0
         
         if (xm_outer < old_xout(1)) then ! there is some accreted material in new
            if (xm_inner <= old_xout(1)) then 
               ! new is entirely accreted material
               !write(*,2) 'new is entirely accreted material', k, new_point_dmbar
               s% omega(k) = 0
               return
            end if
            dm = min(new_point_dmbar, old_xout(1) - xm_outer)
            dm_sum = dm
            j_tot = 0
            xm_outer = old_xout(1)
            k_outer = 1
         else ! new entirely composed of old material
            if (k == k_dbg) write(*,*) 'new entirely composed of old material'
            j_tot = 0
            if (xm_outer >= old_xout(nz)) then
               ! new contained entirely in old nz
               k_outer = nz
            else
               ! binary search for k_outer such that
               ! xm_outer >= old_xout(k_outer)
               ! and old_xout(k_outer+1) > xm_outer
               k_outer = binary_search(nz, old_xout, 0, xm_outer)
               
               if (k == k_dbg) write(*,2) 'k_outer', k_outer, old_xout(k_outer), old_xout(k_outer+1)
               
               ! check
               if (k_outer <= 0 .or. k_outer > nz) then

                  ierr = -1
                  !return

                  write(*,2) 'k', k
                  write(*,2) 'k_outer', k_outer
                  write(*,1) 'xm_outer', xm_outer
                  write(*,2) 'old_xout(1)', 1, old_xout(1)
                  write(*,2) 'old_xout(nz)', nz, old_xout(nz)
                  stop 'debugging: set1_omega'
               end if
               
               if (xm_outer < old_xout(k_outer)) then

                  ierr = -1
                  !return

                  write(*,*) 'k', k
                  write(*,*) 'k_outer', k_outer
                  write(*,1) 'xm_outer', xm_outer
                  write(*,1) 'old_xout(k_outer)', old_xout(k_outer)
                  write(*,*) '(xm_outer < old_xout(k_outer))'
                  stop 'debugging: set1_omega'
               end if
               
               if (k_outer < nz) then
                  if (old_xout(k_outer+1) <= xm_outer) then

                     ierr = -1
                     !return

                     write(*,*) 'k', k
                     write(*,*) 'k_outer', k_outer
                     write(*,1) 'xm_outer', xm_outer
                     write(*,1) 'old_xout(k_outer+1)', old_xout(k_outer+1)
                     write(*,*) '(old_xout(k_outer+1) <= xm_outer)'
                     stop 'debugging: set1_omega'
                  end if
               end if
               
            end if
         end if
         
         if (k == -1) then
            ierr = -1
            !return
            
            write(*,2) 'nz', nz
            write(*,2) 'k_outer', k_outer
            write(*,1) 'xm_outer', xm_outer
            write(*,1) 'xm_inner', xm_inner
				stop 'debugging: set1_omega'
         end if

         do kk = k_outer, nz ! loop until reach m_inner
            xm0 = old_xout(kk)
               
            if (k == k_dbg) write(*,2) 'kk', kk, old_xout(kk), old_xout(kk+1)
            
            if (xm0 >= xm_inner) then
               if (dm_sum < new_point_dmbar .and. kk > 1) then 
                  ! need to add a bit more from the previous source
                  dm = new_point_dmbar - dm_sum
                  dm_sum = new_point_dmbar
                  j_tot = j_tot + old_j_rot(kk-1)*dm
               
                  if (.false. .or. k == k_dbg) &
                     write(*,3) 'new k contains some of old kk-1', &
                        k, kk, old_j_rot(kk-1)*dm, old_j_rot(kk-1), dm, j_tot/dm_sum, j_tot, dm_sum

                  end if
               exit
            end if
            
            if (kk == nz) then
               xm1 = mmax - s% M_center
            else
               xm1 = old_xout(kk+1)
            end if
            
            if (xm1 < xm_outer) then
               ierr = -1
               !return
               write(*,*)
               write(*,*) 'k', k
               write(*,*) 'kk', kk
               write(*,1) 'xm1', xm1
               write(*,1) 'xm_outer', xm_outer
               write(*,*) 'xm1 < xm_outer'
               stop 'debugging: set1_omega'
            end if
            
            if (xm0 >= xm_outer .and. xm1 <= xm_inner) then ! entire old kk is in new k
               
               dm = old_dmbar(kk)
               dm_sum = dm_sum + dm
               
               if (dm_sum > new_point_dmbar) then 
                  ! dm too large -- numerical roundoff problems
                  dm = dm - (new_point_dmbar - dm_sum)
                  dm_sum = new_point_dmbar
               end if
               
               j_tot = j_tot + old_j_rot(kk)*dm
               
               if (.false. .or. k == k_dbg) &
                  write(*,3) 'new k contains all of old kk', &
                     k, kk, old_j_rot(kk)*dm, old_j_rot(kk), dm, j_tot/dm_sum, j_tot, dm_sum
               
            else if (xm0 <= xm_outer .and. xm1 >= xm_inner) then ! entire new k is in old kk
            
               dm = new_dmbar(k)
               dm_sum = dm_sum + dm
               j_tot = j_tot + old_j_rot(kk)*dm
               
               if (.false. .or. k == k_dbg) &
                  write(*,3) 'all new k is in old kk', &
                     k, kk, old_j_rot(kk)*dm, old_j_rot(kk), dm, j_tot/dm_sum, j_tot, dm_sum
               
            else ! only use the part of old kk that is in new k
            
               if (k == k_dbg) then
                  write(*,*) 'only use the part of old kk that is in new k', xm_inner <= xm1
                  write(*,1) 'xm_outer', xm_outer
                  write(*,1) 'xm_inner', xm_inner
                  write(*,1) 'xm0', xm0
                  write(*,1) 'xm1', xm1
                  write(*,1) 'dm_sum', dm_sum
                  write(*,1) 'new_point_dmbar', new_point_dmbar
                  write(*,1) 'new_point_dmbar - dm_sum', new_point_dmbar - dm_sum
               end if
            
               if (xm_inner <= xm1) then ! this is the last part of new k
               
                  if (k == k_dbg) write(*,3) 'this is the last part of new k', k, kk

                  dm = new_point_dmbar - dm_sum
                  dm_sum = new_point_dmbar

               else ! notice that we avoid this case if possible because of numerical roundoff
               
                  if (k == k_dbg) write(*,3) 'we avoid this case if possible', k, kk
               
                  dm = max(0d0, xm1 - xm_outer)
                  if (dm_sum + dm > new_point_dmbar) dm = new_point_dmbar - dm_sum
                  dm_sum = dm_sum + dm

               end if
               
               j_tot = j_tot + old_j_rot(kk)*dm
               
               if (.false. .or. k == k_dbg) &
                  write(*,3) 'new k use only part of old kk', &
                     k, kk, old_j_rot(kk)*dm, old_j_rot(kk), dm, j_tot/dm_sum, j_tot, dm_sum
               
               if (dm <= 0) then
                  ierr = -1
                  !return
                  write(*,*) 'dm <= 0', dm
                  stop 'debugging: set1_omega'
               end if
               
            end if
            
            if (dm_sum >= new_point_dmbar) then
               if (k == k_dbg) then
                  write(*,2) 'exit for k', k
                  write(*,2) 'dm_sum', kk, dm_sum
                  write(*,2) 'new_point_dmbar', kk, new_point_dmbar
               end if
               dm_sum = new_point_dmbar
               exit
            end if
            
         end do
         
			if (dm_sum /= new_point_dmbar) then
            write(*,2) 'dm_sum', k, dm_sum
            write(*,2) 'new_point_dmbar', k, new_point_dmbar
            stop 'debugging: set1_omega'
			end  if
			
			s% j_rot(k) = j_tot/new_point_dmbar
         s% omega(k) = s% j_rot(k)/s% i_rot(k)
         
         if (k_dbg == k) then
            write(*,2) 's% omega(k)', k, s% omega(k)
            write(*,2) 's% j_rot(k)', k, s% j_rot(k)
            write(*,2) 's% i_rot(k)', k, s% i_rot(k)
            stop 'debugging: set1_omega'
         end if
               
      end subroutine set1_omega


      subroutine set_lnd_lnT_lnPgas_for_d_dt( &
            s, nz, k_const_mass, k_below_just_added, &
            rxm_old, rxm_new, old_cell_mass, new_cell_mass, &
            delta_m, old_xmstar, new_xmstar, &
            oldloc, newloc, oldval, newval, work, ierr)
         use interp_1d_lib
         use interp_1d_def
         type (star_info), pointer :: s
         integer, intent(in) :: nz, k_const_mass, k_below_just_added
         real(dp), dimension(:), intent(in), pointer :: &
            rxm_old, rxm_new, old_cell_mass, new_cell_mass ! (nz)
         real(dp), intent(in) :: delta_m, old_xmstar, new_xmstar
         real(dp), pointer, dimension(:) :: oldloc, newloc, oldval, newval
         real(dp), pointer :: work(:)
         integer, intent(out) :: ierr         
         
         integer :: n, nwork, j
         logical :: dbg
         
         include 'formats'
         
         ierr = 0
         
         dbg = .false.
         n = k_const_mass
         nwork = pm_work_size
         
         oldloc(1) = 0
         do j=2,n
            oldloc(j) = rxm_old(j) + 0.5d0*old_cell_mass(j)
         end do
         do j=1,n
            newloc(j) = rxm_new(j) + 0.5d0*new_cell_mass(j)
         end do
         
         if (s% i_xlnd /= 0) then
            do j=1,n
               oldval(j) = s% lnd_for_d_dt(j)
            end do
            call interpolate_vector( &
               n, oldloc, n, newloc, oldval, newval, interp_pm, nwork, work, ierr)
            if (ierr /= 0) return
            do j=1,k_below_just_added-1
               s% lnd_for_d_dt(j) = 0
            end do
            do j=k_below_just_added,n
               s% lnd_for_d_dt(j) = newval(j)
            end do
         end if
         
         if (s% i_lnPgas /= 0) then
            do j=1,n
               oldval(j) = s% lnPgas_for_d_dt(j)
            end do
            call interpolate_vector( &
               n, oldloc, n, newloc, oldval, newval, interp_pm, nwork, work, ierr)
            if (ierr /= 0) return
            do j=1,k_below_just_added-1
               s% lnPgas_for_d_dt(j) = 0
            end do
            do j=k_below_just_added,n
               s% lnPgas_for_d_dt(j) = newval(j)
            end do
         end if

         do j=1,n
            oldval(j) = s% lnT_for_d_dt(j)
         end do
         call interpolate_vector( &
            n, oldloc, n, newloc, oldval, newval, interp_pm, nwork, work, ierr)
         if (ierr /= 0) return
         do j=1,k_below_just_added-1
            s% lnT_for_d_dt(j) = 0
         end do
         do j=k_below_just_added,n
            s% lnT_for_d_dt(j) = newval(j)
         end do
         
         
         return
         
         
         write(*,2) 'oldval(1)', 1, oldval(1)
         write(*,2) 's% xh(i_lnT,1)', 1, s% xh(s% i_lnT,1)
         write(*,2) 'newval(k_below_just_added)', k_below_just_added, newval(k_below_just_added)
         write(*,2) 'oldloc(1)', 1, oldloc(1)
         write(*,2) 'newloc(k_below_just_added)', k_below_just_added, newloc(k_below_just_added)
         write(*,2) 'rxm_old(1)', 1, rxm_old(1)
         write(*,2) 'rxm_new(k_below_just_added)', k_below_just_added, rxm_new(k_below_just_added)
         write(*,2) 'old_cell_mass(1)', 1, old_cell_mass(1)
         write(*,2) 'new_cell_mass(k_below_just_added)', k_below_just_added, new_cell_mass(k_below_just_added)
         write(*,2) 'delta_m', 1, delta_m
         stop 'set_lnd_lnT_lnPgas_for_d_dt'
         
      end subroutine set_lnd_lnT_lnPgas_for_d_dt
      
      
      subroutine set_lnR_v_for_d_dt( &
            s, nz, k_const_mass, k_below_just_added, &
            rxm_old, rxm_new, delta_m, old_xmstar, new_xmstar, &
            oldloc, newloc, oldval, newval, work, ierr)
         use interp_1d_lib
         use interp_1d_def
         type (star_info), pointer :: s
         integer, intent(in) :: nz, k_const_mass, k_below_just_added
         real(dp), dimension(:), intent(in), pointer :: rxm_old, rxm_new ! (nz)
         real(dp), intent(in) :: delta_m, old_xmstar, new_xmstar
         real(dp), pointer, dimension(:) :: oldloc, newloc, oldval, newval
         
         real(dp), pointer :: work(:)
         
         integer, intent(out) :: ierr         
         
         integer :: n, nwork, k
         logical :: dbg
         
         include 'formats'
         
         ierr = 0
         
         dbg = .false.
         n = k_const_mass
         nwork = pm_work_size
         
         oldloc(1) = 0
         do k=2,n
            oldloc(k) = rxm_old(k)
         end do
         do k=1,n 
            newloc(k) = rxm_new(k)
            oldval(k) = s% lnR_for_d_dt(k)
         end do
         call interpolate_vector( &
            n, oldloc, n, newloc, oldval, newval, interp_pm, nwork, work, ierr)
         if (ierr /= 0) return
         do k=1,k_below_just_added-1
            s% lnR_for_d_dt(k) = 0
         end do
         do k=k_below_just_added,n 
            s% lnR_for_d_dt(k) = newval(k)
         end do
         
         if (.not. s% v_flag) return
         
         do k=1,n 
            oldval(k) = s% v_for_d_dt(k)
         end do
         call interpolate_vector( &
            n, oldloc, n, newloc, oldval, newval, interp_pm, nwork, work, ierr)
         if (ierr /= 0) return
         do k=1,k_below_just_added-1
            s% v_for_d_dt(k) = 0
         end do
         do k=k_below_just_added,n 
            s% v_for_d_dt(k) = newval(k)
         end do

      end subroutine set_lnR_v_for_d_dt


      end module adjust_mass







         
         



