! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************
 
      module reaclib_lib
      use reaclib_def
      
      
      ! for details about sources for reaclib reaclib rates
      ! http://groups.nscl.msu.edu/reaclib/reaclib/db/labels.php
      
      implicit none

      contains
      
      
      ! call this routine to initialize the reaclib module. 
      ! only needs to be done once at start of run.
      
      subroutine reaclib_init(ierr)      
      	use reaclib_input, only: do_read_reaclib
         integer, intent(out) :: ierr ! 0 means AOK.         
         call reaclib_def_init
      	call do_read_reaclib(reaclib,nreaclib,ierr)
      end subroutine reaclib_init
      
      
      subroutine reaclib_shutdown
      end subroutine reaclib_shutdown
      
            
      subroutine extract_reaclib_rates(set,nuclides,rates,use_weaklib,ierr)
         use chem_def, only: nuclide_set, nuclide_data
      	use reaclib_input, only: do_extract_rates
      	type(nuclide_set), dimension(:), intent(in) :: set
      	type(nuclide_data), intent(in), target :: nuclides
      	logical, intent(in) :: use_weaklib
      	type(reaction_data), intent(out) :: rates
      	integer, intent(out) :: ierr
      	call do_extract_rates(set,nuclides,rates,use_weaklib,ierr)
      end subroutine extract_reaclib_rates
      

      subroutine output_reaclib_rates(unitno,rates,nuclides,format)
      	use reaclib_print
      	integer, intent(in) :: unitno
      	type(reaction_data),intent(in) :: rates
      	type(nuclide_data), intent(in) :: nuclides
      	integer, intent(in) :: format
      	integer :: err = 0
      	select case (format)
         	case(internal_format)
         		call write_reaction_data(unitno,rates,err)
         	case(pretty_print_format)
         		call pretty_print_reactions(unitno,rates,nuclides,err)
         	case(short_format)
         		call print_short_format_reactions(unitno,rates,nuclides,err)
      	end select
      end subroutine output_reaclib_rates
      
      
      subroutine reaclib_pretty_print_reaction(unitno, i, rates, nuclides, reverse, str, ierr)
      	use reaclib_print, only: do_pretty_print_reaction
      	integer, intent(in) :: unitno, i
      	type(reaction_data), intent(in) :: rates
      	type(nuclide_data), intent(in) :: nuclides
      	logical, intent(in) :: reverse
      	character (len=100), intent(inout) :: str
      	integer, intent(out) :: ierr
      	call do_pretty_print_reaction(unitno, i, rates, nuclides, reverse, str, ierr)
      end subroutine reaclib_pretty_print_reaction
      
      
      subroutine reaclib_parse_handle(handle, num_in, num_out, iso_ids, op, ierr)
      	use reaclib_support, only: do_parse_reaction_handle
      	character (len=*), intent(in) :: handle
      	integer, intent(out) :: num_in, num_out
      	integer, intent(out) :: iso_ids(:) ! holds chem_ids for input and output species
      	character (len=*), intent(out) :: op ! e.g., 'pg', 'wk', 'to', or ...
      	integer, intent(out) :: ierr
      	call do_parse_reaction_handle(handle, num_in, num_out, iso_ids, op, ierr)	
      end subroutine reaclib_parse_handle
      
      
      subroutine reaclib_create_handle(num_in, num_out, iso_ids, handle)
      	use reaclib_support, only: reaction_handle
      	integer, intent(in) :: num_in, num_out
      	integer, intent(in) :: iso_ids(:) ! holds chem_ids for input and output species
      	character (len=*), intent(out) :: handle
      	character (len=1) :: reaction_flag = '-'
      	call reaction_handle(num_in, num_out, iso_ids, reaction_flag, handle)	
      end subroutine reaclib_create_handle
      
      
      subroutine reaclib_create_ec_handle(num_in, num_out, iso_ids, handle)
      	use reaclib_support, only: reaction_handle
      	integer, intent(in) :: num_in, num_out
      	integer, intent(in) :: iso_ids(:) ! holds chem_ids for input and output species
      	character (len=*), intent(out) :: handle
      	character (len=1) :: reaction_flag = 'e'
      	call reaction_handle(num_in, num_out, iso_ids, reaction_flag, handle)	
      end subroutine reaclib_create_ec_handle
      
      
      subroutine reaclib_create_wk_handle(num_in, num_out, iso_ids, handle)
      	use reaclib_support, only: reaction_handle
      	integer, intent(in) :: num_in, num_out
      	integer, intent(in) :: iso_ids(:) ! holds chem_ids for input and output species
      	character (len=*), intent(out) :: handle
      	character (len=1) :: reaction_flag = 'w'
      	call reaction_handle(num_in, num_out, iso_ids, reaction_flag, handle)	
      end subroutine reaclib_create_wk_handle
      
      
      subroutine reaclib_create_reverse_handle(num_in, num_out, iso_ids, handle)
      	use reaclib_support, only: reverse_reaction_handle
      	integer, intent(in) :: num_in, num_out
      	integer, intent(in) :: iso_ids(:) ! holds chem_ids for input and output species
      	character (len=*), intent(out) :: handle
      	call reverse_reaction_handle(num_in, num_out, iso_ids, handle)	
      end subroutine reaclib_create_reverse_handle
      
      
      integer function reaclib_lookup(handle, rates_dict) result(indx)
         ! returns first reaction index that matches handle. 
         ! there may be several following that one having the same handle.
         ! returns 0 if handle doesn't match any of the reactions
      	use utils_lib, only: integer_dict_lookup
      	character(len=*), intent(in) :: handle ! as in rates% reaction_handle
         type (integer_dict), pointer :: rates_dict ! from create_reaclib_rates_dict
         integer :: ierr
         ierr = 0
      	call integer_dict_lookup(rates_dict, handle, indx, ierr)
      	if (ierr /= 0) indx = 0
      end function reaclib_lookup
      
      
      subroutine create_reaction_handle( &
            num_in, num_out, pspecies, nuclides, reverse, reaction_flag, handle)
      	use reaclib_support, only: get1_reaction_handle
      	integer, intent(in) :: num_in, num_out
      	integer, intent(in) :: pspecies(:)
      	type(nuclide_data), intent(in) :: nuclides
      	logical, intent(in) :: reverse
      	character (len=*), intent(in) :: reaction_flag
      	character (len=*), intent(out) :: handle
         call get1_reaction_handle( &
            num_in, num_out, pspecies, nuclides, reverse, reaction_flag, handle)
      end subroutine create_reaction_handle
      
      
      subroutine reaclib_indices_for_reaction(handle, rates, lo, hi, ierr)
      	character(len=*), intent(in) :: handle ! as in rates% reaction_handle
      	type(reaction_data), intent(in) :: rates
      	integer, intent(out) :: lo, hi
      	integer, intent(out) :: ierr
      	integer :: ir
      	include 'formats.dek'
      	ierr = 0
      	lo = 0
      	hi = 0
      	lo = reaclib_lookup(handle, rates% reaction_dict)
      	if (lo == 0) then
      	   ierr = -1
      	   return
      	end if
         hi = rates% nreactions
      	do ir = lo+1, rates% nreactions
            if (rates% reaction_handle(ir) /= handle) then
               hi = ir-1
               exit
            end if
      	end do
      	do ir = lo-1, 1, -1
            if (rates% reaction_handle(ir) /= handle) then
               lo = ir+1
               exit
            end if
      	end do
      end subroutine reaclib_indices_for_reaction
      
      
      subroutine reaclib_reaction_rates( &
            lo, hi, T9, rates, nuclides, forward_only, &
            lambda, dlambda_dlnT, &
            rlambda, drlambda_dlnT, &
            ierr)
      	use reaclib_eval, only: reaction_rates
      	integer, intent(in) :: lo, hi ! from reaclib_indices_for_reaction
      	double precision, intent(in) :: T9
      	type(reaction_data), intent(in) :: rates
      	type(nuclide_data), intent(in) :: nuclides
      	logical, intent(in) :: forward_only
      	double precision, intent(out) :: lambda, dlambda_dlnT
      	double precision, intent(out) :: rlambda, drlambda_dlnT
      	integer, intent(out) :: ierr
         call reaction_rates( &
            hi-lo+1, lo, hi, T9, rates, nuclides, forward_only, &
            lambda, dlambda_dlnT, &
            rlambda, drlambda_dlnT, &
            ierr)
      end subroutine reaclib_reaction_rates
      

      end module reaclib_lib


!  instead of the following, call reaclib_indices_for_reaction and then reaclib_reaction_rates


!      subroutine reaclib_lambda(T9,rates,ln_lambda,lambda,dlambda_dlnT)
!      	use reaclib_eval, only: compute_lambda
!      	double precision, intent(in) :: T9
!      	type(reaction_data), intent(in) :: rates
!      	double precision, dimension(:), intent(out) :: ln_lambda, lambda, dlambda_dlnT
!         call compute_lambda(T9,rates,ln_lambda,lambda,dlambda_dlnT)
!      end subroutine reaclib_lambda
!      
!      
!      subroutine reaclib_inverse_lambda(T9,rates, &
!            ln_lambda,lambda,dlambda_dlnT,inv_lambda,dinv_lambda_dlnT)
!      	use reaclib_eval, only: compute_inverse_lambda
!      	double precision, intent(in) :: T9
!      	type(reaction_data), intent(in) :: rates
!      	double precision, dimension(:), intent(in) :: ln_lambda, lambda, dlambda_dlnT
!      	double precision, dimension(:), intent(out) :: inv_lambda, dinv_lambda_dlnT
!         call compute_inverse_lambda(T9,rates, &
!            ln_lambda,lambda,dlambda_dlnT,inv_lambda,dinv_lambda_dlnT)
!      end subroutine reaclib_inverse_lambda
!
!
!      subroutine reaclib_get_some_lambdas(lo, hi, T9, rates, ln_lambda, lambda, dlambda_dlnT)
!      	use reaclib_eval, only: compute_some_lambdas
!         integer, intent(in) :: lo, hi ! range of rates to do
!      	double precision, intent(in) :: T9
!      	type(reaction_data), intent(in) :: rates
!      	double precision, dimension(:), intent(out) :: ln_lambda, lambda, dlambda_dlnT
!         call compute_some_lambdas(lo, hi, T9, rates, ln_lambda, lambda, dlambda_dlnT)
!      end subroutine reaclib_get_some_lambdas
!      
!      
!      subroutine reaclib_get_some_inverse_lambdas(lo, hi, T9, rates, &
!            ln_lambda, lambda, dlambda_dlnT, inv_lambda, dinv_lambda_dlnT)
!      	use reaclib_eval, only: compute_some_inverse_lambdas
!         integer, intent(in) :: lo, hi ! range of rates to do
!      	double precision, intent(in) :: T9
!      	type(reaction_data), intent(in) :: rates
!      	double precision, dimension(:), intent(in) :: ln_lambda, lambda, dlambda_dlnT
!      	double precision, dimension(:), intent(out) :: inv_lambda, dinv_lambda_dlnT
!         call compute_some_inverse_lambdas(lo, hi, T9, rates, &
!            ln_lambda, lambda, dlambda_dlnT, inv_lambda, dinv_lambda_dlnT)
!      end subroutine reaclib_get_some_inverse_lambdas
