! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module rates_lib
      
      use const_def, only: dp
      
      implicit none


      contains
      
      
      ! call this routine to initialize the rates module. 
      ! only needs to be done once at start of run.
            
      subroutine rates_init(reactionlist_filename, rates_cache_dir, ierr)      
         use rates_def, only : start_rates_def_init, rate_table_info, raw_rates_records,
     >      rates_reaction_id_max, num_predefined_reactions, reaction_Name
         use rates_names, only: set_reaction_names
         use rates_initialize, only: do_rates_init, read_reaction_parameters, finish_rates_def_init
         character (len=*), intent(in) :: reactionlist_filename
         character (len=*), intent(in) :: rates_cache_dir ! '' means use default
         integer, intent(out) :: ierr ! 0 means AOK.  
         type (rate_table_info), pointer :: ri
         integer :: i
         ierr = 0
         rates_reaction_id_max = num_predefined_reactions
         allocate(raw_rates_records(rates_reaction_id_max), reaction_Name(rates_reaction_id_max))
         do i = 1, rates_reaction_id_max
            ri => raw_rates_records(i)
            ri% nT8s = 0
            ri% use_rate_table = .false.
            ri% need_to_read = .false.
            nullify(ri% T8s)
            nullify(ri% f1)
         end do
         call set_reaction_names
         call start_rates_def_init(rates_cache_dir)
         call read_reaction_parameters(reactionlist_filename, ierr)
         if (ierr /= 0) then
            write(*,*) 'rates_init failed in read_reaction_parameters'
            return
         end if
         call finish_rates_def_init
         call do_rates_init(ierr)
         if (ierr /= 0) then
            write(*,*) 'rates_init failed in do_rates_init'
            return
         end if
      end subroutine rates_init
      
      
      subroutine rates_shutdown
         use rates_initialize, only: free_raw_rates_records
         call free_raw_rates_records
      end subroutine rates_shutdown
      
      
      subroutine read_raw_rates_records(rate_tables_dir, ierr)
         use rates_initialize, only: init_raw_rates_records, set_rate_ids
         character (len=*), intent(in) :: rate_tables_dir ! try local first, then try data/rates_data
         integer, intent(out) :: ierr ! 0 means AOK.  
         ierr = 0
         call init_raw_rates_records(rate_tables_dir, ierr)
         if (ierr /= 0) then
            write(*,*) 'rates_init failed in init_raw_rates_records'
            return
         end if
         call set_rate_ids
      end subroutine read_raw_rates_records
      
      
      subroutine eval_tfactors(tf, logT, temp)
         use rates_def, only : T_Factors
         use ratelib, only: tfactors
         type (T_Factors), pointer :: tf ! allocate this before calling
         real(dp), intent(in) :: logT, temp
         call tfactors(tf, logT, temp)
      end subroutine eval_tfactors
      

      subroutine get_raw_rate(ir, which_rate, temp, tf, raw_rate, ierr)
         use rates_def, only : T_Factors
         use raw_rates
         integer, intent(in) :: ir, which_rate
         real(dp), intent(in) :: temp
         type (T_Factors), pointer :: tf
         real(dp), intent(out) :: raw_rate
         integer, intent(out) :: ierr
         call set_raw_rate(ir, which_rate, temp, tf, raw_rate, ierr)
      end subroutine get_raw_rate


      subroutine get_raw_rates(n, irs, which_rates, temp, tf, rates, ierr)
         use rates_def, only : T_Factors
         use raw_rates, only: set_raw_rates
         integer, intent(in) :: n
         integer, intent(in) :: irs(:) ! (n) maps 1..n to reaction id
         integer, intent(in) :: which_rates(:) ! (rates_reaction_id_max)
         real(dp), intent(in) :: temp
         type (T_Factors), pointer :: tf
         real(dp), intent(out) :: rates(:)
         integer, intent(out) :: ierr
         call set_raw_rates(n, irs, which_rates, temp, tf, rates, ierr)
      end subroutine get_raw_rates


      subroutine eval_mazurek_rate(btemp,bden,y56,ye,rn56ec,sn56ec)       
         use ratelib, only: mazurek
         real(dp), intent(in) :: btemp,bden,y56,ye
         real(dp), intent(out) :: rn56ec,sn56ec
         call mazurek(btemp,bden,y56,ye,rn56ec,sn56ec)
      end subroutine eval_mazurek_rate
      

      subroutine eval_G05_epsnuc_CC(T, Rho, X12, eps, deps_dT, deps_dRho)
         
         ! from Gasques, et al, Nuclear fusion in dense matter.
         ! Phys Review C, 72, 025806 (2005)
         
         use pycno, only: G05_epsnuc_CC
         real(dp), intent(in) :: T
         real(dp), intent(in) :: Rho
         real(dp), intent(in) :: X12 ! mass fraction of c12
         real(dp), intent(out) :: eps ! rate in ergs/g/sec
         real(dp), intent(out) :: deps_dT ! partial wrt temperature
         real(dp), intent(out) :: deps_dRho ! partial wrt density
         
         call G05_epsnuc_CC(T, Rho, X12, eps, deps_dT, deps_dRho)
         
      end subroutine eval_G05_epsnuc_CC


      subroutine eval_FL_epsnuc_3alf(T, Rho, Y, UE, eps_nuc, deps_nuc_dT, deps_nuc_dRho)       
         ! based on analytic expressions in Fushiki and Lamb, Apj, 317, 368-388, 1987.
         
         ! Note: if you plot the results of this, you'll see abrupt changes in rate at
         ! logRho about 9.74 and 10.25 -- these aren't bugs in the code. 
         ! They are discussed in F&L, and show up as step functions in their expressions.
         
         ! They provide expressions for both pyconuclear regime and strong screening regime.
         ! The transition between the regimes happens at U = 1, where U is defined below.
         ! Unfortunately, at U = 1, their expressions for pycnonuclear rate and strong screening rate disagree!
         ! Bummer.  For example, at logRho = 8.0, U = 1 for logT = 7.1955.  For these values, and pure He,
         ! their strong screening expression is larger than their pycno expression by a factor of about 25.
         
         ! need to add transition region in U instead of having an abrupt change at U = 1
         
         use pycno, only: FL_epsnuc_3alf
         real(dp), intent(in) :: T ! temperature
         real(dp), intent(in) :: Rho ! density
         real(dp), intent(in) :: Y ! helium mass fraction
         real(dp), intent(in) :: UE ! electron molecular weight
         real(dp), intent(out) :: eps_nuc ! eps_nuc in ergs/g/sec
         real(dp), intent(out) :: deps_nuc_dT ! partial wrt temperature
         real(dp), intent(out) :: deps_nuc_dRho ! partial wrt density
         call FL_epsnuc_3alf(T, Rho, Y, UE, eps_nuc, deps_nuc_dT, deps_nuc_dRho)
      end subroutine eval_FL_epsnuc_3alf


      subroutine eval_ecapnuc_rate(etakep,temp,rpen,rnep,spen,snep)
         use ratelib, only: ecapnuc
         real(dp), intent(in) :: etakep,temp
         real(dp), intent(out) :: rpen,rnep,spen,snep
         !  given the electron degeneracy parameter etakep (chemical potential
         !  without the electron's rest mass divided by kt) and the temperature temp,
         !  this routine calculates rates for 
         !  electron capture on protons rpen (captures/sec/proton),
         !  positron capture on neutrons rnep (captures/sec/neutron), 
         !  and their associated neutrino energy loss rates 
         !  spen (ergs/sec/proton) and snep (ergs/sec/neutron)
         call ecapnuc(etakep,temp,rpen,rnep,spen,snep)
      end subroutine eval_ecapnuc_rate
      
      
      subroutine eval_n14_electron_capture_rate(T,Rho,UE,rate)
         use ratelib, only: n14_electron_capture_rate
         real(dp), intent(in) :: T ! temperature
         real(dp), intent(in) :: Rho ! density
         real(dp), intent(in) :: UE ! electron molecular weight
         real(dp), intent(out) :: rate ! (s^-1)
         call n14_electron_capture_rate(T,Rho,UE,rate)
      end subroutine eval_n14_electron_capture_rate
      
      
		integer function rates_reaction_id(rname)
         use rates_def, only: get_rates_reaction_id
			character (len=*), intent(in)  :: rname ! reaction name such as 'rpp' 
			! returns id for the reaction if there is a matching entry in reaction_Name
			! returns 0 otherwise.
			rates_reaction_id = get_rates_reaction_id(rname)
		end function rates_reaction_id
		
		
		subroutine set_which_rate_c12ag(which_rates, choice)
		   use rates_def
		   integer, intent(inout) :: which_rates(:)
		   integer, intent(in) :: choice
		   which_rates(ir_c12_ag_o16) = choice
		   which_rates(ir_o16_ga_c12) = choice
		end subroutine set_which_rate_c12ag
		
		
		subroutine set_which_rate_n14pg(which_rates, choice)
		   use rates_def
		   integer, intent(inout) :: which_rates(:)
		   integer, intent(in) :: choice
		   which_rates(ir_n14_pg_o15) = choice
		   which_rates(irn14pg_aux) = choice
		   which_rates(irn14_to_n15) = choice
		   which_rates(irn14_to_o16) = choice
		   which_rates(irn14_to_c12) = choice
		   which_rates(ir_o15_gp_n14) = choice
		end subroutine set_which_rate_n14pg
		
		
		subroutine set_which_rate_3a(which_rates, choice)
		   use rates_def
		   integer, intent(inout) :: which_rates(:)
		   integer, intent(in) :: choice
		   which_rates(ir_he4_he4_he4_to_c12) = choice
		   which_rates(ir_c12_to_he4_he4_he4) = choice
		end subroutine set_which_rate_3a
		
		
		subroutine set_which_rate_1212(which_rates, choice)
		   use rates_def
		   integer, intent(inout) :: which_rates(:)
		   integer, intent(in) :: choice
		   which_rates(ir1212) = choice
		end subroutine set_which_rate_1212
		
		
		integer function eval_num_reaction_inputs(ir)
		   use rates_def, only: get_num_reaction_inputs
		   integer, intent(in) :: ir
		   eval_num_reaction_inputs = get_num_reaction_inputs(ir)
		end function eval_num_reaction_inputs
		
		
		integer function eval_num_reaction_outputs(ir)
		   use rates_def, only: get_num_reaction_outputs
		   integer, intent(in) :: ir
		   eval_num_reaction_outputs = get_num_reaction_outputs(ir)
		end function eval_num_reaction_outputs
         
         
      subroutine add_reaction_from_reaclib(reaction_handle, reverse_handle, indx, ierr)
         use rates_initialize, only: do_add_reaction_from_reaclib
         character (len=*), intent(in) :: reaction_handle ! to be added
         character (len=*), intent(in) :: reverse_handle ! = '' if not a reverse
         integer, intent(in) :: indx ! index in reaclib rates
         integer, intent(out) :: ierr
         call do_add_reaction_from_reaclib(reaction_handle, reverse_handle, indx, ierr)
      end subroutine add_reaction_from_reaclib
         
         
      subroutine add_reaclib_reaction(handle, ierr)
         use rates_initialize, only: do_add_reaction_from_reaclib
         use rates_def
         character (len=*), intent(in) :: handle ! to be added
         integer, intent(out) :: ierr
         integer :: indx
         ierr = 0
         indx = reaclib_index(handle)
         if (indx /= 0) then
            call do_add_reaction_from_reaclib(handle, '', indx, ierr)
         else
            indx = reaclib_reverse(handle)
            if (indx /= 0) then
               call do_add_reaction_from_reaclib(
     >                  handle, reaclib_rates% reaction_handle(indx), indx, ierr)
            else
               ierr = -1
            end if
         end if
      end subroutine add_reaclib_reaction
         
         
      subroutine add_reaction_for_handle(handle, ierr)
         use rates_initialize, only: do_add_reaction_for_handle
         character (len=*), intent(in) :: handle ! to be added
         integer, intent(out) :: ierr
         call do_add_reaction_for_handle(handle, ierr)
      end subroutine add_reaction_for_handle


      subroutine make_rate_tables(
     >      num_reactions, cache_suffix, net_reaction_id, which_rates, 
     >      rattab, rattab_f1, nT8s, ttab, logttab, ierr)  
         use rates_support, only : do_make_rate_tables
         integer, intent(in) :: num_reactions, nT8s, net_reaction_id(:), which_rates(:)
         character (len=*), intent(in) :: cache_suffix
         real(dp) :: rattab(:,:), ttab(:), logttab(:)
         real(dp), pointer :: rattab_f1(:)
         integer, intent(out) :: ierr
         call do_make_rate_tables(
     >         num_reactions, cache_suffix, net_reaction_id, 
     >         which_rates, rattab, rattab_f1, nT8s, ttab, logttab, ierr)
      end subroutine make_rate_tables
      
         
      subroutine eval_using_rate_tables(
     >      num_reactions, reaction_id, rattab, rattab_f1, nT8s,
     >      ye, logtemp, btemp, bden, raw_rate_factor, logttab, rate_raw)
         use rates_support, only : do_get_raw_rates
         integer, intent(in) :: num_reactions, reaction_id(:), nT8s
         real(dp), intent(in) :: 
     >      ye, logtemp, btemp, bden, raw_rate_factor(:), 
     >      rattab(:,:), logttab(:)
         real(dp), pointer :: rattab_f1(:)
         real(dp), intent(out) :: rate_raw(:,:)
         call do_get_raw_rates(num_reactions, reaction_id, rattab, rattab_f1, nT8s,
     >         ye, logtemp, btemp, bden, raw_rate_factor, logttab, rate_raw)
      end subroutine eval_using_rate_tables
      
      
      subroutine show_reaction_rates_from_cache(cache_filename, ierr) 
         use rates_support, only: do_show_reaction_from_cache
         character (len=*) :: cache_filename
         integer, intent(out) :: ierr
         call do_show_reaction_from_cache(cache_filename, ierr) 
      end subroutine show_reaction_rates_from_cache

      
      subroutine rates_eval_reaclib_21(
     >      ir, temp, den, rate_raw, reverse_rate_raw, ierr)
         use rates_support, only: do_eval_reaclib_21
         integer, intent(in) :: ir ! reaction_id
         real(dp), intent(in) :: temp, den
         real(dp), intent(out) :: rate_raw(:), reverse_rate_raw(:)
         integer, intent(out) :: ierr
         call do_eval_reaclib_21(
     >      ir, temp, den, rate_raw, reverse_rate_raw, ierr)
      end subroutine rates_eval_reaclib_21

      
      subroutine rates_eval_reaclib_22(
     >      ir, temp, den, rate_raw, reverse_rate_raw, ierr)
         use rates_support, only: do_eval_reaclib_22
         integer, intent(in) :: ir ! reaction_id
         real(dp), intent(in) :: temp, den
         real(dp), intent(out) :: rate_raw(:), reverse_rate_raw(:)
         integer, intent(out) :: ierr
         call do_eval_reaclib_22(
     >      ir, temp, den, rate_raw, reverse_rate_raw, ierr)
      end subroutine rates_eval_reaclib_22
      
      
      subroutine rates_two_to_one_coeffs_for_reverse_factor(
     >      Q, iso_A, iso_B, iso_C, a, b, ierr)
         use chem_def, only: chem_isos
         real(dp), intent(in) :: Q
         integer, intent(in) :: iso_A, iso_B, iso_C
         real(dp), intent(out) :: a, b
         integer, intent(out) :: ierr
         real(dp) :: W_A, W_B, W_C, g_A, g_B, g_C         
         if (Q < 0) then
            ierr = -1
            return
         end if
         ierr = 0         
         W_A = chem_isos% W(iso_A)
         W_B = chem_isos% W(iso_B)
         W_C = chem_isos% W(iso_C)
         g_A = 2d0*chem_isos% spin(iso_A) + 1d0
         g_B = 2d0*chem_isos% spin(iso_B) + 1d0
         g_C = 2d0*chem_isos% spin(iso_C) + 1d0         
         ! Arnett, Supernovae and Nucleosynthesis, eqn 3.136
         a = 9.8678d9*(g_A*g_B/g_C)*(W_A*W_B/W_C)**1.5d0  
         b = -11.605d0*Q         
      end subroutine rates_two_to_one_coeffs_for_reverse_factor
         
      
      ! note: assumes ground state spins and requires Q > 0.
      ! i.e., A + B -> C exothermic
      subroutine rates_two_to_one_reverse_factor(
     >      Q, T9, T932, iso_A, iso_B, iso_C, rev, d_rev_dT, ierr) ! A + B <-> C
         use chem_def, only: chem_isos
         real(dp), intent(in) :: Q, T9, T932
         integer, intent(in) :: iso_A, iso_B, iso_C
         real(dp), intent(out) :: rev, d_rev_dT
         integer, intent(out) :: ierr         
         real(dp) :: a, b      
         call rates_two_to_one_coeffs_for_reverse_factor(
     >      Q, iso_A, iso_B, iso_C, a, b, ierr)
         if (ierr /= 0) return
         rev = a*T932*exp(b/T9)
         d_rev_dT = rev*(1.5d0*T9 - b)/(T9*T9*1d9)         
      end subroutine rates_two_to_one_reverse_factor


      subroutine rates_two_to_two_coeffs_for_reverse_factor(
     >      Q, iso_A, iso_B, iso_C, iso_D, a, b, ierr)
         use chem_def, only: chem_isos
         real(dp), intent(in) :: Q
         integer, intent(in) :: iso_A, iso_B, iso_C, iso_D
         real(dp), intent(out) :: a, b
         integer, intent(out) :: ierr
         real(dp) :: W_A, W_B, W_C, W_D, g_A, g_B, g_C, g_D         
         if (Q < 0) then
            ierr = -1
            return
         end if
         ierr = 0         
         W_A = chem_isos% W(iso_A)
         W_B = chem_isos% W(iso_B)
         W_C = chem_isos% W(iso_C)
         W_D = chem_isos% W(iso_D)
         g_A = 2d0*chem_isos% spin(iso_A) + 1d0
         g_B = 2d0*chem_isos% spin(iso_B) + 1d0
         g_C = 2d0*chem_isos% spin(iso_C) + 1d0
         g_D = 2d0*chem_isos% spin(iso_D) + 0.5d0         
         ! Arnett, Supernovae and Nucleosynthesis, eqn 3.137
         a = (g_A*g_B/(g_C*g_D))*((W_A*W_B)/(W_C*W_D))**1.5d0 
         b = -11.605d0*Q         
      end subroutine rates_two_to_two_coeffs_for_reverse_factor
      
      
      ! note: assumes ground state spins and requires Q > 0.
      ! i.e., A + B -> C + D exothermic
      subroutine rates_two_to_two_reverse_factor(
     >      Q, T9, iso_A, iso_B, iso_C, iso_D, rev, d_rev_dT, ierr) ! A + B <-> C + D
         use chem_def, only: chem_isos
         real(dp), intent(in) :: Q, T9
         integer, intent(in) :: iso_A, iso_B, iso_C, iso_D
         real(dp), intent(out) :: rev, d_rev_dT
         integer, intent(out) :: ierr         
         real(dp) :: a, b
         call rates_two_to_two_coeffs_for_reverse_factor(
     >      Q, iso_A, iso_B, iso_C, iso_D, a, b, ierr)
         if (ierr /= 0) return
         rev = a*exp(b/T9)
         d_rev_dT = -rev*b/(T9*T9*1d9)
      end subroutine rates_two_to_two_reverse_factor
            

      end module rates_lib

