! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License, 
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful, 
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module raw_rates
      
      use rates_def
      use const_def !, only: missing_value, dp
      
      implicit none
      
      
      logical, parameter :: show_rates = .false.

      
      contains
      
      
      subroutine set_raw_rates(n, irs, which_rates, temp, tf, rates, ierr)
         use rates_def, only : T_Factors
         integer, intent(in) :: n
         integer, intent(in) :: irs(:) ! (n) maps 1..n to reaction id
         integer, intent(in) :: which_rates(:) ! (rates_reaction_id_max)
         real(dp), intent(in) :: temp
         type (T_Factors), pointer :: tf
         real(dp), intent(out) :: rates(:)
         integer, intent(out) :: ierr
         integer :: i, ir, op_err
         include 'formats.dek'
         ierr = 0
!x$OMP PARALLEL DO PRIVATE(i,ir,op_err)
         do i=1,n
            ir = irs(i)
            if (ir <= 0) cycle
            op_err = 0
            call set_raw_rate(ir, which_rates(ir), temp, tf, rates(i), op_err)
            if (op_err /= 0) then
               ierr = op_err
               cycle
            end if
         end do
!x$OMP END PARALLEL DO
      end subroutine set_raw_rates
      

      subroutine set_raw_rate(ir, which_rate, temp, tf, raw_rate, ierr)
         use ratelib
         integer, intent(in) :: ir, which_rate
         real(dp), intent(in) :: temp
         type (T_Factors), pointer :: tf
         real(dp), intent(out) :: raw_rate
         integer, intent(out) :: ierr

         real(dp) :: rr

         include 'formats.dek'
         
         ierr = 0
         
         select case(ir)

            case(ir_h1_h1_wk_h2) ! p(p, e+nu)h2
               call do1_of_3(rate_pp_nacre, rate_pp_jina, rate_pp_fxt)

            case(irpp_to_he3) ! p(p, e+nu)h2(p, g)he3
               call do1_of_3(rate_pp_nacre, rate_pp_jina, rate_pp_fxt)

            case(ir_h1_h1_ec_h2)        ! p(e-p, nu)h2
               call do1_of_3(rate_pep_fxt, rate_pep_jina, rate_pep_fxt)

            case(irpep_to_he3)        ! p(e-p, nu)h2(p, g)he3
               call do1_of_3(rate_pep_fxt, rate_pep_jina, rate_pep_fxt)

            case(ir_h2_h2_to_he4)        ! h2(h2, g)he4
               call do1_of_2(rate_ddg_reaclib, rate_ddg_jina)

            case(ir_h1_he3_wk_he4)  ! he3(p, e+nu)he4
               call do1_of_3(rate_hep_fxt, rate_hep_jina, rate_hep_fxt)

            case(ir_h2_he3_to_h1_he4)  ! he3(d, p)he4
               call do1_of_2(rate_he3d_reaclib, rate_he3d_jina)

            case(ir_he3_he3_to_h1_h1_he4) ! he3(he3, 2p)he4
               ! no he3+he3 in jina reaclib
               call do1_of_3(rate_he3he3_nacre, rate_he3he3_nacre, rate_he3he3_fxt)

            case(ir_h1_h1_he4_to_he3_he3) ! he4(2p, he3)he3
               call do1_of_3_reverse(rate_he3he3_nacre, rate_he3he3_nacre, rate_he3he3_fxt)

            case(ir_he3_ag_be7) ! he3(he4, g)be7
               call do1_of_3(rate_he3he4_nacre, rate_he3he4_jina, rate_he3he4_fxt)

            case(ir34_pp2) ! he4(he3, g)be7(e-, nu)li7(p, a)he4
               call do1_of_3(rate_he3he4_nacre, rate_he3he4_jina, rate_he3he4_fxt)

            case(ir34_pp3) ! he4(he3, g)be7(p, g)b8(e+, nu)be8(, a)he4
               call do1_of_3(rate_he3he4_nacre, rate_he3he4_jina, rate_he3he4_fxt)
      
      
      
            case(ir_he4_he4_he4_to_c12) ! triple alpha to c12 
               call do1_of_3(
     >            rate_tripalf_nacre, rate_tripalf_jina, rate_tripalf_fxt)
      
            case(ir_c12_to_he4_he4_he4) ! c12 to 3 alpha
               call do1_of_3_reverse(
     >            rate_tripalf_nacre, rate_tripalf_jina, rate_tripalf_fxt)



            case(ir_li7_pa_he4) ! li7(p, a)he4
               call do1_of_3(rate_li7pa_nacre, rate_li7pa_jina, rate_li7pa_fxt)

            case(ir_be7_ec_li7)      ! be7(e-, nu)li7
               call do1_of_2(rate_be7em_fxt, rate_be7em_jina)

            case(irbe7ec_li7_aux)      ! be7(e-, nu)li7(p, a)he4 
               call do1_of_2(rate_be7em_fxt, rate_be7em_jina)


            case(ir_h2_be7_to_h1_he4_he4)      ! be7(d, p)2he4
               call do1_of_2(rate_be7dp_reaclib, rate_be7dp_jina)

            case(ir_he3_be7_to_h1_h1_he4_he4)      ! be7(he3, 2p)2he4
               call do1_of_2(rate_be7he3_reaclib, rate_be7he3_jina)

            case(irbe7pg_b8_aux) ! be7(p, g)b8(e+, nu)be8(, a)he4
               call do1_of_3(rate_be7pg_nacre, rate_be7pg_jina, rate_be7pg_fxt)

            case(ir_b8_wk_he4_he4) ! b8(p=>n)be8=>2 he4
               call do1_of_2(rate_b8ep, rate_b8_wk_he4_he4_jina)               
               

            case(ir_b8_gp_be7) ! be7(p, g)b8
                ! no jina rate
               call do1_of_3_reverse(rate_be7pg_nacre, rate_be7pg_nacre, rate_be7pg_fxt)

            case(irc12_to_c13) ! c12(p, g)n13(e+nu)c13
               call do1_of_3(rate_c12pg_nacre, rate_c12pg_jina, rate_c12pg_fxt)

            case(irc12_to_n14) ! c12(p, g)n13(e+nu)c13(p, g)n14
               call do1_of_3(rate_c12pg_nacre, rate_c12pg_jina, rate_c12pg_fxt)
                       
            case(ir_c12_ap_n15) ! c12(a, p)n15
               call do1_of_3_reverse(rate_n15pa_nacre, rate_n15pa_jina, rate_n15pa_fxt)
                       
            case(irc12ap_to_o16) ! c12(a, p)n15(p, g)o16
               call do1_of_3_reverse(rate_n15pa_nacre, rate_n15pa_jina, rate_n15pa_fxt)
                       
            case(irc12ng_to_o16) ! c12(n, g)c13(a, n)o16
               call do1_of_2(rate_c12ng_reaclib, rate_c12ng_jina)

            case(ir_c12_ag_o16)
               call do1_of_4(
     >            rate_c12ag_nacre, rate_c12ag_jina, rate_c12ag_kunz, rate_c12ag_fxt)
      
            case(ir1212a) ! c12(c12,a)ne20
               call do1_of_2(rate_c12c12a_fxt, rate_c12c12a_jina)
      
            case(ir1212p) ! c12(c12, p)na23
               call do1_of_2(rate_c12c12p_fxt, rate_c12c12p_jina)
      
            case(ir1212) ! c12(c12,n)mg23, c12(c12,p)na23, c12(c12,a)ne20
               call do1_of_2(rate_c12c12_fxt_multi, rate_c12c12_fxt_basic)
               ! NOTE: Gasques option for c12+c12 is implemented in net, not in rates.
      
            case(ir1216_to_mg24) ! ! c12 + o16 -> mg24 + he4
               call do1_of_2(rate_c12o16_to_mg24_fxt, rate_c12o16a_jina)
      
            case(ir1216_to_si28) ! ! c12 + o16 -> si28
               call do1_of_2(rate_c12o16_to_si28_fxt, rate_c12o16p_jina)
                
            case(irc12ap_aux)  ! c12(a, p)n15
               call do1_of_2_reverse(rate_n15pa_nacre, rate_n15pa_jina)
            
            case(ir_c13_an_o16) ! c13(a, n)o16
               call do1_of_2(rate_c13an_fxt, rate_c13an_jina)
            
            case(ir_n13_ap_o16) ! n13(a, p)o16
               call do1_of_2(rate_n13ap_reaclib, rate_n13ap_jina)
    
            case(ir_n13_gp_c12) ! n13(g, p)c12
               call do1_of_3_reverse(rate_c12pg_nacre, rate_c12pg_jina, rate_c12pg_fxt)

            case(irn14pg_aux) ! n14(p, g)o15
               call do1_of_3(rate_n14pg_nacre, rate_n14pg_jina, rate_n14pg_fxt)

            case(irn14_to_n15) ! n14(p, g)o15(e+nu)n15
               call do1_of_3(rate_n14pg_nacre, rate_n14pg_jina, rate_n14pg_fxt)

            case(irn14_to_o16) ! n14(p, g)o15(e+nu)n15(p, g)o16
               call do1_of_3(rate_n14pg_nacre, rate_n14pg_jina, rate_n14pg_fxt)

            case(irn14_to_c12) ! n14(p, g)o15(e+nu)n15(p, a)c12
               call do1_of_3(rate_n14pg_nacre, rate_n14pg_jina, rate_n14pg_fxt)
            
            case(ir_n14_ap_o17) ! n14(a, p)o17
               call do1_of_2_reverse(rate_o17pa_nacre, rate_o17pa_jina)
               
            case(irn14ecap) ! n14(e-, nu)c14
               raw_rate = -1 ! rate calculated by special routine.

            case(ir_n14_ag_f18) ! n14(a, g)f18
               call do1_of_3(rate_n14ag_nacre, rate_n14ag_jina, rate_n14ag_fxt)
            
            case(irn14ag_to_o18) ! n14(a, g)f18(e+nu)o18
               call do1_of_3(rate_n14ag_nacre, rate_n14ag_jina, rate_n14ag_fxt)
            
            case(irn14ag_to_ne22) ! n14(a, g)f18(e+nu)o18(a, g)ne22
               call do1_of_3(rate_n14ag_nacre, rate_n14ag_jina, rate_n14ag_fxt)
            
            case(irn14ag_to_ne20_2n) ! n14(a,g)f18(e+nu)o18(a,g)ne22(g,n)ne21(g,n)ne20
               call do1_of_3(rate_n14ag_nacre, rate_n14ag_jina, rate_n14ag_fxt)
            
            case(irn14ag_lite) ! n14 + 1.5 alpha => ne20
               call do1_of_3(rate_n14ag_nacre, rate_n14ag_jina, rate_n14ag_fxt)
              
            case(ir_n14_gp_c13) ! n14(g, p)c13
               call do1_of_2_reverse(rate_c13pg_nacre, rate_c13pg_jina)

            case(irn15pg_aux) ! n15(p, g)o16
               call do1_of_2(rate_n15pg_nacre, rate_n15pg_jina)

            case(ir_n15_pa_c12) ! n15(p, a)c12
               call do1_of_3(rate_n15pa_nacre, rate_n15pa_jina, rate_n15pa_fxt)

            case(irn15pa_aux) ! n15(p, a)c12
               call do1_of_3(rate_n15pa_nacre, rate_n15pa_jina, rate_n15pa_fxt)
            
            case(ir_n15_ap_o18) ! n15(a, p)o18
               call do1_of_2_reverse(rate_o18pa_nacre, rate_o18pa_jina)
            
            case(ir_o14_ag_ne18) ! o14(a, g)ne18
               call do1_of_2(rate_o14ag_reaclib, rate_o14ag_jina)
            
            case(ir_o14_ap_f17) ! o14(a, p)f17
               call do1_of_2(rate_o14ap_fxt, rate_o14ap_jina)
            
            case(ir_o14_gp_n13) ! o14(g, p)n13
               call do1_of_2_reverse(rate_n13pg_nacre, rate_n13pg_jina)
            
            case(ir_o15_ap_f18) ! o15(a, p)f18
               call do1_of_2_reverse(rate_f18pa_wk82, rate_f18pa_jina)
            
            case(ir_o15_ag_ne19) ! o15(a, g)ne19 
               call do1_of_2(rate_o15ag_fxt, rate_o15ag_jina)
                
            case(ir_o15_gp_n14)  ! o15(g, p)n14
               call do1_of_3_reverse(rate_n14pg_nacre, rate_n14pg_jina, rate_n14pg_fxt)
            
            case(iro16_to_o17) ! o16(p, g)f17(e+nu)o17
               call do1_of_3(rate_o16pg_nacre, rate_o16pg_jina, rate_o16pg_fxt)
            
            case(iro16_to_n14) ! o16(p, g)f17(e+nu)o17(p, a)n14
               call do1_of_3(rate_o16pg_nacre, rate_o16pg_jina, rate_o16pg_fxt)
                
            case(ir_o16_ap_f19)  ! o16(a, p)f19
               call do1_of_2_reverse(rate_f19pa_nacre, rate_f19pa_jina)
                
            case(iro16ap_aux)  ! o16(a, p)f19
               call do1_of_2_reverse(rate_f19pa_nacre, rate_f19pa_jina)
                
            case(iro16ap_to_ne20)  ! o16(a, p)f19(p, a)ne20
               call do1_of_2_reverse(rate_f19pa_nacre, rate_f19pa_jina)

            case(ir_o16_ag_ne20) ! o16(a, g)ne20
               call do1_of_3(rate_o16ag_nacre, rate_o16ag_jina, rate_o16ag_fxt)
             
            case(ir_o16_gp_n15)  ! o16(g, p)n15
               call do1_of_2_reverse(rate_n15pg_nacre, rate_n15pg_jina)
             
            case(iro16gp_aux)  ! o16(g, p)n15 
               call do1_of_2_reverse(rate_n15pg_nacre, rate_n15pg_jina)
             
            case(iro16gp_to_c12)  ! o16(g, p)n15(p, a)c12
               call do1_of_2_reverse(rate_n15pg_nacre, rate_n15pg_jina)
            
            case(ir_o16_ga_c12) ! o16(g, a)c12
               call do1_of_4_reverse(
     >            rate_c12ag_nacre, rate_c12ag_jina, rate_c12ag_kunz, rate_c12ag_fxt)

            case(iro16ng_to_mg24) ! o16(n, g)o17(a, g)ne21(a, n)mg24
               call do1(rate_o16ng_jina)
      
            case(ir1616) ! o16 + o16 -> si28 + he4
               call do1_of_2(rate_o16o16_fxt, rate_o16o16_jina)

            case(ir1616a) ! o16(o16, a)si28 
               call do1_of_2(rate_o16o16a_fxt, rate_o16o16a_jina)

            case(ir1616ppa) ! o16(o16, p)p31(p, a)si28 
               call do1_of_2(rate_o16o16p_reaclib, rate_o16o16p_jina)

            case(ir1616ppg) ! o16(o16, p)p31(p, g)s32 
               call do1_of_2(rate_o16o16p_reaclib, rate_o16o16p_jina)

            case(ir1616p) ! o16(o16, p)p31 
               call do1_of_2(rate_o16o16p_reaclib, rate_o16o16p_jina)

            case(ir1616p_aux) ! o16(o16, p)p31
               call do1_of_2(rate_o16o16p_fxt, rate_o16o16p_jina)

            case(ir1616g) ! o16(o16, g)s32 
               ! no jina rate
               call do1(rate_o16o16g_fxt)

            case(ir_o17_pa_n14) ! o17(p, a)n14
               call do1_of_2(rate_o17pa_nacre, rate_o17pa_jina)

            case(ir_o17_ag_ne21) ! o17(a, g)ne21
               call do1_of_2(rate_o17ag_reaclib, rate_o17ag_jina)

            case(iro17_to_o18) ! o17(p, g)f18(e+nu)o18
               call do1_of_2(rate_o17pg_nacre, rate_o17pg_jina)

            case(ir_o18_pa_n15) ! o18(p, a)n15 and n15(a, p)o18
               call do1_of_2(rate_o18pa_nacre, rate_o18pa_jina)
            
            case(ir_o18_ag_ne22) ! o18(a, g)ne22 
               call do1_of_2(rate_o18ag_nacre, rate_o18ag_jina)
             
            case(ir_f17_pa_o14) ! f17(p, a)o14
               call do1_of_2_reverse(rate_o14ap_fxt, rate_o14ap_jina)
             
            case(ir_f17_ap_ne20) ! f17(a, p)ne20
               call do1_of_2(rate_f17ap_reaclib, rate_f17ap_jina)
              
            case(ir_f17_gp_o16) ! f17(g, p)o16
               call do1_of_3_reverse(
     >                  rate_o16pg_nacre, rate_o16pg_jina, rate_o16pg_fxt)
            
            case(ir_f18_ap_ne21) ! f18(a, p)ne21
               call do1(rate_rf18ap)
            
            case(ir_f18_pa_o15) ! f18(p, a)o15
               call do1_of_2(rate_f18pa_wk82, rate_f18pa_jina)
            
            case(ir_f18_gp_o17) ! f18(g, p)o17
               call do1_of_2_reverse(rate_o17pg_nacre, rate_o17pg_jina)
            
            case(irf19pg_aux) ! f19(p, g)ne20
               call do1_of_2(rate_f19pg_nacre, rate_f19pg_jina)
            
            case(ir_f19_pa_o16)  ! f19(p, a)o16
               call do1_of_2(rate_f19pa_nacre, rate_f19pa_jina)
            
            case(irf19pa_aux)  ! f19(p, a)o16
               call do1_of_2(rate_f19pa_nacre, rate_f19pa_jina)
            
            case(ir_f19_ap_ne22) ! f19(a, p)ne22
               call do1_of_2(rate_f19ap_cf88, rate_f19ap_jina)
            
            case(ir_f19_gp_o18) ! f19(g, p)o18
               call do1_of_2_reverse(rate_o18pg_nacre, rate_o18pg_jina)
             
            case(irne18ap_to_mg22) ! ne18(a, p)na21(p, g)mg22
               call do1_of_2(rate_ne18ap_fxt, rate_ne18ap_jina)
             
            case(irne18ap_to_mg24) ! ne18(a, p)na21(p, g)mg22 -> mg24
               call do1_of_2(rate_ne18ap_fxt, rate_ne18ap_jina)
            
            case(irne18ag_to_mg24) ! ne18(a, g)mg22 -> mg24
               call do1(rate_ne18ag_jina)
            
            case(irne19pg_to_mg21) ! ne19(p, g)na20(p, g)mg21
               call do1(rate_ne19pg_fxt)

            case(irne19pg_to_mg22) ! ne19(p, g)na20(p, g)mg21(e+nu)na21(p, g)mg22 
               call do1_of_2(rate_ne19pg_fxt, rate_ne19pg_jina)
            
            case(irne19pg_to_mg24) ! ne19(p, g)na20(p, g)mg21(e+nu)na21(p, g)mg22 -> mg24
               call do1_of_2(rate_ne19pg_fxt, rate_ne19pg_jina)
            
            case(ir_ne19_ga_o15) ! ne19(g, a)o15 
               call do1_of_2_reverse(rate_o15ag_fxt, rate_o15ag_jina)
            
            case(irne20pg_to_mg22) ! ne20(p, g)na21(p, g)mg22 
               call do1(rate_ne20pg_nacre)
            
            case(irne20pg_to_mg24) ! ne20(p, g)na21(p, g)mg22 -> mg24
               call do1_of_2(rate_ne20pg_nacre, rate_ne20pg_jina)
            
            case(ir_ne20_ag_mg24) ! ne20(a, g)mg24
               call do1_of_2(rate_ne20ag_nacre, rate_ne20ag_jina)
            
            case(ir_ne20_ga_o16) ! ne20(g, a)o16
               call do1_of_3_reverse(
     >                  rate_o16ag_nacre, rate_o16ag_jina, rate_o16ag_fxt)
            
            case(ir_ne20_gp_f19) ! ne20(g, p)f19
               call do1_of_2_reverse(rate_f19pg_nacre, rate_f19pg_jina)
            
            case(irne20gp_aux) ! ne20(g, p)f19
               call do1_of_2_reverse(rate_f19pg_nacre, rate_f19pg_jina)
            
            case(irne20gp_to_o16) ! ne20(g, p)f19(p, a)o16
               call do1_of_2_reverse(rate_f19pg_nacre, rate_f19pg_jina)

            case(ir_ne20_ap_na23) ! ne20(a, p)na23
               call do1_of_2(rate_ne20ap_reaclib, rate_ne20ap_jina)

            case(irne20ap_aux) ! ne20(a, p)na23
               call do1_of_2(rate_ne20ap_reaclib, rate_ne20ap_jina)

            case(irne20ap_to_mg24) ! ne20(a, p)na23(p, g)mg24
               call do1_of_2(rate_ne20ap_reaclib, rate_ne20ap_jina)

            case(irne20ng_to_mg24) ! ne20(n, g)ne21(a, n)mg24
               call do1_of_2(rate_ne20ng_reaclib, rate_ne20ng_jina)

            case(ir_ne22_ag_mg26) ! ne22(a, g)mg26
               call do1_of_2(rate_ne22ag_fxt, rate_ne22ag_jina)
            
            case(ir_ne22_ga_o18) ! ne22(g, a)o18
               call do1_of_2_reverse(rate_o18ag_nacre, rate_o18ag_jina)

            case(irne22ag_to_mg24_2n) ! ne22(a,g)mg26(g,neut)mg25(g,neut)mg24
               call do1_of_2(rate_ne22ag_fxt, rate_ne22ag_jina)
            case(irne22an_to_mg24_2n) ! ne22(a,neut)mg25(g,neut)mg24
               call do1_of_2(rate_ne22an_fxt, rate_ne22an_jina)
            case(irne22ap_to_mg24_2n) 
               call do1(rate_ne22ap_jina)

            case(irne22ag_fake) ! ne22(0.5a, g)mg24
               call do1_of_2(rate_ne22ag_fxt, rate_ne22ag_jina)
            
            case(irne22ga_fake) ! ne22(g, 0.5a)ne20
               call do1_of_2_reverse(rate_o18ag_nacre, rate_o18ag_jina)
                               
            case(ir_na23_gp_ne22) ! na23(g, p)ne22
               call do1_of_2_reverse(rate_ne22pg_nacre, rate_ne22pg_jina)

            case(ir_na23_ap_mg26) ! na23(a, p)mg26 
               call do1_of_2(rate_na23ap_reaclib, rate_na23ap_jina)

            case(irna23ap_aux) ! na23(a, p)mg26 
               call do1_of_2(rate_na23ap_reaclib, rate_na23ap_jina)

            case(ir_na23_pa_ne20) ! na23(p, a)ne20 
               call do1_of_2(rate_na23pa_fxt, rate_na23pa_jina)
                           
            case(irna23pa_aux) ! na23(p, a)ne20 
               call do1_of_2(rate_na23pa_fxt, rate_na23pa_jina)
            
            case(irna23pg_aux) ! na23(p, g)mg24 
               call do1_of_2(rate_na23pg_fxt, rate_na23pg_jina)
            
            case(ir_mg24_ap_al27) ! mg24(a, p)al27 
               call do1_of_2(rate_mg24ap_fxt, rate_mg24ap_jina)
            
            case(irmg24ap_aux) ! mg24(a, p)al27 
               call do1_of_2(rate_mg24ap_fxt, rate_mg24ap_jina)
      
            case(irmg24ap_to_si28)
               call do1_of_2(rate_mg24ap_fxt, rate_mg24ap_jina)

            case(ir_mg24_ag_si28) ! mg24(a, g)si28 
               call do1_of_2(rate_mg24ag_fxt, rate_mg24ag_jina)
            
            case(ir_mg24_ga_ne20) ! mg24(g, a)ne20
               call do1_of_2_reverse(rate_ne20ag_nacre, rate_ne20ag_jina)
            
            case(ir_mg24_gp_na23) ! mg24(g, p)na23 
               call do1_of_2_reverse(rate_na23pg_fxt, rate_na23pg_jina)
            
            case(irmg24gp_aux) ! mg24(g, p)na23 
               call do1_of_2_reverse(rate_na23pg_fxt, rate_na23pg_jina)
            
            case(irmg24gp_to_ne20) ! mg24(g, p)na23(p, a)ne20 
               call do1_of_2_reverse(rate_na23pg_fxt, rate_na23pg_jina)

            case(ir_mg26_ag_si30) ! mg26(a, g)si30
               call do1_of_2(rate_mg26ag_fxt, rate_mg26ag_jina)
           
            case(iral27pg_aux) ! al27(p, g)si28
               call do1_of_2(rate_al27pg_c96, rate_al27pg_jina)

            case(ir_al27_gp_mg26) ! al27(g, p)mg26
               call do1_of_2_reverse(rate_mg26pg_fxt, rate_mg26pg_jina)
      
            case(ir_al27_pa_mg24) ! al27(p, a)mg24 
               call do1_of_2_reverse(rate_mg24ap_fxt, rate_mg24ap_jina)
      
            case(iral27pa_aux) ! al27(p, a)mg24 
               call do1_of_2_reverse(rate_mg24ap_fxt, rate_mg24ap_jina)
      
            case(ir_al27_ap_si30) ! al27(a, p)si30
               call do1_of_2(rate_mg24ap_fxt, rate_mg24ap_jina)
            
            case(ir_si28_ap_p31) ! si28(a, p)p31 
               call do1_of_2(rate_si28ap_fxt, rate_si28ap_jina)
            
            case(irsi28ap_aux) ! si28(a, p)p31 
               call do1_of_2(rate_si28ap_fxt, rate_si28ap_jina)
                           
            case(irsi28ap_to_s32) 
               call do1_of_2(rate_si28ap_fxt, rate_si28ap_jina)
            
            case(ir_si28_ag_s32) ! si28(a, g)s32 
               call do1_of_2(rate_si28ag_fxt, rate_si28ag_jina)
           
            case(ir_si28_gp_al27) ! si28(g, p)al27
               call do1_of_2_reverse(rate_al27pg_c96, rate_al27pg_jina)
           
            case(irsi28gp_aux) ! si28(g, p)al27
               call do1_of_2_reverse(rate_al27pg_c96, rate_al27pg_jina)

            case(ir_si28_ga_mg24) ! si28(g, a)mg24
               call do1_of_2_reverse(rate_mg24ag_fxt, rate_mg24ag_jina)
            
            case(irsi28gp_to_mg24) 
               call do1_of_2_reverse(rate_al27pg_c96, rate_al27pg_jina)

            case(ir_si30_ga_mg26) ! si30(g, a)mg26
               call do1_of_2_reverse(rate_mg26ag_fxt, rate_mg26ag_jina)

            case(ir_p31_pg_s32) ! p31(g, p)si30 
               call do1_of_2_reverse(rate_si30pg_fxt, rate_si30pg_jina)
            
            case(irp31pg_aux) ! p31(p, g)s32 
               call do1_of_2(rate_p31pg_fxt, rate_p31pg_jina)
      
            case(ir_p31_pa_si28) ! p31(p, a)si28 
               call do1_of_2_reverse(rate_si28ap_fxt, rate_si28ap_jina)
      
            case(irp31pa_aux) ! p31(p, a)si28 
               call do1_of_2_reverse(rate_si28ap_fxt, rate_si28ap_jina)
            
            case(ir_s32_ap_cl35) ! s32(a, p)cl35 
               call do1_of_2(rate_s32ap_fxt, rate_s32ap_jina)
            
            case(irs32ap_aux) ! s32(a, p)cl35 
               call do1_of_2(rate_s32ap_fxt, rate_s32ap_jina)
      
            case(irs32ap_to_ar36) 
               call do1_of_2(rate_s32ap_fxt, rate_s32ap_jina)
            
            case(ir_s32_ag_ar36) ! s32(a, g)ar36 
               call do1_of_2(rate_s32ag_fxt, rate_s32ag_jina)
            
            case(ir_s32_gp_p31) ! s32(g, p)p31 
               call do1_of_2_reverse(rate_p31pg_fxt, rate_p31pg_jina)
            
            case(irs32gp_aux) ! s32(g, p)p31 
               call do1_of_2_reverse(rate_p31pg_fxt, rate_p31pg_jina)

            case(ir_s32_ga_si28) ! s32(g, a)si28 
               call do1_of_2_reverse(rate_si28ag_fxt, rate_si28ag_jina)
                
            case(irs32gp_to_si28)    
               call do1_of_2_reverse(rate_p31pg_fxt, rate_p31pg_jina)
            
            case(ircl35pg_aux) ! cl35(p, g)ar36
               call do1_of_2(rate_cl35pg_fxt, rate_cl35pg_jina)

            case(ir_cl35_pa_s32) ! cl35(p, a)s32 
               call do1_of_2_reverse(rate_s32ap_fxt, rate_s32ap_jina)

            case(ircl35pa_aux) ! cl35(p, a)s32 
               call do1_of_2_reverse(rate_s32ap_fxt, rate_s32ap_jina)
            
            case(ir_ar36_ap_k39) ! ar36(a, p)k39
               call do1_of_2(rate_ar36ap_fxt, rate_ar36ap_jina)
            
            case(irar36ap_aux) ! ar36(a, p)k39
               call do1_of_2(rate_ar36ap_fxt, rate_ar36ap_jina)
                    
            case(irar36ap_to_ca40)  
               call do1_of_2(rate_ar36ap_fxt, rate_ar36ap_jina)
            
            case(ir_ar36_ag_ca40) ! ar36(a, g)ca40 
               call do1_of_2(rate_ar36ag_fxt, rate_ar36ag_jina)
            
            case(ir_ar36_gp_cl35) ! ar36(g, p)cl35
               call do1_of_2_reverse(rate_cl35pg_fxt, rate_cl35pg_jina)
            
            case(irar36gp_aux) ! ar36(g, p)cl35
               call do1_of_2_reverse(rate_cl35pg_fxt, rate_cl35pg_jina)

            case(ir_ar36_ga_s32) ! ar36(g, a)s32 
               call do1_of_2_reverse(rate_s32ag_fxt, rate_s32ag_jina)
               
            case(irar36gp_to_s32) 
               call do1_of_2_reverse(rate_cl35pg_fxt, rate_cl35pg_jina)
            
            case(irk39pg_aux) ! k39(p, g)ca40 
               call do1_of_2(rate_k39pg_fxt, rate_k39pg_jina)
      
            case(ir_k39_pa_ar36) ! k39(p, a)ar36
               call do1_of_2_reverse(rate_ar36ap_fxt, rate_ar36ap_jina)
      
            case(irk39pa_aux) ! k39(p, a)ar36
               call do1_of_2_reverse(rate_ar36ap_fxt, rate_ar36ap_jina)
            
            case(ir_ca40_ap_sc43) ! ca40(a, p)sc43 
               call do1_of_2(rate_ca40ap_fxt, rate_ca40ap_jina)
            
            case(irca40ap_aux) ! ca40(a, p)sc43 
               call do1_of_2(rate_ca40ap_fxt, rate_ca40ap_jina)
             
            case(irca40ap_to_ti44)
               call do1_of_2(rate_ca40ap_fxt, rate_ca40ap_jina)
            
            case(ir_ca40_ag_ti44) ! ca40(a, g)ti44 
               call do1_of_2(rate_ca40ag_fxt, rate_ca40ag_jina)
            
            case(ir_ca40_gp_k39) ! ca40(g, p)k39 
               call do1_of_2_reverse(rate_k39pg_fxt, rate_k39pg_jina)
            
            case(irca40gp_aux) ! ca40(g, p)k39 
               call do1_of_2_reverse(rate_k39pg_fxt, rate_k39pg_jina)
      
            case(ir_ca40_ga_ar36) ! ca40(g, a)ar36 
               call do1_of_2_reverse(rate_ar36ag_fxt, rate_ar36ag_jina)
              
            case(irca40gp_to_ar36)  
               call do1_of_2_reverse(rate_k39pg_fxt, rate_k39pg_jina)
            
            case(irsc43pg_aux) ! sc43(p, g)ti44 
               call do1_of_2(rate_sc43pg_fxt, rate_sc43pg_jina)
      
            case(ir_sc43_pa_ca40) ! sc43(p, a)ca40 
               call do1_of_2_reverse(rate_ca40ap_fxt, rate_ca40ap_jina)
      
            case(irsc43pa_aux) ! sc43(p, a)ca40 
               call do1_of_2_reverse(rate_ca40ap_fxt, rate_ca40ap_jina)
            
            case(ir_ti44_ap_v47) ! ti44(a, p)v47 
               call do1_of_2(rate_ti44ap_fxt, rate_ti44ap_jina)
            
            case(irti44ap_aux) ! ti44(a, p)v47 
               call do1_of_2(rate_ti44ap_fxt, rate_ti44ap_jina)
            
            case(irti44ap_to_cr48) 
               call do1_of_2(rate_ti44ap_fxt, rate_ti44ap_jina)
            
            case(ir_ti44_ag_cr48) ! ti44(a, g)cr48
               call do1_of_2(rate_ti44ag_fxt, rate_ti44ag_jina)
            
            case(ir_ti44_gp_sc43) ! ti44(g, p)sc43 
               call do1_of_2_reverse(rate_sc43pg_fxt, rate_sc43pg_jina)
            
            case(irti44gp_aux) ! ti44(g, p)sc43 
               call do1_of_2_reverse(rate_sc43pg_fxt, rate_sc43pg_jina)

            case(ir_ti44_ga_ca40) ! ti44(g, a)ca40 
               call do1_of_2_reverse(rate_ca40ag_fxt, rate_ca40ag_jina)
            
            case(irti44gp_to_ca40)
               call do1_of_2_reverse(rate_sc43pg_fxt, rate_sc43pg_jina)
            
            case(irv47pg_aux) ! v47(p, g)cr48 
               call do1_of_2(rate_v47pg_fxt, rate_v47pg_jina)
      
            case(ir_v47_pa_ti44) ! v47(p, a)ti44 
               call do1_of_2_reverse(rate_ti44ap_fxt, rate_ti44ap_jina)
      
            case(irv47pa_aux) ! v47(p, a)ti44 
               call do1_of_2_reverse(rate_ti44ap_fxt, rate_ti44ap_jina)
            
            case(ir_cr48_ap_mn51) ! cr48(a, p)mn51 
               call do1_of_2(rate_cr48ap_fxt, rate_cr48ap_jina)
            
            case(ircr48ap_aux) ! cr48(a, p)mn51 
               call do1_of_2(rate_cr48ap_fxt, rate_cr48ap_jina)
            
            case(ircr48ap_to_fe52)
               call do1_of_2(rate_cr48ap_fxt, rate_cr48ap_jina)
            
            case(ir_cr48_ag_fe52) ! cr48(a, g)fe52
               call do1_of_2(rate_cr48ag_fxt, rate_cr48ag_jina)
            
            case(ir_cr48_gp_v47) ! cr48(g, p)v47 
               call do1_of_2_reverse(rate_v47pg_fxt, rate_v47pg_jina)
            
            case(ircr48gp_aux) ! cr48(g, p)v47 
               call do1_of_2_reverse(rate_v47pg_fxt, rate_v47pg_jina)

            case(ir_cr48_ga_ti44) ! cr48(g, a)ti44
               call do1_of_2_reverse(rate_ti44ag_fxt, rate_ti44ag_jina)
            
            case(ircr48gp_to_ti44)
               call do1_of_2_reverse(rate_v47pg_fxt, rate_v47pg_jina)
            
            case(irmn51pg_aux) ! mn51(p, g)fe52 
               call do1_of_2(rate_mn51pg_fxt, rate_mn51pg_jina)
      
            case(ir_mn51_pa_cr48) ! mn51(p, a)cr48 
               call do1_of_2_reverse(rate_cr48ap_fxt, rate_cr48ap_jina)
      
            case(irmn51pa_aux) ! mn51(p, a)cr48 
               call do1_of_2_reverse(rate_cr48ap_fxt, rate_cr48ap_jina)
            
            case(irfe52aprot) ! fe52(a, prot)co55 
               call do1_of_2(rate_fe52ap_fxt, rate_fe52ap_jina)
            
            case(irfe52ap_aux) ! fe52(a, p)co55 
               call do1_of_2(rate_fe52ap_fxt, rate_fe52ap_jina)
            
            case(irfe52ap_to_ni56) ! fe52(a, p)co55(p, g)ni56
               call do1_of_2(rate_fe52ap_fxt, rate_fe52ap_jina)
            
            case(ir_fe52_gp_mn51) ! fe52(g, p)mn51 
               call do1_of_2_reverse(rate_mn51pg_fxt, rate_mn51pg_jina)
            
            case(irfe52gp_aux) ! fe52(g, p)mn51 
               call do1_of_2_reverse(rate_mn51pg_fxt, rate_mn51pg_jina)
      
            case(ir_fe52_ga_cr48) ! fe52(g, a)cr48
               call do1_of_2_reverse(rate_cr48ag_fxt, rate_cr48ag_jina)
            
            case(irfe52gp_to_cr48)
               call do1_of_2_reverse(rate_mn51pg_fxt, rate_mn51pg_jina)
            
            
            case(irfe52ng_to_fe54) ! fe52(n, g)fe53(neut, g)fe54
               call do1_of_2(rate_fe52ng_fxt, rate_fe52ng_jina)
            case(irfe54gn_to_fe52) ! fe54(g, n)fe53(g, n)fe52
               call do1_of_2_reverse(rate_fe53ng_fxt, rate_fe53ng_jina)
            
            case(irfe52ng_aux) ! fe52(n, g)fe53
               call do1_of_2(rate_fe52ng_fxt, rate_fe52ng_jina)
               

            case(irca40ng_to_ca42) ! ca40(neut, g)ca41(neut, g)ca42
               raw_rate = -1 ! rate calculated by special routine.
            case(irca42gn_to_ca40) ! ca42(g, neut)ca41(g, neut)ca40
               raw_rate = -1 ! rate calculated by special routine.

            case(irca40ng_aux) ! ca40(neut, g)ca41                        
               call do1(rate_ca40ng_jina)
            case(irca41ng_aux) ! ca41(neut, g)ca42                            
               call do1(rate_ca41ng_jina)
            case(irca41gn_aux) ! ca41(g, neut)ca40                            
               call do1_reverse(rate_ca40ng_jina)
            case(irca42gn_aux) ! ca42(g, neut)ca41                            
               call do1_reverse(rate_ca41ng_jina)

            case(irca42ng_to_ca44) ! ca42(neut, g)ca43(neut, g)ca44
               raw_rate = -1 ! rate calculated by special routine.
            case(irca44gn_to_ca42) ! ca44(g, neut)ca43(g, neut)ca42
               raw_rate = -1 ! rate calculated by special routine.

            case(irca42ng_aux) ! ca42(neut, g)ca43                        
               call do1(rate_ca42ng_jina)
            case(irca43ng_aux) ! ca43(neut, g)ca44                            
               call do1(rate_ca43ng_jina)
            case(irca43gn_aux) ! ca43(g, neut)ca42                            
               call do1_reverse(rate_ca42ng_jina)
            case(irca44gn_aux) ! ca44(g, neut)ca43                            
               call do1_reverse(rate_ca43ng_jina)

            case(irca44ng_to_ca46) ! ca44(neut, g)ca45(neut, g)ca46
               raw_rate = -1 ! rate calculated by special routine.
            case(irca46gn_to_ca44) ! ca46(g, neut)ca45(g, neut)ca44
               raw_rate = -1 ! rate calculated by special routine.

            case(irca44ng_aux) ! ca44(neut, g)ca45                        
               call do1(rate_ca44ng_jina)
            case(irca45ng_aux) ! ca45(neut, g)ca46                            
               call do1(rate_ca45ng_jina)
            case(irca45gn_aux) ! ca45(g, neut)ca44                            
               call do1_reverse(rate_ca44ng_jina)
            case(irca46gn_aux) ! ca46(g, neut)ca45                            
               call do1_reverse(rate_ca45ng_jina)

            case(irca46ng_to_ca48) ! ca46(neut, g)ca47(neut, g)ca48
               raw_rate = -1 ! rate calculated by special routine.
            case(irca48gn_to_ca46) ! ca48(g, neut)ca47(g, neut)ca46
               raw_rate = -1 ! rate calculated by special routine.

            case(irca46ng_aux) ! ca46(neut, g)ca47                        
               call do1(rate_ca46ng_jina)
            case(irca47ng_aux) ! ca47(neut, g)ca48                            
               call do1(rate_ca47ng_jina)
            case(irca47gn_aux) ! ca47(g, neut)ca46                            
               call do1_reverse(rate_ca46ng_jina)
            case(irca48gn_aux) ! ca48(g, neut)ca47                            
               call do1_reverse(rate_ca47ng_jina)

            case(irti46ng_to_ti48) ! ti46(neut, g)ti47(neut, g)ti48
               raw_rate = -1 ! rate calculated by special routine.
            case(irti48gn_to_ti46) ! ti48(g, neut)ti47(g, neut)ti46
               raw_rate = -1 ! rate calculated by special routine.

            case(irti46ng_aux) ! ti46(neut, g)ti47                        
               call do1(rate_ti46ng_jina)
            case(irti47ng_aux) ! ti47(neut, g)ti48                            
               call do1(rate_ti47ng_jina)
            case(irti47gn_aux) ! ti47(g, neut)ti46                            
               call do1_reverse(rate_ti46ng_jina)
            case(irti48gn_aux) ! ti48(g, neut)ti47                            
               call do1_reverse(rate_ti47ng_jina)

            case(irti48ng_to_ti50) ! ti48(neut, g)ti49(neut, g)ti50
               raw_rate = -1 ! rate calculated by special routine.
            case(irti50gn_to_ti48) ! ti50(g, neut)ti49(g, neut)ti48
               raw_rate = -1 ! rate calculated by special routine.

            case(irti48ng_aux) ! ti48(neut, g)ti49                        
               call do1(rate_ti48ng_jina)
            case(irti49ng_aux) ! ti49(neut, g)ti50                            
               call do1(rate_ti49ng_jina)
            case(irti49gn_aux) ! ti49(g, neut)ti48                            
               call do1_reverse(rate_ti48ng_jina)
            case(irti50gn_aux) ! ti50(g, neut)ti49                            
               call do1_reverse(rate_ti49ng_jina)

            case(ircr48ng_to_cr50) ! cr48(neut, g)cr42(neut, g)cr50
               raw_rate = -1 ! rate calculated by special roucrne.
            case(ircr50gn_to_cr48) ! cr50(g, neut)cr42(g, neut)cr48
               raw_rate = -1 ! rate calculated by special roucrne.

            case(ircr48ng_aux) ! cr48(neut, g)cr49                        
               call do1(rate_cr48ng_jina)
            case(ircr49ng_aux) ! cr49(neut, g)cr50                            
               call do1(rate_cr49ng_jina)
            case(ircr49gn_aux) ! cr49(g, neut)cr48                            
               call do1_reverse(rate_cr48ng_jina)
            case(ircr50gn_aux) ! cr50(g, neut)cr49                            
               call do1_reverse(rate_cr49ng_jina)

            case(ircr50ng_to_cr52) ! cr50(neut, g)cr51(neut, g)cr52
               raw_rate = -1 ! rate calculated by special roucrne.
            case(ircr52gn_to_cr50) ! cr52(g, neut)cr51(g, neut)cr50
               raw_rate = -1 ! rate calculated by special roucrne.

            case(ircr50ng_aux) ! cr50(neut, g)cr51                        
               call do1(rate_cr50ng_jina)
            case(ircr51ng_aux) ! cr51(neut, g)cr52                            
               call do1(rate_cr51ng_jina)
            case(ircr51gn_aux) ! cr51(g, neut)cr50                            
               call do1_reverse(rate_cr50ng_jina)
            case(ircr52gn_aux) ! cr52(g, neut)cr51                            
               call do1_reverse(rate_cr51ng_jina)

            case(ircr52ng_to_cr54) ! cr52(neut, g)cr53(neut, g)cr54
               raw_rate = -1 ! rate calculated by special roucrne.
            case(ircr54gn_to_cr52) ! cr54(g, neut)cr53(g, neut)cr52
               raw_rate = -1 ! rate calculated by special roucrne.

            case(ircr52ng_aux) ! cr52(neut, g)cr53                        
               call do1(rate_cr52ng_jina)
            case(ircr53ng_aux) ! cr53(neut, g)cr54                            
               call do1(rate_cr53ng_jina)
            case(ircr53gn_aux) ! cr53(g, neut)cr52                            
               call do1_reverse(rate_cr52ng_jina)
            case(ircr54gn_aux) ! cr54(g, neut)cr53                            
               call do1_reverse(rate_cr53ng_jina)
               
            
            case(irfe52neut_to_fe54) ! fe52(neut, g)fe53(neut, g)fe54
               raw_rate = -1 ! rate calculated by special routine.
            
            case(irfe53ng_aux) ! fe53(n, g)fe54
               call do1_of_2(rate_fe53ng_fxt, rate_fe53ng_jina)

            case(irfe53gn_aux) ! fe53(g, n)fe52
               call do1_of_2_reverse(rate_fe52ng_fxt, rate_fe52ng_jina)


            case(irfe54gn_aux) ! fe54(g, n)fe53
               call do1_of_2_reverse(rate_fe53ng_fxt, rate_fe53ng_jina)
            
            case(irfe54g_to_fe52) ! fe54(g, neut)fe53(g, neut)fe52
               call do1_of_2_reverse(rate_fe53ng_fxt, rate_fe53ng_jina)
            
            case(irfe54protg) ! fe54(prot, g)co55
               call do1_of_2(rate_fe54pg_fxt, rate_fe54pg_jina)
            
            case(irfe54a_to_ni56) ! fe54 + alpha -> ni56 + 2 neut
               call do1(rate_fe54a_jina)
         
         
            case(irfe54ng_aux) ! fe54(neut, g)fe55                        
               call do1_of_2(rate_fe54ng_reaclib, rate_fe54ng_jina)
            case(irfe55ng_aux) ! fe55(neut, g)fe56                            
               call do1_of_2(rate_fe55ng_reaclib, rate_fe55ng_jina)
               
            case(irfe55gn_aux) ! fe55(g, neut)fe54                            
               call do1_reverse(rate_fe54ng_jina)
            case(irfe56gn_aux) ! fe56(g, neut)fe55                            
               call do1_reverse(rate_fe55ng_jina)

            case(irfe54ng_to_fe56) ! fe54(neut, g)fe55(neut, g)fe56
               raw_rate = -1 ! rate calculated by special routine.
            case(irfe56gn_to_fe54) ! fe56(g, neut)fe55(g, neut)fe54
               raw_rate = -1 ! rate calculated by special routine.




            case(irfe56ng_aux) ! fe56(neut, g)fe57                        
               call do1(rate_fe56ng_jina)
            case(irfe57ng_aux) ! fe57(neut, g)fe58                            
               call do1(rate_fe57ng_jina)
            case(irfe57gn_aux) ! fe57(g, neut)fe56                            
               call do1_reverse(rate_fe56ng_jina)
            case(irfe58gn_aux) ! fe58(g, neut)fe57                            
               call do1_reverse(rate_fe57ng_jina)
            case(irfe56ng_to_fe58) ! fe56(neut, g)fe57(neut, g)fe58
               raw_rate = -1 ! rate calculated by special routine.
            case(irfe58gn_to_fe56) ! fe58(g, neut)fe57(g, neut)fe56
               raw_rate = -1 ! rate calculated by special routine.
      
               
            case(irfe58ng_aux) ! fe58(neut, g)fe59                        
               call do1(rate_fe58ng_jina)
            case(irfe59ng_aux) ! fe59(neut, g)fe60                            
               call do1(rate_fe59ng_jina)
            case(irfe59gn_aux) ! fe59(g, neut)fe58                            
               call do1_reverse(rate_fe58ng_jina)
            case(irfe60gn_aux) ! fe60(g, neut)fe59                            
               call do1_reverse(rate_fe59ng_jina)
            case(irfe58ng_to_fe60) ! fe58(neut, g)fe59(neut, g)fe60
               raw_rate = -1 ! rate calculated by special routine.
            case(irfe60gn_to_fe58) ! fe60(g, neut)fe59(g, neut)fe58
               raw_rate = -1 ! rate calculated by special routine.
               
            case(irfe60ng_aux) ! fe60(neut, g)fe61                        
               call do1(rate_fe60ng_jina)
            case(irfe61ng_aux) ! fe61(neut, g)fe62                            
               call do1(rate_fe61ng_jina)
            case(irfe61gn_aux) ! fe61(g, neut)fe60                            
               call do1_reverse(rate_fe60ng_jina)
            case(irfe62gn_aux) ! fe62(g, neut)fe61                            
               call do1_reverse(rate_fe61ng_jina)
            case(irfe60ng_to_fe62) ! fe60(neut, g)fe61(neut, g)fe62
               raw_rate = -1 ! rate calculated by special routine.
            case(irfe62gn_to_fe60) ! fe62(g, neut)fe61(g, neut)fe60
               raw_rate = -1 ! rate calculated by special routine.
               
            case(irfe62ng_aux) ! fe62(neut, g)fe63                        
               call do1(rate_fe62ng_jina)
            case(irfe63ng_aux) ! fe63(neut, g)fe64                            
               call do1(rate_fe63ng_jina)
            case(irfe63gn_aux) ! fe63(g, neut)fe62                            
               call do1_reverse(rate_fe62ng_jina)
            case(irfe64gn_aux) ! fe64(g, neut)fe63                            
               call do1_reverse(rate_fe63ng_jina)
            case(irfe62ng_to_fe64) ! fe62(neut, g)fe63(neut, g)fe64
               raw_rate = -1 ! rate calculated by special routine.
            case(irfe64gn_to_fe62) ! fe64(g, neut)fe63(g, neut)fe62
               raw_rate = -1 ! rate calculated by special routine.

            case(irfe52aprot_to_fe54) ! fe52(a, prot)co55(g, prot)fe54
               call do1_of_2(rate_fe52ap_fxt, rate_fe52ap_jina)
            case(irfe54prot_to_fe52) ! fe54(prot, g)co55(prot, a)fe52
               raw_rate = -1 ! rate calculated by special routine.


      
            case(ircr54ap_to_cr56) ! cr54(a, p)mn57(g, p)cr56
               call do1(rate_cr54ap_jina)            
            case(ircr56p_to_cr54) ! cr56(p, g)mn57(p, a)cr54
               raw_rate = -1 ! rate calculated by special routine.



      
            case(irfe54aprot_to_fe56) ! fe54(a, prot)co57(g, prot)fe56
               call do1(rate_fe54ap_jina)            
            case(irfe56prot_to_fe54) ! fe56(prot, g)co57(prot, a)fe54
               raw_rate = -1 ! rate calculated by special routine.


            case(irfe56aprot_to_fe58) ! fe56(a, prot)co59(g, prot)fe58
               call do1(rate_fe56ap_jina)            
            case(irfe58prot_to_fe56) ! fe58(prot, g)co59(prot, a)fe56
               raw_rate = -1 ! rate calculated by special routine.

            case(irfe58aprot_to_fe60) ! fe58(a, prot)co61(g, prot)fe60
               call do1(rate_fe58ap_jina)            
            case(irfe60prot_to_fe58) ! fe60(prot, g)co61(prot, a)fe58
               raw_rate = -1 ! rate calculated by special routine.

            case(irfe60aprot_to_fe62) ! fe60(a, prot)co63(g, prot)fe62
               call do1(rate_fe60ap_jina)            
            case(irfe62prot_to_fe60) ! fe62(prot, g)co63(prot, a)fe60
               raw_rate = -1 ! rate calculated by special routine.

            case(irfe62prot_to_ni64) ! fe62(prot, g)co63(prot, g)ni64
               raw_rate = -1 ! rate calculated by special routine.                       
            case(irni64gprot_to_fe62) ! ni64(g, prot)co63(g, prot)fe62 
               raw_rate = -1 ! rate calculated by special routine.

            case(irfe62protg_aux) ! fe62(prot, g)co63
               call do1(rate_fe62pg_jina)          
            case(irco63gprot_aux) ! co63(g, prot)fe62 
               call do1_reverse(rate_fe62pg_jina)



            case(irfe56ec_fake_to_cr56) ! fe56 + 2 e- => cr56
               raw_rate = -1 ! rate calculated by special routine.


            case(irco55g_prot) ! co55(g, prot)fe54 
               call do1_of_2_reverse(rate_fe54pg_fxt, rate_fe54pg_jina)

            case(irco55prot_a) ! co55(p, a)fe52 
               call do1_of_2_reverse(rate_fe52ap_fxt, rate_fe52ap_jina)

            case(irco55pa_aux) ! co55(p, a)fe52 
               call do1_of_2_reverse(rate_fe52ap_fxt, rate_fe52ap_jina)
            
            case(irco55protg) ! co55(prot, g)ni56 
               call do1_of_2(rate_co55pg_fxt, rate_co55pg_jina)
            
            case(irco55pg_aux) ! co55(p, g)ni56 
               call do1_of_2(rate_co55pg_fxt, rate_co55pg_jina)
            
            case(irni56g_prot) ! ni56(g, p)co55 
               call do1_of_2_reverse(rate_co55pg_fxt, rate_co55pg_jina)
            
            case(irni56gp_aux) ! ni56(g, p)co55 
               call do1_of_2_reverse(rate_co55pg_fxt, rate_co55pg_jina)
                        
            case(irni56gp_to_fe52) ! ni56(g, p)co55(p, a)fe52 
               raw_rate = -1 ! rate calculated by special routine.

            case(irni56ec_to_fe54) ! ni56 + 2 e- => 56/54*fe54
               raw_rate = -1 ! rate calculated by special routine.

            case(irni62ec_to_fe62) ! ni62 + 2 e- => fe62
               raw_rate = -1 ! rate calculated by special routine.
            case(irni64ec_to_fe64) ! ni64 + 2 e- => fe64
               raw_rate = -1 ! rate calculated by special routine.

            case(irv53ec_to_ti52_n)
               raw_rate = -1 ! rate calculated by special routine.
            case(ircr54ec_to_v53_n)
               raw_rate = -1 ! rate calculated by special routine.
            case(ircr55ec_to_v53_2n)
               raw_rate = -1 ! rate calculated by special routine.
            case(ircr56ec_to_v53_3n)
               raw_rate = -1 ! rate calculated by special routine.
            case(irfe59ec_to_mn57_2n)
               raw_rate = -1 ! rate calculated by special routine.
            case(irni63ec_to_co61_2n)
               raw_rate = -1 ! rate calculated by special routine.
            case(irni64ec_to_co61_3n)
               raw_rate = -1 ! rate calculated by special routine.
            case(irfe60ec_to_mn57_3n)
               raw_rate = -1 ! rate calculated by special routine.
            case(irmn57ec_to_cr56_n)
               raw_rate = -1 ! rate calculated by special routine.
            case(irfe58ec_to_mn57_n)
               raw_rate = -1 ! rate calculated by special routine.
            case(irco61ec_to_fe60_n)
               raw_rate = -1 ! rate calculated by special routine.
            case(irni62ec_to_co61_n)
               raw_rate = -1 ! rate calculated by special routine.

            case(irprot_to_neut) ! prot(e-nu)neut
               raw_rate = -1 ! rate calculated by special routine.

            case(irneut_to_prot) ! neut(e+nu)prot
               raw_rate = -1 ! rate calculated by special routine.

            case(irhe4gn_to_pieces) ! he4(g, neut)he3(g, prot)h2(g, neut)prot
               call do1(rate_he4gn_jina)

            case(irhe4g_neut_aux) ! he4(g, neut)he3
               call do1_reverse(rate_he3ng_fxt) ! no jina rate
               
            case(irhe3gprot_aux) ! he3(g, prot)h2
               call do1_of_2_reverse(rate_dpg_fxt, rate_dpg_jina)
               
            case(irh2g_neut_aux) ! h2(g, neut)prot
               call do1_reverse(rate_png_fxt)
               
            case(irne22_to_ne20_2n) ! ne22(g,neut)ne21(g,neut)ne20
               call do1_reverse(rate_ne21ng_jina) 
               
            case(iro18_to_o16_2n) ! o18(g,neut)o17(g,neut)o16
               call do1_reverse(rate_o17ng_jina) 
               
            case(irhe4_to_2h1_2neut) ! he4(g, neut)he3(g, prot)h2(g, neut)prot
               raw_rate = -1 ! rate calculated by special routine.
               
            case(irhe4_from_2h1_2neut) ! prot(neut, g)h2(prot, g)he3(neut, g)he4
               raw_rate = -1 ! rate calculated by special routine.
               
            case(irhe4_breakup) ! he4(g, neut)he3(g, prot)h2(g, neut)prot
               raw_rate = -1 ! rate calculated by special routine.

            case(irprot_neutg_aux) ! prot(neut, g)h2
               call do1(rate_png_fxt)
               
            case(irh2_protg_aux) ! h2(prot, g)he3
               call do1(rate_dpg_fxt)
               
            case(irhe3_neutg_aux) ! he3(neut, g)he4
               call do1(rate_he3ng_fxt)
               
            case(irhe4_rebuild) ! prot(neut, g)h2(prot, g)he3(neut, g)he4
               raw_rate = -1 ! rate calculated by special routine.
               
            case(irn14gp_to_c12_n) ! n14 => c12 + neut + h1
               call do1_of_2_reverse(rate_c13pg_nacre, rate_c13pg_jina)
               
            case(irn14gc12) ! n14 => c12 + neut + prot
               call do1_of_2_reverse(rate_c13pg_nacre, rate_c13pg_jina)
            
            case(ir_h2_pg_he3)
               call do1_of_2(rate_dpg_nacre, rate_dpg_jina)
            
            case(ir_be7_pg_b8)
               call do1_of_2(rate_be7pg_nacre, rate_be7pg_jina)

            case(ir_c12_pg_n13)
               call do1_of_2(rate_c12pg_nacre, rate_c12pg_jina)
            
            case(ir_c13_pg_n14)
               call do1_of_2(rate_c13pg_nacre, rate_c13pg_jina)
            
            case(ir_n13_pg_o14)
               call do1_of_2(rate_n13pg_nacre, rate_n13pg_jina)

            case(ir_n14_pg_o15)
               call do1_of_2(rate_n14pg_nacre, rate_n14pg_jina)

            case(ir_n15_pg_o16)
               call do1_of_2(rate_n15pg_nacre, rate_n15pg_jina)

            case(ir_o16_pg_f17)
               call do1_of_2(rate_o16pg_nacre, rate_o16pg_jina)

            case(ir_o17_pg_f18)
               call do1_of_2(rate_o17pg_nacre, rate_o17pg_jina)

            case(ir_o18_pg_f19)
               call do1_of_2(rate_o18pg_nacre, rate_o18pg_jina)
            
            case(ir_f19_pg_ne20)
               call do1_of_2(rate_f19pg_nacre, rate_f19pg_jina)

            case(ir_ne22_pg_na23)
               call do1_of_2(rate_ne22pg_nacre, rate_ne22pg_jina)
         
         ! ni56
            
            case(ir_fe52_ag_ni56) ! fe52(a, g)ni56
               call do1_of_2(rate_fe52ag_fxt, rate_fe52ag_jina)
            case(ir_ni56_ga_fe52) ! ni56(g, a)fe52
               call do1_of_2_reverse(rate_fe52ag_fxt, rate_fe52ag_jina)

            case(irfe52aprot_to_ni56) ! fe52(a, prot)co55(prot, g)ni56
               call do1_of_2(rate_fe52ap_fxt, rate_fe52ap_jina)
            case(irni56gprot_to_fe52) ! ni56(g, prot)co55(prot, a)fe52 
               raw_rate = -1 ! rate calculated by special routine.

            case(irfe52aprot_aux) ! fe52(a, prot)co55 
               call do1_of_2(rate_fe52ap_fxt, rate_fe52ap_jina)
            case(irco55prota_aux) ! co55(prot, a)fe52 
               call do1_of_2_reverse(rate_fe52ap_fxt, rate_fe52ap_jina)

            case(irni56gprot_aux) ! ni56(g, prot)co55 
               call do1_of_2_reverse(rate_co55pg_fxt, rate_co55pg_jina)
            case(irco55protg_aux) ! co55(prot, g)ni56 
               call do1_of_2(rate_co55pg_fxt, rate_co55pg_jina)

            case(irfe54prot_to_ni56) ! fe54(prot, g)co55(prot, g)ni56
               raw_rate = -1 ! rate calculated by special routine.                       
            case(irni56gprot_to_fe54) ! ni56(g, prot)co55(g, prot)fe54 
               raw_rate = -1 ! rate calculated by special routine.

            case(irfe54protg_aux) ! fe54(prot, g)co55
               call do1_of_2(rate_fe54pg_fxt, rate_fe54pg_jina)
            case(irco55gprot_aux) ! co55(g, prot)fe54 
               call do1_of_2_reverse(rate_fe54pg_fxt, rate_fe54pg_jina)

            case(irni56ec_to_fe56) ! ni56 + 2 e- => fe56
               raw_rate = -1 ! rate calculated by special routine.
         
         
         ! ni58
         
            case(ir_fe54_ag_ni58) ! fe54(a, g)ni58
               call do1(rate_fe54ag_jina)
            case(ir_ni58_ga_fe54) ! ni58(g, a)fe54
               call do1_reverse(rate_fe54ag_jina)

            case(irfe54aprot_to_ni58) ! fe54(a, prot)co57(prot, g)ni58
               call do1(rate_fe54ap_jina)                        
            case(irni58gprot_to_fe54) ! ni58(g, prot)co57(prot, a)fe54 
               raw_rate = -1 ! rate calculated by special routine.

            case(irfe54aprot_aux) ! fe54(a, prot)co57 
               call do1(rate_fe54ap_jina)
            case(irco57prota_aux) ! co57(prot, a)fe54 
               call do1_reverse(rate_fe54ap_jina)

            case(irni58gprot_aux) ! ni58(g, prot)co57 
               call do1_reverse(rate_co57pg_jina)            
            case(irco57protg_aux) ! co57(prot, g)ni58 
               call do1(rate_co57pg_jina)

            case(irfe56prot_to_ni58) ! fe56(prot, g)co57(prot, g)ni58
               raw_rate = -1 ! rate calculated by special routine.                       
            case(irni58gprot_to_fe56) ! ni58(g, prot)co57(g, prot)fe56 
               raw_rate = -1 ! rate calculated by special routine.

            case(irfe60ec_to_cr48_12n)
               raw_rate = -1 ! rate calculated by special routine.

            case(irfe56protg_aux) ! fe56(prot, g)co57
               call do1(rate_fe56pg_jina)          
            case(irco57gprot_aux) ! co57(g, prot)fe56 
               call do1_reverse(rate_fe56pg_jina)

            case(irni56aprot_to_ni58) ! ni56(a, prot)cu59(g, prot)ni58
               call do1(rate_ni56ap_jina)            
            case(irni58prot_to_ni56) ! ni58(prot, g)cu59(prot, a)ni56
               raw_rate = -1 ! rate calculated by special routine.

            case(irni56aprot_aux) ! ni56(a, prot)cu59 
               call do1(rate_ni56ap_jina)
            case(ircu59prota_aux) ! cu59(prot, a)ni56 
               call do1_reverse(rate_ni56ap_jina)

            case(irni58protg_aux) ! ni58(prot, g)cu59
               call do1(rate_ni58pg_jina)          
            case(ircu59gprot_aux) ! cu59(g, prot)ni58 
               call do1_reverse(rate_ni58pg_jina)

            case(irni56ng_to_ni58) ! ni56(neut, g)ni57(neut, g)ni58
               raw_rate = -1 ! rate calculated by special routine.
            case(irni58gn_to_ni56) ! ni58(g, neut)ni57(g, neut)ni56
               raw_rate = -1 ! rate calculated by special routine.
            case(irni56ng_aux) ! ni56(neut, g)ni57                        
               call do1(rate_ni56ng_jina)
            case(irni57ng_aux) ! ni57(neut, g)ni58                            
               call do1(rate_ni57ng_jina)
            case(irni57gn_aux) ! ni57(g, neut)ni56                            
               call do1_reverse(rate_ni56ng_jina)
            case(irni58gn_aux) ! ni58(g, neut)ni57                            
               call do1_reverse(rate_ni57ng_jina)

            case(irni58ec_to_fe58) ! ni58 + 2 e- => fe58
               raw_rate = -1 ! rate calculated by special routine.
            case(irni58ec_to_fe56_2n) ! ni58 + 2 e- => fe56 + 2 n
               raw_rate = -1 ! rate calculated by special routine.

            case(ircr55ec_to_ti52_3n) ! cr55 + 2 e- => ti52 + 2 n
               raw_rate = -1 ! rate calculated by special routine.
            case(irfe59ec_to_cr56_3n) ! fe59 + 2 e- => cr56 + 2 n
               raw_rate = -1 ! rate calculated by special routine.
            case(irfe57ec_to_cr56_n) ! fe57 + 2 e- => cr56 + 2 n
               raw_rate = -1 ! rate calculated by special routine.
            case(irsc50ec_to_ca48_2n) ! sc50 + 2 e- => ca48 + 2 n
               raw_rate = -1 ! rate calculated by special routine.
            case(irsc51ec_to_ca48_3n) ! sc51 + 2 e- => ca48 + 3 n
               raw_rate = -1 ! rate calculated by special routine.
            case(irti52ec_to_sc51_n) ! ti52 + 2 e- => sc51 + 1 n
               raw_rate = -1 ! rate calculated by special routine.
            case(irv54ec_to_ti52_2n) ! v54 + 2 e- => ti52 + 2 n
               raw_rate = -1 ! rate calculated by special routine.
            case(irv55ec_to_ti52_3n) ! v55 + 2 e- => ti52 + 3 n
               raw_rate = -1 ! rate calculated by special routine.
            case(ircr56ec_to_v55_n) ! cr56 + 2 e- => v55 + 1 n
               raw_rate = -1 ! rate calculated by special routine.
            case(irmn58ec_to_cr56_2n) ! mn58 + 2 e- => cr56 + 2 n
               raw_rate = -1 ! rate calculated by special routine.
            case(irmn59ec_to_cr56_3n) ! mn59 + 2 e- => cr56 + 3 n
               raw_rate = -1 ! rate calculated by special routine.
            case(irfe60ec_to_mn59_n) ! fe60 + 2 e- => mn59 + 1 n
               raw_rate = -1 ! rate calculated by special routine.
            case(irco62ec_to_fe60_2n) ! co62 + 2 e- => fe60 + 2 n
               raw_rate = -1 ! rate calculated by special routine.
            case(irco63ec_to_fe60_3n) ! co63 + 2 e- => fe60 + 3 n
               raw_rate = -1 ! rate calculated by special routine.
            case(irco64ec_to_fe60_4n) ! co64 + 2 e- => fe60 + 4 n
               raw_rate = -1 ! rate calculated by special routine.

            case(ircr56ap_to_fe60) ! cr56(a, p)mn59(p, g)fe60
               call do1(rate_cr56ap_jina)                        
            case(irfe60gp_to_cr56) ! fe60(g, p)mn59(p, a)cr56 
               raw_rate = -1 ! rate calculated by special routine.

            case(ircr56ap_aux) ! cr56(a, p)mn59 
               call do1(rate_cr56ap_jina)
            case(irmn59pa_aux) ! mn59(p, a)cr56 
               call do1_reverse(rate_cr56ap_jina)

            case(irfe60gp_aux) ! fe60(g, p)mn59 
               call do1_reverse(rate_mn59pg_jina)            
            case(irmn59pg_aux) ! mn59(p, g)fe60 
               call do1(rate_mn59pg_jina)

            case(ircr56p_to_fe58) ! cr56(p, g)mn57(p, g)fe58
               raw_rate = -1 ! rate calculated by special routine.                       
            case(irfe58gp_to_cr56) ! fe58(g, p)mn57(g, p)cr56 
               raw_rate = -1 ! rate calculated by special routine.

            case(ircr56pg_aux) ! cr56(p, g)mn57
               call do1(rate_cr56pg_jina)          
            case(irmn57gp_aux) ! mn57(g, p)cr56 
               call do1_reverse(rate_cr56pg_jina)


            case(ircr54ap_to_fe58) ! cr54(a, p)mn57(p, g)fe58
               call do1(rate_cr54ap_jina)                        
            case(irfe58gp_to_cr54) ! fe58(g, p)mn57(p, a)cr54 
               raw_rate = -1 ! rate calculated by special routine.

            case(ircr54ap_aux) ! cr54(a, p)mn57 
               call do1(rate_cr54ap_jina)
            case(irmn57pa_aux) ! mn57(p, a)cr54 
               call do1_reverse(rate_cr54ap_jina)

            case(irfe58gp_aux) ! fe58(g, p)mn57 
               call do1_reverse(rate_mn57pg_jina)            
            case(irmn57pg_aux) ! mn57(p, g)fe58 
               call do1(rate_mn57pg_jina)

            case(irmn59gp_aux) ! mn59(g, p)cr58 
               call do1_reverse(rate_cr58pg_jina)            



         
         ! ni60
         
            case(ir_fe56_ag_ni60) ! fe56(a, g)ni60
               call do1(rate_fe56ag_jina)
            case(ir_ni60_ga_fe56) ! ni60(g, a)fe56
               call do1_reverse(rate_fe56ag_jina)

            case(irfe56aprot_to_ni60) ! fe56(a, prot)co59(prot, g)ni60
               call do1(rate_fe56ap_jina)                        
            case(irni60gprot_to_fe56) ! ni60(g, prot)co59(prot, a)fe56 
               raw_rate = -1 ! rate calculated by special routine.

            case(irfe56aprot_aux) ! fe56(a, prot)co59 
               call do1(rate_fe56ap_jina)
            case(irco59prota_aux) ! co59(prot, a)fe56 
               call do1_reverse(rate_fe56ap_jina)

            case(irni60gprot_aux) ! ni60(g, prot)co59 
               call do1_reverse(rate_co59pg_jina)            
            case(irco59protg_aux) ! co59(prot, g)ni60 
               call do1(rate_co59pg_jina)

            case(irfe58prot_to_ni60) ! fe58(prot, g)co59(prot, g)ni60
               raw_rate = -1 ! rate calculated by special routine.                       
            case(irni60gprot_to_fe58) ! ni60(g, prot)co59(g, prot)fe58 
               raw_rate = -1 ! rate calculated by special routine.

            case(irfe58protg_aux) ! fe58(prot, g)co59
               call do1(rate_fe58pg_jina)          
            case(irco59gprot_aux) ! co59(g, prot)fe58 
               call do1_reverse(rate_fe58pg_jina)

            case(irni58aprot_to_ni60) ! ni58(a, prot)cu61(g, prot)ni60
               call do1(rate_ni58ap_jina)            
            case(irni60prot_to_ni58) ! ni60(prot, g)cu61(prot, a)ni58
               raw_rate = -1 ! rate calculated by special routine.

            case(irni58aprot_aux) ! ni58(a, prot)cu61 
               call do1(rate_ni58ap_jina)
            case(ircu61prota_aux) ! cu61(prot, a)ni58 
               call do1_reverse(rate_ni58ap_jina)

            case(irni60protg_aux) ! ni60(prot, g)cu61
               call do1(rate_ni60pg_jina)          
            case(ircu61gprot_aux) ! cu61(g, prot)ni60 
               call do1_reverse(rate_ni60pg_jina)

            case(irni58ng_to_ni60) ! ni58(neut, g)ni59(neut, g)ni60
               raw_rate = -1 ! rate calculated by special routine.
            case(irni60gn_to_ni58) ! ni60(g, neut)ni59(g, neut)ni58
               raw_rate = -1 ! rate calculated by special routine.

            case(irni58ng_aux) ! ni58(neut, g)ni59                        
               call do1(rate_ni58ng_jina)
            case(irni59ng_aux) ! ni59(neut, g)ni60                            
               call do1(rate_ni59ng_jina)
            case(irni59gn_aux) ! ni59(g, neut)ni58                            
               call do1_reverse(rate_ni58ng_jina)
            case(irni60gn_aux) ! ni60(g, neut)ni59                            
               call do1_reverse(rate_ni59ng_jina)

            case(irni60ec_to_fe60) ! ni60 + 2 e- => fe60
               raw_rate = -1 ! rate calculated by special routine.
            
            ! ni62
            case(ir_fe58_ag_ni62) ! fe58(a, g)ni62
               call do1(rate_fe58ag_jina)
            case(ir_ni62_ga_fe58) ! ni62(g, a)fe58
               call do1_reverse(rate_fe58ag_jina)

            case(irfe58aprot_to_ni62) ! fe58(a, prot)co61(prot, g)ni62
               call do1(rate_fe58ap_jina)                        
            case(irni62gprot_to_fe58) ! ni62(g, prot)co61(prot, a)fe58 
               raw_rate = -1 ! rate calculated by special routine.

            case(irfe58aprot_aux) ! fe58(a, prot)co61 
               call do1(rate_fe58ap_jina)
            case(irco61prota_aux) ! co61(prot, a)fe58 
               call do1_reverse(rate_fe58ap_jina)

            case(irni62gprot_aux) ! ni62(g, prot)co61 
               call do1_reverse(rate_co61pg_jina)            
            case(irco61protg_aux) ! co61(prot, g)ni62 
               call do1(rate_co61pg_jina)

            case(irfe60prot_to_ni62) ! fe60(prot, g)co61(prot, g)ni62
               raw_rate = -1 ! rate calculated by special routine.                       
            case(irni62gprot_to_fe60) ! ni62(g, prot)co61(g, prot)fe60 
               raw_rate = -1 ! rate calculated by special routine.

            case(irfe60protg_aux) ! fe60(prot, g)co61
               call do1(rate_fe60pg_jina)          
            case(irco61gprot_aux) ! co61(g, prot)fe60 
               call do1_reverse(rate_fe60pg_jina)
!xxxx
            case(irni60aprot_to_ni62) ! ni60(a, prot)cu63(g, prot)ni62
               call do1(rate_ni60ap_jina)            
            case(irni62prot_to_ni60) ! ni62(prot, g)cu63(prot, a)ni60
               raw_rate = -1 ! rate calculated by special routine.

            case(irni60aprot_aux) ! ni60(a, prot)cu63 
               call do1(rate_ni60ap_jina)
            case(ircu63prota_aux) ! cu63(prot, a)ni60 
               call do1_reverse(rate_ni60ap_jina)

            case(irni62protg_aux) ! ni62(prot, g)cu63
               call do1(rate_ni62pg_jina)          
            case(ircu63gprot_aux) ! cu63(g, prot)ni62 
               call do1_reverse(rate_ni62pg_jina)
!xxxx

            case(irni60ng_to_ni62) ! ni60(neut, g)ni61(neut, g)ni62
               raw_rate = -1 ! rate calculated by special routine.
            case(irni62gn_to_ni60) ! ni62(g, neut)ni61(g, neut)ni60
               raw_rate = -1 ! rate calculated by special routine.

            case(irni60ng_aux) ! ni60(neut, g)ni61                        
               call do1(rate_ni60ng_jina)
            case(irni61ng_aux) ! ni61(neut, g)ni62                            
               call do1(rate_ni61ng_jina)
            case(irni61gn_aux) ! ni61(g, neut)ni60                            
               call do1_reverse(rate_ni60ng_jina)
            case(irni62gn_aux) ! ni62(g, neut)ni61                            
               call do1_reverse(rate_ni61ng_jina)

            case(irni62ec_to_fe60_2n) ! ni62 + 2 e- => fe60 + 2 n
               raw_rate = -1 ! rate calculated by special routine.
            
            ! ni64
!xxxx
            case(ir_fe60_ag_ni64) ! fe60(a, g)ni64
               call do1(rate_fe60ag_jina)
            case(ir_ni64_ga_fe60) ! ni64(g, a)fe60
               call do1_reverse(rate_fe60ag_jina)
!xxxx

            case(irfe60aprot_to_ni64) ! fe60(a, prot)co63(prot, g)ni64
               call do1(rate_fe60ap_jina)                        
            case(irni64gprot_to_fe60) ! ni64(g, prot)co63(prot, a)fe60 
               raw_rate = -1 ! rate calculated by special routine.

            case(irfe60aprot_aux) ! fe60(a, prot)co63 
               call do1(rate_fe60ap_jina)
            case(irco63prota_aux) ! co63(prot, a)fe60 
               call do1_reverse(rate_fe60ap_jina)

            case(irni64gprot_aux) ! ni64(g, prot)co63 
               call do1_reverse(rate_co63pg_jina)            
            case(irco63protg_aux) ! co63(prot, g)ni64 
               call do1(rate_co63pg_jina)
!xxxx
            case(irni62aprot_to_ni64) ! ni62(a, prot)cu65(g, prot)ni64
               call do1(rate_ni62ap_jina)            
            case(irni64prot_to_ni62) ! ni64(prot, g)cu65(prot, a)ni62
               raw_rate = -1 ! rate calculated by special routine.

            case(irni62aprot_aux) ! ni62(a, prot)cu65 
               call do1(rate_ni62ap_jina)
            case(ircu65prota_aux) ! cu65(prot, a)ni62 
               call do1_reverse(rate_ni62ap_jina)

            case(irni64protg_aux) ! ni64(prot, g)cu65
               call do1(rate_ni64pg_jina)          
            case(ircu65gprot_aux) ! cu65(g, prot)ni64 
               call do1_reverse(rate_ni64pg_jina)
!xxxx

            case(irni62ng_to_ni64) ! ni62(neut, g)ni63(neut, g)ni64
               raw_rate = -1 ! rate calculated by special routine.
            case(irni64gn_to_ni62) ! ni64(g, neut)ni63(g, neut)ni62
               raw_rate = -1 ! rate calculated by special routine.

            case(irni62ng_aux) ! ni62(neut, g)ni63                        
               call do1(rate_ni62ng_jina)
            case(irni63ng_aux) ! ni63(neut, g)ni64                            
               call do1(rate_ni63ng_jina)
            case(irni63gn_aux) ! ni63(g, neut)ni62                            
               call do1_reverse(rate_ni62ng_jina)
            case(irni64gn_aux) ! ni64(g, neut)ni63                            
               call do1_reverse(rate_ni63ng_jina)

            ! ni66

            case(ir_fe62_ag_ni66) ! fe62(a, g)ni66
               call do1(rate_fe62ag_jina)
            case(ir_ni66_ga_fe62) ! ni66(g, a)fe62
               call do1_reverse(rate_fe62ag_jina)

            case(irfe62aprot_to_ni66) ! fe62(a, prot)co65(prot, g)ni66
               call do1(rate_fe62ap_jina)                        
            case(irni66gprot_to_fe62) ! ni66(g, prot)co65(prot, a)fe62 
               raw_rate = -1 ! rate calculated by special routine.

            case(irfe62aprot_aux) ! fe62(a, prot)co65 
               call do1(rate_fe62ap_jina)
            case(irco65prota_aux) ! co65(prot, a)fe62 
               call do1_reverse(rate_fe62ap_jina)

            case(irni66gprot_aux) ! ni66(g, prot)co65 
               call do1_reverse(rate_co65pg_jina)            
            case(irco65protg_aux) ! co65(prot, g)ni66 
               call do1(rate_co65pg_jina)

            case(irni64aprot_to_ni66) ! ni64(a, prot)cu67(g, prot)ni66
               call do1(rate_ni64ap_jina)            
            case(irni66prot_to_ni64) ! ni66(prot, g)cu67(prot, a)ni64
               raw_rate = -1 ! rate calculated by special routine.

            case(irni64ng_to_ni66) ! ni64(neut, g)ni65(neut, g)ni66
               raw_rate = -1 ! rate calculated by special routine.
            case(irni66gn_to_ni64) ! ni66(g, neut)ni65(g, neut)ni64
               raw_rate = -1 ! rate calculated by special routine.

            case(irni64ng_aux) ! ni64(neut, g)ni65                        
               call do1(rate_ni64ng_jina)
            case(irni65ng_aux) ! ni65(neut, g)ni66                            
               call do1(rate_ni65ng_jina)
            case(irni65gn_aux) ! ni65(g, neut)ni64                            
               call do1_reverse(rate_ni64ng_jina)
            case(irni66gn_aux) ! ni66(g, neut)ni65                            
               call do1_reverse(rate_ni65ng_jina)

            case(irni66ec_to_fe60_6n)
               raw_rate = -1 ! rate calculated by special routine.

            ! zn64

            case(ir_ni60_ag_zn64) ! ni60(a, g)zn64
               call do1(rate_ni60ag_jina)
            case(ir_zn64_ga_ni60) ! zn64(g, a)ni60
               call do1_reverse(rate_ni60ag_jina)

            case(irni60aprot_to_zn64) ! ni60(a, prot)cu63(prot, g)zn64
               call do1(rate_ni60ap_jina)                        
            case(irzn64gprot_to_ni60) ! zn64(g, prot)cu63(prot, a)ni60 
               raw_rate = -1 ! rate calculated by special routine.

            case(irzn64gprot_aux) ! zn64(g, prot)cu63 
               call do1_reverse(rate_cu63pg_jina)            
            case(ircu63protg_aux) ! cu63(prot, g)zn64 
               call do1(rate_cu63pg_jina)

            case(irni62prot_to_zn64) ! ni62(prot, g)cu63(prot, g)zn64
               raw_rate = -1 ! rate calculated by special routine.                       
            case(irzn64gprot_to_ni62) ! zn64(g, prot)cu63(g, prot)ni62 
               raw_rate = -1 ! rate calculated by special routine.

            case(irzn62aprot_to_zn64) ! zn62(a, prot)ga65(g, prot)zn64
               call do1(rate_zn62ap_jina)            
            case(irzn64prot_to_zn62) ! zn64(prot, g)ga65(prot, a)zn62
               raw_rate = -1 ! rate calculated by special routine.

            case(irzn62aprot_aux) ! zn62(a, prot)ga65 
               call do1(rate_zn62ap_jina)
            case(irga65prota_aux) ! ga65(prot, a)zn62 
               call do1_reverse(rate_zn62ap_jina)

            case(irzn64protg_aux) ! zn64(prot, g)ga65
               call do1(rate_zn64pg_jina)          
            case(irga65gprot_aux) ! ga65(g, prot)zn64 
               call do1_reverse(rate_zn64pg_jina)

            case(irni64ec_to_fe60_4n) 
               raw_rate = -1 ! rate calculated by special routine.
            case(irzn64ec_to_ni64) ! zn64 + 2 e- => ni64
               raw_rate = -1 ! rate calculated by special routine.

            ! zn66

            case(ir_ni62_ag_zn66) ! ni62(a, g)zn66
               call do1(rate_ni62ag_jina)
            case(ir_zn66_ga_ni62) ! zn66(g, a)ni62
               call do1_reverse(rate_ni62ag_jina)

            case(irni62aprot_to_zn66) ! ni62(a, prot)cu65(prot, g)zn66
               call do1(rate_ni62ap_jina)                        
            case(irzn66gprot_to_ni62) ! zn66(g, prot)cu65(prot, a)ni62 
               raw_rate = -1 ! rate calculated by special routine.

            case(irzn66gprot_aux) ! zn66(g, prot)cu65 
               call do1_reverse(rate_cu65pg_jina)            
            case(ircu65protg_aux) ! cu65(prot, g)zn66 
               call do1(rate_cu65pg_jina)

            case(irni64prot_to_zn66) ! ni64(prot, g)cu65(prot, g)zn66
               raw_rate = -1 ! rate calculated by special routine.                       
            case(irzn66gprot_to_ni64) ! zn66(g, prot)cu65(g, prot)ni64 
               raw_rate = -1 ! rate calculated by special routine.

            case(irzn64aprot_to_zn66) ! zn64(a, prot)ga67(g, prot)zn66
               call do1(rate_zn64ap_jina)            
            case(irzn66prot_to_zn64) ! zn66(prot, g)ga67(prot, a)zn64
               raw_rate = -1 ! rate calculated by special routine.

            case(irzn64aprot_aux) ! zn64(a, prot)ga67 
               call do1(rate_zn64ap_jina)
            case(irga67prota_aux) ! ga67(prot, a)zn64 
               call do1_reverse(rate_zn64ap_jina)

            case(irzn66protg_aux) ! zn66(prot, g)ga67
               call do1(rate_zn66pg_jina)          
            case(irga67gprot_aux) ! ga67(g, prot)zn66 
               call do1_reverse(rate_zn66pg_jina)

            case(irzn64ng_to_zn66) ! zn64(neut, g)zn65(neut, g)zn66
               raw_rate = -1 ! rate calculated by special routine.
            case(irzn66gn_to_zn64) ! zn66(g, neut)zn65(g, neut)zn64
               raw_rate = -1 ! rate calculated by special routine.


            case(irzn60ng_to_zn62) ! zn60(neut, g)zn61(neut, g)zn62
               raw_rate = -1 ! rate calculated by special routine.
            case(irzn62gn_to_zn60) ! zn62(g, neut)zn61(g, neut)zn60
               raw_rate = -1 ! rate calculated by special routine.

            case(irzn60ng_aux) ! zn60(neut, g)zn61                        
               call do1(rate_zn60ng_jina)
            case(irzn61ng_aux) ! zn61(neut, g)zn62                            
               call do1(rate_zn61ng_jina)
            case(irzn61gn_aux) ! zn61(g, neut)zn60                            
               call do1_reverse(rate_zn60ng_jina)
            case(irzn62gn_aux) ! zn62(g, neut)zn61                            
               call do1_reverse(rate_zn61ng_jina)


            case(irzn62ng_to_zn64) ! zn62(neut, g)zn63(neut, g)zn64
               raw_rate = -1 ! rate calculated by special routine.
            case(irzn64gn_to_zn62) ! zn64(g, neut)zn63(g, neut)zn62
               raw_rate = -1 ! rate calculated by special routine.

            case(irzn62ng_aux) ! zn62(neut, g)zn63                        
               call do1(rate_zn62ng_jina)
            case(irzn63ng_aux) ! zn63(neut, g)zn64                            
               call do1(rate_zn63ng_jina)
            case(irzn63gn_aux) ! zn63(g, neut)zn62                            
               call do1_reverse(rate_zn62ng_jina)
            case(irzn64gn_aux) ! zn64(g, neut)zn63                            
               call do1_reverse(rate_zn63ng_jina)

            case(irzn64ng_aux) ! zn64(neut, g)zn65                        
               call do1(rate_zn64ng_jina)
            case(irzn65ng_aux) ! zn65(neut, g)zn66                            
               call do1(rate_zn65ng_jina)
            case(irzn65gn_aux) ! zn65(g, neut)zn64                            
               call do1_reverse(rate_zn64ng_jina)
            case(irzn66gn_aux) ! zn66(g, neut)zn65                            
               call do1_reverse(rate_zn65ng_jina)

            case(irzn66ec_to_ni66) ! zn66 + 2 e- => ni66
               raw_rate = -1 ! rate calculated by special routine.

            ! zn68
 
            case(ir_ni64_ag_zn68) ! ni64(a, g)zn68
               call do1(rate_ni64ag_jina)
            case(ir_zn68_ga_ni64) ! zn68(g, a)ni64
               call do1_reverse(rate_ni64ag_jina)

            case(irni64aprot_to_zn68) ! ni64(a, prot)cu67(prot, g)zn68
               call do1(rate_ni64ap_jina)                        
            case(irzn68gprot_to_ni64) ! zn68(g, prot)cu67(prot, a)ni64 
               raw_rate = -1 ! rate calculated by special routine.

            case(irni64aprot_aux) ! ni64(a, prot)cu67 
               call do1(rate_ni64ap_jina)
            case(ircu67prota_aux) ! cu67(prot, a)ni64 
               call do1_reverse(rate_ni64ap_jina)

            case(irzn68gprot_aux) ! zn68(g, prot)cu67 
               call do1_reverse(rate_cu67pg_jina)            
            case(ircu67protg_aux) ! cu67(prot, g)zn68 
               call do1(rate_cu67pg_jina)

            case(irni66prot_to_zn68) ! ni66(prot, g)cu67(prot, g)zn68
               raw_rate = -1 ! rate calculated by special routine.                       
            case(irzn68gprot_to_ni66) ! zn68(g, prot)cu67(g, prot)ni66 
               raw_rate = -1 ! rate calculated by special routine.

            case(irni66protg_aux) ! ni66(prot, g)cu67
               call do1(rate_ni66pg_jina)          
            case(ircu67gprot_aux) ! cu67(g, prot)ni66 
               call do1_reverse(rate_ni66pg_jina)

            case(irzn66aprot_to_zn68) ! zn66(a, prot)ga69(g, prot)zn68
               call do1(rate_zn66ap_jina)            
            case(irzn68prot_to_zn66) ! zn68(prot, g)ga69(prot, a)zn66
               raw_rate = -1 ! rate calculated by special routine.

            case(irzn68protg_aux) ! zn68(prot, g)ga69
               call do1(rate_zn68pg_jina)          
            case(irga69gprot_aux) ! ga69(g, prot)zn68 
               call do1_reverse(rate_zn68pg_jina)

            case(irzn66ng_to_zn68) ! zn66(neut, g)zn67(neut, g)zn68
               raw_rate = -1 ! rate calculated by special routine.
            case(irzn68gn_to_zn66) ! zn68(g, neut)zn67(g, neut)zn66
               raw_rate = -1 ! rate calculated by special routine.

            case(irzn66ng_aux) ! zn66(neut, g)zn67                        
               call do1(rate_zn66ng_jina)
            case(irzn67ng_aux) ! zn67(neut, g)zn68                            
               call do1(rate_zn67ng_jina)
            case(irzn67gn_aux) ! zn67(g, neut)zn66                            
               call do1_reverse(rate_zn66ng_jina)
            case(irzn68gn_aux) ! zn68(g, neut)zn67                            
               call do1_reverse(rate_zn67ng_jina)

            case(irzn68ec_to_ni68) ! zn68 + 2 e- => ni68
               raw_rate = -1 ! rate calculated by special routine.


            ! zn70

            case(ir_ni66_ag_zn70) ! ni66(a, g)zn70
               call do1(rate_ni66ag_jina)
            case(ir_zn70_ga_ni66) ! zn70(g, a)ni66
               call do1_reverse(rate_ni66ag_jina)

            case(irni66aprot_to_zn70) ! ni66(a, prot)cu69(prot, g)zn70
               call do1(rate_ni66ap_jina)                        
            case(irzn70gprot_to_ni66) ! zn70(g, prot)cu69(prot, a)ni66 
               raw_rate = -1 ! rate calculated by special routine.

            case(irni66aprot_aux) ! ni66(a, prot)cu69 
               call do1(rate_ni66ap_jina)
            case(ircu69prota_aux) ! cu69(prot, a)ni66 
               call do1_reverse(rate_ni66ap_jina)

            case(irzn70gprot_aux) ! zn70(g, prot)cu69 
               call do1_reverse(rate_cu69pg_jina)            
            case(ircu69protg_aux) ! cu69(prot, g)zn70 
               call do1(rate_cu69pg_jina)

            case(irni68prot_to_zn70) ! ni68(prot, g)cu69(prot, g)zn70
               raw_rate = -1 ! rate calculated by special routine.                       
            case(irzn70gprot_to_ni68) ! zn70(g, prot)cu69(g, prot)ni68 
               raw_rate = -1 ! rate calculated by special routine.

            case(irni68protg_aux) ! ni68(prot, g)cu69
               call do1(rate_ni68pg_jina)          
            case(ircu69gprot_aux) ! cu69(g, prot)ni68 
               call do1_reverse(rate_ni68pg_jina)

            case(irzn68aprot_to_zn70) ! zn68(a, prot)ga71(g, prot)zn70
               call do1(rate_zn68ap_jina)            
            case(irzn70prot_to_zn68) ! zn70(prot, g)ga71(prot, a)zn68
               raw_rate = -1 ! rate calculated by special routine.

            case(irzn68aprot_aux) ! zn68(a, prot)ga71 
               call do1(rate_zn68ap_jina)
            case(irga71prota_aux) ! ga71(prot, a)zn68 
               call do1_reverse(rate_zn68ap_jina)

            case(irzn70protg_aux) ! zn70(prot, g)ga71
               call do1(rate_zn70pg_jina)          
            case(irga71gprot_aux) ! ga71(g, prot)zn70 
               call do1_reverse(rate_zn70pg_jina)

            case(irzn68ng_to_zn70) ! zn68(neut, g)zn69(neut, g)zn70
               raw_rate = -1 ! rate calculated by special routine.
            case(irzn70gn_to_zn68) ! zn70(g, neut)zn69(g, neut)zn68
               raw_rate = -1 ! rate calculated by special routine.

            case(irzn68ng_aux) ! zn68(neut, g)zn69                        
               call do1(rate_zn68ng_jina)
            case(irzn69ng_aux) ! zn69(neut, g)zn70                            
               call do1(rate_zn69ng_jina)
            case(irzn69gn_aux) ! zn69(g, neut)zn68                            
               call do1_reverse(rate_zn68ng_jina)
            case(irzn70gn_aux) ! zn70(g, neut)zn69                            
               call do1_reverse(rate_zn69ng_jina)

            case(irzn70ec_to_ni70) ! zn70 + 2 e- => ni70
               raw_rate = -1 ! rate calculated by special routine.

            ! zn72

            case(ir_ni68_ag_zn72) ! ni68(a, g)zn72
               call do1(rate_ni68ag_jina)
            case(ir_zn72_ga_ni68) ! zn72(g, a)ni68
               call do1_reverse(rate_ni68ag_jina)

            case(irni68aprot_to_zn72) ! ni68(a, prot)cu71(prot, g)zn72
               call do1(rate_ni68ap_jina)                        
            case(irzn72gprot_to_ni68) ! zn72(g, prot)cu71(prot, a)ni68 
               raw_rate = -1 ! rate calculated by special routine.

            case(irni68aprot_aux) ! ni68(a, prot)cu71 
               call do1(rate_ni68ap_jina)
            case(ircu71prota_aux) ! cu71(prot, a)ni68 
               call do1_reverse(rate_ni68ap_jina)

            case(irzn72gprot_aux) ! zn72(g, prot)cu71 
               call do1_reverse(rate_cu71pg_jina)            
            case(ircu71protg_aux) ! cu71(prot, g)zn72 
               call do1(rate_cu71pg_jina)

            case(irni70prot_to_zn72) ! ni70(prot, g)cu71(prot, g)zn72
               raw_rate = -1 ! rate calculated by special routine.                       
            case(irzn72gprot_to_ni70) ! zn72(g, prot)cu71(g, prot)ni70 
               raw_rate = -1 ! rate calculated by special routine.

            case(irni70protg_aux) ! ni70(prot, g)cu71
               call do1(rate_ni70pg_jina)          
            case(ircu71gprot_aux) ! cu71(g, prot)ni70 
               call do1_reverse(rate_ni70pg_jina)

            case(irzn70aprot_to_zn72) ! zn70(a, prot)ga73(g, prot)zn72
               call do1(rate_zn70ap_jina)            
            case(irzn72prot_to_zn70) ! zn72(prot, g)ga73(prot, a)zn70
               raw_rate = -1 ! rate calculated by special routine.

            case(irzn70aprot_aux) ! zn70(a, prot)ga73 
               call do1(rate_zn70ap_jina)
            case(irga73prota_aux) ! ga73(prot, a)zn70 
               call do1_reverse(rate_zn70ap_jina)

            case(irzn72protg_aux) ! zn72(prot, g)ga73
               call do1(rate_zn72pg_jina)          
            case(irga73gprot_aux) ! ga73(g, prot)zn72 
               call do1_reverse(rate_zn72pg_jina)

            case(irzn70ng_to_zn72) ! zn70(neut, g)zn71(neut, g)zn72
               raw_rate = -1 ! rate calculated by special routine.
            case(irzn72gn_to_zn70) ! zn72(g, neut)zn71(g, neut)zn70
               raw_rate = -1 ! rate calculated by special routine.

            case(irzn70ng_aux) ! zn70(neut, g)zn71                        
               call do1(rate_zn70ng_jina)
            case(irzn71ng_aux) ! zn71(neut, g)zn72                            
               call do1(rate_zn71ng_jina)
            case(irzn71gn_aux) ! zn71(g, neut)zn70                            
               call do1_reverse(rate_zn70ng_jina)
            case(irzn72gn_aux) ! zn72(g, neut)zn71                            
               call do1_reverse(rate_zn71ng_jina)

            case(irzn72ec_to_ni72) ! zn72 + 2 e- => ni72
               raw_rate = -1 ! rate calculated by special routine.

            ! ge70

            case(ir_zn66_ag_ge70) ! zn66(a, g)ge70
               call do1(rate_zn66ag_jina)
            case(ir_ge70_ga_zn66) ! ge70(g, a)zn66
               call do1_reverse(rate_zn66ag_jina)

            case(irzn66aprot_to_ge70) ! zn66(a, prot)ga69(prot, g)ge70
               call do1(rate_zn66ap_jina)                        
            case(irge70gprot_to_zn66) ! ge70(g, prot)ga69(prot, a)zn66 
               raw_rate = -1 ! rate calculated by special routine.

            case(irzn66aprot_aux) ! zn66(a, prot)ga69 
               call do1(rate_zn66ap_jina)
            case(irga69prota_aux) ! ga69(prot, a)zn66 
               call do1_reverse(rate_zn66ap_jina)

            case(irge70gprot_aux) ! ge70(g, prot)ga69 
               call do1_reverse(rate_ga69pg_jina)            
            case(irga69protg_aux) ! ga69(prot, g)ge70 
               call do1(rate_ga69pg_jina)

            case(irzn68prot_to_ge70) ! zn68(prot, g)ga69(prot, g)ge70
               raw_rate = -1 ! rate calculated by special routine.                       
            case(irge70gprot_to_zn68) ! ge70(g, prot)ga69(g, prot)zn68 
               raw_rate = -1 ! rate calculated by special routine.

            case(irge68aprot_to_ge70) ! ge68(a, prot)as71(g, prot)ge70
               call do1(rate_ge68ap_jina)            
            case(irge70prot_to_ge68) ! ge70(prot, g)as71(prot, a)ge68
               raw_rate = -1 ! rate calculated by special routine.

            case(irge68aprot_aux) ! ge68(a, prot)as71 
               call do1(rate_ge68ap_jina)
            case(iras71prota_aux) ! as71(prot, a)ge68 
               call do1_reverse(rate_ge68ap_jina)

            case(irge70protg_aux) ! ge70(prot, g)as71
               call do1(rate_ge70pg_jina)          
            case(iras71gprot_aux) ! as71(g, prot)ge70 
               call do1_reverse(rate_ge70pg_jina)

            case(irge68ng_to_ge70) ! ge68(neut, g)ge69(neut, g)ge70
               raw_rate = -1 ! rate calculated by special routine.
            case(irge70gn_to_ge68) ! ge70(g, neut)ge69(g, neut)ge68
               raw_rate = -1 ! rate calculated by special routine.

            case(irge68ng_aux) ! ge68(neut, g)ge69                        
               call do1(rate_ge68ng_jina)
            case(irge69ng_aux) ! ge69(neut, g)ge70                            
               call do1(rate_ge69ng_jina)
            case(irge69gn_aux) ! ge69(g, neut)ge68                            
               call do1_reverse(rate_ge68ng_jina)
            case(irge70gn_aux) ! ge70(g, neut)ge69                            
               call do1_reverse(rate_ge69ng_jina)

            case(irge70ec_to_zn70) ! ge70 + 2 e- => zn70
               raw_rate = -1 ! rate calculated by special routine.

            ! ge72

            case(ir_zn68_ag_ge72) ! zn68(a, g)ge72
               call do1(rate_zn68ag_jina)
            case(ir_ge72_ga_zn68) ! ge72(g, a)zn68
               call do1_reverse(rate_zn68ag_jina)

            case(irzn68aprot_to_ge72) ! zn68(a, prot)ga71(prot, g)ge72
               call do1(rate_zn68ap_jina)                        
            case(irge72gprot_to_zn68) ! ge72(g, prot)ga71(prot, a)zn68 
               raw_rate = -1 ! rate calculated by special routine.

            case(irge72gprot_aux) ! ge72(g, prot)ga71 
               call do1_reverse(rate_ga71pg_jina)            
            case(irga71protg_aux) ! ga71(prot, g)ge72 
               call do1(rate_ga71pg_jina)

            case(irzn70prot_to_ge72) ! zn70(prot, g)ga71(prot, g)ge72
               raw_rate = -1 ! rate calculated by special routine.                       
            case(irge72gprot_to_zn70) ! ge72(g, prot)ga71(g, prot)zn70 
               raw_rate = -1 ! rate calculated by special routine.

            case(irge70aprot_to_ge72) ! ge70(a, prot)as73(g, prot)ge72
               call do1(rate_ge70ap_jina)            
            case(irge72prot_to_ge70) ! ge72(prot, g)as73(prot, a)ge70
               raw_rate = -1 ! rate calculated by special routine.

            case(irge70aprot_aux) ! ge70(a, prot)as73 
               call do1(rate_ge70ap_jina)
            case(iras73prota_aux) ! as73(prot, a)ge70 
               call do1_reverse(rate_ge70ap_jina)

            case(irge72protg_aux) ! ge72(prot, g)as73
               call do1(rate_ge72pg_jina)          
            case(iras73gprot_aux) ! as73(g, prot)ge72 
               call do1_reverse(rate_ge72pg_jina)

            case(irge70ng_to_ge72) ! ge70(neut, g)ge71(neut, g)ge72
               raw_rate = -1 ! rate calculated by special routine.
            case(irge72gn_to_ge70) ! ge72(g, neut)ge71(g, neut)ge70
               raw_rate = -1 ! rate calculated by special routine.

            case(irge70ng_aux) ! ge70(neut, g)ge71                        
               call do1(rate_ge70ng_jina)
            case(irge71ng_aux) ! ge71(neut, g)ge72                            
               call do1(rate_ge71ng_jina)
            case(irge71gn_aux) ! ge71(g, neut)ge70                            
               call do1_reverse(rate_ge70ng_jina)
            case(irge72gn_aux) ! ge72(g, neut)ge71                            
               call do1_reverse(rate_ge71ng_jina)

            case(irge72ec_to_zn72) ! ge72 + 2 e- => zn72
               raw_rate = -1 ! rate calculated by special routine.

            ! ge74

            case(ir_zn70_ag_ge74) ! zn70(a, g)ge74
               call do1(rate_zn70ag_jina)
            case(ir_ge74_ga_zn70) ! ge74(g, a)zn70
               call do1_reverse(rate_zn70ag_jina)

            case(irzn70aprot_to_ge74) ! zn70(a, prot)ga73(prot, g)ge74
               call do1(rate_zn70ap_jina)                        
            case(irge74gprot_to_zn70) ! ge74(g, prot)ga73(prot, a)zn70 
               raw_rate = -1 ! rate calculated by special routine.

            case(irge74gprot_aux) ! ge74(g, prot)ga73 
               call do1_reverse(rate_ga73pg_jina)            
            case(irga73protg_aux) ! ga73(prot, g)ge74 
               call do1(rate_ga73pg_jina)

            case(irzn72prot_to_ge74) ! zn72(prot, g)ga73(prot, g)ge74
               raw_rate = -1 ! rate calculated by special routine.                       
            case(irge74gprot_to_zn72) ! ge74(g, prot)ga73(g, prot)zn72 
               raw_rate = -1 ! rate calculated by special routine.

            case(irge72aprot_to_ge74) ! ge72(a, prot)as75(g, prot)ge74
               call do1(rate_ge72ap_jina)            
            case(irge74prot_to_ge72) ! ge74(prot, g)as75(prot, a)ge72
               raw_rate = -1 ! rate calculated by special routine.

            case(irge72aprot_aux) ! ge72(a, prot)as75 
               call do1(rate_ge72ap_jina)
            case(iras75prota_aux) ! as75(prot, a)ge72 
               call do1_reverse(rate_ge72ap_jina)

            case(irge74protg_aux) ! ge74(prot, g)as75
               call do1(rate_ge74pg_jina)          
            case(iras75gprot_aux) ! as75(g, prot)ge74 
               call do1_reverse(rate_ge74pg_jina)

            case(irge72ng_to_ge74) ! ge72(neut, g)ge73(neut, g)ge74
               raw_rate = -1 ! rate calculated by special routine.
            case(irge74gn_to_ge72) ! ge74(g, neut)ge73(g, neut)ge72
               raw_rate = -1 ! rate calculated by special routine.

            case(irge72ng_aux) ! ge72(neut, g)ge73                        
               call do1(rate_ge72ng_jina)
            case(irge73ng_aux) ! ge73(neut, g)ge74                            
               call do1(rate_ge73ng_jina)
            case(irge73gn_aux) ! ge73(g, neut)ge72                            
               call do1_reverse(rate_ge72ng_jina)
            case(irge74gn_aux) ! ge74(g, neut)ge73                            
               call do1_reverse(rate_ge73ng_jina)

            case(irge74ec_to_zn74) ! ge74 + 2 e- => zn74
               raw_rate = -1 ! rate calculated by special routine.

            case(irge74ng_to_ge76) ! ge74(neut, g)ge75(neut, g)ge76
               raw_rate = -1 ! rate calculated by special routine.
            case(irge76gn_to_ge74) ! ge76(g, neut)ge75(g, neut)ge74
               raw_rate = -1 ! rate calculated by special routine.

            case(irge74ng_aux) ! ge74(neut, g)ge75                        
               call do1(rate_ge74ng_jina)
            case(irge75ng_aux) ! ge75(neut, g)ge76                            
               call do1(rate_ge75ng_jina)
            case(irge75gn_aux) ! ge75(g, neut)ge74                            
               call do1_reverse(rate_ge74ng_jina)
            case(irge76gn_aux) ! ge76(g, neut)ge75                            
               call do1_reverse(rate_ge75ng_jina)
            
            case(irne22p_to_mg24) ! ne22(p,g)na23(p,g)mg24
               raw_rate = -1 ! rate calculated by special routine.                       
            case(irmg24gp_to_ne22) ! mg24(g,p)na23(g,p)ne22 
               raw_rate = -1 ! rate calculated by special routine.
            

            case(irne22ap_to_mg26) ! ne22(a, p)na25(p, g)mg26
               call do1(rate_ne22ap_jina)                        
            case(irmg26gp_to_ne22) ! mg26(g, p)na25(p, a)ne22 
               raw_rate = -1 ! rate calculated by special routine.

            case(irne22ap_aux) ! ne22(a, p)na25 
               call do1(rate_ne22ap_jina)
            case(irna25pa_aux) ! na25(p, a)ne22 
               call do1_reverse(rate_ne22ap_jina)

            case(irmg26gp_aux) ! mg26(g, p)na25 
               call do1_reverse(rate_na25pg_jina)            
            case(irna25pg_aux) ! na25(p, g)mg26 
               call do1(rate_na25pg_jina)

            case(irmg24ap_to_mg26) ! mg24(a, p)al27(g, p)mg26
               call do1(rate_mg24ap_jina)            
            case(irmg26p_to_mg24) ! mg26(p, g)al27(p, a)mg24
               raw_rate = -1 ! rate calculated by special routine.

            case(irmg26pg_aux) ! mg26(p, g)al27
               call do1(rate_mg26pg_jina)          
            case(iral27gp_aux) ! al27(g, p)mg26 
               call do1_reverse(rate_mg26pg_jina)

            case(irmg24ng_to_mg26) ! mg24(neut, g)mg25(neut, g)mg26
               raw_rate = -1 ! rate calculated by special routine.
            case(irmg26gn_to_mg24) ! mg26(g, neut)mg25(g, neut)mg24
               raw_rate = -1 ! rate calculated by special routine.

            case(irmg24ng_aux) ! mg24(neut, g)mg25                        
               call do1(rate_mg24ng_jina)
            case(irmg25ng_aux) ! mg25(neut, g)mg26                            
               call do1(rate_mg25ng_jina)
            case(irmg25gn_aux) ! mg25(g, neut)mg24                            
               call do1_reverse(rate_mg24ng_jina)
            case(irmg26gn_aux) ! mg26(g, neut)mg25                            
               call do1_reverse(rate_mg25ng_jina)

            case(iro18ap_to_ne22) ! o18(a, p)f21(p, g)ne22
               call do1(rate_o18ap_jina)                        
            case(irne22gp_to_o18) ! ne22(g, p)f21(p, a)o18 
               raw_rate = -1 ! rate calculated by special routine.

            case(iro18ap_aux) ! o18(a, p)f21 
               call do1(rate_o18ap_jina)
            case(irf21pa_aux) ! f21(p, a)o18 
               call do1_reverse(rate_o18ap_jina)

            case(irne22gp_aux) ! ne22(g, p)f21 
               call do1_reverse(rate_f21pg_jina)            
            case(irf21pg_aux) ! f21(p, g)ne22 
               call do1(rate_f21pg_jina)

            case(irne20ap_to_ne22) ! ne20(a, p)na23(g, p)ne22
               call do1(rate_ne20ap_jina)            
            case(irne22p_to_ne20) ! ne22(p, g)na23(p, a)ne20
               raw_rate = -1 ! rate calculated by special routine.

            case(irne22pg_aux) ! ne22(p, g)na23
               call do1(rate_ne22pg_jina)          
            case(irna23gp_aux) ! na23(g, p)ne22 
               call do1_reverse(rate_ne22pg_jina)

            case(irne20ng_to_ne22) ! ne20(neut, g)ne21(neut, g)ne22
               raw_rate = -1 ! rate calculated by special routine.
            case(irne22gn_to_ne20) ! ne22(g, neut)ne21(g, neut)ne20
               raw_rate = -1 ! rate calculated by special routine.

            case(irne20ng_aux) ! ne20(neut, g)ne21                        
               call do1(rate_ne20ng_jina)
            case(irne21ng_aux) ! ne21(neut, g)ne22                            
               call do1(rate_ne21ng_jina)
            case(irne21gn_aux) ! ne21(g, neut)ne20                            
               call do1_reverse(rate_ne20ng_jina)
            case(irne22gn_aux) ! ne22(g, neut)ne21                            
               call do1_reverse(rate_ne21ng_jina)

            ! si30

            case(irmg26ap_to_si30) ! mg26(a, p)al29(p, g)si30
               call do1(rate_mg26ap_jina)                        
            case(irsi30gp_to_mg26) ! si30(g, p)al29(p, a)mg26 
               raw_rate = -1 ! rate calculated by special routine.

            case(irmg26ap_aux) ! mg26(a, p)al29 
               call do1(rate_mg26ap_jina)
            case(iral29pa_aux) ! al29(p, a)mg26 
               call do1_reverse(rate_mg26ap_jina)

            case(irsi30gp_aux) ! si30(g, p)al29 
               call do1_reverse(rate_al29pg_jina)            
            case(iral29pg_aux) ! al29(p, g)si30 
               call do1(rate_al29pg_jina)

            case(irmg28p_to_si30) ! mg28(p, g)al29(p, g)si30
               raw_rate = -1 ! rate calculated by special routine.                       
            case(irsi30gp_to_mg28) ! si30(g, p)al29(g, p)mg28 
               raw_rate = -1 ! rate calculated by special routine.

            case(irmg28pg_aux) ! mg28(p, g)al29
               call do1(rate_mg28pg_jina)          
            case(iral29gp_aux) ! al29(g, p)mg28 
               call do1_reverse(rate_mg28pg_jina)

            case(irsi28ap_to_si30) ! si28(a, p)p31(g, p)si30
               call do1(rate_si28ap_jina)            
            case(irsi30p_to_si28) ! si30(p, g)p31(p, a)si28
               raw_rate = -1 ! rate calculated by special routine.

            case(irsi30pg_aux) ! si30(p, g)p31
               call do1(rate_si30pg_jina)          
            case(irp31gp_aux) ! p31(g, p)si30 
               call do1_reverse(rate_si30pg_jina)

            case(irsi28ng_to_si30) ! si28(neut, g)si29(neut, g)si30
               raw_rate = -1 ! rate calculated by special routine.
            case(irsi30gn_to_si28) ! si30(g, neut)si29(g, neut)si28
               raw_rate = -1 ! rate calculated by special routine.

            case(irsi28ng_aux) ! si28(neut, g)si29                        
               call do1(rate_si28ng_jina)
            case(irsi29ng_aux) ! si29(neut, g)si30                            
               call do1(rate_si29ng_jina)
            case(irsi29gn_aux) ! si29(g, neut)si28                            
               call do1_reverse(rate_si28ng_jina)
            case(irsi30gn_aux) ! si30(g, neut)si29                            
               call do1_reverse(rate_si29ng_jina)

            ! s34

            case(ir_si30_ag_s34) ! si30(a, g)s34
               call do1(rate_si30ag_jina)
            case(ir_s34_ga_si30) ! s34(g, a)si30
               call do1_reverse(rate_si30ag_jina)

            case(irsi30ap_to_s34) ! si30(a, p)p33(p, g)s34
               call do1(rate_si30ap_jina)                        
            case(irs34gp_to_si30) ! s34(g, p)p33(p, a)si30 
               raw_rate = -1 ! rate calculated by special routine.

            case(irsi30ap_aux) ! si30(a, p)p33 
               call do1(rate_si30ap_jina)
            case(irp33pa_aux) ! p33(p, a)si30 
               call do1_reverse(rate_si30ap_jina)

            case(irs34gp_aux) ! s34(g, p)p33 
               call do1_reverse(rate_p33pg_jina)            
            case(irp33pg_aux) ! p33(p, g)s34 
               call do1(rate_p33pg_jina)

            case(irsi32p_to_s34) ! si32(p, g)p33(p, g)s34
               raw_rate = -1 ! rate calculated by special routine.                       
            case(irs34gp_to_si32) ! s34(g, p)p33(g, p)si32 
               raw_rate = -1 ! rate calculated by special routine.

            case(irsi32pg_aux) ! si32(p, g)p33
               call do1(rate_si32pg_jina)          
            case(irp33gp_aux) ! p33(g, p)si32 
               call do1_reverse(rate_si32pg_jina)

            case(irs32ap_to_s34) ! s32(a, p)cl35(g, p)s34
               call do1(rate_s32ap_jina)            
            case(irs34p_to_s32) ! s34(p, g)cl35(p, a)s32
               raw_rate = -1 ! rate calculated by special routine.

            case(irs34pg_aux) ! s34(p, g)cl35
               call do1(rate_s34pg_jina)          
            case(ircl35gp_aux) ! cl35(g, p)s34 
               call do1_reverse(rate_s34pg_jina)

            case(irs32ng_to_s34) ! s32(neut, g)s33(neut, g)s34
               raw_rate = -1 ! rate calculated by special routine.
            case(irs34gn_to_s32) ! s34(g, neut)s33(g, neut)s32
               raw_rate = -1 ! rate calculated by special routine.

            case(irs32ng_aux) ! s32(neut, g)s33                        
               call do1(rate_s32ng_jina)
            case(irs33ng_aux) ! s33(neut, g)s34                            
               call do1(rate_s33ng_jina)
            case(irs33gn_aux) ! s33(g, neut)s32                            
               call do1_reverse(rate_s32ng_jina)
            case(irs34gn_aux) ! s34(g, neut)s33                            
               call do1_reverse(rate_s33ng_jina)

            ! ar38

            case(ir_s34_ag_ar38) ! s34(a, g)ar38
               call do1(rate_s34ag_jina)
            case(ir_ar38_ga_s34) ! ar38(g, a)s34
               call do1_reverse(rate_s34ag_jina)

            case(irs34ap_to_ar38) ! s34(a, p)cl37(p, g)ar38
               call do1(rate_s34ap_jina)                        
            case(irar38gp_to_s34) ! ar38(g, p)cl37(p, a)s34 
               raw_rate = -1 ! rate calculated by special routine.

            case(irs34ap_aux) ! s34(a, p)cl37 
               call do1(rate_s34ap_jina)
            case(ircl37pa_aux) ! cl37(p, a)s34 
               call do1_reverse(rate_s34ap_jina)

            case(irar38gp_aux) ! ar38(g, p)cl37 
               call do1_reverse(rate_cl37pg_jina)            
            case(ircl37pg_aux) ! cl37(p, g)ar38 
               call do1(rate_cl37pg_jina)

            case(irs36p_to_ar38) ! s36(p, g)cl37(p, g)ar38
               raw_rate = -1 ! rate calculated by special routine.                       
            case(irar38gp_to_s36) ! ar38(g, p)cl37(g, p)s36 
               raw_rate = -1 ! rate calculated by special routine.

            case(irs36pg_aux) ! s36(p, g)cl37
               call do1(rate_s36pg_jina)          
            case(ircl37gp_aux) ! cl37(g, p)s36 
               call do1_reverse(rate_s36pg_jina)

            case(irar36ap_to_ar38) ! ar36(a, p)k39(g, p)ar38
               call do1(rate_ar36ap_jina)            
            case(irar38p_to_ar36) ! ar38(p, g)k39(p, a)ar36
               raw_rate = -1 ! rate calculated by special routine.

            case(irar38pg_aux) ! ar38(p, g)k39
               call do1(rate_ar38pg_jina)          
            case(irk39gp_aux) ! k39(g, p)ar38 
               call do1_reverse(rate_ar38pg_jina)

            case(irar36ng_to_ar38) ! ar36(neut, g)ar37(neut, g)ar38
               raw_rate = -1 ! rate calculated by special routine.
            case(irar38gn_to_ar36) ! ar38(g, neut)ar37(g, neut)ar36
               raw_rate = -1 ! rate calculated by special routine.

            case(irar36ng_aux) ! ar36(neut, g)ar37                        
               call do1(rate_ar36ng_jina)
            case(irar37ng_aux) ! ar37(neut, g)ar38                            
               call do1(rate_ar37ng_jina)
            case(irar37gn_aux) ! ar37(g, neut)ar36                            
               call do1_reverse(rate_ar36ng_jina)
            case(irar38gn_aux) ! ar38(g, neut)ar37                            
               call do1_reverse(rate_ar37ng_jina)

            ! ca42

            case(ir_ar38_ag_ca42) ! ar38(a, g)ca42
               call do1(rate_ar38ag_jina)
            case(ir_ca42_ga_ar38) ! ca42(g, a)ar38
               call do1_reverse(rate_ar38ag_jina)

            case(irar38ap_to_ca42) ! ar38(a, p)k41(p, g)ca42
               call do1(rate_ar38ap_jina)                        
            case(irca42gp_to_ar38) ! ca42(g, p)k41(p, a)ar38 
               raw_rate = -1 ! rate calculated by special routine.

            case(irar38ap_aux) ! ar38(a, p)k41 
               call do1(rate_ar38ap_jina)
            case(irk41pa_aux) ! k41(p, a)ar38 
               call do1_reverse(rate_ar38ap_jina)

            case(irca42gp_aux) ! ca42(g, p)k41 
               call do1_reverse(rate_k41pg_jina)            
            case(irk41pg_aux) ! k41(p, g)ca42 
               call do1(rate_k41pg_jina)

            case(irar40p_to_ca42) ! ar40(p, g)k41(p, g)ca42
               raw_rate = -1 ! rate calculated by special routine.                       
            case(irca42gp_to_ar40) ! ca42(g, p)k41(g, p)ar40 
               raw_rate = -1 ! rate calculated by special routine.

            case(irar40pg_aux) ! ar40(p, g)k41
               call do1(rate_ar40pg_jina)          
            case(irk41gp_aux) ! k41(g, p)ar40 
               call do1_reverse(rate_ar40pg_jina)

            case(irca40ap_to_ca42) ! ca40(a, p)sc43(g, p)ca42
               call do1(rate_ca40ap_jina)            
            case(irca42p_to_ca40) ! ca42(p, g)sc43(p, a)ca40
               raw_rate = -1 ! rate calculated by special routine.

            case(irca42pg_aux) ! ca42(p, g)sc43
               call do1(rate_ca42pg_jina)          
            case(irsc43gp_aux) ! sc43(g, p)ca42 
               call do1_reverse(rate_ca42pg_jina)

            ! ti46

            case(ir_ca42_ag_ti46) ! ca42(a, g)ti46
               call do1(rate_ca42ag_jina)
            case(ir_ti46_ga_ca42) ! ti46(g, a)ca42
               call do1_reverse(rate_ca42ag_jina)

            case(irca42ap_to_ti46) ! ca42(a, p)sc45(p, g)ti46
               call do1(rate_ca42ap_jina)                        
            case(irti46gp_to_ca42) ! ti46(g, p)sc45(p, a)ca42 
               raw_rate = -1 ! rate calculated by special routine.

            case(irca42ap_aux) ! ca42(a, p)sc45 
               call do1(rate_ca42ap_jina)
            case(irsc45pa_aux) ! sc45(p, a)ca42 
               call do1_reverse(rate_ca42ap_jina)

            case(irti46gp_aux) ! ti46(g, p)sc45 
               call do1_reverse(rate_sc45pg_jina)            
            case(irsc45pg_aux) ! sc45(p, g)ti46 
               call do1(rate_sc45pg_jina)

            case(irca44p_to_ti46) ! ca44(p, g)sc45(p, g)ti46
               raw_rate = -1 ! rate calculated by special routine.                       
            case(irti46gp_to_ca44) ! ti46(g, p)sc45(g, p)ca44 
               raw_rate = -1 ! rate calculated by special routine.

            case(irca44pg_aux) ! ca44(p, g)sc45
               call do1(rate_ca44pg_jina)          
            case(irsc45gp_aux) ! sc45(g, p)ca44 
               call do1_reverse(rate_ca44pg_jina)

            case(irti44ap_to_ti46) ! ti44(a, p)v47(g, p)ti46
               call do1(rate_ti44ap_jina)            
            case(irti46p_to_ti44) ! ti46(p, g)v47(p, a)ti44
               raw_rate = -1 ! rate calculated by special routine.

            case(irti46pg_aux) ! ti46(p, g)v47
               call do1(rate_ti46pg_jina)          
            case(irv47gp_aux) ! v47(g, p)ti46 
               call do1_reverse(rate_ti46pg_jina)

            case(irti44ng_to_ti46) ! ti44(neut, g)ti45(neut, g)ti46
               raw_rate = -1 ! rate calculated by special routine.
            case(irti46gn_to_ti44) ! ti46(g, neut)ti45(g, neut)ti44
               raw_rate = -1 ! rate calculated by special routine.

            case(irti44ng_aux) ! ti44(neut, g)ti45                        
               call do1(rate_ti44ng_jina)
            case(irti45ng_aux) ! ti45(neut, g)ti46                            
               call do1(rate_ti45ng_jina)
            case(irti45gn_aux) ! ti45(g, neut)ti44                            
               call do1_reverse(rate_ti44ng_jina)
            case(irti46gn_aux) ! ti46(g, neut)ti45                            
               call do1_reverse(rate_ti45ng_jina)

            ! cr50

            case(ir_ti46_ag_cr50) ! ti46(a, g)cr50
               call do1(rate_ti46ag_jina)
            case(ir_cr50_ga_ti46) ! cr50(g, a)ti46
               call do1_reverse(rate_ti46ag_jina)

            case(irti46ap_to_cr50) ! ti46(a, p)v49(p, g)cr50
               call do1(rate_ti46ap_jina)                        
            case(ircr50gp_to_ti46) ! cr50(g, p)v49(p, a)ti46 
               raw_rate = -1 ! rate calculated by special routine.

            case(irti46ap_aux) ! ti46(a, p)v49 
               call do1(rate_ti46ap_jina)
            case(irv49pa_aux) ! v49(p, a)ti46 
               call do1_reverse(rate_ti46ap_jina)

            case(ircr50gp_aux) ! cr50(g, p)v49 
               call do1_reverse(rate_v49pg_jina)            
            case(irv49pg_aux) ! v49(p, g)cr50 
               call do1(rate_v49pg_jina)

            case(irti48p_to_cr50) ! ti48(p, g)v49(p, g)cr50
               raw_rate = -1 ! rate calculated by special routine.                       
            case(ircr50gp_to_ti48) ! cr50(g, p)v49(g, p)ti48 
               raw_rate = -1 ! rate calculated by special routine.

            case(irti48pg_aux) ! ti48(p, g)v49
               call do1(rate_ti48pg_jina)          
            case(irv49gp_aux) ! v49(g, p)ti48 
               call do1_reverse(rate_ti48pg_jina)

            case(ircr48ap_to_cr50) ! cr48(a, p)mn51(g, p)cr50
               call do1(rate_cr48ap_jina)            
            case(ircr50p_to_cr48) ! cr50(p, g)mn51(p, a)cr48
               raw_rate = -1 ! rate calculated by special routine.

            case(ircr50pg_aux) ! cr50(p, g)mn51
               call do1(rate_cr50pg_jina)          
            case(irmn51gp_aux) ! mn51(g, p)cr50 
               call do1_reverse(rate_cr50pg_jina)
            
            ! ti52
            case(irti50ng_to_ti52) ! ti50(neut, g)ti51(neut, g)ti52
               raw_rate = -1 ! rate calculated by special routine.
            case(irti52gn_to_ti50) ! ti52(g, neut)ti51(g, neut)ti50
               raw_rate = -1 ! rate calculated by special routine.
            case(irti50ng_aux) ! ti50(neut, g)ti51                        
               call do1(rate_ti50ng_jina)
            case(irti51ng_aux) ! ti51(neut, g)ti52                            
               call do1(rate_ti51ng_jina)
            case(irti51gn_aux) ! ti51(g, neut)ti50                            
               call do1_reverse(rate_ti50ng_jina)
            case(irti52gn_aux) ! ti52(g, neut)ti51                            
               call do1_reverse(rate_ti51ng_jina)

            ! cr56
            case(ircr54ng_to_cr56) ! cr54(neut, g)cr55(neut, g)cr56
               raw_rate = -1 ! rate calculated by special routine.
            case(ircr56gn_to_cr54) ! cr56(g, neut)cr55(g, neut)cr54
               raw_rate = -1 ! rate calculated by special routine.
            case(ircr54ng_aux) ! cr54(neut, g)cr55                        
               call do1(rate_cr54ng_jina)
            case(ircr55ng_aux) ! cr55(neut, g)cr56                            
               call do1(rate_cr55ng_jina)
            case(ircr55gn_aux) ! cr55(g, neut)cr54                            
               call do1_reverse(rate_cr54ng_jina)
            case(ircr56gn_aux) ! cr56(g, neut)cr55                            
               call do1_reverse(rate_cr55ng_jina)

            ! fe54

            case(ir_cr50_ag_fe54) ! cr50(a, g)fe54
               call do1(rate_cr50ag_jina)
            case(ir_fe54_ga_cr50) ! fe54(g, a)cr50
               call do1_reverse(rate_cr50ag_jina)

            case(ircr50ap_to_fe54) ! cr50(a, p)mn53(p, g)fe54
               call do1(rate_cr50ap_jina)                        
            case(irfe54gp_to_cr50) ! fe54(g, p)mn53(p, a)cr50 
               raw_rate = -1 ! rate calculated by special routine.

            case(ircr50ap_aux) ! cr50(a, p)mn53 
               call do1(rate_cr50ap_jina)
            case(irmn53pa_aux) ! mn53(p, a)cr50 
               call do1_reverse(rate_cr50ap_jina)
      
            case(ircr50ap_to_cr52) ! cr50(a, p)mn53(g, p)cr52
               call do1(rate_cr50ap_jina)            
            case(ircr52p_to_cr50) ! cr52(p, g)mn53(p, a)cr50
               raw_rate = -1 ! rate calculated by special routine.

            case(irfe54gp_aux) ! fe54(g, p)mn53 
               call do1_reverse(rate_mn53pg_jina)            
            case(irmn53pg_aux) ! mn53(p, g)fe54 
               call do1(rate_mn53pg_jina)

            case(ircr52p_to_fe54) ! cr52(p, g)mn53(p, g)fe54
               raw_rate = -1 ! rate calculated by special routine.                       
            case(irfe54gp_to_cr52) ! fe54(g, p)mn53(g, p)cr52 
               raw_rate = -1 ! rate calculated by special routine.

            case(ircr52pg_aux) ! cr52(p, g)mn53
               call do1(rate_cr52pg_jina)          
            case(irmn53gp_aux) ! mn53(g, p)cr52 
               call do1_reverse(rate_cr52pg_jina)

            case(ircr48ec_to_ti48)
               raw_rate = -1 ! rate calculated by special routine.
            case(ircr50ec_to_ti50)
               raw_rate = -1 ! rate calculated by special routine.
            case(ircr52ec_to_ti52)
               raw_rate = -1 ! rate calculated by special routine.
            case(ircr54ec_to_ti52_2n)
               raw_rate = -1 ! rate calculated by special routine.
            case(ircr56ec_to_ti52_4n)
               raw_rate = -1 ! rate calculated by special routine.
            case(irti48ec_to_ca48)
               raw_rate = -1 ! rate calculated by special routine.
            case(irti46ec_to_ca46)
               raw_rate = -1 ! rate calculated by special routine.
            case(irti44ec_to_ca44)
               raw_rate = -1 ! rate calculated by special routine.
            case(irti50ec_to_ca48_2n)
               raw_rate = -1 ! rate calculated by special routine.
            case(irti52ec_to_ca48_4n)
               raw_rate = -1 ! rate calculated by special routine.
            case(irfe52ec_to_cr52)
               raw_rate = -1 ! rate calculated by special routine.
            case(irfe54ec_to_cr54)
               raw_rate = -1 ! rate calculated by special routine.
            case(irfe56ec_to_cr56)
               raw_rate = -1 ! rate calculated by special routine.
            case(irfe58ec_to_cr56_2n)
               raw_rate = -1 ! rate calculated by special routine.
            case(irfe60ec_to_cr56_4n)
               raw_rate = -1 ! rate calculated by special routine.

            case(ircr54ee_to_fe54)
               raw_rate = -1 ! rate calculated by special routine.
            case(ircr56ee_to_fe56)
               raw_rate = -1 ! rate calculated by special routine.

            case(irfe56ee_to_ni56)
               raw_rate = -1 ! rate calculated by special routine.
            case(irti50ee_to_cr50)
               raw_rate = -1 ! rate calculated by special routine.
            case(irti52ee_to_cr52)
               raw_rate = -1 ! rate calculated by special routine.
            case(ircr52ee_to_fe52)
               raw_rate = -1 ! rate calculated by special routine.

            case(irti44_ecnp_ca42)
               raw_rate = -1 ! rate calculated by special routine.
            case(irti46_ecnp_ca44)
               raw_rate = -1 ! rate calculated by special routine.
            case(irti48_ecnp_ca46)
               raw_rate = -1 ! rate calculated by special routine.
            case(irti50_ecnp_ca48)
               raw_rate = -1 ! rate calculated by special routine.
            case(irti52_ecnp_ca48)
               raw_rate = -1 ! rate calculated by special routine.

            case(ircr48_ecnp_ti46)
               raw_rate = -1 ! rate calculated by special routine.
            case(ircr50_ecnp_ti48)
               raw_rate = -1 ! rate calculated by special routine.
            case(ircr52_ecnp_ti50)
               raw_rate = -1 ! rate calculated by special routine.
            case(ircr54_ecnp_ti52)
               raw_rate = -1 ! rate calculated by special routine.
            case(ircr56_ecnp_ti52)
               raw_rate = -1 ! rate calculated by special routine.

            case(irfe52_ecnp_cr50)
               raw_rate = -1 ! rate calculated by special routine.
            case(irfe54_ecnp_cr52)
               raw_rate = -1 ! rate calculated by special routine.
            case(irfe56_ecnp_cr54)
               raw_rate = -1 ! rate calculated by special routine.
            case(irfe58_ecnp_cr56)
               raw_rate = -1 ! rate calculated by special routine.
            case(irfe60_ecnp_cr56)
               raw_rate = -1 ! rate calculated by special routine.

            case(irni56_ecnp_fe54)
               raw_rate = -1 ! rate calculated by special routine.
            case(irni58_ecnp_fe56)
               raw_rate = -1 ! rate calculated by special routine.
            case(irni60_ecnp_fe58)
               raw_rate = -1 ! rate calculated by special routine.
            case(irni62_ecnp_fe60)
               raw_rate = -1 ! rate calculated by special routine.
            case(irni64_ecnp_fe60) 
               raw_rate = -1 ! rate calculated by special routine.


            case(irfe54an_to_ni58) ! fe54(a,n)ni57(n,g)ni58
               call do1(rate_fe54an_jina)            
            case(irni58gn_to_fe54) ! ni58(g,n)ni57(n,a)fe54
               raw_rate = -1 ! rate calculated by special routine.

            case(irfe54an_to_ni56) ! fe54(a,n)ni57(g,n)ni56
               call do1(rate_fe54an_jina)            
            case(irni56ng_to_fe54) ! ni56(n,g)ni57(n,a)fe54
               raw_rate = -1 ! rate calculated by special routine.

            case(irfe54an_aux) ! fe54(a,n)ni57                        
               call do1(rate_fe54an_jina)
            case(irni57na_aux) ! ni57(n,a)fe54                            
               call do1_reverse(rate_fe54an_jina)

            case(irfe56an_to_ni60) ! fe56(a,n)ni59(n,g)ni60
               call do1(rate_fe56an_jina)            
            case(irni60gn_to_fe56) ! ni60(g,n)ni59(n,a)fe56
               raw_rate = -1 ! rate calculated by special routine.

            case(irfe56an_to_ni58) ! fe56(a,n)ni59(g,n)ni58
               call do1(rate_fe56an_jina)            
            case(irni58ng_to_fe56) ! ni58(n,g)ni59(n,a)fe56
               raw_rate = -1 ! rate calculated by special routine.

            case(irfe56an_aux) ! fe56(a,n)ni59                        
               call do1(rate_fe56an_jina)
            case(irni59na_aux) ! ni59(n,a)fe56                            
               call do1_reverse(rate_fe56an_jina)

            case(irfe58an_to_ni62) ! fe58(a,n)ni61(n,g)ni62
               call do1(rate_fe58an_jina)            
            case(irni62gn_to_fe58) ! ni62(g,n)ni61(n,a)fe58
               raw_rate = -1 ! rate calculated by special routine.

            case(irfe58an_to_ni60) ! fe58(a,n)ni61(g,n)ni60
               call do1(rate_fe58an_jina)            
            case(irni60ng_to_fe58) ! ni60(n,g)ni61(n,a)fe58
               raw_rate = -1 ! rate calculated by special routine.

            case(irfe58an_aux) ! fe58(a,n)ni61                        
               call do1(rate_fe58an_jina)
            case(irni61na_aux) ! ni61(n,a)fe58                            
               call do1_reverse(rate_fe58an_jina)

            case(ircr54an_to_fe58) ! cr54(a,n)fe57(n,g)fe58
               call do1(rate_cr54an_jina)            
            case(irfe58gn_to_cr54) ! fe58(g,n)fe57(n,a)cr54
               raw_rate = -1 ! rate calculated by special routine.

            case(ircr54an_to_fe56) ! cr54(a,n)fe57(g,n)fe56
               call do1(rate_cr54an_jina)            
            case(irfe56ng_to_cr54) ! fe56(n,g)fe57(n,a)cr54
               raw_rate = -1 ! rate calculated by special routine.

            case(ircr54an_aux) ! cr54(a,n)fe57                        
               call do1(rate_cr54an_jina)
            case(irfe57na_aux) ! fe57(n,a)cr54                            
               call do1_reverse(rate_cr54an_jina)

            case(ircr56an_to_fe60) ! cr56(a,n)fe59(n,g)fe60
               call do1(rate_cr56an_jina)            
            case(irfe60gn_to_cr56) ! fe60(g,n)fe59(n,a)cr56
               raw_rate = -1 ! rate calculated by special routine.

            case(ircr56an_to_fe58) ! cr56(a,n)fe59(g,n)fe58
               call do1(rate_cr56an_jina)            
            case(irfe58ng_to_cr56) ! fe58(n,g)fe59(n,a)cr56
               raw_rate = -1 ! rate calculated by special routine.

            case(ircr56an_aux) ! cr56(a,n)fe59                        
               call do1(rate_cr56an_jina)
            case(irfe59na_aux) ! fe59(n,a)cr56                            
               call do1_reverse(rate_cr56an_jina)

            case(ircr56pn_to_fe56) ! cr56(p,n)mn56(p,n)fe56
               call do1(rate_cr56pn_jina)            
            case(irfe56np_to_cr56) ! fe56(n,p)mn56(n,p)cr56
               raw_rate = -1 ! rate calculated by special routine.

            case(ircr56pn_aux) ! cr56(p,n)mn56                        
               call do1(rate_cr56pn_jina)
            case(irmn56np_aux) ! mn56(n,p)cr56                             
               call do1_reverse(rate_cr56pn_jina)
            case(irmn56pn_aux) ! mn56(p,n)fe56                            
               call do1(rate_mn56pn_jina)
            case(irfe56np_aux) ! fe56(n,p)mn56                        
               call do1_reverse(rate_mn56pn_jina)

            case(ircr54pn_to_fe54) ! cr54(p,n)mn54(p,n)fe54
               call do1(rate_cr54pn_jina)            
            case(irfe54np_to_cr54) ! fe54(n,p)mn54(n,p)cr54
               raw_rate = -1 ! rate calculated by special routine.

            case(ircr54pn_aux) ! cr54(p,n)mn54                        
               call do1(rate_cr54pn_jina)
            case(irmn54np_aux) ! mn54(n,p)cr54                             
               call do1_reverse(rate_cr54pn_jina)
            case(irmn54pn_aux) ! mn54(p,n)fe54                            
               call do1(rate_mn54pn_jina)
            case(irfe54np_aux) ! fe54(n,p)mn54                        
               call do1_reverse(rate_mn54pn_jina)

            case(irfe58pn_to_ni58) ! fe58(p,n)co58(p,n)ni58
               call do1(rate_fe58pn_jina)            
            case(irni58np_to_fe58) ! ni58(n,p)co58(n,p)fe58
               raw_rate = -1 ! rate calculated by special routine.

            case(irfe58pn_aux) ! fe58(p,n)co58                        
               call do1(rate_fe58pn_jina)
            case(irco58np_aux) ! co58(n,p)fe58                             
               call do1_reverse(rate_fe58pn_jina)
            case(irco58pn_aux) ! co58(p,n)ni58                            
               call do1(rate_co58pn_jina)
            case(irni58np_aux) ! ni58(n,p)co58                        
               call do1_reverse(rate_co58pn_jina)

            case(irfe56pn_to_ni56) ! fe56(p,n)co56(p,n)ni56
               call do1(rate_fe56pn_jina)            
            case(irni56np_to_fe56) ! ni56(n,p)co56(n,p)fe56
               raw_rate = -1 ! rate calculated by special routine.

            case(irfe56pn_aux) ! fe56(p,n)co56                        
               call do1(rate_fe56pn_jina)
            case(irco56np_aux) ! co56(n,p)fe56                             
               call do1_reverse(rate_fe56pn_jina)
            case(irco56pn_aux) ! co56(p,n)ni56                            
               call do1(rate_co56pn_jina)
            case(irni56np_aux) ! ni56(n,p)co56                        
               call do1_reverse(rate_co56pn_jina)

            case(ircr52ap_to_fe56) ! cr52(a, p)mn55(p, g)fe56
               call do1(rate_cr52ap_jina)                        
            case(irfe56gp_to_cr52) ! fe56(g, p)mn55(p, a)cr52 
               raw_rate = -1 ! rate calculated by special routine.
      
            case(ircr52ap_to_cr54) ! cr52(a, p)mn55(g, p)cr54
               call do1(rate_cr52ap_jina)            
            case(ircr54p_to_cr52) ! cr54(p, g)mn55(p, a)cr52
               raw_rate = -1 ! rate calculated by special routine.

            case(ircr54p_to_fe56) ! cr54(p, g)mn55(p, g)fe56
               raw_rate = -1 ! rate calculated by special routine.                       
            case(irfe56gp_to_cr54) ! fe56(g, p)mn55(g, p)cr54 
               raw_rate = -1 ! rate calculated by special routine.

            case(ircr54pg_aux) ! cr54(p, g)mn55
               call do1(rate_cr54pg_jina)          
            case(irmn55gp_aux) ! mn55(g, p)cr54 
               call do1_reverse(rate_cr54pg_jina)

            case(ircr52ap_aux) ! cr52(a, p)mn55 
               call do1(rate_cr52ap_jina)
            case(irmn55pa_aux) ! mn55(p, a)cr52 
               call do1_reverse(rate_cr52ap_jina)

            case(irfe56gp_aux) ! fe56(g, p)mn55 
               call do1_reverse(rate_mn55pg_jina)            
            case(irmn55pg_aux) ! mn55(p, g)fe56 
               call do1(rate_mn55pg_jina)

            case(irti52ap_to_cr56) ! ti52(a, p)v55(p, g)cr56
               call do1(rate_ti52ap_jina)                        
            case(ircr56gp_to_ti52) ! cr56(g, p)v55(p, a)ti52 
               raw_rate = -1 ! rate calculated by special routine.
      
            case(irti52ap_to_ti54) ! ti52(a, p)v55(g, p)ti54
               call do1(rate_ti52ap_jina)            
            case(irti54p_to_ti52) ! ti54(p, g)v55(p, a)ti52
               raw_rate = -1 ! rate calculated by special routine.

            case(irti54p_to_cr56) ! ti54(p, g)v55(p, g)cr56
               raw_rate = -1 ! rate calculated by special routine.                       
            case(ircr56gp_to_ti54) ! cr56(g, p)v55(g, p)ti54 
               raw_rate = -1 ! rate calculated by special routine.

            case(irti54pg_aux) ! ti54(p, g)v55
               call do1(rate_ti54pg_jina)          
            case(irv55gp_aux) ! v55(g, p)ti54 
               call do1_reverse(rate_ti54pg_jina)

            case(irti52ap_aux) ! ti52(a, p)v55 
               call do1(rate_ti52ap_jina)
            case(irv55pa_aux) ! v55(p, a)ti52 
               call do1_reverse(rate_ti52ap_jina)

            case(ircr56gp_aux) ! cr56(g, p)v55 
               call do1_reverse(rate_v55pg_jina)            
            case(irv55pg_aux) ! v55(p, g)cr56 
               call do1(rate_v55pg_jina)

            case(irti50ap_to_cr54) ! ti50(a, p)v53(p, g)cr54
               call do1(rate_ti50ap_jina)                        
            case(ircr54gp_to_ti50) ! cr54(g, p)v53(p, a)ti50 
               raw_rate = -1 ! rate calculated by special routine.
      
            case(irti50ap_to_ti52) ! ti50(a, p)v53(g, p)ti52
               call do1(rate_ti50ap_jina)            
            case(irti52p_to_ti50) ! ti52(p, g)v53(p, a)ti50
               raw_rate = -1 ! rate calculated by special routine.

            case(irti52p_to_cr54) ! ti52(p, g)v53(p, g)cr54
               raw_rate = -1 ! rate calculated by special routine.                       
            case(ircr54gp_to_ti52) ! cr54(g, p)v53(g, p)ti52 
               raw_rate = -1 ! rate calculated by special routine.

            case(irti52pg_aux) ! ti52(p, g)v53
               call do1(rate_ti52pg_jina)          
            case(irv53gp_aux) ! v53(g, p)ti52 
               call do1_reverse(rate_ti52pg_jina)

            case(irti50ap_aux) ! ti50(a, p)v53 
               call do1(rate_ti50ap_jina)
            case(irv53pa_aux) ! v53(p, a)ti50 
               call do1_reverse(rate_ti50ap_jina)

            case(ircr54gp_aux) ! cr54(g, p)v53 
               call do1_reverse(rate_v53pg_jina)            
            case(irv53pg_aux) ! v53(p, g)cr54 
               call do1(rate_v53pg_jina)

            case(irti48ap_to_cr52) ! ti48(a, p)v51(p, g)cr52
               call do1(rate_ti48ap_jina)                        
            case(ircr52gp_to_ti48) ! cr52(g, p)v51(p, a)ti48 
               raw_rate = -1 ! rate calculated by special routine.
      
            case(irti48ap_to_ti50) ! ti48(a, p)v51(g, p)ti50
               call do1(rate_ti48ap_jina)            
            case(irti50p_to_ti48) ! ti50(p, g)v51(p, a)ti48
               raw_rate = -1 ! rate calculated by special routine.

            case(irti50p_to_cr52) ! ti50(p, g)v51(p, g)cr52
               raw_rate = -1 ! rate calculated by special routine.                       
            case(ircr52gp_to_ti50) ! cr52(g, p)v51(g, p)ti50 
               raw_rate = -1 ! rate calculated by special routine.

            case(irti50pg_aux) ! ti50(p, g)v51
               call do1(rate_ti50pg_jina)          
            case(irv51gp_aux) ! v51(g, p)ti50 
               call do1_reverse(rate_ti50pg_jina)

            case(irti48ap_aux) ! ti48(a, p)v51 
               call do1(rate_ti48ap_jina)
            case(irv51pa_aux) ! v51(p, a)ti48 
               call do1_reverse(rate_ti48ap_jina)

            case(ircr52gp_aux) ! cr52(g, p)v51 
               call do1_reverse(rate_v51pg_jina)            
            case(irv51pg_aux) ! v51(p, g)cr52 
               call do1(rate_v51pg_jina)
      
            case(irti46ap_to_ti48) ! ti46(a, p)v49(g, p)ti48
               call do1(rate_ti46ap_jina)            
            case(irti48p_to_ti46) ! ti48(p, g)v49(p, a)ti46
               raw_rate = -1 ! rate calculated by special routine.

            case(ircr52an_to_fe56) ! cr52(a,n)fe55(n,g)fe56
               call do1(rate_cr52an_jina)            
            case(irfe56gn_to_cr52) ! fe56(g,n)fe55(n,a)cr52
               raw_rate = -1 ! rate calculated by special routine.

            case(ircr52an_to_fe54) ! cr52(a,n)fe55(g,n)fe54
               call do1(rate_cr52an_jina)            
            case(irfe54ng_to_cr52) ! fe54(n,g)fe55(n,a)cr52
               raw_rate = -1 ! rate calculated by special routine.

            case(ircr52an_aux) ! cr52(a,n)fe55                        
               call do1(rate_cr52an_jina)
            case(irfe55na_aux) ! fe55(n,a)cr52                            
               call do1_reverse(rate_cr52an_jina)

            case(ircr50an_to_fe54) ! cr50(a,n)fe53(n,g)fe54
               call do1(rate_cr50an_jina)            
            case(irfe54gn_to_cr50) ! fe54(g,n)fe53(n,a)cr50
               raw_rate = -1 ! rate calculated by special routine.

            case(ircr50an_to_fe52) ! cr50(a,n)fe53(g,n)fe52
               call do1(rate_cr50an_jina)            
            case(irfe52ng_to_cr50) ! fe52(n,g)fe53(n,a)cr50
               raw_rate = -1 ! rate calculated by special routine.

            case(ircr50an_aux) ! cr50(a,n)fe53                        
               call do1(rate_cr50an_jina)
            case(irfe53na_aux) ! fe53(n,a)cr50                            
               call do1_reverse(rate_cr50an_jina)

            case(irti52an_to_cr56) ! ti52(a,n)cr55(n,g)cr56
               call do1(rate_ti52an_jina)            
            case(ircr56gn_to_ti52) ! cr56(g,n)cr55(n,a)ti52
               raw_rate = -1 ! rate calculated by special routine.

            case(irti52an_to_cr54) ! ti52(a,n)cr55(g,n)cr54
               call do1(rate_ti52an_jina)            
            case(ircr54ng_to_ti52) ! cr54(n,g)cr55(n,a)ti52
               raw_rate = -1 ! rate calculated by special routine.

            case(irti52an_aux) ! ti52(a,n)cr55                        
               call do1(rate_ti52an_jina)
            case(ircr55na_aux) ! cr55(n,a)ti52                            
               call do1_reverse(rate_ti52an_jina)

            case(irti50an_to_cr54) ! ti50(a,n)cr53(n,g)cr54
               call do1(rate_ti50an_jina)            
            case(ircr54gn_to_ti50) ! cr54(g,n)cr53(n,a)ti50
               raw_rate = -1 ! rate calculated by special routine.

            case(irti50an_to_cr52) ! ti50(a,n)cr53(g,n)cr52
               call do1(rate_ti50an_jina)            
            case(ircr52ng_to_ti50) ! cr52(n,g)cr53(n,a)ti50
               raw_rate = -1 ! rate calculated by special routine.

            case(irti50an_aux) ! ti50(a,n)cr53                        
               call do1(rate_ti50an_jina)
            case(ircr53na_aux) ! cr53(n,a)ti50                            
               call do1_reverse(rate_ti50an_jina)

            case(irti48an_to_cr52) ! ti48(a,n)cr51(n,g)cr52
               call do1(rate_ti48an_jina)            
            case(ircr52gn_to_ti48) ! cr52(g,n)cr51(n,a)ti48
               raw_rate = -1 ! rate calculated by special routine.

            case(irti48an_to_cr50) ! ti48(a,n)cr51(g,n)cr50
               call do1(rate_ti48an_jina)            
            case(ircr50ng_to_ti48) ! cr50(n,g)cr51(n,a)ti48
               raw_rate = -1 ! rate calculated by special routine.

            case(irti48an_aux) ! ti48(a,n)cr51                        
               call do1(rate_ti48an_jina)
            case(ircr51na_aux) ! cr51(n,a)ti48                            
               call do1_reverse(rate_ti48an_jina)

            case(irti46an_to_cr50) ! ti46(a,n)cr49(n,g)cr50
               call do1(rate_ti46an_jina)            
            case(ircr50gn_to_ti46) ! cr50(g,n)cr49(n,a)ti46
               raw_rate = -1 ! rate calculated by special routine.

            case(irti46an_to_cr48) ! ti46(a,n)cr49(g,n)cr48
               call do1(rate_ti46an_jina)            
            case(ircr48ng_to_ti46) ! cr48(n,g)cr49(n,a)ti46
               raw_rate = -1 ! rate calculated by special routine.

            case(irti46an_aux) ! ti46(a,n)cr49                        
               call do1(rate_ti46an_jina)
            case(ircr49na_aux) ! cr49(n,a)ti46                            
               call do1_reverse(rate_ti46an_jina)

            case(ircr52pn_to_fe52) ! cr52(p,n)mn52(p,n)fe52
               call do1(rate_cr52pn_jina)            
            case(irfe52np_to_cr52) ! fe52(n,p)mn52(n,p)cr52
               raw_rate = -1 ! rate calculated by special routine.

            case(ircr52pn_aux) ! cr52(p,n)mn52                        
               call do1(rate_cr52pn_jina)
            case(irmn52np_aux) ! mn52(n,p)cr52                             
               call do1_reverse(rate_cr52pn_jina)
            case(irmn52pn_aux) ! mn52(p,n)fe52                            
               call do1(rate_mn52pn_jina)
            case(irfe52np_aux) ! fe52(n,p)mn52                        
               call do1_reverse(rate_mn52pn_jina)

            case(irti52pn_to_cr52) ! ti52(p,n)v52(p,n)cr52
               call do1(rate_ti52pn_jina)            
            case(ircr52np_to_ti52) ! cr52(n,p)v52(n,p)ti52
               raw_rate = -1 ! rate calculated by special routine.

            case(irti52pn_aux) ! ti52(p,n)v52                        
               call do1(rate_ti52pn_jina)
            case(irv52np_aux) ! v52(n,p)ti52                             
               call do1_reverse(rate_ti52pn_jina)
            case(irv52pn_aux) ! v52(p,n)cr52                            
               call do1(rate_v52pn_jina)
            case(ircr52np_aux) ! cr52(n,p)v52                        
               call do1_reverse(rate_v52pn_jina)

            case(irti50pn_to_cr50) ! ti50(p,n)v50(p,n)cr50
               call do1(rate_ti50pn_jina)            
            case(ircr50np_to_ti50) ! cr50(n,p)v50(n,p)ti50
               raw_rate = -1 ! rate calculated by special routine.

            case(irti50pn_aux) ! ti50(p,n)v50                        
               call do1(rate_ti50pn_jina)
            case(irv50np_aux) ! v50(n,p)ti50                             
               call do1_reverse(rate_ti50pn_jina)
            case(irv50pn_aux) ! v50(p,n)cr50                            
               call do1(rate_v50pn_jina)
            case(ircr50np_aux) ! cr50(n,p)v50                        
               call do1_reverse(rate_v50pn_jina)

            case(irti48pn_to_cr48) ! ti48(p,n)v48(p,n)cr48
               call do1(rate_ti48pn_jina)            
            case(ircr48np_to_ti48) ! cr48(n,p)v48(n,p)ti48
               raw_rate = -1 ! rate calculated by special routine.

            case(irti48pn_aux) ! ti48(p,n)v48                        
               call do1(rate_ti48pn_jina)
            case(irv48np_aux) ! v48(n,p)ti48                             
               call do1_reverse(rate_ti48pn_jina)
            case(irv48pn_aux) ! v48(p,n)cr48                            
               call do1(rate_v48pn_jina)
            case(ircr48np_aux) ! cr48(n,p)v48                        
               call do1_reverse(rate_v48pn_jina)

            case(irti44an_to_cr48) ! ti44(a,n)cr47(n,g)cr48
               call do1(rate_ti44an_jina)            
            case(ircr48gn_to_ti44) ! cr48(g,n)cr47(n,a)ti44
               raw_rate = -1 ! rate calculated by special routine.

            case(irti44an_to_cr46) ! ti44(a,n)cr47(g,n)cr46
               call do1(rate_ti44an_jina)            
            case(ircr46ng_to_ti44) ! cr46(n,g)cr47(n,a)ti44
               raw_rate = -1 ! rate calculated by special routine.

            case(irti44an_aux) ! ti44(a,n)cr47                        
               call do1(rate_ti44an_jina)
            case(ircr47na_aux) ! cr47(n,a)ti44                            
               call do1_reverse(rate_ti44an_jina)

            
            case default
               call do_default(ierr)
               
         end select
         
         
         contains


         subroutine do_default(ierr) 
            integer, intent(out) :: ierr ! set ierr to -1 if cannot find rate                  	   
      	   real(dp) :: lambda, dlambda_dlnT, rlambda, drlambda_dlnT        	   
      	   include 'formats.dek'           
            ierr = 0            
            if (raw_rates_records(ir)% use_rate_table) then ! use the table
               call eval_table(ir, tf, temp, raw_rate, rlambda, ierr)
               return
            end if           
            ! look for rate in reaclib
            call get_reaclib_rate_and_dlnT(
     >         ir, temp, lambda, dlambda_dlnT, rlambda, drlambda_dlnT, ierr)
            raw_rate = lambda
         end subroutine do_default


         subroutine do1(rate_fcn)
            interface
               subroutine rate_fcn(tf, temp, fr, rr)
                  use const_def, only: dp
                  use ratelib, only: T_factors
                  type (T_Factors), pointer :: tf
                  real(dp), intent(in) :: temp
                  real(dp), intent(out) :: fr, rr
               end subroutine rate_fcn
            end interface
            call eval_raw_rate(ir, rate_fcn, tf, temp, raw_rate, rr, ierr)
         end subroutine do1
         
         
         subroutine do1_reverse(rate_fcn)
            interface
               subroutine rate_fcn(tf, temp, fr, rr)
                  use const_def, only: dp
                  use ratelib, only: T_factors
                  type (T_Factors), pointer :: tf
                  real(dp), intent(in) :: temp
                  real(dp), intent(out) :: fr, rr
               end subroutine rate_fcn
            end interface
            call eval_raw_rate(ir, rate_fcn, tf, temp, rr, raw_rate, ierr)
         end subroutine do1_reverse
         
         
         subroutine do1_of_2(rate_fcn1, rate_fcn2)
            interface
               subroutine rate_fcn1(tf, temp, fr, rr)
                  use const_def, only: dp
                  use ratelib, only: T_factors
                  type (T_Factors), pointer :: tf
                  real(dp), intent(in) :: temp
                  real(dp), intent(out) :: fr, rr
               end subroutine rate_fcn1
               subroutine rate_fcn2(tf, temp, fr, rr)
                  use const_def, only: dp
                  use ratelib, only: T_factors
                  type (T_Factors), pointer :: tf
                  real(dp), intent(in) :: temp
                  real(dp), intent(out) :: fr, rr
               end subroutine rate_fcn2
            end interface
            call eval_which_raw_rate( 
     >         ir, min(2,which_rate), 
     >         rate_fcn1, rate_fcn2, rate_fcn_null, rate_fcn_null, 
     >         tf, temp, raw_rate, rr, ierr)
         end subroutine do1_of_2
         
         
         subroutine do1_of_2_reverse(rate_fcn1, rate_fcn2)
            interface
               subroutine rate_fcn1(tf, temp, fr, rr)
                  use const_def, only: dp
                  use ratelib, only: T_factors
                  type (T_Factors), pointer :: tf
                  real(dp), intent(in) :: temp
                  real(dp), intent(out) :: fr, rr
               end subroutine rate_fcn1
               subroutine rate_fcn2(tf, temp, fr, rr)
                  use const_def, only: dp
                  use ratelib, only: T_factors
                  type (T_Factors), pointer :: tf
                  real(dp), intent(in) :: temp
                  real(dp), intent(out) :: fr, rr
               end subroutine rate_fcn2
            end interface
            real(dp) :: r
            call do1_of_2(rate_fcn1, rate_fcn2)
            r = raw_rate; raw_rate = rr; rr = r
         end subroutine do1_of_2_reverse
         
         
         subroutine do1_of_3(rate_fcn1, rate_fcn2, rate_fcn3)
            interface
               subroutine rate_fcn1(tf, temp, fr, rr)
                  use const_def, only: dp
                  use ratelib, only: T_factors
                  type (T_Factors), pointer :: tf
                  real(dp), intent(in) :: temp
                  real(dp), intent(out) :: fr, rr
               end subroutine rate_fcn1
               subroutine rate_fcn2(tf, temp, fr, rr)
                  use const_def, only: dp
                  use ratelib, only: T_factors
                  type (T_Factors), pointer :: tf
                  real(dp), intent(in) :: temp
                  real(dp), intent(out) :: fr, rr
               end subroutine rate_fcn2
               subroutine rate_fcn3(tf, temp, fr, rr)
                  use const_def, only: dp
                  use ratelib, only: T_factors
                  type (T_Factors), pointer :: tf
                  real(dp), intent(in) :: temp
                  real(dp), intent(out) :: fr, rr
               end subroutine rate_fcn3
            end interface
            call eval_which_raw_rate( 
     >         ir, min(3,which_rate), 
     >         rate_fcn1, rate_fcn2, rate_fcn3, rate_fcn_null, 
     >         tf, temp, raw_rate, rr, ierr)
         end subroutine do1_of_3
         
         
         subroutine do1_of_3_reverse(rate_fcn1, rate_fcn2, rate_fcn3)
            interface
               subroutine rate_fcn1(tf, temp, fr, rr)
                  use const_def, only: dp
                  use ratelib, only: T_factors
                  type (T_Factors), pointer :: tf
                  real(dp), intent(in) :: temp
                  real(dp), intent(out) :: fr, rr
               end subroutine rate_fcn1
               subroutine rate_fcn2(tf, temp, fr, rr)
                  use const_def, only: dp
                  use ratelib, only: T_factors
                  type (T_Factors), pointer :: tf
                  real(dp), intent(in) :: temp
                  real(dp), intent(out) :: fr, rr
               end subroutine rate_fcn2
               subroutine rate_fcn3(tf, temp, fr, rr)
                  use const_def, only: dp
                  use ratelib, only: T_factors
                  type (T_Factors), pointer :: tf
                  real(dp), intent(in) :: temp
                  real(dp), intent(out) :: fr, rr
               end subroutine rate_fcn3
            end interface
            real(dp) :: r
            call do1_of_3(rate_fcn1, rate_fcn2, rate_fcn3)
            r = raw_rate; raw_rate = rr; rr = r
         end subroutine do1_of_3_reverse
         
         
         subroutine do1_of_4(rate_fcn1, rate_fcn2, rate_fcn3, rate_fcn4)
            interface
               subroutine rate_fcn1(tf, temp, fr, rr)
                  use const_def, only: dp
                  use ratelib, only: T_factors
                  type (T_Factors), pointer :: tf
                  real(dp), intent(in) :: temp
                  real(dp), intent(out) :: fr, rr
               end subroutine rate_fcn1
               subroutine rate_fcn2(tf, temp, fr, rr)
                  use const_def, only: dp
                  use ratelib, only: T_factors
                  type (T_Factors), pointer :: tf
                  real(dp), intent(in) :: temp
                  real(dp), intent(out) :: fr, rr
               end subroutine rate_fcn2
               subroutine rate_fcn3(tf, temp, fr, rr)
                  use const_def, only: dp
                  use ratelib, only: T_factors
                  type (T_Factors), pointer :: tf
                  real(dp), intent(in) :: temp
                  real(dp), intent(out) :: fr, rr
               end subroutine rate_fcn3
               subroutine rate_fcn4(tf, temp, fr, rr)
                  use const_def, only: dp
                  use ratelib, only: T_factors
                  type (T_Factors), pointer :: tf
                  real(dp), intent(in) :: temp
                  real(dp), intent(out) :: fr, rr
               end subroutine rate_fcn4
            end interface
            call eval_which_raw_rate( 
     >         ir, min(4,which_rate), 
     >         rate_fcn1, rate_fcn2, rate_fcn3, rate_fcn4, 
     >         tf, temp, raw_rate, rr, ierr)
         end subroutine do1_of_4
         
         
         subroutine do1_of_4_reverse(rate_fcn1, rate_fcn2, rate_fcn3, rate_fcn4)
            interface
               subroutine rate_fcn1(tf, temp, fr, rr)
                  use const_def, only: dp
                  use ratelib, only: T_factors
                  type (T_Factors), pointer :: tf
                  real(dp), intent(in) :: temp
                  real(dp), intent(out) :: fr, rr
               end subroutine rate_fcn1
               subroutine rate_fcn2(tf, temp, fr, rr)
                  use const_def, only: dp
                  use ratelib, only: T_factors
                  type (T_Factors), pointer :: tf
                  real(dp), intent(in) :: temp
                  real(dp), intent(out) :: fr, rr
               end subroutine rate_fcn2
               subroutine rate_fcn3(tf, temp, fr, rr)
                  use const_def, only: dp
                  use ratelib, only: T_factors
                  type (T_Factors), pointer :: tf
                  real(dp), intent(in) :: temp
                  real(dp), intent(out) :: fr, rr
               end subroutine rate_fcn3
               subroutine rate_fcn4(tf, temp, fr, rr)
                  use const_def, only: dp
                  use ratelib, only: T_factors
                  type (T_Factors), pointer :: tf
                  real(dp), intent(in) :: temp
                  real(dp), intent(out) :: fr, rr
               end subroutine rate_fcn4
            end interface
            real(dp) :: r
            call do1_of_4(rate_fcn1, rate_fcn2, rate_fcn3, rate_fcn4)
            r = raw_rate; raw_rate = rr; rr = r
         end subroutine do1_of_4_reverse


      end subroutine set_raw_rate


      subroutine rate_from_table(tf, temp, fr, rr)
         type (T_Factors), pointer :: tf
         real(dp), intent(in) :: temp
         real(dp), intent(out) :: fr, rr
         fr = missing_value; rr = fr
      end subroutine rate_from_table
      

      subroutine rate_fcn_null(tf, temp, fr, rr)
         use ratelib, only: T_factors
         type (T_Factors), pointer :: tf
         real(dp), intent(in) :: temp
         real(dp), intent(out) :: fr, rr
         fr = -1; rr = -1
      end subroutine rate_fcn_null


      subroutine eval_which_raw_rate( 
     >      ir, which_rate, 
     >      rate_fcn1, rate_fcn2, rate_fcn3, rate_fcn4, 
     >      tf, temp, fr, rr, ierr)
         use interp_1d_lib, only: interp_values
         use ratelib
         use const_def, only: pi
         integer, intent(in) :: ir, which_rate
         interface
            subroutine rate_fcn1(tf, temp, fr, rr)
               use const_def, only: dp
                  use ratelib, only: T_factors
               type (T_Factors), pointer :: tf
               real(dp), intent(in) :: temp
               real(dp), intent(out) :: fr, rr
            end subroutine rate_fcn1
            subroutine rate_fcn2(tf, temp, fr, rr)
               use const_def, only: dp
                  use ratelib, only: T_factors
               type (T_Factors), pointer :: tf
               real(dp), intent(in) :: temp
               real(dp), intent(out) :: fr, rr
            end subroutine rate_fcn2
            subroutine rate_fcn3(tf, temp, fr, rr)
               use const_def, only: dp
                  use ratelib, only: T_factors
               type (T_Factors), pointer :: tf
               real(dp), intent(in) :: temp
               real(dp), intent(out) :: fr, rr
            end subroutine rate_fcn3
            subroutine rate_fcn4(tf, temp, fr, rr)
               use const_def, only: dp
                  use ratelib, only: T_factors
               type (T_Factors), pointer :: tf
               real(dp), intent(in) :: temp
               real(dp), intent(out) :: fr, rr
            end subroutine rate_fcn4
         end interface
         type (T_Factors), pointer :: tf
         real(dp), intent(in) :: temp
         real(dp), intent(out) :: fr, rr
         integer, intent(out) :: ierr
         real(dp) :: fr1, rr1, alfa, beta
         include 'formats.dek'
         ierr = 0
         if (which_rate == 2 .and. temp < JR_T_full_on) then
            call eval_raw_rate(ir, rate_fcn1, tf, temp, fr1, rr1, ierr)
            if (ierr /= 0) return
            if (temp <= JR_T_full_off) then
               fr = fr1; rr = rr1; return
            end if
         end if
         select case(which_rate)
            case (1)
               call eval_raw_rate(ir, rate_fcn1, tf, temp, fr, rr, ierr)
            case (2)
               call eval_raw_rate(ir, rate_fcn2, tf, temp, fr, rr, ierr)
            case (3)
               call eval_raw_rate(ir, rate_fcn3, tf, temp, fr, rr, ierr)
            case (4)
               call eval_raw_rate(ir, rate_fcn4, tf, temp, fr, rr, ierr)
            case default         
               write(*,*) 'bad which rate ' // trim(reaction_Name(ir)), which_rate
               ierr = -1
               return
         end select
         if (which_rate == 2 .and. temp < JR_T_full_on) then ! blend 1 and 2
         	! alfa = fraction from rate_fcn2; beta = 1-alfa = fraction from rate_fcn1
      	   alfa = (temp - JR_T_full_off) / (JR_T_full_on - JR_T_full_off)
      	   alfa = 0.5d0*(1 - cos(alfa*pi))
         	beta = 1 - alfa
         	fr = beta*fr1 + alfa*fr
         	rr = beta*rr1 + alfa*rr
         end if
      end subroutine eval_which_raw_rate


      subroutine eval_raw_rate(ir, rate_fcn, tf, temp, fr, rr, ierr)
         use ratelib
         integer, intent(in) :: ir ! reaction id
         interface
            subroutine rate_fcn(tf, temp, fr, rr)
               use const_def, only: dp
                  use ratelib, only: T_factors
               type (T_Factors), pointer :: tf
               real(dp), intent(in) :: temp
               real(dp), intent(out) :: fr, rr
            end subroutine rate_fcn
         end interface
         type (T_Factors), pointer :: tf
         real(dp), intent(in) :: temp
         real(dp), intent(out) :: fr, rr
         integer, intent(out) :: ierr
         include 'formats.dek'
         ierr = 0
         if (raw_rates_records(ir)% use_rate_table) then ! use the table
            call eval_table(ir, tf, temp, fr, rr, ierr)
            return
         end if
         call rate_fcn(tf, temp, fr, rr)
         if (fr < 0 .or. rr < 0) then
            write(*,1) 'invalid which rate for ' // trim(reaction_Name(ir)), fr, rr
            ierr = -1
            return
         end if
         if (fr == missing_value .or. rr == missing_value) then
            write(*,1) 'missing value for ' // trim(reaction_Name(ir)), fr, rr
            ierr = -1
            return
         end if
      end subroutine eval_raw_rate
      
      
      subroutine eval_table(ir, tf, temp, fr, rr, ierr)
         use interp_1d_lib, only: interp_values
         use ratelib
         integer, intent(in) :: ir ! reaction id
         type (T_Factors), pointer :: tf
         real(dp), intent(in) :: temp
         real(dp), intent(out) :: fr, rr
         integer, intent(out) :: ierr
         integer, parameter :: nv = 1
         real(dp) :: x(nv), vals(nv)
         type (rate_table_info), pointer :: ri
         include 'formats.dek'
         ierr = 0
         ri => raw_rates_records(ir)
         if (.not. ri% use_rate_table) then
            ierr = -1
            return
         end if         
         if (ri% need_to_read) then
!$omp critical (load_rate_table)
            if (ri% need_to_read) then
               call get_interp_table(ri% rate_fname, ri% nT8s, ri% T8s, ri% f1, ierr)
               ri% need_to_read = .false.
            end if
!$omp end critical (load_rate_table)
            if (ierr /= 0) then
               write(*,*) 'failed to load table ' // trim(ri% rate_fname)
               return
            end if
         end if        
         x(1) = temp*1d-8
         call interp_values(ri% T8s, ri% nT8s, ri% f1, nv, x, vals, ierr)
         fr = vals(1)
         rr = 0 ! no reverse rates for tables
      end subroutine eval_table
      
      
      subroutine get_interp_table(f_name, nT8s, T8s_out, f1_out, ierr)
         use interp_1d_def, only: pm_work_size
         use interp_1d_lib, only: interp_pm
         use utils_lib
         character (len=*), intent(in) :: f_name
         integer, intent(out) :: nT8s
         real(dp), pointer :: T8s_out(:) ! will be allocated.  (nT8s)
         real(dp), pointer :: f1_out(:) ! will be allocated.  (4,nT8s)
         integer, intent(out) :: ierr
         
         integer :: iounit, j
         real(dp) :: tmp
         real(dp), pointer :: work(:)
         real(dp), pointer :: T8s(:)
         real(dp), pointer :: f1(:), f(:,:)
         character (len=256) :: line, rate_file
         
         ierr = 0
         iounit = alloc_iounit(ierr)
         if (ierr /= 0) return

         rate_file = trim(rates_table_dir) // '/' // trim(f_name)
         !write(*,*) 'load table ' // trim(rate_file)
         
         open(unit=iounit,file=trim(rate_file),action='read',status='old',iostat=ierr)
         if (ierr /= 0) then
            write(*,*) 'ERROR: cannot open rate info file ' // trim(rate_file)
            !return
            stop 1
         end if

         do ! read until reach line starting with an integer (nT8s)
            ierr = 0
            read(iounit, fmt=*, iostat=ierr) nT8s
            if (ierr == 0) exit
         end do
         if (failed('skip to find line starting with an integer for nT8s')) return

         allocate(T8s(nT8s), f1(4*nT8s), stat=ierr)
         if (failed('allocate')) return
         f(1:4,1:nT8s) => f1(1:4*nT8s)
         
         do j=1,nT8s
            read(iounit, fmt=*, iostat=ierr) T8s(j), tmp
            if (failed('read data')) return
            f(1,j) = tmp
         end do
         
         allocate(work(nT8s*pm_work_size), stat=ierr)
         if (failed('allocate')) return

         call interp_pm(T8s, nT8s, f1, pm_work_size, work, ierr)
         deallocate(work)
         
         if (failed('interp_pm')) return
         
         close(iounit)
         call free_iounit(iounit)
         
         ! don't set the pointers until have finished setting up the data
         
         if (associated(T8s_out)) deallocate(T8s_out)
         if (associated(f1_out)) deallocate(f1_out)

         T8s_out => T8s
         f1_out => f1
         
         contains
         
         logical function failed(str)
            character (len=*), intent(in) :: str
            failed = .false.
            if (ierr == 0) return
            close(iounit)
            call free_iounit(iounit)
            write(*,*) trim(str) // ' failed in reading ' // trim(rate_file)
            failed = .true.
            return
         end function failed
      
      end subroutine get_interp_table
      

      subroutine get_reaclib_rate_and_dlnT(
     >      ir, temp, lambda, dlambda_dlnT, rlambda, drlambda_dlnT, ierr)
         use ratelib
         integer, intent(in) :: ir
         real(dp), intent(in) :: temp
         real(dp), intent(out) :: lambda, dlambda_dlnT, rlambda, drlambda_dlnT
         integer, intent(out) :: ierr
      	integer :: reverse
      	
      	include 'formats'
         ierr = 0
      	
      	reverse = reaction_is_reverse(ir)
      	if (reverse == 0) then ! that means don't know
            reverse = reaclib_reverse(reaction_Name(ir))
            if (reverse > 0) then
               reaction_is_reverse(ir) = reverse
            else
               reaction_is_reverse(ir) = -1
            end if
         else if (reverse == -1) then ! that means reaclib_reverse returned 0
            reverse = 0
      	end if
         if (reverse > 0) then
            call do_jina_reaclib_reverse(reaclib_rates% reaction_handle(reverse))
         else
            !write(*,1) 'call do_jina_reaclib ' // trim(reaction_Name(ir))
            call do_jina_reaclib
         end if 
         
         return
         
         write(*,1) 'temp', temp
         write(*,1) 'lambda', lambda
         write(*,1) 'dlambda_dlnT', dlambda_dlnT
         write(*,1) 'rlambda', rlambda
         write(*,1) 'drlambda_dlnT', drlambda_dlnT
         
         
         contains
         
         
         subroutine get_reaclib_lo_hi(ir, handle, lo, hi, ierr)
            use reaclib_lib, only: reaclib_indices_for_reaction
            integer, intent(in) :: ir
            character (len=*) :: handle
            integer, intent(out) :: lo, hi, ierr
            include 'formats'
            lo = reaction_reaclib_lo(ir)
            hi = reaction_reaclib_hi(ir)
            if (lo > 0 .and. hi > 0) return
            ierr = 0
            call reaclib_indices_for_reaction(handle, reaclib_rates, lo, hi, ierr)
            if (ierr /= 0) return
            reaction_reaclib_lo(ir) = lo
            reaction_reaclib_hi(ir) = hi
         end subroutine get_reaclib_lo_hi
         
         
         subroutine do_jina_reaclib
            integer :: ierr, lo, hi
            ierr = 0
            call get_reaclib_lo_hi(ir, reaction_Name(ir), lo, hi, ierr)
            if (ierr /= 0) then
               write(*,'(a,3x,i5)') 
     >            trim(reaction_Name(ir)) // ' failed in do_jina_reaclib', ir
               stop 'raw_rates'
               return
            end if
            call reaclib_rate_and_dlnT(
     >         lo, hi, reaction_Name(ir), temp*1d-9, 
     >         lambda, dlambda_dlnT, rlambda, drlambda_dlnT, ierr)
            if (ierr /= 0) then
               write(*,'(a,3x,i5)') 
     >            trim(reaction_Name(ir)) // ' failed in get_reaclib_rate_and_dlnT', ir
               return
            end if
         end subroutine do_jina_reaclib
         
         
         subroutine do_jina_reaclib_reverse(reverse_handle)
            character (len=*) :: reverse_handle
            integer :: ierr, lo, hi, r_id
      	   include 'formats.dek'
      	   ierr = 0
      	   r_id = reverse_reaction_id(ir)
      	   if (r_id == 0) then ! don't know
      	      r_id = get_rates_reaction_id(reverse_handle)
      	      if (r_id == 0) then 
                  write(*,'(a,3x,i5)') 
     >               trim(reverse_handle) // ' failed in reaclib_index', r_id
                  stop 'raw_rates'
      	      end if
      	      reverse_reaction_id(ir) = r_id
      	   end if
            call get_reaclib_lo_hi(r_id, reverse_handle, lo, hi, ierr)
            if (ierr /= 0) then
               write(*,'(a,3x,i5)') 
     >            trim(reverse_handle) // ' failed in do_jina_reaclib_reverse', r_id
               stop 'raw_rates'
               return
            end if
            call reaclib_rate_and_dlnT(
     >         lo, hi, reverse_handle, temp*1d-9, 
     >         rlambda, drlambda_dlnT, lambda, dlambda_dlnT, ierr)
            if (ierr /= 0) then
               write(*,'(a)') trim(reverse_handle) // ' failed in get_reaclib_rate_and_dlnT'
               return
            end if
         end subroutine do_jina_reaclib_reverse

                
      end subroutine get_reaclib_rate_and_dlnT




      end module raw_rates

