c ***********************************************************************
!
!   Copyright (C) 2008  Bill Paxton
!
!   This file is part of MEBDFI_95.
!
!   MEBDFI_95 is free software; you can redistribute it and/or modify
!   it under the terms of the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License, or
!   (at your option) any later version.
!
!   MEBDFI_95 is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!   GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
c ***********************************************************************


		module test_mebdfi_pollu

		use mebdfi_solver_def
		use num_lib, only: MEBDFI_95_ibvp, null_monitor
		
		implicit none
		
		integer, parameter :: num_test_ipars = 0
		integer, parameter :: num_test_rpars = 0
		
		
		contains
		
		
      subroutine do_mebdfi_pollu_test(verbose)
			
			implicit none
			
			logical, intent(in) :: verbose
	
			integer, parameter :: m = 20, n = 1, neqns = m*n

			double precision :: Z(m,n), Zprime(m,n), rdata(num_rdata), rpar(num_test_rpars)
     		double precision :: t0, tend, h0, hmax, abs_tol(m), rel_tol(m), scd, inf_nrms(4)
			integer :: max_order, max_steps, idata(num_idata), ipar(num_test_ipars), max_n,
     >			stencil_pts_lower, stencil_pts_higher, ierr, i, lout, verbosity, stage2_min
			logical :: numerical_jacobian, jdebug, equilibrate
			
			scd = 0; scd = scd ! for compiler
			
			write(*,*) 'test mebdfi pollu'
			
			t0 = 0
			tend = 60d0
			abs_tol = 1d-7
			rel_tol = abs_tol
			h0 = 1d-7
			hmax = 0
			
			inf_nrms = 1
			
			max_order = 7
			max_steps = 10000
			stencil_pts_lower = 0
			stencil_pts_higher = 0
			stage2_min = 1
			numerical_jacobian = .false.
			jdebug = .false.
			equilibrate = .false.
		

			call pollu_init(m,Z(:,1),Zprime(:,1))
			
			lout = 6
			verbosity = 0
			
			call MEBDFI_95_ibvp(
     >		m, n, Z, Zprime, 
     >		t0, tend, h0, hmax, abs_tol, rel_tol, inf_nrms,
     >      max_order, max_steps,
     >		num_test_rpars, rpar, num_test_ipars, ipar,
     >		num_rdata, rdata, num_idata, idata,
     >		stencil_pts_lower, stencil_pts_higher,
     >		eval_resid, call_back, eval_pderv, null_monitor,
     >		numerical_jacobian, equilibrate, stage2_min, jdebug, 
     >		lout, verbosity, ierr)	
     
     		if (ierr /= 0) stop 1

			if (verbose) call report(m, n, Z, num_idata, idata, abs_tol, rel_tol, scd)
			
			write(*,*)
			do i = 1, m
			   if (Z(i,1) < 1d-6) cycle
				write(*,'(i6,1pe24.16)') i, Z(i,1)
			end do
			write(*,*)

      end subroutine do_mebdfi_pollu_test


      subroutine pollu_init(neqn,y,yprime)
	      integer, intent(in) :: neqn
	      double precision, intent(out) :: y(neqn), yprime(neqn)
         y = 0d0
         y(2)  = 0.2d0
         y(4)  = 0.04d0
         y(7)  = 0.1d0
         y(8)  = 0.3d0
         y(9)  = 0.01d0
         y(17) = 0.007d0
			yprime = 0
      end subroutine pollu_init


		subroutine eval_resid(neqns,time,h,con,
     >		   y,delta,yprime,num_rpar,rpar,num_ipar,ipar,stage,iteration,ierr)
     		integer, intent(in) :: neqns, num_ipar, num_rpar, stage, iteration
     		double precision, intent(in) :: time, h, con, y(neqns), yprime(neqns)
     		double precision, intent(inout) :: rpar(num_rpar)
     		double precision, intent(out) :: delta(neqns)
     		integer, intent(inout) :: ipar(num_ipar)
			integer, intent(out) :: ierr
			
			integer :: i
			
			ierr = 0
			
			! delta = dydt - yprime, where dydt comes from feval
			
			call pollu_feval(neqns,y,delta) ! store dydt in delta
			
			do i=1,neqns
			   delta(i) = delta(i) - yprime(i)
			end do
		
		end subroutine eval_resid
      
      
      subroutine pollu_feval(neqn,y,f)
      integer neqn
      double precision y(neqn),f(neqn)

      double precision k1,k2,k3,k4,k5,k6,k7,k8,k9,k10,k11,k12,k13,k14,
     +                 k15,k16,k17,k18,k19,k20,k21,k22,k23,k24,k25,r(25)
      parameter (k1=.35d0,   k2=.266d2,
     +           k3=.123d5,  k4=.86d-3,
     +           k5=.82d-3,  k6=.15d5,
     +           k7=.13d-3,  k8=.24d5,
     +           k9=.165d5,  k10=.9d4,
     +           k11=.22d-1, k12=.12d5,
     +           k13=.188d1, k14=.163d5,
     +           k15=.48d7,  k16=.35d-3,
     +           k17=.175d-1,k18=.1d9,
     +           k19=.444d12,k20=.124d4,
     +           k21=.21d1,  k22=.578d1,
     +           k23=.474d-1,k24=.178d4,
     +           k25=.312d1)

      r( 1) = k1 *y( 1)
      r( 2) = k2 *y( 2)*y(4)
      r( 3) = k3 *y( 5)*y(2)
      r( 4) = k4 *y( 7)
      r( 5) = k5 *y( 7)
      r( 6) = k6 *y( 7)*y(6)
      r( 7) = k7 *y( 9)
      r( 8) = k8 *y( 9)*y(6)
      r( 9) = k9 *y(11)*y(2)
      r(10) = k10*y(11)*y(1)
      r(11) = k11*y(13)
      r(12) = k12*y(10)*y(2)
      r(13) = k13*y(14)
      r(14) = k14*y( 1)*y(6)
      r(15) = k15*y( 3)
      r(16) = k16*y( 4)
      r(17) = k17*y( 4)
      r(18) = k18*y(16)
      r(19) = k19*y(16)
      r(20) = k20*y(17)*y(6)
      r(21) = k21*y(19)
      r(22) = k22*y(19)
      r(23) = k23*y( 1)*y(4)
      r(24) = k24*y(19)*y(1)
      r(25) = k25*y(20)

      f(1)  = -r(1)-r(10)-r(14)-r(23)-r(24)+
     +        r(2)+r(3)+r(9)+r(11)+r(12)+r(22)+r(25)
      f(2)  = -r(2)-r(3)-r(9)-r(12)+r(1)+r(21)
      f(3)  = -r(15)+r(1)+r(17)+r(19)+r(22)
      f(4)  = -r(2)-r(16)-r(17)-r(23)+r(15)
      f(5)  = -r(3)+r(4)+r(4)+r(6)+r(7)+r(13)+r(20)
      f(6)  = -r(6)-r(8)-r(14)-r(20)+r(3)+r(18)+r(18)
      f(7)  = -r(4)-r(5)-r(6)+r(13)
      f(8)  = r(4)+r(5)+r(6)+r(7)
      f(9)  = -r(7)-r(8)
      f(10) = -r(12)+r(7)+r(9)
      f(11) = -r(9)-r(10)+r(8)+r(11)
      f(12) = r(9)
      f(13) = -r(11)+r(10)
      f(14) = -r(13)+r(12)
      f(15) = r(14)
      f(16) = -r(18)-r(19)+r(16)
      f(17) = -r(20)
      f(18) = r(20)
      f(19) = -r(21)-r(22)-r(24)+r(23)+r(25)
      f(20) = -r(25)+r(24)

      return
      end subroutine pollu_feval





		subroutine eval_pderv(
     >			time,h,con,y,jac,neqns,yprime,ldj,num_rpar,rpar,num_ipar,ipar,ierr)			
     		integer, intent(in) :: neqns, ldj, num_ipar, num_rpar
     		double precision, intent(in) :: time,h,con,y(neqns),yprime(neqns)
     		double precision, intent(inout) :: rpar(num_rpar)
     		double precision, intent(inout) :: jac(ldj,neqns)
     		integer, intent(inout) :: ipar(num_ipar)
			integer, intent(out) :: ierr
			integer :: j
			ierr = 0
			call pollu_jeval(ldj,neqns,y,jac)
         do j=1,neqns 
            jac(j,j) = jac(j,j) - 1d0/con 
         end do			
		end subroutine eval_pderv
 
 
      subroutine pollu_jeval(ldim,neqn,y,dfdy)
      integer ldim,neqn
      double precision y(neqn),dfdy(ldim,neqn)

      integer i,j
      double precision k1,k2,k3,k4,k5,k6,k7,k8,k9,k10,k11,k12,k13,k14,
     +                 k15,k16,k17,k18,k19,k20,k21,k22,k23,k24,k25
      parameter (k1=.35d0,   k2=.266d2,
     +           k3=.123d5,  k4=.86d-3,
     +           k5=.82d-3,  k6=.15d5,
     +           k7=.13d-3,  k8=.24d5,
     +           k9=.165d5,  k10=.9d4,
     +           k11=.22d-1, k12=.12d5,
     +           k13=.188d1, k14=.163d5,
     +           k15=.48d7,  k16=.35d-3,
     +           k17=.175d-1,k18=.1d9,
     +           k19=.444d12,k20=.124d4,
     +           k21=.21d1,  k22=.578d1,
     +           k23=.474d-1,k24=.178d4,
     +           k25=.312d1)

      do 20 j=1,neqn
         do 10 i=1,neqn
            dfdy(i,j) = 0d0
   10    continue
   20 continue

      dfdy(1,1)   = -k1-k10*y(11)-k14*y(6)-k23*y(4)-k24*y(19)
      dfdy(1,11)  = -k10*y(1)+k9*y(2)
      dfdy(1,6)   = -k14*y(1)
      dfdy(1,4)   = -k23*y(1)+k2*y(2)
      dfdy(1,19)  = -k24*y(1)+k22
      dfdy(1,2)   = k2*y(4)+k9*y(11)+k3*y(5)+k12*y(10)
      dfdy(1,13)  = k11
      dfdy(1,20)  = k25
      dfdy(1,5)   = k3*y(2)
      dfdy(1,10)  = k12*y(2)
c
      dfdy(2,4)   = -k2*y(2)
      dfdy(2,5)   = -k3*y(2)
      dfdy(2,11)  = -k9*y(2)
      dfdy(2,10)  = -k12*y(2)
      dfdy(2,19)  = k21
      dfdy(2,1)   = k1
      dfdy(2,2)   = -k2*y(4)-k3*y(5)-k9*y(11)-k12*y(10)
c
      dfdy(3,1)   = k1
      dfdy(3,4)   = k17
      dfdy(3,16)  = k19
      dfdy(3,19)  = k22
      dfdy(3,3)   = -k15
c
      dfdy(4,4)   = -k2*y(2)-k16-k17-k23*y(1)
      dfdy(4,2)   = -k2*y(4)
      dfdy(4,1)   = -k23*y(4)
      dfdy(4,3)   = k15
c
      dfdy(5,5)   = -k3*y(2)
      dfdy(5,2)   = -k3*y(5)
      dfdy(5,7)   = 2d0*k4+k6*y(6)
      dfdy(5,6)   = k6*y(7)+k20*y(17)
      dfdy(5,9)   = k7
      dfdy(5,14)  = k13
      dfdy(5,17)  = k20*y(6)
c
      dfdy(6,6)   = -k6*y(7)-k8*y(9)-k14*y(1)-k20*y(17)
      dfdy(6,7)   = -k6*y(6)
      dfdy(6,9)   = -k8*y(6)
      dfdy(6,1)   = -k14*y(6)
      dfdy(6,17)  = -k20*y(6)
      dfdy(6,2)   = k3*y(5)
      dfdy(6,5)   = k3*y(2)
      dfdy(6,16)  = 2d0*k18
c
      dfdy(7,7)   = -k4-k5-k6*y(6)
      dfdy(7,6)   = -k6*y(7)
      dfdy(7,14)  = k13
c
      dfdy(8,7)   = k4+k5+k6*y(6)
      dfdy(8,6)   = k6*y(7)
      dfdy(8,9)   = k7
c
      dfdy(9,9)   = -k7-k8*y(6)
      dfdy(9,6)   = -k8*y(9)
c
      dfdy(10,10) = -k12*y(2)
      dfdy(10,2)  = -k12*y(10)+k9*y(11)
      dfdy(10,9)  = k7
      dfdy(10,11) = k9*y(2)
c
      dfdy(11,11) = -k9*y(2)-k10*y(1)
      dfdy(11,2)  = -k9*y(11)
      dfdy(11,1)  = -k10*y(11)
      dfdy(11,9)  = k8*y(6)
      dfdy(11,6)  = k8*y(9)
      dfdy(11,13) = k11
c
      dfdy(12,11) = k9*y(2)
      dfdy(12,2)  = k9*y(11)
c
      dfdy(13,13) = -k11
      dfdy(13,11) = k10*y(1)
      dfdy(13,1)  = k10*y(11)
c
      dfdy(14,14) = -k13
      dfdy(14,10) = k12*y(2)
      dfdy(14,2)  = k12*y(10)
c
      dfdy(15,1)  = k14*y(6)
      dfdy(15,6)  = k14*y(1)
c
      dfdy(16,16) = -k18-k19
      dfdy(16,4)  = k16
c
      dfdy(17,17) = -k20*y(6)
      dfdy(17,6)  = -k20*y(17)
c
      dfdy(18,17) = k20*y(6)
      dfdy(18,6)  = k20*y(17)
c
      dfdy(19,19) = -k21-k22-k24*y(1)
      dfdy(19,1)  = -k24*y(19)+k23*y(4)
      dfdy(19,4)  = k23*y(1)
      dfdy(19,20) = k25
c
      dfdy(20,20) = -k25
      dfdy(20,1)  = k24*y(19)
      dfdy(20,19) = k24*y(1)

      return
      end subroutine pollu_jeval
		

		subroutine call_back(neqns,time,h,con,y,yprime,tout,tend,hmax,lout,verbosity,
     >			num_rd,num_id,rdata,idata,maxder,maxstp,equil,jdebug,
     >			itol,num_tol,rtol,atol,inf_nrms,
     >         num_rpar,rpar,num_ipar,ipar,idid,kgo,ierr)
	      integer, intent(in) :: neqns,num_rd,num_id,num_tol,num_ipar,num_rpar,kgo
	      double precision, intent(in) :: time,h,con,y(neqns),yprime(neqns)					
			integer, intent(inout) :: idid,ierr,itol,lout,verbosity
			integer, intent(inout) :: ipar(num_ipar),maxder,maxstp,equil,jdebug,idata(num_id)		
			double precision, intent(inout) :: rtol(num_tol),atol(num_tol),inf_nrms(4),
     >			rpar(num_rpar),rdata(num_rd)
			double precision, intent(inout) :: tout,tend,hmax
			
			integer :: i
			double precision :: jnk
			jnk = y(1); jnk = yprime(1); jnk = jnk ! to keep the compiler quiet

			ierr = 0
			
		end subroutine call_back
		
		
		subroutine report(m,n,Z,num_id,idata,abs_tol,rel_tol,scd)
			integer, intent(in) :: m, n, num_id, idata(num_id)
			double precision, intent(in) :: Z(m,n), abs_tol(m), rel_tol(m)
			double precision, intent(out) :: scd
			
			double precision :: hi_res_solution(m,n), mescd
			integer :: nsteps
			
			call pollu_solut(m*n, hi_res_solution)
			call getscd(mescd, scd, m*n, hi_res_solution, Z)

			nsteps = idata(i_nstep)+idata(i_nfail)

 1			format(a)
			write(*,1)
			write(*,1) '	"atol" is the requested absolute error tolerance.'
			write(*,1) '	"rtol" is the requested relative error tolerance.'
			write(*,1) '	"tries" is the total number of attempts to take a step.'
			write(*,1) '	"steps" is the sum of steps taken to reach the ending time for the problem.'
			write(*,1) '	"fail" is the number of tries that had to be redone with a smaller timestep.'
			write(*,1) '	"jacob" is the total number of jacobians created.'
			write(*,1) '	"solve" is the total number matrix solves performed.'
			write(*,1) '	"avg k" is the average order used in the solution.'
			write(*,*)
      	write(*,'(2a12,3x,7a8)')
     >			'atol', 'rtol', 'tries', 'steps', 'fail', 'jacob',
     >			'solve', 'avg k'
			
			write(*,'(3x,2e12.3,5i8,2f8.3)') 
     +      abs_tol(1), rel_tol(1), nsteps, idata(i_nstep), idata(i_nfail), idata(i_nje), 
     >		idata(i_nbsol), dble(idata(i_sum_nq))/nsteps

			write(*,*) 
			write(*,'(f14.6,a)') scd, ' significant correct digits in result'
			write(*,*) 

		end subroutine report
		

      subroutine getscd(mescd,scd,neqn,yref,y)
	      integer, intent(in) :: neqn
	      double precision, intent(in) :: yref(neqn), y(neqn)
	      double precision, intent(out) :: mescd, scd

	      integer :: i
	      double precision :: aerr, aerrmx, aerrl2, rerr, rerrmx, rerrl2
	      double precision ::  mixed_rerr, mixed_rerrmx
	      integer :: numa, numr

	      numa   = 0
	      numr   = 0
	      aerrl2 = 0d0
	      rerrl2 = 0d0 
	      aerrmx = 0d0
	      rerrmx = 0d0
			mixed_rerrmx = 0d0
      
	      do i=1,neqn

	         aerr = abs(yref(i)-y(i))
	         aerrmx = max(aerrmx,aerr)
	         aerrl2 = aerrl2 + aerr**2
	         numa   = numa + 1

	         if (yref(i).ne.0d0) then       
	            rerr = abs((yref(i)-y(i))/(yref(i)))
	            rerrmx = max(rerrmx,rerr)
	            rerrl2 = rerrl2 + rerr**2
	            numr   = numr + 1
	         endif

	         mixed_rerr = abs((yref(i)-y(i)) / (1+abs(yref(i))))
	         mixed_rerrmx = max(mixed_rerrmx,mixed_rerr)

	      end do

	      mescd = -log10(mixed_rerrmx)
	      scd = -log10(aerrmx)

      end subroutine getscd


      subroutine pollu_solut(neqn,y)
      	integer, intent(in) :: neqn
      	double precision, intent(out) :: y(neqn)
         y( 1) = 0.5646255480022769d-01
         y( 2) = 0.1342484130422339d+00
         y( 3) = 0.4139734331099427d-08
         y( 4) = 0.5523140207484359d-02
         y( 5) = 0.2018977262302196d-06
         y( 6) = 0.1464541863493966d-06
         y( 7) = 0.7784249118997964d-01
         y( 8) = 0.3245075353396018d+00
         y( 9) = 0.7494013383880406d-02
         y(10) = 0.1622293157301561d-07
         y(11) = 0.1135863833257075d-07
         y(12) = 0.2230505975721359d-02
         y(13) = 0.2087162882798630d-03
         y(14) = 0.1396921016840158d-04
         y(15) = 0.8964884856898295d-02
         y(16) = 0.4352846369330103d-17
         y(17) = 0.6899219696263405d-02
         y(18) = 0.1007803037365946d-03
         y(19) = 0.1772146513969984d-05
         y(20) = 0.5682943292316392d-04
      end subroutine pollu_solut


		end module test_mebdfi_pollu

