! ***********************************************************************
!
!   Copyright (C) 2013  Frank Timmes, Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************
 
      module mod_nse_solve
      
      use nse_def
      use mod_nse_init
      use const_def, only: dp, mesa_data_dir
      use chem_def, only: num_chem_isos

      implicit none


      contains
      


      subroutine do_nse_solve(temp,den,ye,ipartition,icoulomb,xmnse,xmun,xmup,ierr)

! this routine puts a chosen reaction network into its nse distribution.

! input:
! temp = temperature
! den  = density
! ye   = electron mole number
! ipartition = 0 = use temperature independent partiction functions
!            = 1 = use temperature dependent partition functions
! icoulomb   = 0 = do not use coulomb corrections
!            = 1 = use coulomb corrections

! output:
! xmmnse  = output nse mass fractions
! xmun    = chemical potential of neutrons
! xmup    = chemical potential of protons


! declare the pass
      integer, intent(in)    :: ipartition,icoulomb
      real(dp), intent(in)     :: temp,den,ye
      real(dp), intent(out)    :: xmnse(:),xmun,xmup
      integer, intent(out) :: ierr


! for the nse solution
      real(dp) :: yei(ionmax), mu_c(ionmax), wpart(ionmax)

! for the newton solver
      logical  :: check
      integer  :: nfev,ntaken,i
      integer, parameter :: ntrial = 200, n = 2
      real(dp)   :: x(n),resid(n),dum
      real(dp), parameter :: tolf = 1.0d-10,  tolx = 1.0d-12


! for the initial guess
      real(dp) :: fac1,fac2,amass
      real(dp), parameter :: hinv = 1.0d0/h, mev2erg = ev2erg*1.0d6, &
                        mev2gr  = mev2erg/clight**2,  a56 = 56.0d0, &
                        z56 = 28.0d0, n56 = 28.0d0,  b56  = 4.8398d2


! for the temperature dependent partition functions
      integer :: jd1,jd2,jd3,jd4,jd5,jxx
      real(dp)  :: gi,g0,dg0,aa,t9,t9i,t92


! for the coulomb corrections
      real(dp)   :: xne,a3,ge,sqrtgi,beta,mu_c_p
      real(dp), parameter :: twopi    = 2.0d0*pi,          esqu     = qe*qe, &
                           forthpi  = 4.0d0 * pi/3.0d0,  third    = 1.0d0/3.0d0, &
                           fivth    = 5.0d0/3.0d0,       a1       = -0.9052d0, &
                           a2       = 0.6322d0,          a2inv    = 1.0d0/a2
      
      logical, parameter :: dbg = .false.
      
      ierr = 0
      if (temp < min_T_for_NSE) then
         ierr = -1
         return
      end if

! initialize module values
      do i=1,ionmax
         yei(i) = zion(i)/aion(i)
      end do
      beta    = 1.0d0/(kerg * temp)

!      write(6,'(1x,i4,1pe10.2)') (i,yei(i), i=1,ionmax)
!      read(5,*)


! set the partition functions wpart(i).
! wpart(i) = ground state patition function * g(i) = as(jd1) * g(i)
! above t9=10 the fitting functions can go INF, so limit t9.

       wpart(ineut) = 1.0d0
       wpart(iprot) = 1.0d0
       wpart(ihe4)  = 1.0d0
       t9  = min(temp * 1.0d-9, 10.0d0)
       t9i = 1.0d0/t9
       t92 = t9*t9

       do i=ionbeg,ionend
        jd1  = 5*(i-1) + 1 ; jd2  = jd1 + 1 ; jd3  = jd2 + 1 ; jd4  = jd3 + 1 ; jd5  = jd4 + 1
        gi   = 0.0d0
        g0   = 1.0d0
        if (as(jd2) .ne. 0.0) then
         aa   = as(jd2)*t9i + as(jd3) + as(jd4)*t9 + as(jd5)*t92
         gi   = exp(aa)
         if (ist(i) .ne. 0) then
          do jxx=6*(i-1)+1,6*(i-1)+2*ist(i)-1,2
           aa  = gs(jxx+1) * exp(-gs(jxx)*t9i)
           g0  = g0 + aa
          enddo
         end if
        end if
        gi = g0  + gi

! using ground state spins only
        if (ipartition .eq. 0) then
         wpart(i) = as(jd1)

! or with temperature dependence
        else if (ipartition .eq. 1) then
         wpart(i) = as(jd1) * gi

        else
         write(*,*) 'nse_solve: unknown ipartition function value'
         ierr = -1
         return
        end if
       enddo

!      write(6,'(1x,i4,1pe10.2)') (i,wpart(i), i=1,40)
!      read(5,*)


! set the coulomb corrections
! calder et al, apj 656 313 2007, eq a1
      
      if (dbg) write(*,*) 'set the coulomb corrections'
      
      mu_c_p = 0.0d0
      mu_c(1:ionmax) = 0.0d0
      if (icoulomb .eq. 1) then
       xne = ye * avo * den
       ge  = esqu * beta * (forthpi * xne)**third
       a3  = -0.5d0*sqrt(3.0d0) - a1 / sqrt(a2)

       do i=1,ionmax
        gi      = zion(i)**(fivth)  * ge
        sqrtgi  = sqrt(gi)
        mu_c(i) = a1*(sqrt(gi*(a2+gi)) &
                   - a2*log(sqrt(gi*a2inv) + sqrt(1.0d0 + gi*a2inv)) &
                   + 2.0d0*a3*(sqrtgi - atan(sqrtgi)))
       enddo
       mu_c_p = mu_c(iprot)
      end if

!      write(6,'(1x,i4,1pe10.2)') (i,mu_c(i), i=1,40)
!      read(5,*)




! here is an initial guess for the neutron and proton chemical potentials,
! (x1) and (x2) respectively. obtained by setting xmass(ini56) = 1,
! setting mup = mun, and inverting the saha equation.
! here we use pure ni56 as it emprically appears to be a
! robust guess for all temp, rho, ye combinations.

       amass  = n56*mn + z56*mp - b56*mev2gr
       fac1   = a56/(avo * den)
       fac2   = twopi/(beta*h) * amass*hinv
       fac2   = fac2 * sqrt(fac2)
       x(1)   = -(log(fac1*fac2)/beta + b56*ev2erg*1.0d6)/a56
       x(2)   = x(1)



! root find on mass and charge conservation for
! the chemical potentials of protons and neutrons

      
      if (dbg) write(*,*) 'call xnewt_nse'
      call xnewt_nse(ntrial,x,n,tolx,tolf,ntaken,check,nfev, &
                     temp,den,ye,beta,mu_c_p,xmnse,wpart,yei,mu_c,ierr)
      if (ierr /= 0) return

! check convergence
      if (check .or. ntaken .eq. ntrial) then
       !write(6,*)
       !write(6,*) 'check convergence of root finder'
       !write(6,*)
         !write 'nse_solve: failed to converge'
         ierr = -1
         return
      end if


! some optional diagnostics on the root find
!       write(6,*)
!       write(6,110) 'iterations taken             =',ntaken
!       write(6,110) 'function evals               =',nfev
!       write(6,111) 'roots                        =',x(1),x(2)
!       call nsefunc(dum,x,resid,beta,mu_c_p,xmnse,wpart,yei,mu_c)
!       write(6,111) 'mass conservation   residual =',resid(1)
!       write(6,111) 'charge conservation residual =',resid(2)
! 110   format(1x,a,i4)
! 111   format(1x,a,1p2e14.6)



! fill the output array using the converged values
      if (dbg) write(*,*) 'fill the output array'
      call nsefunc(dum,x,resid, &
                   temp,den,ye,beta,mu_c_p,xmnse,wpart,yei,mu_c)


! bound the output nse abundances
      do i=1,ionmax
         xmnse(i) = min(1.0d0,max(xmnse(i),1.0d-30))
      end do
      xmun = x(1)
      xmup = x(2)

      
      end subroutine do_nse_solve


      subroutine nsefunc(x,y,f, &
                         temp,den,ye,beta,mu_c_p,xmnse,wpart,yei,mu_c)

! this routine returns the root find functions.
! input is the point x and y a vector of the unknowns.
! output is the vector of root find functions f, which should be the
! zero vector upon convergence.

! x is irrelevant for nse
! y(1) is input as the neutron chemical potential
! y(2) is input as the proton chemical potential


! stuff passed to make it threadsafe
! temp   = temperature
! den    = density
! ye     = average charge to baryon number
! beta   = 1/(k * T)
! mu_c_p = coulomb correction for proton
! xmnse  = nse mass fractions of each isotope
! waprt  = partitsion functions
! yei    = charge to baryon number of each isotope
! mu_c   = coulomb correction for each isotope



! declare the pass
      real(dp), intent(in)  :: x,y(:),temp,den,ye,beta,mu_c_p,wpart(:),yei(:),mu_c(:)
      real(dp), intent(out) :: f(:),xmnse(:)

! locals
      integer :: i,j,ifirst
      real(dp)  ::  ye_calc,mu,fac1,fac2,fac3,xmsum,sum2,deninv,ww,amass
      real(dp), parameter :: hinv = 1.0d0/h, twopih = 2.0d0 * pi/h, &
                           mev2erg = ev2erg*1.0d6, mev2gr  = mev2erg/clight**2


! chemical potential and mass fraction of each isotope
! hartmann et al, apj 297 837 1985, eq 2
! calder et al, apj 656 313 2007, eq a1

      deninv = 1.0d0/den
      ww     = twopih/beta

      do i=1,ionmax
       amass    = aion(i) * amu
       mu       = nion(i)*y(1) + zion(i)*y(2)
       fac1     = amass * deninv * wpart(i)
       fac2     = ww * amass * hinv
       fac2     = fac2*sqrt(fac2)
       fac3     = exp( beta * (mu + bion(i)*mev2erg) &
                       - mu_c(i) + zion(i)*mu_c_p)
       xmnse(i) = fac1 * fac2 * fac3
      enddo

! mass conservation
      xmsum = sum(xmnse(1:ionmax))


! charge conservation
      ye_calc = sum(yei(1:ionmax) * xmnse(1:ionmax))


! mass and charge conservation are the requirements
      f(1) = xmsum - 1.0d0
      f(2) = ye_calc - ye

      return
      end subroutine nsefunc


      subroutine nsejac(x,y,f,dfdy,n,np, &
                        temp,den,ye,beta,mu_c_p,xmnse,wpart,yei,mu_c)

! this routine returns the functions and the jacobian to do the root find on
! input is x, and y(n) a vector of the unknowns. output is f(n)
! and its jacobian dfdy(np,np).

! x is not relevant for nse
! y(1) is the neutron chemical potential
! y(2) is the proton chemical potential

! stuff passed to make it threadsafe
! temp   = temperature
! den    = density
! ye     = average charge to baryon number
! beta   = 1/(k * T)
! mu_c_p = coulomb correction for proton
! xmnse  = nse mass fractions of each isotope
! waprt  = partitsion functions
! yei    = charge to baryon number of each isotope
! mu_c   = coulomb correction for each isotope


! declare the pass
      integer, intent(in) :: n,np
      real(dp), intent(in)  :: x,y(:), temp,den,ye,beta,mu_c_p,wpart(:),yei(:),mu_c(:)
      real(dp), intent(out) :: f(:),dfdy(:,:),xmnse(:)


! locals
      integer :: i,j
      real(dp)  :: xmbn,xmbp,mu,mubn,mubp,fac1,fac2,fac3,fac4,fac5, amass, &
                 xmsum,xmsumbn,xmsumbp,ye_calc,yebn,yebp,deninv,ww,sum2,sum2bn,sum2bp
      real(dp), parameter :: hinv  = 1.0d0/h, twopih  = 2.0d0 * pi/h, &
                           mev2erg = ev2erg*1.0d6, mev2gr  = mev2erg/clight**2


! chemical potential and mass fraction of each isotope
! hartmann et al, apj 297 837 1985, eq 2
! calder et al, apj 656 313 2007, eq a1

      deninv  = 1.0d0/den
      ww      = twopih/beta
      xmsum   = 0.0d0
      xmsumbn = 0.0d0
      xmsumbp = 0.0d0

      ye_calc = 0.0d0
      yebn    = 0.0d0
      yebp    = 0.0d0

! loop over isotopes, do the sums in place
      do i=1,ionmax

       amass        = aion(i) * amu

       mu       = nion(i) * y(1) + zion(i) * y(2)
       mubn     = nion(i)
       mubp     = zion(i)

       fac1     = amass * deninv * wpart(i)
       fac2     = ww * amass * hinv
       fac2     = fac2 * sqrt(fac2)
       fac3     = exp( beta * (mu + bion(i) * ev2erg * 1.0d6) &
                           - mu_c(i) + zion(i)*mu_c_p)
       fac4     = fac1 * fac2 * fac3

       xmnse(i) = fac4


       xmbn  = fac4 * beta * mubn
       xmbp  = fac4 * beta * mubp

! mass conservation
       xmsum   = xmsum + xmnse(i)
       xmsumbn = xmsumbn + xmbn
       xmsumbp = xmsumbp + xmbp

! charge conservation
       ye_calc = ye_calc + yei(i) * xmnse(i)
       yebn    = yebn + yei(i) * xmbn
       yebp    = yebp + yei(i) * xmbp

      enddo


! mass and charge conservation are the requirements
      f(1) = xmsum - 1.0d0
      f(2) = ye_calc - ye

! jacobian
      dfdy(1,1) = xmsumbn ; dfdy(1,2) = xmsumbp
      dfdy(2,1) = yebn    ; dfdy(2,2) = yebp

      return
      end subroutine nsejac




      subroutine xnewt_nse(ntrial,x,n,tolx,tolf,ntaken,check,nfev, &
                           temp,den,ye,beta,mu_c_p,xmnse,wpart,yei,mu_c,ierr)

! given an initial guess x(1:n) for the root of n equations, this routine
! finds the root by a globally convergent newtons method. the vector of
! functions to be zeroed, called fvec(1:n) in the routine below, is
! returned by the routine nsefunc. the output quantity check
! is false on nomal return and true if the routine has converged to a
! local minimum of the function fmin_nse. if so, try restarting from a
! different initial guess.

! np is the maximum number of equations n
! ntrial is the maximum number of iterations to try
! ntaken is the number of iterations done
! tolf sets the convergence on function values
! tolmin sets the criterion for deciding wheather spurious convergence to
!        a false minimum of fmin_nse has occured
! tolx is the convergence criteria on deltax
! stpmx is the scaled maximum step length allowed in the line searches
! nfev is the number of function evaluations

! stuff passed to make it threadsafe
! temp   = temperature
! den    = density
! ye     = average charge to baryon number
! beta   = 1/(k * T)
! mu_c_p = coulomb correction for proton
! xmnse  = nse mass fractions of each isotope
! waprt  = partitsion functions
! yei    = charge to baryon number of each isotope
! mu_c   = coulomb correction for each isotope


! declare the pass
      logical, intent(out)  :: check
      integer, intent(in)   :: ntrial,n
      integer,intent(out)   :: ntaken,nfev
      real(dp), intent(inout) :: x(:)
      real(dp), intent(in)    :: tolx,tolf,temp,den,ye,beta,mu_c_p,wpart(:),yei(:),mu_c(:)
      real(dp), intent(out)   :: xmnse(:)
      integer, intent(out) :: ierr

! locals
      integer, parameter :: np = 2
      integer :: i,its,j
      real(dp)  :: d,denom,f,fold,stpmax,xsum,test,dum
      real(dp), parameter :: tolmin = 1.0d-12, stpmx = 2.0d0
      
      integer, target :: indx_ary(np)
      real(dp), target :: fjac_ary(np*np)
      real(dp), target, dimension(np) :: fvec_ary, g_ary, p_ary, xold_ary
      
      integer, pointer :: indx(:) ! (np)
      real(dp), pointer :: fjac(:,:) ! (np,np)
      real(dp), pointer, dimension(:) :: fvec, g, p, xold ! (np)
      
      logical, parameter :: dbg = .false.


! initialize
      ierr = 0
      indx(1:np) => indx_ary(1:np)
      fjac(1:np,1:np) => fjac_ary(1:np*np)
      fvec(1:np) => fvec_ary(1:np)
      g(1:np) => g_ary(1:np)
      p(1:np) => p_ary(1:np)
      xold(1:np) => xold_ary(1:np)
      
      f = fmin_nse(x,n,fvec,temp,den,ye,beta,mu_c_p,xmnse,wpart,yei,mu_c)
      nfev = 1 
      ntaken = 0

!  test for the initial guess being a root, using a more stringent tolf
      test = maxval(abs(fvec(1:n)))
      if (test .lt. 0.01*tolf) then
       check = .false.
       return
      end if

! get stpmax for the line search
      xsum = sum(x(1:n)*x(1:n))
      stpmax = stpmx * max(sqrt(xsum),dfloat(n))


! start of iteration loop; get the jacobian
      do its = 1, ntrial
       ntaken = its


! analytic jacobian
       if (dbg) write(*,*) 'xnewt_nse its', its
       if (dbg) write(*,*) 'call nsejac'
       call nsejac(dum,x,fvec,fjac,n,np, &
                   temp,den,ye,beta,mu_c_p,xmnse,wpart,yei,mu_c)
       nfev = nfev + 1


! compute grad f for the line searches
       do i=1,n
        g(i) = sum(fjac(1:n,i)*fvec(1:n))
       enddo


! store x, and f and form right hand sides
       fold = f
       do i=1,n
         xold(i) = x(i)
         p(i) = -fvec(i)
       end do


! solve the linear systems

!       write(6,*)
!       write(6,112) x(1),x(2)
!       write(6,112) fvec(1),fvec(2)
!       write(6,112) fjac(1,1),fjac(1,2),fjac(2,1),fjac(2,2)
! 112   format(1x,1p2e14.6)


       if (dbg) write(*,*) 'call ludcmp'
       call ludcmp(fjac,n,np,indx,d,ierr)
       if (ierr /= 0) then
         return
       end if

       if (dbg) write(*,*) 'call lubksb'
       call lubksb(fjac,n,np,indx,p,ierr)
       if (ierr /= 0) then
         return
       end if

! line search returns new x and f
! it also gets fvec at the new x when it calls fmin_nse

       if (dbg) write(*,*) 'call lnsrch_nse'
       call lnsrch_nse(n,xold,fold,g,p,x,f,stpmax,check,nfev, &
                       fvec,temp,den,ye,beta,mu_c_p,xmnse,wpart,yei,mu_c,ierr)
       if (ierr /= 0) then
         return
       end if

!       write(6,112) x(1),x(2)
!       write(6,112) f
!       write(6,*)

! test for convergence on function value
       test = maxval(abs(fvec(1:n)))
       if (test .lt. tolf) then
        check = .false.
        return
       end if

! check for zero gradiant, i.e. spurious convergence
       if (check) then
        denom  = max(f, 0.5d0 * n)
        test = maxval( abs(g(1:n) * max(abs(x(1:n)),1.0d0)/denom) )
        if (test .lt. tolmin) then
         check = .true.
        else
         check = .false.
        end if
        return
       end if

! test for convergence on deltax
       test = maxval((abs(x(1:n)-xold(1:n)))/max(abs(x(1:n)),1.0d0))
       if (test .lt. tolx) return

!       write(6,*) its,test

! back for another iteration
      enddo
      
      check = .true.

      end subroutine xnewt_nse




      subroutine lnsrch_nse(n,xold,fold,g,p,x,f,stpmax,check,nfev, &
            fvec,temp,den,ye,beta,mu_c_p,xmnse,wpart,yei,mu_c,ierr)

! given an n dimensional point xold(1:n), the value of the function fold
! and the gradient g(1:n) at the point, and a direction p(1:n), this routine
! finds a new point x(1:n) along the direction of p from xold where the
! function fmin_nse has decreased "sufficiently". the new function value is
! returned in f. stpmax is an input quanity that limits the length of the
! steps so that the function is not evaluated in regions where it is
! undefined or subject to overflow. p is usually the newton direction. the
! output quantity check is false on normal exit, and true when x is too
! close to xold. in a minimization routine, this usually signals
! convergence and can be ignored. however, in a root finding routine, the
! calling routine should check wheather the convergence is spurious.

! stuff passed to make it threadsafe
! temp   = temperature
! den    = density
! ye     = average charge to baryon number
! beta   = 1/(k * T)
! mu_c_p = coulomb correction for proton
! xmnse  = nse mass fractions of each isotope
! waprt  = partitsion functions
! yei    = charge to baryon number of each isotope
! mu_c   = coulomb correction for each isotope


! declare the pass
      logical, intent(out)  :: check
      integer, intent(in)   :: n
      integer,intent(out)   :: nfev
      real(dp), intent(in)    :: fold,stpmax,xold(:),g(:)
      real(dp), intent(out)   :: f,p(:),x(:),fvec(:)
      real(dp), intent(in)    :: temp,den,ye,beta,mu_c_p, & 
                               wpart(:),yei(:),mu_c(:)
      real(dp), intent(out)   :: xmnse(:)
      integer, intent(out) :: ierr

! locals
      integer :: i
      real(dp)  :: a,alam,alam2,alamin,b,disc,f2,rhs1, &
                 rhs2,slope,xsum,test,tmplam

! alf ensures decreases in the function value, tolx is the convergence criterion on deltax
      real(dp), parameter :: alf  = 1.0d-4, tolx = 3.0d-12, alam_start = 1.0d0

      ierr = 0
! initialize and scale if the attempted step is too big
      check = .false.
      xsum = sqrt(sum(p(1:n)*p(1:n)))

      if (xsum .gt. stpmax) then
       xsum = 1.0d0/xsum
       p(1:n) = p(1:n) * stpmax*xsum
      end if

      slope = sum(g(1:n)*p(1:n))
      if (slope .ge. 0.0) then
         ierr = -1
         return
      end if


! compute lambda_min
      test = maxval(abs(p(1:n))/max(abs(xold(1:n)),1.0d0))
      alamin = tolx/test


! always try a full newton step, start of iteration loop
      alam = alam_start
      f2 = f
      alam2 = alam
1     continue
      do i=1,n
       x(i) = xold(i) + alam*p(i)


! for the nse problem make sure the neutron and proton
! chemical potentials are less than or equal to zero
! hmmmm. for low ye (0.2) and high density (1e14), the neutron
! chemical potential does indeed go positive. let's try
! letting it go wherever it wants

!       if (x(i) .gt. 0.0) x(i) = xold(i) + 1.0d-4*alam*p(i)
       if (x(i) .gt. 0.0) x(i) = 1.0d-10
      enddo

      f    = fmin_nse(x,n,fvec,temp,den,ye,beta,mu_c_p,xmnse,wpart,yei,mu_c)
      nfev = nfev + 1


! convergence on deltax, for root finding, the calling routine
! should verify the convergence
      if (alam .lt. alamin) then
       do i=1,n
         x(i) = xold(i)
       end do
       check = .true.
       return

! sufficient function decrease
      else if (f .le. fold + alf*alam*slope) then
       return

! backtrack
      else
       if (alam .eq. alam_start) then
        tmplam = -slope / (2.0d0 * (f-fold-slope))
       else
        rhs1 = f  - fold - alam*slope
        rhs2 = f2 - fold - alam2*slope
        a    = (rhs1/alam**2 - rhs2/alam2**2)/(alam-alam2)
        b    = (-alam2*rhs1/alam**2 + alam*rhs2/alam2**2) / (alam-alam2)
        if (a .eq. 0.0) then
         tmplam = -slope/(2.0d0 * b)
        else
         disc = b*b - 3.0d0 * a * slope
         if (disc .lt. 0.0) then
          tmplam = 0.5d0 * alam
         else if (b .le. 0.0) then
          tmplam = (-b + sqrt(disc)) / (3.0d0 * a)
         else
          tmplam = -slope/(b + sqrt(disc))
         end if
        end if
        if (tmplam .gt. 0.5d0*alam) tmplam = 0.5d0*alam
       end if
      end if

! store for the next trip through
      alam2 = alam
      f2    = f
      alam  = max(tmplam, 0.1d0*alam)
      goto 1
      end subroutine lnsrch_nse



      function fmin_nse(x,n,fvec,temp,den,ye,beta,mu_c_p,xmnse,wpart,yei,mu_c)

! returns f = 0.5 f dot f at x. func is a user supplied routine of the
! functions to be root found.

! stuff passed to make it threadsafe
! temp   = temperature
! den    = density
! ye     = average charge to baryon number
! beta   = 1/(k * T)
! mu_c_p = coulomb correction for proton
! xmnse  = nse mass fractions of each isotope
! waprt  = partitsion functions
! yei    = charge to baryon number of each isotope
! mu_c   = coulomb correction for each isotope


! declare the pass
      integer, intent(in) :: n
      real(dp), intent(in)  :: x(:)
      real(dp), intent(out) :: fvec(:),xmnse(:)
      real(dp), intent(in)  :: temp,den,ye,beta,mu_c_p,wpart(:),yei(:),mu_c(:)

! locals
      integer :: i
      real(dp)  :: dum,fmin_nse

      call nsefunc(dum,x,fvec,temp,den,ye,beta,mu_c_p,xmnse,wpart,yei,mu_c)
      fmin_nse = 0.5d0 * sum(fvec(1:2)*fvec(1:2))
  
      return
      end function fmin_nse







      subroutine ludcmp(a,n,np,indx,d,ierr)


! given th matrix a(n,n), with physical dimsnsions a(np,ap) this routine
! replaces a by the lu decompostion of a row-wise permutation of itself.
! input are a,n,np. output is a, indx which records the row
! permutations effected by the partial pivoting, and d which is 1 if
! the number of interchanges is even, -1 if odd.
! use routine lubksb to solve a system of linear equations.
!
! nmax is the largest expected value of n
!
! declare
      integer          n,np,indx(:),nmax,i,j,k,imax,ierr
      parameter        (nmax=500)
      real(dp) a(:,:),d,tiny,vv(nmax),aamax,sum,dum
      parameter        (tiny=1.0d-20)
      
      
! bullet check
      ierr = 0
      if (np .gt. nmax) then
       write(6,*) 'np=',np,' nmax=',nmax
       ierr = -1
       return
      end if

! vv stores the implicit scaling of each row
! loop over the rows to get the scaling information
      !write(*,*) 'loop over the rows'
      d = 1.0d0
      do i=1,n
       aamax = 0.0d0
       do j=1,n
        if (abs(a(i,j)) .gt. aamax) aamax = abs(a(i,j))
       enddo
       if (aamax .eq. 0.0) then
         ! singular matrix in ludcmp
         !write(*,*) 'singular matrix in ludcmp'
         ierr = -1
         return
       end if
       vv(i) = 1.0d0/aamax
      enddo
      
      !write(*,*) 'apply crouts method'
! for each column apply crouts method; see equation 2.3.12
      do j=1,n
       do i=1,j-1
        sum = a(i,j)
        do k=1,i-1
         sum = sum - a(i,k)*a(k,j)
        enddo
        a(i,j) = sum
       enddo

      !write(*,*) 'find the largest pivot element'
! find the largest pivot element
       aamax = 0.0d0
       imax = 0
       do i=j,n
        sum=a(i,j)
        do k=1,j-1
         sum = sum - a(i,k)*a(k,j)
        enddo
        a(i,j) = sum
        dum = vv(i)*abs(sum)
        if (dum .ge. aamax) then
         imax  = i
         aamax = dum
        end if
       enddo
       if (imax == 0) then
         ierr = -1
         return
       end if

! if we need to interchange rows
       if (j .ne. imax) then
        !write(*,*) 'interchange rows', j, imax, n
        do k=1,n
         dum       = a(imax,k)
         a(imax,k) = a(j,k)
         a(j,k)    = dum
        enddo
        d          = -d
        vv(imax)   = vv(j)
       end if

! divide by the pivot element
      !write(*,*) 'divide by the pivot element'
       indx(j) = imax
       if (a(j,j) .eq. 0.0) a(j,j) = tiny
       if (j .ne. n) then
        dum = 1.0d0/a(j,j)
        do i=j+1,n
         a(i,j) = a(i,j)*dum
        enddo
       end if

! and go back for another column of crouts method
      enddo
      return
      end subroutine ludcmp




      subroutine lubksb(a,n,np,indx,b,ierr)


! solves a set of n linear equations ax=b. a is input in its lu decomposition
! form, determined by the routine above ludcmp. indx is input as the
! permutation vector also returned by ludcmp. b is input as the right hand
! side vector and returns with the solution vector x.
! a,n ans np are not modified by this routine and thus can be left in place
! for successive calls (i.e matrix inversion)
!
!
! declare
      integer :: n,np,indx(:),i,ii,j,ll,ierr
      real(dp) :: a(:,:),b(:),sum
      
      ierr = 0
! when ii is > 0, ii becomes the index of the first nonzero element of b
! this is forward substitution of equation 2.3.6, and unscamble in place
      ii = 0
      do i=1,n
       ll = indx(i)
       sum = b(ll)
       b(ll) = b(i)
       if (ii .ne. 0) then
        do j=ii,i-1
         sum = sum - a(i,j) * b(j)
        enddo

! nonzero element was found, so do the sums in the loop above
       else if (sum .ne. 0.0) then
        ii  = i
       end if
       b(i) = sum
      enddo

! back substitution equation 2.3.7
      do i = n,1,-1
       sum = b(i)
       if (i .lt. n) then
        do j=i+1,n
         sum = sum - a(i,j) * b(j)
        enddo
       end if
       b(i) = sum/a(i,i)
      enddo
      return
      end subroutine lubksb






      end module mod_nse_solve

