! ***********************************************************************
!
!   Copyright (C) 2013  Frank Timmes, Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************
 
      module mod_nse_init
      
      use nse_def
      use const_def, only: dp, mesa_data_dir
      use chem_def, only: num_chem_isos

      implicit none


! math constants
      real(dp), parameter :: pi = 3.1415926535897932384d0

! physical constants
      real(dp), parameter :: &
                           h       = 6.6260689633d-27, &
                           qe      = 4.8032042712d-10, &
                           avo     = 6.0221417930d23, &
                           clight  = 2.99792458d10, &
                           kerg    = 1.380650424d-16, &
                           ev2erg  = 1.60217648740d-12, &
                           amu     = 1.66053878283d-24, &
                           mn      = 1.67492721184d-24, &
                           mp      = 1.67262163783d-24, &
                           me      = 9.1093821545d-28





! NOTE: any changes to the following must be reflected in the routines that read/write the cache


! ionmax is the number is isotopes in the network - sets the size of all allocatable arrays
! ionbeg is the location of the first isotope
! ionend is the location of the last isotope

      integer :: ionmax,ionbeg,ionend

! mass fractions, other composition variables
! ionam     = name of the isotope in the network
! wion      = atomic weight or molar mass of isotope
! aion      = number of nucleons
! zion      = number of protons
! nion      = number of neutrons
! bion      = binding energies

      logical :: have_allocated = .false.

      character*5, allocatable :: ionam(:)
      real(dp), allocatable :: mion(:),wion(:),aion(:),zion(:),nion(:),bion(:)
      integer, allocatable :: torch_chem_id(:),torch_net_iso(:)

! for the links between isotopes
      integer, allocatable :: nrr(:,:), nrrneut(:,:)

! for the temperature dependent partition functions
      integer, allocatable :: ist(:)
      real(dp), allocatable  :: as(:),gs(:)

! for easy identification of key isotopes
      integer :: &
          ih1,iprot,ineut,ihe4,ih2,ih3,ihe3,ili6,ili7,ili8, &
          ibe7,ibe9,ib8,ib9,ib10,ib11,ic11,ic12,ic13,ic14, &
          in12,in13,in14,in15,io14,io15,io16,io17,io18,if17, &
          if18,if19,ine18,ine19,ine20,ine21,ine22,ine23,ina20,ina21,ina22, &
          ina23,img22,img23,img24,img25,img26,ial25,ial26,ial27,ial28, &
          isi26,isi27,isi28,isi29,isi30,ip28,ip29,ip30,ip31,is30, &
          is31,is32,icl35,iar36,ik39,ica40,isc43,isc45,iti44,iti46, &
          iti48,iti50,iv46,iv47,iv48,iv51,icr47,icr48,icr49,icr50, &
          icr51,icr52,icr53,icr54,imn50,imn51,imn52,imn55,ife52,ife54, &
          ife55,ife56,ife57,ife58,ico54,ico55,ico56,ico59,ini56,ini58, &
          ini59,ini64,ini66,icu63,izn60,izn64

            
      
      contains
      
      
      subroutine do_alloc_for_nse(ierr)
         integer, intent(out) :: ierr
         include 'formats.dek'
         ierr = 0
         if (have_allocated) return
         allocate( ionam(ionmax), mion(ionmax),wion(ionmax), &
                aion(ionmax),zion(ionmax),nion(ionmax),bion(ionmax), &
                nrr(7,ionmax), nrrneut(7,ionmax), &
                ist(ionmax), as(6*ionmax), gs(6*ionmax), &
                torch_chem_id(ionmax), torch_net_iso(num_chem_isos), stat=ierr)
         if (ierr /= 0) then
            write(*,*) 'alloc failed for nse init'
            write(*,2) 'ionmax', ionmax
            write(*,2) 'num_chem_isos', num_chem_isos
            stop 'do_alloc_for_nse' 
            return
         end if
         have_allocated = .true.
      end subroutine do_alloc_for_nse
      

      subroutine init_nse(ierr)
         use chem_def, only: chem_isos
         use chem_lib, only: chem_get_iso_id
         integer, intent(out) :: ierr
         integer :: i, cid, inetin, izzz(100), inmin(100), inmax(100)
         include 'formats.dek'
      
         ierr = 0

         ! get the size of the desired network
         call zet489(inetin,izzz,inmin,inmax)
         call get_ionmax(inetin,izzz,inmin,inmax,ierr)
         if (ierr /= 0) return
      
         if (ionmax /= nse_species) then
            write(*,3) 'ionmax /= nse_species', ionmax, nse_species
            ierr = -1
            return
         end if

         ! allocate the space
         call do_alloc_for_nse(ierr)
         if (ierr /= 0) return

         ! get the nuclear data for this network
         call init_torch(inetin,izzz,inmin,inmax,ierr)
         if (ierr /= 0) return
      
         do i=1,ionmax
            if (ionam(i) == 'prot') then
               cid = chem_get_iso_id('h1')
            else
               cid = chem_get_iso_id(ionam(i))
            end if
            !write(*,3) ionam(i), i, cid
            if (cid < 1 .or. cid > num_chem_isos) then
               write(*,*) 'failed to find iso for nse net: ' // trim(ionam(i))
               ierr = -1
               return
            end if
            torch_chem_id(i) = cid
            torch_net_iso(cid) = i
            if (nse_net_iso(cid) == 0) then
               write(*,*) 'iso in torch nse net is not in nse_net_isos: ' // trim(ionam(i))
               ierr = -1
               !return
            end if
         end do
      
         do i=1,ionmax
            cid = nse_chem_id(i)
            if (torch_net_iso(cid) == 0) then
               write(*,3) 'nse iso missing from torch: ' // trim(chem_isos% name(cid)), i, cid
            else
               !write(*,3) trim(chem_isos% name(cid)), i, cid
            end if
         end do

      end subroutine init_nse



      subroutine init_torch(inetin,izzz,inmin,inmax,ierr)


! this routine initializes stuff for the torch network

! declare the pass
      integer ::  inetin,izzz(*),inmin(*),inmax(*)
      integer, intent(out) :: ierr


! for reading bdat

! local variables
      logical            :: ibhere,downarrow
      integer, parameter :: zmax=85, fil14=14
      character*132      :: string,word
      character*2        :: zsymb(zmax)
      character*10       :: aname
      integer            :: i,j,k,l,ii,jj,ll,llmin,llmax,mm,j1,nz,na,nn,kk, &
                            aidmin(zmax),aidmax(zmax),inta,intz, & 
                            idum1,idum2,ibs1(10),ibs2(10)
      real(dp)                dum1

! iener = location of energy equation
! itemp = location of temperature equation
! iden  = location of density equation
! ivelx = location of velocity equation
! iposx = location of position equation
! neqs    = number of equations
! nrat    = number of reaction rates in the network

      !integer :: iener,itemp,iden,ivelx,iposx,neqs
      
      integer :: nrat,kmax

      
      real(dp), parameter :: mev2erg = ev2erg*1.0d6, mev2gr  = mev2erg/clight**2
      real(dp) :: dum,qful,xx


! here are the root isotope names
      data  zsymb/'h ','he','li','be','b ','c ','n ','o ','f ','ne', &
                  'na','mg','al','si','p ','s ','cl','ar','k ','ca', &
                  'sc','ti','v ','cr','mn','fe','co','ni','cu','zn', &
                  'ga','ge','as','se','br','kr','rb','sr','y ','zr', &
                  'nb','mo','tc','ru','rh','pd','ag','cd','in','sn', &
                  'sb','te','i' ,'xe','cs','ba','la','ce','pr','nd', &
                  'pm','sm','eu','gd','tb','dy','ho','er','tm','yb', &
                  'lu','hf','ta','w' ,'re','os','ir','pt','au','hg', &
                  'tl','pb','bi','po','at'/

! here are the min and max a's for each z
      data  aidmin/  2,   3,   6,   7,   8,   9,  11,  13,  14,  16, &
                    17,  18,  20,  22,  23,  24,  25,  27,  30,  30, &
                    34,  34,  38,  38,  42,  42,  46,  46,  50,  51, &
                    55,  55,  59,  59,  63,  63,  68,  68,  72,  72, &
                    76,  77,  81,  81,  85,  86,  88,  90,  92,  94, &
                    97,  99, 101, 103, 106, 108, 110, 113, 115, 118, &
                   120, 123, 125, 128, 130, 133, 136, 138, 141, 143, &
                   146, 150, 153, 154, 160, 160, 164, 165, 168, 170, &
                   172, 174, 176, 182, 188/

      data  aidmax/  3,   6,   9,  12,  14,  18,  21,  22,  26,  31, &
                    44,  47,  51,  54,  57,  60,  63,  67,  70,  73, &
                    76,  80,  83,  86,  89,  92,  96,  99, 102, 105, &
                   108, 112, 115, 118, 121, 124, 128, 131, 134, 137, &
                   140, 144, 147, 150, 153, 156, 160, 163, 166, 169, &
                   171, 173, 175, 177, 179, 181, 183, 185, 187, 189, &
                   191, 193, 195, 197, 199, 201, 203, 205, 207, 209, &
                   213, 214, 219, 220, 225, 226, 231, 234, 237, 240, &
                   245, 246, 251, 237, 239/



! popular format statements
01    format(a,i4)
06    format(2i5,f10.4)
07    format(f10.3)
08    format(6f10.3)
09    format(1x,i4,i4,i4,'    ',a5)




! zero all the isotope and rate pointers



! decide on the arrow orientation
! downarrow true puts neutron, protons, alfa to the end
! downarrow false (i.e uparrow) puts neutron, protons, alfa at the beginning
! in general downarrowtrue  is faster for dense linear algebra, and either
! orientation for sparse linear algebra. gift routines, however, do
! much better with uparrow (downarrow false).

      downarrow = .true.
!      downarrow = .false.


! set the beginning isotope index
      if (downarrow) then
       ionbeg = 1
      else
       ionbeg = 4
      end if




! open the nuclear reaction rate data file
! use a soft link to connect bdat to the desired burn data file

      open(unit=fil14, file=trim(mesa_data_dir) // '/nse_data/BDAT', &
         action='read', status='old', iostat=ierr)
      if (ierr /= 0) then
         write(*,*) 'NSE ERROR --- failed to open ' // trim(mesa_data_dir) // '/nse_data/BDAT'
         return
      end if



! now start reading the nuclear reaction rate data file
! i is the code number of element z(i),n(i).
! j = 1 = ng                 j = 6 = an
! j = 2 = pn                 j = 7  = ag
! j = 3 = ground state b-    j = 8 for semi-empirical electron captur
! j = 4 = pg                 j = 9 for semi-empirical positron decay
! j = 5 = ap                 j = 10 for semi-empirical beta decay
! ic1(j,i) = type formula to be used to calculate rate
! ic2(j,i) = number of constants in fitting reaction j on species i
! ic3(j,i) = where to start counting ic2 from


! initialize counters
      k        = 1
      i        = ionbeg - 1


! put neutrons, protons and alfa first for up-arrow orientation
      if (.not.downarrow) then
       ineut        = 1
       aion(ineut)  = 1.0d0
       nion(ineut)  = 1.0d0
       zion(ineut)  = 0.0d0
       bion(ineut)  = 0.0d0
       mion(ineut)  = nion(ineut)*mn + zion(ineut)*(mp+me)  - bion(ineut)*mev2gr
       wion(ineut)  = avo * mion(ineut)
       ionam(ineut) = 'neut'

       iprot        = 2
       aion(iprot)  = 1.0d0
       nion(iprot)  = 0.0d0
       zion(iprot)  = 1.0d0
       bion(iprot)  = 0.0d0
       mion(iprot)  = nion(iprot)*mn + zion(iprot)*(mp+me) - bion(iprot)*mev2gr
       wion(iprot)  = avo * mion(iprot)
       ionam(iprot) = 'prot'

       ihe4         = 3
       aion(ihe4)   = 4.0d0
       nion(ihe4)   = 2.0d0
       zion(ihe4)   = 2.0d0
       bion(ihe4)   = 28.29603d0
       mion(ihe4)   = nion(ihe4)*mn + zion(ihe4)*(mp+me) - bion(ihe4)*mev2gr
       wion(ihe4)  = avo * mion(ihe4)
       ionam(ihe4)  = 'he4'
      end if


! we keep returning here from various goto and loop constructions
60    i = i+1

! read in the z and a and any fitting constants
!      write(6,*) 'reading', i

      read(fil14,02) nz,na,(ibs1(j),ibs2(j), j=1,10)
02    format(2i6,20i3)

!      write(6,*) 'read', nz,na


      zion(i) = nz
      aion(i) = na
      if (nz .eq. 99) go to 120

! temperature dependent partition function information
      llmin = 5*(i-1)+1
      llmax = llmin + 4
      if (llmax .gt. 6*ionmax) then
         ierr = -1
         return
      end if
      read(fil14,03) nz,nn,bion(i),(as(ll),ll=llmin,llmax),ist(i),aname
03    format(2i3,f11.4,f5.1,4e12.3,i2,a10)

      nion(i) = nn

      if (ist(i).ne.0) then
      if (6*i-6+2*ist(i) .gt. 6*ionmax) then
         ierr = -1
         return
      end if
      read(fil14,04) (gs(ll),ll=6*i-5,6*i-6+2*ist(i))
      end if
04    format(f10.4,f10.3,f10.4,f10.3,f10.4,f10.3,f10.4,f10.3)


! decide if this isotope is in the network and branch accordingly

      do jj=1,inetin
       if (int(zion(i)) .eq. izzz(jj)  .and. &
           int(nion(i)) .ge. inmin(jj) .and. &
           int(nion(i)) .le. inmax(jj)) goto 90
      enddo

! not using this isotope, but
! do the read, backup i by one, and go back to 60 for another isotope
      do jj=1,10
       if (ibs1(jj) .gt. 0) then
        read(fil14,240) (dum1,j1=1,ibs2(jj))
240     format (7e13.6)
       end if
      enddo
      i = i - 1
      go to 60



! using this isotope, read parameters for reaction j on species i
90    continue
      if (i .gt. ionmax) then
         write(*,*) 'ionmax too small in init_torch'
         ierr = -1
         return
      end if


! here are the isotopes we are using
!      call sqeeze(aname)
!      write(6,117) i,int(zion(i)),int(aion(i)),aname
! 117  format(1x,3i4,' ',a)


      do j=1,8
       if (ibs1(j) .gt. 0) then
        kmax = k + ibs2(j)-1
        read(fil14,240)  (dum1,j1=k,kmax)
        k = kmax + 1
       end if
      enddo
! and go back for another isotope
      go to 60

! all done with the this part of the loading
120   continue




! append neutrons, protons and alfa if down arrow
! set the ending isotope index ionend
      if (downarrow) then
       ineut        = i
       aion(ineut)  = 1.0d0
       nion(ineut)  = 1.0d0
       zion(ineut)  = 0.0d0
       bion(ineut)  = 0.0d0
       mion(ineut)  = nion(ineut)*mn + zion(ineut)*(mp+me) - bion(ineut)*mev2gr
       wion(ineut)  = avo * mion(ineut)
       ionam(ineut) = 'neut'

       iprot        = i + 1
       aion(iprot)  = 1.0d0
       nion(iprot)  = 0.0d0
       zion(iprot)  = 1.0d0
       bion(iprot)  = 0.0d0
       mion(iprot)  = nion(iprot)*mn + zion(iprot)*(mp+me) - bion(iprot)*mev2gr
       wion(iprot)  = avo * mion(iprot)
       ionam(iprot) = 'prot'

       ihe4         = i + 2
       aion(ihe4)   = 4.0d0
       nion(ihe4)   = 2.0d0
       zion(ihe4)   = 2.0d0
       bion(ihe4)   = 28.29603d0
       mion(ihe4)   = nion(ihe4)*mn + zion(ihe4)*(mp+me) - bion(ihe4)*mev2gr
       wion(ihe4)   = avo * mion(ihe4)
       ionam(ihe4)  = 'he4'
      end if


! for either orientation, append energy, temperature, and denisty pointers

!       iener = ionmax + 1
!       itemp = ionmax + 2
!       iden  = ionmax + 3
!       ivelx = ionmax + 4
!       iposx = ionmax + 5
!       neqs  = iposx
!
!       ionam(iener) = 'ener '
!       ionam(itemp) = 'temp '
!       ionam(iden)  = 'den  '
!       ionam(ivelx) = 'velx '
!       ionam(iposx) = 'posx '


! number of neutrons, mass, molar mass
       nion(ionbeg:ionend) = aion(ionbeg:ionend) - zion(ionbeg:ionend)
       mion(ionbeg:ionend) = nion(ionbeg:ionend)*mn + zion(ionbeg:ionend)*(mp+me) - bion(ionbeg:ionend)*mev2gr
       wion(ionbeg:ionend) = avo * mion(ionbeg:ionend)
       wion(ionbeg:ionend) = aion(ionbeg:ionend)


! build the links between the isotopes in the network
! before reading the weak reaction rates

      call naray



! finally all done reading the nuclear reaction rate file
190   continue
      close(unit=fil14)



! set the isotope names and pointers
!       write(6,*) ' '
!       write(6,*) ' using isotopes:'
!       write(6,*) '   i   z   a      name'

       if (.not.downarrow) then
        i    = ineut
        inta = aion(i)
        intz = zion(i)
!        write(6,09) ineut,intz,inta,ineut,ionam(i)

        i    = iprot
        inta = aion(i)
        intz = zion(i)
!        write(6,09) iprot,intz,inta,ionam(i)

        i    = ihe4
        inta = aion(i)
        intz = zion(i)
!        write(6,09) ihe4,intz,inta,ionam(i)
       endif


       do i=ionbeg,ionend
        inta = aion(i)
        intz = zion(i)
        if (intz .ge. 1  .and.  intz .le. zmax) then
         if (inta .ge. aidmin(intz)  .and.  inta .le. aidmax(intz)) then
          do ii = aidmin(intz),aidmax(intz)
           if (ii .eq. inta) then

            write(string,01) zsymb(intz),inta
            call sqeeze(string)
            ionam(i) = string

! using this isotope
!            write(6,09) i,intz,inta,ionam(i)

           end if
          enddo
         else
          write(6,*) ' bad aion',inta,' in routine init_torch'
          write(6,*) ' zion=',intz
          write(6,*) ' amin=',aidmin(intz),' amax=',aidmax(intz)
          write(*,*) 'error: bad inta in routine init_torch'
          ierr = -1
          return
         end if
        else
         write(6,*) 'bad zion',intz,' in routine init_torch'
         write(6,*) 'inta =',inta,' zmax=', zmax
         write(*,*) 'error: bad intz in routine init_torch'
         ierr = -1
         return
        end if
       enddo

       if (downarrow) then
        i    = ineut
        inta = aion(i)
        intz = zion(i)
!        write(6,09) ionmax-2,intz,inta,ionam(i)

        i    = iprot
        inta = aion(i)
        intz = zion(i)
!        write(6,09) ionmax-1,intz,inta,ionam(i)

        i    = ihe4
        inta = aion(i)
        intz = zion(i)
!        write(6,09) ionmax,intz,inta,ionam(i)
       endif



! check some things
!      do i=1,ionmax
!       write(6,888) ionam(i), &
!                   int(zion(i)),int(nion(i)),int(aion(i)), &
!                   mion(i)*avo,(mion(i)*avo-aion(i)), &
!                   (mion(i)*avo-aion(i))/(mev2gr*avo)
! 888   format(1x,a,3i4,1p5e18.10)
!      enddo
!      read(5,*)



! set the id numbers of certain key isotopes
       do i=ionbeg,ionend
        if (ionam(i) .eq. 'h2   ')   then
         ih2 = i
        else if (ionam(i) .eq. 'h3   ') then
         ih3 = i
        else if (ionam(i) .eq. 'he3  ') then
         ihe3 = i
        else if (ionam(i) .eq. 'li6  ') then
         ili6 = i
        else if (ionam(i) .eq. 'li7  ') then
         ili7 = i
        else if (ionam(i) .eq. 'li8  ') then
         ili8 = i
        else if (ionam(i) .eq. 'be7  ') then
         ibe7 = i
        else if (ionam(i) .eq. 'be9  ') then
         ibe9 = i
        else if (ionam(i) .eq. 'b8   ') then
         ib8 = i
        else if (ionam(i) .eq. 'b9   ') then
         ib9 = i
        else if (ionam(i) .eq. 'b10  ') then
         ib10 = i
        else if (ionam(i) .eq. 'b11  ') then
         ib11 = i
        else if (ionam(i) .eq. 'c11  ') then
         ic11 = i
        else if (ionam(i) .eq. 'c12  ') then
         ic12 = i
        else if (ionam(i) .eq. 'c13  ') then
         ic13 = i
        else if (ionam(i) .eq. 'c14  ') then
         ic14 = i
        else if (ionam(i) .eq. 'n12  ') then
         in12 = i
        else if (ionam(i) .eq. 'n13  ') then
         in13 = i
        else if (ionam(i) .eq. 'n14  ') then
         in14 = i
        else if (ionam(i) .eq. 'n15  ') then
         in15 = i
        else if (ionam(i) .eq. 'o14  ') then
         io14 = i
        else if (ionam(i) .eq. 'o15  ') then
         io15 = i
        else if (ionam(i) .eq. 'o16  ') then
         io16 = i
        else if (ionam(i) .eq. 'o17  ') then
         io17 = i
        else if (ionam(i) .eq. 'o18  ') then
         io18 = i
        else if (ionam(i) .eq. 'f17  ') then
         if17 = i
        else if (ionam(i) .eq. 'f18  ') then
         if18 = i
        else if (ionam(i) .eq. 'f19  ') then
         if19 = i
        else if (ionam(i) .eq. 'ne18 ') then
         ine18 = i
        else if (ionam(i) .eq. 'ne19 ') then
         ine19 = i
        else if (ionam(i) .eq. 'ne20 ') then
         ine20 = i
        else if (ionam(i) .eq. 'ne21 ') then
         ine21 = i
        else if (ionam(i) .eq. 'ne22 ') then
         ine22 = i
        else if (ionam(i) .eq. 'ne23 ') then
         ine23 = i
        else if (ionam(i) .eq. 'na20 ') then
         ina20 = i
        else if (ionam(i) .eq. 'na21 ') then
         ina21 = i
        else if (ionam(i) .eq. 'na22 ') then
         ina22 = i
        else if (ionam(i) .eq. 'na23 ') then
         ina23 = i
        else if (ionam(i) .eq. 'mg22 ') then
         img22 = i
        else if (ionam(i) .eq. 'mg23 ') then
         img23 = i
        else if (ionam(i) .eq. 'mg24 ') then
         img24 = i
        else if (ionam(i) .eq. 'mg25 ') then
         img25 = i
        else if (ionam(i) .eq. 'mg26 ') then
         img26 = i
        else if (ionam(i) .eq. 'al25 ') then
         ial25 = i
        else if (ionam(i) .eq. 'al26 ') then
         ial26 = i
        else if (ionam(i) .eq. 'al27 ') then
         ial27 = i
        else if (ionam(i) .eq. 'si27 ') then
         isi27 = i
        else if (ionam(i) .eq. 'si28 ') then
         isi28 = i
        else if (ionam(i) .eq. 'si29 ') then
         isi29 = i
        else if (ionam(i) .eq. 'si30 ') then
         isi30 = i
        else if (ionam(i) .eq. 'p30  ') then
         ip30  = i
        else if (ionam(i) .eq. 'p31  ') then
         ip31  = i
        else if (ionam(i) .eq. 's30  ') then
         is30  = i
        else if (ionam(i) .eq. 's31  ') then
         is31  = i
        else if (ionam(i) .eq. 's32  ') then
         is32  = i
        else if (ionam(i) .eq. 'cl35 ') then
         icl35  = i
        else if (ionam(i) .eq. 'ar36 ') then
         iar36  = i
        else if (ionam(i) .eq. 'k39  ') then
         ik39  = i
        else if (ionam(i) .eq. 'ca40 ') then
         ica40  = i
        else if (ionam(i) .eq. 'sc43 ') then
         isc43  = i
        else if (ionam(i) .eq. 'sc45 ') then
         isc45  = i
        else if (ionam(i) .eq. 'ti44 ') then
         iti44  = i
        else if (ionam(i) .eq. 'ti46 ') then
         iti46  = i
        else if (ionam(i) .eq. 'ti48 ') then
         iti48  = i
        else if (ionam(i) .eq. 'ti50 ') then
         iti50  = i
        else if (ionam(i) .eq. 'v46  ') then
         iv46  = i
        else if (ionam(i) .eq. 'v47  ') then
         iv47  = i
        else if (ionam(i) .eq. 'v48  ') then
         iv48  = i
        else if (ionam(i) .eq. 'v51  ') then
         iv51  = i
        else if (ionam(i) .eq. 'cr47 ') then
         icr47  = i
        else if (ionam(i) .eq. 'cr48 ') then
         icr48  = i
        else if (ionam(i) .eq. 'cr49 ') then
         icr49  = i
        else if (ionam(i) .eq. 'cr50 ') then
         icr50  = i
        else if (ionam(i) .eq. 'cr51 ') then
         icr51  = i
        else if (ionam(i) .eq. 'cr52 ') then
         icr52  = i
        else if (ionam(i) .eq. 'cr53 ') then
         icr53  = i
        else if (ionam(i) .eq. 'cr54 ') then
         icr54  = i
        else if (ionam(i) .eq. 'mn50 ') then
         imn50  = i
        else if (ionam(i) .eq. 'mn51 ') then
         imn51  = i
        else if (ionam(i) .eq. 'mn52 ') then
         imn52  = i
        else if (ionam(i) .eq. 'mn55 ') then
         imn55  = i
        else if (ionam(i) .eq. 'fe52 ') then
         ife52 = i
        else if (ionam(i) .eq. 'fe54 ') then
         ife54 = i
        else if (ionam(i) .eq. 'fe55 ') then
         ife55 = i
        else if (ionam(i) .eq. 'fe56 ') then
         ife56 = i
        else if (ionam(i) .eq. 'fe57 ') then
         ife57 = i
        else if (ionam(i) .eq. 'fe58 ') then
         ife58 = i
        else if (ionam(i) .eq. 'co54 ') then
         ico54 = i
        else if (ionam(i) .eq. 'co55 ') then
         ico55 = i
        else if (ionam(i) .eq. 'co56 ') then
         ico56 = i
        else if (ionam(i) .eq. 'co59 ') then
         ico59 = i
        else if (ionam(i) .eq. 'ni56 ') then
         ini56 = i
        else if (ionam(i) .eq. 'ni58 ') then
         ini58 = i
        else if (ionam(i) .eq. 'ni59 ') then
         ini59 = i
        else if (ionam(i) .eq. 'ni64 ') then
         ini64 = i
        else if (ionam(i) .eq. 'ni66 ') then
         ini66 = i
        else if (ionam(i) .eq. 'cu63 ') then
         icu63 = i
        else if (ionam(i) .eq. 'zn60 ') then
         izn60 = i
        else if (ionam(i) .eq. 'zn64 ') then
         izn64 = i
        end if
       enddo


! say how many isotopes and rates are in this network
!       write(6,*)
!       write(6,*) 'ionmax=',ionmax,'  nrates=',nrat
!       write(6,*) 'minimum size of cx array ',kmax
!       write(6,*)

       return
       end subroutine init_torch


      subroutine sqeeze(line)

! this routine takes line and removes all blanks, such as
! those from writing to string with fortran format statements

! declare the pass
      character*(*) ::line

! declare local variables
      character*1    achar
      integer        l,n,k,lend,lsiz


! find the end of the line
      lsiz = len(line)
      lend = lenstr(line,lsiz)
      n    = 0
      l    = 0

! do the compression in place
10    continue
      l = l + 1
      achar = line(l:l)
      if (achar .eq. ' ') goto 10
      n = n + 1
      line(n:n) = achar
      if (l .lt. lend) goto 10

! blank the rest of the line
      do k=n+1,lsiz
       line(k:k) = ' '
      enddo
      return
      end subroutine sqeeze




      integer function lenstr(string,istrln)

! lenstr returns the non-blank length of a string.

! declare the pass
      integer, intent(in) ::   istrln
      character*(*), intent(in) :: string

! declare local variables
      integer :: i

! go
      lenstr=0
      do i=istrln,1,-1
       if (string(i:i).ne. ' ') then
        if (ichar(string(i:i)) .ne. 0) then
         lenstr=i
         return
        end if
       end if
      enddo
      return
      end function lenstr


      subroutine zet489(inetin,izzz,inmin,inmax)
!
! this routine sets up a 489 isotope torch network
!

! declare the pass
      integer          inetin,izzz(*),inmin(*),inmax(*)


! initialize
      inetin    = 0

! deuterium and tritium
      inetin         = inetin + 1
      izzz(inetin)   = 1
      inmin(inetin)  = 1
      inmax(inetin)  = 2

! helium 3
      inetin         = inetin + 1
      izzz(inetin)   = 2
      inmin(inetin)  = 1
      inmax(inetin)  = 1

! lithium 6-7
      inetin         = inetin + 1
      izzz(inetin)   = 3
      inmin(inetin)  = 3
      inmax(inetin)  = 4

! berylium 7-9
      inetin         = inetin + 1
      izzz(inetin)   = 4
      inmin(inetin)  = 3
      inmax(inetin)  = 5

! boron 8-11
      inetin         = inetin + 1
      izzz(inetin)   = 5
      inmin(inetin)  = 3
      inmax(inetin)  = 6

! carbon 11-14
      inetin         = inetin + 1
      izzz(inetin)   = 6
      inmin(inetin)  = 5
      inmax(inetin)  = 8

! nitrogen 12-15
      inetin         = inetin + 1
      izzz(inetin)   = 7
      inmin(inetin)  = 5
      inmax(inetin)  = 8

! oxygen 14-19
      inetin         = inetin + 1
      izzz(inetin)   = 8
      inmin(inetin)  = 6
      inmax(inetin)  = 11

! flourine 17-21
      inetin         = inetin + 1
      izzz(inetin)   = 9
      inmin(inetin)  = 8
      inmax(inetin)  = 12

! neon 17-24
      inetin         = inetin + 1
      izzz(inetin)   = 10
      inmin(inetin)  = 7
      inmax(inetin)  = 14

! sodium 19-27
      inetin         = inetin + 1
      izzz(inetin)   = 11
      inmin(inetin)  = 8
      inmax(inetin)  = 16

! magnesium 20-29
      inetin         = inetin + 1
      izzz(inetin)   = 12
      inmin(inetin)  = 8
      inmax(inetin)  = 17

! aluminum 22-31
      inetin         = inetin + 1
      izzz(inetin)   = 13
      inmin(inetin)  = 9
      inmax(inetin)  = 18

! silicon 23-34
      inetin        = inetin + 1
      izzz(inetin)  = 14
      inmin(inetin) = 9
      inmax(inetin) = 20

! phosphorus 27-38
      inetin        = inetin + 1
      izzz(inetin)  = 15
      inmin(inetin) = 12
      inmax(inetin) = 23

! sulfer 28-42
      inetin        = inetin + 1
      izzz(inetin)  = 16
      inmin(inetin) = 12
      inmax(inetin) = 26

! clorine 31-45
      inetin        = inetin + 1
      izzz(inetin)  = 17
      inmin(inetin) = 14
      inmax(inetin) = 28

! argon 32-46
      inetin        = inetin + 1
      izzz(inetin)  = 18
      inmin(inetin) = 14
      inmax(inetin) = 28

! potasium 35-49
      inetin        = inetin + 1
      izzz(inetin)  = 19
      inmin(inetin) = 16
      inmax(inetin) = 30

! calcium 36-49
      inetin        = inetin + 1
      izzz(inetin)  = 20
      inmin(inetin) = 16
      inmax(inetin) = 29

! scandium 40-51
      inetin        = inetin + 1
      izzz(inetin)  = 21
      inmin(inetin) = 19
      inmax(inetin) = 30

! titanium 41-53
      inetin        = inetin + 1
      izzz(inetin)  = 22
      inmin(inetin) = 19
      inmax(inetin) = 31

! vandium 43-55
      inetin        = inetin + 1
      izzz(inetin)  = 23
      inmin(inetin) = 20
      inmax(inetin) = 32

! chromium 44-58
      inetin        = inetin + 1
      izzz(inetin)  = 24
      inmin(inetin) = 20
      inmax(inetin) = 34

! manganese 46-61
      inetin        = inetin + 1
      izzz(inetin)  = 25
      inmin(inetin) = 21
      inmax(inetin) = 36

! iron 47-63
      inetin        = inetin + 1
      izzz(inetin)  = 26
      inmin(inetin) = 21
      inmax(inetin) = 37

! cobolt 50-65
      inetin        = inetin + 1
      izzz(inetin)  = 27
      inmin(inetin) = 23
      inmax(inetin) = 38

! nickel 51-67
      inetin        = inetin + 1
      izzz(inetin)  = 28
      inmin(inetin) = 23
      inmax(inetin) = 39

! copper 55-69
      inetin        = inetin + 1
      izzz(inetin)  = 29
      inmin(inetin) = 26
      inmax(inetin) = 40

! zinc 57-72
      inetin        = inetin + 1
      izzz(inetin)  = 30
      inmin(inetin) = 27
      inmax(inetin) = 42

! gallium 59-75
      inetin        = inetin + 1
      izzz(inetin)  = 31
      inmin(inetin) = 28
      inmax(inetin) = 44

! germanium 62-78
      inetin        = inetin + 1
      izzz(inetin)  = 32
      inmin(inetin) = 30
      inmax(inetin) = 46

! arsenic 65-79
      inetin        = inetin + 1
      izzz(inetin)  = 33
      inmin(inetin) = 32
      inmax(inetin) = 46

! selenium 67-83
      inetin        = inetin + 1
      izzz(inetin)  = 34
      inmin(inetin) = 33
      inmax(inetin) = 49

! bromine 68-83
      inetin        = inetin + 1
      izzz(inetin)  = 35
      inmin(inetin) = 33
      inmax(inetin) = 48

! krypton 69-87
      inetin        = inetin + 1
      izzz(inetin)  = 36
      inmin(inetin) = 33
      inmax(inetin) = 51

! rubidium 73-85
      inetin        = inetin + 1
      izzz(inetin)  = 37
      inmin(inetin) = 36
      inmax(inetin) = 48

! strontium 74-84
      inetin        = inetin + 1
      izzz(inetin)  = 38
      inmin(inetin) = 36
      inmax(inetin) = 46

! yttrium 77-87
      inetin        = inetin + 1
      izzz(inetin)  = 39
      inmin(inetin) = 36
      inmax(inetin) = 48

! zirconium 78-90
      inetin        = inetin + 1
      izzz(inetin)  = 40
      inmin(inetin) = 38
      inmax(inetin) = 50

! niobium 82-90
      inetin        = inetin + 1
      izzz(inetin)  = 41
      inmin(inetin) = 41
      inmax(inetin) = 49

! molybdenum 83-90
      inetin        = inetin + 1
      izzz(inetin)  = 42
      inmin(inetin) = 41
      inmax(inetin) = 48

! technetium 89-91
      inetin        = inetin + 1
      izzz(inetin)  = 43
      inmin(inetin) = 46
      inmax(inetin) = 48
      return
      end subroutine zet489




      subroutine get_ionmax(inetin,izzz,inmin,inmax,ierr)


! this routine gets the size of the network, setting
! the module integers ionmax, ionbeg, ionend

! declare the pass
      integer ::  inetin,izzz(*),inmin(*),inmax(*)
      integer, intent(out) :: ierr

! local variables
      logical            :: downarrow
      integer, parameter :: fil14=14
      character*10       :: aname
      integer            :: i,j,k,l,jj,ll,llmin,llmax,j1,nz,na,nn,kmax,&
                            idum1,idum2,ibs1(10),ibs2(10)
      real(dp)                bener,dum1

      ierr = 0
      
! decide on the arrow orientation
! downarrow true puts neutron, protons, alfa to the end
! downarrow false (i.e., uparrow configuration) puts neutron, protons, alfa at the beginning
! in general downarrowtrue is faster for dense linear algebra, and either
! orientation for sparse linear algebra. gift routines, however, do
! much better with downarrow false.

      downarrow = .true.
!      downarrow = .false.


! set the beginning isotope index
      if (downarrow) then
       ionbeg = 1
      else
       ionbeg = 4
      end if



! open the nuclear recation rate data file
! use a soft link to connect bdat to the desired burn data file

      open(unit=fil14, file=trim(mesa_data_dir) // '/nse_data/BDAT', &
         action='read', status='old', iostat=ierr)
      if (ierr /= 0) then
         write(*,*) 'NSE ERROR --- failed to open ' // trim(mesa_data_dir) // '/nse_data/BDAT'
         return
      end if

! initialize counters
      k        = 1
      i        = ionbeg - 1

! we keep returning here from various goto and loop constructions
60    i = i+1

! read in the z and a and any fitting constants
!      write(6,*) 'reading', i

      read(fil14,02) nz,na,(ibs1(j),ibs2(j), j=1,10)
02    format(2i6,20i3)

      if (nz .eq. 99) go to 120

! temperature dependent partition function information
      llmin = 5*(i-1)+1
      llmax = llmin + 4
      read(fil14,03) nz,nn,bener,(dum1,ll=llmin,llmax),idum1,aname
03    format(2i3,f11.4,f5.1,4e12.3,i2,a10)

      if (idum1 .ne. 0) then
       read(fil14,04) (dum1,ll=6*i-5,6*i-6+2*idum1)
      end if
04    format(f10.4,f10.3,f10.4,f10.3,f10.4,f10.3,f10.4,f10.3)


! decide if this isotope is in the network and branch accordingly

      do jj=1,inetin
       if (nz .eq. izzz(jj)  .and. &
           nn .ge. inmin(jj) .and. &
           nn .le. inmax(jj)) goto 90
      enddo

! not using this isotope, but
! do the read, backup i by one, and go back to 60 for another isotope
      do jj=1,10
       if (ibs1(jj) .gt. 0) then
        read(fil14,240) (dum1,j1=1,ibs2(jj))
240     format (7e13.6)
       end if
      enddo
      i = i - 1
      go to 60



! using this isotope, read parameters for reaction j on species i
90    continue
      do j=1,8
       if (ibs1(j) .gt. 0) then
        kmax = k + ibs2(j)-1
        read(fil14,240)  (dum1,j1=k,kmax)
        k = kmax + 1
       end if
      enddo

! and go back for another isotope
      go to 60

! all done with the this part of the loading
120   continue


! set the ending isotope index ionend
      if (downarrow) then
       ionmax       = i + 2
       ionend       = ionmax - 3

! for up-arrow configurations
      else
       ionmax = i - 1
       ionend = ionmax
      end if

      close(unit=fil14)

      return
      end subroutine get_ionmax












      subroutine naray


! this routine builds the nrr(7,i) and nrrneut(7,i) arrays, which specify
! the location of isotopes coupled to i by various reactions.
! the first index on nrr refers to reactions of the form
!      1=ng   2=pn   3=pg   4=ap   5=an   6=ag   7=b-
!
! while the first index on nrrneut refers to reactions of the form
!      1=nu,e-,n  2=nu,e-  3=nu,e-,p   4=nu e+,n   5=nu,e+   6=nu,e+,p


! declare
      integer      i,k,n,kz,kn,jz(7),jn(7)


! initialize for nrr
      jz(1) = 0
      jz(2) = 1
      jz(3) = 1
      jz(4) = 1
      jz(5) = 2
      jz(6) = 2
      jz(7) = -1

      jn(1) = 1
      jn(2) = -1
      jn(3) = 0
      jn(4) = 2
      jn(5) = 1
      jn(6) = 2
      jn(7) = 1

! build nrr
      do i=ionbeg,ionend
       do n=1,7
        nrr(n,i) = 0
        kz = int(zion(i)) + jz(n)
        kn = int(nion(i)) + jn(n)
        do k=ionbeg,ionend
         if (kz.eq.int(zion(k)) .and. kn.eq.int(nion(k))) nrr(n,i)=k
        enddo
       enddo
      enddo


! initialize for nrrneut
      jz(1) =  1
      jz(2) =  1
      jz(3) =  0
      jz(4) = -1
      jz(5) = -1
      jz(6) = -2
      jz(7) = -2
      jn(1) = -2
      jn(2) = -1
      jn(3) = -1
      jn(4) = 0
      jn(5) = 1
      jn(6) = 1
      jn(7) = -2

! build nrrneut
      do i=ionbeg,ionend
       do n=1,7
        nrrneut(n,i) = 0
        kz = int(zion(i)) + jz(n)
        kn = int(nion(i)) + jn(n)
        do k=ionbeg,ionend
         if (kz.eq.int(zion(k)) .and. kn.eq.int(nion(k))) nrrneut(n,i)=k
        enddo
       enddo
      enddo
      return
      end subroutine naray





      end module mod_nse_init

