! ***********************************************************************
!
!   Copyright (C) 2013  Frank Timmes, Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************
 
      module mod_nse
      
      use nse_def
      use chem_def
      use chem_lib
      use const_def, only: dp

      implicit none
      
      
      integer, parameter :: r_T9 = 1
      integer, parameter :: r_N_init = 2
      integer, parameter :: r_Rho = 3
      integer, parameter :: r_eta = 4
      integer, parameter :: r_dt = 5
      integer, parameter :: r_log10_Rho = 6
      integer, parameter :: r_log10_T = 7
      integer, parameter :: r_eps_neu = 8
      integer, parameter :: num_rparams = 8
      
      integer, parameter :: i_f_call = 1
      integer, parameter :: num_iparams = 1

      
      contains
      
      
      real(dp) function do_eval_nse_avg_eps_binding( &
            X_nse_init, X_nse, dt) result(eps_binding)
         ! average binding energy released in ergs/g/s
         ! from change in abundances over period dt
         use const_def, only: Qconv
         real(dp), pointer, intent(in) :: X_nse_init(:), X_nse(:)
		   real(dp), intent(in) :: dt
		   real(dp) :: eps_bind1, dY
		   integer :: i, ci
         eps_binding = 0.0d0
         do i=1,nse_species
            ci = nse_chem_id(i)
            dY = (X_nse(i) - X_nse_init(i))/chem_isos% Z_plus_N(ci)
            eps_bind1 = &
               dY*(chem_isos% binding_energy(ci) - &
                   chem_isos% Z(ci)*del_Mp - &
                   chem_isos% N(ci)*del_Mn)
            eps_binding = eps_binding + eps_bind1
         end do
         eps_binding = eps_binding*Qconv/dt
		end function do_eval_nse_avg_eps_binding
      

      ! sets X_nse to mass fractions for given T, Rho, ye
      subroutine do_get_nse_abundances(logT, logRho, ye, X_nse, ierr)
         use mod_nse_solve, only: do_nse_solve, torch_net_iso
         real(dp), intent(in) :: logT, logRho, ye
         real(dp), pointer, intent(out) :: X_nse(:) ! (nse_species)
         integer, intent(out) :: ierr
         
         real(dp) :: T, Rho, xmun, xmup, xsum
         real(dp), target :: xmnse_ary(nse_species)
         real(dp), pointer :: xmnse(:)
         integer, parameter :: ipartition = 1, icoulomb = 1
         integer :: i    
         include 'formats.dek'     
         ierr = 0
         T = 10d0**logT
         Rho = 10d0**logRho
         xmnse => xmnse_ary
         call do_nse_solve( &
            T, Rho, ye, ipartition, icoulomb, xmnse, xmun, xmup, ierr)
         if (ierr == 0) then
            xsum = sum(xmnse(1:nse_species))  
            do i=1,nse_species
               X_nse(i) = xmnse(torch_net_iso(nse_chem_id(i)))/xsum
            end do
         end if
      end subroutine do_get_nse_abundances

      
      ! sets X_net from X_nse while preserving ye
      subroutine do_convert_from_nse_abundances( &
            X_nse, species, net_isos, chem_id, X_net, ierr)
         use num_lib, only: qsort
         real(dp), pointer, intent(in) :: X_nse(:) ! (nse_species)
         integer, intent(in) :: species
         integer, pointer, intent(in) :: net_isos(:) ! (num_chem_isos)
            ! maps chem id to net iso number
         integer, pointer, intent(in) :: chem_id(:) ! (species)
            ! maps net iso number to chem id
         real(dp), pointer, intent(out) :: X_net(:) ! (species)
            ! net mass fractions
         integer, intent(out) :: ierr
         
         integer :: i, nse_cid, nse_Z, nse_N, j, cid, &
            max_Z, nse_i, nse_j, N, Z, Z_nxt, jj, alt_i, &
            alt_cid, alt_N, alt_Z
         integer, dimension(species) :: index
         real(dp), dimension(species) :: sort_values
         real(dp) :: xh, xhe, abar, ye_nse, &
            zbar, z2bar, ye, mass_correction, sumx, &
            alpha, beta, Y, n1, p1, n2, p2, x1, x2, &
            ye1, ye2, x1_nse, x2_nse, &
            nse_iso_ye, net_iso_ye, alt_net_iso_ye
         logical :: okay
         
         logical, parameter :: dbg = .false.
         
         include 'formats.dek'
         
         ierr = 0
         X_net(1:species) = 0d0
         
         call basic_composition_info( &
            nse_species, nse_chem_id, X_nse, xh, xhe, & 
            abar, zbar, z2bar, ye_nse, mass_correction, sumx)         
         if (dbg) write(*,1) 'do_convert_from_nse_abundances ye_nse', ye_nse

         max_Z = 0
         do i=1,species
            cid = chem_id(i)
            Z = chem_isos% Z(cid)
            N = chem_isos% N(cid)
            sort_values(i) = 1000*Z + N
            if (Z > max_Z) max_Z = Z
            index(i) = i
         end do
         
         okay = .true.
         do i=2,species
            if (sort_values(i) >= sort_values(i-1)) cycle
            okay = .false.
         end do
         
         if (.not. okay) then ! need to sort them
            call qsort(index, species, sort_values)
            if (dbg) then
               do j=1,species
                  i = index(j) ! i is the index in X_net for the species
                  cid = chem_id(i)
                  write(*,3) 'sort ' // chem_isos% name(cid), j, i, sort_values(i)
               end do
            end if
         end if
         
         ! use sorted species to partition sorted nse_species

         nse_j = nse_species
         do j=species,1,-1 ! j is for sorted list of species
            i = index(j) ! i is the index in X_net for the species
            cid = chem_id(i)
            Z = chem_isos% Z(cid)
            N = chem_isos% N(cid)
            net_iso_ye = Z/dble(Z + N)
            if (j > 1) then
               Z_nxt = chem_isos% Z(chem_id(index(j-1)))
            else
               Z_nxt = -1
            end if
            if (dbg) write(*,5) '***** net ' // chem_isos% name(cid), i, N, Z, Z_nxt
            do while (nse_j > 0)
               nse_i = nse_index(nse_j)
               nse_cid = nse_chem_id(nse_i)
               nse_Z = chem_isos% Z(nse_cid)
               nse_N = chem_isos% N(nse_cid)
               ! now decide if this nse iso goes with the current net iso
               if ((nse_Z == Z .and. (nse_N >= N .or. nse_Z > Z_nxt)) .or. &
                        (nse_Z > Z)) then
                  nse_iso_ye = nse_Z/dble(nse_Z + nse_N)
                  if (nse_iso_ye <= net_iso_ye) then
                     X_net(i) = X_net(i) + X_nse(nse_i)
                  else ! search for next net iso with ye >= nse iso ye
                     do jj=j,1,-1
                        alt_i = index(jj)
                        alt_cid = chem_id(alt_i)
                        alt_Z = chem_isos% Z(alt_cid)
                        alt_N = chem_isos% N(alt_cid)
                        alt_net_iso_ye = alt_Z/dble(alt_Z + alt_N)
                        if (alt_net_iso_ye >= nse_iso_ye .or. jj == 1) then
                           X_net(alt_i) = X_net(alt_i) + X_nse(nse_i)
                           exit
                        end if
                     end do
                  end if
                  
                  if (dbg) then
                     write(*,6) 'nse ' // chem_isos% name(nse_cid) // &
                         ' to net ' // chem_isos% name(cid), &
                         nse_i, nse_N, nse_Z, Z, Z_nxt, X_net(i)
                     if (nse_Z > Z) write(*,*) 'nse_Z > Z', nse_Z > Z
                     if (Z > Z_nxt) write(*,*) 'Z > Z_nxt', Z > Z_nxt
                     if (nse_Z == Z .and. nse_N >= N) &
                        write(*,*) 'nse_Z == Z .and. nse_N >= N', nse_Z == Z .and. nse_N >= N
                  end if
                  nse_j = nse_j - 1
               else ! no
                  exit
               end if
            end do
         end do
         
         X_net(1:species) = X_net(1:species)/sum(X_net(1:species))
            
         n1 = 0
         p1 = 0
         n2 = 0
         p2 = 0
         do i=1,species
            cid = chem_id(i)
            Z = chem_isos% Z(cid)
            N = chem_isos% N(cid)
            Y = X_net(i)/dble(Z+N)
            if (dble(Z)/dble(Z+N) >= ye_nse) then
               if (dbg) write(*,2) chem_isos% name(cid), 1
               n1 = n1 + N*Y
               p1 = p1 + Z*Y
            else ! more n rich than ye_nse
               if (dbg) write(*,2) chem_isos% name(cid), 2
               n2 = n2 + N*Y
               p2 = p2 + Z*Y
            end if
         end do

         x1 = (n1+p1)/(n1+p1+n2+p2)
         x2 = (n2+p2)/(n1+p1+n2+p2)
         ye1 = p1/(n1+p1)
         ye2 = p2/(n2+p2)
         x1_nse = (ye2 - ye_nse)/(ye2 - ye1)
         x2_nse = 1d0 - x1_nse
         alpha = x1_nse/x1
         beta = x2_nse/x2
         
         if (dbg) then
            write(*,1) 'x1', x1
            write(*,1) 'x2', x2
            write(*,1) 'ye1', ye1
            write(*,1) 'ye2', ye2
            write(*,1) 'ye', (p1+p2)/(n1+p1+n2+p2)
            write(*,1) 'x1*ye1 + x2*ye2', x1*ye1 + x2*ye2
            write(*,1) 'x1_nse', x1_nse
            write(*,1) 'x2_nse', x2_nse
            write(*,1) 'x1_nse*ye1 + x2_nse*ye2', x1_nse*ye1 + x2_nse*ye2
            write(*,1) 'ye_nse', ye_nse
            write(*,1) 'alpha', alpha
            write(*,1) 'beta', beta
         end if
      
         do i=1,species
            cid = chem_id(i)
            Z = chem_isos% Z(cid)
            N = chem_isos% N(cid)
            Y = X_net(i)/dble(Z+N)
            if (dble(Z)/dble(Z+N) >= ye_nse) then
               X_net(i) = X_net(i)*alpha
            else
               X_net(i) = X_net(i)*beta
            end if
         end do
         
         if (dbg) then
         
            do j=1,species
               cid = chem_id(j)
               write(*,2) chem_isos% name(cid), j, X_net(j)
            end do

            call basic_composition_info( &
               species, chem_id, X_net, xh, xhe, & 
               abar, zbar, z2bar, ye, mass_correction, sumx)
         
            if (abs(ye - ye_nse) > 1d-6) then
               write(*,1) 'ye - ye_nse', ye - ye_nse, ye, ye_nse, alpha
               stop 'nse: do_convert_from_nse_abundances'
            end if

            write(*,1) 'sum X_net', sum(X_net(1:species))
            write(*,1) 'ye_final - ye_nse', ye - ye_nse, ye, ye_nse, sumx
            stop
            
         end if

      end subroutine do_convert_from_nse_abundances


      subroutine do_advance_nse( &
            ! inputs
            dt, X_nse_init, ye_nse_init, log10_T, log10_Rho, eta, &
            atol, newt_imax, maxiters, &
            ! outputs
            X_nse, eps_binding, nse_eps_neu, ye_nse_final, &
            ierr)
            
         use num_lib, only: safe_root_without_brackets
         
         real(dp), intent(in) :: dt ! in seconds
         real(dp), pointer, intent(in) :: X_nse_init(:) ! (nse_species)
         real(dp), intent(in) :: ye_nse_init
         real(dp), intent(in) :: log10_T, log10_Rho, eta, atol
         integer, intent(in) :: newt_imax, maxiters
         real(dp), pointer, intent(out) :: X_nse(:) ! (nse_species)
         real(dp), intent(out) :: eps_binding, nse_eps_neu, ye_nse_final
         integer, intent(out) :: ierr
         
         real(dp) :: T9, Rho, Ye, YeRho, drate_dYe, N_init, &
            rate, total_N, total_Z_plus_N, final_total_N, final_total_Z, &
            equ, dequ_dY, Ye_lower_limit, Ye_upper_limit, dequ_dYe, dYe
         logical :: hit_limit
         real(dp), target :: Y_nse_a(nse_species)
         real(dp), pointer :: Y_nse(:)
         integer :: j, cid, N, Z_plus_N, iter, lrpar
         
         real(dp), target :: rpar_a(num_rparams + nse_species)
         real(dp), pointer :: rpar(:)
         
         integer, parameter :: lipar = num_iparams
         integer, target :: ipar_a(lipar)
         integer, pointer :: ipar(:)
         
         logical, parameter :: dbg = .false.
         
         include 'formats.dek'
          
         ierr = 0
         if (dbg) write(*,*)

         Y_nse => Y_nse_a

         do j=1,nse_species
            X_nse(j) = X_nse_init(j)
         end do

         T9 = 10**(log10_T - 9d0)
         Rho = 10**log10_Rho
         Ye = ye_nse_init
         YeRho = Ye*Rho
         
         N_init = 0 ! for compiler
         
         if (dbg) write(*,1) 'ye_nse_init', ye_nse_init
         
         total_N = 0
         total_Z_plus_N = 0
         do j=1,nse_species
            cid = nse_chem_id(j)
            N = chem_isos% N(cid)
            Z_plus_N = chem_isos% Z_plus_N(cid)
            Y_nse(j) = X_nse(j)/dble(Z_plus_N)
            total_N = total_N + N*Y_nse(j)
            total_Z_plus_N = total_Z_plus_N + Z_plus_N*Y_nse(j)
         end do
         N_init = total_N
         
         ipar => ipar_a
         ipar(i_f_call) = 0
         
         lrpar = num_rparams + nse_species
         rpar => rpar_a
         rpar(r_T9) = T9
         rpar(r_Rho) = Rho
         rpar(r_eta) = eta
         rpar(r_N_init) = N_init
         rpar(r_dt) = dt
         rpar(r_log10_Rho) = log10_Rho
         rpar(r_log10_T) = log10_T
         do j=1,nse_species
            rpar(num_rparams+j) = X_nse(j)
         end do

         ye_nse_final = safe_root_without_brackets( &
            nse_f, Ye, 0.05d0, newt_imax, maxiters, &
            atol, atol, lrpar, rpar, lipar, ipar, ierr)
         nse_eps_neu = rpar(r_eps_neu)
         do j=1,nse_species
            X_nse(j) = rpar(num_rparams+j)
         end do
                  
         if (ierr /= 0 .or. ye_nse_final > 1 .or. ye_nse_final < 0) then
            ierr = -1
            return

            write(*,*)
            write(*,2) 'do_advance_nse bogus ye_nse_final or ierr /= 0', ierr, ye_nse_final
            write(*,1) 'T9', T9
            write(*,1) 'Ye', Ye
            write(*,1) 'Rho', Rho
            write(*,1) 'eta', eta
            write(*,1) 'dt', dt
            write(*,*)
            write(*,1) 'ye_nse_final', ye_nse_final
            write(*,1) 'ye_nse_init', ye_nse_init
            write(*,1) 'final_total_N', final_total_N
            write(*,1) 'final_total_Z', final_total_Z
            write(*,1) 'rate dN/dt', rate
            write(*,1) 'drate_dYe', drate_dYe
            write(*,1) 'rate*dt', rate*dt
            write(*,1) 'total_N', total_N
            write(*,1) 'total_Z', total_Z_plus_N - total_N
            write(*,1) 'total_Z/(-rate)', (total_Z_plus_N - total_N)/(-rate)
            write(*,*)
            !stop 'do_advance_nse'


         end if
         
         eps_binding = &
            do_eval_nse_avg_eps_binding(X_nse_init, X_nse, dt)
         
         if (.not. dbg) return
         
         write(*,1) 'ye_nse_init', ye_nse_init
         write(*,1) 'ye_nse_final', ye_nse_final
         write(*,1) 'eps_binding', eps_binding
         write(*,1) 'nse_eps_neu', nse_eps_neu
         write(*,*)
         stop 'do_advance_nse'

      end subroutine do_advance_nse


      real(dp) function nse_f(Ye, df_dYe, lrpar, rpar, lipar, ipar, ierr)
         integer, intent(in) :: lrpar, lipar
         real(dp), intent(in) :: Ye
         real(dp), intent(out) :: df_dYe
         integer, intent(inout), pointer :: ipar(:) ! (lipar)
         real(dp), intent(inout), pointer :: rpar(:) ! (lrpar)
         integer, intent(out) :: ierr
         
         
         real(dp), target :: Y_nse_a(nse_species)
         real(dp), pointer :: Y_nse(:), X_nse(:)
         integer :: j, cid, N, Z_plus_N, f_call
         real(dp) :: T9, Rho, eta, dt, N_init, log10_Rho, log10_T, &
            total_N, nse_eps_neu, rate, drate_dYe
            
         logical, parameter :: dbg = .false.
         
         include 'formats'
         
         ierr = 0
         
         T9 = rpar(r_T9)
         Rho = rpar(r_Rho)
         eta = rpar(r_eta)
         dt = rpar(r_dt)
         N_init = rpar(r_N_init)
         log10_Rho = rpar(r_log10_Rho)
         log10_T = rpar(r_log10_T)
         
         f_call = ipar(i_f_call) + 1
         ipar(i_f_call) = f_call

         X_nse => rpar(num_rparams+1:lrpar)
         Y_nse => Y_nse_a
         
         call do_get_nse_abundances( &
            log10_T, log10_Rho, Ye, X_nse, ierr)
         if (ierr /= 0) then
            if (dbg) write(*,2) 'nse_f failed in do_get_nse_abundances', f_call
            return
         end if
         
         total_N = 0
         do j=1,nse_species
            cid = nse_chem_id(j)
            N = chem_isos% N(cid)
            Z_plus_N = chem_isos% Z_plus_N(cid)
            Y_nse(j) = X_nse(j)/dble(Z_plus_N)
            total_N = total_N + N*Y_nse(j)
         end do
   
         call do_eval_nse_eps_neu( &
            Y_nse, T9, Ye, Rho, eta, nse_eps_neu, rate, drate_dYe, ierr)
         if (ierr /= 0) then
            if (dbg) write(*,2) 'nse_f failed in do_eval_nse_eps_neu', f_call
            return
         end if
         rpar(r_eps_neu) = nse_eps_neu

         nse_f = N_init + dt*rate - total_N
         df_dYe = dt*drate_dYe + 1
         
         if (dbg) write(*,2) 'nse_f Ye nse_f', f_call, Ye, nse_f
      
         !df_dYe = df_dYe/0.6 ! undercorrection 
            ! necessary because not very accurate drate_dYe
            ! nothing magic about 0.6 -- just works well for a test case
         
      end function nse_f
      
      
      subroutine do_eval_nse_eps_neu( &
            Y_nse, T9, Ye, Rho, eta, nse_eps_neu, rate, drate_dYe, ierr)
         use weak_lib, only: eval_weak_reaction_info
         use weak_def, only: weak_lhs_nuclide_id, weak_rhs_nuclide_id
         use const_def, only: Qconv, ln10
         
         real(dp), pointer, intent(in) :: Y_nse(:)
		   real(dp), intent(in) :: T9, Ye, Rho, eta
		   real(dp), intent(out) :: nse_eps_neu ! ergs/g/s weak reaction neutrinos
		   real(dp), intent(out) :: rate ! weak reactions/s; dN/dt
		   real(dp), intent(out) :: drate_dYe
		   integer, intent(out) :: ierr

		   real(dp), dimension(nse_num_weaklib_rates), target :: &
		      ldecay_a, d_ldecay_dT9_a, d_ldecay_dlYeRho_a, &
		      lcapture_a, d_lcapture_dT9_a, d_lcapture_dlYeRho_a, &
		      lneutrino_a, d_lneutrino_dT9_a, d_lneutrino_dlYeRho_a, &
		      lambda_a, dlambda_dlnT_a, dlambda_dlnRho_a, &
		      Q_a, dQ_dlnT_a, dQ_dlnRho_a, &
		      Qneu_a, dQneu_dlnT_a, dQneu_dlnRho_a

		   real(dp), dimension(:), pointer :: &
		      ldecay, d_ldecay_dT9, d_ldecay_dlYeRho, &
		      lcapture, d_lcapture_dT9, d_lcapture_dlYeRho, &
		      lneutrino, d_lneutrino_dT9, d_lneutrino_dlYeRho, &
		      lambda, dlambda_dlnT, dlambda_dlnRho, &
		      Q, dQ_dlnT, dQ_dlnRho, &
		      Qneu, dQneu_dlnT, dQneu_dlnRho

         real(dp) :: d_eta_dlnT, d_eta_dlnRho, Y, YeRho, &
            d_lambda_dlYeRho, d_lambda_dYe
         integer :: j, ir, cid, i, delta_N

         include 'formats.dek'
         
         ierr = 0
         
         YeRho = Ye*Rho

         ldecay => ldecay_a
         d_ldecay_dT9 => d_ldecay_dT9_a
         d_ldecay_dlYeRho => d_ldecay_dlYeRho_a

         lcapture => lcapture_a
         d_lcapture_dT9 => d_lcapture_dT9_a
         d_lcapture_dlYeRho => d_lcapture_dlYeRho_a

         lneutrino => lneutrino_a
         d_lneutrino_dT9 => d_lneutrino_dT9_a
         d_lneutrino_dlYeRho => d_lneutrino_dlYeRho_a

         lambda => lambda_a
         dlambda_dlnT => dlambda_dlnT_a
         dlambda_dlnRho => dlambda_dlnRho_a

         Q => Q_a
         dQ_dlnT => dQ_dlnT_a
         dQ_dlnRho => dQ_dlnRho_a

         Qneu => Qneu_a
         dQneu_dlnT => dQneu_dlnT_a
         dQneu_dlnRho => dQneu_dlnRho_a
         
         d_eta_dlnT = 0
         d_eta_dlnRho = 0
         call eval_weak_reaction_info( &
		      nse_weaklib_ids, T9, YeRho, &
		      eta, d_eta_dlnT, d_eta_dlnRho, &
		      ldecay, d_ldecay_dT9, d_ldecay_dlYeRho, &
		      lcapture, d_lcapture_dT9, d_lcapture_dlYeRho, &
		      lneutrino, d_lneutrino_dT9, d_lneutrino_dlYeRho, &
		      lambda, dlambda_dlnT, dlambda_dlnRho, &
		      Q, dQ_dlnT, dQ_dlnRho, &
		      Qneu, dQneu_dlnT, dQneu_dlnRho, &
		      ierr)
		   if (ierr /= 0) return
		   
		   nse_eps_neu = 0
		   rate = 0
		   drate_dYe = 0
		   do j=1,nse_num_weaklib_rates
		      ir = nse_weaklib_ids(j)
		      cid = weak_lhs_nuclide_id(ir)
		      i = nse_net_iso(cid)
		      Y = Y_nse(i)
		      nse_eps_neu = nse_eps_neu + Y*Qneu(j)
		      delta_N = &
		         chem_isos% N(weak_rhs_nuclide_id(ir)) - chem_isos% N(cid)
		      rate = rate + delta_N*Y*lambda(j)		      
		      ! lambda = 10^ldecay + 10^lcapture
		      d_lambda_dlYeRho = ln10 * &
		         (10d0**ldecay(j)*d_ldecay_dlYeRho(j) + &
		          10d0**lcapture(j)*d_lcapture_dlYeRho(j))
		      d_lambda_dYe = d_lambda_dlYeRho/Ye		      
		      drate_dYe = drate_dYe + delta_N*Y*d_lambda_dYe
		   end do
		   nse_eps_neu = nse_eps_neu*Qconv
      
      end subroutine do_eval_nse_eps_neu      
      
      
      

      end module mod_nse

