! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module test_burn_const_P
      use const_def
      use chem_def
      use chem_lib
      use net_def
      use rates_def, only: num_rvs
      use test_net_support
      use mtx_def
      
      implicit none

      ! for burner
      real(dp), pointer :: x_initial(:) ! (species)

      real(dp) :: burn_T, burn_Rho, burn_neu_total, burn_lnE, burn_lnS
      real(dp) :: burn_logT, burn_logRho, burn_eta, burn_cp
      
      ! info at start of step
      real(dp) :: T_prev, time_prev, eps_nuc_prev, eps_neu_prev, cp_prev, pressure
      real(dp), pointer :: x_previous(:) ! (species)

      integer :: eos_handle, net_handle
      real(dp) :: total_energy ! u + P/rho - binding energy; for conservation checks
      real(dp) :: eps_nuc_max
      
      logical, parameter :: dbg = .true.
      
      logical :: show_by_step
      
      
      contains

      
      subroutine Do_One_Test_Burn_Const_P(net_file_in, show)
         use num_lib, only: isolve_name
         use mtx_lib, only: lapack_decsol, lapack_work_sizes
         use eos_lib
         
         character (len=*), intent(in) :: net_file_in
         logical, intent(in) :: show
         
         character (len=256) :: net_file
         real(dp) :: logRho, logT, Rho, T, xsum, tend, starting_temp,
     >     eps_nuc, d_eps_nuc_dRho, d_eps_nuc_dT, ending_temp, ending_rho, initial_rho,
     >     eps_nuc_categories(num_rvs, num_categories)
         real(dp), dimension(:,:), pointer :: rate_screened, rate_raw
         integer :: i, j, info
         
         integer :: which_solver ! as defined in num_def.f
         integer, parameter :: num_times = 1
         real(dp), dimension(:), pointer :: times, dxdt_source_term
         real(dp), dimension(:), pointer :: log10Ts_f1, log10Rhos_f1, etas_f1, log10Ps_f1
         real(dp), dimension(:,:), pointer :: log10Ts_f, log10Rhos_f, etas_f, log10Ps_f

         real(dp), pointer :: rate_factors(:) ! (num_reactions)
         real(dp) :: category_factors(num_categories)
         integer :: screening_mode
         real(dp) :: theta_e_for_graboske_et_al
         
         ! args to control the solver -- see num/public/num_isolve.dek
         real(dp) :: h 
         real(dp) :: max_step_size ! maximal step size.
         integer :: max_steps ! maximal number of allowed steps.
         ! absolute and relative error tolerances
         real(dp) :: rtol(1) ! relative error tolerance(s)
         real(dp) :: atol(1) ! absolute error tolerance(s)
         integer :: itol ! switch for rtol and atol
         
         real(dp), pointer :: ending_x(:) ! (species)
         integer :: nfcn    ! number of function evaluations
         integer :: njac    ! number of jacobian evaluations
         integer :: nstep   ! number of computed steps
         integer :: naccpt  ! number of accepted steps
         integer :: nrejct  ! number of rejected steps
         integer, parameter :: solver_name_len = 10
         character (len=solver_name_len) :: solver_name
         
         integer :: ierr, iout, caller_id, nvar
         real(dp) :: Prad, Pgas, lgPgas 
         real(dp) :: time_doing_net, time_doing_eos
         logical :: do_test, clip
         
         include 'formats.dek'
         
         ierr = 0
         eps_nuc_max = 0
         show_by_step = show
         net_file = net_file_in
         clip = .true.
                  
         do_test = (trim(net_file) == 'do test')
         if (do_test) net_file = 'ns_he.net'
         call test_net_setup(net_file) ! sets species, etc.
         net_handle = handle
         call Setup_eos(eos_handle)
         
         nvar = species + 1
         
         allocate(
     >      rate_factors(num_reactions), ending_x(species), 
     >      x_initial(species), x_previous(species), times(num_times),
     >      log10Ts_f1(4*num_times), log10Rhos_f1(4*num_times),
     >      etas_f1(4*num_times), log10Ps_f1(4*num_times),
     >      stat=ierr)
         if (ierr /= 0) then
            write(*,*) 'allocate failed for Do_One_Test_Burn'
            stop 1
         end if
         
         log10Ts_f(1:4,1:num_times) => log10Ts_f1(1:4*num_times)
         log10Rhos_f(1:4,1:num_times) => log10Rhos_f1(1:4*num_times)
         etas_f(1:4,1:num_times) => etas_f1(1:4*num_times)
         log10Ps_f(1:4,1:num_times) => log10Ps_f1(1:4*num_times)

         xin = 0
         eta = 0
         rate_factors(:) = 1
         category_factors(:) = 1
         screening_mode = extended_screening
         which_rates(:) = rates_NACRE_if_available
         eta = 0
         theta_e_for_graboske_et_al = 1
         max_step_size = 0
         max_steps = 5000
         iout = 1
         itol = 0         

         which_solver = 6 ! rodas4_solver
      
         rtol(:) = 1d-6
         atol(:) = 1d-6

         if (do_test) then ! for CK
                            
            tend = 1d4
         
            logT = 8.75d0
            pressure = 10d0**21.2d0
         
            T = 10**logT
         
            xin = 0
            xin(net_iso(ihe4)) =  0.930d0
            xin(net_iso(ic12)) =  0.045d0
            xin(net_iso(in14)) =  0.023d0
            xin(net_iso(io16)) =  0.002d0
         
         else if (net_file == 'approx20.net' .or. net_file == 'approx21.net') then
                            
            tend = 1d3
         
            T = 10**9.35
            pressure = 5d24
         
            xin = 0
            !xin(net_iso(ihe4)) =  1
            xin(net_iso(ic12)) =  0.5d0
            xin(net_iso(io16)) =  0.5d0
            
            logT = log10(T)
            
         else
                            
            tend = 1d6
         
            T = 10**8.718047058848855d0
            pressure = 10**2.1698890878720999d1
         
            xin = 0
            xin(net_iso(ihe4)) =  9.1616641498639240d-001
            xin(net_iso(ic12)) =  5.9277668539794576d-002
            xin(net_iso(in14)) =  9.5605170046636538d-005
            xin(net_iso(io16)) =  3.4350017605175656d-003
            xin(net_iso(ine20)) = 6.7988949426830893d-003
            xin(net_iso(img24)) = 1.4225931769121859d-002
            xin(net_iso(isi28)) = 1 - sum(xin)
            
            logT = log10(T)


            
         end if

         
         starting_temp = T
         times(1) = tend
         x_initial(1:species) = xin(1:species)
         x_previous(1:species) = xin(1:species)
         burn_neu_total = 0
         caller_id = 0
         log10Ps_f(1,1) = log10(pressure)
            
         if (.not. qt) write(*,*) 'Do_One_Test_Burn_Const_P ' // trim(net_file)
         
         if (show_by_step .and. .not. qt) then
            write(*,*) 'max_steps', max_steps
            write(*,fmt='(a7,99(a26,1x))',advance='no') 'i', 'time', 'lg_time', 'lg_eps_nuc', 'lg_ergs'
            do j=1,species
               write(*,fmt='(a26,1x)',advance='no') 'lg_' // trim(chem_isos% name(chem_id(j)))
            end do
            write(*,fmt='(99(a26,1x))') 'lg_abs_1_sub_sum_x', 'ye',
     >         'logT', 'logRho', 'lg_enthalpy', 'logS', 'logE', 'logP', 'lg_neu_total', 
     >         'log_Cp', 'eps_nuc_div_Cp_T', 'lg_age', 'pgas_div_ptotal'  
         end if
         
         time_doing_net = -1
         time_doing_eos = -1
         call net_1_zone_burn_const_P( 
     >         handle, eos_handle, species, num_reactions,
     >         which_solver, starting_temp, xin, clip,
     >         num_times, times, log10Ps_f1,
     >         rate_factors, category_factors, std_reaction_Qs, std_reaction_neuQs,
     >         screening_mode,
     >         h, max_step_size, max_steps, rtol, atol, itol,
     >         lapack, caller_id, burn_solout, iout,
     >         ending_x(:), ending_temp, ending_rho, initial_rho,
     >         nfcn, njac, nstep, naccpt, nrejct, time_doing_net, time_doing_eos, ierr)

         if (ierr /= 0) then
            write(*,*) 'net_1_zone_burn ierr', ierr
            stop 1
         end if
         
         if (.not. qt) then
         
            write(*,*)
            call isolve_name(which_solver, solver_name_len, solver_name)
            write(*,2) 'using ' // trim(solver_name)
            write(*,1) 'rtol', rtol(1)
            write(*,1) 'atol', atol(1)
            write(*,*)
            write(*,1) 'at time (sec)', tend
            write(*,1) 'at time (yr)', tend/secyer
            write(*,1) 'pressure', pressure, log10(pressure)
            write(*,1) 'starting_temp', starting_temp, log10(starting_temp)
            write(*,1) 'ending_temp', ending_temp, log10(ending_temp)
            write(*,1) 'starting_density', initial_rho, log10(initial_rho)
            write(*,1) 'ending_density', ending_rho, log10(ending_rho)
            if (show_by_step) then
               write(*,*)
               write(*,'(a64,99a26)') 'init mass fraction', 'final mass fraction', 'change since start'
               do j=1,species
                  if (chem_id(j) /= ihe4) then
                     if (x_initial(j) < 1d-5 .and. ending_x(j) < 1d-5) cycle
                     if (abs(ending_x(j) - x_initial(j)) < 1d-10) cycle
                  end if
                  write(*,1) trim(chem_isos% name(chem_id(j))), x_initial(j), ending_x(j), 
     >               ending_x(j) - x_initial(j)
               end do
               write(*,*)
               write(*,2) 'nfcn', nfcn
               write(*,2) 'njac', njac
               write(*,2) 'nstep', nstep
               write(*,2) 'naccpt', naccpt
               write(*,2) 'nrejct', nrejct
            else 
               do j=1,species
                  if (ending_x(j) < 1d-2) cycle
                  write(*,1) trim(chem_isos% name(chem_id(j))), x_initial(j), ending_x(j), 
     >               ending_x(j) - x_initial(j)
               end do
            end if
         
            write(*,*)
            write(*,*) 'Do_One_Test_Burn_Const_P ' // trim(net_file)
            write(*,*)
         
         end if
         
         
         deallocate(
     >      rate_factors, ending_x,
     >      x_initial, x_previous, times,
     >      log10Ts_f1, log10Rhos_f1,
     >      etas_f1, log10Ps_f1)
         
         
         contains
         
         
         subroutine turn_off(rate_id)
            integer, intent(in) :: rate_id
            integer :: rn
            rn = reaction_table(rate_id)
            if (rn == 0) return
            rate_factors(rn) = 0
            write(*,*) 'turn off ' // trim(reaction_name(rate_id))
         end subroutine turn_off
         
      end subroutine Do_One_Test_Burn_Const_P


      subroutine burn_solout(
     >         nr, told, time, nvar, v, rwork_y, iwork_y, interp_y, lrpar, rpar, lipar, ipar, irtrn)
         use chem_lib
         use chem_def
         use eos_def
         use num_lib, only: safe_log10
         use eos_lib, only: Radiation_Pressure, eosPT_get
         integer, intent(in) :: nr, nvar, lrpar, lipar
         real(dp), intent(in) :: told, time
         real(dp), intent(inout) :: v(nvar)
         ! v can be modified if necessary to keep it in valid range of possible solutions.
         real(dp), intent(inout), target :: rwork_y(*)
         integer, intent(inout), target :: iwork_y(*)
         integer, intent(inout), pointer :: ipar(:) ! (lipar)
         real(dp), intent(inout), pointer :: rpar(:) ! (lrpar)
         interface
            include 'num_interp_y.dek'
         end interface
         integer, intent(out) :: irtrn ! < 0 causes solver to return to calling program.


         real(dp) :: logRho, logT, Prad, Pgas, lgPgas, xh, Y
         real(dp) :: eps_neu_total, eps_nuc, d_eps_nuc_dRho, d_eps_nuc_dT,
     >     eps_nuc_categories(num_rvs, num_categories), mass_correction !approx_abar, approx_zbar
         real(dp), dimension(:, :), pointer :: reaction_eps_nuc
         real(dp), dimension(:, :), pointer :: rate_screened, rate_raw

         real(dp) :: Rho, T, Cp, xsum, dx, dt, enuc, burn_ergs, energy, enthalpy
         real(dp) :: dlnRho_dlnPgas_const_T, dlnRho_dlnT_const_Pgas
         real(dp) :: dlnRho_dlnT_const_P, d_epsnuc_dlnT_const_P, d_Cp_dlnT
         real(dp) :: res(num_eos_basic_results)
         real(dp) :: d_dlnRho_const_T(num_eos_basic_results) 
         real(dp) :: d_dlnT_const_Rho(num_eos_basic_results) 
         real(dp) :: d_dabar_const_TRho(num_eos_basic_results) 
         real(dp) :: d_dzbar_const_TRho(num_eos_basic_results) 
         
         integer :: i, j, lwork, adjustment_iso, cid, ierr, num_isos
         real(dp), dimension(nvar-1) :: dabar_dx, dzbar_dx, dmc_dx, eps_nuc_dx, x, d_eps_nuc_dx
         real(dp), pointer :: work(:), rate_factors(:), category_factors(:)

         include 'formats.dek'
         
         irtrn = 0
         if (time == 0) return

         ierr = 0
         call net_ptr(handle, g, ierr)
         if (ierr /= 0) then
            write(*,*) 'invalid handle for net_get -- did you call alloc_net_handle?'
            stop
         end if

         lwork = net_work_size(net_handle, ierr) 
         if (ierr /= 0) then
            write(*,*) 'invalid handle for net_get -- did you call alloc_net_handle?'
            stop 1
         end if
         
         num_isos = nvar - 1
         
         v(1:num_isos) = max(0d0, min(1d0, v(1:num_isos)))
         xsum = sum(v(1:num_isos))
         v(1:num_isos) = v(1:num_isos)/xsum
         
         allocate(work(lwork),
     >         rate_factors(num_reactions), category_factors(num_categories),
     >         rate_screened(num_rvs, num_reactions),     
     >         reaction_eps_nuc(num_rvs, num_reactions),      
     >         rate_raw(num_rvs, num_reactions), stat=ierr)
         if (ierr /= 0) stop 2
         
         v(1:num_isos) = max(1d-30, min(1d0, v(1:num_isos))) ! positive definite mass fractions
         x(1:num_isos) = v(1:num_isos)

         call composition_info(
     >            num_isos, chem_id, x, xh, Y, abar, zbar, z2bar, ye, 
     >            mass_correction, xsum, dabar_dx, dzbar_dx, dmc_dx)
     
         logT = v(nvar)/ln10
         T = 10**logT
         Prad = Radiation_Pressure(T)
         Pgas = pressure - Prad
         lgPgas = log10(Pgas)
         
         call eosPT_get(
     >         eos_handle, 1 - (xh + Y), xh, abar, zbar, 
     >         num_isos, chem_id, net_iso, x,
     >         Pgas, lgPgas, T, logT, 
     >         Rho, logRho, dlnRho_dlnPgas_const_T, dlnRho_dlnT_const_Pgas, 
     >         res, d_dlnRho_const_T, d_dlnT_const_Rho, 
     >         d_dabar_const_TRho, d_dzbar_const_TRho, ierr)
         if (ierr /= 0) then
            write(*,*) 'eosPT_get failed'
            write(*,1) 'xh', xh
            write(*,1) 'Y', Y
            write(*,1) 'Z', 1 - (xh + Y)
            write(*,1) 'abar', abar
            write(*,1) 'zbar', zbar
            write(*,1) 'pressure', pressure
            write(*,1) 'Prad', Prad
            write(*,1) 'Pgas', Pgas
            write(*,1) 'lgPgas', lgPgas
            write(*,1) 'T', T
            write(*,1) 'logT', logT
            write(*,1) 'Rho', Rho
            write(*,1) 'logRho', logRho
            stop
            return
         end if
         Cp = res(i_Cp)
         burn_lnE = res(i_lnE)
         burn_lnS = res(i_lnS)
         energy = exp(burn_lnE)
         enthalpy = energy + pressure/Rho
         
         rate_factors(:) = 1
         category_factors(:) = 1
         call net_get(handle, species, num_reactions, 
     >            xin, T, logT, Rho, logRho, 
     >            abar, zbar, z2bar, ye, eta, d_eta_dlnT, d_eta_dlnRho,
     >            rate_factors, category_factors, 
     >            std_reaction_Qs, std_reaction_neuQs,
     >            eps_nuc, d_eps_nuc_dRho, d_eps_nuc_dT, d_eps_nuc_dx, 
     >            dxdt, d_dxdt_dRho, d_dxdt_dT, d_dxdt_dx, 
     >            screening_mode, theta_e_for_graboske_et_al,     
     >            rate_screened, rate_raw, 
     >            reaction_eps_nuc, eps_nuc_categories, eps_neu_total,
     >            lwork, work, ierr)
         if (ierr /= 0) then
            write(*,1) 'logT', logT
            write(*,1) 'logRho', logRho
            write(*, *) 'bad return from net_get'
            stop 1
         end if
         if (.false. .and. time > 1d45) then
            write(*,1) 'time', time
            write(*,1) 'eps_nuc', eps_nuc
            write(*,1) 'T =', T
            write(*,1) 'logT =', logT
            write(*,1) 'Rho =', Rho
            write(*,1) 'logRho =', logRho
            write(*,1) 'abar =', abar
            write(*,1) 'zbar =', zbar
            write(*,1) 'z2bar =', z2bar
            write(*,1) 'ye =', ye
            write(*,1) 'eta =', eta
            write(*,*)
            do j=1,species
               write(*,1) 'xin(net_iso(i' // trim(chem_isos% name(chem_id(j))) // ')) =', x(j)
            end do
            write(*,*)
            do j=1,num_reactions
               if (abs(reaction_eps_nuc(i_rate,j)) < 1d8) cycle
               write(*, 1) trim(reaction_name(reaction_id(j))), 
     >               reaction_eps_nuc(i_rate,j), rate_screened(i_rate,j)
            end do
            write(*,*)
            stop 'solout'
         end if
         deallocate(work, rate_factors, category_factors, 
     >            rate_screened, reaction_eps_nuc, rate_raw)
         
         dt = time - told

         ! set burn_ergs according to change from initial abundances
         burn_neu_total = burn_neu_total + eps_neu_total*dt
         burn_ergs = 0
         do i=1,num_isos
            cid = chem_id(i)
            dx = x(i) - x_initial(i)
            burn_ergs = burn_ergs + 
     >            (chem_isos% binding_energy(cid) - chem_isos% Z(cid)*del_Mp - 
     >                 chem_isos% N(cid)*del_Mn)*dx/chem_isos% Z_plus_N(cid)
         end do
         burn_ergs = burn_ergs*Qconv - burn_neu_total

         total_energy = 1
         burn_logRho=logRho
         burn_logT = logT
         
         x_previous(1:species) = x(1:species)
                  
         if (show_by_step .and. .not. qt) then
            write(*,'(i7,99(1pe26.16,1x))') nr, 
     >         time, safe_log10(time), safe_log10(eps_nuc), safe_log10(burn_ergs),
     >         safe_log10(max(1d-16,x(1:species))), safe_log10(abs(1d0 - sum(xin(1:species)))), ye, 
     >         burn_logT, burn_logRho, safe_log10(enthalpy), burn_lnS/ln10, burn_lnE/ln10, 
     >         safe_log10(pressure), safe_log10(burn_neu_total),
     >         safe_log10(Cp), eps_nuc/(Cp*T), safe_log10(time/secyer), Pgas/pressure 
         end if

      end subroutine burn_solout


      end module test_burn_const_P

