! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module net_def
      
      use const_def, only: dp
      
      implicit none
      
      
      
      
      logical, parameter :: do_new_way = .false.
      
      
      
      
      ! nets
      
         ! the 'basic net' is for standard hydrogen and helium burning.
         
            ! it includes 3 PP chains, 4 CNO cycles, triple alpha, and alpha captures up to mg24.      
            ! the 8 isotopes included are h1, he3, he4, c12, n14, o16, ne20, and mg24.
            ! see basic_net.dek for the details
               
         ! to go beyond the basics, use any combination of the following extensions.
         
            ! to add C13, see add_c13.dek
            ! to add Ne22, see add_ne22.dek
            ! to add O18 and Ne22, see add_o18_and_ne22.dek
            ! for more PP reactions, see add_pp_extras.dek
            ! for more CNO reactions including hot CNO, see add_cno_extras.dek
            ! for carbon/oxygen burning, see add_co_burn.dek
            ! for links of alpha chain to ni56, see add_alpha_s32.dek, etc.
            
         ! there are also several special purpose nets you can use.
            
            ! 'net_18_to_mg24' includes 18 isotopes from h1 to mg24.
            ! it gives you all the energy generation you need
            ! for hydrogen and helium burning, so for stars with less 
            ! than 6-8 Msun this is good to get you to the WD stage.
            ! includes basic PP chains, CNO cycles, and helium burning.
            
            ! 'net_16_to56' uses only the basic 8 isotopes plus
            ! 8 more "multi-alpha" isotopes from si28 to ni56
            ! to cover evolution of massive stars from zams 
            ! through advance burning by alpha captures up to ni56.
            
            ! 'net_18_to_ni56' adds o18 and ne22 to net_16_to_ni56.
      


   ! reactions

      
      ! predefined nets and extensions      
         
         integer, parameter :: bit_for_Basic = 0  ! net number bits start at 0
         integer, parameter :: num_Basic = 2**bit_for_Basic
         integer, parameter :: num_isos_for_Basic = 8
            
         ! bits 1 to 29 are for extensions to the basic net -- only some numbers are used.
         
         integer, parameter :: bit_for_PP_extras = 1  
         integer, parameter :: bit_for_CNO_extras = 2  
         integer, parameter :: bit_for_C13 = 3  
         integer, parameter :: bit_for_O18_and_Ne22 = 4  
         integer, parameter :: bit_for_CO_burn = 5  

         integer, parameter :: bit_for_alpha_to_S32 = 10
         integer, parameter :: bit_for_alpha_to_Ar36 = 11
         integer, parameter :: bit_for_alpha_to_Ca40 = 12
         integer, parameter :: bit_for_alpha_to_Ti44 = 13 
         integer, parameter :: bit_for_alpha_to_Cr48 = 14  
         integer, parameter :: bit_for_alpha_to_Fe52 = 15
         integer, parameter :: bit_for_alpha_to_Ni56 = 16

         integer, parameter :: bit_for_add_Na23_to_alpha_Mg24 = 17
         integer, parameter :: bit_for_add_Al27_to_alpha_Si28 = 18
         integer, parameter :: bit_for_add_P31_to_alpha_S32 = 19
         integer, parameter :: bit_for_add_Cl35_to_alpha_Ar36 = 20
         integer, parameter :: bit_for_add_K39_to_alpha_Ca40 = 21
         integer, parameter :: bit_for_add_Sc43_to_alpha_Ti44 = 22
         integer, parameter :: bit_for_add_V47_to_alpha_Cr48 = 23
         integer, parameter :: bit_for_add_Mn51_to_alpha_Fe52 = 24
         ! note: Co55 is automatically included in alpha_Ni56

         integer, parameter :: bit_for_specific_nets = 30 
         
         ! "special nets" are nets that are not combos of the basic net plus extensions
         ! NOTE: all nets are required to include the basic net isos.
         ! beyond that, the special nets can do anything they want.
         ! 2**bit_for_specific_nets = 2**30 = 1073741824
         
         integer, parameter :: num_19_to_Ni56 = 2**bit_for_specific_nets ! 1073741824
         integer, parameter :: num_18_to_Mg24 = 2**bit_for_specific_nets+1 ! 1073741825

      ! for reaction_kind array in Net_General_Info
         integer, parameter :: other_kind = 0
         integer, parameter :: pg_kind = other_kind + 1
         integer, parameter :: ag_kind = pg_kind + 1
         integer, parameter :: ng_kind = ag_kind + 1
         integer, parameter :: ap_kind = ng_kind + 1
         integer, parameter :: an_kind = ap_kind + 1
         integer, parameter :: pn_kind = an_kind + 1

         
      type Net_General_Info ! things that are constant for the particular net
      ! it is okay to have multiple threads using the same instance of this simultaneously.

         integer :: num_isos ! total number in current net            
         integer :: num_reactions ! total number of reactions for current net
         
         character (len=32) :: cache_suffix

         ! isotopes
         integer, pointer :: net_iso(:) ! maps chem id to net iso number
         ! index from 1 to num_chem_isos
         ! value is 0 if the iso is not in the current net
         ! else is value between 1 and num_isos in current net
         integer, pointer :: chem_id(:) ! maps net iso number to chem id
         ! index from 1 to num_isos in current net
         ! value is between 1 and num_chem_isos         

         ! reactions

         integer, pointer :: which_rates(:) ! maps reaction id to small integer indicating choice for rate
                  
         integer, pointer :: net_reaction(:) ! maps reaction id to net reaction number
         ! index from 1 to rates_reaction_id_max (in rates_def)   
         ! value is 0 if the reaction is not in the current net
         ! else is value between 1 and num_reactions in current net
         integer, pointer :: reaction_id(:) ! maps net reaction number to reaction id
         ! index from 1 to num_reactions in current net
         ! value is between 1 and rates_reaction_id_max (in rates_def)     

         integer, pointer :: reaction_kind(:)
            ! pg_kind, ag_kind, ng_kind for 2 to 1 reactions
            ! ap_kind, an_kind, pn_kind for 2 to 2 reactions
            ! other_kind for the rest
         integer, pointer :: reverse_id_for_kind_ne_other(:)
         
         integer, pointer :: reaction_max_Z(:)
         integer, pointer :: reaction_max_Z_plus_N_for_max_Z(:)
         
         ! T limit controls
         real(dp) :: 
     >      T_lo_prot, T_hi_prot,
     >      lnT_lo_prot, lnT_hi_prot,
     >      min_ln_factor_prot, min_factor_prot
         
         real(dp) :: 
     >      T_lo_combo_a_capture, T_hi_combo_a_capture,
     >      lnT_lo_combo_a_capture, lnT_hi_combo_a_capture,
     >      min_ln_factor_combo_a_capture, min_factor_combo_a_capture
         
         real(dp) :: 
     >      T_lo_a_cap_high_mass, T_hi_a_cap_high_mass,
     >      lnT_lo_a_cap_high_mass, lnT_hi_a_cap_high_mass,
     >      min_ln_factor_a_cap_high_mass, min_factor_a_cap_high_mass
         
         real(dp) :: 
     >      T_lo_a_cap_intermediate, T_hi_a_cap_intermediate,
     >      lnT_lo_a_cap_intermediate, lnT_hi_a_cap_intermediate,
     >      min_ln_factor_a_cap_intermediate, min_factor_a_cap_intermediate
         
         real(dp) :: 
     >      T_lo_neut, T_hi_neut,
     >      lnT_lo_neut, lnT_hi_neut,
     >      min_ln_factor_neut, min_factor_neut
         
         ! extra info
         
         ! rates cutoff smoothly for logT < logTcut_lim
         real(dp) :: logTcut_lim
         ! rates are zero logT < logTcut_lo
         real(dp) :: logTcut_lo
   
         ! the following is private info for the implementation
         
         ! tables for graboske screening
         real(dp), pointer :: zg1(:) ! (num_reactions)
         real(dp), pointer :: zg2(:) ! (num_reactions)
         real(dp), pointer :: zg3(:) ! (num_reactions)
         real(dp), pointer :: zg4(:) ! (num_reactions)
         
         ! tables for screen5
         real(dp), pointer :: zs13(:) ! (num_reactions) ! zs13 = (z1+z2)**(1./3.)
         real(dp), pointer :: zhat(:) ! (num_reactions)
         real(dp), pointer :: zhat2(:) ! (num_reactions)
         real(dp), pointer :: lzav(:) ! (num_reactions)
         real(dp), pointer :: aznut(:) ! (num_reactions)
         real(dp), pointer :: zs13inv(:) ! (num_reactions) ! zs13inv = 1 / zs13
   
         ! info for evaluation of the raw reaction rates
         real(dp), pointer :: rattab(:,:) ! (num_reactions, nrattab)
         real(dp), pointer :: ttab(:) ! (nrattab)
         real(dp), pointer :: logttab(:) ! (nrattab)
         real(dp), pointer :: rattab_f1(:) ! =(4,nrattab,num_reactions) ! for interpolation

         ! info for evaluation of weak rates by weaklib
         integer :: num_weaklib_rates
         integer, pointer :: weaklib_ids(:), weak_reaction_index(:), weak_reaction_num(:)
         
         ! top level file name for net
         character (len=256) :: net_filename
         
         ! timing
         logical :: doing_timing
         logical :: doing_derivs_timing
         ! the following are sums of results from system_clock.  divide by clock_rate to get seconds.
         ! must set all of these to 0 before change doing_timing to true.
         integer :: clock_net_eval
         integer :: clock_net_weak_rates
         integer :: clock_net_rate_tables
         integer :: clock_net_screen
         integer :: clock_net_derivs
         integer :: clock_derivs_select
         integer :: clock_derivs_setup
         integer :: clock_derivs_general
         integer :: clock_net_get
         
         ! bookkeeping
         integer :: handle
         logical :: net_has_been_defined
         logical :: in_use

      end type Net_General_Info
               
               
      type Net_Info
         ! this is private working storage for the nuclear reaction calculations

         type (Net_General_Info), pointer  :: g

         real(dp), pointer :: reaction_Qs(:) ! if null, use standard values         
         real(dp), pointer :: reaction_neuQs(:) ! if null, use standard values

         ! molar fractions and their rates of change
         real(dp), pointer :: y(:) ! units [moles/gram]     (num_isos)
         real(dp), pointer :: dydt(:, :) ! units [moles/(gram-second)]  (num_rvs, num_isos)
         real(dp), pointer :: d_dydt_dy(:, :) ! units [1/second] (num_isos, num_isos)

         integer :: screening_mode
         real(dp) :: theta_e_for_graboske_et_al
         real(dp), pointer :: graboske_cache(:, :, :)

         real(dp), pointer :: rate_screened(:, :) ! (num_rvs, num_rates)
         ! the units here depend on the number of reactants.
         ! in all cases, the rate_screened times as many molar fractions as there are reactants
            ! gives a number with the same units as dy/dt.
         ! so for a 2-body reaction, there are 2 Y factors, each with units [moles/gram]
            ! and the rate_screened units for such a reaction are [grams/(mole-sec)], 
            ! which when multiplied by [moles/gram]^2 gives the same units as dydt.
         ! for a 1-body reaction (e.g., a decay), there is only 1 Y factor, so the units are [1/second].
         ! similarly, a 3 body reaction will have rate_screened with units of [gram^2/(mole^2-sec)].

         real(dp), pointer :: rate_raw(:, :) ! (num_rvs, num_rates)
         ! raw rates are unscreened (but include density factors)

         real(dp), pointer :: reaction_eps_nuc(:, :) ! (num_rvs, num_rates)
         
         real(dp), pointer :: d_eps_nuc_dy(:) ! (num_isos)

         real(dp), pointer :: eps_nuc_categories(:, :) ! (num_rvs, num_categories)
         ! eps_nuc subtotals for each reaction category

         real(dp) :: eps_neu_total
         
         ! weaklib results
         real(dp), dimension(:), pointer ::
     >      ldecay, d_ldecay_dT9, d_ldecay_dlYeRho, 
     >      lcapture, d_lcapture_dT9, d_lcapture_dlYeRho, 
     >      lneutrino, d_lneutrino_dT9, d_lneutrino_dlYeRho, 
     >      lambda, dlambda_dlnT, dlambda_dlnRho, 
     >      Q, dQ_dlnT, dQ_dlnRho, 
     >      Qneu, dQneu_dlnT, dQneu_dlnRho
     
         ! args for derivs
         real(dp) ::
         
     >      factor_prot, d_factor_prot_dT, 
     >      factor_combo_a_capture, d_factor_combo_a_capture_dT, 
     >      factor_a_cap_high_mass, d_factor_a_cap_high_mass_dT, 
     >      factor_a_cap_intermediate, d_factor_a_cap_intermediate_dT, 
     >      factor_neut, d_factor_neut_dT, 
     
     >      rpen, rnep, spen, snep, 
     >      d_rhe4_breakup_dneut, d_rhe4_breakup_dprot,
     >      d_rhe4_rebuild_dneut, d_rhe4_rebuild_dprot,
     >      d_rhe4_to_2h1_2neut_dneut, d_rhe4_to_2h1_2neut_dh1,
     >      d_rhe4_from_2h1_2neut_dneut, d_rhe4_from_2h1_2neut_dh1,
     
     >      d_ro18ap_to_ne22_dp, d_rne22gp_to_o18_dp, 
     >      d_rne22p_to_mg24_dp, d_rmg24gp_to_ne22_dp,
     >      d_rne20ap_to_ne22_dp, d_rne22p_to_ne20_dp,
     >      d_rne20ap_to_mg24_dp, d_rmg24gp_to_ne20_dp,

     >      d_rne20neut_to_ne22_dneut, d_rne22g_to_ne20_dneut,

     >      d_rca40neut_to_ca42_dneut, d_rca42g_to_ca40_dneut,
     >      d_rca42neut_to_ca44_dneut, d_rca44g_to_ca42_dneut,
     >      d_rca44neut_to_ca46_dneut, d_rca46g_to_ca44_dneut,
     >      d_rca46neut_to_ca48_dneut, d_rca48g_to_ca46_dneut,

     >      d_rti54prot_to_cr56_dprot, d_rcr56gprot_to_ti54_dprot,
     >      d_rti52aprot_to_ti54_dprot, d_rti54prot_to_ti52_dprot,
     >      d_rti52aprot_to_cr56_dprot, d_rcr56gprot_to_ti52_dprot,  

     >      d_rti52prot_to_cr54_dprot, d_rcr54gprot_to_ti52_dprot,
     >      d_rti50aprot_to_ti52_dprot, d_rti52prot_to_ti50_dprot,
     >      d_rti50aprot_to_cr54_dprot, d_rcr54gprot_to_ti50_dprot,  

     >      d_rti52aneut_to_cr54_dneut, d_rcr54neut_to_ti52_dneut,
     >      d_rti52aneut_to_cr56_dneut, d_rcr56gneut_to_ti52_dneut,  

     >      d_rti52pn_to_cr52_dprot, d_rti52pn_to_cr52_dneut,       
     >      d_rcr52np_to_ti52_dprot, d_rcr52np_to_ti52_dneut,       

     >      d_rti50prot_to_cr52_dprot, d_rcr52gprot_to_ti50_dprot,
     >      d_rti48aprot_to_ti50_dprot, d_rti50prot_to_ti48_dprot,
     >      d_rti48aprot_to_cr52_dprot, d_rcr52gprot_to_ti48_dprot,  

     >      d_rti50pn_to_cr50_dprot, d_rti50pn_to_cr50_dneut,       
     >      d_rcr50np_to_ti50_dprot, d_rcr50np_to_ti50_dneut,       

     >      d_rti48pn_to_cr48_dprot, d_rti48pn_to_cr48_dneut,       
     >      d_rcr48np_to_ti48_dprot, d_rcr48np_to_ti48_dneut,       

     >      d_rti50aneut_to_cr52_dneut, d_rcr52neut_to_ti50_dneut,
     >      d_rti50aneut_to_cr54_dneut, d_rcr54gneut_to_ti50_dneut,  

     >      d_rti48prot_to_cr50_dprot, d_rcr50gprot_to_ti48_dprot,
     >      d_rti46aprot_to_ti48_dprot, d_rti48prot_to_ti46_dprot,
     >      d_rti46aprot_to_cr50_dprot, d_rcr50gprot_to_ti46_dprot,  

     >      d_rti48aneut_to_cr50_dneut, d_rcr50neut_to_ti48_dneut,
     >      d_rti48aneut_to_cr52_dneut, d_rcr52gneut_to_ti48_dneut,  

     >      d_rti46aneut_to_cr48_dneut, d_rcr48neut_to_ti46_dneut,
     >      d_rti46aneut_to_cr50_dneut, d_rcr50gneut_to_ti46_dneut,  

     >      d_rcr46neut_to_cr48_dneut, d_rcr48g_to_cr46_dneut,
     >      d_rti44aneut_to_cr46_dneut, d_rcr46neut_to_ti44_dneut,
     >      d_rti44aneut_to_cr48_dneut, d_rcr48gneut_to_ti44_dneut,  

     >      d_rti44neut_to_ti46_dneut, d_rti46g_to_ti44_dneut,
     >      d_rti46neut_to_ti48_dneut, d_rti48g_to_ti46_dneut,
     >      d_rti48neut_to_ti50_dneut, d_rti50g_to_ti48_dneut,

     >      d_rcr54prot_to_fe56_dprot, d_rfe56gprot_to_cr54_dprot,
     >      d_rcr52aprot_to_cr54_dprot, d_rcr54prot_to_cr52_dprot,
     >      d_rcr52aprot_to_fe56_dprot, d_rfe56gprot_to_cr52_dprot,  

     >      d_rcr52aneut_to_fe54_dneut, d_rfe54neut_to_cr52_dneut,
     >      d_rcr52aneut_to_fe56_dneut, d_rfe56gneut_to_cr52_dneut,  

     >      d_rcr52pn_to_fe52_dprot, d_rcr52pn_to_fe52_dneut,       
     >      d_rfe52np_to_cr52_dprot, d_rfe52np_to_cr52_dneut,       

     >      d_rcr52prot_to_fe54_dprot, d_rfe54gprot_to_cr52_dprot,
     >      d_rcr50aprot_to_cr52_dprot, d_rcr52prot_to_cr50_dprot,
     >      d_rcr50aprot_to_fe54_dprot, d_rfe54gprot_to_cr50_dprot,  

     >      d_rcr50aneut_to_fe52_dneut, d_rfe52neut_to_cr50_dneut,
     >      d_rcr50aneut_to_fe54_dneut, d_rfe54gneut_to_cr50_dneut,  

     >      d_rcr48neut_to_cr50_dneut, d_rcr50g_to_cr48_dneut,
     >      d_rcr50neut_to_cr52_dneut, d_rcr52g_to_cr50_dneut,
     >      d_rcr52neut_to_cr54_dneut, d_rcr54g_to_cr52_dneut,

     >      d_rcr56prot_to_fe58_dprot, d_rfe58gprot_to_cr56_dprot,
     >      d_rcr54aprot_to_cr56_dprot, d_rcr56prot_to_cr54_dprot,
     >      d_rcr54aprot_to_fe58_dprot, d_rfe58gprot_to_cr54_dprot,  
       
     >      d_rti50neut_to_ti52_dneut, d_rti52g_to_ti50_dneut,
     >      d_rcr54neut_to_cr56_dneut, d_rcr56g_to_cr54_dneut,
     >      d_rcr58prot_to_fe60_dprot, d_rfe60gprot_to_cr58_dprot,
     >      d_rcr56aprot_to_cr58_dprot, d_rcr58prot_to_cr56_dprot,
     >      d_rcr56aprot_to_fe60_dprot, d_rfe60gprot_to_cr56_dprot,  

     >      d_rfe54prot_to_ni56_dprot, d_rni56gprot_to_fe54_dprot,
     >      d_rfe52aprot_to_fe54_dprot, d_rfe54prot_to_fe52_dprot,
     >      d_rfe52aprot_to_ni56_dprot, d_rni56gprot_to_fe52_dprot,
     >      d_rfe56prot_to_ni58_dprot, d_rni58gprot_to_fe56_dprot,
     >      d_rfe54aprot_to_fe56_dprot, d_rfe56prot_to_fe54_dprot,
     >      d_rfe54aprot_to_ni58_dprot, d_rni58gprot_to_fe54_dprot,
     >      d_rfe58prot_to_ni60_dprot, d_rni60gprot_to_fe58_dprot,
     >      d_rfe56aprot_to_fe58_dprot, d_rfe58prot_to_fe56_dprot,
     >      d_rfe56aprot_to_ni60_dprot, d_rni60gprot_to_fe56_dprot,
     >      d_rfe60prot_to_ni62_dprot, d_rni62gprot_to_fe60_dprot,
     >      d_rfe58aprot_to_fe60_dprot, d_rfe60prot_to_fe58_dprot,
     >      d_rfe58aprot_to_ni62_dprot, d_rni62gprot_to_fe58_dprot,
     >      d_rfe62prot_to_ni64_dprot, d_rni64gprot_to_fe62_dprot,
     >      d_rfe60aprot_to_fe62_dprot, d_rfe62prot_to_fe60_dprot,
     >      d_rfe60aprot_to_ni64_dprot, d_rni64gprot_to_fe60_dprot,
     >      d_rni66prot_to_zn68_dprot, d_rzn68gprot_to_ni66_dprot,
     >      d_rni64aprot_to_ni66_dprot, d_rni66prot_to_ni64_dprot,
     >      d_rni64aprot_to_zn68_dprot, d_rzn68gprot_to_ni64_dprot,
     >      d_rzn70prot_to_ge72_dprot, d_rge72gprot_to_zn70_dprot,
     >      d_rzn68aprot_to_zn70_dprot, d_rzn70prot_to_zn68_dprot,
     >      d_rzn68aprot_to_ge72_dprot, d_rge72gprot_to_zn68_dprot,
     >      d_rni58prot_to_zn60_dprot, d_rzn60gprot_to_ni58_dprot,
     >      d_rni56aprot_to_ni58_dprot, d_rni58prot_to_ni56_dprot,
     >      d_rni56aprot_to_zn60_dprot, d_rzn60gprot_to_ni56_dprot,
     >      d_rni60prot_to_zn62_dprot, d_rzn62gprot_to_ni60_dprot,
     >      d_rni58aprot_to_ni60_dprot, d_rni60prot_to_ni58_dprot,
     >      d_rni58aprot_to_zn62_dprot, d_rzn62gprot_to_ni58_dprot,
     >      d_rni62prot_to_zn64_dprot, d_rzn64gprot_to_ni62_dprot,
     >      d_rni60aprot_to_ni62_dprot, d_rni62prot_to_ni60_dprot,
     >      d_rni60aprot_to_zn64_dprot, d_rzn64gprot_to_ni60_dprot,
     >      d_rni64prot_to_zn66_dprot, d_rzn66gprot_to_ni64_dprot,
     >      d_rni62aprot_to_ni64_dprot, d_rni64prot_to_ni62_dprot,
     >      d_rni62aprot_to_zn66_dprot, d_rzn66gprot_to_ni62_dprot,
     
     >      d_rfe52neut_to_fe54_dneut, d_rfe54g_to_fe52_dneut,
     
     >      d_rfe54neut_to_fe56_dneut, d_rfe56g_to_fe54_dneut,
     
     >      d_rfe56neut_to_fe58_dneut, d_rfe58g_to_fe56_dneut,
     >      d_rcr54aneut_to_fe56_dneut, d_rfe56neut_to_cr54_dneut,
     >      d_rcr54aneut_to_fe58_dneut, d_rfe58gneut_to_cr54_dneut,  

     >      d_rcr54pn_to_fe54_dprot, d_rcr54pn_to_fe54_dneut,       
     >      d_rfe54np_to_cr54_dprot, d_rfe54np_to_cr54_dneut,       

     >      d_rcr56pn_to_fe56_dprot, d_rcr56pn_to_fe56_dneut,       
     >      d_rfe56np_to_cr56_dprot, d_rfe56np_to_cr56_dneut,       

     >      d_rfe56pn_to_ni56_dprot, d_rfe56pn_to_ni56_dneut,       
     >      d_rni56np_to_fe56_dprot, d_rni56np_to_fe56_dneut,       

     >      d_rfe58pn_to_ni58_dprot, d_rfe58pn_to_ni58_dneut,       
     >      d_rni58np_to_fe58_dprot, d_rni58np_to_fe58_dneut,       
     
     >      d_rfe58neut_to_fe60_dneut, d_rfe60g_to_fe58_dneut,
     >      d_rcr56aneut_to_fe58_dneut, d_rfe58neut_to_cr56_dneut,
     >      d_rcr56aneut_to_fe60_dneut, d_rfe60gneut_to_cr56_dneut,  
     
     >      d_rfe60neut_to_fe62_dneut, d_rfe62g_to_fe60_dneut,
     
     >      d_rni56neut_to_ni58_dneut, d_rni58g_to_ni56_dneut,
     >      d_rfe54aneut_to_ni56_dneut, d_rni56neut_to_fe54_dneut,
     >      d_rfe54aneut_to_ni58_dneut, d_rni58gneut_to_fe54_dneut,  
     
     >      d_rni58neut_to_ni60_dneut, d_rni60g_to_ni58_dneut,
     >      d_rfe56aneut_to_ni58_dneut, d_rni58neut_to_fe56_dneut,
     >      d_rfe56aneut_to_ni60_dneut, d_rni60gneut_to_fe56_dneut,  
     
     >      d_rni60neut_to_ni62_dneut, d_rni62g_to_ni60_dneut,
     >      d_rfe58aneut_to_ni60_dneut, d_rni60neut_to_fe58_dneut,
     >      d_rfe58aneut_to_ni62_dneut, d_rni62gneut_to_fe58_dneut,  
     
     >      d_rni62neut_to_ni64_dneut, d_rni64g_to_ni62_dneut,
     
     >      d_rni64neut_to_ni66_dneut, d_rni66g_to_ni64_dneut,

     >      d_rzn60neut_to_zn62_dneut, d_rzn62g_to_zn60_dneut,
     >      d_rzn62neut_to_zn64_dneut, d_rzn64g_to_zn62_dneut,
     >      d_rzn64neut_to_zn66_dneut, d_rzn66g_to_zn64_dneut,
     >      d_rzn66neut_to_zn68_dneut, d_rzn68g_to_zn66_dneut,
     >      d_rzn68neut_to_zn70_dneut, d_rzn70g_to_zn68_dneut,

     >      d_rge70neut_to_ge72_dneut, d_rge72g_to_ge70_dneut,
     >      d_rge72neut_to_ge74_dneut, d_rge74g_to_ge72_dneut,
     >      d_rge74neut_to_ge76_dneut, d_rge76g_to_ge74_dneut
     
      end type Net_Info
      
   ! private to the implementation
      integer, parameter :: max_net_handles = 10
      type (Net_General_Info), target :: net_handles(max_net_handles)
      
      character (len=256) :: net_dir

      integer :: weak_rate_id_for_sc43_ec
      integer :: weak_rate_id_for_sc45_ec
      integer :: weak_rate_id_for_sc47_ec
      integer :: weak_rate_id_for_sc49_ec
      integer :: weak_rate_id_for_sc50_ec
      integer :: weak_rate_id_for_sc51_ec
      
      integer :: weak_rate_id_for_ti43_ec
      integer :: weak_rate_id_for_ti44_ec
      integer :: weak_rate_id_for_ti45_ec
      integer :: weak_rate_id_for_ti46_ec
      integer :: weak_rate_id_for_ti47_ec
      integer :: weak_rate_id_for_ti48_ec
      integer :: weak_rate_id_for_ti49_ec
      integer :: weak_rate_id_for_ti50_ec
      integer :: weak_rate_id_for_ti51_ec
      integer :: weak_rate_id_for_ti52_ec

      integer :: weak_rate_id_for_v47_ec
      integer :: weak_rate_id_for_v49_ec
      integer :: weak_rate_id_for_v51_ec
      integer :: weak_rate_id_for_v53_ec
      integer :: weak_rate_id_for_v54_ec
      integer :: weak_rate_id_for_v55_ec
      
      integer :: weak_rate_id_for_cr47_ec
      integer :: weak_rate_id_for_cr48_ec
      integer :: weak_rate_id_for_cr49_ec
      integer :: weak_rate_id_for_cr50_ec
      integer :: weak_rate_id_for_cr51_ec
      integer :: weak_rate_id_for_cr52_ec
      integer :: weak_rate_id_for_cr53_ec
      integer :: weak_rate_id_for_cr54_ec
      integer :: weak_rate_id_for_cr55_ec
      integer :: weak_rate_id_for_cr56_ec

      integer :: weak_rate_id_for_mn51_ec
      integer :: weak_rate_id_for_mn53_ec
      integer :: weak_rate_id_for_mn54_ec
      integer :: weak_rate_id_for_mn55_ec
      integer :: weak_rate_id_for_mn56_ec
      integer :: weak_rate_id_for_mn57_ec
      integer :: weak_rate_id_for_mn58_ec
      integer :: weak_rate_id_for_mn59_ec

!      integer :: weak_rate_id_for_fe51_ec      << not in weaklib
      integer :: weak_rate_id_for_fe52_ec
      integer :: weak_rate_id_for_fe53_ec
      integer :: weak_rate_id_for_fe54_ec
      integer :: weak_rate_id_for_fe55_ec
      integer :: weak_rate_id_for_fe56_ec
      integer :: weak_rate_id_for_fe57_ec
      integer :: weak_rate_id_for_fe58_ec
      integer :: weak_rate_id_for_fe59_ec
      integer :: weak_rate_id_for_fe60_ec
      
      integer :: weak_rate_id_for_co55_ec
      integer :: weak_rate_id_for_co56_ec
      integer :: weak_rate_id_for_co57_ec
      integer :: weak_rate_id_for_co59_ec
      integer :: weak_rate_id_for_co61_ec
      integer :: weak_rate_id_for_co62_ec
      integer :: weak_rate_id_for_co63_ec
      integer :: weak_rate_id_for_co64_ec
!      integer :: weak_rate_id_for_co65_ec      << not in weaklib
      
      integer :: weak_rate_id_for_ni55_ec
      integer :: weak_rate_id_for_ni56_ec
      integer :: weak_rate_id_for_ni57_ec
      integer :: weak_rate_id_for_ni58_ec
      integer :: weak_rate_id_for_ni59_ec
      integer :: weak_rate_id_for_ni60_ec
      integer :: weak_rate_id_for_ni61_ec
      integer :: weak_rate_id_for_ni62_ec
      integer :: weak_rate_id_for_ni63_ec
      integer :: weak_rate_id_for_ni64_ec
      integer :: weak_rate_id_for_ni65_ec
!      integer :: weak_rate_id_for_ni66_ec      << not in weaklib
      
!      integer :: weak_rate_id_for_zn68_ec      << not in weaklib
!      integer :: weak_rate_id_for_ge72_ec      << not in weaklib

      integer :: weak_rate_id_for_cr54_ee
      integer :: weak_rate_id_for_mn54_ee
      
      integer :: weak_rate_id_for_cr56_ee
      integer :: weak_rate_id_for_mn56_ee

      integer :: weak_rate_id_for_fe56_ee
      integer :: weak_rate_id_for_ti50_ee
      integer :: weak_rate_id_for_ti52_ee
      integer :: weak_rate_id_for_cr52_ee

      
      contains


      subroutine net_def_init
         use const_def, only: mesa_data_dir
         use weak_lib, only: get_weak_rate_id
         integer :: i
         net_dir = trim(mesa_data_dir) // '/net_data'
         do i=1, max_net_handles
            net_handles(i)% handle = i
            net_handles(i)% in_use = .false.
            net_handles(i)% net_has_been_defined = .false.
            net_handles(i)% num_isos = 0
            net_handles(i)% num_reactions = 0
         end do
              
         weak_rate_id_for_sc43_ec = get_id('sc43','ca43')         
         weak_rate_id_for_sc45_ec = get_id('sc45','ca45')         
         weak_rate_id_for_sc47_ec = get_id('sc47','ca47')         
         weak_rate_id_for_sc49_ec = get_id('sc49','ca49')         
         weak_rate_id_for_sc50_ec = get_id('sc50','ca50')         
         weak_rate_id_for_sc51_ec = get_id('sc51','ca51')         
              
         weak_rate_id_for_ti43_ec = get_id('ti43','sc43')         
         weak_rate_id_for_ti44_ec = get_id('ti44','sc44')         
         weak_rate_id_for_ti45_ec = get_id('ti45','sc45')         
         weak_rate_id_for_ti46_ec = get_id('ti46','sc46')         
         weak_rate_id_for_ti47_ec = get_id('ti47','sc47')         
         weak_rate_id_for_ti48_ec = get_id('ti48','sc48')         
         weak_rate_id_for_ti49_ec = get_id('ti49','sc49')         
         weak_rate_id_for_ti50_ec = get_id('ti50','sc50')         
         weak_rate_id_for_ti51_ec = get_id('ti51','sc51')         
         weak_rate_id_for_ti52_ec = get_id('ti52','sc52')  
           
         weak_rate_id_for_v47_ec = get_id('v47','ti47')
         weak_rate_id_for_v49_ec = get_id('v49','ti49')
         weak_rate_id_for_v51_ec = get_id('v51','ti51')
         weak_rate_id_for_v53_ec = get_id('v53','ti53')
         weak_rate_id_for_v54_ec = get_id('v54','ti54')
         weak_rate_id_for_v55_ec = get_id('v55','ti55')
         
         weak_rate_id_for_cr47_ec = get_id('cr47','v47')         
         weak_rate_id_for_cr48_ec = get_id('cr48','v48')         
         weak_rate_id_for_cr49_ec = get_id('cr49','v49')         
         weak_rate_id_for_cr50_ec = get_id('cr50','v50')         
         weak_rate_id_for_cr51_ec = get_id('cr51','v51')         
         weak_rate_id_for_cr52_ec = get_id('cr52','v52')         
         weak_rate_id_for_cr53_ec = get_id('cr53','v53')         
         weak_rate_id_for_cr54_ec = get_id('cr54','v54')         
         weak_rate_id_for_cr55_ec = get_id('cr55','v55')         
         weak_rate_id_for_cr56_ec = get_id('cr56','v56')    
         
         weak_rate_id_for_mn51_ec = get_id('mn51','cr51')
         weak_rate_id_for_mn53_ec = get_id('mn53','cr53')
         weak_rate_id_for_mn54_ec = get_id('mn54','cr54')
         weak_rate_id_for_mn55_ec = get_id('mn55','cr55')
         weak_rate_id_for_mn56_ec = get_id('mn56','cr56')
         weak_rate_id_for_mn57_ec = get_id('mn57','cr57')
         weak_rate_id_for_mn58_ec = get_id('mn58','cr58')
         weak_rate_id_for_mn59_ec = get_id('mn59','cr59')
              
!         weak_rate_id_for_fe51_ec = get_id('fe51','mn51')         
         weak_rate_id_for_fe52_ec = get_id('fe52','mn52')         
         weak_rate_id_for_fe53_ec = get_id('fe53','mn53')         
         weak_rate_id_for_fe54_ec = get_id('fe54','mn54')         
         weak_rate_id_for_fe55_ec = get_id('fe55','mn55')         
         weak_rate_id_for_fe56_ec = get_id('fe56','mn56')         
         weak_rate_id_for_fe57_ec = get_id('fe57','mn57')
         weak_rate_id_for_fe58_ec = get_id('fe58','mn58')         
         weak_rate_id_for_fe59_ec = get_id('fe59','mn59')
         weak_rate_id_for_fe60_ec = get_id('fe60','mn60') 

         weak_rate_id_for_co55_ec = get_id('co55','fe55')
         weak_rate_id_for_co56_ec = get_id('co56','fe56')
         weak_rate_id_for_co57_ec = get_id('co57','fe57')
         weak_rate_id_for_co59_ec = get_id('co59','fe59')
         weak_rate_id_for_co61_ec = get_id('co61','fe61')
         weak_rate_id_for_co62_ec = get_id('co62','fe62')
         weak_rate_id_for_co63_ec = get_id('co63','fe63')
         weak_rate_id_for_co64_ec = get_id('co64','fe64')
!         weak_rate_id_for_co65_ec = get_id('co65','fe65')
                 
         weak_rate_id_for_ni55_ec = get_id('ni55','co55')
         weak_rate_id_for_ni56_ec = get_id('ni56','co56')
         weak_rate_id_for_ni57_ec = get_id('ni57','co57')
         weak_rate_id_for_ni58_ec = get_id('ni58','co58')
         weak_rate_id_for_ni59_ec = get_id('ni59','co59')
         weak_rate_id_for_ni60_ec = get_id('ni60','co60')
         weak_rate_id_for_ni61_ec = get_id('ni61','co61')
         weak_rate_id_for_ni62_ec = get_id('ni62','co62')
         weak_rate_id_for_ni63_ec = get_id('ni63','co63')
         weak_rate_id_for_ni64_ec = get_id('ni64','co64')
         weak_rate_id_for_ni65_ec = get_id('ni65','co65')
!         weak_rate_id_for_ni66_ec = get_id('ni66','co66')   
               
!         weak_rate_id_for_zn68_ec = get_id('zn68','cu68')
!         weak_rate_id_for_ge72_ec = get_id('ge72','ga72')

         weak_rate_id_for_cr54_ee = get_id('cr54','mn54')
         weak_rate_id_for_mn54_ee = get_id('mn54','fe54')
         weak_rate_id_for_cr56_ee = get_id('cr56','mn56')
         weak_rate_id_for_mn56_ee = get_id('mn56','fe56')

         weak_rate_id_for_fe56_ee = get_id('fe56','co56')
         weak_rate_id_for_ti50_ee = get_id('ti50','v50')
         weak_rate_id_for_ti52_ee = get_id('ti52','v52')
         weak_rate_id_for_cr52_ee = get_id('cr52','mn52')
         
         contains
         
         integer function get_id(iso1, iso2)
            character(len=*), intent(in) :: iso1, iso2
            include 'formats'
            get_id = get_weak_rate_id(iso1, iso2)
            if (get_id == 0) then
               write(*,2) 'failed to find weak reaction for ' // trim(iso1)
     >            // ' to ' // trim(iso2) 
            end if
         end function get_id
         
      end subroutine net_def_init


      integer function do_alloc_net(ierr)
         use rates_def
         integer, intent(out) :: ierr
         integer :: i
         type (Net_General_Info), pointer :: g
         ierr = 0
         do_alloc_net = -1
!$omp critical (net_handle)
         do i = 1, max_net_handles
            if (.not. net_handles(i)% in_use) then
               net_handles(i)% in_use = .true.
               do_alloc_net = i
               exit
            end if
         end do
!$omp end critical (net_handle)
         if (do_alloc_net == -1) then
            ierr = -1
            return
         end if
         if (net_handles(do_alloc_net)% handle /= do_alloc_net) then
            ierr = -1
            return
         end if
         g => net_handles(do_alloc_net)
         nullify(g% which_rates)
         nullify(g% net_iso)
         nullify(g% chem_id)
         nullify(g% net_reaction)
         nullify(g% reaction_id)
         nullify(g% reaction_kind)
         nullify(g% reverse_id_for_kind_ne_other)
         nullify(g% reaction_max_Z)
         nullify(g% reaction_max_Z_plus_N_for_max_Z)
         g% net_has_been_defined = .false.
         g% doing_timing = .false.
         g% doing_derivs_timing = .false.
         g% num_isos = 0
         g% num_reactions = 0
         g% logTcut_lo = rattab_tlo
         g% logTcut_lim = rattab_tlo + 0.1d0
         g% cache_suffix = '0'
         
         call do_set_default_T_limits(g)
         
      end function do_alloc_net
      
      
      subroutine do_set_default_T_limits(g)
         type (Net_General_Info), pointer :: g
         call do_turn_off_T_limits(g) ! for now, the default is off
      end subroutine do_set_default_T_limits
      
      
      subroutine do_turn_off_T_limits(g)
         use const_def, only: ln10
         type (Net_General_Info), pointer :: g

         g% T_lo_prot = 1d99
         g% T_hi_prot = 1d99
         g% lnT_lo_prot = 99*ln10
         g% lnT_hi_prot = 99*ln10
         g% min_ln_factor_prot = 0
         g% min_factor_prot = 1
     
         g% T_lo_combo_a_capture = 1d99
         g% T_hi_combo_a_capture = 1d99
         g% lnT_lo_combo_a_capture = 99*ln10
         g% lnT_hi_combo_a_capture = 99*ln10
         g% min_ln_factor_combo_a_capture = 0
         g% min_factor_combo_a_capture = 1
     
         g% T_lo_a_cap_high_mass = 1d99
         g% T_hi_a_cap_high_mass = 1d99
         g% lnT_lo_a_cap_high_mass = 99*ln10
         g% lnT_hi_a_cap_high_mass = 99*ln10
         g% min_ln_factor_a_cap_high_mass = 0
         g% min_factor_a_cap_high_mass = 1
     
         g% T_lo_a_cap_intermediate = 1d99
         g% T_hi_a_cap_intermediate = 1d99
         g% lnT_lo_a_cap_intermediate = 99*ln10
         g% lnT_hi_a_cap_intermediate = 99*ln10
         g% min_ln_factor_a_cap_intermediate = 0
         g% min_factor_a_cap_intermediate = 1

         g% T_lo_neut = 1d99
         g% T_hi_neut = 1d99
         g% lnT_lo_neut = 99*ln10
         g% lnT_hi_neut = 99*ln10
         g% min_ln_factor_neut = 0      
         g% min_factor_neut = 1
      
      end subroutine do_turn_off_T_limits
      
      
      subroutine do_free_net(handle)
         use rates_def
         integer, intent(in) :: handle
         type (Net_General_Info), pointer :: g
         if (handle >= 1 .and. handle <= max_net_handles) then
            g => net_handles(handle)
            deallocate(g% which_rates)
            deallocate(g% net_iso)
            deallocate(g% chem_id)
            deallocate(g% net_reaction)
            deallocate(g% reaction_id)
            deallocate(g% reaction_kind)
            deallocate(g% reverse_id_for_kind_ne_other)
            deallocate(g% reaction_max_Z)
            deallocate(g% reaction_max_Z_plus_N_for_max_Z)
            deallocate(g% zg1)
            deallocate(g% zg2)
            deallocate(g% zg3)
            deallocate(g% zg4)
            deallocate(g% zs13)
            deallocate(g% zhat)
            deallocate(g% zhat2)
            deallocate(g% lzav)
            deallocate(g% aznut)
            deallocate(g% zs13inv)
            deallocate(g% rattab)
            deallocate(g% ttab)
            deallocate(g% logttab)
            deallocate(g% rattab_f1)
            if (associated(g% weaklib_ids)) deallocate(g% weaklib_ids)
            if (associated(g% weak_reaction_num)) deallocate(g% weak_reaction_num)
            if (associated(g% weak_reaction_index)) deallocate(g% weak_reaction_index)
            g% in_use = .false.
            g% net_has_been_defined = .false.
            g% num_isos = 0
            g% num_reactions = 0
            g% num_weaklib_rates = 0
         end if
      end subroutine do_free_net
      

      subroutine get_net_ptr(handle, g, ierr)
         integer, intent(in) :: handle
         type (Net_General_Info), pointer :: g
         integer, intent(out):: ierr         
         if (handle < 1 .or. handle > max_net_handles) then
            ierr = -1
            return
         end if
         g => net_handles(handle)
         ierr = 0
      end subroutine get_net_ptr


      integer function get_net_timing_total(g)
         type (Net_General_Info), pointer :: g
         get_net_timing_total = 0
         if (.not. g% doing_timing) return
         get_net_timing_total = 
     >      g% clock_net_eval +
     >      g% clock_net_weak_rates +
     >      g% clock_net_rate_tables + 
     >      g% clock_net_screen + 
     >      g% clock_net_derivs
      end function get_net_timing_total


      subroutine zero_net_timing(g)
         type (Net_General_Info), pointer :: g
         g% clock_net_eval = 0
         g% clock_net_weak_rates = 0
         g% clock_net_rate_tables = 0 
         g% clock_net_screen = 0 
         g% clock_net_derivs = 0
         
         g% clock_derivs_setup = 0
         g% clock_derivs_select = 0
         g% clock_derivs_general = 0
         g% clock_net_get = 0
      end subroutine zero_net_timing


      end module net_def

