! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module net_eval
      use const_def
      use chem_def
      use net_def
      use ecapture_def, only : do_ecapture
      implicit none
      
      
      contains
      

      subroutine eval_net(
     >         g, num_isos, num_reactions, num_weaklib_rates,
     >         x, atemp, alogtemp, arho, alogrho, 
     >         abar, zbar, z2bar, ye, eta, d_eta_dlnT, d_eta_dlnRho,
     >         rate_factors, category_factors,
     >         reaction_Qs, reaction_neuQs,
     >         eps_nuc, d_eps_nuc_dRho, d_eps_nuc_dT, d_eps_nuc_dx, 
     >         dxdt, d_dxdt_dRho, d_dxdt_dT, d_dxdt_dx, 
     >         screening_mode, theta_e_for_graboske_et_al, 
     >         rate_screened, rate_raw,
     >         reaction_eps_nuc, eps_nuc_categories, eps_neu_total,
     >         lwork, work, actual_Qs, actual_neuQs, from_weaklib, symbolic, ierr)
         use net_initialize
         use net_screen
         use net_derivs

         type (Net_General_Info), pointer :: g
         integer, intent(in) :: num_isos
         integer, intent(in) :: num_reactions, num_weaklib_rates
         real(dp), intent(in)  :: x(:)
         real(dp), intent(in)  :: atemp, alogtemp
         real(dp), intent(in)  :: arho, alogrho
         real(dp), intent(in)  :: abar  ! mean number of nucleons per nucleus
         real(dp), intent(in)  :: zbar  ! mean charge per nucleus
         real(dp), intent(in)  :: z2bar ! mean charge squared per nucleus
         real(dp), intent(in)  :: ye    
         real(dp), intent(in)  :: eta, d_eta_dlnT, d_eta_dlnRho ! eta and derivatives
         real(dp), intent(in) :: rate_factors(:)
         real(dp), intent(in) :: category_factors(:)
         real(dp), pointer, intent(in) :: reaction_Qs(:) ! (rates_reaction_id_max)
         real(dp), pointer, intent(in) :: reaction_neuQs(:) ! (rates_reaction_id_max)
         real(dp), intent(out) :: eps_nuc ! ergs/gram/second from burning 
         real(dp), intent(out) :: d_eps_nuc_dT
         real(dp), intent(out) :: d_eps_nuc_dRho
         real(dp), intent(out) :: d_eps_nuc_dx(:) 
         real(dp), intent(out) :: dxdt(num_isos)
         real(dp), intent(out) :: d_dxdt_dRho(:)
         real(dp), intent(out) :: d_dxdt_dT(:)
         real(dp), intent(out) :: d_dxdt_dx(:, :)
         real(dp), intent(out) :: reaction_eps_nuc(:, :)
         real(dp), intent(out) :: eps_nuc_categories(:, :)
         real(dp), intent(inout), dimension(:, :) :: 
     >         rate_screened, rate_raw
         real(dp), intent(out) :: eps_neu_total
         integer, intent(in) :: screening_mode
         real(dp), intent(in)  :: theta_e_for_graboske_et_al
         integer, intent(in) :: lwork
         real(dp), pointer :: work(:) ! (lwork)
         real(dp), pointer, dimension(:) :: actual_Qs, actual_neuQs ! ignore if null
         logical, pointer :: from_weaklib(:) ! ignore if null
         logical, intent(in) :: symbolic
         integer, intent(out) :: ierr

         type (Net_Info), target :: netinf
         type (Net_Info), pointer:: n
         integer, parameter :: max_z_for_cache = 14
         real(dp), target :: 
     >         graboske_cache(3, max_z_for_cache, max_z_for_cache)
         real(dp) :: enuc, temp, logtemp, rho, logrho, total, prev, curr, prev_T
         real(dp) :: btemp, bden, eps_total
         integer :: ci, i, j, ir, weak_id, h1
         integer, pointer :: chem_id(:)
         integer :: time0, time1
         logical :: doing_timing, reuse_given_rates
         real(dp), dimension(num_weaklib_rates), target :: 
     >         ldecay, d_ldecay_dT9, d_ldecay_dlYeRho, 
     >         lcapture, d_lcapture_dT9, d_lcapture_dlYeRho, 
     >         lneutrino, d_lneutrino_dT9, d_lneutrino_dlYeRho, 
     >         lambda, dlambda_dlnT, dlambda_dlnRho, 
     >         Q, dQ_dlnT, dQ_dlnRho, 
     >         Qneu, dQneu_dlnT, dQneu_dlnRho
         real(dp), parameter :: ! new names for values defined in chem_def
     >         deltap     = del_Mp, 
     >         deltan     = del_Mn
         
         logical, parameter :: dbg = .false.
         
         include 'formats.dek'
         
         reuse_given_rates = .false.

         doing_timing = g% doing_timing
         if (doing_timing) then
            call system_clock(time0)
            g% doing_timing = .false.
         end if

         ierr = 0

         eps_nuc = 0
         d_eps_nuc_dT = 0
         d_eps_nuc_dRho = 0
         d_eps_nuc_dx(:) = 0
                  
         dxdt(:) = 0
         d_dxdt_dRho(:) = 0
         d_dxdt_dT(:) = 0
         d_dxdt_dx(: ,:) = 0
         reaction_eps_nuc = 0
         eps_nuc_categories = 0
         eps_neu_total = 0

         temp = atemp; logtemp = alogtemp; rho = arho; logrho = alogrho
         call get_T_rho_args(temp, logtemp, rho, logrho, ierr)
         if (ierr /= 0) return
         
         if (logtemp < rattab_tlo) then ! clip to table so can eval beta decays
            logtemp = rattab_tlo
            temp = 10**logtemp
            !write(*,1) 'logtemp', logtemp
            !write(*,1) 'rattab_tlo', rattab_tlo
            !return 
         end if
         
         chem_id => g% chem_id
         n => netinf
         n% graboske_cache => graboske_cache
         n% reaction_Qs => reaction_Qs
         n% reaction_neuQs => reaction_neuQs
         n% eps_neu_total = 0

         if (dbg) write(*,*) 'call setup_net_info'
         call setup_net_info(
     >         g, n, num_reactions, reaction_eps_nuc, eps_nuc_categories, 
     >         screening_mode, theta_e_for_graboske_et_al,
     >         rate_screened, rate_raw, lwork, work, reuse_given_rates, ierr)
         if (ierr /= 0) return
         
         if (.not. g% net_has_been_defined) then
            ierr = -1
            return
         end if
         
         if (doing_timing) then
            call system_clock(time1)
            g% clock_net_eval = g% clock_net_eval + (time1 - time0)
            time0 = time1
         end if
         
         if (num_weaklib_rates > 0) then
            if (dbg) write(*,*) 'call get_weaklib_rates'
            call get_weaklib_rates(ierr)
            if (ierr /= 0) return
            if (do_ecapture) then 
               if (dbg) write (*,*) "call get_ecapture_rates"
               call get_ecapture_rates(ierr)
               if (ierr /= 0) return
            end if
         end if
         
         if (associated(actual_Qs) .and. associated(actual_neuQs)) then
            do i = 1, g% num_reactions
               ir = g% reaction_id(i)
               from_weaklib(i) = .false.
               actual_Qs(i) = n% reaction_Qs(ir)
               actual_neuQs(i) = n% reaction_neuQs(ir)
               weak_id = g% weak_reaction_index(i)
               if (weak_id > 0) then
                  if (g% weaklib_ids(weak_id) > 0) then
                     from_weaklib(i) = .true.
                     actual_Qs(i) = n% Q(weak_id)
                     actual_neuQs(i) = n% Qneu(weak_id)
                  end if
               end if
            end do
         end if

         n% d_eps_nuc_dy(:) = 0

         if (dbg) write(*,*) 'call set_molar_abundances'
         call set_molar_abundances(g, num_isos, x, n% y, ierr)
         if (ierr /= 0) return

         !..limit range of temperatures and densities
         btemp = min(10**rattab_thi, max(temp, 10**rattab_tlo))
         bden = min(1.0d11, max(rho, 1.0d-10))
         
         if (doing_timing) then
            call system_clock(time1)
            g% clock_net_eval = g% clock_net_eval + (time1 - time0)
            time0 = time1
         end if

         if (.not. reuse_given_rates) then
            call get_rates_with_screening(ierr)
            if (ierr /= 0) return
         end if
         
         call get_T_limit_factor(
     >      temp, logtemp*ln10, 
     >      g% T_lo_prot, g% T_hi_prot, 
     >      g% lnT_lo_prot, g% lnT_hi_prot,
     >      g% min_ln_factor_prot, g% min_factor_prot,
     >      n% factor_prot, n% d_factor_prot_dT)
         
         call get_T_limit_factor(
     >      temp, logtemp*ln10, 
     >      g% T_lo_combo_a_capture, g% T_hi_combo_a_capture, 
     >      g% lnT_lo_combo_a_capture, g% lnT_hi_combo_a_capture,
     >      g% min_ln_factor_combo_a_capture, g% min_factor_combo_a_capture,
     >      n% factor_combo_a_capture, n% d_factor_combo_a_capture_dT)
         
         call get_T_limit_factor(
     >      temp, logtemp*ln10, 
     >      g% T_lo_a_cap_high_mass, g% T_hi_a_cap_high_mass, 
     >      g% lnT_lo_a_cap_high_mass, g% lnT_hi_a_cap_high_mass,
     >      g% min_ln_factor_a_cap_high_mass, g% min_factor_a_cap_high_mass,
     >      n% factor_a_cap_high_mass, n% d_factor_a_cap_high_mass_dT)
         
         call get_T_limit_factor(
     >      temp, logtemp*ln10, 
     >      g% T_lo_a_cap_intermediate, g% T_hi_a_cap_intermediate, 
     >      g% lnT_lo_a_cap_intermediate, g% lnT_hi_a_cap_intermediate,
     >      g% min_ln_factor_a_cap_intermediate, g% min_factor_a_cap_intermediate,
     >      n% factor_a_cap_intermediate, n% d_factor_a_cap_intermediate_dT)
         
         call get_T_limit_factor(
     >      temp, logtemp*ln10, 
     >      g% T_lo_neut, g% T_hi_neut, 
     >      g% lnT_lo_neut, g% lnT_hi_neut,
     >      g% min_ln_factor_neut, g% min_factor_neut,
     >      n% factor_neut, n% d_factor_neut_dT)
     
         if (dbg) write(*,*) 'call get_derivs'
         call get_derivs( 
     >       n, category_factors, eta, ye, btemp, bden, abar, zbar, 
     >       num_reactions, rate_factors, symbolic, ierr)
         if (ierr /= 0) return
         if (symbolic) then
            do j=1, num_isos
               do i=1, num_isos
                  d_dxdt_dx(i,j) = n% d_dydt_dy(i,j)
               end do
            end do
            return
         end if
         
         if (doing_timing) then
            call system_clock(time1)
            g% clock_net_derivs = g% clock_net_derivs + (time1 - time0)
            time0 = time1
         end if

         !..convert the reaction_eps_nuc Qs to ergs/g/sec
         n% reaction_eps_nuc(: ,:) = Qconv*n% reaction_eps_nuc(: ,:)

         !..convert the eps_nuc_categories
         n% eps_nuc_categories(: ,:) = Qconv*n% eps_nuc_categories(: ,:)

         !..store the results
         do i = 1, num_isos
            ci = chem_id(i)
            dxdt(i) = chem_isos% Z_plus_N(ci)*n% dydt(i_rate, i)
            d_dxdt_dRho(i) = chem_isos% Z_plus_N(ci)*n% dydt(i_rate_dRho, i)
            d_dxdt_dT(i) = chem_isos% Z_plus_N(ci)*n% dydt(i_rate_dT, i)
            do j=1, num_isos
               d_dxdt_dx(i, j) =
     >            chem_isos% Z_plus_N(ci)*n% d_dydt_dy(i, j) / chem_isos% Z_plus_N(chem_id(j))
            end do
         enddo
   
         eps_nuc = sum(n% reaction_eps_nuc(i_rate,:))
         d_eps_nuc_dT = sum(n% reaction_eps_nuc(i_rate_dT,:))
         d_eps_nuc_dRho = sum(n% reaction_eps_nuc(i_rate_dRho,:))         
         d_eps_nuc_dx(:) = Qconv*n% d_eps_nuc_dy(:) / chem_isos% Z_plus_N(chem_id(:))
         
         eps_neu_total = n% eps_neu_total*Qconv

         if (doing_timing) then
            call system_clock(time1)
            g% clock_net_eval = g% clock_net_eval + (time1 - time0)
            g% doing_timing = .true.
         end if
         
         
         contains

         
         subroutine get_rates_with_screening(ierr)
            use rates_lib, only: eval_using_rate_tables
            integer, intent(out) :: ierr
            integer :: ir, i
            include 'formats.dek'
            !..get the raw reaction rates
            if (dbg) write(*,*) 'call eval_using_rate_tables'
            call eval_using_rate_tables(
     >         num_reactions, g% reaction_id, g% rattab, g% rattab_f1, nrattab, 
     >         ye, logtemp, btemp, bden, rate_factors, g% logttab, n% rate_raw) 
            if (doing_timing) then
               call system_clock(time1)
               g% clock_net_rate_tables = g% clock_net_rate_tables + (time1 - time0)
               time0 = time1
            end if
            !..get the reaction rates including screening factors
            if (dbg) write(*,*) 'call screen_net'
            call screen_net(
     >            n, num_isos, n% y, btemp, bden, logtemp, logrho, .false., 
     >            zbar, abar, z2bar, ye, ierr)
            if (ierr /= 0) return
            if (doing_timing) then
               call system_clock(time1)
               g% clock_net_screen = g% clock_net_screen + (time1 - time0)
               time0 = time1
            end if
         end subroutine get_rates_with_screening
         
         
         subroutine get_weaklib_rates(ierr)
            use weak_lib, only: eval_weak_reaction_info
            use weak_def, only: weak_lhs_nuclide_name, weak_rhs_nuclide_name
            use num_lib, only: safe_log10
            integer, intent(out) :: ierr
            real(dp), parameter :: lntwo = log(2d0) ! 0.6931471805599453d0
            integer :: i, id
            include 'formats.dek'
            n% ldecay => ldecay
            n% d_ldecay_dT9 => d_ldecay_dT9
            n% d_ldecay_dlYeRho => d_ldecay_dlYeRho
            n% lcapture => lcapture
            n% d_lcapture_dT9 => d_lcapture_dT9
            n% d_lcapture_dlYeRho => d_lcapture_dlYeRho
            n% lneutrino => lneutrino
            n% d_lneutrino_dT9 => d_lneutrino_dT9
            n% d_lneutrino_dlYeRho => d_lneutrino_dlYeRho
            n% lambda => lambda
            n% dlambda_dlnT => dlambda_dlnT
            n% dlambda_dlnRho => dlambda_dlnRho
            n% lambda => lambda
            n% dlambda_dlnT => dlambda_dlnT
            n% dlambda_dlnRho => dlambda_dlnRho
            n% Q => Q
            n% dQ_dlnT => dQ_dlnT
            n% dQ_dlnRho => dQ_dlnRho
            n% Qneu => Qneu
            n% dQneu_dlnT => dQneu_dlnT
            n% dQneu_dlnRho => dQneu_dlnRho
            call eval_weak_reaction_info(
     >         g% weaklib_ids(1:num_weaklib_rates), temp*1d-9, ye*rho,
     >         eta, d_eta_dlnT, d_eta_dlnRho,
     >         n% ldecay, n% d_ldecay_dT9, n% d_ldecay_dlYeRho,
     >         n% lcapture, n% d_lcapture_dT9, n% d_lcapture_dlYeRho,
     >         n% lneutrino, n% d_lneutrino_dT9, n% d_lneutrino_dlYeRho,
     >         n% lambda, n% dlambda_dlnT, n% dlambda_dlnRho,
     >         n% Q, n% dQ_dlnT, n% dQ_dlnRho,
     >         n% Qneu, n% dQneu_dlnT, n% dQneu_dlnRho,
     >         ierr)
            if (doing_timing) then
               call system_clock(time1)
               g% clock_net_weak_rates = g% clock_net_weak_rates + (time1 - time0)
               time0 = time1
            end if

         end subroutine get_weaklib_rates

         subroutine get_ecapture_rates(ierr)
            use ecapture_lib, only: eval_ecapture_reaction_info
            use num_lib, only: safe_log10
            integer, intent(out) :: ierr
            real(dp), parameter :: lntwo = log(2d0) ! 0.6931471805599453d0
            integer :: i, id
            include 'formats.dek'
            n% ldecay => ldecay
            n% d_ldecay_dT9 => d_ldecay_dT9
            n% d_ldecay_dlYeRho => d_ldecay_dlYeRho
            n% lcapture => lcapture
            n% d_lcapture_dT9 => d_lcapture_dT9
            n% d_lcapture_dlYeRho => d_lcapture_dlYeRho
            n% lneutrino => lneutrino
            n% d_lneutrino_dT9 => d_lneutrino_dT9
            n% d_lneutrino_dlYeRho => d_lneutrino_dlYeRho
            n% lambda => lambda
            n% dlambda_dlnT => dlambda_dlnT
            n% dlambda_dlnRho => dlambda_dlnRho
            n% lambda => lambda
            n% dlambda_dlnT => dlambda_dlnT
            n% dlambda_dlnRho => dlambda_dlnRho
            n% Q => Q
            n% dQ_dlnT => dQ_dlnT
            n% dQ_dlnRho => dQ_dlnRho
            n% Qneu => Qneu
            n% dQneu_dlnT => dQneu_dlnT
            n% dQneu_dlnRho => dQneu_dlnRho
            call eval_ecapture_reaction_info(
     >         g% weaklib_ids(1:num_weaklib_rates), temp*1d-9, ye*rho, zbar,
     >         eta, d_eta_dlnT, d_eta_dlnRho,
     >         n% ldecay, n% d_ldecay_dT9, n% d_ldecay_dlYeRho,
     >         n% lcapture, n% d_lcapture_dT9, n% d_lcapture_dlYeRho,
     >         n% lneutrino, n% d_lneutrino_dT9, n% d_lneutrino_dlYeRho,
     >         n% lambda, n% dlambda_dlnT, n% dlambda_dlnRho,
     >         n% Q, n% dQ_dlnT, n% dQ_dlnRho,
     >         n% Qneu, n% dQneu_dlnT, n% dQneu_dlnRho,
     >         ierr)
            if (doing_timing) then
               call system_clock(time1)
               g% clock_net_weak_rates = g% clock_net_weak_rates + (time1 - time0)
               time0 = time1
            end if

         end subroutine get_ecapture_rates
      
      end subroutine eval_net
         
         
      subroutine get_T_limit_factor(
     >      temp, lnT, T_lo, T_hi, lnT_lo, lnT_hi,
     >      min_ln_factor, min_factor,
     >      factor, d_factor_dT)
         real(dp), intent(in) :: 
     >      temp, lnT, T_lo, T_hi, lnT_lo, lnT_hi,
     >      min_ln_factor, min_factor
         real(dp), intent(out) ::
     >      factor, d_factor_dT
         real(dp) :: ln_factor, d_ln_factor_dlnT
         factor = 1d0
         d_factor_dT = 0d0
         if (temp <= T_lo) return
         if (temp >= T_hi) then
            factor = min_factor
            return
         end if
         ln_factor = min_ln_factor*(lnT - lnT_lo)/(lnT_hi - lnT_lo)
         d_ln_factor_dlnT = min_ln_factor/(lnT_hi - lnT_lo)
         factor = exp(ln_factor)
         d_factor_dT = d_ln_factor_dlnT*factor/temp
      end subroutine get_T_limit_factor

         
      subroutine set_molar_abundances(g, num_isos, x, y, ierr)
         type (Net_General_Info), pointer :: g
         integer, intent(in) :: num_isos
         real(dp), intent(in) :: x(:)
         real(dp), intent(out) :: y(:)
         integer, intent(out) :: ierr
         
         real(dp) :: sum
         integer :: i, ci
         character (len=256) :: message
         include 'formats.dek'
         sum = 0
         do i = 1, g% num_isos
            sum = sum + x(i)
            ci = g% chem_id(i)
            if (ci <= 0) then
               write(*,*) 'problem in set_molar_abundances'
               write(*,*) 'i', i
               write(*,*) 'g% num_isos', g% num_isos
               write(*,*) 'g% chem_id(i)', g% chem_id(i)
               stop 'set_molar_abundances' 
            end if
            y(i) = min(1d0, max(x(i), 0d0)) / chem_isos% Z_plus_N(ci)
         enddo
   
         if (abs(sum - 1d0) > 1d-2) then
            ierr = -1
            !do i = 1, g% num_isos
            !   ci = g% chem_id(i)
            !   write(*,2) chem_isos% name(ci), i, x(i)
            !end do
            return
         end if
      
      end subroutine set_molar_abundances

      
      subroutine get_T_rho_args(temp, logtemp, rho, logrho, info)
         real(dp), intent(inout) :: temp, logtemp ! log10 of temp
         real(dp), intent(inout) :: rho, logrho ! log10 of rho
         integer, intent(out) :: info
         info = 0
         if (temp == arg_not_provided .and. logtemp == arg_not_provided) then
            info = -2
            return
         end if
         if (logtemp == arg_not_provided) logtemp = LOG10(temp)
         if (temp == arg_not_provided) temp = 10**logtemp
         if (temp <= 0) then
            info = -1
            return
         end if
         if (rho == arg_not_provided .and. logrho == arg_not_provided) then
            info = -3
            return
         end if
         if (logrho == arg_not_provided) logrho = LOG10(rho)
         if (rho == arg_not_provided) rho = 10**logrho
         if (rho <= 0) then
            info = -1
            return
         end if
      end subroutine get_T_rho_args
      
      
      subroutine do_clean_up_fractions(nzlo, nzhi, species, nz, xa, max_sum_abs, xsum_tol, ierr)
         integer, intent(in) :: nzlo, nzhi, species, nz
         real(dp), intent(inout) :: xa(:,:) ! (species, nz)
         real(dp), intent(in) :: max_sum_abs, xsum_tol
         integer, intent(out) :: ierr
         integer :: k, op_err
         ierr = 0
         if (nzlo == nzhi) then
            call do_clean1(species, xa(1: species, nzlo), nzlo, max_sum_abs, xsum_tol, ierr)
            return
         end if         
!x$OMP  PARALLEL DO PRIVATE(k, op_err)
         do k = nzlo, nzhi
            op_err = 0
            call do_clean1(species, xa(1: species, k), k, max_sum_abs, xsum_tol, op_err)
            if (op_err /= 0) ierr = op_err
         end do
!x$OMP  END PARALLEL DO
      end subroutine do_clean_up_fractions
      

      subroutine do_clean1(species, xa, k, max_sum_abs, xsum_tol, ierr)
         use utils_lib
         integer, intent(in) :: species, k
         real(dp), intent(inout) :: xa(:) ! (species)
         real(dp), intent(in) :: max_sum_abs, xsum_tol
         integer, intent(out) :: ierr
         integer :: j
         real(dp) :: xsum
         real(dp), parameter :: tiny_x = 1d-99
         character (len=256) :: message
 2       format(a40,i6,1pe26.16)
         if (max_sum_abs > 1) then ! check for crazy values
            xsum = sum(abs(xa(1: species)))
            if (is_bad_num(xsum) .or. xsum > max_sum_abs) then
               ierr = -1
               return
            end if
         end if
         ierr = 0
         do j = 1, species
            if (xa(j) < tiny_x) xa(j) = tiny_x
            if (xa(j) > 1) xa(j) = 1
         end do
         
         xsum = sum(xa(1: species))         
         if (abs(xsum-1) > xsum_tol) then
            ierr = -1
            return
         end if
         xa(1: species) = xa(1: species)/xsum
      end subroutine do_clean1
      

      end module net_eval

