! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module net_derivs
      use net_def
      use const_def
      use chem_def
      use net_derivs_support
      use net_combos_support
      use rates_def
      use rates_lib

      implicit none
      private
      public :: get_derivs


      logical, parameter :: combine_beta_rates = .true.
      
      real(dp), parameter :: tiny_rate = 1d-50
      
      contains


      subroutine get_derivs(
     >         n, category_factors, eta, ye, temp, den, abar, zbar,
     >         num_reactions, rate_factors, symbolic, ierr)
         use rates_lib, only: eval_ecapnuc_rate
         type (Net_Info), pointer :: n
         integer, intent(in) :: num_reactions
         real(dp), intent(in) :: 
     >      category_factors(:), eta, ye, temp, den, abar, zbar,
     >      rate_factors(:)
         logical, intent(in) :: symbolic
         integer, intent(out) :: ierr

         logical :: all_okay
         real(dp), pointer :: rates(:,:)
         integer, pointer :: itab(:), rtab(:)
         integer :: ir, i, op_err
         logical, target :: deriv_flgs_data(num_reactions)
         logical, pointer :: deriv_flgs(:)
         type (Net_General_Info), pointer  :: g
         real(dp), pointer :: y(:)
         integer :: time0, time1
         real(dp) :: T9, T932
         
         ierr = 0

         T9 = temp*1d-9
         T932 = T9**1.5d0
         
         y => n% y
         g => n% g
         if (g% doing_derivs_timing) call system_clock(time0)

         itab => g% net_iso
         rtab => g% net_reaction
         rates => n% rate_screened
         
         deriv_flgs => deriv_flgs_data
         deriv_flgs(:) = .false. ! we'll check this later to see if we got them all.

         n% dydt = 0
         n% d_dydt_dy = 0

         if (rtab(irprot_to_neut) /= 0 .or. rtab(irneut_to_prot) /= 0) then
            call eval_ecapnuc_rate(eta, temp, n% rpen, n% rnep, n% spen, n% snep)
         end if
         
         if (rtab(irhe4_breakup) /= 0 .or. rtab(irhe4_rebuild) /= 0) then
            call get_he4_breakup_and_rebuild(n, g, y, num_reactions, rate_factors,
     >         irhe4_breakup, irhe4_rebuild, .true.,
     >         n% d_rhe4_breakup_dneut, n% d_rhe4_breakup_dprot,
     >         n% d_rhe4_rebuild_dneut, n% d_rhe4_rebuild_dprot)
         end if
         
         if (rtab(irhe4_to_2h1_2neut) /= 0 .or. rtab(irhe4_from_2h1_2neut) /= 0) then
            call get_he4_breakup_and_rebuild(n, g, y, num_reactions, rate_factors,
     >         irhe4_to_2h1_2neut, irhe4_from_2h1_2neut, .false.,
     >         n% d_rhe4_to_2h1_2neut_dneut, n% d_rhe4_to_2h1_2neut_dh1,
     >         n% d_rhe4_from_2h1_2neut_dneut, n% d_rhe4_from_2h1_2neut_dh1)
         end if
         
         call T_limit_high_mass_a_captures(n)
         call T_limit_intermediate_combo_a_captures(n)
         call setup_combo_reactions

         if (g% doing_derivs_timing) then
            call system_clock(time1)
            g% clock_derivs_setup = g% clock_derivs_setup + (time1 - time0)
         end if
         
         ierr = 0
         do i=1, n% g% num_reactions
            call get1_derivs( 
     >         n, i, category_factors, eta, ye, temp, den, abar, zbar, 
     >         num_reactions, rate_factors, rates, rtab, itab,
     >         deriv_flgs, symbolic, ierr)
            if (ierr /= 0) exit
         end do
         
         
         contains

         
         subroutine setup_combo_reactions

            call get_rates_for_v49_combos( ! also does ti46(a,g)cr50  
     >         n, g, den, temp, T9, T932, y, num_reactions, rate_factors,
     >         n% d_rti48prot_to_cr50_dprot, n% d_rcr50gprot_to_ti48_dprot,
     >         n% d_rti46aprot_to_ti48_dprot, n% d_rti48prot_to_ti46_dprot,
     >         n% d_rti46aprot_to_cr50_dprot, n% d_rcr50gprot_to_ti46_dprot
     >         ) 
            call get_rates_for_v51_combos( ! also does ti48(a,g)cr52  
     >         n, g, den, temp, T9, T932, y, num_reactions, rate_factors,
     >         n% d_rti50prot_to_cr52_dprot, n% d_rcr52gprot_to_ti50_dprot,
     >         n% d_rti48aprot_to_ti50_dprot, n% d_rti50prot_to_ti48_dprot,
     >         n% d_rti48aprot_to_cr52_dprot, n% d_rcr52gprot_to_ti48_dprot
     >         ) 
            call get_rates_for_v53_combos( ! also does ti50(a,g)cr54  
     >         n, g, den, temp, T9, T932, y, num_reactions, rate_factors,
     >         n% d_rti52prot_to_cr54_dprot, n% d_rcr54gprot_to_ti52_dprot,
     >         n% d_rti50aprot_to_ti52_dprot, n% d_rti52prot_to_ti50_dprot,
     >         n% d_rti50aprot_to_cr54_dprot, n% d_rcr54gprot_to_ti50_dprot
     >         ) 
            call get_rates_for_v55_combos( ! also does ti52(a,g)cr56  
     >         n, g, den, temp, T9, T932, y, num_reactions, rate_factors,
     >         n% d_rti54prot_to_cr56_dprot, n% d_rcr56gprot_to_ti54_dprot,
     >         n% d_rti52aprot_to_ti54_dprot, n% d_rti54prot_to_ti52_dprot,
     >         n% d_rti52aprot_to_cr56_dprot, n% d_rcr56gprot_to_ti52_dprot
     >         ) 

            call get_rates_for_mn53_combos( ! also does cr50(a,g)fe54  
     >         n, g, den, temp, T9, T932, y, num_reactions, rate_factors,
     >         n% d_rcr52prot_to_fe54_dprot, n% d_rfe54gprot_to_cr52_dprot,
     >         n% d_rcr50aprot_to_cr52_dprot, n% d_rcr52prot_to_cr50_dprot,
     >         n% d_rcr50aprot_to_fe54_dprot, n% d_rfe54gprot_to_cr50_dprot
     >         ) 
            call get_rates_for_mn55_combos( ! also does cr52(a,g)fe56  
     >         n, g, den, temp, T9, T932, y, num_reactions, rate_factors,
     >         n% d_rcr54prot_to_fe56_dprot, n% d_rfe56gprot_to_cr54_dprot,
     >         n% d_rcr52aprot_to_cr54_dprot, n% d_rcr54prot_to_cr52_dprot,
     >         n% d_rcr52aprot_to_fe56_dprot, n% d_rfe56gprot_to_cr52_dprot
     >         )             
            call get_rates_for_mn57_combos( ! also does cr54(a,g)fe58  
     >         n, g, den, temp, T9, T932, y, num_reactions, rate_factors,
     >         n% d_rcr56prot_to_fe58_dprot, n% d_rfe58gprot_to_cr56_dprot,
     >         n% d_rcr54aprot_to_cr56_dprot, n% d_rcr56prot_to_cr54_dprot,
     >         n% d_rcr54aprot_to_fe58_dprot, n% d_rfe58gprot_to_cr54_dprot
     >         ) 
            call get_rates_for_mn59_combos( ! also does cr56(a,g)fe60  
     >         n, g, den, temp, T9, T932, y, num_reactions, rate_factors,
     >         n% d_rcr58prot_to_fe60_dprot, n% d_rfe60gprot_to_cr58_dprot,
     >         n% d_rcr56aprot_to_cr58_dprot, n% d_rcr58prot_to_cr56_dprot,
     >         n% d_rcr56aprot_to_fe60_dprot, n% d_rfe60gprot_to_cr56_dprot
     >         )
            call get_rates_for_co55_combos( ! also does fe52(a,g)ni56
     >         n, g, den, temp, T9, T932, y, num_reactions, rate_factors,
     >         n% d_rfe54prot_to_ni56_dprot, n% d_rni56gprot_to_fe54_dprot,
     >         n% d_rfe52aprot_to_fe54_dprot, n% d_rfe54prot_to_fe52_dprot,
     >         n% d_rfe52aprot_to_ni56_dprot, n% d_rni56gprot_to_fe52_dprot
     >         )
            call get_rates_for_co57_combos( ! also does fe54(a,g)ni58
     >         n, g, den, temp, T9, T932, y, num_reactions, rate_factors,
     >         n% d_rfe56prot_to_ni58_dprot, n% d_rni58gprot_to_fe56_dprot,
     >         n% d_rfe54aprot_to_fe56_dprot, n% d_rfe56prot_to_fe54_dprot,
     >         n% d_rfe54aprot_to_ni58_dprot, n% d_rni58gprot_to_fe54_dprot
     >         )
            call get_rates_for_co59_combos( ! also does fe56(a,g)ni60  
     >         n, g, den, temp, T9, T932, y, num_reactions, rate_factors,
     >         n% d_rfe58prot_to_ni60_dprot, n% d_rni60gprot_to_fe58_dprot,
     >         n% d_rfe56aprot_to_fe58_dprot, n% d_rfe58prot_to_fe56_dprot,
     >         n% d_rfe56aprot_to_ni60_dprot, n% d_rni60gprot_to_fe56_dprot
     >         )
            call get_rates_for_co61_combos( ! also does fe58(a,g)ni62   
     >         n, g, den, temp, T9, T932, y, num_reactions, rate_factors,
     >         n% d_rfe60prot_to_ni62_dprot, n% d_rni62gprot_to_fe60_dprot,
     >         n% d_rfe58aprot_to_fe60_dprot, n% d_rfe60prot_to_fe58_dprot,
     >         n% d_rfe58aprot_to_ni62_dprot, n% d_rni62gprot_to_fe58_dprot
     >         )
            call get_rates_for_co63_combos( ! also does fe60(a,g)ni64 
     >         n, g, den, temp, T9, T932, y, num_reactions, rate_factors,
     >         n% d_rfe62prot_to_ni64_dprot, n% d_rni64gprot_to_fe62_dprot,
     >         n% d_rfe60aprot_to_fe62_dprot, n% d_rfe62prot_to_fe60_dprot,
     >         n% d_rfe60aprot_to_ni64_dprot, n% d_rni64gprot_to_fe60_dprot
     >         )
            call get_rates_for_cu59_combos(
     >         n, g, den, temp, T9, T932, y, num_reactions, rate_factors,
     >         n% d_rni58prot_to_zn60_dprot, n% d_rzn60gprot_to_ni58_dprot,
     >         n% d_rni56aprot_to_ni58_dprot, n% d_rni58prot_to_ni56_dprot,
     >         n% d_rni56aprot_to_zn60_dprot, n% d_rzn60gprot_to_ni56_dprot
     >         )
            call get_rates_for_cu61_combos(
     >         n, g, den, temp, T9, T932, y, num_reactions, rate_factors,
     >         n% d_rni60prot_to_zn62_dprot, n% d_rzn62gprot_to_ni60_dprot,
     >         n% d_rni58aprot_to_ni60_dprot, n% d_rni60prot_to_ni58_dprot,
     >         n% d_rni58aprot_to_zn62_dprot, n% d_rzn62gprot_to_ni58_dprot
     >         )
            call get_rates_for_cu63_combos(
     >         n, g, den, temp, T9, T932, y, num_reactions, rate_factors,
     >         n% d_rni62prot_to_zn64_dprot, n% d_rzn64gprot_to_ni62_dprot,
     >         n% d_rni60aprot_to_ni62_dprot, n% d_rni62prot_to_ni60_dprot,
     >         n% d_rni60aprot_to_zn64_dprot, n% d_rzn64gprot_to_ni60_dprot
     >         )
            call get_rates_for_cu65_combos(
     >         n, g, den, temp, T9, T932, y, num_reactions, rate_factors,
     >         n% d_rni64prot_to_zn66_dprot, n% d_rzn66gprot_to_ni64_dprot,
     >         n% d_rni62aprot_to_ni64_dprot, n% d_rni64prot_to_ni62_dprot,
     >         n% d_rni62aprot_to_zn66_dprot, n% d_rzn66gprot_to_ni62_dprot
     >         )
            call get_rates_for_cu67_combos( ! also does ni64(a,g)zn68
     >         n, g, den, temp, T9, T932, y, num_reactions, rate_factors,
     >         n% d_rni66prot_to_zn68_dprot, n% d_rzn68gprot_to_ni66_dprot,
     >         n% d_rni64aprot_to_ni66_dprot, n% d_rni66prot_to_ni64_dprot,
     >         n% d_rni64aprot_to_zn68_dprot, n% d_rzn68gprot_to_ni64_dprot
     >         )
            call get_rates_for_ga71_combos( ! also does zn68(a,g)ge72
     >         n, g, den, temp, T9, T932, y, num_reactions, rate_factors,
     >         n% d_rzn70prot_to_ge72_dprot, n% d_rge72gprot_to_zn70_dprot,
     >         n% d_rzn68aprot_to_zn70_dprot, n% d_rzn70prot_to_zn68_dprot,
     >         n% d_rzn68aprot_to_ge72_dprot, n% d_rge72gprot_to_zn68_dprot
     >         )

            call get_rates_for_ne20_ne22(
     >         n, g, den, temp, T9, T932, y, num_reactions, rate_factors)

            call get_rates_for_ca40_ca42(
     >         n, g, den, temp, T9, T932, y, num_reactions, rate_factors)
            call get_rates_for_ca42_ca44(
     >         n, g, den, temp, T9, T932, y, num_reactions, rate_factors)
            call get_rates_for_ca44_ca46(
     >         n, g, den, temp, T9, T932, y, num_reactions, rate_factors)
            call get_rates_for_ca46_ca48(
     >         n, g, den, temp, T9, T932, y, num_reactions, rate_factors)
         
            call get_rates_for_ti44_ti46(
     >         n, g, den, temp, T9, T932, y, num_reactions, rate_factors)
            call get_rates_for_ti46_ti48(
     >         n, g, den, temp, T9, T932, y, num_reactions, rate_factors)
            call get_rates_for_ti48_ti50(
     >         n, g, den, temp, T9, T932, y, num_reactions, rate_factors)
            call get_rates_for_ti50_ti52(
     >         n, g, den, temp, T9, T932, y, num_reactions, rate_factors)

            call get_rates_for_cr48_cr50(
     >         n, g, den, temp, T9, T932, y, num_reactions, rate_factors)
            call get_rates_for_cr50_cr52(
     >         n, g, den, temp, T9, T932, y, num_reactions, rate_factors)
            call get_rates_for_cr52_cr54(
     >         n, g, den, temp, T9, T932, y, num_reactions, rate_factors)
            call get_rates_for_cr54_cr56(
     >         n, g, den, temp, T9, T932, y, num_reactions, rate_factors)

            call get_rates_for_v48_combos(
     >         n, g, den, temp, y, num_reactions, rate_factors)
            call get_rates_for_v50_combos(
     >         n, g, den, temp, y, num_reactions, rate_factors)
            call get_rates_for_v52_combos(
     >         n, g, den, temp, y, num_reactions, rate_factors)

            call get_rates_for_cr49_combos(
     >         n, g, den, temp, y, num_reactions, rate_factors)
            call get_rates_for_cr51_combos(
     >         n, g, den, temp, y, num_reactions, rate_factors)
            call get_rates_for_cr53_combos(
     >         n, g, den, temp, y, num_reactions, rate_factors)
            call get_rates_for_cr55_combos(
     >         n, g, den, temp, y, num_reactions, rate_factors)

            call get_rates_for_mn52_combos(
     >         n, g, den, temp, y, num_reactions, rate_factors)
            call get_rates_for_mn54_combos(
     >         n, g, den, temp, y, num_reactions, rate_factors)
            call get_rates_for_mn56_combos(
     >         n, g, den, temp, y, num_reactions, rate_factors)
         
            call get_rates_for_fe52_fe54(
     >         n, g, den, temp, T9, T932, y, num_reactions, rate_factors)
            call get_rates_for_fe54_fe56(
     >         n, g, den, temp, T9, T932, y, num_reactions, rate_factors)

            call get_rates_for_fe53_combos(
     >         n, g, den, temp, y, num_reactions, rate_factors)
            call get_rates_for_fe55_combos(
     >         n, g, den, temp, y, num_reactions, rate_factors)
            call get_rates_for_fe57_combos(
     >         n, g, den, temp, y, num_reactions, rate_factors)
!            call get_rates_for_fe56_fe58(
!     >         n, g, den, temp, T9, T932, y, num_reactions, rate_factors)
            call get_rates_for_fe59_combos(
     >         n, g, den, temp, y, num_reactions, rate_factors)
!            call get_rates_for_fe58_fe60(
!     >         n, g, den, temp, T9, T932, y, num_reactions, rate_factors)
            call get_rates_for_fe60_fe62(
     >         n, g, den, temp, T9, T932, y, num_reactions, rate_factors)

            call get_rates_for_co56_combos(
     >         n, g, den, temp, y, num_reactions, rate_factors)
            call get_rates_for_co58_combos(
     >         n, g, den, temp, y, num_reactions, rate_factors)
            
            call get_rates_for_ni57_combos(
     >         n, g, den, temp, y, num_reactions, rate_factors)
!            call get_rates_for_ni56_ni58(
!     >         n, g, den, temp, T9, T932, y, num_reactions, rate_factors)
            call get_rates_for_ni59_combos(
     >         n, g, den, temp, y, num_reactions, rate_factors)
!            call get_rates_for_ni58_ni60(
!     >         n, g, den, temp, T9, T932, y, num_reactions, rate_factors)
            call get_rates_for_ni61_combos(
     >         n, g, den, temp, y, num_reactions, rate_factors)
!            call get_rates_for_ni60_ni62(
!     >         n, g, den, temp, T9, T932, y, num_reactions, rate_factors)
            call get_rates_for_ni62_ni64(
     >         n, g, den, temp, T9, T932, y, num_reactions, rate_factors)
            call get_rates_for_ni64_ni66(
     >         n, g, den, temp, T9, T932, y, num_reactions, rate_factors)
            call get_rates_for_ni64_ni66(
     >         n, g, den, temp, T9, T932, y, num_reactions, rate_factors)

            call get_rates_for_zn60_zn62(
     >         n, g, den, temp, T9, T932, y, num_reactions, rate_factors)
            call get_rates_for_zn62_zn64(
     >         n, g, den, temp, T9, T932, y, num_reactions, rate_factors)
            call get_rates_for_zn64_zn66(
     >         n, g, den, temp, T9, T932, y, num_reactions, rate_factors)
            call get_rates_for_zn66_zn68(
     >         n, g, den, temp, T9, T932, y, num_reactions, rate_factors)
            call get_rates_for_zn68_zn70(
     >         n, g, den, temp, T9, T932, y, num_reactions, rate_factors)

            call get_rates_for_ge70_ge72(
     >         n, g, den, temp, T9, T932, y, num_reactions, rate_factors)
            call get_rates_for_ge72_ge74(
     >         n, g, den, temp, T9, T932, y, num_reactions, rate_factors)
            call get_rates_for_ge74_ge76(
     >         n, g, den, temp, T9, T932, y, num_reactions, rate_factors)
         
         end subroutine setup_combo_reactions


         subroutine Check
            integer :: nrates
            nrates = n% g% num_reactions
            
            do ir = 1, nrates
               if (.not. deriv_flgs(ir)) then
                  all_okay = .false.
                  write(*,'(a,i4,2x,a)') 'missing derivs for ', ir,
     >                  trim(reaction_Name(n% g% reaction_id(ir)))
               end if
            end do
         
         end subroutine Check
         
      
      end subroutine get_derivs


      subroutine get1_derivs(
     >         n, i, category_factors, eta, ye, temp, den, abar, zbar,
     >         num_reactions, rate_factors, rates, rtab, itab,
     >         deriv_flgs, symbolic, ierr)
         type (Net_Info), pointer :: n
         integer, intent(in) :: i, num_reactions
         real(dp), intent(in) :: category_factors(:), 
     >         eta, ye, temp, den, abar, zbar, rate_factors(:)
         real(dp), pointer, intent(in) :: rates(:,:)
         integer, pointer, intent(in) :: rtab(:), itab(:)
         logical, pointer :: deriv_flgs(:)
         logical, intent(in) :: symbolic
         integer, intent(out) :: ierr
         
         integer :: ir, j, prot, neut, h1, he4, c14, n14, ne20, ne22, 
     >         mg21, mg22, mg23, mg24, al23, al24, si24, si25, si26,  
     >         s28, s29, s30, cl31, ar32, ar33, ar34, k35, ca36, ca37, ca38, 
     >         ti41, ti42, v43, cr44, cr45, cr46, cr56, 
     >         fe48, fe49, fe50, fe51, fe52, fe54, fe56, fe58, fe60, fe62, fe64,
     >         ni52, ni53, ni54, ni55, ni56, ni58, ni60, ni62,
     >         zn57, zn58, zn59, zn60, ge62, ge63, ge64, 
     >         se68, kr72, sr76, mo84, sn104
         integer :: weak_id, num_reaction_inputs, in1, in2, in3, in4, in5
         integer :: cin1, cin2, cin3, cin4, cin5
         real(dp) :: din1, din2, din3, din4, din5
         integer :: num_reaction_outputs, out1, out2, out3, out4, out5
         integer :: cout1, cout2, cout3, cout4, cout5
         real(dp) :: dout1, dout2, dout3, dout4, dout5
         type (Net_General_Info), pointer  :: g
         integer, pointer :: reaction_id(:)
         real(dp), pointer :: y(:)
         integer :: i1, i2, i3, idr1, idr2, idr3, o1, o2, o3
         real(dp) :: r, dr1, dr2, dr3, rate, d_rate_dlnT, d_rate_dlnRho, Q, Qneu, rn14ec
         real(dp), parameter :: lntwo = log(2d0)

         integer, parameter :: npart = 24	! no. entries in partition fcn table
         real(dp), dimension(npart) :: Tpart = (/
     >         0.10,0.15,0.20,0.30,0.40,0.50,0.60,0.70,0.80,0.90,1.00,1.50,
     >         2.00,2.50,3.00,3.50,4.00,4.50,5.00,6.00,7.00,8.00,9.00,10.0 /)            
         
         logical :: done, has_prot, has_neut, has_h1, switch_to_prot
         integer :: time0, time1, max_Z, Z_plus_N_for_max_Z
         integer, parameter :: min_Z_for_switch_to_prot = 15

         logical, parameter :: dbg = .false.
         
         include 'formats.dek'
         
         y => n% y
         g => n% g
         reaction_id => g% reaction_id

         ierr = 0
         prot = itab(iprot)
         neut = itab(ineut)
         h1 = itab(ih1)
         he4 = itab(ihe4)

         ir = reaction_id(i)         

         if (reaction_outputs(1,ir) == 0) return ! skip aux reactions
         
         if (dbg) write(*,'(/,a)') ' reaction name <' // trim(reaction_Name(ir)) // '>'

         if (g% doing_derivs_timing) call system_clock(time0)
         
         max_Z = g% reaction_max_Z(i)
         Z_plus_N_for_max_Z = g% reaction_max_Z_plus_N_for_max_Z(i)
         
         select case(max_Z)
            
         case(20) ! ca
         
            select case(Z_plus_N_for_max_Z)

            case(42) ! ca42
               select case(ir)
            
               case(irca40ng_to_ca42) ! ca40(neut,g)ca41(neut,g)ca42
                  call do_neutg_neutg_reaction(i, ica40, ica42, n% d_rca40neut_to_ca42_dneut)
                  return
                  
               case(irca42gn_to_ca40) ! ca42(g,neut)ca41(g,neut)ca40
                  call do_gneut_gneut_reaction(i, ica40, ica42, n% d_rca42g_to_ca40_dneut)
                  return

               end select
            
            case(44) ! ca44
               select case(ir)

               case(irca42ng_to_ca44) ! ca42(neut,g)ca43(neut,g)ca44
                  call do_neutg_neutg_reaction(i, ica42, ica44, n% d_rca42neut_to_ca44_dneut)
                  return
                  
               case(irca44gn_to_ca42) ! ca44(g,neut)ca43(g,neut)ca42
                  call do_gneut_gneut_reaction(i, ica42, ica44, n% d_rca44g_to_ca42_dneut)
                  return

               end select
            
            case(46) ! ca46
               select case(ir)

               case(irca44ng_to_ca46) ! ca44(neut,g)ca45(neut,g)ca46
                  call do_neutg_neutg_reaction(i, ica44, ica46, n% d_rca44neut_to_ca46_dneut)
                  return
                  
               case(irca46gn_to_ca44) ! ca46(g,neut)ca45(g,neut)ca44
                  call do_gneut_gneut_reaction(i, ica44, ica46, n% d_rca46g_to_ca44_dneut)
                  return

               end select
            
            case(48) ! ca48
               select case(ir)

               case(irca46ng_to_ca48) ! ca46(neut,g)ca47(neut,g)ca48
                  call do_neutg_neutg_reaction(i, ica46, ica48, n% d_rca46neut_to_ca48_dneut)
                  return
               
               case(irca48gn_to_ca46) ! ca48(g,neut)ca47(g,neut)ca46
                  call do_gneut_gneut_reaction(i, ica46, ica48, n% d_rca48g_to_ca46_dneut)
                  return

               end select
            
            end select
            
         case(21) ! sc
            select case(ir)
               
            case(irsc50ec_to_ca48_2n) ! sc50 + 2 e- => ca48 + 2n
               call do_ec_nn_reaction(
     >            i, isc50, ica48, 2, weak_rate_id_for_sc50_ec, ierr)
               return
               
            case(irsc51ec_to_ca48_3n) ! sc51 + 2 e- => ca48 + 3n
               call do_ec_nn_reaction(
     >            i, isc51, ica48, 3, weak_rate_id_for_sc51_ec, ierr)
               return
      
            end select
            
         case(22) ! ti
         
            select case(Z_plus_N_for_max_Z)
            
            case(44) ! ti44
               select case(ir)

               case(irti44ec_to_ca44) ! ti44 + 2 e- => ca44
                  call do_2ec_reaction(i, iti44, ica44, weak_rate_id_for_ti44_ec, ierr)
                  return
                  
               case(irti44_ecnp_ca42)
                  call do_ecnp_reaction(i, iti44, isc44, ica42, weak_rate_id_for_ti44_ec, 
     >               weak_rate_id_for_ti43_ec, weak_rate_id_for_sc43_ec, ierr)
                  return

               end select
            
            case(46) ! ti46
               select case(ir)

               case(irti44ng_to_ti46) ! ti44(neut,g)ti45(neut,g)ti46
                  call do_neutg_neutg_reaction(i, iti44, iti46, n% d_rti44neut_to_ti46_dneut)
                  return
                  
               case(irti46gn_to_ti44) ! ti46(g,neut)ti45(g,neut)ti44
                  call do_gneut_gneut_reaction(i, iti44, iti46, n% d_rti46g_to_ti44_dneut)
                  return

               case(irti46ec_to_ca46) ! ti46 + 2 e- => ca46
                  call do_2ec_reaction(i, iti46, ica46, weak_rate_id_for_ti46_ec, ierr)
                  return

               case(irti46_ecnp_ca44)
                  call do_ecnp_reaction(i, iti46, isc46, ica44, weak_rate_id_for_ti46_ec, 
     >               weak_rate_id_for_ti45_ec, weak_rate_id_for_sc45_ec, ierr)
                  return

               end select
            
            case(48) ! ti48
               select case(ir)

               case(irti46ng_to_ti48) ! ti46(neut,g)ti47(neut,g)ti48
                  call do_neutg_neutg_reaction(i, iti46, iti48, n% d_rti46neut_to_ti48_dneut)
                  return                  
               case(irti48gn_to_ti46) ! ti48(g,neut)ti47(g,neut)ti46
                  call do_gneut_gneut_reaction(i, iti46, iti48, n% d_rti48g_to_ti46_dneut)
                  return

               case(irti46ap_to_ti48) ! ti46(a,p)v49(g,p)ti48
                  call do_aprot_gprot_reaction(i, iti46, iti48, n% d_rti46aprot_to_ti48_dprot)
                  return
               case(irti48p_to_ti46) ! ti48(p,g)v49(p,a)ti46
                  call do_protg_prota_reaction(i, iti46, iti48, n% d_rti48prot_to_ti46_dprot)
                  return

               case(irti48ec_to_ca48) ! ti48 + 2 e- => ca48
                  call do_2ec_reaction(i, iti48, ica48, weak_rate_id_for_ti48_ec, ierr)
                  return

               case(irti48_ecnp_ca46)
                  call do_ecnp_reaction(i, iti48, isc48, ica46, weak_rate_id_for_ti48_ec, 
     >               weak_rate_id_for_ti47_ec, weak_rate_id_for_sc47_ec, ierr)
                  return

               end select
            
            case(50) ! ti50
               select case(ir)

               case(irti48ng_to_ti50) ! ti48(neut,g)ti49(neut,g)ti50
                  call do_neutg_neutg_reaction(i, iti48, iti50, n% d_rti48neut_to_ti50_dneut)
                  return                 
               case(irti50gn_to_ti48) ! ti50(g,neut)ti49(g,neut)ti48
                  call do_gneut_gneut_reaction(i, iti48, iti50, n% d_rti50g_to_ti48_dneut)
                  return

               case(irti48ap_to_ti50) ! ti48(a,p)v51(g,p)ti50
                  call do_aprot_gprot_reaction(i, iti48, iti50, n% d_rti48aprot_to_ti50_dprot)
                  return
               case(irti50p_to_ti48) ! ti50(p,g)v51(p,a)ti48
                  call do_protg_prota_reaction(i, iti48, iti50, n% d_rti50prot_to_ti48_dprot)
                  return

               case(irti50ec_to_ca48_2n) ! ti50 + 2 e- => ca48
                  call do_ec_nn_reaction(
     >               i, iti50, ica48, 2, weak_rate_id_for_ti50_ec, ierr)
                  return

               case(irti50_ecnp_ca48)
                  call do_ecnp_reaction(i, iti50, isc50, ica48, weak_rate_id_for_ti50_ec, 
     >               weak_rate_id_for_ti49_ec, weak_rate_id_for_sc49_ec, ierr)
                  return

               end select
            
            case(52) ! ti52
               select case(ir)

               case(irti50ap_to_ti52) ! ti50(a,p)v53(g,p)ti52
                  call do_aprot_gprot_reaction(i, iti50, iti52, n% d_rti50aprot_to_ti52_dprot)
                  return
               case(irti52p_to_ti50) ! ti52(p,g)v53(p,a)ti50
                  call do_protg_prota_reaction(i, iti50, iti52, n% d_rti52prot_to_ti50_dprot)
                  return

               case(irti50ng_to_ti52) ! ti50(neut,g)ti51(neut,g)ti52
                  call do_neutg_neutg_reaction(i, iti50, iti52, n% d_rti50neut_to_ti52_dneut)
                  return                 
               case(irti52gn_to_ti50) ! ti52(g,neut)ti51(g,neut)ti50
                  call do_gneut_gneut_reaction(i, iti50, iti52, n% d_rti52g_to_ti50_dneut)
                  return

               case(irti52ec_to_ca48_4n) ! ti52 + 2 e- => ca48
                  call do_ec_nn_reaction(
     >               i, iti52, ica48, 4, weak_rate_id_for_ti52_ec, ierr)
                  return
               
               case(irti52ec_to_sc51_n) ! ti52 + 2 e- => sc51 + 1n
                  call do_ec_nn_reaction(
     >               i, iti52, isc51, 1, weak_rate_id_for_ti52_ec, ierr)
                  return

               case(irti52_ecnp_ca48)
                  call do_ecnp_nn_reaction(
     >               i, iti52, isc52, ica48, 2, weak_rate_id_for_ti52_ec, 
     >               weak_rate_id_for_ti51_ec, weak_rate_id_for_sc51_ec, temp, ierr)
                  return

               end select
      
            end select
         
         case(23) ! v
            select case(ir)
         
            case(irv53ec_to_ti52_n)
               call do_ec_nn_reaction(
     >            i, iv53, iti52, 1, weak_rate_id_for_v53_ec, ierr)
               return

            case(irv54ec_to_ti52_2n) ! v54 + 2 e- => ti52 + 2n
               call do_ec_nn_reaction(
     >            i, iv54, iti52, 2, weak_rate_id_for_v54_ec, ierr)
               return
               
            case(irv55ec_to_ti52_3n) ! v55 + 2 e- => ti52 + 3n
               call do_ec_nn_reaction(
     >            i, iv55, iti52, 3, weak_rate_id_for_v55_ec, ierr)
               return
      
            end select
         
         case(24) ! cr
         
            select case(Z_plus_N_for_max_Z)

            case(48) ! cr48
               select case(ir)
      
               case(irti46an_to_cr48) ! ti46(a,n)cr49(g,n)cr48
                  call do_aneut_gneut_reaction(i, iti46, icr48, n% d_rti46aneut_to_cr48_dneut)
                  return         
               case(ircr48ng_to_ti46) ! cr48(n,g)cr49(n,a)ti46
                  call do_neutg_neuta_reaction(i, iti46, icr48, n% d_rcr48neut_to_ti46_dneut)
                  return
      
               case(irti48pn_to_cr48) ! ti48(p,n)v48(p,n)cr48
                  call do_pn_pn_reaction(i, iti48, icr48,
     >               n% d_rti48pn_to_cr48_dprot, n% d_rti48pn_to_cr48_dneut)
                  return         
               case(ircr48np_to_ti48) ! cr48(n,p)v48(n,p)ti48
                  call do_np_np_reaction(i, iti48, icr48, 
     >               n% d_rcr48np_to_ti48_dprot, n% d_rcr48np_to_ti48_dneut)
                  return

               case(ircr48ec_to_ti48) ! cr48 + 2 e- => ti48
                  call do_2ec_reaction(i, icr48, iti48, weak_rate_id_for_cr48_ec, ierr)
                  return

               case(ircr48_ecnp_ti46)
                  call do_ecnp_reaction(i, icr48, iv48, iti46, weak_rate_id_for_cr48_ec, 
     >               weak_rate_id_for_cr47_ec, weak_rate_id_for_v47_ec, ierr)
                  return

               end select

            case(50) ! cr50
               select case(ir)

               case(ircr48ng_to_cr50) ! cr48(neut,g)cr49(neut,g)cr50
                  call do_neutg_neutg_reaction(i, icr48, icr50, n% d_rcr48neut_to_cr50_dneut)
                  return                
               case(ircr50gn_to_cr48) ! cr50(g,neut)cr49(g,neut)cr48
                  call do_gneut_gneut_reaction(i, icr48, icr50, n% d_rcr50g_to_cr48_dneut)
                  return

               case(irti46ap_to_cr50) ! ti46(a,prot)v49(prot,g)cr50
                  call do_aprot_protg_reaction(i, iti46, icr50, n% d_rti46aprot_to_cr50_dprot)
                  return                              
               case(ircr50gp_to_ti46) ! cr50(g,prot)v49(prot,a)ti46
                  call do_gprot_prota_reaction(i, iti46, icr50, n% d_rcr50gprot_to_ti46_dprot)
                  return
      
               case(irti50pn_to_cr50) ! ti50(p,n)v50(p,n)cr50
                  call do_pn_pn_reaction(i, iti50, icr50,
     >               n% d_rti50pn_to_cr50_dprot, n% d_rti50pn_to_cr50_dneut)
                  return         
               case(ircr50np_to_ti50) ! cr50(n,p)v50(n,p)ti50
                  call do_np_np_reaction(i, iti50, icr50, 
     >               n% d_rcr50np_to_ti50_dprot, n% d_rcr50np_to_ti50_dneut)
                  return
      
               case(irti48an_to_cr50) ! ti48(a,n)cr51(g,n)cr50
                  call do_aneut_gneut_reaction(i, iti48, icr50, n% d_rti48aneut_to_cr50_dneut)
                  return         
               case(ircr50ng_to_ti48) ! cr50(n,g)cr51(n,a)ti48
                  call do_neutg_neuta_reaction(i, iti48, icr50, n% d_rcr50neut_to_ti48_dneut)
                  return
      
               case(irti46an_to_cr50) ! ti46(a,n)cr49(n,g)cr50
                  call do_aneut_neutg_reaction(i, iti46, icr50, n% d_rti46aneut_to_cr50_dneut)
                  return         
               case(ircr50gn_to_ti46) ! cr50(g,n)cr49(n,a)ti46
                  call do_gneut_neuta_reaction(i, iti46, icr50, n% d_rcr50gneut_to_ti46_dneut)
                  return
         
               case(irti48p_to_cr50) ! ti48(prot,g)v49(prot,g)cr50
                  call do_protg_protg_reaction(i, iti48, icr50, n% d_rti48prot_to_cr50_dprot)
                  return                              
               case(ircr50gp_to_ti48) ! cr50(g,prot)v49(g,prot)ti48
                  call do_gprot_gprot_reaction(i, iti48, icr50, n% d_rcr50gprot_to_ti48_dprot)
                  return

               case(ircr50ec_to_ti50) ! cr50 + 2 e- => ti50
                  call do_2ec_reaction(i, icr50, iti50, weak_rate_id_for_cr50_ec, ierr)
                  return
               case(irti50ee_to_cr50) ! ti50 => cr50 + 2 e-
                  call do_2ee_reaction(i, iti50, icr50, weak_rate_id_for_ti50_ee, ierr)
                  return

               case(ircr50_ecnp_ti48)
                  call do_ecnp_reaction(i, icr50, iv50, iti48, weak_rate_id_for_cr50_ec, 
     >               weak_rate_id_for_cr49_ec, weak_rate_id_for_v49_ec, ierr)
                  return

               end select

            case(52) ! cr52
               select case(ir)

               case(ircr50ng_to_cr52) ! cr50(neut,g)cr51(neut,g)cr52
                  call do_neutg_neutg_reaction(i, icr50, icr52, n% d_rcr50neut_to_cr52_dneut)
                  return               
               case(ircr52gn_to_cr50) ! cr52(g,neut)cr51(g,neut)cr50
                  call do_gneut_gneut_reaction(i, icr50, icr52, n% d_rcr52g_to_cr50_dneut)
                  return

               case(ircr50ap_to_cr52) ! cr50(a,p)mn53(g,p)cr52
                  call do_aprot_gprot_reaction(i, icr50, icr52, n% d_rcr50aprot_to_cr52_dprot)
                  return
               case(ircr52p_to_cr50) ! cr52(p,g)mn53(p,a)cr50
                  call do_protg_prota_reaction(i, icr50, icr52, n% d_rcr52prot_to_cr50_dprot)
                  return
      
               case(irti50an_to_cr52) ! ti50(a,n)cr53(g,n)cr52
                  call do_aneut_gneut_reaction(i, iti50, icr52, n% d_rti50aneut_to_cr52_dneut)
                  return         
               case(ircr52ng_to_ti50) ! cr52(n,g)cr53(n,a)ti50
                  call do_neutg_neuta_reaction(i, iti50, icr52, n% d_rcr52neut_to_ti50_dneut)
                  return
      
               case(irti52pn_to_cr52) ! ti52(p,n)v52(p,n)cr52
                  call do_pn_pn_reaction(i, iti52, icr52,
     >               n% d_rti52pn_to_cr52_dprot, n% d_rti52pn_to_cr52_dneut)
                  return         
               case(ircr52np_to_ti52) ! cr52(n,p)v52(n,p)ti52
                  call do_np_np_reaction(i, iti52, icr52, 
     >               n% d_rcr52np_to_ti52_dprot, n% d_rcr52np_to_ti52_dneut)
                  return
      
               case(irti48an_to_cr52) ! ti48(a,n)cr51(n,g)cr52
                  call do_aneut_neutg_reaction(i, iti48, icr52, n% d_rti48aneut_to_cr52_dneut)
                  return         
               case(ircr52gn_to_ti48) ! cr52(g,n)cr51(n,a)ti48
                  call do_gneut_neuta_reaction(i, iti48, icr52, n% d_rcr52gneut_to_ti48_dneut)
                  return

               case(irti48ap_to_cr52) ! ti48(a,prot)v51(prot,g)cr52
                  call do_aprot_protg_reaction(i, iti48, icr52, n% d_rti48aprot_to_cr52_dprot)
                  return                              
               case(ircr52gp_to_ti48) ! cr52(g,prot)v51(prot,a)ti48
                  call do_gprot_prota_reaction(i, iti48, icr52, n% d_rcr52gprot_to_ti48_dprot)
                  return
         
               case(irti50p_to_cr52) ! ti50(prot,g)v51(prot,g)cr52
                  call do_protg_protg_reaction(i, iti50, icr52, n% d_rti50prot_to_cr52_dprot)
                  return                              
               case(ircr52gp_to_ti50) ! cr52(g,prot)v51(g,prot)ti50
                  call do_gprot_gprot_reaction(i, iti50, icr52, n% d_rcr52gprot_to_ti50_dprot)
                  return

               case(ircr52ec_to_ti52) ! cr52 + 2 e- => ti52
                  call do_2ec_reaction(i, icr52, iti52, weak_rate_id_for_cr52_ec, ierr)
                  return
               case(irti52ee_to_cr52) ! ti52 => cr52 + 2 e-
                  call do_2ee_reaction(i, iti52, icr52, weak_rate_id_for_ti52_ee, ierr)
                  return

               case(ircr52_ecnp_ti50)
                  call do_ecnp_reaction(i, icr52, iv52, iti50, weak_rate_id_for_cr52_ec, 
     >               weak_rate_id_for_cr51_ec, weak_rate_id_for_v51_ec, ierr)
                  return

               end select
            
            case(54) ! cr54
               select case(ir)

               case(ircr52ng_to_cr54) ! cr52(neut,g)cr53(neut,g)cr54
                  call do_neutg_neutg_reaction(i, icr52, icr54, n% d_rcr52neut_to_cr54_dneut)
                  return
               case(ircr54gn_to_cr52) ! cr54(g,neut)cr53(g,neut)cr52
                  call do_gneut_gneut_reaction(i, icr52, icr54, n% d_rcr54g_to_cr52_dneut)
                  return

               case(ircr52ap_to_cr54) ! cr52(a,p)mn55(g,p)cr54
                  call do_aprot_gprot_reaction(i, icr52, icr54, n% d_rcr52aprot_to_cr54_dprot)
                  return
               case(ircr54p_to_cr52) ! cr54(p,g)mn55(p,a)cr52
                  call do_protg_prota_reaction(i, icr52, icr54, n% d_rcr54prot_to_cr52_dprot)
                  return
      
               case(irti52an_to_cr54) ! ti52(a,n)cr55(g,n)cr54
                  call do_aneut_gneut_reaction(i, iti52, icr54, n% d_rti52aneut_to_cr54_dneut)
                  return         
               case(ircr54ng_to_ti52) ! cr54(n,g)cr55(n,a)ti52
                  call do_neutg_neuta_reaction(i, iti52, icr54, n% d_rcr54neut_to_ti52_dneut)
                  return
      
               case(irti50an_to_cr54) ! ti50(a,n)cr53(n,g)cr54
                  call do_aneut_neutg_reaction(i, iti50, icr54, n% d_rti50aneut_to_cr54_dneut)
                  return         
               case(ircr54gn_to_ti50) ! cr54(g,n)cr53(n,a)ti50
                  call do_gneut_neuta_reaction(i, iti50, icr54, n% d_rcr54gneut_to_ti50_dneut)
                  return

               case(irti50ap_to_cr54) ! ti50(a,prot)v53(prot,g)cr54
                  call do_aprot_protg_reaction(i, iti50, icr54, n% d_rti50aprot_to_cr54_dprot)
                  return                              
               case(ircr54gp_to_ti50) ! cr54(g,prot)v53(prot,a)ti50
                  call do_gprot_prota_reaction(i, iti50, icr54, n% d_rcr54gprot_to_ti50_dprot)
                  return
         
               case(irti52p_to_cr54) ! ti52(prot,g)v53(prot,g)cr54
                  call do_protg_protg_reaction(i, iti52, icr54, n% d_rti52prot_to_cr54_dprot)
                  return                              
               case(ircr54gp_to_ti52) ! cr54(g,prot)v53(g,prot)ti52
                  call do_gprot_gprot_reaction(i, iti52, icr54, n% d_rcr54gprot_to_ti52_dprot)
                  return

               case(ircr54ec_to_ti52_2n) ! cr54 + 2 e- => ti52 + 2n
                  call do_ec_nn_reaction(
     >               i, icr54, iti52, 2, 
     >               weak_rate_id_for_cr53_ec, ierr)
                  return
               
               case(ircr54ec_to_v53_n)
                  call do_ec_nn_reaction(
     >               i, icr54, iv53, 1, weak_rate_id_for_cr54_ec, ierr)
                  return

               case(ircr54_ecnp_ti52)
                  call do_ecnp_reaction(i, icr54, iv54, iti52, weak_rate_id_for_cr54_ec, 
     >               weak_rate_id_for_cr53_ec, weak_rate_id_for_v53_ec, ierr)
                  return

               end select
            
            case(55) ! cr55
               select case(ir)
               
               case(ircr55ec_to_v53_2n)
                  call do_ec_nn_reaction(
     >               i, icr55, iv53, 2, weak_rate_id_for_cr55_ec, ierr)
                  return
               
               case(ircr55ec_to_ti52_3n) ! cr55 + 2 e- => ti52 + 3n
                  call do_ec_nn_reaction(
     >               i, icr55, iti52, 3, weak_rate_id_for_cr55_ec, ierr)
                  return

               end select
            
            case(56) ! cr56
               select case(ir)

               case(ircr54ng_to_cr56) ! cr54(neut,g)cr55(neut,g)cr56
                  call do_neutg_neutg_reaction(i, icr54, icr56, n% d_rcr54neut_to_cr56_dneut)
                  return
               case(ircr56gn_to_cr54) ! cr56(g,neut)cr55(g,neut)cr54
                  call do_gneut_gneut_reaction(i, icr54, icr56, n% d_rcr56g_to_cr54_dneut)
                  return

               case(ircr54ap_to_cr56) ! cr54(a,p)mn57(g,p)cr56
                  call do_aprot_gprot_reaction(i, icr54, icr56, n% d_rcr54aprot_to_cr56_dprot)
                  return
               case(ircr56p_to_cr54) ! cr56(p,g)mn57(p,a)cr54
                  call do_protg_prota_reaction(i, icr54, icr56, n% d_rcr56prot_to_cr54_dprot)
                  return

               case(irti52ap_to_cr56) ! ti52(a,prot)v55(prot,g)cr56
                  call do_aprot_protg_reaction(i, iti52, icr56, n% d_rti52aprot_to_cr56_dprot)
                  return                              
               case(ircr56gp_to_ti52) ! cr56(g,prot)v55(prot,a)ti52
                  call do_gprot_prota_reaction(i, iti52, icr56, n% d_rcr56gprot_to_ti52_dprot)
                  return
      
               case(irti52an_to_cr56) ! ti52(a,n)cr55(n,g)cr56
                  call do_aneut_neutg_reaction(i, iti52, icr56, n% d_rti52aneut_to_cr56_dneut)
                  return         
               case(ircr56gn_to_ti52) ! cr56(g,n)cr55(n,a)ti52
                  call do_gneut_neuta_reaction(i, iti52, icr56, n% d_rcr56gneut_to_ti52_dneut)
                  return
         
               case(irti54p_to_cr56) ! ti54(prot,g)v55(prot,g)cr56
                  call do_protg_protg_reaction(i, iti54, icr56, n% d_rti54prot_to_cr56_dprot)
                  return                              
               case(ircr56gp_to_ti54) ! cr56(g,prot)v55(g,prot)ti54
                  call do_gprot_gprot_reaction(i, iti54, icr56, n% d_rcr56gprot_to_ti54_dprot)
                  return

               case(ircr56ec_to_ti52_4n) ! cr56 + 2 e- => ti52 + 4n
                  call do_ec_nn_reaction(
     >               i, icr56, iti52, 4, weak_rate_id_for_cr56_ec, ierr)
                  return
               
               case(ircr56ec_to_v53_3n)
                  call do_ec_nn_reaction(
     >               i, icr56, iv53, 3,weak_rate_id_for_cr56_ec, ierr)
                  return
               
               case(ircr56ec_to_v55_n) ! cr56 + 2 e- => v55 + 1n
                  call do_ec_nn_reaction(
     >               i, icr56, iv55, 1, weak_rate_id_for_cr56_ec, ierr)
                  return

               case(ircr56_ecnp_ti52)
                  call do_ecnp_nn_reaction(
     >               i, icr56, iv56, iti52, 2, weak_rate_id_for_cr56_ec, 
     >               weak_rate_id_for_cr55_ec, weak_rate_id_for_v55_ec, temp, ierr)
                  return

               end select
      
            end select
         
         case(25) ! mn
            select case(ir)
            
            case(irmn57ec_to_cr56_n)
               call do_ec_nn_reaction(
     >            i, imn57, icr56, 1, weak_rate_id_for_mn57_ec, ierr)
               return

            case(irmn58ec_to_cr56_2n) ! mn58 + 2 e- => cr56 + 2n
               call do_ec_nn_reaction(
     >            i, imn58, icr56, 2, weak_rate_id_for_mn58_ec, ierr)
               return
               
            case(irmn59ec_to_cr56_3n) ! mn59 + 2 e- => cr56 + 3n
               call do_ec_nn_reaction(
     >            i, imn59, icr56, 3, weak_rate_id_for_mn59_ec, ierr)
               return
      
            end select
         
         case(26) ! fe
         
            select case(Z_plus_N_for_max_Z)

            case(52) ! fe52
               select case(ir)
      
               case(ircr50an_to_fe52) ! cr50(a,n)fe53(g,n)fe52
                  call do_aneut_gneut_reaction(i, icr50, ife52, n% d_rcr50aneut_to_fe52_dneut)
                  return         
               case(irfe52ng_to_cr50) ! fe52(n,g)fe53(n,a)cr50
                  call do_neutg_neuta_reaction(i, icr50, ife52, n% d_rfe52neut_to_cr50_dneut)
                  return
      
               case(ircr52pn_to_fe52) ! cr52(p,n)mn52(p,n)fe52
                  call do_pn_pn_reaction(i, icr52, ife52,
     >               n% d_rcr52pn_to_fe52_dprot, n% d_rcr52pn_to_fe52_dneut)
                  return         
               case(irfe52np_to_cr52) ! fe52(n,p)mn52(n,p)cr52
                  call do_np_np_reaction(i, icr52, ife52, 
     >               n% d_rfe52np_to_cr52_dprot, n% d_rfe52np_to_cr52_dneut)
                  return

               case(irfe52ec_to_cr52) ! fe52 + 2 e- => cr52
                  call do_2ec_reaction(i, ife52, icr52, weak_rate_id_for_fe52_ec, ierr)
                  return
               case(ircr52ee_to_fe52) ! cr52 => fe52 + 2 e-
                  call do_2ee_reaction(i, icr52, ife52, weak_rate_id_for_cr52_ee, ierr)
                  return

               case(irfe52_ecnp_cr50)
                  call do_ecnp_reaction(i, ife52, imn52, icr50, weak_rate_id_for_fe52_ec,  
     >               0, weak_rate_id_for_mn51_ec, ierr)
                  return

               end select
            
            case(54) ! fe54
               select case(ir)
         
               case(irfe52aprot_to_fe54) ! fe52(a,prot)co55(g,prot)fe54
                  call do_aprot_gprot_reaction(i, ife52, ife54, n% d_rfe52aprot_to_fe54_dprot)
                  return                  
               case(irfe54prot_to_fe52) ! fe54(prot,g)co55(prot,a)fe52
                  call do_protg_prota_reaction(i, ife52, ife54, n% d_rfe54prot_to_fe52_dprot)
                  return
      
               case(ircr54pn_to_fe54) ! cr54(p,n)mn54(p,n)fe54
                  call do_pn_pn_reaction(i, icr54, ife54,
     >               n% d_rcr54pn_to_fe54_dprot, n% d_rcr54pn_to_fe54_dneut)
                  return         
               case(irfe54np_to_cr54) ! fe54(n,p)mn54(n,p)cr54
                  call do_np_np_reaction(i, icr54, ife54, 
     >               n% d_rfe54np_to_cr54_dprot, n% d_rfe54np_to_cr54_dneut)
                  return
      
               case(ircr50an_to_fe54) ! cr50(a,n)fe53(n,g)fe54
                  call do_aneut_neutg_reaction(i, icr50, ife54, n% d_rcr50aneut_to_fe54_dneut)
                  return         
               case(irfe54gn_to_cr50) ! fe54(g,n)fe53(n,a)cr50
                  call do_gneut_neuta_reaction(i, icr50, ife54, n% d_rfe54gneut_to_cr50_dneut)
                  return
      
               case(ircr52an_to_fe54) ! cr52(a,n)fe55(g,n)fe54
                  call do_aneut_gneut_reaction(i, icr52, ife54, n% d_rcr52aneut_to_fe54_dneut)
                  return         
               case(irfe54ng_to_cr52) ! fe54(n,g)fe55(n,a)cr52
                  call do_neutg_neuta_reaction(i, icr52, ife54, n% d_rfe54neut_to_cr52_dneut)
                  return

               case(irfe52neut_to_fe54) ! fe52(neut,g)fe53(neut,g)fe54               
                  call do_neutg_neutg_reaction(i, ife52, ife54, n% d_rfe52neut_to_fe54_dneut)
                  return                 
               case(irfe54g_to_fe52) ! fe54(g,neut)fe53(g,neut)fe52
                  call do_gneut_gneut_reaction(i, ife52, ife54, n% d_rfe54g_to_fe52_dneut)
                  return

               case(ircr50ap_to_fe54) ! cr50(a,prot)mn53(prot,g)fe54
                  call do_aprot_protg_reaction(i, icr50, ife54, n% d_rcr50aprot_to_fe54_dprot)
                  return                              
               case(irfe54gp_to_cr50) ! fe54(g,prot)mn53(prot,a)cr50
                  call do_gprot_prota_reaction(i, icr50, ife54, n% d_rfe54gprot_to_cr50_dprot)
                  return
         
               case(ircr52p_to_fe54) ! cr52(prot,g)mn53(prot,g)fe54
                  call do_protg_protg_reaction(i, icr52, ife54, n% d_rcr52prot_to_fe54_dprot)
                  return                              
               case(irfe54gp_to_cr52) ! fe54(g,prot)mn53(g,prot)cr52
                  call do_gprot_gprot_reaction(i, icr52, ife54, n% d_rfe54gprot_to_cr52_dprot)
                  return

               case(irfe54ec_to_cr54) ! fe54 + 2 e- => cr54
                  call do_2ec_reaction(i, ife54, icr54, weak_rate_id_for_fe54_ec, ierr)
                  return
               case(ircr54ee_to_fe54) ! cr54 => fe54 + 2 e-
                  call do_2ee_reaction(i, icr54, ife54, weak_rate_id_for_cr54_ee, ierr)
                  return

               case(irfe54_ecnp_cr52)
                  call do_ecnp_reaction(i, ife54, imn54, icr52, weak_rate_id_for_fe54_ec, 
     >               weak_rate_id_for_fe53_ec, weak_rate_id_for_mn53_ec, ierr)
                  return

               end select
            
            case(56) ! fe56
               select case(ir)
      
               case(irfe54aprot_to_fe56) ! fe54(a,prot)co57(g,prot)fe56
                  call do_aprot_gprot_reaction(i, ife54, ife56, n% d_rfe54aprot_to_fe56_dprot)
                  return                  
               case(irfe56prot_to_fe54) ! fe56(prot,g)co57(prot,a)fe54
                  call do_protg_prota_reaction(i, ife54, ife56, n% d_rfe56prot_to_fe54_dprot)
                  return
      
               case(ircr54an_to_fe56) ! cr54(a,n)fe57(g,n)fe56
                  call do_aneut_gneut_reaction(i, icr54, ife56, n% d_rcr54aneut_to_fe56_dneut)
                  return         
               case(irfe56ng_to_cr54) ! fe56(n,g)fe57(n,a)cr54
                  call do_neutg_neuta_reaction(i, icr54, ife56, n% d_rfe56neut_to_cr54_dneut)
                  return
      
               case(ircr56pn_to_fe56) ! cr56(p,n)mn56(p,n)fe56
                  call do_pn_pn_reaction(i, icr56, ife56,
     >               n% d_rcr56pn_to_fe56_dprot, n% d_rcr56pn_to_fe56_dneut)
                  return         
               case(irfe56np_to_cr56) ! fe56(n,p)mn56(n,p)cr56
                  call do_np_np_reaction(i, icr56, ife56, 
     >               n% d_rfe56np_to_cr56_dprot, n% d_rfe56np_to_cr56_dneut)
                  return

               case(irfe54ng_to_fe56) ! fe54(neut,g)fe55(neut,g)fe56
                  call do_neutg_neutg_reaction(i, ife54, ife56, n% d_rfe54neut_to_fe56_dneut)
                  return                  
               case(irfe56gn_to_fe54) ! fe56(g,neut)fe55(g,neut)fe54
                  call do_gneut_gneut_reaction(i, ife54, ife56, n% d_rfe56g_to_fe54_dneut)
                  return

               case(ircr52ap_to_fe56) ! cr52(a,prot)mn55(prot,g)fe56
                  call do_aprot_protg_reaction(i, icr52, ife56, n% d_rcr52aprot_to_fe56_dprot)
                  return                              
               case(irfe56gp_to_cr52) ! fe56(g,prot)mn55(prot,a)cr52
                  call do_gprot_prota_reaction(i, icr52, ife56, n% d_rfe56gprot_to_cr52_dprot)
                  return
      
               case(ircr52an_to_fe56) ! cr52(a,n)fe55(n,g)fe56
                  call do_aneut_neutg_reaction(i, icr52, ife56, n% d_rcr52aneut_to_fe56_dneut)
                  return         
               case(irfe56gn_to_cr52) ! fe56(g,n)fe55(n,a)cr52
                  call do_gneut_neuta_reaction(i, icr52, ife56, n% d_rfe56gneut_to_cr52_dneut)
                  return
         
               case(ircr54p_to_fe56) ! cr54(prot,g)mn55(prot,g)fe56
                  call do_protg_protg_reaction(i, icr54, ife56, n% d_rcr54prot_to_fe56_dprot)
                  return                              
               case(irfe56gp_to_cr54) ! fe56(g,prot)mn55(g,prot)cr54
                  call do_gprot_gprot_reaction(i, icr54, ife56, n% d_rfe56gprot_to_cr54_dprot)
                  return

               case(irfe56ec_to_cr56) ! fe56 + 2 e- => cr56
                  call do_2ec_reaction(i, ife56, icr56, weak_rate_id_for_fe56_ec, ierr)
                  return
               case(ircr56ee_to_fe56) ! cr56 => fe56 + 2 e-
                  call do_2ee_reaction(i, icr56, ife56, weak_rate_id_for_cr56_ee, ierr)
                  return

               case(irfe56_ecnp_cr54)
                  call do_ecnp_reaction(i, ife56, imn56, icr54, weak_rate_id_for_fe56_ec,  
     >               weak_rate_id_for_fe55_ec, weak_rate_id_for_mn55_ec, ierr)
                  return

               case(irfe56ec_fake_to_cr56) ! fe56 + 2 e- => cr56
                  cr56 = itab(icr56)
                  fe56 = itab(ife56)
                  r = y(fe56)
                  ! NOTE: for this fake reaction, use ni56 ec rate
                  call eval_ni56_ec_rate(temp, den, r, ye, eta, 
     >               rate, d_rate_dlnT, d_rate_dlnRho, Q, Qneu, ierr)
                  if (ierr /= 0) return
                  if (temp < 3d9) return ! only at high T
                  n% rate_raw(i_rate,i) = rate
                  n% rate_raw(i_rate_dT,i) = 0d0 ! d_rate_dlnT/temp
                  n% rate_raw(i_rate_dRho,i) = 0d0 ! d_rate_dlnRho/den
                  if (temp < 4d9) then
                     n% rate_raw(i_rate_dT,i) = 
     >                  n% rate_raw(i_rate,i)/1d9 + n% rate_raw(i_rate_dT,i)*(temp - 3d9)/1d9
                     n% rate_raw(i_rate,i) = n% rate_raw(i_rate,i)*(temp - 3d9)/1d9
                  end if                  
                  n% rate_raw(:,i) = n% rate_raw(:,i)*rate_factors(i)
                  n% rate_screened(:,i) = n% rate_raw(:,i)
                  dr1 = 1d0
                  Q = 2*Q; Qneu = 2*Qneu ! 2 ec's, so double the fe56 ec Q and Qneu
                  call do_one_one_neu(n, i, 1d0, fe56, 
     >               r, 1d0, cr56, fe56, dr1, Q, Qneu, deriv_flgs, category_factors, symbolic)
                  if (g% doing_derivs_timing) call done_select
                  return

               end select
            
            case(57) ! fe57
               select case(ir)

               case(irfe57ec_to_cr56_n) ! fe57 + 2 e- => cr56 + n
                  call do_ec_nn_reaction(
     >               i, ife57, icr56, 1, weak_rate_id_for_fe57_ec, ierr)
                  return            

               end select
            
            case(58) ! fe58
               select case(ir)
      
               case(irfe56aprot_to_fe58) ! fe56(a,prot)co59(g,prot)fe58
                  call do_aprot_gprot_reaction(i, ife56, ife58, n% d_rfe56aprot_to_fe58_dprot)
                  return                  
               case(irfe58prot_to_fe56) ! fe58(prot,g)co59(prot,a)fe56
                  call do_protg_prota_reaction(i, ife56, ife58, n% d_rfe58prot_to_fe56_dprot)
                  return
      
               case(ircr54an_to_fe58) ! cr54(a,n)fe57(n,g)fe58
                  call do_aneut_neutg_reaction(i, icr54, ife58, n% d_rcr54aneut_to_fe58_dneut)
                  return         
               case(irfe58gn_to_cr54) ! fe58(g,n)fe57(n,a)cr54
                  call do_gneut_neuta_reaction(i, icr54, ife58, n% d_rfe58gneut_to_cr54_dneut)
                  return

               case(ircr54ap_to_fe58) ! cr54(a,prot)mn57(prot,g)fe58
                  call do_aprot_protg_reaction(i, icr54, ife58, n% d_rcr54aprot_to_fe58_dprot)
                  return                              
               case(irfe58gp_to_cr54) ! fe58(g,prot)mn57(prot,a)cr54
                  call do_gprot_prota_reaction(i, icr54, ife58, n% d_rfe58gprot_to_cr54_dprot)
                  return
         
               case(ircr56p_to_fe58) ! cr56(prot,g)mn57(prot,g)fe58
                  call do_protg_protg_reaction(i, icr56, ife58, n% d_rcr56prot_to_fe58_dprot)
                  return                              
               case(irfe58gp_to_cr56) ! fe58(g,prot)mn57(g,prot)cr56
                  call do_gprot_gprot_reaction(i, icr56, ife58, n% d_rfe58gprot_to_cr56_dprot)
                  return
      
               case(ircr56an_to_fe58) ! cr56(a,n)fe59(g,n)fe58
                  call do_aneut_gneut_reaction(i, icr56, ife58, n% d_rcr56aneut_to_fe58_dneut)
                  return         
               case(irfe58ng_to_cr56) ! fe58(n,g)fe59(n,a)cr56
                  call do_neutg_neuta_reaction(i, icr56, ife58, n% d_rfe58neut_to_cr56_dneut)
                  return

               case(irfe56ng_to_fe58) ! fe56(neut,g)fe57(neut,g)fe58
                  call do_neutg_neutg_reaction(i, ife56, ife58, n% d_rfe56neut_to_fe58_dneut)
                  return                  
               case(irfe58gn_to_fe56) ! fe58(g,neut)fe57(g,neut)fe56
                  call do_gneut_gneut_reaction(i, ife56, ife58, n% d_rfe58g_to_fe56_dneut)
                  return

               case(irfe58ec_to_mn57_n)
                  call do_ec_nn_reaction(
     >               i, ife58, imn57, 1, weak_rate_id_for_fe58_ec, ierr)
                  return

               case(irfe58ec_to_cr56_2n) ! fe58 + 2 e- => cr56 + 2n
                  call do_ec_nn_reaction(
     >               i, ife58, icr56, 2, weak_rate_id_for_fe58_ec, ierr)
                  return
 
               case(irfe58_ecnp_cr56)
                  call do_ecnp_reaction(i, ife58, imn58, icr56, weak_rate_id_for_fe58_ec,  
     >               weak_rate_id_for_fe57_ec, weak_rate_id_for_mn57_ec, ierr)
                  return

               end select
            
            case(59) ! fe59
               select case(ir)

               case(irfe59ec_to_mn57_2n)
                  call do_ec_nn_reaction(
     >               i, ife59, imn57, 2, weak_rate_id_for_fe59_ec, ierr)
                  return

               case(irfe59ec_to_cr56_3n) ! fe59 + 2 e- => cr56 + 3n
                  call do_ec_nn_reaction(
     >               i, ife59, icr56, 3, weak_rate_id_for_fe59_ec, ierr)
                  return

               end select
            
            case(60) ! fe60
               select case(ir)
      
               case(irfe58aprot_to_fe60) ! fe58(a,prot)co61(g,prot)fe60
                  call do_aprot_gprot_reaction(i, ife58, ife60, n% d_rfe58aprot_to_fe60_dprot)
                  return                  
               case(irfe60prot_to_fe58) ! fe60(prot,g)co61(prot,a)fe58
                  call do_protg_prota_reaction(i, ife58, ife60, n% d_rfe60prot_to_fe58_dprot)
                  return
      
               case(ircr56an_to_fe60) ! cr56(a,n)fe59(n,g)fe60
                  call do_aneut_neutg_reaction(i, icr56, ife60, n% d_rcr56aneut_to_fe60_dneut)
                  return         
               case(irfe60gn_to_cr56) ! fe60(g,n)fe59(n,a)cr56
                  call do_gneut_neuta_reaction(i, icr56, ife60, n% d_rfe60gneut_to_cr56_dneut)
                  return

               case(irfe58ng_to_fe60) ! fe58(neut,g)fe59(neut,g)fe60
                  call do_neutg_neutg_reaction(i, ife58, ife60, n% d_rfe58neut_to_fe60_dneut)
                  return
               case(irfe60gn_to_fe58) ! fe60(g,neut)fe59(g,neut)fe58
                  call do_gneut_gneut_reaction(i, ife58, ife60, n% d_rfe60g_to_fe58_dneut)
                  return
                  
               case(ircr56ap_to_fe60) ! cr56(a,prot)mn59(prot,g)fe60
                  call do_aprot_protg_reaction(i, icr56, ife60, n% d_rcr56aprot_to_fe60_dprot)
                  return 
               case(irfe60gp_to_cr56) ! fe60(g,prot)mn59(prot,a)cr56
                  call do_gprot_prota_reaction(i, icr56, ife60, n% d_rfe60gprot_to_cr56_dprot)
                  return                  

               case(irfe60ec_to_mn57_3n)
                  call do_ec_nn_reaction(
     >               i, ife60, imn57, 3, weak_rate_id_for_fe60_ec, ierr)
                  return
         
               case(irfe60ec_to_cr48_12n) ! fe60 + 2 e- => cr48 + 12n
                  call do_ec_nn_reaction(
     >               i, ife60, icr48, 12, weak_rate_id_for_fe60_ec, ierr)
                  return
               
               case(irfe60ec_to_mn59_n) ! fe60 + 2 e- => mn59 + 1n
                  call do_ec_nn_reaction(
     >               i, ife60, imn59, 1, weak_rate_id_for_fe60_ec, ierr)
                  return

               case(irfe60ec_to_cr56_4n) ! fe60 + 2 e- => cr56 + 4n
                  call do_ec_nn_reaction(
     >               i, ife60, icr56, 4, weak_rate_id_for_fe60_ec, ierr)
                  return

               case(irfe60_ecnp_cr56)
                  call do_ecnp_nn_reaction(
     >               i, ife60, imn60, icr56, 2, weak_rate_id_for_fe60_ec, 
     >               weak_rate_id_for_fe59_ec, weak_rate_id_for_mn59_ec, temp, ierr)
                  return

               end select
            
            case(62) ! fe62
               select case(ir)

               case(irfe60ng_to_fe62) ! fe60(neut,g)fe61(neut,g)fe62
                  call do_neutg_neutg_reaction(i, ife60, ife62, n% d_rfe60neut_to_fe62_dneut)
                  return
               
               case(irfe62gn_to_fe60) ! fe62(g,neut)fe61(g,neut)fe60
                  call do_gneut_gneut_reaction(i, ife60, ife62, n% d_rfe62g_to_fe60_dneut)
                  return

               end select
      
            end select
         
         case(27) ! co
            select case(ir)
        
            case(irco61ec_to_fe60_n)
               call do_ec_nn_reaction(
     >            i, ico61, ife60, 1, weak_rate_id_for_co61_ec, ierr)
               return

            case(irco62ec_to_fe60_2n) ! co62 + 2 e- => fe60 + 2n
               call do_ec_nn_reaction(
     >            i, ico62, ife60, 2, weak_rate_id_for_co62_ec, ierr)
               return
               
            case(irco63ec_to_fe60_3n) ! co63 + 2 e- => fe60 + 3n
               call do_ec_nn_reaction(
     >            i, ico63, ife60, 3, weak_rate_id_for_co63_ec, ierr)
               return
               
            case(irco64ec_to_fe60_4n) ! co64 + 2 e- => fe60 + 4n
               call do_ec_nn_reaction(
     >            i, ico64, ife60, 4, weak_rate_id_for_co64_ec, ierr)
               return
      
            end select
         
         case(28) ! ni
         
            select case(Z_plus_N_for_max_Z)
            
            case(56) ! ni56
               select case(ir)
         
               case(irfe52aprot_to_ni56) ! fe52(a,prot)co55(prot,g)ni56
                  call do_aprot_protg_reaction(i, ife52, ini56, n% d_rfe52aprot_to_ni56_dprot)
                  return                            
               case(irni56gprot_to_fe52) ! ni56(g,prot)co55(prot,a)fe52
                  call do_gprot_prota_reaction(i, ife52, ini56, n% d_rni56gprot_to_fe52_dprot)
                  return

               case(irfe54an_to_ni56) ! fe54(a,n)ni57(g,n)ni56
                  call do_aneut_gneut_reaction(i, ife54, ini56, n% d_rfe54aneut_to_ni56_dneut)
                  return         
               case(irni56ng_to_fe54) ! ni56(n,g)ni57(n,a)fe54
                  call do_neutg_neuta_reaction(i, ife54, ini56, n% d_rni56neut_to_fe54_dneut)
                  return
         
               case(irfe54prot_to_ni56) ! fe54(prot,g)co55(prot,g)ni56
                  call do_protg_protg_reaction(i, ife54, ini56, n% d_rfe54prot_to_ni56_dprot)
                  return                              
               case(irni56gprot_to_fe54) ! ni56(g,prot)co55(g,prot)fe54
                  call do_gprot_gprot_reaction(i, ife54, ini56, n% d_rni56gprot_to_fe54_dprot)
                  return
      
               case(irfe56pn_to_ni56) ! fe56(p,n)co56(p,n)ni56
                  call do_pn_pn_reaction(i, ife56, ini56,
     >               n% d_rfe56pn_to_ni56_dprot, n% d_rfe56pn_to_ni56_dneut)
                  return         
               case(irni56np_to_fe56) ! ni56(n,p)co56(n,p)fe56
                  call do_np_np_reaction(i, ife56, ini56, 
     >               n% d_rni56np_to_fe56_dprot, n% d_rni56np_to_fe56_dneut)
                  return

               case(irni56ec_to_fe56) ! ni56 + 2 e- => fe56
                  call do_2ec_reaction(i, ini56, ife56, weak_rate_id_for_ni56_ec, ierr)               
                  return
               case(irfe56ee_to_ni56) ! fe56 => ni56 + 2 e-
                  call do_2ee_reaction(i, ife56, ini56, weak_rate_id_for_fe56_ee, ierr)
                  return

               case(irni56_ecnp_fe54)
                  call do_ecnp_reaction(i, ini56, ico56, ife54, weak_rate_id_for_ni56_ec, 
     >               weak_rate_id_for_ni55_ec, weak_rate_id_for_co55_ec, ierr)               
                  return

               case(irni56ec_to_fe54) ! ni56 + 2 e- => 56/54 fe54 in place of fe56
                  fe54 = itab(ife54)
                  ni56 = itab(ini56)
                  r = y(ni56)
                  call eval_ni56_ec_rate(temp, den, r, ye, eta, 
     >               rate, d_rate_dlnT, d_rate_dlnRho, Q, Qneu, ierr)
                  if (ierr /= 0) return
                  n% rate_raw(i_rate,i) = rate
                  n% rate_raw(i_rate_dT,i) = 0d0 ! d_rate_dlnT/temp
                  n% rate_raw(i_rate_dRho,i) = 0d0 ! d_rate_dlnRho/den
                  n% rate_raw(:,i) = n% rate_raw(:,i)*rate_factors(i)
                  n% rate_screened(:,i) = n% rate_raw(:,i)
                  dr1 = 1d0
                  Q = 2*Q; Qneu = 2*Qneu ! 2 ec's, so double the ni56 ec Q and Qneu
                  call do_one_one_neu(n, i, 1d0, ni56, 
     >               r, 56d0/54d0, fe54, ni56, dr1, Q, Qneu, deriv_flgs, category_factors, symbolic)
                  if (g% doing_derivs_timing) call done_select
                  return

               end select
            
            case(58) ! ni58
               select case(ir)
         
               case(irfe54aprot_to_ni58) ! fe54(a,prot)co57(prot,g)ni58
                  call do_aprot_protg_reaction(i, ife54, ini58, n% d_rfe54aprot_to_ni58_dprot)
                  return            
               case(irni58gprot_to_fe54) ! ni58(g,prot)co57(prot,a)fe54
                  call do_gprot_prota_reaction(i, ife54, ini58, n% d_rni58gprot_to_fe54_dprot)
                  return
      
               case(irfe54an_to_ni58) ! fe54(a,n)ni57(n,g)ni58
                  call do_aneut_neutg_reaction(i, ife54, ini58, n% d_rfe54aneut_to_ni58_dneut)
                  return         
               case(irni58gn_to_fe54) ! ni58(g,n)ni57(n,a)fe54
                  call do_gneut_neuta_reaction(i, ife54, ini58, n% d_rni58gneut_to_fe54_dneut)
                  return
         
               case(irfe56prot_to_ni58) ! fe56(prot,g)co57(prot,g)ni58
                  call do_protg_protg_reaction(i, ife56, ini58, n% d_rfe56prot_to_ni58_dprot)
                  return            
               case(irni58gprot_to_fe56) ! ni58(g,prot)co57(g,prot)fe56
                  call do_gprot_gprot_reaction(i, ife56, ini58, n% d_rni58gprot_to_fe56_dprot)
                  return
      
               case(irfe58pn_to_ni58) ! fe58(p,n)co58(p,n)ni58
                  call do_pn_pn_reaction(i, ife58, ini58,
     >               n% d_rfe58pn_to_ni58_dprot, n% d_rfe58pn_to_ni58_dneut)
                  return         
               case(irni58np_to_fe58) ! ni58(n,p)co58(n,p)fe58
                  call do_np_np_reaction(i, ife58, ini58, 
     >               n% d_rni58np_to_fe58_dprot, n% d_rni58np_to_fe58_dneut)
                  return
      
               case(irfe56an_to_ni58) ! fe56(a,n)ni59(g,n)ni58
                  call do_aneut_gneut_reaction(i, ife56, ini58, n% d_rfe56aneut_to_ni58_dneut)
                  return         
               case(irni58ng_to_fe56) ! ni58(n,g)ni59(n,a)fe56
                  call do_neutg_neuta_reaction(i, ife56, ini58, n% d_rni58neut_to_fe56_dneut)
                  return
      
               case(irni56aprot_to_ni58) ! ni56(a,prot)cu59(g,prot)ni58
                  call do_aprot_gprot_reaction(i, ini56, ini58, n% d_rni56aprot_to_ni58_dprot)
                  return         
               case(irni58prot_to_ni56) ! ni58(prot,g)cu59(prot,a)ni56
                  call do_protg_prota_reaction(i, ini56, ini58, n% d_rni58prot_to_ni56_dprot)
                  return

               case(irni56ng_to_ni58) ! ni56(neut,g)ni57(neut,g)ni58
                  call do_neutg_neutg_reaction(i, ini56, ini58, n% d_rni56neut_to_ni58_dneut)
                  return            
               case(irni58gn_to_ni56) ! ni58(g,neut)ni57(g,neut)ni56
                  call do_gneut_gneut_reaction(i, ini56, ini58, n% d_rni58g_to_ni56_dneut)
                  return

               case(irni58ec_to_fe58) ! ni58 + 2 e- => fe58
                  call do_2ec_reaction(i, ini58, ife58, weak_rate_id_for_ni58_ec, ierr)
                  return

               case(irni58_ecnp_fe56)
                  call do_ecnp_reaction(i, ini58, ico58, ife56, weak_rate_id_for_ni58_ec, 
     >               weak_rate_id_for_ni57_ec, weak_rate_id_for_co57_ec, ierr)
                  return

               end select
            
            case(60) ! ni60
               select case(ir)
         
               case(irfe56aprot_to_ni60) ! fe56(a,prot)co59(prot,g)ni60
                  call do_aprot_protg_reaction(i, ife56, ini60, n% d_rfe56aprot_to_ni60_dprot)
                  return            
               case(irni60gprot_to_fe56) ! ni60(g,prot)co59(prot,a)fe56
                  call do_gprot_prota_reaction(i, ife56, ini60, n% d_rni60gprot_to_fe56_dprot)
                  return
      
               case(irfe56an_to_ni60) ! fe56(a,n)ni59(n,g)ni60
                  call do_aneut_neutg_reaction(i, ife56, ini60, n% d_rfe56aneut_to_ni60_dneut)
                  return         
               case(irni60gn_to_fe56) ! ni60(g,n)ni59(n,a)fe56
                  call do_gneut_neuta_reaction(i, ife56, ini60, n% d_rni60gneut_to_fe56_dneut)
                  return
         
               case(irfe58prot_to_ni60) ! fe58(prot,g)co59(prot,g)ni60
                  call do_protg_protg_reaction(i, ife58, ini60, n% d_rfe58prot_to_ni60_dprot)
                  return           
               case(irni60gprot_to_fe58) ! ni60(g,prot)co59(g,prot)fe58
                  call do_gprot_gprot_reaction(i, ife58, ini60, n% d_rni60gprot_to_fe58_dprot)
                  return
      
               case(irni58aprot_to_ni60) ! ni58(a,prot)cu61(g,prot)ni60
                  call do_aprot_gprot_reaction(i, ini58, ini60, n% d_rni58aprot_to_ni60_dprot)
                  return         
               case(irni60prot_to_ni58) ! ni60(prot,g)cu61(prot,a)ni58
                  call do_protg_prota_reaction(i, ini58, ini60, n% d_rni60prot_to_ni58_dprot)
                  return
      
               case(irfe58an_to_ni60) ! fe58(a,n)ni61(g,n)ni60
                  call do_aneut_gneut_reaction(i, ife58, ini60, n% d_rfe58aneut_to_ni60_dneut)
                  return         
               case(irni60ng_to_fe58) ! ni60(n,g)ni61(n,a)fe58
                  call do_neutg_neuta_reaction(i, ife58, ini60, n% d_rni60neut_to_fe58_dneut)
                  return

               case(irni58ng_to_ni60) ! ni58(neut,g)ni59(neut,g)ni60
                  call do_neutg_neutg_reaction(i, ini58, ini60, n% d_rni58neut_to_ni60_dneut)
                  return           
               case(irni60gn_to_ni58) ! ni60(g,neut)ni59(g,neut)ni58
                  call do_gneut_gneut_reaction(i, ini58, ini60, n% d_rni60g_to_ni58_dneut)
                  return

               case(irni60ec_to_fe60) ! ni60 + 2 e- => fe60
                  call do_2ec_reaction(i, ini60, ife60, weak_rate_id_for_ni60_ec, ierr)
                  return

               case(irni60_ecnp_fe58)
                  call do_ecnp_reaction(i, ini60, ico60, ife58, weak_rate_id_for_ni60_ec, 
     >               weak_rate_id_for_ni59_ec, weak_rate_id_for_co59_ec, ierr)
                  return

               end select
            
            case(62) ! ni62
               select case(ir)
         
               case(irfe58aprot_to_ni62) ! fe58(a,prot)co61(prot,g)ni62
                  call do_aprot_protg_reaction(i, ife58, ini62, n% d_rfe58aprot_to_ni62_dprot)
                  return            
               case(irni62gprot_to_fe58) ! ni62(g,prot)co61(prot,a)fe58
                  call do_gprot_prota_reaction(i, ife58, ini62, n% d_rni62gprot_to_fe58_dprot)
                  return
      
               case(irfe58an_to_ni62) ! fe58(a,n)ni61(n,g)ni62
                  call do_aneut_neutg_reaction(i, ife58, ini62, n% d_rfe58aneut_to_ni62_dneut)
                  return         
               case(irni62gn_to_fe58) ! ni62(g,n)ni61(n,a)fe58
                  call do_gneut_neuta_reaction(i, ife58, ini62, n% d_rni62gneut_to_fe58_dneut)
                  return
         
               case(irfe60prot_to_ni62) ! fe60(prot,g)co61(prot,g)ni62
                  call do_protg_protg_reaction(i, ife60, ini62, n% d_rfe60prot_to_ni62_dprot)
                  return            
               case(irni62gprot_to_fe60) ! ni62(g,prot)co61(g,prot)fe60
                  call do_gprot_gprot_reaction(i, ife60, ini62, n% d_rni62gprot_to_fe60_dprot)
                  return
      
               case(irni60aprot_to_ni62) ! ni60(a,prot)cu63(g,prot)ni62
                  call do_aprot_gprot_reaction(i, ini60, ini62, n% d_rni60aprot_to_ni62_dprot)
                  return         
               case(irni62prot_to_ni60) ! ni62(prot,g)cu63(prot,a)ni60
                  call do_protg_prota_reaction(i, ini60, ini62, n% d_rni62prot_to_ni60_dprot)
                  return

               case(irni60ng_to_ni62) ! ni60(neut,g)ni61(neut,g)ni62
                  call do_neutg_neutg_reaction(i, ini60, ini62, n% d_rni60neut_to_ni62_dneut)
                  return            
               case(irni62gn_to_ni60) ! ni62(g,neut)ni61(g,neut)ni60
                  call do_gneut_gneut_reaction(i, ini60, ini62, n% d_rni62g_to_ni60_dneut)
                  return
         
               case(irni62ec_to_fe60_2n) ! ni62 + 2 e- => fe60 + 2n
                  call do_ec_nn_reaction(
     >               i, ini62, ife60, 2, weak_rate_id_for_ni62_ec, ierr)
                  return

               case(irni62ec_to_fe62) ! ni62 + 2 e- => fe62
                  call do_2ec_reaction(i, ini62, ife62, weak_rate_id_for_ni62_ec, ierr)
                  return
               
               case(irni62ec_to_co61_n)
                  call do_ec_nn_reaction(
     >               i, ini62, ico61, 1, weak_rate_id_for_ni62_ec, ierr)
                  return

               case(irni62_ecnp_fe60)
                  call do_ecnp_reaction(i, ini62, ico62, ife60, weak_rate_id_for_ni62_ec, 
     >               weak_rate_id_for_ni61_ec, weak_rate_id_for_co61_ec, ierr)
                  return

               end select
            
            case(63) ! ni63
               select case(ir)
         
               case(irni63ec_to_co61_2n)
                  call do_ec_nn_reaction(
     >               i, ini63, ico61, 2, weak_rate_id_for_ni63_ec, ierr)
                  return

               end select
            
            case(64) ! ni64
               select case(ir)
         
               case(irfe60aprot_to_ni64) ! fe60(a,prot)co63(prot,g)ni64
                  call do_aprot_protg_reaction(i, ife60, ini64, n% d_rfe60aprot_to_ni64_dprot)
                  return
               case(irni64gprot_to_fe60) ! ni64(g,prot)co63(prot,a)fe60
                  call do_gprot_prota_reaction(i, ife60, ini64, n% d_rni64gprot_to_fe60_dprot)
                  return
      
               case(irni62aprot_to_ni64) ! ni62(a,prot)cu65(g,prot)ni64
                  call do_aprot_gprot_reaction(i, ini62, ini64, n% d_rni62aprot_to_ni64_dprot)
                  return         
               case(irni64prot_to_ni62) ! ni64(prot,g)cu65(prot,a)ni62
                  call do_protg_prota_reaction(i, ini62, ini64, n% d_rni64prot_to_ni62_dprot)
                  return

               case(irni62ng_to_ni64) ! ni62(neut,g)ni63(neut,g)ni64
                  call do_neutg_neutg_reaction(i, ini62, ini64, n% d_rni62neut_to_ni64_dneut)
                  return            
               case(irni64gn_to_ni62) ! ni64(g,neut)ni63(g,neut)ni62
                  call do_gneut_gneut_reaction(i, ini62, ini64, n% d_rni64g_to_ni62_dneut)
                  return
               
               case(irni64ec_to_co61_3n)
                  call do_ec_nn_reaction(
     >               i, ini64, ico61, 3, weak_rate_id_for_ni64_ec, ierr)
                  return
         
               case(irni64ec_to_fe60_4n) ! ni64 + 2 e- => fe60 + 4n
                  call do_ec_nn_reaction(
     >               i, ini64, ife60, 4, weak_rate_id_for_ni64_ec, ierr)
                  return

               case(irni64_ecnp_fe60)
                  call do_ecnp_nn_reaction(
     >               i, ini64, ico64, ife60, 2, weak_rate_id_for_ni64_ec, 
     >               weak_rate_id_for_ni63_ec, weak_rate_id_for_co63_ec, temp, ierr)
                  return

               end select
            
            case(66) ! ni66
               select case(ir)
      
               case(irni64aprot_to_ni66) ! ni64(a,prot)cu67(g,prot)ni66
                  call do_aprot_gprot_reaction(i, ini64, ini66, n% d_rni64aprot_to_ni66_dprot)
                  return         
               case(irni66prot_to_ni64) ! ni66(prot,g)cu67(prot,a)ni64
                  call do_protg_prota_reaction(i, ini64, ini66, n% d_rni66prot_to_ni64_dprot)
                  return

               case(irni64ng_to_ni66) ! ni64(neut,g)ni65(neut,g)ni66
                  call do_neutg_neutg_reaction(i, ini64, ini66, n% d_rni64neut_to_ni66_dneut)
                  return            
               case(irni66gn_to_ni64) ! ni66(g,neut)ni65(g,neut)ni64
                  call do_gneut_gneut_reaction(i, ini64, ini66, n% d_rni66g_to_ni64_dneut)
                  return

               end select
      
            end select
         
         case(30) ! zn
            select case(ir)

            case(irni64aprot_to_zn68) ! ni64(a,prot)cu67(prot,g)zn68
               call do_aprot_protg_reaction(i, ini64, izn68, n% d_rni64aprot_to_zn68_dprot)
               return            
            case(irzn68gprot_to_ni64) ! zn68(g,prot)cu67(prot,a)ni64
               call do_gprot_prota_reaction(i, ini64, izn68, n% d_rzn68gprot_to_ni64_dprot)
               return

            case(irzn60ng_to_zn62) ! zn60(neut,g)zn61(neut,g)zn62
               call do_neutg_neutg_reaction(i, izn60, izn62, n% d_rzn60neut_to_zn62_dneut)
               return
            case(irzn62gn_to_zn60) ! zn62(g,neut)zn61(g,neut)zn60
               call do_gneut_gneut_reaction(i, izn60, izn62, n% d_rzn62g_to_zn60_dneut)
               return

            case(irzn62ng_to_zn64) ! zn62(neut,g)zn63(neut,g)zn64
               call do_neutg_neutg_reaction(i, izn62, izn64, n% d_rzn62neut_to_zn64_dneut)
               return
            case(irzn64gn_to_zn62) ! zn64(g,neut)zn63(g,neut)zn62
               call do_gneut_gneut_reaction(i, izn62, izn64, n% d_rzn64g_to_zn62_dneut)
               return

            case(irzn64ng_to_zn66) ! zn64(neut,g)zn65(neut,g)zn66
               call do_neutg_neutg_reaction(i, izn64, izn66, n% d_rzn64neut_to_zn66_dneut)
               return
            case(irzn66gn_to_zn64) ! zn66(g,neut)zn65(g,neut)zn64
               call do_gneut_gneut_reaction(i, izn64, izn66, n% d_rzn66g_to_zn64_dneut)
               return

            case(irzn66ng_to_zn68) ! zn66(neut,g)zn67(neut,g)zn68
               call do_neutg_neutg_reaction(i, izn66, izn68, n% d_rzn66neut_to_zn68_dneut)
               return
            case(irzn68gn_to_zn66) ! zn68(g,neut)zn67(g,neut)zn66
               call do_gneut_gneut_reaction(i, izn66, izn68, n% d_rzn68g_to_zn66_dneut)
               return

            case(irzn68ng_to_zn70) ! zn68(neut,g)zn69(neut,g)zn70
               call do_neutg_neutg_reaction(i, izn68, izn70, n% d_rzn68neut_to_zn70_dneut)
               return
            case(irzn70gn_to_zn68) ! zn70(g,neut)zn69(g,neut)zn68
               call do_gneut_gneut_reaction(i, izn68, izn70, n% d_rzn70g_to_zn68_dneut)
               return
      
            end select
         
         case(32) ! ge
            select case(ir)
         
            case(irzn68aprot_to_ge72) ! zn68(a,prot)ga71(prot,g)ge72
               call do_aprot_protg_reaction(i, izn68, ige72, n% d_rzn68aprot_to_ge72_dprot)
               return            
            case(irge72gprot_to_zn68) ! ge72(g,prot)ga71(prot,a)zn68
               call do_gprot_prota_reaction(i, izn68, ige72, n% d_rge72gprot_to_zn68_dprot)
               return

            case(irge70ng_to_ge72) ! ge70(neut,g)ge71(neut,g)ge72
               call do_neutg_neutg_reaction(i, ige70, ige72, n% d_rge70neut_to_ge72_dneut)
               return
            case(irge72gn_to_ge70) ! ge72(g,neut)ge71(g,neut)ge70
               call do_gneut_gneut_reaction(i, ige70, ige72, n% d_rge72g_to_ge70_dneut)
               return

            case(irge72ng_to_ge74) ! ge72(neut,g)ge73(neut,g)ge74
               call do_neutg_neutg_reaction(i, ige72, ige74, n% d_rge72neut_to_ge74_dneut)
               return
            case(irge74gn_to_ge72) ! ge74(g,neut)ge73(g,neut)ge72
               call do_gneut_gneut_reaction(i, ige72, ige74, n% d_rge74g_to_ge72_dneut)
               return

            case(irge74ng_to_ge76) ! ge74(neut,g)ge75(neut,g)ge76
               call do_neutg_neutg_reaction(i, ige74, ige76, n% d_rge74neut_to_ge76_dneut)
               return
            case(irge76gn_to_ge74) ! ge76(g,neut)ge75(g,neut)ge74
               call do_gneut_gneut_reaction(i, ige74, ige76, n% d_rge76g_to_ge74_dneut)
               return
      
            end select
         
         end select

         select case(ir)
               
         case(irprot_to_neut) ! prot(e-nu)neut
            n% rate_raw(i_rate,i) = n% rpen*rate_factors(i)
            n% rate_raw(i_rate_dT,i) = 0
            n% rate_raw(i_rate_dRho,i) = 0
            n% rate_screened(:,i) = n% rate_raw(:,i)
            r = y(prot) 
            dr1 = 1  
            if (n% rpen > 1d-50) then
               Qneu = n% spen/n% rpen
            else
               Qneu = 0
            end if
            call do_one_one_neu(
     >         n, i, 1d0, prot, r, 1d0, neut, prot, dr1, 
     >         n% reaction_Qs(ir), Qneu, deriv_flgs, category_factors, symbolic)
            if (g% doing_derivs_timing) call done_select
            return
         
         case(irneut_to_prot) ! neut(e+nu)prot
            n% rate_raw(i_rate,i) = n% rnep*rate_factors(i)
            n% rate_raw(i_rate_dT,i) = 0
            n% rate_raw(i_rate_dRho,i) = 0
            n% rate_screened(:,i) = n% rate_raw(:,i)
            r = y(neut)
            dr1 = 1 
            if (n% rnep > 1d-50) then
               Qneu = n% snep/n% rnep
            else
               Qneu = 0
            end if
            call do_one_one_neu(
     >         n, i, 1d0, neut, r, 1d0, prot, neut, dr1, 
     >         n% reaction_Qs(ir), Qneu, deriv_flgs, category_factors, symbolic)
            if (g% doing_derivs_timing) call done_select
            return
         
         case(irhe4_breakup) ! he4(g,neut)he3(g,prot)h2(g,neut)prot
            ! since rate depends on abundances of neut and prot,
            ! need to write this as a 3 to 2 reaction
            r = y(he4)
            dr1 = 1d0 ! dr/dhe4
            if (n% rate_screened(i_rate,i) > tiny_rate) then
               dr2 = r*n% d_rhe4_breakup_dneut/n% rate_screened(i_rate,i)
               dr3 = r*n% d_rhe4_breakup_dprot/n% rate_screened(i_rate,i)
            else
               dr2 = 0
               dr3 = 0
            end if
            call do_three_two(
     >         n, i, 1d0, he4, 1d0, neut, 1d0, prot,
     >         r, 3d0, neut, 3d0, prot, 
     >         he4, dr1, neut, dr2, prot, dr3,
     >         deriv_flgs, category_factors, symbolic)                   
            if (g% doing_derivs_timing) call done_select
            return
         
         case(irhe4_rebuild) ! prot(neut,g)h2(prot,g)he3(neut,g)he4
            r = y(neut)**2 * y(prot)**2
            dr1 = 2*y(neut)**2*y(prot)
            dr2 = 2*y(neut)*y(prot)**2
            if (n% rate_screened(i_rate,i) > tiny_rate) then
               dr1 = dr1 + r*n% d_rhe4_rebuild_dprot/n% rate_screened(i_rate,i)
               dr2 = dr2 + r*n% d_rhe4_rebuild_dneut/n% rate_screened(i_rate,i)
            end if
            call do_two_one(n, i, 2d0, neut, 2d0, prot, 
     >         r, 1d0, he4, prot, dr1, neut, dr2, 
     >         deriv_flgs, category_factors, symbolic)       
            if (g% doing_derivs_timing) call done_select
            return
         
         case(irhe4_to_2h1_2neut) ! he4(g,neut)he3(g,h1)h2(g,neut)h1
            ! since rate depends on abundances of neut and h1,
            ! need to write this as a 3 to 2 reaction
            r = y(he4)
            dr1 = 1d0 ! dr/dhe4
            if (n% rate_screened(i_rate,i) > tiny_rate) then
               dr2 = r*n% d_rhe4_to_2h1_2neut_dneut/n% rate_screened(i_rate,i)
               dr3 = r*n% d_rhe4_to_2h1_2neut_dh1/n% rate_screened(i_rate,i)
            else
               dr2 = 0
               dr3 = 0
            end if
            call do_three_two(
     >         n, i, 1d0, he4, 1d0, neut, 1d0, h1,
     >         r, 3d0, neut, 3d0, h1, 
     >         he4, dr1, neut, dr2, h1, dr3,
     >         deriv_flgs, category_factors, symbolic)                   
            if (g% doing_derivs_timing) call done_select
            return
         
         case(irhe4_from_2h1_2neut) ! h1(neut,g)h2(h1,g)he3(neut,g)he4
            r = y(neut)**2 * y(h1)**2
            dr1 = 2*y(neut)**2*y(h1)
            dr2 = 2*y(neut)*y(h1)**2
            if (n% rate_screened(i_rate,i) > tiny_rate) then
               dr1 = dr1 + r*n% d_rhe4_from_2h1_2neut_dh1/n% rate_screened(i_rate,i)
               dr2 = dr2 + r*n% d_rhe4_from_2h1_2neut_dneut/n% rate_screened(i_rate,i)
            end if
            call do_two_one(n, i, 2d0, neut, 2d0, h1,
     >         r, 1d0, he4, h1, dr1, neut, dr2, 
     >         deriv_flgs, category_factors, symbolic)       
            if (g% doing_derivs_timing) call done_select
            return
            
            case(irn14ag_lite) ! n14 + 1.5 alpha => ne20
               n14 = itab(in14)
               ne20 = itab(ine20)
               r = y(n14) * y(he4)
               dr1 = y(n14)
               dr2 = y(he4)               
               call do_two_one(n, i, 1d0, n14, 1.5d0, he4, 
     >            r, 1d0, ne20, he4, dr1, n14, dr2, 
     >            deriv_flgs, category_factors, symbolic)
               if (g% doing_derivs_timing) call done_select
               return
            
            case(irne22ga_fake) ! fake reaction to get ne22 to the c12 alpha chain: ne22(g,0.5a)ne20
               ne20 = itab(ine20)
               ne22 = itab(ine22)
               r = y(ne22)
               dr1 = 1d0       
               call do_one_two(
     >            n, i, 1d0, ne22, r, 0.5d0, he4, 1d0, ne20, ne22, dr1, 
     >            deriv_flgs, category_factors, symbolic)
               if (g% doing_derivs_timing) call done_select
               return
            
            case(irne22ag_fake) ! fake reaction to get ne22 to the c12 alpha chain: ne22(0.5a,g)mg24
               ne22 = itab(ine22)
               mg24 = itab(img24)
               r = y(ne22) * y(he4)
               dr1 = y(ne22)
               dr2 = y(he4)
               call do_two_one(n, i, 1d0, ne22, 0.5d0, he4, 
     >                  r, 1d0, mg24, he4, dr1, ne22, dr2, 
     >                  deriv_flgs, category_factors, symbolic)       
               if (g% doing_derivs_timing) call done_select
               return

            case(irne20ng_to_ne22) ! ne20(neut,g)ne21(neut,g)ne22
               call do_neutg_neutg_reaction(i, ine20, ine22, n% d_rne20neut_to_ne22_dneut)
               return
            case(irne22gn_to_ne20) ! ne22(g,neut)ne21(g,neut)ne20
               call do_gneut_gneut_reaction(i, ine20, ine22, n% d_rne22g_to_ne20_dneut)
               return

            case(ir_he4_he4_he4_to_c12) ! triple alpha               
               if (g% which_rates(ir) == use_rate_3a_FL87) then 
                  ! use Fushiki and Lamb, Apj, 317, 368-388, 1987
                  call do_FL_3alf(i)
                  if (g% doing_derivs_timing) call done_select
                  return
               end if

            case(ir1212) ! c12 + c12 -> ne20 + he4; rate includes -> na23 + h1
               if (g% which_rates(ir) == use_rate_1212_G05) then 
                  ! use Gasques, et al. Phys Review C, 72, 025806 (2005)
                  call do_G05_1212_to_ne20(i)
                  if (g% doing_derivs_timing) call done_select
                  return
               end if

            case(irn14ecap) ! n14(e-,nu)c14
               n14 = itab(in14)
               c14 = itab(ic14)
               r = y(n14)
               call eval_n14_electron_capture_rate(temp,den,1/ye,rn14ec)
               n% rate_raw(i_rate,i) = rn14ec*rate_factors(i)
               n% rate_raw(i_rate_dT,i) = 0
               n% rate_raw(i_rate_dRho,i) = 0
               n% rate_screened(:,i) = n% rate_raw(:,i)
               dr1 = 1d0
               call do_one_one(n, i, 1d0, n14, 
     >            r, 1d0, c14, n14, dr1, deriv_flgs, category_factors, symbolic)
               if (g% doing_derivs_timing) call done_select
               return

         end select
         
         if (g% doing_derivs_timing) call done_select

         num_reaction_inputs = get_num_reaction_inputs(ir)
         num_reaction_outputs = get_num_reaction_outputs(ir)
         
         if (dbg) write(*,*) 'num_reaction_inputs', num_reaction_inputs
         if (dbg) write(*,*) 'num_reaction_outputs', num_reaction_outputs
         if (dbg) write(*,*)
         
         switch_to_prot = .false.
         cout1 = 0; out1 = 0; o1 = 0
         cout2 = 0; out2 = 0; o2 = 0
         cout3 = 0; out3 = 0; o3 = 0
         cin1 = 0; in1 = 0; i1 = 0
         cin2 = 0; in2 = 0; i2 = 0
         cin3 = 0; in3 = 0; i3 = 0

         if (num_reaction_outputs >= 1) then
            cout1 = reaction_outputs(1,ir); dout1 = cout1
            out1 = reaction_outputs(2,ir)
            o1 = itab(out1)
            if (o1 == 0) then
               write(*,*) trim(reaction_Name(ir))
               stop 'get1_derivs: itab(out1) = 0'
            end if
         end if
            
         if (num_reaction_outputs >= 2) then
            cout2 = reaction_outputs(3,ir); dout2 = cout2
            out2 = reaction_outputs(4,ir)
            o2 = itab(out2)
            if (o2 == 0) then
               write(*,*) trim(reaction_Name(ir))
               stop 'get1_derivs: itab(out2) = 0'
            end if
         end if
            
         if (num_reaction_outputs >= 3) then
            cout3 = reaction_outputs(5,ir); dout3 = cout3
            out3 = reaction_outputs(6,ir)
            o3 = itab(out3)
            if (o3 == 0) then
               write(*,*) trim(reaction_Name(ir))
               stop 'get1_derivs: itab(out3) = 0'
            end if
         end if
            
         if (num_reaction_outputs >= 4) then
            write(*,*) trim(reaction_Name(ir))
            stop 'get1_derivs: num_reaction_outputs >= 4'
         end if

         
         if (num_reaction_inputs == 1) then            
            cin1 = reaction_inputs(1,ir); din1 = cin1
            in1 = reaction_inputs(2,ir)
            i1 = itab(in1)            
         else if (num_reaction_inputs == 2 .or. num_reaction_inputs == 3) then            
            cin1 = reaction_inputs(1,ir); din1 = cin1
            in1 = reaction_inputs(2,ir)
            i1 = itab(in1)                        
            cin2 = reaction_inputs(3,ir); din2 = cin2
            in2 = reaction_inputs(4,ir)
            i2 = itab(in2)            
            if (num_reaction_inputs == 3) then
               cin3 = reaction_inputs(5,ir); din3 = cin3
               in3 = reaction_inputs(6,ir)
               i3 = itab(in3)
            end if
         end if
            
         switch_to_prot = (prot /= 0) .and. (max_Z >= min_Z_for_switch_to_prot)
         if (switch_to_prot) then
            if (i1 == h1) then
               in1 = iprot
               i1 = prot
            else if (i2 == h1) then
               in2 = iprot
               i2 = prot
            else if (i3 == h1) then
               in3 = iprot
               i3 = prot
            end if
            if (o1 == h1) then
               out1 = iprot
               o1 = prot
            else if (o2 == h1) then
               out2 = iprot
               o2 = prot
            else if (o3 == h1) then
               out3 = iprot
               o3 = prot
            end if
         end if



         
         if (num_reaction_inputs == 1) then
            
            if (i1 == 0) then
               write(*,*) trim(reaction_Name(ir))
               write(*,2) 'num_reaction_inputs', num_reaction_inputs
               stop 'get1_derivs: itab(in1) = 0'
            end if
            
            if (cin1 == 1) then
               r = y(i1)
               idr1 = i1
               dr1 = 1
            else if (cin1 == 3 .and. in1 /= ih1) then ! 3 he4
               !write(*,'(/,a)') '1/6*r  reaction name <' // trim(reaction_Name(ir)) // '>'
               r = (1d0/6d0)*y(i1)**3
               idr1 = i1
               dr1 = 0.5d0*y(i1)**2
            else ! 2 body
               !write(*,'(/,a)') '1/2*r  reaction name <' // trim(reaction_Name(ir)) // '>'
               !write(*,'(i3,3x,99e20.10)') i, n% rate_raw(i_rate,i), n% rate_screened(i_rate,i)
               r = 0.5d0*y(i1)**2
               idr1 = i1
               dr1 = y(i1)
               !stop
            end if
            
         else if (num_reaction_inputs == 2 .or. num_reaction_inputs == 3) then
                        
            if (reaction_ye_rho_exponents(2,ir) == 0) then
               ! treat as 1 body reaction
               r = y(i1)
               idr1 = i1
               dr1 = 1
               idr2 = i2
               dr2 = 0
               !write(*,*) 'get1_derivs rho=0: ' // trim(reaction_Name(ir))
               !stop 'net_derivs'
            else if ((cin1 == 1 .and. cin2 == 1) .or. reaction_ye_rho_exponents(2,ir) == 1) then
               ! treat as 2 body reaction
               r = y(i1)*y(i2)
               dr1 = y(i1)
               idr1 = i2
               dr2 = y(i2)
               idr2 = i1
            else if (cin1 == 2 .and. cin2 == 1) then 
               r = 0.5d0*y(i1)**2*y(i2)
               dr1 = 0.5d0*y(i1)**2
               idr1 = i2
               dr2 = y(i1)*y(i2)
               idr2 = i1
            else if (cin1 == 1 .and. cin2 == 2) then 
               ! e.g., rhe4p, r_neut_he4_he4_to_be9, r_neut_h1_h1_to_h1_h2
               r = y(i1)*0.5d0*y(i2)**2
               dr1 = y(i1)*y(i2)
               idr1 = i2
               dr2 = 0.5d0*y(i2)**2
               idr2 = i1
            else if (cin1 == 2 .and. cin2 == 2) then 
               ! e.g., r_neut_neut_he4_he4_to_h3_li7, r_h1_h1_he4_he4_to_he3_be7
               r = 0.5d0*y(i1)**2*0.5d0*y(i2)**2
               dr1 = 0.5d0*y(i1)**2*y(i2)
               idr1 = i2
               dr2 = y(i1)*0.5d0*y(i2)**2
               idr2 = i1
            else
               write(*,*) 'get1_derivs: ' // trim(reaction_Name(ir)) // ' invalid coefficient'
               stop 'get1_derivs'
            end if            
            
            if (num_reaction_inputs == 3) then
               ! we assume that the 3rd kind of input is just "along for the ride"
               ! e.g., some compound reactions such as r34_pp2 are in this category.
               dr3 = 0
               idr3 = i3
               if (i3 == 0) then
                  write(*,*) trim(reaction_Name(ir))
                  stop 'get1_derivs: itab(in3) = 0'
               end if
            end if

         else

            write(*,*) 'get1_derivs: ' // trim(reaction_Name(ir)) // ' invalid specification'
            stop 'get1_derivs'

         end if
         
         if (temp > 1d9) then
            has_neut = (i1 == neut .or. i2 == neut .or. i3 == neut .or.
     >         o1 == neut .or. o2 == neut .or. o3 == neut)       
            if (has_neut) then
               call limit_rate_at_very_high_T(
     >             ir, rates, rtab, n% factor_neut, n% d_factor_neut_dT, .false.) 
            else if (prot /= 0) then
               has_prot = (i1 == prot .or. i2 == prot .or. i3 == prot .or.
     >            o1 == prot .or. o2 == prot .or. o3 == prot)                  
               if (has_prot) then
                  call limit_rate_at_very_high_T(
     >               ir, rates, rtab, n% factor_prot, n% d_factor_prot_dT, .false.) 
               end if
            else
               has_h1 = (i1 == h1 .or. i2 == h1 .or. i3 == h1 .or.
     >            o1 == h1 .or. o2 == h1 .or. o3 == h1)                  
               if (has_h1) then
                  call limit_rate_at_very_high_T(
     >               ir, rates, rtab, n% factor_prot, n% d_factor_prot_dT, .false.) 
               end if
            end if
         end if
         
         if (num_reaction_inputs == 1) then
         
            if (num_reaction_outputs == 1) then 
               ! reaction of form din1 in1 -> dout1 out1
               if (dbg) write(*,*) ' do_one_one din1', din1, trim(chem_isos% name(g% chem_id(i1)))
               if (dbg) write(*,*) 'do_one_one dout1', dout1, trim(chem_isos% name(g% chem_id(o1)))
               
               weak_id = g% weak_reaction_index(i)
               done = .false.
               if (weak_id > 0) then
                  if (g% weaklib_ids(weak_id) > 0) then ! 0 means not included in weaklib
                     n% rate_screened(i_rate,i) = n% lambda(weak_id)
                     n% rate_screened(i_rate_dT,i) = 0
                     n% rate_screened(i_rate_dRho,i) = 0
                     !write(*,*)
                     !stop
                     call do_one_one_neu(
     >                  n, i, din1, i1, r, dout1, o1, 
     >                  idr1, dr1, n% Q(weak_id), n% Qneu(weak_id), 
     >                  deriv_flgs, category_factors, symbolic)
                     done = .true.
                  end if
               end if
               if (.not. done) then
                  call do_one_one(
     >               n, i, din1, i1, r, dout1, o1, 
     >               idr1, dr1, deriv_flgs, category_factors, symbolic)
               end if
     
            else if (num_reaction_outputs == 2) then
               ! reaction of form cin1 in1 -> dout1 out1 + dout2 out2
               if (dbg) write(*,*) ' do_one_two din1', din1, trim(chem_isos% name(g% chem_id(i1)))
               if (dbg) write(*,*) 'do_one_two dout1', dout1, trim(chem_isos% name(g% chem_id(o1)))
               if (dbg) write(*,*) 'do_one_two dout2', dout2, trim(chem_isos% name(g% chem_id(o2)))
               call do_one_two(
     >            n, i, din1, i1, r, dout1, o1, dout2, o2, 
     >            idr1, dr1, deriv_flgs, category_factors, symbolic)
            else if (num_reaction_outputs == 3) then
               ! reaction of form cin1 in1 -> dout1 out1 + dout2 out2 + dout3 out3
               if (dbg) write(*,*) ' do_one_three din1', din1, trim(chem_isos% name(g% chem_id(i1)))
               if (dbg) write(*,*) 'do_one_three dout1', dout1, trim(chem_isos% name(g% chem_id(o1)))
               if (dbg) write(*,*) 'do_one_three dout2', dout2, trim(chem_isos% name(g% chem_id(o2)))
               if (dbg) write(*,*) 'do_one_three dout3', dout3, trim(chem_isos% name(g% chem_id(o3)))
               call do_one_three(
     >            n, i, din1, i1, r, dout1, o1, dout2, o2, dout3, o3, 
     >            idr1, dr1, deriv_flgs, category_factors, symbolic)
            else
               write(*,*) trim(reaction_Name(ir))
               stop 'too many reaction_outputs for num_reaction_inputs == 1'
            end if
            
         else if (num_reaction_inputs == 2) then
         
            if (num_reaction_outputs == 1) then 
               ! reaction of form din1 in1 + din2 in2 -> dout1 out1
               if (dbg) write(*,*) ' do_two_one din1', din1, trim(chem_isos% name(g% chem_id(i1)))
               if (dbg) write(*,*) ' do_two_one din2', din2, trim(chem_isos% name(g% chem_id(i2)))
               if (dbg) write(*,*) 'do_two_one dout1', dout1, trim(chem_isos% name(g% chem_id(o1)))
               if (.false. .and. reaction_Name(ir) == 'rc12_to_n14' .and. r > 0) then
                  write(*,'(i3,3x,a,2x,99e20.10)') i, 
     >               'do_two_one ' // trim(reaction_Name(ir)) // ' ' //
     >               trim(chem_isos% name(g% chem_id(i1))) // ' + ' // 
     >               trim(chem_isos% name(g% chem_id(i2))) // ' => ' // 
     >               trim(chem_isos% name(g% chem_id(o1))), 
     >               r, dr1, dr2, y(i1), y(i2)
                  !stop
               end if
               call do_two_one(
     >            n, i, din1, i1, din2, i2, r, dout1, o1, 
     >            idr1, dr1, idr2, dr2, deriv_flgs, category_factors, symbolic)
            else if (num_reaction_outputs == 2) then
               ! reaction of form din1 in1 + din2 in2 -> dout1 out1 + dout2 out2
               if (dbg) write(*,*) ' do_two_two din1', din1, trim(chem_isos% name(g% chem_id(i1)))
               if (dbg) write(*,*) ' do_two_two din2', din2, trim(chem_isos% name(g% chem_id(i2)))
               if (dbg) write(*,*) 'do_two_two dout1', dout1, trim(chem_isos% name(g% chem_id(o1)))
               if (dbg) write(*,*) 'do_two_two dout2', dout2, trim(chem_isos% name(g% chem_id(o2)))
               call do_two_two(
     >            n, i, din1, i1, din2, i2, r, dout1, o1, dout2, o2, 
     >            idr1, dr1, idr2, dr2, deriv_flgs, category_factors, symbolic)
            else if (num_reaction_outputs == 3) then
               ! reaction of form din1 in1 + din2 in2 -> dout1 out1 + dout2 out2 + dout3 out3
               if (dbg) write(*,*) ' do_two_three din1', din1, trim(chem_isos% name(g% chem_id(i1)))
               if (dbg) write(*,*) ' do_two_three din2', din2, trim(chem_isos% name(g% chem_id(i2)))
               if (dbg) write(*,*) 'do_two_three dout1', dout1, trim(chem_isos% name(g% chem_id(o1)))
               if (dbg) write(*,*) 'do_two_three dout2', dout2, trim(chem_isos% name(g% chem_id(o2)))
               if (dbg) write(*,*) 'do_two_three dout3', dout3, trim(chem_isos% name(g% chem_id(o3)))
               
               call do_two_three(
     >            n, i, din1, i1, din2, i2, r, dout1, o1, dout2, o2, dout3, o3, 
     >            idr1, dr1, idr2, dr2, deriv_flgs, category_factors, symbolic)
            else
               write(*,*) trim(reaction_Name(ir))
               stop 'too many reaction_outputs for num_reaction_inputs == 2'
            end if
            
         else if (num_reaction_inputs == 3) then

            if (num_reaction_outputs == 1) then 
               ! reaction of form din1 in1 + din2 in2 + din3 in3 -> dout1 out1
               if (dbg) write(*,*) ' do_three_one din1', din1, trim(chem_isos% name(g% chem_id(i1)))
               if (dbg) write(*,*) ' do_three_one din2', din2, trim(chem_isos% name(g% chem_id(i2)))
               if (dbg) write(*,*) ' do_three_one din3', din3, trim(chem_isos% name(g% chem_id(i3)))
               if (dbg) write(*,*) 'do_three_one dout1', dout1, trim(chem_isos% name(g% chem_id(o1)))
               call do_three_one(
     >            n, i, din1, i1, din2, i2, din3, i3, r, dout1, o1, 
     >            idr1, dr1, idr2, dr2, deriv_flgs, category_factors, symbolic)
            else if (num_reaction_outputs == 2) then
               ! reaction of form din1 in1 + din2 in2 + din3 in3 -> dout1 out1 + dout2 out2
               if (dbg) write(*,*) ' do_three_two din1', din1, trim(chem_isos% name(g% chem_id(i1)))
               if (dbg) write(*,*) ' do_three_two din2', din2, trim(chem_isos% name(g% chem_id(i2)))
               if (dbg) write(*,*) ' do_three_two din3', din3, trim(chem_isos% name(g% chem_id(i3)))
               if (dbg) write(*,*) 'do_three_two dout1', dout1, trim(chem_isos% name(g% chem_id(o1)))
               if (dbg) write(*,*) 'do_three_two dout2', dout2, trim(chem_isos% name(g% chem_id(o2)))
               call do_three_two(
     >            n, i, din1, i1, din2, i2, din3, i3, r, dout1, o1, dout2, o2,
     >            idr1, dr1, idr2, dr2, idr3, dr3, deriv_flgs, category_factors, symbolic)
            else
               write(*,*) trim(reaction_Name(ir))
               stop 'too many reaction_outputs for num_reaction_inputs == 3'
            end if
            
         else
            stop 'too many reaction_inputs'
         end if
         


         if (g% doing_derivs_timing) then
            call system_clock(time1)
            g% clock_derivs_general = g% clock_derivs_general + (time1 - time0)
            time0 = time1
         end if
               
               
         
         contains
         
         
         subroutine done_select
            call system_clock(time1)
            g% clock_derivs_select = g% clock_derivs_select + (time1 - time0)
            time0 = time1
         end subroutine done_select 
         
         subroutine check_ec_rates
            real(dp) :: rate_ni56, rate_ni55, rate_co56, rate_co55, 
     >            Q, Qneu, d_rate_dlnT, d_rate_dlnRho
            include 'formats'
            call eval1_weak_rate(weak_rate_id_for_ni56_ec,
     >         temp, ye, den, eta, rate_ni56, d_rate_dlnT, d_rate_dlnRho, Q, Qneu, ierr)
            if (ierr /= 0) return
            call eval1_weak_rate(weak_rate_id_for_ni55_ec,
     >         temp, ye, den, eta, rate_ni55, d_rate_dlnT, d_rate_dlnRho, Q, Qneu, ierr)
            if (ierr /= 0) return
            call eval1_weak_rate(weak_rate_id_for_co56_ec,
     >         temp, ye, den, eta, rate_co56, d_rate_dlnT, d_rate_dlnRho, Q, Qneu, ierr)
            if (ierr /= 0) return
            call eval1_weak_rate(weak_rate_id_for_co55_ec,
     >         temp, ye, den, eta, rate_co55, d_rate_dlnT, d_rate_dlnRho, Q, Qneu, ierr)
            if (ierr /= 0) return
            write(*,1) 'rate_ni56', rate_ni56
            write(*,1) 'rate_ni55', rate_ni55
            write(*,1) 'rate_co56', rate_co56
            write(*,1) 'rate_co55', rate_co55
            stop 'check_ec_rates'
         end subroutine check_ec_rates

         subroutine do_2ec_reaction(
     >         i, i_ni66, i_fe66, weak_rate_id_for_ec, ierr)
            integer, intent(in) :: i, i_ni66, i_fe66, weak_rate_id_for_ec
            integer, intent(out) :: ierr
            call do_ec_nn_reaction(
     >         i, i_ni66, i_fe66, 0, weak_rate_id_for_ec, ierr)
         end subroutine do_2ec_reaction

         subroutine do_ec_nn_reaction(
     >         i, i_ni66, i_fe60, n_neut, weak_rate_id_for_ec, ierr)
            integer, intent(in) :: i, i_ni66, i_fe60, n_neut, weak_rate_id_for_ec
            integer, intent(out) :: ierr
            
            integer :: neut, fe60, ni66
            real(dp) :: r, dr1, rate, d_rate_dlnT, d_rate_dlnRho, Q, Qneu
            
            logical, parameter :: only_main_rate = .true.
            
            include 'formats'
            ierr = 0
            neut = itab(ineut)
            fe60 = itab(i_fe60)
            ni66 = itab(i_ni66)
            r = y(ni66)
            
            call eval1_weak_rate(weak_rate_id_for_ec, temp, ye, den, eta, 
     >         rate, d_rate_dlnT, d_rate_dlnRho, Q, Qneu, ierr)
            if (ierr /= 0) return
            Q = 2*Q
            Qneu = 2*Qneu
            
            n% rate_raw(i_rate,i) = rate
            n% rate_raw(i_rate_dT,i) = 0d0 ! d_rate_dlnT/temp << bad partials from weaklib?
            n% rate_raw(i_rate_dRho,i) = 0d0 ! d_rate_dlnRho/den
            n% rate_raw(:,i) = n% rate_raw(:,i)*rate_factors(i)
            n% rate_screened(:,i) = n% rate_raw(:,i)
            dr1 = 1d0
            
            if (n_neut == 0) then
               call do_one_one_neu(n, i, 1d0, ni66, 
     >            r, 1d0, fe60, ni66, dr1, 
     >            Q, Qneu, deriv_flgs, category_factors, symbolic)
            else
               call do_one_two_neu(n, i, 1d0, ni66, 
     >            r, 1d0, fe60, dble(n_neut), neut, ni66, dr1, 
     >            Q, Qneu, deriv_flgs, category_factors, symbolic)
            end if
            
            if (g% doing_derivs_timing) call done_select
            
         end subroutine do_ec_nn_reaction

         subroutine do_2ee_reaction(
     >         i, i_cr56, i_fe56, weak_rate_id_for_ee, ierr)
            integer, intent(in) :: i, i_cr56, i_fe56, weak_rate_id_for_ee
            integer, intent(out) :: ierr
            
            integer :: fe56, cr56
            real(dp) :: r, dr1, rate, d_rate_dlnT, d_rate_dlnRho, Q, Qneu
            
            include 'formats'
            ierr = 0
            fe56 = itab(i_fe56)
            cr56 = itab(i_cr56)
            r = y(cr56)
            
            call eval1_weak_rate(weak_rate_id_for_ee, temp, ye, den, eta, 
     >         rate, d_rate_dlnT, d_rate_dlnRho, Q, Qneu, ierr)
            if (ierr /= 0) return
            Q = 2*Q
            Qneu = 2*Qneu
            
            n% rate_raw(i_rate,i) = rate
            n% rate_raw(i_rate_dT,i) = 0d0 ! d_rate_dlnT/temp << bad partials from weaklib?
            n% rate_raw(i_rate_dRho,i) = 0d0 ! d_rate_dlnRho/den
            n% rate_raw(:,i) = n% rate_raw(:,i)*rate_factors(i)
            n% rate_screened(:,i) = n% rate_raw(:,i)
            dr1 = 1d0
            
            call do_one_one_neu(n, i, 1d0, cr56, 
     >         r, 1d0, fe56, cr56, dr1, 
     >         Q, Qneu, deriv_flgs, category_factors, symbolic)
            
            if (g% doing_derivs_timing) call done_select
            
         end subroutine do_2ee_reaction
         
         subroutine do_ecnp_reaction(
     >         i, i_ni66, i_co66, i_fe64, 
     >         weak_rate1_id_for_ec, weak_rate2_id_for_ec, weak_rate3_id_for_ec,
     >         ierr)
            integer, intent(in) :: i, i_ni66, i_co66, i_fe64, 
     >         weak_rate1_id_for_ec, weak_rate2_id_for_ec, weak_rate3_id_for_ec
            integer, intent(out) :: ierr
            call do_ecnp_nn_reaction(
     >         i, i_ni66, i_co66, i_fe64, 0, 
     >         weak_rate1_id_for_ec, weak_rate2_id_for_ec, weak_rate3_id_for_ec,  
     >         temp, ierr)
         end subroutine do_ecnp_reaction

         subroutine do_ecnp_nn_reaction(
     >         i, i_ni66, i_co66, i_fe62, extra_neut, 
     >         weak_rate1_id_for_ec, weak_rate2_id_for_ec, weak_rate3_id_for_ec,  
     >         temp, ierr)
            use chem_def, only: chem_isos
            use chem_lib, only: reaction_Qtotal
            integer, intent(in) :: 
     >         i, i_ni66, i_co66, i_fe62, extra_neut, 
     >         weak_rate1_id_for_ec, weak_rate2_id_for_ec, weak_rate3_id_for_ec
            real(dp), intent(in) :: temp
            integer, intent(out) :: ierr
            
            integer :: neut, prot, fe62, ni66, c1, c2, c3, cnt, num_reactants
            integer :: reactants(10)
            real(dp) :: r, dr1, 
     >         rate, d_rate_dlnT, d_rate_dlnRho, Q, Qneu, Qbind,
     >         rate1, d_rate1_dlnT, d_rate1_dlnRho, Q1, Qneu1, 
     >         rate2, d_rate2_dlnT, d_rate2_dlnRho, Q2, Qneu2,
     >         rate3, d_rate3_dlnT, d_rate3_dlnRho, Q3, Qneu3
            
            logical, parameter :: only_main_rate = .true.
            
            include 'formats'
            ierr = 0
            neut = itab(ineut)
            prot = itab(iprot)
            fe62 = itab(i_fe62)
            ni66 = itab(i_ni66)
            r = y(ni66)
            c1 = 0
            c2 = 0
            c3 = 0
            
            if (weak_rate1_id_for_ec == 0) then
               rate1=0; d_rate1_dlnT=0; d_rate1_dlnRho=0; Q1=0; Qneu1=0
            else
               call eval1_weak_rate(weak_rate1_id_for_ec, temp, ye, den, eta, 
     >            rate1, d_rate1_dlnT, d_rate1_dlnRho, Q1, Qneu1, ierr)
               if (ierr /= 0) return
               c1 = 2 ! extra weight for "base" rate
            end if
            
            if (only_main_rate .or. weak_rate2_id_for_ec == 0) then
               rate2=0; d_rate2_dlnT=0; d_rate2_dlnRho=0; Q2=0; Qneu2=0
            else
               call eval1_weak_rate(weak_rate2_id_for_ec, temp, ye, den, eta, 
     >            rate2, d_rate2_dlnT, d_rate2_dlnRho, Q2, Qneu2, ierr)
               if (ierr /= 0) return
               c2 = 1
            end if
            
            if (only_main_rate .or. weak_rate3_id_for_ec == 0) then
               rate3=0; d_rate3_dlnT=0; d_rate3_dlnRho=0; Q3=0; Qneu3=0
            else
               call eval1_weak_rate(weak_rate3_id_for_ec, temp, ye, den, eta, 
     >            rate3, d_rate3_dlnT, d_rate3_dlnRho, Q3, Qneu3, ierr)
               if (ierr /= 0) return
               c3 = 1
            end if
            
            cnt = c1 + c2 + c3
            if (cnt == 0) then
!$omp critical
               write(*,1) 'bad args for ec ' // trim(reaction_name(ir))
               write(*,2) 'i_fe62 ' // trim(chem_isos% name(i_fe62)), i_fe62
               write(*,2) 'i_ni66 ' // trim(chem_isos% name(i_ni66)), i_ni66
               write(*,2) 'weak_rate1_id_for_ec', weak_rate1_id_for_ec
               write(*,2) 'weak_rate2_id_for_ec', weak_rate2_id_for_ec
               write(*,2) 'weak_rate3_id_for_ec', weak_rate3_id_for_ec
               stop 'do_ecnp_nn_reaction'
!$omp end critical
               ierr = -1
               return
            end if
            
            rate = (c1*rate1 + c2*rate2 + c3*rate3)/cnt
            d_rate_dlnT = (c1*d_rate1_dlnT + c2*d_rate2_dlnT + c3*d_rate3_dlnT)/cnt
            d_rate_dlnRho = (c1*d_rate1_dlnRho + c2*d_rate2_dlnRho + c3*d_rate3_dlnRho)/cnt
            Q = (c1*Q1 + c2*Q2 + c3*Q3)/cnt
            Qneu = (c1*Qneu1 + c2*Qneu2 + c3*Qneu3)/cnt
            
            ! add Q for splitting co66 -> fe62 + (1+extra) n + 1 p
            num_reactants = 4 + extra_neut
            reactants(1) = i_co66
            reactants(2) = i_fe62
            reactants(3) = iprot
            reactants(4:num_reactants) = ineut
            Qbind = reaction_Qtotal(1, num_reactants-1, reactants, chem_isos)
            
            Q = Q + Qbind
            
            n% rate_raw(i_rate,i) = rate
            n% rate_raw(i_rate_dT,i) = 0d0 ! d_rate_dlnT/temp << bad partials from weaklib?
            n% rate_raw(i_rate_dRho,i) = 0d0 ! d_rate_dlnRho/den
            n% rate_raw(:,i) = n% rate_raw(:,i)*rate_factors(i)
            n% rate_screened(:,i) = n% rate_raw(:,i)
            dr1 = 1d0
            
            call do_one_three_neu(n, i, 1d0, ni66, 
     >         r, 1d0, fe62, dble(1+extra_neut), neut, 1d0, prot, ni66, dr1, 
     >         Q, Qneu, deriv_flgs, category_factors, symbolic)
            
            if (g% doing_derivs_timing) call done_select
            
         end subroutine do_ecnp_nn_reaction
         
         subroutine do_protg_protg_reaction(i, ife54, ini56, d_rfe54p_to_ni56_dp)
            integer, intent(in) :: i, ife54, ini56
            real(dp), intent(in) :: d_rfe54p_to_ni56_dp
            integer :: prot, fe54, ni56
            real(dp) :: r, dr1, dr2
            prot = itab(iprot)
            fe54 = itab(ife54)
            ni56 = itab(ini56)
            r = y(fe54) * y(prot)**2
            dr1 = 2 * y(fe54) * y(prot)
            dr2 = y(prot)**2
            if (n% rate_screened(i_rate,i) > tiny_rate) then
               dr1 = dr1 + r*d_rfe54p_to_ni56_dp/n% rate_screened(i_rate,i)
            end if
            call do_two_one(n, i, 1d0, fe54, 2d0, prot, 
     >         r, 1d0, ni56, prot, dr1, fe54, dr2, 
     >         deriv_flgs, category_factors, symbolic)       
            if (g% doing_derivs_timing) call done_select
         end subroutine do_protg_protg_reaction

         subroutine do_gprot_gprot_reaction(i, ife54, ini56, d_rni56gp_to_fe54_dp)
            integer, intent(in) :: i, ife54, ini56
            real(dp), intent(in) :: d_rni56gp_to_fe54_dp
            integer :: prot, fe54, ni56
            real(dp) :: r, dr1, dr2
            prot = itab(iprot)
            fe54 = itab(ife54)
            ni56 = itab(ini56)
            r = y(ni56)
            dr1 = 1d0        
            if (n% rate_screened(i_rate,i) > tiny_rate) then
               dr2 = r*d_rni56gp_to_fe54_dp/n% rate_screened(i_rate,i)
            else
               dr2 = 0
            end if
            call do_two_two(n, i, 1d0, ni56, 1d0, prot, 
     >         r, 3d0, prot, 1d0, fe54, ni56, dr1, prot, dr2, 
     >         deriv_flgs, category_factors, symbolic)       
            if (g% doing_derivs_timing) call done_select
         end subroutine do_gprot_gprot_reaction

         subroutine do_aprot_gprot_reaction(i, ife52, ife54, d_rfe52aprot_to_fe54_dprot)
            integer, intent(in) :: i, ife52, ife54
            real(dp), intent(in) :: d_rfe52aprot_to_fe54_dprot
            integer :: prot, he4, fe52, fe54
            real(dp) :: r, dr1, dr2
            ! since rate depends on abundance of prot,
            ! need to write this as a 3 to 2 reaction
            prot = itab(iprot)
            he4 = itab(ihe4)
            fe52 = itab(ife52)
            fe54 = itab(ife54)
            r = y(fe52) * y(he4)
            dr1 = y(he4)
            dr2 = y(fe52)
            if (n% rate_screened(i_rate,i) > tiny_rate) then
               dr3 = r*d_rfe52aprot_to_fe54_dprot/n% rate_screened(i_rate,i)
            else
               dr3 = 0
            end if
            call do_three_two(
     >         n, i, 1d0, fe52, 1d0, he4, 1d0, prot,
     >         r, 1d0, fe54, 3d0, prot, 
     >         fe52, dr1, he4, dr2, prot, dr3,
     >         deriv_flgs, category_factors, symbolic) 
            if (g% doing_derivs_timing) call done_select
         end subroutine do_aprot_gprot_reaction

         subroutine do_protg_prota_reaction(i, ife52, ife54, d_rfe54prot_to_fe52_dprot)
            integer, intent(in) :: i, ife52, ife54
            real(dp), intent(in) :: d_rfe54prot_to_fe52_dprot
            integer :: prot, he4, fe52, fe54
            real(dp) :: r, dr1, dr2
            prot = itab(iprot)
            he4 = itab(ihe4)
            fe52 = itab(ife52)
            fe54 = itab(ife54)
            r = y(fe54) * y(prot)**2
            dr1 = 2 * y(fe54)* y(prot)
            dr2 = y(prot)**2
            if (n% rate_screened(i_rate,i) > tiny_rate) then
               dr1 = dr1 + r*d_rfe54prot_to_fe52_dprot/n% rate_screened(i_rate,i)
            end if
            call do_two_two(n, i, 1d0, fe54, 2d0, prot, 
     >         r, 1d0, he4, 1d0, fe52, prot, dr1, fe54, dr2, 
     >         deriv_flgs, category_factors, symbolic)
            if (g% doing_derivs_timing) call done_select
         end subroutine do_protg_prota_reaction

         subroutine do_aprot_protg_reaction(i, ife52, ini56, d_rfe52aprot_to_ni56_dprot)
            integer, intent(in) :: i, ife52, ini56
            real(dp), intent(in) :: d_rfe52aprot_to_ni56_dprot
            integer :: prot, he4, fe52, ni56
            real(dp) :: r, dr1, dr2
            ! since rate depends on abundance of prot,
            ! need to write this as a 3 to 2 reaction
            prot = itab(iprot)
            he4 = itab(ihe4)
            fe52 = itab(ife52)
            ni56 = itab(ini56)
            r = y(fe52) * y(he4) * y(prot)
            dr1 = y(he4) * y(prot)
            dr2 = y(fe52) * y(prot)
            dr3 = y(fe52) * y(he4)
            if (n% rate_screened(i_rate,i) > tiny_rate) then
               dr3 = dr3 + r*d_rfe52aprot_to_ni56_dprot/n% rate_screened(i_rate,i)
            end if
            call do_three_two(
     >         n, i, 1d0, fe52, 1d0, he4, 1d0, prot,
     >         r, 1d0, ni56, 1d0, prot, 
     >         fe52, dr1, he4, dr2, prot, dr3,
     >         deriv_flgs, category_factors, symbolic)                   
            if (g% doing_derivs_timing) call done_select
         end subroutine do_aprot_protg_reaction

         subroutine do_gprot_prota_reaction(i, ife52, ini56, d_rni56gprot_to_fe52_dprot)
            integer, intent(in) :: i, ife52, ini56
            real(dp), intent(in) :: d_rni56gprot_to_fe52_dprot
            integer :: prot, he4, fe52, ni56
            real(dp) :: r, dr1, dr2
            prot = itab(iprot)
            he4 = itab(ihe4)
            fe52 = itab(ife52)
            ni56 = itab(ini56)
            r = y(ni56) * y(prot)
            dr1 = y(prot)
            dr2 = y(ni56)
            if (n% rate_screened(i_rate,i) > tiny_rate) then
               dr2 = dr2 + r*d_rni56gprot_to_fe52_dprot/n% rate_screened(i_rate,i)
            end if
            call do_two_three(n, i, 1d0, ni56, 1d0, prot, 
     >         r, 1d0, prot, 1d0, he4, 1d0, fe52, 
     >         ni56, dr1, prot, dr2, deriv_flgs, category_factors, symbolic)       
            if (g% doing_derivs_timing) call done_select
         end subroutine do_gprot_prota_reaction




         
         subroutine do_neutg_neutg_reaction(i, ini56, ini58, d_rni56p_to_ni58_dneut)
            integer, intent(in) :: i, ini56, ini58
            real(dp), intent(in) :: d_rni56p_to_ni58_dneut
            integer :: neut, ni56, ni58
            real(dp) :: r, dr1, dr2
            neut = itab(ineut)
            ni56 = itab(ini56)
            ni58 = itab(ini58)
            r = y(ni56) * y(neut)**2
            dr1 = 2 * y(ni56) * y(neut)
            dr2 = y(neut)**2
            if (n% rate_screened(i_rate,i) > tiny_rate) then
               dr1 = dr1 + r*d_rni56p_to_ni58_dneut/n% rate_screened(i_rate,i)
            end if
            call do_two_one(n, i, 1d0, ni56, 2d0, neut, 
     >         r, 1d0, ni58, neut, dr1, ni56, dr2, 
     >         deriv_flgs, category_factors, symbolic)       
            if (g% doing_derivs_timing) call done_select
         end subroutine do_neutg_neutg_reaction

         subroutine do_gneut_gneut_reaction(i, ini56, ini58, d_rni58gp_to_ni56_dneut)
            integer, intent(in) :: i, ini56, ini58
            real(dp), intent(in) :: d_rni58gp_to_ni56_dneut
            integer :: neut, ni56, ni58
            real(dp) :: r, dr1, dr2
            neut = itab(ineut)
            ni56 = itab(ini56)
            ni58 = itab(ini58)
            r = y(ni58)
            dr1 = 1d0        
            if (n% rate_screened(i_rate,i) > tiny_rate) then
               dr2 = r*d_rni58gp_to_ni56_dneut/n% rate_screened(i_rate,i)
            else
               dr2 = 0
            end if
            call do_two_two(n, i, 1d0, ni58, 1d0, neut, 
     >         r, 3d0, neut, 1d0, ni56, ni58, dr1, neut, dr2, 
     >         deriv_flgs, category_factors, symbolic)       
            if (g% doing_derivs_timing) call done_select
         end subroutine do_gneut_gneut_reaction

         subroutine do_aneut_gneut_reaction(i, ife54, ini56, d_rfe54aneut_to_ni56_dneut)
            integer, intent(in) :: i, ife54, ini56
            real(dp), intent(in) :: d_rfe54aneut_to_ni56_dneut
            integer :: neut, he4, fe54, ni56
            real(dp) :: r, dr1, dr2
            ! since rate depends on abundance of neut,
            ! need to write this as a 3 to 2 reaction
            neut = itab(ineut)
            he4 = itab(ihe4)
            fe54 = itab(ife54)
            ni56 = itab(ini56)
            r = y(fe54) * y(he4)
            dr1 = y(he4)
            dr2 = y(fe54)
            if (n% rate_screened(i_rate,i) > tiny_rate) then
               dr3 = r*d_rfe54aneut_to_ni56_dneut/n% rate_screened(i_rate,i)
            else
               dr3 = 0
            end if
            call do_three_two(
     >         n, i, 1d0, fe54, 1d0, he4, 1d0, neut,
     >         r, 1d0, ni56, 3d0, neut, 
     >         fe54, dr1, he4, dr2, neut, dr3,
     >         deriv_flgs, category_factors, symbolic) 
            if (g% doing_derivs_timing) call done_select
         end subroutine do_aneut_gneut_reaction

         subroutine do_neutg_neuta_reaction(i, ife54, ini56, d_rni56neut_to_fe54_dneut)
            integer, intent(in) :: i, ife54, ini56
            real(dp), intent(in) :: d_rni56neut_to_fe54_dneut
            integer :: neut, he4, fe54, ni56
            real(dp) :: r, dr1, dr2
            neut = itab(ineut)
            he4 = itab(ihe4)
            fe54 = itab(ife54)
            ni56 = itab(ini56)
            r = y(ni56) * y(neut)**2
            dr1 = 2 * y(ni56)* y(neut)
            dr2 = y(neut)**2
            if (n% rate_screened(i_rate,i) > tiny_rate) then
               dr1 = dr1 + r*d_rni56neut_to_fe54_dneut/n% rate_screened(i_rate,i)
            end if
            call do_two_two(n, i, 1d0, ni56, 2d0, neut, 
     >         r, 1d0, he4, 1d0, fe54, neut, dr1, ni56, dr2, 
     >         deriv_flgs, category_factors, symbolic)
            if (g% doing_derivs_timing) call done_select
         end subroutine do_neutg_neuta_reaction

         subroutine do_aneut_neutg_reaction(i, ife54, ini58, d_rfe54aneut_to_ni58_dneut)
            integer, intent(in) :: i, ife54, ini58
            real(dp), intent(in) :: d_rfe54aneut_to_ni58_dneut
            integer :: neut, he4, fe54, ni58
            real(dp) :: r, dr1, dr2
            ! since rate depends on abundance of neut,
            ! need to write this as a 3 to 2 reaction
            neut = itab(ineut)
            he4 = itab(ihe4)
            fe54 = itab(ife54)
            ni58 = itab(ini58)
            r = y(fe54) * y(he4) * y(neut)
            dr1 = y(he4) * y(neut)
            dr2 = y(fe54) * y(neut)
            dr3 = y(fe54) * y(he4)
            if (n% rate_screened(i_rate,i) > tiny_rate) then
               dr3 = dr3 + r*d_rfe54aneut_to_ni58_dneut/n% rate_screened(i_rate,i)
            end if
            call do_three_two(
     >         n, i, 1d0, fe54, 1d0, he4, 1d0, neut,
     >         r, 1d0, ni58, 1d0, neut, 
     >         fe54, dr1, he4, dr2, neut, dr3,
     >         deriv_flgs, category_factors, symbolic)                   
            if (g% doing_derivs_timing) call done_select
         end subroutine do_aneut_neutg_reaction

         subroutine do_gneut_neuta_reaction(i, ife54, ini58, d_rni58gneut_to_fe54_dneut)
            integer, intent(in) :: i, ife54, ini58
            real(dp), intent(in) :: d_rni58gneut_to_fe54_dneut
            integer :: neut, he4, fe54, ni58
            real(dp) :: r, dr1, dr2
            neut = itab(ineut)
            he4 = itab(ihe4)
            fe54 = itab(ife54)
            ni58 = itab(ini58)
            r = y(ni58) * y(neut)
            dr1 = y(neut)
            dr2 = y(ni58)
            if (n% rate_screened(i_rate,i) > tiny_rate) then
               dr2 = dr2 + r*d_rni58gneut_to_fe54_dneut/n% rate_screened(i_rate,i)
            end if
            call do_two_three(n, i, 1d0, ni58, 1d0, neut, 
     >         r, 1d0, neut, 1d0, he4, 1d0, fe54, 
     >         ni58, dr1, neut, dr2, deriv_flgs, category_factors, symbolic)       
            if (g% doing_derivs_timing) call done_select
         end subroutine do_gneut_neuta_reaction
         
         subroutine do_pn_pn_reaction( ! cr56(p,n)mn56(p,n)fe56
     >         i, icr56, ife56, d_rcr56pn_to_fe56_dprot, d_rcr56pn_to_fe56_dneut)
            integer, intent(in) :: i, icr56, ife56
            real(dp), intent(in) :: d_rcr56pn_to_fe56_dprot, d_rcr56pn_to_fe56_dneut
            integer :: neut, prot, cr56, fe56
            real(dp) :: r, dr1, dr2
            neut = itab(ineut)
            prot = itab(iprot)
            cr56 = itab(icr56)
            fe56 = itab(ife56)
            r = y(cr56)*y(prot)**2
            dr1 = 2*y(cr56)*y(prot)
            dr2 = y(prot)**2
            if (n% rate_screened(i_rate,i) > tiny_rate) then
               dr1 = dr1 + r*d_rcr56pn_to_fe56_dprot/n% rate_screened(i_rate,i)
               ! ignore d_rcr56pn_to_fe56_dneut
            end if
            call do_two_two(n, i, 1d0, cr56, 2d0, prot, 
     >         r, 2d0, neut, 1d0, fe56, prot, dr1, cr56, dr2, 
     >         deriv_flgs, category_factors, symbolic)       
            if (g% doing_derivs_timing) call done_select
         end subroutine do_pn_pn_reaction

         subroutine do_np_np_reaction( ! fe56(n,p)mn56(n,p)cr56
     >         i, icr56, ife56, d_rfe56np_to_cr56_dprot, d_rfe56np_to_cr56_dneut)
            integer, intent(in) :: i, icr56, ife56
            real(dp), intent(in) :: d_rfe56np_to_cr56_dprot, d_rfe56np_to_cr56_dneut
            integer :: neut, prot, cr56, fe56
            real(dp) :: r, dr1, dr2
            neut = itab(ineut)
            prot = itab(iprot)
            cr56 = itab(icr56)
            fe56 = itab(ife56)
            r = y(fe56)*y(neut)**2
            dr1 = 2*y(fe56)*y(neut)
            dr2 = y(neut)**2
            if (n% rate_screened(i_rate,i) > tiny_rate) then
               dr1 = dr1 + r*d_rfe56np_to_cr56_dneut/n% rate_screened(i_rate,i)
               ! ignore d_rfe56np_to_cr56_dprot
            end if
            call do_two_two(n, i, 1d0, fe56, 2d0, neut, 
     >         r, 2d0, prot, 1d0, cr56, neut, dr1, fe56, dr2, 
     >         deriv_flgs, category_factors, symbolic)       
            if (g% doing_derivs_timing) call done_select
         end subroutine do_np_np_reaction

         
         subroutine do_FL_3alf(i) ! Fushiki and Lamb, Apj, 317, 368-388, 1987
            integer, intent(in) :: i
            integer :: he4, c12
            real(dp) :: UE, XHe4, YHe4, 
     >            FLeps_nuc, dFLeps_nuc_dT, dFLeps_nuc_dRho, r, drdT, drdRho, conv
            include 'formats.dek'
            he4 = itab(ihe4)
            c12 = itab(ic12)
            UE = abar/zbar
            YHe4 = y(he4)
            XHe4 = 4*YHe4
            if (YHe4 < 1d-50) then
               n% rate_raw(:,i) = 0
               n% rate_screened(:,i) = 0
               r = 0
               dr1 = 0
            else
               call eval_FL_epsnuc_3alf(
     >                  temp, den, XHe4, UE, FLeps_nuc, dFLeps_nuc_dT, dFLeps_nuc_dRho)
               conv = Qconv*n% reaction_Qs(ir)
               r = YHe4**3/6d0
               dr1 = 0.5d0*YHe4**2
               n% rate_raw(i_rate,i) = FLeps_nuc/r*rate_factors(i)/conv
               n% rate_raw(i_rate_dT,i) = dFLeps_nuc_dT/r*rate_factors(i)/conv
               n% rate_raw(i_rate_dRho,i) = dFLeps_nuc_dRho/r*rate_factors(i)/conv         
               n% rate_screened(:,i) = n% rate_raw(:,i)
            end if
            call do_one_one(n, i, 3d0, he4, r, 1d0, c12, he4, dr1, 
     >               deriv_flgs, category_factors, symbolic)
         end subroutine do_FL_3alf
         
         
         subroutine do_G05_1212_to_ne20(i) ! use Gasques, et al. Phys Review C, 72, 025806 (2005)
            integer, intent(in) :: i
            integer :: he4, c12, ne20
            real(dp) :: X12, Y12, eps, deps_dT, deps_dRho, r, drdT, drdRho, conv
            include 'formats.dek'
            he4 = itab(ihe4)
            c12 = itab(ic12)
            ne20 = itab(ine20)
            Y12 = y(c12)
            X12 = 12*Y12
            if (Y12 < 1d-50) then
               n% rate_raw(:,i) = 0
               n% rate_screened(:,i) = 0
               r = 0
               dr1 = 0
            else
               call eval_G05_epsnuc_CC(temp, den, X12, eps, deps_dT, deps_dRho)
               conv = Qconv*n% reaction_Qs(ir)
               r = Y12**2/2d0
               dr1 = Y12
               n% rate_raw(i_rate,i) = eps/r*rate_factors(i)/conv
               n% rate_raw(i_rate_dT,i) = deps_dT/r*rate_factors(i)/conv
               n% rate_raw(i_rate_dRho,i) = deps_dRho/r*rate_factors(i)/conv         
               n% rate_screened(:,i) = n% rate_raw(:,i)
            end if
            call do_one_two( n, i, 2d0, c12, r,
     >         1d0, ne20, 1d0, he4, c12, dr1, deriv_flgs, category_factors, symbolic)            
         end subroutine do_G05_1212_to_ne20

          
      end subroutine get1_derivs


      subroutine eval_ni56_ec_rate(temp, den, y56, ye, eta, 
     >      rate, d_rate_dlnT, d_rate_dlnRho, Q, Qneu, ierr)       
         use weak_lib, only: eval_weak_reaction_info
         real(dp), intent(in) :: temp, den, y56, ye, eta
         real(dp), intent(out) :: rate, d_rate_dlnT, d_rate_dlnRho, Q, Qneu
		   integer, intent(out) :: ierr
         call eval1_weak_rate(weak_rate_id_for_ni56_ec,
     >      temp, ye, den, eta, rate, d_rate_dlnT, d_rate_dlnRho, Q, Qneu, ierr)
      end subroutine eval_ni56_ec_rate


      subroutine eval1_weak_rate(id, temp, ye, rho, eta,
     >      rate_out, d_rate_dlnT, d_rate_dlnRho, Q_out, Qneu_out, ierr)       
         use weak_lib, only: eval_weak_reaction_info
		   integer, intent(in) :: id
         real(dp), intent(in) :: temp, ye, rho, eta
         real(dp), intent(out) :: 
     >      rate_out, d_rate_dlnT, d_rate_dlnRho, Q_out, Qneu_out
		   integer, intent(out) :: ierr
         
		   integer :: ids(1)
		   real(dp) :: T9, YeRho, d_eta_dlnT, d_eta_dlnRho
		   ! lambda = combined rate = 10**ldecay + 10**lcapture
		   ! Q and Qneu are for combined rate of beta decay and electron capture.
		   ! Q is total, so Q-Qneu is the actual thermal energy.
   	   ! note: lambdas include Ye Rho factors for electron captures.
   	   ! so treat the rates as if just beta decays
		   real(dp), dimension(1), target ::
     >      ldecay_a, d_ldecay_dT9_a, d_ldecay_dlYeRho_a,
     >      lcapture_a, d_lcapture_dT9_a, d_lcapture_dlYeRho_a,
     >      lneutrino_a, d_lneutrino_dT9_a, d_lneutrino_dlYeRho_a,
     >      lambda_a, dlambda_dlnT_a, dlambda_dlnRho_a,
     >      Q_a, dQ_dlnT_a, dQ_dlnRho_a,
     >      Qneu_a, dQneu_dlnT_a, dQneu_dlnRho_a
		   real(dp), dimension(:), pointer ::
     >      ldecay, d_ldecay_dT9, d_ldecay_dlYeRho,
     >      lcapture, d_lcapture_dT9, d_lcapture_dlYeRho,
     >      lneutrino, d_lneutrino_dT9, d_lneutrino_dlYeRho,
     >      lambda, dlambda_dlnT, dlambda_dlnRho,
     >      Q, dQ_dlnT, dQ_dlnRho,
     >      Qneu, dQneu_dlnT, dQneu_dlnRho
		   
	      ldecay => ldecay_a
	      d_ldecay_dT9 => d_ldecay_dT9_a
	      d_ldecay_dlYeRho => d_ldecay_dlYeRho_a
	      
	      lcapture => lcapture_a
	      d_lcapture_dT9 => d_lcapture_dT9_a
	      d_lcapture_dlYeRho => d_lcapture_dlYeRho_a
	      
	      lneutrino => lneutrino_a
	      d_lneutrino_dT9 => d_lneutrino_dT9_a
	      d_lneutrino_dlYeRho => d_lneutrino_dlYeRho_a
	      
	      lambda => lambda_a
	      dlambda_dlnT => dlambda_dlnT_a
	      dlambda_dlnRho => dlambda_dlnRho_a
	      
	      Q => Q_a
	      dQ_dlnT => dQ_dlnT_a
	      dQ_dlnRho => dQ_dlnRho_a
	      
	      Qneu => Qneu_a
	      dQneu_dlnT => dQneu_dlnT_a
	      dQneu_dlnRho => dQneu_dlnRho_a
	      
	      ids(1) = id
	      T9 = temp*1d-9
	      YeRho = ye*rho
	      d_eta_dlnT = 0
	      d_eta_dlnRho = 0
         
         ierr = 0
         call eval_weak_reaction_info(
     >      ids, T9, YeRho,
     >      eta, d_eta_dlnT, d_eta_dlnRho,
     >      ldecay, d_ldecay_dT9, d_ldecay_dlYeRho,
     >      lcapture, d_lcapture_dT9, d_lcapture_dlYeRho,
     >      lneutrino, d_lneutrino_dT9, d_lneutrino_dlYeRho,
     >      lambda, dlambda_dlnT, dlambda_dlnRho,
     >      Q, dQ_dlnT, dQ_dlnRho,
     >      Qneu, dQneu_dlnT, dQneu_dlnRho,
     >      ierr)
		    rate_out = lambda(1)
		    d_rate_dlnT = dlambda_dlnT(1)
		    d_rate_dlnRho = dlambda_dlnRho(1)
		    Q_out = Q(1) 
		    Qneu_out = Qneu(1) 
         
      end subroutine eval1_weak_rate


      end module net_derivs















