! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module net_burn
      use const_def
      use chem_def
      use net_def
      use rates_def, only: num_rvs
      
      implicit none
      
      integer, parameter :: i_burn_caller_id = 1
      integer, parameter :: i_handle = 2
      integer, parameter :: i_screening_mode = 3
      integer, parameter :: i_reuse_rates = 4
      integer, parameter :: i_net_lwork = 5
      integer, parameter :: i_sparse_format = 6
      integer, parameter :: i_ntimes = 7
      integer, parameter :: i_clip = 8
      
      integer, parameter :: burn_lipar = 8

      
      ! ugh... these are used in test/src/one_zone_burn -- move to net_def
      integer, parameter :: r_temp = 1
      integer, parameter :: r_lgT = 2
      integer, parameter :: r_rho = 3
      integer, parameter :: r_lgRho = 4
      integer, parameter :: r_eta = 5
      integer, parameter :: r_theta = 6
      integer, parameter :: r_time_net = 7
      integer, parameter :: r_prev_lgT = 8
      integer, parameter :: r_prev_lgRho = 9
      integer, parameter :: r_prev_eta = 10

      integer, parameter :: burn_lrpar = 10
      
      logical, parameter :: dbg = .false.
      
      
      contains
      
      
      subroutine burn_1_zone(
     >         handle, which_solver, num_isos, num_reactions, 
     >         t_start, t_end, starting_x, clip,
     >         ntimes, times, log10Ts_f1, log10Rhos_f1, etas_f1,
     >         dxdt_source_term, rate_factors, category_factors, reaction_Qs, reaction_neuQs,
     >         screening_mode, theta_e_for_graboske_et_al, 
     >         h, max_step_size, max_steps, rtol, atol, itol, 
     >         which_decsol, caller_id, solout, iout, ending_x, 
     >         nfcn, njac, nstep, naccpt, nrejct, time_doing_net, ierr)
         use num_def
         use num_lib 
         use mtx_lib
         use mtx_def
         use rates_def, only: rates_reaction_id_max
         use net_initialize, only: work_size
         
         integer, intent(in) :: handle
         integer, intent(in) :: num_isos
         integer, intent(in) :: num_reactions
         real(dp), intent(in) :: t_start, t_end, starting_x(:) ! (num_isos)
         logical, intent(in) :: clip ! if true, set negative x's to zero during burn.
         integer, intent(in) :: which_solver ! as defined in num_def.f
         integer, intent(in) :: ntimes ! ending time is times(num_times); starting time is 0
         real(dp), pointer, intent(in) :: times(:) ! (num_times) 
         real(dp), pointer, intent(in) :: log10Ts_f1(:) ! =(4,numtimes) interpolant for log10T(time)
         real(dp), pointer, intent(in) :: log10Rhos_f1(:) ! =(4,numtimes) interpolant for log10Rho(time)
         real(dp), pointer, intent(in) :: etas_f1(:) ! =(4,numtimes) interpolant for eta(time)
         real(dp), pointer, intent(in) :: dxdt_source_term(:) ! (num_isos)  or null if no source term.
         real(dp), intent(in) :: rate_factors(:) ! (num_reactions)
         real(dp), intent(in) :: category_factors(:) ! (num_categories)
         real(dp), pointer, intent(in) :: reaction_Qs(:) ! (rates_reaction_id_max)
         real(dp), pointer, intent(in) :: reaction_neuQs(:) ! (rates_reaction_id_max)
         integer, intent(in) :: screening_mode ! see screen_def
         real(dp), intent(in) :: theta_e_for_graboske_et_al
         real(dp), intent(inout) :: h 
         real(dp), intent(in) :: max_step_size ! maximal step size.
         integer, intent(in) :: max_steps ! maximal number of allowed steps.
         real(dp), intent(inout) :: rtol(*) ! relative error tolerance(s)
         real(dp), intent(inout) :: atol(*) ! absolute error tolerance(s)
         integer, intent(in) :: itol ! switch for rtol and atol
         integer, intent(in) :: which_decsol ! from mtx_def
         integer, intent(in) :: caller_id ! only provided for use by caller's solout routine
         interface ! subroutine called after each successful step
            include "num_solout.dek"
         end interface
         integer, intent(in)  :: iout ! switch for calling the subroutine solout:
         real(dp), intent(out) :: ending_x(:) ! (num_isos)
         integer, intent(out) :: nfcn    ! number of function evaluations
         integer, intent(out) :: njac    ! number of jacobian evaluations
         integer, intent(out) :: nstep   ! number of computed steps
         integer, intent(out) :: naccpt  ! number of accepted steps
         integer, intent(out) :: nrejct  ! number of rejected steps
         real(dp), intent(inout) :: time_doing_net
         integer, intent(out) :: ierr
         
         type (Net_General_Info), pointer :: g
         integer :: ijac, nzmax, isparse, mljac, mujac, imas, mlmas, mumas, 
     >         lrd, lid, lout, liwork, lwork, i, j, lrpar, lipar, idid, net_lwork
         integer, pointer :: ipar(:), iwork(:), ipar_decsol(:)
         real(dp), pointer :: rpar(:), work(:), rpar_decsol(:)
         real(dp) :: temp, rho, eta, lgT, lgRho
         
         include 'formats.dek'
         
         ierr = 0
         call get_net_ptr(handle, g, ierr)
         if (ierr /= 0) then
            write(*,*) 'invalid handle for net_work_size -- did you call alloc_net_handle?'
            return
         end if
         
         if (g% num_isos /= num_isos) then
            write(*,*) 'invalid num_isos', num_isos
            return
         end if
         
         if (g% num_reactions /= num_reactions) then
            write(*,*) 'invalid num_reactions', num_reactions
            return
         end if
         
         nfcn = 0
         njac = 0
         nstep = 0
         naccpt = 0
         nrejct = 0
         
         if (which_decsol == klu) then
            nzmax = num_isos**2 ! max number of non-zero entries
            isparse = klu_compressed_format
            call klu_dble_work_sizes(num_isos, nzmax, lrd, lid)
         else if (which_decsol == lapack) then
            nzmax = 0
            isparse = 0
            call lapack_work_sizes(num_isos, lrd, lid)
         else
            write(*,'(a,i4)') 'net 1 zone burn: unknown value for which_decsol', which_decsol
            stop 1
         end if
      
         ijac = 1
         mljac = num_isos ! square matrix
         mujac = num_isos

         imas = 0
         mlmas = 0
         mumas = 0        
         
         lout = 0
         
         net_lwork = work_size(g, num_isos, num_reactions)

         call isolve_work_sizes(num_isos, nzmax, imas, mljac, mujac, mlmas, mumas, liwork, lwork)

         lipar = burn_lipar     
             
         ! contents of rpar   size
         ! params             burn_lrpar
         ! source term        num_isos
         ! times              ntimes
         ! log10Ts_f          4*ntimes
         ! log10Rhos_f        4*ntimes
         ! etas_f             4*ntimes
         ! rate factors       num_reactions
         ! category factors   num_categories
         ! reaction eps_nuc   num_rvs*num_reactions
         ! reaction Qs        rates_reaction_id_max
         ! reaction Qneu's    rates_reaction_id_max
         ! work for eval_net  net_lwork
         ! raw rates          num_rvs*num_reactions
         ! screened rates     num_rvs*num_reactions
         lrpar = burn_lrpar + num_isos + 13*ntimes + num_reactions + num_categories +
     >         num_rvs*num_reactions + rates_reaction_id_max + rates_reaction_id_max + 
     >         net_lwork + num_rvs*num_reactions + num_rvs*num_reactions
         
         allocate(iwork(liwork), work(lwork), rpar(lrpar), ipar(lipar), 
     >         ipar_decsol(lid), rpar_decsol(lrd), stat=ierr)
         if (ierr /= 0) then
            write(*, *) 'allocate ierr', ierr
            stop 1
         end if
         
         rpar(r_theta) = theta_e_for_graboske_et_al
         rpar(r_time_net) = time_doing_net         
         rpar(r_lgT) = log10Ts_f1(1)
         rpar(r_temp) = 10**rpar(r_lgT)
         rpar(r_lgRho) = log10Rhos_f1(1)
         rpar(r_rho) = 10**rpar(r_lgRho)
         rpar(r_eta) = etas_f1(1)
         rpar(r_prev_lgT) = -1d99
         rpar(r_prev_lgRho) = -1d99
         rpar(r_prev_eta) = -1d99
         
         i = burn_lrpar
         ! source term
         if (associated(dxdt_source_term)) then
            do j=1,num_isos
               rpar(i+j) = dxdt_source_term(j)
            end do
         else
            rpar(i+1:i+num_isos) = 0
         end if
         i = i+num_isos
         ! times              ntimes
         rpar(i+1:i+ntimes) = times(1:ntimes); i = i+ntimes
         ! log10Ts_f          4*ntimes
         rpar(i+1:i+4*ntimes) = log10Ts_f1(1:4*ntimes); i = i+4*ntimes
         ! log10Rhos_f        4*ntimes
         rpar(i+1:i+4*ntimes) = log10Rhos_f1(1:4*ntimes); i = i+4*ntimes
         ! etas_f             4*ntimes
         rpar(i+1:i+4*ntimes) = etas_f1(1:4*ntimes); i = i+4*ntimes
         ! rate_factors
         rpar(i+1:i+num_reactions) = rate_factors(1:num_reactions)
         i = i+num_reactions
         ! category_factors
         rpar(i+1:i+num_categories) = category_factors(1:num_categories)
         i = i+num_categories
         ! room for reaction_eps_nuc
         i = i+num_rvs*num_reactions
         ! reaction_Qs
         rpar(i+1:i+rates_reaction_id_max) = reaction_Qs(1:rates_reaction_id_max)
         i = i+rates_reaction_id_max
         ! reaction_neuQs
         rpar(i+1:i+rates_reaction_id_max) = reaction_neuQs(1:rates_reaction_id_max)
         i = i+rates_reaction_id_max
         ! room for eval_net work
         i = i + net_lwork
         ! room for rate_raw and rate_screened
         i = i + 2*num_rvs*num_reactions
         
         if (i /= lrpar) then
            write(*,3) 'burn_1_zone: (i /= lrpar)', i, lrpar
            stop
         end if
         
         ipar(i_burn_caller_id) = caller_id
         
         ipar(i_handle) = handle
         ipar(i_reuse_rates) = 0
         ipar(i_screening_mode) = screening_mode
         ipar(i_net_lwork) = net_lwork
         ipar(i_sparse_format) = isparse
         ipar(i_ntimes) = ntimes
         if (clip) then
            ipar(i_clip) = 1
         else
            ipar(i_clip) = 0
         end if
         
         ending_x(:) = starting_x(:)
         iwork = 0
         work = 0
                  
         if (which_decsol == lapack) then
            call do_isolve(lapack_decsol, null_decsols, ierr)
         else if (which_decsol == klu) then
            call do_isolve(null_decsol, klu_dble_decsols, ierr)
         else
            write(*,*) 'unknown value for which_decsol', which_decsol
            stop 1
         end if
         if (ierr /= 0) then
            call dealloc
            return
         end if
      
         nfcn = nfcn + iwork(14)
         njac = njac + iwork(15)
         nstep = nstep + iwork(16)
         naccpt = naccpt + iwork(17)
         nrejct = nrejct + iwork(18)
         if (time_doing_net >= 0) time_doing_net = time_doing_net + rpar(r_time_net)
         
         if (.false. .and. ierr /= 0) then
            write(*, *) 'isolve failed: idid', idid
            write(*,1) 'rpar(r_temp)', rpar(r_temp)
            write(*,1) 'rpar(r_lgT)', rpar(r_lgT)
            write(*,1) 'rpar(r_rho)', rpar(r_rho)
            write(*,1) 'rpar(r_lgRho)', rpar(r_lgRho)
            write(*,1) 'rpar(r_eta)', rpar(r_eta)
            write(*,1) 'sum(starting_x)', sum(starting_x)
            write(*,1) 'maxval(starting_x)', maxval(starting_x)
            write(*,1) 'minval(starting_x)', minval(starting_x)
            stop 1
         end if
         
         call dealloc
         
         
         contains
         
         
         subroutine dealloc
            deallocate(iwork, work, rpar, ipar, ipar_decsol, rpar_decsol)
         end subroutine dealloc
         
         
         subroutine do_isolve(decsol, decsols, ierr)
            interface
               include "mtx_decsol.dek"
               include "mtx_decsols.dek"
            end interface
            integer, intent(out) :: ierr
            real(dp) :: t
            include 'formats.dek'
            t = t_start
            ierr = 0
            call isolve(
     >         which_solver, num_isos, burn_derivs, t, ending_x, t_end,  
     >         h, max_step_size, max_steps, 
     >         rtol, atol, itol, 
     >         burn_jacob, ijac, burn_sjac, nzmax, isparse, mljac, mujac, 
     >         null_mas, imas, mlmas, mumas, 
     >         solout, iout, 
     >         decsol, decsols, lrd, rpar_decsol, lid, ipar_decsol, 
     >         work, lwork, iwork, liwork, 
     >         lrpar, rpar, lipar, ipar, 
     >         lout, idid)
            if (idid < 0) ierr = -1
         end subroutine do_isolve


      end subroutine burn_1_zone


      subroutine burn_derivs(num_isos, t, x, f, lrpar, rpar, lipar, ipar, ierr)
         integer, intent(in) :: num_isos, lrpar, lipar
         real(dp), intent(in) :: t
         real(dp), intent(inout) :: x(num_isos)
         real(dp), intent(out) :: f(num_isos) ! dxdt
         integer, intent(inout), pointer :: ipar(:) ! (lipar)
         real(dp), intent(inout), pointer :: rpar(:) ! (lrpar)
         integer, intent(out) :: ierr
         integer, parameter :: ld_dfdx = 0
         real(dp) :: dfdx(ld_dfdx,num_isos)
         ierr = 0
         call burn_jacob(num_isos, t, x, f, dfdx, ld_dfdx, lrpar, rpar, lipar, ipar, ierr)
      end subroutine burn_derivs


      subroutine burn_jacob(num_isos, time, x, f, dfdx, ld_dfdx, lrpar, rpar, lipar, ipar, ierr)
         use chem_lib, only: basic_composition_info
         use net_eval, only: eval_net
         use rates_def, only: rates_reaction_id_max
         use interp_1d_lib, only: interp_value
         
         integer, intent(in) :: num_isos, ld_dfdx, lrpar, lipar
         real(dp), intent(in) :: time
         real(dp), intent(inout) :: x(num_isos)
         real(dp), intent(out) :: f(num_isos), dfdx(ld_dfdx, num_isos)
         integer, intent(inout), pointer :: ipar(:) ! (lipar)
         real(dp), intent(inout), pointer :: rpar(:) ! (lrpar)
         integer, intent(out) :: ierr
         
         integer :: handle, num_reactions
         real(dp) :: xh, xhe, abar, zbar, z2bar, ye, mass_correction, sumx,
     >         rho, lgRho, T, lgT
         real(dp) :: eta, d_eta_dlnT, d_eta_dlnRho
         real(dp), pointer :: category_factors(:)
         real(dp) :: eps_neu_total, eps_nuc
         real(dp) :: d_eps_nuc_dT
         real(dp) :: d_eps_nuc_dRho
         real(dp) :: d_eps_nuc_dx(num_isos) 
         real(dp) :: dxdt(num_isos)
         real(dp) :: d_dxdt_dRho(num_isos)
         real(dp) :: d_dxdt_dT(num_isos)
         real(dp) :: d_dxdt_dx(num_isos, num_isos)
         real(dp), pointer :: reaction_eps_nuc(:,:) ! (num_rvs, num_reactions)
         real(dp), target :: eps_nuc_categories(num_rvs, num_categories)
         real(dp), pointer :: rate_screened(:,:) ! (num_rvs, num_reactions)
         real(dp), pointer :: rate_raw(:,:) ! (num_rvs, num_reactions)
         real(dp), pointer :: rate_factors(:) ! (num_reactions)
         real(dp), pointer :: reaction_Qs(:) ! (rates_reaction_id_max)
         real(dp), pointer :: reaction_neuQs(:) ! (rates_reaction_id_max)
         logical :: reuse_given_rates
         integer :: screening_mode, net_lwork, i, j, species, time0, time1, clock_rate, ntimes
         real(dp), pointer :: dxdt_source_term(:) ! (num_isos)
         real(dp), pointer :: work(:) ! (lwork)
         real(dp), pointer :: times(:) ! (ntimes)
         real(dp), pointer :: log10Ts_f1(:) ! (4,ntimes)
         real(dp), pointer :: log10Rhos_f1(:) ! (4,ntimes)
         real(dp), pointer :: etas_f1(:) ! (4,ntimes)

         type (Net_General_Info), pointer :: g
         real(dp), pointer, dimension(:) :: actual_Qs, actual_neuQs
         logical, pointer :: from_weaklib(:)
         actual_Qs => null()
         actual_neuQs => null()
         from_weaklib => null()
         
         include 'formats.dek'
         
         ierr = 0
         f = 0
         dfdx = 0
         
         handle = ipar(i_handle)
         call get_net_ptr(handle, g, ierr)
         if (ierr /= 0) then
            write(*,*) 'invalid handle for net_get -- did you call alloc_net_handle?'
            return
         end if
         
         num_reactions = g% num_reactions

         net_lwork = ipar(i_net_lwork)
         ntimes = ipar(i_ntimes)

         i = burn_lrpar
         ! source term
         dxdt_source_term => rpar(i+1:i+num_isos)
         i = i+num_isos
         ! times              ntimes
         times => rpar(i+1:i+ntimes)
         i = i+ntimes
         ! log10Ts_f          4*ntimes
         log10Ts_f1(1:4*ntimes) => rpar(i+1:i+4*ntimes)
         i = i+4*ntimes
         ! log10Rhos_f        4*ntimes
         log10Rhos_f1(1:4*ntimes) => rpar(i+1:i+4*ntimes)
         i = i+4*ntimes
         ! etas_f             4*ntimes
         etas_f1(1:4*ntimes) => rpar(i+1:i+4*ntimes)
         i = i+4*ntimes
         ! rate_factors
         rate_factors => rpar(i+1:i+num_reactions)
         i = i+num_reactions
         ! category_factors
         category_factors => rpar(i+1:i+num_categories)
         i = i+num_categories
         ! room for reaction_eps_nuc
         call set_Aptr(reaction_eps_nuc, rpar(i+1:i+num_rvs*num_reactions), num_rvs, num_reactions)
         i = i+num_rvs*num_reactions
         ! reaction_Qs
         reaction_Qs => rpar(i+1:i+rates_reaction_id_max)
         i = i+rates_reaction_id_max
         ! reaction_neuQs
         reaction_neuQs => rpar(i+1:i+rates_reaction_id_max)
         i = i+rates_reaction_id_max
         ! for eval_net work
         work => rpar(i+1:i+net_lwork)
         i = i+net_lwork
         ! rate_raw
         call set_Aptr(rate_raw, rpar(i+1:i+num_rvs*num_reactions), num_rvs, num_reactions)
         i = i+num_rvs*num_reactions
         ! rate_screened
         call set_Aptr(rate_screened, rpar(i+1:i+num_rvs*num_reactions), num_rvs, num_reactions)
         i = i+num_rvs*num_reactions
         
         if (i /= lrpar) then
            write(*,3) 'burn_jacob (i /= lrpar)', i, lrpar
            write(*,2) 'burn_lrpar', burn_lrpar
            write(*,2) 'num_reactions', num_reactions
            write(*,2) 'num_categories', num_categories
            write(*,2) 'num_rvs', num_rvs
            write(*,2) 'net_lwork', net_lwork
            write(*,2) 'i', i
            write(*,2) 'lrpar', lrpar
            write(*,2) 'sum', burn_lrpar + num_reactions + num_categories + 
     >         3*num_rvs*num_reactions + 2*rates_reaction_id_max + net_lwork
            stop
         end if
         
         if (ntimes == 1) then
         
            lgT = log10Ts_f1(1)
            lgRho = log10Rhos_f1(1)
            eta = etas_f1(1)
            
         else
         
            call interp_value(times, ntimes, log10Ts_f1, time, lgT, ierr)
            if (ierr /= 0) then
               write(*,1) 'interp_value for lgT failed in burn_jacob for 1 zone burn', time
               return
            end if

            call interp_value(times, ntimes, log10Rhos_f1, time, lgRho, ierr)
            if (ierr /= 0) then
               write(*,1) 'interp_value for lgRho failed in burn_jacob for 1 zone burn', time
               return
            end if
         
            call interp_value(times, ntimes, etas_f1, time, eta, ierr)
            if (ierr /= 0) then
               write(*,1) 'interp_value for eta failed in burn_jacob for 1 zone burn', time
               return
            end if
            
         end if

         rho = 10**lgRho
         T = 10**lgT
         
         if (ipar(i_clip) /= 0) then
            do i=1,num_isos
               x(i) = max(0d0, min(1d0, x(i)))
            end do
         end if

         call basic_composition_info(
     >      num_isos, g% chem_id, x, xh, xhe, abar, zbar, z2bar, ye, mass_correction, sumx)
         
         if (rpar(r_time_net) >= 0) call system_clock(time0,clock_rate)  
         
         reuse_given_rates = (ipar(i_reuse_rates) /= 0) .and. 
     >         rpar(r_prev_lgT) == lgT .and. 
     >         rpar(r_prev_lgRho) == lgRho .and. 
     >         rpar(r_prev_eta) == eta


         call eval_net(
     >         g, num_isos, num_reactions, g% num_weaklib_rates,
     >         x, T, lgT, rho, lgRho,
     >         abar, zbar, z2bar, ye, eta, d_eta_dlnT, d_eta_dlnRho,
     >         rate_factors, category_factors,
     >         reaction_Qs, reaction_neuQs,
     >         eps_nuc, d_eps_nuc_dRho, d_eps_nuc_dT, d_eps_nuc_dx, 
     >         dxdt, d_dxdt_dRho, d_dxdt_dT, d_dxdt_dx, 
     >         ipar(i_screening_mode), rpar(r_theta), 
     >         rate_screened, rate_raw,
     >         reaction_eps_nuc, eps_nuc_categories, eps_neu_total,
     >         net_lwork, work, actual_Qs, actual_neuQs, from_weaklib, .false., ierr)
     
         if (.false. .and. ierr /= 0) then
            write(*,1) 'time', time
            write(*,1) 'T', T
            write(*,1) 'lgT', lgT
            write(*,1) 'rho', rho
            write(*,1) 'lgRho', lgRho
            write(*,1) 'eta', eta
            write(*,1) 'abar', abar
            write(*,1) 'zbar', zbar
            write(*,1) 'z2bar', z2bar
            write(*,1) 'ye', ye
            write(*,1) 'xh', xh
            write(*,1) 'xhe', xhe
            write(*,1) 'sumx', sumx
            write(*,1) 'sum(x)', sum(x)
            write(*,1) 'maxval(x)', maxval(x)
            write(*,1) 'minval(x)', minval(x)
            write(*,1) 'eps_nuc', eps_nuc
            write(*,1) 'eps_neu_total', eps_neu_total
            write(*,*)
            if (time > 0) stop
            !write(*,2) 'eval_net ierr', ierr
            !write(*,*) 'failed in burn_jacob'
            !stop '1 zone burn burn_jacob'
         end if

         if (rpar(r_time_net) >= 0) then
            call system_clock(time1,clock_rate)
            rpar(r_time_net) = rpar(r_time_net) + dble(time1 - time0) / clock_rate
         end if

         if (ntimes == 1) ipar(i_reuse_rates) = 1 ! okay if constant T and Rho
         rpar(r_prev_lgT) = lgT
         rpar(r_prev_lgRho) = lgRho
         rpar(r_prev_eta) = eta

         
      	do j = 1, num_isos
      	   f(j) = dxdt(j) + dxdt_source_term(j)
      	end do
         if (ld_dfdx > 0) dfdx = d_dxdt_dx
         
         
       	return
       	
       	species = num_isos
      	write(*,1) 'time', time
      	do j = 1, species
      	   write(*,2) 'x', j, x(j)
      	end do
         write(*,1) 'T', T
         write(*,1) 'lgT', lgT
         write(*,1) 'rho', rho
         write(*,1) 'lgRho', lgRho
         write(*,1) 'eta', eta
      	write(*,1) 'abar', abar
      	write(*,1) 'zbar', zbar
      	write(*,1) 'z2bar', z2bar
      	write(*,1) 'd_eta_dlnT', 0
      	write(*,1) 'd_eta_dlnRho', 0
      	
      	do j = 1, species
      	   write(*,2) 'dxdt', j, dxdt(j)
      	end do
      	do j = 1, species
      	   do i = 1, species
      	      write(*,3) 'd_dxdt_dx', i, j, d_dxdt_dx(i,j)
      	   end do
      	end do
      	do j = 1, species
      	   write(*,2) 'd_dxdt_dlnRho', j, d_dxdt_dRho(j)*rpar(r_rho)
      	end do
      	
      	write(*,1) 'eps_nuc', eps_nuc
      	do j = 1, species
      	   write(*,2) 'd_eps_nuc_dx', j, d_eps_nuc_dx(j)
      	end do
      	write(*,1) 'd_eps_nuc_dlnRho', d_eps_nuc_dRho*rpar(r_rho)
      	
      	do j = 1, species
      	   write(*,2) 'd_dxdt_dlnT', j, d_dxdt_dT(j)*rpar(r_temp), rpar(r_temp)
      	end do
      	write(*,1) 'd_eps_nuc_dlnT', d_eps_nuc_dT*rpar(r_temp)
      	
      	stop 'burn_jacob'         	
      	
        
         contains
   
         subroutine set_Aptr(Aptr, dest, n1, n2)
            real(dp), pointer :: Aptr(:, :)
            real(dp), target :: dest(n1, n2) ! reshape work section
            integer, intent(in) :: n1, n2
            Aptr => dest
         end subroutine set_Aptr
         
      end subroutine burn_jacob


      subroutine burn_sjac(n,time,y,f,nzmax,ia,ja,values,lrpar,rpar,lipar,ipar,ierr)  
         use mtx_lib, only: dense_to_sparse_with_diag
         use mtx_def
         integer, intent(in) :: n, nzmax, lrpar, lipar
         real(dp), intent(in) :: time
         real(dp), intent(inout) :: y(n)
         integer, intent(out) :: ia(n+1), ja(nzmax)
         real(dp), intent(out) :: f(n), values(nzmax)
         integer, intent(inout), pointer :: ipar(:) ! (lipar)
         real(dp), intent(inout), pointer :: rpar(:) ! (lrpar)
         integer, intent(out) :: ierr ! nonzero means terminate integration
         real(dp), pointer :: dfdv(:,:) ! (n,n)
         integer :: ld_dfdv, nz, i, j, cnt, nnz
      	include 'formats.dek'
      	!write(*,1) 'burn_sjac', x
      	ierr = 0
         ld_dfdv = n
         allocate(dfdv(n,n),stat=ierr)
         if (ierr /= 0) return
         call burn_jacob(n,time,y,f,dfdv,ld_dfdv,lrpar,rpar,lipar,ipar,ierr)
         if (ierr /= 0) then
            deallocate(dfdv)
            return
         end if
         ! remove entries with abs(value) < 1d-16
         cnt = 0; nnz = 0
         do i=1,n
            do j=1,n
               if (dfdv(i,j) /= 0) then
                  nnz = nnz + 1
                  if (abs(dfdv(i,j)) < 1d-16) then
                     cnt = cnt+1; dfdv(i,j) = 0
                  end if
               end if
            end do
         end do
         call dense_to_sparse_with_diag(ipar(i_sparse_format),n,n,dfdv,nzmax,nz,ia,ja,values,ierr)
         deallocate(dfdv)
      	!write(*,2) 'done burn_sjac: nz', nz
      end subroutine burn_sjac
      

      end module net_burn

