! ***********************************************************************
!
!   Copyright (C) 2011  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************



      module thomas_block_sparse
      use mtx_support
      use mod_klu_dble, only: num_klu_ipar_decsol, num_klu_rpar_decsol
      use const_def, only: dp
      use my_lapack95_dble      


      implicit none


      integer, parameter :: lid = num_klu_ipar_decsol
      integer, parameter :: lrd = num_klu_rpar_decsol
      

      type sparse_info
         integer :: sprs_nz ! number of nonzeros
         integer :: input_nz ! number of nonzeros in the input block on diagonal
         ! allocated
         integer, pointer :: ia(:) ! (mblk+1)
         integer, pointer :: ja(:) ! (mblk)
         real(dp), pointer :: values(:) ! (sprs_nz)
         real(dp) :: rpar_decsol(lrd)
         integer :: ipar_decsol(lid)
      end type sparse_info

      
      type thomas_info
         ! args
         integer :: nblk ! number of zones
         integer :: mblk ! number of variables per zone
         real(dp), pointer, dimension(:,:,:) :: lblk, dblk, ublk ! (mblk,mblk,nblk)
         ! allocated
         integer, pointer :: same_pattern_as(:) ! (nblk)
         type (sparse_info), dimension(:), pointer :: s ! (nblk)
         ! stats
         integer :: factor ! number of times did block factor 
         integer :: refactor ! number of times did block refactor 
         integer :: rejected_refactor ! number of times did block refactor but rejected it
         integer :: solve ! number of times did block solve 
         real(dp) :: min_rcond
         ! bookkeeping
         integer :: handle
         logical :: in_use
      end type thomas_info
      
      
      real(dp) :: clip_limit = 1d-20
      
      
      integer, parameter :: max_handles = 100
      type (thomas_info), target :: handles(max_handles)      
      logical :: have_initialized = .false.
      


      contains


      subroutine thomas_factor( &
            id, compressed_format, &
            decsols_nrhs, okay_to_refactor, can_eval_rcond, &
            mblk, nblk, lblk1, dblk1, ublk1, ipiv1, ierr)
         use mtx_def
         integer, intent(in) :: id, nblk, mblk, compressed_format
         real(dp), dimension(:), pointer, intent(inout) :: lblk1, dblk1, ublk1 ! =(nvar,nvar,nz)
         integer, pointer, intent(inout) :: ipiv1(:) ! =(nvar,nz)
         interface
            subroutine decsols_nrhs( &
                  iopt,nrhs,n,nz,ia,ja,values,b,lrd,rpar_decsol,lid,ipar_decsol,ierr)
               use const_def, only: dp
               integer, intent(in) :: iopt, nrhs, n, nz, lrd, lid
               integer, intent(inout) :: ia(:) ! (n+1)
               integer, intent(inout) :: ja(:) ! (nz)
               real(dp), intent(inout) :: values(:) ! (nz)
               real(dp), intent(inout) :: b(:) ! (n)
               real(dp), intent(inout), pointer :: rpar_decsol(:) ! (lrd)
               integer, intent(inout), pointer :: ipar_decsol(:) ! (lid)
               integer, intent(out) :: ierr
            end subroutine decsols_nrhs
         end interface
         logical, intent(in) :: okay_to_refactor, can_eval_rcond
         integer, intent(out) :: ierr

         integer :: ns, ns1, i, j, nrhs, &
            sprs_nzmax, sprs_nz, iounit, num_refactor, other, n
         real(dp), pointer :: dmat(:,:), umat(:,:), lmat(:,:)
         type (thomas_info), pointer :: t
         integer, pointer :: ia(:) ! (mblk+1)
         integer, pointer :: ja(:) ! (mblk)
         real(dp), pointer :: values(:) ! (sprs_nzmax)
         real(dp) :: b(1), rcond, rcond_prev
         logical :: refactor
         character (len=256) :: filename

         real(dp), pointer :: rpar(:), l1(:), lblk(:,:,:), dblk(:,:,:), ublk(:,:,:)
         integer, pointer :: ipiv(:,:)
         integer, pointer :: ipar(:)
         
         include 'formats.dek'

         ierr = 0
         ipiv(1:mblk,1:nblk) => ipiv1(1:mblk*nblk)
         lblk(1:mblk,1:mblk,1:nblk) => lblk1(1:mblk*mblk*nblk)
         dblk(1:mblk,1:mblk,1:nblk) => dblk1(1:mblk*mblk*nblk)
         ublk(1:mblk,1:mblk,1:nblk) => ublk1(1:mblk*mblk*nblk)
         
         call get_ptr(id,t,ierr)
         if (ierr /= 0) return
         
         t% lblk => lblk
         t% dblk => dblk
         t% ublk => ublk
         t% mblk = mblk
         t% nblk = nblk
         t% factor = 0
         t% refactor = 0
         t% rejected_refactor = 0
         t% solve = 0
         t% min_rcond = 1d99
         
         allocate(t% s(nblk), t% same_pattern_as(nblk))
         t% s(1:nblk)% sprs_nz = -1 ! mark as unallocated

         do ns = nblk, 1, -1
            t% s(ns)% input_nz = 0
            do j=1,mblk
               do i=1,mblk
                  if (abs(dblk(i,j,ns)) >= clip_limit) &
                     t% s(ns)% input_nz = t% s(ns)% input_nz + 1
                  if (abs(ublk(i,j,ns)) >= clip_limit) &
                     t% s(ns)% input_nz = t% s(ns)% input_nz + 1
                  if (abs(lblk(i,j,ns)) >= clip_limit) &
                     t% s(ns)% input_nz = t% s(ns)% input_nz + 1
               end do
            end do
         end do
         
         num_refactor = 0
         do ns = nblk, 1, -1
            dmat => dblk(:,:,ns)
            
            sprs_nzmax = 0
            do j=1,mblk
               do i=1,mblk
                  if (i == j) then ! don't clip diagonals
                     sprs_nzmax = sprs_nzmax + 1
                     cycle
                  end if
                  if (abs(dmat(i,j)) < clip_limit) dmat(i,j) = 0d0
                  if (dmat(i,j) == 0) cycle
                  sprs_nzmax = sprs_nzmax + 1
               end do
            end do
            
            allocate(t% s(ns)% ia(mblk+1), t% s(ns)% ja(sprs_nzmax), t% s(ns)% values(sprs_nzmax))
            values => t% s(ns)% values
            
            if (compressed_format == compressed_column_sparse) then
               call do_dense_to_column_sparse( &
                  mblk, mblk, dmat, sprs_nzmax, sprs_nz, &
                  t% s(ns)% ia, t% s(ns)% ja, values, .true., ierr)
            else if (compressed_format == compressed_row_sparse) then
               call do_dense_to_row_sparse( &
                  mblk, mblk, dmat, sprs_nzmax, sprs_nz, &
                  t% s(ns)% ia, t% s(ns)% ja, values, .true., ierr)
            else if (compressed_format == compressed_col_sparse_0_based) then
               call do_dense_to_col_sparse_0_based( &
                  mblk, mblk, dmat, sprs_nzmax, sprs_nz, &
                  t% s(ns)% ia, t% s(ns)% ja, values, .true., ierr)
            else if (compressed_format == compressed_row_sparse_0_based) then
               call do_dense_to_row_sparse_0_based( &
                  mblk, mblk, dmat, sprs_nzmax, sprs_nz, &
                  t% s(ns)% ia, t% s(ns)% ja, values, .true., ierr)
            else
               write(*,*) 'bad value for compressed_format', compressed_format
               ierr = -1
            end if
            if (ierr /= 0) then
               write(*,*) 'failed in converting from dense to sparse', ns, compressed_format
               exit
            end if
            if (sprs_nz /= sprs_nzmax) then
               write(*,*) &
                  'failed in converting from dense to sparse: bad sprs_nz', ns, sprs_nzmax, sprs_nz
               exit
            end if
            t% s(ns)% sprs_nz = sprs_nz
            
            call set_same_pattern_as(ns)
            other = t% same_pattern_as(ns)
            
            if (other /= 0) then
               n = other
               t% refactor = t% refactor + 1
               ipar => t% s(n)% ipar_decsol
               rpar => t% s(n)% rpar_decsol
               call decsols_nrhs( & ! refactor with new values
                  3, mblk, mblk, sprs_nz, &
                  t% s(n)% ia, t% s(n)% ja, t% s(ns)% values, b, &
                  lrd, rpar, lid, ipar, ierr)
               if (ierr /= 0) then
                  !write(*,3) 'thomas block sparse bad refactor', ns, n
                  ierr = 0
                  other = 0
                  t% same_pattern_as(ns) = 0
               else if (can_eval_rcond) then
                  call decsols_nrhs( &
                     4, mblk, mblk, sprs_nz, &
                     t% s(n)% ia, t% s(n)% ja, t% s(ns)% values, b, &
                     lrd, rpar, lid, ipar, ierr)
                  rcond = rpar(1)
                  if (rcond < t% min_rcond) t% min_rcond = rcond
                  if (ierr /= 0 .or. rcond < 1d-14) then
                     !write(*,3) 'thomas block sparse bad rcond', ns, n, rcond
                     ierr = 0
                     other = 0
                     t% same_pattern_as(ns) = 0
                     t% rejected_refactor = t% rejected_refactor + 1
                  else if (rcond < t% min_rcond) then
                     t% min_rcond = rcond
                  end if
               end if
            end if

            if (ierr == 0 .and. other == 0) then
               t% factor = t% factor + 1
               ipar => t% s(ns)% ipar_decsol
               rpar => t% s(ns)% rpar_decsol
               call decsols_nrhs( & ! factor
                  0, mblk, mblk, sprs_nz, &
                  t% s(ns)% ia, t% s(ns)% ja, t% s(ns)% values, b, &
                  lrd, rpar, lid, ipar, ierr)
               if (can_eval_rcond) then
                  call decsols_nrhs( & ! rcond
                     4, mblk, mblk, sprs_nz, &
                     t% s(ns)% ia, t% s(ns)% ja, t% s(ns)% values, b, &
                     lrd, rpar, lid, ipar, ierr)
                  rcond = rpar(1)
                  if (ierr /= 0 .and. rcond < t% min_rcond) t% min_rcond = rcond
               end if
            end if            
            
            if (ierr /= 0) then
               exit
               !          < 0:  if info = -i, the i-th argument had an illegal value
               !          > 0:  if info = i, u(i,i) is exactly zero. the factorization
               write (6, '(x,a,i4)') 'error factoring matrix in thomas_factor: block = ', ns
               if (ierr < 0) write (6,'(i4, a)') ierr, 'th argument has illegal value'
               if (ierr > 0) write (6,'(i4, a)') ierr, 'th diagonal factor exactly zero'
               exit
            end if
            
            if (ns == 1) exit
            lmat => lblk(:,:,ns)
            
            n = other
            if (n == 0) n = ns
            ! replace lmat by dmat^-1*lmat            
            t% solve = t% solve + 1
            ipar => t% s(n)% ipar_decsol
            rpar => t% s(n)% rpar_decsol
            l1(1:mblk*mblk) => lblk1(1+mblk*mblk*(ns-1):mblk*mblk*ns)
            call decsols_nrhs( & ! solve
               1, mblk, mblk, sprs_nz, &
               t% s(n)% ia, t% s(n)% ja, t% s(ns)% values, l1, &
               lrd, rpar, lid, ipar, ierr)            
            if (ierr /= 0) exit
            ns1 = ns-1 
            umat => ublk(:,:,ns1)
            dmat => dblk(:,:,ns1)
            ! dmat = dmat - umat*lmat
            call my_gemm(mblk,mblk,mblk,umat,mblk,lmat,mblk,dmat,mblk)
            
         end do
         
         
         contains
         
         
         subroutine set_same_pattern_as(ns)
            integer, intent(in) :: ns
            integer :: other, nsp1
            t% same_pattern_as(ns) = 0
            if (.not. okay_to_refactor) return
            if (ns == nblk) return
            nsp1 = ns+1
            other = t% same_pattern_as(nsp1)
            if (other == 0) other = nsp1
            if (t% s(ns)% sprs_nz /= t% s(nsp1)% sprs_nz) return
            if (any(t% s(ns)% ia /= t% s(nsp1)% ia)) return
            if (any(t% s(ns)% ja /= t% s(nsp1)% ja)) return
            t% same_pattern_as(ns) = other
         end subroutine set_same_pattern_as

         
      end subroutine thomas_factor

      
      subroutine thomas_solve( &
            id, decsols_nrhs, okay_to_refactor, can_eval_rcond, &
            mblk, nblk, lblk1, dblk1, ublk1, brhs1, nrhs, ipiv1, ierr)
         integer, intent(in) :: id, nblk, mblk
         interface
            subroutine decsols_nrhs( &
                  iopt,nrhs,n,nz,ia,ja,values,b,lrd,rpar_decsol,lid,ipar_decsol,ierr)
               use const_def, only: dp
               integer, intent(in) :: iopt, nrhs, n, nz, lrd, lid
               integer, intent(inout) :: ia(:) ! (n+1)
               integer, intent(inout) :: ja(:) ! (nz)
               real(dp), intent(inout) :: values(:) ! (nz)
               real(dp), intent(inout) :: b(:) ! (n)
               real(dp), intent(inout), pointer :: rpar_decsol(:) ! (lrd)
               integer, intent(inout), pointer :: ipar_decsol(:) ! (lid)
               integer, intent(out) :: ierr
            end subroutine decsols_nrhs
         end interface
         logical, intent(in) :: okay_to_refactor, can_eval_rcond
         real(dp), dimension(:), pointer, intent(inout) :: lblk1, dblk1, ublk1 ! =(nvar,nvar,nz)
         integer, pointer, intent(inout) :: ipiv1(:) ! =(nvar,nz)
         integer, intent(in) :: nrhs
         real(fltp), pointer, intent(inout) :: brhs1(:) ! =(mblk,nrhs,nblk)
         integer, intent(out) :: ierr

         integer :: ns, ns1, i, j, k, other
         integer, pointer :: ipivot(:), iwork(:)
         real(dp), pointer, dimension(:,:) :: dmat, umat, lmat, bptr, x1
         type (thomas_info), pointer :: t

         real(fltp), pointer :: lblk(:,:,:), dblk(:,:,:), ublk(:,:,:)
         integer, pointer :: ipiv(:,:)
         real(dp), pointer :: brhs(:,:,:)
         
         include 'formats.dek'
         
         ierr = 0
         ipiv(1:mblk,1:nblk) => ipiv1(1:mblk*nblk)
         lblk(1:mblk,1:mblk,1:nblk) => lblk1(1:mblk*mblk*nblk)
         dblk(1:mblk,1:mblk,1:nblk) => dblk1(1:mblk*mblk*nblk)
         ublk(1:mblk,1:mblk,1:nblk) => ublk1(1:mblk*mblk*nblk)
         brhs(1:mblk,1:nrhs,1:nblk) => brhs1(1:mblk*nrhs*nblk)

         call get_ptr(id,t,ierr)
         if (ierr /= 0) return
         
         do ns = nblk, 1, -1
            bptr => brhs(:,:,ns)
            ! solve dmat*x = bptr with x returned in bptr
            call refactor_and_solve(ns, ierr)
            if (ierr /= 0) exit
            if (ns == 1) exit
            ns1 = ns-1
            umat => ublk(:,:,ns1)
            bptr => brhs(:,:,ns1)
            x1 => brhs(:,:,ns)
            ! bptr = bptr - umat*x1
            if (nrhs == 1) then
               call my_gemv(mblk,mblk,umat,mblk,x1(:,1),bptr(:,1))
            else
               call my_gemm(mblk,nrhs,mblk,umat,mblk,x1,mblk,bptr,mblk)
            end if
         end do

         if (ierr /= 0) return

         do ns = 2, nblk
            ns1 = ns-1
            bptr => brhs(:,:,ns)
            x1 => brhs(:,:,ns1)
            lmat => lblk(:,:,ns)
            ! bptr = bptr - lmat*x1
            if (nrhs == 1) then
               call my_gemv(mblk,mblk,lmat,mblk,x1(:,1),bptr(:,1))
            else
               call my_gemm(mblk,nrhs,mblk,lmat,mblk,x1,mblk,bptr,mblk)
            end if
         end do
         
         
         contains
         
         
         subroutine refactor_and_solve(ns, ierr)
            integer, intent(in) :: ns
            integer, intent(out) :: ierr
            integer :: n
            integer, pointer :: ipar(:)
            real(dp), pointer :: rpar(:), b(:)
            include 'formats.dek'
            n = t% same_pattern_as(ns)
            if (n == 0) n = ns
            ierr = 0
            ipar => t% s(n)% ipar_decsol
            rpar => t% s(n)% rpar_decsol
            b(1:mblk*nrhs) => brhs1(1+(ns-1)*mblk*nrhs:ns*mblk*nrhs)
            if (okay_to_refactor) then
               call decsols_nrhs( & ! refactor
                  3, nrhs, mblk, t% s(n)% sprs_nz, &
                  t% s(n)% ia, t% s(n)% ja, t% s(ns)% values, b, &
                  lrd, rpar, lid, ipar, ierr)
               if (ierr /= 0) return
            end if           
            call decsols_nrhs( & ! solve
               1, nrhs, mblk, t% s(n)% sprs_nz, &
               t% s(n)% ia, t% s(n)% ja, t% s(ns)% values, b, &
               lrd, rpar, lid, ipar, ierr)
         end subroutine refactor_and_solve


      end subroutine thomas_solve

      
      subroutine thomas_solve1( &
            id, &
            decsols_nrhs, okay_to_refactor, can_eval_rcond, &
            mblk, nblk, lblk1, dblk1, ublk1, b1rhs1, ipiv1, ierr) ! nrhs = 1
         integer, intent(in) :: id, nblk, mblk
         interface
            subroutine decsols_nrhs( &
                  iopt,nrhs,n,nz,ia,ja,values,b,lrd,rpar_decsol,lid,ipar_decsol,ierr)
               use const_def, only: dp
               integer, intent(in) :: iopt, nrhs, n, nz, lrd, lid
               integer, intent(inout) :: ia(:) ! (n+1)
               integer, intent(inout) :: ja(:) ! (nz)
               real(dp), intent(inout) :: values(:) ! (nz)
               real(dp), intent(inout) :: b(:) ! (n)
               real(dp), intent(inout), pointer :: rpar_decsol(:) ! (lrd)
               integer, intent(inout), pointer :: ipar_decsol(:) ! (lid)
               integer, intent(out) :: ierr
            end subroutine decsols_nrhs
         end interface
         logical, intent(in) :: okay_to_refactor, can_eval_rcond

         real(fltp), dimension(:), pointer, intent(inout) :: &
            lblk1, dblk1, ublk1 ! =(nvar,nvar,nz)
            ! row(i) of mtx has lblk(:,:,i), dblk(:,:,i), ublk(:,:,i)
            ! lblk(:,:,1) is not used; ublk(:,:,nz) is not used.
         real(fltp), pointer, intent(inout) :: b1rhs1(:) ! =(mblk,nblk)
         integer, pointer, intent(inout) :: ipiv1(:) ! =(mblk,nblk)

         integer, intent(out) :: ierr
         
         call thomas_solve(id, &
            decsols_nrhs, okay_to_refactor, can_eval_rcond, &
            mblk, nblk, lblk1, dblk1, ublk1, b1rhs1, 1, ipiv1, ierr)
            
      end subroutine thomas_solve1

      
      subroutine thomas_stats( &
            id, nblk, factor, refactor, rejected_refactor, solve, sum_nz, max_nz, min_rcond, ierr)
         integer, intent(in) :: id
         integer, intent(out) :: nblk ! number of blocks 
         integer, intent(out) :: factor ! number of times did block factor 
         integer, intent(out) :: refactor ! number of times did block refactor 
         integer, intent(out) :: rejected_refactor ! number of times rejected block refactor
         integer, intent(out) :: solve ! number of times did block solve 
         integer, intent(out) :: sum_nz ! sum of non-zeros in factored blocks 
         integer, intent(out) :: max_nz ! max number of non-zeros in a factored block 
         real(dp), intent(out) :: min_rcond
         integer, intent(out) :: ierr
         type (thomas_info), pointer :: t
         integer :: ns, nnz
         ierr = 0
         call get_ptr(id,t,ierr)
         if (ierr /= 0) return
         nblk = t% nblk
         factor = t% factor
         refactor = t% refactor
         rejected_refactor = t% rejected_refactor
         solve = t% solve
         min_rcond = t% min_rcond
         max_nz = 0; sum_nz = 0
         do ns = 1, nblk
            !nnz = t% s(ns)% sprs_nz
            nnz = t% s(ns)% input_nz
            if (nnz > max_nz) max_nz = nnz
            sum_nz = sum_nz + nnz
         end do
      end subroutine thomas_stats
      
      
      subroutine thomas_dealloc(id,decsols_nrhs,ierr)
         integer, intent(in) :: id
         interface
            subroutine decsols_nrhs( &
                  iopt,nrhs,n,nz,ia,ja,values,b,lrd,rpar_decsol,lid,ipar_decsol,ierr)
               use const_def, only: dp
               integer, intent(in) :: iopt, nrhs, n, nz, lrd, lid
               integer, intent(inout) :: ia(:) ! (n+1)
               integer, intent(inout) :: ja(:) ! (nz)
               real(dp), intent(inout) :: values(:) ! (nz)
               real(dp), intent(inout) :: b(:) ! (n)
               real(dp), intent(inout), pointer :: rpar_decsol(:) ! (lrd)
               integer, intent(inout), pointer :: ipar_decsol(:) ! (lid)
               integer, intent(out) :: ierr
            end subroutine decsols_nrhs
         end interface
         integer, intent(out) :: ierr
         integer :: i
         type (thomas_info), pointer :: t
         integer :: ns, mblk, op_err
         real(dp) :: b(1)         
         integer, pointer :: ipar(:)
         real(dp), pointer :: rpar(:)
         include 'formats.dek'         
         ierr = 0
         call get_ptr(id,t,ierr)
         if (ierr /= 0) then
            write(*,*) 'thomas_dealloc invalid id', id
            return
         end if
         if (t% nblk > 0) then
            mblk = t% mblk
            do ns = 1, t% nblk
               op_err = 0
               if (t% s(ns)% sprs_nz < 0) cycle ! never allocated
               if (t% same_pattern_as(ns) == 0) then
                  ipar => t% s(ns)% ipar_decsol
                  rpar => t% s(ns)% rpar_decsol
                  call decsols_nrhs( & ! dealloc
                     2, mblk, mblk, t% s(ns)% sprs_nz, &
                     t% s(ns)% ia, t% s(ns)% ja, t% s(ns)% values, b, &
                     lrd, rpar, lid, ipar, op_err)
                  if (op_err /= 0) ierr = op_err
               end if
               deallocate(t% s(ns)% ia, t% s(ns)% ja, t% s(ns)% values)
            end do        
            deallocate(t% s, t% same_pattern_as)
         end if
         call do_free_handle(id)         
      end subroutine thomas_dealloc

      
      subroutine thomas_block_init
         integer :: i
         if (have_initialized) return
!$omp critical (init_thomas_block_sparse)
         if (.not. have_initialized) then
            do i=1,max_handles
               handles(i)% handle = i
               handles(i)% in_use = .false.
            end do
            have_initialized = .true.
         end if
!$omp end critical (init_thomas_block_sparse)
      end subroutine thomas_block_init

      
      integer function thomas_handle(ierr)
         integer, intent(out) :: ierr
         integer :: i
         type (thomas_info), pointer :: t
         if (.not. have_initialized) call thomas_block_init
         ierr = 0
         thomas_handle = -1
!$omp critical (alloc_thomas_sparse_handle)
         do i = 1, max_handles
            if (.not. handles(i)% in_use) then
               handles(i)% in_use = .true.
               thomas_handle = i
               exit
            end if
         end do
         !write(*,*) 'alloc thomas handle', thomas_handle
!$omp end critical (alloc_thomas_sparse_handle)
         if (thomas_handle == -1) then
            ierr = -1
            return
         end if
         if (handles(thomas_handle)% handle /= thomas_handle) then
            ierr = -1
            return
         end if
         t => handles(thomas_handle)
         t% nblk = -1 ! to indicate hasn't been setup yet
      end function thomas_handle
            
      
      subroutine do_free_handle(handle)
         integer, intent(in) :: handle
         type (thomas_info), pointer :: d
         !write(*,*) 'free thomas handle', handle
         if (handle >= 1 .and. handle <= max_handles) then
            d => handles(handle)
            handles(handle)% in_use = .false.
         end if
      end subroutine do_free_handle
      

      subroutine get_ptr(handle,t,ierr)
         integer, intent(in) :: handle
         type (thomas_info), pointer :: t
         integer, intent(out):: ierr
         ierr = 0
         if (handle < 1 .or. handle > max_handles) then
            ierr = -1
            return
         end if
         t => handles(handle)
      end subroutine get_ptr


      end module thomas_block_sparse




      module thomas_block_klu
      
      use thomas_block_sparse, only: &
         thomas_handle, thomas_factor, thomas_solve, thomas_dealloc, thomas_stats
      use mod_klu_dble, only: do_klu_decsols_nrhs_0_based
      use const_def, only: dp
      
      implicit none
      
      logical, parameter :: okay_to_refactor = .true.
      logical, parameter :: can_eval_rcond = .true.
      
      contains
      

      integer function thomas_klu_handle(ierr)
         integer, intent(out) :: ierr
         thomas_klu_handle = thomas_handle(ierr)
      end function thomas_klu_handle


      subroutine thomas_klu_factor(id, nvar, nz, lblk, dblk, ublk, ipiv, &
               lrd, rpar_decsol, lid, ipar_decsol, ierr)
         use mtx_def, only: compressed_col_sparse_0_based
         integer, intent(in) :: id, nvar, nz, lrd, lid
         real(dp), dimension(:), pointer, intent(inout) :: lblk, dblk, ublk ! =(nvar,nvar,nz)
         integer, pointer, intent(inout) :: ipiv(:) ! =(nvar,nz)
         real(dp), pointer, intent(inout) :: rpar_decsol(:) ! (lrd)
         integer, pointer, intent(inout) :: ipar_decsol(:) ! (lid)
         integer, intent(out) :: ierr
         call thomas_factor( &
            id, compressed_col_sparse_0_based, do_klu_decsols_nrhs_0_based, &
            okay_to_refactor, can_eval_rcond, nvar, nz, lblk, dblk, ublk, ipiv, ierr)
      end subroutine thomas_klu_factor
      
      
      subroutine thomas_klu_solve(id, nvar, nz, lblk, dblk, ublk, brhs, nrhs, ipiv, &
               lrd, rpar_decsol, lid, ipar_decsol, ierr)
         integer, intent(in) :: id, nvar, nz, lrd, lid
         real(dp), dimension(:), pointer, intent(inout) :: lblk, dblk, ublk ! =(nvar,nvar,nz)
         real(dp), pointer, intent(inout)  :: brhs(:) ! =(nvar,nz)     
         integer, intent(in) :: nrhs
         integer, pointer, intent(inout) :: ipiv(:) ! =(nvar,nz)
         real(dp), pointer, intent(inout) :: rpar_decsol(:) ! (lrd)
         integer, pointer, intent(inout) :: ipar_decsol(:) ! (lid)
         integer, intent(out) :: ierr
         call thomas_solve(id, do_klu_decsols_nrhs_0_based, &
            okay_to_refactor, can_eval_rcond, nvar, nz, lblk, dblk, ublk, brhs, nrhs, ipiv, ierr)
      end subroutine thomas_klu_solve
      
      
      subroutine thomas_klu_stats( &
            id, nblk, factor, refactor, rejected_refactor, solve, sum_nz, max_nz, min_rcond, ierr)
         integer, intent(in) :: id
         integer, intent(out) :: nblk ! number of blocks 
         integer, intent(out) :: factor ! number of times did block factor 
         integer, intent(out) :: refactor ! number of times did block refactor 
         integer, intent(out) :: rejected_refactor ! number of times rejected block refactor
         integer, intent(out) :: solve ! number of times did block solve 
         integer, intent(out) :: sum_nz ! sum of non-zeros in factored blocks 
         integer, intent(out) :: max_nz ! max number of non-zeros in a factored block 
         real(dp), intent(out) :: min_rcond
         integer, intent(out) :: ierr
         call thomas_stats( &
            id, nblk, factor, refactor, rejected_refactor, solve, sum_nz, max_nz, min_rcond, ierr)
      end subroutine thomas_klu_stats
      
      
      subroutine thomas_klu_dealloc(id,ierr)
         integer, intent(in) :: id
         integer, intent(out) :: ierr
         call thomas_dealloc(id, do_klu_decsols_nrhs_0_based, ierr)
      end subroutine thomas_klu_dealloc


      end module thomas_block_klu
