! ***********************************************************************
!
!   Copyright (C) 2009  Aaron Dotter, Bill Paxton
!
!   This file is part of MESA.
!
!   MESA is free software; you can redistribute it and/or modify
!   it under the terms of the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License, or
!   (at your option) any later version.
!
!   MESA is distributed in the hope that it will be useful, 
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!   GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************
      program sample_kap
      use kap_lib
      use kap_def
      use chem_lib
      use chem_def
      use const_def
      use const_lib
      
      implicit none
      !this program demonstrates how to use mesa/kap in a stellar structure code
      !it reads in a mesa/star model of an AGB star so that it makes full use
      !of mesa/kap's capabilities. modules kap_lib and kap_def contain access to
      !all of the pieces required to set up and use mesa/kap.

      character(len=256) :: kappa_file_prefix, kappa_CO_prefix, &
         kappa_lowT_prefix, model_file, output_file
      logical :: use_cache, cubic_X_interpolation, &
         cubic_Z_interpolation, include_electron_conduction
      integer :: handle, i, ii, iounit, Npts, Nspec, omp_err
      integer, parameter :: maxpts = 2000, maxspec = 31
      integer :: ierr, omp_get_thread_num !for OpenMP
      
      integer, parameter :: h1 = 1
      integer, parameter :: h2 = 2
      integer, parameter :: he3 = 3
      integer, parameter :: he4 = 4
      integer, parameter :: li7 = 5
      integer, parameter :: be7 = 6
      integer, parameter :: b8 = 7
      integer, parameter :: c12 = 8
      integer, parameter :: c13 = 9
      integer, parameter :: n13 = 10
      integer, parameter :: n14 = 11
      integer, parameter :: n15 = 12
      integer, parameter :: o16 = 13
      integer, parameter :: o17 = 14
      integer, parameter :: o18 = 15
      integer, parameter :: f19 = 16
      integer, parameter :: ne20 = 17
      integer, parameter :: ne21 = 18
      integer, parameter :: ne22 = 19
      integer, parameter :: na22 = 20
      integer, parameter :: na23 = 21
      integer, parameter :: mg24 = 22
      integer, parameter :: mg25 = 23
      integer, parameter :: mg26 = 24
      integer, parameter :: al26 = 25
      integer, parameter :: al27 = 26
      integer, parameter :: si28 = 27
      integer, parameter :: si29 = 28
      integer, parameter :: si30 = 29
      integer, parameter :: p31 = 30
      integer, parameter :: s32 = 31
      
      real(dp) :: Mstar, Xc, Xn, Xo, Xne, xheavy, &
         xc_base, xn_base, xo_base, xne_base, &
         zbar, frac_Type2, lnfree_e, d_lnfree_e_dlnRho, d_lnfree_e_dlnT
      real(dp) :: Z_init, Z, dXC, dXO, type2_logT_lower_bdy, Zbase
      real(dp) :: kappa_blend_logT_upper_bdy, kappa_blend_logT_lower_bdy
      real(dp) :: lnRho(maxpts), lnT(maxpts), logRho(maxpts), logT(maxpts), X(maxspec,maxpts)
      real(dp) :: lnR(maxpts), L, dq(maxpts), dqsum
      real(dp) :: kappa(maxpts), dlnkap_dlnRho(maxpts), dlnkap_dlnT(maxpts), q(maxpts)
      real(dp) :: kappaCO(maxpts), dlnkapCO_dlnRho(maxpts), dlnkapCO_dlnT(maxpts)
      real(dp) :: &
         base_fC = 0d0, base_fN = 0d0, base_fO = 0d0, base_fNe = 0d0, &
         kap_Type2_full_off_X = 0d0, kap_Type2_full_on_X = 0d0
      character (len=32) :: my_mesa_dir
      
      kappa_file_prefix = 'gn93'
      !kappa_lowT_prefix = 'lowT_Freedman11'
      kappa_lowT_prefix = 'lowT_fa05_gn93'
      
      kappa_CO_prefix = '' ! use default
      kappa_lowT_prefix = '' ! use default
      kappa_blend_logT_upper_bdy = 0 ! use default
      kappa_blend_logT_lower_bdy = 0 ! use default
      type2_logT_lower_bdy = 0 ! use default
      use_cache = .false.
      model_file = 'sample_kap_agb.model'
      output_file = 'kap_test.data'
      ierr = 0

      ! initialization and setup

      my_mesa_dir = '../..'         
      call const_init(my_mesa_dir,ierr)     
   	if (ierr /= 0) then
   	   write(*,*) 'const_init failed'
   	   stop 1
   	end if        
      call chem_init('isotopes.data', ierr)
      call kap_init( &
         kappa_file_prefix, kappa_CO_prefix, kappa_lowT_prefix, &
         kappa_blend_logT_upper_bdy, kappa_blend_logT_lower_bdy, &
         type2_logT_lower_bdy, use_cache, '', ierr) 
      if(ierr/=0) stop 'problem in kap_init'

      !next it is necessary to create a 'handle' for the general kap structure
      !using handles, it is possible to simultaneously access more than one
      !working copy of the opacity subroutines with different settings
      handle = alloc_kap_handle(ierr)
      if(ierr/=0) stop 'problem in alloc_kap_handle'

      !read in AGB model
      iounit=99
      open(unit=iounit,file=trim(model_file),status='old',iostat=ierr)
      if(ierr/=0) stop 'problem opening agb.mod file'
      read(iounit,*)
      read(iounit,*)            !skip 3 header lines
      read(iounit,*)
      read(iounit,1) Mstar      !read stellar mass
      read(iounit,1) Z_init     !read initial Z
      read(iounit,2) Npts       !read number of points in model
      read(iounit,*)            !skip
      read(iounit,2) Nspec      !read number of chemical species in model
      read(iounit,*)            !skip 2 lines
      read(iounit,*)
      
      write(*,*) ' Npts', Npts
      write(*,*) 'Nspec', Nspec
      
      do i=1,Npts               !read model
         read(iounit,*) ii, lnRho(i), lnT(i), lnR(i), L, dq(i), X(1:Nspec,i)
         if (ii /= i) then
            write(*,*) 'bad data for zone', i
            stop
         end if
      enddo
      close(iounit)

      write(*,*)
      write(*,*) 'Z_init', Z_init
      write(*,*)

      XC_base = GN93_element_zfrac(e_C)*Z_init
      XN_base = GN93_element_zfrac(e_N)*Z_init
      XO_base = GN93_element_zfrac(e_O)*Z_init
      XNe_base = GN93_element_zfrac(e_Ne)*Z_init

      !the final step is to set the type of X and Z interpolation, either linear or cubic spline
      !this is achieved by two logical variables, in this example both X and Z use cubic splines
      cubic_X_interpolation = .true.
      cubic_Z_interpolation = .true.
      include_electron_conduction = .true.
      call kap_set_choices( &
         handle, cubic_X_interpolation, cubic_X_interpolation, &
         include_electron_conduction, base_fC, base_fN, base_fO, base_fNe, &
         kap_Type2_full_off_X, kap_Type2_full_on_X, &
         ierr)
      if(ierr/=0) stop 'problem in kap_set_interpolation_choices'
      !end of initialization and setup

      logRho(:) = lnRho(:)/ln10 !convert ln's to log10's
      logT(:)   = lnT(:)  /ln10
      
      q(1) = 1d0; dqsum = 0d0
      do i=2,Npts
         dqsum = dqsum + dq(i-1)
         q(i) = 1 - dqsum
      end do

      do i=1,Npts            !X=H     Y=He3+He             metals
         XC = X(c12,i) + X(c13,i)
         XN = X(n14,i) + X(n15,i)
         XO = X(o16,i) + X(o17,i) + X(o18,i)
         XNe = X(ne20,i) + X(ne21,i) + X(ne22,i)
         Z = 1d0 - (X(h1,i)+X(he3,i)+X(he4,i))
         Xheavy = Z - (XC + XN + XO + XNe)
         
         zbar = 1.5 ! needed for electron conduction at high rho
         lnfree_e = 0 ! needed for Compton at high T
         d_lnfree_e_dlnRho = 0
         d_lnfree_e_dlnT = 0
         
         call kap_get_Type1( &
            handle, zbar, X(h1,i), Z, logRho(i), logT(i), &
            lnfree_e, d_lnfree_e_dlnRho, d_lnfree_e_dlnT, &
            kappa(i), dlnkap_dlnRho(i), dlnkap_dlnT(i), ierr)
         if(ierr/=0) then
            write(*,*) 'kap_get failed at i=', i
            stop
         endif
 22      format(a40,i6,1pe26.16)
         if (.false. .and. i >= 117 .and. i <= 119) then
            write(*,22) 'xh =', i, X(h1,i)
            write(*,22) 'y =', i, X(he3,i)+X(he4,i)
            write(*,22) 'z =', i, Z
            write(*,22) 'xc =', i, XC
            write(*,22) 'xn =', i, XN
            write(*,22) 'xo =', i, XO
            write(*,22) 'xne =', i, XNe
            write(*,22) 'xheavy =', i, Xheavy
            write(*,22) 'logRho =', i, logRho(i)
            write(*,22) 'logT =', i, logT(i)
            write(*,*)
            write(*,22) 'kappa =', i, kappa(i)
            write(*,*)
            write(*,*)
         end if
      enddo

      do i=1,Npts 
         XC = X(c12,i) + X(c13,i) - XC_base
         XN = X(n14,i) + X(n15,i) - XN_base
         XO = X(o16,i) + X(o17,i) + X(o18,i) - XO_base
         XNe = X(ne20,i) + X(ne21,i) + X(ne22,i) - XNe_base
         
         zbar = 1.5 ! needed for electron conduction at high rho
         lnfree_e = 0 ! needed for Compton at high T
         d_lnfree_e_dlnRho = 0
         d_lnfree_e_dlnT = 0
         Zbase = -1

         call kap_get_Type2( &
            handle, zbar, X(h1,i), Z, Zbase, XC, XN, XO, XNe, logRho(i), logT(i), &
            frac_Type2, lnfree_e, d_lnfree_e_dlnRho, d_lnfree_e_dlnT, &
            kappaCO(i), dlnkapCO_dlnRho(i), dlnkapCO_dlnT(i), ierr)
         if(ierr/=0) then
            write(*,*) 'kap_get failed at i=', i
            stop
         endif
      enddo

      open(unit=iounit,file=trim(output_file),iostat=ierr)
      if(ierr/=0) stop 'problem opening agb.mod file'
      
      write(*,*) 'write ' // trim(output_file)

      write(iounit,4) '#','grid', 'lgK', 'lgKCO', 'lgK_sub_lgKCO', 'kap', 'kapCO', 'kap_sub_kapCO', &
            'mass', 'r', 'lgR', 'lgT', 'lgRho', &
            'x', 'y', 'z', 'xc', 'xn', 'xo', 'dxc', 'dxn', 'dxo', 'dz', &
            'dlnK_dlnRho', 'dlnK_dlnT'

      do i=1,Npts !'
         XC = X(c12,i) + X(c13,i)
         XN = X(n14,i) + X(n15,i)
         XO = X(o16,i) + X(o17,i) + X(o18,i)
         Z = 1d0 - (X(h1,i)+X(he3,i)+X(he4,i))
         write(iounit,5) i, log10(kappa(i)), log10(kappaCO(i)), log10(kappa(i)/kappaCO(i)), &
            kappa(i), kappaCO(i), kappa(i) - kappaCO(i), &
            Mstar*q(i), exp(lnR(i)), lnR(i)/ln10, logT(i), logRho(i), &
            X(h1,i), X(he3,i)+X(he4,i), Z, &
            XC, XN, XO, XC - XC_base, XN - XN_base, XO - XO_base, Z - Z_init, &
            dlnkap_dlnRho(i), dlnkap_dlnT(i)
       enddo
       
       close(iounit)

      !all finished? then deallocate the handle and unload the opacity tables
      call free_kap_handle(handle)
      call kap_shutdown


 1    format(37x,e23.16)
 2    format(37x,i6)
 3    format(5x,99(4x,e23.6))
 4    format(a1,a7,99(a14))
 5    format(i8,1p,99(1x,e13.6),0p)
      
      end program
