      module plot_kap_model_support
      
      use kap_def
      use kap_lib
      use const_def
      use chem_def
      
      implicit none
      
      ! for micro data
      integer :: nz, species, net_num, nci
      integer, pointer :: chem_id(:), net_iso(:)
      double precision, pointer :: lnd(:), lnT(:), q(:), xa(:,:)
      double precision :: mstar, Zbase
      
      ! for results
      double precision, pointer, dimension(:) :: log10kap, dlnkap_dlnRho, dlnkap_dlnT

      contains
      
      
      subroutine read_model_data(ierr)
         integer, intent(out) :: ierr
         integer, parameter :: iounit = 33
         integer :: k, i
         22 format(999(1pe26.16, 1x))         
         ierr = 0
         open(iounit, file='model.data', status='old', action='read', iostat=ierr)
         if (ierr /= 0) then
            write(*,*) 'failed to open model.data'
            return
         end if         
         read(iounit,*)
         read(iounit,*) nz, species, net_num, nci
         if (nci /= num_chem_isos) stop 1    
         allocate(chem_id(species), net_iso(num_chem_isos), &
            lnd(nz), lnT(nz), q(nz), xa(species,nz))
         read(iounit,*)
         do i=1, species
            read(iounit, *) chem_id(i)
         end do         
         read(iounit,*)
         do i=1, num_chem_isos
            read(iounit, *) net_iso(i)
         end do         
         read(iounit,*)
         do k=1, nz
            read(iounit, 22) lnd(k), lnT(k), q(k)
         end do         
         read(iounit,*)
         read(iounit,*) mstar
         read(iounit,*)
         do k=1, nz
            read(iounit, 22) xa(1:species, k)
         end do        
         close(iounit)
      end subroutine read_model_data
      
      
      subroutine eval_for_model_data(kap_handle, Zbase, ierr)
         integer, intent(in) :: kap_handle
         double precision, intent(in) :: Zbase
         integer, intent(out) :: ierr      
         integer :: k, op_err         
         allocate(log10kap(nz), dlnkap_dlnRho(nz), dlnkap_dlnT(nz))
         ierr = 0
!$OMP PARALLEL DO PRIVATE(k,op_err)
         do k = 1, nz
            op_err = 0
            call eval1_for_model_data(kap_handle, k, Zbase, op_err)
            if (op_err /= 0) ierr = op_err
         end do
!$OMP END PARALLEL DO         
      end subroutine eval_for_model_data
      
      
      subroutine eval1_for_model_data(kap_handle, k, Zbase, ierr)
         use chem_def
         use num_lib, only: safe_log10
         integer, intent(in) :: kap_handle
         integer, intent(in) :: k
         double precision, intent(in) :: Zbase
         integer, intent(out) :: ierr
         
         double precision :: xh, dxc, dxo, kap

         xh = xa(net_iso(ih1),k)
         dxc = 0
         dxo = 0

         call kap_get_Type1( &
                  kap_handle, xh, Zbase, lnd(k)/ln10, lnT(k)/ln10,  &
                  kap, dlnkap_dlnRho(k), dlnkap_dlnT(k), ierr)
         log10kap(k) = safe_log10(kap)
         
      end subroutine eval1_for_model_data
      
      
      subroutine write_results_for_model_data
         character (len=256) :: filename
         integer :: ierr, iounit, k
         
         filename = 'plot_data/kap_model.data'
         ierr = 0
         iounit = 33
         open(iounit, file=trim(filename), action='write', status='replace', iostat=ierr)
         if (ierr == 0) then
            write(*, *) 'write results to ' // trim(filename)
            write(iounit, '(99(a23, 1x))') &
               'm', 'log10kap', 'dlnkap_dlnRho', 'dlnkap_dlnT', &
               'logRho', 'logT', 'xh', 'xhe', 'xc', 'xn', 'xo', 'xne', 'z'
            do k=1, nz
               write(iounit, '(99e24.10)') &
                     mstar*q(k)/Msun, log10kap(k), dlnkap_dlnRho(k), dlnkap_dlnT(k), &
                     lnd(k)/ln10, lnT(k)/ln10, &
                     xa(net_iso(ih1),k), &
                     xa(net_iso(ihe4),k), &
                     xa(net_iso(ic12),k), &
                     xa(net_iso(in14),k), &
                     xa(net_iso(io16),k), &
                     xa(net_iso(ine20),k), &
                     1 - (xa(net_iso(ih1),k) + xa(net_iso(ihe4),k))
            end do
            close(iounit)
         else
            write(*, *) 'failed to open file ' // trim(filename)
         end if
      
      end subroutine write_results_for_model_data
      
      
      subroutine plot_results_for_model_data(kap_handle, ierr)
         integer, intent(in) :: kap_handle
         integer, intent(out) :: ierr
         ierr = 0
         call read_model_data(ierr)
         if (ierr /= 0) return
         call eval_for_model_data(kap_handle, Zbase, ierr)
         if (ierr /= 0) return
         call write_results_for_model_data
      end subroutine plot_results_for_model_data
      
      
      end module plot_kap_model_support



      program plot_kap_model
      use plot_kap_model_support
      use test_kap_support
      implicit none
      
      integer :: ierr

      call setup(.true.) ! quietly, please.
      
      ierr = 0
      call plot_results_for_model_data(handle,ierr)
      if (ierr /= 0) stop 'failed to make plot data'

      call Finish

      end program plot_kap_model