      module mod_ion_create_tables
      
      use const_def
      use utils_lib

      implicit none
      
      
      integer :: version = 51
      real(dp) :: X, Z
      integer :: num_logQs
      real(dp) :: logQ_min, logQ_max, del_logQ
      integer :: num_logTs
      real(dp) :: logT_min, logT_max, del_logT
      real(dp), pointer, dimension(:) :: logQs, logTs
      real(dp), pointer, dimension(:) :: &
         logPgas, logE, logS, chiRho, chiT, Cp, Cv, &
         dE_dRho, dS_dT, dS_dRho, mu, log_free_e, gamma1, gamma3, grad_ad, eta, &
         fneut_H, fneut_He, fneut_C, fneut_N, fneut_O, fneut_Ne, fneut_Mg, fneut_Si, fneut_Fe, &
         Z_H, Z_He, Z_C, Z_N, Z_O, Z_Ne, Z_Mg, Z_Si, Z_Fe, &
         logpp_H, logpp_He, logpp_C, logpp_N, logpp_O, logpp_Ne, logpp_Mg, logpp_Si, logpp_Fe
      integer :: ili_logQs    ! =1: logRho grid is "nearly" equally spaced
      integer :: ili_logTs    ! =1: logT grid is "nearly" equally spaced
      
      integer :: eos_handle ! for standard eos
      
      
      integer, parameter :: H1 = 1
      integer, parameter :: He4 = H1+1
      integer, parameter :: C12 = He4+1
      integer, parameter :: N14 = C12+1
      integer, parameter :: O16 = N14+1
      integer, parameter :: Ne20 = O16+1
      integer, parameter :: Mg24 = Ne20+1
      integer, parameter :: Si28 = Mg24+1
      integer, parameter :: Fe56 = Si28+1
      integer, parameter :: species = Fe56
      integer, pointer :: chem_id(:), net_iso(:)
      real(dp) :: xa(species), abar, zbar


      integer, parameter :: i_fneut_H = 1
      integer, parameter :: i_fneut_He = i_fneut_H+1
      integer, parameter :: i_fneut_C = i_fneut_He+1
      integer, parameter :: i_fneut_N = i_fneut_C+1
      integer, parameter :: i_fneut_O = i_fneut_N+1
      integer, parameter :: i_fneut_Ne = i_fneut_O+1
      integer, parameter :: i_fneut_Mg = i_fneut_Ne+1
      integer, parameter :: i_fneut_Si = i_fneut_Mg+1
      integer, parameter :: i_fneut_Fe = i_fneut_Si+1
      integer, parameter :: i_Z_H = i_fneut_Fe+1
      integer, parameter :: i_Z_He = i_Z_H+1
      integer, parameter :: i_Z_C = i_Z_He+1
      integer, parameter :: i_Z_N = i_Z_C+1
      integer, parameter :: i_Z_O = i_Z_N+1
      integer, parameter :: i_Z_Ne = i_Z_O+1
      integer, parameter :: i_Z_Mg = i_Z_Ne+1
      integer, parameter :: i_Z_Si = i_Z_Mg+1
      integer, parameter :: i_Z_Fe = i_Z_Si+1
      integer, parameter :: i_logpp_H = i_Z_Fe+1
      integer, parameter :: i_logpp_He = i_logpp_H+1
      integer, parameter :: i_logpp_C = i_logpp_He+1
      integer, parameter :: i_logpp_N = i_logpp_C+1
      integer, parameter :: i_logpp_O = i_logpp_N+1
      integer, parameter :: i_logpp_Ne = i_logpp_O+1
      integer, parameter :: i_logpp_Mg = i_logpp_Ne+1
      integer, parameter :: i_logpp_Si = i_logpp_Mg+1
      integer, parameter :: i_logpp_Fe = i_logpp_Si+1
      
      integer, parameter :: num_eos_extra_results = i_logpp_Fe

      
      contains
      
      
      subroutine do_create_ion_table_files( &
            in_dir, out_dir_ion, out_dir_eosDT, out_dir_eosPT)
         character (len=*), intent(in) :: &
            in_dir, out_dir_ion, out_dir_eosDT, out_dir_eosPT

         ! standard low Z cases
         call convert1('x00z04', '_04z00x')
         call convert1('x02z04', '_04z20x')
         call convert1('x04z04', '_04z40x')
         call convert1('x06z04', '_04z60x')
         call convert1('x08z04', '_04z80x')

         call convert1('x00z02', '_02z00x')
         call convert1('x02z02', '_02z20x')
         call convert1('x04z02', '_02z40x')
         call convert1('x06z02', '_02z60x')
         call convert1('x08z02', '_02z80x')

         call convert1('x00z00', '_00z00x')
         call convert1('x02z00', '_00z20x')
         call convert1('x04z00', '_00z40x')
         call convert1('x06z00', '_00z60x')
         call convert1('x08z00', '_00z80x')
         call convert1('x10z00', '_00z100x')
         
         
         !return
         
         ! high Z cases
         call convert1('z01C495O495', '_100z00x_CO_1')
         call convert1('C50O50', '_100z00x_CO_0')
         call convert1('x00z100', '_100z00x')
         
         call convert1('x00z20', '_20z00x')
         call convert1('x02z20', '_20z20x')
         call convert1('x04z20', '_20z40x')
         call convert1('x06z20', '_20z60x')
         call convert1('x08z20', '_20z80x')
         
         
         
         contains
         
         subroutine convert1(str_mac, str_mesa)
            character (len=*), intent(in) :: str_mac, str_mesa
            character (len=128) :: infile, outfile_ionization, outfile_eosDT, outfile_eosPT
            infile = trim(in_dir) // '/p' // trim(str_mac) // '.txt'
            if (len_trim(out_dir_ion) > 0) then
               outfile_ionization = trim(out_dir_ion) // '/ion' // trim(str_mesa) // '.data'
            else
               outfile_ionization = ''
            end if
            if (len_trim(out_dir_eosDT) > 0) then
               outfile_eosDT = trim(out_dir_eosDT) // '/macdonald-eosDT' // trim(str_mesa) // '.data'
            else
               outfile_eosDT = ''
            end if
            if (len_trim(out_dir_eosPT) > 0) then
               outfile_eosPT = trim(out_dir_eosPT) // '/macdonald-eosPT' // trim(str_mesa) // '.data'
            else
               outfile_eosPT = ''
            end if
            call convert_eos_p_to_d(infile, outfile_ionization, outfile_eosDT, outfile_eosPT)
         end subroutine convert1
      
      
      end subroutine do_create_ion_table_files
      
      
      subroutine setup_composition
         use chem_lib, only: basic_composition_info
         use chem_def
         real(dp) :: Y, z2bar, ye, sumx, mass_correction
         
         allocate(chem_id(species), net_iso(num_chem_isos))
         chem_id(H1) = iH1
         chem_id(He4) = iHe4
         chem_id(C12) = iC12
         chem_id(N14) = iN14
         chem_id(O16) = iO16
         chem_id(Ne20) = iNe20
         chem_id(Mg24) = iMg24
         chem_id(Si28) = iSi28
         chem_id(Fe56) = iFe56
         
         net_iso(iH1) = H1
         net_iso(iHe4) = He4
         net_iso(iC12) = C12
         net_iso(iN14) = N14
         net_iso(iO16) = O16
         net_iso(iNe20) = Ne20
         net_iso(iMg24) = Mg24
         net_iso(iSi28) = Si28
         net_iso(iFe56) = Fe56

         Y = 1d0 - (X + Z)

         xa(H1) = X
         xa(He4) = Y
         xa(C12) = 0.173285d0*Z
         xa(N14) = 0.053152d0*Z
         xa(O16) = 0.482273d0*Z
         xa(Ne20) = 0.098668d0*Z
         xa(Mg24) = 0.042810d0*Z
         xa(Si28) = 0.071796d0*Z
         xa(Fe56) = 0.078017d0*Z
         if (sum(xa(:) - 1d0) > 1d-5) then
            write(*,*) 'error in setup_composition'
            stop 1
         end if

         call basic_composition_info( &
            species, chem_id, xa, X, Y, abar, zbar, z2bar, ye, &
            mass_correction, sumx)

      end subroutine setup_composition
      
      
      subroutine eval_eos(Rho, logRho, T, logT_in, res, res_extra, ierr)
         use eos_def
         real(dp), intent(in) :: Rho, logRho, T, logT_in
         real(dp), dimension(num_eos_basic_results) :: res
         real(dp), dimension(num_eos_extra_results) :: res_extra
         integer, intent(out) :: ierr
         
         real(dp) :: logQ, result, logT

         ierr = 0
         logT = max(logT_min, min(logT_max, logT_in))
         logQ = logRho - 2*logT + 12
         logQ = max(logQ_min, min(logQ_max, logQ))
      
         call eval1(logPgas, 'logPgas', result)
         res(i_lnPgas) = result*ln10
         call eval1(logE, 'logE', result)
         res(i_lnE) = result*ln10
         call eval1(logS, 'logS', result)
         res(i_lnS) = result*ln10
         call eval1(chiRho, 'chiRho', res(i_chiRho))
         call eval1(chiT, 'chiT', res(i_chiT))
         call eval1(Cp, 'Cp', res(i_Cp))
         call eval1(Cv, 'Cv', res(i_Cv))
         call eval1(dE_dRho, 'dE_dRho', res(i_dE_dRho))
         call eval1(dS_dT, 'dS_dT', res(i_dS_dT))
         call eval1(dS_dRho, 'dS_dRho', res(i_dS_dRho))
         call eval1(mu, 'mu', res(i_mu))
         call eval1(log_free_e, 'log_free_e', result)
         res(i_lnfree_e) = result*ln10
         call eval1(gamma1, 'gamma1', res(i_gamma1))
         call eval1(gamma3, 'gamma3', res(i_gamma3))
         call eval1(grad_ad, 'grad_ad', res(i_grad_ad))
         call eval1(eta, 'eta', res(i_eta))
         
         call eval1(fneut_H, 'fneut_H', res_extra(i_fneut_H))
         call eval1(fneut_He, 'fneut_He', res_extra(i_fneut_He))
         call eval1(fneut_C, 'fneut_C', res_extra(i_fneut_C))
         call eval1(fneut_N, 'fneut_N', res_extra(i_fneut_N))
         call eval1(fneut_O, 'fneut_O', res_extra(i_fneut_O))
         call eval1(fneut_Ne, 'fneut_Ne', res_extra(i_fneut_Ne))
         call eval1(fneut_Mg, 'fneut_Mg', res_extra(i_fneut_Mg))
         call eval1(fneut_Si, 'fneut_Si', res_extra(i_fneut_Si))
         call eval1(fneut_Fe, 'fneut_Fe', res_extra(i_fneut_Fe))
         call eval1(Z_H, 'Z_H', res_extra(i_Z_H))
         call eval1(Z_He, 'Z_He', res_extra(i_Z_He))
         call eval1(Z_C, 'Z_C', res_extra(i_Z_C))
         call eval1(Z_N, 'Z_N', res_extra(i_Z_N))
         call eval1(Z_O, 'Z_O', res_extra(i_Z_O))
         call eval1(Z_Ne, 'Z_Ne', res_extra(i_Z_Ne))
         call eval1(Z_Mg, 'Z_Mg', res_extra(i_Z_Mg))
         call eval1(Z_Si, 'Z_Si', res_extra(i_Z_Si))
         call eval1(Z_Fe, 'Z_Fe', res_extra(i_Z_Fe))
         call eval1(logpp_H, 'logpp_H', res_extra(i_logpp_H))
         call eval1(logpp_He, 'logpp_He', res_extra(i_logpp_He))
         call eval1(logpp_C, 'logpp_C', res_extra(i_logpp_C))
         call eval1(logpp_N, 'logpp_N', res_extra(i_logpp_N))
         call eval1(logpp_O, 'logpp_O', res_extra(i_logpp_O))
         call eval1(logpp_Ne, 'logpp_Ne', res_extra(i_logpp_Ne))
         call eval1(logpp_Mg, 'logpp_Mg', res_extra(i_logpp_Mg))
         call eval1(logpp_Si, 'logpp_Si', res_extra(i_logpp_Si))
         call eval1(logpp_Fe, 'logpp_Fe', res_extra(i_logpp_Fe))
         
         
         contains
         
         
         subroutine eval1(f, str, val)
            real(dp), pointer :: f(:,:,:)
            character (len=*), intent(in) :: str
            real(dp), intent(out) :: val
            real(dp) :: v
            include 'formats.dek'
            call eval_bicubic(logQ, logT, f, v, ierr)
            val = dble(v)
            if (ierr == 0) return
            write(*,*) 'failed in evaluation of ' // str
            write(*,1) 'logRho', logRho
            write(*,1) 'logQ_min', logQ_min
            write(*,1) 'logQ', logQ
            write(*,1) 'logQ_max', logQ_max
            write(*,1) 'logT_min', logT_min
            write(*,1) 'logT', logT
            write(*,1) 'logT_max', logT_max
            write(*,*) 'logQ > logQ_max', logQ > logQ_max
            write(*,*) 'logQ < logQ_min', logQ < logQ_min
            write(*,*) 'logT > logT_max', logT > logT_max
            write(*,*) 'logT < logT_min', logT < logT_min
            stop 1
         end subroutine eval1
         
      
      end subroutine eval_eos


      subroutine setup_eos_data(fname)
         character (len=*), intent(in) :: fname
      
         integer :: i, j, ierr, iounit, line, version_in
         real(dp) :: logT
         
         include 'formats.dek'
         
         write(*,*) 'read ' // trim(fname)
         
         iounit = 37
         open(iounit, file=trim(fname), action='read', status='old', iostat=ierr)
         if (ierr /= 0) then
            write(*, *) 'failed to open ' // trim(fname)
            call free_iounit(iounit)
            return
         end if           
         
         line = 0
         call skip_lines(1,iounit,ierr)
         if (ierr /= 0) stop 1
         line = line+1
         read(iounit,*,iostat=ierr) version_in, X, Z, &
            num_logTs, logT_min, logT_max, del_logT, &
            num_logQs, logQ_min, logQ_max, del_logQ
         if (ierr /= 0) stop 2
      
         !write(*,2) 'num_logTs', num_logTs
         !write(*,2) 'num_logQs', num_logQs
      
         allocate( &
            logTs(num_logTs), logQs(num_logQs), &
            logPgas(4*num_logQs*num_logTs), &
            logE(4*num_logQs*num_logTs), &
            logS(4*num_logQs*num_logTs), &
            chiRho(4*num_logQs*num_logTs), &
            chiT(4*num_logQs*num_logTs), &
            Cp(4*num_logQs*num_logTs), &
            Cv(4*num_logQs*num_logTs), &
            dE_dRho(4*num_logQs*num_logTs), &
            dS_dT(4*num_logQs*num_logTs), &
            dS_dRho(4*num_logQs*num_logTs), &
            mu(4*num_logQs*num_logTs), &
            log_free_e(4*num_logQs*num_logTs), &
            gamma1(4*num_logQs*num_logTs), &
            gamma3(4*num_logQs*num_logTs), &
            grad_ad(4*num_logQs*num_logTs), &
            eta(4*num_logQs*num_logTs), &
            fneut_H(4*num_logQs*num_logTs), &
            fneut_He(4*num_logQs*num_logTs), &
            fneut_C(4*num_logQs*num_logTs), &
            fneut_N(4*num_logQs*num_logTs), &
            fneut_O(4*num_logQs*num_logTs), &
            fneut_Ne(4*num_logQs*num_logTs), &
            fneut_Mg(4*num_logQs*num_logTs), &
            fneut_Si(4*num_logQs*num_logTs), &
            fneut_Fe(4*num_logQs*num_logTs), &
            Z_H(4*num_logQs*num_logTs), &
            Z_He(4*num_logQs*num_logTs), &
            Z_C(4*num_logQs*num_logTs), &
            Z_N(4*num_logQs*num_logTs), &
            Z_O(4*num_logQs*num_logTs), &
            Z_Ne(4*num_logQs*num_logTs), &
            Z_Mg(4*num_logQs*num_logTs), &
            Z_Si(4*num_logQs*num_logTs), &
            Z_Fe(4*num_logQs*num_logTs), &
            logpp_H(4*num_logQs*num_logTs), &
            logpp_He(4*num_logQs*num_logTs), &
            logpp_C(4*num_logQs*num_logTs), &
            logpp_N(4*num_logQs*num_logTs), &
            logpp_O(4*num_logQs*num_logTs), &
            logpp_Ne(4*num_logQs*num_logTs), &
            logpp_Mg(4*num_logQs*num_logTs), &
            logpp_Si(4*num_logQs*num_logTs), &
            logpp_Fe(4*num_logQs*num_logTs))

         call skip_lines(2,iounit,ierr)
         if (ierr /= 0) stop 3
         line = line+2

         do i = 1, num_logQs
            read(iounit,*,iostat=ierr) logQs(i)
            if (ierr /= 0) stop 4
            line = line+1
            call skip_lines(2,iounit,ierr)
            if (ierr /= 0) stop 5
            line = line+2
            !write(*,2) 'logQs(i)', i, logQs(i)
            do j = 1, num_logTs
               read(iounit,*,iostat=ierr) logT, &
                  logPgas(1,i,j), logE(1,i,j), logS(1,i,j), chiRho(1,i,j), chiT(1,i,j), &
                  Cp(1,i,j), Cv(1,i,j), dE_dRho(1,i,j), dS_dT(1,i,j), dS_dRho(1,i,j), mu(1,i,j), &
                  log_free_e(1,i,j), gamma1(1,i,j), gamma3(1,i,j), grad_ad(1,i,j), eta(1,i,j), &
                  fneut_H(1,i,j), fneut_He(1,i,j), fneut_C(1,i,j), fneut_N(1,i,j), &
                  fneut_O(1,i,j), fneut_Ne(1,i,j), fneut_Mg(1,i,j), fneut_Si(1,i,j), fneut_Fe(1,i,j), &
                  Z_H(1,i,j), Z_He(1,i,j), Z_C(1,i,j), Z_N(1,i,j), Z_O(1,i,j), &
                  Z_Ne(1,i,j), Z_Mg(1,i,j), Z_Si(1,i,j), Z_Fe(1,i,j), &
                  logpp_H(1,i,j), logpp_He(1,i,j), logpp_C(1,i,j), logpp_N(1,i,j), &
                  logpp_O(1,i,j), logpp_Ne(1,i,j), logpp_Mg(1,i,j), logpp_Si(1,i,j), logpp_Fe(1,i,j)
               if (ierr /= 0) then
                  write(*,2) 'failed on line', line, logQs(i), logTs(j)
                  stop 6
               end if
               line = line+1
               if (i == 1) then
                  logTs(j) = logT
               else if (abs(logTs(j) - logT) > 1d-4) then
                  write(*,2) 'inconsistent logT on line', line, logTs(j) - logT, logTs(j), logT
                  stop 7
               end if
            end do
            if (i == num_logQs) exit
            call skip_lines(3,iounit,ierr)
            if (ierr /= 0) stop 8
            line = line+3
         end do

         close(iounit)
      
         call create_bicubic(logPgas, 'logPgas')
         call create_bicubic(logE, 'logE')
         call create_bicubic(logS, 'logS')
         call create_bicubic(chiRho, 'chiRho')
         call create_bicubic(chiT, 'chiT')
         call create_bicubic(Cp, 'Cp')
         call create_bicubic(Cv, 'Cv')
         call create_bicubic(dE_dRho, 'dE_dRho')
         call create_bicubic(dS_dT, 'dS_dT')
         call create_bicubic(dS_dRho, 'dS_dRho')
         call create_bicubic(mu, 'mu')
         call create_bicubic(log_free_e, 'log_free_e')
         call create_bicubic(gamma1, 'gamma1')
         call create_bicubic(gamma3, 'gamma3')
         call create_bicubic(grad_ad, 'grad_ad')
         call create_bicubic(eta, 'eta')
         call create_bicubic(fneut_H, 'fneut_H')
         call create_bicubic(fneut_He, 'fneut_He')
         call create_bicubic(fneut_C, 'fneut_C')
         call create_bicubic(fneut_N, 'fneut_N')
         call create_bicubic(fneut_O, 'fneut_O')
         call create_bicubic(fneut_Ne, 'fneut_Ne')
         call create_bicubic(fneut_Mg, 'fneut_Mg')
         call create_bicubic(fneut_Si, 'fneut_Si')
         call create_bicubic(fneut_Fe, 'fneut_Fe')
         call create_bicubic(Z_H, 'Z_H')
         call create_bicubic(Z_He, 'Z_He')
         call create_bicubic(Z_C, 'Z_C')
         call create_bicubic(Z_N, 'Z_N')
         call create_bicubic(Z_O, 'Z_O')
         call create_bicubic(Z_Ne, 'Z_Ne')
         call create_bicubic(Z_Mg, 'Z_Mg')
         call create_bicubic(Z_Si, 'Z_Si')
         call create_bicubic(Z_Fe, 'Z_Fe')
         call create_bicubic(logpp_H, 'logpp_H')
         call create_bicubic(logpp_He, 'logpp_He')
         call create_bicubic(logpp_C, 'logpp_C')
         call create_bicubic(logpp_N, 'logpp_N')
         call create_bicubic(logpp_O, 'logpp_O')
         call create_bicubic(logpp_Ne, 'logpp_Ne')
         call create_bicubic(logpp_Mg, 'logpp_Mg')
         call create_bicubic(logpp_Si, 'logpp_Si')
         call create_bicubic(logpp_Fe, 'logpp_Fe')
         
      end subroutine setup_eos_data
      
      
      subroutine create_bicubic(f,str)
         use interp_2d_lib_db, only: interp_mkbicub_db        
         real(dp), pointer :: f(:,:,:)
         character (len=*), intent(in) :: str
         integer :: ibcxmin                   ! bc flag for x=xmin
         real(dp) :: bcxmin(num_logTs)    ! bc data vs. y at x=xmin
         integer :: ibcxmax                   ! bc flag for x=xmax
         real(dp) :: bcxmax(num_logTs)     ! bc data vs. y at x=xmax
         integer :: ibcymin                   ! bc flag for y=ymin
         real(dp) :: bcymin(num_logQs)   ! bc data vs. x at y=ymin
         integer :: ibcymax                   ! bc flag for y=ymax
         real(dp) :: bcymax(num_logQs)   ! bc data vs. x at y=ymax
         integer :: ierr
         ierr = 0
         ! just use "not a knot" bc's at edges of tables
         ibcxmin = 0; bcxmin(:) = 0
         ibcxmax = 0; bcxmax(:) = 0
         ibcymin = 0; bcymin(:) = 0
         ibcymax = 0; bcymax(:) = 0
         call interp_mkbicub_db( &
            logQs, num_logQs, logTs, num_logTs, f, num_logQs, &
            ibcxmin, bcxmin, ibcxmax, bcxmax, &
            ibcymin, bcymin, ibcymax, bcymax, &
            ili_logQs, ili_logTs, ierr)
         if (ierr /= 0) then
            write(*,*) 'failed to create bicubic interpolant for ' // trim(str)
            stop 1
         end if         
      end subroutine create_bicubic
      
      
      subroutine eval_bicubic(logQ,logT,f,val,ierr)
         use interp_2d_lib_db, only: interp_evbicub_db         
         real(dp), intent(in) :: logQ, logT
         real(dp), pointer :: f(:,:,:)
         real(dp), intent(out) :: val
         integer, intent(out) :: ierr         
         integer, parameter :: ict(6) = (/ 1, 0, 0, 0, 0, 0 /) ! just return fval
         real(dp) :: fval(6)     
         call interp_evbicub_db( &
            logQ, logT, logQs, num_logQs, logTs, num_logTs, &
            ili_logQs, ili_logTs, f, num_logQs, ict, fval, ierr)      
         val = fval(1)
      end subroutine eval_bicubic   
      
      
      subroutine setup_std_eos
         use const_lib
         use chem_lib
         use eos_def
         use eos_lib      
         character (len=256) :: data_dir, eos_file_prefix
         integer :: ierr
         logical, parameter :: use_cache = .true.
         call const_init         
         data_dir = '../../data'
         ierr = 0
         call chem_init(data_dir, 'isotopes.data_approx', ierr)
         if (ierr /= 0) then
            write(*,*) 'chem_init failed'
            stop 1
         end if
         data_dir = '../../data'
         eos_file_prefix = 'mesa'
         call eos_init(data_dir, eos_file_prefix, use_cache, ierr)
         if (ierr /= 0) then
            write(*,*) 'eos_init failed'
            stop 1
         end if         
         eos_handle = alloc_eos_handle(ierr)
         if (ierr /= 0) then
            write(*,*) 'failed trying to allocate eos handle'
            stop 1
         end if      
      end subroutine setup_std_eos
      
      
      subroutine convert_eos_p_to_d( &
            fname_in, outfile_ionization, outfile_eosDT, outfile_eosPT)
         character (len=*), intent(in) :: &
            fname_in, outfile_ionization, outfile_eosDT, outfile_eosPT
         integer :: iounit, ierr, i, j, k
         
         real(dp) :: X, Z, logT_min, logT_max, del_logT
         integer :: num_logTs
         real(dp), pointer, dimension(:) :: logTs

         ! pgas data
         integer :: num_logWs
         real(dp) :: logW_min, logW_max, del_logW
         real(dp), pointer, dimension(:) :: logWs
         real(dp), pointer, dimension(:,:) :: logRho_p
         real(dp), pointer, dimension(:) :: &
            logPgas_p, logE_p, logS_p, chiRho_p, chiT_p, Cp_p, Cv_p, &
            dE_dRho_p, dS_dT_p, dS_dRho_p, mu_p, log_free_e_p, &
            gamma1_p, gamma3_p, grad_ad_p, eta_p, &
            fneut_H_p, fneut_He_p, fneut_C_p, fneut_N_p, fneut_O_p, &
            fneut_Ne_p, fneut_Mg_p, fneut_Si_p, fneut_Fe_p, &
            Z_H_p, Z_He_p, Z_C_p, Z_N_p, Z_O_p, Z_Ne_p, Z_Mg_p, Z_Si_p, Z_Fe_p, &
            logpp_H_p, logpp_He_p, logpp_C_p, logpp_N_p, logpp_O_p, &
            logpp_Ne_p, logpp_Mg_p, logpp_Si_p, logpp_Fe_p
            
         ! density data
         integer :: num_logQs
         real(dp) :: logQ_min, logQ_max, del_logQ
         real(dp), pointer, dimension(:) :: logQs
         real(dp), pointer, dimension(:,:) :: &
            logPgas_d, logE_d, logS_d, chiRho_d, chiT_d, Cp_d, Cv_d, &
            dE_dRho_d, dS_dT_d, dS_dRho_d, mu_d, log_free_e_d, gamma1_d, gamma3_d, grad_ad_d, eta_d, &
            fneut_H_d, fneut_He_d, fneut_C_d, fneut_N_d, fneut_O_d, &
            fneut_Ne_d, fneut_Mg_d, fneut_Si_d, fneut_Fe_d, &
            Z_H_d, Z_He_d, Z_C_d, Z_N_d, Z_O_d, Z_Ne_d, Z_Mg_d, Z_Si_d, Z_Fe_d, &
            logpp_H_d, logpp_He_d, logpp_C_d, logpp_N_d, logpp_O_d, &
            logpp_Ne_d, logpp_Mg_d, logpp_Si_d, logpp_Fe_d
         
         include 'formats.dek'
         
         ierr = 0
         write(*,*) 'read ' // trim(fname_in)

         iounit = alloc_iounit(ierr)
         if (ierr /= 0) stop 1
         
         call setup_pgas_data
         
         if (len_trim(outfile_eosPT) > 0) call write_eosPT_data
         
         call create_density_data
         
         if (len_trim(outfile_eosDT) > 0) call write_eosDT_data
         
         if (len_trim(outfile_ionization) > 0) call write_ion_data
         
         call free_iounit(iounit)
         
         
         contains

         
         subroutine write_eosPT_data
            integer :: i, j, num_mesa_logWs
            real(dp) :: logW, logT, logW_max_mesa
            include 'formats.dek'
            
            write(*,*) 'write ' // trim(outfile_eosPT)
         
            open(iounit, file=trim(outfile_eosPT), action='write', iostat=ierr)
            if (ierr /= 0) then
               write(*, *) 'failed to open ' // trim(outfile_eosPT)
               call free_iounit(iounit)
               return
            end if
            
            logW_max_mesa = -2.9d0
            num_mesa_logWs = 143

            write(iounit,'(99(a14))') &
               'version', 'X', 'Z', 'num logTs', 'logT min', 'logT max', 'del logT',  &
            	'num logWs', 'logW min', 'logW max', 'del logW'
      
            write(iounit,'(i14,2f14.4,2(i10,4x,3(f14.4)))') &
               version, X, Z, num_logTs, logT_min, logT_max, del_logT, &
         		num_mesa_logWs, logW_min, logW_max_mesa, del_logW

            do i = 1, num_mesa_logWs
               logW = logW_min + (i-1) * del_logW

               write(iounit,'(/,7x,a)') 'logW = logPgas - 4*logT' 
               write(iounit,'(2x,f14.6/)') logW
               write(iounit,'(a5,1x,3(a9,1x),7(a12,1x),1(a7,1x),1(a10,1x),3(a9,1x),1(a10,1x))')  &
                  'logT', 'logRho', 'logE', 'logS', 'chiRho', 'chiT', &
                  'Cp', 'Cv', 'dE_dRho', 'dS_dT', 'dS_dRho', &
                  'mu', 'log_free_e', 'gamma1', 'gamma3', 'grad_ad', 'eta'

               do j = 1, num_logTs
                  logT = logT_min + (j-1) * del_logT                  
                  write(iounit,'(f5.2,3(f10.5),7(1pe13.5),0p,f8.4,f11.5,3f10.5,f11.5)') &
                     logT, logRho_p(i,j), logE_p(1,i,j), max(-99d0,logS_p(1,i,j)), & 
                     chiRho_p(1,i,j), chiT_p(1,i,j), Cp_p(1,i,j), Cv_p(1,i,j), &
                     dE_dRho_p(1,i,j), dS_dT_p(1,i,j), dS_dRho_p(1,i,j), & 
                     mu_p(1,i,j), &
                     log_free_e_p(1,i,j), &
                     gamma1_p(1,i,j), gamma3_p(1,i,j), grad_ad_p(1,i,j), &
                     eta_p(1,i,j)
               end do
                        
               write(iounit,*)
         
            end do
      
            write(iounit,*)
            write(iounit,*)

            close(iounit)

         end subroutine write_eosPT_data


         subroutine write_eosDT_data
            integer :: i, j
            real(dp) :: logQ, logT
            include 'formats.dek'
            
            write(*,*) 'write ' // trim(outfile_eosDT)
         
            open(iounit, file=trim(outfile_eosDT), action='write', iostat=ierr)
            if (ierr /= 0) then
               write(*, *) 'failed to open ' // trim(outfile_eosDT)
               call free_iounit(iounit)
               return
            end if
      
            write(iounit,'(99(a14))') &
               'version', 'X', 'Z', 'num logTs', 'logT min', 'logT max', 'del logT', &
            	'num logQs', 'logQ min', 'logQ max', 'del logQ'
      
            write(iounit,'(i14,2f14.4,2(i10,4x,3(f14.4)))') & 
               version, X, Z, num_logTs, logT_min, logT_max, del_logT, &
            	num_logQs, logQ_min, logQ_max, del_logQ
            
            do i = 1, num_logQs
               logQ = logQ_min + (i-1) * del_logQ

               write(iounit,'(/,7x,a)') 'logQ = logRho - 2*logT + 12' 
               write(iounit,'(2x,f14.6/)') logQ
               write(iounit, &
                  '(a5,1x,3(a9,1x),7(a12,1x),1(a7,1x),1(a10,1x),3(a9,1x),1(a10,1x))') & 
                  'logT', 'logPgas', 'logE', 'logS', &  ! a5,1x,3(a9,1x)
                  'chiRho', 'chiT', 'Cp', 'Cv', 'dE_dRho', 'dS_dT', 'dS_dRho', & ! 7(a12,1x)
                  'mu', & ! 1(a7,1x)
                  'log_free_e', & ! 1(a10,1x)
                  'gamma1', 'gamma3', 'grad_ad', & ! 3(a9,1x)
                  'eta' ! 1(a10,1x)
         
               do j = 1, num_logTs
                  logT = logT_min + (j-1) * del_logT                  
                  write(iounit, &
                     '(f5.2,3(f10.5),7(1pe13.5),0p,f8.4,f11.5,3f10.5,f11.5)') &
                     logT, logPgas_d(i,j), logE_d(i,j), max(-99d0,logS_d(i,j)), & ! f5.2,3(f10.5)
                     chiRho_d(i,j), chiT_d(i,j), Cp_d(i,j), Cv_d(i,j), &
                     dE_dRho_d(i,j), dS_dT_d(i,j), dS_dRho_d(i,j), & ! 7(1pe13.5)
                     mu_d(i,j), & ! 1(0pf8.4)
                     log_free_e_d(i,j), & ! 1(0pf11.5)
                     gamma1_d(i,j), gamma3_d(i,j), grad_ad_d(i,j), & ! 3(0pf10.5)
                     eta_d(i,j) ! 1(0pf11.5)
               end do
                        
               write(iounit,*)
         
            end do
      
            write(iounit,*)
            write(iounit,*)

            close(iounit)

         end subroutine write_eosDT_data

         
         subroutine write_ion_data
            integer :: i, j, ion_num_logQs, ion_num_logTs
            real(dp) :: logQ, logT, ion_logQ_min, ion_logQ_max, ion_del_logQ, &
               ion_logT_min, ion_logT_max, ion_del_logT
            integer, parameter :: skQ=3, skT=3
            include 'formats.dek'
            
            ion_del_logQ = skQ*del_logQ
            ion_num_logQs = (num_logQs - 1)/skQ + 1
            ion_logQ_min = logQ_min
            ion_logQ_max = logQ_min + (ion_num_logQs - 1)*ion_del_logQ
            
            ion_del_logT = skT*del_logT
            ion_num_logTs = (num_logTs - 1)/skT + 1
            ion_logT_min = logT_min
            ion_logT_max = logT_min + (ion_num_logTs - 1)*ion_del_logT
            
            write(*,*) 'write ' // trim(outfile_ionization)
         
            open(iounit, file=trim(outfile_ionization), action='write', iostat=ierr)
            if (ierr /= 0) then
               write(*, *) 'failed to open ' // trim(outfile_ionization)
               call free_iounit(iounit)
               return
            end if
      
            write(iounit,'(99(a14))') &
               'version', 'X', 'Z', 'num logTs', 'logT min', 'logT max', 'del logT', &
            	'num logQs', 'logQ min', 'logQ max', 'del logQ'
      
            write(iounit,'(i14,2f14.4,2(i10,4x,3(f14.4)))') & 
               version, X, Z, ion_num_logTs, ion_logT_min, ion_logT_max, ion_del_logT, &
            	ion_num_logQs, ion_logQ_min, ion_logQ_max, ion_del_logQ

            do i = 1, num_logQs, skQ
               logQ = logQ_min + (i-1) * del_logQ

               write(iounit,'(/,7x,a)') 'logQ = logRho - 2*logT + 12' 
               write(iounit,'(2x,f14.6/)') logQ
               write(iounit, &
                  '(a5,1x,99(a7,1x))') & 
                  'logT', 'logPgas', &
                  'lpp_H', 'lpp_He', 'lpp_C', 'lpp_N', 'lpp_O', &
                  'lpp_Ne', 'lpp_Mg', 'lpp_Si', 'lpp_Fe', &
                  'Z_H', 'Z_He', 'Z_C', 'Z_N', 'Z_O', &
                  'Z_Ne', 'Z_Mg', 'Z_Si', 'Z_Fe', &
                  'f_H', 'f_He', 'f_C', 'f_N', 'f_O', &
                  'f_Ne', 'f_Mg', 'f_Si', 'f_Fe'
         
               do j = 1, num_logTs, skT
                  logT = logT_min + (j-1) * del_logT
                  
                  write(iounit, &
                     '(f5.2,99f8.3)') &                     
                     logT, logPgas_d(i,j), &
                     logpp_H_d(i,j), logpp_He_d(i,j), logpp_C_d(i,j), logpp_N_d(i,j), logpp_O_d(i,j), &
                     logpp_Ne_d(i,j), logpp_Mg_d(i,j), logpp_Si_d(i,j), logpp_Fe_d(i,j), &
                     Z_H_d(i,j), Z_He_d(i,j), Z_C_d(i,j), Z_N_d(i,j), Z_O_d(i,j), &
                     Z_Ne_d(i,j), Z_Mg_d(i,j), Z_Si_d(i,j), Z_Fe_d(i,j), &
                     fneut_H_d(i,j), fneut_He_d(i,j), fneut_C_d(i,j), fneut_N_d(i,j), fneut_O_d(i,j), & 
                     fneut_Ne_d(i,j), fneut_Mg_d(i,j), fneut_Si_d(i,j), fneut_Fe_d(i,j)
                     
               end do
         
               write(iounit,*)
         
            end do
      
            write(iounit,*)
            write(iounit,*)

            close(iounit)

         end subroutine write_ion_data

         
         subroutine create_density_data
            integer :: i, j
            real(dp) :: logRho
         
            include 'formats.dek'
         
            num_logQs = 524
            logQ_min = -10d0
            logQ_max = 5.69d0
            del_logQ = 0.03d0
         
            allocate( &
               logQs(num_logQs), &
               logPgas_d(num_logQs, num_logTs), &
               logE_d(num_logQs, num_logTs), &
               logS_d(num_logQs, num_logTs), &
               chiRho_d(num_logQs, num_logTs), &
               chiT_d(num_logQs, num_logTs), &
               Cp_d(num_logQs, num_logTs), &
               Cv_d(num_logQs, num_logTs), &
               dE_dRho_d(num_logQs, num_logTs), &
               dS_dT_d(num_logQs, num_logTs), &
               dS_dRho_d(num_logQs, num_logTs), &
               mu_d(num_logQs, num_logTs), &
               log_free_e_d(num_logQs, num_logTs), &
               gamma1_d(num_logQs, num_logTs), &
               gamma3_d(num_logQs, num_logTs), &
               grad_ad_d(num_logQs, num_logTs), &
               eta_d(num_logQs, num_logTs), &
               fneut_H_d(num_logQs, num_logTs), &
               fneut_He_d(num_logQs, num_logTs), &
               fneut_C_d(num_logQs, num_logTs), &
               fneut_N_d(num_logQs, num_logTs), &
               fneut_O_d(num_logQs, num_logTs), &
               fneut_Ne_d(num_logQs, num_logTs), &
               fneut_Mg_d(num_logQs, num_logTs), &
               fneut_Si_d(num_logQs, num_logTs), &
               fneut_Fe_d(num_logQs, num_logTs), &
               Z_H_d(num_logQs, num_logTs), &
               Z_He_d(num_logQs, num_logTs), &
               Z_C_d(num_logQs, num_logTs), &
               Z_N_d(num_logQs, num_logTs), &
               Z_O_d(num_logQs, num_logTs), &
               Z_Ne_d(num_logQs, num_logTs), &
               Z_Mg_d(num_logQs, num_logTs), &
               Z_Si_d(num_logQs, num_logTs), &
               Z_Fe_d(num_logQs, num_logTs), &
               logpp_H_d(num_logQs, num_logTs), &
               logpp_He_d(num_logQs, num_logTs), &
               logpp_C_d(num_logQs, num_logTs), &
               logpp_N_d(num_logQs, num_logTs), &
               logpp_O_d(num_logQs, num_logTs), &
               logpp_Ne_d(num_logQs, num_logTs), &
               logpp_Mg_d(num_logQs, num_logTs), &
               logpp_Si_d(num_logQs, num_logTs), &
               logpp_Fe_d(num_logQs, num_logTs))
            
            do i = 1, num_logQs
               logQs(i) = logQ_min + (i-1)*del_logQ
            end do
            if (abs(logQs(num_logQs) - logQ_max) > 1d-8*logQ_max) then
               write(*,2) 'bad logQs', num_logQs, logQs(num_logQs) - logQ_max, logQs(num_logQs), logQ_max
               stop 1
            end if
               
            do i = 1, num_logQs               
               do j = 1, num_logTs
                  logRho = logQs(i) + 2*logTs(j) - 12
                  call set_val(logPgas_d, 'logPgas_d', logRho, i, j, logPgas_p)
                  call set_val(logE_d, 'logE_d', logRho, i, j, logE_p)
                  call set_val(logS_d, 'logS_d', logRho, i, j, logS_p)
                  call set_val(chiRho_d, 'chiRho_d', logRho, i, j, chiRho_p)
                  call set_val(chiT_d, 'chiT_d', logRho, i, j, chiT_p)
                  call set_val(Cp_d, 'Cp_d', logRho, i, j, Cp_p)
                  call set_val(Cv_d, 'Cv_d', logRho, i, j, Cv_p)
                  call set_val(dE_dRho_d, 'dE_dRho_d', logRho, i, j, dE_dRho_p)
                  call set_val(dS_dT_d, 'dS_dT_d', logRho, i, j, dS_dT_p)
                  call set_val(dS_dRho_d, 'dS_dRho_d', logRho, i, j, dS_dRho_p)
                  call set_val(mu_d, 'mu_d', logRho, i, j, mu_p)
                  call set_val(log_free_e_d, 'log_free_e_d', logRho, i, j, log_free_e_p)
                  call set_val(gamma1_d, 'gamma1_d', logRho, i, j, gamma1_p)
                  call set_val(gamma3_d, 'gamma3_d', logRho, i, j, gamma3_p)
                  call set_val(grad_ad_d, 'grad_ad_d', logRho, i, j, grad_ad_p)
                  call set_val(eta_d, 'eta_d', logRho, i, j, eta_p)
                  call set_val(fneut_H_d, 'fneut_H_d', logRho, i, j, fneut_H_p)
                  call set_val(fneut_He_d, 'fneut_He_d', logRho, i, j, fneut_He_p)
                  call set_val(fneut_C_d, 'fneut_C_d', logRho, i, j, fneut_C_p)
                  call set_val(fneut_N_d, 'fneut_N_d', logRho, i, j, fneut_N_p)
                  call set_val(fneut_O_d, 'fneut_O_d', logRho, i, j, fneut_O_p)
                  call set_val(fneut_Ne_d, 'fneut_Ne_d', logRho, i, j, fneut_Ne_p)
                  call set_val(fneut_Mg_d, 'fneut_Mg_d', logRho, i, j, fneut_Mg_p)
                  call set_val(fneut_Si_d, 'fneut_Si_d', logRho, i, j, fneut_Si_p)
                  call set_val(fneut_Fe_d, 'fneut_Fe_d', logRho, i, j, fneut_Fe_p)
                  call set_val(Z_H_d, 'Z_H_d', logRho, i, j, Z_H_p)
                  call set_val(Z_He_d, 'Z_He_d', logRho, i, j, Z_He_p)
                  call set_val(Z_C_d, 'Z_C_d', logRho, i, j, Z_C_p)
                  call set_val(Z_N_d, 'Z_N_d', logRho, i, j, Z_N_p)
                  call set_val(Z_O_d, 'Z_O_d', logRho, i, j, Z_O_p)
                  call set_val(Z_Ne_d, 'Z_Ne_d', logRho, i, j, Z_Ne_p)
                  call set_val(Z_Mg_d, 'Z_Mg_d', logRho, i, j, Z_Mg_p)
                  call set_val(Z_Si_d, 'Z_Si_d', logRho, i, j, Z_Si_p)
                  call set_val(Z_Fe_d, 'Z_Fe_d', logRho, i, j, Z_Fe_p)
                  call set_val(logpp_H_d, 'logpp_H_d', logRho, i, j, logpp_H_p)
                  call set_val(logpp_He_d, 'logpp_He_d', logRho, i, j, logpp_He_p)
                  call set_val(logpp_C_d, 'logpp_C_d', logRho, i, j, logpp_C_p)
                  call set_val(logpp_N_d, 'logpp_N_d', logRho, i, j, logpp_N_p)
                  call set_val(logpp_O_d, 'logpp_O_d', logRho, i, j, logpp_O_p)
                  call set_val(logpp_Ne_d, 'logpp_Ne_d', logRho, i, j, logpp_Ne_p)
                  call set_val(logpp_Mg_d, 'logpp_Mg_d', logRho, i, j, logpp_Mg_p)
                  call set_val(logpp_Si_d, 'logpp_Si_d', logRho, i, j, logpp_Si_p)
                  call set_val(logpp_Fe_d, 'logpp_Fe_d', logRho, i, j, logpp_Fe_p)  
               end do
            end do
         
         end subroutine create_density_data
         
         
         subroutine set_val(v,str,logRho,i,j,f1)
            use interp_1d_lib
            real(dp) :: v(:,:) ! (num_logQs,num_logTs)
            character (len=*), intent(in) :: str
            real(dp), intent(in) :: logRho
            integer, intent(in) :: i, j
            real(dp), intent(in), pointer :: f1(:) ! =(4,num_logWs,num_logTs)
            real(dp), pointer :: f(:)
            include 'formats.dek'
            integer :: ierr
            ierr = 0
            if (logRho >= logRho_p(num_logWs,j)) then
               v(i,j) = f(1,num_logWs,j)
               return
            end if
            f(1:4*num_logWs) => f1(1+(j-1)*4*num_logWs:j*4*num_logWs)
            call interp_value(logRho_p(:,j), num_logWs, f, logRho, v(i,j), ierr)
            if (ierr /= 0) then
               write(*,3) 'failed to interp value for ' // str, i, j
               stop 1
            end if
         end subroutine set_val
         
         
         subroutine setup_pgas_data
         
            real(dp) :: logT
            integer :: version_in
            
            include 'formats.dek'
         
            open(iounit, file=trim(fname_in), action='read', status='old', iostat=ierr)
            if (ierr /= 0) then
               write(*, *) 'failed to open ' // trim(fname_in)
               call free_iounit(iounit)
               return
            end if           
                
            call skip_lines(2,iounit,ierr)
            if (ierr /= 0) stop 1
            read(iounit,*,iostat=ierr) version_in, X, Z, &
               num_logTs, logT_min, logT_max, del_logT, &
               num_logWs, logW_min, logW_max, del_logW
            if (ierr /= 0) stop 1
         
            !write(*,2) 'num_logTs', num_logTs
            !write(*,2) 'num_logWs', num_logWs
            !stop
         
            allocate( &
               logTs(num_logTs), logWs(num_logWs), &
               logRho_p(num_logWs, num_logTs), &
               logPgas_p(4*num_logWs*num_logTs), &
               logE_p(4*num_logWs*num_logTs), &
               logS_p(4*num_logWs*num_logTs), &
               chiRho_p(4*num_logWs*num_logTs), &
               chiT_p(4*num_logWs*num_logTs), &
               Cp_p(4*num_logWs*num_logTs), &
               Cv_p(4*num_logWs*num_logTs), &
               dE_dRho_p(4*num_logWs*num_logTs), &
               dS_dT_p(4*num_logWs*num_logTs), &
               dS_dRho_p(4*num_logWs*num_logTs), &
               mu_p(4*num_logWs*num_logTs), &
               log_free_e_p(4*num_logWs*num_logTs), &
               gamma1_p(4*num_logWs*num_logTs), &
               gamma3_p(4*num_logWs*num_logTs), &
               grad_ad_p(4*num_logWs*num_logTs), &
               eta_p(4*num_logWs*num_logTs), &
               fneut_H_p(4*num_logWs*num_logTs), &
               fneut_He_p(4*num_logWs*num_logTs), &
               fneut_C_p(4*num_logWs*num_logTs), &
               fneut_N_p(4*num_logWs*num_logTs), &
               fneut_O_p(4*num_logWs*num_logTs), &
               fneut_Ne_p(4*num_logWs*num_logTs), &
               fneut_Mg_p(4*num_logWs*num_logTs), &
               fneut_Si_p(4*num_logWs*num_logTs), &
               fneut_Fe_p(4*num_logWs*num_logTs), &
               Z_H_p(4*num_logWs*num_logTs), &
               Z_He_p(4*num_logWs*num_logTs), &
               Z_C_p(4*num_logWs*num_logTs), &
               Z_N_p(4*num_logWs*num_logTs), &
               Z_O_p(4*num_logWs*num_logTs), &
               Z_Ne_p(4*num_logWs*num_logTs), &
               Z_Mg_p(4*num_logWs*num_logTs), &
               Z_Si_p(4*num_logWs*num_logTs), &
               Z_Fe_p(4*num_logWs*num_logTs), &
               logpp_H_p(4*num_logWs*num_logTs), &
               logpp_He_p(4*num_logWs*num_logTs), &
               logpp_C_p(4*num_logWs*num_logTs), &
               logpp_N_p(4*num_logWs*num_logTs), &
               logpp_O_p(4*num_logWs*num_logTs), &
               logpp_Ne_p(4*num_logWs*num_logTs), &
               logpp_Mg_p(4*num_logWs*num_logTs), &
               logpp_Si_p(4*num_logWs*num_logTs), &
               logpp_Fe_p(4*num_logWs*num_logTs))

            call skip_lines(2,iounit,ierr)
            if (ierr /= 0) stop 1

            do i = 1, num_logWs
               read(iounit,*,iostat=ierr) logWs(i)
               if (ierr /= 0) stop 1
               call skip_lines(2,iounit,ierr)
               if (ierr /= 0) stop 1
               !write(*,2) 'logWs(i)', i, logWs(i)
               do j = 1, num_logTs
                  k = 1 + (i-1)*num_logWs + (j-1)*num_logWs*num_logTs
                  read(iounit,*,iostat=ierr) logT, logRho_p(i,j), &
                     logE_p(k), logS_p(k), chiRho_p(k), chiT_p(k), &
                     Cp_p(k), Cv_p(k), dE_dRho_p(k), dS_dT_p(k), dS_dRho_p(k), mu_p(k), &
                     log_free_e_p(k), gamma1_p(k), gamma3_p(k), grad_ad_p(k), eta_p(k), &
                     fneut_H_p(k), fneut_He_p(k), fneut_C_p(k), fneut_N_p(k), &
                     fneut_O_p(k), fneut_Ne_p(k), fneut_Mg_p(k), fneut_Si_p(k), fneut_Fe_p(k), &
                     Z_H_p(k), Z_He_p(k), Z_C_p(k), Z_N_p(k), Z_O_p(k), &
                     Z_Ne_p(k), Z_Mg_p(k), Z_Si_p(k), Z_Fe_p(k), &
                     logpp_H_p(k), logpp_He_p(k), logpp_C_p(k), logpp_N_p(k), &
                     logpp_O_p(k), logpp_Ne_p(k), logpp_Mg_p(k), logpp_Si_p(k), logpp_Fe_p(k)
                  if (ierr /= 0) stop 1
                  if (i == 1) then
                     logTs(j) = logT
                  else if (abs(logTs(j) - logT) > 1d-4) then
                     write(*,3) 'inconsistent logT', i, j, logTs(j) - logT, logTs(j), logT
                     stop 1
                  end if
                  logPgas_p(1,i,j) = logWs(i) + 4*logT
                  logE_p(1,i,j) = log10(max(1d-99,logE_p(1,i,j)))
               end do
               if (i == num_logWs) exit
               call skip_lines(5,iounit,ierr)
               if (ierr /= 0) stop 1
            end do

            close(iounit)

            write(*,*) 'done reading ' // trim(fname_in)
         
            call smooth_where_necessary
         
            call create_interpolant(logPgas_p, 'logPgas_p')
            call create_interpolant(logE_p, 'logE_p')
            call create_interpolant(logS_p, 'logS_p')
            call create_interpolant(chiRho_p, 'chiRho_p')
            call create_interpolant(chiT_p, 'chiT_p')
            call create_interpolant(Cp_p, 'Cp_p')
            call create_interpolant(Cv_p, 'Cv_p')
            call create_interpolant(dE_dRho_p, 'dE_dRho_p')
            call create_interpolant(dS_dT_p, 'dS_dT_p')
            call create_interpolant(dS_dRho_p, 'dS_dRho_p')
            call create_interpolant(mu_p, 'mu_p')
            call create_interpolant(log_free_e_p, 'log_free_e_p')
            call create_interpolant(gamma1_p, 'gamma1_p')
            call create_interpolant(gamma3_p, 'gamma3_p')
            call create_interpolant(grad_ad_p, 'grad_ad_p')
            call create_interpolant(eta_p, 'eta_p')
            call create_interpolant(fneut_H_p, 'fneut_H_p')
            call create_interpolant(fneut_He_p, 'fneut_He_p')
            call create_interpolant(fneut_C_p, 'fneut_C_p')
            call create_interpolant(fneut_N_p, 'fneut_N_p')
            call create_interpolant(fneut_O_p, 'fneut_O_p')
            call create_interpolant(fneut_Ne_p, 'fneut_Ne_p')
            call create_interpolant(fneut_Mg_p, 'fneut_Mg_p')
            call create_interpolant(fneut_Si_p, 'fneut_Si_p')
            call create_interpolant(fneut_Fe_p, 'fneut_Fe_p')
            call create_interpolant(Z_H_p, 'Z_H_p')
            call create_interpolant(Z_He_p, 'Z_He_p')
            call create_interpolant(Z_C_p, 'Z_C_p')
            call create_interpolant(Z_N_p, 'Z_N_p')
            call create_interpolant(Z_O_p, 'Z_O_p')
            call create_interpolant(Z_Ne_p, 'Z_Ne_p')
            call create_interpolant(Z_Mg_p, 'Z_Mg_p')
            call create_interpolant(Z_Si_p, 'Z_Si_p')
            call create_interpolant(Z_Fe_p, 'Z_Fe_p')
            call create_interpolant(logpp_H_p, 'logpp_H_p')
            call create_interpolant(logpp_He_p, 'logpp_He_p')
            call create_interpolant(logpp_C_p, 'logpp_C_p')
            call create_interpolant(logpp_N_p, 'logpp_N_p')
            call create_interpolant(logpp_O_p, 'logpp_O_p')
            call create_interpolant(logpp_Ne_p, 'logpp_Ne_p')
            call create_interpolant(logpp_Mg_p, 'logpp_Mg_p')
            call create_interpolant(logpp_Si_p, 'logpp_Si_p')
            call create_interpolant(logpp_Fe_p, 'logpp_Fe_p')

         end subroutine setup_pgas_data
         
         
         subroutine smooth_where_necessary
            real(dp) :: work(4*num_logWs*num_logTs)
            call smooth1_where_necessary(logPgas_p, 'logPgas_p')
            call smooth1_where_necessary(logE_p, 'logE_p')
            call smooth1_where_necessary(logS_p, 'logS_p')
            call smooth1_where_necessary(chiRho_p, 'chiRho_p')
            call smooth1_where_necessary(chiT_p, 'chiT_p')
            call smooth1_where_necessary(Cp_p, 'Cp_p')
            call smooth1_where_necessary(Cv_p, 'Cv_p')
            call smooth1_where_necessary(dE_dRho_p, 'dE_dRho_p')
            call smooth1_where_necessary(dS_dT_p, 'dS_dT_p')
            call smooth1_where_necessary(dS_dRho_p, 'dS_dRho_p')
            call smooth1_where_necessary(mu_p, 'mu_p')
            call smooth1_where_necessary(log_free_e_p, 'log_free_e_p')
            call smooth1_where_necessary(gamma1_p, 'gamma1_p')
            call smooth1_where_necessary(gamma3_p, 'gamma3_p')
            call smooth1_where_necessary(grad_ad_p, 'grad_ad_p')
            call smooth1_where_necessary(eta_p, 'eta_p')
            call smooth1_where_necessary(fneut_H_p, 'fneut_H_p')
            call smooth1_where_necessary(fneut_He_p, 'fneut_He_p')
            call smooth1_where_necessary(fneut_C_p, 'fneut_C_p')
            call smooth1_where_necessary(fneut_N_p, 'fneut_N_p')
            call smooth1_where_necessary(fneut_O_p, 'fneut_O_p')
            call smooth1_where_necessary(fneut_Ne_p, 'fneut_Ne_p')
            call smooth1_where_necessary(fneut_Mg_p, 'fneut_Mg_p')
            call smooth1_where_necessary(fneut_Si_p, 'fneut_Si_p')
            call smooth1_where_necessary(fneut_Fe_p, 'fneut_Fe_p')
            call smooth1_where_necessary(Z_H_p, 'Z_H_p')
            call smooth1_where_necessary(Z_He_p, 'Z_He_p')
            call smooth1_where_necessary(Z_C_p, 'Z_C_p')
            call smooth1_where_necessary(Z_N_p, 'Z_N_p')
            call smooth1_where_necessary(Z_O_p, 'Z_O_p')
            call smooth1_where_necessary(Z_Ne_p, 'Z_Ne_p')
            call smooth1_where_necessary(Z_Mg_p, 'Z_Mg_p')
            call smooth1_where_necessary(Z_Si_p, 'Z_Si_p')
            call smooth1_where_necessary(Z_Fe_p, 'Z_Fe_p')
            call smooth1_where_necessary(logpp_H_p, 'logpp_H_p')
            call smooth1_where_necessary(logpp_He_p, 'logpp_He_p')
            call smooth1_where_necessary(logpp_C_p, 'logpp_C_p')
            call smooth1_where_necessary(logpp_N_p, 'logpp_N_p')
            call smooth1_where_necessary(logpp_O_p, 'logpp_O_p')
            call smooth1_where_necessary(logpp_Ne_p, 'logpp_Ne_p')
            call smooth1_where_necessary(logpp_Mg_p, 'logpp_Mg_p')
            call smooth1_where_necessary(logpp_Si_p, 'logpp_Si_p')
            call smooth1_where_necessary(logpp_Fe_p, 'logpp_Fe_p')
         end subroutine smooth_where_necessary
         
         
         subroutine smooth1_where_necessary(f1,str)
            real(dp), pointer :: f1(:)
            character (len=*), intent(in) :: str
            real(dp) :: work(num_logWs, num_logTs)
            real(dp) :: logT_smooth_min, logT_smooth_max
            real(dp) :: logRho_smooth_min, logRho_smooth_max
            integer :: i, j, n, nmax, im1, i00, ip1, jm1, j00, jp1
            real(dp), pointer :: f(:,:,:) ! 4*num_logWs*num_logTs
            f(1:4,1:num_logWs,1:num_logTs) => f1(1:4*num_logWs*num_logTs)
            logT_smooth_min = 2.1d0
            logT_smooth_max = 5.7d0
            logRho_smooth_min = 0.2d0
            logRho_smooth_max = 3.3d0
            nmax = 100
            if (Z > 0.999) then
               logT_smooth_min = 2.1d0
               logT_smooth_max = 6.7d0
               logRho_smooth_min = 0.2d0
               logRho_smooth_max = 100d0
               nmax = 200
            end if
            do n = 1, nmax
               work(:,:) = f(1,:,:)
               do i = 1, num_logWs
                  im1 = max(1,i-1)
                  i00 = i
                  ip1 = min(num_logWs,i+1)
                  do j = 1, num_logTs
                     if (logRho_p(i,j) < logRho_smooth_min .or. logRho_p(i,j) > logRho_smooth_max) cycle
                     if (logTs(j) < logT_smooth_min .or. logTs(j) > logT_smooth_max) cycle
                     jm1 = max(1,j-1)
                     j00 = j
                     jp1 = min(num_logTs,j+1)
                     f(1,i,j) = ( &
                        work(im1,jm1)*1 + work(i00,jm1)*2 + work(ip1,jm1)*1 + &
                        work(im1,j00)*2 + work(i00,j00)*4 + work(ip1,j00)*2 + &
                        work(im1,jp1)*1 + work(i00,jp1)*2 + work(ip1,jp1)*1) / 16
                  end do
               end do
               logT_smooth_min = logT_smooth_min + 0.005
               logT_smooth_max = logT_smooth_max - 0.0075
               logRho_smooth_min = logRho_smooth_min + 0.005
               logRho_smooth_max = logRho_smooth_max - 0.005
            end do
         end subroutine smooth1_where_necessary
         
         
         subroutine create_interpolant(f1,str)
            use interp_1d_lib
            use interp_1d_def
            real(dp), pointer :: f1(:)
            character (len=*), intent(in) :: str
            integer, parameter :: nwork = pm_work_size
            real(dp), target :: work_ary(num_logWs*nwork)
            real(dp), pointer :: work(:), f(:)
            integer :: j, ierr
            ierr = 0
            work => work_ary
            do j = 1, num_logTs
               f(1:4*num_logWs) => f1(1+(j-1)*4*num_logWs:j*4*num_logWs)
               call interp_pm(logRho_p(:,j), num_logWs, f, nwork, work, ierr)
               if (ierr /= 0) then
                  write(*,*) 'failed to create interpolant for ' // trim(str)
                  stop 1
               end if
            end do
         end subroutine create_interpolant
      
      
      end subroutine convert_eos_p_to_d      
      
      
      subroutine skip_lines(n,iounit,ierr)
         integer, intent(in) :: n, iounit
         integer, intent(out) :: ierr
         integer :: i
         ierr = 0
         do i=1,n
            read(iounit, *, iostat=ierr) 
            if (ierr /= 0) then
               close(iounit)
               call free_iounit(iounit)
               return
            end if
         end do
      end subroutine skip_lines
      

      end module mod_ion_create_tables


