! ***********************************************************************
!
!   Copyright (C) 2013  Josiah Schwab, Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************




program test_ecapture

  use const_def
  use ecapture_lib
  use weak_lib
  use num_lib
  use utils_lib
  implicit none

  real(dp) :: X, Z, Y, abar, zbar, z2bar, ye
  integer, parameter :: species = 7
  integer, parameter :: h1=1, he4=2, c12=3, n14=4, o16=5, ne20=6, mg24=7
  integer, pointer, dimension(:) :: net_iso, chem_id
  real(dp) :: xa(species)
  character (len=256) :: my_mesa_dir
  character (len=32) :: filename
  integer :: handle

  do_ecapture = .true.

  call do_test_startup

  ! first test the phase space integral functions

  call do_test_derivs(-10.0_dp,1d-3,0.0_dp)
  call do_test_derivs(-10.0_dp,0.0_dp,1d-3)

  call do_test_derivs(10.0_dp,1d-3,0.0_dp)
  call do_test_derivs(10.0_dp,0.0_dp,1d-3)

  call do_test_coulomb

  call do_test_ecapture


!  call do_test_betadecay

contains

  subroutine do_test_betadecay

    real(dp) :: beta  ! mec2 / kT
    real(dp) :: zeta  ! Q_n / kT
    real(dp) :: eta   ! chemical potential / kT
    real(dp) :: deta_dlnT, deta_dlnRho ! and derivs

    real(dp) :: I, J   ! phase space integral
    real(dp) :: dI_dlnT, dI_dlnRho ! and derivatives
    real(dp) :: dJ_dlnT, dJ_dlnRho ! and derivatives

    integer :: k, neta
    real(dp) :: etamin, etamax, deltaeta
    real(dp) :: dx

    beta = 1
    zeta = 10

    etamin = 1
    etamax = 100
    deltaeta = 1
    
    neta = int((etamax - etamin)/deltaeta)

    do k = 0, neta

       eta = etamin + deltaeta * k
       deta_dlnRho = eta / 3.0
       deta_dlnT = -eta
       
       call psi_Iee_and_Jee(beta, zeta, eta, deta_dlnT, deta_dlnRho, &
            I, dI_dlnT, dI_dlnRho, J, dJ_dlnT, dJ_dlnRho)

       write(*,*) eta, I, J

    end do

  end subroutine do_test_betadecay



  subroutine do_test_derivs(q,dx,dy)

    real(dp) :: beta  ! mec2 / kT
    real(dp) :: zeta  ! Q_n / kT
    real(dp) :: eta   ! chemical potential / kT
    real(dp) :: deta_dlnT, deta_dlnRho ! and derivs

    real(dp) :: I, J   ! phase space integral
    real(dp) :: dI_dlnT, dI_dlnRho ! and derivatives
    real(dp) :: dJ_dlnT, dJ_dlnRho ! and derivatives

    real(dp) :: Im, Jm   ! phase space integral
    real(dp) :: dIm_dlnT, dIm_dlnRho ! and derivatives
    real(dp) :: dJm_dlnT, dJm_dlnRho ! and derivatives

    real(dp) :: Ip, Jp   ! phase space integral
    real(dp) :: dIp_dlnT, dIp_dlnRho ! and derivatives
    real(dp) :: dJp_dlnT, dJp_dlnRho ! and derivatives

    integer :: k, neta
    real(dp) :: etamin, etamax, deltaeta
    real(dp) :: dx, dy
    real(dp) :: q,mu,D9,T8
    character(len=80) :: fmt

    do k = 1, 3

       select case (k)
       case (1)
          D9 = 1
          T8 = 1
       case (2)
          D9 = 1 + dx
          T8 = 1 + dy
       case(3)
          D9 = 1 - dx
          T8 = 1 - dy
       end select
          
       mu= 10.1 * D9**(1.0/3.0)
       beta = 59.3 / T8

       zeta = q * beta
       eta = mu * beta
       deta_dlnRho = eta / 3.0
       deta_dlnT = -eta

       select case (k)
       case(1)

          if (q < 0) then 
             call psi_Iec_and_Jec(beta, zeta, eta, deta_dlnT, deta_dlnRho, &
                  I, dI_dlnT, dI_dlnRho, J, dJ_dlnT, dJ_dlnRho)
          else
             call psi_Iee_and_Jee(beta, zeta, eta, deta_dlnT, deta_dlnRho, &
                  I, dI_dlnT, dI_dlnRho, J, dJ_dlnT, dJ_dlnRho)
          end if

       case(2)

          if (q < 0) then 
             call psi_Iec_and_Jec(beta, zeta, eta, deta_dlnT, deta_dlnRho, &
                  Ip, dIp_dlnT, dIp_dlnRho, Jp, dJp_dlnT, dJp_dlnRho)
          else
             call psi_Iee_and_Jee(beta, zeta, eta, deta_dlnT, deta_dlnRho, &
                  Ip, dIp_dlnT, dIp_dlnRho, Jp, dJp_dlnT, dJp_dlnRho)
          end if
             
       case(3)

          if (q < 0) then
             call psi_Iec_and_Jec(beta, zeta, eta, deta_dlnT, deta_dlnRho, &
                  Im, dIm_dlnT, dIm_dlnRho, Jm, dJm_dlnT, dJm_dlnRho)
          else
             call psi_Iee_and_Jee(beta, zeta, eta, deta_dlnT, deta_dlnRho, &
                  Im, dIm_dlnT, dIm_dlnRho, Jm, dJm_dlnT, dJm_dlnRho)
          end if
                
       end select
    end do

    fmt = "(A10, 3ES20.12)"
    if (dx.ne.0) then 
       write(*,fmt) "dI_lnRho", dI_dlnRho, (Ip - Im) / (2 * dx), dI_dlnRho/(Ip - Im) * dx * 2
       write(*,fmt) "dJ_lnRho", dJ_dlnRho, (Jp - Jm) / (2 * dx), dJ_dlnRho/(Jp - Jm) * dx * 2
    end if
    if (dy.ne.0) then 
       write(*,fmt) "dI_dlnT", dI_dlnT, (Ip - Im) / (2 * dy), dI_dlnT/(Ip - Im) * dy * 2
       write(*,fmt) "dJ_dlnT", dJ_dlnT, (Jp - Jm) / (2 * dy), dJ_dlnT/(Jp - Jm) * dy * 2
    end if

  end subroutine do_test_derivs



  subroutine do_test_coulomb

    use mod_eval_coulomb

    real(dp) :: mu1, mu2
    real(dp) :: z1, z2, rhoye, T

    rhoye = 10**9.3
    T = 10**8.5

    z1 = 12
    z2 = 11

    zbar = 9.5

    mu1 = do_mu_coulomb(z1, rhoye, T, zbar)
    mu2 = do_mu_coulomb(z2, rhoye, T, zbar)

    write(*,*) mu1, mu2, abs(mu1-mu2) * (8.617e-11) * T

  end subroutine do_test_coulomb


  subroutine do_test_ecapture

    use const_lib
    use utils_lib
    use eos_def
    use eos_lib
    use chem_def
    use chem_lib

    real(dp) :: Rho, T, Pgas, log10Rho, log10T
    real(dp) :: dlnRho_dlnPgas_const_T, dlnRho_dlnT_const_Pgas, d_dlnRho_const_T, d_dlnT_const_Rho
    real(dp), dimension(num_eos_basic_results) :: res, d_dlnd, d_dlnT, d_dabar, d_dzbar
    integer :: ierr

    integer :: i, ir, nr
    integer, pointer :: ids(:)
    real(dp), dimension(:), pointer :: &
         ldecay, d_ldecay_dT9, d_ldecay_dlYeRho, &
         lcapture, d_lcapture_dT9, d_lcapture_dlYeRho, &
         lneutrino, d_lneutrino_dT9, d_lneutrino_dlYeRho, &
         ec_lambda, ec_dlambda_dlnT, ec_dlambda_dlnRho, &
         Q, dQ_dlnT, dQ_dlnRho, &
         Qneu, dQneu_dlnT, dQneu_dlnRho
    real(dp) :: lntwo, logT, T9, dT9, dlnT, YeRho, &
         ye, logRho, dlogRho, eta, d_eta_dlnT, d_eta_dlnRho
    character(len=iso_name_length) :: weak_lhs, weak_rhs
    character(len=2*iso_name_length+1) :: keyp

    integer :: j,k, ND, NT
    real(dp), allocatable, dimension(:) :: logTpts, logDpts
    real(dp) :: log

    real(dp) :: logTmin,logTmax, deltaT
    real(dp) :: logDmin,logDmax, deltaD

    real(dp), dimension(:,:,:), allocatable :: tneu, tec, &
         decay, capture, neutrino, &
         d_decay_dT9, d_decay_dlYeRho, &
         d_capture_dT9, d_capture_dlYeRho, &
         d_neutrino_dT9, d_neutrino_dlYeRho, &
         Q_out, Qneu_out, & 
         lambda, dlambda_dlnT, dlambda_dlnRho

    real(dp), dimension(:,:), allocatable :: neucool

    integer, parameter :: datafile = 12 
    character (len=32) :: outfile

    outfile = "test_data/test-ecapture-"

    allocate(net_iso(num_chem_isos), chem_id(species), stat=ierr)
    if (ierr /= 0) stop 'allocate failed'
    X = 0.70
    Z = 0.02
    call Init_Composition

    nr = 8

    allocate( &
         ids(nr), ldecay(nr), d_ldecay_dT9(nr), d_ldecay_dlYeRho(nr), &
         lcapture(nr), d_lcapture_dT9(nr), d_lcapture_dlYeRho(nr), &
         lneutrino(nr), d_lneutrino_dT9(nr), d_lneutrino_dlYeRho(nr), &
         ec_lambda(nr), ec_dlambda_dlnT(nr), ec_dlambda_dlnRho(nr), &
         Q(nr), dQ_dlnT(nr), dQ_dlnRho(nr), &
         Qneu(nr), dQneu_dlnT(nr), dQneu_dlnRho(nr), &
         stat=ierr)


    ! pick reactions
    ids(1) =  get_weak_rate_id('mg24','na24')
    ids(2) =  get_weak_rate_id('na24','mg24')
    ids(3) =  get_weak_rate_id('na24','ne24')
    ids(4) =  get_weak_rate_id('ne24','na24')
    ids(5) =  get_weak_rate_id('ne20','f20')
    ids(6) =  get_weak_rate_id('f20','ne20')
    ids(7) =  get_weak_rate_id('f20','o20')
    ids(8) =  get_weak_rate_id('o20','f20')

    write(*,'(8i12)') ids

    ! ids(1) =  get_weak_rate_id('ne20','f20')
    ! ids(2) =  get_weak_rate_id('f20','ne20')

    ! now loop over temperature and density
    !fill in arrays
    ND = 121; logDmin = 8.0; logDmax = 11.0
    NT = 121; logTmin = 7.0; logTmax = 10.0

    !         write(*,*) 'NT', NT, 'Tmin', logTmin, 'Tmax', logTmax
    !         write(*,*) 'ND', ND, 'Dmin', logDmin, 'Dmax', logDmax

    deltaT = (logTmax - logTmin) / real(NT-1,8)
    deltaD = (logDmax - logDmin) / real(ND-1,8)

    allocate(logTpts(NT))
    logTpts(1) = logTmin
    do j=2,NT
       logTpts(j) = logTpts(j-1) + deltaT
    end do

    allocate(logDpts(ND))
    logDpts(1) = logDmin
    do j=2,ND
       logDpts(j) = logDpts(j-1) + deltaD
    end do

    allocate(lambda(nr,NT,ND))
    allocate(dlambda_dlnRho(nr,NT,ND))
    allocate(dlambda_dlnT(nr,NT,ND))

    allocate(Q_out(nr,NT,ND))
    allocate(Qneu_out(nr,NT,ND))

    do j = 1, NT
       do k = 1, ND


          logT = logTpts(j)
          logRho = logDpts(k)
          Ye = 0.5d0

          T = 10**logT
          T9 = T*1d-9
          rho = 10**logRho
          YeRho = Ye*rho

          ! get a set of results for given temperature and density
          call eosDT_get( &
               handle, Z, X, abar, zbar,  &
               species, chem_id, net_iso, xa, &
               Rho, logRho, T, logT,  &
               res, d_dlnd, d_dlnT, d_dabar, d_dzbar, ierr)



          eta = res(i_eta) 
          d_eta_dlnT = d_dlnT(i_eta)
          d_eta_dlnRho = d_dlnd(i_eta)

          
          ec_lambda = 1
          if (do_ecapture) then 
             call eval_ecapture_reaction_info( &
                  ids, T9, YeRho, zbar, &
                  eta, d_eta_dlnT, d_eta_dlnRho, &
                  ldecay, d_ldecay_dT9, d_ldecay_dlYeRho, &
                  lcapture, d_lcapture_dT9, d_lcapture_dlYeRho, &
                  lneutrino, d_lneutrino_dT9, d_lneutrino_dlYeRho, &
                  ec_lambda, ec_dlambda_dlnT, ec_dlambda_dlnRho, &
                  Q, dQ_dlnT, dQ_dlnRho, &
                  Qneu, dQneu_dlnT, dQneu_dlnRho, &
                  ierr)
          else
             call eval_weak_reaction_info( &
                  ids, T9, YeRho, &
                  eta, d_eta_dlnT, d_eta_dlnRho, &
                  ldecay, d_ldecay_dT9, d_ldecay_dlYeRho, &
                  lcapture, d_lcapture_dT9, d_lcapture_dlYeRho, &
                  lneutrino, d_lneutrino_dT9, d_lneutrino_dlYeRho, &
                  ec_lambda, ec_dlambda_dlnT, ec_dlambda_dlnRho, &
                  Q, dQ_dlnT, dQ_dlnRho, &
                  Qneu, dQneu_dlnT, dQneu_dlnRho, &
                  ierr)
          end if


          if (ierr /= 0) then
             write(*,*) 'failed in eval_ecapture_reaction_info'
             stop 1
          end if


          ! pull out the data

          lambda(1:nr,j,k) = ec_lambda(1:nr)
          dlambda_dlnRho(1:nr,j,k) = ec_dlambda_dlnRho(1:nr)
          dlambda_dlnT(1:nr,j,k) = ec_dlambda_dlnT(1:nr)
          Q_out(1:nr,j,k) = Q(1:nr)
          Qneu_out(1:nr,j,k) = Qneu(1:nr)

       end do
    end do


    do i = 1, nr

       write(*,*) trim(outfile), trim(weak_lhs_nuclide_name(ids(i))), trim(weak_rhs_nuclide_name(ids(i)))
       open(unit=datafile, file=trim(outfile) // &
            trim(weak_lhs_nuclide_name(ids(i))) // trim(weak_rhs_nuclide_name(ids(i))))
       write(datafile, "(2I6)") NT,ND
       write(datafile, "(999ES12.4)") logTpts
       write(datafile, "(999ES12.4)") logDpts

       do k = 1, ND
          do j = 1, NT
             write(datafile,"(ES12.4)",advance='no') safe_log10(lambda(i,j,k))
          end do
          write(datafile,*)
       end do

       do k = 1, ND
          do j = 1, NT
             write(datafile,"(ES12.4)",advance='no') safe_log10(dlambda_dlnT(i,j,k))
          end do
          write(datafile,*)
       end do

       do k = 1, ND
          do j = 1, NT
             write(datafile,"(ES12.4)",advance='no') safe_log10(dlambda_dlnRho(i,j,k))
          end do
          write(datafile,*)
       end do
       ! end do

       do k = 1, ND
          do j = 1, NT
             write(datafile,"(ES12.4)",advance='no') Q_out(i,j,k)
          end do
          write(datafile,*)
       end do

       do k = 1, ND
          do j = 1, NT
             write(datafile,"(ES12.4)",advance='no') Qneu_out(i,j,k)
          end do
          write(datafile,*)
       end do

       close(datafile)

    end do

    ! deallocate the eos tables
    call Shutdown_eos(handle)

    deallocate(net_iso, chem_id)

    if (ierr /= 0) then
       write(*,*) 'bad result from eos_get'
       stop 1
    end if


    deallocate( &
         ids, ldecay, d_ldecay_dT9, d_ldecay_dlYeRho, &
         lcapture, d_lcapture_dT9, d_lcapture_dlYeRho, &
         lneutrino, d_lneutrino_dT9, d_lneutrino_dlYeRho, &
         lambda, dlambda_dlnT, dlambda_dlnRho, &
         Q, dQ_dlnT, dQ_dlnRho, &
         Qneu, dQneu_dlnT, dQneu_dlnRho)


    close(datafile)

  end subroutine do_test_ecapture

  subroutine Init_Composition
    use chem_def
    use chem_lib

    real(dp), parameter :: Zfrac_C = 0.173312d0
    real(dp), parameter :: Zfrac_N = 0.053177d0
    real(dp), parameter :: Zfrac_O = 0.482398d0
    real(dp), parameter :: Zfrac_Ne = 0.098675d0

    real(dp) :: frac, dabar_dx(species), dzbar_dx(species),  &
         sumx, xh, xhe, mass_correction, dmc_dx(species)

    net_iso(:) = 0

    chem_id(h1) = ih1; net_iso(ih1) = h1
    chem_id(he4) = ihe4; net_iso(ihe4) = he4
    chem_id(c12) = ic12; net_iso(ic12) = c12
    chem_id(n14) = in14; net_iso(in14) = n14
    chem_id(o16) = io16; net_iso(io16) = o16
    chem_id(ne20) = ine20; net_iso(ine20) = ne20
    chem_id(mg24) = img24; net_iso(img24) = mg24


    xa(h1) = 0
    xa(he4) = 0
    xa(c12) = 0
    xa(n14) = 0
    xa(o16) = 0.0
    xa(ne20) = 1.0
    xa(mg24) = 0.0

    call composition_info( &
         species, chem_id, xa, xh, xhe, abar, zbar, z2bar, ye,  &
         mass_correction, sumx, dabar_dx, dzbar_dx, dmc_dx)

  end subroutine Init_Composition


  subroutine do_test_startup

    use const_lib
    use utils_lib
    use eos_def
    use eos_lib
    use chem_def
    use chem_lib
    use weak_def
    use weak_lib

    integer :: ierr
    character (len=32) :: my_mesa_dir

    character (len=256) :: eos_file_prefix
    logical, parameter :: use_cache = .true.

    eos_file_prefix = 'mesa'

    include 'formats.dek'

    ierr = 0

    my_mesa_dir = '../..'         
    call const_init(my_mesa_dir,ierr)     
    if (ierr /= 0) then
       write(*,*) 'const_init failed'
       stop 1
    end if

    call chem_init('isotopes.data', ierr)
    if (ierr /= 0) then
       write(*,*) 'chem_init failed'
       stop 1
    end if

    ! call eos_init(eos_file_prefix, '', '', use_cache, ierr)
    ! if (ierr /= 0) then
    !    write(*,*) 'eos_init failed'
    !    stop 1
    ! end if

    ! handle = alloc_eos_handle(ierr)
    ! if (ierr /= 0) then
    !    write(*,*) 'failed trying to allocate eos handle'
    !    stop 1
    ! end if

    ! allocate and initialize the eos tables
    call Setup_eos(handle)

    call weak_init('',ierr)
    if (ierr /= 0) then
       write(*,*) 'weak_init failed'
       stop 1
    end if

    call ecapture_init(ierr)
    if (ierr /= 0) then
       write(*,*) 'ecapture_init failed'
       stop 1
    end if

  end subroutine do_test_startup


  subroutine Setup_eos(handle)
    ! allocate and load the eos tables
    use eos_def
    use eos_lib
    integer, intent(out) :: handle

    character (len=256) :: eos_file_prefix
    integer :: ierr
    logical, parameter :: use_cache = .true.

    eos_file_prefix = 'mesa'

    call eos_init(eos_file_prefix, '', '', use_cache, ierr)
    if (ierr /= 0) then
       write(*,*) 'eos_init failed in Setup_eos'
       stop 1
    end if

    handle = alloc_eos_handle(ierr)
    if (ierr /= 0) then
       write(*,*) 'failed trying to allocate eos handle'
       stop 1
    end if

  end subroutine Setup_eos


  subroutine Shutdown_eos(handle)
    use eos_def
    use eos_lib
    integer, intent(in) :: handle
    call free_eos_handle(handle)
    call eos_shutdown
  end subroutine Shutdown_eos


end program test_ecapture




