! ***********************************************************************
!
!   Copyright (C) 2013  Josiah Schwab, Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

module ecapture_lib

  use ecapture_def
  use const_def, only: dp

  implicit none

contains

  ! call this routine to initialize the weak reactions module. 
  ! only needs to be done once at start of run.

  subroutine ecapture_init(ierr)      
    use ecapture_def, only : ecapture_def_init
    use mod_load_ecapture, only: load_ecapture_data_if_necessary
    integer, intent(out) :: ierr ! 0 means AOK.         
    call ecapture_def_init(ierr)
    if (ierr /= 0) return
    call load_ecapture_data_if_necessary(ierr)
  end subroutine ecapture_init


  subroutine ecapture_shutdown
    use ecapture_def, only : free_ecapture_info
    call free_ecapture_info
  end subroutine ecapture_shutdown


  integer function get_ecapture_rate_id(lhs, rhs) ! returns 0 if reaction not found
    use ecapture_def
    use mod_load_ecapture, only: load_ecapture_data_if_necessary
    use utils_lib
    character (len=*), intent(in)  :: lhs, rhs 
    ! names of the nuclides as given in ecapturereactions.tables (e.g. 'p', 'n', 'ca42', etc.)
    integer :: ierr, i
    character(len=2*iso_name_length+1) :: key
    character (len=iso_name_length) :: lhs_name, rhs_name
    ierr = 0
    get_ecapture_rate_id = 0
    if (.not. ecapture_data_loaded) then
       call load_ecapture_data_if_necessary(ierr)
       if (ierr /= 0) then
          write(*,*) 'failed in loading ecapture reactions data'
          return
       end if
    end if
    lhs_name = adjustl(lhs)
    rhs_name = adjustl(rhs)
    call create_ecapture_dict_key(lhs_name, rhs_name, key)
    call integer_dict_lookup(ecapture_reactions_dict, key, i, ierr)
    if (ierr /= 0) then
       !write(*,*) 'failed in integer_dict_lookup for key ' // trim(key)
       return
    end if
    get_ecapture_rate_id = i
  end function get_ecapture_rate_id


  integer function get_ecapture_info_list_id(lhs, rhs) ! returns 0 if reaction not found
    ! value can be used to index ecapture_info_life_halflife and ecapture_info_list_Qneu
    use ecapture_def
    use utils_lib
    character (len=*), intent(in)  :: lhs, rhs ! names as in ecapture_info.list file
    integer :: ierr, i
    character(len=2*iso_name_length+1) :: key
    character (len=iso_name_length) :: lhs_name, rhs_name
    ierr = 0
    get_ecapture_info_list_id = 0
    lhs_name = adjustl(lhs)
    rhs_name = adjustl(rhs)
    call create_ecapture_dict_key(lhs_name, rhs_name, key)
    call integer_dict_lookup(ecapture_reactions_dict, key, i, ierr)
    if (ierr /= 0) then
       !write(*,'(a)') 'get_ecapture_info_list_id failed for ' // trim(key)
       return
    end if
    get_ecapture_info_list_id = i
  end function get_ecapture_info_list_id


  subroutine eval_ecapture_reaction_info( &
       ids, T9, YeRho, zbar, &
       eta, d_eta_dlnT, d_eta_dlnRho, &
       ldecay, d_ldecay_dT9, d_ldecay_dlYeRho, &
       lcapture, d_lcapture_dT9, d_lcapture_dlYeRho, &
       lneutrino, d_lneutrino_dT9, d_lneutrino_dlYeRho, &
       lambda, dlambda_dlnT, dlambda_dlnRho, &
       Q, dQ_dlnT, dQ_dlnRho, &
       Qneu, dQneu_dlnT, dQneu_dlnRho, &
       ierr)
    use mod_eval_ecapture, only: do_eval_ecapture_reaction_info
    integer, intent(in) :: ids(:)
    real(dp), intent(in) :: T9, YeRho, zbar, eta, d_eta_dlnT, d_eta_dlnRho
    ! lambda = combined rate = 10**ldecay + 10**lcapture
    ! Q and Qneu are for combined rate of beta decay and electron capture.
    ! Q is total, so Q-Qneu is the actual thermal energy.
    ! note: lambdas include Ye Rho factors for electron captures.
    ! so treat the rates as if just beta decays
    real(dp), dimension(:), pointer :: &
         ldecay, d_ldecay_dT9, d_ldecay_dlYeRho, &
         lcapture, d_lcapture_dT9, d_lcapture_dlYeRho, &
         lneutrino, d_lneutrino_dT9, d_lneutrino_dlYeRho, &
         lambda, dlambda_dlnT, dlambda_dlnRho, &
         Q, dQ_dlnT, dQ_dlnRho, &
         Qneu, dQneu_dlnT, dQneu_dlnRho
    integer, intent(out) :: ierr
    call do_eval_ecapture_reaction_info( &
         ids, T9, YeRho, zbar, &
         eta, d_eta_dlnT, d_eta_dlnRho, &
         ldecay, d_ldecay_dT9, d_ldecay_dlYeRho, &
         lcapture, d_lcapture_dT9, d_lcapture_dlYeRho, &
         lneutrino, d_lneutrino_dT9, d_lneutrino_dlYeRho, &
         lambda, dlambda_dlnT, dlambda_dlnRho, &
         Q, dQ_dlnT, dQ_dlnRho, &
         Qneu, dQneu_dlnT, dQneu_dlnRho, &
         ierr)
  end subroutine eval_ecapture_reaction_info


  subroutine psi_Iec_and_Jec(beta, zeta, eta, deta_dlnT, deta_dlnRho, &
       I, dI_dlnT, dI_dlnRho, J, dJ_dlnT, dJ_dlnRho)

    use mod_eval_psi, only : do_psi_Iec_and_Jec

    ! calulate the phase space integral for electron emission (beta-decay)

    implicit none

    real(dp), intent(in) :: beta  ! mec2 / kT
    real(dp), intent(in) :: zeta  ! Q_n / kT
    real(dp), intent(in) :: eta   ! chemical potential / kT
    real(dp), intent(in) :: deta_dlnT, deta_dlnRho ! and derivs

    real(dp), intent(out) :: I, J   ! phase space integral
    real(dp), intent(out) :: dI_dlnT, dI_dlnRho ! and derivatives
    real(dp), intent(out) :: dJ_dlnT, dJ_dlnRho ! and derivatives

    call do_psi_Iec_and_Jec(beta, zeta, eta, deta_dlnT, deta_dlnRho, &
         I, dI_dlnT, dI_dlnRho, J, dJ_dlnT, dJ_dlnRho)

  end subroutine psi_Iec_and_Jec


  subroutine psi_Iee_and_Jee(beta, zeta, eta, deta_dlnT, deta_dlnRho, &
       I, dI_dlnT, dI_dlnRho, J, dJ_dlnT, dJ_dlnRho)

    use mod_eval_psi, only : do_psi_Iee_and_Jee

    ! calulate the phase space integral for electron emission (beta-decay)

    implicit none

    real(dp), intent(in) :: beta  ! mec2 / kT
    real(dp), intent(in) :: zeta  ! Q_n / kT
    real(dp), intent(in) :: eta   ! chemical potential / kT
    real(dp), intent(in) :: deta_dlnT, deta_dlnRho ! and derivs

    real(dp), intent(out) :: I, J   ! phase space integral
    real(dp), intent(out) :: dI_dlnT, dI_dlnRho ! and derivatives
    real(dp), intent(out) :: dJ_dlnT, dJ_dlnRho ! and derivatives

    call do_psi_Iee_and_Jee(beta, zeta, eta, deta_dlnT, deta_dlnRho, &
         I, dI_dlnT, dI_dlnRho, J, dJ_dlnT, dJ_dlnRho)

  end subroutine psi_Iee_and_Jee

end module ecapture_lib

