! ***********************************************************************
!
!   Copyright (C) 2013  Josiah Schwab, Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

module mod_eval_ecapture

  use ecapture_def
  use const_def, only: dp

  implicit none


contains

  subroutine do_eval_ecapture_reaction_info( &
       ids, T9, YeRho, zbar, &
       etak, d_etak_dlnT, d_etak_dlnRho, &  ! these are kinetic chemical potentials
       ldecay, d_ldecay_dT9, d_ldecay_dlYeRho, &
       lcapture, d_lcapture_dT9, d_lcapture_dlYeRho, &
       lneutrino, d_lneutrino_dT9, d_lneutrino_dlYeRho, &
       lambda, dlambda_dlnT, dlambda_dlnRho, &
       Q, dQ_dlnT, dQ_dlnRho, &
       Qneu, dQneu_dlnT, dQneu_dlnRho, &
       ierr)
    use mod_load_ecapture, only: load_ecapture_data_if_necessary
    use mod_eval_psi
    use mod_eval_coulomb
    use const_def, only: ln10, kerg, mev_to_ergs, keV, me, clight, ev2erg, fine
    use const_def, only: const_pi => pi
    use chem_def
    use utils_lib, only: is_bad_real, is_bad_num, integer_dict_lookup
    use weak_def, only: weak_lhs_nuclide_name, weak_rhs_nuclide_name, &
         weak_lhs_nuclide_id, weak_rhs_nuclide_id

    integer, intent(in) :: ids(:)
    real(dp), intent(in) :: T9, YeRho, zbar, etak, d_etak_dlnT, d_etak_dlnRho
    real(dp), dimension(:), intent(out), pointer :: &
         ldecay, d_ldecay_dT9, d_ldecay_dlYeRho, &
         lcapture, d_lcapture_dT9, d_lcapture_dlYeRho, &
         lneutrino, d_lneutrino_dT9, d_lneutrino_dlYeRho, &
         lambda, dlambda_dlnT, dlambda_dlnRho, &
         Q, dQ_dlnT, dQ_dlnRho, &
         Qneu, dQneu_dlnT, dQneu_dlnRho
    integer, intent(out) :: ierr

    logical, parameter :: dbg = .false.

    integer :: n, i, ir, in, out, j, lhs, rhs
    integer :: offset, ntrans, lo, hi
    integer :: offset_lhs, nstates_lhs, lo_lhs, hi_lhs
    integer :: offset_rhs, nstates_rhs, lo_rhs, hi_rhs

    real(dp) :: mue, dmue_dlnRho, dmue_dlnT ! chem pot and derivatives

    character(len=iso_name_length) :: ecapture_lhs, ecapture_rhs

    real(dp) :: Qx, T, beta, kT, mec2, Z, Eavg, Ei, Ef, G_GM
    real(dp) :: eta, deta_dlnT, deta_dlnRho

    real(dp), dimension(max_num_ecapture_states) :: E_lhs, E_rhs, J_lhs, J_rhs
    real(dp), dimension(max_num_ecapture_states) :: bf, Pi, dPi_dlnT

    integer, dimension(max_num_ecapture_transitions) :: Si, Sf
    real(dp), dimension(max_num_ecapture_transitions) :: logft

    real(dp), dimension(max_num_ecapture_transitions) :: zeta
    real(dp), dimension(max_num_ecapture_transitions) :: Ie, dIe_dlnT, dIe_dlnRho
    real(dp), dimension(max_num_ecapture_transitions) :: Je, dJe_dlnT, dJe_dlnRho
    real(dp), dimension(max_num_ecapture_transitions) :: Pj, dPj_dlnT

    real(dp), dimension(max_num_ecapture_transitions) :: lambdaj, lambdaj_neu

    real(dp) :: Z_lhs, Z_rhs, muC_lhs, muC_rhs, delta_muC ! for coulomb corrections

    real(dp) :: neutrino, dneutrino_dlnT, dneutrino_dlnRho, &
         gammaray, dgammaray_dlnT, dgammaray_dlnRho, lambdam1
    character(len=2*iso_name_length+1) :: key

    integer :: rxn_type
    integer, parameter :: rxn_ecapture = 1, rxn_betadecay = -1

    include 'formats.dek'

    ierr = 0
    if (.not. ecapture_data_loaded) then
       call load_ecapture_data_if_necessary(ierr)
       if (ierr /= 0) then
          write(*,*) 'failed in loading ecapture reactions data'
          return
       end if
    end if

    n = size(ids)

    ! first, translate the density, temperature, etc into appropriate units

    kT = 1e3_dp * keV * T9 ! in MeV
    mec2 = me * clight**2 / mev_to_ergs ! in MeV
    beta = mec2/kT ! dimesionless

    ! the chemical potentials from the equation of state are kinetic
    ! so add in the rest mass terms

    eta = etak + beta
    deta_dlnT = d_etak_dlnT - beta
    deta_dlnRho = d_etak_dlnRho

    ! only evaluate this for really degenerate stuff
    if (eta .lt. 2*beta) return

    ! also need chemical potential in MeV
    mue = eta * kT
    dmue_dlnRho = deta_dlnRho * kT
    dmue_dlnT = deta_dlnT * kT + mue

    do i = 1, n ! loop over reactions

       ! if there's not a weak reaction index, don't bother
       ir = ids(i)
       if (dbg .and. (ir.le.0)) write(*,'(a,i3)') "No weak reaction for ", ir
       if (ir <= 0) cycle

       ! get reactant names & ids
       ecapture_lhs = weak_lhs_nuclide_name(ir)
       ecapture_rhs = weak_rhs_nuclide_name(ir)

       ! generate reaction dictionary key (e.g. "mg24 na24")
       call create_ecapture_dict_key(ecapture_lhs, ecapture_rhs, key)

       lhs = weak_lhs_nuclide_id(ir)
       rhs = weak_rhs_nuclide_id(ir)

       ! get species charges (which will be one different)
       Z_lhs = chem_isos% Z(lhs)
       Z_rhs = chem_isos% Z(rhs)

       ! determine whether this is a beta-decay or an electron capture
       rxn_type = int(Z_lhs - Z_rhs)

       Qx = 0
       G_GM = 1
       select case (rxn_type)
       case(rxn_ecapture)
          ! use the *atomic* isotope data to get the *nuclear* mass excess
          Qx = chem_isos% mass_excess(lhs) - chem_isos% mass_excess(rhs) - mec2
          G_GM = exp(-const_pi* fine * Z_lhs)
          if (dbg) write(*,*) "ecapture ", key
       case(rxn_betadecay)
          ! use the *atomic* isotope data to get the *nuclear* mass excess
          Qx = chem_isos% mass_excess(lhs) - chem_isos% mass_excess(rhs) + mec2
          G_GM = exp(-const_pi * fine * Z_rhs)
          if (dbg) write(*,*) "betadecay ", key
       end select


       ! in either case, get the transition data

       call integer_dict_lookup(ecapture_transitions_number_dict, key, ntrans, ierr)
       if (ierr /=0) stop "ERROR: ecapture (transitions)"
       call integer_dict_lookup(ecapture_transitions_offset_dict, key, offset, ierr)
       if (ierr /=0) stop "ERROR: ecapture (transitions)"

       lo = offset + 1
       hi = offset + ntrans

       Si(1:ntrans) = ecapture_transitions_data(lo:hi, i_Si)
       Sf(1:ntrans) = ecapture_transitions_data(lo:hi, i_Sf)
       logft(1:ntrans) = ecapture_logft_data(lo:hi)

       ! get the left state info (energies and spins)

       call integer_dict_lookup(ecapture_states_number_dict, ecapture_lhs, nstates_lhs, ierr)
       if (ierr /=0) stop "ERROR: ecapture (lhs states)"
       call integer_dict_lookup(ecapture_states_offset_dict, ecapture_lhs, offset_lhs, ierr)
       if (ierr /=0) stop "ERROR: ecapture (lhs states)"

       lo_lhs = offset_lhs + 1
       hi_lhs = offset_lhs + nstates_lhs

       E_lhs(1:nstates_lhs) = ecapture_states_data(lo_lhs:hi_lhs, i_E)
       J_lhs(1:nstates_lhs) = ecapture_states_data(lo_lhs:hi_lhs, i_J)

       ! get the right state info (energies and spins)

       call integer_dict_lookup(ecapture_states_number_dict, ecapture_rhs, nstates_rhs, ierr)
       if (ierr /=0) stop "ERROR: ecapture (rhs states)"
       call integer_dict_lookup(ecapture_states_offset_dict, ecapture_rhs, offset_rhs, ierr)
       if (ierr /=0) stop "ERROR: ecapture (rhs states)"

       lo_rhs = offset_rhs + 1
       hi_rhs = offset_rhs + nstates_rhs

       E_rhs(1:nstates_rhs) = ecapture_states_data(lo_rhs:hi_rhs, i_E)
       J_rhs(1:nstates_rhs) = ecapture_states_data(lo_rhs:hi_rhs, i_J)

       ! we assume the left hand side states have thermal occupation fractions

       ! calculate boltzmann factor (bf), partition function (Z), and <E>
       bf(1:nstates_lhs) = (2 * J_lhs(1:nstates_lhs) + 1) * exp(-E_lhs(1:nstates_lhs)/ kT)
       Z = sum(bf(1:nstates_lhs))
       Eavg = sum(bf(1:nstates_lhs)* E_lhs(1:nstates_lhs))/ Z

       ! occupation probability (& derivative)
       Pi(1:nstates_lhs) = bf(1:nstates_lhs)/Z
       dPi_dlnT(1:nstates_lhs) = Pi(1:nstates_lhs) * (E_lhs(1:nstates_lhs) - Eavg) / kT

       ! now rearrange these rates to they apply to the transitions
       do j = 1, ntrans
          Pj(j) = Pi(Si(j))
          dPj_dlnT(j) = dPi_dlnT(Si(j))
       end do

       if (dbg) then
          do j = 1, ntrans
             write(*,"(2(F5.2, I2, F5.2))")  E_lhs(Si(j)), int(J_lhs(Si(j))), Pj(j), &
             E_rhs(Sf(j)), int(J_rhs(Sf(j))), logft(j)
          end do
       end if

       ! get chemical potential difference, related to coulomb correction
       if (do_ecapture_coulomb) then

          ! get chemical potentials (already in units of kT)
          muC_lhs = do_mu_coulomb(Z_lhs, YeRho, 1e9_dp * T9, zbar)
          muC_rhs = do_mu_coulomb(Z_rhs, YeRho, 1e9_dp * T9, zbar)

          ! shift is negative (for e-capture) given our definitions
          delta_muC = muC_lhs - muC_rhs

          if (dbg) write(*,*) eta, muC_lhs, muC_rhs, delta_muC

       else
          ! no corrections
          delta_muC = 0
       end if

       ! do the phase space integrals
       do j = 1, ntrans

          Ei = E_lhs(Si(j))
          Ef = E_rhs(Sf(j))

          ! calculate energy difference, including electron rest mass
          zeta(j) = (Qx - Ef + Ei)/ kT + delta_muC

          select case(rxn_type)
          case(rxn_ecapture)
             call do_psi_Iec_and_Jec(beta, zeta(j), eta, deta_dlnT, deta_dlnRho, &
                  Ie(j), dIe_dlnT(j), dIe_dlnRho(j), Je(j), dJe_dlnT(j), dJe_dlnRho(j))
          case(rxn_betadecay)
             call do_psi_Iee_and_Jee(beta, zeta(j), eta, deta_dlnT, deta_dlnRho, &
                  Ie(j), dIe_dlnT(j), dIe_dlnRho(j), Je(j), dJe_dlnT(j), dJe_dlnRho(j))
          end select

       end do

       ! apply fermi-function correction factor
       Ie(1:ntrans) = G_GM * Ie(1:ntrans)
       dIe_dlnT(1:ntrans) = G_GM * dIe_dlnT(1:ntrans)
       dIe_dlnRho(1:ntrans) = G_GM * dIe_dlnRho(1:ntrans)

       Je(1:ntrans) = G_GM * Je(1:ntrans)
       dJe_dlnT(1:ntrans) = G_GM * dJe_dlnT(1:ntrans)
       dJe_dlnRho(1:ntrans) = G_GM * dJe_dlnRho(1:ntrans)

       ! add approximate fermi-function corrections

       lambdaj(1:ntrans) =  Ie(1:ntrans) * 10.0_dp**(-logft(1:ntrans)) * log(2.0_dp)
       lambdaj_neu(1:ntrans) = Je(1:ntrans) * 10.0_dp**(-logft(1:ntrans)) * log(2.0_dp)

       lambda(i) = sum(lambdaj(1:ntrans) * Pj(1:ntrans))
       dlambda_dlnT(i) = sum(lambdaj(1:ntrans) *  &
            ( dPj_dlnT(1:ntrans) + dIe_dlnT(1:ntrans)/Ie(1:ntrans)))
       dlambda_dlnRho(i) = sum(lambdaj(1:ntrans) * &
            dIe_dlnRho(1:ntrans)/Ie(1:ntrans))

       neutrino = mec2 * sum(lambdaj_neu(1:ntrans) * Pj(1:ntrans))
       dneutrino_dlnT = mec2 * sum(lambdaj_neu(1:ntrans) * &
            ( dPj_dlnT(1:ntrans) + dJe_dlnT(1:ntrans)/Je(1:ntrans)))
       dneutrino_dlnRho = mec2 * sum(lambdaj_neu(1:ntrans) * &
            dJe_dlnRho(1:ntrans)/Je(1:ntrans))

       if (lambda(i) .gt. 1d-30) then
          lambdam1 = 1.0_dp / lambda(i)
       else
          lambdam1 = 0
       end if

       Qneu(i) = neutrino * lambdam1
       dQneu_dlnT(i) = (dneutrino_dlnT - dlambda_dlnT(i) * neutrino) * lambdam1
       dQneu_dlnRho(i) = (dneutrino_dlnRho - dlambda_dlnRho(i) * neutrino) * lambdam1

       ! this is the *total* energy per decay (neu losses are subtracted later)
       select case(rxn_type)
       case(rxn_ecapture)

          Q(i) = Qx + mue
          dQ_dlnT(i) = dmue_dlnT
          dQ_dlnRho(i) = dmue_dlnRho

       case(rxn_betadecay)

          Q(i) = Qx - mue
          dQ_dlnT(i) = -dmue_dlnT
          dQ_dlnRho(i) = -dmue_dlnRho

       end select

       ! this routine does not evaluate the following quantities
       ! because they don't actually ever get used anywhere else

       ldecay(i) = 0.0
       d_ldecay_dT9(i) = 0.0
       d_ldecay_dlYeRho(i) = 0.0

       lcapture(i) = 0.0
       d_lcapture_dT9(i) = 0.0
       d_lcapture_dlYeRho(i) = 0.0

       lneutrino(i) = 0.0
       d_lneutrino_dT9(i) = 0.0
       d_lneutrino_dlYeRho(i) = 0.0

    end do

    ierr = 0

  end subroutine do_eval_ecapture_reaction_info

end module mod_eval_ecapture
