! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
!
! ***********************************************************************

      module mod_colors
		use colors_def
      implicit none
		private
		public :: init_colors, free_colors, Eval_Colors

      contains


      subroutine init_colors(ierr)
			integer, intent(out) :: ierr
         call init_colors_names
			call Read_Colors_Data(ierr)		
		end subroutine init_colors


      subroutine free_colors
			type (lgg_list), pointer :: glist, gnxt
			
			integer :: num_gs, num_ts, num_zs
			num_gs=0; num_ts=0; num_zs=0

			glist => ghead
			do while (associated(glist))
				gnxt => glist% nxt
				call free_tlist(glist% tlist)
				deallocate(glist); num_gs = num_gs+1
				glist => gnxt
			end do
			nullify(ghead)
			
			contains
			
			subroutine free_tlist(tptr)
				type (lgt_list), pointer :: tptr				
				type (lgt_list), pointer :: tlist				
				type (lgt_list), pointer :: tnxt		
				tlist => tptr		
				do while (associated(tlist))
					tnxt => tlist% nxt
					call free_zlist(tlist% zlist)
					deallocate(tlist); num_ts = num_ts+1
					tlist => tnxt
				end do			
			end subroutine free_tlist
			
			subroutine free_zlist(zptr)
				type (lgz_list), pointer :: zptr				
				type (lgz_list), pointer :: zlist				
				type (lgz_list), pointer :: znxt	
				zlist => zptr			
				do while (associated(zlist))
					znxt => zlist% nxt
					deallocate(zlist); num_zs = num_zs+1
					zlist => znxt
				end do			
			end subroutine free_zlist
			
		end subroutine free_colors


      subroutine show_tree(ghead)
			type (lgg_list), pointer :: ghead
			type (lgg_list), pointer :: glist, gnxt
			
			glist => ghead
			do while (associated(glist))
				write(*,*)
				write(*,*) 'glist lgg', glist% lgg
				gnxt => glist% nxt
				call show_tlist(glist% tlist)
				glist => gnxt
			end do
			
			contains
			
			subroutine show_tlist(tptr)
				type (lgt_list), pointer :: tptr
				type (lgt_list), pointer :: tlist, tnxt
				
				tlist => tptr
				do while (associated(tlist))
					write(*,*) 'tlist% lgt', 10**(tlist% lgt)
					tnxt => tlist% nxt
					call show_zlist(tlist% zlist)
					tlist => tnxt
				end do			
			end subroutine show_tlist
			
			subroutine show_zlist(zptr)
				type (lgz_list), pointer :: zptr				
				type (lgz_list), pointer :: zlist				
				type (lgz_list), pointer :: znxt		
				zlist => zptr		
				do while (associated(zlist))
					write(*,*) 'zlist% lgz', zlist% lgz
					znxt => zlist% nxt
					zlist => znxt
				end do			
			end subroutine show_zlist
			
		end subroutine show_tree


		subroutine Eval_Colors(log_Teff, log_L, mass, Fe_H_in, results, log_g, ierr)
			double precision, intent(in)  :: log_Teff ! log10 of surface temp
			double precision, intent(in)  :: log_L ! log10 of luminosity in solar units
			double precision, intent(in)  :: mass ! mass in solar units
			double precision, intent(in)  :: Fe_H_in ! [Fe/H]
			double precision, intent(out) :: results(n_colors)
			double precision, intent(out) :: log_g
			integer, intent(out) :: ierr
			
			double precision, parameter :: Zsol = 0.02d0, colors_bol_sun = 4.746d0
			double precision :: colors_bol
			
			double precision :: lgg, lgz, lgt, results1(n_colors), results2(n_colors), alfa, beta
			type (lgg_list), pointer :: glist, gnxt
			
	      lgg = log10(mass) + 4.0D0*log_Teff - log_L - 10.6071D0
			lgt = log_Teff
			lgz = Fe_H_in
			
			log_g = lgg
			
			colors_bol = colors_bol_sun - 2.5d0 * log_L
			
			if (.not. associated(ghead)) then
				ierr = -1; return
			end if
			
			ierr = 0
			
			if (lgg >= ghead% lgg) then ! use the largest lgg
				call get_tlist_results(ghead% tlist, lgt, lgz, results1)
				results = results1
			else
				
				glist => ghead
				do while (associated(glist% nxt))
					gnxt => glist% nxt
					if (lgg == gnxt% lgg) then ! use gnxt
						call get_tlist_results(gnxt% tlist, lgt, lgz, results1)
						results = results1
						exit
					end if
					if (lgg >= gnxt% lgg) then ! interpolate between glist and gnxt
						call get_tlist_results(glist% tlist, lgt, lgz, results1)
						call get_tlist_results(gnxt% tlist, lgt, lgz, results2)
						alfa = (lgg - gnxt% lgg) / (glist% lgg - gnxt% lgg)
						beta = 1 - alfa
						results = alfa * results1 + beta * results2
						exit
					end if
					glist => gnxt
				end do
		
				if (.not. (associated(glist% nxt))) then
					! use the smallest lgg
					call get_tlist_results(glist% tlist, lgt, lgz, results1)
					results = results1
				end if
			
			end if
			
			results(bol) = colors_bol
			
		end subroutine Eval_Colors
		
		
		subroutine get_tlist_results(tptr, lgt, lgz, results)
			type (lgt_list), pointer :: tptr
			double precision, intent(in) :: lgt, lgz
			double precision, intent(out) :: results(n_colors)
			
			type (lgt_list), pointer :: tlist, tnxt
			double precision :: results1(n_colors), results2(n_colors), alfa, beta
			
			tlist => tptr
			
			if (.not. associated(tlist)) stop 'bad tlist for get_tlist_results'
			
			if (lgt >= tlist% lgt) then ! use the largest lgt
				call get_zlist_results(tlist% zlist, lgz, results1)
				results = results1
				return
			end if

			do while (associated(tlist% nxt))
				tnxt => tlist% nxt
				if (lgt == tnxt% lgt) then ! use tnxt
					call get_zlist_results(tnxt% zlist, lgz, results1)
					results = results1
					return
				end if
				if (lgt >= tnxt% lgt) then ! interpolate between tlist and tnxt
					call get_zlist_results(tlist% zlist, lgz, results1)
					call get_zlist_results(tnxt% zlist, lgz, results2)					
					alfa = (lgt - tnxt% lgt) / (tlist% lgt - tnxt% lgt)
					beta = 1 - alfa
					results = alfa * results1 + beta * results2
					return
				end if
				tlist => tnxt
			end do
		
			! use the smallest lgt
			call get_zlist_results(tlist% zlist, lgz, results1)
			results = results1
		
		end subroutine get_tlist_results
		
		
		subroutine get_zlist_results(zptr, lgz, results)
			type (lgz_list), pointer :: zptr
			double precision, intent(in) :: lgz
			double precision, intent(out) :: results(n_colors)
			
			type (lgz_list), pointer :: zlist, znxt
			double precision :: alfa, beta
			
			zlist => zptr
			
			if (.not. associated(zlist)) stop 'bad zlist for get_zlist_results'
			
			if (lgz >= zlist% lgz) then ! use the largest lgz
				results = zlist% colors
				return
			end if

			do while (associated(zlist% nxt))
				znxt => zlist% nxt
				if (lgz == znxt% lgz) then ! use znxt
					results = znxt% colors
					return
				end if
				if (lgz >= znxt% lgz) then ! interpolate between zlist and znxt
					alfa = (lgz - znxt% lgz) / (zlist% lgz - znxt% lgz)
					beta = 1 - alfa
					results = alfa * zlist% colors + beta * znxt% colors
					return
				end if
				zlist => znxt
			end do
		
			! use the smallest lgz
			results = zlist% colors
		
		end subroutine get_zlist_results
		
		
		subroutine Read_Colors_Data(ierr)
		   use utils_lib
		   use const_def, only: mesa_data_dir
			integer, intent(out) :: ierr ! 0 means ok

			! read file and build lists
			integer :: ios, cnt
			character (len=256) :: fname
			double precision :: lgt, lgg, colors(n_colors), lgz
			type (lgg_list), pointer :: glist
			type (lgt_list), pointer :: tlist
			type (lgz_list), pointer :: zlist
			integer :: num_entries, num_made, IO_UBV
			
			include 'formats.dek'
			
			IO_UBV = alloc_iounit(ierr)

	      fname = trim(mesa_data_dir) // '/colors_data/lcb98cor.dat'
         open(UNIT=IO_UBV, FILE=trim(fname), ACTION='READ', STATUS='OLD', IOSTAT=ios)
			if (ios /= 0) then ! failed to open lcb98cor.dat
			   write(*,*) 'colors_init: failed to open ' // trim(fname)
			   write(*,*) 'please check the path to the mesa data directory'
				ierr = 1; return
			end if
			
			ierr = 0
			num_entries = 0
			cnt = 0
			do while (.true.)
				
				read(IO_UBV,fmt=*,iostat=ios) lgt, lgg, lgz, colors
				if (ios /= 0) exit
				cnt = cnt + 1

				lgt = log10(lgt)				
				
				call get_glist(ghead, lgg, glist, ierr)
				if (ierr /= 0) exit
				
				call get_tlist(glist% tlist, lgt, tlist, ierr)
				if (ierr /= 0) exit
				
				call get_zlist(tlist% zlist, lgz, zlist, num_entries, ierr)
				if (ierr /= 0) exit
				
				zlist% colors = colors

			end do
			
			close(IO_UBV)
			call free_iounit(IO_UBV)
			if (ierr /= 0) return
			
			num_made = 0
			
			glist => ghead
			lgg = 1d99
			do while (associated(glist))
				
				if (glist% lgg >= lgg) then ! bad glist order
					ierr = -1; return
				end if
				lgg = glist% lgg
				tlist => glist% tlist
				lgt = 1d99
				do while (associated(tlist))
					if (tlist% lgt >= lgt) then ! bad tlist order
						ierr = -2; return
					end if
					lgt = tlist% lgt
					zlist => tlist% zlist
					lgz = 1d99
					do while (associated(zlist))
						if (zlist% lgz >= lgz) then ! bad zlist order
							ierr = -3; return
						end if
						lgz = zlist% lgz
						num_made = num_made + 1
						zlist => zlist% nxt
					end do
					tlist => tlist% nxt
				end do
				glist => glist% nxt
			end do
			
		end subroutine Read_Colors_Data


		subroutine get_glist(head, lgg, glist, ierr)
			type (lgg_list), pointer :: head
			double precision, intent(in) :: lgg
			type (lgg_list), pointer :: glist
			integer, intent(out) :: ierr
			
			type (lgg_list), pointer :: g1, g2
			
			ierr = 0
			
			if (.not. associated(head)) then ! first time
				if (.not. alloc_glist()) return
				head => glist
				return
			end if
			
			if (head% lgg == lgg) then ! matches head of list
				glist => head
				return
			end if
			
			if (head% lgg < lgg) then ! becomes new head of list
				if (.not. alloc_glist()) return
				glist% nxt => head
				head => glist
				return
			end if
			
			! check list
			g1 => head
			do while (associated(g1% nxt))
				g2 => g1% nxt
				if (g2% lgg == lgg) then
					glist => g2; return
				end if
				if (g2% lgg < lgg) then ! insert new one before g2 
					if (.not. alloc_glist()) return
					glist% nxt => g2
					g1% nxt => glist
					return
				end if
				g1 => g2
			end do
			! add to end of list after g1
			if (.not. alloc_glist()) return
			g1% nxt => glist
			
			contains
			
			logical function alloc_glist()
				integer :: istat
				allocate(glist,stat=istat)
				if (istat /= 0) then ! allocate failed in alloc_glist
					alloc_glist = .false.; ierr = -1; return
				end if
				nullify(glist% tlist)
				nullify(glist% nxt)
				glist% lgg = lgg	
				alloc_glist = .true.		
			end function alloc_glist
			
		end subroutine get_glist
							

		subroutine get_tlist(head, lgt, tlist, ierr)
			type (lgt_list), pointer :: head
			double precision, intent(in) :: lgt
			type (lgt_list), pointer :: tlist
			integer, intent(out) :: ierr ! 0 means ok
			
			type (lgt_list), pointer :: t1, t2
			
			ierr = 0
			
			if (.not. associated(head)) then ! first time
				if (.not. alloc_tlist()) return
				head => tlist
				return
			end if
			
			if (head% lgt == lgt) then ! matches head of list
				tlist => head
				return
			end if
			
			if (head% lgt < lgt) then ! becomes new head of list
				if (.not. alloc_tlist()) return
				tlist% nxt => head
				head => tlist
				return
			end if
			
			! check list
			t1 => head
			do while (associated(t1% nxt))
				t2 => t1% nxt
				if (t2% lgt == lgt) then
					tlist => t2; return
				end if
				if (t2% lgt < lgt) then ! insert new one before t2 
					if (.not. alloc_tlist()) return
					tlist% nxt => t2
					t1% nxt => tlist
					return
				end if
				t1 => t2
			end do
			! add to end of list after t1
			if (.not. alloc_tlist()) return
			t1% nxt => tlist
			
			contains
			
			logical function alloc_tlist()
				integer :: istat
				allocate(tlist,stat=istat)
				if (istat /= 0) then
					alloc_tlist = .false.; ierr = -1; return
				end if
				nullify(tlist% zlist)
				nullify(tlist% nxt)
				tlist% lgt = lgt			
				alloc_tlist = .true.
			end function alloc_tlist

		end subroutine get_tlist
							

		subroutine get_zlist(head, lgz, zlist, num_created, ierr)
			type (lgz_list), pointer :: head
			double precision, intent(in) :: lgz
			type (lgz_list), pointer :: zlist
			integer, intent(inout) :: num_created
			integer, intent(out) :: ierr ! 0 means ok
			
			type (lgz_list), pointer :: z1, z2
			
			ierr = 0
			
			if (.not. associated(head)) then ! first time
				if (.not. alloc_zlist()) return
				head => zlist
				return
			end if
			
			if (head% lgz == lgz) then ! matches head of list
				zlist => head
				return
			end if
			
			if (head% lgz < lgz) then ! becomes new head of list
				if (.not. alloc_zlist()) return
				zlist% nxt => head
				head => zlist
				return
			end if
			
			! check list
			z1 => head
			do while (associated(z1% nxt))
				z2 => z1% nxt
				if (z2% lgz == lgz) then
					zlist => z2; return
				end if
				if (z2% lgz < lgz) then ! insert new one before z2 
					if (.not. alloc_zlist()) return
					zlist% nxt => z2
					z1% nxt => zlist
					return
				end if
				z1 => z2
			end do
			! add to end of list after z1
			if (.not. alloc_zlist()) return
			z1% nxt => zlist
			
			contains
			
			logical function alloc_zlist()
				integer :: istat
				allocate(zlist,stat=istat)
				if (istat /= 0) then
					alloc_zlist = .false.; ierr = -1; return
				end if
				nullify(zlist% nxt)
				zlist% lgz = lgz		
				num_created = num_created + 1	
				alloc_zlist = .true.
			end function alloc_zlist
			
		end subroutine get_zlist
							

      end module mod_colors

