! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
!
! ***********************************************************************

      module chem_lib
      use chem_def, only: chem_has_been_initialized
      use const_def, only: dp
      
      implicit none


      contains
      
      
      subroutine chem_init(isotopes_filename, ierr) 
         ! uses mesa_data_dir from const_def
         use chem_def
         use chem_isos_io, only: do_read_chem_isos
         use nuclide_set_mod, only: set_some_isos
				 use lodders_mod, only : read_lodders03_data
         character (len=*), intent(in) :: isotopes_filename
         integer, intent(out) :: ierr
         ierr = 0
         if (chem_has_been_initialized) return
         call init_chem_tables
         call read_lodders03_data('lodders03.data',ierr)
         if (ierr /= 0) then
            write(*,*) 'failed in read_lodders03_data'
            return
         end if
      	call do_read_chem_isos(isotopes_filename, ierr)
         if (ierr /= 0) then
            write(*,*) 'failed in do_read_chem_isos'
            return
         end if
      	call set_some_isos
      	chem_has_been_initialized = .true.
      end subroutine chem_init
      
      
      subroutine basic_composition_info(
     >      num_isos, chem_id, x, xh, xhe, 
     >      abar, zbar, z2bar, ye, mass_correction, sumx)
         integer, intent(in) :: num_isos
         integer, intent(in) :: chem_id(:) ! (num_isos) ! the nuclide indices for the entries in x
         real(dp), intent(in)  :: x(:) ! (num_isos) ! baryon fractions.  should sum to 1.0
         real(dp), intent(out) :: 
     >         xh, xhe, abar, zbar, z2bar, ye, mass_correction, sumx
         real(dp), dimension(0) :: dabar_dx, dzbar_dx, dmc_dx
         call get_composition_info(
     >      num_isos, chem_id, x, xh, xhe, 
     >      abar, zbar, z2bar, ye, mass_correction,
     >      sumx, .true., dabar_dx, dzbar_dx, dmc_dx)
      end subroutine basic_composition_info
      
         
      subroutine composition_info(
     >      num_isos, chem_id, x, xh, xhe, 
     >      abar, zbar, z2bar, ye, mass_correction,
     >      sumx, dabar_dx, dzbar_dx, dmc_dx)
         integer, intent(in) :: num_isos
         integer, intent(in) :: chem_id(:) ! (num_isos) ! the nuclide indices for the entries in x
         real(dp), intent(in)  :: x(:) ! (num_isos) ! baryon fractions.  should sum to 1.0
         real(dp), intent(out) :: 
     >         xh, xhe, abar, zbar, z2bar, ye, mass_correction, sumx
         real(dp), dimension(:) :: dabar_dx, dzbar_dx, dmc_dx
         call get_composition_info(
     >      num_isos, chem_id, x, xh, xhe, 
     >      abar, zbar, z2bar, ye, mass_correction,
     >      sumx, .false., dabar_dx, dzbar_dx, dmc_dx)
      end subroutine composition_info
      
         
      subroutine get_composition_info(
     >      num_isos, chem_id, x, xh, xhe, 
     >      abar, zbar, z2bar, ye, mass_correction,
     >      sumx, skip_partials, dabar_dx, dzbar_dx, dmc_dx)

         ! here's a reminder of definitions:
         ! X(i) ion baryon fraction (g/g)
				 ! A(i) ion atomic mass number
         ! W(i) ion atomic weight (g/mole)
         ! Z(i) ion charge (number of protons)
         ! Y(i) = X(i)/A(i), ion abundance 
         ! n(i) = rho*avo*Y(i), ion number density (g/cm^3)*(#/mole)*(mole/g) -> (#/cm^3)
         
         ! abar = sum(n(i)*A(i))/sum(n(i)) -- average atomic mass number
         ! zbar = sum(n(i)*Z(i))/sum(n(i)) -- average charge number
         ! z2bar = sum(n(i)*Z(i)^2)/sum(n(i)) -- average charge^2
         ! n = rho*avo/abar -- total number density (#/cm^3)
         ! ye = zbar/abar -- average charge per baryon = proton fraction
         ! xh = mass fraction hydrogen
         ! xhe = mass fraction helium
				 ! mass_correction = sum(n(i)*W(i))/sum(n(i)*A(i)) --
				 ! (mass density) = (baryon density) * m_u * mass_correction

         use chem_def
         integer, intent(in) :: num_isos
         integer, intent(in) :: chem_id(:) ! (num_isos) ! the nuclide indices for the entries in x
         real(dp), intent(in)  :: x(:) ! (num_isos) ! baryon fractions.  should sum to 1.0
         real(dp), intent(out) :: 
     >         xh, xhe, abar, zbar, z2bar, ye, mass_correction, sumx
         logical, intent(in) :: skip_partials
         real(dp), dimension(:) :: dabar_dx, dzbar_dx, dmc_dx
     
         real(dp), dimension(num_isos) :: y, z, w, a
         integer :: i, cid, iz
         if (.not. chem_has_been_initialized) then
            write(*,*) 'must call chem_init before calling any other routine in chem_lib'
            xh=0; xhe=0; abar=0; zbar=0; z2bar=0; ye=0; mass_correction = 0; sumx=0
            return
         end if
         xh = 0d0; xhe = 0d0
         do i=1,num_isos
            cid = chem_id(i)
            iz = chem_isos% Z(cid)
            z(i) = iz
            y(i) = x(i)/dble(chem_isos% Z_plus_N(cid))
   			w(i) = chem_isos% W(cid)
            a(i) = chem_isos% Z_plus_N(cid)
            select case(iz)
               case (1)
                  xh = xh + x(i)
               case (2)
                  xhe = xhe + x(i)
            end select
         end do

         abar = 1d0/sum(min(1d0,max(y(1:num_isos),1.0d-50)))				 
         zbar = abar * sum(y(1:num_isos)*z(1:num_isos))
         z2bar = abar * sum(y(1:num_isos)*z(1:num_isos)**2)
         ye = zbar/abar ! assume complete ionization
         mass_correction = sum(y(1:num_isos)*w(1:num_isos))
         sumx = sum(x(1:num_isos))		! this should be one, always, since we define x as a baryon fraction
         
         if (skip_partials) return
         
         do i=1,num_isos
            dabar_dx(i) = abar*(a(i)-abar)/a(i)
            dzbar_dx(i) = abar*(z(i)-zbar)/a(i)
            dmc_dx(i) = w(i)/a(i) - mass_correction
         end do
         
      end subroutine get_composition_info
      
      
      real(dp) function reaction_Qtotal(num_in,num_out,reactants,nuclides)
         use chem_def
         integer, intent(in) :: num_in,num_out,reactants(:)
      	type(nuclide_data), intent(in) :: nuclides
         integer :: j, l
         real(dp) :: Q
         reaction_Qtotal = 0
			do j=1,num_in+num_out
			   l = reactants(j)
			   Q = nuclides% binding_energy(l) - nuclides% Z(l)*del_Mp - nuclides% N(l)*del_Mn
			   if (j <= num_in) then
			      reaction_Qtotal = reaction_Qtotal - Q
			   else
			      reaction_Qtotal = reaction_Qtotal + Q
			   end if
			end do
      end function reaction_Qtotal
            
      
		integer function chem_get_element_id(cname) 
		! NOTE: this is for elements like 'h', not for isotopes like 'h1'
		! use chem_get_iso_id for looking up isotope names
         use chem_def
         use utils_lib
			character (len=*), intent(in)  :: cname 
		   ! name of the element (e.g. 'h', 'he', 'ne')
		   ! same names as in chem_element_Name
			! returns id for the element if there is a matching name
			! returns 0 otherwise.
			integer :: ierr, value
         if (.not. chem_has_been_initialized) then
            write(*,*) 'must call chem_init before calling any other routine in chem_lib'
            chem_get_element_id = -1
            return
         end if
			call integer_dict_lookup(chem_element_names_dict, cname, value, ierr)
			if (ierr /= 0) value = -1
			chem_get_element_id = value
		end function chem_get_element_id
		
		
		real(dp) function chem_Xsol(nam)
			character (len=*), intent(in)  :: nam 
			   ! name of the isotope (e.g. 'h1', 'he4', 'ne20')
		   real(dp) :: z, a, xelem
		   integer :: ierr
         if (.not. chem_has_been_initialized) then
            write(*,*) 'must call chem_init before calling any other routine in chem_lib'
            chem_Xsol = -1
            return
         end if
		   call chem_get_solar(nam, z, a, xelem, ierr)
		   if (ierr /= 0) then
		      chem_Xsol = 0d0
		   else
		      chem_Xsol = xelem
		   end if
		end function chem_Xsol
      

		subroutine chem_get_solar(nam, z, a, xelem, ierr)
         use chem_def
		   use utils_lib
		   ! returns data from Anders and Grevesse, 1989
			character (len=*), intent(in)  :: nam 
			   ! name of the isotope (e.g. 'h1', 'he4', 'ne20')
				! note that these names match those in the nuclear net library iso_Names array
				! but some net isotopes are not here (ex/ be7, n13, o14, o15, f17, f18, ... fe52, ni56 )
				! and some isotopes are here that are not in the nets (ex/ hf176)
			real(dp), intent(out) :: z ! charge
			real(dp), intent(out) :: a ! number of nucleons (protons and neutrons)
			real(dp), intent(out) :: xelem ! elemental mass fraction associated with this isotope
			integer, intent(out) :: ierr ! == 0 means AOK; == -1 means did not find the requested name
			integer :: i
			ierr = 0
         if (.not. chem_has_been_initialized) then
            write(*,*) 'must call chem_init before calling any other routine in chem_lib'
            ierr = -1; z = 0; a = 0; xelem = 0; return
         end if
			call integer_dict_lookup(Xsol_names_dict, nam, i, ierr)
			if (ierr /= 0) then
   			z = 0; a = 0; xelem = 0; return
			end if
			z = dble(izsol(i))
			a = dble(iasol(i))
		   xelem = solx(i)
		end subroutine chem_get_solar

      
      subroutine chem_create_set_from_file(set_filename, default_dir, set, ierr)
         ! looks first in local directory, then in default_dir
         use chem_def, only: nuclide_set, iso_name_length
         use utils_lib
         character (len=*), intent(in) :: set_filename, default_dir
      	type(nuclide_set), dimension(:), pointer :: set ! will be allocated
      	integer, intent(out) :: ierr
         
         integer :: nnuc, iounit
      	character(len=iso_name_length), pointer :: names(:)
         integer, dimension(:), pointer :: Z, A ! will be allocated by chem_read_ZA
         character (len=256) :: filename
         
         if (.not. chem_has_been_initialized) then
            write(*,*) 'must call chem_init before calling any other routine in chem_lib'
            ierr = -1
            return
         end if
         ierr = 0
         iounit = alloc_iounit(ierr)
         if (ierr /= 0) return
         filename = trim(set_filename)
         open(unit=iounit, file=trim(filename), action='read', status='old', iostat=ierr)
         close(iounit)
         if (ierr /= 0) then ! if don't find that file, look in default_dir
            if (len_trim(default_dir) > 0) then
               filename = trim(default_dir) // '/' // trim(set_filename)
               ierr = 0
               open(unit=iounit, file=trim(filename), action='read', status='old', iostat=ierr)
               close(iounit)
            end if
            if (ierr /= 0) then
               call free_iounit(iounit)
               write(*,*) 'chem_create_set_from_file failed for open ' // trim(set_filename)
               return
            end if
            call free_iounit(iounit)
         end if

      	call chem_read_ZA(filename, Z, A, nnuc, ierr)
      	if (ierr /= 0) then
      	   write(*,*) 'chem_create_set_from_file failed reading ' // trim(set_filename)
      	   return
      	end if
      	
      	allocate(names(nnuc), set(nnuc), stat=ierr)
      	if (ierr /= 0) return
      	
      	call generate_nuclide_names(Z,A,names)	
      	call generate_nuclide_set(names,set)
      	deallocate(names, Z, A)
      	
      end subroutine chem_create_set_from_file

      
      
      ! create arrays of Z's and A's from file of Z, A_lo, A_hi triples.
      subroutine chem_read_ZA(filename, Z, A, nnuc, ierr)
         use nuclide_set_mod, only: read_ZA
         character (len=*), intent(in) :: filename
         integer, dimension(:), pointer :: Z, A ! will be allocated
         integer, intent(out) :: nnuc, ierr
         call read_ZA(filename, Z, A, nnuc, ierr)
      end subroutine chem_read_ZA
      
      
      ! given an array of Z, A, returns an array of names in chem_isos format
      subroutine generate_nuclide_names(Z, A, names)
         use chem_def
      	integer, dimension(:), intent(in) :: Z, A
      	character(len=iso_name_length), dimension(size(Z)), intent(out) :: names
      	integer :: i, ierr, count_isomer
      	character(len=80) :: message
         logical :: use_al26_isomers

      	count_isomer = 0
         use_al26_isomers = .false.
      	do i = 1, size(Z)
           if (A(i) == 26 .and. Z(i) == 13) count_isomer = count_isomer + 1
         enddo
         if (count_isomer > 1) use_al26_isomers = .true.

      	count_isomer = 1
      	do i = 1, size(Z)
      		select case(Z(i))
      		case (0)
      			names(i) = el_name(Z(i))
      		case (1:max_el_z)
      			write(names(i), '(a, i0)') trim(el_name(Z(i))), A(i)
      			if (A(i) == 26 .and. Z(i) == 13 .and. use_al26_isomers) then
      				count_isomer = count_isomer + 1
      				!if (count_isomer > 1) names(i) = al_isomers(count_isomer)
      				names(i) = al_isomers(count_isomer)
      			end if
      		case default
      			write(*, '(a, i0, a, i0)') 'warning: ', Z(i), ' greater than Zmax = ', max_el_z
      			names(i) = '*****'
      			ierr = nuclide_not_found
      		end select
      		names(i) = adjustr(names(i))
      	end do
      end subroutine generate_nuclide_names
      

      subroutine generate_long_nuclide_names(Z, A, long_names)
         use chem_def
      	integer, dimension(:), intent(in) :: Z, A
      	character(len=long_name_length), dimension(size(Z)), intent(out) :: long_names
      	integer :: i, ierr, count_isomer
      	character(len=80) :: message
         logical :: use_al26_isomers

      	count_isomer = 0
         use_al26_isomers = .false.
      	do i = 1, size(Z)
           if (A(i) == 26 .and. Z(i) == 13) count_isomer = count_isomer + 1
         enddo
         if (count_isomer > 1) use_al26_isomers = .true.

      	count_isomer = 1
      	do i = 1, size(Z)
      		select case(Z(i))
      		case (0)	! neutrons are special?
      			long_names(i) = el_long_name(Z(i))
      		case (1:max_el_z)
      			write(long_names(i), '(a, "-", i0)') trim(el_long_name(Z(i))), A(i)
      			if (A(i) == 26 .and. Z(i) == 13 .and. use_al26_isomers) then
      				count_isomer = count_isomer + 1
      				!if (count_isomer > 1) long_names(i) = long_al_isomers(count_isomer)
                                    long_names(i) = long_al_isomers(count_isomer)
      			end if
      		case default
      			write(*, '(a, i0, a, i0)') 'warning: ', Z(i), ' greater than Zmax = ', max_el_z
      			long_names(i) = '********'
      			ierr = nuclide_not_found
      		end select
      	end do
      end subroutine generate_long_nuclide_names
      
      
      ! nuclide information comes from the chem_isos tables
      ! the storage container for the data is called 'chem_isos'
      ! it has name, A, Z, N, spin, and B for each nuclide
      ! use the function get_nuclide_index to find the index given the name.

      ! returns the index of a particular nuclide in the full chem_isos set
      ! returns nuclide_not_found if name not found
      function get_nuclide_index(nuclei) result(indx)
         use chem_def
      	use utils_lib, only: integer_dict_lookup
      	character(len=*), intent(in) :: nuclei
      	integer :: indx, ierr
         if (.not. chem_has_been_initialized) then
            write(*,*) 'must call chem_init before calling any other routine in chem_lib'
            indx = nuclide_not_found
            return
         end if
      	ierr = 0
      	call integer_dict_lookup(chem_isos_dict, nuclei, indx, ierr)
      	if (ierr /= 0) indx = nuclide_not_found
      end function get_nuclide_index
            
      
		integer function chem_get_iso_id(cname) ! same as get_nuclide_index
      	character(len=*), intent(in) :: cname
      	chem_get_iso_id = get_nuclide_index(cname)
		end function chem_get_iso_id  
            
      
		integer function lookup_ZN(Z,N)
		   use chem_def, only: chem_isos, num_chem_isos
      	integer, intent(in) :: Z, N
      	lookup_ZN = lookup_ZN_isomeric_state(Z,N,0)
		end function lookup_ZN  
            
      
		integer function lookup_ZN_isomeric_state(Z,N,isomeric_state)
		   use chem_def, only: chem_isos, num_chem_isos
      	integer, intent(in) :: Z, N, isomeric_state
      	integer :: cid, i      	
      	iso_loop: do cid = 1, num_chem_isos
      	   if (chem_isos% Z(cid) == Z .and. chem_isos% N(cid) == N) then
   	         if (chem_isos% isomeric_state(cid) == isomeric_state) then
   	            lookup_ZN_isomeric_state = cid
   	            return
   	         end if
      	      do i = cid+1, num_chem_isos
      	         if (chem_isos% Z(i) /= Z .or. chem_isos% N(i) /= N) exit iso_loop
      	         if (chem_isos% isomeric_state(i) == isomeric_state) then
      	            lookup_ZN_isomeric_state = i
      	            return
      	         end if
      	      end do
      	   end if
      	end do iso_loop
         lookup_ZN_isomeric_state = 0 ! indicating failure
		end function lookup_ZN_isomeric_state  
      

      subroutine generate_nuclide_set(names, set)
         use chem_def
      	use nuclide_set_mod, only: sort_nuclide_set
      	character(len=iso_name_length), dimension(:), intent(in) :: names
      	type(nuclide_set), dimension(size(names)), intent(out) :: set
      	integer :: i
      	set = [(nuclide_set(names(i), i), i=1, size(names))]
      	call sort_nuclide_set(set)
      end subroutine generate_nuclide_set
      

      ! returns the index of a particular nuclide in a particular set
      ! returns nuclide_not_found if name not found
      function get_nuclide_index_in_set(nuclei, set) result(indx)
         use chem_def
      	use nuclide_set_mod
      	character(len=*), intent(in) :: nuclei
      	type(nuclide_set), dimension(:), intent(in) :: set
      	integer :: indx
      	character(len=iso_name_length) :: name
      	name = nuclei
      	indx = rank_in_set(name, set)
      end function get_nuclide_index_in_set
      
	
      subroutine generate_nuclide_permutation(iso_array, set, permutation, ierr)
         use chem_def
      	use nuclide_set_mod, only: rank_in_set
      	character(len=iso_name_length), dimension(:), intent(in) :: iso_array
      	type(nuclide_set), dimension(size(iso_array)), intent(in) :: set
      	integer, dimension(size(iso_array)), intent(out) :: permutation
      	integer, intent(out) :: ierr
      	integer :: i	
      	ierr = 0
      	permutation = [ (rank_in_set(iso_array(i), set), i=1, size(iso_array)) ]
      	if (any(permutation == nuclide_not_found)) ierr = -1
      end subroutine generate_nuclide_permutation
      

      subroutine extract_nuclides_from_chem_isos(set, nuclides, ierr)
         use chem_def
      	use chem_isos_io, only : do_extract_nuclides_from_chem_isos
      	type(nuclide_set), dimension(:), intent(in) :: set
      	type(nuclide_data), intent(out) :: nuclides ! contents will be allocated
      	   ! call free_nuclide_data to release when done
      	integer, intent(out) :: ierr
      	call do_extract_nuclides_from_chem_isos(chem_isos, nuclides, set, ierr)
      end subroutine extract_nuclides_from_chem_isos
      
      
      subroutine create_nuclides_dict(nuclides, nuclides_dict, ierr)
         use chem_def
      	use chem_isos_io, only : do_create_nuclides_dict
      	type(nuclide_data), intent(in) :: nuclides
         type (integer_dict), pointer :: nuclides_dict ! will be allocated
      	integer, intent(out) :: ierr
         call do_create_nuclides_dict(nuclides, nuclides_dict, ierr)
      end subroutine create_nuclides_dict
      
      
      subroutine nuclides_composition_info(Y, nuclides, abar, zbar, z2bar, ye)
         use chem_def
         use nuclide_set_mod, only: do_composition_info
      	real(dp), intent(in) :: Y(:)
      	type(nuclide_data), intent(in) :: nuclides
         real(dp), intent(out) :: abar, zbar, z2bar, ye
         call do_composition_info(Y, nuclides, abar, zbar, z2bar, ye)
      end subroutine nuclides_composition_info
            
      
		integer function rates_category_id(cname)
		   use chem_def, only: category_names_dict
         use utils_lib
			character (len=*), intent(in)  :: cname 
			! returns id for the category if there is a matching name
			! returns 0 otherwise.
			integer :: ierr, value
			call integer_dict_lookup(category_names_dict, cname, value, ierr)
			if (ierr /= 0) value = 0
			rates_category_id = value
		end function rates_category_id
      

      function binding_energy(nuclides, Y) result (B)
         use chem_def
      	type(nuclide_data), intent(in) :: nuclides
      	real(dp), dimension(size(nuclides% binding_energy)), intent(in) :: Y
      	real(dp) :: B
      	B = dot_product(nuclides% binding_energy, Y)
      end function binding_energy
      

      ! returns the indx corresponding to Tpart just less than T9
      ! T9 is the temperature in units of GK
      ! returns a value of 0 or npart if value is less than the minimum or maximum of the partition function
      ! temperature array Tpart
      function get_partition_fcn_indx(T9) result(indx)
      	use chem_def, only: Tpart, npart
      	real(dp), intent(in) :: T9
      	integer :: indx
      	integer, parameter :: max_iterations = 8
      	integer :: low, high, mid, i

      	low = 1
      	high = npart

      	if (T9 < Tpart(low)) then
      		indx = low-1
      		return
      	end if
      	if (T9 > Tpart(high)) then
      		indx = high + 1
      	end if

      	do i = 1, max_iterations
      		if (high-low <= 1) then
      			indx = low
      			return
      		end if
      		mid = (high+low)/2
      		if (T9 < Tpart(mid)) then
      			high = mid
      		else
      			low = mid
      		end if
      	end do
      	! should never get here
      	indx = low-1
      	
      end function get_partition_fcn_indx


      function lodders03_element_atom_percent(nuclei) result(percent)
      ! Katharina Lodders,
      ! "Solar System Abundances and Condensation Temperatures of the Elements",
      ! ApJ, 591, 1220-1247 (2003).
      ! Table 6: Abundances of the Isotopes in the Solar System
      
      ! These are element atom percentages (i.e., by number, not by mass)
      
      ! NOTE: The data here stops at ge -- the table in the paper goes to 92U
      
      ! TO DO: add the rest of the info from the paper
				use chem_def
				use lodders_mod
				character(len=*), intent(in) :: nuclei
				real(dp) :: percent
				integer :: ierr
				
				if (.not. chem_has_been_initialized) then
					write(*,*) 'must call chem_init before calling any other routine in chem_lib'
				  percent = -1.0
					return
				end if
				percent = get_lodders03_isotopic_abundance(nuclei, ierr)
				if (ierr /= 0) percent = 0.0
      end function lodders03_element_atom_percent



      end module chem_lib

