! ***********************************************************************
!
!   Copyright (C) 2011  Bill Paxton, Aaron Dotter
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
!
! ***********************************************************************

      module chem_def
      
      use utils_def, only: integer_dict
      use const_def, only: dp

      
      implicit none




      integer, parameter :: iso_name_length = 8	! no. characters in an nuclide name
         ! 2 for element, 3 for A, and 3 for isomeric state as '_' + 1 or 2 digit integer 0 to 99)
      integer, parameter :: long_name_length = 16 ! no. characters in a long name
      integer, parameter :: npart = 24	! no. entries in partition fcn table
      integer, parameter :: max_el_z = 112	! max. Z in the winvn database
      integer, parameter :: nhname = 3	! no. isotopes for h in the winvn database

      logical, parameter :: convert_mass_excess_to_binding_energy = .true.
      logical, parameter :: report_solar = .false.
      integer, parameter :: nuclide_not_found = -1	! warning flag

      ! element names
      character(len=iso_name_length), dimension(0:max_el_z) :: el_name = [character(len=iso_name_length) ::
     >      'neut','h','he','li','be','b','c','n','o','f','ne', 
     >      'na','mg','al','si','p','s','cl','ar','k','ca', 
     >      'sc','ti','v','cr','mn','fe','co','ni','cu','zn', 
     >      'ga','ge','as','se','br','kr','rb','sr','y','zr', 
     >      'nb','mo','tc','ru','rh','pd','ag','cd','in','sn', 
     >      'sb','te','i','xe','cs','ba','la','ce','pr','nd', 
     >      'pm','sm','eu','gd','tb','dy','ho','er','tm','yb', 
     >      'lu','hf','ta','w','re','os','ir','pt','au','hg', 
     >      'tl','pb','bi','po','at','rn','fr','ra','ac','th',
     >      'pa','u','np','pu','am','cm','bk','cf','es','fm','md',
     >      'no','lr','rf','db','sg','bh','hs','mt','ds','rg','cn']
      		
      character(len=long_name_length), dimension(0:max_el_z) :: el_long_name = [character(len=long_name_length) :: 
     >      'neutron','hydrogen','helium','lithium','beryllium','boron','carbon','nitrogen','oxygen','fluorine','neon', 
     >      'sodium','magnesium','aluminum','silicon','phosphorus','sulfur','chlorine','argon','potassium','calcium', 
     >      'scandium','titanium','vanadium','chromium','manganese','iron','cobalt','nickel','copper','zinc','gallium', 
     >      'germanium','arsenic','selenium','bromine','krypton','rubidium','strontium','yttrium','zirconium','niobium', 
     >      'molybdenum','technetium','ruthenium','rhodium','palladium','silver','cadmium','indium','tin','antimony', 
     >      'tellurium','iodine','xenon','cesium','barium','lanthanum','cerium','praseodymium','neodymium','promethium', 
     >      'samarium','europium','gadolinium','terbium','dysprosium','holmium','erbium','thulium','ytterbium','lutetium', 
     >      'hafnium','tantalum','tungsten','rhenium','osmium','iridium','platinum','gold','mercury','thallium','lead', 
     >      'bisumth','polonium','astatine','radon','francium','radium','actinium','thorium','protactinium','uranium',
     >      'neptunium','plutonium','americium','curium','berkelium','californium','einsteinium','fermium','mendelevium',
     >      'nobelium','lawrencium','rutherfordium','dubnium','seaborgium','bohrium','hassium','meitnerium','darmstadtium',
     >      'roentgenium','copernicum' ]

      ! aluminum isomers
      character(len=iso_name_length), dimension(2:3) :: al_isomers = [character(len=iso_name_length) :: 
     >      'al-6','al*6']
      character(len=long_name_length), dimension(2:3) :: long_al_isomers = [character(len=long_name_length) :: 
     >      'Aluminum-gs','Aluminum-ex']
      
      
      ! chem element id numbers (up to Cn)
      ! note: for isotope i, the element id number = chem_Z(i)

      !periodic table, row 1
      integer, parameter :: e_h = 1
      integer, parameter :: e_he = 2

      !periodic table, row 2
      integer, parameter :: e_li = 3
      integer, parameter :: e_be = 4
      integer, parameter :: e_b = 5
      integer, parameter :: e_c = 6
      integer, parameter :: e_n = 7
      integer, parameter :: e_o = 8
      integer, parameter :: e_f = 9
      integer, parameter :: e_ne = 10

      !periodic table, row 3
      integer, parameter :: e_na = 11
      integer, parameter :: e_mg = 12
      integer, parameter :: e_al = 13
      integer, parameter :: e_si = 14
      integer, parameter :: e_p = 15
      integer, parameter :: e_s = 16
      integer, parameter :: e_cl = 17
      integer, parameter :: e_ar = 18

      !periodic table, row 4
      integer, parameter :: e_k = 19
      integer, parameter :: e_ca = 20
      integer, parameter :: e_sc = 21
      integer, parameter :: e_ti = 22
      integer, parameter :: e_v = 23
      integer, parameter :: e_cr = 24
      integer, parameter :: e_mn = 25
      integer, parameter :: e_fe = 26
      integer, parameter :: e_co = 27
      integer, parameter :: e_ni = 28
      integer, parameter :: e_cu = 29
      integer, parameter :: e_zn = 30
      integer, parameter :: e_ga = 31
      integer, parameter :: e_ge = 32
      integer, parameter :: e_as = 33
      integer, parameter :: e_se = 34
      integer, parameter :: e_br = 35
      integer, parameter :: e_kr = 36
      
      !periodic table, row 5
      integer, parameter :: e_rb = 37
      integer, parameter :: e_sr = 38
      integer, parameter :: e_y = 39
      integer, parameter :: e_zr = 40
      integer, parameter :: e_nb = 41
      integer, parameter :: e_mo = 42
      integer, parameter :: e_tc = 43
      integer, parameter :: e_ru = 44
      integer, parameter :: e_rh = 45
      integer, parameter :: e_pd = 46
      integer, parameter :: e_ag = 47
      integer, parameter :: e_cd = 48
      integer, parameter :: e_in = 49
      integer, parameter :: e_sn = 50
      integer, parameter :: e_sb = 51
      integer, parameter :: e_te = 52
      integer, parameter :: e_i = 53
      integer, parameter :: e_xe = 54

      !periodic table, row 6
      integer, parameter :: e_cs = 55
      integer, parameter :: e_ba = 56      
      integer, parameter :: e_la = 57
      integer, parameter :: e_ce = 58
      integer, parameter :: e_pr = 59
      integer, parameter :: e_nd = 60
      integer, parameter :: e_pm = 61
      integer, parameter :: e_sm = 62
      integer, parameter :: e_eu = 63
      integer, parameter :: e_gd = 64
      integer, parameter :: e_tb = 65
      integer, parameter :: e_dy = 66
      integer, parameter :: e_ho = 67
      integer, parameter :: e_er = 68
      integer, parameter :: e_tm = 69
      integer, parameter :: e_yb = 70
      integer, parameter :: e_lu = 71
      integer, parameter :: e_hf = 72
      integer, parameter :: e_ta = 73
      integer, parameter :: e_w  = 74
      integer, parameter :: e_re = 75
      integer, parameter :: e_os = 76
      integer, parameter :: e_ir = 77
      integer, parameter :: e_pt = 78
      integer, parameter :: e_au = 79
      integer, parameter :: e_hg = 80
      integer, parameter :: e_tl = 81
      integer, parameter :: e_pb = 82
      integer, parameter :: e_bi = 83
      integer, parameter :: e_po = 84
      integer, parameter :: e_at = 85
		integer, parameter :: e_rn = 86
		
		!periodic table, row 7
		integer, parameter :: e_fr = 87
		integer, parameter :: e_ra = 88
		integer, parameter :: e_ac = 89
		integer, parameter :: e_th = 90
		integer, parameter :: e_pa = 91
		integer, parameter :: e_u  = 92
		integer, parameter :: e_np = 93
		integer, parameter :: e_pu = 94
		integer, parameter :: e_am = 95
		integer, parameter :: e_cm = 96
		integer, parameter :: e_bk = 97
		integer, parameter :: e_cf = 98
		integer, parameter :: e_es = 99
		integer, parameter :: e_fm = 100
		integer, parameter :: e_md = 101
		integer, parameter :: e_no = 102
		integer, parameter :: e_lr = 103
		integer, parameter :: e_rf = 104
		integer, parameter :: e_db = 105
		integer, parameter :: e_sg = 106
		integer, parameter :: e_bh = 107
		integer, parameter :: e_hs = 108
		integer, parameter :: e_mt = 109
		integer, parameter :: e_ds = 110
		integer, parameter :: e_rg = 111
		integer, parameter :: e_cn = 112
			
      integer, parameter :: num_chem_elements = max_el_z

      
      ! anders & grevesse 1989
	   integer, parameter :: solsiz = 286
	   integer, parameter :: solnamelen = 5
      character (len=solnamelen) :: namsol(solsiz)
      integer :: izsol(solsiz),iasol(solsiz)
      real(dp) :: solx(solsiz), zsol, yesol ! according to AG89
      type (integer_dict), pointer :: Xsol_names_dict
      
      
      ! various values for current solar Z and Y (at photosphere)
      ! note that these have been reduced by diffusion from pre-MS values.
      ! values updated from Asplund et al. ARAA, 2009, 47, 481

      real(dp), parameter :: AG89_zsol   = 0.0201d0
      real(dp), parameter :: GN93_zsol   = 0.0179d0
      real(dp), parameter :: GS98_zsol   = 0.0169d0
      real(dp), parameter :: L03_zsol    = 0.0133d0
      real(dp), parameter :: AGS04_zsol  = 0.0122d0
      real(dp), parameter :: AGSS09_zsol = 0.0134d0
      real(dp), parameter :: L09_zsol    = 0.0154d0
      
      real(dp), parameter :: AG89_ysol   = 0.2485d0
      real(dp), parameter :: GN93_ysol   = 0.2485d0
      real(dp), parameter :: GS98_ysol   = 0.2485d0
      real(dp), parameter :: L03_ysol    = 0.2377d0
      real(dp), parameter :: AGS04_ysol  = 0.2485d0
      real(dp), parameter :: AGSS09_ysol = 0.2485d0
      real(dp), parameter :: L09_ysol    = 0.2751d0
      
      character(len=iso_name_length) :: chem_element_main_iso_name(num_chem_elements)
      integer, parameter :: chem_element_name_len = iso_name_length
      character (len=chem_element_name_len)  :: chem_element_Name(num_chem_elements) ! names for elements
      
      
      ! identifiers for different Z fractions.
      integer, parameter :: AG89_zfracs = 1
      integer, parameter :: GN93_zfracs = 2
      integer, parameter :: GS98_zfracs = 3
      integer, parameter :: L03_zfracs = 4
      integer, parameter :: AGS04_zfracs = 5
      integer, parameter :: AGSS09_zfracs = 6
      integer, parameter :: L09_zfracs = 7
      
         
      real(dp) :: AG89_element_zfrac(num_chem_elements) ! fraction by mass of total Z
         ! Anders & Grevesse 1989
         
      real(dp) :: GN93_element_zfrac(num_chem_elements) ! fraction by mass of total Z
         ! Grevesse and Noels 1993 abundances
         
      real(dp) :: GS98_element_zfrac(num_chem_elements) ! fraction by mass of total Z
         ! Grevesse and Sauval 1998 abundances
         
      real(dp) :: L03_element_zfrac(num_chem_elements) ! fraction by mass of total Z
         ! Lodders 2003 abundances
         
      real(dp) :: AGS04_element_zfrac(num_chem_elements) ! fraction by mass of total Z
         ! Asplund, Grevesse, and Sauval 2004 abundances
         
      real(dp) :: AGSS09_element_zfrac(num_chem_elements) ! fraction by mass of total Z
         ! Asplund, Grevesse, Sauval, and Scott 2009 abundances
         ! Annu. Rev. Astron. Astrophys. 2009. 47:481–522
         
      real(dp) :: L09_element_zfrac(num_chem_elements) ! fraction by mass of total Z
         ! Lodders and Palme, 2009.  (http://adsabs.harvard.edu/abs/2009M%26PSA..72.5154L)
         
         

      type (integer_dict), pointer :: chem_element_names_dict

         
      real(dp) :: element_atomic_weight(num_chem_elements)
         ! de Laeter et al, Pure and Applied Chemistry 75(6), 683–799, 2003. (IUPAC Technical Report)
      


      ! temperature values at which partition function is defined
      real(dp), dimension(npart) :: Tpart = (/ 
     >      0.10,0.15,0.20,0.30,0.40,0.50,0.60,0.70,0.80,0.90,1.00,1.50,
     >      2.00,2.50,3.00,3.50,4.00,4.50,5.00,6.00,7.00,8.00,9.00,10.0 /)
      	

      ! mass excess of proton, neutron in MeV (for calculating binding energies)
      real(dp), parameter :: del_Mp = 7.288969d0, del_Mn = 8.071323d0
      

      type nuclide_data
      	integer :: nnuclides
      	character(len=iso_name_length), dimension(:), pointer :: name	! name of nuclide
      	integer, dimension(:), pointer :: chem_id ! (nnuclides) gives chem_id for member of nuclide_data
      	integer, dimension(:), pointer :: nuclide ! (num_chem_isos)
      	   ! gives index in nuclide_data 1 .. nnuclides or 0 if not included
      	real(dp), dimension(:), pointer :: W ! atomic weight (mass in amu units)
      	integer, dimension(:), pointer :: Z	! number of protons
      	integer, dimension(:), pointer :: N	! number of neutrons
      	integer, dimension(:), pointer :: Z_plus_N ! number of baryons
      	integer, dimension(:), pointer :: isomeric_state ! 0 is default
      	real(dp), dimension(:), pointer :: spin	! ground-state spin
      	real(dp), dimension(:), pointer :: binding_energy
            ! the binding energy is B = Z*del_Mp + N*del_Mn - mass_excess
      	real(dp), dimension(:,:), pointer :: pfcn	! table of partition function
      	real(dp), dimension(:), pointer :: mass_excess
      end type nuclide_data
      

      ! storage container for a set of nuclides, used for extracting a subset of the full winvn database
      ! a set of nuclides is actually an array of these items
      type nuclide_set
      	character(len=iso_name_length) :: nuclide
      	integer :: rank
      end type nuclide_set

      type(nuclide_data) :: chem_isos ! from winvn
      type (integer_dict), pointer :: chem_isos_dict
      integer :: num_chem_isos	! no. entries in isotopes database

			! storage for Lodders (2003) isotopic percentages
      type isotopic_abundance_table_type
				type (integer_dict), pointer :: name_dict
				real(dp), dimension(:), allocatable :: isotopic_percent
			end type isotopic_abundance_table_type

			type(isotopic_abundance_table_type) :: lodders03_tab6
			
      
      ! reaction categories

      integer, parameter :: ipp = 1 ! pp chains
      integer, parameter :: icno = 2 ! cno cycles
      integer, parameter :: i3alf = 3 ! triple alpha  
      
      ! "burn" in the following means decays or captures of protons, alphas, or neutrons
      integer, parameter :: i_burn_c = 4
      integer, parameter :: i_burn_n = 5
      integer, parameter :: i_burn_o = 6
      integer, parameter :: i_burn_ne = 7
      integer, parameter :: i_burn_na = 8
      integer, parameter :: i_burn_mg = 9
      integer, parameter :: i_burn_si = 10
      integer, parameter :: i_burn_s = 11
      integer, parameter :: i_burn_ar = 12
      integer, parameter :: i_burn_ca = 13
      integer, parameter :: i_burn_ti = 14
      integer, parameter :: i_burn_cr = 15
      integer, parameter :: i_burn_fe = 16
      
      integer, parameter :: icc = 17 ! c12 + c12
      integer, parameter :: ico = 18 ! c12 + o16
      integer, parameter :: ioo = 19 ! o16 + o16
      
      integer, parameter :: ipnhe4 = 20 ! 2prot + 2neut -> he4
      
      integer, parameter :: iphoto = 21 ! photodisintegration
         ! note: for photodisintegrations, eps_nuc will be negative.
         
      integer, parameter :: iother = 22 ! misc.
   
      integer, parameter :: num_categories = 22

      integer, parameter :: maxlen_category_name = 16
      character (len=maxlen_category_name) :: category_name(num_categories)      
      type (integer_dict), pointer :: category_names_dict

      
      ! some commonly used values of get_nuclide_index      
      integer ::
     >   ih1, ih2, ih3, 
     >   ihe3, ihe4, 
     >   ili6, ili7, ili8, 
     >   ibe7, ibe8, ibe9, ibe10, ibe11, 
     >   ib8, ib10, ib11, ib12, ib13, ib14, 
     >   ic9, ic10, ic11, ic12, ic13, ic14, ic15, ic16, 
     >   in12, in13, in14, in15, in16, in17, in18, in19, in20, 
     >   io13, io14, io15, io16, io17, io18, io19, io20, 
     >   if15, if16, if17, if18, if19, if20, if21, if22, if23, if24, 
     >   ine17, ine18, ine19, ine20, ine21, ine22, ine23, ine24, ine25, 
     >      ine26, ine27, ine28, 
     >   ina20, ina21, ina22, ina23, ina24, ina25, ina26, ina27, ina28, 
     >      ina29, ina30, ina31, 
     >   img20, img21, img22, img23, img24, img25, img26, img27, img28, 
     >      img29, img30, img31, img32, img33, 
     >   ial22, ial23, ial24, ial25, ial26, ial27, ial28, ial29, ial30, 
     >      ial31, ial32, ial33, ial34, ial35, 
     >   isi22, isi23, isi24, isi25, isi26, isi27, isi28, isi29, isi30, 
     >      isi31, isi32, isi33, isi34, isi35, isi36, isi37, isi38, 
     >   ip26, ip27, ip28, ip29, ip30, ip31, ip32, ip33, ip34, ip35, 
     >      ip36, ip37, ip38, ip39, ip40, 
     >   is27, is28, is29, is30, is31, is32, is33, is34, is35, is36, 
     >      is37, is38, is39, is40, is41, is42, 
     >   icl31, icl32, icl33, icl34, icl35, icl36, icl37, icl38, icl39, 
     >      icl40, icl41, icl42, icl43, icl44, 
     >   iar31, iar32, iar33, iar34, iar35, iar36, iar37, iar38, iar39, 
     >      iar40, iar41, iar42, iar43, iar44, iar45, iar46, iar47, 
     >   ik35, ik36, ik37, ik38, ik39, ik40, ik41, ik42, ik43, ik44, ik45, ik46, ik47,
     >   ica35, ica36, ica37, ica38, ica39, ica40, ica41, ica42, ica43, 
     >      ica44, ica45, ica46, ica47, ica48, ica49, ica50, ica51, ica52, ica53, 
     >   isc40, isc41, isc42, isc43, isc44, isc45, isc46, isc47, isc48, 
     >      isc49, isc50, isc51, isc52, isc53, 
     >   iti39, iti40, iti41, iti42, iti43, iti44, iti45, iti46, iti47, 
     >      iti48, iti49, iti50, iti51, iti52, iti53, iti54, iti55, 
     >   iv43, iv44, iv45, iv46, iv47, iv48, iv49, iv50, iv51, iv52, iv53, 
     >      iv54, iv55, iv56, iv57, 
     >   icr43, icr44, icr45, icr46, icr47, icr48, icr49, icr50, icr51, 
     >      icr52, icr53, icr54, icr55, icr56, icr57, icr58, icr59, icr60, 
     >   imn46, imn47, imn48, imn49, imn50, imn51, imn52, imn53, imn54, 
     >      imn55, imn56, imn57, imn58, imn59, imn60, imn61, imn62, imn63, 
     >   ife46, ife47, ife48, ife49, ife50, ife51, ife52, ife53, ife54, ife55, ife56, 
     >      ife57, ife58, ife59, ife60, ife61, ife62, ife63, ife64, ife65, ife66, ife68,
     >   ico50, ico51, ico52, ico53, ico54, ico55, ico56, ico57, ico58, 
     >      ico59, ico60, ico61, ico62, ico63, ico64, ico65, ico66, ico67, 
     >   ini50, ini51, ini52, ini53, ini54, ini55, ini56, ini57, ini58, 
     >      ini59, ini60, ini61, ini62, ini63, ini64, ini65, ini66, ini67, 
     >      ini68, ini69, ini70, ini71, ini72, ini73, 
     >   icu56, icu57, icu58, icu59, icu60, icu61, icu62, icu63, icu64, 
     >      icu65, icu66, icu67, icu68, icu69, icu70, icu71, icu72, 
     >   izn55, izn56, izn57, izn58, izn59, izn60, izn61, izn62, izn63, izn64, 
     >      izn65, izn66, izn67, izn68, izn69, izn70, izn71, izn72, izn73, izn74,
     >   iga60, iga61, iga62, iga63, iga64, iga65, iga66, iga67, iga68, 
     >      iga69, iga70, iga71, iga72, iga73, iga74, iga75, 
     >   ige59, ige60, ige61, ige62, ige63, ige64, ige65, ige66, ige67, 
     >      ige68, ige69, ige70, ige71, ige72, ige73, ige74, ige75, ige76, 
     >   ias71,ias72, ias73, ias74, ias75, ias76, ias77, ias78, ias79,
     >   ise68, ise69, ise70, ise71, ise72, ise73, ise74, ise75, ise76, 
     >      ikr70, ikr72, isr74, isr75, isr76, izr77, izr80, imo82, imo84, 
     >      iru86, iru87, iru88, ipd89, ipd91, ipd92, icd93, icd96, 
     >      isn98, isn100, isn102, isn104,
     >   ineut, iprot


      logical :: chem_has_been_initialized = .false.

      
      contains
      
      
      subroutine init_chem_tables
         use utils_lib, only: integer_dict_define, integer_dict_create_hash
      
         integer :: i, ierr
         
         call init_ag_data
      
         call init_chem_element_names
      
         call init_chem_element_main_iso_names
      
         call init_element_atomic_weights
         
         call init_AG89_data
         
         call init_GN93_data
         
         call init_GS98_data
         
         call init_L03_data
         
         call init_AGS04_data
         
         call init_AGSS09_data
         
         call init_L09_data
         
         nullify(chem_element_names_dict)
         do i=1,num_chem_elements
            call integer_dict_define(chem_element_names_dict, chem_element_Name(i), i, ierr)
            if (ierr /= 0) then
               write(*,*) 'FATAL ERROR: init_chem_tables failed in integer_dict_define'
               stop 1
            end if
         end do
         call integer_dict_create_hash(chem_element_names_dict, ierr)
         if (ierr /= 0) then
            write(*,*) 'FATAL ERROR: init_chem_tables failed in integer_dict_create_hash'
            stop 1
         end if
         
         call set_category_names
         nullify(category_names_dict)
         do i=1,num_categories
            call integer_dict_define(category_names_dict, category_name(i), i, ierr)
            if (ierr /= 0) then
               write(*,*) 'FATAL ERROR: rates_def_init failed in integer_dict_define'
               stop 1
            end if
         end do
         call integer_dict_create_hash(category_names_dict, ierr)
         if (ierr /= 0) then
            write(*,*) 'FATAL ERROR: rates_def_init failed in integer_dict_create_hash'
            stop 1
         end if
         
      end subroutine init_chem_tables
      
      
		subroutine init_ag_data
		   use utils_lib
		
			real(dp) :: sum
			integer :: i, j, ierr

!..names of the stable isotopes
      	namsol(1:120) = ( /
     1 'h1   ','h2   ','he3  ','he4  ','li6  ','li7  ','be9  ','b10  ',
     2 'b11  ','c12  ','c13  ','n14  ','n15  ','o16  ','o17  ','o18  ',
     3 'f19  ','ne20 ','ne21 ','ne22 ','na23 ','mg24 ','mg25 ','mg26 ',
     4 'al27 ','si28 ','si29 ','si30 ','p31  ','s32  ','s33  ','s34  ',
     5 's36  ','cl35 ','cl37 ','ar36 ','ar38 ','ar40 ','k39  ','k40  ',
     6 'k41  ','ca40 ','ca42 ','ca43 ','ca44 ','ca46 ','ca48 ','sc45 ',
     7 'ti46 ','ti47 ','ti48 ','ti49 ','ti50 ','v50  ','v51  ','cr50 ',
     8 'cr52 ','cr53 ','cr54 ','mn55 ','fe54 ','fe56 ','fe57 ','fe58 ',
     9 'co59 ','ni58 ','ni60 ','ni61 ','ni62 ','ni64 ','cu63 ','cu65 ',
     & 'zn64 ','zn66 ','zn67 ','zn68 ','zn70 ','ga69 ','ga71 ','ge70 ',
     1 'ge72 ','ge73 ','ge74 ','ge76 ','as75 ','se74 ','se76 ','se77 ',
     2 'se78 ','se80 ','se82 ','br79 ','br81 ','kr78 ','kr80 ','kr82 ',
     3 'kr83 ','kr84 ','kr86 ','rb85 ','rb87 ','sr84 ','sr86 ','sr87 ',
     4 'sr88 ','y89  ','zr90 ','zr91 ','zr92 ','zr94 ','zr96 ','nb93 ',
     5 'mo92 ','mo94 ','mo95 ','mo96 ','mo97 ','mo98 ','mo100','ru96 ' / )

      	namsol(121:240) = ( /
     1 'ru98 ','ru99 ','ru100','ru101','ru102','ru104','rh103','pd102',
     2 'pd104','pd105','pd106','pd108','pd110','ag107','ag109','cd106',
     3 'cd108','cd110','cd111','cd112','cd113','cd114','cd116','in113',
     4 'in115','sn112','sn114','sn115','sn116','sn117','sn118','sn119',
     5 'sn120','sn122','sn124','sb121','sb123','te120','te122','te123',
     6 'te124','te125','te126','te128','te130','i127 ','xe124','xe126',
     7 'xe128','xe129','xe130','xe131','xe132','xe134','xe136','cs133',
     8 'ba130','ba132','ba134','ba135','ba136','ba137','ba138','la138',
     9 'la139','ce136','ce138','ce140','ce142','pr141','nd142','nd143',
     & 'nd144','nd145','nd146','nd148','nd150','sm144','sm147','sm148',
     1 'sm149','sm150','sm152','sm154','eu151','eu153','gd152','gd154',
     2 'gd155','gd156','gd157','gd158','gd160','tb159','dy156','dy158',
     3 'dy160','dy161','dy162','dy163','dy164','ho165','er162','er164',
     4 'er166','er167','er168','er170','tm169','yb168','yb170','yb171',
     5 'yb172','yb173','yb174','yb176','lu175','lu176','hf174','hf176'/ )

      	namsol(241:286) = ( /
     1 'hf177','hf178','hf179','hf180','ta180','ta181','w180 ','w182 ',
     2 'w183 ','w184 ','w186 ','re185','re187','os184','os186','os187',
     3 'os188','os189','os190','os192','ir191','ir193','pt190','pt192',
     4 'pt194','pt195','pt196','pt198','au197','hg196','hg198','hg199',
     5 'hg200','hg201','hg202','hg204','tl203','tl205','pb204','pb206',
     6 'pb207','pb208','bi209','th232','u235 ','u238 '/ )


!..anders & grevesse 1989 solar mass fractions
        	solx(1:45) = ( /
     1     7.0573D-01, 4.8010D-05, 2.9291D-05, 2.7521D-01, 6.4957D-10,
     2     9.3490D-09, 1.6619D-10, 1.0674D-09, 4.7301D-09, 3.0324D-03,
     3     3.6501D-05, 1.1049D-03, 4.3634D-06, 9.5918D-03, 3.8873D-06,
     4     2.1673D-05, 4.0515D-07, 1.6189D-03, 4.1274D-06, 1.3022D-04,
     5     3.3394D-05, 5.1480D-04, 6.7664D-05, 7.7605D-05, 5.8052D-05,
     6     6.5301D-04, 3.4257D-05, 2.3524D-05, 8.1551D-06, 3.9581D-04,
     7     3.2221D-06, 1.8663D-05, 9.3793D-08, 2.5320D-06, 8.5449D-07,
     8     7.7402D-05, 1.5379D-05, 2.6307D-08, 3.4725D-06, 4.4519D-10,
     9     2.6342D-07, 5.9898D-05, 4.1964D-07, 8.9734D-07, 1.4135D-06/ )

        	solx(46:90)  = ( /
     1       2.7926D-09, 1.3841D-07, 3.8929D-08, 2.2340D-07, 2.0805D-07,
     2       2.1491D-06, 1.6361D-07, 1.6442D-07, 9.2579D-10, 3.7669D-07,
     3       7.4240D-07, 1.4863D-05, 1.7160D-06, 4.3573D-07, 1.3286D-05,
     4       7.1301D-05, 1.1686D-03, 2.8548D-05, 3.6971D-06, 3.3579D-06,
     5       4.9441D-05, 1.9578D-05, 8.5944D-07, 2.7759D-06, 7.2687D-07,
     6       5.7528D-07, 2.6471D-07, 9.9237D-07, 5.8765D-07, 8.7619D-08,
     7       4.0593D-07, 1.3811D-08, 3.9619D-08, 2.7119D-08, 4.3204D-08,
     8       5.9372D-08, 1.7136D-08, 8.1237D-08, 1.7840D-08, 1.2445D-08,
     9       1.0295D-09, 1.0766D-08, 9.1542D-09, 2.9003D-08, 6.2529D-08/ )

        	solx(91:135)  = ( /
     1       1.1823D-08, 1.1950D-08, 1.2006D-08, 3.0187D-10, 2.0216D-09,
     2       1.0682D-08, 1.0833D-08, 5.4607D-08, 1.7055D-08, 1.1008D-08,
     3       4.3353D-09, 2.8047D-10, 5.0468D-09, 3.6091D-09, 4.3183D-08,
     4       1.0446D-08, 1.3363D-08, 2.9463D-09, 4.5612D-09, 4.7079D-09,
     5       7.7706D-10, 1.6420D-09, 8.7966D-10, 5.6114D-10, 9.7562D-10,
     6       1.0320D-09, 5.9868D-10, 1.5245D-09, 6.2225D-10, 2.5012D-10,
     7       8.6761D-11, 5.9099D-10, 5.9190D-10, 8.0731D-10, 1.5171D-09,
     8       9.1547D-10, 8.9625D-10, 3.6637D-11, 4.0775D-10, 8.2335D-10,
     9       1.0189D-09, 1.0053D-09, 4.5354D-10, 6.8205D-10, 6.4517D-10/ )

        	solx(136:180)  = ( /
     1       5.3893D-11, 3.9065D-11, 5.5927D-10, 5.7839D-10, 1.0992D-09,
     2       5.6309D-10, 1.3351D-09, 3.5504D-10, 2.2581D-11, 5.1197D-10,
     3       1.0539D-10, 7.1802D-11, 3.9852D-11, 1.6285D-09, 8.6713D-10,
     4       2.7609D-09, 9.8731D-10, 3.7639D-09, 5.4622D-10, 6.9318D-10,
     5       5.4174D-10, 4.1069D-10, 1.3052D-11, 3.8266D-10, 1.3316D-10,
     6       7.1827D-10, 1.0814D-09, 3.1553D-09, 4.9538D-09, 5.3600D-09,
     7       2.8912D-09, 1.7910D-11, 1.6223D-11, 3.3349D-10, 4.1767D-09,
     8       6.7411D-10, 3.3799D-09, 4.1403D-09, 1.5558D-09, 1.2832D-09,
     9       1.2515D-09, 1.5652D-11, 1.5125D-11, 3.6946D-10, 1.0108D-09/ )

        	solx(181:225)  = ( /
     1       1.2144D-09, 1.7466D-09, 1.1240D-08, 1.3858D-12, 1.5681D-09,
     2       7.4306D-12, 9.9136D-12, 3.5767D-09, 4.5258D-10, 5.9562D-10,
     3       8.0817D-10, 3.6533D-10, 7.1757D-10, 2.5198D-10, 5.2441D-10,
     4       1.7857D-10, 1.7719D-10, 2.9140D-11, 1.4390D-10, 1.0931D-10,
     5       1.3417D-10, 7.2470D-11, 2.6491D-10, 2.2827D-10, 1.7761D-10,
     6       1.9660D-10, 2.5376D-12, 2.8008D-11, 1.9133D-10, 2.6675D-10,
     7       2.0492D-10, 3.2772D-10, 2.9180D-10, 2.8274D-10, 8.6812D-13,
     8       1.4787D-12, 3.7315D-11, 3.0340D-10, 4.1387D-10, 4.0489D-10,
     9       4.6047D-10, 3.7104D-10, 1.4342D-12, 1.6759D-11, 3.5397D-10/ )

        	solx(226:270)  = ( /
     1       2.4332D-10, 2.8557D-10, 1.6082D-10, 1.6159D-10, 1.3599D-12,
     2       3.2509D-11, 1.5312D-10, 2.3624D-10, 1.7504D-10, 3.4682D-10,
     3       1.4023D-10, 1.5803D-10, 4.2293D-12, 1.0783D-12, 3.4992D-11,
     4       1.2581D-10, 1.8550D-10, 9.3272D-11, 2.4131D-10, 1.1292D-14,
     5       9.4772D-11, 7.8768D-13, 1.6113D-10, 8.7950D-11, 1.8989D-10,
     6       1.7878D-10, 9.0315D-11, 1.5326D-10, 5.6782D-13, 5.0342D-11,
     7       5.1086D-11, 4.2704D-10, 5.2110D-10, 8.5547D-10, 1.3453D-09,
     8       1.1933D-09, 2.0211D-09, 8.1702D-13, 5.0994D-11, 2.1641D-09,
     9       2.2344D-09, 1.6757D-09, 4.8231D-10, 9.3184D-10, 2.3797D-12/ )

        	solx(271:286)  = ( /
     1       1.7079D-10, 2.8843D-10, 3.9764D-10, 2.2828D-10, 5.1607D-10,
     2       1.2023D-10, 2.7882D-10, 6.7411D-10, 3.1529D-10, 3.1369D-09,
     3       3.4034D-09, 9.6809D-09, 7.6127D-10, 1.9659D-10, 3.8519D-13,
     4       5.3760D-11/ )
                                                          

!..charge of the stable isotopes

        	izsol(1:117)  = ( /
     1   1,   1,   2,   2,   3,   3,   4,   5,   5,   6,   6,   7,   7,
     2   8,   8,   8,   9,  10,  10,  10,  11,  12,  12,  12,  13,  14,
     3  14,  14,  15,  16,  16,  16,  16,  17,  17,  18,  18,  18,  19,
     4  19,  19,  20,  20,  20,  20,  20,  20,  21,  22,  22,  22,  22,
     5  22,  23,  23,  24,  24,  24,  24,  25,  26,  26,  26,  26,  27,
     6  28,  28,  28,  28,  28,  29,  29,  30,  30,  30,  30,  30,  31,
     7  31,  32,  32,  32,  32,  32,  33,  34,  34,  34,  34,  34,  34,
     8  35,  35,  36,  36,  36,  36,  36,  36,  37,  37,  38,  38,  38,
     9  38,  39,  40,  40,  40,  40,  40,  41,  42,  42,  42,  42,  42/ )
 
        	izsol(118:234)  = ( /
     1  42,  42,  44,  44,  44,  44,  44,  44,  44,  45,  46,  46,  46,
     2  46,  46,  46,  47,  47,  48,  48,  48,  48,  48,  48,  48,  48,
     3  49,  49,  50,  50,  50,  50,  50,  50,  50,  50,  50,  50,  51,
     4  51,  52,  52,  52,  52,  52,  52,  52,  52,  53,  54,  54,  54,
     5  54,  54,  54,  54,  54,  54,  55,  56,  56,  56,  56,  56,  56,
     6  56,  57,  57,  58,  58,  58,  58,  59,  60,  60,  60,  60,  60,
     7  60,  60,  62,  62,  62,  62,  62,  62,  62,  63,  63,  64,  64,
     8  64,  64,  64,  64,  64,  65,  66,  66,  66,  66,  66,  66,  66,
     9  67,  68,  68,  68,  68,  68,  68,  69,  70,  70,  70,  70,  70/ )

        	izsol(235:286)  = ( /
     1  70,  70,  71,  71,  72,  72,  72,  72,  72,  72,  73,  73,  74,
     2  74,  74,  74,  74,  75,  75,  76,  76,  76,  76,  76,  76,  76,
     3  77,  77,  78,  78,  78,  78,  78,  78,  79,  80,  80,  80,  80,
     4  80,  80,  80,  81,  81,  82,  82,  82,  82,  83,  90,  92,  92/ )


!..number of nucleons (protons and neutrons) in the stable isotopes

        	iasol(1:117)  = ( /
     1   1,   2,   3,   4,   6,   7,   9,  10,  11,  12,  13,  14,  15,
     2  16,  17,  18,  19,  20,  21,  22,  23,  24,  25,  26,  27,  28,
     3  29,  30,  31,  32,  33,  34,  36,  35,  37,  36,  38,  40,  39,
     4  40,  41,  40,  42,  43,  44,  46,  48,  45,  46,  47,  48,  49,
     5  50,  50,  51,  50,  52,  53,  54,  55,  54,  56,  57,  58,  59,
     6  58,  60,  61,  62,  64,  63,  65,  64,  66,  67,  68,  70,  69,
     7  71,  70,  72,  73,  74,  76,  75,  74,  76,  77,  78,  80,  82,
     8  79,  81,  78,  80,  82,  83,  84,  86,  85,  87,  84,  86,  87,
     9  88,  89,  90,  91,  92,  94,  96,  93,  92,  94,  95,  96,  97/ )

        	iasol(118:234)  = ( /
     1  98, 100,  96,  98,  99, 100, 101, 102, 104, 103, 102, 104, 105,
     2 106, 108, 110, 107, 109, 106, 108, 110, 111, 112, 113, 114, 116,
     3 113, 115, 112, 114, 115, 116, 117, 118, 119, 120, 122, 124, 121,
     4 123, 120, 122, 123, 124, 125, 126, 128, 130, 127, 124, 126, 128,
     5 129, 130, 131, 132, 134, 136, 133, 130, 132, 134, 135, 136, 137,
     6 138, 138, 139, 136, 138, 140, 142, 141, 142, 143, 144, 145, 146,
     7 148, 150, 144, 147, 148, 149, 150, 152, 154, 151, 153, 152, 154,
     8 155, 156, 157, 158, 160, 159, 156, 158, 160, 161, 162, 163, 164,
     9 165, 162, 164, 166, 167, 168, 170, 169, 168, 170, 171, 172, 173/ )

        	iasol(235:286)  = ( /
     1 174, 176, 175, 176, 174, 176, 177, 178, 179, 180, 180, 181, 180,
     2 182, 183, 184, 186, 185, 187, 184, 186, 187, 188, 189, 190, 192,
     3 191, 193, 190, 192, 194, 195, 196, 198, 197, 196, 198, 199, 200,
     4 201, 202, 204, 203, 205, 204, 206, 207, 208, 209, 232, 235, 238/ )
		
	! get sum and stuff residual into hydrogen
        sum = 0.0d0
        do j=1,solsiz
           sum = sum + solx(j)
        enddo
        sum = 1.0d0 - sum
        solx(1) = solx(1) + sum
        
        sum  = 0.0d0
        do j=1,solsiz
           if (izsol(j) .ge. 3) then
              sum  = sum + solx(j)
           endif
        enddo
        zsol = sum
        
        sum = 0.0d0
        do j=1,solsiz
           sum = sum + float(izsol(j))/float(iasol(j))*solx(j)
        enddo
        yesol = sum
        
        nullify(Xsol_names_dict)
        ierr = 0
        do i=1,solsiz
           call integer_dict_define(Xsol_names_dict, namsol(i), i, ierr)
           if (ierr /= 0) then
              write(*,*) 'FATAL ERROR: chem_init, init_ag_data failed in integer_dict_define'
              stop 1
           end if
        end do
        call integer_dict_define(Xsol_names_dict, 'p', 1, ierr)
        if (ierr /= 0) then
           write(*,*) 'FATAL ERROR: chem_init, init_ag_data failed in integer_dict_define'
           stop 1
        end if
        call integer_dict_define(Xsol_names_dict, 'd', 2, ierr)
        if (ierr /= 0) then
           write(*,*) 'FATAL ERROR: chem_init, init_ag_data failed in integer_dict_define'
           stop 1
        end if
        call integer_dict_create_hash(Xsol_names_dict, ierr)
        if (ierr /= 0) then
           write(*,*) 'FATAL ERROR: chem_init, init_ag_data failed in integer_dict_create_hash'
           stop 1
        end if
         
        end subroutine init_ag_data
            
      
      subroutine init_chem_element_names
         integer :: i
         chem_element_name(:) =''

         chem_element_name(1:num_chem_elements) = el_name(1:num_chem_elements)

         do i=1,num_chem_elements
            if (len_trim(chem_element_name(i)) == 0) then
               write(*,*)'missing chem_element_name(i)', i
               stop 1 
            end if
         end do

      end subroutine init_chem_element_names

      
      
      subroutine init_chem_element_main_iso_names
         ! the iso with the largest number abundance according to Lodders03
         integer :: i
         chem_element_main_iso_name(:) = ''

         !periodic table, row 1
         chem_element_main_iso_name(e_h) = 'h1'
         chem_element_main_iso_name(e_he) = 'he4'

         !periodic table, row 2
         chem_element_main_iso_name(e_li) = 'li7'
         chem_element_main_iso_name(e_be) = 'be9'
         chem_element_main_iso_name(e_b) = 'b11'
         chem_element_main_iso_name(e_c) = 'c12'
         chem_element_main_iso_name(e_n) = 'n14'
         chem_element_main_iso_name(e_o) = 'o16'
         chem_element_main_iso_name(e_f) = 'f19'
         chem_element_main_iso_name(e_ne) = 'ne20'
         
         !periodic table, row 3
         chem_element_main_iso_name(e_na) = 'na23'
         chem_element_main_iso_name(e_mg) = 'mg24'
         chem_element_main_iso_name(e_al) = 'al27'
         chem_element_main_iso_name(e_si) = 'si28'
         chem_element_main_iso_name(e_p) = 'p31'
         chem_element_main_iso_name(e_s) = 's32'
         chem_element_main_iso_name(e_cl) = 'cl35'
         chem_element_main_iso_name(e_ar) = 'ar36'

         !periodic table, row 4
         chem_element_main_iso_name(e_k) = 'k39'
         chem_element_main_iso_name(e_ca) = 'ca40'
         chem_element_main_iso_name(e_sc) = 'sc45'
         chem_element_main_iso_name(e_ti) = 'ti48'
         chem_element_main_iso_name(e_v) = 'v51'
         chem_element_main_iso_name(e_cr) = 'cr52'
         chem_element_main_iso_name(e_mn) = 'mn55'
         chem_element_main_iso_name(e_fe) = 'fe56'
         chem_element_main_iso_name(e_co) = 'co59'
         chem_element_main_iso_name(e_ni) = 'ni58'
         chem_element_main_iso_name(e_cu) = 'cu63'
         chem_element_main_iso_name(e_zn) = 'zn64'
         chem_element_main_iso_name(e_ga) = 'ga69'
         chem_element_main_iso_name(e_ge) = 'ge74'
         chem_element_main_iso_name(e_as) = 'as75'
         chem_element_main_iso_name(e_se) = 'se80'
         chem_element_main_iso_name(e_br) = 'br79'
         chem_element_main_iso_name(e_kr) = 'kr84'

         !periodic table, row 5
         chem_element_main_iso_name(e_rb) = 'rb85'
         chem_element_main_iso_name(e_sr) = 'sr88'
         chem_element_main_iso_name(e_y) = 'y89'
         chem_element_main_iso_name(e_zr) = 'zr90'
         chem_element_main_iso_name(e_nb) = 'nb93'
         chem_element_main_iso_name(e_mo) = 'mo98'
         chem_element_main_iso_name(e_tc) = 'tc97'
         chem_element_main_iso_name(e_ru) = 'ru102'
         chem_element_main_iso_name(e_rh) = 'rh103'
         chem_element_main_iso_name(e_pd) = 'pd106'
         chem_element_main_iso_name(e_ag) = 'ag107'
         chem_element_main_iso_name(e_cd) = 'cd114'
         chem_element_main_iso_name(e_in) = 'in115'
         chem_element_main_iso_name(e_sn) = 'sn120'
         chem_element_main_iso_name(e_sb) = 'sb121'
         chem_element_main_iso_name(e_te) = 'te130'
         chem_element_main_iso_name(e_i ) = 'i127'
         chem_element_main_iso_name(e_xe) = 'xe132'

         !periodic table, row 6
         chem_element_main_iso_name(e_cs) = 'cs133'
         chem_element_main_iso_name(e_ba) = 'ba138'
         chem_element_main_iso_name(e_la) = 'la139'
         chem_element_main_iso_name(e_ce) = 'ce140'
         chem_element_main_iso_name(e_pr) = 'pr141'
         chem_element_main_iso_name(e_nd) = 'nd142'
         chem_element_main_iso_name(e_pm) = 'pm145'
         chem_element_main_iso_name(e_sm) = 'sm152'
         chem_element_main_iso_name(e_eu) = 'eu153'
         chem_element_main_iso_name(e_gd) = 'gd158'
         chem_element_main_iso_name(e_tb) = 'tb159'
         chem_element_main_iso_name(e_dy) = 'dy164'
         chem_element_main_iso_name(e_ho) = 'ho165'
         chem_element_main_iso_name(e_er) = 'er166'
         chem_element_main_iso_name(e_tm) = 'tm169'
         chem_element_main_iso_name(e_yb) = 'yb174'
         chem_element_main_iso_name(e_lu) = 'lu175'
         chem_element_main_iso_name(e_hf) = 'hf180'
         chem_element_main_iso_name(e_ta) = 'ta181'
         chem_element_main_iso_name(e_w ) = 'w184'
         chem_element_main_iso_name(e_re) = 're187'
         chem_element_main_iso_name(e_os) = 'os192'
         chem_element_main_iso_name(e_ir) = 'ir193'
         chem_element_main_iso_name(e_pt) = 'pt195'
         chem_element_main_iso_name(e_au) = 'au197'
         chem_element_main_iso_name(e_hg) = 'hg202'
         chem_element_main_iso_name(e_tl) = 'tl205'
         chem_element_main_iso_name(e_pb) = 'pb208'
         chem_element_main_iso_name(e_bi) = 'bi209'
         chem_element_main_iso_name(e_po) = 'po210'
         chem_element_main_iso_name(e_at) = 'at210'
         ! need a rule here to continue for the unstable nuclei -- longest half-life?
         chem_element_main_iso_name(e_rn) = 'rn222'
         chem_element_main_iso_name(e_fr) = 'fr223'
         chem_element_main_iso_name(e_ra) = 'ra226'
         chem_element_main_iso_name(e_ac) = 'ac227'
         chem_element_main_iso_name(e_th) = 'th232'
         chem_element_main_iso_name(e_pa) = 'pa231'
         chem_element_main_iso_name(e_u)  = 'u238'
         chem_element_main_iso_name(e_np) = 'np237'
         chem_element_main_iso_name(e_pu) = 'pu244'
         chem_element_main_iso_name(e_am) = 'am243'
         chem_element_main_iso_name(e_cm) = 'cm247'
         chem_element_main_iso_name(e_bk) = 'bk247'
         chem_element_main_iso_name(e_cf) = 'cf251'
         chem_element_main_iso_name(e_es) = 'es252'
         chem_element_main_iso_name(e_fm) = 'fm257'
         chem_element_main_iso_name(e_md) = 'md258'
         chem_element_main_iso_name(e_no) = 'no259'
         chem_element_main_iso_name(e_lr) = 'lr262'
         chem_element_main_iso_name(e_rf) = 'rf261'
         chem_element_main_iso_name(e_db) = 'db268'
         chem_element_main_iso_name(e_sg) = 'sg271'
         chem_element_main_iso_name(e_bh) = 'bh274'
         chem_element_main_iso_name(e_hs) = 'hs270'
         chem_element_main_iso_name(e_mt) = 'mt278'
         chem_element_main_iso_name(e_ds) = 'ds281'
         chem_element_main_iso_name(e_rg) = 'rg281'
         chem_element_main_iso_name(e_cn) = 'cn285'


         do i=1,num_chem_elements
            if (len_trim(chem_element_main_iso_name(i)) == 0) then
               write(*,*)'missing chem_element_main_iso_name', i
               stop 1 
            end if
         end do
         
      end subroutine init_chem_element_main_iso_names
      
      
      subroutine init_element_atomic_weights
         ! de Laeter et al, Pure and Applied Chemistry 75(6), 683–799, 2003.
         element_atomic_weight(:) = 0

         !periodic table, row 1
         element_atomic_weight(e_h ) = 1.00794
         element_atomic_weight(e_he) = 4.002602

         !periodic table, row 2
         element_atomic_weight(e_li) = 6.941
         element_atomic_weight(e_be) = 9.012
         element_atomic_weight(e_b)  = 10.811
         element_atomic_weight(e_c)  = 12.0107
         element_atomic_weight(e_n)  = 14.0067
         element_atomic_weight(e_o)  = 15.9994
         element_atomic_weight(e_f)  = 18.9984032
         element_atomic_weight(e_ne) = 20.1797

         !periodic table, row 3
         element_atomic_weight(e_na) = 22.989770
         element_atomic_weight(e_mg) = 24.3050
         element_atomic_weight(e_al) = 26.981538
         element_atomic_weight(e_si) = 28.0855
         element_atomic_weight(e_p)  = 30.973761
         element_atomic_weight(e_s)  = 32.065
         element_atomic_weight(e_cl) = 35.453
         element_atomic_weight(e_ar) = 39.948

         !periodic table, row 4
         element_atomic_weight(e_k)  = 39.0983
         element_atomic_weight(e_ca) = 40.078
         element_atomic_weight(e_sc) = 44.955910
         element_atomic_weight(e_ti) = 47.867
         element_atomic_weight(e_v)  = 50.9415
         element_atomic_weight(e_cr) = 51.9961
         element_atomic_weight(e_mn) = 54.938049
         element_atomic_weight(e_fe) = 55.845
         element_atomic_weight(e_co) = 58.933200
         element_atomic_weight(e_ni) = 58.6934
         element_atomic_weight(e_cu) = 63.546
         element_atomic_weight(e_zn) = 65.409
         element_atomic_weight(e_ga) = 69.723
         element_atomic_weight(e_ge) = 72.64
         element_atomic_weight(e_as) = 74.921
         element_atomic_weight(e_se) = 78.96
         element_atomic_weight(e_br) = 79.904
         element_atomic_weight(e_kr) = 83.798

         !periodic table, row 5
         element_atomic_weight(e_rb) = 85.4678
         element_atomic_weight(e_sr) = 87.62
         element_atomic_weight(e_y) =  88.905
         element_atomic_weight(e_zr) = 91.224
         element_atomic_weight(e_nb) = 92.906
         element_atomic_weight(e_mo) = 95.94
         element_atomic_weight(e_tc) = 97.9072
         element_atomic_weight(e_ru) = 101.07
         element_atomic_weight(e_rh) = 102.905
         element_atomic_weight(e_pd) = 106.42
         element_atomic_weight(e_ag) = 107.8682
         element_atomic_weight(e_cd) = 112.411
         element_atomic_weight(e_in) = 114.818
         element_atomic_weight(e_sn) = 118.710
         element_atomic_weight(e_sb) = 121.760
         element_atomic_weight(e_te) = 127.60
         element_atomic_weight(e_i ) = 126.904
         element_atomic_weight(e_xe) = 131.293

         !periodic table, row 6
         element_atomic_weight(e_cs) = 132.905
         element_atomic_weight(e_ba) = 137.327
         element_atomic_weight(e_la) = 138.9055
         element_atomic_weight(e_ce) = 140.115
         element_atomic_weight(e_pr) = 140.90765
         element_atomic_weight(e_nd) = 144.24
         element_atomic_weight(e_pm) = 144.9127
         element_atomic_weight(e_sm) = 150.36
         element_atomic_weight(e_eu) = 151.965
         element_atomic_weight(e_gd) = 157.25
         element_atomic_weight(e_tb) = 158.92534
         element_atomic_weight(e_dy) = 162.50
         element_atomic_weight(e_ho) = 164.93032
         element_atomic_weight(e_er) = 167.26
         element_atomic_weight(e_tm) = 168.93421
         element_atomic_weight(e_yb) = 173.04
         element_atomic_weight(e_lu) = 174.967
         element_atomic_weight(e_hf) = 178.49
         element_atomic_weight(e_ta) = 180.9479
         element_atomic_weight(e_w ) = 183.84
         element_atomic_weight(e_re) = 186.207
         element_atomic_weight(e_os) = 190.23
         element_atomic_weight(e_ir) = 192.22
         element_atomic_weight(e_pt) = 195.08
         element_atomic_weight(e_au) = 196.96654
         element_atomic_weight(e_hg) = 200.59
         element_atomic_weight(e_tl) = 204.3833
         element_atomic_weight(e_pb) = 207.2
         element_atomic_weight(e_bi) = 208.98037
         element_atomic_weight(e_po) = 208.9824	! where did this come from?
         element_atomic_weight(e_at) = 209.9871	! where did this come from?
         element_atomic_weight(e_th) = 232.0381
         element_atomic_weight(e_pa) = 231.03588
         element_atomic_weight(e_u)  = 238.02891
      end subroutine init_element_atomic_weights
      
      
      subroutine init_AG89_data ! fraction by mass of total Z
         ! anders & grevesse 1989, paper not available on ADS
         AG89_element_zfrac(:) = 0
         
         AG89_element_zfrac(e_li) =     5.2663415161043120D-07
         AG89_element_zfrac(e_be) =     8.7533846996258026D-09
         AG89_element_zfrac(e_b)  =     3.0535981584981396D-07
         AG89_element_zfrac(e_c)  =     1.6164192224602159D-01
         AG89_element_zfrac(e_n)  =     5.8425953868553440D-02
         AG89_element_zfrac(e_o)  =     5.0655547566525427D-01
         AG89_element_zfrac(e_f)  =     2.1339634220190109D-05
         AG89_element_zfrac(e_ne) =     9.2345201069972446D-02
         AG89_element_zfrac(e_na) =     1.7588936076737712D-03
         AG89_element_zfrac(e_mg) =     3.4766459385626718D-02
         AG89_element_zfrac(e_al) =     3.0576538214253392D-03
         AG89_element_zfrac(e_si) =     3.7438035164761560D-02
         AG89_element_zfrac(e_p)  =     4.2953684074804963D-04
         AG89_element_zfrac(e_s)  =     2.2000396518735602D-02
         AG89_element_zfrac(e_cl) =     1.7836963566662123D-04
         AG89_element_zfrac(e_ar) =     4.8868631434862610D-03
         AG89_element_zfrac(e_k)  =     1.8289986382724957D-04
         AG89_element_zfrac(e_ca) =     3.2987013574392004D-03
         AG89_element_zfrac(e_sc) =     2.0504272999081344D-06
         AG89_element_zfrac(e_ti) =     1.5319766333496369D-04
         AG89_element_zfrac(e_v)  =     1.9889381301661414D-05
         AG89_element_zfrac(e_cr) =     9.3528485499287748D-04
         AG89_element_zfrac(e_mn) =     6.9978620325668458D-04
         AG89_element_zfrac(e_fe) =     6.7005139944813982D-02
         AG89_element_zfrac(e_co) =     1.7686377328884702D-04
         AG89_element_zfrac(e_ni) =     3.8650578305194534D-03
         AG89_element_zfrac(e_cu) =     4.4243068859971588D-05
         AG89_element_zfrac(e_zn) =     1.0994428157112287D-04
         
         if ( abs(sum(AG89_element_zfrac)-1) > 1d-6) then
            write(*,*) 'sum(AG89_element_zfrac)', sum(AG89_element_zfrac)
            stop 1
         end if
         
         AG89_element_zfrac(:) = AG89_element_zfrac(:) / sum(AG89_element_zfrac(:))
         
      end subroutine init_AG89_data


      subroutine init_GN93_data ! fraction by mass of total Z
         ! Grevesse and Noels 1993
         integer :: i
         real(dp) :: Y_H, sum_X, Y_i  
         include 'formats.dek'         


         GN93_element_zfrac(:) = 0
         
        !GN93_element_zfrac(e_H)=12.00
        !GN93_element_zfrac(e_He)=10.99
         GN93_element_zfrac(e_Li)=3.31 !meteor
         GN93_element_zfrac(e_Be)=1.42 !meteor
         GN93_element_zfrac(e_B )=2.79  !meteor
         GN93_element_zfrac(e_C )=8.55
         GN93_element_zfrac(e_N )=7.97
         GN93_element_zfrac(e_O )=8.87
         GN93_element_zfrac(e_F )=4.48
         GN93_element_zfrac(e_Ne)=8.08
         GN93_element_zfrac(e_Na)=6.33
         GN93_element_zfrac(e_Mg)=7.58
         GN93_element_zfrac(e_Al)=6.47
         GN93_element_zfrac(e_Si)=7.55
         GN93_element_zfrac(e_P )=5.45
         GN93_element_zfrac(e_S )=7.20
         GN93_element_zfrac(e_Cl)=5.28
         GN93_element_zfrac(e_Ar)=6.52
         GN93_element_zfrac(e_K )=5.12
         GN93_element_zfrac(e_Ca)=6.36
         GN93_element_zfrac(e_Sc)=3.17
         GN93_element_zfrac(e_Ti)=5.02
         GN93_element_zfrac(e_V )=4.00
         GN93_element_zfrac(e_Cr)=5.67
         GN93_element_zfrac(e_Mn)=5.39
         GN93_element_zfrac(e_Fe)=7.50
         GN93_element_zfrac(e_Co)=4.92
         GN93_element_zfrac(e_Ni)=6.25
         GN93_element_zfrac(e_Cu)=4.21
         GN93_element_zfrac(e_Zn)=4.60
         GN93_element_zfrac(e_Ga)=3.13
         GN93_element_zfrac(e_Ge)=3.41
         GN93_element_zfrac(e_As)=2.37
         GN93_element_zfrac(e_Se)=3.38
         GN93_element_zfrac(e_Br)=2.63
         GN93_element_zfrac(e_Kr)=3.23
         GN93_element_zfrac(e_Rb)=2.41
         GN93_element_zfrac(e_Sr)=2.97
         GN93_element_zfrac(e_Y )=2.24
         GN93_element_zfrac(e_Zr)=2.60
         GN93_element_zfrac(e_Nb)=1.42
         GN93_element_zfrac(e_Mo)=1.92
         GN93_element_zfrac(e_Ru)=1.84
         GN93_element_zfrac(e_Rh)=1.12
         GN93_element_zfrac(e_Pd)=1.69
         GN93_element_zfrac(e_Ag)=1.24
         GN93_element_zfrac(e_Cd)=1.77
         GN93_element_zfrac(e_In)=0.82
         GN93_element_zfrac(e_Sn)=2.14
         GN93_element_zfrac(e_Sb)=1.03
         GN93_element_zfrac(e_Te)=2.24
         GN93_element_zfrac(e_I )=1.51
         GN93_element_zfrac(e_Xe)=2.23
         GN93_element_zfrac(e_Cs)=1.13
         GN93_element_zfrac(e_Ba)=2.13
         GN93_element_zfrac(e_La)=1.17
         GN93_element_zfrac(e_Ce)=1.58
         GN93_element_zfrac(e_Pr)=0.71
         GN93_element_zfrac(e_Nd)=1.50
         GN93_element_zfrac(e_Sm)=1.01
         GN93_element_zfrac(e_Eu)=0.51
         GN93_element_zfrac(e_Gd)=1.12
         GN93_element_zfrac(e_Tb)=0.35
         GN93_element_zfrac(e_Dy)=1.14
         GN93_element_zfrac(e_Ho)=0.51
         GN93_element_zfrac(e_Er)=0.93
         GN93_element_zfrac(e_Tm)=0.15
         GN93_element_zfrac(e_Yb)=1.08
         GN93_element_zfrac(e_Lu)=0.13
         GN93_element_zfrac(e_Hf)=0.88
         GN93_element_zfrac(e_Ta)=-0.13
         GN93_element_zfrac(e_W )=0.69
         GN93_element_zfrac(e_Re)=0.28
         GN93_element_zfrac(e_Os)=1.45
         GN93_element_zfrac(e_Ir)=1.37
         GN93_element_zfrac(e_Pt)=1.69
         GN93_element_zfrac(e_Au)=0.87
         GN93_element_zfrac(e_Hg)=1.17
         GN93_element_zfrac(e_Tl)=0.83
         GN93_element_zfrac(e_Pb)=2.06
         GN93_element_zfrac(e_Bi)=0.71
         GN93_element_zfrac(e_Th)=0.09
         GN93_element_zfrac(e_U)=-0.50

         ! convert to mass fractions
         Y_H = 1.0 - (GN93_zsol + GN93_ysol)/element_atomic_weight(e_h)
         do i = e_li, e_u
            Y_i = Y_H*10**(GN93_element_zfrac(i) - 12d0)
            GN93_element_zfrac(i) = Y_i*element_atomic_weight(i)
         end do

         if (report_solar .or. abs(sum(GN93_element_zfrac)-GN93_zsol) > 1d-3 ) then
            write(*,1)'sum(GN93_element_zfrac)', sum(GN93_element_zfrac)
            write(*,1)'GN93_zsol', GN93_zsol
            write(*,1)'sum - GN93_zsol', sum(GN93_element_zfrac)-GN93_zsol
            write(*,*)
            write(*,1) 'X_C/Z', GN93_element_zfrac(e_c)/GN93_zsol
            write(*,1) 'X_N/Z', GN93_element_zfrac(e_n)/GN93_zsol
            write(*,1) 'X_O/Z', GN93_element_zfrac(e_o)/GN93_zsol
            write(*,1) 'X_Ne/Z', GN93_element_zfrac(e_ne)/GN93_zsol
            write(*,*)
            !stop 1
         end if
         
         GN93_element_zfrac = GN93_element_zfrac / sum(GN93_element_zfrac(:))

      end subroutine init_GN93_data
      
      
      subroutine init_GS98_data ! fraction by mass of total Z
         ! Grevesse and Sauval 1998, Table 1; photospheric unless otherwise stated
         integer :: i
         real(dp) :: Y_H, sum_X, Y_i  
         GS98_element_zfrac(:) = 0
         include 'formats.dek'         

         !GS98_element_zfrac(e_H)=12.00
         !GS98_element_zfrac(e_He)=10.93
         GS98_element_zfrac(e_Li)=3.31 !meteor
         GS98_element_zfrac(e_Be)=1.42 !meteor
         GS98_element_zfrac(e_B)=2.79  !meteor
         GS98_element_zfrac(e_C)=8.52
         GS98_element_zfrac(e_N)=7.92
         GS98_element_zfrac(e_O)=8.83
         GS98_element_zfrac(e_F)=4.48
         GS98_element_zfrac(e_Ne)=8.08
         GS98_element_zfrac(e_Na)=6.32
         GS98_element_zfrac(e_Mg)=7.58
         GS98_element_zfrac(e_Al)=6.49
         GS98_element_zfrac(e_Si)=7.56
         GS98_element_zfrac(e_P)=5.56
         GS98_element_zfrac(e_S)=7.20
         GS98_element_zfrac(e_Cl)=5.28
         GS98_element_zfrac(e_Ar)=6.40
         GS98_element_zfrac(e_K)=5.13
         GS98_element_zfrac(e_Ca)=6.35
         GS98_element_zfrac(e_Sc)=3.10
         GS98_element_zfrac(e_Ti)=4.94
         GS98_element_zfrac(e_V)=4.02
         GS98_element_zfrac(e_Cr)=5.69
         GS98_element_zfrac(e_Mn)=5.53
         GS98_element_zfrac(e_Fe)=7.50
         GS98_element_zfrac(e_Co)=4.91
         GS98_element_zfrac(e_Ni)=6.25
         GS98_element_zfrac(e_Cu)=4.29
         GS98_element_zfrac(e_Zn)=4.67
         GS98_element_zfrac(e_Ga)=3.13
         GS98_element_zfrac(e_Ge)=3.63
         GS98_element_zfrac(e_As)=2.37
         GS98_element_zfrac(e_Se)=3.41
         GS98_element_zfrac(e_Br)=2.63
         GS98_element_zfrac(e_Kr)=3.31
         GS98_element_zfrac(e_Rb)=2.41
         GS98_element_zfrac(e_Sr)=2.92
         GS98_element_zfrac(e_Y)=2.23
         GS98_element_zfrac(e_Zr)=2.61
         GS98_element_zfrac(e_Nb)=1.40
         GS98_element_zfrac(e_Mo)=1.97
         GS98_element_zfrac(e_Ru)=1.83
         GS98_element_zfrac(e_Rh)=1.10
         GS98_element_zfrac(e_Pd)=1.70
         GS98_element_zfrac(e_Ag)=1.24
         GS98_element_zfrac(e_Cd)=1.76
         GS98_element_zfrac(e_In)=0.82
         GS98_element_zfrac(e_Sn)=2.14
         GS98_element_zfrac(e_Sb)=1.03
         GS98_element_zfrac(e_Te)=2.24
         GS98_element_zfrac(e_I)=1.51
         GS98_element_zfrac(e_Xe)=2.17
         GS98_element_zfrac(e_Cs)=1.13
         GS98_element_zfrac(e_Ba)=2.22
         GS98_element_zfrac(e_La)=1.22
         GS98_element_zfrac(e_Ce)=1.63
         GS98_element_zfrac(e_Pr)=0.80
         GS98_element_zfrac(e_Nd)=1.49
         GS98_element_zfrac(e_Sm)=0.98
         GS98_element_zfrac(e_Eu)=0.55
         GS98_element_zfrac(e_Gd)=1.09
         GS98_element_zfrac(e_Tb)=0.35
         GS98_element_zfrac(e_Dy)=1.17
         GS98_element_zfrac(e_Ho)=0.51
         GS98_element_zfrac(e_Er)=0.97
         GS98_element_zfrac(e_Tm)=0.15
         GS98_element_zfrac(e_Yb)=0.96
         GS98_element_zfrac(e_Lu)=0.13
         GS98_element_zfrac(e_Hf)=0.75
         GS98_element_zfrac(e_Ta)=-0.13
         GS98_element_zfrac(e_W)=0.69
         GS98_element_zfrac(e_Re)=0.28
         GS98_element_zfrac(e_Os)=1.39
         GS98_element_zfrac(e_Ir)=1.37
         GS98_element_zfrac(e_Pt)=1.69
         GS98_element_zfrac(e_Au)=0.85
         GS98_element_zfrac(e_Hg)=1.13
         GS98_element_zfrac(e_Tl)=0.83
         GS98_element_zfrac(e_Pb)=2.06
         GS98_element_zfrac(e_Bi)=0.71
         GS98_element_zfrac(e_Th)=0.09
         GS98_element_zfrac(e_U)=-0.50


         ! convert to mass fractions
         Y_H = 1.0 - (GS98_zsol + GS98_ysol)/element_atomic_weight(e_h)
         do i = e_li, e_u
            Y_i = Y_H*10**(GS98_element_zfrac(i) - 12d0)
            GS98_element_zfrac(i) = Y_i*element_atomic_weight(i)
         end do

         if (report_solar .or. abs(sum(GS98_element_zfrac)-GS98_zsol) > 1d-3 ) then
            write(*,1)'sum(GS98_element_zfrac)', sum(GS98_element_zfrac)
            write(*,1)'GS98_zsol', GS98_zsol
            write(*,1)'sum - GS98_zsol', sum(GS98_element_zfrac)-GS98_zsol
            write(*,*)
            write(*,1) 'X_C/Z', GS98_element_zfrac(e_c)/GS98_zsol
            write(*,1) 'X_N/Z', GS98_element_zfrac(e_n)/GS98_zsol
            write(*,1) 'X_O/Z', GS98_element_zfrac(e_o)/GS98_zsol
            write(*,1) 'X_Ne/Z', GS98_element_zfrac(e_ne)/GS98_zsol
            write(*,*)
            !stop 1
         end if

         GS98_element_zfrac(:) = GS98_element_zfrac(:) / sum(GS98_element_zfrac(:))
         
      end subroutine init_GS98_data
            
      subroutine init_L03_data ! fraction by mass of total Z
         ! Lodders 2003, ApJ, Table 1 recommended abundance
         integer :: i
         real(dp) :: Y_H, sum_X, Y_i
         include 'formats.dek'

         L03_element_zfrac(:) = 0
         
        !L03_element_zfrac(e_H)=12
        !L03_element_zfrac(e_He)=10.89
        L03_element_zfrac(e_Li)=3.28
        L03_element_zfrac(e_Be)=1.41
        L03_element_zfrac(e_B)=2.78
        L03_element_zfrac(e_C)=8.39
        L03_element_zfrac(e_N)=7.83
        L03_element_zfrac(e_O)=8.69
        L03_element_zfrac(e_F)=4.46
        L03_element_zfrac(e_Ne)=7.87
        L03_element_zfrac(e_Na)=6.30
        L03_element_zfrac(e_Mg)=7.55
        L03_element_zfrac(e_Al)=6.46
        L03_element_zfrac(e_Si)=7.54
        L03_element_zfrac(e_P)=5.46
        L03_element_zfrac(e_S)=7.19
        L03_element_zfrac(e_Cl)=5.26
        L03_element_zfrac(e_Ar)=6.55
        L03_element_zfrac(e_K)=5.11
        L03_element_zfrac(e_Ca)=6.34
        L03_element_zfrac(e_Sc)=3.07
        L03_element_zfrac(e_Ti)=4.92
        L03_element_zfrac(e_V)=4.00
        L03_element_zfrac(e_Cr)=5.65
        L03_element_zfrac(e_Mn)=5.50
        L03_element_zfrac(e_Fe)=7.47
        L03_element_zfrac(e_Co)=4.91
        L03_element_zfrac(e_Ni)=6.22
        L03_element_zfrac(e_Cu)=4.26
        L03_element_zfrac(e_Zn)=4.63
        L03_element_zfrac(e_Ga)=3.10
        L03_element_zfrac(e_Ge)=3.62
        L03_element_zfrac(e_As)=2.32
        L03_element_zfrac(e_Se)=3.36
        L03_element_zfrac(e_Br)=2.59
        L03_element_zfrac(e_Kr)=3.28
        L03_element_zfrac(e_Rb)=2.36
        L03_element_zfrac(e_Sr)=2.91
        L03_element_zfrac(e_Y)=2.20
        L03_element_zfrac(e_Zr)=2.60
        L03_element_zfrac(e_Nb)=1.42
        L03_element_zfrac(e_Mo)=1.96
        L03_element_zfrac(e_Ru)=1.82
        L03_element_zfrac(e_Rh)=1.11
        L03_element_zfrac(e_Pd)=1.70
        L03_element_zfrac(e_Ag)=1.23
        L03_element_zfrac(e_Cd)=1.74
        L03_element_zfrac(e_In)=0.80
        L03_element_zfrac(e_Sn)=2.11
        L03_element_zfrac(e_Sb)=1.06
        L03_element_zfrac(e_Te)=2.22
        L03_element_zfrac(e_I)=1.54
        L03_element_zfrac(e_Xe)=2.27
        L03_element_zfrac(e_Cs)=1.10
        L03_element_zfrac(e_Ba)=2.18
        L03_element_zfrac(e_La)=1.18
        L03_element_zfrac(e_Ce)=1.61
        L03_element_zfrac(e_Pr)=0.78
        L03_element_zfrac(e_Nd)=1.46
        L03_element_zfrac(e_Sm)=0.95
        L03_element_zfrac(e_Eu)=0.52
        L03_element_zfrac(e_Gd)=1.06
        L03_element_zfrac(e_Tb)=0.31
        L03_element_zfrac(e_Dy)=1.13
        L03_element_zfrac(e_Ho)=0.49
        L03_element_zfrac(e_Er)=0.95
        L03_element_zfrac(e_Tm)=0.11
        L03_element_zfrac(e_Yb)=0.94
        L03_element_zfrac(e_Lu)=0.09
        L03_element_zfrac(e_Hf)=0.77
        L03_element_zfrac(e_Ta)=-0.14
        L03_element_zfrac(e_W)=0.65
        L03_element_zfrac(e_Re)=0.26
        L03_element_zfrac(e_Os)=1.37
        L03_element_zfrac(e_Ir)=1.35
        L03_element_zfrac(e_Pt)=1.67
        L03_element_zfrac(e_Au)=0.83
        L03_element_zfrac(e_Hg)=1.16
        L03_element_zfrac(e_Tl)=0.81
        L03_element_zfrac(e_Pb)=2.05
        L03_element_zfrac(e_Bi)=0.68
        L03_element_zfrac(e_Th)=0.09
        L03_element_zfrac(e_U)=-0.49


         ! convert to mass fractions
         Y_H = 1.0 - (L03_zsol + L03_ysol)/element_atomic_weight(e_h)
         do i = e_li, e_u
            Y_i = Y_H*10**(L03_element_zfrac(i) - 12d0)
            L03_element_zfrac(i) = Y_i*element_atomic_weight(i)
         end do
         
         if (report_solar .or. abs(sum(L03_element_zfrac)-L03_zsol) > 1d-3 ) then
            write(*,1)'sum(L03_element_zfrac)', sum(L03_element_zfrac)
            write(*,1)'L03_zsol', L03_zsol
            write(*,1)'sum - L03_zsol', sum(L03_element_zfrac)-L03_zsol
            write(*,*)
            write(*,1) 'X_C/Z', L03_element_zfrac(e_c)/L03_zsol
            write(*,1) 'X_N/Z', L03_element_zfrac(e_n)/L03_zsol
            write(*,1) 'X_O/Z', L03_element_zfrac(e_o)/L03_zsol
            write(*,1) 'X_Ne/Z', L03_element_zfrac(e_ne)/L03_zsol
            write(*,*)
            !stop 1
         end if
         
         L03_element_zfrac(:) = L03_element_zfrac(:) / sum(L03_element_zfrac(:))
         
      end subroutine init_L03_data
      
      
      subroutine init_AGS04_data ! fraction by mass of total Z
         ! Asplund, Grevesse and Sauval 2004      
         integer :: i
         real(dp) :: Y_H, sum_X, Y_i
         include 'formats.dek'

         AGS04_element_zfrac(:) = 0
                  
         ! first store log abundances from the paper
         AGS04_element_zfrac(e_li) = 3.25
         AGS04_element_zfrac(e_be) = 1.38
         AGS04_element_zfrac(e_b ) = 2.70
         AGS04_element_zfrac(e_c ) = 8.39
         AGS04_element_zfrac(e_n ) = 7.78
         AGS04_element_zfrac(e_o ) = 8.66
         AGS04_element_zfrac(e_f ) = 4.56
         AGS04_element_zfrac(e_ne) = 7.84
         AGS04_element_zfrac(e_na) = 6.17
         AGS04_element_zfrac(e_mg) = 7.53
         AGS04_element_zfrac(e_al) = 6.37
         AGS04_element_zfrac(e_si) = 7.51
         AGS04_element_zfrac(e_p ) = 5.36
         AGS04_element_zfrac(e_s ) = 7.14
         AGS04_element_zfrac(e_cl) = 5.50
         AGS04_element_zfrac(e_ar) = 6.18
         AGS04_element_zfrac(e_k ) = 5.08
         AGS04_element_zfrac(e_ca) = 6.31
         AGS04_element_zfrac(e_sc) = 3.05
         AGS04_element_zfrac(e_ti) = 4.90
         AGS04_element_zfrac(e_v ) = 4.00
         AGS04_element_zfrac(e_cr) = 5.64
         AGS04_element_zfrac(e_mn) = 5.39
         AGS04_element_zfrac(e_fe) = 7.45
         AGS04_element_zfrac(e_co) = 4.92
         AGS04_element_zfrac(e_ni) = 6.23
         AGS04_element_zfrac(e_cu) = 4.21
         AGS04_element_zfrac(e_zn) = 4.60
         AGS04_element_zfrac(e_ga) = 2.88
         AGS04_element_zfrac(e_ge) = 3.58
         AGS04_element_zfrac(e_as) = 2.29 !meteor
         AGS04_element_zfrac(e_se) = 3.33 !meteor
         AGS04_element_zfrac(e_br) = 2.56 !meteor
         AGS04_element_zfrac(e_kr) = 3.28 !indirect
         AGS04_element_zfrac(e_rb) = 2.60
         AGS04_element_zfrac(e_sr) = 2.92
         AGS04_element_zfrac(e_y ) = 2.21
         AGS04_element_zfrac(e_zr) = 2.59
         AGS04_element_zfrac(e_nb) = 1.42
         AGS04_element_zfrac(e_mo) = 1.92
         AGS04_element_zfrac(e_Ru) = 1.84
         AGS04_element_zfrac(e_Rh) = 1.12
         AGS04_element_zfrac(e_Pd) = 1.69
         AGS04_element_zfrac(e_Ag) = 0.94
         AGS04_element_zfrac(e_Cd) = 1.77
         AGS04_element_zfrac(e_In) = 1.60
         AGS04_element_zfrac(e_Sn) = 2.00
         AGS04_element_zfrac(e_Sb) = 1.00
         AGS04_element_zfrac(e_Te) = 2.19
         AGS04_element_zfrac(e_I ) = 1.51
         AGS04_element_zfrac(e_Xe) = 2.27
         AGS04_element_zfrac(e_Cs) = 1.07
         AGS04_element_zfrac(e_Ba) = 2.17
         AGS04_element_zfrac(e_La) = 1.13
         AGS04_element_zfrac(e_Ce) = 1.58
         AGS04_element_zfrac(e_Pr) = 0.71
         AGS04_element_zfrac(e_Nd) = 1.45
         AGS04_element_zfrac(e_Sm) = 1.01
         AGS04_element_zfrac(e_Eu) = 0.52
         AGS04_element_zfrac(e_Gd) = 1.12
         AGS04_element_zfrac(e_Tb) = 0.28
         AGS04_element_zfrac(e_Dy) = 1.14
         AGS04_element_zfrac(e_Ho) = 0.51
         AGS04_element_zfrac(e_Er) = 0.93
         AGS04_element_zfrac(e_Tm) = 0.00
         AGS04_element_zfrac(e_Yb) = 1.08
         AGS04_element_zfrac(e_Lu) = 0.06
         AGS04_element_zfrac(e_Hf) = 0.88
         AGS04_element_zfrac(e_Ta) = -0.17
         AGS04_element_zfrac(e_W ) = 1.11
         AGS04_element_zfrac(e_Re) = 0.23
         AGS04_element_zfrac(e_Os) = 1.45
         AGS04_element_zfrac(e_Ir) = 1.38
         AGS04_element_zfrac(e_Pt) = 1.64
         AGS04_element_zfrac(e_Au) = 1.01
         AGS04_element_zfrac(e_Hg) = 1.13
         AGS04_element_zfrac(e_Tl) = 0.90
         AGS04_element_zfrac(e_Pb) = 2.00
         AGS04_element_zfrac(e_Bi) = 0.65
         AGS04_element_zfrac(e_Th) = 0.06
         AGS04_element_zfrac(e_U) = -0.52

         ! convert to mass fractions
         Y_H = 1.0 - (AGS04_zsol + AGS04_ysol)/element_atomic_weight(e_h)
         do i = e_li, e_u
            Y_i = Y_H*10**(AGS04_element_zfrac(i) - 12d0)
            AGS04_element_zfrac(i) = Y_i*element_atomic_weight(i)
         end do
         
         if (report_solar .or.  abs(sum(AGS04_element_zfrac)-AGS04_zsol) > 1d-3 ) then
            write(*,1)'sum(AGS04_element_zfrac)', sum(AGS04_element_zfrac)
            write(*,1)'AGS04_zsol', AGS04_zsol
            write(*,1)'sum - AGS04_zsol', sum(AGS04_element_zfrac)-AGS04_zsol
            write(*,*)
            write(*,1) 'X_C/Z', AGS04_element_zfrac(e_c)/AGS04_zsol
            write(*,1) 'X_N/Z', AGS04_element_zfrac(e_n)/AGS04_zsol
            write(*,1) 'X_O/Z', AGS04_element_zfrac(e_o)/AGS04_zsol
            write(*,1) 'X_Ne/Z', AGS04_element_zfrac(e_ne)/AGS04_zsol
            write(*,*)
            !stop 1
         end if


         AGS04_element_zfrac(:) = AGS04_element_zfrac(:) / sum(AGS04_element_zfrac(:))
         
      end subroutine init_AGS04_data
      
      
      subroutine init_AGSS09_data ! fraction by mass of total Z
         ! Asplund, Grevesse, Sauval, and Scott 2009 abundances
         ! Annu. Rev. Astron. Astrophys. 2009. 47:481–522
         integer :: i
         real(dp) :: Y_H, sum_X, Y_i
         include 'formats.dek'
         
         AGSS09_element_zfrac(:) = 0
         
         ! first store log abundances from the paper
         AGSS09_element_zfrac(e_li) = 3.26
         AGSS09_element_zfrac(e_be) = 1.38
         AGSS09_element_zfrac(e_b ) = 2.70
         AGSS09_element_zfrac(e_c ) = 8.43
         AGSS09_element_zfrac(e_n ) = 7.83
         AGSS09_element_zfrac(e_o ) = 8.69
         AGSS09_element_zfrac(e_f ) = 4.56
         AGSS09_element_zfrac(e_ne) = 7.93
         AGSS09_element_zfrac(e_na) = 6.24
         AGSS09_element_zfrac(e_mg) = 7.60
         AGSS09_element_zfrac(e_al) = 6.45
         AGSS09_element_zfrac(e_si) = 7.51
         AGSS09_element_zfrac(e_p ) = 5.41
         AGSS09_element_zfrac(e_s ) = 7.12
         AGSS09_element_zfrac(e_cl) = 5.50
         AGSS09_element_zfrac(e_ar) = 6.40
         AGSS09_element_zfrac(e_k ) = 5.03
         AGSS09_element_zfrac(e_ca) = 6.34
         AGSS09_element_zfrac(e_sc) = 3.15
         AGSS09_element_zfrac(e_ti) = 4.95
         AGSS09_element_zfrac(e_v ) = 3.93
         AGSS09_element_zfrac(e_cr) = 5.64
         AGSS09_element_zfrac(e_mn) = 5.43
         AGSS09_element_zfrac(e_fe) = 7.50
         AGSS09_element_zfrac(e_co) = 4.99
         AGSS09_element_zfrac(e_ni) = 6.22
         AGSS09_element_zfrac(e_cu) = 4.19
         AGSS09_element_zfrac(e_zn) = 4.56
         AGSS09_element_zfrac(e_ga) = 3.04
         AGSS09_element_zfrac(e_ge) = 3.65
         AGSS09_element_zfrac(e_as) = 2.30 !meteor
         AGSS09_element_zfrac(e_se) = 3.34 !meteor
         AGSS09_element_zfrac(e_br) = 2.54 !meteor
         AGSS09_element_zfrac(e_kr) = 3.25 !indirect
         AGSS09_element_zfrac(e_rb) = 2.52
         AGSS09_element_zfrac(e_sr) = 2.87
         AGSS09_element_zfrac(e_y ) = 2.21
         AGSS09_element_zfrac(e_zr) = 2.58
         AGSS09_element_zfrac(e_nb) = 1.46
         AGSS09_element_zfrac(e_mo) = 1.88
         AGSS09_element_zfrac(e_Ru) = 1.75
         AGSS09_element_zfrac(e_Rh) = 0.91
         AGSS09_element_zfrac(e_Pd) = 1.57
         AGSS09_element_zfrac(e_Ag) = 0.94
         AGSS09_element_zfrac(e_Cd) = 1.71
         AGSS09_element_zfrac(e_In) = 0.80
         AGSS09_element_zfrac(e_Sn) = 2.04
         AGSS09_element_zfrac(e_Sb) = 1.01
         AGSS09_element_zfrac(e_Te) = 2.18
         AGSS09_element_zfrac(e_I ) = 1.55
         AGSS09_element_zfrac(e_Xe) = 2.24
         AGSS09_element_zfrac(e_Cs) = 1.08
         AGSS09_element_zfrac(e_Ba) = 2.18
         AGSS09_element_zfrac(e_La) = 1.10
         AGSS09_element_zfrac(e_Ce) = 1.58
         AGSS09_element_zfrac(e_Pr) = 0.72
         AGSS09_element_zfrac(e_Nd) = 1.42
         AGSS09_element_zfrac(e_Sm) = 0.96
         AGSS09_element_zfrac(e_Eu) = 0.52
         AGSS09_element_zfrac(e_Gd) = 1.07
         AGSS09_element_zfrac(e_Tb) = 0.30
         AGSS09_element_zfrac(e_Dy) = 1.10
         AGSS09_element_zfrac(e_Ho) = 0.48
         AGSS09_element_zfrac(e_Er) = 0.92
         AGSS09_element_zfrac(e_Tm) = 0.10
         AGSS09_element_zfrac(e_Yb) = 0.84
         AGSS09_element_zfrac(e_Lu) = 0.10
         AGSS09_element_zfrac(e_Hf) = 0.85
         AGSS09_element_zfrac(e_Ta) = -0.12
         AGSS09_element_zfrac(e_W ) = 0.85
         AGSS09_element_zfrac(e_Re) = 0.26
         AGSS09_element_zfrac(e_Os) = 1.40
         AGSS09_element_zfrac(e_Ir) = 1.38
         AGSS09_element_zfrac(e_Pt) = 1.62
         AGSS09_element_zfrac(e_Au) = 0.92
         AGSS09_element_zfrac(e_Hg) = 1.17
         AGSS09_element_zfrac(e_Tl) = 0.90
         AGSS09_element_zfrac(e_Pb) = 1.75
         AGSS09_element_zfrac(e_Bi) = 0.65
         AGSS09_element_zfrac(e_Th) = 0.02
         AGSS09_element_zfrac(e_U) = -0.54
         
         ! convert to mass fractions
         Y_H = 1.0 - (AGSS09_zsol + AGSS09_ysol)/element_atomic_weight(e_h)
         do i = e_li, e_u
            Y_i = Y_H*10**(AGSS09_element_zfrac(i) - 12d0)
            AGSS09_element_zfrac(i) = Y_i*element_atomic_weight(i)
         end do
         
         if (report_solar .or. abs(sum(AGSS09_element_zfrac)-AGSS09_zsol) > 1d-3 ) then
            write(*,1)'sum(AGSS09_element_zfrac)', sum(AGSS09_element_zfrac)
            write(*,1)'AGSS09_zsol', AGSS09_zsol
            write(*,1)'sum - AGSS09_zsol', sum(AGSS09_element_zfrac)-AGSS09_zsol
            write(*,*)
            write(*,1) 'X_C/Z', AGSS09_element_zfrac(e_c)/AGSS09_zsol
            write(*,1) 'X_N/Z', AGSS09_element_zfrac(e_n)/AGSS09_zsol
            write(*,1) 'X_O/Z', AGSS09_element_zfrac(e_o)/AGSS09_zsol
            write(*,1) 'X_Ne/Z', AGSS09_element_zfrac(e_ne)/AGSS09_zsol
            write(*,*)
            !stop 1
         end if
         
         AGSS09_element_zfrac(:) = AGSS09_element_zfrac(:) / sum(AGSS09_element_zfrac(:))
         
         return
         
         do i = e_li, e_u
            if (AGSS09_element_zfrac(i) > 1d-3) 
     >            write(*,1) trim(chem_element_Name(i)), AGSS09_element_zfrac(i)
         end do
         stop 'init_AGSS09_data'
         
      end subroutine init_AGSS09_data
      
      
      subroutine init_L09_data ! fraction by mass of total Z
         ! Lodders 09
         include 'formats.dek'
         
         L09_element_zfrac(:) = 0
         
         ! mass fractions
         L09_element_zfrac(e_li) = 1.054594933683d-08
         L09_element_zfrac(e_be) = 1.5087555571d-10
         L09_element_zfrac(e_b ) = 5.5633306761d-09
         L09_element_zfrac(e_c ) = 0.002365544090848
         L09_element_zfrac(e_n ) = 0.0008161934768925
         L09_element_zfrac(e_o ) = 0.0068991682737478
         L09_element_zfrac(e_f ) = 4.1844135602D-07
         L09_element_zfrac(e_ne) = 0.0018162023028794
         L09_element_zfrac(e_na) = 3.6352024324D-05
         L09_element_zfrac(e_mg) = 0.000683514477408
         L09_element_zfrac(e_al) = 6.2568980455D-05
         L09_element_zfrac(e_si) = 0.000769722819625
         L09_element_zfrac(e_p ) = 7.0479812061D-06
         L09_element_zfrac(e_s ) = 0.000370123705833609
         L09_element_zfrac(e_cl) = 5.0250763788D-06
         L09_element_zfrac(e_ar) = 9.2220355099056D-05
         L09_element_zfrac(e_k ) = 4.0297305059079D-06
         L09_element_zfrac(e_ca) = 6.63135868398494D-05
         L09_element_zfrac(e_sc) = 4.2402933957D-08
         L09_element_zfrac(e_ti) = 3.24207135484D-06
         L09_element_zfrac(e_v ) = 4.0049954611698D-07
         L09_element_zfrac(e_cr) = 1.870484358513D-05
         L09_element_zfrac(e_mn) = 1.3890521841D-05
         L09_element_zfrac(e_fe) = 0.0012986862725768
         L09_element_zfrac(e_co) = 3.7979113651D-06
         L09_element_zfrac(e_ni) = 7.901833309378D-05
         L09_element_zfrac(e_cu) = 9.4275308656D-07
         L09_element_zfrac(e_zn) = 2.324135489336D-06
         L09_element_zfrac(e_ga) = 6.9975797859D-08
         L09_element_zfrac(e_ge) = 2.27918509226D-07
         L09_element_zfrac(e_as) = 1.2531874861D-08
         L09_element_zfrac(e_se) = 1.460613983153D-07
         L09_element_zfrac(e_br) = 2.1909826069D-07
         L09_element_zfrac(e_kr) = 1.2837104766341D-07
         L09_element_zfrac(e_rb) = 1.69493949899D-08
         L09_element_zfrac(e_sr) = 5.58119030984D-08
         L09_element_zfrac(e_y ) = 1.1287452840D-08
         L09_element_zfrac(e_zr) = 2.696974516888D-08
         L09_element_zfrac(e_nb) = 1.9870212075D-09
         L09_element_zfrac(e_mo) = 6.70675811282D-09
         L09_element_zfrac(e_Ru) = 4.935148192679D-09
         L09_element_zfrac(e_Rh) = 1.0439120240D-09
         L09_element_zfrac(e_Pd) = 3.958834334296D-09
         L09_element_zfrac(e_Ag) = 1.44909561511D-09
         L09_element_zfrac(e_Cd) = 4.850725813727D-09
         L09_element_zfrac(e_In) = 5.60277526584D-10
         L09_element_zfrac(e_Sn) = 1.1748571050167D-08
         L09_element_zfrac(e_Sb) = 1.04476117832D-09
         L09_element_zfrac(e_Te) = 1.64381492798D-08
         L09_element_zfrac(e_I ) = 3.8266730451D-09
         L09_element_zfrac(e_Xe) = 1.9631791443838D-08
         L09_element_zfrac(e_Cs) = 1.3516072159D-09
         L09_element_zfrac(e_Ba) = 1.6848783894182D-08
         L09_element_zfrac(e_La) = 1.7400268564D-09
         L09_element_zfrac(e_Ce) = 4.5166246603333D-09
         L09_element_zfrac(e_Pr) = 6.6431263199D-10
         L09_element_zfrac(e_Nd) = 5.82366496835D-09
         L09_element_zfrac(e_Sm) = 1.10017534847D-09
         L09_element_zfrac(e_Eu) = 4.1023195079D-10
         L09_element_zfrac(e_Gd) = 1.5505832574086D-09
         L09_element_zfrac(e_Tb) = 2.7612856334D-10
         L09_element_zfrac(e_Dy) = 1.79997715438722D-09
         L09_element_zfrac(e_Ho) = 4.1129202414D-10
         L09_element_zfrac(e_Er) = 1.2035968713737D-09
         L09_element_zfrac(e_Tm) = 1.8794799164D-10
         L09_element_zfrac(e_Yb) = 1.2153809425605D-09
         L09_element_zfrac(e_Lu) = 1.826667993422D-10
         L09_element_zfrac(e_Hf) = 7.619544268076D-10
         L09_element_zfrac(e_Ta) = 1.04130101121661D-10
         L09_element_zfrac(e_W ) = 6.9059258989518D-10
         L09_element_zfrac(e_Re) = 2.9647265833D-10
         L09_element_zfrac(e_Os) = 3.52828020217507D-09
         L09_element_zfrac(e_Ir) = 3.5336600059D-09
         L09_element_zfrac(e_Pt) = 6.8100262401586D-09
         L09_element_zfrac(e_Au) = 1.0522666072D-09
         L09_element_zfrac(e_Hg) = 2.5169209215851D-09
         L09_element_zfrac(e_Tl) = 1.02465539439D-09
         L09_element_zfrac(e_Pb) = 1.879940103275D-08
         L09_element_zfrac(e_Bi) = 7.9004226175D-10
         L09_element_zfrac(e_Th) = 2.7961831384D-10
         L09_element_zfrac(e_U) = 1.546830543D-10
         
         ! convert from mass fraction to fraction of Z by mass
         L09_element_zfrac(:) = L09_element_zfrac(:) / sum(L09_element_zfrac(:))
                  
      end subroutine init_L09_data


      subroutine allocate_nuclide_data(d,n,ierr)
      	type(nuclide_data), intent(out) :: d
      	integer, intent(in) :: n
      	integer, intent(out) :: ierr	
      	ierr = 0
      	allocate(d% name(n), d% W(n), d% Z(n), d% N(n), d% Z_plus_N(n), 
     >      	d% spin(N), d% binding_energy(n), 
     >      	d% isomeric_state(n), d% mass_excess(n), d% pfcn(npart,n), 
     >         d% chem_id(n), d% nuclide(num_chem_isos), stat=ierr)
      	if (ierr /= 0) return
      	d% nnuclides = n
      end subroutine allocate_nuclide_data
      

      subroutine free_nuclide_data(n)
      	type(nuclide_data), intent(inout) :: n
      	if (associated(n% name))
     >      deallocate(n% name, n% W, n% Z, n% N, n% Z_plus_N, n% spin, n% binding_energy,  
     >         n% isomeric_state, n% mass_excess, n% pfcn, n% chem_id, n% nuclide)
      	n% nnuclides = 0
      end subroutine free_nuclide_data
      
      
      subroutine free_lodders03_table()
				use utils_lib, only : integer_dict_free
				deallocate(lodders03_tab6% isotopic_percent)
				call integer_dict_free(lodders03_tab6% name_dict)
				nullify(lodders03_tab6% name_dict)
			end subroutine free_lodders03_table
			
		integer function category_id(cname)
			character (len=*), intent(in)  :: cname 
			! returns id for the category if there is a matching name
			! returns 0 otherwise.
			integer :: i, len
			character (len=maxlen_category_name) :: nam
			len = len_trim(cname)
			do i = 1, maxlen_category_name
			   if (i <= len) then
			      nam(i:i) = cname(i:i)
			   else
			      nam(i:i) = ' '
			   end if
			end do
			do i = 1, num_categories
			   if (category_name(i)==nam) then
			      category_id = i
			      return
			   end if
			end do
			category_id = 0
		end function category_id
      
      
      subroutine set_category_names
         integer :: i
         category_name(:) = ''

         category_name(ipp) = 'pp'
         category_name(icno) = 'cno'
         category_name(i3alf) = 'tri_alfa'

         category_name(i_burn_c) = 'burn_c'
         category_name(i_burn_n) = 'burn_n'
         category_name(i_burn_o) = 'burn_o'
         category_name(i_burn_ne) = 'burn_ne'
         category_name(i_burn_na) = 'burn_na'
         category_name(i_burn_mg) = 'burn_mg'
         category_name(i_burn_si) = 'burn_si'
         category_name(i_burn_s) = 'burn_s'
         category_name(i_burn_ar) = 'burn_ar'
         category_name(i_burn_ca) = 'burn_ca'
         category_name(i_burn_ti) = 'burn_ti'
         category_name(i_burn_cr) = 'burn_cr'
         category_name(i_burn_fe) = 'burn_fe'

         category_name(icc) = 'c12_c12'
         category_name(ico) = 'c12_o16'
         category_name(ioo) = 'o16_o16'

         category_name(iphoto) = 'photo'
         category_name(ipnhe4) = 'pnhe4'
         category_name(iother) = 'other'
         
         do i=1,num_categories
            if (len_trim(category_name(i)) == 0) then
               write(*,*) 'missing name for category', i
               if (i > 1) write(*,*) 'following ' // trim(category_name(i-1))
               stop 'set_category_names'
            end if
         end do
         
      end subroutine set_category_names


      end module chem_def

