! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
!
! ***********************************************************************
 
      module chem_isos_io
      use chem_def
      
      implicit none


      contains
      

      subroutine do_read_chem_isos(isotopes_filename, ierr)
      	use utils_lib
      	use const_def, only: mesa_data_dir
         character (len=*), intent(in) :: isotopes_filename
      	integer, intent(out) :: ierr
      	integer :: i, iounit, pass
      	character (len=256) :: filename
      	
      	ierr = 0
	
      	iounit = alloc_iounit(ierr)
      	if (ierr /= 0) then
      	   write(*,*) 'failed to alloc iounit'
      	   return
      	end if
      	
      	filename = trim(mesa_data_dir) // '/chem_data/' // trim(isotopes_filename)
         num_chem_isos = 0
      	
      	do pass = 1, 2
      	
         	open(unit=iounit, file=trim(filename), iostat=ierr, status='old',action='read')
         	if ( ierr /= 0 ) then
         		write(*,*) 'unable to open '// trim(filename)
         		return
         	end if
         	
         	if (pass == 1) then
         	
         	   do ! 4 lines per nuclide
            		read(iounit, *, iostat=ierr) 
            		if (ierr /= 0) exit
            		read(iounit, *, iostat=ierr) 
            		if (ierr /= 0) exit
            		read(iounit, *, iostat=ierr) 
            		if (ierr /= 0) exit
            		read(iounit, *, iostat=ierr) 
            		if (ierr /= 0) exit
            		num_chem_isos = num_chem_isos+1
         	   end do
         	   if (num_chem_isos == 0) then
							write (*,*) 'unable to retrieve isotopes from '//trim(filename)
							return
						end if
         	else
         	
            	call allocate_nuclide_data(chem_isos, num_chem_isos, ierr)
            	if (ierr /= 0) then
            	   write(*,*) 'unable to allocate nuclide data'
            	   return
            	end if
	
            	do i = 1, num_chem_isos
            		read(iounit, *, iostat=ierr) &
            		   chem_isos% name(i), chem_isos% W(i), chem_isos% Z(i), chem_isos% N(i), &
            		 	chem_isos% spin(i), chem_isos% mass_excess(i)
            		if (ierr /= 0) exit
            		read(iounit, *, iostat=ierr) chem_isos% pfcn(1:8, i)
            		if (ierr /= 0) exit
            		read(iounit, *, iostat=ierr) chem_isos% pfcn(9:16, i)
            		if (ierr /= 0) exit
            		read(iounit, *, iostat=ierr) chem_isos% pfcn(17:24, i)
            		if (ierr /= 0) exit
            		chem_isos% chem_id(i) = i
            		chem_isos% nuclide(i) = i
            		chem_isos% isomeric_state(i) = get_isomeric_state(chem_isos% name(i), ierr)
            	end do
            	if (ierr /= 0) then
								write (*,*) 'something went wrong in read of '//trim(filename)
								return
							end if

         	end if
         	
         	close(iounit)
         	
         end do
         
      	call free_iounit(iounit)

      	if (ierr /= 0) return
      	
         call do_create_nuclides_dict(chem_isos, chem_isos_dict, ierr)
      	if (ierr /= 0) return
	
         chem_isos% Z_plus_N = chem_isos% Z + chem_isos% N
         
      	chem_isos% binding_energy = chem_isos% Z*del_Mp + chem_isos% N*del_Mn - chem_isos% mass_excess
      	where (chem_isos% Z_plus_N <= 1) chem_isos% binding_energy = 0.0 ! neutrons and protons by various names
      	
      	contains
      	
      	integer function get_isomeric_state(name, ierr)
      	   character (len=*), intent(in) :: name
      	   integer, intent(out) :: ierr
      	   integer :: i, len
      	   ierr = 0
      	   get_isomeric_state = 0
      	   len = len_trim(name)
      	   do i=1,len
      	      if (name(i:i) == '-') then
      	         read(name(i+1:len),*,iostat=ierr) get_isomeric_state
      	         if (ierr /= 0 .or. get_isomeric_state < 0 .or. get_isomeric_state > 99) then
      	            write(*,*) 'ERROR: invalid name for iso ' // trim(name) // ' in ' // trim(filename)
      	            return
      	         end if
      	         return
      	      end if
      	   end do
      	end function get_isomeric_state
      	

      end subroutine do_read_chem_isos
            
      
      subroutine do_create_nuclides_dict(nuclides, nuclides_dict, ierr)
         use utils_lib, only: integer_dict_define, integer_dict_create_hash, integer_dict_lookup
      	type(nuclide_data), intent(in) :: nuclides
         type (integer_dict), pointer :: nuclides_dict ! will be allocated
      	integer, intent(out) :: ierr
         integer :: i
         
         ierr = 0
         nullify(nuclides_dict)
         do i=1,nuclides% nnuclides
            call integer_dict_define(nuclides_dict, nuclides% name(i), i, ierr)
            if (ierr /= 0) return
         end do

         call integer_dict_create_hash(nuclides_dict, ierr)
         if (ierr /= 0) return
         
      end subroutine do_create_nuclides_dict
      

      subroutine set_B(n)
      	type(nuclide_data), intent(inout) :: n
         ! mass excess of proton, neutron in MeV
      end subroutine set_B
      

      subroutine do_extract_nuclides_from_chem_isos( &
            chem_isos, nuclides_out, set, err)
      	use nuclide_set_mod, only : rank_in_set
      	type(nuclide_data), intent(in) :: chem_isos
      	type(nuclide_data), intent(out) :: nuclides_out
      	type(nuclide_set), dimension(:), intent(in) :: set
      	integer, intent(out) :: err
      	integer :: niso, i, l, count
      	character(len=80) :: message
	
      	err = 0
      	niso = size(set)
      	call allocate_nuclide_data(nuclides_out, niso, err)
      	if (err /= 0) return

      	count = 0
      	nuclides_out% nuclide(:) = 0
      	loop_over_isotopes : do i = 1, chem_isos% nnuclides
      		l = rank_in_set(chem_isos% name(i), set)
      		got_one: if (l > 0) then
      			nuclides_out% name(l) = chem_isos% name(i)
      			nuclides_out% W(l) = chem_isos% W(i)
      			nuclides_out% Z(l) = chem_isos% Z(i)
      			nuclides_out% N(l) = chem_isos% N(i)
      			nuclides_out% Z_plus_N(l) = chem_isos% Z_plus_N(i)
      			nuclides_out% spin(l) = chem_isos% spin(i)
      			nuclides_out% binding_energy(l) = chem_isos% binding_energy(i)
      			nuclides_out% pfcn(:, l) = chem_isos% pfcn(:, i)
      			nuclides_out% chem_id(l) = i
      			nuclides_out% nuclide(i) = l
      			count = count + 1
      			if (count > niso) exit
      		end if got_one
      	end do loop_over_isotopes
      	if (count /= niso) then 
      		write(*, '(a, i0, a, i0, a)') 'only', count, '/', niso, ' nuclei extracted!'
      		do i = 1, niso
      			if (adjustl(nuclides_out% name(set(i)% rank)) /= adjustl(set(i)% nuclide)) &
      			   write(*,*) 'missing ', set(i)% nuclide
      		end do
      		err = -1
      	end if
      end subroutine do_extract_nuclides_from_chem_isos
      

      subroutine write_nuclides(unitno, nuclides, err)
      	integer, intent(in) :: unitno
      	type(nuclide_data), intent(in) :: nuclides
      	integer, intent(out) :: err
      	integer :: i
      	character(len=*), parameter :: fmt = &
      	   '(a5, tr1, f8.3, tr1, i3, tr1, i4, tr1, f4.1, tr1, f8.3, tr1, 3(/, 8(f9.2, tr1)))'
      	write(unitno, *) size(nuclides% name)
      	do i = 1, size(nuclides% name)
      		write(unitno, fmt=fmt, iostat=err) nuclides% name(i), nuclides% W(i), nuclides% Z(i), nuclides% N(i), &
      			nuclides% spin(i), nuclides% binding_energy(i), nuclides% pfcn(:, i)
      	end do
      end subroutine write_nuclides
      

      end module chem_isos_io