module chem_support
	use const_def
	use const_lib
	
	real, parameter :: no_mass_table_entry = -999.0
	character(len=256) :: data_dir, output_dir, masstable_filename, winvn_filename
	integer :: masstable_header_length
	integer :: winvn_header_length
	integer :: number_nuclides
	! integer, parameter :: number_masstable_header_lines = 15
	! integer, parameter :: number_winvn_header_lines = 7857
	! character(len=*), parameter :: masstable_filename = 'masslib_library_5.data'
	real, dimension(:,:), allocatable :: mass_table
	
	namelist /chem/  &
		& data_dir, output_dir, masstable_filename, winvn_filename,  &
		&	masstable_header_length, winvn_header_length, number_nuclides
	
	contains
	
	subroutine read_input_parameters(inlist_fname)
		character(len=*), intent(in) :: inlist_fname
		integer :: iounit, ios
		
		! set default values
		data_dir = 'chem_input_data'
		output_dir = 'data/chem_data'
		masstable_filename = 'masslib_library_5.data'
		winvn_filename = 'winvne_v2.data'
		masstable_header_length = 15
		winvn_header_length = 4
		number_nuclides = 7853
		
		iounit = 50
		open(unit=iounit, file=trim(inlist_fname), iostat=ios, status="old", action="read",delim='quote')
		if ( ios /= 0 ) then
			write(*,*)
     	write(*,*)
     	write(*,*) 'Failed to open namelist file ',trim(inlist_fname)
      write(*,*)
      write(*,*)
			stop
		end if
		read(iounit,nml=chem,iostat=ios)
		if (ios /= 0) then
			write (*,*)
			write (*,*)
			write (*,*) 'Failed to read namelist file ',trim(inlist_fname)
			write (*,*)
			write (*,*)
			stop
		end if
	end subroutine read_input_parameters
	
	subroutine init_preprocessor()
		call const_init
	end subroutine init_preprocessor
	
	subroutine read_mass_table()
		integer, parameter :: mass_unit = 20
		integer :: ios, i, nlines, zmin, zmax, nmin, nmax, Z, A, N
		real(dp), parameter :: keV_to_MeV = 1.0e-3
		real(dp) :: mass
		character(len=256) :: filename
		character(len=24) :: eval, error
		
		write(filename,'(a)') trim(data_dir)//'/'//trim(masstable_filename)
		open(unit=mass_unit, file=trim(filename), iostat=ios, status="old", action="read")
		
		if (ios /= 0) stop 'unable to open mass table for reading'
		
		! first pass
		call skip_header
		nlines = 0
		zmin = 999; zmax = -1; nmin = 999; nmax = -1
		do
			read(mass_unit,*,iostat = ios) Z,A, eval, mass, error
			if (ios /= 0) exit
			nlines = nlines + 1
			N = A-Z
			if (Z < zmin) zmin  = Z
			if (Z > zmax) zmax = Z
			if (N < nmin) nmin = N
			if (N > nmax) nmax = N
		end do
		
		! now read the table
		allocate(mass_table(zmin:zmax,nmin:nmax))
		mass_table = no_mass_table_entry
		rewind(mass_unit)
		call skip_header
		do i = 1, nlines
			read(mass_unit,*) Z, A, eval, mass, error
			N = A-Z
			! store mass and convert from keV to MeV
			mass_table(Z,N) = mass*keV_to_MeV
		end do
		close(mass_unit)
		
		contains
		subroutine skip_header()
			integer :: i,ios
			do i = 1, masstable_header_length
				read(mass_unit,*,iostat=ios)
				if (ios /= 0) stop 'error while reading header of masstable'
			end do
		end subroutine skip_header
	end subroutine read_mass_table

	subroutine process_winvn_table()
		use iso_fortran_env, only: error_unit
		integer, parameter :: in_unit = 30, out_unit = 40
		integer :: ios, i
		character(len=256) :: infile_name,outfile_name
		real(dp) :: fac
		integer :: zmin, zmax, nmin, nmax
		integer :: Z, N
		real(dp) :: W, spin, mass_excess, pfcn(24)
		character(len=8) :: name, ref
		logical :: ground_state
		
		fac = mev_to_ergs/amu/clight**2
		
		! the mass table must be allocated first
		if (.not.allocated(mass_table)) then
			write (error_unit,*) 'mass_table must be allocated first'
			return
		end if
		
		zmin = lbound(mass_table,dim=1)
    zmax = ubound(mass_table,dim=1)
    nmin = lbound(mass_table,dim=2)
    nmax = ubound(mass_table,dim=2)

		write(infile_name,'(a)') trim(data_dir)//'/'//trim(winvn_filename)
		write(outfile_name,'(a)') trim(output_dir)//'/isotopes.data'
		
		open(unit=in_unit, file=trim(infile_name), iostat=ios, status="old", action="read")
		if ( ios /= 0 ) stop "Error opening raw winvn file"
		
		open(unit=out_unit, file=trim(outfile_name), iostat=ios, action="write")
		if ( ios /= 0 ) stop "Error opening processed winvn file"
		
		! skim off the header
		do i = 1, winvn_header_length + number_nuclides
			read(in_unit,*)
		end do
		
		do i = 1, number_nuclides	! 4 lines per nuclide
			read(in_unit,*,iostat=ios) name,W, Z, N, spin, mass_excess, ref
			if (ios /= 0) exit
			read(in_unit,*,iostat=ios) pfcn(1:8)
			if (ios /= 0) exit
			read(in_unit,*,iostat=ios) pfcn(9:16)
			if (ios /= 0) exit
			read(in_unit,*,iostat=ios) pfcn(17:24)
			if (ios /= 0) exit
			
			if (name /= 'al-6' .and. name /= 'al*6') then
				ground_state = .true.
			else
				ground_state = .false.
			end if
			
			! lookup the mass information if it is available
      if (Z >= zmin .and. Z <= zmax .and. N >= nmin .and. N <= nmax .and. ground_state) then
				if (mass_table(Z,N) /= no_mass_table_entry)	mass_excess = mass_table(Z,N)
			end if
			
			! set the atomic weight
			W = Z + N + mass_excess*fac
			
			! convert the name
			if (name == 'n') name = 'neut'
			if (name == 'p') name = 'h1'
			if (name == 'd') name = 'h2'
			if (name == 't') name = 'h3'
			if (name == 'al-6') name = 'al26-1'
			if (name == 'al*6') name = 'al26-2'
			
			! write to the processed datafile
			call write_entry
			
			! duplicate entry for h1 as prot
			if (name == 'h1') then
				name = 'prot'
				call write_entry
			end if
		end do
		! write the 'xtra' entries for the rates
		name = 'xtra1'; W = 100.0; Z = 0; N = 100; spin = 0.0; mass_excess = 0.0; pfcn = 1.0
		call write_entry
		
		name = 'xtra2'; W = 200.0; Z = 0; N = 200; spin = 0.0; mass_excess = 0.0; pfcn = 1.0
		call write_entry
		
		close(in_unit)
		close(out_unit)
		
		contains
		subroutine write_entry()
			write (out_unit,'(a8,f13.7,i5,i5,f6.1,f14.9)') name, W, Z, N, spin, mass_excess
			write (out_unit,'(8(es12.5,tr1))') pfcn(1:8)
			write (out_unit,'(8(es12.5,tr1))') pfcn(9:16)
			write (out_unit,'(8(es12.5,tr1))') pfcn(17:24)
		end subroutine write_entry
	end subroutine process_winvn_table

	subroutine cleanup()
		deallocate(mass_table)
	end subroutine cleanup

end module chem_support
