! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton and Pablo Marchant
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************


      module binary_timestep

      use const_def
      use star_lib
      use star_def
      use num_lib, only: safe_log10
      use binary_def

      implicit none

      contains

      subroutine set_star_timesteps() ! sets the smallest next timestep for all stars
         integer :: i
         real(dp) :: dt_min, rel_overlap
         type (star_info), pointer :: s
         type (binary_info), pointer :: b
         integer :: ierr
         ierr = 0
         call binary_ptr(b, ierr)
         dt_min = b% max_timestep
         do i = 1, num_stars
            call star_ptr(b% star_ids(i), s, ierr)
            if (ierr /= 0) then
                write(*, *) trim('star_ptr') // ' ierr', ierr
                return
            end if
            if (s% dt_next < dt_min) then
               dt_min = s% dt_next
            end if
         end do
         do i = 1, num_stars
            call star_ptr(b% star_ids(i), s, ierr)
            if (ierr /= 0) then
                write(*, *) trim('star_ptr') // ' ierr', ierr
                return
            end if
            s% dt_next = dt_min
         end do

         if (.not. b% trace_binary_rlo) return
         call star_ptr(b% star_ids(1), s, ierr)
         if (ierr /= 0) then
             write(*, *) trim('star_ptr') // ' ierr', ierr
             return
         end if
         if (mod(s% model_number, s% terminal_interval) == 0) then
            write(*,'(99a20)') 'star', 'model', 'age', 'mass', 'lg_mdot', '(r-rl)/rl', 'last photo'
         else if (num_stars > 1) then
            call star_ptr(b% star_ids(2), s, ierr)
            if (ierr /= 0) then
                write(*, *) trim('star_ptr') // ' ierr', ierr
                return
            end if
            if (mod(s% model_number, s% terminal_interval) == 0) &
               write(*,'(99a20)') 'star', 'model', 'age', 'mass', 'lg_mdot', '(r-rl)/rl', 'last photo'
         end if
         do i = 1, num_stars
            call star_ptr(b% star_ids(i), s, ierr)
            if (ierr /= 0) then
                write(*, *) trim('star_ptr') // ' ierr', ierr
                return
            end if
            if (i == b% donor_star_number) then
               rel_overlap = (b% r1 - b% rl1) / b% rl1
            else
               rel_overlap = (b% r2 - b% rl2) / b% rl2
            end if
            write(*,'(2i20,4(4x,1pe16.9),8x,a)') i, &
               s% model_number, s% star_age, s% star_mass, &
               log10(max(1d-99,abs(s% star_mdot))), rel_overlap, &
               trim(s% most_recent_photo_name)
         end do
         
      end subroutine set_star_timesteps

      integer function binary_pick_next_timestep(b)
         type (binary_info), pointer :: b
         type (star_info), pointer :: s
         
         real(dp) :: &
            env_change, dtm, dtj, dta, dtr, dtmdot, &
            j_change, sep_change, rel_gap_change, set_dt
         character (len=8) :: why_str

         include 'formats.inc'

         binary_pick_next_timestep = keep_going

         s => b% s_donor

         ! release control during implicit scheme after b% num_limit_timestep steps
         if (abs(b% mtransfer_rate) > 0 .and. &
             b% max_tries_to_achieve > 0 .and. &
             b% rl_rel_overlap_tolerance > 0d0) then
            if (b% num_rlo_timesteps < b% num_limit_timestep) then
               b% max_timestep = s% dt
               b% num_rlo_timesteps = b% num_rlo_timesteps +1
            else
               b% max_timestep = 1d99
            end if
            return
         end if

         ! release control during explicit scheme after b% num_limit_timestep steps
         ! past the point where rl_rel_gap < - rl_rel_overlap_tolerance
         if (b% rl_relative_gap < - b% rl_rel_overlap_tolerance .and. &
             .not. (b% max_tries_to_achieve > 0 .and. &
             b% rl_rel_overlap_tolerance > 0d0)) then
            if (b% num_rlo_timesteps < b% num_limit_timestep) then
               b% max_timestep = s% dt
               b% num_rlo_timesteps = b% num_rlo_timesteps +1
            else
               b% max_timestep = 1d99
            end if
            return
         end if

         b% env = s% star_mass - s% he_core_mass 
         if (b% env_old /= 0) then
            env_change = b% env - b% env_old
         else
            env_change = 0
         end if
         
         if (b% rl_relative_gap_old /= 0) then
            rel_gap_change = b% rl_relative_gap_old - b% rl_relative_gap
         else
            rel_gap_change = 0
         end if
         
         if (b% angular_momentum_j_old /= 0) then
            j_change = b% angular_momentum_j - b% angular_momentum_j_old
         else
            j_change = 0
         end if
         
         if (b% separation_old /= 0) then
            sep_change = b% separation - b% separation_old
         else
            sep_change = 0
         end if
   
         ! get limits for dt based on relative changes
         if (b% fm <= 0) then
             dtm = 1d99
         else if (b% env < 1d-6*s% star_mass) then
            dtm = s% time_step
         else
            dtm = s% time_step/(abs(env_change/b% env)/b% fm+1d-99)
         end if
         
         if (b% fr <= 0) then
             dtr = 1d99
         else if (abs(b% rl_relative_gap) < b% rl_rel_overlap_tolerance) then
            dtr = s% time_step
         else 
            dtr = s% time_step/(abs(rel_gap_change/b% rl_relative_gap)/b% fr+1d-99)
         end if

         if (b% fj <= 0) then
             dtj = 1d99
         else if (b% angular_momentum_j <= 0) then
            dtj = s% time_step
         else
            dtj = s% time_step/(abs(j_change/b% angular_momentum_j)/b% fj+1d-99)
         end if

         if (b% fa <= 0) then
             dta = 1d99
         else if (b% separation <=0) then
            dta = s% time_step 
         else
            dta = s% time_step/(abs(sep_change/b% separation)/b% fa+1d-99)
         end if
            
         if (-b% mtransfer_rate <= b% max_abs_mdot*Msun/secyer) then 
            dtmdot = 1.3*s% time_step
         else 
            dtmdot = s% time_step
         end if

         set_dt = min(dtm, dtr, dtj, dta, dtmdot)
         
         if (set_dt == dtmdot) then
            why_str = 'dtmdot'
         else if (set_dt == dtm) then
            why_str = 'dtm'
         else if (set_dt == dtr) then
            why_str = 'dtr'
         else if (set_dt == dtj) then
            why_str = 'dtj'
         else if (set_dt == dta) then
            why_str = 'dta'
         else
            why_str = '   '
         end if

         if (set_dt < 1d-7) set_dt = 1d-7 ! there's a limit to everything

         if (rlo_dbg) write(*,'(i6,3x,a,3x,e20.10,12x,a,3x,f8.2,10x,a)') s% model_number, &
            '(rl-r)/r', b% rl_relative_gap, &
            'signed_lg_rel_gap_x_1e4', &
            sign(1d0,b% rl_relative_gap)*log10(max(1d0,1d6*abs(b% rl_relative_gap))), &
            why_str

         b% max_timestep = set_dt*secyer
         
      end function binary_pick_next_timestep
      

      end module binary_timestep
