! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton and Pablo Marchant
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************


      module binary_jdot

      use const_def
      use star_lib
      use star_def
      use num_lib, only: safe_log10
      use binary_def

      implicit none

      contains

      real(dp) function get_jdot(mdot, xfer_fraction)
         real(dp), intent(in) :: mdot, xfer_fraction

         integer :: ierr
         type (binary_info), pointer :: b
         ! L-S feedback on jdot
         real(dp) :: jdot_ls_feedback
         
         jdot_ls_feedback = 0

         call binary_ptr(b, ierr)
         

         ! calculate jdot from magnetic braking
         if (.not. b% do_jdot_mb) then
             b% jdot_mb = 0d0
         else if (.not. b% use_other_jdot_mb) then
             call default_jdot_mb(ierr)
         else
             call b% other_jdot_mb(ierr)
         end if
         
         ! calculate jdot from gravitational wave radiation
         if (.not. b% do_jdot_gr) then
             b% jdot_gr = 0d0
         else if (.not. b% use_other_jdot_gr) then
             call default_jdot_gr(ierr)
         else
             call b% other_jdot_gr(ierr)
         end if
            
         ! calculate jdot for mass ejected from system
         if (.not. b% do_jdot_ml) then
             b% jdot_ml = 0d0
         else if (.not. b% use_other_jdot_ml) then
             call default_jdot_ml(ierr)
         else
             call b% other_jdot_ml(ierr)
         end if
         
         ! calculate tidal jdot
         if (.not. b% do_jdot_tide) then
             b% jdot_tide = 0d0
         else if (.not. b% use_other_jdot_tide) then
             call default_jdot_tide(ierr)
         else
             call b% other_jdot_tide(ierr)
         end if
         
         ! calculate extra jdot
         if (.not. b% use_other_extra_jdot) then
             b% extra_jdot = 0
         else 
             call b% other_extra_jdot(ierr)
         end if

         ! solve jdot due to L-S coupling, ignore in first step
         if (b% do_rotation .and. .not. b% doing_first_model_of_run) then
            b% jdot_ls = -(b% s_donor % total_angular_momentum - &
               b% s_donor % total_angular_momentum_old + &
               b% s_donor% angular_momentum_removed) / b% s_donor% dt
            if (b% evolve_both_stars) &
               b% jdot_ls = b% jdot_ls &
                  -(b% s_accretor % total_angular_momentum - &
                  b% s_accretor % total_angular_momentum_old + &
                  b% s_accretor% angular_momentum_removed) / b% s_accretor% dt
         else
            b% jdot_ls = 0
         end if

         
         get_jdot = (b% jdot_mb + b% jdot_gr + b% jdot_ml + &
             b% jdot_tide + b% jdot_ls + b% extra_jdot)*b% jdot_multiplier
         
      end function get_jdot

      logical function do_mb(s, b)
         use mlt_def, only: convective_mixing
         type (star_info), pointer :: s
         type (binary_info), pointer :: b
         
         real(dp) :: sum_conv, q_loc, sum_div_qloc 
         integer :: k

         include 'formats.inc'

         do_mb = .false.

         if (s% star_mass > 1.4) then
             return
         end if

         ! calculate how much of inner region is convective
         sum_conv = 0; q_loc = 0
         do k = s% nz, 1, -1
            q_loc = s% q(k)
            if (q_loc > 0.2d0) exit 
            if (s% mixing_type(k) == convective_mixing) &
               sum_conv = sum_conv + s% dq(k)
         end do
         
         sum_div_qloc = (b% sum_div_qloc + sum_conv/q_loc)/2
         b% sum_div_qloc = sum_div_qloc
         
         if (b% have_radiative_core) then ! check if still have rad core
            if (sum_div_qloc > 0.75d0) then
               b% have_radiative_core = .false.
               write(*,*)
               write(*,*) 'turn off magnetic braking because radiative core has gone away'
               write(*,*)
               ! required mdot for the implicit scheme may drop drastically,
               ! so its neccesary to increase change factor to avoid implicit 
               ! scheme from getting stuck
               b% change_factor = b% max_change_factor
               do_mb = .false.
            else
               do_mb = .true.
            end if
         else if (sum_div_qloc < 0.25d0) then ! check if now have rad core
            b% have_radiative_core = .true.
            if (.not. do_mb) then
               write(*,*)
               write(*,*) 'turn on magnetic braking'
               write(*,*)
               do_mb = .true.
            end if
         end if
            
      end function do_mb

      subroutine default_jdot_mb(ierr)
         integer, intent(out) :: ierr
         type (binary_info), pointer :: b
         ierr = 0
         call binary_ptr(b, ierr)
         b% jdot_mb = 0
         ! use the formula from rappaport, verbunt, and joss.  apj, 275, 713-731. 1983.
         if (do_mb(b% s_donor,b)) b% jdot_mb = -3.8d-30*b% m2*(rsun**4)* &
                           ((min(b% r1,b% rl1)/rsun)**b% magnetic_braking_gamma)* &
                           ((2.0*pi/b% period)**3)

         if (b% evolve_both_stars .and. b% include_accretor_mb) then
             !if (do_mb(s_accretor)) def_jdot_mb = def_jdot_mb - &
             b% jdot_mb = b% jdot_mb - &
                           3.8d-30*b% m1*(rsun**4)* &
                           ((min(b% r2,b% rl2)/rsun)**b% magnetic_braking_gamma)* &
                           ((2.0*pi/b% period)**3)
         end if

      end subroutine default_jdot_mb

      subroutine default_jdot_gr(ierr)
         integer, intent(out) :: ierr
         type (binary_info), pointer :: b
         ierr = 0
         call binary_ptr(b, ierr)
         b% jdot_gr = -32d0 * b% s_donor% cgrav(1)**3 * b% m1 * b% m2 * (b% m1 + b% m2) / &
             (5d0 * clight**5 * b% separation**4) * b% angular_momentum_j
      end subroutine default_jdot_gr

      subroutine default_jdot_ml(ierr)
         integer, intent(out) :: ierr
         type (binary_info), pointer :: b
         real(dp) :: alfa
         ierr = 0
         call binary_ptr(b, ierr)
         if (b% alpha < 0) then
            alfa = (b% m1/(b% m1+b% m2))**2
         else
            alfa = b% alpha
         end if
         b% jdot_ml = b% mdot_system_1*alfa*(b% separation**2)*2*pi/b% period
         if (b% alpha < 0) then
            alfa = (b% m2/(b% m1+b% m2))**2
         else
            alfa = b% alpha
         end if
         b% jdot_ml = b% jdot_ml + b% mdot_system_2*alfa*(b% separation**2)*2*pi/b% period
      end subroutine default_jdot_ml

      subroutine default_jdot_tide(ierr)
         integer, intent(out) :: ierr
         type (binary_info), pointer :: b
         real(dp) :: rc
         ierr = 0
         call binary_ptr(b, ierr)
         b% jdot_tide = 0
         if (b% tidal_Q <= 0 .or. b% period <= 0) then
            b% jdot_tide = 0
         else
            rc = b% R_companion
            if (rc <= 0) rc = b% companion_mass**0.78d0
            b% jdot_tide = -(9d0/2d0)*(2*pi/b% period)* &
               (b% m2/b% m1)*(1/b% tidal_Q)*(rc*Rsun/b% separation)**5 * b% angular_momentum_j
         end if
      end subroutine default_jdot_tide

      end module binary_jdot
