! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton and Pablo Marchant
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************


      module binary_history

      use const_def
      use star_lib
      use star_def
      use num_lib, only: safe_log10
      use binary_def
      use binary_private_def
      use binary_history_specs

      implicit none

      contains


      subroutine write_binary_history_info ( b, &
            how_many_extra_binary_history_columns, data_for_extra_binary_history_columns, ierr)
         use utils_lib, only:alloc_iounit, free_iounit
         type (binary_info), pointer :: b
         interface
            include 'extra_binary_history_cols.inc'
         end interface
         integer, intent(out) :: ierr
         character (len=maxlen_profile_column_name), pointer :: names(:) ! (num_history_columns)
         real(dp), pointer :: vals(:) ! (num_history_columns)
         logical, pointer :: is_int(:)
         logical, parameter :: write_flag = .true.
         names => null()
         vals => null()
         is_int => null()
         call do_binary_history_info( &
            b, how_many_extra_binary_history_columns, data_for_extra_binary_history_columns, &
            write_flag, names, vals, is_int, ierr)
      end subroutine write_binary_history_info


      subroutine do_binary_history_info( &
            b, how_many_extra_binary_history_columns, data_for_extra_binary_history_columns, &
            write_flag, names, vals, is_int, ierr)
         use utils_lib, only: alloc_iounit, free_iounit
         type (binary_info), pointer :: b
         interface
            include 'extra_binary_history_cols.inc'
         end interface
         logical, intent(in) :: write_flag
         character (len=maxlen_profile_column_name), pointer :: names(:) ! (num_history_columns)
         real(dp), pointer :: vals(:) ! (num_history_columns)
         logical, pointer :: is_int(:)
         integer, intent(out) :: ierr
         
         character (len=strlen) :: fname, dbl_fmt, int_fmt, txt_fmt
         integer :: numcols, io, i, nz, col, j, i0, num_extra_cols
         character (len=maxlen_binary_history_column_name), pointer :: extra_col_names(:)
         real(dp), pointer :: extra_col_vals(:)
         
         include 'formats'
         
         dbl_fmt = b% history_dbl_format
         int_fmt = b% history_int_format
         txt_fmt = b% history_txt_format
         
         ierr = 0
         
         if (write_flag) then
            io = alloc_iounit(ierr)
            if (ierr /= 0) return
         else
            io = -1 ! set to invalid value to trigger complaint if use it by mistake
         end if
         
         if (.not. associated(b% history_column_spec)) then
            numcols = 0
         else
            numcols = size(b% history_column_spec, dim=1)
         end if
         
         if (numcols == 0) then
            write(*,*) 'WARNING: do not have any output specified for binary logs.'
            if (io > 0) call free_iounit(io)
            return
         end if
         
         num_extra_cols = how_many_extra_binary_history_columns(b)
         if (num_extra_cols > 0) then
            allocate( &
               extra_col_names(num_extra_cols), extra_col_vals(num_extra_cols), stat=ierr)
            if (ierr /= 0) then
               if (io > 0) call free_iounit(io)
               return
            end if
            call data_for_extra_binary_history_columns( &
               b, num_extra_cols, extra_col_names, extra_col_vals, ierr)
            if (ierr /= 0) then
               deallocate(extra_col_names, extra_col_vals)
               if (io > 0) call free_iounit(io)
               return
            end if
         end if
         
         i0 = 1
         if (write_flag .and. (open_close_log .or. b% s_donor% model_number == -100)) then
            fname = trim(b% log_directory) // '/' // trim(b% history_name)
            if ((.not. history_file_exists(fname,io)) .or. b% doing_first_model_of_run) then
               ierr = 0
               open(unit=io, file=trim(fname), action='write', iostat=ierr)
            else
               i0 = 3            
               open(unit=io, file=trim(fname), action='write', position='append', iostat=ierr)
            end if
            if (ierr /= 0) then
               write(*,*) 'failed to open ' // trim(fname)
               if (io > 0) call free_iounit(io)
               return
            end if
         end if
         
         if (write_flag .and. i0 == 1) then ! write parameters at start of log
            !call b% other_binary_history_data_initialize(b, ierr)
            !if (ierr /= 0) return
            do i=1,3
               col = 0
               call write_integer(io, col, i, 'version_number', version_number)
               call write_val(io, col, i, 'initial_don_mass', initial_donor_mass)
               call write_val(io, col, i, 'initial_acc_mass', initial_companion_mass)
               call write_val(io, col, i, 'initial_period_days', &
                   initial_binary_period/(3600*24))
               write(io,*)
            end do
            write(io,*)
         end if

         do i=i0,3 ! add a row to the log
            !call b% other_binary_history_data_add_model(s% id, ierr)
            !if (ierr /= 0) return
            col = 0
            if (i==3) then
            
               !if (write_flag .and. i0 == 1) then
                  !close(io)
                  !stop "enough"
                  !fname = trim(b% log_directory) // '/' // trim(b% history_name)
                  !open(unit=io, file=trim(fname), action='write',status='replace', iostat=ierr)
                  !if (ierr /= 0) then
                  !   call free_iounit(io); return
                  !end if
               !end if
               
            end if
            do j=1,numcols
               call do_col(i, j)
            end do
            do j=1,num_extra_cols
               call do_extra_col(i, j)
            end do
            if (write_flag) write(io, *)
         end do
         
         if (open_close_log) close(io)
         
         if (io > 0) call free_iounit(io)

         if (associated(extra_col_names)) deallocate(extra_col_names)
         if (associated(extra_col_vals)) deallocate(extra_col_vals)
         
         
         contains


         subroutine do_extra_col(pass, j)
            integer, intent(in) :: pass, j
            if (pass == 1) then
               if (write_flag) write(io, fmt=int_fmt, advance='no') j + numcols
            else if (pass == 2) then
               call do_name(j + numcols, extra_col_names(j))
            else if (pass == 3) then
               call do_val(j + numcols, extra_col_vals(j))
            end if
         end subroutine do_extra_col


         subroutine do_name(j, col_name)
            integer, intent(in) :: j
            character (len=*), intent(in) :: col_name
            if (write_flag) then
               write(io, fmt=txt_fmt, advance='no') trim(col_name)
            else
               names(j) = trim(col_name)
            end if
         end subroutine do_name
         

         subroutine do_col(pass, j)
            integer, intent(in) :: pass, j
            if (pass == 1) then
               call do_col_pass1
            else if (pass == 2) then
               call do_col_pass2(j)
            else if (pass == 3) then
               call do_col_pass3(b% history_column_spec(j))
            end if
         end subroutine do_col
         
         
         subroutine do_col_pass1 ! write the column number
            col = col+1
            if (write_flag) write(io, fmt=int_fmt, advance='no') col
         end subroutine do_col_pass1
         
         
         subroutine do_col_pass2(j) ! get the column name
            integer, intent(in) :: j
            character (len=100) :: col_name
            character (len=10) :: str
            integer :: c, i, ii
            c = b% history_column_spec(j)
            col_name = trim(binary_history_column_name(c))
            call do_name(j, col_name)
         end subroutine do_col_pass2
         
         
         subroutine do_col_pass3(c) ! get the column value
            integer, intent(in) :: c
            integer :: i, ii, k, int_val
            logical :: is_int_val
            real(dp) :: val, val1, Ledd, power_photo, frac
            int_val = 0; val = 0; is_int_val = .false.
            call binary_history_getval( &
               b, c, val, int_val, is_int_val, ierr)
            if (ierr /= 0) then
               write(*,*) 'missing log info for ' // trim(binary_history_column_name(c)), j, k
               return
            end if
            if (is_int_val) then
               call do_int_val(j,int_val)
            else
               call do_val(j,val)
            end if
         end subroutine do_col_pass3
         
         
         subroutine do_val(j, val)
            use utils_lib, only: is_bad_num
            integer, intent(in) :: j
            real(dp), intent(in) :: val
            if (write_flag) then
               if (is_bad_num(val)) then
                  write(io, fmt=dbl_fmt, advance='no') -1d99
               else
                  write(io, fmt=dbl_fmt, advance='no') val
               end if
            else
               vals(j) = val
               is_int(j) = .false.
            end if
         end subroutine do_val
         
         
         subroutine do_int_val(j, val)
            integer, intent(in) :: j
            integer, intent(in) :: val
            if (write_flag) then
               write(io, fmt=int_fmt, advance='no') val
            else
               vals(j) = dble(val)
               is_int(j) = .true.
            end if
         end subroutine do_int_val
                  
         
      end subroutine do_binary_history_info
      


      subroutine write_integer(io, col, pass, name, val)
         integer, intent(in) :: io, pass
         integer, intent(inout) :: col
         character (len=*), intent(in) :: name
         integer, intent(in) :: val
         if (pass == 1) then
            col = col+1
            write(io, fmt='(i28, 1x)', advance='no') col
         else if (pass == 2) then
            write(io, fmt='(a28, 1x)', advance='no') trim(name)
         else if (pass == 3) then
            write(io, fmt='(i28, 1x)', advance='no') val
         end if
      end subroutine write_integer
      
      
      subroutine write_val(io, col, pass, name, val)
         integer, intent(in) :: io, pass
         integer, intent(inout) :: col
         character (len=*), intent(in) :: name
         real(dp), intent(in) :: val
         if (pass == 1) then
            col = col+1
            write(io, fmt='(i28, 1x)', advance='no') col
         else if (pass == 2) then
            write(io, fmt='(a28, 1x)', advance='no') trim(name)
         else if (pass == 3) then
            write(io, fmt='(1pe28.16e3, 1x)', advance='no') val
         end if
      end subroutine write_val
      
      
      subroutine binary_history_getval( &
            b, c, val, int_val, is_int_val, ierr)
         type (binary_info), pointer :: b
         integer, intent(in) :: c
         real(dp), intent(out) :: val
         integer, intent(out) :: int_val
         logical, intent(out) :: is_int_val
         integer, intent(out) :: ierr
         integer :: k, i
         
         include 'formats'
         
         ierr = 0
         is_int_val = .false.
         int_val = 0
         val = 0
            
         select case(c)
         
         case(bh_model_number)
            int_val = b% s_donor% model_number
            is_int_val = .true.
         case(bh_age)
            val = b% s_donor% star_age
         case(bh_period_days)
            val = b% period/(60d0*60d0*24d0)
         case(bh_period_hr)
            val = b% period/(60d0*60d0)
         case(bh_period_minutes)
            val = b% period/60d0
         case(bh_lg_separation)
            val = safe_log10(b% separation)
         case(bh_binary_separation)
            val = b% separation/Rsun
         case(bh_rl_donor)
            val = b% rl1/Rsun
         case(bh_rl_accretor)
            val = b% rl2/Rsun
         case(bh_rl_overflow)
            val = (b% r1-b% rl1)/Rsun
         case(bh_rl_relative_overflow)
            val = (b% r1-b% rl1)/b% rl1
         case(bh_donor_mass)
            val = b% m2
         case(bh_lg_donor_mass)
            val = safe_log10(b% m2)
         case(bh_companion_mass)
            val = b% m1
         case(bh_lg_companion_mass)
            val = safe_log10(b% m1)
         case(bh_sum_of_masses)
            val = b% m2 + b% m1
         case(bh_donor_div_companion_mass)
            val = b% m2 / b% m1
         case(bh_delta_companion_mass)
            val = b% companion_mass - initial_companion_mass
         case(bh_lg_F_irr)
            val = safe_log10(b% s_donor% irradiation_flux)
         case(bh_xfer_fraction)
            val = b% xfer_fraction
         case(bh_log_J)
            val = safe_log10(b% angular_momentum_j)
         case(bh_log_abs_Jdot)
            val = safe_log10(abs(b% jdot))
         case(bh_log_abs_jdot_mb)
            val = safe_log10(abs(b% jdot_mb))
         case(bh_log_abs_jdot_gr)
            val = safe_log10(abs(b% jdot_gr))
         case(bh_log_abs_jdot_ml)
            val = safe_log10(abs(b% jdot_ml))
         case(bh_log_abs_jdot_tide)
            val = safe_log10(abs(b% jdot_tide))
         case(bh_log_abs_jdot_ls)
            val = safe_log10(abs(b% jdot_ls))
         case(bh_log_abs_extra_jdot)
            val = safe_log10(abs(b% extra_jdot))
         
         case default
            ierr = -1
         
         end select
         
         
         contains
         
      end subroutine binary_history_getval
      

      integer function binary_how_many_extra_history_columns(s, id, id_extra) result(n)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra
         integer :: other_extra_columns
         if (.not. binary% evolve_both_stars) then
            n = rlo_extra_history_columns
         else if (is_donor(s)) then
            n = how_many_donor_history_columns(s, id, id_extra)
         else
            n = how_many_accretor_history_columns(s, id, id_extra)
         end if
         other_extra_columns = &
            binary% how_many_extra_history_columns(s, id, id_extra)
         n = n + other_extra_columns
      end function binary_how_many_extra_history_columns
      
      
      subroutine binary_data_for_extra_history_columns( &
            s, id, id_extra, n, names, vals, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra, n
         character (len=maxlen_binary_history_column_name) :: names(n)
         real(dp) :: vals(n)
         integer, intent(out) :: ierr
         integer :: num_extra
         ierr = 0
         num_extra = binary% how_many_extra_history_columns(s, id, id_extra)
         if (num_extra > n) then
            ierr = -1
            return
         end if
         call binary% data_for_extra_history_columns( &
            s, id, id_extra, num_extra, names, vals, ierr)
         if (ierr /= 0) return         
         if (num_extra == n) return
         if (.not. binary% evolve_both_stars) then
            call rlo_data_for_extra_history_columns( &
               s, id, id_extra,  &
               n-num_extra, names(num_extra+1:n), vals(num_extra+1:n), ierr)
         else if (is_donor(s)) then
            call data_for_donor_history_columns( &
               s, id, id_extra,  &
               n-num_extra, names(num_extra+1:n), vals(num_extra+1:n), ierr)
         else
            call data_for_accretor_history_columns( &
               s, id, id_extra,  &
               n-num_extra, names(num_extra+1:n), vals(num_extra+1:n), ierr)
         end if
      end subroutine binary_data_for_extra_history_columns

      
      integer function binary_how_many_extra_profile_columns(s, id, id_extra) result(n)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra
         integer :: other_extra_columns
         if (is_donor(s)) then
            n = how_many_donor_profile_columns(s, id, id_extra)
         else
            n = how_many_accretor_profile_columns(s, id, id_extra)
         end if
         other_extra_columns = binary% how_many_extra_profile_columns(s, id, id_extra)
         n = n + other_extra_columns
      end function binary_how_many_extra_profile_columns
      
      
      subroutine binary_data_for_extra_profile_columns( &
            s, id, id_extra, n, nz, names, vals, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra, n, nz
         character (len=maxlen_profile_column_name) :: names(n)
         real(dp) :: vals(nz,n)
         integer, intent(out) :: ierr
         integer :: num_extra
         ierr = 0
         num_extra = binary% how_many_extra_profile_columns( &
            s, id, id_extra)
         if (num_extra > n) then
            ierr = -1
            return
         end if
         call binary% data_for_extra_profile_columns( &
               s, id, id_extra, num_extra, nz, names, vals, ierr)
         if (ierr /= 0) return         
         if (num_extra == n) return
         if (is_donor(s)) then
            call data_for_donor_profile_columns( &
               s, id, id_extra, n-num_extra, nz, &
               names(num_extra+1:n), vals(1:nz,num_extra+1:n), ierr)
         else
            call data_for_accretor_profile_columns( &
               s, id, id_extra, n-num_extra, nz, &
               names(num_extra+1:n), vals(1:nz,num_extra+1:n), ierr)
         end if
      end subroutine binary_data_for_extra_profile_columns
      

      integer function how_many_accretor_history_columns(s, id, id_accretor)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_accretor
         how_many_accretor_history_columns = 0
      end function how_many_accretor_history_columns
      
      
      subroutine data_for_accretor_history_columns(s, id, id_accretor, n, names, vals, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_accretor, n
         character (len=maxlen_history_column_name) :: names(:) ! (n)
         real(dp) :: vals(:) ! (n)
         integer, intent(out) :: ierr
         ierr = 0
      end subroutine data_for_accretor_history_columns

      
      integer function how_many_accretor_profile_columns(s, id, id_accretor)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_accretor
         how_many_accretor_profile_columns = 0
      end function how_many_accretor_profile_columns
      
      
      subroutine data_for_accretor_profile_columns(s, id, id_accretor, n, nz, names, vals, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_accretor, n, nz
         character (len=maxlen_profile_column_name) :: names(:) ! (n)
         real(dp) :: vals(:,:) ! (nz,n)
         integer, intent(out) :: ierr
         ierr = 0
      end subroutine data_for_accretor_profile_columns
      

      integer function how_many_donor_history_columns(s, id, id_donor)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_donor
         how_many_donor_history_columns = binary_rlo_extra_history_columns
      end function how_many_donor_history_columns
      
      
      subroutine data_for_donor_history_columns(s, id, id_donor, n, names, vals, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_donor, n
         character (len=maxlen_history_column_name) :: names(n)
         real(dp) :: vals(n)
         integer, intent(out) :: ierr
         call data_for_history_columns(s, id, id_donor, n, names, vals, ierr)
      end subroutine data_for_donor_history_columns
      
      
      subroutine data_for_history_columns(s, id, id_donor, n, names, vals, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_donor, n
         character (len=maxlen_history_column_name) :: names(n)
         real(dp) :: vals(n), rl_accretor, rl_donor
         integer, intent(out) :: ierr
         integer :: i
         type (binary_info), pointer :: b

         ierr = 0
         call binary_ptr(b, ierr)
         if (n /= binary_rlo_extra_history_columns) then
            stop 'bad value for n in data_for_history_columns'
         end if
         
         vals = 0

         i = 0
         
         i=i+1; names(i) = 'period_days'
         vals(i) = b% period/(60d0*60d0*24d0)
         
         i=i+1; names(i) = 'period_hr'
         vals(i) = b% period/(60d0*60d0)
         
         i=i+1; names(i) = 'period_minutes'
         vals(i) = b% period/60d0     
             
         i=i+1; names(i) = 'lg_separation'
         vals(i) = safe_log10(b% separation)
         
         i=i+1; names(i) = 'binary_separation'
         vals(i) = b% separation/Rsun
         
         i=i+1; names(i) = 'rl_donor'
         vals(i) = b% rl1/Rsun
         
         i=i+1; names(i) = 'rl_accretor'
         vals(i) = b% rl2/Rsun
         
         i=i+1; names(i) = 'donor_mass'
         vals(i) = b% s_donor% star_mass
         
         i=i+1; names(i) = 'companion_mass'
         vals(i) = b% companion_mass
         
         i=i+1; names(i) = 'sum_of_masses'
         vals(i) = b% s_donor% star_mass + b% companion_mass
         
         i=i+1; names(i) = 'donor_div_companion_mass'
         vals(i) = b% s_donor% star_mass / b% companion_mass
         
         i=i+1; names(i) = 'delta_companion_mass'
         vals(i) = b% companion_mass - initial_companion_mass
         
         i=i+1; names(i) = 'xfer_fraction'
         vals(i) = b% xfer_fraction
         
         i=i+1; names(i) = 'log_J'
         vals(i) = safe_log10(b% angular_momentum_j)
         
         i=i+1; names(i) = 'log_abs_Jdot'
         vals(i) = safe_log10(abs(b% jdot))
         
         i=i+1; names(i) = 'log_abs_jdot_mb'
         vals(i) = safe_log10(abs(b% jdot_mb))
         
         i=i+1; names(i) = 'log_abs_jdot_gr'
         vals(i) = safe_log10(abs(b% jdot_gr))
         
         i=i+1; names(i) = 'log_abs_jdot_ml'
         vals(i) = safe_log10(abs(b% jdot_ml))
         
         i=i+1; names(i) = 'log_abs_extra_jdot'
         vals(i) = safe_log10(abs(b% extra_jdot))

         i=i+1; names(i) = 'r_donor'
         vals(i) = b% s_donor% photosphere_r
         
         if (binary% evolve_both_stars) then
            i=i+1; names(i) = 'r_accretor'
            vals(i) = b% s_accretor% photosphere_r
         end if
         
         i=i+1; names(i) = 'donor_rl_div_r'
         vals(i) = b% rl1/(b% s_donor% photosphere_r*Rsun)
         
         if (binary% evolve_both_stars) then
            i=i+1; names(i) = 'accretor_rl_div_r'
            vals(i) = b% rl2/(b% s_accretor% photosphere_r*Rsun)
            i=i+1; names(i) = 'separation_div_r_accretor'
            vals(i) = b% separation/(b% s_accretor% photosphere_r*Rsun)
         end if
         
         if (i /= n) then
            stop 'bad value for i in data_for_donor_history_columns'
         end if
      end subroutine data_for_history_columns

      
      integer function how_many_donor_profile_columns(s, id, id_donor)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_donor
         how_many_donor_profile_columns = 0
      end function how_many_donor_profile_columns
      
      
      subroutine data_for_donor_profile_columns(s, id, id_donor, n, nz, names, vals, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_donor, n, nz
         character (len=maxlen_profile_column_name) :: names(n)
         real(dp) :: vals(nz,n)
         integer, intent(out) :: ierr
         integer :: k
         ierr = 0
      end subroutine data_for_donor_profile_columns
      

      subroutine rlo_data_for_extra_history_columns(s, id, id_extra, n, names, vals, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra, n
         character (len=maxlen_history_column_name) :: names(n)
         double precision :: vals(n)
         integer, intent(out) :: ierr
         integer :: i
         double precision :: Lmdot
         type (binary_info), pointer :: b
         ierr = 0
         call binary_ptr(b, ierr)
         
         if (n /= rlo_extra_history_columns) then
            stop 'bad value for n in data_for_extra_history_columns'
         end if
         Lmdot = 6d32*abs(b% mtransfer_rate)/1d-11 ! ergs s^-1
         i = 0
         vals = 0
         
         i=i+1; names(i) = 'period_days'
         vals(i) = b% period/(60d0*60d0*24d0)
         
         i=i+1; names(i) = 'period_hr'
         vals(i) = b% period/(60d0*60d0)
         
         i=i+1; names(i) = 'period_minutes'
         vals(i) = b% period/60d0         
         
         i=i+1; names(i) = 'lg_separation'
         vals(i) = safe_log10(b% separation)
         
         i=i+1; names(i) = 'binary_separation'
         vals(i) = b% separation/Rsun
         
         i=i+1; names(i) = 'rl_donor'
         vals(i) = b% rl1/Rsun
         
         i=i+1; names(i) = 'rl_accretor'
         vals(i) = b% rl2/Rsun
         
         i=i+1; names(i) = 'rl_overflow'
         vals(i) = (b% r1-b% rl1)/Rsun
         
         i=i+1; names(i) = 'rl_relative_overflow'
         vals(i) = (b% r1-b% rl1)/b% rl1
         
         i=i+1; names(i) = 'donor_mass'
         vals(i) = s% star_mass
         
         i=i+1; names(i) = 'lg_donor_mass'
         vals(i) = safe_log10(s% star_mass)
         
         i=i+1; names(i) = 'companion_mass'
         vals(i) = b% companion_mass
         
         i=i+1; names(i) = 'lg_companion_mass'
         vals(i) = safe_log10(b% companion_mass)
         
         i=i+1; names(i) = 'sum_of_masses'
         vals(i) = s% star_mass + b% companion_mass
         
         i=i+1; names(i) = 'donor_div_companion_mass'
         vals(i) = s% star_mass / b% companion_mass
         
         i=i+1; names(i) = 'delta_companion_mass'
         vals(i) = b% companion_mass - initial_companion_mass
         
         i=i+1; names(i) = 'lg_F_irr'
         vals(i) = safe_log10(s% irradiation_flux)
         
         i=i+1; names(i) = 'xfer_fraction'
         vals(i) = b% xfer_fraction
         
         i=i+1; names(i) = 'log_J'
         vals(i) = safe_log10(b% angular_momentum_j)
         
         i=i+1; names(i) = 'log_abs_Jdot'
         vals(i) = safe_log10(abs(b% jdot))
         
         i=i+1; names(i) = 'log_abs_jdot_mb'
         vals(i) = safe_log10(abs(b% jdot_mb))
         
         i=i+1; names(i) = 'log_abs_jdot_gr'
         vals(i) = safe_log10(abs(b% jdot_gr))
         
         i=i+1; names(i) = 'log_abs_jdot_ml'
         vals(i) = safe_log10(abs(b% jdot_ml))
         
         i=i+1; names(i) = 'log_abs_jdot_tide'
         vals(i) = safe_log10(abs(b% jdot_tide))
         
         i=i+1; names(i) = 'log_abs_extra_jdot'
         vals(i) = safe_log10(abs(b% extra_jdot))
         
         if (i /= rlo_extra_history_columns) then
            stop 'bad value for i in data_for_extra_history_columns'
         end if

      end subroutine rlo_data_for_extra_history_columns


      end module binary_history
