! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton and Pablo Marchant
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************


      module binary_evolve

      use const_def
      use star_lib
      use star_def
      use num_lib, only: safe_log10
      use binary_def

      implicit none

      contains

      subroutine binarydata_init(b, evolve_both_stars, trace_binary_rlo)
         use utils_lib, only: is_bad_num
         type (binary_info), pointer :: b
         logical :: evolve_both_stars
         logical :: trace_binary_rlo
         real(dp) :: p, grav, rho
         include 'formats.inc'

         b% doing_first_model_of_run = .true.

         b% evolve_both_stars = evolve_both_stars
         b% trace_binary_rlo = trace_binary_rlo
         b% max_timestep = secyer*1d99

         initial_companion_mass = b% maccretor
         b% companion_mass = initial_companion_mass
         initial_donor_mass = b% mdonor
         if (b% initial_period_in_days <= 0) then ! calculate from initial_separation_in_Rsuns
            b% separation = b% initial_separation_in_Rsuns*Rsun
            b% m1 = initial_companion_mass*Msun
            b% m2 = initial_donor_mass*Msun
            b% initial_period_in_days = &
               (2*pi)*sqrt(b% separation**3/(standard_cgrav*(b% m1+b% m2)))/(24d0*60d0*60d0)
         end if
         initial_binary_period = b% initial_period_in_days*(24d0*60d0*60d0)
         write(*,*)
         write(*,1) 'maccretor', b% maccretor
         write(*,1) 'mdonor', b% mdonor
         write(*,1) 'initial_period_in_days', b% initial_period_in_days
         write(*,1) 'jdot_multiplier', b% jdot_multiplier
         write(*,1) 'fr', b% fr
         write(*,*)
         ! appears that can go to 0.02 Msun okay,
         ! but run into edge of eos valid range if try to go to 0.01 Msun.
         !m_vals = (/ initial_donor_mass, &
         !         .50d0, .25d0, .10d0, .07d0, .05d0, .04d0, .03d0, .025d0, smallest_mass /)
         scale_height_multiple = 10d0
         mdot0 = 5d-10 ! msolar per year for calculating initial envelope density for rlo
         b% lower_limit_on_period_in_hours = -1d99
         b% started_rlo = .false.
         b% have_radiative_core = .false.
         just_evolve = .false.
         b% s_donor% mesh_delta_coeff_pre_ms = 1

         b% period = initial_binary_period
         min_binary_period = b% period
         b% r1 = Rsun*b% s_donor% photosphere_r
         if (b% evolve_both_stars) b% r2 = Rsun*b% s_accretor% photosphere_r
         b% companion_mass = initial_companion_mass
         b% m2 = Msun*b% s_donor% star_mass ! mass of secondary in gm
         b% m1 = Msun*b% companion_mass ! mass of primary in gm
         rho = b% s_donor% rho(1) ! density at surface in g/cm^3
         p = b% s_donor% p(1) ! pressure at surface in dynes/cm^2
         grav = b% s_donor% cgrav(1)*b% m2/(b% r1*b% r1) ! local gravitational acceleration
         b% hp = p/(grav*rho) ! pressure scale height
         b% v_th = sqrt(3.0 * kerg * b% s_donor% T(1) / (mp * b% s_donor% mu(1)))
         rho_rl0 = mdot0 * (msun / secyer) / (b% v_th * 2 * pi * b% hp * b% r1)
         b% separation = &
            ((b% s_donor% cgrav(1)*(b% m1+b% m2))*(b% period/(2*pi))**2)**(1.0/3.0)
         b% min_binary_separation = b% separation
         b% angular_momentum_j = &
            b% m1 * b% m2 * sqrt( b% s_donor% cgrav(1) * b% separation / (b% m1 + b% m2) )
         b% rl1 = eval_rlobe(b% m2, b% m1, b% separation) ! donor roche lobe radius
         b% rl2 = eval_rlobe(b% m1, b% m2, b% separation) ! accretor roche lobe radius
         b% rl_relative_gap = (b% rl1 - b% r1)/b% r1
         if (is_bad_num(b% rl_relative_gap)) stop 'binarydata_init'
          
      end subroutine

      subroutine binary_evolve_step(b)
         use utils_lib, only: is_bad_num
         use binary_jdot, only: get_jdot
         type(binary_info), pointer :: b
         type (star_info), pointer :: s
         
         real(dp) :: grav, rho, P
         
         include 'formats.inc'

         s => b% s_donor

         if (b% evolve_both_stars) then
            b% companion_mass = b% s_accretor% mstar/Msun
         else
            b% companion_mass = b% companion_mass &
               - b% xfer_fraction*b% mtransfer_rate*b% s_donor% dt/Msun
         end if
         
         b% r1 = s% photosphere_r*Rsun ! radius at photosphere in cm
         if (b% evolve_both_stars) b% r2 = b% s_accretor% photosphere_r*Rsun ! radius at photosphere in cm
         b% m2 = Msun*s% star_mass ! mass of secondary in gm
         b% m1 = Msun*b% companion_mass ! mass of primary in gm         
         rho = s% rho(1) ! density at surface in g/cm^3
         p = s% p(1) ! pressure at surface in dynes/cm^2
         grav = s% cgrav(1)*b% m2/(b% r1*b% r1) ! local gravitational acceleration
         b% hp = p/(grav*rho) ! pressure scale height
         b% v_th = sqrt(3.0 * kerg * s% T(1) / (mp * s% mu(1)))
         
         ! get_jdot needs an estimate for rl; use prev binary separation for this
         b% rl1 = eval_rlobe(b% m2, b% m1, b% separation)
         b% rl2 = eval_rlobe(b% m1, b% m2, b% separation)

         b% jdot = get_jdot(b% mtransfer_rate, b% xfer_fraction)
         
         b% angular_momentum_j = b% angular_momentum_j + b% jdot*s% time_step*secyer

         !TODO:CHECK THIS!!
         !if (b% angular_momentum_j < 0) then
         !   write(*,1) 'bad angular_momentum_j', b% angular_momentum_j
         !   ierr = -1
         !   return
         !end if
         
         ! use the new j to calculate new separation
         
         b% separation = ((b% angular_momentum_j/(b% m1*b% m2))**2) * (b% m1+b% m2) / s% cgrav(1)
         if (b% separation < b% min_binary_separation) &
            b% min_binary_separation = b% separation
         
         b% period = 2*pi*sqrt(b% separation**3/(s% cgrav(1)*(b% m1+b% m2))) 
         if (b% period < min_binary_period) min_binary_period = b% period
         
         ! use the new separation to calculate the new roche lobe radius
         
         b% rl1 = eval_rlobe(b% m2, b% m1, b% separation)
         if (b% evolve_both_stars) b% rl2 = eval_rlobe(b% m1, b% m2, b% separation)
         b% rl_relative_gap = (b% rl1 - b% r1)/b% r1 ! gap > 0 means out of contact

         !TODO:CHECK THIS!!
         !if (is_bad_num(b% rl_relative_gap)) then
         !   ierr = -1
         !   return
         !   !stop 'get_info_for_rlo_mdot'
         !end if

         !check things
         !write(*,*) "check things: dt: ", b% s_donor% dt/secyer
         !write(*,*) "period:", b% period/(24*3600), b% period_old/(24*3600), b% period_older/(24*3600)
         !write(*,*) "sep:", b% separation/(rsun), b% separation_old/(rsun), b% separation_older/(rsun)
         !write(*,*) "am:", b% angular_momentum_j, b% angular_momentum_j_old, b% angular_momentum_j_older
         !write(*,*) "m1:", b% m1/(msun)
         !write(*,*) "m2:", b% m2/(msun)
         !write(*,*) "sum of masses:", (b% m2+b% m1)/(msun)
         !write(*,*) "r1:", b% r1/(rsun)
         !write(*,*) "r2:", b% r2/(rsun)
         !write(*,*) "rl1:", b% rl1/(rsun)
         !write(*,*) "rl2:", b% rl2/(rsun)

      end subroutine

      integer function binary_check_model(b)
         use run_star_support, only: pgstar_flag
         use binary_mdot, only: rlo_mdot, check_implicit_rlo
         use binary_irradiation
         type (binary_info), pointer :: b

         integer :: i, j, ierr
         logical :: implicit_rlo
         real(dp) :: new_mdot


         include 'formats.inc'

         binary_check_model = retry
         ierr = 0
! BP: what's a call on pgstar doing in check_model?    already done in run_binary_support.     
!         if (pgstar_flag) call update_pgstar_plots( &
!               b% s_donor, .false., 0, &
!               b% how_many_extra_history_columns, &
!               b% data_for_extra_history_columns, &
!               b% how_many_extra_profile_columns, &
!               b% data_for_extra_profile_columns, &
!               ierr)
!         if (ierr /= 0) return
         
         implicit_rlo = (b% max_tries_to_achieve > 0 .and. b% rl_rel_overlap_tolerance > 0d0)
         
         binary_check_model = keep_going
                  
         if (.not. b% started_rlo) then               
            b% started_rlo = .true.
            b% num_tries = 0
         else if (.not. just_evolve) then
            if (ierr /= 0) then
               b% num_tries = 0
               binary_check_model = retry
               return
            end if
            if (implicit_rlo) then ! check agreement between new r and new rl
               b% s_donor% min_abs_mdot_for_change_limits = 1d99
               binary_check_model = check_implicit_rlo(new_mdot)
               if (binary_check_model == keep_going) then
                  b% num_tries = 0
               end if
            else
               new_mdot = rlo_mdot(b% s_donor) ! grams per second
               ! smooth out the changes in mdot
               new_mdot = b% cur_mdot_frac*b% mtransfer_rate + (1-b% cur_mdot_frac)*new_mdot
               if (-new_mdot/(Msun/secyer) > b% max_abs_mdot) new_mdot = -b% max_abs_mdot*Msun/secyer 
            end if
            b% mtransfer_rate = new_mdot
            call adjust_irradiation(b% s_donor, b% mtransfer_rate, b% xfer_fraction)
         end if
         
         if (b% period/(60d0*60d0) < b% lower_limit_on_period_in_hours) then
            binary_check_model = terminate
            write(*,*) 'terminate because binary period < lower_limit_on_period_in_hours'
         end if
      
         if (b% period/(60d0*60d0) > b% upper_limit_on_period_in_hours) then
            binary_check_model = terminate
            write(*,*) 'terminate because binary period > upper_limit_on_period_in_hours'
         end if

      end function binary_check_model

      integer function binary_finish_step(b)
         type (binary_info), pointer :: b

         binary_finish_step = keep_going
         ! update change factor in case mtransfer_rate has changed
         if(b% mtransfer_rate_old /= b% mtransfer_rate .and. &
             b% mtransfer_rate /= 0 .and. b% mtransfer_rate_old/=0.0) then
            if(b% mtransfer_rate < b% mtransfer_rate_old) then
               b% change_factor = b% change_factor*(1.0-b% implicit_lambda) + b% implicit_lambda* &
                  (1+b% change_factor_fraction*(b% mtransfer_rate/b% mtransfer_rate_old-1))
            else
               b% change_factor = b% change_factor*(1.0-b% implicit_lambda) + b% implicit_lambda* &
                   (1+b% change_factor_fraction*(b% mtransfer_rate_old/b% mtransfer_rate-1))
            end if
            if(b% change_factor > b% max_change_factor) b% change_factor = b% max_change_factor
            if(b% change_factor < b% min_change_factor) b% change_factor = b% min_change_factor
         end if

         ! store all variables into "old" and "older"
         b% mtransfer_rate_older = b% mtransfer_rate_old
         b% angular_momentum_j_older = b% angular_momentum_j_old
         b% separation_older = b% separation_old
         b% rl_relative_gap_older = b% rl_relative_gap_old
         b% dt_older = b% dt_old
         b% env_older = b% env_old
         b% xfer_fraction_older = b% xfer_fraction_old
         b% sum_div_qloc_older = b% sum_div_qloc_old
         b% period_older = b% period_old
         b% companion_mass_older = b% companion_mass_old
         b% started_rlo_older = b% started_rlo_old
         b% have_radiative_core_older = b% have_radiative_core_old

         b% mtransfer_rate_old = b% mtransfer_rate
         b% angular_momentum_j_old = b% angular_momentum_j
         b% separation_old = b% separation
         b% rl_relative_gap_old = b% rl_relative_gap
         b% dt_old = b% dt
         b% env_old = b% env
         b% xfer_fraction_old = b% xfer_fraction
         b% sum_div_qloc_old = b% sum_div_qloc
         b% period_old = b% period
         b% companion_mass_old = b% companion_mass
         b% started_rlo_old = b% started_rlo
         b% have_radiative_core_old = b% have_radiative_core

      end function binary_finish_step

      integer function binary_prepare_to_redo(b)
         type (binary_info), pointer :: b

         binary_prepare_to_redo = redo
         ! restore variables
         ! do not restore mtransfer_rate during implicit rlo
         if (b% num_tries == 0) b% mtransfer_rate = b% mtransfer_rate_old
         b% angular_momentum_j = b% angular_momentum_j_old
         b% separation = b% separation_old
         b% rl_relative_gap = b% rl_relative_gap_old
         b% dt = b% dt_old
         b% env = b% env_old
         b% xfer_fraction = b% xfer_fraction_old
         b% sum_div_qloc = b% sum_div_qloc_old
         b% period = b% period_old
         b% companion_mass = b% companion_mass_old
         b% started_rlo = b% started_rlo_old
         b% have_radiative_core = b% have_radiative_core_old

      end function binary_prepare_to_redo

      integer function binary_prepare_to_retry(b)
         type (binary_info), pointer :: b

         binary_prepare_to_retry = retry
         ! restore variables
         b% mtransfer_rate = b% mtransfer_rate_old
         b% angular_momentum_j = b% angular_momentum_j_old
         b% separation = b% separation_old
         b% rl_relative_gap = b% rl_relative_gap_old
         b% dt = b% dt_old
         b% env = b% env_old
         b% xfer_fraction = b% xfer_fraction_old
         b% sum_div_qloc = b% sum_div_qloc_old
         b% period = b% period_old
         b% companion_mass = b% companion_mass_old
         b% started_rlo = b% started_rlo_old
         b% have_radiative_core = b% have_radiative_core_old

      end function binary_prepare_to_retry

      integer function binary_do1_backup(b)
         type (binary_info), pointer :: b

         binary_do1_backup = retry
         ! restore variables
         b% mtransfer_rate = b% mtransfer_rate_older
         b% angular_momentum_j = b% angular_momentum_j_older
         b% separation = b% separation_older
         b% rl_relative_gap = b% rl_relative_gap_older
         b% dt = b% dt_older
         b% env = b% env_older
         b% xfer_fraction = b% xfer_fraction_older
         b% sum_div_qloc = b% sum_div_qloc_older
         b% period = b% period_older
         b% companion_mass = b% companion_mass_older
         b% started_rlo = b% started_rlo_older
         b% have_radiative_core = b% have_radiative_core_older

         b% mtransfer_rate_old = b% mtransfer_rate_older
         b% angular_momentum_j_old = b% angular_momentum_j_older
         b% separation_old = b% separation_older
         b% rl_relative_gap_old = b% rl_relative_gap_older
         b% dt_old = b% dt_older
         b% env_old = b% env_older
         b% xfer_fraction_old = b% xfer_fraction_older
         b% sum_div_qloc_old = b% sum_div_qloc_older
         b% period_old = b% period_older
         b% companion_mass_old = b% companion_mass_older
         b% started_rlo_old = b% started_rlo_older
         b% have_radiative_core_old = b% have_radiative_core_older

      end function binary_do1_backup

      real(dp) function eval_rlobe(m1, m2, a) result(rlobe)
         real(dp), intent(in) :: m1, m2, a
         real(dp) :: q
         q = (m1/m2)**one_third
      ! Roche lobe size for star of mass m1 with a
      ! companion of mass m2 at separation a, according to
      ! the approximation of Eggleton 1983, apj 268:368-369
         rlobe = a*0.49d0*q*q/(0.6d0*q*q + log(1d0 + q))
      end function eval_rlobe

      end module binary_evolve
