      module test_atm_support
      use const_def
      use atm_def
      use atm_lib
      use chem_lib
      use chem_def
      use eos_def
      use eos_lib
      use kap_def
      use kap_lib

      implicit none

      logical :: test_verbosely
      integer :: eos_handle, kap_handle
      real*8 :: cgrav
      ! composition info
      integer, parameter :: species = 7
      integer, pointer :: chem_id(:), net_iso(:)
      real*8 :: X, Y, Z, XC, XN, XO, xa(species), abar, zbar
      
      
      integer :: ierr, max_tries, iters
      real*8 :: logg, Teff, M, R, L, kap_guess, tau, tau_base, tau_phot, &
         lnT, dlnT_dL, dlnT_dlnR, dlnT_dlnM, & 
         lnP, dlnP_dL, dlnP_dlnR, dlnP_dlnM, &
         atol, rtol, kap, T, err, P, Prad, &
         dlnT_dlnkap, dlnP_dlnkap, &
         logTeff, log_gsurf, log_Rsurf, log_M, g, opacity, &
         T_eq, kap_v, kap_v_div_kap_th, T_int
      logical :: use_kap_th_guess

      contains
      
      
      subroutine do_test_atm

         call setup
         call Do_One_Atm_Table(atm_tau_1m1_tables)        
         call Do_One_Atm_Table(atm_tau_1_tables)         
         call Do_One_Atm_Table(atm_tau_10_tables)              
         call Do_One_Atm_Table(atm_tau_100_tables)             

         call Do_One_Atm_Table(atm_wd_tau_25_tables)         
         call Do_One_Atm_Table(atm_photosphere_tables)      
         
         call Do_One_Atm_Int(atm_Eddington_grey, -1d0)
         call Do_One_Atm_Int(atm_Krishna_Swamy, -1d0)
         call Do_One_Atm_Int(atm_solar_Hopf_grey, -1d0)
         call Do_One_Atm_Int(atm_Eddington_grey, 100d0)
         
         call Do_One_Grey      
         call Do_One_Grey_and_Kap(150d0)
         call Do_One_Grey_Irradiated
         
      end subroutine do_test_atm


      subroutine Do_One_Atm_Table(which_atm_option)
         integer, intent(in) :: which_atm_option
         integer :: off_table_option
         
         include 'formats.dek'

         ierr = 0

         if (test_verbosely) then
            write(*,*)
            call show_option(which_atm_option, 'Do_One_Atm_Table')
         endif

         if (which_atm_option == atm_wd_tau_25_tables) then
            logg = 7.1d0
            Teff = 5000
            M = 0.8*Msun
            R = sqrt(cgrav*M / 10**logg)
            !write(*,*) 'R/Rsun', R/Rsun
            L = pi*crad*clight*R**2*Teff**4
            !write(*,*) 'L/Lsun', L/Lsun
         else
            M = Msun
            R = Rsun
            L = Lsun
         end if
         
         Z = 0.02
         X = 0.70
         XC = 3.2724592105263235d-03
         XN = 9.5023842105263292d-04
         XO = 8.8218000000000601d-03
         call set_composition
         
         kap_guess = 0.5d0
         atol = 1d-6
         rtol = 1d-4
         max_tries = 30
         
         logg = log10(cgrav * M / R**2)
         if (ierr /= 0) then
            if (test_verbosely) write(*,*) 'failed in atm_tau_base'
            stop 1
         end if
         
         off_table_option = atm_grey_and_kap
         
         call atm_get_table( &
            which_atm_option, off_table_option, cgrav, M, R, L, &
            lnT, dlnT_dL, dlnT_dlnR, dlnT_dlnM, &
            lnP, dlnP_dL, dlnP_dlnR, dlnP_dlnM, &
            kap_guess, X, Z, abar, zbar, &
            species, chem_id, net_iso, xa, &
            max_tries, atol, rtol, eos_handle, kap_handle, &
            Teff, ierr) 
         if (ierr /= 0) then
            if (test_verbosely) write(*,*) 'failed in atm_get'
            stop 1
         end if
         
         T = exp(lnT)
         P = exp(lnP)
         
         Prad = crad*T**4/3

         if (test_verbosely) write(*,*)
         if (test_verbosely) write(*,'(99a16)') &
            'T', 'log_T', 'log P', 'M/Msun', 'L/Lsun', 'R/Rsun', 'logPgas'
         if (test_verbosely) write(*,'(i16,99f16.8)') &
            floor(0.5d0 + T), lnT/ln10, log10(P), M/Msun, L/Lsun, R/Rsun, log10(P-Prad)
         if (test_verbosely) write(*,*)
         
      end subroutine Do_One_Atm_Table


      subroutine Do_One_Atm_Int(which_atm_option, tau_base_in)
         
         integer, intent(in) :: which_atm_option
         double precision, intent(in) :: tau_base_in
         logical :: save_atm_structure_info
         integer :: num_atm_structure_points
         real*8, pointer :: atm_structure_results(:,:) ! will be allocated if necessary
         logical, parameter :: skip_partials = .true.
         real*8 :: errtol
         
         include 'formats.dek'
         
         ierr = 0
         tau_base = tau_base_in
         if (tau_base <= 0) then
            tau_base = atm_tau_base(which_atm_option, ierr)
            if (ierr /= 0) stop 1
         end if

         if (test_verbosely) then
            write(*,*)
            call show_option(which_atm_option, 'Do_One_Atm_Int')
         endif
         
         ! TEST SOLAR VALUES
         logTeff = log10(5776d0)
         log_gsurf = log10(cgrav*Msun/Rsun**2)
         log_Rsurf = log10(Rsun)
         log_M = log10(Msun)
         
         Z = 0.02
         X = 0.70
         XC = 3.2724592105263235d-03
         XN = 9.5023842105263292d-04
         XO = 8.8218000000000601d-03
         call set_composition
         
         R = 10**log_Rsurf
         M = 10**log_M
         
         g = 10**log_gsurf
         Teff = 10**logTeff
         L = pi*crad*clight*R**2*Teff**4
         
         save_atm_structure_info = .false.
         errtol = 1d-9
		   
         call atm_get_int_T_tau( &
            errtol, cgrav, M, R, L, X, Z, abar, zbar,  &
            species, chem_id, net_iso, xa, &
            which_atm_option, eos_handle, kap_handle, save_atm_structure_info, &
            tau_base, skip_partials, Teff, &
            lnT, dlnT_dL, dlnT_dlnR, dlnT_dlnM, dlnT_dlnkap, & 
            lnP, dlnP_dL, dlnP_dlnR, dlnP_dlnM, dlnP_dlnkap, &
            num_atm_structure_points, atm_structure_results, &
            ierr) 
         if (ierr /= 0) then
            if (test_verbosely) write(*,*) 'bad return from atm_get'
            stop 1
         end if
         
         T = exp(lnT)
         P = exp(lnP)
         
         if (test_verbosely) write(*,*)
         if (test_verbosely) write(*,'(99a16)') 'T', 'log T', 'log P', 'M/Msun', 'L/Lsun', 'X', 'Z'
         if (test_verbosely) write(*,'(i16,99f16.8)') floor(0.5d0 + T), log10(T), log10(P), M/Msun, L/Lsun, X, Z
         if (test_verbosely) write(*,*)
         
      end subroutine Do_One_Atm_Int
      
      
      subroutine Do_One_Grey         
         include 'formats.dek'
            
         if (test_verbosely) then
            write(*,*)
            write(*,*) 'Do_One_Grey'
         endif

         tau_phot = 1.0000000000000000D+02

         M = 1.9892000000000002D+32
         R = 6.3556231577545586D+10
         L = 2.4015399190199118D+32
         opacity = 5.8850802481174469D-02
         
         call atm_get_grey( &
            tau_phot, cgrav, M, R, L, opacity, Teff, &
            lnT, dlnT_dL, dlnT_dlnR, dlnT_dlnM, dlnT_dlnkap, & 
            lnP, dlnP_dL, dlnP_dlnR, dlnP_dlnM, dlnP_dlnkap, &
            ierr)
         if (ierr /= 0) then
            if (test_verbosely) write(*,*) 'failed in atm_get_grey'
            stop 1
         end if
         if (test_verbosely) write(*,1) 'atm_get_grey logP surf', lnP/ln10
         if (test_verbosely) write(*,1) 'atm_get_grey logT surf', lnT/ln10
         if (test_verbosely) write(*,1) 'atm_get_grey tau_phot', tau_phot
         if (test_verbosely) write(*,*)
      
      end subroutine Do_One_Grey


      subroutine Do_One_Grey_and_Kap(tau_base_in)
         double precision, intent(in) :: tau_base_in
         
         include 'formats.dek'
         
         if (test_verbosely) write(*,*)
         ierr = 0
         tau_base = tau_base_in
         
         logTeff = 3.6284699999999992d0
         log_gsurf =  2.4136404467857595d0
         log_Rsurf = 1.1854708077307119d1
         log_M = 3.3298656601399998E+01
         
         ! TEST SOLAR VALUES
         logTeff = log10(5776d0)
         log_gsurf = log10(cgrav*Msun/Rsun**2)
         log_Rsurf = log10(Rsun)
         log_M = log10(Msun)
         
         Z = 0.02
         X = 0.70
         XC = 3.2724592105263235d-03
         XN = 9.5023842105263292d-04
         XO = 8.8218000000000601d-03
         call set_composition
         
         R = 10**log_Rsurf
         M = 10**log_M
         Teff = 10**logTeff
         L = pi*crad*clight*R**2*Teff**4
         
         kap_guess = 0.5d0
         atol = 1d-6
         rtol = 1d-4
         max_tries = 30
		   
         call atm_get_grey_and_kap( &
            tau_base, kap_guess, &
            cgrav, M, R, L, X, Z, abar, zbar, & 
            species, chem_id, net_iso, xa, &
            max_tries, atol, rtol, eos_handle, kap_handle, &
            lnT, dlnT_dL, dlnT_dlnR, dlnT_dlnM, & 
            lnP, dlnP_dL, dlnP_dlnR, dlnP_dlnM, & 
            kap, Teff, iters, err, ierr) 
         if (ierr /= 0) then
            if (test_verbosely) write(*,*) 'bad return from atm_get_grey_and_kap'
            stop 1
         end if
         
         if (test_verbosely) write(*,1) 'Do_One_Grey_and_Kap: kap', kap
         if (test_verbosely) write(*,1) 'tau', tau_base
         if (test_verbosely) write(*,1) 'logT surf', lnT/ln10
         if (test_verbosely) write(*,1) 'logP surf', lnP/ln10
         if (test_verbosely) write(*,1) 'logkap', log10(kap)
         if (test_verbosely) write(*,*)
         
         deallocate(chem_id, net_iso)
         
      end subroutine Do_One_Grey_and_Kap


      subroutine Do_One_Grey_Irradiated
         
         include 'formats.dek'
         
         if (test_verbosely) write(*,*)
         ierr = 0

         X = 0.70d0
         Z = 1d-2
         XC = 3.2724592105263235d-03
         XN = 9.5023842105263292d-04
         XO = 8.8218000000000601d-03
         
         call set_composition
         
         atol = 1d-6
         rtol = 1d-4
         max_tries = 30
         
         T_eq = 1000
         kap_v = 4d-3
         kap_guess = 1.5d-2
         use_kap_th_guess = .false.
         kap_v_div_kap_th = 0
         P = 1d6
         M = 1.5*M_jupiter
         tau = 10 ! just a guess for use in getting R
         R = sqrt(cgrav*M*tau/(P*kap_guess)) ! g = P*kap/tau = G*M/R^2
         T_int = 900
         L = pi*crad*clight*R**2*T_int**4
		   
         call atm_grey_irradiated_get( &
            T_eq, kap_v, kap_guess, use_kap_th_guess, kap_v_div_kap_th, &
            P, cgrav, M, R, L, &
            X, Z, abar, zbar, species, chem_id, net_iso, xa, &
            max_tries, atol, rtol, eos_handle, kap_handle, &
            lnT, dlnT_dL, dlnT_dlnR, dlnT_dlnM, & 
            kap, tau, Teff, iters, err, ierr) 
         if (ierr /= 0) then
            if (test_verbosely) write(*,*) 'bad return from atm_get_grey_and_kap'
            stop 1
         end if
         
         if (test_verbosely) write(*,*) 
         if (test_verbosely) write(*,1) 'Do_One_Grey_Irradiated: kap', kap
         if (test_verbosely) write(*,1) 'M/M_jupiter', M/M_jupiter
         if (test_verbosely) write(*,1) 'R/R_jupiter', R/R_jupiter
         if (test_verbosely) write(*,1) 'R/Rsun', R/Rsun
         if (test_verbosely) write(*,1) 'kap_v', kap_v
         if (test_verbosely) write(*,1) 'P', P
         if (test_verbosely) write(*,*) 

         if (test_verbosely) write(*,1) 'tau', tau
         if (test_verbosely) write(*,1) 'Teff', Teff
         if (test_verbosely) write(*,1) 'T_eq', T_eq
         if (test_verbosely) write(*,1) 'T_int', T_int
         if (test_verbosely) write(*,1) 'T', exp(lnT)
         if (test_verbosely) write(*,1) 'logT', lnT/ln10
         if (test_verbosely) write(*,*)         
         
      end subroutine Do_One_Grey_Irradiated
      
      
      subroutine setup
         use const_lib
         logical, parameter :: use_cache = .true.
   	   character(len=256) :: my_mesa_dir
      	ierr = 0
         my_mesa_dir = '../..'         
         call const_init(my_mesa_dir,ierr)     
      	if (ierr /= 0) then
      	   write(*,*) 'const_init failed'
      	   stop 1
      	end if        
         cgrav = standard_cgrav
      	call chem_init('isotopes.data', ierr)
      	if (ierr /= 0) then
      	   write(*,*) 'chem_init failed'
      	   stop 1
      	end if        
         call eos_init('mesa', '', '', use_cache, ierr)
         if (ierr /= 0) stop 1         
         eos_handle = alloc_eos_handle(ierr)
         if (ierr /= 0) stop 2        
         call kap_init( &
            'gs98', 'gn93_co', 'lowT_fa05_gs98', &
            0d0, 0d0, 0d0, use_cache, '', ierr)
         if (ierr /= 0) stop 3         
         kap_handle = alloc_kap_handle(ierr)
         if (ierr /= 0) stop 4        
         call atm_init(.false., ierr)
         if (ierr /= 0) then
            if (test_verbosely) write(*,*) 'bad return from atm_init'
            stop 1
         end if         
      end subroutine setup


      subroutine set_composition
         real*8 :: z2bar, ye, mass_correction, sumx, &
            dabar_dx(species), dzbar_dx(species), dmc_dx(species)
         integer :: i
         Y = 1-(X+Z)
      	allocate(chem_id(species), net_iso(num_chem_isos), stat=ierr)
      	if (ierr /= 0) then
      	   write(*,*) 'allocate failed'
      	   stop 1
      	end if
         chem_id(:) = (/ ih1, ihe4, ic12, in14, io16, ine20, img24 /)
         net_iso(:) = 0
         forall (i=1:species) net_iso(chem_id(i)) = i
         xa(:) = (/ X, Y, xc, xn, xo, 0d0, 0d0 /)
         xa(species) = 1 - sum(xa(:))
         call composition_info( &
            species, chem_id, xa, X, Y, abar, zbar, z2bar, ye, &
            mass_correction, sumx, dabar_dx, dzbar_dx, dmc_dx)
      end subroutine set_composition
      
      
      subroutine show_option(which_atm_option, tst_str)
         integer, intent(in) :: which_atm_option
         character (len=*), intent(in) :: tst_str
         character (len=256) :: option_str
         integer :: ierr
         call atm_option_str(which_atm_option, option_str, ierr)
         if (ierr /= 0) stop 1
         write(*,*) trim(tst_str) // ': ' // trim(option_str)
      end subroutine show_option



      end module




