
! ***********************************************************************
!
!   Copyright (C) 2010  Aaron Dotter
!
!   This file is part of MESA.
!
!   MESA is free software; you can redistribute it and/or modify
!   it under the terms of the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License, or
!   (at your option) any later version.
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!   GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      program create_table_atm

         use atm_def
         use atm_lib
         use chem_def
         use chem_lib
         use const_def
         use const_lib
         use eos_def
         use eos_lib
         use kap_lib
         use utils_lib
         implicit none
         integer :: eos_handle, kap_handle, io_out
         integer, parameter :: num_isos = 7, nmet = 5, num_logg = 13, num_Teff = 80
         integer, parameter :: which_atm_option = atm_Eddington_grey

         integer, pointer :: chem_id(:), net_iso(:)
         double precision, pointer :: xa(:)

         integer :: ierr, i_Teff, i_logg
         character(len=256) :: data_dir, clogZ, output_file, ctau_base
         logical, parameter :: use_cache = .true.
         logical, parameter :: skip_partials = .true.
         double precision :: M, R, L, X, Y, Z, XC, XN, XO, XNe, XMg, abar, zbar, z2bar, kap, err
         double precision :: Teff, lnP, lnT, tau_base, Teff_out, ye, sumx
         double precision :: dabar_dx(num_isos), dzbar_dx(num_isos), Xsun, Ysun, Zsun, logZ
         double precision :: dlnT_dL, dlnT_dlnR, dlnT_dlnM, dlnT_dlnkap
         double precision :: dlnP_dL, dlnP_dlnR, dlnP_dlnm, dlnP_dlnkap, Xbbn, Ybbn
         double precision :: logg_array(num_logg), Teff_array(num_Teff)
         double precision :: Pgas(num_logg, num_Teff), T(num_logg, num_Teff)
         integer, parameter :: max_tries = 100
         double precision, parameter :: atol = 1d-6, rtol = 1d-4
         integer :: iters

         !process command line args
         if( iargc() /= 3 ) then
            stop 'usage: ./create_table_atm [logZ/Zsolar] [output file] [tau_base or 0]'
         endif

         call getarg(1,clogZ)
         read(clogZ,*) logZ

         call getarg(2,output_file)

         !mesa initialization
         ierr = 0
         data_dir = '../../data'

         call mesa_init

         call set_table_composition
         
         call getarg(3,ctau_base)
         read(ctau_base,*) tau_base

         if ( tau_base <= 0 ) tau_base = atm_tau_base(which_atm_option, ierr)
         if (ierr /= 0) stop 1

         !for table creation:
         M = Msun

         logg_array(:) = (/ -0.5d0, 0d0, 0.5d0, 1d0, 1.5d0, 2d0, 2.5d0, 3d0, 3.5d0, 4d0, 4.5d0, 5d0, 5.5d0 /)

         Teff_array(1) = 2.5d3
         do i_Teff = 2, num_Teff
            if( Teff_array(i_Teff-1) < 1.3d4) Teff_array(i_Teff) = Teff_array(i_Teff-1) + 2.5d2
            if( Teff_array(i_Teff-1) >=1.3d4) Teff_array(i_Teff) = Teff_array(i_Teff-1) + 1d3
         enddo
         
         write(*,*) trim(output_file)
         
         do i_Teff = 1, num_Teff
            do i_logg = 1, num_logg
               R = sqrt ( standard_cgrav*M / 10d0**logg_array(i_logg) )               
               L = pi*crad*clight * R**2 * Teff_array(i_Teff)**4
               ierr = 0
               call atm_get_grey_and_kap( &
                     tau_base, 0.2d0*(1 + X), &
                     standard_cgrav, M, R, L, X, Z, abar, zbar,  &
                     num_isos, chem_id, net_iso, xa, &
                     max_tries, atol, rtol, eos_handle, kap_handle, &
                     lnT, dlnT_dL, dlnT_dlnR, dlnT_dlnM,  &
                     lnP, dlnP_dL, dlnP_dlnR, dlnP_dlnM,  &
                     kap, Teff_out, iters, err, ierr) 
               if (ierr /= 0) then
                  Pgas(i_logg, i_Teff) = -1
                  T(i_logg, i_Teff) = -1
                  write(*,*) 'failed in integration', logg_array(i_logg), Teff_array(i_Teff)
               else           
                  T(i_logg, i_Teff) = exp(lnT)
                  Pgas(i_logg, i_Teff) = max( 0d0, exp(lnP) - Radiation_Pressure(T(i_logg, i_Teff)) )
               end if
            enddo

         enddo
         
         
         open(io_out, file=output_file)

         write(io_out,'("#Teff(K)| Pgas@",13("  log g =",f5.2,1x))') logg_array(1:num_logg)           
         do i_Teff = 1, num_Teff
            write(io_out,'(1p,14e15.7)') Teff_array(i_Teff), Pgas(1:num_logg, i_Teff)
         enddo

         write(io_out,'("#Teff(K)|    T@",13("  log g =",f5.2,1x))') logg_array(1:num_logg)           
         do i_Teff = 1, num_Teff
            write(io_out,'(1p,14e15.7)') Teff_array(i_Teff), T(1:num_logg, i_Teff)
         enddo

         close (io_out)

         call mesa_shutdown

         contains

         subroutine mesa_init
            character(len=256) :: my_mesa_dir = '../..'
            character(len=256) :: kappa_file_prefix = 'gn93'
            character (len=256) :: kappa_CO_prefix = 'gn93_co'
            character (len=256) :: kappa_lowT_prefix = 'lowT_fa05_gs98'
            real(dp) :: kappa_blend_logT_upper_bdy = 4.1d0
            real(dp) :: kappa_blend_logT_lower_bdy = 3.93d0
            real(dp) :: kappa_type2_logT_lower_bdy = 3.80d0

            eos_handle = alloc_eos_handle(ierr)
            if (ierr /= 0) stop 2
            
            call kap_init( &
               kappa_file_prefix, kappa_CO_prefix, kappa_lowT_prefix, &
               kappa_type2_logT_lower_bdy, kappa_blend_logT_upper_bdy, kappa_blend_logT_lower_bdy, &
               use_cache, ierr) 
            
            kap_handle = alloc_kap_handle(ierr)
            if (ierr /= 0) stop 4
            
            call atm_init(.false., ierr)
            if (ierr /= 0) stop 5

            io_out = alloc_iounit(ierr)
            if (ierr /= 0) stop 6

         end subroutine mesa_init


         subroutine mesa_shutdown
            call eos_shutdown
            call free_eos_handle(eos_handle)
            call kap_shutdown
            call free_kap_handle(kap_handle)
            call atm_shutdown
            call free_iounit(io_out)
         end subroutine mesa_shutdown


         subroutine set_table_composition
            integer :: i
            real*8 :: xh, xhe, mass_correction, approx_abar, approx_zbar

            Xbbn   = 0.75d0
            Ybbn   = 0.25d0
            
            Xsun   = 7.348d-1
            Ysun   = 2.348d-1
            Zsun   = 1.685d-2

            Z = exp(ln10*logZ)*Zsun
            Y = 1.5d0*Z + Ybbn
            X = 1d0 - (Y+Z)

            XC  = 0.1721d0*Z
            XN  = 0.0504d0*Z
            XO  = 0.4680d0*Z
            XNe = 0.1050d0*Z
            XMg = 0.2450d0*Z
         
               
            allocate(xa(num_isos), chem_id(num_isos), net_iso(num_chem_isos))
      
            chem_id(:) = (/ ih1, ihe4, ic12, in14, io16, ine20, img24 /)
            net_iso(:) = 0
            do i=1,num_isos
               net_iso(chem_id(i)) = i
            end do
            xa(:) = (/ X, Y, xc, xn, xo, xne, xmg /)
            xa(num_isos) = 1 - sum(xa(:))
            call basic_composition_info( &
               num_isos, chem_id, xa, xh, xhe, abar, zbar, z2bar, ye, &
               mass_correction, sumx)
               
         end subroutine set_table_composition

         end program create_table_atm
