! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module utils_lib
      
      use utils_def, only: max_io_unit
      use alert_lib
      use const_def, only: dp, qp
      
      implicit none

      logical :: assigned(max_io_unit) = .false.
      
      logical, parameter :: test_with_nans = .false.

      CHARACTER( * ), PRIVATE, PARAMETER :: LOWER_CASE = 'abcdefghijklmnopqrstuvwxyz'
      CHARACTER( * ), PRIVATE, PARAMETER :: UPPER_CASE = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ' 
   
      contains
      
      
      logical function is_bad_quad(x)
         use utils_isnan,only:check_for_bad_quad
         real(qp), intent(in) :: x
         is_bad_quad = check_for_bad_quad(x)
      end function is_bad_quad
      
      
      logical function is_bad_num(x)
         use utils_isnan,only:check_for_bad_num
         real(dp), intent(in) :: x
         is_bad_num = check_for_bad_num(x)
      end function is_bad_num
      
      
      logical function has_bad_num(n,x)
         integer, intent(in) :: n
         real(dp), intent(in) :: x(n)
         integer :: k
         has_bad_num = .true.
         do k=1,n
            if (is_bad_num(x(k))) return
         end do
         has_bad_num = .false.
      end function has_bad_num
      
      
      logical function is_bad_real(x)
         use utils_isnan,only:check_for_bad_real
         real, intent(in) :: x
         is_bad_real = check_for_bad_real(x)
      end function is_bad_real
      
      
      logical function has_bad_real(n,x)
         integer, intent(in) :: n
         real, intent(in) :: x(n)
         integer :: k
         has_bad_real = .true.
         do k=1,n
            if (is_bad_real(x(k))) return
         end do
         has_bad_real = .false.
      end function has_bad_real
      
      
      integer function alloc_iounit(ierr)
         use utils_def
         integer, intent(out) :: ierr
         integer :: i
!$omp critical (utils_alloc_io_unit)
         ierr = 0
         alloc_iounit = -1
         do i = min_io_unit, max_io_unit
            if (.not. assigned(i)) then
               assigned(i) = .true.
               alloc_iounit = i
               exit
            end if
         end do
!$omp end critical (utils_alloc_io_unit)
         if (alloc_iounit == -1) then
            ierr = -1
            call alert(ierr, 'no available iounit numbers')
         end if
      end function alloc_iounit
      
      
      integer function number_iounits_allocated()
         use utils_def
         integer :: i, cnt
         cnt = 0
!$omp critical (utils_alloc_io_unit)
         do i = min_io_unit, max_io_unit
            if (assigned(i)) cnt = cnt + 1
         end do
!$omp end critical (utils_alloc_io_unit)
         number_iounits_allocated = cnt
      end function number_iounits_allocated
      
      
      subroutine free_iounit(iounit)
         use utils_def
         integer, intent(in) :: iounit
         logical :: bad_iounit
         bad_iounit = .false.
!$omp critical (utils_alloc_io_unit)
         if (iounit >= min_io_unit .and. iounit <= max_io_unit) then
            assigned(iounit) = .false.
         else
            bad_iounit = .true.
         end if
!$omp end critical (utils_alloc_io_unit)
         if (bad_iounit) then
            write(*,*) 'called free_iounit with invalid arg', iounit
            stop 'free_iounit'
         end if
      end subroutine free_iounit      

      
      subroutine append_line(n, arry, filename, format_str, initialize, ierr)
         integer, intent(in) :: n
         real(dp), intent(in) :: arry(n)
         character (len=256), intent(in) :: filename, format_str
         logical, intent(in) :: initialize
         integer, intent(out) :: ierr
         integer :: iounit
         ierr = 0
         iounit = alloc_iounit(ierr); if (ierr /= 0) return
         if (initialize) then
            open(unit=iounit, file=trim(filename), action='write', iostat=ierr)
         else
            open(unit=iounit, file=trim(filename), position='append', action='write', iostat=ierr)
         end if
         if (ierr /= 0) then
            call free_iounit(iounit)
            return
         end if
         write(unit=iounit, fmt=format_str) arry
         close(iounit)
         call free_iounit(iounit)
      end subroutine append_line

      
      subroutine append_data(n, arry, filename, format_str, initialize, ierr)
         integer, intent(in) :: n
         real(dp), intent(in) :: arry(n)
         character (len=256), intent(in) :: filename, format_str
         logical, intent(in) :: initialize
         integer, intent(out) :: ierr
         integer :: iounit, i
         ierr = 0
         iounit = alloc_iounit(ierr); if (ierr /= 0) return
         if (initialize) then
            open(unit=iounit, file=trim(filename), action='write', iostat=ierr)
         else
            open(unit=iounit, file=trim(filename), position='append', action='write', iostat=ierr)
         end if
         if (ierr /= 0) then
            call free_iounit(iounit)
            return
         end if
         do i=1,n
            write(unit=iounit, fmt=format_str) arry(i)
         end do
         close(iounit)
         call free_iounit(iounit)
      end subroutine append_data
      
            
      subroutine realloc_double(ptr,new_size,ierr)
         real(dp), pointer :: ptr(:)
         integer, intent(in) :: new_size
         integer, intent(out) :: ierr
         real(dp), pointer :: new_ptr(:)
         integer :: i
         ierr = 0
         allocate(new_ptr(new_size),stat=ierr)
         if (ierr /= 0) return
         if (associated(ptr)) then
            do i = 1, min(new_size,size(ptr,dim=1))
               new_ptr(i) = ptr(i)
            end do
            deallocate(ptr)
         end if
         ptr => new_ptr
      end subroutine realloc_double
      
            
      subroutine realloc_double2(ptr,new_size1,new_size2,ierr)
         real(dp), pointer :: ptr(:,:)
         integer, intent(in) :: new_size1,new_size2
         integer, intent(out) :: ierr
         real(dp), pointer :: new_ptr(:,:)
         integer :: i1,i2, i,j
         ierr = 0
         allocate(new_ptr(new_size1,new_size2),stat=ierr)
         if (ierr /= 0) return
         if (associated(ptr)) then
            i1 = min(new_size1,size(ptr,dim=1))
            i2 = min(new_size2,size(ptr,dim=2))
            ! ifort uses stack for array copy temp storage
            ! for large copies, this can produce seg faults
            ! doing the explicit loops seems to be safe
            !new_ptr(1:i1,1:i2) = ptr(1:i1,1:i2)
            do i=1,i1
               do j=1,i2
                  new_ptr(i,j) = ptr(i,j)
               end do
            end do
            deallocate(ptr)
         end if
         ptr => new_ptr
      end subroutine realloc_double2
      
            
      subroutine realloc_double2_no_copy(ptr,new_size1,new_size2,ierr)
         real(dp), pointer :: ptr(:,:)
         integer, intent(in) :: new_size1,new_size2
         integer, intent(out) :: ierr
         real(dp), pointer :: new_ptr(:,:)
         ierr = 0
         allocate(new_ptr(new_size1,new_size2),stat=ierr)
         if (ierr /= 0) return
         if (associated(ptr)) deallocate(ptr)
         ptr => new_ptr
      end subroutine realloc_double2_no_copy
      
            
      subroutine realloc_double3(ptr,new_size1,new_size2,new_size3,ierr)
         real(dp), pointer :: ptr(:,:,:)
         integer, intent(in) :: new_size1,new_size2,new_size3
         integer, intent(out) :: ierr
         real(dp), pointer :: new_ptr(:,:,:)
         integer :: i1,i2,i3, i,j,k
         ierr = 0
         allocate(new_ptr(new_size1,new_size2,new_size3),stat=ierr)
         if (ierr /= 0) return
         if (associated(ptr)) then
            i1 = min(new_size1,size(ptr,dim=1))
            i2 = min(new_size2,size(ptr,dim=2))
            i3 = min(new_size3,size(ptr,dim=3))
            ! ifort uses stack for array copy temp storage
            ! for large copies, this can produce seg faults
            ! doing the explicit loops seems to be safe
            !new_ptr(1:i1,1:i2,1:i3) = ptr(1:i1,1:i2,1:i3)
            do i=1,i1
               do j=1,i2
                  do k=1,i3
                     new_ptr(i,j,k) = ptr(i,j,k)
                  end do
               end do
            end do
            deallocate(ptr)
         end if
         ptr => new_ptr
      end subroutine realloc_double3
      
            
      subroutine realloc_double3_no_copy(ptr,new_size1,new_size2,new_size3,ierr)
         real(dp), pointer :: ptr(:,:,:)
         integer, intent(in) :: new_size1,new_size2,new_size3
         integer, intent(out) :: ierr
         real(dp), pointer :: new_ptr(:,:,:)
         ierr = 0
         allocate(new_ptr(new_size1,new_size2,new_size3),stat=ierr)
         if (ierr /= 0) return
         if (associated(ptr)) deallocate(ptr)
         ptr => new_ptr
      end subroutine realloc_double3_no_copy
      
            
      subroutine realloc_quad(ptr,new_size,ierr)
         real(qp), pointer :: ptr(:)
         integer, intent(in) :: new_size
         integer, intent(out) :: ierr
         real(qp), pointer :: new_ptr(:)
         integer :: i
         ierr = 0
         allocate(new_ptr(new_size),stat=ierr)
         if (ierr /= 0) return
         if (associated(ptr)) then
            do i = 1, min(new_size,size(ptr,dim=1))
               new_ptr(i) = ptr(i)
            end do
            deallocate(ptr)
         end if
         ptr => new_ptr
      end subroutine realloc_quad
      
            
      subroutine realloc_quad2(ptr,new_size1,new_size2,ierr)
         real(qp), pointer :: ptr(:,:)
         integer, intent(in) :: new_size1,new_size2
         integer, intent(out) :: ierr
         real(qp), pointer :: new_ptr(:,:)
         integer :: i1,i2, i,j
         ierr = 0
         allocate(new_ptr(new_size1,new_size2),stat=ierr)
         if (ierr /= 0) return
         if (associated(ptr)) then
            i1 = min(new_size1,size(ptr,dim=1))
            i2 = min(new_size2,size(ptr,dim=2))
            ! ifort uses stack for array copy temp storage
            ! for large copies, this can produce seg faults
            ! doing the explicit loops seems to be safe
            !new_ptr(1:i1,1:i2) = ptr(1:i1,1:i2)
            do i=1,i1
               do j=1,i2
                  new_ptr(i,j) = ptr(i,j)
               end do
            end do
            deallocate(ptr)
         end if
         ptr => new_ptr
      end subroutine realloc_quad2
      
            
      subroutine realloc_quad2_no_copy(ptr,new_size1,new_size2,ierr)
         real(qp), pointer :: ptr(:,:)
         integer, intent(in) :: new_size1,new_size2
         integer, intent(out) :: ierr
         real(qp), pointer :: new_ptr(:,:)
         ierr = 0
         allocate(new_ptr(new_size1,new_size2),stat=ierr)
         if (ierr /= 0) return
         if (associated(ptr)) deallocate(ptr)
         ptr => new_ptr
      end subroutine realloc_quad2_no_copy
      
            
      subroutine realloc_quad3(ptr,new_size1,new_size2,new_size3,ierr)
         real(qp), pointer :: ptr(:,:,:)
         integer, intent(in) :: new_size1,new_size2,new_size3
         integer, intent(out) :: ierr
         real(qp), pointer :: new_ptr(:,:,:)
         integer :: i1,i2,i3, i,j,k
         ierr = 0
         allocate(new_ptr(new_size1,new_size2,new_size3),stat=ierr)
         if (ierr /= 0) return
         if (associated(ptr)) then
            i1 = min(new_size1,size(ptr,dim=1))
            i2 = min(new_size2,size(ptr,dim=2))
            i3 = min(new_size3,size(ptr,dim=3))
            ! ifort uses stack for array copy temp storage
            ! for large copies, this can produce seg faults
            ! doing the explicit loops seems to be safe
            !new_ptr(1:i1,1:i2,1:i3) = ptr(1:i1,1:i2,1:i3)
            do i=1,i1
               do j=1,i2
                  do k=1,i3
                     new_ptr(i,j,k) = ptr(i,j,k)
                  end do
               end do
            end do
            deallocate(ptr)
         end if
         ptr => new_ptr
      end subroutine realloc_quad3
      
            
      subroutine realloc_quad3_no_copy(ptr,new_size1,new_size2,new_size3,ierr)
         real(qp), pointer :: ptr(:,:,:)
         integer, intent(in) :: new_size1,new_size2,new_size3
         integer, intent(out) :: ierr
         real(qp), pointer :: new_ptr(:,:,:)
         ierr = 0
         allocate(new_ptr(new_size1,new_size2,new_size3),stat=ierr)
         if (ierr /= 0) return
         if (associated(ptr)) deallocate(ptr)
         ptr => new_ptr
      end subroutine realloc_quad3_no_copy
      
            
      subroutine realloc_real(ptr,new_size,ierr)
         real, pointer :: ptr(:)
         integer, intent(in) :: new_size
         integer, intent(out) :: ierr
         real, pointer :: new_ptr(:)
         integer :: i
         ierr = 0
         allocate(new_ptr(new_size),stat=ierr)
         if (ierr /= 0) return
         if (associated(ptr)) then
            do i=1,min(new_size,size(ptr,dim=1))
               new_ptr(i) = ptr(i)
            end do
            deallocate(ptr)
         end if
         ptr => new_ptr
      end subroutine realloc_real
      
            
      subroutine realloc_real2(ptr,new_size1,new_size2,ierr)
         real, pointer :: ptr(:,:)
         integer, intent(in) :: new_size1,new_size2
         integer, intent(out) :: ierr
         real, pointer :: new_ptr(:,:)
         integer :: i1,i2, i,j
         ierr = 0
         allocate(new_ptr(new_size1,new_size2),stat=ierr)
         if (ierr /= 0) return
         if (associated(ptr)) then
            i1 = min(new_size1,size(ptr,dim=1))
            i2 = min(new_size2,size(ptr,dim=2))
            ! ifort uses stack for array copy temp storage
            ! for large copies, this can produce seg faults
            ! doing the explicit loops seems to be safe
            !new_ptr(1:i1,1:i2) = ptr(1:i1,1:i2)
            do i=1,i1
               do j=1,i2
                  new_ptr(i,j) = ptr(i,j)
               end do
            end do
            deallocate(ptr)
         end if
         ptr => new_ptr
      end subroutine realloc_real2
      
            
      subroutine realloc_real3(ptr,new_size1,new_size2,new_size3,ierr)
         real, pointer :: ptr(:,:,:)
         integer, intent(in) :: new_size1,new_size2,new_size3
         integer, intent(out) :: ierr
         real, pointer :: new_ptr(:,:,:)
         integer :: i1,i2,i3, i,j,k
         ierr = 0
         allocate(new_ptr(new_size1,new_size2,new_size3),stat=ierr)
         if (ierr /= 0) return
         if (associated(ptr)) then
            i1 = min(new_size1,size(ptr,dim=1))
            i2 = min(new_size2,size(ptr,dim=2))
            i3 = min(new_size3,size(ptr,dim=3))
            ! ifort uses stack for array copy temp storage
            ! for large copies, this can produce seg faults
            ! doing the explicit loops seems to be safe
            !new_ptr(1:i1,1:i2,1:i3) = ptr(1:i1,1:i2,1:i3)
            do i=1,i1
               do j=1,i2
                  do k=1,i3
                     new_ptr(i,j,k) = ptr(i,j,k)
                  end do
               end do
            end do
            deallocate(ptr)
         end if
         ptr => new_ptr
      end subroutine realloc_real3
            
      
      subroutine realloc_integer(ptr,new_size,ierr)
         integer, pointer :: ptr(:)
         integer, intent(in) :: new_size
         integer, intent(out) :: ierr
         integer, pointer :: new_ptr(:)
         integer :: i
         ierr = 0
         allocate(new_ptr(new_size),stat=ierr)
         if (ierr /= 0) return
         if (associated(ptr)) then
            do i = 1, min(new_size,size(ptr,dim=1))
               new_ptr(i) = ptr(i)
            end do
            deallocate(ptr)
         end if
         ptr => new_ptr
      end subroutine realloc_integer
      
            
      subroutine realloc_integer2(ptr,new_size1,new_size2,ierr)
         integer, pointer :: ptr(:,:)
         integer, intent(in) :: new_size1,new_size2
         integer, intent(out) :: ierr
         integer, pointer :: new_ptr(:,:)
         integer :: i1,i2, i,j
         ierr = 0
         allocate(new_ptr(new_size1,new_size2),stat=ierr)
         if (ierr /= 0) return
         if (associated(ptr)) then
            i1 = min(new_size1,size(ptr,dim=1))
            i2 = min(new_size2,size(ptr,dim=2))
            ! ifort uses stack for array copy temp storage
            ! for large copies, this can produce seg faults
            ! doing the explicit loops seems to be safe
            !new_ptr(1:i1,1:i2) = ptr(1:i1,1:i2)
            do i=1,i1
               do j=1,i2
                  new_ptr(i,j) = ptr(i,j)
               end do
            end do
            deallocate(ptr)
         end if
         ptr => new_ptr
      end subroutine realloc_integer2
            
      
      subroutine realloc_logical(ptr,new_size,ierr)
         logical, pointer :: ptr(:)
         integer, intent(in) :: new_size
         integer, intent(out) :: ierr
         logical, pointer :: new_ptr(:)
         integer :: i
         ierr = 0
         allocate(new_ptr(new_size),stat=ierr)
         if (ierr /= 0) return
         if (associated(ptr)) then
            do i = 1, min(new_size,size(ptr,dim=1))
               new_ptr(i) = ptr(i)
            end do
            deallocate(ptr)
         end if
         ptr => new_ptr
      end subroutine realloc_logical
      
      
            
      subroutine do1D(ptr,sz,dealloc,ierr)
         real(dp),dimension(:),pointer::ptr
         integer, intent(in) :: sz
         logical, intent(in) :: dealloc
         integer, intent(out) :: ierr
         if (dealloc) then
            deallocate(ptr,stat=ierr)
         else
            allocate(ptr(sz),stat=ierr)
            if (test_with_nans) ptr(:) = return_nan()
         end if
      end subroutine do1D
      
      
      subroutine do2D(ptr,sz1,sz2,dealloc,ierr)
         real(dp),dimension(:,:),pointer::ptr
         integer, intent(in) :: sz1,sz2
         logical, intent(in) :: dealloc
         integer, intent(out) :: ierr
         if (dealloc) then
            deallocate(ptr,stat=ierr)
         else
            allocate(ptr(sz1,sz2),stat=ierr)
            if (test_with_nans) ptr(:,:) = return_nan()
         end if
      end subroutine do2D
      
      
      subroutine do3D(ptr,sz1,sz2,sz3,dealloc,ierr)
         real(dp),dimension(:,:,:),pointer::ptr
         integer, intent(in) :: sz1,sz2,sz3
         logical, intent(in) :: dealloc
         integer, intent(out) :: ierr
         if (dealloc) then
            deallocate(ptr,stat=ierr)
         else
            allocate(ptr(sz1,sz2,sz3),stat=ierr)
            if (test_with_nans) ptr(:,:,:) = return_nan()
         end if
      end subroutine do3D
      
      
      subroutine do4D(ptr,sz1,sz2,sz3,sz4,dealloc,ierr)
         real(dp),dimension(:,:,:,:),pointer::ptr
         integer, intent(in) :: sz1,sz2,sz3,sz4
         logical, intent(in) :: dealloc
         integer, intent(out) :: ierr
         if (dealloc) then
            deallocate(ptr,stat=ierr)
         else
            allocate(ptr(sz1,sz2,sz3,sz4),stat=ierr)
            if (test_with_nans) ptr(:,:,:,:) = return_nan()
         end if
      end subroutine do4D
      
      
      subroutine do1D_integer(ptr,sz,dealloc,ierr)
         integer,dimension(:),pointer::ptr
         integer, intent(in) :: sz
         logical, intent(in) :: dealloc
         integer, intent(out) :: ierr
         if (dealloc) then
            deallocate(ptr,stat=ierr)
         else
            allocate(ptr(sz),stat=ierr)
         end if
      end subroutine do1D_integer
      
      
      subroutine do2D_integer(ptr,sz1,sz2,dealloc,ierr)
         integer,dimension(:,:),pointer::ptr
         integer, intent(in) :: sz1,sz2
         logical, intent(in) :: dealloc
         integer, intent(out) :: ierr
         if (dealloc) then
            deallocate(ptr,stat=ierr)
         else
            allocate(ptr(sz1,sz2),stat=ierr)
         end if
      end subroutine do2D_integer
      
      
      subroutine do1D_logical(ptr,sz,dealloc,ierr)
         logical,dimension(:),pointer::ptr
         integer, intent(in) :: sz
         logical, intent(in) :: dealloc
         integer, intent(out) :: ierr
         if (dealloc) then
            deallocate(ptr,stat=ierr)
         else
            allocate(ptr(sz),stat=ierr)
         end if
      end subroutine do1D_logical
            
      
      subroutine alloc1(sz,a,ierr)
         real(dp), dimension(:), pointer :: a
         integer, intent(in) :: sz
         integer, intent(out) :: ierr
         allocate(a(sz),stat=ierr); if (ierr /= 0) return
         if (test_with_nans) a = return_nan()
      end subroutine alloc1
      
      
      subroutine alloc2(sz1,sz2,a,ierr)
         real(dp), dimension(:,:), pointer :: a
         integer, intent(in) :: sz1,sz2
         integer, intent(out) :: ierr
         allocate(a(sz1,sz2),stat=ierr); if (ierr /= 0) return
         if (test_with_nans) a = return_nan()
      end subroutine alloc2
      
      
      subroutine alloc3(sz1,sz2,sz3,a,ierr)
         real(dp), dimension(:,:,:), pointer :: a
         integer, intent(in) :: sz1,sz2,sz3
         integer, intent(out) :: ierr
         allocate(a(sz1,sz2,sz3),stat=ierr); if (ierr /= 0) return
         if (test_with_nans) a = return_nan()
      end subroutine alloc3
      

      subroutine enlarge_if_needed_1(ptr,sz,extra,ierr)
         real(dp), pointer :: ptr(:)
         integer, intent(in) :: sz, extra
         integer, intent(out) :: ierr
         ierr = 0
         if (associated(ptr)) then
            if (size(ptr,dim=1) >= sz) return
            deallocate(ptr)
         end if
         allocate(ptr(sz + extra), stat=ierr)
      end subroutine enlarge_if_needed_1
   
   
      subroutine enlarge_if_needed_2(ptr,sz1,sz2,extra,ierr)
         real(dp), pointer :: ptr(:,:)
         integer, intent(in) :: sz1, sz2, extra
         integer, intent(out) :: ierr
         ierr = 0
         if (associated(ptr)) then
            if (size(ptr,1) == sz1 .and. size(ptr,2) >= sz2) return
            deallocate(ptr)
         end if
         allocate(ptr(sz1, sz2 + extra), stat=ierr)
      end subroutine enlarge_if_needed_2
   
   
      subroutine enlarge_if_needed_3(ptr,sz1,sz2,sz3,extra,ierr)
         real(dp), pointer :: ptr(:,:,:)
         integer, intent(in) :: sz1, sz2, sz3, extra
         integer, intent(out) :: ierr
         ierr = 0
         if (associated(ptr)) then
            if (size(ptr,1) == sz1 .and. size(ptr,2) == sz2 .and. &
                size(ptr,3) >= sz3) return
            deallocate(ptr)
         end if
         allocate(ptr(sz1, sz2, sz3 + extra), stat=ierr)
      end subroutine enlarge_if_needed_3
      

      subroutine quad_enlarge_if_needed_1(ptr,sz,extra,ierr)
         real(qp), pointer :: ptr(:)
         integer, intent(in) :: sz, extra
         integer, intent(out) :: ierr
         ierr = 0
         if (associated(ptr)) then
            if (size(ptr,dim=1) >= sz) return
            deallocate(ptr)
         end if
         allocate(ptr(sz + extra), stat=ierr)
      end subroutine quad_enlarge_if_needed_1
   
   
      subroutine quad_enlarge_if_needed_2(ptr,sz1,sz2,extra,ierr)
         real(qp), pointer :: ptr(:,:)
         integer, intent(in) :: sz1, sz2, extra
         integer, intent(out) :: ierr
         ierr = 0
         if (associated(ptr)) then
            if (size(ptr,1) == sz1 .and. size(ptr,2) >= sz2) return
            deallocate(ptr)
         end if
         allocate(ptr(sz1, sz2 + extra), stat=ierr)
      end subroutine quad_enlarge_if_needed_2
   
   
      subroutine quad_enlarge_if_needed_3(ptr,sz1,sz2,sz3,extra,ierr)
         real(qp), pointer :: ptr(:,:,:)
         integer, intent(in) :: sz1, sz2, sz3, extra
         integer, intent(out) :: ierr
         ierr = 0
         if (associated(ptr)) then
            if (size(ptr,1) == sz1 .and. size(ptr,2) == sz2 .and. &
                size(ptr,3) >= sz3) return
            deallocate(ptr)
         end if
         allocate(ptr(sz1, sz2, sz3 + extra), stat=ierr)
      end subroutine quad_enlarge_if_needed_3
      

      subroutine enlarge_integer_if_needed_1(ptr,sz,extra,ierr)
         integer, pointer :: ptr(:)
         integer, intent(in) :: sz, extra
         integer, intent(out) :: ierr
         ierr = 0
         if (associated(ptr)) then
            if (size(ptr,dim=1) >= sz) return
            deallocate(ptr)
         end if
         allocate(ptr(sz + extra), stat=ierr)
      end subroutine enlarge_integer_if_needed_1
   
   
      subroutine enlarge_integer_if_needed_2(ptr,sz1,sz2,extra,ierr)
         integer, pointer :: ptr(:,:)
         integer, intent(in) :: sz1, sz2, extra
         integer, intent(out) :: ierr
         ierr = 0
         if (associated(ptr)) then
            if (size(ptr,1) == sz1 .and. size(ptr,2) >= sz2) return
            deallocate(ptr)
         end if
         allocate(ptr(sz1, sz2 + extra), stat=ierr)
      end subroutine enlarge_integer_if_needed_2
   
   
      subroutine enlarge_integer_if_needed_3(ptr,sz1,sz2,sz3,extra,ierr)
         integer, pointer :: ptr(:,:,:)
         integer, intent(in) :: sz1, sz2, sz3, extra
         integer, intent(out) :: ierr
         ierr = 0
         if (associated(ptr)) then
            if (size(ptr,1) == sz1 .and. size(ptr,2) == sz2 .and. &
                size(ptr,3) >= sz3) return
            deallocate(ptr)
         end if
         allocate(ptr(sz1, sz2, sz3 + extra), stat=ierr)
      end subroutine enlarge_integer_if_needed_3
      
      
      real(dp) function return_nan()
         use utils_nan
         return_nan = create_nan(-1d0)
      end function return_nan
      
      
      integer function token(iounit, n, i, buffer, string)
         use utils_def
         integer, intent(in) :: iounit
         integer, intent(inout) :: n ! number of characters currently in buffer
         integer, intent(inout) :: i ! number of characters already read from buffer
         character (len=*), intent(inout) :: buffer ! line of text from input file
         character (len=*), intent(inout) :: string ! holds string or name for string or name token
         character (len=1) :: tab_str
         
         integer :: info, j, j1, j2, l, str_len
         
         token = 0
         info = 0
         
        line_loop: do
            do while (i >= n)
               read(iounit,fmt='(a)',iostat=info) buffer
               if (info /= 0) then
                  token = eof_token
                  return 
               end if
               n = len_trim(buffer)
               i = 0
               !write(*,'(i6,3x,a)') n, trim(buffer)
            end do
          token_loop: do while (i < n) ! have non-empty buffer
               i = i+1
               if (buffer(i:i) == char(9)) cycle token_loop ! skip tabs
               select case(buffer(i:i))
                  case ('!')
                     i = n
                     cycle line_loop
                  case (' ')
                     cycle token_loop
                  case ('&') ! ignore &'s
                     cycle token_loop
                  case ('(')
                     token = left_paren_token; return
                  case (')')
                     token = right_paren_token; return
                  case (',')
                     token = comma_token; return
                  case ('"')
                     j = 1; str_len = len(string)
                     do
                        i = i+1
                        if (i > n) exit
                        if (buffer(i:i) == '"') exit
                        if (j > str_len) exit
                        string(j:j) = buffer(i:i)
                        j = j+1
                     end do
                     do while (j <= str_len)
                        string(j:j) = ' '
                        j = j+1
                     end do
                     token = string_token
                     return
                  case ('''')
                     j = 1; str_len = len(string)
                     do
                        i = i+1
                        if (i > n) exit
                        if (buffer(i:i) == '''') exit
                        if (j > str_len) exit
                        string(j:j) = buffer(i:i)
                        j = j+1
                     end do
                     do while (j <= str_len)
                        string(j:j) = ' '
                        j = j+1
                     end do
                     token = string_token
                     return
                  case default
                     j1 = i; j2 = i
                     name_loop: do
                        if (i+1 > n) exit
                        if (buffer(i+1:i+1) == ' ') exit
                        if (buffer(i+1:i+1) == '(') exit
                        if (buffer(i+1:i+1) == ')') exit
                        if (buffer(i+1:i+1) == ',') exit
                        i = i+1
                        j2 = i
                     end do name_loop
                     str_len = len(string)
                     l = j2-j1+1
                     if (l > str_len) then
                        l = str_len
                        j2 = l+j1-1
                     end if
                     string(1:l) = buffer(j1:j2)
                     do j = l+1, str_len
                        string(j:j) = ' '
                     end do
                     token = name_token
                     return
               end select
            end do token_loop
         end do line_loop
         
      end function token
      
      
      subroutine integer_dict_define(dict, key, value, ierr)
         use utils_dict
         type (integer_dict), pointer :: dict ! pass null for empty dict
         character (len=*), intent(in) :: key
         integer, intent(in) :: value
         integer, intent(out) :: ierr ! error if len_trim(key) > maxlen_key_string
         call do_integer_dict_define(dict, key, value, ierr)
      end subroutine integer_dict_define
      
      
      subroutine integer_dict_create_hash(dict, ierr)
         use utils_dict
         type (integer_dict), pointer :: dict
         integer, intent(out) :: ierr
         call do_integer_dict_create_hash(dict, ierr)
      end subroutine integer_dict_create_hash
      
      
      subroutine integer_dict_lookup(dict, key, value, ierr)
         use utils_dict
         type (integer_dict), pointer :: dict
         character (len=*), intent(in) :: key
         integer, intent(out) :: value
         integer, intent(out) :: ierr ! 0 if found key in dict, -1 if didn't
         call do_integer_dict_lookup(dict, key, value, ierr)
      end subroutine integer_dict_lookup
      
      
      integer function integer_dict_size(dict) ! number of entries
         use utils_dict
         type (integer_dict), pointer :: dict
         integer_dict_size = size_integer_dict(dict)
      end function integer_dict_size
      
      
      subroutine integer_dict_free(dict)
         use utils_dict
         type (integer_dict), pointer :: dict
         call do_integer_dict_free(dict)
      end subroutine integer_dict_free


      subroutine set_pointer_1(Aptr, dest, n)
         real(dp), pointer :: Aptr(:)
         real(dp), target :: dest(n)
         integer, intent(in) :: n
         Aptr => dest
      end subroutine set_pointer_1


      subroutine set_pointer_2(Aptr, dest, n1, n2)
         real(dp), pointer :: Aptr(:, :)
         real(dp), target :: dest(n1, n2)
         integer, intent(in) :: n1, n2
         Aptr => dest
      end subroutine set_pointer_2
      

      subroutine set_pointer_3(Aptr, dest, n1, n2, n3)
         real(dp), pointer :: Aptr(:, :, :)
         real(dp), target :: dest(n1, n2, n3)
         integer, intent(in) :: n1, n2, n3
         Aptr => dest
      end subroutine set_pointer_3
      

      subroutine set_pointer_4(Aptr, dest, n1, n2, n3, n4)
         real(dp), pointer :: Aptr(:, :, :, :)
         real(dp), target :: dest(n1, n2, n3, n4)
         integer, intent(in) :: n1, n2, n3, n4
         Aptr => dest
      end subroutine set_pointer_4


      subroutine set_quad_pointer_1(Aptr, dest, n)
         real(qp), pointer :: Aptr(:)
         real(qp), target :: dest(n)
         integer, intent(in) :: n
         Aptr => dest
      end subroutine set_quad_pointer_1


      subroutine set_quad_pointer_2(Aptr, dest, n1, n2)
         real(qp), pointer :: Aptr(:, :)
         real(qp), target :: dest(n1, n2)
         integer, intent(in) :: n1, n2
         Aptr => dest
      end subroutine set_quad_pointer_2
      

      subroutine set_quad_pointer_3(Aptr, dest, n1, n2, n3)
         real(qp), pointer :: Aptr(:, :, :)
         real(qp), target :: dest(n1, n2, n3)
         integer, intent(in) :: n1, n2, n3
         Aptr => dest
      end subroutine set_quad_pointer_3
      

      subroutine set_quad_pointer_4(Aptr, dest, n1, n2, n3, n4)
         real(qp), pointer :: Aptr(:, :, :, :)
         real(qp), target :: dest(n1, n2, n3, n4)
         integer, intent(in) :: n1, n2, n3, n4
         Aptr => dest
      end subroutine set_quad_pointer_4


      subroutine set_logical_pointer_1(Aptr, dest, n)
         logical, pointer :: Aptr(:)
         logical, target :: dest(n)
         integer, intent(in) :: n
         Aptr => dest
      end subroutine set_logical_pointer_1


      subroutine set_logical_pointer_2(Aptr, dest, n1, n2)
         logical, pointer :: Aptr(:, :)
         logical, target :: dest(n1, n2)
         integer, intent(in) :: n1, n2
         Aptr => dest
      end subroutine set_logical_pointer_2


      subroutine set_int_pointer_1(Aptr, dest, n)
         integer, pointer :: Aptr(:)
         integer, target :: dest(n)
         integer, intent(in) :: n
         Aptr => dest
      end subroutine set_int_pointer_1


      subroutine set_int_pointer_2(Aptr, dest, n1, n2)
         integer, pointer :: Aptr(:, :)
         integer, target :: dest(n1, n2)
         integer, intent(in) :: n1, n2
         Aptr => dest
      end subroutine set_int_pointer_2
      

      subroutine set_int_pointer_3(Aptr2, dest, n1, n2, n3)
         integer, pointer :: Aptr2(:, :, :)
         integer, target :: dest(n1, n2, n3)
         integer, intent(in) :: n1, n2, n3
         Aptr2 => dest
      end subroutine set_int_pointer_3


      FUNCTION StrUpCase ( Input_String ) RESULT ( Output_String )
         CHARACTER( * ), INTENT( IN ) :: Input_String
         CHARACTER( LEN( Input_String ) ) :: Output_String
         INTEGER :: i, n
         Output_String = Input_String
         DO i = 1, LEN( Output_String )
            n = INDEX( LOWER_CASE, Output_String( i:i ) )
            IF ( n /= 0 ) Output_String( i:i ) = UPPER_CASE( n:n )
         END DO
      END FUNCTION StrUpCase


      FUNCTION StrLowCase ( Input_String ) RESULT ( Output_String )
         CHARACTER( * ), INTENT( IN ) :: Input_String
         CHARACTER( LEN( Input_String ) ) :: Output_String
         INTEGER :: i, n
         Output_String = Input_String
         DO i = 1, LEN( Output_String )
            n = INDEX( UPPER_CASE, Output_String( i:i ) )
            IF ( n /= 0 ) Output_String( i:i ) = LOWER_CASE( n:n )
         END DO
      END FUNCTION StrLowCase 
      
          
      end module utils_lib

