! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   this file is part of mesa.
!
!   mesa is free software; you can redistribute it and/or modify
!   it under the terms of the gnu general library public license as published
!   by the free software foundation; either version 2 of the license, or
!   (at your option) any later version.
!
!   mesa is distributed in the hope that it will be useful, 
!   but without any warranty; without even the implied warranty of
!   merchantability or fitness for a particular purpose.  see the
!   gnu library general public license for more details.
!
!   you should have received a copy of the gnu library general public license
!   along with this software; if not, write to the free software
!   foundation, inc., 59 temple place, suite 330, boston, ma 02111-1307 usa
!
! ***********************************************************************
 
      module run_star_extras

      use star_lib
      use star_def
      use const_def
      use utils_lib
      use num_lib, only: safe_log10, qsort
      use run_star_support, only: create_pre_main_sequence_model
      
      implicit none

      real*8 :: target_log_luminosity, target_log_radius, target_surface_Z_div_X, &
         target_Rcz, target_surf_He, target_csound_rms
      real*8 :: sigma_log_luminosity, sigma_log_radius, sigma_surface_Z_div_X, &
         sigma_Rcz, sigma_surf_He, sigma_csound_rms
      logical :: use_Rcz_in_search, use_sigma_surf_He_in_search, use_csound_rms_in_search
      logical :: vary_Z_div_X, vary_Y, vary_alpha, vary_f_ov, vary_f0_ov_fraction
      
      real*8 :: first_Z_div_X, first_Y, first_alpha, first_f_ov, first_f0_ov_fraction
      real*8 :: min_Z_div_X, min_Y, min_alpha, min_f_ov, min_f0_ov_fraction
      real*8 :: max_Z_div_X, max_Y, max_alpha, max_f_ov, max_f0_ov_fraction
      
      real*8 :: rhoend = 1d-4

      real*8 :: Y_frac_he3 = 1d-4
            
      character (len=100) :: search_type
      integer :: max_tries = 100 ! for BOBYQA


      integer :: hooke_itermax = 100
      real*8 :: hooke_rho = 0.50
      real*8 :: hooke_eps = 0.05
      real*8 :: hooke_xscale = 0.75



      namelist /calibration_controls/ &
         search_type, max_tries, &
         hooke_itermax, hooke_rho, hooke_eps, hooke_xscale, &
         target_log_luminosity, target_log_radius, target_surface_Z_div_X, &
         target_Rcz, target_surf_He, target_csound_rms, &
         sigma_log_luminosity, sigma_log_radius, sigma_surface_Z_div_X, &
         sigma_Rcz, sigma_surf_He, sigma_csound_rms, &
         use_Rcz_in_search, use_sigma_surf_He_in_search, use_csound_rms_in_search, &
         first_Z_div_X, first_Y, first_alpha, first_f_ov, first_f0_ov_fraction, &
         min_Z_div_X, min_Y, min_alpha, min_f_ov, min_f0_ov_fraction, &
         max_Z_div_X, max_Y, max_alpha, max_f_ov, max_f0_ov_fraction, &
         vary_Z_div_X, vary_Y, vary_alpha, vary_f_ov, vary_f0_ov_fraction, &
         Y_frac_he3, rhoend


      logical :: okay_to_restart

      real*8 :: next_initial_h1_to_try, next_initial_he3_to_try, &
         next_initial_he4_to_try, next_alpha_to_try, &
         next_Z_div_X_to_try, next_Y_to_try, next_f_ov_to_try, next_f0_ov_fraction_to_try

      ! "best_model" results are for particular parameters
      real*8 :: &
         best_model_chi_square, &
         best_model_age, &
         best_model_logR, &
         best_model_logL, &
         best_model_Teff, &
         best_model_Rcz, &
         best_model_surf_He, &
         best_model_surf_Z_div_X, &
         best_model_csound_rms
      integer :: best_model_model_number

      ! samples are for different combinations of parameters Y, Z/X, and alpha
      integer, parameter :: max_num_samples = 1000
      integer :: num_samples
      real*8, dimension(max_num_samples) :: &
         sample_chi_square, &
         sample_age, &
         sample_logR, &
         sample_logL, &
         sample_Teff, &
         sample_init_Y, &
         sample_init_Z_div_X, &
         sample_alpha, &
         sample_f_ov, &
         sample_f0_ov_fraction, &
         sample_Rcz, &
         sample_surf_He, &
         sample_surf_Z_div_X, &
         sample_csound_rms
      integer, dimension(max_num_samples) :: sample_model_number

      ! info about current model
      integer :: star_id, minimize_i_test
      real*8 :: chi_square
      real*8 :: initial_Y, initial_Z_div_X

      integer, parameter :: num_extra_log_columns = 0

      integer :: i_alpha, i_Y, i_Z_div_X, i_f_ov, i_f0_ov_fraction, nvar
      real*8 :: final_alpha, final_Y, final_Z_div_X, final_f_ov, final_f0_ov_fraction


      ! solar sound speed data
      logical :: have_sound_speed_data = .false.
      integer, parameter :: npts = 79
      real*8, dimension(npts) :: data_r, data_csound, data_width


      contains
      
      
      subroutine do_run
         use run_star_support
         type (star_info), pointer :: s
         integer :: id, ierr, i
         real*8 :: chi2, mass
         
         include 'formats.dek'
                  
         ierr = 0
         okay_to_restart = .true.

         call read_calibration_controls(ierr)
         if (failed('read_calibration_controls')) return

         call init_and_alloc(id, s, ierr)
         if (failed('init_and_alloc')) return
         
         star_id = id
                  
         call star_setup(id, 'inlist', ierr)
         if (failed('star_setup')) return
         
         write(*,*) 'search_type == ' // trim(search_type)
         if (search_type == 'just_do_first_values') then
            call do_bobyqa(.true., ierr)
         else if (search_type == 'bobyqa') then
            call do_bobyqa(.false., ierr)
         else if (search_type == 'hooke') then
            call do_hooke(ierr)
         else
            write(*,*) 'bad value for search_type ' // trim(search_type)
            ierr = -1
         end if

         
         contains

         logical function failed(str)
            character (len=*), intent(in) :: str
            failed = (ierr /= 0)
            if (failed) then
               write(*, *) trim(str) // ' ierr', ierr
               write(*, '(a)') trim(alert_message)
            end if
         end function failed
      
      
         subroutine read_calibration_controls(ierr)
            integer, intent(out) :: ierr
            character (len=256) :: filename, message
            integer :: unit
            unit=alloc_iounit(ierr)
            if (ierr /= 0) return
            filename = 'inlist_calibration_controls'
            open(unit=unit, file=trim(filename), action='read', delim='quote', iostat=ierr)
            if (ierr /= 0) then
               write(message, *) 'Failed to open control namelist file ', trim(filename)
               call alert(ierr, message)
               write(*,*) trim(message)
            else
               read(unit, nml=calibration_controls, iostat=ierr)  
               close(unit)
               if (ierr /= 0) then
                  write(message, *) 'Failed while trying to read control namelist file ', trim(filename)
                  write(*, '(a)') trim(message)
                  write(*, '(a)') &
                     'The following runtime error message might help you find the problem'
                  write(*, *) 
                  open(unit=unit, file=trim(filename), action='read', delim='quote', status='old', iostat=ierr)
                  read(unit, nml=calibration_controls)
                  close(unit)
                  call alert(ierr, message)
               end if  
            end if
            call free_iounit(unit)
         end subroutine read_calibration_controls


      end subroutine do_run
      
      
      real*8 function eval1(id)
         use run_star_support, only: run1_star
         integer :: id
         logical :: restart
         logical, parameter :: do_init_and_alloc = .false., do_free_star = .false.
         call run1_star( &
            do_init_and_alloc, do_free_star, okay_to_restart, &
            id, restart, &
            extras_controls, &
            extras_startup, &
            extras_check_model, &
            how_many_extra_log_columns, &
            data_for_extra_log_columns, &
            how_many_extra_profile_columns, &
            data_for_extra_profile_columns, &
            extras_finish_step, &
            extras_after_evolve)
         eval1 = best_model_chi_square
      end function eval1
         

      subroutine bobyqa_fun(n,x,f)
         use run_star_support, only: initial_h1, initial_he3, initial_he4
         integer, intent(in) :: n
         real*8, intent(in) :: x(*)
         real*8, intent(out) :: f
         
         integer :: ierr, id
         type (star_info), pointer :: s

         include 'formats.dek'
         
         ierr = 0
         id = star_id
         call get_star_ptr(id,s,ierr)
         if (ierr /= 0) return
         
         if (vary_alpha) then
            next_alpha_to_try = bobyqa_param(x(i_alpha), first_alpha, min_alpha, max_alpha)
            write(*,1) 'next_alpha_to_try', next_alpha_to_try, x(i_alpha)
         end if
         
         if (vary_Y) then
            next_Y_to_try = bobyqa_param(x(i_Y), first_Y, min_Y, max_Y)
            write(*,1) 'next_Y_to_try', next_Y_to_try, x(i_Y)
         end if

         if (vary_Z_div_X) then
            next_Z_div_X_to_try = bobyqa_param(x(i_Z_div_X), first_Z_div_X, min_Z_div_X, max_Z_div_X)
            write(*,1) 'next_Z_div_X_to_try', next_Z_div_X_to_try, x(i_Z_div_X)
         end if
         
         if (vary_f_ov) then
            next_f_ov_to_try = bobyqa_param(x(i_f_ov), first_f_ov, min_f_ov, max_f_ov)
            write(*,1) 'next_f_ov_to_try', next_f_ov_to_try, x(i_f_ov)
         end if
         
         if (vary_f0_ov_fraction) then
            next_f0_ov_fraction_to_try = &
               bobyqa_param(x(i_f0_ov_fraction), first_f0_ov_fraction, min_f0_ov_fraction, max_f0_ov_fraction)
            if (next_f0_ov_fraction_to_try < 0) next_f0_ov_fraction_to_try = 0
            write(*,1) 'next_f0_ov_fraction_to_try', next_f0_ov_fraction_to_try, x(i_f0_ov_fraction)
         end if
         
         f = eval1(star_id)
         
         minimize_i_test = minimize_i_test + 1
         call save_best_for_this_sample( &
            minimize_i_test, next_alpha_to_try, next_Y_to_try, next_Z_div_X_to_try, &
            s% overshoot_f_below_nonburn, s% overshoot_f0_below_nonburn)

         write(*,*)
         write(*,*) 'current set of sample results'
         call show_all_sample_results(6,minimize_i_test,ierr)
         if (ierr /= 0) then
            write(*,*) 'failed in show_all_sample_results'
            stop 'bobyqa_fun'
         end if

         write(*,*)
         call save_sample_results_to_file( &
            minimize_i_test,'bobyqa_results.txt',ierr)
         if (ierr /= 0) then
            write(*,*) 'failed in save_sample_results_to_file'
            stop 'bobyqa_fun'
         end if
            
         okay_to_restart = .false. ! only allow restart on 1st call
         
      end subroutine bobyqa_fun

      
      real*8 function bobyqa_param(x, first, min, max)
         real*8, intent(in) :: x, first, min, max
         if (x > 0) then
            bobyqa_param = first + x*(max-first)
         else
            bobyqa_param = first + x*(first-min)
         end if
      end function bobyqa_param
            
      
      subroutine do_bobyqa(just_do_first_values, ierr)
         use num_lib
         logical, intent(in) :: just_do_first_values
         integer, intent(out) :: ierr
         
         integer, parameter :: iprint = 0
         real*8, pointer, dimension(:) :: xl, xu, x, w
         real*8 :: min_chi2, rhobeg, max_bobyqa_value
         integer :: i, npt
         include 'formats.dek'
         ierr = 0
         minimize_i_test = 0
         nvar = 0
         
         if (vary_alpha) then
            nvar = nvar+1; i_alpha = nvar
            if (min_alpha >= max_alpha) then
               write(*,1) 'min_alpha >= max_alpha', min_alpha, max_alpha
               ierr = -1
            end if
         end if
         
         if (vary_Y) then
            nvar = nvar+1; i_Y = nvar
            if (min_Y >= max_Y) then
               write(*,1) 'min_Y >= max_Y', min_Y, max_Y
               ierr = -1
            end if
         end if
         
         if (vary_Z_div_X) then
            nvar = nvar+1; i_Z_div_X = nvar
            if (min_Z_div_X >= max_Z_div_X) then
               write(*,1) 'min_Z_div_X >= max_Z_div_X', min_Z_div_X, max_Z_div_X
               ierr = -1
            end if
         end if
         
         if (vary_f_ov) then
            nvar = nvar+1; i_f_ov = nvar
            if (min_f_ov >= max_f_ov) then
               write(*,1) 'min_f_ov >= max_f_ov', min_f_ov, max_f_ov
               ierr = -1
            end if
         end if
         
         if (vary_f0_ov_fraction) then
            nvar = nvar+1; i_f0_ov_fraction = nvar
            if (min_f0_ov_fraction >= max_f0_ov_fraction) then
               write(*,1) 'min_f0_ov_fraction >= max_f0_ov_fraction', &
                  min_f0_ov_fraction, max_f0_ov_fraction
               ierr = -1
            end if
         end if
         if (ierr /= 0) return
         
         npt = 2*nvar + 1
         
         allocate(xl(nvar), xu(nvar), x(nvar), w((npt+5)*(npt+nvar)+3*nvar*(nvar+5)/2))
         
         xl(:) = -1
         x(:) = 0
         xu(:) = 1
         
         if (just_do_first_values .or. nvar == 0) then         
            call bobyqa_fun(nvar,x,min_chi2)         
         else         
            !       RHOBEG and RHOEND must be set to the initial and final values of a trust
            !       region radius, so both must be positive with RHOEND no greater than
            !       RHOBEG. Typically, RHOBEG should be about one tenth of the greatest
            !       expected change to a variable, while RHOEND should indicate the
            !       accuracy that is required in the final values of the variables. An
            !       error return occurs if any of the differences XU(I)-XL(I), I=1,...,N,
            !       is less than 2*RHOBEG.
            rhobeg = 0.45d0
            max_bobyqa_value = 1d99
            call bobyqa(nvar,npt,x,xl,xu,rhobeg,rhoend,iprint,max_tries,w,bobyqa_fun,max_bobyqa_value)         
         end if
      
         if (vary_alpha) &
            final_alpha = bobyqa_param(x(i_alpha), first_alpha, min_alpha, max_alpha)
         
         if (vary_Y) &
            final_Y = bobyqa_param(x(i_Y), first_Y, min_Y, max_Y)

         if (vary_Z_div_X) &
            final_Z_div_X = bobyqa_param(x(i_Z_div_X), first_Z_div_X, min_Z_div_X, max_Z_div_X)
         
         if (vary_f_ov) &
            final_f_ov = bobyqa_param(x(i_f_ov), first_f_ov, min_f_ov, max_f_ov)
         
         if (vary_f0_ov_fraction) &
            final_f0_ov_fraction = &
               bobyqa_param(x(i_f0_ov_fraction), first_f0_ov_fraction, min_f0_ov_fraction, max_f0_ov_fraction)
         
         deallocate(xl, xu, x, w)
      
      end subroutine do_bobyqa



      double precision function hooke_f(x,n)
         integer, intent(in) :: n
         double precision, intent(in) :: x(n)
         
         integer :: ierr, id
         type (star_info), pointer :: s

         include 'formats.dek'
         
         ierr = 0
         id = star_id
         call get_star_ptr(id,s,ierr)
         if (ierr /= 0) return
         
         write(*,*)
         
         if (vary_alpha) then
            next_alpha_to_try = hooke_param(x(i_alpha), first_alpha, min_alpha, max_alpha)
            write(*,1) 'next_alpha_to_try', next_alpha_to_try, x(i_alpha)
         end if
         
         if (vary_Y) then
            next_Y_to_try = hooke_param(x(i_Y), first_Y, min_Y, max_Y)
            write(*,1) 'next_Y_to_try', next_Y_to_try, x(i_Y)
         end if

         if (vary_Z_div_X) then
            next_Z_div_X_to_try = hooke_param(x(i_Z_div_X), first_Z_div_X, min_Z_div_X, max_Z_div_X)
            write(*,1) 'next_Z_div_X_to_try', next_Z_div_X_to_try, x(i_Z_div_X)
         end if
         
         if (vary_f_ov) then
            next_f_ov_to_try = hooke_param(x(i_f_ov), first_f_ov, min_f_ov, max_f_ov)
            write(*,1) 'next_f_ov_to_try', next_f_ov_to_try, x(i_f_ov)
         end if
         
         if (vary_f0_ov_fraction) then
            next_f0_ov_fraction_to_try = &
               hooke_param(x(i_f0_ov_fraction), first_f0_ov_fraction, min_f0_ov_fraction, max_f0_ov_fraction)
            if (next_f0_ov_fraction_to_try < 0) next_f0_ov_fraction_to_try = 0
            write(*,1) 'next_f0_ov_fraction_to_try', next_f0_ov_fraction_to_try, x(i_f0_ov_fraction)
         end if
         
         if (maxval(x(1:n)) > 1 .or. minval(x(1:n)) < -1) then
            write(*,*) 'reject because parameter out of range'
            write(*,*)
            write(*,*)
            hooke_f = 1d99
            return
         end if
         
         hooke_f = eval1(star_id)
         
         minimize_i_test = minimize_i_test + 1
         call save_best_for_this_sample( &
            minimize_i_test, next_alpha_to_try, next_Y_to_try, next_Z_div_X_to_try, &
            s% overshoot_f_below_nonburn, s% overshoot_f0_below_nonburn)

         write(*,*)
         write(*,*) 'current set of sample results'
         call show_all_sample_results(6,minimize_i_test,ierr)
         if (ierr /= 0) then
            write(*,*) 'failed in show_all_sample_results'
            stop 'hooke_fun'
         end if

         write(*,*)
         call save_sample_results_to_file( &
            minimize_i_test,'hooke_results.txt',ierr)
         if (ierr /= 0) then
            write(*,*) 'failed in save_sample_results_to_file'
            stop 'hooke_fun'
         end if
            
         okay_to_restart = .false. ! only allow restart on 1st call
         
      end function hooke_f

      
      real*8 function hooke_param(x, first, min, max)
         real*8, intent(in) :: x, first, min, max
         if (x > 0) then
            hooke_param = first + hooke_xscale*x*(max-first)
         else
            hooke_param = first + hooke_xscale*x*(first-min)
         end if
      end function hooke_param
      
      
      subroutine do_hooke(ierr)
         use num_lib
         integer, intent(out) :: ierr
         real*8, pointer :: startpt(:), endpt(:)
         real*8 :: rho, eps, final_mass, final_alpha, final_Y, final_Z_div_X
         integer :: itermax, num_iters
         include 'formats.dek'
         
         ierr = 0
         minimize_i_test = 0
         
         if (vary_alpha) then
            nvar = nvar+1; i_alpha = nvar
            if (min_alpha >= max_alpha) then
               write(*,1) 'min_alpha >= max_alpha', min_alpha, max_alpha
               ierr = -1
            end if
         end if
         
         if (vary_Y) then
            nvar = nvar+1; i_Y = nvar
            if (min_Y >= max_Y) then
               write(*,1) 'min_Y >= max_Y', min_Y, max_Y
               ierr = -1
            end if
         end if
         
         if (vary_Z_div_X) then
            nvar = nvar+1; i_Z_div_X = nvar
            if (min_Z_div_X >= max_Z_div_X) then
               write(*,1) 'min_Z_div_X >= max_Z_div_X', min_Z_div_X, max_Z_div_X
               ierr = -1
            end if
         end if
         
         if (vary_f_ov) then
            nvar = nvar+1; i_f_ov = nvar
            if (min_f_ov >= max_f_ov) then
               write(*,1) 'min_f_ov >= max_f_ov', min_f_ov, max_f_ov
               ierr = -1
            end if
         end if
         
         if (vary_f0_ov_fraction) then
            nvar = nvar+1; i_f0_ov_fraction = nvar
            if (min_f0_ov_fraction >= max_f0_ov_fraction) then
               write(*,1) 'min_f0_ov_fraction >= max_f0_ov_fraction', &
                  min_f0_ov_fraction, max_f0_ov_fraction
               ierr = -1
            end if
         end if
         if (ierr /= 0) return
         
         allocate(startpt(nvar), endpt(nvar))         
         ! initialize to 0 means start with the given first_values
         startpt(:) = 0
         
         itermax = hooke_itermax
         rho = hooke_rho
         eps = hooke_eps
         
         write(*,2) 'hooke_itermax', hooke_itermax
         write(*,1) 'hooke_rho', hooke_rho
         write(*,1) 'hooke_eps', hooke_eps

         num_iters = hooke(nvar, startpt, endpt, rho, eps, itermax, hooke_f)
         
         if (vary_alpha) &
            final_alpha = hooke_param(endpt(i_alpha), first_alpha, min_alpha, max_alpha)
         
         if (vary_Y) &
            final_Y = hooke_param(endpt(i_Y), first_Y, min_Y, max_Y)

         if (vary_Z_div_X) &
            final_Z_div_X = hooke_param(endpt(i_Z_div_X), first_Z_div_X, min_Z_div_X, max_Z_div_X)
         
         if (vary_f_ov) &
            final_f_ov = hooke_param(endpt(i_f_ov), first_f_ov, min_f_ov, max_f_ov)
         
         if (vary_f0_ov_fraction) &
            final_f0_ov_fraction = &
               hooke_param(endpt(i_f0_ov_fraction), first_f0_ov_fraction, min_f0_ov_fraction, max_f0_ov_fraction)

         deallocate(startpt, endpt)         
      
      end subroutine do_hooke



      
      subroutine save_sample_results_to_file(i_test, results_fname, ierr)
         use utils_lib
         integer, intent(in) :: i_test
         character (len=*), intent(in) :: results_fname
         integer, intent(out) :: ierr
         character (len=256) :: save_model_name
         character (len=4) :: i_test_str
         integer :: iounit
         write(*,*) 'save_sample_results_to_file ' // trim(results_fname)
         iounit = alloc_iounit(ierr)
         if (ierr /= 0) stop 'alloc_iounit failed'
         open(unit=iounit, file=trim(results_fname), action='write', iostat=ierr)
         if (ierr /= 0) return
         call show_all_sample_results(iounit,i_test,ierr)
         close(iounit)
         call free_iounit(iounit)
         if (i_test < 10) then
            write(i_test_str,'(a,i1)') '000', i_test
         else if (i_test < 100) then
            write(i_test_str,'(a,i2)') '00', i_test
         else if (i_test < 1000) then
            write(i_test_str,'(a,i3)') '0', i_test
         else
            write(i_test_str,'(a,i4)') '', i_test
         end if 
         save_model_name = 'trial_out/trial_' // trim(i_test_str) // '.mod'
         write(*,*) 'save final model to ' // trim(save_model_name)
         call star_write_model(star_id, save_model_name, .true., ierr)
         write(*,*)
      end subroutine save_sample_results_to_file
      
      
      subroutine show_all_sample_results(iounit, i_test, ierr)
         integer, intent(in) :: iounit, i_test
         integer, intent(out) :: ierr
         integer :: i, j
         integer, pointer :: index(:)
         ! sort results by increasing sample_chi_square
         allocate(index(i_test), stat=ierr)
         if (ierr /= 0) then
            stop 'failed in allocate before calling qsort from show_all_sample_results'
         end if
         call qsort(index, i_test, sample_chi_square)
         write(iounit,'(8x,99a22)') &
               'chi_square', &
               'init_Y', &
               'init_Z_div_X', &
               'alpha', &
               'f_ov', &
               'f0_ov_fraction', &

               'logR', &
               'logL', &
               'csound_rms', &
               
               'surf_Z_div_X', &
               'target_surf_Z_div_X', &
               
               'radius_cz', &
               'target_radius_cz', &
               
               'surf_he', &
               'target_surf_he', &
               
               'sigma_surf_Z_div_X', &
               'err_surf_Z_div_X', &
               'sigma_radius_cz', &
               'err_radius_cz', &
               'sigma_surf_he', &
               'err_surf_he', &
               'sigma_csound_rms', &
               'err_csound_rms'
               
         do j = 1, i_test
            i = index(j)
            write(iounit,'(3x,i5,99f22.14)') i, &
               sample_chi_square(i), &
               sample_init_Y(i), &
               sample_init_Z_div_X(i), &
               sample_alpha(i), &
               sample_f_ov(i), &
               sample_f0_ov_fraction(i), &
               
               sample_logR(i), &
               sample_logL(i), &   
               sample_csound_rms(i), &

               sample_surf_Z_div_X(i), &
               target_surface_Z_div_X, &
                             
               sample_Rcz(i), &
               target_Rcz, &
                      
               sample_surf_He(i), &
               target_surf_He, &
                         
               sigma_surface_Z_div_X, &
               (sample_surf_Z_div_X(i) - target_surface_Z_div_X)/sigma_surface_Z_div_X, & 
               
               sigma_Rcz, &
               (sample_Rcz(i) - target_Rcz)/sigma_Rcz, &        
               
               sigma_surf_He, &
               (sample_surf_He(i) - target_surf_He)/sigma_surf_He, &     
               
               sigma_csound_rms, &
               (sample_csound_rms(i) - target_csound_rms)/sigma_csound_rms
               
         end do
         deallocate(index)
         
      end subroutine show_all_sample_results


      subroutine save_best_for_this_sample(itry, alpha, Y, Z_div_X, f_ov, f0_ov_fraction)
         integer, intent(in) :: itry
         real*8, intent(in) :: alpha, Y, Z_div_X, f_ov, f0_ov_fraction
      
         sample_alpha(itry) = alpha
         sample_init_Y(itry) = Y 
         sample_init_Z_div_X(itry) = Z_div_X
         sample_f_ov(itry) = f_ov
         sample_f0_ov_fraction(itry) = f0_ov_fraction

         sample_chi_square(itry) = best_model_chi_square
         sample_logR(itry) = best_model_logR
         sample_logL(itry) = best_model_logL
         sample_Teff(itry) = best_model_Teff
         sample_Rcz(itry) = best_model_Rcz
         sample_surf_He(itry) = best_model_surf_He
         sample_surf_Z_div_X(itry) = best_model_surf_Z_div_X
         sample_csound_rms(itry) = best_model_csound_rms
         sample_model_number(itry) = best_model_model_number

         num_samples = itry

      end subroutine save_best_for_this_sample


      subroutine setup_solar_data_for_calc_rms(ierr)
         use run_star_support, only: mesa_data_dir
         integer, intent(out) :: ierr
         
         integer, parameter :: lines_to_skip = 11
         integer :: iounit, i, k
         real*8 :: jnk
         
         character (len=256) :: fname
         
         have_sound_speed_data = .true.
         ierr = 0
         iounit = alloc_iounit(ierr)
         if (ierr /= 0) then
            return
         end if
         fname = trim(mesa_data_dir) // '/star_data/solar_csound.data'
         open(iounit, file=trim(fname), action='read', status='old', iostat=ierr)
         if (ierr /= 0) then
            write(*,*) 'failed to open ' // trim(fname)
            call free_iounit(iounit)
            return
         end if                  
         
         do i=1,lines_to_skip
            read(iounit,fmt=*,iostat=ierr)
            if (ierr /= 0) then
               write(*,*) 'failed when skipping line', i
               close(iounit)
               call free_iounit(iounit)
               return
            end if
         end do
         
         do i=1,npts
            read(iounit,fmt=*,iostat=ierr) &
               data_r(i), jnk, data_csound(i), jnk, jnk, jnk, data_width(i)
            if (ierr /= 0) then
               write(*,*) 'failed when reading data point', i
               close(iounit)
               call free_iounit(iounit)
               return
            end if
         end do

         close(iounit)
         call free_iounit(iounit)
      
      
      end subroutine setup_solar_data_for_calc_rms


      real*8 function calc_current_rms(s, nz) ! dR weighted
         use interp_1d_lib
         use interp_1d_def
         type (star_info), pointer :: s
         integer, intent(in) :: nz
         
         logical, parameter :: dbg = .false.
         real*8 :: calc_rms_f(4,nz), pm_work(nz,pm_work_size), sumy2, sumdr, dr, y2, cs
         real*8, parameter :: min_R = 0.094, max_R = 0.94
         integer :: k, i, ierr
         
         include 'formats.dek'
         
         calc_current_rms = -1  

         ierr = 0
         if (.not. have_sound_speed_data) then
            call setup_solar_data_for_calc_rms(ierr)
            if (ierr /= 0) then
               write(*,*) 'failed in setup_solar_data_for_calc_rms'
               return
            end if
         end if
         
         do k=1,nz
            if (k == 1) then
               calc_rms_f(1,k) = s% csound(k)
            else
               calc_rms_f(1,k) = &
                  (s% csound(k)*s% dq(k-1) + s% csound(k-1)*s% dq(k))/(s% dq(k-1) + s% dq(k))
            end if
         end do

         call interp_pm(s% r, nz, calc_rms_f, pm_work_size, pm_work, ierr)
         if (ierr /= 0) then
            write(*,*) 'failed in interp_pm'
            return
         end if

         sumy2 = 0
         sumdr = 0
         do i=1,npts
            if (data_r(i) < min_R .or. data_r(i) > max_R) cycle
            call interp_value(s% r, nz, calc_rms_f, Rsun*data_r(i), cs, ierr)
            if (ierr /= 0) then
               write(*,*) 'failed in interp_value', i
               return
            end if
            if (i == 1) then
               dr = data_r(2) - data_r(1)
            else
               dr = data_r(i) - data_r(i-1)
            end if
            if (dr < 0) dr = -dr
            ! change to weigh by point rather than by dr
            dr = 1
            
            sumdr = sumdr + dr
            y2 = dr*((cs - data_csound(i))/data_csound(i))**2
            sumy2 = sumy2 + y2
            if (dbg) write(*,2) 'rms cs, data_cs, reldiff, y2, dr', i, cs, data_csound(i), &
               (cs - data_csound(i))/data_csound(i), y2, dr
         end do
         
         calc_current_rms = sqrt(sumy2/sumdr)
         if (.true. .or. dbg) write(*,1) 'calc_current_rms', calc_current_rms

      end function calc_current_rms


      integer function extras_check_model(s, id, id_extra)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra
         include 'formats.dek'
         extras_check_model = keep_going
         if (create_pre_main_sequence_model) return
         if (mod(s% model_number,100) == 0) then
            if (vary_alpha) &
               write(*,'(a20,99f16.8)') 'alpha', s% mixing_length_alpha
            if (vary_Z_div_X) &
               write(*,'(a20,99f16.8)') 'initial Z/X', initial_Z_div_X
            if (vary_Y) &
               write(*,'(a20,99f16.8)') 'initial Y', initial_Y
            if (vary_f_ov) &
               write(*,'(a20,99f16.8)') 'f_ov', s% overshoot_f_below_nonburn
            if (vary_f0_ov_fraction) &
               write(*,'(a20,99f16.8)') 'f0_ov_fraction', &
                  s% overshoot_f0_below_nonburn/s% overshoot_f_below_nonburn
            write(*,*)
         end if
      end function extras_check_model

      
      integer function extras_finish_step(s, id, id_extra)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra
         extras_finish_step = keep_going
         if (create_pre_main_sequence_model) return
         call store_extra_info(s)
      end function extras_finish_step
      

      subroutine extras_controls(s, ierr)
         use run_star_support, only: &
            initial_h1, initial_h2, initial_he3, initial_he4, &
            set_uniform_initial_composition
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         real*8 :: X, Y, Z, Z_div_X
         include 'formats.dek'
         ierr = 0
         if (create_pre_main_sequence_model) return
         ! overwrite various inlist controls
         
         if (vary_alpha) then
            s% mixing_length_alpha = next_alpha_to_try
         else
            s% mixing_length_alpha = first_alpha
         end if
         
         if (vary_Y) then
            Y = next_Y_to_try
         else
            Y = first_Y
         end if
         
         if (vary_Z_div_X) then
            Z_div_X = next_Z_div_X_to_try
         else
            Z_div_X = first_Z_div_X
         end if
         
         if (vary_f_ov) then
            s% overshoot_f_below_nonburn = next_f_ov_to_try
         else
            s% overshoot_f_below_nonburn = first_f_ov
         end if
         
         if (vary_f0_ov_fraction) then
            s% overshoot_f0_below_nonburn = &
               next_f0_ov_fraction_to_try*s% overshoot_f_below_nonburn
         else if (first_f0_ov_fraction >= 0) then
            s% overshoot_f0_below_nonburn = &
               first_f0_ov_fraction*s% overshoot_f_below_nonburn
         end if
         
         initial_Y = Y
         initial_Z_div_X = Z_div_X
         X = (1 - Y)/(1 + Z_div_X)
         Z = X*Z_div_X
         initial_h1 = X
         initial_h2 = 0
         initial_he3 = Y_frac_he3*Y
         initial_he4 = Y - initial_he3
         set_uniform_initial_composition = .true.
      end subroutine extras_controls
      
      
      integer function extras_startup(s, id, restart, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: id
         logical, intent(in) :: restart
         integer, intent(out) :: ierr
         ierr = 0
         extras_startup = 0
         if (create_pre_main_sequence_model) return
         if (.not. restart) then
            call alloc_extra_info(s)
         else ! it is a restart
            call unpack_extra_info(s)
         end if
      end function extras_startup


      integer function how_many_extra_log_columns(s, id, id_extra)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra
         if (create_pre_main_sequence_model) then
            how_many_extra_log_columns = 0
            return
         end if
         how_many_extra_log_columns = num_extra_log_columns
      end function how_many_extra_log_columns
      
      
      subroutine data_for_extra_log_columns(s, id, id_extra, n, names, vals, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra, n
         character (len=maxlen_log_column_name) :: names(n)
         real*8 :: vals(n)
         integer, intent(out) :: ierr
         integer :: i
         
         include 'formats.dek'
         
         ierr = 0
         if (create_pre_main_sequence_model) return


         return
         
         i = 0
         
         i = i+1
         names(i) = 'chi_square'
         vals(i) = chi_square
         
         if (i /= num_extra_log_columns) then
            write(*,2) 'i', i
            write(*,2) 'num_extra_log_columns', num_extra_log_columns
            stop 'bad num_extra_log_columns'
         end if
         
      end subroutine data_for_extra_log_columns

      
      integer function how_many_extra_profile_columns(s, id, id_extra)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra
         how_many_extra_profile_columns = 0
      end function how_many_extra_profile_columns
      
      
      subroutine data_for_extra_profile_columns(s, id, id_extra, n, nz, names, vals, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra, n, nz
         character (len=maxlen_profile_column_name) :: names(n)
         real*8 :: vals(nz,n)
         integer, intent(out) :: ierr
         integer :: k
         ierr = 0
         ! here is an example for adding a profile column
         !if (n /= 1) stop 'data_for_extra_profile_columns'
         !names(1) = 'beta'
         !do k = 1, nz
         !   vals(k,1) = s% Pgas(k)/s% P(k)
         !end do
      end subroutine data_for_extra_profile_columns

      
      subroutine extras_after_evolve(s, id, id_extra, ierr)
         use run_star_support, only: initial_h1, initial_he3, initial_he4
         use const_def, only: Rsun
         use mlt_def, only: no_mixing, convective_mixing, overshoot_mixing
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra
         integer, intent(out) :: ierr
         
         real*8 :: surface_He, Rcz, surface_Z_div_X, csound_rms, &
            chi2term, chi2sum, chi_square
         integer :: chi2N, i, nz
         
         include 'formats.dek'

         ierr = 0         
         if (create_pre_main_sequence_model) return
         nz = s% nz

         surface_He = s% surface_he3 + s% surface_he4
         surface_Z_div_X = &
            (1d0 - (s% surface_h1 + surface_He))/s% surface_h1
         csound_rms = calc_current_rms(s, nz)
         
         Rcz = 0
         do i = 1, nz-1 ! locate bottom of solar convective zone
            if (s% mixing_type(i+1) /= convective_mixing &
                  .and. s% mixing_type(i) == convective_mixing) then
               if (s% r(i+1) > 0.25*Rsun .and. s% r(i) < 0.9*Rsun) then
                  Rcz = s% r(i)/Rsun
                  exit
               end if
            end if
         end do
         
         chi2sum = 0
         chi2N = 0
         
         chi2term = ((s% log_surface_luminosity - target_log_luminosity)/sigma_log_luminosity)**2
         write(*,1) 'log L chi2term', chi2term
         chi2sum = chi2sum + chi2term
         chi2N = chi2N + 1
         
         chi2term = ((s% log_surface_radius - target_log_radius)/sigma_log_radius)**2
         write(*,1) 'log R chi2term', chi2term
         chi2sum = chi2sum + chi2term
         chi2N = chi2N + 1

         chi2term = ((surface_Z_div_X-target_surface_Z_div_X)/sigma_surface_Z_div_X)**2
         write(*,1) 'Z_div_X chi2term', chi2term
         chi2sum = chi2sum + chi2term
         chi2N = chi2N + 1

         if (use_Rcz_in_search) then
            chi2term = ((Rcz-target_Rcz)/sigma_Rcz)**2
            write(*,1) 'Rcz chi2term', chi2term
            chi2sum = chi2sum + chi2term
            chi2N = chi2N + 1
         end if
         
         if (use_sigma_surf_He_in_search) then
            chi2term = ((surface_He-target_surf_He)/sigma_surf_He)**2
            write(*,1) 'surface_He chi2term', chi2term
            chi2sum = chi2sum + chi2term
            chi2N = chi2N + 1
         end if
         
         if (use_csound_rms_in_search) then
            chi2term = ((csound_rms-target_csound_rms)/sigma_csound_rms)**2
            write(*,1) 'csound_rms chi2term', chi2term
            chi2sum = chi2sum + chi2term
            chi2N = chi2N + 1
         end if
         
         ! add more chi^2 terms as desired
         
         chi_square = chi2sum/chi2N
         write(*,1) 'chi_square', chi_square
         
         best_model_chi_square = chi_square
         best_model_age = s% star_age
         best_model_logR = log10(s% photosphere_r)
         best_model_logL = s% log_surface_luminosity
         best_model_Teff = s% Teff
         best_model_Rcz = Rcz
         best_model_surf_He = surface_He
         best_model_surf_Z_div_X = surface_Z_div_X
         best_model_csound_rms = csound_rms
         
         best_model_model_number = s% model_number

      end subroutine extras_after_evolve
      
      
      ! routines for saving and restoring extra data so can do restarts
         
         ! put these defs at the top and delete from the following routines
         !integer, parameter :: extra_info_alloc = 1
         !integer, parameter :: extra_info_get = 2
         !integer, parameter :: extra_info_put = 3
      
      
      subroutine alloc_extra_info(s)
         integer, parameter :: extra_info_alloc = 1
         type (star_info), pointer :: s
         call move_extra_info(s,extra_info_alloc)
      end subroutine alloc_extra_info
      
      
      subroutine unpack_extra_info(s)
         integer, parameter :: extra_info_get = 2
         type (star_info), pointer :: s
         call move_extra_info(s,extra_info_get)
      end subroutine unpack_extra_info
      
      
      subroutine store_extra_info(s)
         integer, parameter :: extra_info_put = 3
         type (star_info), pointer :: s
         call move_extra_info(s,extra_info_put)
      end subroutine store_extra_info
      
      
      subroutine move_extra_info(s,op)
         integer, parameter :: extra_info_alloc = 1
         integer, parameter :: extra_info_get = 2
         integer, parameter :: extra_info_put = 3
         type (star_info), pointer :: s
         integer, intent(in) :: op
         
         integer :: i, j, num_ints, num_dbls, ierr
         
         i = 0
         ! call move_int or move_flg    
         num_ints = i
         
         i = 0
         ! call move_dbl       
         
         num_dbls = i
         
         if (op /= extra_info_alloc) return
         if (num_ints == 0 .and. num_dbls == 0) return
         
         ierr = 0
         call star_alloc_extras(s% id, num_ints, num_dbls, ierr)
         if (ierr /= 0) then
            write(*,*) 'failed in star_alloc_extras'
            write(*,*) 'alloc_extras num_ints', num_ints
            write(*,*) 'alloc_extras num_dbls', num_dbls
            stop 1
         end if
         
         contains
         
         subroutine move_dbl(dbl)
            real*8 :: dbl
            i = i+1
            select case (op)
            case (extra_info_get)
               dbl = s% extra_work(i)
            case (extra_info_put)
               s% extra_work(i) = dbl
            end select
         end subroutine move_dbl
         
         subroutine move_int(int)
            integer :: int
            i = i+1
            select case (op)
            case (extra_info_get)
               int = s% extra_iwork(i)
            case (extra_info_put)
               s% extra_iwork(i) = int
            end select
         end subroutine move_int
         
         subroutine move_flg(flg)
            logical :: flg
            i = i+1
            select case (op)
            case (extra_info_get)
               flg = (s% extra_iwork(i) /= 0)
            case (extra_info_put)
               if (flg) then
                  s% extra_iwork(i) = 1
               else
                  s% extra_iwork(i) = 0
               end if
            end select
         end subroutine move_flg
      
      end subroutine move_extra_info
      

      end module run_star_extras
