! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton, Aaron Dotter
!
!   this file is part of mesa.
!
!   mesa is free software; you can redistribute it and/or modify
!   it under the terms of the gnu general library public license as published
!   by the free software foundation; either version 2 of the license, or
!   (at your option) any later version.
!
!   mesa is distributed in the hope that it will be useful, 
!   but without any warranty; without even the implied warranty of
!   merchantability or fitness for a particular purpose.  see the
!   gnu library general public license for more details.
!
!   you should have received a copy of the gnu library general public license
!   along with this software; if not, write to the free software
!   foundation, inc., 59 temple place, suite 330, boston, ma 02111-1307 usa
!
! ***********************************************************************
 
      module run_star_extras

      use star_lib
      use star_def
      use const_def
      use run_star_support
      
      implicit none

      
      integer :: time0, time1, clock_rate
      double precision, parameter :: expected_runtime = 15 ! minutes

      
      logical :: start_from_photo, save_last_model
      character (len=256) :: masses_filename, pre_zahb_model, pre_zahb_photo
      
      namelist /create_zahb/ &
         masses_filename, pre_zahb_model, pre_zahb_photo, start_from_photo, save_last_model
         
      double precision, pointer :: masses(:)
      integer :: nmasses

      contains


      subroutine do_run
         
         integer :: id, ierr, id_extra
         logical :: restart
         type (star_info), pointer :: s
         double precision :: dt
         
         write(*,*) 'do create_zahb'

         ierr = 0
         id_extra = 0
         call system_clock(time0,clock_rate)

         call init_and_alloc(id, s, ierr)
         if (failed('init_and_alloc')) return
                  
         call star_setup(id, 'inlist', ierr)
         if (failed('star_setup')) return

         call extras_controls(s, ierr)
         if (failed('extras_controls')) return

         call do_star_job_controls_before(id, s, restart, ierr)
         if (failed('do_star_job_controls_before')) return

         call do_isochrone( s, id, id_extra, ierr)
         
         call system_clock(time1,clock_rate)
         dt = dble(time1 - time0) / clock_rate / 60
         if (dt > 10*expected_runtime) then
            write(*,'(/,a30,2f18.6,a,/)') '>>>>>>> EXCESSIVE runtime', &
               dt, expected_runtime, '   <<<<<<<<<  ERROR'
         else
            write(*,'(/,a30,2f18.6,2i10/)') 'runtime (minutes), retries, backups', &
               dt, expected_runtime, s% num_retries, s% num_backups
         end if

         
         contains
         
         logical function failed(str)
            character (len=*), intent(in) :: str
            failed = (ierr /= 0)
            if (failed) then
               write(*, *) trim(str) // ' ierr', ierr
               write(*, '(a)') trim(alert_message)
            end if
         end function failed


      end subroutine do_run    

      
      subroutine read_controls(filename,ierr)
         use utils_lib
         character (len=*) :: filename
         integer, intent(out) :: ierr

         
         character (len=256) :: message
         integer :: unit
         
         ! set defaults
         start_from_photo = .false.
         pre_zahb_photo = 'pre_zahb.photo'
         pre_zahb_model = 'pre_zahb.mod'
         masses_filename = 'zahb_masses.list'
         save_last_model = .false.
         
         unit=alloc_iounit(ierr)
         if (ierr /= 0) return

         open(unit=unit, file=trim(filename), action='read', delim='quote', iostat=ierr)
         if (ierr /= 0) then
            write(message, *) 'Failed to open control namelist file ', trim(filename)
            call alert(ierr, message)
            write(*,*) trim(message)
         else
            read(unit, nml=create_zahb, iostat=ierr)  
            close(unit)
            if (ierr /= 0) then
               write(message, *) 'Failed while trying to read control namelist file ', trim(filename)
               write(*, '(a)') trim(message)
               write(*, '(a)') &
                  'The following runtime error message might help you find the problem'
               write(*, *) 
               open(unit=unit, file=trim(filename), action='read', delim='quote', status='old', iostat=ierr)
               read(unit, nml=create_zahb)
               close(unit)
               call alert(ierr, message)
            end if  
         end if
         call free_iounit(unit)

      end subroutine read_controls
      

      subroutine do_isochrone( s, id, id_extra, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra
         integer, intent(out) :: ierr
         call do_multi_track( &
            s, id, id_extra, & !isochrone_inlist, log_columns_file, profile_columns_file, &
            isochrone_check_model, & !create_pre_main_sequence_model, &
            !pre_ms_T_c, pre_ms_guess_rho_c, pre_ms_d_log10_P, &
            ierr)
      end subroutine do_isochrone
      

      subroutine do_multi_track( s, id, id_extra, check_model, ierr)
         use run_star_support
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra
         interface
            integer function check_model(s, id, id_extra)
               use star_def
               type (star_info), pointer :: s
               integer, intent(in) :: id, id_extra
            end function check_model
         end interface
         integer, intent(out) :: ierr
         
         integer :: i, j, result
         logical :: first_try, prev_flag, just_did_backup
         character (len=256) :: log_name, profiles_index_name, log_data_prefix, zahb_inlist, last_model

         double precision :: next_mass

         logical, parameter :: dbg = .true.
         
         include 'formats.dek'
         
         ierr = 0

         !for saving last model
         prev_flag = .true.
         
         zahb_inlist = 'inlist_create_zahb'
         call read_controls(zahb_inlist,ierr)
         if (ierr /= 0) return

         nullify(masses)
         call read_masses(trim(masses_filename), masses, nmasses, ierr)
         if (ierr /= 0) return
      
         if (dbg) then
            write(*,*)
            write(*,2) 'done read_masses', nmasses
            write(*,*)
            do i=1, nmasses
               write(*,2) 'mass', i, masses(i)
            end do
            write(*,*)
            !stop 'do_isochrone'
         end if
         
         do i=1, nmasses
            
            next_mass = masses(i)
            if (next_mass <= 0) cycle ! skip this one
            
            ! set log/profile names prefix for isochrone mass number            
            if (i < 10) then
               write(log_name, '(a,i1,a)') 'i00', i, '_star.log'
               write(profiles_index_name, '(a,i1,a)') 'i00', i, '_profiles.index'
               write(log_data_prefix, '(a,i1,a)') 'i00', i, '_log'
            else if (i < 100) then
               write(log_name, '(a,i2,a)') 'i0', i, '_star.log'
               write(profiles_index_name, '(a,i2,a)') 'i0', i, '_profiles.index'
               write(log_data_prefix, '(a,i2,a)') 'i0', i, '_log'
            else
               write(log_name, '(a,i3,a)') 'i', i, '_star.log'
               write(profiles_index_name, '(a,i3,a)') 'i', i, '_profiles.index'
               write(log_data_prefix, '(a,i3,a)') 'i', i, '_log'
            end if 
            s% star_log_name = trim(log_name)
            s% profiles_index_name = trim(profiles_index_name)
            s% log_data_prefix = trim(log_data_prefix)
           

            if(start_from_photo)then
               write(*,2) 'reading photo'
               call star_load_restart_photo(id, trim(pre_zahb_photo), ierr)
               if(failed('star_load_restart_photo')) return
            else 
               write(*,2) 'reading saved model'
               call star_read_model(id, trim(pre_zahb_model), ierr)
               if(failed('star_read_model')) return
            endif

            if(dbg)then
               write(*,*)
               write(*,2) 'isochrone mass', i, s% initial_mass
               write(*,2) 'star_log_name: ' // trim(s% star_log_name)
               write(*,2) 'profiles_index_name: ' // trim(s% profiles_index_name)
               write(*,2) 'log_data_prefix: ' // trim(s% log_data_prefix)
               write(*,*)
            endif

            call do_star_job_controls_after(id, s, .false., ierr)
            if (failed('do_star_job_controls_after')) return

            write(*,1) 'relax_mass', next_mass
            lg_max_abs_mdot=-5d0
            call star_relax_mass(id, next_mass, lg_max_abs_mdot, ierr)
            if(failed('star_relax_mass')) return

         
            if (len_trim(log_columns_file) > 0) &
               write(*,*) 'read ' // trim(log_columns_file)
            call star_set_log_columns(id, log_columns_file, ierr)
            if (failed('star_set_log_columns')) return
         
            if (len_trim(profile_columns_file) > 0) &
               write(*,*) 'read ' // trim(profile_columns_file)
            call star_set_profile_columns(id, profile_columns_file, ierr)
            if (failed('star_set_profile_columns')) return

            evolve_loop: do ! evolve one step per loop
               first_try = .true.
               just_did_backup = .false.
               step_loop: do ! repeat for retry or backup
                  result = star_evolve_step(id, first_try, just_did_backup)
                  if (result == keep_going) result = check_model(s, id, id_extra)
                  if (result == keep_going) result = star_pick_next_timestep(id)            
                  if (result == keep_going) exit step_loop
                  if (result == redo) result = star_prepare_to_redo(id)
                  if (result == retry) result = star_prepare_to_retry(id)
                  if (result == backup) then
                     result = star_do1_backup(id)
                     just_did_backup = .true.
                  end if
                  if (result == terminate) exit evolve_loop
                  first_try = .false.
               end do step_loop
               result = star_finish_step(id, id_extra, .false., &
                  no_extra_profile_columns, none_for_extra_profile_columns, &
                  no_extra_log_columns, none_for_extra_log_columns, ierr)
               if (result /= keep_going) exit evolve_loop         
            end do evolve_loop

            call save_profile(id, id_extra, &
                  no_extra_profile_columns, none_for_extra_profile_columns, &
                  5, ierr)
            if (failed('star_after_evolve')) return

            if(save_last_model)then
            ! set filename for isochrone mass number
               if (i < 10) then
                  write(last_model, '(a,i1,a)') 'i00', i, '_star.mod'
               else if (i < 100) then
                  write(last_model, '(a,i2,a)') 'i0', i, '_star.mod'
               else
                  write(last_model, '(a,i3,a)') 'i', i, '_star.mod'
               end if 
               call star_write_model(id, last_model, prev_flag, ierr)
               write(*,'(a25,a13)') '      save last model to ', trim(last_model)
               if(failed('star_write_model')) return
            endif

            call show_terminal_header(id, ierr)
            if (failed('show_terminal_header')) return

            call write_terminal_summary(id, ierr)
            if (failed('write_terminal_summary')) return
            
            call star_after_evolve(id, ierr)
            if (failed('star_after_evolve')) return
            
            do j = 1, 10
               write(*,*)
            end do
                        
         end do
         
         write(*,*)

         write (*, *)
         write (*, *)
         write(*, *)
         write(*, '(a)') 'finished'
         write(*, *)
         
         contains
         
         logical function failed(str)
            character (len=*), intent(in) :: str
            failed = (ierr /= 0)
            if (failed) then
               write(*, *) trim(str) // ' ierr', ierr
               write(*, '(a)') trim(alert_message)
               stop 1
            end if
         end function failed

      end subroutine do_multi_track
      

      integer function isochrone_check_model(s, id, id_extra)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra
         include 'formats.dek'
         isochrone_check_model = star_check_model(s% id)
      end function isochrone_check_model

      
      integer function no_extra_log_columns(s, id, id_extra)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra
         no_extra_log_columns = 0
      end function no_extra_log_columns
      
      
      subroutine none_for_extra_log_columns(s, id, id_extra, n, names, vals, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra, n
         character (len=maxlen_log_column_name) :: names(n)
         double precision :: vals(n)
         integer, intent(out) :: ierr
         ierr = 0
      end subroutine none_for_extra_log_columns

      integer function no_extra_profile_columns(s, id, id_extra)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra
         no_extra_profile_columns = 0
      end function no_extra_profile_columns
      
      
      subroutine none_for_extra_profile_columns(s, id, id_extra, n, nz, names, vals, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra, n, nz
         character (len=maxlen_profile_column_name) :: names(n)
         double precision :: vals(nz,n)
         integer, intent(out) :: ierr
         ierr = 0
      end subroutine none_for_extra_profile_columns


      include 'standard_run_star_extras.dek'

      end module run_star_extras
