! ***********************************************************************
!
!   Copyright (C) 2011  Bill Paxton
!
!   this file is part of mesa.
!
!   mesa is free software; you can redistribute it and/or modify
!   it under the terms of the gnu general library public license as published
!   by the free software foundation; either version 2 of the license, or
!   (at your option) any later version.
!
!   mesa is distributed in the hope that it will be useful, 
!   but without any warranty; without even the implied warranty of
!   merchantability or fitness for a particular purpose.  see the
!   gnu library general public license for more details.
!
!   you should have received a copy of the gnu library general public license
!   along with this software; if not, write to the free software
!   foundation, inc., 59 temple place, suite 330, boston, ma 02111-1307 usa
!
! ***********************************************************************
 
      module run_star_extras

      use star_lib
      use star_def
      use const_def
      use rates_def, only: maxlen_reaction_Name
      
      implicit none
      
      !Interpolation variables & data
      !info:
      integer :: num_mesh_points, num_global_vars, num_local_vars, version
      !globals:
      double precision:: &
         m_tot, r_tot, l_tot, z_tot, x0, alpha, phi, xi, beta, &
         lambda, ddP_drr_c, ddrho_drr_c, age
      !locals:
      double precision, dimension(:,:), pointer :: fgong_data
      
      
      ! fgong model variables at each mesh point
      integer, parameter :: f_r = 1
      integer, parameter :: f_lnq = f_r + 1
      integer, parameter :: f_T = f_lnq + 1
      integer, parameter :: f_P = f_T + 1
      integer, parameter :: f_rho = f_P + 1
      integer, parameter :: f_X = f_rho + 1
      integer, parameter :: f_L = f_X + 1
      integer, parameter :: f_kap = f_L + 1
      integer, parameter :: f_eps = f_kap + 1
      integer, parameter :: f_gamma1 = f_eps + 1
      integer, parameter :: f_grada = f_gamma1 + 1
      integer, parameter :: f_delta = f_grada + 1
      integer, parameter :: f_Cp = f_delta + 1
      integer, parameter :: f_mu_e_inv = f_Cp + 1
      integer, parameter :: f_A = f_mu_e_inv + 1
      integer, parameter :: f_rX = f_A + 1
      integer, parameter :: f_Z = f_rX + 1
      integer, parameter :: f_R_sub_r = f_Z + 1
      integer, parameter :: f_eps_g = f_R_sub_r + 1
      integer, parameter :: f_Lg = f_eps_g + 1
      integer, parameter :: f_he3 = f_Lg + 1
      integer, parameter :: f_c12 = f_he3 + 1
      integer, parameter :: f_c13 = f_c12 + 1
      integer, parameter :: f_n14 = f_c13 + 1
      integer, parameter :: f_o16 = f_n14 + 1
      integer, parameter :: f_dlngam1_drho = f_o16 + 1
      integer, parameter :: f_dlngam1_dP = f_dlngam1_drho + 1
      integer, parameter :: f_dlngam1_dY = f_dlngam1_dP + 1
      integer, parameter :: f_h2 = f_dlngam1_dY + 1
      integer, parameter :: f_he4 = f_h2 + 1
      integer, parameter :: f_li7 = f_he4 + 1
      integer, parameter :: f_be7 = f_li7 + 1
      integer, parameter :: f_n15 = f_be7 + 1
      integer, parameter :: f_o17 = f_n15 + 1
      integer, parameter :: f_o18 = f_o17 + 1
      integer, parameter :: f_ne20 = f_o18 + 1
      
      integer, parameter :: num_fgong_vars = f_ne20
      
      integer, parameter :: num_fgong_species = 15



      ! controls
      
      character (len=256) :: nonstandard_format, fgong_file, output_file
      logical :: skip_net, skip_kap, skip_eos, subtract_fgong_eps_g
      
      integer, parameter :: max_num_rates_to_report = 100
      integer :: num_rates_to_report
      character(len=maxlen_reaction_Name) :: &
         reaction_to_report(max_num_rates_to_report)
      
      integer, parameter :: max_num_categories_to_report = 20
      integer :: num_categories_to_report
      character(len=maxlen_reaction_Name) :: &
         category_to_report(max_num_categories_to_report)
         
      real*8 :: constant_Z
      
      namelist /compare_controls/ &
         nonstandard_format, fgong_file, output_file, skip_net, skip_kap, skip_eos, &
         num_rates_to_report, reaction_to_report, constant_Z, &
         num_categories_to_report, category_to_report, subtract_fgong_eps_g
      

      
      contains

      
      subroutine read_controls(filename,ierr)
         use utils_lib
         character (len=*) :: filename
         integer, intent(out) :: ierr

         
         character (len=256) :: message
         integer :: unit
         
         ! set defaults
         nonstandard_format = ''
         fgong_file = 'fgong.l9bi.d.14c'
         output_file = 'compare.data'
         skip_kap = .false.
         skip_eos = .false.
         skip_net = .false.
         subtract_fgong_eps_g = .false.
         num_rates_to_report = 0
         reaction_to_report(:) = ''
         num_categories_to_report = 0
         category_to_report(:) = ''
         constant_Z = -1
         
         unit=alloc_iounit(ierr)
         if (ierr /= 0) return

         open(unit=unit, file=trim(filename), action='read', delim='quote', iostat=ierr)
         if (ierr /= 0) then
            write(message, *) 'Failed to open control namelist file ', trim(filename)
            call alert(ierr, message)
            write(*,*) trim(message)
         else
            read(unit, nml=compare_controls, iostat=ierr)  
            close(unit)
            if (ierr /= 0) then
               write(message, *) 'Failed while trying to read control namelist file ', trim(filename)
               write(*, '(a)') trim(message)
               write(*, '(a)') &
                  'The following runtime error message might help you find the problem'
               write(*, *) 
               open(unit=unit, file=trim(filename), action='read', delim='quote', status='old', iostat=ierr)
               read(unit, nml=compare_controls)
               close(unit)
               call alert(ierr, message)
            end if  
         end if
         call free_iounit(unit)

      end subroutine read_controls
         
      
      subroutine compare_points(s, species, ierr)
         use screen_lib, only: screening_option
         use num_lib, only: safe_log10
         use utils_lib
         type (star_info), pointer :: s
         integer, intent(in) :: species
         integer, intent(out) :: ierr
         
         integer :: i, screening_mode, net_lwork, io_out, &
            h1, h2, he3, he4, li7, be7, c12, c13, n14, n15, o16, o17, o18, ne20, mg24
         real*8 :: X, Y, Z, abar, zbar, z2bar, ye, approx_abar, approx_zbar, sumx, &
            dabar_dx(species), dzbar_dx(species), xa(species), &
            Rho, log10Rho, T, log10T
         character (len=32) :: fmt_out, label_fmt_out, ifmt_out
         character (len=128) :: fname
         
         include 'formats.dek'
         
         ierr = 0
         
         net_lwork = s% net_lwork
         
         call get_species_info(ierr)
         if (ierr /= 0) return

         screening_mode = screening_option(s% screening_mode, ierr)
         if (ierr /= 0) then
            write(*,*) 'unknown string for screening_mode: ' // trim(s% screening_mode)
            return
         end if
         
         call set_kap_params
         
         io_out = alloc_iounit(ierr)
         if (ierr /= 0) then
            write(*,*) 'failed in alloc_iounit'
            return
         end if
         
         fname = output_file
         write(*,*) 'write ' // trim(fname)
         open(unit=io_out, file=trim(fname), action='write', iostat=ierr)
         if (ierr /= 0) then
            write(*,*) 'failed to open ' // trim(fname)
            return
         end if
         
         fmt_out = '(99e40.20)'
         ifmt_out = '(99i40)'
         label_fmt_out = '(99a40)'
         
         write(io_out,label_fmt_out,advance='no') 'i'
            
         if (.not. skip_kap) &
            write(io_out,label_fmt_out,advance='no') &
               'lnkap_m_sub_f', 'mesa_kap', 'fgong_kap'
            
         if (.not. skip_eos) &
            write(io_out,label_fmt_out,advance='no') &
            'cs_m_sub_f_div_m', 'diff_lnP', 'diff_gamma1', &
            'diff_grada', 'diff_delta', 'diff_lnCp', & 
            'mesa_cs', 'fgong_cs', &
            'mesa_P', 'fgong_P', &
            'mesa_gamma1', 'fgong_gamma1', &
            'mesa_grada', 'fgong_grada', &
            'mesa_delta', 'fgong_delta', &
            'mesa_Cp', 'fgong_Cp'
            
         if (.not. skip_net) then
            write(io_out,label_fmt_out,advance='no') &
               'eps_nuc_m_sub_f', 'mesa_eps_nuc', 'fgong_eps'
            do i=1,num_rates_to_report
               !write(*,2) 'report ' // trim(reaction_to_report(i)), i
               if (len_trim(reaction_to_report(i)) == 0) cycle
               write(io_out,label_fmt_out,advance='no') trim(reaction_to_report(i))
            end do
            do i=1,num_categories_to_report
               !write(*,2) 'report ' // trim(category_to_report(i)), i
               if (len_trim(category_to_report(i)) == 0) cycle
               write(io_out,label_fmt_out,advance='no') trim(category_to_report(i))
            end do
         end if
         
         write(io_out,label_fmt_out,advance='no') &
            'logRho', 'logT', 'r_div_Rtot', 'm_div_Mtot', 'logR_div_Rsun', 'logM_div_Msun', &
            'X', 'Y', 'Z'
               
         write(io_out,*)
         
         do i = 1, num_mesh_points
            
            Rho = fgong_data(f_rho, i)
            log10Rho = log10(Rho)
            
            T = fgong_data(f_T, i)
            log10T = log10(T)
            
            call get_composition_info(i, ierr)
            if (ierr /= 0) exit
            
            write(io_out,ifmt_out,advance='no') i
            if (.not. skip_kap) then
               call compare_kap(i, ierr)
               if (ierr /= 0) exit
            end if

            if (.not. skip_eos) then
               call compare_eos(i, ierr)
               if (ierr /= 0) exit
            end if
         
            if (.not. skip_net) then
               call compare_net(i, ierr)
               if (ierr /= 0) exit
            end if
            
            write(io_out,fmt_out,advance='no') log10Rho, log10T, &
               fgong_data(f_r,i)/r_tot, exp(fgong_data(f_lnq,i)), &
               safe_log10(fgong_data(f_r,i)/Rsun), &
               safe_log10(exp(fgong_data(f_lnq,i))*m_tot/Msun), &
               X, Y, Z
         
            write(io_out,*)
				
         end do
         
         close(io_out)
         call free_iounit(io_out)
         
         
         contains
         
         
         subroutine set_kap_params
            use kap_lib, only: kap_ptr
            use kap_def, only: Kap_General_Info
            type (Kap_General_Info), pointer :: rq
            call kap_ptr(s% kap_handle, rq, ierr)
            if (ierr /= 0) return
            rq% cubic_interpolation_in_X = s% cubic_interpolation_in_X
            rq% cubic_interpolation_in_Z = s% cubic_interpolation_in_Z
         end subroutine set_kap_params
         
         
         subroutine get_species_info(ierr)
            use chem_def
            integer, intent(out) :: ierr
            
            integer :: j
            include 'formats.dek'
         
            h1 = s% net_iso(ih1); if (h1 == 0) ierr = -1
            h2 = s% net_iso(ih2); if (h2 == 0) ierr = -1
            he3 = s% net_iso(ihe3); if (he3 == 0) ierr = -1
            he4 = s% net_iso(ihe4); if (he4 == 0) ierr = -1
            li7 = s% net_iso(ili7); if (li7 == 0) ierr = -1
            be7 = s% net_iso(ibe7); if (be7 == 0) ierr = -1
            c12 = s% net_iso(ic12); if (c12 == 0) ierr = -1
            c13 = s% net_iso(ic13); if (c13 == 0) ierr = -1
            n14 = s% net_iso(in14); if (n14 == 0) ierr = -1
            n15 = s% net_iso(in15); if (n15 == 0) ierr = -1
            o16 = s% net_iso(io16); if (o16 == 0) ierr = -1
            o17 = s% net_iso(io17); if (o17 == 0) ierr = -1
            o18 = s% net_iso(io18); if (o18 == 0) ierr = -1
            ne20 = s% net_iso(ine20); if (ne20 == 0) ierr = -1
            mg24 = s% net_iso(img24); if (mg24 == 0) ierr = -1
            if (ierr /= 0) write(*,*) 'net is missing one of the required isos'
         
         end subroutine get_species_info


         subroutine get_composition_info(i, ierr)
            use chem_lib, only: composition_info
            use chem_def
            integer, intent(in) :: i
            integer, intent(out) :: ierr
            
            integer :: j, cid
            real*8 :: xh, xhe
            include 'formats.dek'
            
            X = fgong_data(f_X,i)
            Z = fgong_data(f_Z,i)
            if (constant_Z > 0) Z = constant_Z
            Y = 1 - (X+Z)
            
            ierr = 0
            xa(:) = 0
            xa(h2) = fgong_data(f_h2,i)
            xa(h1) = X - xa(h2)
            xa(he3) = fgong_data(f_he3,i)
            xa(he4) = Y - xa(he3)
            xa(li7) = fgong_data(f_li7,i)
            xa(be7) = fgong_data(f_be7,i)
            xa(c12) = fgong_data(f_c12,i)
            xa(c13) = fgong_data(f_c13,i)
            xa(n14) = fgong_data(f_n14,i)
            xa(n15) = fgong_data(f_n15,i)
            xa(o16) = fgong_data(f_o16,i)
            xa(o17) = fgong_data(f_o17,i)
            xa(o18) = fgong_data(f_o18,i)
            xa(ne20) = fgong_data(f_ne20,i)
            xa(mg24) = 0
            xa(mg24) = 1d0 - max(0d0,min(1d0,sum(xa)))
            
            if (sum(xa) > 1.000001d0) then
               ierr = -1
               write(*,2) 'bad composition for point', i, sum(xa)
               do j=1,species
                  write(*,1) trim(chem_isos% name(s% chem_id(j))), xa(j)
               end do
               write(*,1) 'X', X
               write(*,1) 'Y', Y
               write(*,1) 'Z', Z
               return
            end if
            
            call composition_info( &
               species, s% chem_id, xa, xh, xhe, abar, zbar, z2bar, ye, &
               approx_abar, approx_zbar, sumx, dabar_dx, dzbar_dx)
            
            if (abs(xh - X) > 1d-9 .or. abs(xhe - Y) > 1d-9) then
               ierr = -1
               write(*,2) 'xh - X', i, xh - X
               write(*,2) 'xhe - Y', i, xhe - Y
            end if
            
            
               
         end subroutine get_composition_info
         
         
         subroutine compare_eos(i, ierr)  
            use eos_def
            use eos_lib
            integer, intent(in) :: i
            integer, intent(out) :: ierr

            real*8, dimension(num_eos_basic_results) :: res, res_a, res_b, d_dlnd, d_dlnT
            real*8 :: Pgas, Prad, P, delta, cs, fgong_cs
               
            include 'formats.dek'
            
            ierr = 0
            call eosDT_get( &
               s% eos_handle, Z, X, approx_abar, approx_zbar, &
               s% species, s% chem_id, s% net_iso, xa, &
               Rho, log10Rho, T, log10T, &
               res, d_dlnd, d_dlnT, ierr)
               
            if (ierr /= 0) then
               write(*,2) 'eosDT_get failed for point', i
               return
            end if
            
            Prad = Radiation_Pressure(T)
            Pgas = exp(res(i_lnPgas))
            P = Prad + Pgas
            delta = res(i_chiT)/res(i_chiRho)
            cs = sqrt(res(i_gamma1)*P/Rho)
            fgong_cs = sqrt(fgong_data(f_gamma1,i)*fgong_data(f_P,i)/Rho)
            write(io_out,fmt_out,advance='no') &
               (cs - fgong_cs)/cs, (P - fgong_data(f_P,i))/P, &
               res(i_gamma1) - fgong_data(f_gamma1,i), &
               res(i_grad_ad) - fgong_data(f_grada,i), &
               delta - fgong_data(f_delta,i), &
               (res(i_cp) - fgong_data(f_cp,i))/res(i_cp), &
               cs, fgong_cs, &
               P, fgong_data(f_P,i), &
               res(i_gamma1), fgong_data(f_gamma1,i), &
               res(i_grad_ad), fgong_data(f_grada,i), &
               delta, fgong_data(f_delta,i), &
               res(i_cp), fgong_data(f_cp,i)
               
         end subroutine compare_eos
         
         
         subroutine compare_kap(i, ierr)
            integer, intent(in) :: i
            integer, intent(out) :: ierr
            real*8 :: kap
         
            !call kap_check ! TESTING

            call do1_kap(X,Z,kap,ierr)
            if (ierr /= 0) return
            write(io_out,fmt_out,advance='no') &
               (kap - fgong_data(f_kap,i))/kap, kap, fgong_data(f_kap,i)
               
         end subroutine compare_kap
         
         
         subroutine do1_kap(X,Z,kap,ierr)
            use kap_lib, only: kap_get_Type1
            real*8, intent(in) :: X, Z
            real*8, intent(out) :: kap
            integer, intent(out) :: ierr
            
            real*8 :: dln_kap_dlnRho, dln_kap_dlnT, &
               lnfree_e, d_lnfree_e_dlnRho, d_lnfree_e_dlnT
               
            include 'formats.dek'
            
            ierr = 0
            lnfree_e = 0
            d_lnfree_e_dlnRho = 0
            d_lnfree_e_dlnT = 0
            call kap_get_Type1( &
               s% kap_handle, zbar, X, Z, log10Rho, log10T, & 
               lnfree_e, d_lnfree_e_dlnRho, d_lnfree_e_dlnT, &
               kap, dln_kap_dlnRho, dln_kap_dlnT, ierr)
               
            if (ierr /= 0) then
               write(*,2) 'kap_get_Type1 failed for point', i
               return
            end if
            
         end subroutine do1_kap
         
         
         
         subroutine kap_check
            integer :: ierr
            real*8 :: X, Y, Z, kap
            include 'formats.dek'
            log10Rho = -5.5d0; log10T = 5.5d0
            
            Z = 0.01d0
            
               X = 0.70d0; Y = 1 - (X+Z)
               call do1_kap(X,Z,kap,ierr)
               if (ierr /= 0) return
               write(*,1) 'log_kap', log10(kap), X, Y, Z
            
               X = 0.80d0; Y = 1 - (X+Z)
               call do1_kap(X,Z,kap,ierr)
               if (ierr /= 0) return
               write(*,1) 'log_kap', log10(kap), X, Y, Z
            
            Z = 0.02d0
            
               X = 0.70d0; Y = 1 - (X+Z)
               call do1_kap(X,Z,kap,ierr)
               if (ierr /= 0) return
               write(*,1) 'log_kap', log10(kap), X, Y, Z
            
               X = 0.80d0; Y = 1 - (X+Z)
               call do1_kap(X,Z,kap,ierr)
               if (ierr /= 0) return
               write(*,1) 'log_kap', log10(kap), X, Y, Z
            
            Z = 0.014
            
               X = 0.72d0; Y = 1 - (X+Z)
               call do1_kap(X,Z,kap,ierr)
               if (ierr /= 0) return
               write(*,1) 'log_kap', log10(kap), X, Y, Z
            
            stop 'kap_check'
            
         end subroutine kap_check
         
         
         subroutine compare_net(i, ierr)
            use rates_def
            use chem_def
            use chem_lib, only: rates_category_id
            use net_lib, only: net_get, get_net_reaction_table_ptr
            use rates_lib, only: rates_reaction_id

            integer, intent(in) :: i
            integer, intent(out) :: ierr
            
            integer, parameter :: k = 1
            integer :: j, ii, ir
            real*8 :: net_work(net_lwork), eps_nuc, eps_binding, eta, theta_e, &
               d_eps_nuc_dRho, d_eps_nuc_dT, fgong_eps_nuc
            integer, pointer :: net_reaction_ptr(:) 
            logical, parameter :: reuse_given_rates = .false.
               
            include 'formats.dek'
            
            ierr = 0
            eta = 0
            theta_e = 0
            
            call net_get( &
               s% net_handle, s% species, s% num_reactions, xa, &
               T, log10T, Rho, log10Rho, &
               abar, zbar, z2bar, ye, &
               eta, s% rate_factors, s% category_factors(:), std_reaction_Qs, std_reaction_neuQs, &
               eps_nuc, d_eps_nuc_dRho, d_eps_nuc_dT, s% d_epsnuc_dx(:,k), & 
               s% dxdt_nuc(:,k), s% dxdt_dRho(:,k), s% dxdt_dT(:,k), s% d_dxdt_dx(:,:,k), &
               screening_mode, theta_e, &
               s% rate_screened(:,:,k), s% rate_raw(:,:,k), reuse_given_rates, &
               s% reaction_eps_nuc(:,:,k), s% eps_nuc_categories(:,:,k), &
               eps_binding, s% eps_nuc_neu_total(k), &
               net_lwork, net_work, ierr)
               
            if (ierr /= 0) then
               write(*,2) 'net_get failed for point', i
               write(*,2) 'sum(xa)', i, sum(xa)
               do j=1,species
                  write(*,1) trim(chem_isos% name(s% chem_id(j))), xa(j)
               end do
               return
            end if
            
            fgong_eps_nuc = fgong_data(f_eps,i)
            if (subtract_fgong_eps_g) &
               fgong_eps_nuc = fgong_eps_nuc - fgong_data(f_eps_g,i)
            write(io_out,fmt_out,advance='no') &
               eps_nuc - fgong_eps_nuc, &
               eps_nuc, fgong_eps_nuc
         
            call get_net_reaction_table_ptr(s% net_handle, net_reaction_ptr, ierr)
            if (ierr /= 0) return
            
            do ii=1,num_rates_to_report
               if (len_trim(reaction_to_report(ii)) == 0) cycle
               ir = rates_reaction_id(reaction_to_report(ii))
               j = 0
               if (ir > 0) j = net_reaction_ptr(ir)
               if (j <= 0) then
                  eps_nuc = 0
               else
                  eps_nuc = s% reaction_eps_nuc(i_rate,j,k)
               end if
               write(io_out,fmt_out,advance='no') eps_nuc
            end do
            
            do ii=1,num_categories_to_report
               if (len_trim(category_to_report(ii)) == 0) cycle
               j = rates_category_id(category_to_report(ii))
               if (j <= 0) then
                  eps_nuc = 0
               else
                  eps_nuc = s% eps_nuc_categories(i_rate,j,k)
               end if
               write(io_out,fmt_out,advance='no') eps_nuc
            end do


         end subroutine compare_net
         
         
      end subroutine compare_points
      
      
      !Read in a file in FGONG format and initializes all the variables associated
      subroutine read_fgong(filename, fmt, ierr)
         use utils_lib
         character(len=*), intent(in) :: filename, fmt ! std fgong fmt = '(5ES16.9)'
         integer, intent(out) :: ierr
         
         integer :: i, j, iounit
         
         include 'formats.dek'

         ierr = 0
         write(*,*) 'Reading FGONG file: ', trim(filename)

         iounit = alloc_iounit(ierr)
         if (ierr /= 0) return
         open(unit=iounit, file=trim(filename), status='old', action='read', iostat=ierr)
         if (ierr /= 0) then
            write(*, '(a)') 'failed to open ' // trim(filename)
            call free_iounit(iounit)
            return
         end if

         !skip the 4 lines of header:
         do i=1,4
            read(iounit,*,iostat=ierr)
            if (ierr /= 0) then
               write(*,*) 'failed reading header lines'
               return
            end if
         end do

         read(iounit,*,iostat=ierr) num_mesh_points, num_global_vars, num_local_vars, version
         if (ierr /= 0) then
            write(*,*) 'failed reading num vars info'
            return
         end if
         write(*,2) 'num_mesh_points', num_mesh_points
         write(*,2) 'num_global_vars', num_global_vars
         write(*,2) 'num_local_vars', num_local_vars
         write(*,2) 'version', version

         allocate(fgong_data(max(num_local_vars,num_fgong_vars),num_mesh_points), stat=ierr)
         if (ierr /= 0) then
            write(*,*) 'failed to allocate fgong_data'
            return
         end if
         fgong_data = 0d0

         !Read in the num_global_vars global variables:
         read(iounit,fmt=fmt,iostat=ierr) m_tot, r_tot, l_tot, z_tot, x0
         if (ierr /= 0) then
            write(*,*) 'failed reading m_tot, r_tot, l_tot, z_tot, x0'
            return
         end if

         read(iounit,fmt=fmt,iostat=ierr) alpha, phi, xi, beta, lambda
         if (ierr /= 0) then
            write(*,*) 'failed reading alpha, phi, xi, beta, lambda'
            return
         end if
         
         read(iounit,fmt=fmt,iostat=ierr) ddP_drr_c, ddrho_drr_c, age
         if (ierr /= 0) then
            write(*,*) 'failed reading ddP_drr_c, ddrho_drr_c, age'
            return
         end if

         !Now read the point data:
         i = 1
         do while (i <= num_mesh_points)

            read(iounit,fmt=fmt,iostat=ierr) fgong_data(1:5,i)
            if (ierr /= 0) then
               write(*,*) 'failed reading point data 1:5 for mesh point', i
               return
            end if
 
            read(iounit,fmt=fmt,iostat=ierr) fgong_data(6:10,i)
            if (ierr /= 0) then
               write(*,*) 'failed reading point data 6:10 for mesh point', i
               return
            end if
 
            read(iounit,fmt=fmt,iostat=ierr) fgong_data(11:15,i)
            if (ierr /= 0) then
               write(*,*) 'failed reading point data 11:15 for mesh point', i
               return
            end if
 
            read(iounit,fmt=fmt,iostat=ierr) fgong_data(16:20,i)
            if (ierr /= 0) then
               write(*,*) 'failed reading point data 16:20 for mesh point', i
               return
            end if
 
            read(iounit,fmt=fmt,iostat=ierr) fgong_data(21:25,i)
            if (ierr /= 0) then
               write(*,*) 'failed reading point data 21:25 for mesh point', i
               return
            end if
 
            if (num_local_vars.gt.25) then
               read(iounit,fmt=fmt,iostat=ierr) fgong_data(26:30,i)
               if (ierr /= 0) then
                  write(*,*) 'failed reading point data 26:30 for mesh point', i
                  return
               end if
               if (num_local_vars > 30) then
                  read(iounit,fmt=fmt,iostat=ierr) fgong_data(31:35,i)
                  if (ierr /= 0) then
                     write(*,*) 'failed reading point data 31:35 for mesh point', i
                     return
                  end if
                  if (num_local_vars > 35) then
                     read(iounit,fmt=fmt,iostat=ierr) fgong_data(36:40,i)
                     if (ierr /= 0) then
                        write(*,*) 'failed reading point data for mesh point', i
                        return
                     end if
                  end if
               end if
            end if

            if (i > 1) then
               !check for repeated mass-coordinates
               if (fgong_data(f_lnq,i) == fgong_data(f_lnq,i-1)) then
                  write(*,2) 'discard repeated mass coordinate: ', i, fgong_data(f_lnq,i)
                  num_mesh_points = num_mesh_points-1
                  i = i-1
               else if (fgong_data(f_r,i) <= 0) then
                  write(*,2) 'discard r=0 point: ', i
                  num_mesh_points = num_mesh_points-1
                  i = i-1
               end if
            end if
            i = i+1

         end do

         close(iounit)
         call free_iounit(iounit)
         
      end subroutine read_fgong
      
      
      subroutine read_guenter_file(filename, ierr)
         use utils_lib
         character(len=*), intent(in) :: filename
         integer, intent(out) :: ierr
         
         integer :: i, j, iounit
         real*8 :: logP, logT, logRho, X, Z, logK
         
         include 'formats.dek'

         ierr = 0
         write(*,*) 'Reading Guenter input file: ', trim(filename)

         iounit = alloc_iounit(ierr)
         if (ierr /= 0) return
         open(unit=iounit, file=trim(filename), status='old', action='read', iostat=ierr)
         if (ierr /= 0) then
            write(*, '(a)') 'failed to open ' // trim(filename)
            call free_iounit(iounit)
            return
         end if

         !skip the 6 lines of header:
         do i=1,6
            read(iounit,*,iostat=ierr)
            if (ierr /= 0) then
               write(*,*) 'failed reading header lines'
               return
            end if
         end do
         
         m_tot = 1; r_tot = 1

         read(iounit,*,iostat=ierr) num_mesh_points
         if (ierr /= 0) then
            write(*,*) 'failed reading num vars info'
            return
         end if
         write(*,2) 'num_mesh_points', num_mesh_points

         allocate(fgong_data(num_fgong_vars,num_mesh_points), stat=ierr)
         if (ierr /= 0) then
            write(*,*) 'failed to allocate fgong_data'
            return
         end if
         fgong_data = 0d0

         !Now read the point data:
         do i = 1, num_mesh_points
            read(iounit,*,iostat=ierr) logP, logT, logRho, X, Z, logK
            if (ierr /= 0) then
               write(*,*) 'failed reading point data for mesh point', i
               return
            end if
            fgong_data(f_P, i) = 10**logP
            fgong_data(f_T, i) = 10**logT
            fgong_data(f_rho, i) = 10**logRho
            fgong_data(f_X, i) = X
            fgong_data(f_Z, i) = Z
            fgong_data(f_kap, i) = 10**logK
         end do

         close(iounit)
         call free_iounit(iounit)
         
      end subroutine read_guenter_file
      
      
      subroutine extras_controls(s, ierr)
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         ierr = 0
      end subroutine extras_controls
      
      
      integer function extras_startup(s, id, restart, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: id
         logical, intent(in) :: restart
         integer, intent(out) :: ierr
         ierr = 0
         extras_startup = 0
         if (.not. restart) then
            call alloc_extra_info(s)
         else ! it is a restart
            call unpack_extra_info(s)
         end if
      end function extras_startup
      
      
      subroutine extras_after_evolve(s, id, id_extra, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra
         integer, intent(out) :: ierr
         double precision :: dt
         ierr = 0
      end subroutine extras_after_evolve
      

      ! returns either keep_going, retry, backup, or terminate.
      integer function extras_check_model(s, id, id_extra)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra
         
         integer :: ierr
         
         ierr = 0         
         extras_check_model = terminate         
         
         call read_controls('inlist_compare_microphysics',ierr)
         if (ierr /= 0) return
         
         if (len_trim(nonstandard_format) == 0) then
            call read_fgong(fgong_file, '(5ES16.9)', ierr)
         else if (nonstandard_format == 'garstec_fgong') then
            call read_fgong(fgong_file, '(5ES25.18)', ierr)
         else if (nonstandard_format == 'guenter') then
            call read_guenter_file(fgong_file, ierr)
         end if
         if (ierr /= 0) return
         
         call compare_points(s, s% species, ierr)
         if (ierr /= 0) return

      end function extras_check_model


      integer function how_many_extra_log_columns(s, id, id_extra)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra
         how_many_extra_log_columns = 0
      end function how_many_extra_log_columns
      
      
      subroutine data_for_extra_log_columns(s, id, id_extra, n, names, vals, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra, n
         character (len=maxlen_log_column_name) :: names(n)
         double precision :: vals(n)
         integer, intent(out) :: ierr
         ierr = 0
      end subroutine data_for_extra_log_columns

      
      integer function how_many_extra_profile_columns(s, id, id_extra)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra
         how_many_extra_profile_columns = 0
      end function how_many_extra_profile_columns
      
      
      subroutine data_for_extra_profile_columns(s, id, id_extra, n, nz, names, vals, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra, n, nz
         character (len=maxlen_profile_column_name) :: names(n)
         double precision :: vals(nz,n)
         integer, intent(out) :: ierr
         integer :: k
         ierr = 0
      end subroutine data_for_extra_profile_columns
      

      ! returns either keep_going or terminate.
      integer function extras_finish_step(s, id, id_extra)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra
         integer :: ierr
         extras_finish_step = keep_going
         call store_extra_info(s)
      end function extras_finish_step
      
      
      ! routines for saving and restoring extra data so can do restarts
         
         ! put these defs at the top and delete from the following routines
         !integer, parameter :: extra_info_alloc = 1
         !integer, parameter :: extra_info_get = 2
         !integer, parameter :: extra_info_put = 3
      
      
      subroutine alloc_extra_info(s)
         integer, parameter :: extra_info_alloc = 1
         type (star_info), pointer :: s
         call move_extra_info(s,extra_info_alloc)
      end subroutine alloc_extra_info
      
      
      subroutine unpack_extra_info(s)
         integer, parameter :: extra_info_get = 2
         type (star_info), pointer :: s
         call move_extra_info(s,extra_info_get)
      end subroutine unpack_extra_info
      
      
      subroutine store_extra_info(s)
         integer, parameter :: extra_info_put = 3
         type (star_info), pointer :: s
         call move_extra_info(s,extra_info_put)
      end subroutine store_extra_info
      
      
      subroutine move_extra_info(s,op)
         integer, parameter :: extra_info_alloc = 1
         integer, parameter :: extra_info_get = 2
         integer, parameter :: extra_info_put = 3
         type (star_info), pointer :: s
         integer, intent(in) :: op
         
         integer :: i, j, num_ints, num_dbls, ierr
         
         i = 0
         ! call move_int or move_flg    
         num_ints = i
         
         i = 0
         ! call move_dbl       
         
         num_dbls = i
         
         if (op /= extra_info_alloc) return
         if (num_ints == 0 .and. num_dbls == 0) return
         
         ierr = 0
         call star_alloc_extras(s% id, num_ints, num_dbls, ierr)
         if (ierr /= 0) then
            write(*,*) 'failed in star_alloc_extras'
            write(*,*) 'alloc_extras num_ints', num_ints
            write(*,*) 'alloc_extras num_dbls', num_dbls
            stop 1
         end if
         
         contains
         
         subroutine move_dbl(dbl)
            double precision :: dbl
            i = i+1
            select case (op)
            case (extra_info_get)
               dbl = s% extra_work(i)
            case (extra_info_put)
               s% extra_work(i) = dbl
            end select
         end subroutine move_dbl
         
         subroutine move_int(int)
            integer :: int
            i = i+1
            select case (op)
            case (extra_info_get)
               int = s% extra_iwork(i)
            case (extra_info_put)
               s% extra_iwork(i) = int
            end select
         end subroutine move_int
         
         subroutine move_flg(flg)
            logical :: flg
            i = i+1
            select case (op)
            case (extra_info_get)
               flg = (s% extra_iwork(i) /= 0)
            case (extra_info_put)
               if (flg) then
                  s% extra_iwork(i) = 1
               else
                  s% extra_iwork(i) = 0
               end if
            end select
         end subroutine move_flg
      
      end subroutine move_extra_info

      end module run_star_extras
      
