! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   this file is part of mesa.
!
!   mesa is free software; you can redistribute it and/or modify
!   it under the terms of the gnu general library public license as published
!   by the free software foundation; either version 2 of the license, or
!   (at your option) any later version.
!
!   mesa is distributed in the hope that it will be useful, 
!   but without any warranty; without even the implied warranty of
!   merchantability or fitness for a particular purpose.  see the
!   gnu library general public license for more details.
!
!   you should have received a copy of the gnu library general public license
!   along with this software; if not, write to the free software
!   foundation, inc., 59 temple place, suite 330, boston, ma 02111-1307 usa
!
! ***********************************************************************
 
      module run_star_extras

      use star_lib
      use star_def
      use const_def
      
      implicit none
      
      integer :: time0, time1, clock_rate
      double precision, parameter :: expected_runtime = 4 ! minutes

      
      contains
      
      
      subroutine extras_controls(s, ierr)
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         ierr = 0
      end subroutine extras_controls
      
      
      integer function extras_startup(s, id, restart, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: id
         logical, intent(in) :: restart
         integer, intent(out) :: ierr
         ierr = 0
         extras_startup = 0
         call system_clock(time0,clock_rate)
         if (.not. restart) then
            call alloc_extra_info(s)
         else ! it is a restart
            call unpack_extra_info(s)
         end if
      end function extras_startup
      
      
      subroutine extras_after_evolve(s, id, id_extra, ierr)
         use num_lib
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra
         integer, intent(out) :: ierr
         double precision :: dt
         integer :: k, k_cntr, k_surf
         
         logical :: okay
         
         include 'formats.dek'

         
         okay = .true.

         ierr = 0
         call system_clock(time1,clock_rate)
         dt = dble(time1 - time0) / clock_rate / 60
         if (dt > 10*expected_runtime) then
            write(*,'(/,a30,2f18.6,a,/)') '>>>>>>> EXCESSIVE runtime', &
               dt, expected_runtime, '   <<<<<<<<<  ERROR'
         else
            write(*,'(/,a30,2f18.6,2i10/)') 'runtime (minutes), retries, backups', &
               dt, expected_runtime, s% num_retries, s% num_backups
         end if
         
         write(*,*)
         call check('star_mass', s% star_mass, 11.5d0, 13.0d0)
         call check('log total_angular_momentum', safe_log10(s% total_angular_momentum), 50d0, 52.5d0)
         call check('log center_omega', safe_log10(s% center_omega), -4.7d0, -4.2d0)
         call check('log h1_bdy_omega', safe_log10(s% h1_bdy_omega), -6.7d0, -4.7d0)
         call check('h1_boundary_mass', s% h1_boundary_mass, 4.9d0, 5.4d0)
         call check('log he4_bdy_omega', safe_log10(s% he4_bdy_omega), -4.7d0, -4.2d0)
         call check('he4_boundary_mass', s% he4_boundary_mass, 2.8d0, 3.1d0)
         call check('surface j_rot', safe_log10(s% j_rot(1)),  17d0, 19d0)
         call check('surface v_rot', s% omega(1)*s% r(1)*1d-5, 0.32d0, 0.45d0)
         
         k_cntr = 0
         k_surf = 0
         do k = s% nz, 1, -1
            if (s% m(k) > 6.0*Msun .and. k_cntr == 0) k_cntr = k
            if (s% m(k) > 6.4*Msun .and. k_surf == 0) k_surf = k
         end do
         
         write(*,*)
         write(*,1) 'avg from 6.0 to 6.4 Msun'
         call check('logT', avg_val(s% lnT)/ln10, 6.5d0, 6.9d0)
         call check('logRho', avg_val(s% lnd)/ln10, -3d0, -1.0d0)
         call check('log j_rot', safe_log10(avg_val(s% j_rot)), 13.2d0, 15.4d0)
         call check('r/Rsun', avg_val(s% r)/Rsun, 2d0, 4.5d0)
         call check('D_mix_non_rotation', safe_log10(avg_val(s% D_mix_non_rotation)), -11111d0, -90d0)
         call check('D_DSI', safe_log10(avg_val(s% D_DSI)),  -11111d0, -90d0)
         call check('D_SH', safe_log10(avg_val(s% D_SH)),  -11111d0, -90d0)
         call check('D_SSI', safe_log10(avg_val(s% D_SSI)),  -11111d0, -90d0)
         call check('D_ES', safe_log10(avg_val(s% D_ES)), 6.1d0, 7.9d0) 
         call check('D_GSF', safe_log10(avg_val(s% D_GSF)), 6.2d0, 7.9d0)
         call check('D_ST', safe_log10(avg_val(s% D_ST)), 7.0d0, 8.8d0)
         call check('nu_ST', safe_log10(avg_val(s% nu_ST)), 9.1d0, 10.9d0)
         call check('dynamo_B_r', safe_log10(avg_val(s% dynamo_B_r)), -1.5d0, -0.1d0)
         call check('dynamo_B_phi', safe_log10(avg_val(s% dynamo_B_phi)), 1d0, 3.5d0)
         write(*,*)
         if (okay) write(*,'(a)') 'all values are within tolerances'
         write(*,*)
         
         
         contains
         
         double precision function avg_val(v)
            double precision :: v(:)
            avg_val = dot_product(v(k_surf:k_cntr), s% dq(k_surf:k_cntr)) / sum(s% dq(k_surf:k_cntr))
         end function avg_val
         
         subroutine check(str, val, low, hi)
            double precision, intent(in) :: val, low, hi
            character (len=*) :: str
            include 'formats.dek'
            if (low <= val .and. val <= hi) then
               write(*,1) trim(str), val, low, hi
            else
               write(*,1) '*** BAD *** ' // trim(str), val, low, hi
               okay = .false.
            end if
         end subroutine check
         
         
      end subroutine extras_after_evolve
      

      ! returns either keep_going, retry, backup, or terminate.
      integer function extras_check_model(s, id, id_extra)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra
         extras_check_model = keep_going         
      end function extras_check_model


      integer function how_many_extra_log_columns(s, id, id_extra)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra
         how_many_extra_log_columns = 0
      end function how_many_extra_log_columns
      
      
      subroutine data_for_extra_log_columns(s, id, id_extra, n, names, vals, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra, n
         character (len=maxlen_log_column_name) :: names(n)
         double precision :: vals(n)
         integer, intent(out) :: ierr
         ierr = 0
      end subroutine data_for_extra_log_columns

      
      integer function how_many_extra_profile_columns(s, id, id_extra)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra
         how_many_extra_profile_columns = 0
      end function how_many_extra_profile_columns
      
      
      subroutine data_for_extra_profile_columns(s, id, id_extra, n, nz, names, vals, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra, n, nz
         character (len=maxlen_profile_column_name) :: names(n)
         double precision :: vals(nz,n)
         integer, intent(out) :: ierr
         integer :: k
         ierr = 0
      end subroutine data_for_extra_profile_columns
      

      ! returns either keep_going or terminate.
      integer function extras_finish_step(s, id, id_extra)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra
         integer :: ierr
         extras_finish_step = keep_going
         call store_extra_info(s)
      end function extras_finish_step
      
      
      ! routines for saving and restoring extra data so can do restarts
         
         ! put these defs at the top and delete from the following routines
         !integer, parameter :: extra_info_alloc = 1
         !integer, parameter :: extra_info_get = 2
         !integer, parameter :: extra_info_put = 3
      
      
      subroutine alloc_extra_info(s)
         integer, parameter :: extra_info_alloc = 1
         type (star_info), pointer :: s
         call move_extra_info(s,extra_info_alloc)
      end subroutine alloc_extra_info
      
      
      subroutine unpack_extra_info(s)
         integer, parameter :: extra_info_get = 2
         type (star_info), pointer :: s
         call move_extra_info(s,extra_info_get)
      end subroutine unpack_extra_info
      
      
      subroutine store_extra_info(s)
         integer, parameter :: extra_info_put = 3
         type (star_info), pointer :: s
         call move_extra_info(s,extra_info_put)
      end subroutine store_extra_info
      
      
      subroutine move_extra_info(s,op)
         integer, parameter :: extra_info_alloc = 1
         integer, parameter :: extra_info_get = 2
         integer, parameter :: extra_info_put = 3
         type (star_info), pointer :: s
         integer, intent(in) :: op
         
         integer :: i, j, num_ints, num_dbls, ierr
         
         i = 0
         ! call move_int or move_flg    
         num_ints = i
         
         i = 0
         ! call move_dbl       
         
         num_dbls = i
         
         if (op /= extra_info_alloc) return
         if (num_ints == 0 .and. num_dbls == 0) return
         
         ierr = 0
         call star_alloc_extras(s% id, num_ints, num_dbls, ierr)
         if (ierr /= 0) then
            write(*,*) 'failed in star_alloc_extras'
            write(*,*) 'alloc_extras num_ints', num_ints
            write(*,*) 'alloc_extras num_dbls', num_dbls
            stop 1
         end if
         
         contains
         
         subroutine move_dbl(dbl)
            double precision :: dbl
            i = i+1
            select case (op)
            case (extra_info_get)
               dbl = s% extra_work(i)
            case (extra_info_put)
               s% extra_work(i) = dbl
            end select
         end subroutine move_dbl
         
         subroutine move_int(int)
            integer :: int
            i = i+1
            select case (op)
            case (extra_info_get)
               int = s% extra_iwork(i)
            case (extra_info_put)
               s% extra_iwork(i) = int
            end select
         end subroutine move_int
         
         subroutine move_flg(flg)
            logical :: flg
            i = i+1
            select case (op)
            case (extra_info_get)
               flg = (s% extra_iwork(i) /= 0)
            case (extra_info_put)
               if (flg) then
                  s% extra_iwork(i) = 1
               else
                  s% extra_iwork(i) = 0
               end if
            end select
         end subroutine move_flg
      
      end subroutine move_extra_info

      end module run_star_extras
      
