! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   this file is part of mesa.
!
!   mesa is free software; you can redistribute it and/or modify
!   it under the terms of the gnu general library public license as published
!   by the free software foundation; either version 2 of the license, or
!   (at your option) any later version.
!
!   mesa is distributed in the hope that it will be useful, 
!   but without any warranty; without even the implied warranty of
!   merchantability or fitness for a particular purpose.  see the
!   gnu library general public license for more details.
!
!   you should have received a copy of the gnu library general public license
!   along with this software; if not, write to the free software
!   foundation, inc., 59 temple place, suite 330, boston, ma 02111-1307 usa
!
! ***********************************************************************
 
      module run_star_extras

      use star_lib
      use star_def
      use const_def
      use num_lib, only: safe_log10
      
      implicit none
      
      double precision :: max_D, age_D, lgTeff_D, lgL_D
      double precision :: max_Li7, age_Li7, lgTeff_Li7, lgL_Li7
      double precision :: age_zams, lgTeff_zams, lgL_zams

      double precision :: age_target_D, lgTeff_target_D, lgL_target_D
      double precision :: age_target_Li7, lgTeff_target_Li7, lgL_target_Li7
      double precision :: age_target_zams, lgTeff_target_zams, lgL_target_zams

      

      integer, parameter :: extra_info_alloc = 1
      integer, parameter :: extra_info_get = 2
      integer, parameter :: extra_info_put = 3

      
      integer :: time0, time1, clock_rate
      double precision, parameter :: expected_runtime = 2 ! minutes

      
      ! these routines are called by the standard run_star check_model
      contains

      
      subroutine extras_controls(s, ierr)
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         ierr = 0
      end subroutine extras_controls
      
      
      integer function extras_startup(s, id, restart, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: id
         logical, intent(in) :: restart
         integer, intent(out) :: ierr
         ierr = 0
         extras_startup = 0
         call system_clock(time0,clock_rate)
         
         max_D = -1
         max_Li7 = -1
         age_D = -1
         age_Li7 = -1
         age_zams = -1
         
         age_target_D = 1.7537325290873423D+06
         lgTeff_target_D = 3.4876691895614527D+00
         lgL_target_D = -7.3613582841842218D-01
         
         age_target_Li7 = 2.9828270099016540D+07
         lgTeff_target_Li7 = 3.5111539469872914D+00
         lgL_target_Li7 = -1.7897008212036463D+00
         
         age_target_zams = 7.9656059635014117D+08
         lgTeff_target_zams = 3.5162543018960135D+00
         lgL_target_zams = -2.3020155336836492D+00
         
         if (.not. restart) then
            call alloc_extra_info(s)
         else ! it is a restart
            call unpack_extra_info(s)
         end if
         
      end function extras_startup


      ! returns either keep_going, retry, backup, or terminate.
      integer function extras_check_model(s, id, id_extra)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra
         extras_check_model = keep_going         
      end function extras_check_model


      integer function how_many_extra_log_columns(s, id, id_extra)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra
         how_many_extra_log_columns = 0
      end function how_many_extra_log_columns
      
      
      subroutine data_for_extra_log_columns(s, id, id_extra, n, names, vals, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra, n
         character (len=maxlen_log_column_name) :: names(n)
         double precision :: vals(n)
         integer, intent(out) :: ierr
         ierr = 0
      end subroutine data_for_extra_log_columns

      
      integer function how_many_extra_profile_columns(s, id, id_extra)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra
         how_many_extra_profile_columns = 0
      end function how_many_extra_profile_columns
      
      
      subroutine data_for_extra_profile_columns(s, id, id_extra, n, nz, names, vals, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra, n, nz
         character (len=maxlen_profile_column_name) :: names(n)
         double precision :: vals(nz,n)
         integer, intent(out) :: ierr
         integer :: k
         ierr = 0
      end subroutine data_for_extra_profile_columns
      

      ! returns either keep_going, retry, backup, or terminate.
      integer function extras_finish_step(s, id, id_extra)
         use chem_def, only: ih2, ili7
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra
         
         double precision :: avg_abundance
         integer :: h2, li7, nz
         include 'formats.dek'
         
         extras_finish_step = keep_going

         h2 = s% net_iso(ih2)
         li7 = s% net_iso(ili7)
         nz = s% nz
         
         if (h2 /= 0) then
            avg_abundance = dot_product(s% dq(1:nz), s% xa(h2,1:nz))
            if (avg_abundance > max_D) max_D = avg_abundance
            if (age_D < 0 .and. avg_abundance <= max_D*0.01d0) then
               age_D = s% star_age
               lgTeff_D = log10(s% Teff)
               lgL_D = s% log_surface_luminosity
               write(*,1) 'age_D', age_D
               write(*,1) 'lgTeff_D', lgTeff_D
               write(*,1) 'lgL_D', lgL_D
            end if
         end if
         
         if (li7 /= 0) then
            avg_abundance = dot_product(s% dq(1:nz), s% xa(li7,1:nz))
            if (avg_abundance > max_Li7) max_Li7 = avg_abundance
            if (age_Li7 < 0 .and. avg_abundance <= max_Li7*0.01d0)  then
               age_Li7 = s% star_age
               lgTeff_Li7 = log10(s% Teff)
               lgL_Li7 = s% log_surface_luminosity
               write(*,1) 'age_Li7', age_Li7
               write(*,1) 'lgTeff_Li7', lgTeff_Li7
               write(*,1) 'lgL_Li7', lgL_Li7
            end if
         end if
         
         if (age_zams < 0 .and. s% power_h_burn >= s% L(1)/Lsun) then
            age_zams = s% star_age
            lgTeff_zams = log10(s% Teff)
            lgL_zams = s% log_surface_luminosity
            write(*,1) 'age_zams', age_zams
            write(*,1) 'lgTeff_zams', lgTeff_zams
            write(*,1) 'lgL_zams', lgL_zams
         end if

         call store_extra_info(s)

      end function extras_finish_step
      
      
      subroutine extras_after_evolve(s, id, id_extra, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra
         integer, intent(out) :: ierr
         
         double precision :: rel_diff, lgTeff_tol, lgL_tol, dt
         double precision :: age_D_tol, age_Li7_tol, age_zams_tol
         logical :: aok
         include 'formats.dek'
         ierr = 0
         aok = .true.
         
         age_D_tol = 1d-2
         age_Li7_tol = 1d-2
         age_zams_tol = 1d-1
         
         lgTeff_tol = 1d-2
         lgL_tol = 1d-2
         
         rel_diff = (age_D-age_target_D)/age_target_D
         if (abs(rel_diff) > age_D_tol) then
            write(*,1) 'age_D bad ***********'
            aok = .false.
         end if
         write(*,1) 'age_D', rel_diff, age_D, age_target_D
         
         rel_diff = (lgTeff_D-lgTeff_target_D)/lgTeff_target_D
         if (abs(rel_diff) > lgTeff_tol) then
            write(*,1) 'lgTeff_D bad ***********'
            aok = .false.
         end if
         write(*,1) 'lgTeff_D', rel_diff, lgTeff_D, lgTeff_target_D
         
         rel_diff = (lgL_D-lgL_target_D)/lgL_D
         if (abs(rel_diff) > lgL_tol) then
            write(*,1) 'lgL_D bad ***********'
            aok = .false.
         end if
         write(*,1) 'lgL_D', rel_diff, lgL_D, lgL_target_D
         write(*,*)
         
         rel_diff = (age_Li7-age_target_Li7)/age_target_Li7
         if (abs(rel_diff) > age_Li7_tol) then
            write(*,1) 'age_Li7 bad ***********'
            aok = .false.
         end if
         write(*,1) 'age_Li7', rel_diff, age_Li7, age_target_Li7
         
         rel_diff = (lgTeff_Li7-lgTeff_target_Li7)/lgTeff_target_Li7
         if (abs(rel_diff) > lgTeff_tol) then
            write(*,1) ' *********** lgTeff_Li7 bad'
            aok = .false.
         end if
         write(*,1) 'lgTeff_Li7', rel_diff, lgTeff_Li7, lgTeff_target_Li7
         
         rel_diff = (lgL_Li7-lgL_target_Li7)/lgL_target_Li7
         if (abs(rel_diff) > lgL_tol) then
            write(*,1) 'lgL_Li7 bad ***********'
            aok = .false.
         end if
         write(*,1) 'lgL_Li7', rel_diff, lgL_Li7, lgL_target_Li7
         write(*,*)
         
         rel_diff = (age_zams-age_target_zams)/age_target_zams
         if (abs(rel_diff) > age_zams_tol) then
            write(*,1) 'age_zams bad ***********'
            aok = .false.
         end if
         write(*,1) 'age_zams', rel_diff, age_zams, age_target_zams
         
         rel_diff = (lgTeff_zams-lgTeff_target_zams)/lgTeff_target_zams
         if (abs(rel_diff) > lgTeff_tol) then
            write(*,1) 'lgTeff_zams bad ***********'
            aok = .false.
         end if
         write(*,1) 'lgTeff_zams', rel_diff, lgTeff_zams, lgTeff_target_zams
         
         rel_diff = (lgL_zams-lgL_target_zams)/lgL_target_zams
         if (abs(rel_diff) > lgL_tol) then
            write(*,1) 'lgL_zams bad ***********'
            aok = .false.
         end if
         write(*,1) 'lgL_zams', rel_diff, lgL_zams, lgL_target_zams
         write(*,*)
         
         if (aok) then
            write(*,'(a)') 'all values are within tolerance'
         else
            write(*,'(a)') 'FAILED -- some values too far from target'
         end if

         call system_clock(time1,clock_rate)
         dt = dble(time1 - time0) / clock_rate / 60
         if (dt > 10*expected_runtime) then
            write(*,'(/,a30,2f18.6,a,/)') '>>>>>>> EXCESSIVE runtime', &
               dt, expected_runtime, '   <<<<<<<<<  ERROR'
         else
            write(*,'(/,a30,2f18.6,2i10/)') 'runtime (minutes), retries, backups', &
               dt, expected_runtime, s% num_retries, s% num_backups
         end if

      end subroutine extras_after_evolve

      
      ! routines for saving and restoring extra data so can do restarts
      
      subroutine alloc_extra_info(s)
         type (star_info), pointer :: s
         call move_extra_info(s,extra_info_alloc)
      end subroutine alloc_extra_info
      
      
      subroutine unpack_extra_info(s)
         type (star_info), pointer :: s
         call move_extra_info(s,extra_info_get)
      end subroutine unpack_extra_info
      
      
      subroutine store_extra_info(s)
         type (star_info), pointer :: s
         call move_extra_info(s,extra_info_put)
      end subroutine store_extra_info
      
      
      subroutine move_extra_info(s,op)
         type (star_info), pointer :: s
         integer, intent(in) :: op
         
         integer :: i, j, num_ints, num_dbls, ierr
         
         i = 0
         ! call move_int or move_flg    
         num_ints = i
         
         i = 0
         ! call move_dbl       
         call move_dbl(max_D)
         call move_dbl(age_D)
         call move_dbl(lgTeff_D)
         call move_dbl(lgL_D)
         call move_dbl(max_Li7)
         call move_dbl(age_Li7)
         call move_dbl(lgTeff_Li7)
         call move_dbl(lgL_Li7)
         call move_dbl(age_zams)
         call move_dbl(lgTeff_zams)
         call move_dbl(lgL_zams)
         
         num_dbls = i
         
         if (op /= extra_info_alloc) return
         if (num_ints == 0 .and. num_dbls == 0) return
         
         ierr = 0
         call star_alloc_extras(s% id, num_ints, num_dbls, ierr)
         if (ierr /= 0) then
            write(*,*) 'failed in star_alloc_extras'
            write(*,*) 'alloc_extras num_ints', num_ints
            write(*,*) 'alloc_extras num_dbls', num_dbls
            stop 1
         end if
         
         contains
         
         subroutine move_dbl(dbl)
            double precision :: dbl
            i = i+1
            select case (op)
            case (extra_info_get)
               dbl = s% extra_work(i)
            case (extra_info_put)
               s% extra_work(i) = dbl
            end select
         end subroutine move_dbl
         
         subroutine move_int(int)
            integer :: int
            i = i+1
            select case (op)
            case (extra_info_get)
               int = s% extra_iwork(i)
            case (extra_info_put)
               s% extra_iwork(i) = int
            end select
         end subroutine move_int
         
         subroutine move_flg(flg)
            logical :: flg
            i = i+1
            select case (op)
            case (extra_info_get)
               flg = (s% extra_iwork(i) /= 0)
            case (extra_info_put)
               if (flg) then
                  s% extra_iwork(i) = 1
               else
                  s% extra_iwork(i) = 0
               end if
            end select
         end subroutine move_flg
      
      end subroutine move_extra_info


      end module run_star_extras
