# profile_plots.rb

class ProfilePlots

    include Math
    include Tioga
    include FigureConstants
    
    def background
        return
        
        t.fill_color = FloralWhite
        t.fill_opacity = 0.5
        t.fill_frame
    end
    
    def show_box_labels(title, xlabel=nil, ylabel=nil)
        if title != nil
            t.show_title(title); t.no_title
        end
        if xlabel != nil
            t.show_xlabel(xlabel); t.no_xlabel
        end
        if ylabel != nil
            t.show_ylabel(ylabel); t.no_ylabel
        end
    end
    
    def show_model_number(pos = 1.15, shift = 4)
        return if t.in_subplot or t.model_number < 0 or @skip_model_number == true
        txt = sprintf('%i', t.model_number)
        t.show_text('text' => txt, 'side' => TOP, 'pos' => pos, 'shift' => shift, 'scale' => 0.8,
            'justification' => RIGHT_JUSTIFIED)
        t.show_text('text' => d.initial_mass_str, 'side' => TOP, 'pos' => pos, 'shift' => shift-0.5, 'scale' => 0.5,
            'justification' => RIGHT_JUSTIFIED)
        t.show_text('text' => d.initial_z_str, 'side' => TOP, 'pos' => pos, 'shift' => shift-1.9, 'scale' => 0.5,
            'justification' => RIGHT_JUSTIFIED)
        if d.star_mass < 0.1
            mstr = '$M\!$=%0.4f'
        elsif d.star_mass < 1
            mstr = '$M\!$=%0.3f'
        else
            mstr = '$M\!$=%0.2f'
        end
        t.show_text('text' => sprintf(mstr, d.star_mass), 
            'side' => TOP, 'pos' => pos, 'shift' => shift-3.4, 'scale' => 0.5,
            'justification' => RIGHT_JUSTIFIED)
    end
    
    def get_star_age_title(fmt)
        return sprintf(fmt, d.star_age_str)
    end
    
    def show_mass_locs(xs, ymin)
        y1 = ymin + t.default_text_height_dy * 1.5
        t.save_legend_separator(1.5)
        t.line_type = Line_Type_Dot
        t.line_width = 2

        if @show_tau10
          x = xs[d.mass.where_closest(d.tau10_mass)]
          t.stroke_color = Crimson
          t.stroke_line(x, ymin, x, y1)
          t.save_legend_info('$\tau = 10$')
        end
        
        t.line_type = Line_Type_Solid

        x = xs[d.mass.where_closest(d.star_mass * 0.9999)]
        t.stroke_color = RoyalPurple
        t.stroke_line(x, ymin, x, y1)
        t.save_legend_info('99.99\%')
        
        x = xs[d.mass.where_closest(d.star_mass * 0.999)]
        t.stroke_color = BrightBlue
        t.stroke_line(x, ymin, x, y1)
        t.save_legend_info('99.9\%')
        
        x = xs[d.mass.where_closest(d.star_mass * 0.99)]
        t.stroke_color = Goldenrod
        t.stroke_line(x, ymin, x, y1)
        t.save_legend_info('99\%')
        
        x = xs[d.mass.where_closest(d.star_mass * 0.9)]
        t.stroke_color = Lilac
        t.stroke_line(x, ymin, x, y1)
        t.save_legend_info('90\%')
        
        x = xs[d.mass.where_closest(d.star_mass * 0.5)]
        t.stroke_color = Coral
        t.stroke_line(x, ymin, x, y1)
        t.save_legend_info('50\% $ M_{tot} $')
        
    end
    
    
    def plot_polyline(xs, ys, color, label, linetype)
      t.show_polyline(xs[@grid_min..@grid_max], ys[@grid_min..@grid_max], color, label, linetype)
    end 
    
    def plot_polydots(xs, ys, color, marker, scale)
      t.show_marker('xs' => xs[@grid_min..@grid_max], 'ys' => ys[@grid_min..@grid_max], 
        'color' => color, 'marker' => marker, 'scale' => scale)
    end 

    
    def mode_propagation(xs, xlabel)
        title = get_star_age_title('Mode Propagation --- Age %s') if title == nil
        title = nil if @no_frills
         puts "@no_title #{@no_title}"
        t.set_aspect_ratio(0.8) if @no_frills
        show_box_labels(title, xlabel, 'log frequency ($\mu$ Hz)')
        t.legend_text_dy = 1        
        ymin = 0.5
        ymax = 5.0
        #unless d.log_lamb_Sl10 == nil
        # ymax10 = d.log_lamb_Sl10.max
        # ymax = ymax10 if ymax10 > ymax
        #end
        xleft = get_plot_xleft(xs)
        xright = get_plot_xright(xs)
        t.show_plot('left_boundary' => xleft, 'right_boundary' => xright,
            'top_boundary' => ymax, 'bottom_boundary' => ymin) do
            show_mesh(xs)
            cnt = 0
            cnt = plot_mode_propagation_line(cnt, xs, d.log_brunt_nu, 'log $\mathrm{N_{BV}}$')
            #cnt = plot_mode_propagation_line(cnt, xs, d.brunt_nu_structure_term.safe_log10, 'structure')
            #cnt = plot_mode_propagation_line(cnt, xs, d.brunt_nu_composition_term.safe_log10, 'composition')
            cnt = plot_mode_propagation_line(cnt, xs, d.log_lamb_Sl1, 'log Lamb S l=1')
            cnt = plot_mode_propagation_line(cnt, xs, d.log_lamb_Sl2, 'log Lamb S l=2')
            #cnt = plot_mode_propagation_line(cnt, xs, d.log_lamb_Sl3, 'log Lamb S l=3')
            cnt = plot_mode_propagation_line(cnt, xs, d.log_lamb_Sl10, 'log Lamb S l=10') unless d.log_lamb_Sl10 == nil
            r = d.photosphere_r
            m = d.star_mass
            teff = d.Teff
            #puts "r #{r}"
            #puts "m #{m}"
            #puts "teff #{teff}"
            teff_sun = 5777.0
            nu_max_sun = 3100.0
            nu_max = nu_max_sun*m/(r**2*sqrt(teff/teff_sun))
            puts "nu_max #{nu_max}"
            lg_nu_max = log10(nu_max)
            lg_2pt0_nu_max = log10(2.0*nu_max)
            lg_0pt5_nu_max = log10(0.5*nu_max)
            xs_nu = [xs.min, xs.max]
            cnt = plot_mode_propagation_line(cnt, xs_nu, 
                     [lg_2pt0_nu_max, lg_2pt0_nu_max], 'log 2.0 $\nu_\mathrm{max}$', true)
            #cnt = plot_mode_propagation_line(cnt, xs_nu, 
            #         [lg_nu_max, lg_nu_max], 'log 1.0 nu max', true)
            cnt = plot_mode_propagation_line(cnt, xs_nu, 
                     [lg_0pt5_nu_max, lg_0pt5_nu_max], 'log 0.5 $\nu_\mathrm{max}$', true)
        end
    end

    
    def plot_mode_propagation_line(cnt, xs, ys_in, txt, use_given_xs_ys = false)
        show_lim = 1e-50 #10**ymin
        return cnt if ys_in == nil
        return cnt unless ys_in.max > show_lim

        ys = ys_in
        if use_given_xs_ys
           xs_plot = xs
           ys_plot = ys
        else
           xs_plot = xs[@grid_min..@grid_max]
           ys_plot = ys[@grid_min..@grid_max]
        end
        
        colors = [ BrightBlue, Goldenrod, Coral, FireBrick, RoyalPurple, Lilac ]
        num_colors = colors.length
        patterns = [ Line_Type_Solid, Line_Type_Dash, Line_Type_Dot, Line_Type_Dot_Dash, Line_Type_Dot_Long_Dash ]
        num_patterns = patterns.length
        
        color = colors[cnt - num_colors*(cnt/num_colors)]
        pattern = patterns[cnt/num_colors]

        t.show_polyline(xs_plot, ys_plot, color, txt, pattern)

        return cnt+1
    end 

    
    def brunt_nu_terms(xs, xlabel)
        title = get_star_age_title('Brunt terms --- Age %s') if title == nil
        title = nil if @no_frills
        t.set_aspect_ratio(0.8) if @no_frills
        show_box_labels(title, xlabel, 'frequency ($\mu$ Hz)')
        t.legend_text_dy = 1 
        ymin = 100
        ymax = 2600
        xleft = get_plot_xleft(xs)
        xright = get_plot_xright(xs)
        t.show_plot('left_boundary' => xleft, 'right_boundary' => xright,
            'top_boundary' => ymax, 'bottom_boundary' => ymin) do
            show_mesh(xs)
            cnt = 0
            cnt = plot_brunt_terms_line(cnt, xs, d.brunt_nu, '$\mathrm{N_{BV}}$')
            cnt = plot_brunt_terms_line(cnt, xs, d.brunt_nu_structure_term, 'structure')
            cnt = plot_brunt_terms_line(cnt, xs, d.brunt_nu_composition_term, 'composition')
            cnt = plot_brunt_terms_line(cnt, xs, d.lamb_Sl1, 'Lamb l=1')
            cnt = plot_brunt_terms_line(cnt, xs, d.lamb_Sl2, 'Lamb l=2')
            cnt = plot_brunt_terms_line(cnt, xs, d.lamb_Sl3, 'Lamb l=3')
        end
    end

    
    def plot_brunt_terms_line(cnt, xs, ys_in, txt, use_given_xs_ys = false)
        show_lim = 1e-50 #10**ymin
        return cnt if ys_in == nil
        #return cnt unless ys_in.max > show_lim

        ys = ys_in
        if use_given_xs_ys
           xs_plot = xs
           ys_plot = ys
        else
           xs_plot = xs[@grid_min..@grid_max]
           ys_plot = ys[@grid_min..@grid_max]
        end
        
        colors = [ BrightBlue, Goldenrod, Coral, FireBrick, RoyalPurple, Lilac ]
        num_colors = colors.length
        patterns = [ Line_Type_Solid, Line_Type_Dash, Line_Type_Dot, Line_Type_Dot_Dash, Line_Type_Dot_Long_Dash ]
        num_patterns = patterns.length
        
        color = colors[cnt - num_colors*(cnt/num_colors)]
        pattern = patterns[cnt/num_colors]

        t.show_polyline(xs_plot, ys_plot, color, txt, pattern)

        return cnt+1
    end 


    
    def abundances(xs, xlabel)
        title = get_star_age_title('Abundances --- Age %s') if title == nil
        title = nil if @no_frills

        t.set_aspect_ratio(0.8) if @no_frills
        show_box_labels(title, xlabel, 'log mass fraction')
        t.legend_text_dy = 1        
        ymin = @log_mass_frac_ymin
        if ymin == nil
            ymin = -15.1
            lgz = log10(d.initial_z+1e-9)-1
            ymin = lgz if lgz-1 < ymin
        end 
        ymax = @log_mass_frac_ymax
        if ymax == nil
            if ymin < -8
                ymax = 0.5
            else
                ymax = 0.5
            end
        end
        show_plot_abundances(xs, ymin, ymax)
    end

    
    
    def plot_abundance_line(cnt, xs, ys_in, chem_name)
        show_lim = 1e-50 #10**ymin
        return cnt if ys_in == nil
        return cnt unless ys_in.max > show_lim
        return cnt+1 if @abundances_to_skip.include?(chem_name)
        ys = ys_in.safe_log10
        xs_plot = xs[@grid_min..@grid_max]
        ys_plot = ys[@grid_min..@grid_max]
        txt = nil
        txt = '\it ' + chem_name if @legend_abundance_lines == true || @doing_multiple != true
        
        colors = [ BrightBlue, Goldenrod, Coral, FireBrick, RoyalPurple, Lilac ]
        num_colors = colors.length
        patterns = [ Line_Type_Solid, Line_Type_Dash, Line_Type_Dot, Line_Type_Dot_Dash, Line_Type_Dot_Long_Dash ]
        num_patterns = patterns.length
        
        color = colors[cnt - num_colors*(cnt/num_colors)]
        pattern = patterns[cnt/num_colors]

        t.show_polyline(xs_plot, ys_plot, color, txt, pattern)
        t.show_marker('xs' => xs_plot, 'ys' => ys_plot, 'marker' => Bullet,
            'color' => color, 'scale' => 0.4) if @show_points
        
        return cnt+1 unless @num_abundance_line_labels > 0
        return cnt+1 if txt == nil
        
        xwidth = xs_plot.max - xs_plot.min
        n = @num_abundance_line_labels
        dx = xwidth/n
        xs_reversed = xs_plot.reverse # need to reverse for use with linear_interpolate
        ys_reversed = ys_plot.reverse
        scale = 0.6
        dy = t.default_text_height_dy*scale/3
        n.times do |j|
            x = xs_plot.min + dx*(j+0.5)
            y = Dvector.linear_interpolate(x, xs_reversed, ys_reversed)
            if y < 0*dy
                y = y + dy
            else
                y = y - dy*0
            end
            t.show_label(
                'x' => x, 'y' => y, 
                'text' => txt, 'scale' => scale, 'alignment' => ALIGNED_AT_BASELINE)
        end
        return cnt+1
    end 

    
    def show_plot_abundances(xs, ymin, ymax)
        xleft = get_plot_xleft(xs)
        xright = get_plot_xright(xs)
        t.show_plot('left_boundary' => xleft, 'right_boundary' => xright,
            'top_boundary' => ymax, 'bottom_boundary' => ymin) do
            show_mesh(xs)
            cnt = 0
            cnt = plot_abundance_line(cnt, xs, d.h1, '$^{1}$\rm H')
            cnt = plot_abundance_line(cnt, xs, d.h2, '$^{2}$\rm H')
            cnt = plot_abundance_line(cnt, xs, d.he3, '$^{3}$\rm He')
            cnt = plot_abundance_line(cnt, xs, d.he4, '$^{4}$\rm He')
            cnt = plot_abundance_line(cnt, xs, d.li7, '$^{7}$\rm Li')
            cnt = plot_abundance_line(cnt, xs, d.be7, '$^{7}$\rm Be')
            cnt = plot_abundance_line(cnt, xs, d.b8, '$^{8}$\rm B')
            cnt = plot_abundance_line(cnt, xs, d.c12, '$^{12}$\rm C')
            cnt = plot_abundance_line(cnt, xs, d.c13, '$^{13}$\rm C')
            cnt = plot_abundance_line(cnt, xs, d.c14, '$^{14}$\rm C')
            cnt = plot_abundance_line(cnt, xs, d.n13, '$^{13}$\rm N')
            cnt = plot_abundance_line(cnt, xs, d.n14, '$^{14}$\rm N')
            cnt = plot_abundance_line(cnt, xs, d.n15, '$^{15}$\rm N')
            cnt = plot_abundance_line(cnt, xs, d.o14, '$^{14}$\rm O')
            cnt = plot_abundance_line(cnt, xs, d.o15, '$^{15}$\rm O')
            cnt = plot_abundance_line(cnt, xs, d.o16, '$^{16}$\rm O')
            cnt = plot_abundance_line(cnt, xs, d.o17, '$^{17}$\rm O')
            cnt = plot_abundance_line(cnt, xs, d.o18, '$^{18}$\rm O')
            cnt = plot_abundance_line(cnt, xs, d.f17, '$^{17}$\rm F')
            cnt = plot_abundance_line(cnt, xs, d.f18, '$^{18}$\rm F')
            cnt = plot_abundance_line(cnt, xs, d.f19, '$^{19}$\rm F')
            cnt = plot_abundance_line(cnt, xs, d.ne18, '$^{18}$\rm Ne')
            cnt = plot_abundance_line(cnt, xs, d.ne19, '$^{19}$\rm Ne')
            cnt = plot_abundance_line(cnt, xs, d.ne20, '$^{20}$\rm Ne')
            cnt = plot_abundance_line(cnt, xs, d.ne21, '$^{21}$\rm Ne')
            cnt = plot_abundance_line(cnt, xs, d.ne22, '$^{22}$\rm Ne')
            cnt = plot_abundance_line(cnt, xs, d.na23, '$^{23}$\rm Na')
            cnt = plot_abundance_line(cnt, xs, d.mg22, '$^{22}$\rm Mg')
            cnt = plot_abundance_line(cnt, xs, d.mg24, '$^{24}$\rm Mg')
            cnt = plot_abundance_line(cnt, xs, d.mg26, '$^{26}$\rm Mg')
            cnt = plot_abundance_line(cnt, xs, d.al27, '$^{27}$\rm Al')
            cnt = plot_abundance_line(cnt, xs, d.si28, '$^{28}$\rm Si')
            cnt = plot_abundance_line(cnt, xs, d.p31, '$^{31}$\rm P')
            cnt = plot_abundance_line(cnt, xs, d.s30, '$^{30}$\rm S')
            cnt = plot_abundance_line(cnt, xs, d.s32, '$^{32}$\rm S')
            cnt = plot_abundance_line(cnt, xs, d.s34, '$^{34}$\rm S')
            cnt = plot_abundance_line(cnt, xs, d.cl35, '$^{35}$\rm Cl')
            cnt = plot_abundance_line(cnt, xs, d.ar36, '$^{36}$\rm Ar')
            cnt = plot_abundance_line(cnt, xs, d.k39, '$^{39}$\rm K')
            cnt = plot_abundance_line(cnt, xs, d.ca40, '$^{40}$\rm Ca')
            cnt = plot_abundance_line(cnt, xs, d.sc43, '$^{43}$\rm Sc')
            cnt = plot_abundance_line(cnt, xs, d.ti44, '$^{44}$\rm Ti')
            cnt = plot_abundance_line(cnt, xs, d.v47, '$^{47}$\rm V')
            cnt = plot_abundance_line(cnt, xs, d.cr48, '$^{48}$\rm Cr')
            cnt = plot_abundance_line(cnt, xs, d.mn51, '$^{51}$\rm Mn')
            cnt = plot_abundance_line(cnt, xs, d.fe52, '$^{52}$\rm Fe')
            cnt = plot_abundance_line(cnt, xs, d.fe53, '$^{53}$\rm Fe')
            cnt = plot_abundance_line(cnt, xs, d.fe54, '$^{54}$\rm Fe')
            cnt = plot_abundance_line(cnt, xs, d.fe56, '$^{56}$\rm Fe')
            cnt = plot_abundance_line(cnt, xs, d.co55, '$^{55}$\rm Co')
            cnt = plot_abundance_line(cnt, xs, d.ni53, '$^{53}$\rm Ni')
            cnt = plot_abundance_line(cnt, xs, d.ni55, '$^{55}$\rm Ni')
            cnt = plot_abundance_line(cnt, xs, d.ni56, '$^{56}$\rm Ni')
            cnt = plot_abundance_line(cnt, xs, d.zn60, '$^{60}$\rm Zn')
            cnt = plot_abundance_line(cnt, xs, d.ge64, '$^{64}$\rm Ge')
            cnt = plot_abundance_line(cnt, xs, d.se68, '$^{68}$\rm Se')
            cnt = plot_abundance_line(cnt, xs, d.kr72, '$^{72}$\rm Kr')
            cnt = plot_abundance_line(cnt, xs, d.sr76, '$^{76}$\rm Sr')
            cnt = plot_abundance_line(cnt, xs, d.sn104, '$^{104}$\rm Sn')
            cnt = plot_abundance_line(cnt, xs, d.cr56, '``$^{56}$\rm Cr"')
            cnt = plot_abundance_line(cnt, xs, d.neut, 'neut')
            cnt = plot_abundance_line(cnt, xs, d.prot, 'prot')
            if false && @no_frills
                t.show_label('x' => 0.175, 'y' => -0.7, 'scale' => 0.7, 
                    'justification' => LEFT_JUSTIFIED, 'alignment' => ALIGNED_AT_MIDHEIGHT, 
                    'text' => 'M_0 = 25 $\mathrm{M_\odot}$')
                t.show_label('x' => 0.175, 'y' => -0.85, 'scale' => 0.7, 
                    'justification' => LEFT_JUSTIFIED, 'alignment' => ALIGNED_AT_MIDHEIGHT, 
                    'text' => 'Z_0 = 0.02')
            end
        end
    end 


    
    def mixing_Ds(xs, xlabel)
        title = get_star_age_title('mixing Ds --- Age %s') if title == nil
        title = nil if @no_frills

        t.set_aspect_ratio(0.8) if @no_frills
        show_box_labels(title, xlabel, 'log D')
        t.legend_text_dy = 1        
        ymin = -2
        ymax_factor = 1.1
        ymax = d.log_D_mix[@grid_min..@grid_max].max
        if (ymax > 0)
            ymax = ymax*ymax_factor
        else
            ymax = 0
        end
        if d.am_log_D_SSI != nil
            ymax2 = d.am_log_D_SSI[@grid_min..@grid_max].max
            if (ymax2 > 0)
                ymax2 = ymax2*ymax_factor
            else
                ymax2 = 0
            end if
            ymax = ymax2 if ymax2 > ymax
        end
        if d.am_log_D_ES != nil
            ymax2 = d.am_log_D_ES[@grid_min..@grid_max].max
            if (ymax2 > 0)
                ymax2 = ymax2*ymax_factor
            else
                ymax2 = 0
            end if
            ymax = ymax2 if ymax2 > ymax
        end
        if d.am_log_D_GSF != nil
            ymax2 = d.am_log_D_GSF[@grid_min..@grid_max].max
            if (ymax2 > 0)
                ymax2 = ymax2*ymax_factor
            else
                ymax2 = 0
            end if
            ymax = ymax2 if ymax2 > ymax
        end
        if d.am_log_D_ST != nil
            ymax2 = d.am_log_D_ST[@grid_min..@grid_max].max
            if (ymax2 > 0)
                ymax2 = ymax2*ymax_factor
            else
                ymax2 = 0
            end if
            ymax = ymax2 if ymax2 > ymax
        end
        show_plot_mixing_Ds(xs, ymin, ymax)
    end
    
    
    def plot_D_line(cnt, xs, ys, mix_D_name)
        show_lim = -3
        return cnt+1 if ys == nil
        return cnt+1 unless ys.max > show_lim
        xs_plot = xs[@grid_min..@grid_max]
        ys_plot = ys[@grid_min..@grid_max]
        txt = nil
        txt = mix_D_name if @legend_mixing_D_lines == true || @doing_multiple != true
        
        colors = [ Teal, Goldenrod, Lilac, Coral, FireBrick, MediumSlateBlue,
            LightSkyBlue, LightOliveGreen, SeaGreen, Beige]
        num_colors = colors.length
        patterns = [ Line_Type_Solid, Line_Type_Dash, Line_Type_Dot, Line_Type_Dot_Dash, Line_Type_Dot_Long_Dash ]
        num_patterns = patterns.length
        
        color = colors[cnt - num_colors*(cnt/num_colors)]
        pattern = patterns[cnt/num_colors]

        t.show_polyline(xs_plot, ys_plot, color, txt, pattern)
        t.show_marker('xs' => xs_plot, 'ys' => ys_plot, 'marker' => Bullet,
            'color' => color, 'scale' => 0.4) if @show_points
        
        return cnt+1 unless @num_mixing_D_line_labels > 0
        return cnt+1 if txt == nil
        
        xwidth = xs_plot.max - xs_plot.min
        n = @num_mixing_D_line_labels
        dx = xwidth/n
        xs_reversed = xs_plot.reverse # need to reverse for use with linear_interpolate
        ys_reversed = ys_plot.reverse
        scale = 0.6
        dy = t.default_text_height_dy*scale/3
        n.times do |j|
            x = xs_plot.min + dx*(j+0.5)
            y = Dvector.linear_interpolate(x, xs_reversed, ys_reversed)
            if y < 0*dy
                y = y + dy
            else
                y = y - dy*0
            end
            t.show_label(
                'x' => x, 'y' => y, 
                'text' => txt, 'scale' => scale, 'alignment' => ALIGNED_AT_BASELINE)
        end
        return cnt+1
    end 

    
    def show_plot_mixing_Ds(xs, ymin, ymax)
        @num_mixing_D_line_labels = 0
        @legend_mixing_D_lines = true
        xleft = get_plot_xleft(xs)
        xright = get_plot_xright(xs)
        t.show_plot('left_boundary' => xleft, 'right_boundary' => xright,
            'top_boundary' => ymax, 'bottom_boundary' => ymin) do
            show_mesh(xs)
            cnt = 0
            cnt = plot_D_line(cnt, xs, d.am_log_D_ST, 'ST')
            cnt = plot_D_line(cnt, xs, d.am_log_D_DSI, 'DSI')
            cnt = plot_D_line(cnt, xs, d.am_log_D_SH, 'SH')
            cnt = plot_D_line(cnt, xs, d.am_log_D_SSI, 'SSI')
            cnt = plot_D_line(cnt, xs, d.am_log_D_ES, 'ES')
            cnt = plot_D_line(cnt, xs, d.am_log_D_GSF, 'GSF')
            cnt = plot_D_line(cnt, xs, d.log_D_conv, 'conv')
            cnt = plot_D_line(cnt, xs, d.log_D_semi, 'semi')
            cnt = plot_D_line(cnt, xs, d.log_D_th, 'th')
            cnt = plot_D_line(cnt, xs, d.log_D_ovr, 'ovr')
        end
    end 

    
    def dynamo_info
        @no_frills = true
        title = get_star_age_title('dynamo information --- Age %s') if title == nil
        title = nil if @no_frills

        t.set_aspect_ratio(0.8) if @no_frills
        
        t.rescale(1.3)

        xs = xs_for_plot()
        xlabel = xlabel_for_plot()

        show_box_labels(title, xlabel, nil)
        xleft = get_plot_xleft(xs)
        xright = get_plot_xright(xs)
        xs = xs[@grid_min..@grid_max]
        
      t.legend_text_dy = 1.0
      #show_model_number
      t.show_plot_with_legend('legend_left_margin' => 0.05, 'legend_top_margin' => 0.01,
                              'plot_scale' => 1,
                              'legend_scale' => 1.3,
                              'plot_right_margin' => 0) { 
        # NOTE: use subfigure rather than subplot to get combined legend info
        # subplot would be correct if each had its own legend
            t.subfigure {
               t.yaxis_loc = t.ylabel_side = LEFT
               t.right_edge_type = AXIS_HIDDEN
               t.show_ylabel('log B')
               ymin = -2.1
               ymax = 10
               t.show_plot('left_boundary' => xleft, 'right_boundary' => xright,
                   'top_boundary' => ymax, 'bottom_boundary' => ymin) do
                    log_B_r = get_by_name_for_plot('dynamo_log_B_r')
                    log_B_phi = get_by_name_for_plot('dynamo_log_B_phi')
                    mixing_type = d.mixing_type[@grid_min..@grid_max]
                    t.show_plot('left_boundary' => xleft, 'right_boundary' => xright,
                       'top_boundary' => ymax, 'bottom_boundary' => ymin) do
                       show_convection_regions(mixing_type, xs, xleft, xright, ymin, ymax)
                    end
                    t.show_polyline(xs, log_B_r, Coral, 'B_r')
                    t.show_polyline(xs, log_B_phi, Teal, 'B_{\phi}')
               end
             }
            t.subfigure {
               t.yaxis_loc = t.ylabel_side = RIGHT
               t.left_edge_type = AXIS_HIDDEN
               t.show_ylabel('log $(\omega)$ and log $(j \times 10^{-20}$)')
               ymin = -11
               ymax = 1
               t.show_plot('left_boundary' => xleft, 'right_boundary' => xright,
                   'top_boundary' => ymax, 'bottom_boundary' => ymin) do
                    log_j = get_by_name_for_plot('log_j_rot')
                    log_j.sub!(20)
                    log_omega = get_by_name_for_plot('log_omega')
                    t.show_polyline(xs, log_omega, BrightBlue, '\omega')
                    t.show_polyline(xs, log_j, FireBrick, 'j')
               end
             }
         }
         
         log_J_inside = d.log_J_inside
         return if log_J_inside == nil
         
         ms = [ 1.5, 2.5, 3.5 ]
         ms.each do |m|
             k = d.mass.where_closest(m)
             log_J = log_J_inside[k]
             puts "m #{m} log_J #{log_J} J #{10**(log_J)}"
         end
        
    end 
    
    
    def show_convection_regions(mixing_type, xs, xleft, xright, ymin, ymax)
        conv_type = 1
        k = 0; k0 = 0
        in_conv_region = (mixing_type[k].to_i == conv_type)
        t.fill_color = DarkSmoke
        mixing_type.length.times do |k|
            if in_conv_region
                if mixing_type[k].to_i != conv_type
                    dx = xs[k]-xs[k0]
                    if dx > 0
                        t.fill_rect(xs[k0], ymin, dx, ymax-ymin)
                    else
                        t.fill_rect(xs[k], ymin, -dx, ymax-ymin)
                    end
                    in_conv_region = false
                end
            else
                if mixing_type[k].to_i == conv_type
                    in_conv_region = true
                    k0 = k
                end
            end
        end
        if in_conv_region
            k = -1
            dx = xs[k]-xs[k0]
            if dx > 0
                t.fill_rect(xs[k0], ymin, dx, ymax-ymin)
            else
                t.fill_rect(xs[k], ymin, -dx, ymax-ymin)
            end
        end
    end

    
    def signed_scaled_log10(v,scale)
      v.map {|x| (x > 0)? log10(1+x*scale) : -log10(1-x*scale) }
    end
    
    
    def show_y0(xleft, xright)
      t.context do
        t.line_type = Line_Type_Dash
        t.stroke_color = Black
        t.line_width = 0.5
        t.stroke_line(xleft,0,xright,0)
      end
    end

    
    def rho_T_2Ys(xs, title, xlabel, xleft, xright)
        t.subplot {
            t.yaxis_loc = t.ylabel_side = LEFT;
            t.right_edge_type = AXIS_HIDDEN; density(xs, title, xlabel, xleft, xright) }
        t.subplot {
            t.yaxis_loc = t.ylabel_side = RIGHT;
            t.left_edge_type = AXIS_HIDDEN; temperature(xs, title, xlabel, xleft, xright) }
    end
    
    def convection(xs, xlabel, ymin = -0.4, ymax = 1.3)
        xleft = get_plot_xleft(xs)
        xright = get_plot_xright(xs)
        title = get_star_age_title('Convection --- Age %s') if title == nil
        show_box_labels(title, xlabel, '$ dlnT/dlnP $')
        #ymin = 0.22; ymax = 0.27
        t.show_plot('left_boundary' => xleft, 'right_boundary' => xright,
            'top_boundary' => ymax, 'bottom_boundary' => ymin) do
            show_mesh(xs)
            plot_polyline(xs, d.grada, BrightBlue, '$ \nabla_{ad} $', Line_Type_Solid)
            plot_polyline(xs, d.gradr, Goldenrod, '$ \nabla_{rad} $', Line_Type_Solid)
            plot_polyline(xs, d.gradT, Crimson, '$ \nabla_{actual} $', Line_Type_Dot)
            show_mass_locs(xs, ymin)
        end
    end
    
    def power_line(cnt, xs, vals, text, ymin)
        return cnt if vals == nil
        ys = @temp_vec2
        ys.replace(vals).safe_log10!
        return cnt if ys.max < ymin
        
        colors = [ BrightBlue, Goldenrod, Coral, Lilac, FireBrick, RoyalPurple ]
        num_colors = colors.length
        patterns = [ Line_Type_Solid, Line_Type_Dash, Line_Type_Dot, Line_Type_Dot_Dash, Line_Type_Dot_Long_Dash ]
        num_patterns = patterns.length
        
        color = colors[cnt - num_colors*(cnt/num_colors)]
        pattern = patterns[cnt/num_colors]

        plot_polyline(xs, ys, color, text, pattern)
        t.show_marker('xs' => xs, 'ys' => ys, 'marker' => Bullet,
            'color' => color, 'scale' => 0.4) if @show_points
        cnt = cnt+1
    end
    
    
    def power(xs, xlabel, ymin = -4.1, ymax = 15)
        power_plot(xs, xlabel, ymin, ymax, false)
    end
    
    
    def power_plus(xs, xlabel, ymin = -4.1, ymax = 15)
        power_plot(xs, xlabel, ymin, ymax, true)
    end

    
    def power_plot(xs, xlabel, ymin, ymax, do_plus)
        title = get_star_age_title('Power --- Age %s') if title == nil
        show_box_labels(title, xlabel, 'log $ergs/g/s$')
        xleft = get_plot_xleft(xs)
        xright = get_plot_xright(xs)
        
        neg_photo = d.photo.neg

        ary = [ d.non_nuc_neu, 
                d.pp,
                d.cno,
                d.tri_alfa,
                d.burn_c,
                d.burn_n,
                d.burn_o,
                d.burn_ne,
                d.burn_na,
                d.burn_mg,
                d.burn_si,
                d.burn_s,
                d.burn_ar,
                d.burn_ca,
                d.burn_ti,
                d.burn_cr,
                d.burn_fe,
                d.c12_c12,
                d.c12_o16,
                d.o16_o16,
                d.pnhe4,
                d.other,
                neg_photo
                ]
        
        ary << d.dL_dm if do_plus
                
        ymax = Dvector.max_of_many(ary)
        ymax = 1e-99 if ymax < 1e-99
        ymax = ymax.log10
        if ymax <= 14
            ymax = 15.1
        else
            ymax = ymax*1.1
            ymin = -0.1
        end

        t.show_plot('left_boundary' => xleft, 'right_boundary' => xright,
            'top_boundary' => ymax, 'bottom_boundary' => ymin) do
            show_mesh(xs)
            
            cnt = 0
            cnt = power_line(cnt, xs, d.non_nuc_neu, '\it neutrino losses', ymin)
            cnt = power_line(cnt, xs, neg_photo, '\it photodisintegration', ymin)
            
            cnt = power_line(cnt, xs, d.pp, '\it PP', ymin)
            cnt = power_line(cnt, xs, d.cno, '\it CNO', ymin)
            cnt = power_line(cnt, xs, d.tri_alfa, '$ 3 alpha $', ymin)

            cnt = power_line(cnt, xs, d.c12_c12, '$ C12+C12 $', ymin)
            cnt = power_line(cnt, xs, d.c12_o16, '$ C12+O16 $', ymin)            
            cnt = power_line(cnt, xs, d.o16_o16, '$ O16+O16 $', ymin)
           
            cnt = power_line(cnt, xs, d.burn_c, '\it C', ymin)
            cnt = power_line(cnt, xs, d.burn_n, '\it N', ymin)
            cnt = power_line(cnt, xs, d.burn_o, '\it O', ymin)
            cnt = power_line(cnt, xs, d.burn_ne, '\it Ne', ymin)
            cnt = power_line(cnt, xs, d.burn_na, '\it Na', ymin)
            cnt = power_line(cnt, xs, d.burn_mg, '\it Mg', ymin)
            cnt = power_line(cnt, xs, d.burn_si, '\it Si', ymin)
            cnt = power_line(cnt, xs, d.burn_s, '\it S', ymin)
            cnt = power_line(cnt, xs, d.burn_ar, '\it Ar', ymin)
            cnt = power_line(cnt, xs, d.burn_ca, '\it Ca', ymin)
            cnt = power_line(cnt, xs, d.burn_ti, '\it Ti', ymin)
            cnt = power_line(cnt, xs, d.burn_cr, '\it Cr', ymin)
            cnt = power_line(cnt, xs, d.burn_fe, '\it Fe', ymin)
            
            cnt = power_line(cnt, xs, d.pnhe4, '$ 2 n + 2 p \rightarrow He4 $', ymin)
            cnt = power_line(cnt, xs, d.other, '\it Other', ymin)
            cnt = power_line(cnt, xs, d.dL_dm, '$ dL/dm $', ymin) if do_plus

        end
    end
    
    
    def neutrinos(xs, xlabel, ymin = -11, ymax = 16)
        xleft = get_plot_xleft(xs)
        xright = get_plot_xright(xs)
        title = get_star_age_title('Neutrinos --- Age %s') if title == nil
        show_box_labels(title, xlabel, 'log $ergs/g/s$')
        t.show_plot('left_boundary' => xleft, 'right_boundary' => xright,
            'top_boundary' => ymax, 'bottom_boundary' => ymin) do
            show_mesh(xs)
            cnt = 0
            cnt = power_line(cnt, xs, d.nonnucneu_phot, 'photo neutrinos', ymin)
            cnt = power_line(cnt, xs, d.nonnucneu_plas, 'plasma neutrinos', ymin)
            cnt = power_line(cnt, xs, d.nonnucneu_brem, 'brem neutrinos', ymin)
            cnt = power_line(cnt, xs, d.nonnucneu_pair, 'pair neutrinos', ymin)
            cnt = power_line(cnt, xs, d.nonnucneu_reco, 'recombination neutrinos', ymin)
            show_mass_locs(xs, ymin) unless xs == d.mass || xs == d.mmid
        end
    end
    
    
    def abundance_C_O_Ne_Mg(xs, xlabel, ymin = -6, ymax = 14)
        xleft = get_plot_xleft(xs)
        xright = get_plot_xright(xs)
        title = get_star_age_title('Combined C, O, Ne, Mg --- Age %s') if title == nil
        show_box_labels(title, xlabel, 'mass fraction')
        ys = d.c12.dup
        ys = ys.add!(d.o16)
        ys = ys.add!(d.ne20)
        ys = ys.add!(d.mg24)
        ys_to_plot = ys[@grid_min..@grid_max]
        ymax = ys_to_plot.max
        ymin = ys_to_plot.min
        t.show_plot('left_boundary' => xleft, 'right_boundary' => xright,
            'top_boundary' => ymax, 'bottom_boundary' => ymin) do
            t.show_polyline(xs,ys,BrightBlue)
            t.show_marker('xs' => xs, 'ys' => ys, 'marker' => Bullet,
                'color' => BrightBlue, 'scale' => 0.4) if @show_points
        end
    end
    
    
    def abundance_Fe_core(xs, xlabel, ymin = -6, ymax = 14)
        xleft = get_plot_xleft(xs)
        xright = get_plot_xright(xs)
        title = get_star_age_title('Combined Fe and Ni --- Age %s') if title == nil
        show_box_labels(title, xlabel, 'mass fraction')
        ys = Dvector.new(xs.length,0)
        ys = ys.add!(d.fe52) unless d.fe52 == nil
        ys = ys.add!(d.fe53) unless d.fe53 == nil
        ys = ys.add!(d.fe54) unless d.fe54 == nil
        ys = ys.add!(d.co55) unless d.co55 == nil
        ys = ys.add!(d.fe56) unless d.fe56 == nil
        ys = ys.add!(d.ni56) unless d.ni56 == nil
        ys = ys.add!(d.cr56) unless d.cr56 == nil
        ys_to_plot = ys[@grid_min..@grid_max]
        ymax = ys_to_plot.max
        ymin = ys_to_plot.min
        t.show_plot('left_boundary' => xleft, 'right_boundary' => xright,
            'top_boundary' => ymax, 'bottom_boundary' => ymin) do
            t.show_polyline(xs,ys,BrightBlue)
            t.show_marker('xs' => xs, 'ys' => ys, 'marker' => Bullet,
                'color' => BrightBlue, 'scale' => 0.4) if @show_points
        end
    end
        
        
    def show_mesh(xs)
        return unless @show_mesh_flag == true
        return if @xaxis_by == 'grid'
        ymin = t.bounds_ymin; ymax = t.bounds_ymax
        t.line_color = LightGray; lw = t.line_width; t.line_width = 0.6
        xs.each_with_index { |x, i| t.stroke_line(x, ymin, x, ymax) if i.mod(10) == 1 }
        t.line_width = lw
    end
    
    def show_properites(name, xs, xlabel)
        t.landscape unless t.in_subplot
        #t.portrait
        show_model_number unless @no_frills
        t.top_edge_type = AXIS_LINE_ONLY
        with_legend = ! @no_legend
        if (with_legend)
            t.rescale(@legend_scale)
            t.legend_text_dy = 1.1
            t.subplot('right_margin' => @plot_with_legend_right_margin) do
                background
                eval_str = sprintf("%s(xs, xlabel)", name)
                eval(eval_str)
            end
            t.set_subframe('left_margin' => @legend_left_margin, 'top_margin' => 0.05)
            t.show_legend
        else
            eval(sprintf("%s(xs, xlabel)", name))
        end
    end

    
    def xs_for_plot()
      if @xaxis_by == 'mass'
         if @use_mmid && d.mmid != nil
            xs = d.mmid
         else
            xs = d.mass
         end
      elsif @xaxis_by == 'q'
          xs = d.q
      elsif @xaxis_by == 'logT'
          xs = d.logT
      elsif @xaxis_by == 'logP'
          xs = d.logP
      elsif @xaxis_by == 'log_column_depth'
          xs = d.log_column_depth
      elsif @xaxis_by == 'radius'
          xs = d.radius
      elsif @xaxis_by == 'r_div_R'
          xs = d.r_div_R
      elsif @xaxis_by == 'acoustic_r_div_R_phot'
          xs = d.acoustic_r_div_R_phot
      elsif @xaxis_by == 'logR'
          xs = d.logR
      elsif @xaxis_by == 'logxm'
          xs = d.logxm
      elsif @xaxis_by == 'logxq'
          xs = d.logxq
      else
          xs = d.zone
          puts("problem in xs_for_plot @xaxis_by = #{@xaxis_by}") unless @xaxis_by == 'grid'
      end
      return xs
    end

    
    def xaxis_reversed?()
      return true if @xaxis_by == 'grid'
      return true if @xaxis_by == 'logP'
      return true if @xaxis_by == 'logxm'
      return true if @xaxis_by == 'logxq'
      return true if @xaxis_by == 'log_column_depth'
      return false
    end


    
    def xlabel_for_plot()
      if @xaxis_by == 'mass'
          xlabel = @mass_xlabel
      elsif @xaxis_by == 'q'
          xlabel = @q_xlabel
      elsif @xaxis_by == 'logT'
          xlabel = @logT_xlabel
      elsif @xaxis_by == 'logP'
          xlabel = @logP_xlabel
      elsif @xaxis_by == 'log_column_depth'
          xlabel = @log_col_depth_xlabel
      elsif @xaxis_by == 'radius'
          xlabel = @radius_xlabel
      elsif @xaxis_by == 'r_div_R'
          xlabel = @r_div_R_xlabel
      elsif @xaxis_by == 'acoustic_r_div_R_phot'
          xlabel = @acoustic_r_div_R_phot_xlabel
      elsif @xaxis_by == 'logR'
          xlabel = @logR_xlabel
      elsif @xaxis_by == 'logxm'
          xlabel = @logxm_xlabel
      elsif @xaxis_by == 'logxq'
          xlabel = @logxq_xlabel
      else
          xlabel = @grid_xlabel
      end
      return xlabel
    end

    
    def show_by_q(name)
        xs = d.q
        show_properites(name, xs, @q_xlabel)
    end

    
    def show_by_grid(name)
        xs = d.zone
        show_properites(name, xs, @grid_xlabel)
    end
    
    def show_by_mass(name)
        if @use_mmid && d.mmid != nil
            xs = d.mmid
        else
            xs = d.mass
        end
        show_properites(name, xs, @mass_xlabel)
    end
    
    def show_by_logP(name)
        xs = d.logP
        show_properites(name, xs, @logP_xlabel)
    end
    
    def show_by_log_column_depth(name)
        xs = d.log_column_depth
        show_properites(name, xs, @log_col_depth_xlabel)
    end
    
    def show_by_logT(name)
        xs = d.logT
        show_properites(name, xs, @logT_xlabel)
    end
    
    def show_by_logR(name)
        xs = d.logR
        show_properites(name, xs, @logR_xlabel)
    end
    
    def show_by_radius(name)
        xs = d.radius
        show_properites(name, xs, @radius_xlabel)
    end
    
    def show_by_r_div_R(name)
        xs = d.r_div_R
        puts "xs.max for r_div_R #{xs.max}"
        show_properites(name, xs, @r_div_R_xlabel)
    end
    
    def show_by_acoustic_r_div_R_phot(name)
        xs = d.acoustic_r_div_R_phot
        show_properites(name, xs, @acoustic_r_div_R_phot_xlabel)
    end
    
    def show_by_logxm(name)
        xs = d.logxm
        show_properites(name, xs, @logxm_xlabel)
    end
    
    def show_by_logxq(name)
        xs = d.logxq
        show_properites(name, xs, @logxq_xlabel)
    end
    
    
    def show_by(name)
      if @xaxis_by == 'grid'
        show_by_grid(name)
      elsif @xaxis_by == 'q'
        show_by_q(name)
      elsif @xaxis_by == 'mass'
        show_by_mass(name)
      elsif @xaxis_by == 'radius'
        show_by_radius(name)
      elsif @xaxis_by == 'r_div_R'
        show_by_r_div_R(name)
      elsif @xaxis_by == 'acoustic_r_div_R_phot'
        show_by_acoustic_r_div_R_phot(name)
      elsif @xaxis_by == 'logR'
        show_by_logR(name)
      elsif @xaxis_by == 'logP'
        show_by_logP(name)
      elsif @xaxis_by == 'log_column_depth'
        show_by_log_column_depth(name)
      elsif @xaxis_by == 'logT'
        show_by_logT(name)
      elsif @xaxis_by == 'logxm'
        show_by_logxm(name)
      elsif @xaxis_by == 'logxq'
        show_by_logxq(name)
      elsif
        puts "problem in show_by @xaxis_by = #{@xaxis_by}"
      end
    end

    def trio_by
        trio('Profiles --- Age %s', 'power', 'abundances', 'convection')
    end

    def power_and_abundances
        duo('Power and Abundances --- Age %s', 'power', 'abundances')
    end

    def trio_by_both
        t.landscape unless t.in_subplot
        show_model_number
        column_margin = 0.28
        t.rescale(0.75)
        show_box_labels(get_star_age_title('Summary -- Age %s'))
        save_xaxis_by = @xaxis_by
        @xaxis_by = 'mass'
        t.subplot(t.column_margins('num_columns' => 2, 'column' => 1, 'column_margin' => column_margin)) { trio_by }
        @xaxis_by = 'logP'
        t.subplot(t.column_margins('num_columns' => 2, 'column' => 2, 'column_margin' => column_margin)) { trio_by }
        @xaxis_by = save_xaxis_by
    end


    def trio2_by_mass
        t.landscape
        trio('mass', 'Profiles', 'power', 'abundances', 'rho_T')
    end

    def trio2_and_history
        plot_and_history { trio2_by_mass }
    end
    
    def draw_pair(text, val, x_num, y)
        x_text = x_num - 0.48
        t.show_text('text' => text,
                'x' => x_text, 'y' => y, 'justification' => RIGHT_JUSTIFIED)
        t.show_text('text' => '{\sffamily ' + sprintf('%0.6f',val) + '}',
                'x' => x_num, 'y' => y, 'justification' => RIGHT_JUSTIFIED)
    end
    
    def draw_pair_log(text, val, x_num, y)
        draw_pair(text, val.safe_log10, x_num, y)
    end
    
    def info
        t.rescale(0.7)
        xloc = 1.86
        t.show_text('text' => sprintf('Age %s', d.star_age_str),
                'side' => TOP, 'position' => xloc, 'scale' => 0.9, 'shift' => 2,
                'justification' => RIGHT_JUSTIFIED, 'color' => Black)
        t.show_text('text' => sprintf('Mass %0.2f ($ \mathrm{M_{\odot}} $)', d.star_mass),
                'side' => TOP, 'position' => xloc, 'scale' => 0.9, 'shift' => 0.75,
                'justification' => RIGHT_JUSTIFIED, 'color' => Black)
        t.show_text('text' => sprintf('Model %i', d.model_number),
                'side' => BOTTOM, 'position' => xloc, 'scale' => 0.9, 'shift' => 1.75,
                'justification' => RIGHT_JUSTIFIED, 'color' => Black)
        t.rescale(0.58)
        y = 0.97; dy = -t.default_text_height_dy * 1.2
        draw_pair('$ \log L $', d.log_luminosity, xloc, y); y += dy;
        draw_pair('$ \log T_{eff} $', d.log_surface_temp, xloc, y); y += dy;
        draw_pair('$ \log T_{c} $', d.log_center_temp, xloc, y); y += dy;
        draw_pair('$ \log \rho_{c} $', d.log_center_density, xloc, y); y += dy;
        draw_pair('$ \Psi_{c} $', d.center_eta, xloc, y); y += dy;
        draw_pair('Mass', d.star_mass, xloc, y); y += dy;
        draw_pair('He Core', d.h1_boundary_mass, xloc, y); y += dy;
        draw_pair('C/O Core', d.he4_boundary_mass, xloc, y); y += dy;
        y += dy/2;
        draw_pair('$ T_{max} $ Mass', d.mass[d.shell_with_max_temp], xloc, y); y += dy;
        draw_pair_log('$ T_{max} $ $\log \rho $', d.density[d.shell_with_max_temp], xloc, y); y += dy;
        draw_pair_log('$ \log T_{max} $', d.temp_max, xloc, y); y += dy;
        draw_pair_log('$ \log \kappa_{max} $', d.opacity_max, xloc, y); y += dy;
        draw_pair_log('$ \log L_{max} $', d.luminosity_max, xloc, y); y += dy;
        draw_pair('$ \log \rho_{max} $', d.log_center_density, xloc, y); y += dy;
        draw_pair_log('$ \log R_{max} $', d.radius[d.surface_shell], xloc, y); y += dy;
        draw_pair_log('$ \log P_{max}  $', d.pressure[d.center_shell], xloc, y); y += dy;
        y += dy/2;
        draw_pair_log('$ \log L_{\nu}$', d.power_neu, xloc, y); y += dy;
        draw_pair_log('$ \log L_{PP}$', d.power_pp, xloc, y); y += dy;
        draw_pair_log('$ \log L_{CNO}$', d.power_cno, xloc, y); y += dy;
        draw_pair_log('$ \log L_{3 \alpha}$', d.power_3_alpha, xloc, y); y += dy;
        draw_pair_log('$ \log L_{C+\alpha}$', d.power_c_alpha, xloc, y); y += dy;
        draw_pair_log('$ \log L_{N+\alpha}$', d.power_n_alpha, xloc, y); y += dy;
        draw_pair_log('$ \log L_{O+\alpha}$', d.power_o_alpha, xloc, y); y += dy;
        y += dy/2;
        draw_pair('Center XH', d.center_h1, xloc, y); y += dy;
        draw_pair('XHe', d.center_he4, xloc, y); y += dy;
        draw_pair('XC', d.center_c12, xloc, y); y += dy;
        draw_pair('XN', d.center_n14, xloc, y); y += dy;
        draw_pair('XO', d.center_o16, xloc, y); y += dy;
        draw_pair('XNe', d.center_ne20, xloc, y); y += dy;
        y += dy/2;
        draw_pair_log('$ \log t_{nuclear}$', d.nuc_timescale, xloc, y); y += dy;
        draw_pair_log('$ \log t_{thermal}$', d.kh_timescale, xloc, y); y += dy;
        seconds_per_year = 3.155692597e7
        draw_pair_log('$ \log t_{dynamic}$', d.dynamic_time/seconds_per_year, xloc, y); y += dy;
        draw_pair_log('$ \log t_{step}$', d.time_step, xloc, y); y += dy;
        y += dy/2;
    end

    def full_profile
        t.title_shift += 0.5
        t.subplot('right_margin' => 0.17) { trio_by_both }
        t.set_subframe('left_margin' => 0.8)
        info
    end

    
    def plot5(title, plot1, plot2, plot3, plot4, plot5)
        show_model_number
        t.rescale(@subplot_scale)
        if @xaxis_by == 'mass'
          xlabel = @mass_xlabel
        elsif @xaxis_by == 'logP'
          xlabel = @logP_xlabel
        else
          xlabel = @grid_xlabel
        end
        num_plots = 5
        t.legend_scale = 0.6
        t.legend_text_ystart = 0.95
        t.legend_text_dy = 1.2
        row_margin = 0.02
        title = get_star_age_title(title)
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 1, 'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            show_plot_by(plot1, title)
        end
        t.reset_legend_info
        t.top_edge_type = AXIS_LINE_ONLY
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 2, 'row_margin' => row_margin)) do 
            t.title_visible = t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            show_plot_by(plot2, nil)
        end
        t.reset_legend_info
        t.top_edge_type = AXIS_LINE_ONLY
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 3, 'row_margin' => row_margin)) do 
            t.title_visible = t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            show_plot_by(plot3, nil)
        end
        t.reset_legend_info
        t.top_edge_type = AXIS_LINE_ONLY
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 4, 'row_margin' => row_margin)) do 
            t.title_visible = t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            show_plot_by(plot4, nil)
        end
        t.reset_legend_info
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 5, 'row_margin' => row_margin)) do 
            t.title_visible = false
            show_plot_by(plot5, nil)
        end
    end

    
    def quad(title, plot1, plot2, plot3, plot4)
        show_model_number
        t.rescale(@subplot_scale)
        if @xaxis_by == 'mass'
          xlabel = @mass_xlabel
        elsif @xaxis_by == 'logP'
          xlabel = @logP_xlabel
        else
          xlabel = @grid_xlabel
        end
        num_plots = 4
        t.legend_scale = 0.6
        t.legend_text_ystart = 0.95
        t.legend_text_dy = 1.2
        row_margin = 0.02
        title = get_star_age_title(title)
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 1, 'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            show_plot_by(plot1, title)
        end
        t.reset_legend_info
        t.top_edge_type = AXIS_LINE_ONLY
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 2, 'row_margin' => row_margin)) do 
            t.title_visible = t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            show_plot_by(plot2, nil)
        end
        t.reset_legend_info
        t.top_edge_type = AXIS_LINE_ONLY
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 3, 'row_margin' => row_margin)) do 
            t.title_visible = t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            show_plot_by(plot3, nil)
        end
        t.reset_legend_info
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 4, 'row_margin' => row_margin)) do 
            t.title_visible = false
            show_plot_by(plot4, nil)
        end
    end

    
    def trio(title, plot1, plot2, plot3)
        show_model_number
        t.rescale(@subplot_scale)
        if @xaxis_by == 'mass'
          xlabel = @mass_xlabel
        elsif @xaxis_by == 'logP'
          xlabel = @logP_xlabel
        else
          xlabel = @grid_xlabel
        end
        num_plots = 3
        t.legend_scale = 0.6
        t.legend_text_ystart = 0.95
        t.legend_text_dy = 1.2
        row_margin = 0.02
        title = get_star_age_title(title)
        @doing_multiple = true
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 1, 'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            show_plot_by(plot1, title)
        end
        t.reset_legend_info
        t.top_edge_type = AXIS_LINE_ONLY
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 2, 'row_margin' => row_margin)) do 
            t.title_visible = t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            show_plot_by(plot2, nil)
        end
        t.reset_legend_info
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 3, 'row_margin' => row_margin)) do 
            t.title_visible = false
            show_plot_by(plot3, nil)
        end
        @doing_multiple = false
    end

    
    def duo(title, plot1, plot2)
        show_model_number
        t.rescale(@subplot_scale)
        if @xaxis_by == 'mass'
          xlabel = @mass_xlabel
        elsif @xaxis_by == 'logP'
          xlabel = @logP_xlabel
        else
          xlabel = @grid_xlabel
        end
        num_plots = 2
        t.legend_scale = 0.6
        t.legend_text_ystart = 0.95
        t.legend_text_dy = 1.2
        row_margin = 0.02
        title = get_star_age_title(title)
        @doing_multiple = true
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 1, 'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            show_plot_by(plot1)
        end
        t.reset_legend_info
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 2, 'row_margin' => row_margin)) do 
            t.title_visible = false
            show_plot_by(plot2)
        end
        @doing_multiple = false
    end

    
    def show_plot_by(plot)
      if @xaxis_by == 'mass'
          show_by_mass(plot)
      elsif @xaxis_by == 'logP'
          show_by_logP(plot)
      elsif @xaxis_by == 'log_column_depth'
          show_by_log_column_depth(plot)
      elsif @xaxis_by == 'logT'
          show_by_logT(plot)
      elsif @xaxis_by == 'radius'
          show_by_radius(plot)
      elsif @xaxis_by == 'logR'
          show_by_logR(plot)
      elsif @xaxis_by == 'logxm'
          show_by_logxm(plot)
      elsif @xaxis_by == 'logxq'
          show_by_logxq(plot)
      else
          show_by_grid(plot)
      end
    end

end
